/*  This file is part of SAIL (https://github.com/HappySeaFox/sail)

    Copyright (c) 2020-2021 Dmitry Baryshev

    The MIT License

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
*/

#include <stdio.h>

#include <sail-manip/sail-manip.h>
#include <sail/sail.h>

#include "munit.h"

static MunitResult test_best_conversion_grayscale(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE, SAIL_PIXEL_FORMAT_BPP24_RGB};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(
            sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE, pixel_formats, pixel_formats_length), ==,
            SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP24_RGB, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(
            sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE, pixel_formats, pixel_formats_length), ==,
            SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP1_INDEXED, SAIL_PIXEL_FORMAT_BPP2_INDEXED};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(
            sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE, pixel_formats, pixel_formats_length), ==,
            SAIL_PIXEL_FORMAT_UNKNOWN);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_indexed(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP24_RGB, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP8_INDEXED, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP24_RGB);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE, SAIL_PIXEL_FORMAT_BPP24_RGB};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP1_INDEXED, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP24_RGB);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE,
                                                      SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP1_INDEXED, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_rgb(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP32_RGBA, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP32_RGBA);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE, SAIL_PIXEL_FORMAT_BPP32_RGBA};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP32_RGBA);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE,
                                                      SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_grayscale_alpha(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA,
                                                      SAIL_PIXEL_FORMAT_BPP24_RGB};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(
            sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA, pixel_formats, pixel_formats_length), ==,
            SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE_ALPHA,
                                                      SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_ALPHA,
                                                      SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_ALPHA};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(
            sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE, pixel_formats, pixel_formats_length), ==,
            SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE_ALPHA);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_rgb555_565(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP16_RGB565, SAIL_PIXEL_FORMAT_BPP16_RGBA};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP16_RGB555, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP16_RGB565);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP16_BGR555, SAIL_PIXEL_FORMAT_BPP16_BGR565};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_BGR, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP16_BGR555);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_cmyk(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP32_CMYK, SAIL_PIXEL_FORMAT_BPP24_RGB};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP24_RGB);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP32_CMYK, SAIL_PIXEL_FORMAT_BPP64_CMYK};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP32_CMYK);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_rgba16(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP16_RGBA, SAIL_PIXEL_FORMAT_BPP32_RGBA};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP32_RGBA, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP32_RGBA);
    }

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP64_RGBA, SAIL_PIXEL_FORMAT_BPP16_RGBA};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP32_RGBA, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP64_RGBA);
    }

    return MUNIT_OK;
}

static MunitResult test_best_conversion_yuv(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    {
        const enum SailPixelFormat pixel_formats[] = {SAIL_PIXEL_FORMAT_BPP24_YUV, SAIL_PIXEL_FORMAT_BPP24_YCBCR,
                                                      SAIL_PIXEL_FORMAT_BPP24_RGB};
        const size_t pixel_formats_length          = sizeof(pixel_formats) / sizeof(pixel_formats[0]);

        munit_assert_int(sail_closest_pixel_format(SAIL_PIXEL_FORMAT_BPP24_RGB, pixel_formats, pixel_formats_length),
                         ==, SAIL_PIXEL_FORMAT_BPP24_YCBCR);
    }

    return MUNIT_OK;
}

static MunitResult test_can_convert_float_formats(const MunitParameter params[], void* user_data)
{
    (void)params;
    (void)user_data;

    /* Test float grayscale conversions */
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE_HALF, SAIL_PIXEL_FORMAT_BPP8_GRAYSCALE));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_FLOAT, SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP16_GRAYSCALE, SAIL_PIXEL_FORMAT_BPP32_GRAYSCALE_FLOAT));

    /* Test float RGB conversions */
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP48_RGB_HALF, SAIL_PIXEL_FORMAT_BPP24_RGB));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT, SAIL_PIXEL_FORMAT_BPP48_RGB));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP24_RGB, SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT));

    /* Test float RGBA conversions */
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF, SAIL_PIXEL_FORMAT_BPP32_RGBA));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT, SAIL_PIXEL_FORMAT_BPP64_RGBA));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP32_RGBA, SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT));

    /* Test float to indexed */
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT, SAIL_PIXEL_FORMAT_BPP8_INDEXED));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT, SAIL_PIXEL_FORMAT_BPP8_INDEXED));

    /* Test bidirectional float conversions */
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP48_RGB_HALF, SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP96_RGB_FLOAT, SAIL_PIXEL_FORMAT_BPP48_RGB_HALF));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF, SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT));
    munit_assert_true(sail_can_convert(SAIL_PIXEL_FORMAT_BPP128_RGBA_FLOAT, SAIL_PIXEL_FORMAT_BPP64_RGBA_HALF));

    return MUNIT_OK;
}

// clang-format off
static MunitTest test_suite_tests[] = {
    { (char *)"/grayscale",         test_best_conversion_grayscale,       NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/indexed",           test_best_conversion_indexed,         NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/rgb",               test_best_conversion_rgb,             NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/grayscale-alpha",   test_best_conversion_grayscale_alpha, NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/rgb555-565",        test_best_conversion_rgb555_565,      NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/cmyk",              test_best_conversion_cmyk,            NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/rgba16",            test_best_conversion_rgba16,          NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/yuv",               test_best_conversion_yuv,             NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },
    { (char *)"/can-convert-float", test_can_convert_float_formats,       NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL },

    { NULL, NULL, NULL, NULL, MUNIT_TEST_OPTION_NONE, NULL }
};

static const MunitSuite test_suite = {
    (char *)"/best-conversion", test_suite_tests, NULL, 1, MUNIT_SUITE_OPTION_NONE
};
// clang-format on

int main(int argc, char* argv[MUNIT_ARRAY_PARAM(argc + 1)])
{
    return munit_suite_main(&test_suite, NULL, argc, argv);
}
