/*
 *   Copyright (C) 2009 Dan Vratil <vratil@progdansoft.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 3 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "kopetecontactsrunner.h"

#include <QtDBus/QDBusInterface>
#include <QtDBus/QDBusReply>

#include <kicon.h>



KopeteContactsRunner::KopeteContactsRunner(QObject *parent, const QVariantList& args)
    : Plasma::AbstractRunner(parent, args)
{
    Q_UNUSED(args);
    setObjectName("KopeteContacts");

    c_icon = KIcon("kopete");
    setIgnoredTypes(Plasma::RunnerContext::Directory | 
		    Plasma::RunnerContext::File |
                    Plasma::RunnerContext::NetworkLocation |
		    Plasma::RunnerContext::Executable |
		    Plasma::RunnerContext::ShellCommand);
    Plasma::RunnerSyntax syntax = Plasma::RunnerSyntax(":q:", i18n("Open Kopete chat window to contact."));
    syntax.addExampleQuery("kopete :q:");
    addSyntax(syntax);
    syntax = Plasma::RunnerSyntax("<status>", i18n("Set Kopete status"));
    syntax.addExampleQuery("status <status>");
    addSyntax(syntax);
    
    setSpeed(AbstractRunner::SlowSpeed);
}

KopeteContactsRunner::~KopeteContactsRunner()
{
}

void KopeteContactsRunner::match(Plasma::RunnerContext &context)
{
     //don't be case-sensitive
    QString searchedName = context.query().toLower();
    
    /* Is set true if "kopete contactname" syntax is used. Then searched terms of any length are 
       accepted. If false, only terms longer then 3 are accepted */
    bool searchAll = false;
    // By default we search for contacts
    bool searchStatus = false;
    
    // if syntax "kopete contactname" is used, remove the "kopete" from the beginning of the string
    if (searchedName.startsWith("kopete ")) {
        searchAll = true;
        searchedName.remove("kopete ", Qt::CaseInsensitive);
    }
    if (searchedName.startsWith("status ")) {
	searchAll = true;
	searchedName.remove("status ", Qt::CaseInsensitive);
	searchStatus = true;
    }

    // search only if term is longer then 3 chars
    if (not searchAll) {
	if (searchedName.size() < 3) {
	    return;
	}
    } else {
	if (searchedName.isEmpty()) {
	    return;
	}
    }


    // Try to open DBus connection to Kopete otherwise this plugin is useless.
    QDBusInterface kopeteDBusTest("org.kde.kopete", "/Kopete", "org.freedesktop.DBus.Introspectable");
    QDBusReply<QString>kopeteReply = kopeteDBusTest.call("Introspect");
    if (!kopeteReply.isValid()) {
            return;
    }

    QList<Plasma::QueryMatch> matches;

    if (searchStatus == false) {
	// get list of all contact's IDs
	QDBusInterface kopeteDBus("org.kde.kopete","/Kopete","org.kde.Kopete");
	QDBusReply<QStringList> kopeteContacts = kopeteDBus.call("contacts");

	QString contactName;
	//find all contacts that matches
	for (int i = 0; i < kopeteContacts.value().size(); i++) {
	    /*Check if the current context is still valid, if not quit (another
	      context is already running) */
	    if (!context.isValid()) {
		return;
	    }
	    //Get informations about the current contact
	    QDBusReply<QVariantMap>contactData = kopeteDBus.call("contactProperties",kopeteContacts.value().at(i));
	    //Store the contact name
	    contactName = contactData.value().value("display_name").toString();
	    //Does the current displayName match the searched?
	    bool matched = contactName.contains(searchedName, Qt::CaseInsensitive);
	    
	    // Yes it does!
	    if (matched) {
		Plasma::QueryMatch matchItem(this);

		matchItem.setIcon(c_icon);
		//User's display name
		matchItem.setText(contactName);
		//Status message
		QString statusMessage = contactData.value().value("status_message").toString();
		if (not statusMessage.isEmpty()) {
		    statusMessage.prepend(": ");
		}
		matchItem.setSubtext(contactData.value().value("status").toString().append(statusMessage));
		//Store contact's ID
		matchItem.setData(QString("contact ").append(kopeteContacts.value().at(i)));

		if (contactName == searchedName) {
		    matchItem.setType(Plasma::QueryMatch::ExactMatch);
		} else {
		    matchItem.setType(Plasma::QueryMatch::PossibleMatch);
		}
		matches.append(matchItem);
	    }
	}
    } else { //searchStatus==true
	if (QString("online").contains(searchedName,Qt::CaseInsensitive)) {
	    Plasma::QueryMatch match(this);
	    match.setIcon(KIcon("user-online"));
	    match.setText("Go online");
	    match.setSubtext("Set status 'online' for all Kopete accounts");
	    match.setData("status online");
	    if (searchedName == "online") {
		match.setType(Plasma::QueryMatch::ExactMatch);
	    } else {
		match.setType(Plasma::QueryMatch::PossibleMatch);
	    }
	    matches.append(match);
	    
	}
	if (QString("offline").contains(searchedName, Qt::CaseInsensitive)) {
	    Plasma::QueryMatch match(this);
	    match.setIcon(KIcon("user-offline"));
	    match.setText("Go offline");
	    match.setSubtext("Set status 'offline' for all Kopete accounts");
	    match.setData("status offline");
	    if (searchedName == "offline") {
		match.setType(Plasma::QueryMatch::ExactMatch);
	    } else {
		match.setType(Plasma::QueryMatch::PossibleMatch);
	    }
	    matches.append(match);
	    
	}
	if (QString("away").contains(searchedName, Qt::CaseInsensitive)) {
	    Plasma::QueryMatch match(this);
	    match.setIcon(KIcon("user-away"));
	    match.setText("Go away");
	    match.setSubtext("Set status 'away' for all Kopete accounts");
	    match.setData("status away");
	    if (searchedName == "away") {
		match.setType(Plasma::QueryMatch::ExactMatch);
	    } else {
		match.setType(Plasma::QueryMatch::PossibleMatch);
	    }
	    matches.append(match);
	    
	}
	if (QString("busy").contains(searchedName, Qt::CaseInsensitive)) {
	    Plasma::QueryMatch match(this);
	    match.setIcon(KIcon("user-busy"));
	    match.setText("Go busy");
	    match.setSubtext("Set status 'busy' for all Kopete accounts");
	    match.setData("status busy");
	    if (searchedName == "busy") {
		match.setType(Plasma::QueryMatch::ExactMatch);
	    } else {
		match.setType(Plasma::QueryMatch::PossibleMatch);
	    }
	    matches.append(match);	
	    
	}
	if (QString("invisible").contains(searchedName, Qt::CaseInsensitive)) {
	    Plasma::QueryMatch match(this);
	    match.setIcon(KIcon("user-invisible"));
	    match.setText("Go invisible");
	    match.setSubtext("Set status 'invisible' for all Kopete accounts");
	    match.setData("status invisible");
	    if (searchedName == "invisible") {
		match.setType(Plasma::QueryMatch::ExactMatch);
	    } else {
		match.setType(Plasma::QueryMatch::PossibleMatch);
	    }
	    matches.append(match);
	}
    } //end of if(searchStatus==false)
	  
      

    context.addMatches(searchedName,matches);
}

void KopeteContactsRunner::run(const Plasma::RunnerContext &context, const Plasma::QueryMatch &match)
{
    Q_UNUSED(context);
    
    QString command = match.data().toString();
    
    // Try to open DBus connection to Kopete otherwise this plugin is useless.
    QDBusInterface kopeteDBusTest("org.kde.kopete", "/Kopete", "org.freedesktop.DBus.Introspectable");
    QDBusReply<QString>kopeteReply = kopeteDBusTest.call("Introspect");
    if (!kopeteReply.isValid()) {
          return;
    }
    QDBusInterface kopeteDBus("org.kde.kopete","/Kopete","org.kde.Kopete");
    
    if (command.startsWith("status ")) {
	command.remove("status ");
	if (command == "offline") {
	  kopeteDBus.call("disconnectAll");
	} else {
	  kopeteDBus.call("setOnlineStatus",command,"");
	}
    } else if (command.startsWith("contact ")) {
	command.remove("contact ");
	kopeteDBus.call("openChat",command);
    }

}

#include "kopetecontactsrunner.moc"
