/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmviewport.c
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include "pgmplugintestsuite.h"

/* test default values */
PGM_START_TEST (test_viewport_default_values)
{
  PgmViewport *viewport;
  PgmError ret;
  gint width, height;
  PgmViewportCursor cursor;
  gboolean fullscreen;
  GdkPixbuf *icon;

  ret = pgm_viewport_factory_make (PGM_TEST_PLUGIN, &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_make error");

  /* size */
  ret = pgm_viewport_get_size (viewport, &width, &height);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_size error");
  fail_if (width != 800 || height != 600, "bad default size");

  /* cursor */
  ret = pgm_viewport_get_cursor (viewport, &cursor);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_cursor error");
  fail_if (cursor != PGM_VIEWPORT_LEFT_ARROW, "bad default cursor");

  /* icon */
  ret = pgm_viewport_get_icon (viewport, &icon);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_icon error");
  fail_if (icon != NULL, "bad default icon");

  /* fullscreen */
  ret = pgm_viewport_get_fullscreen (viewport, &fullscreen);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_fullscreen error");
  fail_if (fullscreen != FALSE, "bad default fullscreen");

  gst_object_unref (GST_OBJECT_CAST (viewport));
  ASSERT_OBJECT_REFCOUNT (viewport, "viewport", 0);

}
PGM_END_TEST;

/* test set/get correctness */
PGM_START_TEST (test_viewport_set_get_correctness)
{
  PgmViewport *viewport;
  PgmError ret;
  gint width, height;
  PgmViewportCursor cursor;
  gboolean fullscreen;
  GdkPixbuf *icon_src, *icon;

  ret = pgm_viewport_factory_make (PGM_TEST_PLUGIN, &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_make error");

  /* size */
  ret = pgm_viewport_set_size (viewport, 640, 480);
  fail_if (ret != PGM_ERROR_OK, "viewport_set_size error");
  ret = pgm_viewport_get_size (viewport, &width, &height);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_size error");
  fail_if (width != 640 || height != 480, "size not set");

  /* cursor */
  ret = pgm_viewport_set_cursor (viewport, PGM_VIEWPORT_NONE);
  fail_if (ret != PGM_ERROR_OK, "viewport_set_cursor error");
  ret = pgm_viewport_get_cursor (viewport, &cursor);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_cursor error");
  fail_if (cursor != PGM_VIEWPORT_NONE, "cursor not set");

  /* icon */
  icon_src = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 32, 32);
  fail_if (icon_src == NULL, "problem in gdk_pixbuf_new");
  ret = pgm_viewport_set_icon (viewport, icon_src);
  fail_if (ret != PGM_ERROR_OK, "viewport_set_icon error");
  ret = pgm_viewport_get_icon (viewport, &icon);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_icon error");
  fail_if (icon != icon_src, "icon not set");

  /* fullscreen */
  ret = pgm_viewport_set_fullscreen (viewport, TRUE);
  fail_if (ret != PGM_ERROR_OK, "viewport_set_fullscreen error");
  ret = pgm_viewport_get_fullscreen (viewport, &fullscreen);
  fail_if (ret != PGM_ERROR_OK, "viewport_get_fullscreen error");
  fail_if (fullscreen != TRUE, "fullscreen not set");

  gst_object_unref (GST_OBJECT_CAST (viewport));
}
PGM_END_TEST

PGM_START_TEST (test_viewport_geometry)
{
  PgmViewport *viewport = NULL;
  PgmCanvas *canvas = NULL;
  gfloat v_x, v_y, v_z, c_x, c_y, c_z;

  ASSERT_PGM_SUCCESS (pgm_viewport_factory_make (PGM_TEST_PLUGIN, &viewport));

  canvas = pgm_canvas_new ();
  fail_if (NULL == canvas, "canvas creation failed");

  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_canvas_set_size (NULL, 4, 3));
  ASSERT_PGM_SUCCESS (pgm_canvas_set_size (canvas, 4, 3));

  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_set_size (NULL, 640, 480));
  ASSERT_PGM_SUCCESS (pgm_viewport_set_size (viewport, 640, 480));

  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_from_canvas (viewport,
                                                &v_x, &v_y, &v_z,
                                                1.1f, 2.2f, 3.3f));
  ASSERT_PGM_SUCCESS (pgm_viewport_set_canvas (viewport, canvas));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_from_canvas (NULL,
                                                            &v_x, &v_y, &v_z,
                                                            1.1f, 2.2f, 3.3f));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_from_canvas (viewport,
                                                            NULL, &v_y, &v_z,
                                                            1.1f, 2.2f, 3.3f));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_from_canvas (viewport,
                                                            &v_x, NULL, &v_z,
                                                            1.1f, 2.2f, 3.3f));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_from_canvas (viewport,
                                                            &v_x, &v_y, NULL,
                                                            1.1f, 2.2f, 3.3f));
  ASSERT_PGM_SUCCESS (pgm_viewport_from_canvas (viewport,
                                                &v_x, &v_y, &v_z,
                                                1.1f, 2.2f, 3.3f));
  ASSERT_PGM_SUCCESS (pgm_viewport_set_canvas (viewport, NULL));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_to_canvas (viewport,
                                                          &c_x, &c_y, &c_z,
                                                          v_x, v_y, v_z));
  ASSERT_PGM_SUCCESS (pgm_viewport_set_canvas (viewport, canvas));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_to_canvas (NULL,
                                                          &c_x, &c_y, &c_z,
                                                          v_x, v_y, v_z));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_to_canvas (viewport,
                                                          NULL, &c_y, &c_z,
                                                          v_x, v_y, v_z));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_to_canvas (viewport,
                                                          &c_x, NULL, &c_z,
                                                          v_x, v_y, v_z));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_to_canvas (viewport,
                                                          &c_x, &c_y, NULL,
                                                          v_x, v_y, v_z));
  ASSERT_PGM_SUCCESS (pgm_viewport_to_canvas (viewport,
                                              &c_x, &c_y, &c_z,
                                              v_x, v_y, v_z));
  ASSERT_PGM_SUCCESS (pgm_viewport_to_canvas (viewport,
                                              &c_x, &c_y, &c_z,
                                              v_x, v_y, v_z));
  fail_unless (fabs (c_x - 1.1f) <= 0.0001, "conversion from canvas then to canvas failed for the x coordinate");
  fail_unless (fabs (c_y - 2.2f) <= 0.0001, "conversion from canvas then to canvas failed for the y coordinate");
  fail_unless (fabs (c_z - 3.3f) <= 0.0001, "conversion from canvas then to canvas failed for the z coordinate");

  gst_object_unref (viewport);
}
PGM_END_TEST

static void
on_pixels_read (PgmViewport *viewport,
                guint        width,
                guint        height,
                gpointer     pixels,
                gpointer     user_data)
{
  gboolean *called = (gboolean *) user_data;

  *called = TRUE;
}

static gpointer
thread_init (gpointer data)
{
  pgm_main ();
}

/* TODO: have pgm_main running in another thread and check that the callback is
 * called when it should */
PGM_START_TEST (test_viewport_read_pixels)
{
  PgmViewport *viewport = NULL;
  guint8 * pixels = NULL;
  gint width = 0, height = 0;
  gulong signal_handler = 0;
  GThread *pgm_thread  = NULL;
  gboolean callback_called = FALSE;

  ASSERT_PGM_SUCCESS (pgm_viewport_factory_make (PGM_TEST_PLUGIN, &viewport));
  ASSERT_PGM_SUCCESS (pgm_viewport_get_size (viewport, &width, &height));

  pgm_thread = g_thread_create ( thread_init, NULL, TRUE, NULL);
  fail_unless (NULL != pgm_thread, "thread creation");

  pixels = g_malloc (width * height * 4);

  /* sleeps to ensure pgm_main() is running */
  g_usleep (1000000);

  /* fails because no signal connected */
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_read_pixels (viewport,
                                                            0, 0, width, height,
                                                            pixels));

  signal_handler = g_signal_connect (G_OBJECT (viewport),
                                     "pixels-read",
                                     G_CALLBACK (on_pixels_read),
                                     &callback_called);

  /* param checks */
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_read_pixels (NULL,
                                                            0, 0, width, height,
                                                            pixels));
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_read_pixels (viewport,
                                                            0, 0, width, height,
                                                            NULL));
  /* wait for 1 sec for possible callback call */
  g_usleep (1000000);
  fail_if (callback_called, "callback has been called, but it shouldn't have");

  ASSERT_PGM_SUCCESS (pgm_viewport_read_pixels (viewport,
                                                0, 0, width, height,
                                                pixels));
  /* wait for 1 sec for possible callback call */
  g_usleep (1000000);
  fail_unless (callback_called, "callback has not been called");
  g_free (pixels);
  pixels = NULL;

  pixels = g_malloc (width * height * 4);
  g_signal_handler_block (G_OBJECT (viewport), signal_handler);
  /* should fail because there is no unblocked signal handler */
  ASSERT_PGM_ERROR_WITH_CRITICAL (pgm_viewport_read_pixels (viewport,
                                                            0, 0, width, height,
                                                            pixels));

  g_signal_handler_unblock (G_OBJECT (viewport), signal_handler);
  ASSERT_PGM_SUCCESS (pgm_viewport_read_pixels (viewport,
                                                0, 0, width, height,
                                                pixels));
  /* wait for 1 sec for possible callback call */
  g_usleep (1000000);
  fail_unless (callback_called, "callback has not been called");
  g_free (pixels);
  pixels = NULL;

  pgm_main_quit ();
  gst_object_unref (viewport);

  g_thread_join (pgm_thread);

}
PGM_END_TEST

Suite *
pgm_viewport_suite (void)
{
  Suite *s = suite_create ("PgmViewport");
  TCase *tc_chain = tcase_create ("pgmviewport tests");

  tcase_set_timeout (tc_chain, 10);

  PGM_USE_STANDARD_FIXTURES (tc_chain);

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_viewport_default_values);
  tcase_add_test (tc_chain, test_viewport_set_get_correctness);
  tcase_add_test (tc_chain, test_viewport_geometry);
  tcase_add_test (tc_chain, test_viewport_read_pixels);

  return s;
}

