/*
 *  Copyright (c) 2007,2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENGTL_DEBUG_H_
#define _OPENGTL_DEBUG_H_

#include <iostream>
#include <stdlib.h>
#include <GTLCore/String.h>
#include <GTLCore/Export.h>

class DebugDeleter;

namespace LLVMBackend {
  class ExpressionResult;
  std::ostream& operator<< (std::ostream& ostr, const ExpressionResult& expressionResult);
}

namespace GTLCore {
  /**
   * @internal
   * @ingroup GTLCore
   *
   * Control the debug flow in OpenGTL.
   */
  class Debug {
    friend class ::DebugDeleter;
    public:
      Debug();
      ~Debug();
      GTLCORE_DEBUG_EXPORT static bool isDebugEnabled( const String& _libraryName, const String& _fileName, const String& _functionName );
      /**
       * @return the debug stream used to output debug messages
       */
      GTLCORE_DEBUG_EXPORT static std::ostream& debug( const String& _libraryName, const String& _fileName, int _line, const String& _functionName );
      /**
       * @return the warning stream used to output warning messages
       */
      GTLCORE_EXPORT static std::ostream& warning( const String& _libraryName, const String& _fileName, int _line, const String& _functionName );
      /**
       * @return the error stream used to output error messages
       */
      GTLCORE_EXPORT static std::ostream& error( const String& _libraryName, const String& _fileName, int _line, const String& _functionName );
    private:
      struct Private;
  };
  class Function;
  class PixelDescription;
  class RegionI;
  class RegionF;
  class ScopedName;
  class Type;
  struct Token;
  class Value;
  class Color;

  std::ostream& operator<< (std::ostream& ostr, const PixelDescription& pixelDescription);
  GTLCORE_EXPORT std::ostream& operator<< (std::ostream& ostr, const RegionI& region);
  GTLCORE_EXPORT std::ostream& operator<< (std::ostream& ostr, const RegionF& region);
  GTLCORE_TESTS_OR_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const ScopedName& name);
  GTLCORE_TESTS_OR_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const Type& type);
  GTLCORE_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const Token& token);
  GTLCORE_TESTS_OR_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const Color& color);
  GTLCORE_TESTS_OR_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const Value& value);
  std::ostream& operator<< (std::ostream& ostr, const Function& function);
}

namespace llvm {
  class Function;
  class Module;
  class Value;
  class Type;
  GTLCORE_EXPORT std::ostream& operator<< (std::ostream& ostr, const llvm::Module& mod);
  GTLCORE_TESTS_OR_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const llvm::Value& val);
  GTLCORE_DEBUG_EXPORT std::ostream& operator<< (std::ostream& ostr, const llvm::Type& type);
}

#if (defined(__GNUC__) && !( defined(__sun) || defined(sun) )) || ( ( defined(hpux) || defined(__hpux) ) && ( defined(__HP_aCC) || __cplusplus >= 199707L )  )
#  define FUNC_INFO __PRETTY_FUNCTION__
#elif defined(_MSC_VER) && _MSC_VER > 1300
#  define FUNC_INFO __FUNCSIG__
#else
#  define FUNC_INFO ""
#endif

/**
 * Use this macro to display in the output stream the name of a variable followed by its value.
 */
#define ppVar( var ) #var << "=" << var << " "

#define GTL_WARNING(msg) \
  GTLCore::Debug::warning(COUMPONENT_NAME, __FILE__, __LINE__, FUNC_INFO ) << msg << std::endl;

#define GTL_ERROR(msg) \
  GTLCore::Debug::error(COUMPONENT_NAME, __FILE__, __LINE__, FUNC_INFO ) << msg << std::endl;

#define GTL_ABORT(msg) \
  GTL_ERROR(msg); \
  abort();

#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT

#include <assert.h>

#define GTL_DEBUG(msg) \
  if(GTLCore::Debug::isDebugEnabled(COUMPONENT_NAME, __FILE__, FUNC_INFO ) ) { \
    GTLCore::Debug::debug(COUMPONENT_NAME, __FILE__, __LINE__, FUNC_INFO ) << msg << std::endl; \
  }

#define GTL_ASSERT(assrt) \
  if( not (assrt ) ) \
  { \
    GTL_ABORT( "Assertion failed: " << #assrt ); \
  }
#define GTL_ASSERT_X(assrt, msg) \
  if( not (assrt ) ) \
  { \
    GTL_ABORT( "Assertion failed: " << #assrt << " (" << msg << ")" ); \
  }
#define GTL_CHECK_PTR(ptr) \
  if( not (assrt ) ) \
  { \
    GTL_ABORT( "Null pointer: " << #ptr ); \
  }
#define GTL_CHECK_EQUAL(val1, val2) \
  if( val1 != val2 ) \
  { \
    GTL_ABORT( #val1 << " != " << #val2 ); \
  }
#define GTL_CHECK_LLVM_EQUAL(val1, val2) \
  if( val1 != val2 ) \
  { \
    GTL_ABORT( *(val1) << " != " << *(val2) ); \
  }


#include <vector>

GTLCORE_DEBUG_EXPORT void compareFunctionParameters( llvm::Value* func, const std::vector<llvm::Value*>& params  );

#define GTL_COMPARE_FUNCTION_PARAMETERS( _FUNC_, _PARAMS_ ) \
  compareFunctionParameters( _FUNC_, _PARAMS_ );

#else

#define GTL_DEBUG(msg)
#define GTL_ASSERT(assrt)
#define GTL_ASSERT_X(assrt, msg)
#define GTL_CHECK_PTR(ptr) (void)ptr;
#define GTL_CHECK_EQUAL(val1, val2) (void)val1; (void)val2;
#define GTL_CHECK_LLVM_EQUAL(val1, val2) (void)val1; (void)val2;
#define GTL_COMPARE_FUNCTION_PARAMETERS( _FUNC_, _PARAMS_ ) (void)_FUNC_; (void)_PARAMS_; \

#endif

#endif
