/* MIT License
 *
 * Copyright (c) 2023 Brad House
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * SPDX-License-Identifier: MIT
 */
#include "ares_setup.h"
#include "ares.h"
#include "ares_private.h"
#include "ares__htable.h"
#include "ares__htable_asvp.h"


struct ares__htable_asvp {
  ares__htable_asvp_val_free_t free_val;
  ares__htable_t              *hash;
};


typedef struct {
  ares_socket_t        key;
  void                *val;
  ares__htable_asvp_t *parent;
} ares__htable_asvp_bucket_t;


void ares__htable_asvp_destroy(ares__htable_asvp_t *htable)
{
  if (htable == NULL)
    return;

  ares__htable_destroy(htable->hash);
  ares_free(htable);
}


static unsigned int hash_func(const void *key, unsigned int seed)
{
  const ares_socket_t *arg = key;
  return ares__htable_hash_FNV1a((const unsigned char *)arg, sizeof(*arg),
                                 seed);
}


static const void *bucket_key(const void *bucket)
{
  const ares__htable_asvp_bucket_t *arg = bucket;
  return &arg->key;
}


static void bucket_free(void *bucket)
{
  ares__htable_asvp_bucket_t *arg = bucket;

  if (arg->parent->free_val)
    arg->parent->free_val(arg->val);

  ares_free(arg);
}


static unsigned int key_eq(const void *key1, const void *key2)
{
  const ares_socket_t *k1 = key1;
  const ares_socket_t *k2 = key2;

  if (*k1 == *k2)
    return 1;

  return 0;
}


ares__htable_asvp_t *ares__htable_asvp_create(
    ares__htable_asvp_val_free_t val_free)
{
  ares__htable_asvp_t *htable = ares_malloc(sizeof(*htable));
  if (htable == NULL)
    goto fail;

  htable->hash = ares__htable_create(hash_func,
                                     bucket_key,
                                     bucket_free,
                                     key_eq);
  if (htable->hash == NULL)
    goto fail;

  htable->free_val = val_free;

  return htable;

fail:
  if (htable) {
    ares__htable_destroy(htable->hash);
    ares_free(htable);
  }
  return NULL;
}


unsigned int ares__htable_asvp_insert(ares__htable_asvp_t *htable,
                                      ares_socket_t key, void *val)
{
  ares__htable_asvp_bucket_t *bucket = NULL;

  if (htable == NULL)
    goto fail;

  bucket = ares_malloc(sizeof(*bucket));
  if (bucket == NULL)
    goto fail;

  bucket->parent = htable;
  bucket->key    = key;
  bucket->val    = val;

  if (!ares__htable_insert(htable->hash, bucket))
    goto fail;

  return 1;

fail:
  if (bucket) {
    ares_free(bucket);
  }
  return 0;
}


unsigned int ares__htable_asvp_get(ares__htable_asvp_t *htable,
                                   ares_socket_t key, void **val)
{
  ares__htable_asvp_bucket_t *bucket = NULL;

  if (val)
    *val = NULL;

  if (htable == NULL)
    return 0;

  bucket = ares__htable_get(htable->hash, &key);
  if (bucket == NULL)
    return 0;

  if (val)
    *val = bucket->val;
  return 1;
}


void *ares__htable_asvp_get_direct(ares__htable_asvp_t *htable,
                                   ares_socket_t key)
{
  void *val = NULL;
  ares__htable_asvp_get(htable, key, &val);
  return val;
}


unsigned int ares__htable_asvp_remove(ares__htable_asvp_t *htable,
                                      ares_socket_t key)
{
  if (htable == NULL)
    return 0;

  return ares__htable_remove(htable->hash, &key);
}


size_t ares__htable_asvp_num_keys(ares__htable_asvp_t *htable)
{
  if (htable == NULL)
    return 0;
  return ares__htable_num_keys(htable->hash);
}
