#include "osl/move_generator/allMoves.h"
#include "osl/move_action/store.h"
#include "osl/move_action/voidAction.h"
#include "osl/move_classifier/classifier.h"
#include "osl/state/simpleState.h"
#include "osl/record/record.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/move.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <boost/progress.hpp>
#include <fstream>
#include <iostream>

class generateAllMovesTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE( generateAllMovesTest );
  CPPUNIT_TEST( testOne );
  CPPUNIT_TEST( testMoveGen );
  CPPUNIT_TEST( testMoveMember );
  CPPUNIT_TEST_SUITE_END();
 private:
  osl::SimpleState state;
 public:
  generateAllMovesTest();
  void testOne();
  void testMoveGen();
  void testMoveMember();
};

using namespace osl;
using namespace osl::move_generator;
using namespace osl::move_action;

CPPUNIT_TEST_SUITE_REGISTRATION(generateAllMovesTest);

generateAllMovesTest::generateAllMovesTest()
{
  state=CsaString(
"P1+NY+TO *  *  *  * -OU-KE-KY\n"
"P2 *  *  *  *  * -GI-KI *  *\n"
"P3 * +RY *  * +UM * -KI-FU-FU\n"
"P4 *  * +FU-FU *  *  *  *  *\n"
"P5 *  * -KE * +FU *  * +FU *\n"
"P6-KE *  * +FU+GI-FU *  * +FU\n"
"P7 *  * -UM *  *  *  *  *  *\n"
"P8 *  *  *  *  *  *  *  *  * \n"
"P9 * +OU * -GI *  *  *  * -NG\n"
"P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
"P-00KI00KY00FU00FU\n"
"P-00AL\n"
"+\n").getInitialState();
}

void generateAllMovesTest::testOne(){
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI-OU * -GI+RY-KY\n"
			   "P2 *  *  *  *  *  * -KI *  * \n"
			   "P3-FU * -GI-FU-FU-FU *  * -FU\n"
			   "P4 *  * -FU *  *  *  *  *  * \n"
			   "P5 *  *  *  * +KA *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			   "P8 * +GI+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  * +OU+KI+GI * +KY\n"
			   "P+00KE00FU00FU\n"
			   "P-00HI00KA00KE00FU00FU00FU\n"
			   "-\n"
			   ).getInitialState());
    MoveVector moves;
    GenerateAllMoves::generate(WHITE,state,moves);
    CPPUNIT_ASSERT(!moves.isMember(Move(Square(3,1),Square(2,2),SILVER,PTYPE_EMPTY,false,WHITE)));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY * +TO *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  * -GI * \n"
			   "P3 *  *  *  *  *  * -KE-FU * \n"
			   "P4-FU *  * +UM-FU-FU *  * +FU\n"
			   "P5 *  *  *  *  *  *  *  * -OU\n"
			   "P6 *  *  *  * +FU * -FU+KE+KE\n"
			   "P7+FU *  *  *  * +FU+FU+FU+KY\n"
			   "P8 *  *  *  *  *  * +KI *  * \n"
			   "P9+KY-RY *  *  *  * +KI+OU * \n"
			   "P+00HI00GI00KE00FU00FU00FU00FU00FU\n"
			   "P-00KA00KI00KI00GI00GI00KY00FU\n"
			   "+\n"
			   ).getInitialState());
    MoveVector moves;
    GenerateAllMoves::generate(BLACK,state,moves);
    CPPUNIT_ASSERT(moves.isMember(Move(Square(1,6),Square(2,4),KNIGHT,PTYPE_EMPTY,false,BLACK)));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  * -KA * -OU-KE-KY\n"
			   "P2-KY * -KI * -HI *  *  *  * \n"
			   "P3 *  * -KE-GI-KI *  * -FU-FU\n"
			   "P4-FU *  *  *  * -FU-FU *  * \n"
			   "P5 * -FU+KA-FU *  * -GI+FU+KY\n"
			   "P6+FU * +FU * +FU+FU *  * +HI\n"
			   "P7 * +FU * +FU * +GI *  *  * \n"
			   "P8 * +OU+KI+GI+KI *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  *  * \n"
			   "P+00FU00FU\n"
			   "P-00KE00FU00FU\n"
			   "-\n"
			   ).getInitialState());
    MoveVector moves;
    GenerateAllMoves::generate(WHITE,state,moves);
    CPPUNIT_ASSERT(moves.isMember(Move(Square(5,3),Square(6,4),GOLD,PTYPE_EMPTY,false,WHITE)));
  }
}

template<class State>
static void myStateTest(State& state0){
  typedef AllMoves<Store> DoAll;

  MoveVector moves;
  {
    Store action(moves);
    DoAll::generate<BLACK>(state0,action);
  }
  //  std::cout << state0 << std::endl;
  for(size_t i=0;i<moves.size();i++){
    const Move move=moves[i];
    CPPUNIT_ASSERT(state0.isValidMove(move));
    CPPUNIT_ASSERT(Classifier::isSafeMove(state0,move));
    CPPUNIT_ASSERT(!move.ignoreUnpromote());
  }
  Move tmpMove=moves[0];
  state0.makeMove(tmpMove);
  moves.clear();
  {
    Store action(moves);
    DoAll::generate<WHITE>(state0,action);
  }
  for(size_t i=0;i<moves.size();i++){
    CPPUNIT_ASSERT(state0.isValidMove(moves[i]));
  }
}

void generateAllMovesTest::testMoveGen(){
  // std::cerr << std::endl << " generateAllMovesTest::testMoveGen" << std::endl;
  NumEffectState state1(state);
  myStateTest(state1);
}

static void testMoveFile(const std::string& fileName){
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state(rec.getInitialState());
  const vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++){
    MoveVector curMoves;
    {
      Store action(curMoves);
      AllMoves<Store>::generate(state.turn(),state,action);
    }
    // 動いた結果王が取られる手は生成しない
    for(size_t j=0;j<curMoves.size();j++){
      CPPUNIT_ASSERT(Classifier::isSafeMove(state,curMoves[j]) ||
		     (std::cerr << "\n" << state << "\n" << curMoves[j] << std::endl,0)
	);
      if(!state.isValidMove(curMoves[j])){
	for(size_t k=0;k<curMoves.size();k++){
	  std::cerr << "curMoves[" << k << "]=" << curMoves[k] << std::endl;
	}
      }
      CPPUNIT_ASSERT(state.isValidMove(curMoves[j]) ||
		     (std::cerr << "------\n" << state << "\n" << curMoves[j] << std::endl,0)
	);
    }
    const Move move=moves[i];
    if(move.ignoreUnpromote()) {
      CPPUNIT_ASSERT(! curMoves.isMember(move.promote()));      
      CPPUNIT_ASSERT(curMoves.isMember(move.promote()));
    }
    else
      CPPUNIT_ASSERT(curMoves.isMember(move));
    state.makeMove(move);
  }
}

void generateAllMovesTest::testMoveMember(){
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=10000;
  if (OslConfig::inUnitTestShort()) 
    count=10;
  std::string filename;
  boost::scoped_ptr<boost::progress_display> progress;
  if (OslConfig::inUnitTestLong())
    progress.reset(new boost::progress_display(count, std::cerr));
  while((ifs >> filename) && filename !="" && ++i<count){
    if (progress)
      ++(*progress);
    testMoveFile(OslConfig::testCsaFile(filename));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
