/*
 * Copyright (C) 2015 Martin Willi
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the Licenseor (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be usefulbut
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <crypto/crypto_tester.h>

/**
 * As I couldn't find any test vectors for common MODP diffie hellman groups
 * these have been generated.
 */

ke_test_vector_t modp768 = {
	.method = MODP_768_BIT, .priv_len = 32, .pub_len = 96, .shared_len = 96,
	.priv_a	= "\x60\x91\xff\xc6\xde\x28\xc1\xcc\xc7\xc6\x5d\xa6\x11\xab\xfa\xe8"
			  "\x6a\x10\x74\xb2\x22\x43\xe3\x70\x6c\xb6\xde\x2f\xe2\x9d\x11\x42",
	.priv_b	= "\x76\xe0\x2f\xc3\xea\xbe\x6a\x0f\xce\xd6\xc3\x1e\x59\x45\xd1\x67"
			  "\xfa\xd0\x02\x00\xb4\xaf\x12\xcd\x6f\xc6\xd7\xe3\x81\x48\x62\x78",
	.pub_a	= "\xd7\xa2\x54\x62\x99\x01\xc8\x89\x53\x2c\x3e\x44\xda\x3d\x0b\x7e"
			  "\x92\x27\x37\x70\xc6\x26\xc3\x91\x90\x52\x2d\xab\x67\x07\xca\xff"
			  "\x36\x59\x10\x9f\x2f\x43\x24\xa4\x17\xeb\x7d\xc4\x56\x3a\x09\xba"
			  "\x04\xcd\x3c\x9b\x27\xd3\xc1\x22\x7e\xb1\x9d\xcb\x69\xfe\xf3\xb6"
			  "\xe2\xaa\x13\x81\x23\x24\x06\x64\x2d\xd1\x50\x78\x57\x07\xf4\x7c"
			  "\x3d\x74\x8a\x3d\x6b\x96\xd0\x00\xc5\x2c\x99\xd1\x0b\x65\xf2\xd1",
	.pub_b	= "\xf0\xe9\xdc\x78\x10\x4c\x97\x99\xb6\x70\x74\xb0\x7d\x8f\x09\x7a"
			  "\xa8\x82\xbd\xe4\x64\xc8\xeb\x9a\x0a\xcd\xef\x13\x86\x19\x4f\x49"
			  "\xc0\x63\xc6\x7d\x68\xf1\x4c\x5e\x3b\x04\x19\x57\x67\x8e\xa7\xcb"
			  "\x39\x7d\x87\x07\x20\x81\x9e\xa1\x08\x5a\x28\xd8\x13\xe3\x73\x9a"
			  "\x64\x00\x6f\x24\x66\xe8\x6c\x09\xe0\xc6\x9c\x2c\xa4\xf1\x0a\x04"
			  "\xc4\x9b\xb3\x01\x86\xbd\xfa\xb5\x4d\xf4\x20\x83\x14\xd9\x3c\x52",
	.shared	= "\x29\xcb\x14\x7d\x2a\x2b\x0d\x02\x59\x4d\xb0\x7b\xf0\x44\x70\x5c"
			  "\xb1\x44\x89\xd0\xf0\xa9\x32\xcc\x87\xf6\x04\x05\x1d\x1c\xb6\xe2"
			  "\xbe\x90\x39\x16\xe2\x5c\x65\x39\x93\xc8\x05\x5e\xd4\x37\x6c\xa4"
			  "\xdb\xa8\x40\x12\x39\x1a\x92\xa3\xf3\x42\x64\xaf\x64\x1f\xd8\x18"
			  "\xe1\xba\x4e\x99\x34\x30\x09\x97\xd7\x32\xd7\x0d\x0d\x73\x73\xcd"
			  "\x3d\xaa\xad\x22\xe8\x68\xb7\xe2\x50\x02\x9d\x30\x7e\xe5\x41\x48",
};

ke_test_vector_t modp1024 = {
	.method = MODP_1024_BIT, .priv_len = 32, .pub_len = 128, .shared_len = 128,
	.priv_a	= "\xe5\x3c\x20\x5d\xa0\xd8\xe4\xbf\xb4\x17\x49\x44\x32\x0f\xc6\xe4"
			  "\xea\x66\xfe\x44\xe3\xc9\x31\xac\x5d\xa1\x45\x0a\xea\x47\xeb\xcf",
	.priv_b	= "\x7f\x9a\xf7\x21\xeb\x7c\xd2\xa9\x00\xa3\x6e\x39\x9e\xbc\x5c\x65"
			  "\xee\xcc\xe6\x62\x9c\x8e\x1c\x5a\x7f\xf3\x32\x93\x64\x5f\xd3\xe3",
	.pub_a	= "\x66\x61\x7c\x9b\xbe\x91\xee\x99\x00\xd8\x06\x41\x5b\x73\x84\xec"
			  "\xb0\xb6\xb8\x0a\x39\xbd\x5b\x07\x03\x96\xee\x32\x37\x5a\x8f\x68"
			  "\x37\x6b\x35\x2b\x97\xba\xf3\x2f\x95\xc4\xd1\x8c\x06\xab\x96\xbf"
			  "\xe3\xf3\x75\x2d\xf1\xe3\xc5\x57\x41\xb6\xf4\x24\x41\x17\xee\xbd"
			  "\xa1\x01\x59\x83\xc6\xba\x00\x8f\xe8\x9f\xe5\x1c\xf4\xc9\x69\x25"
			  "\x92\xeb\xf7\x42\x43\x6c\x39\x3d\xf6\x1a\x60\xcc\xc6\x4f\xd7\x90"
			  "\x7a\x6d\x26\x26\x7c\x0f\x15\x56\x23\x0c\x3e\x33\x2b\x2e\x11\xd1"
			  "\x18\xa6\x98\x25\x89\x79\x3a\x2f\x6d\x66\xb8\x66\x76\xba\xe4\x7a",
	.pub_b	= "\x7b\x90\x10\x60\x9e\xbf\x2c\x49\x70\x17\x45\x8a\xfa\xab\x42\x02"
			  "\x88\x5d\x25\xde\x7b\x5e\x5c\xe9\xb0\x5b\xd4\x42\xa3\xe9\x7b\x52"
			  "\xce\xa7\x60\xd7\xdb\xcb\x21\xdd\x71\xd8\x0c\xd4\x34\x7c\xaa\x9e"
			  "\xdf\xbc\x2d\xf4\xc1\xcd\xc1\x66\x9b\x8a\xd2\x44\xeb\x34\x5b\x33"
			  "\x1d\x87\x54\x92\x88\x3d\xf0\x4a\x3d\x0b\x1a\x8b\x89\x27\xd5\x09"
			  "\x91\xfe\x03\xe7\x35\x7e\xb6\xbd\xfc\xe3\xd8\xc6\x84\xc9\x86\x1b"
			  "\xc5\xce\x03\x96\x2f\xfb\x87\xbf\x05\xda\xbd\x5a\x37\x27\x99\x4d"
			  "\xde\xe5\xd3\xc6\xec\xc5\x89\x85\x99\x91\xb9\x32\x55\x76\x1f\xd5",
	.shared	= "\x95\x79\x0f\x5c\x46\xae\x7c\xa1\xa4\x71\xdd\x78\x6a\xa8\xe4\x44"
			  "\x07\x3e\xce\xc9\x69\x5d\x00\x46\x20\xcd\x7c\x9d\x36\x09\xa6\x97"
			  "\x3c\x89\xa2\x30\x75\x09\x35\x63\x8b\x86\xd1\xe6\x5b\x27\xb7\x84"
			  "\x88\x81\xf9\x01\x3a\xbd\x03\x62\x80\xd1\x86\x2b\xb0\x3c\xa6\x0b"
			  "\xa9\x0b\x70\xf9\xae\x7e\xdf\x71\xff\x80\xf5\xa7\xee\xfb\xe2\x67"
			  "\x29\xb5\xbb\xb1\xbb\x1f\xeb\x0d\x24\x0e\x53\xc9\xd7\x4b\x4f\xe9"
			  "\xd2\x62\xb5\x3b\xd5\xd8\xa7\x38\x3f\x90\xf0\x1e\x35\x96\x47\xd0"
			  "\x48\x02\xd7\x4a\x4f\x75\x3b\x29\x4a\x96\x50\x3f\x26\x05\xd3\xf1",
};

ke_test_vector_t modp1536 = {
	.method = MODP_1536_BIT, .priv_len = 32, .pub_len = 192, .shared_len = 192,
	.priv_a	= "\x22\xd9\xdc\xc7\x30\x79\x93\x6a\x85\x8c\x07\xaa\x85\xed\x07\xb3"
			  "\xd1\xe8\xb6\x70\xe7\xca\xaf\xa1\x92\x83\x76\x96\x07\x0f\xef\x29",
	.priv_b	= "\x32\x34\x74\xde\x77\x88\xe0\x03\x6b\x30\x95\x49\x56\x0b\x00\x0d"
			  "\x81\xf8\x2f\xdb\x69\x78\xf3\xc0\x3b\x70\x16\x80\xde\x6e\x28\x10",
	.pub_a	= "\x3d\x7d\x1d\xd1\xbc\xa7\x13\x7a\x64\x23\x73\xd5\xd9\xb8\x6e\xf9"
			  "\x95\x84\x9e\xa5\x1c\xb6\xcd\x51\xa6\xb9\x3e\x83\xb3\x4f\x96\x7d"
			  "\xbf\x7d\x66\xbc\x7c\xe5\xd9\x58\xbf\x01\x90\x2b\x60\xf1\xc3\x07"
			  "\x6c\xfe\x14\x7b\xeb\x45\xf0\x83\x97\xcf\xf4\xc3\xa8\x02\x7c\xaa"
			  "\xe1\x84\x78\x8e\xf3\xeb\x0d\xd5\x6e\x14\xc6\xdd\x2c\xe9\x54\xe3"
			  "\xd5\xcc\x80\xdb\x84\xde\xb5\x34\xce\x38\x24\x45\xe7\xa4\x41\xdb"
			  "\x97\x12\x30\x02\x0a\x1a\x45\xca\x00\x70\xd6\x94\xf0\x93\xc7\x16"
			  "\xd8\x07\x68\x33\x6e\x61\xb5\x6f\xf7\x8b\x35\x09\x39\xfe\x4e\x9e"
			  "\x03\x2c\x85\xbb\x58\x81\xc4\xc8\xd7\xdb\xd5\x30\xa6\xfc\x50\x13"
			  "\x00\xf7\xe9\xe6\x5b\xff\xb9\x83\x34\x8a\xd0\x5c\xc5\x6e\x2c\x19"
			  "\xf5\x97\xa9\x9f\xb9\x68\x91\x4b\xe9\xb5\x7a\xcf\x91\x11\xe7\x5b"
			  "\x57\x6a\x61\x37\x67\x15\x76\x57\x90\x0d\xcf\x22\xf4\x20\x0c\x3d",
	.pub_b	= "\xd6\x32\x58\xd5\x54\x35\x3d\x6b\x2b\xcc\x0b\x53\x53\xfa\x80\x00"
			  "\xb3\xa3\x54\xa2\x41\x1d\x7f\x17\xab\xca\x69\x1d\xe8\x22\x7c\xd7"
			  "\xd4\x3e\x7f\xef\x8b\x3e\xe7\xa0\xa9\x1a\x66\x3d\x30\xc5\x4f\x3c"
			  "\x5f\x4b\x95\xc9\xfe\x38\xc6\xcf\x19\x39\xb4\x80\x2b\xb6\xf0\xa9"
			  "\x51\x12\x8f\xdc\x39\x1d\x90\xfa\x8b\x40\x48\x4f\x45\xb6\xda\x02"
			  "\xc7\x6c\xf9\x1b\x43\x31\xc4\xcf\x78\x51\xe5\x50\xa2\xd1\xc0\x25"
			  "\x53\x14\x03\xe0\x40\x3a\xf4\x72\xb3\x83\x41\xdc\x56\x2b\xe9\x8d"
			  "\x1a\xa2\x80\x49\x4d\x62\x64\x31\x6a\x6f\x77\x4c\xe0\xee\xd4\x01"
			  "\x57\xb6\x37\xc5\x4e\x69\x50\x3b\xec\xb8\xf0\xd7\x57\xb2\x86\xe4"
			  "\xd8\x43\xc2\x24\x21\x9a\x92\x3f\x73\xab\x57\x83\x15\xaf\x44\x5a"
			  "\xfa\xd9\x56\x9d\xf2\x5a\xcf\xca\x3a\x9f\x7c\x93\xd3\x03\xf4\xef"
			  "\x24\x32\xbf\x62\xce\x52\x5c\x8a\x56\xba\xbd\x2f\xfb\x54\x38\x32",
	.shared	= "\x6e\x3d\xf6\xe2\x52\xba\x11\x53\xca\x45\xe9\xa8\xbb\xe5\x48\x33"
			  "\x7b\x69\x57\x2a\xff\x4e\x61\xb4\x6e\xc9\x86\xb8\x70\xa8\x63\xd3"
			  "\x85\x3a\xb9\xa5\x4d\x9c\x63\x2a\x09\x48\xce\x8c\x65\xf1\xa2\x9b"
			  "\x06\x7a\x14\x51\x19\x8d\xab\x9f\x24\x77\x6e\x86\x42\x5a\x06\xbd"
			  "\xaa\x9f\x26\xdc\xe9\xe8\x9e\x36\x8d\x0e\x16\x70\x27\x74\x01\x5a"
			  "\x14\x30\xaa\xa1\xe8\x33\x22\x7f\x9d\xea\x7a\x58\x18\xce\x47\x43"
			  "\x20\xb3\x9c\xe8\x67\xca\x79\xa4\x9b\x31\xe6\xab\xce\xa6\xc8\xda"
			  "\xff\x00\xb8\x06\x4e\x2a\x75\x73\x72\xd4\x0d\x58\xa5\x92\xe7\xa2"
			  "\xde\xb1\xf9\xa0\xd9\xab\xab\x1f\xc0\x81\x2e\xe1\xff\xa6\x2a\x20"
			  "\xff\x68\xce\x4d\x02\xac\xb8\x4a\x1a\x03\x3d\x03\xe4\xf0\x5e\x97"
			  "\xa4\xfa\xd8\x9e\xc2\x3a\xee\x34\x9e\x26\x4e\xfa\x61\xae\x59\xe9"
			  "\x38\x1b\x1e\x5b\x7a\xa5\xd0\x9a\xb6\x6b\x74\x99\x7c\xba\xed\x20",
};

ke_test_vector_t modp2048 = {
	.method = MODP_2048_BIT, .priv_len = 48, .pub_len = 256, .shared_len = 256,
	.priv_a	= "\xaf\x3b\xfd\x38\x62\xca\xa1\x17\x74\xce\x2b\x74\x84\x08\x07\xc1"
			  "\xde\x5c\xd6\xa7\x61\x9b\xb3\xa0\xc7\xaf\x39\xee\xda\xa6\xeb\x89"
			  "\xe2\xe9\xc1\x44\xb3\x62\x5b\x27\x31\x87\x9c\xb5\x8f\xa3\x76\x6d",
	.priv_b	= "\x77\xa4\x8e\x47\x72\xe2\x3e\x28\x4c\xe4\xaf\x81\x39\x9d\xcd\x58"
			  "\x9b\xeb\x7c\xef\xbc\xc9\xd1\x96\xf0\x6d\xcf\xdf\xc3\xa8\x8e\x3b"
			  "\x4c\x1c\x82\xbe\xfe\xc0\xe6\x4b\xa4\x95\xcc\xde\x32\x99\x36\xce",
	.pub_a	= "\x1a\x27\xc5\xa7\x23\x7a\xe8\xfe\x0a\x71\xc0\xaf\x64\x94\xfa\xec"
			  "\xcd\xd4\xa5\x03\x1d\x47\xa1\xa6\xb7\x16\x2d\xfc\xdf\x03\xbb\x17"
			  "\xb7\xbb\x08\x9b\xd6\x52\xa5\xf3\x1b\x36\x61\xbb\x11\x4d\x7c\x48"
			  "\xd8\x3d\x24\x4f\x46\xdb\x64\x87\x0c\x9c\x83\x27\xf1\xa8\x30\xab"
			  "\xf5\x31\xe5\x18\xdd\x52\x1a\x2e\x94\xe7\x06\x1e\x94\x42\x09\xba"
			  "\x53\x31\x01\x2b\x3d\xff\x00\x84\x9e\xaa\xb5\x8c\x7c\x7a\xf1\x52"
			  "\x65\x21\x0f\xbc\xf0\xf8\x7b\x9d\xd9\x32\xf8\xba\x4e\x9a\x9f\x91"
			  "\xb6\x32\x94\x53\x0a\x12\x00\xb8\x8b\x3a\x03\xe1\xa0\xc7\xfd\x34"
			  "\xde\xec\x6f\xad\x50\x1e\xde\x06\xce\x7e\xc9\xe2\x91\xd4\x7b\x69"
			  "\xb3\xa4\xb6\x1b\xb0\x87\x63\xb6\xd4\x6b\xfb\xda\xb9\x7b\xde\xce"
			  "\xa0\xff\x09\xa8\x72\x22\x1f\xb4\xb6\xcc\xca\x6b\x5c\x6f\xaa\xd8"
			  "\xeb\x87\x3b\x9a\x2f\x39\xb6\x3e\x62\xcf\xad\xd4\x76\x07\xe6\x8e"
			  "\x2c\x29\x05\x83\xf7\x26\x8f\xb4\x3f\xc0\x0a\x7d\xc8\x81\x0e\xcd"
			  "\xac\x1a\x59\xbd\x1b\x47\x7a\xaf\x41\xcb\x4b\x24\xad\xa3\xe5\xf7"
			  "\xb8\xcb\x98\x6f\x1a\xe4\x76\xaf\xc7\x67\x7a\xa6\x25\x70\xa1\xb1"
			  "\x83\x8d\xda\x26\xbe\x78\x63\xee\x2f\x40\xb5\xe5\x38\xa4\xe1\x81",
	.pub_b	= "\x66\x72\x82\x31\x8d\x65\xcb\x05\xcd\x32\x3c\x25\x09\x1e\x66\x4c"
			  "\xc7\x5a\x3a\x20\xc0\x14\xb2\xf6\x38\x9d\x3b\x27\xf5\xc1\xe9\x08"
			  "\xe0\xd4\x0f\xf8\x55\x2b\xd5\x87\x75\x25\x50\x30\x22\x85\xb4\x9f"
			  "\x75\xf8\xff\xae\xc9\x37\xfb\x2e\x69\xc8\x15\x0f\x88\xd6\x50\x8b"
			  "\xf0\xfb\x1f\x16\xe9\x67\x6f\x5e\xcf\x71\x8f\x87\x4f\x3d\x8d\xe3"
			  "\xc8\x68\x87\x49\xdb\xcf\x1a\xff\xd4\xd6\x2f\xf9\xea\x5d\x9e\x30"
			  "\x35\xb2\xce\x4a\x6e\xae\x28\x1c\x40\x8f\xc8\x08\x4a\xe2\x35\xd9"
			  "\x98\xa0\x37\x61\x0a\x8c\xb4\x8c\x87\x8c\xd0\x63\x5f\xa1\x36\x9f"
			  "\xc9\x0b\x86\x4c\x53\x23\x67\x38\x45\x97\xfd\xee\xfd\xb6\x74\x2b"
			  "\x00\x6a\xde\xda\xe2\x01\x5b\xc4\xce\x2f\x9c\x54\x51\xc9\x63\x6a"
			  "\x16\xed\x7d\x10\xb4\xe4\x0d\x82\x5b\x50\xaa\x76\x33\xa0\xe8\xfb"
			  "\x81\x3b\xeb\xc0\x49\xf7\xff\x6a\x71\xec\xfb\x14\xe8\x80\xf5\x09"
			  "\xd3\x8b\x57\x1a\x80\x98\x08\xa4\x96\xb4\x51\xb5\xb5\x56\x7a\x36"
			  "\x08\xb0\x1a\x22\xe4\x99\x83\x55\xf1\x81\xb8\x79\xde\x26\x23\x04"
			  "\xf2\x9d\xe1\x42\xff\x5e\x52\xcc\x56\x4f\x3a\x2d\x1e\x84\xa0\xc3"
			  "\x3d\x8f\x3d\xa7\xbf\x64\x12\xb3\xf0\x18\xe0\x0d\x90\xfa\x5b\x94",
	.shared	= "\x26\xf9\xe8\x45\x7c\xf2\xd7\xd5\x6d\xd8\x19\xc2\xa8\x67\xd7\xc3"
			  "\xa9\x0a\xa2\xb9\x44\xe2\x0b\x8b\xde\xd3\xa5\xbf\x91\xaa\x1f\xf8"
			  "\x1a\x7e\xa8\x9d\x52\x4c\x36\x5f\x38\x23\x51\x08\x35\xcd\x86\x44"
			  "\xc4\xd5\xda\xcf\x89\x3e\x61\x0d\x2c\x1a\x14\x06\x16\xd1\xc8\xc1"
			  "\xd3\x5f\x46\xde\x2c\x5f\xaa\xd9\xae\xce\xc6\xb6\x77\xac\xd9\x84"
			  "\x09\x81\x42\xa8\xaa\x0c\x85\x42\xe9\xac\x98\xb3\xbc\x7b\x57\x7c"
			  "\x59\xb8\xec\x53\x34\x6e\x15\x67\xa9\x00\x39\xac\x92\xb2\x24\x63"
			  "\x5a\xb2\x16\x73\x1b\x06\x35\x39\x25\x64\x2c\x33\x0a\x20\x1b\xa4"
			  "\xb3\x12\x84\xa9\x51\x58\x60\xf3\x5e\x93\x08\xf5\x51\x5a\x77\x99"
			  "\x84\xfb\xd9\xf5\xce\x41\x77\xdb\x78\xd1\xcb\x03\x84\xb6\x3c\x73"
			  "\x9c\x6d\x74\x4a\xd7\xa7\x00\xb9\x5a\x53\x1b\x29\xf3\xb7\x44\xed"
			  "\x38\xe6\x9a\xee\x67\x07\x2c\x45\xa9\x1c\xee\x6b\x14\x21\x5e\x04"
			  "\xf2\x7c\x31\x35\x8c\x86\xdc\xe4\x48\xd6\x0a\x22\x3b\xdc\x55\x4e"
			  "\xda\xa3\xe4\x07\x2e\xf6\x03\xa0\xf4\x61\x9f\x8d\xb3\x9c\xec\x29"
			  "\x1c\x86\x01\x74\x15\x5d\x8a\xbf\x9e\x10\x82\x93\x1d\xf5\x8e\xd9"
			  "\xee\x9c\x12\x15\xdd\x23\x93\xde\x02\xf5\xc1\x76\x7f\x07\x0e\x28",
};

ke_test_vector_t modp3072 = {
	.method = MODP_3072_BIT, .priv_len = 48, .pub_len = 384, .shared_len = 384,
	.priv_a	= "\x51\xf8\xaa\xb6\x63\x20\x1e\xb2\x86\xba\xd2\x99\x32\xb2\xe5\x8a"
			  "\x92\x96\xbf\x2a\xa7\x78\x79\xcc\x8c\x64\x29\xd5\xa6\x68\xad\xf7"
			  "\x60\x57\xad\xc3\x77\xcc\x75\xfd\x86\x47\x96\xb8\xfa\x7b\x42\x8c",
	.priv_b	= "\xaf\x7f\x44\x29\x0b\xaa\xc8\x81\x5e\x70\x75\x9b\x6b\x27\x42\x27"
			  "\x12\x22\x50\xea\x8d\x5a\x12\xef\x4b\x0a\x82\xcf\xe9\x1d\x52\x98"
			  "\x9d\x96\xc1\xa8\x40\x89\x2a\x9f\xfa\x9f\x3a\x65\xc7\x7c\x8c\xd9",
	.pub_a	= "\xae\x98\x27\xa0\x7a\x37\x2e\x16\x01\x67\x20\x63\x72\xc5\x5b\x61"
			  "\x5c\xd4\x9e\x8c\x43\xf7\x1a\x6e\x6b\x22\x83\x8c\x0f\x9b\x9d\x90"
			  "\x3a\xe5\x97\x97\xf3\x95\xc9\xca\x7c\x78\x0f\xa8\x8d\x79\x27\xa3"
			  "\xac\x48\xf1\xb5\x05\xe6\x7a\x0d\xb1\xeb\xad\x57\x8e\xb1\x57\x2b"
			  "\x86\x52\x5e\x84\x0c\x4c\xe2\x43\x5e\xd4\xcd\xe1\x75\x68\xa2\xcb"
			  "\x27\x08\x85\xc9\xd3\xa7\x24\xf8\x73\x46\xe5\x19\xa9\x47\x84\xee"
			  "\x88\xaf\xec\x27\x86\x68\x9f\x0b\xc1\x23\xe0\xe4\x65\x79\x57\xb4"
			  "\xf7\x5f\x24\xb5\x2d\xd4\x80\x39\x09\x3a\xe8\xeb\x5d\xa4\x0d\x1e"
			  "\x31\xfc\x53\xfd\x98\xa0\x58\xb0\x38\x9b\xd5\xc7\xaf\x5b\x0d\xf4"
			  "\x49\xd6\x13\x90\x1c\xa2\x71\xb5\x8e\x47\x0f\x88\xd3\x45\x93\x1f"
			  "\x3f\x64\xe8\x1f\x03\x1d\xe9\x5a\xda\xef\x78\xf6\x0e\x64\x5e\xca"
			  "\x7d\xd5\x61\x8a\xa6\xed\xd3\x9f\xa8\x91\x22\x46\x39\xa9\xae\x42"
			  "\xa8\xa9\x37\x4d\x68\x3f\xd3\xf4\x82\x6b\x97\x4c\x8d\x00\x39\x2f"
			  "\x31\x12\x45\xa7\x1c\x86\xdb\x4e\xb5\xef\xc4\x78\x34\xce\x6b\xaf"
			  "\xb2\x08\xaa\xba\xc6\x8c\x35\xe3\x4f\x79\x15\x78\x40\x1c\x7b\x75"
			  "\x01\x15\xb1\x6e\x5c\x3a\xc3\x62\x42\xf4\xa9\x0a\x99\x73\x4c\x08"
			  "\x36\x8d\x48\xab\x5d\x50\xec\x92\xf3\xf1\x26\x7e\x54\xb9\x33\x34"
			  "\xa5\x0b\x00\xd7\x57\x1d\x1c\xeb\x1e\x66\x21\x58\xd5\x66\x36\x20"
			  "\xd1\xf0\x22\x8a\x64\xf4\x92\xea\x8e\x1f\xe1\x81\x7e\x35\x83\x57"
			  "\x7f\x1e\x48\xa5\x84\x91\xc1\x0d\x98\xff\xa0\xea\xa8\x64\x05\xd0"
			  "\x2c\x94\xb4\x9f\x99\xfd\xdc\x6a\x3b\x7b\xd6\xe3\xbc\xe2\x26\x1e"
			  "\xa3\xcb\xbd\x82\x1f\x49\x5a\x2a\xa7\x1a\xe9\x09\xb3\xcd\x0d\x2c"
			  "\x9b\xec\x0b\x06\x8b\xa7\x26\x20\xff\x06\x32\x4c\xdb\x80\x0b\xf8"
			  "\x56\x5e\x4b\x78\x21\x84\x61\x0e\xf0\x77\xbc\x4e\x8f\xc5\x17\xf7",
	.pub_b	= "\x2b\x27\x0c\x8e\x66\x74\x09\x31\x45\xef\x84\xab\x0c\x34\x56\x0f"
			  "\x5c\x4f\x94\x82\xa5\x0f\x2f\x44\x72\xfd\xe9\x87\xb4\x6e\xb8\x40"
			  "\x34\x02\x03\xd6\x31\x10\x2f\x2c\x7f\x9e\xb4\x8d\x20\x74\x44\xfb"
			  "\x21\x71\x7d\xaf\x76\xf6\x60\x04\xa7\x9e\xa8\xee\xe7\x7c\xc9\x80"
			  "\x19\xba\x12\x34\xf1\x6f\xbf\xf7\xdd\xa9\x45\xa4\x46\x81\x1d\x28"
			  "\x5d\x31\xaf\xd3\x31\xdf\xa8\x0c\x87\x0e\xfb\x19\x96\x68\xbb\x38"
			  "\x76\x18\xe1\x16\x21\xf5\x32\x92\x90\x8f\xd3\x29\xaa\xdb\x91\x63"
			  "\x5e\x47\x5f\x33\x00\x4d\x9e\xff\xca\x65\xe7\x3f\xdf\xe5\x58\xf3"
			  "\x4b\xa9\xee\xf3\xcf\x96\x1d\xab\x56\x5c\x5a\x72\x2a\xfb\x22\x8c"
			  "\x26\xbf\x38\xd3\x9c\x84\xf7\xca\x70\xf8\x51\xbf\x21\x8a\x7e\x2c"
			  "\x2d\xb5\xd3\x56\x93\x7a\x01\x5a\x65\x50\x31\x13\xf6\xc0\xd5\x2d"
			  "\x3b\x2a\x21\x21\xa0\xca\x42\xc6\xcc\xde\xbb\x0b\x2c\xfc\x68\xaa"
			  "\x66\x2f\x3e\x08\xdd\x69\x82\x42\x7a\x0a\x11\xb2\xe1\x44\xe0\xf1"
			  "\xa3\x47\xd1\xd2\x7f\xf8\x96\xee\x56\xf5\xd9\xfb\x25\xf7\x40\xb5"
			  "\xc7\x09\x88\xfc\xdc\x91\x12\xdf\xce\xa2\xde\x11\x7c\x12\x87\xd3"
			  "\x2d\xa8\x42\x70\x49\xce\xef\x09\x7b\xf4\x57\x81\x84\x6a\x02\x1d"
			  "\x38\x3e\x73\x1a\xa6\xe5\xc3\xdc\x0a\x46\x22\x6e\x54\xc5\xa6\x36"
			  "\x7a\xd8\x90\x0b\x8e\xad\xba\x54\x09\x3e\xc6\xf7\x1c\x29\x74\x4a"
			  "\xdc\x73\x02\xe8\x11\x85\xed\x80\xdb\x14\xb3\xe4\x3f\x29\x36\x1a"
			  "\xf4\xe6\x1c\x33\x1e\x5d\x77\x4d\x76\x17\xbf\x31\x55\x71\x3b\x0d"
			  "\x4f\x90\x38\x97\x6f\x2f\xf8\x51\x99\xc1\xd7\x0d\xcb\x3e\xfa\x2b"
			  "\x63\xe0\xc7\x7c\xee\x78\x9f\xa0\x05\x8f\xdf\x52\x67\xc6\xb1\xbd"
			  "\xed\xc1\xd1\xef\x47\xca\x08\x22\x2a\x9a\x27\xe6\x5f\x89\x2e\xef"
			  "\xde\x06\x12\xf4\x6b\x51\xbe\x1a\x14\x77\x4d\x45\x14\xd7\xe7\xca",
	.shared	= "\x45\x3c\xaf\x89\xd6\xbf\xf3\x9b\xda\xe1\x85\x46\x17\x61\xd6\xde"
			  "\xf5\x89\xa5\x82\x66\x9c\x1a\xc8\x6b\x0b\x7e\xe4\x69\x56\xc9\xf9"
			  "\x4d\x74\x77\xdd\xd8\x1c\xde\x12\x75\x76\xda\xeb\xd3\x52\x95\x15"
			  "\xf0\x6a\xa7\xdf\x5e\xb1\x31\x27\x67\x5c\xe1\xb4\xdc\xa2\xac\xb6"
			  "\x47\xe3\x55\x76\xb9\x40\x50\xbc\xc3\x85\xb3\xb4\x89\x44\xdd\x1e"
			  "\xca\xbe\x6c\x76\x70\x45\xcd\xcd\xdd\x3a\x1d\x65\x83\x8c\xcc\xb0"
			  "\x82\xf5\x44\xf0\xfd\x9b\xe6\xc3\xd4\xff\xe7\x55\xe4\xd5\x6d\xfb"
			  "\xb4\x20\x93\x52\x3f\x45\xd2\x41\x5e\xad\xf3\x6a\x18\x5c\x87\xa6"
			  "\xe0\x14\x27\xbb\xcc\x98\x95\x68\xa2\x93\x66\xb1\xc6\x91\x72\x4d"
			  "\x81\x0e\x99\xfb\x05\x83\x00\x26\x72\xa5\x3e\x49\x9f\xf2\x30\x4a"
			  "\x4b\xcc\x10\xa3\xb8\x0b\xf4\x8a\x9e\x31\x5b\xe9\x98\xbf\x17\x0c"
			  "\xe9\x0e\xc4\xeb\x87\x05\x57\x0e\x5e\xc4\x80\xc2\x1c\x4d\xd0\xa5"
			  "\x3a\x1c\xb2\x06\xa3\x42\x45\x2e\xa8\xa6\x84\xd5\x69\xd8\x10\xf8"
			  "\xe9\x8f\xea\xd7\x03\x9c\x64\xef\xef\x59\x35\x24\xf5\x07\x5c\xb8"
			  "\x7e\x75\x63\xa4\x2a\xd1\xb5\x78\x19\xc7\xb0\x19\x96\x3d\x07\x0b"
			  "\x4c\xef\xe2\x2c\x59\x6b\x3a\x96\xa8\xba\x41\xbd\xab\xe0\x54\xa9"
			  "\xa9\xb4\xd1\xc4\x27\x8a\x83\x64\xea\xd9\x96\xa0\x39\xe8\xbd\xdc"
			  "\x00\x60\x1e\xcf\x10\x2d\x20\xd6\xab\xce\xd2\xff\x19\x29\x49\xf5"
			  "\xf8\x1c\x36\xfb\x47\x96\x44\x39\xcd\x44\x03\x6a\x23\xac\x30\x86"
			  "\x8c\xa5\x23\x25\x84\x7b\x31\x00\xd3\xd4\x28\x16\x76\xbf\x94\x84"
			  "\xa0\x05\x66\xf1\x3b\x6a\xe8\x9c\x57\xb7\x73\x2f\x8e\x60\x43\x4e"
			  "\x8e\x48\x85\xcb\x1b\xf9\x47\xce\xd1\x4d\x31\x53\xe1\xb6\xc8\x1b"
			  "\x33\x12\x0f\xfb\xd8\x45\x94\x91\xd2\xc5\x78\xaa\xb0\xa9\x7e\x83"
			  "\xdc\x87\x87\x5b\xe5\x88\xc0\xcd\xee\xee\xfd\x19\xcc\x4f\x1d\x40",
};

ke_test_vector_t modp4096 = {
	.method = MODP_4096_BIT, .priv_len = 64, .pub_len = 512, .shared_len = 512,
	.priv_a	= "\xab\x69\xbc\xe9\x61\xf9\x8a\xa9\xd5\x91\xe3\xfd\x9a\xbc\x46\xc8"
			  "\x0d\xde\x39\x02\x84\xf1\x91\x42\xe8\x81\x5a\xb0\x80\x54\x72\x2b"
			  "\xbd\x2e\x14\x1e\x27\x9e\xc7\xfd\x30\xaa\xfa\xca\x66\x40\x93\x73"
			  "\x1e\xcc\x75\xa2\xbd\x07\xe4\xa5\x88\x3f\x56\x08\x93\xc8\x33\x50",
	.priv_b	= "\xef\x3e\xcb\x88\xf4\xf9\x88\xf7\x2a\xcd\x5a\xa2\x2e\xbe\xbf\x19"
			  "\x0f\xde\x7e\x8e\x4d\x0d\x50\xcc\xde\x80\xf9\x41\xb3\xbb\xd6\xa2"
			  "\xf9\xa1\x2a\xee\x9b\xb3\xe5\xc5\x80\x3f\x67\xaf\xfe\x27\xc8\x0d"
			  "\x05\x46\x9b\x52\x54\xeb\x1f\x32\x6c\x18\x73\x1f\xd3\x4a\xc2\xcd",
	.pub_a	= "\x3e\xc6\x3c\x0c\x68\x32\x05\xc3\x9d\x4b\x97\x72\x39\xfe\x7f\x96"
			  "\x17\x56\x0b\x1a\x56\xc4\x7f\xd9\x07\x02\x6a\xd8\x09\x0a\xa4\x4b"
			  "\xcc\x29\xd3\xc2\xb3\x8d\x34\xe1\x5c\x9e\x14\xc8\x05\x81\xa5\xc4"
			  "\x83\x2f\x5c\xd8\x8e\x43\x73\xbf\x59\x0e\xad\x63\x6f\x68\xc8\x99"
			  "\x59\xdf\xeb\xc4\x33\x03\x48\x97\x4e\xc4\x80\x7c\xa4\x4f\xb7\xd4"
			  "\x4f\x02\xb7\x97\x70\x9e\x16\x0e\x3c\x0e\xc4\x06\x1f\x75\x9a\xec"
			  "\x63\xd9\x96\xd2\x37\x7a\x18\xdc\xc5\x94\xc2\x08\xfa\x83\x7a\xeb"
			  "\xc1\x68\x9d\x5c\x0a\xb4\x30\x5c\x1b\xbe\x86\xb1\xd4\xa1\x6f\x4c"
			  "\xb5\x25\xfc\xcc\xf1\x00\x6e\x21\x23\xc7\x76\xb1\xff\xd1\xfe\xa2"
			  "\x97\x7b\x1e\xac\x82\xd1\xee\xec\x4c\x46\x73\xa5\x17\xdb\xc4\x2e"
			  "\x0f\x89\x30\xdb\x28\xd8\xc8\xe8\xb4\x0f\x8a\x49\x8d\xa5\x83\x05"
			  "\x5c\x9c\x12\x35\x34\xcc\x2c\x53\x34\xd0\xbe\xbe\xa3\x76\x26\x78"
			  "\xd1\xf5\x34\xba\x64\x6f\x75\x22\xf2\x68\x57\xa0\xff\x28\x8f\x7e"
			  "\xfc\x38\xc1\xdf\xa0\xa6\x4c\x3e\xb5\x31\x64\xc1\x8f\x6c\x2a\x4e"
			  "\x51\x5c\x13\xc1\xb7\x39\x35\xd9\xbd\xb4\x58\x27\x33\x55\x34\x29"
			  "\x67\xa8\x54\xc7\x8b\x16\x8b\x58\x90\xf7\xfe\x08\xd0\x8c\x2e\x4a"
			  "\x6f\x16\x63\x32\x2e\x7b\x52\xef\x02\x17\xc8\x15\x13\x72\x2f\x34"
			  "\xc0\x95\x48\x02\xe4\x2c\xfb\x6c\x9e\xe9\xd2\xc3\x98\x68\x71\x1f"
			  "\x1a\xfc\x4e\x47\x78\x52\x2d\x9c\x45\x09\x0b\x26\x23\xf4\xaf\xa3"
			  "\x88\x42\x6b\x89\x75\x64\x05\x93\xbe\x79\x2b\xb8\xa7\x15\xbe\xf2"
			  "\xc4\x6c\x2c\x50\xa6\x8a\xa6\x30\x13\x8a\xb2\x6d\xf9\xda\xe2\xb0"
			  "\xe2\xc4\xeb\xc1\x0a\x86\x36\xda\xd9\xbe\x2f\xfb\x79\x19\xcd\x52"
			  "\x78\x85\x84\x3c\xb7\x6c\x71\xf2\x8e\x71\xf6\xf8\x23\xd9\x9c\x2f"
			  "\xe3\x1a\xd7\xfb\xcc\x9a\xbb\x24\xef\x26\x66\xa1\x21\xcc\xf2\x1f"
			  "\x88\x1f\xa6\x6a\xb7\x18\xba\xf9\x2b\xef\x5c\xec\xaa\xcf\x4c\x92"
			  "\x9c\x9c\xb5\x44\x24\xd1\x7a\xb7\x73\x8c\xa6\x80\x7b\x4e\xfa\x49"
			  "\x36\x0a\x3d\xb6\x52\x7c\x72\x4c\x45\x31\x62\x9e\x9d\xb1\x84\x5c"
			  "\x6a\x1e\xb4\x63\x2e\x0c\x07\xab\x6c\x31\x37\x7c\x80\x83\x70\x49"
			  "\x46\x44\xa3\x5b\xd4\x78\x5f\x49\x5f\x60\x63\x8b\x98\x81\x71\xde"
			  "\xce\x5d\x5c\x44\x90\x35\x7d\x02\x89\xb3\x94\x4b\xc9\xf7\x85\x4d"
			  "\xb8\x06\x97\xd5\xf6\x3b\xf3\xe3\x1d\xb4\x4e\xfb\x9a\x43\xe1\xdc"
			  "\x9d\x17\xee\x47\x01\x2d\xfb\xac\xfd\xaf\x4d\xeb\xfe\xf0\x8b\x5c",
	.pub_b	= "\x20\xb7\xb4\x5f\x69\xe2\xc3\x0b\xcc\xf8\x41\xca\xe8\x04\x2c\x36"
			  "\x35\x37\x25\x42\x05\x99\x33\x5f\xa2\xe8\x7b\xbe\x59\xce\xbb\xc7"
			  "\xa7\xd3\xf6\x1e\xb6\x69\x82\x50\x3a\x75\x76\xc3\x47\x63\xdf\xef"
			  "\x6a\x6b\x18\x0a\x93\xaf\x66\xe4\xf7\x2f\x12\xd5\x8c\x93\x84\x6d"
			  "\x16\x05\x58\xb0\xd3\x16\x03\x9f\x6b\xa9\x9e\xa6\x4f\x00\x5b\xa1"
			  "\x1e\x59\xf3\xa9\xcb\x99\x3d\x28\x27\x1a\x4f\xb8\x30\xc3\xf6\xc4"
			  "\xce\xb9\xb0\x16\x2c\xcc\xa1\x97\xff\x65\x15\x78\x9d\x43\x6c\x94"
			  "\x7e\xb5\xd8\x01\x09\x74\xeb\xcd\x36\x6b\xc4\x76\x83\x41\x09\x0a"
			  "\x6f\xb5\x5c\xa8\x4e\x31\xd2\x48\x9e\x35\x27\xa2\x60\x77\x6f\x9b"
			  "\x8a\x58\x57\x7b\xdc\xd6\x89\xd8\xe4\xb7\x25\x14\xcf\x15\xee\xa5"
			  "\xa4\x96\x29\xa2\xf2\xc4\x86\xc5\x1b\x5d\x14\xd4\x9b\x11\x93\x09"
			  "\xbf\xe9\xc9\x32\xb6\x04\xc6\xf1\xc0\xe9\x2c\x44\x8d\xc1\x9f\x54"
			  "\xf4\x21\x11\x2f\x28\x87\x23\x8c\x91\x37\xc5\x59\xb6\x9a\x93\xac"
			  "\xf3\x6c\xc1\xf0\xbd\xfe\x4c\xca\x0b\x60\x47\x71\xee\x2a\xf1\x7c"
			  "\x34\x04\x5d\x42\x29\xb2\xb8\x0a\xcd\xfb\x7f\x56\xe4\xea\xee\x81"
			  "\xed\x6c\x88\x5a\x2e\x45\xaf\xc9\x8d\xe1\x21\x2e\x5f\x71\x4f\x5f"
			  "\x00\x88\x12\xd7\x17\x06\x89\x6d\x2d\x5e\xd8\x59\x0c\xee\x66\x41"
			  "\x08\x79\xdc\x31\x95\xa9\x21\xef\xe0\x85\xdb\x41\xd6\x87\xec\x2d"
			  "\xe9\x06\xa8\x10\x33\x6f\xa7\x57\x0d\x43\x11\x2d\xec\xf9\xff\xa4"
			  "\xae\x50\xf5\x4f\xad\x3e\xec\x2b\xb3\xed\x86\xdd\xa3\x66\x2b\xc0"
			  "\xfc\x1c\x28\x94\xd4\xde\x7d\xa9\x26\x0f\x74\x73\xbe\x67\x6d\xbf"
			  "\x60\x90\x33\x32\x0d\xba\xa4\x91\x72\x0c\xe2\xd3\x5d\x07\x8d\xbd"
			  "\xde\x84\x5b\x4a\x49\x5a\xd2\xec\xc2\xe6\xda\x5f\x1c\x0c\x89\x20"
			  "\xe4\xea\x0e\xcc\xa0\xe0\x77\xda\xc5\x9b\x97\xf0\xe9\x4b\x6c\xca"
			  "\xd3\xf4\x2a\x34\xb6\x63\x4e\xde\x83\xb2\x9d\x9c\x76\x15\xf7\x41"
			  "\x90\xdf\x33\xb2\x0e\x7d\x69\x08\xd6\x63\x69\x8a\xdb\x8a\x2d\xac"
			  "\xd9\x79\xf1\xca\x13\xf2\xcc\x84\x02\xf8\xf6\xcb\xf9\x11\x92\xf1"
			  "\x11\xdd\xf5\xad\x29\x39\x05\x95\x54\xac\x47\xb4\x78\x74\x7b\x78"
			  "\xa2\x71\x92\x97\xae\xda\x20\xe3\xd8\x53\x38\x26\x9b\x30\x80\xfc"
			  "\x3f\xd7\x8d\xe4\xac\xf3\x4e\x09\x47\x7a\x61\xca\xc7\xc1\xb7\xdc"
			  "\x76\x94\xed\x14\x1a\x51\x48\x0e\x6b\x2b\x43\xc0\x25\x56\xe8\xa8"
			  "\x49\x7c\x4f\xca\x56\xf7\xfd\x56\xc8\xeb\x36\xa1\xf8\xc3\xd1\x24",
	.shared	= "\x23\xb0\x6f\x49\x39\x60\x9b\x0f\x67\x08\x85\x2d\x4f\x87\xb3\x56"
			  "\x98\x8f\xb4\x5b\x1e\x4c\x95\x1b\x0b\x1d\x59\xbb\xa8\xca\x8b\x60"
			  "\xc2\x55\xa1\x8f\x37\x55\xa6\xdb\x05\xe4\x28\x5f\xe8\xf1\xf9\xd2"
			  "\x6a\xdd\x24\x14\x19\xf0\xfc\xa9\x82\x37\xfd\x7a\x5e\x52\x6d\x57"
			  "\xab\x5a\xd0\x69\xc3\xf1\x89\xf3\xc2\x91\x50\x11\x44\xa6\x55\x14"
			  "\xa5\x66\xb7\x4d\x0d\xc6\x41\x9c\x13\x06\xcf\xbf\xf3\x5d\xbe\xb4"
			  "\x3a\xbd\x5e\xd3\x9c\xe1\x37\x53\xb2\x8a\xe5\xe1\x05\xf7\x19\x1b"
			  "\xc6\xd7\x0a\xc3\x55\x2c\x82\x37\x3c\x6b\xd9\xdb\x8e\xd5\xee\x15"
			  "\x46\xfd\xb5\x49\x9a\xe7\x0a\x6b\xcb\xd1\x85\x9a\x31\xec\x43\xdc"
			  "\xcf\x05\x52\x5e\xe4\x0c\x94\x98\x87\xdd\x81\x68\xae\x29\xcc\x53"
			  "\x7a\x6f\x57\xa5\x26\xf4\x25\xdd\x07\x5f\x39\xd6\xee\x71\xcb\x49"
			  "\x7a\x57\x1f\xe5\x79\x58\xc7\xde\x32\x1d\x64\xf4\xe4\x89\x22\x43"
			  "\xe7\x8f\xef\x47\xb3\x31\xa6\x76\x84\x49\x89\x19\x1f\x97\xad\xf7"
			  "\x91\x32\x60\x7a\x14\x8f\x19\x3c\x7d\xd5\xe6\xd5\x99\x25\x7e\x1b"
			  "\xf1\x21\x93\x24\x68\xdb\xbe\x21\x60\xc9\x7a\xf0\x3f\x9d\x1b\x19"
			  "\xb1\x6b\x4f\x8f\xec\x5d\xcb\xc7\x98\x34\x4a\x87\xdb\xd4\x02\xa6"
			  "\x26\x6e\x10\xc0\xe8\xa7\x22\xfe\x9f\x67\xe8\x63\x6c\xb0\xa7\x3e"
			  "\x22\x4d\x53\x23\xde\x9b\x7f\xa7\xc6\x6c\x62\xa1\xf4\x00\x42\x04"
			  "\x1c\xba\xde\xf2\x4b\x4f\xaa\xfd\xa9\x14\x79\xec\x91\x97\x64\xb0"
			  "\xf4\x8b\x95\x9e\x67\x99\xf0\x94\x96\x6d\x24\x61\x27\xc0\x0e\x9c"
			  "\xc7\xd8\xf5\xa8\x43\xd1\xa4\xd6\x1c\x5c\x0a\x64\xb6\xb1\x6c\xa7"
			  "\x32\x44\x93\x75\xd9\xcf\x5d\x32\xd3\x99\xf6\x56\xfd\x51\x4f\xbf"
			  "\xe6\x6e\xea\x82\xe4\x79\xfc\x73\x18\x0b\x72\x1d\xd4\xc5\xbb\x20"
			  "\xd4\x50\xc5\xa1\x95\x9e\x1f\x8f\xed\x9d\xd2\x8c\x97\x05\x12\x72"
			  "\xf6\x64\x00\xcd\xd4\x13\x45\x7d\xdd\xe6\x9a\xc7\x43\x5e\xe4\xa4"
			  "\xf7\x2a\x37\x02\x49\x82\xb4\xa7\xf6\xf5\x5e\x03\x07\x03\x82\xb8"
			  "\x3b\x2c\xed\xb7\x75\x25\x17\xf0\x48\xb7\xc6\x91\xd1\xf1\xd7\xb8"
			  "\x52\xa5\xb7\xcd\x3b\x2b\xde\x97\x62\x0e\x9b\x2c\xd9\xc7\x7f\xd8"
			  "\xcc\xb6\x92\x5a\xde\xf7\x06\xa6\x77\x0a\x2e\xfb\x62\x1c\x93\xf1"
			  "\xca\x24\xf0\x9a\x68\x6e\x8b\x86\x05\x81\x49\x47\x39\x92\x15\x33"
			  "\x9a\x1f\x29\xfb\x57\xac\xf9\xce\x9e\xba\x2c\xbd\x49\x69\xc8\x9e"
			  "\x4f\xb9\x39\x02\x12\xb9\xb2\xa3\x5d\x4a\xfa\x17\xb3\xee\xc0\x8a",
};

ke_test_vector_t modp6144 = {
	.method = MODP_6144_BIT, .priv_len = 64, .pub_len = 768, .shared_len = 768,
	.priv_a	= "\xab\x36\xf0\x65\x7c\x4f\xba\xdc\x2a\x3b\x07\xed\xd1\xc8\xaf\xcb"
			  "\x42\xaf\xcd\x7f\xf9\x1c\x57\x01\x37\x25\x50\x0d\x89\x42\x9f\x34"
			  "\x79\x8f\x99\xf5\xde\x53\xd1\x08\x8f\xd9\xf6\x60\xa1\xa5\x2b\xe4"
			  "\x54\xf9\x63\x4d\x15\xbb\x4e\x4c\xe0\x9e\x5a\x69\xe0\xf9\x9d\x59",
	.priv_b	= "\x59\xa5\x52\xa4\x0d\x17\x80\xaf\x64\x33\xbc\x9e\x8a\x9b\x6d\x48"
			  "\x30\xdd\xd3\x57\x53\x07\x40\x9a\x90\xc9\x2b\xda\x40\x4b\xb7\x99"
			  "\xf0\x09\x3e\xfe\x21\xbe\xad\x85\xf1\x6a\x23\x7b\x05\x90\xc3\x35"
			  "\x6d\x96\x8c\xc0\x9d\xcd\x5b\x2d\x96\x86\x07\xd6\xd1\xbf\x70\xc2",
	.pub_a	= "\x0a\x78\x01\x5c\x3e\x0d\x36\x2a\xff\xe5\xd9\x3a\x9c\x2d\xf2\xdf"
			  "\x5b\xb9\x6b\x7f\xf2\xfc\xc2\xdd\x96\x6c\x53\x9d\x1d\x4c\xdb\xac"
			  "\xec\xe2\x6d\x16\xab\x6d\x3d\xe8\x24\xe0\x75\x87\x29\x12\xeb\xa0"
			  "\x44\x33\x66\x11\x0e\x0d\x7e\x17\x27\x4b\x95\x78\xaf\x12\x46\x63"
			  "\xe6\x55\x17\x0b\xf7\xb7\xb2\xbd\xb4\xbf\xe4\x7a\x88\x97\x68\x95"
			  "\x6a\x98\xf1\xce\x1a\xc8\xc6\xc7\xc4\x8e\x13\xa3\x6c\x16\x1c\x82"
			  "\x00\x7c\x98\x35\xb2\x2e\x7b\x47\x99\x38\x92\xa1\x71\x2b\x27\x5d"
			  "\xdc\x84\x54\xf4\x33\x0b\xd1\xfc\x08\xab\xfc\x48\x8e\x71\x8b\x43"
			  "\xa6\x21\x3a\x09\xc5\x68\x74\xce\xef\xb3\xa9\xfa\xe7\xe1\x11\xc0"
			  "\x0a\x10\x43\x3f\x4b\x23\x75\xef\xab\x04\x2b\xd5\xc1\xc2\x9a\xaf"
			  "\x97\x0c\xeb\xae\xb5\x7f\x10\xf0\x10\x6e\xa6\x3d\x25\x72\x59\x93"
			  "\x0a\xf5\xb3\x3b\xc2\x64\x27\xe6\xef\x47\x32\xde\x43\xdc\xea\x0a"
			  "\x88\x72\x9b\x93\x7e\x6a\x9a\xfc\xf2\x92\xa2\x9f\x8b\xe9\x9b\x71"
			  "\x88\xd8\x2d\xfc\x13\x7e\x69\x19\x9a\x53\x5d\x92\xdc\x61\x37\x60"
			  "\x03\x38\x67\x1e\x6d\x46\x76\x31\xff\xc2\xbd\x69\x59\x42\xcd\x7e"
			  "\xbb\x40\x2c\x54\xea\x39\x1b\x9a\x02\xde\x1f\x28\x99\xfe\xd5\xbd"
			  "\xb7\x1f\x27\xab\x71\x0b\xba\xaa\x32\xb2\xc2\xe0\xec\x8d\x4b\xd4"
			  "\xca\x6c\xc5\x07\xd9\x72\x0b\x20\xaf\x9c\xce\xcd\x7f\x4e\x88\xed"
			  "\x55\x0a\xea\xbc\x43\xdd\x0b\x3d\xc0\x20\xdb\x3e\x14\x89\x76\xc7"
			  "\x61\xf5\x44\x21\x8a\x79\xb7\x3b\x37\x77\x24\x99\xf3\x61\xba\x0b"
			  "\x1d\x3c\xf2\x10\x23\x75\x36\xfb\x89\xd8\x57\x86\x51\xed\x67\x51"
			  "\xd1\xe8\x10\x95\x61\x2b\x0f\xcf\x65\x36\xbc\xb0\xff\x17\x2c\x3d"
			  "\x54\xdc\x07\x13\x19\x99\xd4\x11\x98\xf9\x7e\xa8\x32\x9a\xbb\x04"
			  "\xc3\x75\x3f\x83\xe1\xfd\x3b\x92\x78\x72\x3c\x98\x67\xf4\xc1\xff"
			  "\x19\xe1\xd2\xad\x7d\x34\x65\xf0\xb8\xc2\xdd\x9d\x4c\xcd\x36\x1a"
			  "\xbd\xf8\x56\x66\xd6\xfe\x04\x2c\x98\x04\x2b\xec\xa9\x4b\x66\x4b"
			  "\x71\xcf\x78\x07\x56\xe5\xba\x9c\x8a\x72\xb8\xc9\xe4\x82\xd6\x59"
			  "\x22\x59\x39\x75\xd6\xdd\x00\xf3\x16\xc7\xb2\x0c\x81\xeb\x67\x4f"
			  "\x0b\xbe\xa8\x1e\xed\xe6\x7b\xbf\xf1\x17\x38\x3f\xf4\x86\x0b\x97"
			  "\x75\xa7\x81\x86\x14\xb8\x6d\x48\x5e\x88\x98\xa9\x2f\x54\xfd\x7f"
			  "\x05\x45\xb4\x32\xcd\x5f\xab\xec\x2e\xa8\x42\xd8\x3b\x51\xc2\x18"
			  "\x91\x7a\xb6\x10\x5e\x26\x8b\xc8\x50\x08\x2c\x79\xa1\xd0\x74\x12"
			  "\xce\xc4\xd2\x3c\xb0\x04\x54\xa8\x87\x2b\x9f\xb3\x05\x4a\x89\x64"
			  "\xb5\xaf\x7f\x64\x4b\x45\xcd\xd7\xf2\xb8\xa8\x8c\xd8\x7c\x6e\xe6"
			  "\x19\xd9\xaf\x59\xb5\x2b\x07\x37\x1b\x8f\xb0\x57\xe7\x05\x9f\x21"
			  "\x52\x6d\xc4\x89\x4a\xff\xce\xda\xc8\x5b\x73\xf3\xd4\x07\xc7\x29"
			  "\x02\x7e\xa6\x79\x82\xd3\x01\xba\x93\x0e\x37\x17\x3d\xfc\x38\xd3"
			  "\x25\x7e\x52\xd2\x53\xba\x20\xe8\xe9\xef\xa2\x96\x38\x49\x14\xd2"
			  "\x83\x8b\x2c\x62\xb0\x27\xc6\x5d\x36\x34\xd4\x58\x14\x25\x6e\xc1"
			  "\xcf\xd0\x2d\x21\xa3\xc0\x9c\x9b\x14\x20\x83\xec\x1a\xeb\x14\x2a"
			  "\xd3\x97\x40\xad\xd0\xeb\x51\x8f\xa2\x10\x62\xb4\x50\x94\xff\x35"
			  "\x43\xc2\x29\x88\x0e\xf6\xb9\x4c\x85\x80\x13\xed\x2f\x56\x15\xdc"
			  "\x0f\x09\xd2\xe5\x40\x11\x70\x34\x76\x2c\xed\xb1\xac\xe5\x82\x77"
			  "\x45\x42\x3e\x8e\x8d\x08\x6e\x5b\xbe\x34\xf9\x93\x0f\x8a\x43\xec"
			  "\xa6\x9f\x7c\x56\xe6\x95\x31\x85\x9d\xb1\x97\xaf\x2d\xac\x76\x81"
			  "\x1f\xfb\x4d\x53\xfe\x04\xe2\x48\xbe\xac\x50\xe2\xb3\x74\x77\x5f"
			  "\x48\xec\x26\xd0\x9b\xb9\xa3\x28\x23\xa9\x2c\xc2\x0a\xb7\xd3\x80"
			  "\x87\x03\xa2\x3b\x74\x07\xaf\xa5\x5f\x2b\x9b\x90\xa2\xf8\x89\x3c",
	.pub_b	= "\x75\x3a\x06\x23\x48\x41\x6e\x90\x3b\x5b\x3b\x25\x89\x38\xf1\xa4"
			  "\x3f\xe0\x96\x2a\xcb\x3c\xd2\x7a\x71\xb3\xed\x8a\xd4\xa5\x62\x77"
			  "\x4b\x6f\xf4\xf2\x29\x31\x2a\xfc\xb4\x7b\x34\xfe\x9c\xb0\x83\x62"
			  "\xe7\x45\xc9\x93\x19\x89\xdb\x90\x99\xc5\x77\x85\x06\x97\xa6\x2f"
			  "\xde\x6d\x98\x01\xbc\x4f\x51\x92\x94\x6f\x10\x3a\x7a\x56\x14\x48"
			  "\xad\x7d\x1d\x15\x0c\x8c\xda\xc9\x01\xf1\x3b\xfd\x27\x09\x2e\xf7"
			  "\xec\x0f\x82\x1e\x0e\xa6\xb9\x1b\x63\x90\xc3\x3e\x7e\xf1\xad\x5c"
			  "\xaf\x6f\x6d\x9d\x3f\x25\x4f\xe9\x53\xaf\x03\x6e\xdc\x24\xf3\x2c"
			  "\x65\x67\xc7\x08\x61\x80\x18\x7c\x19\x97\x44\x56\x5e\xf0\xa2\x94"
			  "\x7c\x59\x01\x94\x5b\x46\xa8\x0b\x28\x6c\xa0\xfc\xa0\xad\xe4\x4a"
			  "\x2c\x87\x77\x7b\x44\x28\x25\xd4\xa2\x24\x70\x69\x9a\x83\xf7\x65"
			  "\xde\xe3\xeb\x03\x14\x00\x4c\xba\x87\x87\xf2\x47\x4c\x3e\x44\x67"
			  "\x66\x85\x48\xb4\x12\xa6\x15\x22\x0d\x39\x02\x07\x66\x59\x07\x3a"
			  "\x64\x9e\xba\x6e\xc4\xdc\x29\x07\x5b\x67\xff\x42\xca\xe0\x1d\xcd"
			  "\x39\x08\xff\x63\x03\xb1\x76\xc4\xa3\xdc\x81\x33\xfb\x4c\x28\xa1"
			  "\xe4\x7e\xbe\x5f\x73\x24\x92\x7a\x40\x8d\x75\xc5\x94\x13\x26\x91"
			  "\xef\x9a\xee\x45\xaa\xff\xfc\xae\x61\x34\xdb\x20\x96\x99\xe9\x18"
			  "\x30\x95\x37\x23\xaf\x3d\x2f\x3b\x03\x69\x4b\xfa\x92\x92\x57\x8e"
			  "\x66\xe2\x89\xf0\x62\xe5\x2b\x2c\x23\xca\xcd\x8d\xdd\x88\x92\xb4"
			  "\xc9\x8e\x9d\x57\x62\x69\x3e\xd1\xd7\xc8\x7e\x99\xac\x4e\x7c\x65"
			  "\xaf\xea\x99\xfa\x69\xd8\x46\xb2\xc7\x1f\x76\xf1\x3e\x99\xb7\x23"
			  "\x2c\x7c\x80\x8b\x3a\x5e\x86\x2c\x50\x5a\x36\x48\x0a\x23\x23\xdf"
			  "\x69\x95\xa0\x1b\xfd\x1f\x4e\x06\xc5\x0b\x17\x3c\x62\xce\x5a\x63"
			  "\x82\xcd\x14\x64\xb8\x60\x36\xb9\x74\x9c\xa4\xe1\xa5\x0c\xc0\x77"
			  "\x05\x41\x46\xac\x16\xdb\xb5\x16\x71\x71\x6e\x62\x93\x17\xd6\xdc"
			  "\xbb\xbd\xb3\x01\x5f\x08\xa9\x71\x91\x97\x92\xb1\x1d\xa8\x0a\xf9"
			  "\xc3\xaa\x4c\xc2\x63\x48\xd1\xae\x21\xbb\xf3\xb7\xda\x04\x5e\x6e"
			  "\x82\x89\x5d\xdc\xfb\xae\x76\xaf\x50\x42\x71\x06\x8b\x0c\xfd\xb9"
			  "\x0f\x00\x24\x97\xe0\x0c\x9f\xf2\x95\x11\x63\x6f\xcf\x59\xfb\xd2"
			  "\xcc\x10\xec\xaa\xef\x89\xff\x1b\x48\xc9\xce\x78\x22\x50\xf6\x31"
			  "\x47\x78\x38\x3b\xae\x32\xed\xf6\xaa\xa9\x7a\x53\x71\xc6\xbd\x10"
			  "\xcf\x17\xf4\x1b\x1e\xb0\x90\x4d\xd1\xd2\xa2\x9b\x5c\x37\xd3\x9c"
			  "\x31\xb2\xb8\x5b\x8c\xa2\xde\x11\xf7\x97\x03\xea\x45\x38\xc5\x5c"
			  "\x22\x8e\x3d\x60\x4a\xc7\x32\xaa\xee\x7a\x67\x9a\xa5\x85\x1f\x64"
			  "\xb1\x45\xe7\xe1\x69\x68\x5c\x65\x1e\x0a\xf3\xf3\x11\x26\x98\x7b"
			  "\xf8\x27\x23\xad\xf4\x25\x6f\xab\x83\x48\xc4\x5e\xba\xea\x73\x6a"
			  "\x2b\x82\x66\x02\xf5\x21\x5a\xbc\xf5\xbf\xf1\xa4\x72\x1c\xd9\x9d"
			  "\xb6\x46\xe9\xb5\x61\xbe\xe5\x59\x8a\xf9\x8e\xfa\x79\x2e\xa6\x02"
			  "\xad\x22\xea\x06\x2c\x42\x66\xb9\x0c\x6d\x4b\x2b\x8b\xd9\xa3\x8b"
			  "\x60\xe2\x63\xe0\x44\x54\x02\x2f\x75\xb7\x41\x81\x9c\xe7\xce\xc4"
			  "\x3e\x82\x05\x5e\x0e\x4c\x16\x0a\x59\xfa\xb1\x13\x02\x87\xb8\xd8"
			  "\xa7\xbc\x15\xb2\x5a\xb5\xea\x50\x76\x76\x73\xa1\xf4\xc2\x71\x88"
			  "\x5d\x0c\x8c\xbe\x32\x3d\x60\x15\xdb\xad\xde\x37\xf8\x8e\xb8\xd2"
			  "\x24\xc3\x3c\x97\xe7\x9c\xc6\xdc\xcd\xcd\x43\x93\x06\xd8\x64\x9f"
			  "\xca\x07\x15\x47\xca\x13\x39\x8d\xd6\x75\xe0\x61\x7f\x7f\x15\x28"
			  "\x8f\xe8\x4d\x19\xb6\x41\x20\x93\x17\x03\xaf\x1b\x16\x13\x07\xc7"
			  "\x50\xfe\xeb\x97\x7c\xe3\x72\x32\x9b\x87\xab\xab\x2d\x47\xa0\x93"
			  "\xc3\xc9\x17\x58\xc6\x2c\x8a\xa8\x78\x6e\x6c\x30\x6c\xbf\x3f\x66",
	.shared	= "\xfc\xe8\xe5\xeb\xf2\xb0\x07\xfc\x46\x60\x17\xa8\xed\xf0\xf1\xa6"
			  "\x9f\x5a\xf2\xea\x49\x2a\x09\xd8\x08\xd2\x84\xc2\x2d\x9b\x3d\x07"
			  "\x2b\x9f\x89\x1b\x0c\x0f\x09\xe4\x67\xd1\x6e\x33\x8d\x2e\x6d\xd1"
			  "\xf5\x1b\xdb\xc4\xe8\x64\x61\xf5\x49\x47\xd0\x07\x0f\xbe\xc1\xfc"
			  "\xe3\xe4\xf3\xd8\xa6\xeb\x73\x8c\xb1\xd5\xcf\xc6\x54\x6d\xe2\x07"
			  "\xba\x55\x17\xe8\x2b\x39\xc2\xab\x30\x8d\x2d\x48\xe1\xe7\x3f\x7f"
			  "\x5c\xf8\x4a\xfa\x7b\x71\xb3\xf2\x4c\x52\x45\x5d\xfd\xa0\x94\x98"
			  "\xe9\xde\x3f\x14\x52\x7f\xa5\xd6\xf7\xa4\x67\x27\x4d\x84\x4e\xaf"
			  "\x7f\x20\xef\xc6\xaf\x5d\xac\xaf\xd7\xe1\x52\x02\x6f\xd4\x84\x73"
			  "\x15\x7b\x74\x89\x65\xcd\xa7\xb8\xd5\x8d\x54\x94\x44\x88\xf6\xd2"
			  "\xa9\x7f\xa9\x12\xd1\x19\xa7\x75\x2f\xcf\x3c\xb9\xc2\xc7\x66\x18"
			  "\x91\x83\xd9\xa1\x69\xe2\x09\xc0\x38\x7f\x7b\xff\x5e\x44\x9d\x34"
			  "\x3c\xef\xb3\xbc\x86\x57\xaa\x28\x22\x09\x7a\xa1\x64\xad\xf0\x16"
			  "\xcb\x15\x8f\xaf\x9e\xb8\x0f\x9f\x53\x1c\x8a\xf9\xf3\x2d\xee\xf7"
			  "\x31\xa8\xcd\x44\x6a\x5f\xea\x66\x95\x81\x96\x49\x83\xd7\x1d\x54"
			  "\xaf\x34\x81\x5d\x3c\x4e\x4e\xc5\xe6\x5c\x04\x4d\x7a\x5c\x4b\x7b"
			  "\x54\x14\xa1\x19\x31\xe3\xda\x28\x4f\x8a\x3a\x99\x12\x3c\x22\xb5"
			  "\xce\xe7\x8b\x1c\xf0\x3a\xdf\x88\x5b\xe5\xdb\x0a\x88\xcd\x84\x9d"
			  "\x41\xdd\xf1\x63\x00\xf3\x6c\x23\xa4\x4b\x81\x04\x74\xd1\x35\x8f"
			  "\xef\x99\x9d\xf9\x46\x7f\x96\x98\x15\xbc\xa6\x25\x39\x6e\x32\xd1"
			  "\x7f\xa0\xcc\xe9\x60\x41\xc7\xcf\x0d\xd3\x16\x99\xee\xb7\x63\xbb"
			  "\xb2\x4c\x93\x7f\x50\xd0\xfe\x90\x39\x33\x87\xc6\x17\x97\x44\x83"
			  "\xad\x3c\x20\x2f\xd5\xc0\x30\xf4\xfa\xde\xdf\x42\x43\x16\x46\x6a"
			  "\x6b\x36\x59\xcd\xf1\x8c\x39\x82\xba\x54\xf6\x34\x3f\x6b\xf2\xf2"
			  "\x1f\x58\x58\xea\xdb\x8f\x80\xa2\x4d\x01\xe8\x2d\x70\x0c\x58\xf9"
			  "\x56\x4f\x6e\xea\xe2\x98\x0c\xc7\x01\x74\x40\x25\x24\xcc\x87\x16"
			  "\x98\x67\x5a\x7e\x95\x87\xa1\x41\x0e\xa6\x22\x89\x13\x26\x0f\x6f"
			  "\x81\xdc\xd5\x3d\x75\xe0\xd6\x71\xe7\xde\x0a\x6c\x62\x3b\x64\xab"
			  "\x2f\x67\xa0\xad\xdc\x32\x70\x1c\xa1\xfa\xe9\x08\xed\x5d\x90\x09"
			  "\xad\x42\x74\x5d\xcd\x12\x24\xfb\x14\xe8\xb9\x89\xa4\x4f\x0b\x07"
			  "\xa9\x4a\x7a\x27\xce\x19\x8d\x32\x6f\x74\x9a\xc5\x10\x91\xf1\xfc"
			  "\x03\xfa\x1c\x87\xe8\x70\x5c\x36\xa8\xef\x2c\xc7\xb6\xe4\x47\xac"
			  "\xb7\x3d\x0e\x44\x12\xbd\xa2\xf9\xdb\x49\x5f\xc5\xe6\xdf\x7a\x37"
			  "\x87\x32\x73\xb3\x6d\x08\xac\xef\xb8\xa3\x0a\x14\xb6\xd7\x7f\x76"
			  "\x0f\x02\x9b\x93\x45\x7b\xee\xd4\x92\x53\x40\x6c\x9e\xe1\x52\xe4"
			  "\x22\x31\xcf\xcb\xc5\x30\x11\xe0\xb6\xe4\x17\xad\x03\xe9\x8d\xaf"
			  "\x2e\xc8\x5a\x2c\x89\x49\x9d\xde\x67\xc4\x03\x1d\x91\x8f\x30\x7d"
			  "\x11\xd1\x93\x8c\x7b\xb9\x17\x71\x7c\x3b\x14\x34\x03\xba\x76\x8a"
			  "\x4b\xe0\xd0\x93\xc9\x25\x59\xce\x3f\xcc\x5d\x38\xfe\x18\xfa\xbb"
			  "\x0d\xde\xe0\xd4\x17\x3e\x18\x27\x58\x73\x86\x89\x71\xa0\x7f\xcd"
			  "\xd4\x8e\x54\xcc\x3f\x63\x1e\x44\x3d\xb9\x39\x77\x4c\xb8\xaa\x57"
			  "\x5f\x7b\xd2\xa9\x86\x9c\x15\xbe\xa7\xf1\xea\xa9\xdf\x95\x32\x2c"
			  "\xbf\x93\xb0\x31\xaf\x6e\x74\xe0\x37\x7c\x94\xf0\x4f\x6c\x44\x3c"
			  "\xd9\x1c\x3b\x48\x24\x8b\x28\x2f\x65\x54\xce\x69\xf4\x5b\xb6\x11"
			  "\xef\xab\xec\x45\x4d\x10\x58\xa7\xf2\xa7\xc7\x1a\x9f\xfa\xdc\xcd"
			  "\x54\xfa\x0d\xe8\x96\xbc\x0d\x55\x0a\x49\xf1\x2a\x31\x28\x15\x26"
			  "\x56\xf2\xa0\x6c\x84\x51\xbd\x6b\xee\x20\xc5\xd6\x4b\x36\x5f\x7b"
			  "\xb8\xd0\xeb\x41\xb6\x7c\xfb\x9d\x9d\xfd\x62\x0e\xb7\x99\xca\x17",
};

ke_test_vector_t modp8192 = {
	.method = MODP_8192_BIT, .priv_len = 64, .pub_len = 1024, .shared_len = 1024,
	.priv_a	= "\xa6\x48\x81\x47\x45\xfe\x1e\xd1\x98\x9b\x75\xba\x6d\xd5\x01\xe4"
			  "\x4e\x77\x6d\xc9\x97\xa0\xae\x27\x37\x64\x61\xb0\xee\x79\x65\x94"
			  "\xc2\xe6\xdb\x07\xe5\xf9\xd8\x7d\x94\x4d\x37\x01\x22\x38\xe5\x70"
			  "\xaf\x52\x3a\x47\xf3\xe8\x6e\x2a\x4c\xd0\xdb\x3a\x49\x40\xcf\x0b",
	.priv_b	= "\xd8\xf2\xb0\x9f\x17\xbe\x6f\x13\x64\x5c\xb4\x57\xbe\x0a\xa8\x02"
			  "\x02\xe8\x43\xf5\x10\xd2\x93\xf9\x32\x55\x98\x2f\x89\x87\x26\x61"
			  "\x79\xd6\x01\x90\x95\x70\xa3\xf9\xcc\x58\x50\x8b\x62\xf8\x8d\x64"
			  "\xe0\xd0\x59\xf1\xa2\xed\xea\xd4\x42\x82\x04\xcc\x09\x2a\x2d\x82",
	.pub_a	= "\x1f\x6d\x66\x0a\xfb\x04\x87\x42\x55\xe6\x83\xee\x89\x9a\xd2\xfd"
			  "\xbc\xf7\x35\x7f\xdf\xe2\x47\xef\x3b\x2f\xf9\x41\x19\x48\x45\x26"
			  "\x44\x53\x41\xc5\xdf\xea\x28\xa7\x45\xa0\xff\x56\x95\x13\x24\x72"
			  "\x34\x1b\x11\xb2\xe6\x9b\xd8\xe9\x98\x18\xf0\x5c\x22\x74\x26\x65"
			  "\x9c\xf0\x34\x81\x55\x50\x1d\xce\x5a\x7b\x5d\x11\xae\xaf\xfb\xd0"
			  "\x52\xb5\xff\x8a\xc8\xe3\x5e\x78\x50\x29\x79\x53\x9e\xf4\xe4\xa7"
			  "\xa9\x51\x55\x1a\xcc\x10\x02\x72\x6b\xa8\xee\x95\x95\x70\x19\x92"
			  "\x34\x15\xd8\xec\x61\x27\xbe\x25\x2c\x44\x65\xb0\xef\xae\x8a\xee"
			  "\xcd\x39\x0f\x33\x55\xcf\x94\x52\x5e\xef\x38\xdb\x92\x62\x79\xb8"
			  "\xfd\xce\xe1\x3c\x43\x8e\xdd\xb2\xa5\x22\x44\xa7\x1b\xda\x1e\x85"
			  "\x30\x91\x82\xfa\x8c\x57\xff\x0c\xd4\x06\x9d\x4f\xfb\xcc\x42\xe5"
			  "\xe6\x60\xc1\xf0\x07\x3c\x85\x6a\xc6\x11\xcb\xf7\xf9\xf3\x2c\x95"
			  "\x67\xa1\x8e\x44\x94\xed\xe7\x42\x7b\x1a\x18\x16\x55\xdb\xc7\x91"
			  "\xc7\x9d\x54\x7d\x88\xfd\x49\x45\x9b\xd3\x56\xee\x47\x26\xa9\x5c"
			  "\xd3\x7b\xf2\x42\xe4\x8a\x94\xd9\x22\xcc\x06\x66\xce\x4f\x29\x91"
			  "\x2a\x1e\x4e\xc8\x8e\x3e\x40\xde\xdc\x95\xe1\x2c\xe2\x50\x47\x2e"
			  "\x58\x37\x49\x32\xe4\x64\x81\xe7\x03\x11\x81\x13\xb7\xdd\x6f\xef"
			  "\xae\xf5\xc5\xe6\x1c\x13\x6b\x76\xa1\xb0\xbf\x4d\xf0\x01\xc3\x09"
			  "\xd8\x59\xff\x13\x34\x6d\x52\x11\x9b\xa6\xc9\xca\x38\xe6\x71\x80"
			  "\x39\xae\x37\x4b\x1c\xe1\x13\xfb\xc6\x91\x8f\x32\xc3\x6a\x04\xd8"
			  "\xc6\x80\x08\x4e\xef\xf6\x80\xf9\x8f\xda\xaf\x27\x79\x9b\x21\x2d"
			  "\x2d\xea\x87\x4b\x4d\xeb\x5a\x87\xfb\x72\x36\xe2\x3e\x9b\x42\xcf"
			  "\xa9\xeb\x49\x41\xe6\xc2\xb8\xc0\xad\xbd\xb3\x61\xc1\x61\x5f\x9c"
			  "\xb6\xbd\x8f\x99\x3a\xe8\xca\x86\xb8\xd8\x7f\x2f\xb2\x33\xc0\x9f"
			  "\xd3\x8e\x44\xdc\x6d\x0c\x19\x89\xb9\x05\xc6\x36\xf4\x7a\xc3\x06"
			  "\x46\xa4\x22\x6e\xef\x76\xba\x92\xfd\xaf\xce\x71\xbc\x0c\x05\xfe"
			  "\xec\x8f\x8d\xfb\x59\x46\xf9\x7c\xc1\x1f\x1c\x52\x62\x4d\x01\x57"
			  "\x4e\x86\x03\x94\x8b\xba\xf7\xbd\xca\xbb\x15\xaa\x61\xdd\xbc\x32"
			  "\xdf\x9e\x5f\xad\x3b\xf8\xf6\xfb\x4d\x1e\x9c\x86\x9e\x7e\x0f\xaf"
			  "\xf6\x91\x7e\x08\xf4\xfb\x55\xd8\xe0\x4b\xd9\xcd\x23\x57\x83\x58"
			  "\x59\xd7\x56\x93\xee\x14\x2f\x2b\xd2\x83\xa4\xce\x45\xa2\x90\xba"
			  "\x15\x4d\xca\x0c\x8b\x29\x7a\xe5\xbc\xba\xa0\x45\xd0\x08\x27\x75"
			  "\x61\x3a\x83\x99\x2f\x39\xc6\x41\x03\xc0\xb7\xd6\xfb\x5b\x94\x64"
			  "\xc3\xfe\xca\x32\x1c\xe3\x66\xc8\xb4\x49\x44\x1c\x63\xb0\xaa\x18"
			  "\x31\x4b\x15\x8a\xda\x77\xc5\xfd\xea\x33\x6c\x9a\x45\x4c\x24\xb5"
			  "\x1c\xd2\x5d\x56\x98\x3c\xdf\xb1\x5f\x10\xee\xc1\x17\xec\xbe\x7c"
			  "\xdb\x24\xdd\xdb\x22\xf6\x47\x8a\xfc\xc5\xb5\xa8\x9a\x8f\xb8\x27"
			  "\x72\xa1\xd4\x5e\x43\xcd\x7b\x33\x2e\xe3\x09\x94\x77\x20\xfe\xba"
			  "\xae\x2e\x9b\xc1\x43\xdd\x9d\x44\xd8\xd6\xfe\xff\xe3\x0f\xf6\xd3"
			  "\x71\xa1\xf8\xda\x1c\xff\x41\x21\x6f\x07\xc9\x55\x99\x6f\x0a\xef"
			  "\xd6\x5a\x6c\xa5\xdd\xba\x98\x46\x30\x43\xad\x27\xe4\xf1\x1e\x3a"
			  "\x89\x4a\xb3\xc8\x6d\xf7\xe9\x2f\x48\xd0\xd7\x29\x38\x5c\xe7\xac"
			  "\xbc\x3f\x65\x5e\x23\xdd\xc1\xad\x73\xed\x1a\xee\x81\xf3\x63\x29"
			  "\x7e\x72\x8f\x1a\xfc\x2d\x03\xf9\xbb\x3c\x38\x42\xc2\xfb\x53\x2f"
			  "\x56\xd6\xca\xb9\xeb\xa4\x17\x46\xdb\x53\x56\xf0\xdd\x1d\x8a\xfc"
			  "\x03\x06\x4d\x8c\x97\x7e\xf0\xc6\x5d\x6d\x5a\x23\xed\xee\xf9\x11"
			  "\xed\x04\x34\x0c\x04\xa0\x60\xf9\xa8\xfe\x8f\xfa\xd6\xf3\x27\x3d"
			  "\x32\x48\xbe\x3b\x56\x3a\xe8\x76\xe9\x54\xe7\x81\xef\xe3\x8f\xd9"
			  "\x03\x42\x5b\xa7\xd2\x69\x96\x39\x05\x8f\x41\x25\x35\x3a\x56\x66"
			  "\x5a\xc8\x36\xda\x84\xc5\xe9\x2e\x55\xac\xe9\xeb\xdc\x8f\xd8\x26"
			  "\x06\x3b\x61\x7f\xd7\x78\x4e\x5b\xe5\xd7\xca\x76\xf0\xd0\x71\xf6"
			  "\x45\x21\x30\xdf\xfc\x69\x82\xcf\xc3\x02\xda\x4f\x4e\x51\x59\x4f"
			  "\x56\xd1\x60\x5a\xd9\x12\x21\x7a\xa9\x99\x81\x5a\xa8\xa8\x10\x94"
			  "\xd7\x3e\x58\x7e\xd9\xde\xbc\xf5\x83\xf0\x92\x4f\xe6\xdd\x17\xb3"
			  "\x9c\xdd\xd5\x7b\xc0\xb4\xb5\x8c\xae\x7b\x63\x1a\xaa\xd0\x93\x55"
			  "\x31\x28\xd7\x8e\xec\x8c\x28\x64\x81\x9e\x2e\x47\xa9\xb3\xa0\x06"
			  "\x2a\x3d\xfe\x27\x40\x9c\x13\x96\xa0\xff\x65\xd9\x05\xd8\x92\xa2"
			  "\x63\x0e\xbf\xe2\xc5\xe3\x51\xbf\x20\x74\xdc\xbe\x5a\xfb\xd5\x5c"
			  "\x1b\x8f\x41\x21\x43\xaf\xdf\x35\xd1\x46\x80\x16\xf0\x28\x5d\x55"
			  "\x52\x0c\x05\x24\x63\x68\x5d\x50\x7a\x50\xde\x2d\xfa\xae\xd0\x71"
			  "\x9a\xad\x2a\x56\xb6\xa2\x52\xbc\xe1\x93\xc4\x93\x36\xc6\x08\xec"
			  "\xf3\x26\x14\x0b\xa5\x4c\x9e\x15\xf3\x48\xf6\xd1\xd4\x45\x41\xd7"
			  "\xb7\x5c\xd9\x89\x89\xe6\x62\xe6\x4b\x9f\x3a\xf7\xc7\x77\xd7\xc3"
			  "\xae\x18\x3c\x5c\xee\x2a\xf1\x80\x8a\x73\x83\xa4\x0e\x63\x12\x04",
	.pub_b	= "\x8e\x2b\x1b\xac\x90\x76\x05\x3f\xf4\xd5\xde\xa3\xad\xf0\xe9\x40"
			  "\x27\x69\xb2\x65\x1a\x54\x5c\x89\x0a\x82\xbd\x59\xc3\x4f\x55\x02"
			  "\x77\xdd\xac\x4c\x3b\x43\xb5\x34\xf5\xa2\x15\x85\xba\xa2\x40\x6f"
			  "\xb7\x37\x6c\xdb\x3d\xd1\xc0\xc9\xa0\x6c\x17\x3c\xa3\xc1\xcf\x7f"
			  "\x86\x86\xcb\xaf\x9a\x5c\x27\x36\x8e\x47\xac\x3f\x30\x46\xe7\xd9"
			  "\xd9\x56\x54\x22\xfe\x43\xb6\xc9\x04\xf9\xd0\x63\x02\x02\xcd\x7d"
			  "\x77\xcc\x21\x1b\x7b\xea\x4a\xc1\x9c\xdb\x68\x0e\x34\x00\x43\x4b"
			  "\xd1\x66\x03\x07\x42\xc9\x79\x7f\x81\x0f\xcc\xff\x4b\x2c\x00\xf9"
			  "\xb9\xd4\x70\xde\x92\x65\x45\x32\xac\x64\x76\x49\xe6\xc0\x26\x77"
			  "\xbc\xb8\x09\xa6\xbd\xa1\x22\x73\x13\x4b\x1c\x7c\xa8\x88\xe1\x69"
			  "\xb1\xe2\x64\x48\x3f\x4b\x2f\xd8\xa3\xb4\xf6\x4e\x5a\x79\x06\xf0"
			  "\xcb\x72\x9c\x72\x0e\xe6\x39\xa6\x69\x7b\xa3\x32\x9e\xa0\x81\xb0"
			  "\xd6\xa2\x10\xee\xbb\x5b\xd5\x51\xd0\xd6\xbf\x28\x80\x0c\x67\xf4"
			  "\x38\xf4\xc3\x16\x3b\x83\xd2\x6c\xd3\xf3\x02\x34\x64\xeb\xa1\x6c"
			  "\xb3\xa5\x13\x6a\x64\xb5\xa6\x3a\x1b\x63\x5e\xe7\x03\x96\xdb\x37"
			  "\x4f\xc8\xb8\x60\x86\x45\x30\x61\x97\xfa\x8e\x3d\xae\x48\xa4\x7d"
			  "\xfe\x72\x6a\xe9\x98\xeb\x77\x13\x4a\x4e\x6a\xae\x24\xf2\xd2\xad"
			  "\xa6\xca\x7e\x98\x38\x53\x24\xea\x98\x09\x2c\x5b\x0f\x46\x9c\x6a"
			  "\x50\x0c\x46\x27\x98\x51\x56\x06\x6e\xca\xe9\xb8\x47\xe7\x20\xd7"
			  "\x71\x38\xdc\x17\x74\x4e\x0b\x9d\xa6\xe3\x8e\x69\x28\x2a\x90\x45"
			  "\x3e\x72\xdf\x62\x88\x3a\x8a\x04\x59\x05\x43\x2a\xa3\x22\x90\x1f"
			  "\x62\xa2\x46\x03\x90\x81\xd8\x1a\x12\x77\x37\x08\x34\x58\x0d\x0b"
			  "\x5d\x02\x5f\xa3\x66\xfb\x6e\xeb\x22\x5c\xe3\xbc\xcc\x8d\xa5\x94"
			  "\xe8\x14\xd5\x08\xfe\x8b\x75\x6a\xf1\x9a\xde\x32\x26\x10\xb4\xb9"
			  "\x9f\x5d\x60\x83\x66\xdf\xfc\xfe\x4d\xdc\xd9\x95\xcf\xa9\x89\xc4"
			  "\xe6\x92\x8a\xd5\x99\x35\x7e\xc5\x5b\x42\x62\x92\x3f\x7e\x2a\x32"
			  "\xd0\x64\x8b\x0a\x61\x0f\x3d\xa8\x83\xb8\x95\x77\x34\x3c\xd0\x31"
			  "\xe6\xf9\x01\x69\x2f\x3f\xb4\xa1\x03\x18\x1b\xf1\xbd\xac\xe1\x8d"
			  "\x61\xb8\xd0\xf6\x7f\xd9\x7a\x3f\x26\xc1\x81\x16\x80\x93\xe6\x65"
			  "\x56\x5b\xc1\x56\xfb\xf1\x65\xd9\x5a\x21\xe9\xf2\xd2\xde\x7d\x89"
			  "\x12\xa2\x4f\x8f\xd9\x61\x0c\x51\xbf\xfc\x36\x65\x36\x5b\x51\x20"
			  "\x90\x42\x3a\x99\xd1\x88\x21\xf1\xd7\x1a\xe1\xed\xca\xdd\x08\x4e"
			  "\xb5\x29\xa3\x72\x83\xed\x85\x15\xe5\x20\x5c\x6e\xbb\x39\x2a\x38"
			  "\x12\xde\xe0\x5a\x1c\x10\x7b\xb2\x5b\x46\xa2\x80\xb4\x3e\x59\x1a"
			  "\x7d\x09\x70\x9d\xbc\x6f\x76\x8a\xf4\xb4\xc9\x35\x26\xfb\xcc\x7a"
			  "\xb5\x52\x8b\x06\xe3\x6f\x50\xb1\xf6\x4d\xf4\xbd\xa9\x1e\x39\x37"
			  "\x56\x1b\x25\x62\x0f\x7a\x4a\xfb\xe1\xff\xf5\x28\xf8\x40\xd4\xf2"
			  "\x99\xd1\xad\xd9\x23\x5b\x4e\x3b\x8a\x57\x87\x54\x23\xd7\x73\x54"
			  "\xf9\xa0\x0e\xa5\x49\x5f\x8e\x93\x8e\x41\xeb\x69\x7b\xef\x48\xeb"
			  "\x1b\xb6\x6b\x48\x29\xff\x4e\x89\xb1\xf2\xbd\xef\x77\x7a\x92\x71"
			  "\x97\xac\x21\x76\x6e\x54\xd3\x89\xe2\xca\x07\xbe\x00\xe8\xb4\xd4"
			  "\xd5\x63\xf1\x4b\xbb\x76\x16\xc8\xdb\xf1\x14\x18\x4b\xbe\x39\x00"
			  "\x75\xf2\x85\x2b\x8e\xd8\xc0\x81\x12\x9b\xe4\x9f\x98\x74\x6f\x70"
			  "\xa8\xa4\x7f\x14\x66\x21\x91\x61\x86\x2b\xa6\x01\xf5\xd1\x00\x6c"
			  "\x7b\x3a\x39\xb8\x3b\x6c\xe8\x4b\x16\xd0\x99\xcc\x2a\x9f\x63\xdd"
			  "\x77\xc1\x1d\x14\x98\x38\xca\x3f\x5a\xdf\xc4\x9d\xf5\xe6\xa8\x2a"
			  "\x4b\xfb\x53\x68\x6e\x31\x39\x12\x07\x2b\x7e\xe3\x78\x3d\x23\x79"
			  "\xc4\x2b\xc1\xac\x29\x20\x59\x91\xf0\xd8\xfb\x9d\x20\x5b\x21\x70"
			  "\x0e\xa8\x46\xca\xfc\x09\x38\x12\x1c\x2c\xa4\x67\xa8\x94\x04\x87"
			  "\xa4\xaa\x2c\x25\xea\x57\x35\xc2\x4e\xa9\x15\x0e\x7c\x46\xe3\xdf"
			  "\x40\x77\x81\xd6\xf3\x9c\xee\xd4\x22\x3b\xe9\x20\x98\x1a\x18\x4e"
			  "\xd3\x3d\xe6\xd0\xfc\xcc\x2d\x5c\x72\x69\xe7\xc4\xca\x12\xe9\xb7"
			  "\xbe\x84\x31\x9e\x5d\x6f\xcf\x91\x37\x15\xeb\xf8\x08\xba\x50\xc3"
			  "\xb4\xad\x36\x09\xe0\xd3\x61\xea\x9b\xe6\x28\x56\x33\x73\x2b\xe7"
			  "\x51\x52\xc5\x40\x66\x34\x47\x03\xe4\xd9\xff\x4b\x23\xaf\x80\x63"
			  "\xbd\x30\xe0\x2d\xe2\x08\x4a\x09\xd3\x04\x0a\x4d\xbc\x7b\xac\x80"
			  "\x94\x8c\x4d\xd7\x4a\x35\x72\xa4\x18\x66\xf2\x0e\x01\x6a\x0f\x31"
			  "\x17\x31\x07\xb9\x65\x68\xd6\xac\x2f\x53\x36\x01\x3f\x06\x18\x77"
			  "\xae\xbe\xb4\xac\x60\x9a\x73\x3d\x7d\x40\xdc\xed\x19\x27\x03\x57"
			  "\x0a\xca\xe4\xcc\xc6\xfe\x3b\x7a\x9d\x73\xb2\xa4\xf1\x21\x32\x11"
			  "\x21\x86\x77\x1e\x25\x96\xc8\xb0\xce\xd5\x1e\x4c\xea\xd0\x04\xde"
			  "\x67\xa8\x36\x23\x89\x85\x7b\xa9\x2a\x0c\x52\xd5\x63\xbe\x17\xdb"
			  "\x82\xd5\xf6\x51\xc3\x9a\x9f\xab\x58\x46\x87\x0c\xb3\xc1\x23\x87"
			  "\x61\xac\x00\xed\x46\xe5\x79\x2c\x5b\xd6\x6e\xaa\xee\x4b\x08\xb2",
	.shared	= "\x7d\xc5\xc9\x35\xc9\x62\x79\x93\x1d\xfc\x55\xc6\x20\x36\x95\x7c"
			  "\xa1\x5d\x9c\x0f\xdf\x95\xa7\x1f\x59\xae\xb7\xa6\x39\x0f\xd1\x7e"
			  "\x93\x34\x7f\xb6\x72\x96\xf2\xe3\x20\x30\x40\xe7\x77\x51\xc1\x0a"
			  "\x08\xf3\xad\x86\x40\xec\xf0\xd2\x9d\x88\xba\xff\x50\xf7\x4e\x88"
			  "\x1f\xf0\xdd\x12\x17\x05\x97\x1e\x3c\xe3\x91\x65\x10\xd9\x16\x32"
			  "\x21\x44\xd8\x81\xed\x17\x9a\x60\x34\xee\x4d\x81\x03\xbc\x05\x56"
			  "\xba\x7f\xa7\x31\xc9\x3b\x57\x5e\x38\x1c\x45\xbf\xfc\x51\x48\xf3"
			  "\x05\xa9\x74\x39\xdf\xa1\x34\x48\x62\x31\x5d\x58\x45\x16\xc1\x9e"
			  "\x26\x38\xb4\x59\x95\xdd\x92\x52\x1e\x26\x20\xed\xd2\xb2\xb3\x98"
			  "\x6b\xde\xbe\xf1\xa0\xbc\x52\xc9\xfe\x97\x65\x78\xd6\xce\x91\xb1"
			  "\x8e\x9b\x04\xfc\x74\xb9\x1d\x52\x7d\x0b\xf0\x1d\x2a\x3c\xde\x2e"
			  "\x4c\x49\xee\x62\x9c\x59\x09\x12\xd7\x4a\xd7\x0d\x03\x72\x3d\x04"
			  "\x58\xd5\x19\x9e\x42\x97\x2c\x32\xb9\xda\x5c\xee\xaf\x40\x44\xc9"
			  "\xfd\x69\xcf\x6e\x81\x6b\x2b\xa4\x21\x3b\xde\x2d\xc3\xb7\x03\xe3"
			  "\xa3\xf2\x77\xd8\x70\xf5\x6f\x24\x48\xbf\x48\xb5\xa5\xff\x86\x06"
			  "\x0c\x23\xc2\xc9\x8e\xfa\x0f\xa6\xaa\x77\x95\x1a\x1a\x91\xfe\x30"
			  "\x6b\x18\xc5\xfb\x13\x76\x7a\xa5\x29\xa3\x02\xce\x2b\x46\x74\x5a"
			  "\x1c\xe6\x7c\x3a\xb7\x43\xea\xf9\x3f\x53\xa9\xfb\x63\x94\x26\x42"
			  "\x82\xdc\x9a\x4a\xd6\xbb\xa9\xd1\x6e\xc2\xc7\x92\x28\x3b\x33\xc3"
			  "\x58\xf6\xfb\x44\x5c\x5d\xea\xe0\x62\xaa\x30\x97\x30\x38\x36\x47"
			  "\xb1\xae\x3b\x99\x35\xab\xc9\x45\x2a\x07\xec\x34\x88\x61\x70\xb0"
			  "\x47\xf9\xd6\x61\x23\x48\x71\x6f\x85\x3b\xf1\x8a\x17\x72\xc1\x99"
			  "\x6d\x4f\x94\x2b\x11\xf2\x75\x33\x08\x67\x08\xba\xa6\x50\x2a\x1f"
			  "\x6b\x0e\x38\x0e\x52\xea\xe0\xd8\xcd\xd0\x11\x80\xa7\xb9\x97\xd9"
			  "\x68\x1b\x21\x7c\xe2\x69\x8e\x14\x61\x39\x49\x65\xbb\xc2\x7c\x2e"
			  "\x8e\xb1\x83\x28\x7b\xf0\xf0\x62\xdc\x35\x33\xea\xc9\x14\x3f\x83"
			  "\x60\xd3\xec\x37\xd4\xe3\x87\x86\x2c\x43\x53\x80\x3f\x39\x36\x4d"
			  "\x1e\x73\x2a\x5c\x8c\xfb\x2d\x2e\x78\xd1\xf1\xe5\x8c\x89\x35\x1e"
			  "\x0a\xd3\x07\x3c\x05\x0d\x4e\xa0\x81\x6a\xee\xb5\x27\xef\xa6\x2e"
			  "\x04\x07\x01\xf7\x40\x9b\xbf\x7b\xf8\x3d\x0f\x68\x79\xc6\x96\x16"
			  "\x53\x5b\xb0\x8f\xb1\xbd\xd4\x28\xa7\x73\x37\x7b\x06\xbf\x9c\xd2"
			  "\xb4\x28\x82\xb4\xd2\x63\xf9\xb2\x56\x7c\x28\x02\x4b\x91\x63\x03"
			  "\xdc\x58\x6e\xb5\x88\x2e\x76\x3a\x53\x4a\xd4\x22\x7c\xa5\xca\xdc"
			  "\x96\x14\xa6\xfb\x64\xe9\xa5\x09\xf2\x3e\x4d\x7e\x40\x01\x93\x05"
			  "\xe0\xad\x2e\x20\x7c\x4e\x17\x45\xa8\x70\x7e\xd1\xc7\xbb\x10\x94"
			  "\x81\x36\x22\x62\x3f\x44\xe2\xfa\x36\x9e\x4e\x37\x75\xdf\x8d\x6f"
			  "\x15\x73\x41\xf0\x6f\x32\xce\xa9\xfc\x06\x8f\x39\xcf\xfb\x34\x16"
			  "\xba\x74\x7d\xe6\x02\x18\xd3\xe4\x8c\xee\xd1\xef\xdc\xd4\x59\x8d"
			  "\xeb\x25\x7a\x09\xc8\xcc\x38\x7b\xcb\x8b\x3c\xb8\x77\xb4\xad\x1a"
			  "\x9e\x78\x20\x58\x6b\x36\x4d\x5d\xbb\x07\x38\xba\x65\xf4\xfe\x4d"
			  "\x86\x94\xdf\x77\x52\x08\xb1\x75\x07\x85\xaf\x77\xe9\xb3\xab\x3a"
			  "\x35\x30\x8a\xfd\x34\x99\x3b\x8f\xfc\x9a\x9e\xe1\x46\xcb\xbf\xf3"
			  "\xbd\xb9\x9c\x8c\x4e\x29\x6d\xa3\x29\x5a\x1f\x88\x23\xdf\x89\xa7"
			  "\x88\x7c\x90\x75\x5a\xd5\x97\x60\x17\xdf\x5e\x96\x04\xf8\xae\x24"
			  "\xd8\xf1\xa3\x3a\xa1\x40\x40\xd1\x45\x8e\xa5\xe5\x32\x32\xcc\xcc"
			  "\x1c\x8a\x9f\x6d\xa0\xa8\x89\x9d\x75\x2b\xd2\xc2\x07\xb2\xe1\x1a"
			  "\xcd\x4f\x31\x80\xb2\x19\x5f\x06\x9d\x8e\x7a\xf1\x2e\x15\x1d\xbc"
			  "\x4c\xb4\x75\x6f\x7b\xb7\xe4\x70\xd6\x0c\xe1\x27\x93\xce\x4a\x63"
			  "\xde\xe7\x48\x87\x5c\x75\xae\x34\x82\x5a\x71\xda\x1f\x93\x72\x64"
			  "\x7c\x7b\xda\x6a\xd7\xf9\xe8\x1f\x72\x67\x71\x68\x6f\x85\x64\x53"
			  "\x72\x80\x8f\x1f\x7a\x15\x1c\x11\xfd\x11\xf7\x8c\xc4\x43\x41\x60"
			  "\x03\xf2\x79\x19\x27\xdf\x98\x37\xaf\x07\xb3\x8d\x0b\xee\x01\x85"
			  "\x6f\x6d\xa9\xbd\x0e\x1f\x79\xdf\x21\x78\x24\xbf\xc5\x09\x29\x9e"
			  "\x64\x0d\x8e\xff\xc9\xe7\xe8\x51\xd3\xe8\xfe\xa5\xe0\xa9\x8d\x54"
			  "\x17\x3a\x34\xaf\xd9\xbd\xfd\xaa\x8c\x81\x79\xfb\x65\x85\x67\x73"
			  "\x32\x2d\x32\x00\xcb\x3b\xe2\x0e\x43\xe8\x40\xf6\x42\xf2\x74\x00"
			  "\xa0\xfd\xe6\x96\x4a\x2b\xbb\xa9\xd9\x95\xc4\x42\x75\x12\x3f\xbb"
			  "\x79\x35\x9c\x91\xb5\x24\x10\xc4\xc5\xbd\x4a\x4c\x47\xd8\x89\x92"
			  "\x70\xa5\xe6\xc9\xed\x2e\xbd\x98\xc0\x17\xb0\xad\x8c\x31\x95\x81"
			  "\x84\x86\xb1\xaa\x42\xf6\x2e\x10\x92\x2f\x67\x73\x33\xb9\x02\x43"
			  "\x52\x24\x05\xdb\x9c\xec\xc5\xf1\x3e\x78\x05\xcb\x04\xd6\x91\xa8"
			  "\x51\x9e\x48\xa8\xae\xa8\x8d\x13\x2d\xcd\xa1\xbe\x23\x9e\x00\x4c"
			  "\x0a\x59\xf8\x18\xb0\x0a\x06\xe2\x0a\xb4\x16\x02\xa7\x21\x4c\xac"
			  "\x9a\x80\x62\x7f\xb6\xd6\xa0\x3b\x11\xd3\x30\xf9\x3d\xfd\x26\x27",
};
