/**
 * @file    special_fn.c
 * @brief   Routines for performing manipulations of plot data.
 *
 *          This file contains a number of routines for manipulating
 *          data sets in order to generate new data which are functions
 *          of raw data.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @todo    This file is becoming large and could probably be split by
 *          function.
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <string.h>

#include "ygraph.h"

GtkWidget* file_entry_A;
GtkWidget* file_entry_B;

GtkWidget* file_entry_1;
GtkWidget* file_entry_2;
GtkWidget* file_entry_4;

GtkWidget* new_window_check;

void
dataset_calc_subtract(DataSet* data_set)
{
  DataSet* s0;
  DataSet* s1;
  Frame* f0;
  Frame* f1;
  Frame* new_frame;
  gdouble* xy_point;
  gdouble x_val;
  gdouble y_val_0;
  gdouble y_val_1;
  gint s0_idx;
  gint s1_idx;
  gint i;
  gint j;

  if ((data_set->type != YG_SUBTRACT) || (data_set->cmpt_set->len != 2))
    return;
 
  s0_idx = g_array_index(data_set->cmpt_set, gint, 0);
  s1_idx = g_array_index(data_set->cmpt_set, gint, 1);

  s0 = g_array_index(global_data_set_list, DataSet*, s0_idx);
  s1 = g_array_index(global_data_set_list, DataSet*, s1_idx);

 if (data_set->frame != NULL)
    g_array_free(data_set->frame, TRUE);

  data_set->frame = g_array_new(FALSE, FALSE, sizeof(Frame*));
  data_set->nframes = 0;

  for (i=0; ((i < s0->nframes) && (i < s1->nframes)); ++i)
    {
      f0 = g_array_index(s0->frame, Frame*, i);
      f1 = g_array_index(s1->frame, Frame*, i);

      new_frame = frame_init();
      new_frame->time = f0->time;

      for (j=0; j<f0->npoints; ++j)
        {
          xy_point = g_array_index(f0->xy_data, gdouble*, j);

          x_val = xy_point[0];

          y_val_0 = xy_point[1];
          y_val_1 = interp (f1->xy_data, x_val, INTERP_ORDER);

          frame_append_data_point(new_frame, x_val, y_val_0 - y_val_1);
        }

      dataset_append_frame(data_set, new_frame);
    }
}

/**
 * @brief    Read a pair of files and subtract individual data values.
 *
 *           Each of the files is read in turn and added to the 
 *           global_data_set_list. A new DataSet is created corresponding
 *           to the subtraction of individual frames of file0-file1.
 *           A new data set name is created based on the names of the
 *           given data sets (basically "label0 - label1"), and given to
 *           the new DataSet. The new DataSet is given the type YG_SUBTRACT,
 *           and the cmpt_set contains index references to the two DataSets
 *           which were used to generate the new DataSet.
 *
 * @param    file0  The name of the first file.
 * @param    file1  The name of the file which is to be subtracted from the
 *                  first.
 * @returns  The index of the new DataSet within the global_data_set_list
 *           if the operation was successful. Otherwise FAIL.
 */
gint
dataset_read_subtract_files(gchar* file0, gchar* file1)
{
  DataSet* data_set_0;
  DataSet* data_set_1;
  DataSet* subtract_data_set;
  gint data_set_idx_0;
  gint data_set_idx_1;
  gint subtract_data_set_idx;
  gint name_len;

  data_set_idx_0 = dataset_read_from_file(file0, option_read_skip_step);
  data_set_idx_1 = dataset_read_from_file(file1, option_read_skip_step);

  if ((data_set_idx_0 == FAIL) || (data_set_idx_1 == FAIL))
    return FAIL;
  
  data_set_0 = g_array_index(global_data_set_list, DataSet*, data_set_idx_0);
  data_set_1 = g_array_index(global_data_set_list, DataSet*, data_set_idx_1);

  subtract_data_set = data_set_init(NULL);
  subtract_data_set->type = YG_SUBTRACT;
  
  /* 6 = "( - )" */
  name_len = strlen(data_set_0->name) + strlen(data_set_1->name) + 6;
  subtract_data_set->name = g_malloc(name_len*sizeof(gchar));

  g_snprintf(subtract_data_set->name, name_len, "(%s - %s)", data_set_0->name,
             data_set_1->name);

  g_array_append_val(subtract_data_set->cmpt_set, data_set_idx_0);
  g_array_append_val(subtract_data_set->cmpt_set, data_set_idx_1);

  dataset_calc_subtract(subtract_data_set);

  subtract_data_set_idx = global_data_set_list_append(subtract_data_set);

  return subtract_data_set_idx;
}

/**
 * @brief    Carry out simple integration
 */
void dataset_calc_integrate(DataSet* s)
{
  gint      i;
  guint     k;
  gdouble* xypoint;
  gdouble* xypoint2;
  gdouble  min_y, max_y;
  Frame*   f;
  
  min_y=s->y_range[0];
  max_y=s->y_range[1];

  /* going over all frames */
  for (i=0; i<s->nframes; i++)
    {
      f = g_array_index(s->frame, Frame*, i);
      /* going over all points of an frame */
      if (f->xy_data->len > 1)
        for (k=1; k<f->xy_data->len; k++)
          {
            xypoint = g_array_index(f->xy_data, gdouble*, k);
            /* add last point */
            xypoint2 = g_array_index(f->xy_data, gdouble*, k-1);
            xypoint[1] *= (xypoint[0]-xypoint2[0]);
            xypoint[1] += xypoint2[1];

            if (xypoint[1] < min_y)
                min_y = xypoint[1];
            if (xypoint[1] > max_y)
                max_y = xypoint[1];
          }
    }
  s->type = YG_INT;
  s->y_range[0]=min_y;
  s->y_range[1]=max_y;
}

/**
 * @brief    Carry out integration of frames
 */
void dataset_calc_frame_integrate(DataSet* s)
{
  gint     i;
  guint    k;
  gdouble* xypoint;
  gdouble* xypoint2;
  gdouble  min_y, max_y;
  Frame*   f;
  Frame*   f2;
  
  min_y=s->y_range[0];
  max_y=s->y_range[1];

  /* going over all frames */
  for (i=1; i<s->nframes; i++)
    {
      f = g_array_index(s->frame, Frame*, i);
      /* going over all points of an frame */
      for (k=0; k<f->xy_data->len; k++)
        {
          xypoint = g_array_index(f->xy_data, gdouble*, k);
          /* add last frame (which is already integrated) */
          f2 = g_array_index(s->frame, Frame*, i-1);
          xypoint2 = g_array_index(f2->xy_data, gdouble*, k);
          xypoint[1] += xypoint2[1];

          if (xypoint[1] < min_y)
              min_y = xypoint[1];
          if (xypoint[1] > max_y)
              max_y = xypoint[1];
        }
    }
  s->type = YG_FRAME_INT;
  s->y_range[0]=min_y;
  s->y_range[1]=max_y;
}

/**
 * @brief    Carry out integration of frames with time multiplied
 */
void dataset_calc_time_integrate(DataSet* s)
{
  gint     i;
  guint    k;
  gdouble* xypoint;
  gdouble* xypoint2;
  gdouble* xypointl;
  gdouble  save, min_y, max_y;
  Frame*   f;
  Frame*   f2;
  Frame*   fl;
  
  min_y=s->y_range[0];
  max_y=s->y_range[1];

  /* set 0-th frame to 0.0, but safe it to last frame */
  f = g_array_index(s->frame, Frame*, 0);
  f2 = g_array_index(s->frame, Frame*, s->nframes-1);
  for (k=0; k<f->xy_data->len; k++)
  {
    xypoint = g_array_index(f->xy_data, gdouble*, k);
    xypoint2= g_array_index(f2->xy_data, gdouble*, k);
    xypoint2[1] = xypoint[1];
    xypoint [1] = 0.0;
  }

  /* going over all frames */
  for (i=1; i<s->nframes-1; i++)
    {
      f  = g_array_index(s->frame, Frame*, i);
      f2 = g_array_index(s->frame, Frame*, i-1);
      fl = g_array_index(s->frame, Frame*, s->nframes-1);
      /* going over all points of an frame */
      for (k=0; k<f->xy_data->len; k++)
        {
          xypoint  = g_array_index(f->xy_data, gdouble*, k);
          xypoint2 = g_array_index(f2->xy_data, gdouble*, k);
          xypointl = g_array_index(fl->xy_data, gdouble*, k);

          /* save value from this frame (for use in the next) in temp. var */
          save = xypoint[1];
          /* update this frame */
          xypoint[1] = xypoint2[1] + xypointl[1] * (f->time - f2->time);
          /* save old value in final place and also save old time */
          xypointl[1] = save;

          if (xypoint[1] < min_y)
              min_y = xypoint[1];
          if (xypoint[1] > max_y)
              max_y = xypoint[1];
        }
    }
  /* do last frame */
  f2 = g_array_index(s->frame, Frame*, s->nframes-2);
  fl = g_array_index(s->frame, Frame*, s->nframes-1);
  for (k=0; k<f->xy_data->len; k++)
    {
      xypoint2 = g_array_index(f2->xy_data, gdouble*, k);
      xypointl = g_array_index(fl->xy_data, gdouble*, k);
      xypointl[1] = xypoint2[1] + xypointl[1] * (fl->time - f2->time);
    }

  s->type = YG_TIME_INT;
  s->y_range[0]=min_y;
  s->y_range[1]=max_y;
}

gint
dataset_read_frame_integrate_file(gchar* file, gint option_read_skip_step,
                                               gint option_t_int )
{
  gint idx;
  DataSet* s;

  idx=dataset_read_from_file(file, option_read_skip_step);
  if (idx != FAIL)
    {
      s = g_array_index(global_data_set_list, DataSet*, idx);
      if (option_t_int)
          dataset_calc_time_integrate(s);
      else
          dataset_calc_frame_integrate(s);
    }
  return idx;
}

gint
dataset_read_integrate_file(gchar* file, gint option_read_skip_step)
{
  gint idx;
  DataSet* s;

  idx=dataset_read_from_file(file, option_read_skip_step);
  if (idx != FAIL)
    {
      s = g_array_index(global_data_set_list, DataSet*, idx);
      dataset_calc_integrate(s);
    }
  return idx;
}

/**
 * @brief    Carry out the pointwise rescaling of a pair of DataSets.
 *
 * @param    data_set  A DataSet
 *           scale     A number
 */
/**
 * @brief    Carry out the pointwise re-scaling of a dataset.
 *
 *           The DataSet is modified by multiplication by a scale factor.
 *
 * @param    data_set  The DataSet to be re-scaled.
 * @param    scale     The factor by which the data_set is to be re-scaled.
 */
void
dataset_calc_rescale(DataSet* data_set, gdouble scale)
{
  Frame* frame;
  gdouble* xy_point;
  gdouble rescale_factor;
  gint i;
  gint j;

  rescale_factor = scale / data_set->scale;
  data_set->scale = scale;
  data_set->y_range[0] = G_MAXDOUBLE;
  data_set->y_range[1] = -G_MAXDOUBLE;
 
  for (i=0; i < data_set->nframes; ++i)
    {
      frame = g_array_index(data_set->frame, Frame*, i);
      frame->y_range[0] = G_MAXDOUBLE;
      frame->y_range[1] = -G_MAXDOUBLE;

      for (j=0; j<frame->npoints; ++j)
        {
          xy_point = g_array_index(frame->xy_data, gdouble*, j);
          xy_point[1] *= rescale_factor;

          frame->y_range[0] = MIN(frame->y_range[0], xy_point[1]);
          frame->y_range[1] = MAX(frame->y_range[1], xy_point[1]);
        }
      data_set->y_range[0] = MIN(data_set->y_range[0], frame->y_range[0]);
      data_set->y_range[1] = MAX(data_set->y_range[1], frame->y_range[1]);
    }
}

/**
 * @brief    Take the derivative of the given frame.
 *
 *           Does a centred finite difference at each interior point of a
 *           frame. One-sided differences are done at the boundaries.
 *
 * @param    f  The Frame to be differenced.
 * @returns  A new Frame containing the derivative data.
 */
Frame*
frame_derivative(Frame* f)
{
  Frame* df;
  gdouble* xypoint_m;
  gdouble* xypoint_p;
  gdouble xval_m;
  gdouble xval_p;
  gdouble yval_m;
  gdouble yval_p;
  gdouble xval;
  gint i;

  df = frame_init();
  df->time = f->time;

  xypoint_m = g_array_index(f->xy_data, gdouble*, 0);
  xval_m = xypoint_m[0];
  yval_m = xypoint_m[1];

  xypoint_p = g_array_index(f->xy_data, gdouble*, 0);
  xval_p = xypoint_p[0];
  yval_p = xypoint_p[1];

  frame_append_data_point(df, xval_m, 0.0);

  for (i=1; i<f->npoints-1; ++i)
    {
      xypoint_m = g_array_index(f->xy_data, gdouble*, i-1);
      xval_m = xypoint_m[0];
      yval_m = xypoint_m[1];

      xval = g_array_index(f->xy_data, gdouble*, i)[0];

      xypoint_p = g_array_index(f->xy_data, gdouble*, i+1);
      xval_p = xypoint_p[0];
      yval_p = xypoint_p[1];
      
      frame_append_data_point(df, xval,
                              (yval_p - yval_m) / (xval_p - xval_m));
    }

  xypoint_m = g_array_index(f->xy_data, gdouble*, f->npoints-2);
  xval_m = xypoint_m[0];
  yval_m = xypoint_m[1];

  xypoint_p = g_array_index(f->xy_data, gdouble*, f->npoints-1);
  xval_p = xypoint_p[0];
  yval_p = xypoint_p[1];

  frame_append_data_point(df, xval_p, (yval_p - yval_m) / (xval_p - xval_m));

  return(df);
}

/**
 * @brief    Take the derivative of each frame in a DataSet.
 *
 *           A new DataSet is created by taking the derivative of
 *           each frame of an existing DataSet. The new DataSet is
 *           added to the global_data_set_list, and given a name
 *           "D<label>" where <label> is the label of the original DataSet.
 *           The new DataSet is given the type YG_DERIVATIVE, and its
 *           cmpt_set points to the original data.
 *
 * @param    idx  The DataSet index of the original data set within the
 *                global_data_set_list.
 * @returns  The newly created DataSet containing the derivative data.
 */
DataSet*
dataset_derivative(gint idx)
{
  Frame* f;
  Frame* df;
  DataSet* data;
  DataSet* d_data;
  gint i;
  gint name_len;
  
  data = g_array_index(global_data_set_list, DataSet*, idx);

  d_data = data_set_init(NULL);
  d_data->type = YG_DERIVATIVE;
  g_array_append_val(d_data->cmpt_set, idx);

  name_len = strlen(data->name) + 2;
  d_data->name = g_malloc(name_len*sizeof(gchar));
  g_snprintf(d_data->name, name_len, "D%s", data->name);

  for (i=0; i<data->nframes; ++i)
    {
      f = g_array_index(data->frame, Frame*, i);
      df = frame_derivative(f);
      dataset_append_frame(d_data, df);
    }

  return d_data;
}

/**
 * @brief    Call the derivative calculator on a given DataSet.
 *
 *           Takes the derivative at each point of an existing DataSet.
 *
 * @param    data_set  The DataSet whose derivative is to be taken.
 */
void
dataset_calc_derivative(DataSet* data_set)
{
  DataSet* s;
  Frame* f;
  Frame* df;
  gint s_idx;
  gint i;

  if ((data_set->type != YG_DERIVATIVE) || (data_set->cmpt_set->len != 1))
    return;
 
  s_idx = g_array_index(data_set->cmpt_set, gint, 0);
  s = g_array_index(global_data_set_list, DataSet*, s_idx);

 if (data_set->frame != NULL)
    g_array_free(data_set->frame, TRUE);

  data_set->frame = g_array_new(FALSE, FALSE, sizeof(Frame*));
  data_set->nframes = 0;

  for (i=0; i < s->nframes; ++i)
    {
      f = g_array_index(s->frame, Frame*, i);
      df = frame_derivative(f);
      dataset_append_frame(data_set, df);
    }
}

/**
 * @brief    Creates a dialog box for loading a pair of files to be subtracted.
 *
 * @param    plot    The calling Plot.
 * @param    action  The calling action (unused).
 * @param    button  The calling button (unused).
 */
void
subtract_select_dialog_create(Plot* plot, gint action, GtkItem* button)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* hbox_A;
  GtkWidget* file_label_A;
  GtkWidget* file_button_A;
  GtkWidget* hbox_B;
  GtkWidget* file_label_B;
  GtkWidget* file_button_B;
  GtkWidget* okay_button;
  GtkWidget* cancel_button;

  UNUSED(action);
  UNUSED(button);
  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), SUBTRACT_DIALOG_TITLE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  label = gtk_label_new(SUBTRACT_MESSAGE);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  hbox_A = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_A);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_A, FALSE,
                     FALSE, 0);

  file_label_A = gtk_label_new(FILE_A_LABEL);
  gtk_widget_show(file_label_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_label_A, FALSE, FALSE, 0);

  file_entry_A = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_A), FILE_STR_SIZE);
  gtk_widget_show(file_entry_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_entry_A, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_A, FILE_ENTRY_LENGTH, -2);

  file_button_A = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_A), "clicked",
                     GTK_SIGNAL_FUNC(file_select_A), plot);
  gtk_widget_show(file_button_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_button_A, FALSE, FALSE, 0);

  hbox_B = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_B);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_B, FALSE,
                     FALSE, 0);

  file_label_B = gtk_label_new(FILE_B_LABEL);
  gtk_widget_show(file_label_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_label_B, FALSE, FALSE, 0);

  file_entry_B = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_B), FILE_STR_SIZE);
  gtk_widget_show(file_entry_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_entry_B, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_B, FILE_ENTRY_LENGTH, -2);

  file_button_B = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_B), "clicked",
                     GTK_SIGNAL_FUNC(file_select_B), NULL);
  gtk_widget_show(file_button_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_button_B, FALSE, FALSE, 0);

  new_window_check = gtk_check_button_new_with_label(NEW_WINDOW_LABEL);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), new_window_check,
                     FALSE, FALSE, 0);
  gtk_widget_show(new_window_check);

  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(okay_button),
                     "clicked", GTK_SIGNAL_FUNC(file_subtract), plot);
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_destroy),
                             GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                     okay_button);
  gtk_widget_show(okay_button);

  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_destroy),
                             GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                     cancel_button);
  gtk_widget_show(cancel_button);

  gtk_widget_show_all (dialog);
}

/**
 * @brief    Read the file selection for the first subtract file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_A(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  UNUSED(ok_button);
  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_A), selection);
  gtk_widget_show(file_entry_A);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the first subtract file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling even.
 * @param    plot           The calling Plot data.
 */
void
file_select_A(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  UNUSED(select_button);
  UNUSED(event);
  UNUSED(plot);
  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                     "clicked", GTK_SIGNAL_FUNC(file_entry_set_A), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Read the file selection for the second subtract file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_B(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  UNUSED(ok_button);
  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_B), selection);
  gtk_widget_show(file_entry_B);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the first subtract file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling even.
 * @param    plot           The calling Plot data.
 */
void
file_select_B(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  UNUSED(select_button);
  UNUSED(event);
  UNUSED(plot);
  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                     "clicked", GTK_SIGNAL_FUNC(file_entry_set_B), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                            (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Prompts the user for a pair of files to subtract, then performs
 *           and plots the subtraction.
 *
 * @param    ok_button  The calling button.
 * @param    plot       The calling plot.
 * @returns  TRUE if things worked out.
 */
gboolean
file_subtract(GtkObject* ok_button, Plot* plot)
{
  Plot* new_plot;
  GArray* data;
  gchar* file_A;
  gchar* file_B;
  gint data_idx;
  gboolean new_window;

  UNUSED(ok_button);
  file_A = gtk_entry_get_text(GTK_ENTRY(file_entry_A));
  file_B = gtk_entry_get_text(GTK_ENTRY(file_entry_B));

  new_window = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(new_window_check));
  
  data_idx = dataset_read_subtract_files(file_A, file_B);

  if (data_idx == FAIL)
    return TRUE;

  if (new_window)
    {
      data = g_array_new(FALSE, FALSE, sizeof(gint));
      g_array_append_val(data, data_idx);
      new_plot = plot_data_init(data);
      gtk_widget_show(new_plot->window);
    }
  else
    {
      plot_data_append(plot, data_idx);
      plot_window_display_all(plot);
    }

  return TRUE;
}

/**
 * @brief    Call a calculation operation on a specified data set.
 *
 * @param    data_set  The DataSet to be recalculated.
 */
void
dataset_recalc(DataSet* data_set)
{
  switch(data_set->type)
    {
    case YG_DATAFILE:
      dataset_read_from_file(data_set->fname, option_read_skip_step);
      break;
    case YG_SUBTRACT:
      dataset_calc_subtract(data_set);
      break;
    case YG_DERIVATIVE:
      dataset_calc_derivative(data_set);
      break;
    case YG_INT:
      dataset_read_integrate_file(data_set->fname, option_read_skip_step);
      break;
    case YG_FRAME_INT:
      dataset_read_frame_integrate_file(data_set->fname, option_read_skip_step, 0);
      break;
    case YG_TIME_INT:
      dataset_read_frame_integrate_file(data_set->fname, option_read_skip_step, 1);
      break;
    }
}

