"""Character table datasets of point groups."""

# Copyright (C) 2020 Atsushi Togo
# All rights reserved.
#
# This file is part of phonopy.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# * Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
# * Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
# * Neither the name of the phonopy project nor the names of its
#   contributors may be used to endorse or promote products derived
#   from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

# from Wikipedia http://en.wikipedia.org/wiki/List_of_character_tables_for_chemically_important_3D_point_groups  # noqa E501
character_table = {
    # C1 (1)
    "1": [
        {
            "rotation_list": ("E"),
            "character_table": {"A": (1)},
            "mapping_table": {"E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),)},
        }
    ],
    # Ci (2)
    "-1": [
        {
            "rotation_list": ("E", "i"),
            "character_table": {"Ag": (1, 1), "Au": (1, -1)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
            },
        }
    ],
    # C2 (3)
    "2": [
        {
            "rotation_list": ("E", "C2"),
            "character_table": {"A": (1, 1), "B": (1, -1)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, 1, 0), (0, 0, -1)),),
            },
        }
    ],
    # C1v = Cs = C1h (4)
    "m": [
        {
            "rotation_list": ("E", "sgh"),
            "character_table": {"A'": (1, 1), "A''": (1, -1)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "sgh": (((1, 0, 0), (0, -1, 0), (0, 0, 1)),),
            },
        }
    ],
    # C2h (5)
    "2/m": [
        {
            "rotation_list": ("E", "C2", "i", "sgh"),
            "character_table": {
                "Ag": (1, 1, 1, 1),
                "Bg": (1, -1, 1, -1),
                "Au": (1, 1, -1, -1),
                "Bu": (1, -1, -1, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "sgh": (((1, 0, 0), (0, -1, 0), (0, 0, 1)),),
            },
        }
    ],
    # D2 (6)
    "222": [
        {
            "rotation_list": ("E", "C2", "C2x", "C2y"),
            "character_table": {
                "A": (1, 1, 1, 1),
                "B1": (1, 1, -1, -1),
                "B2": (1, -1, -1, 1),
                "B3": (1, -1, 1, -1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2y": (((-1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "C2x": (((1, 0, 0), (0, -1, 0), (0, 0, -1)),),
            },
        }
    ],
    # C2v (7)
    "mm2": [
        {
            "rotation_list": ("E", "C2", "sgvxz", "sgvyz"),
            "character_table": {
                "A1": (1, 1, 1, 1),
                "A2": (1, 1, -1, -1),
                "B1": (1, -1, 1, -1),
                "B2": (1, -1, -1, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "sgvxz": (((1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "sgvyz": (((-1, 0, 0), (0, 1, 0), (0, 0, 1)),),
            },
        }
    ],
    # D2h (8)
    "mmm": [
        {
            "rotation_list": ("E", "C2", "C2x", "C2y", "i", "sgxy", "sgxz", "sgyz"),
            "character_table": {
                "Ag": (1, 1, 1, 1, 1, 1, 1, 1),
                "B1g": (1, 1, -1, -1, 1, 1, -1, -1),
                "B2g": (1, -1, -1, 1, 1, -1, 1, -1),
                "B3g": (1, -1, 1, -1, 1, -1, -1, 1),
                "Au": (1, 1, 1, 1, -1, -1, -1, -1),
                "B1u": (1, 1, -1, -1, -1, -1, 1, 1),
                "B2u": (1, -1, -1, 1, -1, 1, -1, 1),
                "B3u": (1, -1, 1, -1, -1, 1, 1, -1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2y": (((-1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "C2x": (((1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "sgxy": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "sgxz": (((1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "sgyz": (((-1, 0, 0), (0, 1, 0), (0, 0, 1)),),
            },
        }
    ],
    # C4 (9)
    "4": [
        {
            "rotation_list": ("E", "C4", "C2"),
            "character_table": {"A": (1, 1, 1), "B": (1, -1, 1), "E": (2, 0, -2)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
            },
        }
    ],
    # S4 (10)
    "-4": [
        {
            "rotation_list": ("E", "S4", "C2"),
            "character_table": {"A": (1, 1, 1), "B": (1, -1, 1), "E": (2, 0, -2)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # C4h (11)
    "4/m": [
        {
            "rotation_list": ("E", "C4", "C2", "i", "S4", "sgh"),
            "character_table": {
                "Ag": (1, 1, 1, 1, 1, 1),
                "Bg": (1, -1, 1, 1, -1, 1),
                "Eg": (2, 0, -2, 2, 0, -2),
                "Au": (1, 1, 1, -1, -1, -1),
                "Bu": (1, -1, 1, -1, 1, -1),
                "Eu": (2, 0, -2, -2, 0, 2),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S4": (
                    (
                        (0, 1, 0),
                        (-1, 0, 0),
                        (0, 0, -1),
                    ),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
            },
        }
    ],
    # D4 (12)
    "422": [
        {
            "rotation_list": ("E", "C4", "C2", "C2'", "C2''"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, 1, 1, -1, -1),
                "B1": (1, -1, 1, 1, -1),
                "B2": (1, -1, 1, -1, 1),
                "E": (2, 0, -2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2'": (
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                ),
                "C2''": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # C4v (13)
    "4mm": [
        {
            "rotation_list": ("E", "C4", "C2", "sgv", "sgd"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, 1, 1, -1, -1),
                "B1": (1, -1, 1, 1, -1),
                "B2": (1, -1, 1, -1, 1),
                "E": (2, 0, -2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "sgv": (
                    ((-1, 0, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, 1)),
                ),
                "sgd": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # D2d (14)
    "-42m": [
        {
            "rotation_list": ("E", "S4", "C2z", "C2'", "sgd"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, 1, 1, -1, -1),
                "B1": (1, -1, 1, 1, -1),
                "B2": (1, -1, 1, -1, 1),
                "E": (2, 0, -2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "C2z": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2'": (
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                ),
                "sgd": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                ),
            },
        },
        {
            "rotation_list": ("E", "S4", "C2z", "C2''", "sgv"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, 1, 1, -1, -1),
                "B1": (1, -1, 1, 1, -1),
                "B2": (1, -1, 1, -1, 1),
                "E": (2, 0, -2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "C2z": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2''": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                ),
                "sgv": (
                    ((-1, 0, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, 1)),
                ),
            },
        },
    ],
    #  D4h (15)
    "4/mmm": [
        {
            "rotation_list": (
                "E",
                "C4",
                "C2",
                "C2'",
                "C2''",
                "i",
                "S4",
                "sgh",
                "sgv",
                "sgd",
            ),
            "character_table": {
                "A1g": (1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
                "A2g": (1, 1, 1, -1, -1, 1, 1, 1, -1, -1),
                "B1g": (1, -1, 1, 1, -1, 1, -1, 1, 1, -1),
                "B2g": (1, -1, 1, -1, 1, 1, -1, 1, -1, 1),
                "Eg": (2, 0, -2, 0, 0, 2, 0, -2, 0, 0),
                "A1u": (1, 1, 1, 1, 1, -1, -1, -1, -1, -1),
                "A2u": (1, 1, 1, -1, -1, -1, -1, -1, 1, 1),
                "B1u": (1, -1, 1, 1, -1, -1, 1, -1, -1, 1),
                "B2u": (1, -1, 1, -1, 1, -1, 1, -1, 1, -1),
                "Eu": (2, 0, -2, 0, 0, -2, 0, 2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2'": (
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                ),
                "C2''": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "sgv": (
                    ((-1, 0, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, 1)),
                ),
                "sgd": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # C3 (16)
    "3": [
        {
            "rotation_list": ("E", "C3"),
            "character_table": {"A": (1, 1), "E": (2, -1)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # S6 (17)
    "-3": [
        {
            "rotation_list": ("E", "C3", "i", "S6"),
            "character_table": {
                "Ag": (1, 1, 1, 1),
                "Eg": (2, -1, 2, -1),
                "Au": (1, 1, -1, -1),
                "Eu": (2, -1, -2, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S6": (
                    ((0, 1, 0), (-1, 1, 0), (0, 0, -1)),
                    ((1, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # D3 (18)
    "32": [
        {
            "rotation_list": ("E", "C3", "C2'"),
            "character_table": {"A1": (1, 1, 1), "A2": (1, 1, -1), "E": (2, -1, 0)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2'": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # C3v (19)
    "3m": [
        {
            "rotation_list": ("E", "C3", "sgv"),
            "character_table": {"A1": (1, 1, 1), "A2": (1, 1, -1), "E": (2, -1, 0)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "sgv": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, 1)),
                ),
            },
        },
        {
            "rotation_list": ("E", "C3", "sgv"),
            "character_table": {"A1": (1, 1, 1), "A2": (1, 1, -1), "E": (2, -1, 0)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "sgv": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, 1)),
                ),
            },
        },
    ],
    # D3d (20)
    "-3m": [
        {
            "rotation_list": ("E", "C3", "C2'", "i", "S6", "sgd"),
            "character_table": {
                "A1g": (1, 1, 1, 1, 1, 1),
                "A2g": (1, 1, -1, 1, 1, -1),
                "Eg": (2, -1, 0, 2, -1, 0),
                "A1u": (1, 1, 1, -1, -1, -1),
                "A2u": (1, 1, -1, -1, -1, 1),
                "Eu": (2, -1, 0, -2, 1, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2'": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S6": (
                    ((0, 1, 0), (-1, 1, 0), (0, 0, -1)),
                    ((1, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgd": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, 1)),
                ),
            },
        },
        {
            "rotation_list": ("E", "C3", "C2'", "i", "S6", "sgd"),
            "character_table": {
                "A1g": (1, 1, 1, 1, 1, 1),
                "A2g": (1, 1, -1, 1, 1, -1),
                "Eg": (2, -1, 0, 2, -1, 0),
                "A1u": (1, 1, 1, -1, -1, -1),
                "A2u": (1, 1, -1, -1, -1, 1),
                "Eu": (2, -1, 0, -2, 1, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2'": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S6": (
                    ((0, 1, 0), (-1, 1, 0), (0, 0, -1)),
                    ((1, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgd": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, 1)),
                ),
            },
        },
    ],
    # C6 (21)
    "6": [
        {
            "rotation_list": ("E", "C6", "C3", "C2"),
            "character_table": {
                "A": (1, 1, 1, 1),
                "B": (1, -1, 1, -1),
                "E1": (2, 1, -1, -2),
                "E2": (2, -1, -1, 2),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C6": (
                    ((1, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
            },
        }
    ],
    # C3h (22)
    "-6": [
        {
            "rotation_list": ("E", "C3", "sgh", "S3"),
            "character_table": {
                "A'": (1, 1, 1, 1),
                "E'": (2, -1, 2, -1),
                "A''": (1, 1, -1, -1),
                "E''": (2, -1, -2, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "S3": (
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, -1, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # C6h (23)
    "6/m": [
        {
            "rotation_list": ("E", "C6", "C3", "C2", "i", "S3", "S6", "sgh"),
            "character_table": {
                "Ag": (1, 1, 1, 1, 1, 1, 1, 1),
                "Bg": (1, -1, 1, -1, 1, -1, 1, -1),
                "E1g": (2, 1, -1, -2, 2, 1, -1, -2),
                "E2g": (2, -1, -1, 2, 2, -1, -1, 2),
                "Au": (1, 1, 1, 1, -1, -1, -1, -1),
                "Bu": (1, -1, 1, -1, -1, 1, -1, 1),
                "E1u": (2, 1, -1, -2, -2, -1, 1, 2),
                "E2u": (2, -1, -1, 2, -2, 1, 1, -2),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C6": (
                    ((1, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S3": (
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, -1, 0), (0, 0, -1)),
                ),
                "S6": (
                    ((0, 1, 0), (-1, 1, 0), (0, 0, -1)),
                    ((1, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
            },
        }
    ],
    # D6 (24)
    "622": [
        {
            "rotation_list": ("E", "C6", "C3", "C2", "C2'", "C2''"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1, 1),
                "A2": (1, 1, 1, 1, -1, -1),
                "B1": (1, -1, 1, -1, 1, -1),
                "B2": (1, -1, 1, -1, -1, 1),
                "E1": (2, 1, -1, -2, 0, 0),
                "E2": (2, -1, -1, 2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C6": (
                    ((1, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2'": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, -1)),
                ),
                "C2''": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # C6v (25)
    "6mm": [
        {
            "rotation_list": ("E", "C6", "C3", "C2", "sgv", "sgd"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1, 1),
                "A2": (1, 1, 1, 1, -1, -1),
                "B1": (1, -1, 1, -1, 1, -1),
                "B2": (1, -1, 1, -1, -1, 1),
                "E1": (2, 1, -1, -2, 0, 0),
                "E2": (2, -1, -1, 2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C6": (
                    ((1, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "sgv": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, 1)),
                ),
                "sgd": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # D3h (26)
    "-6m2": [
        {
            "rotation_list": ("E", "C3", "C'2", "sgh", "S3", "sgv"),
            "character_table": {
                "A1'": (1, 1, 1, 1, 1, 1),
                "A2'": (1, 1, -1, 1, 1, -1),
                "E'": (2, -1, 0, 2, -1, 0),
                "A1''": (1, 1, 1, -1, -1, -1),
                "A2''": (1, 1, -1, -1, -1, 1),
                "E''": (2, -1, 0, -2, 1, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C'2": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, -1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, -1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "S3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, -1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, -1)),
                ),
                "sgv": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # D6h (27)
    "6/mmm": [
        {
            "rotation_list": (
                "E",
                "C6",
                "C3",
                "C2",
                "C2'",
                "C2''",
                "i",
                "S3",
                "S6",
                "sgh",
                "sgd",
                "sgv",
            ),
            "character_table": {
                "A1g": (1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
                "A2g": (1, 1, 1, 1, -1, -1, 1, 1, 1, 1, -1, -1),
                "B1g": (1, -1, 1, -1, 1, -1, 1, -1, 1, -1, 1, -1),
                "B2g": (1, -1, 1, -1, -1, 1, 1, -1, 1, -1, -1, 1),
                "E1g": (2, 1, -1, -2, 0, 0, 2, 1, -1, -2, 0, 0),
                "E2g": (2, -1, -1, 2, 0, 0, 2, -1, -1, 2, 0, 0),
                "A1u": (1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1),
                "A2u": (1, 1, 1, 1, -1, -1, -1, -1, -1, -1, 1, 1),
                "B1u": (1, -1, 1, -1, 1, -1, -1, 1, -1, 1, -1, 1),
                "B2u": (1, -1, 1, -1, -1, 1, -1, 1, -1, 1, 1, -1),
                "E1u": (2, 1, -1, -2, 0, 0, -2, -1, 1, 2, 0, 0),
                "E2u": (2, -1, -1, 2, 0, 0, -2, 1, 1, -2, 0, 0),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C6": (
                    ((1, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "C3": (
                    ((0, -1, 0), (1, -1, 0), (0, 0, 1)),
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, 1)),
                ),
                "C2": (((-1, 0, 0), (0, -1, 0), (0, 0, 1)),),
                "C2'": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, -1)),
                ),
                "C2''": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S3": (
                    ((-1, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, -1, 0), (0, 0, -1)),
                ),
                "S6": (
                    ((0, 1, 0), (-1, 1, 0), (0, 0, -1)),
                    ((1, -1, 0), (1, 0, 0), (0, 0, -1)),
                ),
                "sgh": (((1, 0, 0), (0, 1, 0), (0, 0, -1)),),
                "sgv": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, -1, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (-1, 1, 0), (0, 0, 1)),
                ),
                "sgd": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((-1, 1, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (1, -1, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # T (28)
    "23": [
        {
            "rotation_list": ("E", "C3", "C2"),
            "character_table": {"A": (1, 1, 1), "E": (2, -1, 2), "T": (3, 0, -1)},
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, 0, 1), (1, 0, 0), (0, 1, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, -1, 0)),
                    ((0, 0, -1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, -1, 0)),
                    ((0, 1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, -1, 0), (0, 0, 1), (-1, 0, 0)),
                    ((0, 1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, -1, 0), (0, 0, -1), (1, 0, 0)),
                ),
                "C2": (
                    ((-1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                ),
            },
        }
    ],
    # Th (29)
    "m-3": [
        {
            "rotation_list": ("E", "C3", "C2", "i", "S6", "sgh"),
            "character_table": {
                "Ag": (1, 1, 1, 1, 1, 1),
                "Au": (1, 1, 1, -1, -1, -1),
                "Eg": (2, -1, 2, 2, -1, 2),
                "Eu": (2, -1, 2, -2, 1, -2),
                "Tg": (3, 0, -1, 3, 0, -1),
                "Tu": (3, 0, -1, -3, 0, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, 0, 1), (1, 0, 0), (0, 1, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, -1, 0)),
                    ((0, 0, -1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, -1, 0)),
                    ((0, 1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, -1, 0), (0, 0, 1), (-1, 0, 0)),
                    ((0, 1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, -1, 0), (0, 0, -1), (1, 0, 0)),
                ),
                "C2": (
                    ((-1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S6": (
                    ((0, 0, -1), (-1, 0, 0), (0, -1, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, 1, 0)),
                    ((0, 0, 1), (1, 0, 0), (0, -1, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, 1, 0)),
                    ((0, -1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, 1, 0), (0, 0, -1), (1, 0, 0)),
                    ((0, -1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, 1, 0), (0, 0, 1), (-1, 0, 0)),
                ),
                "sgh": (
                    ((1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, 1)),
                ),
            },
        }
    ],
    # O (30)
    "432": [
        {
            "rotation_list": ("E", "C4", "C2", "C3", "C2'"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, -1, 1, 1, -1),
                "E": (2, 0, 2, -1, 0),
                "T1": (3, 1, -1, 0, -1),
                "T2": (3, -1, -1, 0, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, 1), (0, -1, 0)),
                    ((1, 0, 0), (0, 0, -1), (0, 1, 0)),
                    ((0, 0, 1), (0, 1, 0), (-1, 0, 0)),
                    ((0, 0, -1), (0, 1, 0), (1, 0, 0)),
                ),
                "C2": (
                    ((-1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                ),
                "C3": (
                    ((0, 0, 1), (1, 0, 0), (0, 1, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, -1, 0)),
                    ((0, 0, -1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, -1, 0)),
                    ((0, 1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, -1, 0), (0, 0, 1), (-1, 0, 0)),
                    ((0, 1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, -1, 0), (0, 0, -1), (1, 0, 0)),
                ),
                "C2'": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 0, 1), (0, 1, 0)),
                    ((-1, 0, 0), (0, 0, -1), (0, -1, 0)),
                    ((0, 0, 1), (0, -1, 0), (1, 0, 0)),
                    ((0, 0, -1), (0, -1, 0), (-1, 0, 0)),
                ),
            },
        }
    ],
    # Td (31)
    "-43m": [
        {
            "rotation_list": ("E", "C3", "C2", "S4", "sgd"),
            "character_table": {
                "A1": (1, 1, 1, 1, 1),
                "A2": (1, 1, 1, -1, -1),
                "E": (2, -1, 2, 0, 0),
                "T1": (3, 0, -1, 1, -1),
                "T2": (3, 0, -1, -1, 1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, 0, 1), (1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, -1), (1, 0, 0)),
                    ((0, 0, -1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, 1), (-1, 0, 0)),
                ),
                "C2": (
                    ((-1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                ),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 0, 1), (0, -1, 0)),
                    ((-1, 0, 0), (0, 0, -1), (0, 1, 0)),
                    ((0, 0, -1), (0, -1, 0), (1, 0, 0)),
                    ((0, 0, 1), (0, -1, 0), (-1, 0, 0)),
                ),
                "sgd": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, -1), (0, -1, 0)),
                    ((0, 0, -1), (0, 1, 0), (-1, 0, 0)),
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, 1), (0, 1, 0)),
                    ((0, 0, 1), (0, 1, 0), (1, 0, 0)),
                ),
            },
        }
    ],
    # Oh (32)
    "m-3m": [
        {
            "rotation_list": (
                "E",
                "C3",
                "C2",
                "C4",
                "C4^2",
                "i",
                "S4",
                "S6",
                "sgh",
                "sgd",
            ),
            "character_table": {
                "A1g": (1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
                "A2g": (1, 1, -1, -1, 1, 1, -1, 1, 1, -1),
                "Eg": (2, -1, 0, 0, 2, 2, 0, -1, 2, 0),
                "T1g": (3, 0, -1, 1, -1, 3, 1, 0, -1, -1),
                "T2g": (3, 0, 1, -1, -1, 3, -1, 0, -1, 1),
                "A1u": (1, 1, 1, 1, 1, -1, -1, -1, -1, -1),
                "A2u": (1, 1, -1, -1, 1, -1, 1, -1, -1, 1),
                "Eu": (2, -1, 0, 0, 2, -2, 0, 1, -2, 0),
                "T1u": (3, 0, -1, 1, -1, -3, -1, 0, 1, 1),
                "T2u": (3, 0, 1, -1, -1, -3, 1, 0, 1, -1),
            },
            "mapping_table": {
                "E": (((1, 0, 0), (0, 1, 0), (0, 0, 1)),),
                "C3": (
                    ((0, 0, 1), (1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, -1), (1, 0, 0)),
                    ((0, 0, -1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, 1), (-1, 0, 0)),
                ),
                "C2": (
                    ((0, 1, 0), (1, 0, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 0, 1), (0, 1, 0)),
                    ((0, 0, 1), (0, -1, 0), (1, 0, 0)),
                    ((0, -1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 0, -1), (0, -1, 0)),
                    ((0, 0, -1), (0, -1, 0), (-1, 0, 0)),
                ),
                "C4": (
                    ((0, -1, 0), (1, 0, 0), (0, 0, 1)),
                    ((0, 1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, -1), (0, 1, 0)),
                    ((1, 0, 0), (0, 0, 1), (0, -1, 0)),
                    ((0, 0, 1), (0, 1, 0), (-1, 0, 0)),
                    ((0, 0, -1), (0, 1, 0), (1, 0, 0)),
                ),
                "C4^2": (
                    ((-1, 0, 0), (0, -1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, -1)),
                ),
                "i": (((-1, 0, 0), (0, -1, 0), (0, 0, -1)),),
                "S4": (
                    ((0, 1, 0), (-1, 0, 0), (0, 0, -1)),
                    ((0, -1, 0), (1, 0, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 0, 1), (0, -1, 0)),
                    ((-1, 0, 0), (0, 0, -1), (0, 1, 0)),
                    ((0, 0, -1), (0, -1, 0), (1, 0, 0)),
                    ((0, 0, 1), (0, -1, 0), (-1, 0, 0)),
                ),
                "S6": (
                    ((0, 0, -1), (-1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, -1), (-1, 0, 0)),
                    ((0, 0, 1), (-1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, 1), (-1, 0, 0)),
                    ((0, 0, 1), (1, 0, 0), (0, -1, 0)),
                    ((0, -1, 0), (0, 0, 1), (1, 0, 0)),
                    ((0, 0, -1), (1, 0, 0), (0, 1, 0)),
                    ((0, 1, 0), (0, 0, -1), (1, 0, 0)),
                ),
                "sgh": (
                    ((1, 0, 0), (0, 1, 0), (0, 0, -1)),
                    ((-1, 0, 0), (0, 1, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, -1, 0), (0, 0, 1)),
                ),
                "sgd": (
                    ((0, -1, 0), (-1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, -1), (0, -1, 0)),
                    ((0, 0, -1), (0, 1, 0), (-1, 0, 0)),
                    ((0, 1, 0), (1, 0, 0), (0, 0, 1)),
                    ((1, 0, 0), (0, 0, 1), (0, 1, 0)),
                    ((0, 0, 1), (0, 1, 0), (1, 0, 0)),
                ),
            },
        }
    ],
}
