/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "../nb_global.h"
#include "nb_ui_interface_callbacks.h"
#include "nb_ui_interface.h"

#define _(text) gettext(text)

void
nb_ui_create_main_window (void)
{
	extern GladeXML * xml;
	GtkWidget * main_window;

	/* load the interface */
	xml = glade_xml_new (g_build_path ("/", GLADE_DIR, "main.glade", NULL), NULL, NULL);

	/* connect the signals in the interface */
	glade_xml_signal_autoconnect(xml);

	nb_ui_treeview_create ();
	nb_ui_note_view_create ();

	/* resize the window and vpan to last values  */
	/* FIXME: resize is buggy 
	 * nb_ui_load_interface_positions (); */

	/* set drag and drap on main_window */
	main_window = glade_xml_get_widget (xml, "mainwindow");
	gtk_drag_dest_set (main_window, GTK_DEST_DEFAULT_ALL, drag_drop_target, 2, GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);
	g_signal_connect (main_window, "drag_data_received", G_CALLBACK (nb_ui_cb_drag_received), NULL);

	/* set the icon */
	gtk_window_set_icon ( GTK_WINDOW (main_window),
			     gdk_pixbuf_new_from_file (g_build_path ("/", PIXMAPS_DIR, "notebook.svg", NULL), NULL));
}

/* dialogs */
gboolean
nb_ui_dialog_question (gchar * text, gchar * title)
{
	GtkWidget * dialog;
	int button;

	dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, text);
	g_object_set (dialog, "title", title, "allow_grow", TRUE, NULL);

	button = gtk_dialog_run ((GtkDialog *) dialog);
	gtk_widget_destroy (dialog);

	if (button == GTK_RESPONSE_NO)
		return FALSE;
	else
		return TRUE;
}

void
nb_ui_dialog_error (gchar * text)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, text);

	gtk_dialog_run ((GtkDialog *) dialog);
	gtk_widget_destroy (dialog);

	return;
}

void
nb_ui_dialog_import ()
{
	extern Global * g;
	extern GladeXML * xml;
	GtkWidget * dialog, * mainwindow;

	mainwindow = glade_xml_get_widget (xml, "mainwindow");

	dialog = gtk_file_chooser_dialog_new ((gchar *) _("Select file to import."), GTK_WINDOW (mainwindow),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					      NULL);

	if ( gtk_dialog_run ( GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT )
	{
		gchar * filename;

		filename = gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER (dialog));
		if ( filename != NULL )
			nb_global_import_note_list (g, filename);

		g_free (filename);
	}

	gtk_widget_destroy (dialog);
}

void
nb_ui_dialog_export ()
{
	extern GladeXML * xmlexport;

	/* create the export dialog */
	xmlexport = glade_xml_new (g_build_path ("/", GLADE_DIR, "exportdialog.glade", NULL), NULL, NULL);
	glade_xml_signal_autoconnect(xmlexport);
}

void
nb_ui_about_dialog_create (void)
{
	const gchar * authors[] = { "Mathias André <mathias@openbrookes.org>", NULL};
	const gchar * version = "0.2";
	const gchar * comment = (gchar *) _("Notebook is a simple logbook editor.");
	const gchar * translator = (gchar *) _("Mathias Andre <mathias@openbrookes.org>");
	GdkPixbuf * pixbuf;
	
	pixbuf = gdk_pixbuf_new_from_file_at_size (g_build_path ("/", PIXMAPS_DIR, "notebook.svg", NULL),
						   200, 200, NULL);

	gtk_widget_show ( GTK_WIDGET (gnome_about_new (PACKAGE, VERSION,
					"Copyright \xc2\xa9 2004 Mathias André",
					comment,
					(const gchar **) authors,
					NULL,
					translator,
					pixbuf)));
}

/* store/load interface settings */
void
nb_ui_store_interface_positions (void)
{
	extern GladeXML * xml;
	GConfClient * client;
	GtkWidget * mainwindow = glade_xml_get_widget (xml, "mainwindow");
	GtkWidget * vpan = glade_xml_get_widget (xml, "vpaned1");
	gint width, height, position;

	position = gtk_paned_get_position ( GTK_PANED (vpan));
	gdk_drawable_get_size ( GDK_DRAWABLE (mainwindow->window), &width, &height);

	client = gconf_client_get_default ();

	gconf_client_set_int (client, "/apps/notebook/ui/width", width, NULL);
	gconf_client_set_int (client, "/apps/notebook/ui/height", height, NULL);
	gconf_client_set_int (client, "/apps/notebook/ui/position", position, NULL);
}

void
nb_ui_load_interface_positions (void)
{
	extern GladeXML * xml;
	GConfClient * client;
	GtkWidget * mainwindow = glade_xml_get_widget (xml, "mainwindow");
	GtkWidget * vpan = glade_xml_get_widget (xml, "vpaned1");
	gint width = 0, height = 0, position = 0;

	client = gconf_client_get_default ();

	width = gconf_client_get_int (client, "/apps/notebook/ui/width", NULL);
	height = gconf_client_get_int (client, "/apps/notebook/ui/height", NULL);
	position = gconf_client_get_int (client, "/apps/notebook/ui/position", NULL);

	if ( (width != 0) && (height != 0) )
		gtk_window_resize ( GTK_WINDOW (mainwindow), width, height);

	if ( position != 0 )
		gtk_paned_set_position ( GTK_PANED (vpan), position);
}

/* drag and drop callbacks */
void
nb_ui_cb_drag_received (GtkWidget * wgt, GdkDragContext * context, int x, int y, GtkSelectionData * seldata, guint info, guint time, gpointer userdata)
{
	extern Global * g;
	gchar ** split, ** copy ;
	GSList * attachments = NULL;

	if ( seldata == NULL )
		return ;
	if ( seldata->length < 0 )
		return ;

	split = g_strsplit ((gchar *) seldata->data, "\n", 0);
	
	for ( copy = split ; *copy != NULL ; copy++ )
	{
		if ( strlen (*copy) > 0 )
			attachments = g_slist_append (attachments, g_strchomp (*copy));
	}

	nb_global_add_attachments (g, attachments);

	gtk_drag_finish (context, TRUE, FALSE, time);
	g_signal_stop_emission_by_name (wgt, "drag_data_received");
}

