// Copyright Mirage authors & contributors <https://github.com/mirukana/mirage>
// SPDX-License-Identifier: LGPL-3.0-or-later

import QtQuick 2.12
import QtQuick.Controls 2.12
import QtQuick.Layouts 1.12
import QtQuick.Window 2.12
import QtGraphicalEffects 1.12
import "."
import "Base"
import "MainPane"
import "Popups"

Item {
    id: mainUI

    enum NotificationLevel { Mute, HighlightsOnly, Enable }

    property int notificationLevel:
        settings.Notifications.start_level === "highlights_only" ?
        UI.NotificationLevel.HighlightsOnly :
        settings.Notifications.start_level === "mute" ?
        UI.NotificationLevel.Mute :
        UI.NotificationLevel.Enable

    property bool accountsPresent:
        ModelStore.get("accounts").count > 0 || py.startupAnyAccountsSaved

    readonly property var accountIds: {
        const ids   = []
        const model = ModelStore.get("accounts")

        for (let i = 0; i < model.count; i++)
            ids.push(model.get(i).id)

        return ids
    }

    readonly property alias debugConsole: debugConsole
    readonly property alias mainPane: mainPane
    readonly property alias pageLoader: pageLoader
    readonly property alias unexpectedErrorPopup: unexpectedErrorPopup
    readonly property alias fontMetrics: fontMetrics
    readonly property alias idleManager: idleManager

    focus: true
    Component.onCompleted: window.mainUI = mainUI

    HShortcut {
        sequences: window.settings.Keys.python_debugger
        onActivated: py.call("BRIDGE.pdb")
    }

    HShortcut {
        sequences: window.settings.Keys.python_remote_debugger
        onActivated: py.call("BRIDGE.pdb", [[], true])
    }

    HShortcut {
        sequences: window.settings.Keys.zoom_in
        onActivated: {
            window.settings.General.zoom += 0.1
            window.saveSettings()
        }
    }

    HShortcut {
        sequences: window.settings.Keys.zoom_out
        onActivated: {
            window.settings.General.zoom =
                Math.max(0.1, window.settings.General.zoom - 0.1)

            window.saveSettings()
        }
    }

    HShortcut {
        sequences: window.settings.Keys.reset_zoom
        onActivated: {
            window.settings.General.zoom = 1
            window.saveSettings()
        }
    }

    HShortcut {
        sequences: window.settings.Keys.compact
        onActivated: {
            window.settings.General.compact = ! window.settings.General.compact
            window.saveSettings()
        }
    }

    HShortcut {
        sequences: window.settings.Keys.quit
        onActivated: Qt.quit()
    }

    HShortcut {
        sequences: window.settings.Keys.notifications_highlights_only
        onActivated:
            mainUI.notificationLevel =
                mainUI.notificationLevel ===
                UI.NotificationLevel.HighlightsOnly ?
                UI.NotificationLevel.Enable :
                UI.NotificationLevel.HighlightsOnly
    }

    HShortcut {
        sequences: window.settings.Keys.notifications_mute
        onActivated:
            mainUI.notificationLevel =
                mainUI.notificationLevel === UI.NotificationLevel.Mute ?
                UI.NotificationLevel.Enable :
                UI.NotificationLevel.Mute
    }

    FontMetrics {
        id: fontMetrics
        font.family: theme.fontFamily.sans
        font.pixelSize: theme.fontSize.normal
        font.pointSize: -1
    }

    IdleManager {
        id: idleManager
    }

    DebugConsole {
        id: debugConsole
        target: mainUI
        visible: false
    }

    LinearGradient {
        id: mainUIGradient
        visible: ! image.visible
        anchors.fill: parent
        start: theme.ui.gradientStart
        end: theme.ui.gradientEnd

        gradient: Gradient {
            GradientStop { position: 0.0; color: theme.ui.gradientStartColor }
            GradientStop { position: 1.0; color: theme.ui.gradientEndColor }
        }
    }

    HImage {
        id: image
        visible: Boolean(Qt.resolvedUrl(source))
        fillMode: Image.PreserveAspectCrop
        animatedFillMode: AnimatedImage.PreserveAspectCrop
        source: theme.ui.image
        sourceSize.width: Screen.width
        sourceSize.height: Screen.height
        anchors.fill: parent
        asynchronous: false
    }

    MainPane {
        id: mainPane
        maximumSize: parent.width - theme.minimumSupportedWidth * 1.5

        // Drawers (side panes) are actually Popups, which are considered to be
        // different "layer". Input handlers will only get events occuring
        // in the layer they were declared in.
        GlobalTapHandlers { pageLoader: pageLoader }
    }

    PageLoader {
        id: pageLoader
        anchors.fill: parent
        anchors.leftMargin:
            mainPane.requireDefaultSize &&
            mainPane.minimumSize > mainPane.maximumSize ?
            mainPane.calculatedSizeNoRequiredMinimum :
            mainPane.visibleSize

        visible: mainPane.visibleSize < mainUI.width

        GlobalTapHandlers { pageLoader: parent }
    }

    UnexpectedErrorPopup {
        id: unexpectedErrorPopup
    }
}
