/*
 * @(#)SPISingletonStore.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.classes.v1;

import java.io.IOException;

import org.apache.log4j.Logger;



/**
 * Aids pluggable factories and related classes by being a central repository
 * for storing SPI singletons, and creating means to load and change the
 * singletons.
 * <P>
 * This uses the same pattern as the <tt>SingletonStore</tt> class, but this
 * one allows for multiple singleton instances, and loads them from the
 * SPILoader class.
 *
 * @author  Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version $Date: 2003/02/10 22:52:36 $
 * @since   June 28, 2002
 * @see     SingletonStore
 */
public class SPISingletonStore extends AbstractMultipleStore
{
    private static final Logger LOG = Logger.getLogger(
        SPISingletonStore.class.getName() );
    
    private Class baseClass;
    
    
    /**
     * Constructor specifying all the parameters for using a singleton in this
     * framework.
     *
     * @param instanceOf singletons must be of this class.  This is also used
     *      to load the SPI classes.
     * @param ama <tt>true</tt> if this store should allow
     *      multiple instances of the exact same class, or <tt>false</tt>
     *      if it should prevent multiple instances sharing the exact
     *      same class.  This helps to enforce the idea of 'singleton'.
     */
    public SPISingletonStore( Class instanceOf, AllowMultiplesAction ama )
    {
        super( instanceOf, ama );
        
        if (instanceOf == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        
        this.baseClass = instanceOf;
    }
    
    
    /**
     * Adds the default inner singletons, which is an implementation
     * specific method.
     */
    protected void addDefaultSingletons()
    {
        try
        {
            addSPI( null );
        }
        catch (IOException ioe)
        {
            LOG.warn( "adding SPI caused IOException", ioe );
            throw new IllegalStateException( ioe.toString() );
        }
    }
    
    
    /**
     * Add a set of SPIs from the given class loader.
     */
    public void addSPI( ClassLoader cl )
            throws IOException
    {
        SPILoader spil = new SPILoader( this.baseClass, cl );
        
        while (spil.hasNext())
        {
            addSingleton( spil.nextProvier() );
        }
    }
}

