<?PHP
# Copyright 2004, Revolution Linux Inc., Benoit des Ligneris
#
# This file is part of the MILLE-XTERM distribution.
# See the MILLE-XTERM (english) and/or the MILLE (french) project web site
#
#   http://www.revolutionlinux.com/mille-xterm/
#   http://www.mille.ca/
#
# The MILLE-XTERM framework is covered by the GNU General Public License.  See
# the COPYING file in the top-level MILLE-XTERM directory.  Software packages
# that are included in the MILLE-XTERM distribution have their own licenses.
#
# -------------------------------------------------------------------------

/**
*
* This script is called by the XTERMINAL to obtain it's configuration
*
* @package MILLE-CONFIGURATOR
* @author Benoit des Ligneris
*
* <code>
* code 1 ?ip=192.168.2.221/mac=00:90:F5:2A:ED:A7/bootservip=192.168.2.253/code=1
*      XTERM is booting and has its bootservip
* code 2 ?ip=192.168.2.221/appservip=192.168.2.254/display=0.0/code=2
*     XTERM has it appserver & display
* code 3 * ?ip=192.168.2.221/appservip=192.168.2.253/display=0.0/code=3/username=3
*      a user is connected
* code 4 * ?ip=192.168.2.221/appservip=192.168.2.253/display=0.0/code=4/username=3
*      a user is de connected
* </code>
*/

openlog("ltsp-directory", LOG_PID, LOG_USER);

# Ensure the Db is configured
if (!is_readable("util/config.php")) {
    $first_start = true;
    include "util/setup.php";
    die();
}
require_once 'util/functions.php';
require_once 'util/Node.php';
require_once 'util/EntityNodeFinder.php';
require_once 'util/HWGroupFinder.php';

# Security : only the expected variable are assigned
# No cookies 
unset($_COOKIES);


# Expected variables
$get["debug"]='';
$get["ip"]='';
$get["password"]='';
$get["username"]='';
$get["groups"]='';
$get["mac"]='';
$get["bootservip"]='';
$get["appservip"]='';
$get["display"]='';
$get["code"]='';
$get["hwlist"]='';
$get["publickey"]='';

# Get the allowed variables from _GET and _POST
foreach (array_merge($_GET,$_POST) as $key=>$value) {
    if (isset($get[$key])) {
        if (($key == "display") && ($value == 0)) {
            $get[$key]="0.0";
        }
        else {
            $get[$key]="$value";
        }
    }
}

# Delete the _GET and _POST variable
unset($_GET);
unset($_POST);

// JMD: fix in case we mount nfs via DNS
if (!isIP($get["bootservip"]) && $get["bootservip"]!='') {
    $get["bootservip"]=gethostbyname($get["bootservip"]);
}
// ND: same fix in case terminal ip is a dns
// And why not doing it for all! Adding appservip
if (!isIP($get["ip"]) && $get["ip"]!='') {
    $get["ip"]=gethostbyname($get["ip"]);
}
if (!isIP($get["appservip"]) && $get["appservip"]!='') {
    $get["appservip"]=gethostbyname($get["appservip"]);
}

// Make sure mac is uppercase
if ($get["mac"]!='') {
    $get["mac"] = strtoupper($get["mac"]);
}

# Clean the $get variable
foreach ($get as $key=>$value) {
    if (empty($value)) {
        unset($get[$key]);
    }
}

# Code 999 : Testing purpose
if ($get["code"] == 999) {
    print "ALIVE";
    die();
}

# Retrieve current ip address
if ($CONFIG['terminal_auth'] == 'true') {
    $ip=getIP();
} else {
    $ip=$get["ip"];
}
# Code 0 : get config, logs nothing
if ($get["code"] == 0) {
    # Connection authentication
    $auth = $ip==$get["ip"];
    # Access node
    $node=new Node($get["mac"]);
    # Compute hw groups
    if ($node->hasHWgroup()) {
        $nodesIds = $node->getHWgroupIDs();
    }
    $get["hwgroups"]=$hwGroups;
    # Set status and return LTSP
    returnLtsp($node,$get,$error,$auth,$nodesIds);
}
# Code 1 : initialize with MAC
elseif ($get["code"] == 1) {
    # Connection authentication
    $auth = $ip==$get["ip"];
    # Access node
    $node=new Node($get["mac"]);
    # If it is not a node, let's create it !
    if ($node->notFound() && $auth) {
        $node=createNode($get);
    }
    # Save inventory
    if (isset($get["hwlist"])) {
        $hwList = decodeHwList($get["hwlist"]);
        $node->setHardware($hwList);
        unset($get["hwlist"]);
    }
    # Compute hw groups
    if ($node->hasHWgroup()) {
        $nodesIds = $node->getHWgroupIDs();
    }
    $get["hwgroups"]=$hwGroups;
    # Set status and return LTSP
    if ($auth) 
        $error=setStatus($node,$get);
    else
        $error = "Cannot set status, auth error, client ip: ".$ip." expected: ".$get["ip"];
    logAccess($get);
    returnLtsp($node,$get,$error,$auth,$nodesIds);
}
# Code 2 : update status with appserver ip
elseif ($get["code"] == 2) {
    # Connection authentication
    $auth = $ip==$get["ip"];
    # Access node
    $node=new Node($get["ip"]);
    $status=$node->getStatus();
    if ($node->hasHWgroup()) {
        $nodesIds = $node->getHWgroupIDs();
    }
    # Set status and return LTSP
    if ($auth) 
        $error=setStatus($node,$get);
    else
        $error = "Cannot set status, auth error, client ip: ".$ip." expected: ".$get["ip"];
    logAccess($get);
    returnLtsp($node,$get,$error,$auth,$nodesIds);
}
# Code 3 : register user and groups
elseif ($get["code"] == 3) {
    $node=new Node($get["ip"]);
    # Retrieve entity nodes
    $entityFinder = new EntityNodeFinder(strtolower($get["username"]),split(",", strtolower($get["groups"])));
    $nodesIds = $entityFinder->getNodesIds();
    # Add HW groups
    $status=$node->getStatus();
    if (!empty($status["hwgroups"])) {
        $nodesIds = array_merge($nodesIds,split(",",$status["hwgroups"]));
    }
    # Retrieve ltsp attributes
    $array=$node->getExtendedLTS($nodesIds);
    # Authentication required to set termaccess and public keys
    $auth = (!($CONFIG['terminal_auth']=='true') || $get["password"]==$CONFIG["terminal_password"]);
    $status["termaccess"] = 0;
    if ($auth && $array["IS_TEACHER"]=="Y") {
        $status["termaccess"] = 2;
    }
    # Set status and return LTSP
    unset($get["password"]);
    unset($get["groups"]);
    $error=setStatus($node,$get,$status);
    logAccess($get);
    returnLtsp($array,$get,$error,$auth,$nodesIds);
}
# Code 4 : Logout, remove security infos
elseif ($get["code"] == 4) {
    $node=new Node($get["ip"]);
    # Cleanup security parameters
    $get["termaccess"]=0;
    $get["publickey"]='';
    # Set status and return LTSP
    $error=setStatus($node,$get);
    logAccess($get);
    returnLtsp($node,$get,$error);
}
else {
    returnLtsp("Invalid code : ".$get["code"],$get);
    log_error("Invalid code : ".$get["code"],$get);
}

die();

// No more code below this point: functions 
/**
*
* Create a node if it does not exists
*
*/
function createNode($get) {
    # Try to insert the node until success
    $success=FALSE;
    $counter=0;
    while ((!$success) && ($counter < 10)) {
        @include("autocreate.php");
        if(!$parent)  $parent=new Node();
        if ($node=$parent->createChildComputer($get["mac"])) {
            $success=TRUE;
        }
        else {
            log_error("Try number $counter, mac=".$get["mac"].", node =\"$node\"",$parent->lastError());
        }
        $counter++;
        sleep(1);
    }
    if (!$success) {
        returnLtsp($counter.$parent->lastError(),$get);
        log_error($parent->lastError(),$get);
        die();
    }
    return $node;
}

/**
* True if is a IP, false otherwise.
*
* @return boolean
*/
function isIP($ip) {
    return ereg("^([0-2]{0,1}[0-9]{1,2}\.){3}[0-2]{0,1}[0-9]{1,2}$", $ip);
}

/**
*
* Update the status using the $get variable (including $get[code])
*
*/
function setStatus($node,$get,$status=NULL) {
    # Read status from the DB
    if (!isset($status)) {
        $status=$node->getStatus();
    }

    # Merge the arrays
    if (!isset($status)) {
        $status=$get;
    } else {
        $status=array_merge($status,$get);
    }   
    
    # Unset the debug column
    unset($status["debug"]);
    # Continue on setStatus error (for read only databases)
    # If error, add MILLE_CONFIG_ERROR to the conf file and return the default file
    if (!$node->setStatus($status)) {
        log_error($node->lastError(),$get);
        return $node->lastError();
    }
}

/**
*
* Decode the hwList parameter
*
*/
function decodeHwList($hardware) {
    $hwList = array();
    foreach (split("\n",$hardware) as $line) {
        if (!empty($line)) {
            $pos=strpos($line,"=");
            $hwkey = substr($line,0,$pos);
            if ($pos < strlen($line))
                $hwval = substr($line,$pos+1,strlen($line));
            else
                $hwval = '';
            $hwList[] = array('hwkey'=>$hwkey,'hwvalue'=>$hwval);
        }
    }
    return $hwList;
}

# Return the LTSP.conf file generated for this terminal
function returnLtsp($node,$get,$error=NULL,$auth=true,$groups=NULL) {

    if (is_array($node)) {
        $array=$node;
    }
    elseif (strtolower(get_class($node)) != "node" || $node->notFound()) {
        $error=$node;
        $node=new Node(0);
        $array=$node->getLTS();
    }
    elseif (isset($groups) && $groups!='') {
        $array=$node->getExtendedLTS($groups);
    }
    else {
        $array=$node->getLTS();
    }

    if (is_object($error)) {
        log_error($error->lastError(),$get);
        if (!$get["debug"]) {
            $array["MILLE_CONFIGURATOR_ERROR"]="yes";
        }
    }
    if (!$auth) {
        $array["MILLE_AUTH_ERROR"]="yes";
    }

    print "[Default]\n";
    foreach ($array as $key=>$value) {
        if ( $value == "%LOADBALANCER%" )
            $value=returnGetLBServer("default");
        elseif ( preg_match('/^%LOADBALANCER%(.*)%$/',$value,$group) )
            $value=returnGetLBServer($group[1]);
        print ("$key=\"".returnGetLtspVariable($value)."\"\n");
    }
    if ($node->id_parent > 0)
        print "CLUSTER_CONFIGURED=\"True\"\n";
    else
        print "CLUSTER_CONFIGURED=\"False\"\n";
    return $array;
}

# Get an IP address from the loadbalancer
function returnGetLBServer ($group) {
    global $CONFIG;
    $lbserver=fopen("http://".$CONFIG['loadbalancer'].":8008/?group=".$group,"r");
    $server=trim(fgets($lbserver));
    fclose($lbserver);
    return $server;
}

# Remove the initial "'" and final "'" produced by escapeshellarg
function returnGetLtspVariable($string) {
    $string=escapeshellarg($string);
    $result=substr($string,1);
    $result=substr($result,0,-1);
    return $result;
}

# Function to log access in the access log
function logAccess($get) {

    $line = ',';
    foreach ($get as $key=>$value) {
        $line .= "$key=$value,";
    }
    $line = substr($line, 0, -1);
    syslog(LOG_INFO,$line);
}

# Function to log error on a file
function log_error($message,$get) {

    if (is_array($get)) {
        $line = ',';
        foreach ($get as $key=>$value) {
            $line .= "$key=$value,";
        }
    }
    else {
        $line=",$get";
    }
    $line .= " - " . $message;
    syslog(LOG_ERR,$line);
}

function getIP() {
   if (getenv("HTTP_CLIENT_IP") && strcasecmp(getenv("HTTP_CLIENT_IP"), "unknown"))
           $ip = getenv("HTTP_CLIENT_IP");
       else if (getenv("HTTP_X_FORWARDED_FOR") && strcasecmp(getenv("HTTP_X_FORWARDED_FOR"), "unknown"))
           $ip = getenv("HTTP_X_FORWARDED_FOR");
       else if (getenv("REMOTE_ADDR") && strcasecmp(getenv("REMOTE_ADDR"), "unknown"))
           $ip = getenv("REMOTE_ADDR");
       else if (isset($_SERVER['REMOTE_ADDR']) && $_SERVER['REMOTE_ADDR'] && strcasecmp($_SERVER['REMOTE_ADDR'], "unknown"))
           $ip = $_SERVER['REMOTE_ADDR'];
       else
           $ip = "unknown";
   return($ip);
}

# Function for debugging on the screen
function debug($msg){
    $conf = array('error_prepend' => '<font color="#ff0000"><tt>',
                  'error_append'  => '</tt></font>');
    $debug = &Log::singleton('display', '', '', $conf, PEAR_LOG_DEBUG);
    $debug->log($msg);
}

?>
