package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Figure;
import latexDraw.figures.FramedBox;
import latexDraw.figures.Text;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.components.ButtonIcon;
import latexDraw.ui.components.ColorButton;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class allows to set the parameters of the text<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersTextFrame extends AbstractParametersFrame implements FocusListener, MouseListener
{
	private static final long serialVersionUID = 1L;

	/** This field allows the set the text */
	protected JTextField textField;
	
	/** This field allows the set the X-coordinate of the first character of the text */
	protected JSpinner xCoord;
	
	/** This field allows the set the Y-coordinate of the first character of the text */
	protected JSpinner yCoord;

	/** Allows to set the size of the text */
	protected JComboBox sizeChoice;
		
	/** Allows to choice the font */
	protected JComboBox fontChoice;
	
	/** Allows to set the text in italic */
	protected JToggleButton buttonItalic;
	
	/** Allows to set the text in bold */
	protected JToggleButton buttonBold;
	
	/** Allows to set if there is framed box or not. */
	protected JCheckBox framedBoxCB;
	
	/** Allows to set that there is only a simple framed box for the text. */
	protected JRadioButton simpleRB;
	
	/** Allows to set that there is only a several framed boxes for the text. */
	protected JRadioButton multipleRB;
	
	/** Allows to set that the box is a circle. */
	protected JRadioButton circleRB;
	
	/** Allows to set that the box is a rectangle. */
	protected JRadioButton rectangleRB;
	
	/** Allows to set that the box is an ellipse. */
	protected JRadioButton ellipseRB;
	
	/** Allows to set that the box is a triangle. */
	protected JRadioButton triangleRB;
	
	/** Allows to set that the box is a diamond. */
	protected JRadioButton diamondRB;
	
	/** Allows to add a new box. */
	protected JButton addBox;
	
	/** Allows to modify the position of a box. */
	protected JButton modifyBox;
	
	/** Allows to remove a box. */
	protected JButton removeBox;
	
	/** The list of all the boxes. */
	protected JList boxesList;
	
	/** Allows to set the frameSep attribute. */
	protected JSpinner frameSepField;
	
	/** Allows to open the parameters frame of the box. */
	protected JButton editBoxB;
	
	/** Allows to set the boxSep attribute. */
	protected JCheckBox boxSepCB;
	
	/** Allows to set the isOpaque attribute. */
	protected JCheckBox opaqueCB;
	
	/** The label of each box in the list. */
	protected Vector<FramedBox> boxes;
	
	/** Allows to select parts of the text to box. */
	protected SelectTextFrame textSelFrame;
	
	/** Can choose the colour of the opacity. */
	protected ColorButton opacityColorButton;
	
	/** The frame of parameters to change the rectangle and the ellipse. */
	protected ParametersEllipseRectangleFrame rectFrame;
	
	/** The frame of parameters to change the circle. */
	protected ParametersCircleSquareFrame circleFrame;
	
	/** The frame of parameters to change the triangle. */
	protected ParametersTriangleFrame triangleFrame;
	
	/** The frame of parameters to change the diamond. */
	protected ParametersRhombusFrame diamondFrame;
	
	/** The label of the button opacityColorButton. */
	public static final String LABEL_OPACITY_BUTTON = LaTeXDrawLang.getString1_7("ParametersTextFrame.0"); //$NON-NLS-1$
	
	/** The label of simpleRB */
	public static final String LABEL_BOX_LIST = LaTeXDrawLang.getString1_7("ParametersTextFrame.1"); //$NON-NLS-1$
	
	/** The label of opaqueCB */
	public static final String LABEL_OPAQUE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.2"); //$NON-NLS-1$
	
	/** The label of circleRB */
	public static final String LABEL_CIRCLE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.3"); //$NON-NLS-1$
	
	/** The label of rectangleRB */
	public static final String LABEL_RECTANGLE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.4"); //$NON-NLS-1$
	
	/** The label of triangleRB */
	public static final String LABEL_TRIANGLE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.5"); //$NON-NLS-1$
	
	/** The label of diamondRB */
	public static final String LABEL_DIAMOND_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.6"); //$NON-NLS-1$
	
	/** The label of ellipseRB */
	public static final String LABEL_ELLIPSE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.7"); //$NON-NLS-1$
	
	/** The label of simpleRB */
	public static final String LABEL_SINGLE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.8"); //$NON-NLS-1$
	
	/** The label of multipleRB */
	public static final String LABEL_MULTIPLE_RB = LaTeXDrawLang.getString1_7("ParametersTextFrame.9"); //$NON-NLS-1$
	
	/** The label of framedBoxCB */
	public static final String LABEL_FRAMED_BOX_CB = LaTeXDrawLang.getString1_7("ParametersTextFrame.10"); //$NON-NLS-1$
	
	/** The label of modifyBox */
	public static final String LABEL_ADD_BOX = LaTeXDrawLang.getString1_7("ParametersTextFrame.11"); //$NON-NLS-1$
	
	/** The label of modifyBox */
	public static final String LABEL_MODIFY_BOX = LaTeXDrawLang.getString1_7("ParametersTextFrame.12"); //$NON-NLS-1$
	
	/** The label of removeBox */
	public static final String LABEL_REMOVE_BOX = LaTeXDrawLang.getString1_7("ParametersTextFrame.13"); //$NON-NLS-1$
	
	/** The label of frameSepField */
	public static final String LABEL_FRAME_SEP = LaTeXDrawLang.getString1_7("ParametersTextFrame.14"); //$NON-NLS-1$
	
	/** The label of editBoxB */
	public static final String LABEL_EDIT_BOX_B = LaTeXDrawLang.getString1_7("ParametersTextFrame.15"); //$NON-NLS-1$
	
	/** The label of boxSepCB */
	public static final String LABEL_BOX_SEP_CB = LaTeXDrawLang.getString1_7("ParametersTextFrame.16"); //$NON-NLS-1$
	
	/** The label of the bold button */
	public static final String LABEL_BOLD = LaTeXDrawLang.getDialogFrameString("ParametersTextFrame.0"); //$NON-NLS-1$
	
	/** The label of the italic button */
	public static final String LABEL_ITALIC = LaTeXDrawLang.getDialogFrameString("ParametersTextFrame.1"); //$NON-NLS-1$
	
	/** The label of the font choice */
	public static final String LABEL_FONT = LaTeXDrawLang.getDialogFrameString("ParametersTextFrame.2"); //$NON-NLS-1$
	
	/** The label of the size choice */
	public static final String LABEL_SIZE = LaTeXDrawLang.getDialogFrameString("ParametersTextFrame.3"); //$NON-NLS-1$
	
	/** The label of the text field */
	public static final String LABEL_TEXT = LaTeXDrawLang.getDialogFrameString("ParametersTextFrame.4"); //$NON-NLS-1$
	
	
	
	/**
	 * The constructor by default
	 * @param p The panel which contains the draw
	 */
	public ParametersTextFrame(Frame parent, DrawPanel p, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		JTabbedPane tabbedPane = new JTabbedPane();

	  		JToolBar toolbar = new JToolBar();
	  		toolbar.setLayout(new FlowLayout(FlowLayout.LEFT));
	  		toolbar.setName("Text toolbar");//$NON-NLS-1$
	  		toolbar.setPreferredSize(new Dimension(100, 30));
	  		toolbar.setMaximumSize(new Dimension(600, 30));
	  		toolbar.setFloatable(false);
	  		buttonItalic = new JToggleButton(new ImageIcon(getClass().getResource("/res/Italic.png")));//$NON-NLS-1$
	  		buttonItalic.setActionCommand(LABEL_ITALIC);
	  		buttonItalic.addActionListener(this);
	  		buttonBold = new JToggleButton(new ImageIcon(getClass().getResource("/res/Bold.png")));//$NON-NLS-1$
	  		buttonBold.setActionCommand(LABEL_BOLD);
	  		buttonBold.addActionListener(this);
	  		toolbar.add(buttonItalic);
	  		toolbar.add(buttonBold);
	  		
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel(new GridBagLayout());
	  		
	  		textField = new JTextField();
	  		textField.setColumns(40);
	  		textField.setName(LABEL_TEXT);
	  		textField.addKeyListener(this);
	  		Dimension dimension = new Dimension(350, 30);//fixes #1562954
	  		textField.setMinimumSize(dimension);
	  		textField.setMaximumSize(dimension);
	  		textField.setPreferredSize(dimension);
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 5;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	pGeneral.add(textField, constraint);
	  		
	     	constraint.anchor = GridBagConstraints.WEST;
	     	constraint.gridx = 0;	constraint.gridy++;
	     	constraint.gridwidth = 1;
	     	pGeneral.add(new JLabel(LABEL_X), constraint);
	     	
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	xCoord = new JSpinner(model);
	     	xCoord.setEditor(new JSpinner.NumberEditor(xCoord, "0.0"));//$NON-NLS-1$
			xCoord.addChangeListener(this);
	     	xCoord.setName(LABEL_X);
	     	constraint.gridx++;
	     	pGeneral.add(xCoord, constraint);

	    	constraint.gridx = 0;	constraint.gridy++;
	     	pGeneral.add(new JLabel(LABEL_Y), constraint);
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	yCoord = new JSpinner(model);
	     	yCoord.setEditor(new JSpinner.NumberEditor(yCoord, "0.0"));//$NON-NLS-1$
	     	yCoord.addChangeListener(this);
	     	yCoord.setName(LABEL_Y);
	     	constraint.gridx = 1;
	     	pGeneral.add(yCoord, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	constraint.gridwidth = 1;
	     	pGeneral.add(new JLabel(LABEL_ROTATION), constraint);
	     	
	     	model = new SpinnerNumberModel(Figure.DEFAULT_ROTATION_ANGLE, 
					-360, 360,0.1);
	     	rotationField = new JSpinner(model);
	     	rotationField.setEditor(new JSpinner.NumberEditor(rotationField, "0.0"));	//$NON-NLS-1$     	
	     	rotationField.setName(LaTeXDrawResources.LABEL_ROTATE);
	     	rotationField.addChangeListener(this);
	     	constraint.gridx = 1;
	     	pGeneral.add(rotationField, constraint);
	     	
	     	bordersButton = new ColorButton(LaTeXDrawResources.LABEL_COLOR_BUTTON, new ButtonIcon(Color.BLACK));
	     	bordersButton.setActionCommand(LaTeXDrawResources.LABEL_COLOR_BUTTON);
	     	bordersButton.addActionListener(this);
	     	constraint.gridx = 0;	constraint.gridy++;
	     	pGeneral.add(bordersButton, constraint);
	     	
	     	constraint.gridx=0;		constraint.gridy++;
	     	pGeneral.add(new JLabel(LABEL_SIZE), constraint);
	     	
	     	sizeChoice = Text.getChoiceSize();
	     	sizeChoice.setName(LABEL_SIZE);
	     	sizeChoice.addItemListener(this);
	     	constraint.gridx++;
	     	pGeneral.add(sizeChoice, constraint);
	     	
	     	constraint.gridx=0;		constraint.gridy++;
	     	pGeneral.add(new JLabel(LABEL_FONT), constraint);
	     	
	     	fontChoice = Text.getChoiceFont();
	     	fontChoice.setName(LABEL_FONT);
	     	fontChoice.addItemListener(this);
	     	constraint.gridx++;
	     	pGeneral.add(fontChoice, constraint);
	     	
	     	constraint.gridx=0;		constraint.gridy++;
	     	opaqueCB = new JCheckBox(LABEL_OPAQUE_RB);
	     	opaqueCB.setActionCommand(LABEL_OPAQUE_RB);
	     	opaqueCB.addActionListener(this);
	     	opaqueCB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.17")); //$NON-NLS-1$
	     	opaqueCB.setSelected(Text.DEFAULT_IS_OPAQUE);
	     	pGeneral.add(opaqueCB, constraint);
	     	
	     	opacityColorButton = new ColorButton(LABEL_OPACITY_BUTTON, new ButtonIcon(Color.WHITE));
	     	opacityColorButton.setActionCommand(LABEL_OPACITY_BUTTON);
	     	opacityColorButton.addActionListener(this);
	     	constraint.gridx++;
	     	pGeneral.add(opacityColorButton, constraint);
	     	
	     	pGeneral.setBorder(new CompoundBorder(new TitledBorder(null, TITLE_GENERAL, 
							  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	tabbedPane.add(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	tabbedPane.add(LaTeXDrawLang.getString1_7("ParametersTextFrame.18"), createFramedBoxPanel()); //$NON-NLS-1$

	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(toolbar);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}
		catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	/**
	 * @return The panel which set the framed box(es) of the text.
	 * @since 1.7
	 */
	private JPanel createFramedBoxPanel()
	{
		JPanel fbP = new JPanel(new GridBagLayout());
		
		fbP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("ParametersTextFrame.18"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
		fbP.setPreferredSize(new Dimension(390, 280));
		
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;  		constraint.gridy = 0;
     	constraint.gridwidth = 2; 	constraint.gridheight = 1;
     	constraint.weightx = 0.1; 	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.WEST;
     	ButtonGroup shapeBox = new ButtonGroup();
     	
     	framedBoxCB = new JCheckBox(LABEL_FRAMED_BOX_CB);
     	framedBoxCB.setActionCommand(LABEL_FRAMED_BOX_CB);
     	framedBoxCB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.20")); //$NON-NLS-1$
     	framedBoxCB.setSelected(false);
     	framedBoxCB.addActionListener(this);
     	framedBoxCB.setEnabled(false);
     	fbP.add(framedBoxCB, constraint);
     	
     	rectangleRB = new JRadioButton(LABEL_RECTANGLE_RB);
    	rectangleRB.setActionCommand(LABEL_RECTANGLE_RB);
    	rectangleRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.21")); //$NON-NLS-1$
     	rectangleRB.addActionListener(this);
     	rectangleRB.setEnabled(false);
     	rectangleRB.setSelected(true);
     	constraint.gridwidth = 1; 
     	constraint.gridx=2;
     	fbP.add(rectangleRB, constraint);
     	shapeBox.add(rectangleRB);
     	
     	triangleRB = new JRadioButton(LABEL_TRIANGLE_RB);
     	triangleRB.setActionCommand(LABEL_TRIANGLE_RB);
     	triangleRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.22")); //$NON-NLS-1$
     	triangleRB.addActionListener(this);
     	triangleRB.setEnabled(false);
     	triangleRB.setSelected(true);
     	constraint.gridx=3;
     	fbP.add(triangleRB, constraint);
     	shapeBox.add(triangleRB);
     	
     	constraint.gridy++;	constraint.gridx=0;
     	constraint.gridwidth = 2; //Fixes #1562954 
     	ButtonGroup typeBox = new ButtonGroup();
     	
     	simpleRB = new JRadioButton(LABEL_SINGLE_RB);
     	simpleRB.setActionCommand(LABEL_SINGLE_RB);
     	simpleRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.23")); //$NON-NLS-1$
     	simpleRB.addActionListener(this);
     	simpleRB.setSelected(true);
     	simpleRB.setEnabled(false);
     	typeBox.add(simpleRB);
     	fbP.add(simpleRB, constraint);
     	
     	ellipseRB = new JRadioButton(LABEL_ELLIPSE_RB);
     	ellipseRB.setActionCommand(LABEL_ELLIPSE_RB);
     	ellipseRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.24")); //$NON-NLS-1$
     	ellipseRB.addActionListener(this);
     	ellipseRB.setEnabled(false);
     	ellipseRB.setSelected(true);
     	constraint.gridx=2;	
     	fbP.add(ellipseRB, constraint);
     	shapeBox.add(ellipseRB);
     	
     	diamondRB = new JRadioButton(LABEL_DIAMOND_RB);
     	diamondRB.setActionCommand(LABEL_DIAMOND_RB);
     	diamondRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.25")); //$NON-NLS-1$
     	diamondRB.addActionListener(this);
     	diamondRB.setEnabled(false);
     	diamondRB.setSelected(true);
     	constraint.gridx=3;
     	fbP.add(diamondRB, constraint);
     	shapeBox.add(diamondRB);
     	
     	constraint.gridy++;	constraint.gridx=0;
     	constraint.gridwidth = 2; 
     	multipleRB = new JRadioButton(LABEL_MULTIPLE_RB);
     	multipleRB.setActionCommand(LABEL_MULTIPLE_RB);
     	multipleRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.26")); //$NON-NLS-1$
     	multipleRB.addActionListener(this);
     	multipleRB.setEnabled(false);
     	typeBox.add(multipleRB);
     	fbP.add(multipleRB, constraint);
     	
     	circleRB = new JRadioButton(LABEL_CIRCLE_RB);
     	circleRB.setActionCommand(LABEL_CIRCLE_RB);
     	circleRB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.27")); //$NON-NLS-1$
     	circleRB.addActionListener(this);
     	circleRB.setEnabled(false);
     	circleRB.setSelected(true);
     	constraint.gridx=2;		constraint.gridwidth = 1; 
     	fbP.add(circleRB, constraint);
     	shapeBox.add(circleRB);
     	
     	constraint.gridy++; constraint.gridx=0;	
     	addBox = new JButton(LABEL_ADD_BOX);
     	addBox.setActionCommand(LABEL_ADD_BOX);
     	addBox.addActionListener(this);
     	addBox.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.28")); //$NON-NLS-1$
     	fbP.add(addBox, constraint);
     	addBox.setEnabled(false);
     	
     	constraint.gridx+=2; constraint.gridwidth = 4;
     	constraint.gridheight=3;
     	boxes = new Vector<FramedBox>();
     	boxesList = new JList(boxes);
     	boxesList.setVisibleRowCount(5);
     	boxesList.setEnabled(false);
     	boxesList.addFocusListener(this);
     	boxesList.addMouseListener(this);
     	boxesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
     	boxesList.setName(LABEL_BOX_LIST);
     	JScrollPane scrollPane = new JScrollPane(boxesList);
     	Dimension dimension = new Dimension(250,100);
     	scrollPane.setPreferredSize(dimension);
     	scrollPane.setMinimumSize(dimension);
     	scrollPane.setMaximumSize(dimension);
     	fbP.add(scrollPane, constraint);
     	
     	constraint.gridx=0;	constraint.gridheight=1;
     	constraint.gridy++; constraint.gridwidth = 2;
     	modifyBox = new JButton(LABEL_MODIFY_BOX);
     	modifyBox.setActionCommand(LABEL_MODIFY_BOX);
     	modifyBox.addActionListener(this);
     	modifyBox.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.29")); //$NON-NLS-1$
     	fbP.add(modifyBox, constraint);
     	modifyBox.setEnabled(false);
     	
     	constraint.gridy++;
     	removeBox = new JButton(LABEL_REMOVE_BOX);
     	removeBox.setActionCommand(LABEL_REMOVE_BOX);
     	removeBox.addActionListener(this);
     	removeBox.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.30")); //$NON-NLS-1$
     	fbP.add(removeBox, constraint);
     	removeBox.setEnabled(false);
     	
     	constraint.gridy++;
     	constraint.gridwidth = 1;
     	fbP.add(new JLabel(LABEL_FRAME_SEP), constraint);
     	
     	constraint.gridx++;
     	SpinnerModel model = new SpinnerNumberModel(
     			PSTricksConstants.DEFAULT_FRAME_SEP*DrawContainer.DEFAULT_PIXPERCM,0,2000,1);
     	frameSepField = new JSpinner(model);
     	frameSepField.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.31")); //$NON-NLS-1$
     	frameSepField.setEditor(new JSpinner.NumberEditor(frameSepField, "0"));//$NON-NLS-1$
     	frameSepField.addChangeListener(this);
     	frameSepField.setName(LABEL_FRAME_SEP);
     	fbP.add(frameSepField, constraint);	 
     	frameSepField.setEnabled(false);
     	
     	constraint.gridx++;
     	boxSepCB = new JCheckBox(LABEL_BOX_SEP_CB);
     	boxSepCB.setActionCommand(LABEL_BOX_SEP_CB);
     	boxSepCB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.32")); //$NON-NLS-1$
     	boxSepCB.setSelected(PSTricksConstants.DEFAULT_BOX_SEP);
     	boxSepCB.addActionListener(this);
     	fbP.add(boxSepCB, constraint);
     	boxSepCB.setEnabled(false);
     	
     	constraint.gridx++;
     	editBoxB = new JButton(LABEL_EDIT_BOX_B);
     	editBoxB.setActionCommand(LABEL_EDIT_BOX_B);
     	editBoxB.addActionListener(this);
     	editBoxB.setToolTipText(LaTeXDrawLang.getString1_7("ParametersTextFrame.33")); //$NON-NLS-1$
     	fbP.add(editBoxB, constraint);
     	editBoxB.setEnabled(false);
		
		return fbP;
	}
	
	
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		try
		{
			Object o = e.getSource();
			Text t = (Text) glimpsePanel.getGlimpseFigure();
			
			if(o instanceof AbstractButton)
			{
				String label = ((AbstractButton)o).getActionCommand();
				
				if(label.equals(LABEL_ADD_BOX))
				{
					if(textSelFrame==null)
						textSelFrame = new SelectTextFrame(this);
					
					textSelFrame.setVisible(true);
					int[] is 	= textSelFrame.getSelectedIndex();
					String txt 	= textSelFrame.getSelectedText();
					
					if(txt!=null && txt.length()>0 && is!=null && is.length==2)
					{
						FramedBox fb = new FramedBox(t);
						fb.setStart(is[0]);
						fb.setEnd(is[1]);
						
						if(t.addBox(fb))
						{
							setModified(true);
							boxes.add(fb);
							int id = boxesList.getSelectedIndex();
							frameSepField.setEnabled(id!=-1);
							boxSepCB.setEnabled(id!=-1);
							editBoxB.setEnabled(id!=-1);
							removeBox.setEnabled(id!=-1);
							modifyBox.setEnabled(id!=-1);
							rectangleRB.setEnabled(id!=-1);
							circleRB.setEnabled(id!=-1);
							ellipseRB.setEnabled(id!=-1);
							diamondRB.setEnabled(id!=-1);
							triangleRB.setEnabled(id!=-1);
							boxesList.updateUI();
							glimpsePanel.repaint();
						}
					}
					return;
				}
				
				if(label.equals(LABEL_REMOVE_BOX))
				{
					Object selected = boxesList.getSelectedValue();
					boxes.remove(selected);
					boxesList.updateUI();
					boxesList.clearSelection();
					
					if(selected instanceof FramedBox)
						t.removeBox((FramedBox)selected);
					
					setModified(true);
					frameSepField.setEnabled(false);
					boxSepCB.setEnabled(false);
					editBoxB.setEnabled(false);
					removeBox.setEnabled(false);
					modifyBox.setEnabled(false);
					rectangleRB.setEnabled(false);
					circleRB.setEnabled(false);
					ellipseRB.setEnabled(false);
					diamondRB.setEnabled(false);
					triangleRB.setEnabled(false);
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(LABEL_MODIFY_BOX))
				{
					if(textSelFrame==null)
						textSelFrame = new SelectTextFrame(this);
					
					Object selected = boxesList.getSelectedValue();
					
					if(selected instanceof FramedBox)
					{
						FramedBox fb = (FramedBox)selected;
						int index = t.getMultipleBox().indexOf(fb);
						t.getMultipleBox().remove(index);
						
						textSelFrame.show(fb.getStart(), fb.getEnd());
						int[] is = textSelFrame.getSelectedIndex();
						
						t.getMultipleBox().add(index, fb);
						
						if(is!=null && is.length==2)
						{
							setModified(true);
							fb.setStart(is[0]);
							fb.setEnd(is[1]);
							glimpsePanel.repaint();
							boxesList.updateUI();
						}
					}
					
					return ;
				}
				
				if(label.equals(LABEL_OPAQUE_RB))
				{
					setModified(true);
					t.setOpaque(opaqueCB.isSelected());
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_TRIANGLE_RB))
				{
					if(simpleRB.isSelected())
						 t.getSimpleBox().setBoxType(FramedBox.BOX_TRIANGLE);
					else 
						((FramedBox)boxesList.getSelectedValue()).setBoxType(FramedBox.BOX_TRIANGLE);
					
					setModified(true);
					t.updateFramedBoxes();
					t.updateBorders();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_DIAMOND_RB))
				{
					if(simpleRB.isSelected())
						 t.getSimpleBox().setBoxType(FramedBox.BOX_DIAMOND);
					else 
						((FramedBox)boxesList.getSelectedValue()).setBoxType(FramedBox.BOX_DIAMOND);
					
					setModified(true);
					t.updateFramedBoxes();
					t.updateBorders();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_RECTANGLE_RB))
				{
					if(simpleRB.isSelected())
						 t.getSimpleBox().setBoxType(FramedBox.BOX_RECTANGLE);
					else 
						((FramedBox)boxesList.getSelectedValue()).setBoxType(FramedBox.BOX_RECTANGLE);
					
					setModified(true);
					t.updateFramedBoxes();
					t.updateBorders();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_ELLIPSE_RB))
				{
					if(simpleRB.isSelected())
						 t.getSimpleBox().setBoxType(FramedBox.BOX_ELLIPSE);
					else 
						((FramedBox)boxesList.getSelectedValue()).setBoxType(FramedBox.BOX_ELLIPSE);
					
					setModified(true);
					t.updateFramedBoxes();
					t.updateBorders();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_CIRCLE_RB))
				{
					if(simpleRB.isSelected())
						 t.getSimpleBox().setBoxType(FramedBox.BOX_CIRCLE);
					else 
						((FramedBox)boxesList.getSelectedValue()).setBoxType(FramedBox.BOX_CIRCLE);
					
					setModified(true);
					t.updateFramedBoxes();
					t.updateBorders();
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_EDIT_BOX_B))
				{
					Figure f;
					
					if(simpleRB.isSelected())
						 f = t.getSimpleBox().getBox();
					else 
						f = ((FramedBox)boxesList.getSelectedValue()).getBox();
						
					if(rectangleRB.isSelected() || ellipseRB.isSelected())
					{
						if(rectFrame==null)
							rectFrame = new ParametersEllipseRectangleFrame((JFrame)getParent(), drawPanel, false);
						rectFrame.setVisible(true, f, false, true);
					}
					else if(circleRB.isSelected())
					{
						if(circleFrame==null)
							circleFrame = new ParametersCircleSquareFrame((JFrame)getParent(), drawPanel, false);
						circleFrame.setVisible(true, f, false, true);
					}
					else if(diamondRB.isSelected())
					{
						if(diamondFrame==null)
							diamondFrame = new ParametersRhombusFrame((JFrame)getParent(), drawPanel, false);
						diamondFrame.setVisible(true, f, false, true);
					}
					else if(triangleRB.isSelected())
					{
						if(triangleFrame==null)
							triangleFrame = new ParametersTriangleFrame((JFrame)getParent(), drawPanel, false);
						triangleFrame.setVisible(true, f, false, true);
					}
					
					if(multipleRB.isSelected())
						t.updateFramedBoxes();
					
					setModified(true);
					glimpsePanel.repaint();
					return;
				}
				
				if(label.equals(LABEL_FRAMED_BOX_CB))
				{
					boolean ok = framedBoxCB.isSelected();
					int id = boxesList.getSelectedIndex();
					boolean simple = simpleRB.isSelected();
					
					setModified(true);
					opaqueCB.setEnabled(!ok);
					simpleRB.setEnabled(ok);
					boxesList.setEnabled(ok && !simple);
					multipleRB.setEnabled(ok);
					frameSepField.setEnabled(ok && ( simple || id!=-1));
					boxSepCB.setEnabled(ok && id!=-1);
					editBoxB.setEnabled(ok && (simple || id!=-1));
					addBox.setEnabled(ok && !simple);
					removeBox.setEnabled(ok && !simple && id!=-1);
					modifyBox.setEnabled(ok && !simple && id!=-1);
					rectangleRB.setEnabled(ok && (simple || id!=-1));
					circleRB.setEnabled(ok && (simple || id!=-1));
					ellipseRB.setEnabled(ok && (simple || id!=-1));
					diamondRB.setEnabled(ok && (simple || id!=-1));
					triangleRB.setEnabled(ok && (simple || id!=-1));
					t.setHasFramedBox(ok);
					t.setHasSimpleFramedBox(simpleRB.isSelected());
					
					if(simpleRB.isSelected())
					{
						FramedBox fb = t.getSimpleBox();
						fb.setBoxSep(boxSepCB.isSelected());
						fb.setFrameSep(Double.valueOf(frameSepField.getValue().toString()).doubleValue());
						if(rectangleRB.isSelected())
							 fb.setBoxType(FramedBox.BOX_RECTANGLE);
						else if(ellipseRB.isSelected())
							 fb.setBoxType(FramedBox.BOX_ELLIPSE);
						else if(circleRB.isSelected())
							 fb.setBoxType(FramedBox.BOX_CIRCLE);
						else if(diamondRB.isSelected())
							 fb.setBoxType(FramedBox.BOX_DIAMOND);
						else fb.setBoxType(FramedBox.BOX_TRIANGLE);
					}
					
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(LABEL_BOLD))
				{
					setModified(true);
					t.setIsBold(buttonBold.isSelected());
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(LABEL_ITALIC))
				{
					setModified(true);
					t.setIsItalic(buttonItalic.isSelected());
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}

				if(label.equals(LABEL_SINGLE_RB))
				{
					setModified(true);
					frameSepField.setEnabled(true);
					boxSepCB.setEnabled(true);
					editBoxB.setEnabled(true);
					boxesList.setEnabled(false);
					addBox.setEnabled(false);
					removeBox.setEnabled(false);
					modifyBox.setEnabled(false);
					t.setHasSimpleFramedBox(true);
					FramedBox fb = t.getSimpleBox();
					frameSepField.setValue(fb.getFrameSep());
					boxSepCB.setEnabled(false);
					rectangleRB.setEnabled(true);
					circleRB.setEnabled(true);
					ellipseRB.setEnabled(true);
					diamondRB.setEnabled(true);
					triangleRB.setEnabled(true);
					
					int style = fb.getBoxType();
					if(style==FramedBox.BOX_RECTANGLE)
						rectangleRB.setSelected(true);
					else if(style==FramedBox.BOX_CIRCLE)
						circleRB.setSelected(true);
					else if(style==FramedBox.BOX_ELLIPSE)
						ellipseRB.setSelected(true);
					else if(style==FramedBox.BOX_TRIANGLE)
						triangleRB.setSelected(true);
					else if(style==FramedBox.BOX_DIAMOND)
						diamondRB.setSelected(true);
					boxSepCB.setSelected(fb.isBoxSep());
					frameSepField.setValue(fb.getFrameSep());
					
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(LABEL_MULTIPLE_RB))
				{
					int id = boxesList.getSelectedIndex();
					
					setModified(true);
					frameSepField.setEnabled(id!=-1);
					boxSepCB.setEnabled(id!=-1);
					editBoxB.setEnabled(id!=-1);
					boxesList.setEnabled(true);
					addBox.setEnabled(true);
					removeBox.setEnabled(id!=-1);
					modifyBox.setEnabled(id!=-1);
					t.setHasSimpleFramedBox(false);
					rectangleRB.setEnabled(id!=-1);
					circleRB.setEnabled(id!=-1);
					ellipseRB.setEnabled(id!=-1);
					diamondRB.setEnabled(id!=-1);
					triangleRB.setEnabled(id!=-1);
					
					if(id!=-1)
					{
						FramedBox fb = (FramedBox)boxesList.getSelectedValue();
						int style = fb.getBoxType();
						if(style==FramedBox.BOX_RECTANGLE)
							rectangleRB.setSelected(true);
						else if(style==FramedBox.BOX_CIRCLE)
							circleRB.setSelected(true);
						else if(style==FramedBox.BOX_ELLIPSE)
							ellipseRB.setSelected(true);
						else if(style==FramedBox.BOX_TRIANGLE)
							triangleRB.setSelected(true);
						else if(style==FramedBox.BOX_DIAMOND)
							diamondRB.setSelected(true);
						boxSepCB.setSelected(fb.isBoxSep());
						frameSepField.setValue(fb.getFrameSep());
					}
					
					glimpsePanel.repaint();
					return ;
				}
				
				if(label.equals(LABEL_BOX_SEP_CB))
				{
					if(multipleRB.isSelected())
					{
						setModified(true);
						FramedBox fb = (FramedBox)boxesList.getSelectedValue();
						fb.setBoxSep(boxSepCB.isSelected());
						t.updateFramedBoxes();
						t.updateBorders();
						glimpsePanel.repaint();
					}
					return ;
				}
				
				
				if(label.equals(LABEL_OPACITY_BUTTON))
				{
					Color c = JColorChooser.showDialog((Component) e.getSource(), 
							LaTeXDrawLang.getDialogFrameString("AbstractParametersFrame.34"),  //$NON-NLS-1$
							((ColorButton)o).getColor());
						
					if(c!=null)
					{
						setModified(true);
						((ColorButton)o).setColor(c);
						t.setOpacityColor(c);
						glimpsePanel.repaint();
					}
					return ;
				}
			}
			super.actionPerformed(e);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	

	@Override
	public void setFigureFrameField()
	{
		try
		{
			if(!(figure instanceof Text))
					throw new ClassCastException();
			
			Text t = (Text)figure;
			LaTeXDrawPoint2D p = t.getPosition();
			opacityColorButton.setColor(t.getOpacityColor());
			boxes.clear();
			Text txt = (Text)glimpsePanel.glimpseFigure;
			for(FramedBox bf : txt.getMultipleBox())
				boxes.add(bf);
			boxesList.clearSelection();
			boxesList.updateUI();
			framedBoxCB.setSelected(t.hasFramedBox());
			framedBoxCB.setEnabled(t.getText().length()>0);
			simpleRB.setEnabled(framedBoxCB.isEnabled() && framedBoxCB.isSelected());
			multipleRB.setEnabled(framedBoxCB.isEnabled() && framedBoxCB.isSelected());
			simpleRB.setSelected(t.hasFramedBox() && t.hasSimpleFramedBox());
			multipleRB.setSelected(t.hasSeveralBoxes());
			frameSepField.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			boxSepCB.setEnabled(false);
			circleRB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			rectangleRB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			triangleRB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			diamondRB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			ellipseRB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			editBoxB.setEnabled(framedBoxCB.isSelected() && simpleRB.isSelected());
			frameSepField.removeChangeListener(this);
			if(t.hasFramedBox() && t.hasSimpleFramedBox())
				frameSepField.setValue(t.getSimpleBox().getFrameSep());
			else frameSepField.setValue(PSTricksConstants.DEFAULT_FRAME_SEP*Figure.PPC);
			frameSepField.addChangeListener(this);
			textField.setText(t.getText());
			xCoord.setValue(p.x);
			yCoord.setValue(p.y);
			sizeChoice.setSelectedItem(t.getTextSize().getLabel());
			buttonItalic.setSelected(t.isItalic());
			buttonBold.setSelected(t.isBold());
			fontChoice.setSelectedItem(t.getCurrentTextFont().getName());
			if(t.getText().length()==0)
				 buttonOk.setEnabled(false);
			else buttonOk.setEnabled(true);
			opaqueCB.setSelected(t.isOpaque());
			opaqueCB.setEnabled(!t.hasFramedBox());
			addBox.setEnabled(t.hasSeveralBoxes());
			boxesList.setEnabled(t.hasSeveralBoxes());
			
			if(t.hasSimpleFramedBox() && t.getSimpleBox()!=null)
			{
				int style = t.getSimpleBox().getBoxType();
				if(style==FramedBox.BOX_RECTANGLE)
					rectangleRB.setSelected(true);
				else if(style==FramedBox.BOX_CIRCLE)
					circleRB.setSelected(true);
				else if(style==FramedBox.BOX_ELLIPSE)
					ellipseRB.setSelected(true);
				else if(style==FramedBox.BOX_TRIANGLE)
					triangleRB.setSelected(true);
				else if(style==FramedBox.BOX_DIAMOND)
					diamondRB.setSelected(true);
			}
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}


	
	@Override
	public void setDefaultsValues()
	{
		try
		{
			super.setDefaultsValues();
			Text t = (Text)glimpsePanel.getGlimpseFigure();
			t.setOpacityColor(Text.DEFAULT_OPACITY_COLOR);
			opacityColorButton.setColor(Text.DEFAULT_OPACITY_COLOR);
			fontChoice.setSelectedItem(Text.DEFAULT_TEXTFONT.getName());
			buttonItalic.setSelected(Text.DEFAULT_ISITALIC);
			buttonBold.setSelected(Text.DEFAULT_ISBOLD);
			t.setIsBold(Text.DEFAULT_ISBOLD);
			t.setIsItalic(Text.DEFAULT_ISITALIC);
			t.setTextFont(Text.DEFAULT_TEXTFONT.getName());
			t.setSize(Text.DEFAULT_SIZE.getLabel());
			sizeChoice.setSelectedItem(Text.DEFAULT_SIZE.getLabel());
			t.setSimpleBox(new FramedBox(t));
			t.setHasFramedBox(Text.DEFAULT_HAS_BOX);
			framedBoxCB.setSelected(Text.DEFAULT_HAS_BOX);
			rectangleRB.setSelected(true);
			boxSepCB.setSelected(PSTricksConstants.DEFAULT_BOX_SEP);
			frameSepField.setValue(PSTricksConstants.DEFAULT_FRAME_SEP*Figure.PPC);
			opaqueCB.setSelected(Text.DEFAULT_IS_OPAQUE);
			
			boolean ok = framedBoxCB.isSelected();
			int id = boxesList.getSelectedIndex();
			boolean simple = simpleRB.isSelected();
			
			opaqueCB.setEnabled(!ok);
			simpleRB.setEnabled(ok);
			boxesList.setEnabled(ok && !simple);
			multipleRB.setEnabled(ok);
			frameSepField.setEnabled(ok && ( simple || id!=-1));
			boxSepCB.setEnabled(ok && id!=-1);
			editBoxB.setEnabled(ok && (simple || id!=-1));
			addBox.setEnabled(ok && !simple && id!=-1);
			removeBox.setEnabled(ok && !simple && id!=-1);
			modifyBox.setEnabled(ok && !simple && id!=-1);
			rectangleRB.setEnabled(ok && (simple || id!=-1));
			circleRB.setEnabled(ok && (simple || id!=-1));
			ellipseRB.setEnabled(ok && (simple || id!=-1));
			diamondRB.setEnabled(ok && (simple || id!=-1));
			triangleRB.setEnabled(ok && (simple || id!=-1));
			boxes.clear();
			boxesList.updateUI();
			t.setHasFramedBox(ok);
			t.setHasSimpleFramedBox(simpleRB.isSelected());
		}
		catch(Exception e)
		{
			e.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		if(!(figure instanceof Text))
				throw new ClassCastException();
		
		Text t2 = (Text) glimpsePanel.getGlimpseFigure();
		Text t = (Text)figure;
		LaTeXDrawPoint2D p = t.getPosition();
		
		t.setOpacityColor(opacityColorButton.getColor());
		p.x = Double.valueOf(xCoord.getValue().toString());
		p.y = Double.valueOf(yCoord.getValue().toString());
		t.setText(textField.getText());
		t.setSize(sizeChoice.getSelectedItem().toString());
		t.setIsBold(buttonBold.isSelected());
		t.setIsItalic(buttonItalic.isSelected());
		t.setTextFont(fontChoice.getSelectedItem().toString());
		t.updateFontsBorders();
		t.setOpaque(opaqueCB.isSelected());
		t.setHasFramedBox(framedBoxCB.isSelected() && (simpleRB.isSelected() || !t2.getMultipleBox().isEmpty()));
		if(framedBoxCB.isSelected())
			t.setHasSimpleFramedBox(simpleRB.isSelected());
		
		if(t2.hasSimpleFramedBox())
		{
			t2.getSimpleBox().setText(t);
			t.setSimpleBox(t2.getSimpleBox());
		}
		
		if(t2.hasSeveralBoxes())
		{
			t.removeAllBoxes();
			
			if(!t2.getMultipleBox().isEmpty())
				t.setHasFramedBox(true);
			
			for(FramedBox fb : t2.getMultipleBox())
				t.addBox(fb);
		}
		
		t.updateBorders();
		drawPanel.getFrameParent().setExportAsMenu(true);
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}
	
	
	
	
	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Text))
				throw new ClassCastException();
			
			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
	
	
	
	
	@Override
	public void itemStateChanged(ItemEvent e) 
	{
		try
		{
			Object o = e.getSource();
			Text t = (Text) glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JComboBox)
			{
				String name = ((JComboBox)o).getName();
				
				if(name.equals(LABEL_FONT))
				{
					setModified(true);
					t.setTextFont(fontChoice.getSelectedItem().toString());
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_SIZE))
				{
					setModified(true);
					t.setSize(sizeChoice.getSelectedItem().toString());
					glimpsePanel.repaint();
					return ;
				}
			}
			
			super.itemStateChanged(e);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
	
	
	
	
	@Override
	public void keyReleased(KeyEvent e)
	{
		try
		{
			Object o = e.getSource();
			Text t = (Text)glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JTextField)
			{
				String name = ((JTextField)o).getName();
				
				if(name.equals(LABEL_TEXT))
				{
					if(multipleRB.isSelected())
					{
						boxes.removeAllElements();
						boxesList.updateUI();
						t.getMultipleBox().removeAllElements();
						t.getTextPos().removeAllElements();
						editBoxB.setEnabled(false);
						circleRB.setEnabled(false);
						ellipseRB.setEnabled(false);
						rectangleRB.setEnabled(false);
						triangleRB.setEnabled(false);
						diamondRB.setEnabled(false);
						boxSepCB.setEnabled(false);
						frameSepField.setEnabled(false);
						modifyBox.setEnabled(false);
						removeBox.setEnabled(false);
					}
					
					setModified(true);
					t.setText(textField.getText());
					buttonOk.setEnabled(textField.getText().length()!=0);
					framedBoxCB.setEnabled(textField.getText().length()!=0);
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
			}
			
			super.keyReleased(e);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Text t = (Text)glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				
				if(name.equals(LABEL_X))
				{
					setModified(true);
					t.setX(Double.valueOf(xCoord.getValue().toString()).doubleValue());
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_Y))
				{
					setModified(true);
					t.setY(Double.valueOf(yCoord.getValue().toString()).doubleValue());
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_FRAME_SEP))
				{
					double val = Double.valueOf(frameSepField.getValue().toString()).doubleValue();
					
					if(simpleRB.isSelected())
					{
						t.getSimpleBox().setFrameSep(val);
						t.getSimpleBox().updateBoxSize();
					}
					else
					{
						((FramedBox)boxesList.getSelectedValue()).setFrameSep(val);
						t.updateFramedBoxes();
					}
					
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
			}
			
			if(o instanceof JTextField)
			{
				String name = ((JTextField)o).getName();
				
				if(name.equals(LABEL_TEXT))
				{
					setModified(true);
					t.setText(textField.getText());
					if(t.getSimpleBox()!=null) t.getSimpleBox().updateBoxSize();
					glimpsePanel.repaint();
					return ;
				}
			}
			
			super.stateChanged(e);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}



	public void focusGained(FocusEvent e)
	{
		try
		{
			Object o = e.getSource();
			
			if(o instanceof JList)
			{
				String name = ((JList)o).getName();
				
				if(name.equals(LABEL_BOX_LIST))
				{
					int id = boxesList.getSelectedIndex();
					frameSepField.setEnabled(id!=-1);
					boxSepCB.setEnabled(id!=-1);
					editBoxB.setEnabled(id!=-1);
					removeBox.setEnabled(id!=-1);
					modifyBox.setEnabled(id!=-1);
					rectangleRB.setEnabled(id!=-1);
					circleRB.setEnabled(id!=-1);
					ellipseRB.setEnabled(id!=-1);
					diamondRB.setEnabled(id!=-1);
					triangleRB.setEnabled(id!=-1);
					return ;
				}
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}



	public void focusLost(FocusEvent e)
	{
		try
		{
			Object o = e.getSource();
			
			if(o instanceof JList)
			{
				String name = ((JList)o).getName();
				
				if(name.equals(LABEL_BOX_LIST))
				{
					int id = boxesList.getSelectedIndex();
					frameSepField.setEnabled(id!=-1);
					boxSepCB.setEnabled(id!=-1);
					editBoxB.setEnabled(id!=-1);
					removeBox.setEnabled(id!=-1);
					modifyBox.setEnabled(id!=-1);
					return ;
				}
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}


	public void mouseClicked(MouseEvent e)
	{
		// Nothing to do.
	}

	public void mouseEntered(MouseEvent e)
	{
		// Nothing to do.
	}

	public void mouseExited(MouseEvent e)
	{
		// Nothing to do.
	}

	public void mousePressed(MouseEvent e)
	{
		// Nothing to do.
	}


	public void mouseReleased(MouseEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof JList && boxesList!=null)
		{
			Object selection = boxesList.getSelectedValue();
			
			if(multipleRB.isSelected() && selection instanceof FramedBox)
			{
				FramedBox fb = (FramedBox)selection;
				int style = fb.getBoxType();
				
				if(style==FramedBox.BOX_RECTANGLE)
					rectangleRB.setSelected(true);
				else if(style==FramedBox.BOX_CIRCLE)
					circleRB.setSelected(true);
				else if(style==FramedBox.BOX_ELLIPSE)
					ellipseRB.setSelected(true);
				else if(style==FramedBox.BOX_TRIANGLE)
					triangleRB.setSelected(true);
				else if(style==FramedBox.BOX_DIAMOND)
					diamondRB.setSelected(true);
				
				setModified(true);
				frameSepField.setValue(fb.getFrameSep());
				boxSepCB.setSelected(fb.isBoxSep());
			}
		}
	}
}



/**
 * This class defines a frame to select the text to box.
 * @since 1.7
 * @author Arnaud BLOUIN
 */
class SelectTextFrame extends JDialog implements ActionListener, MouseListener
{
	private static final long serialVersionUID = 1L;

	protected JButton buttonOk;
	
	/** This field allows the set the text */
	protected JTextField textField;
	
	/** The frame of parameters. */
	protected ParametersTextFrame textFrame;
	
	/** The framed box use to test the validity of the new box. */
	protected FramedBox framedBox;
	
	/** Allows to display a message in the frame. */
	protected JLabel msg;
	
	/** The position of the first selected character when the frame is displayed. */
	protected int startPos;
	
	/** The position of the last selected character when the frame is displayed. */
	protected int endPos;
	
	
	
	/**
	 * The constructor.
	 * @param frame The frame of parameters of a text.
	 */
	public SelectTextFrame(ParametersTextFrame frame)
	{
		super(frame, LaTeXDrawLang.getString1_7("ParametersTextFrame.34"), true); //$NON-NLS-1$
		startPos = 0;
		endPos = 0;
		setResizable(false);
		textFrame = frame;
		framedBox = new FramedBox((Text)frame.glimpsePanel.getGlimpseFigure());
		
		JPanel pButtons = new JPanel();
  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
  		setLocation(dim.width*1/3, dim.height*1/3);
  		
  		JPanel pGeneral = new JPanel(new GridBagLayout());
  		
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 0;
     	constraint.gridwidth = 1;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.WEST;
     	
     	pGeneral.add(new JLabel(LaTeXDrawLang.getString1_7("ParametersTextFrame.35")), constraint); //$NON-NLS-1$
     	
  		textField = new JTextField();
  		textField.setColumns(40);
  		textField.setPreferredSize(new Dimension(250, 20));
  		textField.setMinimumSize(new Dimension(250, 20));
  		textField.setName(ParametersTextFrame.LABEL_TEXT);
  		textField.setText(textFrame.textField.getText());
  		textField.setEditable(false);
  		textField.addMouseListener(this);
  		constraint.gridy++;
  		constraint.gridwidth = 5;
     	pGeneral.add(textField, constraint);
		
     	msg = new JLabel(" ");//$NON-NLS-1$
     	constraint.gridy++;
     	pGeneral.add(msg, constraint);
     	
     	pGeneral.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("ParametersTextFrame.36"),  //$NON-NLS-1$
				  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
     	pGeneral.setPreferredSize(new Dimension(270, 100));
     	
		buttonOk = new JButton(AbstractParametersFrame.LABEL_OK);
		JButton buttonCancel  = new JButton(AbstractParametersFrame.LABEL_CANCEL);  		
		GridLayout gridButton = new GridLayout(1, 2);
		
  		buttonOk.setActionCommand(AbstractParametersFrame.NAME_BUTTON_OK);
  		buttonOk.setEnabled(false);
  		buttonCancel.setActionCommand(AbstractParametersFrame.NAME_BUTTON_CANCEL);
  		
  		pButtons.setLayout(gridButton);
  		pButtons.add(buttonOk);
  		pButtons.add(buttonCancel);
  		buttonOk.addActionListener(this);
  		buttonCancel.addActionListener(this);
  		pButtons.setPreferredSize(new Dimension(30, 30));
  		pButtons.setMaximumSize(new Dimension(30, 30));
  		
  		getContentPane().setLayout(new BorderLayout());
     	getContentPane().add(pGeneral, BorderLayout.CENTER);
		getContentPane().add(pButtons, BorderLayout.SOUTH);
  		
  		setSize(270, 170);
	}

	
	
	/**
	 * Allows to do actions when an event is created.
	 */
	public void actionPerformed(ActionEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			if(o instanceof JButton || o instanceof JCheckBox)
			{
				String label = ((AbstractButton)o).getActionCommand();
				
				if(label.equals(AbstractParametersFrame.NAME_BUTTON_CANCEL))
				{
					textField.select(0, 0);
					setVisible(false);
					return;
				}
				
				if(label.equals(AbstractParametersFrame.NAME_BUTTON_OK))
				{
					setVisible(false);
					return;
				}
				
				return ;
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}

	
	
	/**
	 * @return The indexes (start and end) of the selected text.
	 */
	public int[] getSelectedIndex()
	{
		if(textField!=null)
		{
			String text = textField.getSelectedText();
			
			if(text==null || text.length()==0)
				return null;
			
			int[] is = {textField.getSelectionStart(), textField.getSelectionEnd()};
			return is;
		}
		return null;
	}
	
	
	/**
	 * @return The selected text.
	 */
	public String getSelectedText()
	{
		if(textField==null) return null;
		
		return textField.getSelectedText();
	}
	
	

	
	public void mouseClicked(MouseEvent e)
	{
//		 Nothing to do.
	}


	public void mouseEntered(MouseEvent e)
	{
//		 Nothing to do.
	}


	public void mouseExited(MouseEvent e)
	{
//		 Nothing to do.
	}


	public void mousePressed(MouseEvent e)
	{
		// Nothing to do.
	}


	public void mouseReleased(MouseEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof JTextField)
		{
			String text = textField.getSelectedText();
			int start = textField.getSelectionStart(), end = textField.getSelectionEnd();
			Text txt = (Text)textFrame.glimpsePanel.getGlimpseFigure();
			framedBox.setStart(start);
			framedBox.setEnd(end);
			boolean overlap =  framedBox.overlap(txt.getMultipleBox());
			
			if(overlap)
				 msg.setText(LaTeXDrawLang.getString1_7("ParametersTextFrame.37")); //$NON-NLS-1$
			else msg.setText(" ");//$NON-NLS-1$
			
			if(text==null || text.length()==0)
				msg.setText(" ");//$NON-NLS-1$
			
			buttonOk.setEnabled(text!=null && text.length()>0 && !overlap);
		}
	}



	@Override
	public void setVisible(boolean ok)
	{
		if(ok)
		{
			textField.setText(((Text)textFrame.glimpsePanel.getGlimpseFigure()).getText());
			textField.setSelectionStart(0);
			textField.setSelectionEnd(0);
			buttonOk.setEnabled(false);
			msg.setText(" ");//$NON-NLS-1$
		}
		super.setVisible(ok);
	}


	
	/**
	 * Allows to show the frame and define the selected text.
	 * @param start The position of the first selected character.
	 * @param end The position of the last selected character.
	 */
	public void show(int start, int end)
	{
		startPos = start;
		endPos = end;
		textField.select(start, end);
		buttonOk.setEnabled(start<end && start>=0);
		
		setVisible(true);
	}


	
	@Override
	public void paint(Graphics g)
	{
		textField.select(startPos, endPos);
		buttonOk.setEnabled(startPos<endPos && startPos>=0);
		super.paint(g);
	}
}
