/* $Id: kmo_fits_stack-test.c,v 1.4 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmclipm_functions.h"
#include "kmclipm_constants.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
const char* kmo_test_esorex_verbose();

const char      *path_recipe        = "kmo_fits_stack/";

const char      *valid_files[]      = {"v_vec1.fits",
                                       "v_vec2.fits",
                                       "v_vec3.fits",
                                       "ohspec.fits",
                                       "Ar_prediction_K.txt",
                                       "f2l.txt",
                                       "Ar_prediction_K_2.txt"};

// const char      *invalid_files[]    = {};

/**
    @defgroup kmo_fits_stack   kmo_fits_stack unit tests

    @{
*/

/**
 * @brief test with data and mask
 */
static void test_fits_stack(const char *file1,
                              const char *file2,
                              const char *file3,
                              int ret_val)
{
    // create sof-file
    const char *sof_path = "test_fits_stack.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, file1);
    fprintf (fh, "   STACK_DATA\n");
    fprintf (fh, file2);
    fprintf (fh, "   STACK_DATA\n");
    fprintf (fh, file3);
    fprintf (fh, "   STACK_DATA\n");
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE %s kmo_fits_stack --type=F1I %s",
            kmo_test_esorex_verbose(), sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_fits_stack.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
 * @brief stack a F1S
 */
static void test_fits_stack2(const char *file,
                             const char *type,
                             const char *format,
                             const char *title,
                             int ret_val)
{
    // create sof-file
    const char *sof_path = "test_fits_stack2.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   STACK_DATA\n", file);
    fclose(fh);

    // create esorex-command
    char esorex_command[512], tmp[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=%s ", type);
    if (strlen(format) > 0) {
        sprintf(tmp, " --format=%s ", format);
        strcat(esorex_command, tmp);
    }
    if (strlen(title) > 0) {
        sprintf(tmp, " --title=%s ",title);
        strcat(esorex_command, tmp);
    }
    strcat(esorex_command, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_fits_stack.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_fits_stack.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_fits_stack(const char *path)
{
    char        file_path[256];
    cpl_vector  *vec;
    /* ----- valid test data ----- */
    vec = cpl_vector_new(5);
    cpl_vector_fill(vec, 0.5);
    cpl_vector_save(vec, kmo_test_cat_strings(file_path, path, valid_files[0]),
                    CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);

    cpl_vector_fill(vec, 0.6);
    cpl_vector_save(vec, kmo_test_cat_strings(file_path, path, valid_files[1]),
                    CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);

    cpl_vector_fill(vec, 0.7);
    cpl_vector_save(vec, kmo_test_cat_strings(file_path, path, valid_files[2]),
                    CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_DEFAULT);


    /* ----- invalid test data ----- */
    /* none */
    cpl_vector_delete(vec);
    return 0;
}

/**
  @brief    Test of kmo_fits_stack recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_fits_stack-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_fits_stack-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            file_path[256],
            name[256],
            file1[256],
            file2[256],
            file3[256];

    cpl_propertylist *prl = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_fits_stack(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_fits_stack.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */
    // stack F1I
    kmo_test_cat_strings(file1, test_path, valid_files[0]);
    kmo_test_cat_strings(file2, test_path, valid_files[1]);
    kmo_test_cat_strings(file3, test_path, valid_files[2]);

    test_fits_stack(file1, file2, file3, 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 1.8, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "IFU.1.DATA"));
    cpl_propertylist_delete(prl);
    prl = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "IFU.2.DATA"));
    cpl_propertylist_delete(prl);

    // stack F1S
    char *test_global_path_ref_data = cpl_sprintf("%s/ref_data/", getenv("srcdir"));
    kmo_test_cat_strings(file1, test_global_path_ref_data, valid_files[3]);

    test_fits_stack2(file1, F1S, "", "", 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 1.29, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "SPEC"));
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, "CUNIT1"), "MICRON"));
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, "CTYPE1"), "WAVE"));
    cpl_test_abs(cpl_propertylist_get_float(prl, "CRVAL1"), 0.8, tol);
    cpl_propertylist_delete(prl);

    // stack F1L from ascii-file
    kmo_test_cat_strings(file1, test_global_path_ref_data, valid_files[6]);

    test_fits_stack2(file1, F1L, "\"%f;%f\"", "\"wavelength;strength\"", 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 883.872, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "LIST"));
    cpl_propertylist_delete(prl);

    cpl_test_eq(0, system("mv fits_stack.fits table_f1l.fits"));

    // stack F1L from binary FITS-table-file
    strcpy(file1, "table_f1l.fits");

    test_fits_stack2(file1, F1L, "", "", 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 883.872, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "LIST"));
    cpl_propertylist_delete(prl);

    // stack F2L from ascii-file
    kmo_test_cat_strings(file1, test_global_path_ref_data, valid_files[5]);

    test_fits_stack2(file1, F2L, "\"%f;%f;%f;%f\"", "\"A;B;C;D\"", 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 13.0033, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "LIST"));
    cpl_propertylist_delete(prl);

    cpl_test_eq(0, system("mv fits_stack.fits table_f2l.fits"));

    // stack F2L from binary FITS-table-file
    strcpy(file1, "table_f2l.fits");

    test_fits_stack2(file1, F2L, "", "", 0);
    cpl_test_abs(kmo_test_esorex_data(FITS_STACK, 0), 13.0033, tol);

    strcpy(name, FITS_STACK);
    kmo_strlower(name);
    kmo_test_cat_strings(file_path, name, ".fits");
    prl = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(0, strcmp(cpl_propertylist_get_string(prl, EXTNAME), "LIST"));

    cpl_propertylist_delete(prl);

    /* ----- invalid tests ----- */
    // stack F1L as F2L
    kmo_test_cat_strings(file1, test_global_path_ref_data, valid_files[4]);

    test_fits_stack2(file1, F2L, "\"%f;%f;%f;%f\"", "\"A;B;C;D\"", 1);

    // stack F2L as F1L
    kmo_test_cat_strings(file1, test_global_path_ref_data, valid_files[5]);

    test_fits_stack2(file1, F1L, "\"%f;%f\"", "\"A;B\"", 1);

    cpl_free(test_global_path_ref_data);

    return cpl_test_end(0);
}

/**@}*/
