//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ParticleItem.cpp
//! @brief     Implements class ParticleItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/ParticleItem.h"
#include "GUI/Model/Sample/FormFactorItems.h"
#include "GUI/Model/Sample/MaterialItem.h"
#include "Sample/Particle/IFormFactor.h"
#include "Sample/Particle/Particle.h"
#include "Sample/Scattering/Rotations.h"

namespace {
namespace Tag {

const QString FormFactor("FormFactor");
const QString MaterialData("MaterialData");
const QString BaseData("BaseData");
const QString ExpandParticleGroupbox("ExpandParticleGroupbox");

} // namespace Tag

const QString abundance_tooltip = "Proportion of this type of particles normalized to the \n"
                                  "total number of particles in the layout";

const QString position_tooltip = "Relative position of the particle's reference point \n"
                                 "in the coordinate system of the parent (nm)";

} // namespace

ParticleItem::ParticleItem(const MaterialModel* materials)
    : ItemWithMaterial(materials)
    , ItemWithParticles(abundance_tooltip, position_tooltip)
{
    m_formFactor.init("Form Factor", "");
    m_formFactor.setCurrentItem(new CylinderItem());
}

void ParticleItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // parameters from base class
    w->writeStartElement(Tag::BaseData);
    ItemWithParticles::writeTo(w);
    w->writeEndElement();

    // material data
    w->writeStartElement(Tag::MaterialData);
    ItemWithMaterial::writeTo(w);
    w->writeEndElement();

    // form factor
    w->writeStartElement(Tag::FormFactor);
    m_formFactor.writeTo(w);
    w->writeEndElement();

    // particle groupbox: is expanded?
    w->writeStartElement(Tag::ExpandParticleGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandParticle);
    w->writeEndElement();
}

void ParticleItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // parameters from base class
        if (tag == Tag::BaseData) {
            ItemWithParticles::readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // material data
        } else if (tag == Tag::MaterialData) {
            ItemWithMaterial::readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // form factor
        } else if (tag == Tag::FormFactor) {
            m_formFactor.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // particle groupbox: is expanded?
        } else if (tag == Tag::ExpandParticleGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandParticle);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

std::unique_ptr<Particle> ParticleItem::createParticle() const
{
    auto domainMaterial = materialItem()->createMaterial();

    auto particle = std::make_unique<Particle>(*domainMaterial, *m_formFactor->createFormFactor());
    particle->setAbundance(abundance());
    if (auto r = createRotation(); r && !r->isIdentity())
        particle->rotate(*r);
    particle->translate(position());

    return particle;
}

void ParticleItem::setFormFactor(FormFactorItem* p)
{
    m_formFactor.setCurrentItem(p);
}

FormFactorItem* ParticleItem::formFactorItem() const
{
    return m_formFactor.currentItem();
}

QVector<ItemWithParticles*> ParticleItem::containedItemsWithParticles() const
{
    return {};
}
