; JSON Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "JSON")

(include-book "abstract-syntax")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ parser-output-to-abstract-syntax
  :parents (json)
  :short "A translator
          from the JSON parser's output to the JSON abstract syntax."
  :long
  (xdoc::topstring
   (xdoc::p
    "The JSON parser in @('[books]/kestrel/json-parser/')
     produces output (parsed and abstracted JSON) in a form
     that is slightly different from the JSON abstract syntax
     formalized in this JSON library (see @(see abstract-syntax));
     in particular, the parser's output is not a fixtype.")
   (xdoc::p
    "Thus, here we provide a translator
     from the parser's output to the JSON abstract syntax.
     Since currently the parser does not include
     a type (i.e. recognizer) for its output,
     the translator is defined over all possible ACL2 values,
     but it returns an error if given something
     that does not belong to the parser's implicit output type."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define value-irrelevant ()
  :returns (value valuep)
  :short "An irrelevant JSON value."
  (with-guard-checking :none (ec-call (value-fix :irrelevant))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define member-irrelevant ()
  :returns (member memberp)
  :short "An irrelevant JSON member."
  (with-guard-checking :none (ec-call (member-fix :irrelevant))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defines parsed-to-value
  :short "Translate the JSON values generated by the parser
          to the corresponding fixtype JSON values."

  (define parsed-to-value (x)
    :returns (mv (erp booleanp)
                 (value valuep))
    (cond ((eq x :null) (mv nil (value-null)))
          ((eq x :true) (mv nil (value-true)))
          ((eq x :false) (mv nil (value-false)))
          ((stringp x) (mv nil (value-string x)))
          ((rationalp x) (mv nil (value-number x)))
          ((consp x)
           (cond ((and (eq (car x) :object)
                       (consp (cdr x))
                       (null (cddr x)))
                  (b* (((mv erp members) (parsed-to-member-list (cadr x))))
                    (if erp
                        (mv erp (value-irrelevant))
                      (mv nil (value-object members)))))
                 ((and (eq (car x) :array)
                       (consp (cdr x))
                       (null (cddr x)))
                  (b* (((mv erp values) (parsed-to-value-list (cadr x))))
                    (if erp
                        (mv erp (value-irrelevant))
                      (mv nil (value-array values)))))
                 (t (mv t (value-irrelevant)))))
          (t (mv t (value-irrelevant)))))

  (define parsed-to-value-list (x)
    :returns (mv (erp booleanp)
                 (values value-listp))
    (b* (((when (atom x)) (if (null x)
                              (mv nil nil)
                            (mv t nil)))
         ((mv erp value) (parsed-to-value (car x)))
         ((when erp) (mv erp nil))
         ((mv erp values) (parsed-to-value-list (cdr x)))
         ((when erp) (mv erp nil)))
      (mv nil (cons value values))))

  (define parsed-to-member (x)
    :returns (mv (erp booleanp)
                 (member memberp))
    (if (and (consp x)
             (stringp (car x)))
        (b* (((mv erp value) (parsed-to-value (cdr x))))
          (if erp
              (mv erp (member-irrelevant))
            (mv nil (member (car x) value))))
      (mv t (member-irrelevant))))

  (define parsed-to-member-list (x)
    :returns (mv (erp booleanp)
                 (members member-listp))
    (b* (((when (atom x)) (if (null x)
                              (mv nil nil)
                            (mv t nil)))
         ((mv erp member) (parsed-to-member (car x)))
         ((when erp) (mv erp nil))
         ((mv erp members) (parsed-to-member-list (cdr x)))
         ((when erp) (mv erp nil)))
      (mv nil (cons member members))))

  :verify-guards nil ; done below
  ///
  (verify-guards parsed-to-value))
