/* Mapping tables for CNS 11643, planes 2 to 7 handling.
   Copyright (C) 1998-2025 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <https://www.gnu.org/licenses/>.  */

#include <stdint.h>

/* To generate a Unicode 3.2 CNS11643.TXT, take from
   http://www.unicode.org/Public/Mappings/EASTASIA/OTHER/CNS11643.TXT
   the planes 1+2

     grep -v '^#' CNS11643.TXT | grep '^0x[12]' | sed -e 's/[ 	]*#.*$//'

   and merge with the extracted table

     grep '^U.*kIRG_TSource	' UNIHAN.TXT | sed -e 's,^U+\([0-9A-F][0-9A-F][0-9A-F][0-9A-F][0-9A-F]*\)	kIRG_TSource	\(.\)-\(....\)$,0x\2\3	0x\1,' | sed -e 's,^0x64627	0x2F958$,0x64267	0x2F958,' | sort

   using a "cat ... ... | sort | uniq" command.
   The following duplicate must be treated by hand:
       U+5344 = plane 1, 0x243F = plane 3, 0x2138
 */

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x2' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x22121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint16_t __cns11643l2_to_ucs4_tab[] =
{
  [0x0000] = 0x4e42, [0x0001] = 0x4e5c, [0x0002] = 0x51f5, [0x0003] = 0x531a,
  [0x0004] = 0x5382, [0x0005] = 0x4e07, [0x0006] = 0x4e0c, [0x0007] = 0x4e47,
  [0x0008] = 0x4e8d, [0x0009] = 0x56d7, [0x000a] = 0x5c6e, [0x000b] = 0x5f73,
  [0x000c] = 0x4e0f, [0x000d] = 0x5187, [0x000e] = 0x4e0e, [0x000f] = 0x4e2e,
  [0x0010] = 0x4e93, [0x0011] = 0x4ec2, [0x0012] = 0x4ec9, [0x0013] = 0x4ec8,
  [0x0014] = 0x5198, [0x0015] = 0x52fc, [0x0016] = 0x536c, [0x0017] = 0x53b9,
  [0x0018] = 0x5720, [0x0019] = 0x5903, [0x001a] = 0x592c, [0x001b] = 0x5c10,
  [0x001c] = 0x5dff, [0x001d] = 0x65e1, [0x001e] = 0x6bb3, [0x001f] = 0x6bcc,
  [0x0020] = 0x6c14, [0x0021] = 0x723f, [0x0022] = 0x4e31, [0x0023] = 0x4e3c,
  [0x0024] = 0x4ee8, [0x0025] = 0x4edc, [0x0026] = 0x4ee9, [0x0027] = 0x4ee1,
  [0x0028] = 0x4edd, [0x0029] = 0x4eda, [0x002a] = 0x520c, [0x002b] = 0x5209,
  [0x002c] = 0x531c, [0x002d] = 0x534c, [0x002e] = 0x5722, [0x002f] = 0x5723,
  [0x0030] = 0x5917, [0x0031] = 0x592f, [0x0032] = 0x5b81, [0x0033] = 0x5b84,
  [0x0034] = 0x5c12, [0x0035] = 0x5c3b, [0x0036] = 0x5c74, [0x0037] = 0x5c73,
  [0x0038] = 0x5e04, [0x0039] = 0x5e80, [0x003a] = 0x5e82, [0x003b] = 0x5fc9,
  [0x003c] = 0x6209, [0x003d] = 0x6250, [0x003e] = 0x6c15, [0x003f] = 0x6c36,
  [0x0040] = 0x6c43, [0x0041] = 0x6c3f, [0x0042] = 0x6c3b, [0x0043] = 0x72ae,
  [0x0044] = 0x72b0, [0x0045] = 0x738a, [0x0046] = 0x79b8, [0x0047] = 0x808a,
  [0x0048] = 0x961e, [0x0049] = 0x4f0e, [0x004a] = 0x4f18, [0x004b] = 0x4f2c,
  [0x004c] = 0x4ef5, [0x004d] = 0x4f14, [0x004e] = 0x4ef1, [0x004f] = 0x4f00,
  [0x0050] = 0x4ef7, [0x0051] = 0x4f08, [0x0052] = 0x4f1d, [0x0053] = 0x4f02,
  [0x0054] = 0x4f05, [0x0055] = 0x4f22, [0x0056] = 0x4f13, [0x0057] = 0x4f04,
  [0x0058] = 0x4ef4, [0x0059] = 0x4f12, [0x005a] = 0x51b1, [0x005b] = 0x5213,
  [0x005c] = 0x5210, [0x005d] = 0x52a6, [0x005e] = 0x5322, [0x005f] = 0x531f,
  [0x0060] = 0x534d, [0x0061] = 0x538a, [0x0062] = 0x5407, [0x0063] = 0x56e1,
  [0x0064] = 0x56df, [0x0065] = 0x572e, [0x0066] = 0x572a, [0x0067] = 0x5734,
  [0x0068] = 0x593c, [0x0069] = 0x5980, [0x006a] = 0x597c, [0x006b] = 0x5985,
  [0x006c] = 0x597b, [0x006d] = 0x597e, [0x006e] = 0x5977, [0x006f] = 0x597f,
  [0x0070] = 0x5b56, [0x0071] = 0x5c15, [0x0072] = 0x5c25, [0x0073] = 0x5c7c,
  [0x0074] = 0x5c7a, [0x0075] = 0x5c7b, [0x0076] = 0x5c7e, [0x0077] = 0x5ddf,
  [0x0078] = 0x5e75, [0x0079] = 0x5e84, [0x007a] = 0x5f02, [0x007b] = 0x5f1a,
  [0x007c] = 0x5f74, [0x007d] = 0x5fd5, [0x007e] = 0x5fd4, [0x007f] = 0x5fcf,
  [0x0080] = 0x625c, [0x0081] = 0x625e, [0x0082] = 0x6264, [0x0083] = 0x6261,
  [0x0084] = 0x6266, [0x0085] = 0x6262, [0x0086] = 0x6259, [0x0087] = 0x6260,
  [0x0088] = 0x625a, [0x0089] = 0x6265, [0x008a] = 0x6537, [0x008b] = 0x65ef,
  [0x008c] = 0x65ee, [0x008d] = 0x673e, [0x008e] = 0x6739, [0x008f] = 0x6738,
  [0x0090] = 0x673b, [0x0091] = 0x673a, [0x0092] = 0x673f, [0x0093] = 0x673c,
  [0x0094] = 0x6733, [0x0095] = 0x6c18, [0x0096] = 0x6c46, [0x0097] = 0x6c52,
  [0x0098] = 0x6c5c, [0x0099] = 0x6c4f, [0x009a] = 0x6c4a, [0x009b] = 0x6c54,
  [0x009c] = 0x6c4b, [0x009d] = 0x6c4c, [0x009e] = 0x7071, [0x009f] = 0x725e,
  [0x00a0] = 0x72b4, [0x00a1] = 0x72b5, [0x00a2] = 0x738e, [0x00a3] = 0x752a,
  [0x00a4] = 0x767f, [0x00a5] = 0x7a75, [0x00a6] = 0x7f51, [0x00a7] = 0x8278,
  [0x00a8] = 0x827c, [0x00a9] = 0x8280, [0x00aa] = 0x827d, [0x00ab] = 0x827f,
  [0x00ac] = 0x864d, [0x00ad] = 0x897e, [0x00ae] = 0x9099, [0x00af] = 0x9097,
  [0x00b0] = 0x9098, [0x00b1] = 0x909b, [0x00b2] = 0x9094, [0x00b3] = 0x9622,
  [0x00b4] = 0x9624, [0x00b5] = 0x9620, [0x00b6] = 0x9623, [0x00b7] = 0x4f56,
  [0x00b8] = 0x4f3b, [0x00b9] = 0x4f62, [0x00ba] = 0x4f49, [0x00bb] = 0x4f53,
  [0x00bc] = 0x4f64, [0x00bd] = 0x4f3e, [0x00be] = 0x4f67, [0x00bf] = 0x4f52,
  [0x00c0] = 0x4f5f, [0x00c1] = 0x4f41, [0x00c2] = 0x4f58, [0x00c3] = 0x4f2d,
  [0x00c4] = 0x4f33, [0x00c5] = 0x4f3f, [0x00c6] = 0x4f61, [0x00c7] = 0x518f,
  [0x00c8] = 0x51b9, [0x00c9] = 0x521c, [0x00ca] = 0x521e, [0x00cb] = 0x5221,
  [0x00cc] = 0x52ad, [0x00cd] = 0x52ae, [0x00ce] = 0x5309, [0x00cf] = 0x5363,
  [0x00d0] = 0x5372, [0x00d1] = 0x538e, [0x00d2] = 0x538f, [0x00d3] = 0x5430,
  [0x00d4] = 0x5437, [0x00d5] = 0x542a, [0x00d6] = 0x5454, [0x00d7] = 0x5445,
  [0x00d8] = 0x5419, [0x00d9] = 0x541c, [0x00da] = 0x5425, [0x00db] = 0x5418,
  [0x00dc] = 0x543d, [0x00dd] = 0x544f, [0x00de] = 0x5441, [0x00df] = 0x5428,
  [0x00e0] = 0x5424, [0x00e1] = 0x5447, [0x00e2] = 0x56ee, [0x00e3] = 0x56e7,
  [0x00e4] = 0x56e5, [0x00e5] = 0x5741, [0x00e6] = 0x5745, [0x00e7] = 0x574c,
  [0x00e8] = 0x5749, [0x00e9] = 0x574b, [0x00ea] = 0x5752, [0x00eb] = 0x5906,
  [0x00ec] = 0x5940, [0x00ed] = 0x59a6, [0x00ee] = 0x5998, [0x00ef] = 0x59a0,
  [0x00f0] = 0x5997, [0x00f1] = 0x598e, [0x00f2] = 0x59a2, [0x00f3] = 0x5990,
  [0x00f4] = 0x598f, [0x00f5] = 0x59a7, [0x00f6] = 0x59a1, [0x00f7] = 0x5b8e,
  [0x00f8] = 0x5b92, [0x00f9] = 0x5c28, [0x00fa] = 0x5c2a, [0x00fb] = 0x5c8d,
  [0x00fc] = 0x5c8f, [0x00fd] = 0x5c88, [0x00fe] = 0x5c8b, [0x00ff] = 0x5c89,
  [0x0100] = 0x5c92, [0x0101] = 0x5c8a, [0x0102] = 0x5c86, [0x0103] = 0x5c93,
  [0x0104] = 0x5c95, [0x0105] = 0x5de0, [0x0106] = 0x5e0a, [0x0107] = 0x5e0e,
  [0x0108] = 0x5e8b, [0x0109] = 0x5e89, [0x010a] = 0x5e8c, [0x010b] = 0x5e88,
  [0x010c] = 0x5e8d, [0x010d] = 0x5f05, [0x010e] = 0x5f1d, [0x010f] = 0x5f78,
  [0x0110] = 0x5f76, [0x0111] = 0x5fd2, [0x0112] = 0x5fd1, [0x0113] = 0x5fd0,
  [0x0114] = 0x5fed, [0x0115] = 0x5fe8, [0x0116] = 0x5fee, [0x0117] = 0x5ff3,
  [0x0118] = 0x5fe1, [0x0119] = 0x5fe4, [0x011a] = 0x5fe3, [0x011b] = 0x5ffa,
  [0x011c] = 0x5fef, [0x011d] = 0x5ff7, [0x011e] = 0x5ffb, [0x011f] = 0x6000,
  [0x0120] = 0x5ff4, [0x0121] = 0x623a, [0x0122] = 0x6283, [0x0123] = 0x628c,
  [0x0124] = 0x628e, [0x0125] = 0x628f, [0x0126] = 0x6294, [0x0127] = 0x6287,
  [0x0128] = 0x6271, [0x0129] = 0x627b, [0x012a] = 0x627a, [0x012b] = 0x6270,
  [0x012c] = 0x6281, [0x012d] = 0x6288, [0x012e] = 0x6277, [0x012f] = 0x627d,
  [0x0130] = 0x6272, [0x0131] = 0x6274, [0x0132] = 0x65f0, [0x0133] = 0x65f4,
  [0x0134] = 0x65f3, [0x0135] = 0x65f2, [0x0136] = 0x65f5, [0x0137] = 0x6745,
  [0x0138] = 0x6747, [0x0139] = 0x6759, [0x013a] = 0x6755, [0x013b] = 0x674c,
  [0x013c] = 0x6748, [0x013d] = 0x675d, [0x013e] = 0x674d, [0x013f] = 0x675a,
  [0x0140] = 0x674b, [0x0141] = 0x6bd0, [0x0142] = 0x6c19, [0x0143] = 0x6c1a,
  [0x0144] = 0x6c78, [0x0145] = 0x6c67, [0x0146] = 0x6c6b, [0x0147] = 0x6c84,
  [0x0148] = 0x6c8b, [0x0149] = 0x6c8f, [0x014a] = 0x6c71, [0x014b] = 0x6c6f,
  [0x014c] = 0x6c69, [0x014d] = 0x6c9a, [0x014e] = 0x6c6d, [0x014f] = 0x6c87,
  [0x0150] = 0x6c95, [0x0151] = 0x6c9c, [0x0152] = 0x6c66, [0x0153] = 0x6c73,
  [0x0154] = 0x6c65, [0x0155] = 0x6c7b, [0x0156] = 0x6c8e, [0x0157] = 0x7074,
  [0x0158] = 0x707a, [0x0159] = 0x7263, [0x015a] = 0x72bf, [0x015b] = 0x72bd,
  [0x015c] = 0x72c3, [0x015d] = 0x72c6, [0x015e] = 0x72c1, [0x015f] = 0x72ba,
  [0x0160] = 0x72c5, [0x0161] = 0x7395, [0x0162] = 0x7397, [0x0163] = 0x7393,
  [0x0164] = 0x7394, [0x0165] = 0x7392, [0x0166] = 0x753a, [0x0167] = 0x7539,
  [0x0168] = 0x7594, [0x0169] = 0x7595, [0x016a] = 0x7681, [0x016b] = 0x793d,
  [0x016c] = 0x8034, [0x016d] = 0x8095, [0x016e] = 0x8099, [0x016f] = 0x8090,
  [0x0170] = 0x8092, [0x0171] = 0x809c, [0x0172] = 0x8290, [0x0173] = 0x828f,
  [0x0174] = 0x8285, [0x0175] = 0x828e, [0x0176] = 0x8291, [0x0177] = 0x8293,
  [0x0178] = 0x828a, [0x0179] = 0x8283, [0x017a] = 0x8284, [0x017b] = 0x8c78,
  [0x017c] = 0x8fc9, [0x017d] = 0x8fbf, [0x017e] = 0x909f, [0x017f] = 0x90a1,
  [0x0180] = 0x90a5, [0x0181] = 0x909e, [0x0182] = 0x90a7, [0x0183] = 0x90a0,
  [0x0184] = 0x9630, [0x0185] = 0x9628, [0x0186] = 0x962f, [0x0187] = 0x962d,
  [0x0188] = 0x4e33, [0x0189] = 0x4f98, [0x018a] = 0x4f7c, [0x018b] = 0x4f85,
  [0x018c] = 0x4f7d, [0x018d] = 0x4f80, [0x018e] = 0x4f87, [0x018f] = 0x4f76,
  [0x0190] = 0x4f74, [0x0191] = 0x4f89, [0x0192] = 0x4f84, [0x0193] = 0x4f77,
  [0x0194] = 0x4f4c, [0x0195] = 0x4f97, [0x0196] = 0x4f6a, [0x0197] = 0x4f9a,
  [0x0198] = 0x4f79, [0x0199] = 0x4f81, [0x019a] = 0x4f78, [0x019b] = 0x4f90,
  [0x019c] = 0x4f9c, [0x019d] = 0x4f94, [0x019e] = 0x4f9e, [0x019f] = 0x4f92,
  [0x01a0] = 0x4f82, [0x01a1] = 0x4f95, [0x01a2] = 0x4f6b, [0x01a3] = 0x4f6e,
  [0x01a4] = 0x519e, [0x01a5] = 0x51bc, [0x01a6] = 0x51be, [0x01a7] = 0x5235,
  [0x01a8] = 0x5232, [0x01a9] = 0x5233, [0x01aa] = 0x5246, [0x01ab] = 0x5231,
  [0x01ac] = 0x52bc, [0x01ad] = 0x530a, [0x01ae] = 0x530b, [0x01af] = 0x533c,
  [0x01b0] = 0x5392, [0x01b1] = 0x5394, [0x01b2] = 0x5487, [0x01b3] = 0x547f,
  [0x01b4] = 0x5481, [0x01b5] = 0x5491, [0x01b6] = 0x5482, [0x01b7] = 0x5488,
  [0x01b8] = 0x546b, [0x01b9] = 0x547a, [0x01ba] = 0x547e, [0x01bb] = 0x5465,
  [0x01bc] = 0x546c, [0x01bd] = 0x5474, [0x01be] = 0x5466, [0x01bf] = 0x548d,
  [0x01c0] = 0x546f, [0x01c1] = 0x5461, [0x01c2] = 0x5460, [0x01c3] = 0x5498,
  [0x01c4] = 0x5463, [0x01c5] = 0x5467, [0x01c6] = 0x5464, [0x01c7] = 0x56f7,
  [0x01c8] = 0x56f9, [0x01c9] = 0x576f, [0x01ca] = 0x5772, [0x01cb] = 0x576d,
  [0x01cc] = 0x576b, [0x01cd] = 0x5771, [0x01ce] = 0x5770, [0x01cf] = 0x5776,
  [0x01d0] = 0x5780, [0x01d1] = 0x5775, [0x01d2] = 0x577b, [0x01d3] = 0x5773,
  [0x01d4] = 0x5774, [0x01d5] = 0x5762, [0x01d6] = 0x5768, [0x01d7] = 0x577d,
  [0x01d8] = 0x590c, [0x01d9] = 0x5945, [0x01da] = 0x59b5, [0x01db] = 0x59ba,
  [0x01dc] = 0x59cf, [0x01dd] = 0x59ce, [0x01de] = 0x59b2, [0x01df] = 0x59cc,
  [0x01e0] = 0x59c1, [0x01e1] = 0x59b6, [0x01e2] = 0x59bc, [0x01e3] = 0x59c3,
  [0x01e4] = 0x59d6, [0x01e5] = 0x59b1, [0x01e6] = 0x59bd, [0x01e7] = 0x59c0,
  [0x01e8] = 0x59c8, [0x01e9] = 0x59b4, [0x01ea] = 0x59c7, [0x01eb] = 0x5b62,
  [0x01ec] = 0x5b65, [0x01ed] = 0x5b93, [0x01ee] = 0x5b95, [0x01ef] = 0x5c44,
  [0x01f0] = 0x5c47, [0x01f1] = 0x5cae, [0x01f2] = 0x5ca4, [0x01f3] = 0x5ca0,
  [0x01f4] = 0x5cb5, [0x01f5] = 0x5caf, [0x01f6] = 0x5ca8, [0x01f7] = 0x5cac,
  [0x01f8] = 0x5c9f, [0x01f9] = 0x5ca3, [0x01fa] = 0x5cad, [0x01fb] = 0x5ca2,
  [0x01fc] = 0x5caa, [0x01fd] = 0x5ca7, [0x01fe] = 0x5c9d, [0x01ff] = 0x5ca5,
  [0x0200] = 0x5cb6, [0x0201] = 0x5cb0, [0x0202] = 0x5ca6, [0x0203] = 0x5e17,
  [0x0204] = 0x5e14, [0x0205] = 0x5e19, [0x0206] = 0x5f28, [0x0207] = 0x5f22,
  [0x0208] = 0x5f23, [0x0209] = 0x5f24, [0x020a] = 0x5f54, [0x020b] = 0x5f82,
  [0x020c] = 0x5f7e, [0x020d] = 0x5f7d, [0x020e] = 0x5fde, [0x020f] = 0x5fe5,
  [0x0210] = 0x602d, [0x0211] = 0x6026, [0x0212] = 0x6019, [0x0213] = 0x6032,
  [0x0214] = 0x600b, [0x0215] = 0x6034, [0x0216] = 0x600a, [0x0217] = 0x6017,
  [0x0218] = 0x6033, [0x0219] = 0x601a, [0x021a] = 0x601e, [0x021b] = 0x602c,
  [0x021c] = 0x6022, [0x021d] = 0x600d, [0x021e] = 0x6010, [0x021f] = 0x602e,
  [0x0220] = 0x6013, [0x0221] = 0x6011, [0x0222] = 0x600c, [0x0223] = 0x6009,
  [0x0224] = 0x601c, [0x0225] = 0x6214, [0x0226] = 0x623d, [0x0227] = 0x62ad,
  [0x0228] = 0x62b4, [0x0229] = 0x62d1, [0x022a] = 0x62be, [0x022b] = 0x62aa,
  [0x022c] = 0x62b6, [0x022d] = 0x62ca, [0x022e] = 0x62ae, [0x022f] = 0x62b3,
  [0x0230] = 0x62af, [0x0231] = 0x62bb, [0x0232] = 0x62a9, [0x0233] = 0x62b0,
  [0x0234] = 0x62b8, [0x0235] = 0x653d, [0x0236] = 0x65a8, [0x0237] = 0x65bb,
  [0x0238] = 0x6609, [0x0239] = 0x65fc, [0x023a] = 0x6604, [0x023b] = 0x6612,
  [0x023c] = 0x6608, [0x023d] = 0x65fb, [0x023e] = 0x6603, [0x023f] = 0x660b,
  [0x0240] = 0x660d, [0x0241] = 0x6605, [0x0242] = 0x65fd, [0x0243] = 0x6611,
  [0x0244] = 0x6610, [0x0245] = 0x66f6, [0x0246] = 0x670a, [0x0247] = 0x6785,
  [0x0248] = 0x676c, [0x0249] = 0x678e, [0x024a] = 0x6792, [0x024b] = 0x6776,
  [0x024c] = 0x677b, [0x024d] = 0x6798, [0x024e] = 0x6786, [0x024f] = 0x6784,
  [0x0250] = 0x6774, [0x0251] = 0x678d, [0x0252] = 0x678c, [0x0253] = 0x677a,
  [0x0254] = 0x679f, [0x0255] = 0x6791, [0x0256] = 0x6799, [0x0257] = 0x6783,
  [0x0258] = 0x677d, [0x0259] = 0x6781, [0x025a] = 0x6778, [0x025b] = 0x6779,
  [0x025c] = 0x6794, [0x025d] = 0x6b25, [0x025e] = 0x6b80, [0x025f] = 0x6b7e,
  [0x0260] = 0x6bde, [0x0261] = 0x6c1d, [0x0262] = 0x6c93, [0x0263] = 0x6cec,
  [0x0264] = 0x6ceb, [0x0265] = 0x6cee, [0x0266] = 0x6cd9, [0x0267] = 0x6cb6,
  [0x0268] = 0x6cd4, [0x0269] = 0x6cad, [0x026a] = 0x6ce7, [0x026b] = 0x6cb7,
  [0x026c] = 0x6cd0, [0x026d] = 0x6cc2, [0x026e] = 0x6cba, [0x026f] = 0x6cc3,
  [0x0270] = 0x6cc6, [0x0271] = 0x6ced, [0x0272] = 0x6cf2, [0x0273] = 0x6cd2,
  [0x0274] = 0x6cdd, [0x0275] = 0x6cb4, [0x0276] = 0x6c8a, [0x0277] = 0x6c9d,
  [0x0278] = 0x6c80, [0x0279] = 0x6cde, [0x027a] = 0x6cc0, [0x027b] = 0x6d30,
  [0x027c] = 0x6ccd, [0x027d] = 0x6cc7, [0x027e] = 0x6cb0, [0x027f] = 0x6cf9,
  [0x0280] = 0x6ccf, [0x0281] = 0x6ce9, [0x0282] = 0x6cd1, [0x0283] = 0x7094,
  [0x0284] = 0x7098, [0x0285] = 0x7085, [0x0286] = 0x7093, [0x0287] = 0x7086,
  [0x0288] = 0x7084, [0x0289] = 0x7091, [0x028a] = 0x7096, [0x028b] = 0x7082,
  [0x028c] = 0x709a, [0x028d] = 0x7083, [0x028e] = 0x726a, [0x028f] = 0x72d6,
  [0x0290] = 0x72cb, [0x0291] = 0x72d8, [0x0292] = 0x72c9, [0x0293] = 0x72dc,
  [0x0294] = 0x72d2, [0x0295] = 0x72d4, [0x0296] = 0x72da, [0x0297] = 0x72cc,
  [0x0298] = 0x72d1, [0x0299] = 0x73a4, [0x029a] = 0x73a1, [0x029b] = 0x73ad,
  [0x029c] = 0x73a6, [0x029d] = 0x73a2, [0x029e] = 0x73a0, [0x029f] = 0x73ac,
  [0x02a0] = 0x739d, [0x02a1] = 0x74dd, [0x02a2] = 0x74e8, [0x02a3] = 0x753f,
  [0x02a4] = 0x7540, [0x02a5] = 0x753e, [0x02a6] = 0x758c, [0x02a7] = 0x7598,
  [0x02a8] = 0x76af, [0x02a9] = 0x76f3, [0x02aa] = 0x76f1, [0x02ab] = 0x76f0,
  [0x02ac] = 0x76f5, [0x02ad] = 0x77f8, [0x02ae] = 0x77fc, [0x02af] = 0x77f9,
  [0x02b0] = 0x77fb, [0x02b1] = 0x77fa, [0x02b2] = 0x77f7, [0x02b3] = 0x7942,
  [0x02b4] = 0x793f, [0x02b5] = 0x79c5, [0x02b6] = 0x7a78, [0x02b7] = 0x7a7b,
  [0x02b8] = 0x7afb, [0x02b9] = 0x7c75, [0x02ba] = 0x7cfd, [0x02bb] = 0x8035,
  [0x02bc] = 0x808f, [0x02bd] = 0x80ae, [0x02be] = 0x80a3, [0x02bf] = 0x80b8,
  [0x02c0] = 0x80b5, [0x02c1] = 0x80ad, [0x02c2] = 0x8220, [0x02c3] = 0x82a0,
  [0x02c4] = 0x82c0, [0x02c5] = 0x82ab, [0x02c6] = 0x829a, [0x02c7] = 0x8298,
  [0x02c8] = 0x829b, [0x02c9] = 0x82b5, [0x02ca] = 0x82a7, [0x02cb] = 0x82ae,
  [0x02cc] = 0x82bc, [0x02cd] = 0x829e, [0x02ce] = 0x82ba, [0x02cf] = 0x82b4,
  [0x02d0] = 0x82a8, [0x02d1] = 0x82a1, [0x02d2] = 0x82a9, [0x02d3] = 0x82c2,
  [0x02d4] = 0x82a4, [0x02d5] = 0x82c3, [0x02d6] = 0x82b6, [0x02d7] = 0x82a2,
  [0x02d8] = 0x8670, [0x02d9] = 0x866f, [0x02da] = 0x866d, [0x02db] = 0x866e,
  [0x02dc] = 0x8c56, [0x02dd] = 0x8fd2, [0x02de] = 0x8fcb, [0x02df] = 0x8fd3,
  [0x02e0] = 0x8fcd, [0x02e1] = 0x8fd6, [0x02e2] = 0x8fd5, [0x02e3] = 0x8fd7,
  [0x02e4] = 0x90b2, [0x02e5] = 0x90b4, [0x02e6] = 0x90af, [0x02e7] = 0x90b3,
  [0x02e8] = 0x90b0, [0x02e9] = 0x9639, [0x02ea] = 0x963d, [0x02eb] = 0x963c,
  [0x02ec] = 0x963a, [0x02ed] = 0x9643, [0x02ee] = 0x4fcd, [0x02ef] = 0x4fc5,
  [0x02f0] = 0x4fd3, [0x02f1] = 0x4fb2, [0x02f2] = 0x4fc9, [0x02f3] = 0x4fcb,
  [0x02f4] = 0x4fc1, [0x02f5] = 0x4fd4, [0x02f6] = 0x4fdc, [0x02f7] = 0x4fd9,
  [0x02f8] = 0x4fbb, [0x02f9] = 0x4fb3, [0x02fa] = 0x4fdb, [0x02fb] = 0x4fc7,
  [0x02fc] = 0x4fd6, [0x02fd] = 0x4fba, [0x02fe] = 0x4fc0, [0x02ff] = 0x4fb9,
  [0x0300] = 0x4fec, [0x0301] = 0x5244, [0x0302] = 0x5249, [0x0303] = 0x52c0,
  [0x0304] = 0x52c2, [0x0305] = 0x533d, [0x0306] = 0x537c, [0x0307] = 0x5397,
  [0x0308] = 0x5396, [0x0309] = 0x5399, [0x030a] = 0x5398, [0x030b] = 0x54ba,
  [0x030c] = 0x54a1, [0x030d] = 0x54ad, [0x030e] = 0x54a5, [0x030f] = 0x54cf,
  [0x0310] = 0x54c3, [0x0311] = 0x830d, [0x0312] = 0x54b7, [0x0313] = 0x54ae,
  [0x0314] = 0x54d6, [0x0315] = 0x54b6, [0x0316] = 0x54c5, [0x0317] = 0x54c6,
  [0x0318] = 0x54a0, [0x0319] = 0x5470, [0x031a] = 0x54bc, [0x031b] = 0x54a2,
  [0x031c] = 0x54be, [0x031d] = 0x5472, [0x031e] = 0x54de, [0x031f] = 0x54b0,
  [0x0320] = 0x57b5, [0x0321] = 0x579e, [0x0322] = 0x579f, [0x0323] = 0x57a4,
  [0x0324] = 0x578c, [0x0325] = 0x5797, [0x0326] = 0x579d, [0x0327] = 0x579b,
  [0x0328] = 0x5794, [0x0329] = 0x5798, [0x032a] = 0x578f, [0x032b] = 0x5799,
  [0x032c] = 0x57a5, [0x032d] = 0x579a, [0x032e] = 0x5795, [0x032f] = 0x58f4,
  [0x0330] = 0x590d, [0x0331] = 0x5953, [0x0332] = 0x59e1, [0x0333] = 0x59de,
  [0x0334] = 0x59ee, [0x0335] = 0x5a00, [0x0336] = 0x59f1, [0x0337] = 0x59dd,
  [0x0338] = 0x59fa, [0x0339] = 0x59fd, [0x033a] = 0x59fc, [0x033b] = 0x59f6,
  [0x033c] = 0x59e4, [0x033d] = 0x59f2, [0x033e] = 0x59f7, [0x033f] = 0x59db,
  [0x0340] = 0x59e9, [0x0341] = 0x59f3, [0x0342] = 0x59f5, [0x0343] = 0x59e0,
  [0x0344] = 0x59fe, [0x0345] = 0x59f4, [0x0346] = 0x59ed, [0x0347] = 0x5ba8,
  [0x0348] = 0x5c4c, [0x0349] = 0x5cd0, [0x034a] = 0x5cd8, [0x034b] = 0x5ccc,
  [0x034c] = 0x5cd7, [0x034d] = 0x5ccb, [0x034e] = 0x5cdb, [0x034f] = 0x5cde,
  [0x0350] = 0x5cda, [0x0351] = 0x5cc9, [0x0352] = 0x5cc7, [0x0353] = 0x5cca,
  [0x0354] = 0x5cd6, [0x0355] = 0x5cd3, [0x0356] = 0x5cd4, [0x0357] = 0x5ccf,
  [0x0358] = 0x5cc8, [0x0359] = 0x5cc6, [0x035a] = 0x5cce, [0x035b] = 0x5cdf,
  [0x035c] = 0x5cf8, [0x035d] = 0x5df9, [0x035e] = 0x5e21, [0x035f] = 0x5e22,
  [0x0360] = 0x5e23, [0x0361] = 0x5e20, [0x0362] = 0x5e24, [0x0363] = 0x5eb0,
  [0x0364] = 0x5ea4, [0x0365] = 0x5ea2, [0x0366] = 0x5e9b, [0x0367] = 0x5ea3,
  [0x0368] = 0x5ea5, [0x0369] = 0x5f07, [0x036a] = 0x5f2e, [0x036b] = 0x5f56,
  [0x036c] = 0x5f86, [0x036d] = 0x6037, [0x036e] = 0x6039, [0x036f] = 0x6054,
  [0x0370] = 0x6072, [0x0371] = 0x605e, [0x0372] = 0x6045, [0x0373] = 0x6053,
  [0x0374] = 0x6047, [0x0375] = 0x6049, [0x0376] = 0x605b, [0x0377] = 0x604c,
  [0x0378] = 0x6040, [0x0379] = 0x6042, [0x037a] = 0x605f, [0x037b] = 0x6024,
  [0x037c] = 0x6044, [0x037d] = 0x6058, [0x037e] = 0x6066, [0x037f] = 0x606e,
  [0x0380] = 0x6242, [0x0381] = 0x6243, [0x0382] = 0x62cf, [0x0383] = 0x630d,
  [0x0384] = 0x630b, [0x0385] = 0x62f5, [0x0386] = 0x630e, [0x0387] = 0x6303,
  [0x0388] = 0x62eb, [0x0389] = 0x62f9, [0x038a] = 0x630f, [0x038b] = 0x630c,
  [0x038c] = 0x62f8, [0x038d] = 0x62f6, [0x038e] = 0x6300, [0x038f] = 0x6313,
  [0x0390] = 0x6314, [0x0391] = 0x62fa, [0x0392] = 0x6315, [0x0393] = 0x62fb,
  [0x0394] = 0x62f0, [0x0395] = 0x6541, [0x0396] = 0x6543, [0x0397] = 0x65aa,
  [0x0398] = 0x65bf, [0x0399] = 0x6636, [0x039a] = 0x6621, [0x039b] = 0x6632,
  [0x039c] = 0x6635, [0x039d] = 0x661c, [0x039e] = 0x6626, [0x039f] = 0x6622,
  [0x03a0] = 0x6633, [0x03a1] = 0x662b, [0x03a2] = 0x663a, [0x03a3] = 0x661d,
  [0x03a4] = 0x6634, [0x03a5] = 0x6639, [0x03a6] = 0x662e, [0x03a7] = 0x670f,
  [0x03a8] = 0x6710, [0x03a9] = 0x67c1, [0x03aa] = 0x67f2, [0x03ab] = 0x67c8,
  [0x03ac] = 0x67ba, [0x03ad] = 0x67dc, [0x03ae] = 0x67bb, [0x03af] = 0x67f8,
  [0x03b0] = 0x67d8, [0x03b1] = 0x67c0, [0x03b2] = 0x67b7, [0x03b3] = 0x67c5,
  [0x03b4] = 0x67eb, [0x03b5] = 0x67e4, [0x03b6] = 0x67df, [0x03b7] = 0x67b5,
  [0x03b8] = 0x67cd, [0x03b9] = 0x67b3, [0x03ba] = 0x67f7, [0x03bb] = 0x67f6,
  [0x03bc] = 0x67ee, [0x03bd] = 0x67e3, [0x03be] = 0x67c2, [0x03bf] = 0x67b9,
  [0x03c0] = 0x67ce, [0x03c1] = 0x67e7, [0x03c2] = 0x67f0, [0x03c3] = 0x67b2,
  [0x03c4] = 0x67fc, [0x03c5] = 0x67c6, [0x03c6] = 0x67ed, [0x03c7] = 0x67cc,
  [0x03c8] = 0x67ae, [0x03c9] = 0x67e6, [0x03ca] = 0x67db, [0x03cb] = 0x67fa,
  [0x03cc] = 0x67c9, [0x03cd] = 0x67ca, [0x03ce] = 0x67c3, [0x03cf] = 0x67ea,
  [0x03d0] = 0x67cb, [0x03d1] = 0x6b28, [0x03d2] = 0x6b82, [0x03d3] = 0x6b84,
  [0x03d4] = 0x6bb6, [0x03d5] = 0x6bd6, [0x03d6] = 0x6bd8, [0x03d7] = 0x6be0,
  [0x03d8] = 0x6c20, [0x03d9] = 0x6c21, [0x03da] = 0x6d28, [0x03db] = 0x6d34,
  [0x03dc] = 0x6d2d, [0x03dd] = 0x6d1f, [0x03de] = 0x6d3c, [0x03df] = 0x6d3f,
  [0x03e0] = 0x6d12, [0x03e1] = 0x6d0a, [0x03e2] = 0x6cda, [0x03e3] = 0x6d33,
  [0x03e4] = 0x6d04, [0x03e5] = 0x6d19, [0x03e6] = 0x6d3a, [0x03e7] = 0x6d1a,
  [0x03e8] = 0x6d11, [0x03e9] = 0x6d00, [0x03ea] = 0x6d1d, [0x03eb] = 0x6d42,
  [0x03ec] = 0x6d01, [0x03ed] = 0x6d18, [0x03ee] = 0x6d37, [0x03ef] = 0x6d03,
  [0x03f0] = 0x6d0f, [0x03f1] = 0x6d40, [0x03f2] = 0x6d07, [0x03f3] = 0x6d20,
  [0x03f4] = 0x6d2c, [0x03f5] = 0x6d08, [0x03f6] = 0x6d22, [0x03f7] = 0x6d09,
  [0x03f8] = 0x6d10, [0x03f9] = 0x70b7, [0x03fa] = 0x709f, [0x03fb] = 0x70be,
  [0x03fc] = 0x70b1, [0x03fd] = 0x70b0, [0x03fe] = 0x70a1, [0x03ff] = 0x70b4,
  [0x0400] = 0x70b5, [0x0401] = 0x70a9, [0x0402] = 0x7241, [0x0403] = 0x7249,
  [0x0404] = 0x724a, [0x0405] = 0x726c, [0x0406] = 0x7270, [0x0407] = 0x7273,
  [0x0408] = 0x726e, [0x0409] = 0x72ca, [0x040a] = 0x72e4, [0x040b] = 0x72e8,
  [0x040c] = 0x72eb, [0x040d] = 0x72df, [0x040e] = 0x72ea, [0x040f] = 0x72e6,
  [0x0410] = 0x72e3, [0x0411] = 0x7385, [0x0412] = 0x73cc, [0x0413] = 0x73c2,
  [0x0414] = 0x73c8, [0x0415] = 0x73c5, [0x0416] = 0x73b9, [0x0417] = 0x73b6,
  [0x0418] = 0x73b5, [0x0419] = 0x73b4, [0x041a] = 0x73eb, [0x041b] = 0x73bf,
  [0x041c] = 0x73c7, [0x041d] = 0x73be, [0x041e] = 0x73c3, [0x041f] = 0x73c6,
  [0x0420] = 0x73b8, [0x0421] = 0x73cb, [0x0422] = 0x74ec, [0x0423] = 0x74ee,
  [0x0424] = 0x752e, [0x0425] = 0x7547, [0x0426] = 0x7548, [0x0427] = 0x75a7,
  [0x0428] = 0x75aa, [0x0429] = 0x7679, [0x042a] = 0x76c4, [0x042b] = 0x7708,
  [0x042c] = 0x7703, [0x042d] = 0x7704, [0x042e] = 0x7705, [0x042f] = 0x770a,
  [0x0430] = 0x76f7, [0x0431] = 0x76fb, [0x0432] = 0x76fa, [0x0433] = 0x77e7,
  [0x0434] = 0x77e8, [0x0435] = 0x7806, [0x0436] = 0x7811, [0x0437] = 0x7812,
  [0x0438] = 0x7805, [0x0439] = 0x7810, [0x043a] = 0x780f, [0x043b] = 0x780e,
  [0x043c] = 0x7809, [0x043d] = 0x7803, [0x043e] = 0x7813, [0x043f] = 0x794a,
  [0x0440] = 0x794c, [0x0441] = 0x794b, [0x0442] = 0x7945, [0x0443] = 0x7944,
  [0x0444] = 0x79d5, [0x0445] = 0x79cd, [0x0446] = 0x79cf, [0x0447] = 0x79d6,
  [0x0448] = 0x79ce, [0x0449] = 0x7a80, [0x044a] = 0x7a7e, [0x044b] = 0x7ad1,
  [0x044c] = 0x7b00, [0x044d] = 0x7b01, [0x044e] = 0x7c7a, [0x044f] = 0x7c78,
  [0x0450] = 0x7c79, [0x0451] = 0x7c7f, [0x0452] = 0x7c80, [0x0453] = 0x7c81,
  [0x0454] = 0x7d03, [0x0455] = 0x7d08, [0x0456] = 0x7d01, [0x0457] = 0x7f58,
  [0x0458] = 0x7f91, [0x0459] = 0x7f8d, [0x045a] = 0x7fbe, [0x045b] = 0x8007,
  [0x045c] = 0x800e, [0x045d] = 0x800f, [0x045e] = 0x8014, [0x045f] = 0x8037,
  [0x0460] = 0x80d8, [0x0461] = 0x80c7, [0x0462] = 0x80e0, [0x0463] = 0x80d1,
  [0x0464] = 0x80c8, [0x0465] = 0x80c2, [0x0466] = 0x80d0, [0x0467] = 0x80c5,
  [0x0468] = 0x80e3, [0x0469] = 0x80d9, [0x046a] = 0x80dc, [0x046b] = 0x80ca,
  [0x046c] = 0x80d5, [0x046d] = 0x80c9, [0x046e] = 0x80cf, [0x046f] = 0x80d7,
  [0x0470] = 0x80e6, [0x0471] = 0x80cd, [0x0472] = 0x81ff, [0x0473] = 0x8221,
  [0x0474] = 0x8294, [0x0475] = 0x82d9, [0x0476] = 0x82fe, [0x0477] = 0x82f9,
  [0x0478] = 0x8307, [0x0479] = 0x82e8, [0x047a] = 0x8300, [0x047b] = 0x82d5,
  [0x047c] = 0x833a, [0x047d] = 0x82eb, [0x047e] = 0x82d6, [0x047f] = 0x82f4,
  [0x0480] = 0x82ec, [0x0481] = 0x82e1, [0x0482] = 0x82f2, [0x0483] = 0x82f5,
  [0x0484] = 0x830c, [0x0485] = 0x82fb, [0x0486] = 0x82f6, [0x0487] = 0x82f0,
  [0x0488] = 0x82ea, [0x0489] = 0x82e4, [0x048a] = 0x82e0, [0x048b] = 0x82fa,
  [0x048c] = 0x82f3, [0x048d] = 0x82ed, [0x048e] = 0x8677, [0x048f] = 0x8674,
  [0x0490] = 0x867c, [0x0491] = 0x8673, [0x0492] = 0x8841, [0x0493] = 0x884e,
  [0x0494] = 0x8867, [0x0495] = 0x886a, [0x0496] = 0x8869, [0x0497] = 0x89d3,
  [0x0498] = 0x8a04, [0x0499] = 0x8a07, [0x049a] = 0x8d72, [0x049b] = 0x8fe3,
  [0x049c] = 0x8fe1, [0x049d] = 0x8fee, [0x049e] = 0x8fe0, [0x049f] = 0x90f1,
  [0x04a0] = 0x90bd, [0x04a1] = 0x90bf, [0x04a2] = 0x90d5, [0x04a3] = 0x90c5,
  [0x04a4] = 0x90be, [0x04a5] = 0x90c7, [0x04a6] = 0x90cb, [0x04a7] = 0x90c8,
  [0x04a8] = 0x91d4, [0x04a9] = 0x91d3, [0x04aa] = 0x9654, [0x04ab] = 0x964f,
  [0x04ac] = 0x9651, [0x04ad] = 0x9653, [0x04ae] = 0x964a, [0x04af] = 0x964e,
  [0x04b0] = 0x501e, [0x04b1] = 0x5005, [0x04b2] = 0x5007, [0x04b3] = 0x5013,
  [0x04b4] = 0x5022, [0x04b5] = 0x5030, [0x04b6] = 0x501b, [0x04b7] = 0x4ff5,
  [0x04b8] = 0x4ff4, [0x04b9] = 0x5033, [0x04ba] = 0x5037, [0x04bb] = 0x502c,
  [0x04bc] = 0x4ff6, [0x04bd] = 0x4ff7, [0x04be] = 0x5017, [0x04bf] = 0x501c,
  [0x04c0] = 0x5020, [0x04c1] = 0x5027, [0x04c2] = 0x5035, [0x04c3] = 0x502f,
  [0x04c4] = 0x5031, [0x04c5] = 0x500e, [0x04c6] = 0x515a, [0x04c7] = 0x5194,
  [0x04c8] = 0x5193, [0x04c9] = 0x51ca, [0x04ca] = 0x51c4, [0x04cb] = 0x51c5,
  [0x04cc] = 0x51c8, [0x04cd] = 0x51ce, [0x04ce] = 0x5261, [0x04cf] = 0x525a,
  [0x04d0] = 0x5252, [0x04d1] = 0x525e, [0x04d2] = 0x525f, [0x04d3] = 0x5255,
  [0x04d4] = 0x5262, [0x04d5] = 0x52cd, [0x04d6] = 0x530e, [0x04d7] = 0x539e,
  [0x04d8] = 0x5526, [0x04d9] = 0x54e2, [0x04da] = 0x5517, [0x04db] = 0x5512,
  [0x04dc] = 0x54e7, [0x04dd] = 0x54f3, [0x04de] = 0x54e4, [0x04df] = 0x551a,
  [0x04e0] = 0x54ff, [0x04e1] = 0x5504, [0x04e2] = 0x5508, [0x04e3] = 0x54eb,
  [0x04e4] = 0x5511, [0x04e5] = 0x5505, [0x04e6] = 0x54f1, [0x04e7] = 0x550a,
  [0x04e8] = 0x54fb, [0x04e9] = 0x54f7, [0x04ea] = 0x54f8, [0x04eb] = 0x54e0,
  [0x04ec] = 0x550e, [0x04ed] = 0x5503, [0x04ee] = 0x550b, [0x04ef] = 0x5701,
  [0x04f0] = 0x5702, [0x04f1] = 0x57cc, [0x04f2] = 0x5832, [0x04f3] = 0x57d5,
  [0x04f4] = 0x57d2, [0x04f5] = 0x57ba, [0x04f6] = 0x57c6, [0x04f7] = 0x57bd,
  [0x04f8] = 0x57bc, [0x04f9] = 0x57b8, [0x04fa] = 0x57b6, [0x04fb] = 0x57bf,
  [0x04fc] = 0x57c7, [0x04fd] = 0x57d0, [0x04fe] = 0x57b9, [0x04ff] = 0x57c1,
  [0x0500] = 0x590e, [0x0501] = 0x594a, [0x0502] = 0x5a19, [0x0503] = 0x5a16,
  [0x0504] = 0x5a2d, [0x0505] = 0x5a2e, [0x0506] = 0x5a15, [0x0507] = 0x5a0f,
  [0x0508] = 0x5a17, [0x0509] = 0x5a0a, [0x050a] = 0x5a1e, [0x050b] = 0x5a33,
  [0x050c] = 0x5b6c, [0x050d] = 0x5ba7, [0x050e] = 0x5bad, [0x050f] = 0x5bac,
  [0x0510] = 0x5c03, [0x0511] = 0x5c56, [0x0512] = 0x5c54, [0x0513] = 0x5cec,
  [0x0514] = 0x5cff, [0x0515] = 0x5cee, [0x0516] = 0x5cf1, [0x0517] = 0x5cf7,
  [0x0518] = 0x5d00, [0x0519] = 0x5cf9, [0x051a] = 0x5e29, [0x051b] = 0x5e28,
  [0x051c] = 0x5ea8, [0x051d] = 0x5eae, [0x051e] = 0x5eaa, [0x051f] = 0x5eac,
  [0x0520] = 0x5f33, [0x0521] = 0x5f30, [0x0522] = 0x5f67, [0x0523] = 0x605d,
  [0x0524] = 0x605a, [0x0525] = 0x6067, [0x0526] = 0x6041, [0x0527] = 0x60a2,
  [0x0528] = 0x6088, [0x0529] = 0x6080, [0x052a] = 0x6092, [0x052b] = 0x6081,
  [0x052c] = 0x609d, [0x052d] = 0x6083, [0x052e] = 0x6095, [0x052f] = 0x609b,
  [0x0530] = 0x6097, [0x0531] = 0x6087, [0x0532] = 0x609c, [0x0533] = 0x608e,
  [0x0534] = 0x6219, [0x0535] = 0x6246, [0x0536] = 0x62f2, [0x0537] = 0x6310,
  [0x0538] = 0x6356, [0x0539] = 0x632c, [0x053a] = 0x6344, [0x053b] = 0x6345,
  [0x053c] = 0x6336, [0x053d] = 0x6343, [0x053e] = 0x63e4, [0x053f] = 0x6339,
  [0x0540] = 0x634b, [0x0541] = 0x634a, [0x0542] = 0x633c, [0x0543] = 0x6329,
  [0x0544] = 0x6341, [0x0545] = 0x6334, [0x0546] = 0x6358, [0x0547] = 0x6354,
  [0x0548] = 0x6359, [0x0549] = 0x632d, [0x054a] = 0x6347, [0x054b] = 0x6333,
  [0x054c] = 0x635a, [0x054d] = 0x6351, [0x054e] = 0x6338, [0x054f] = 0x6357,
  [0x0550] = 0x6340, [0x0551] = 0x6348, [0x0552] = 0x654a, [0x0553] = 0x6546,
  [0x0554] = 0x65c6, [0x0555] = 0x65c3, [0x0556] = 0x65c4, [0x0557] = 0x65c2,
  [0x0558] = 0x664a, [0x0559] = 0x665f, [0x055a] = 0x6647, [0x055b] = 0x6651,
  [0x055c] = 0x6712, [0x055d] = 0x6713, [0x055e] = 0x681f, [0x055f] = 0x681a,
  [0x0560] = 0x6849, [0x0561] = 0x6832, [0x0562] = 0x6833, [0x0563] = 0x683b,
  [0x0564] = 0x684b, [0x0565] = 0x684f, [0x0566] = 0x6816, [0x0567] = 0x6831,
  [0x0568] = 0x681c, [0x0569] = 0x6835, [0x056a] = 0x682b, [0x056b] = 0x682d,
  [0x056c] = 0x682f, [0x056d] = 0x684e, [0x056e] = 0x6844, [0x056f] = 0x6834,
  [0x0570] = 0x681d, [0x0571] = 0x6812, [0x0572] = 0x6814, [0x0573] = 0x6826,
  [0x0574] = 0x6828, [0x0575] = 0x682e, [0x0576] = 0x684d, [0x0577] = 0x683a,
  [0x0578] = 0x6825, [0x0579] = 0x6820, [0x057a] = 0x6b2c, [0x057b] = 0x6b2f,
  [0x057c] = 0x6b2d, [0x057d] = 0x6b31, [0x057e] = 0x6b34, [0x057f] = 0x6b6d,
  [0x0580] = 0x8082, [0x0581] = 0x6b88, [0x0582] = 0x6be6, [0x0583] = 0x6be4,
  [0x0584] = 0x6be8, [0x0585] = 0x6be3, [0x0586] = 0x6be2, [0x0587] = 0x6be7,
  [0x0588] = 0x6c25, [0x0589] = 0x6d7a, [0x058a] = 0x6d63, [0x058b] = 0x6d64,
  [0x058c] = 0x6d76, [0x058d] = 0x6d0d, [0x058e] = 0x6d61, [0x058f] = 0x6d92,
  [0x0590] = 0x6d58, [0x0591] = 0x6d62, [0x0592] = 0x6d6d, [0x0593] = 0x6d6f,
  [0x0594] = 0x6d91, [0x0595] = 0x6d8d, [0x0596] = 0x6def, [0x0597] = 0x6d7f,
  [0x0598] = 0x6d86, [0x0599] = 0x6d5e, [0x059a] = 0x6d67, [0x059b] = 0x6d60,
  [0x059c] = 0x6d97, [0x059d] = 0x6d70, [0x059e] = 0x6d7c, [0x059f] = 0x6d5f,
  [0x05a0] = 0x6d82, [0x05a1] = 0x6d98, [0x05a2] = 0x6d2f, [0x05a3] = 0x6d68,
  [0x05a4] = 0x6d8b, [0x05a5] = 0x6d7e, [0x05a6] = 0x6d80, [0x05a7] = 0x6d84,
  [0x05a8] = 0x6d16, [0x05a9] = 0x6d83, [0x05aa] = 0x6d7b, [0x05ab] = 0x6d7d,
  [0x05ac] = 0x6d75, [0x05ad] = 0x6d90, [0x05ae] = 0x70dc, [0x05af] = 0x70d3,
  [0x05b0] = 0x70d1, [0x05b1] = 0x70dd, [0x05b2] = 0x70cb, [0x05b3] = 0x7f39,
  [0x05b4] = 0x70e2, [0x05b5] = 0x70d7, [0x05b6] = 0x70d2, [0x05b7] = 0x70de,
  [0x05b8] = 0x70e0, [0x05b9] = 0x70d4, [0x05ba] = 0x70cd, [0x05bb] = 0x70c5,
  [0x05bc] = 0x70c6, [0x05bd] = 0x70c7, [0x05be] = 0x70da, [0x05bf] = 0x70ce,
  [0x05c0] = 0x70e1, [0x05c1] = 0x7242, [0x05c2] = 0x7278, [0x05c3] = 0x7277,
  [0x05c4] = 0x7276, [0x05c5] = 0x7300, [0x05c6] = 0x72fa, [0x05c7] = 0x72f4,
  [0x05c8] = 0x72fe, [0x05c9] = 0x72f6, [0x05ca] = 0x72f3, [0x05cb] = 0x72fb,
  [0x05cc] = 0x7301, [0x05cd] = 0x73d3, [0x05ce] = 0x73d9, [0x05cf] = 0x73e5,
  [0x05d0] = 0x73d6, [0x05d1] = 0x73bc, [0x05d2] = 0x73e7, [0x05d3] = 0x73e3,
  [0x05d4] = 0x73e9, [0x05d5] = 0x73dc, [0x05d6] = 0x73d2, [0x05d7] = 0x73db,
  [0x05d8] = 0x73d4, [0x05d9] = 0x73dd, [0x05da] = 0x73da, [0x05db] = 0x73d7,
  [0x05dc] = 0x73d8, [0x05dd] = 0x73e8, [0x05de] = 0x74de, [0x05df] = 0x74df,
  [0x05e0] = 0x74f4, [0x05e1] = 0x74f5, [0x05e2] = 0x7521, [0x05e3] = 0x755b,
  [0x05e4] = 0x755f, [0x05e5] = 0x75b0, [0x05e6] = 0x75c1, [0x05e7] = 0x75bb,
  [0x05e8] = 0x75c4, [0x05e9] = 0x75c0, [0x05ea] = 0x75bf, [0x05eb] = 0x75b6,
  [0x05ec] = 0x75ba, [0x05ed] = 0x768a, [0x05ee] = 0x76c9, [0x05ef] = 0x771d,
  [0x05f0] = 0x771b, [0x05f1] = 0x7710, [0x05f2] = 0x7713, [0x05f3] = 0x7712,
  [0x05f4] = 0x7723, [0x05f5] = 0x7711, [0x05f6] = 0x7715, [0x05f7] = 0x7719,
  [0x05f8] = 0x771a, [0x05f9] = 0x7722, [0x05fa] = 0x7727, [0x05fb] = 0x7823,
  [0x05fc] = 0x782c, [0x05fd] = 0x7822, [0x05fe] = 0x7835, [0x05ff] = 0x782f,
  [0x0600] = 0x7828, [0x0601] = 0x782e, [0x0602] = 0x782b, [0x0603] = 0x7821,
  [0x0604] = 0x7829, [0x0605] = 0x7833, [0x0606] = 0x782a, [0x0607] = 0x7831,
  [0x0608] = 0x7954, [0x0609] = 0x795b, [0x060a] = 0x794f, [0x060b] = 0x795c,
  [0x060c] = 0x7953, [0x060d] = 0x7952, [0x060e] = 0x7951, [0x060f] = 0x79eb,
  [0x0610] = 0x79ec, [0x0611] = 0x79e0, [0x0612] = 0x79ee, [0x0613] = 0x79ed,
  [0x0614] = 0x79ea, [0x0615] = 0x79dc, [0x0616] = 0x79de, [0x0617] = 0x79dd,
  [0x0618] = 0x7a86, [0x0619] = 0x7a89, [0x061a] = 0x7a85, [0x061b] = 0x7a8b,
  [0x061c] = 0x7a8c, [0x061d] = 0x7a8a, [0x061e] = 0x7a87, [0x061f] = 0x7ad8,
  [0x0620] = 0x7b10, [0x0621] = 0x7b04, [0x0622] = 0x7b13, [0x0623] = 0x7b05,
  [0x0624] = 0x7b0f, [0x0625] = 0x7b08, [0x0626] = 0x7b0a, [0x0627] = 0x7b0e,
  [0x0628] = 0x7b09, [0x0629] = 0x7b12, [0x062a] = 0x7c84, [0x062b] = 0x7c91,
  [0x062c] = 0x7c8a, [0x062d] = 0x7c8c, [0x062e] = 0x7c88, [0x062f] = 0x7c8d,
  [0x0630] = 0x7c85, [0x0631] = 0x7d1e, [0x0632] = 0x7d1d, [0x0633] = 0x7d11,
  [0x0634] = 0x7d0e, [0x0635] = 0x7d18, [0x0636] = 0x7d16, [0x0637] = 0x7d13,
  [0x0638] = 0x7d1f, [0x0639] = 0x7d12, [0x063a] = 0x7d0f, [0x063b] = 0x7d0c,
  [0x063c] = 0x7f5c, [0x063d] = 0x7f61, [0x063e] = 0x7f5e, [0x063f] = 0x7f60,
  [0x0640] = 0x7f5d, [0x0641] = 0x7f5b, [0x0642] = 0x7f96, [0x0643] = 0x7f92,
  [0x0644] = 0x7fc3, [0x0645] = 0x7fc2, [0x0646] = 0x7fc0, [0x0647] = 0x8016,
  [0x0648] = 0x803e, [0x0649] = 0x8039, [0x064a] = 0x80fa, [0x064b] = 0x80f2,
  [0x064c] = 0x80f9, [0x064d] = 0x80f5, [0x064e] = 0x8101, [0x064f] = 0x80fb,
  [0x0650] = 0x8100, [0x0651] = 0x8201, [0x0652] = 0x822f, [0x0653] = 0x8225,
  [0x0654] = 0x8333, [0x0655] = 0x832d, [0x0656] = 0x8344, [0x0657] = 0x8319,
  [0x0658] = 0x8351, [0x0659] = 0x8325, [0x065a] = 0x8356, [0x065b] = 0x833f,
  [0x065c] = 0x8341, [0x065d] = 0x8326, [0x065e] = 0x831c, [0x065f] = 0x8322,
  [0x0660] = 0x8342, [0x0661] = 0x834e, [0x0662] = 0x831b, [0x0663] = 0x832a,
  [0x0664] = 0x8308, [0x0665] = 0x833c, [0x0666] = 0x834d, [0x0667] = 0x8316,
  [0x0668] = 0x8324, [0x0669] = 0x8320, [0x066a] = 0x8337, [0x066b] = 0x832f,
  [0x066c] = 0x8329, [0x066d] = 0x8347, [0x066e] = 0x8345, [0x066f] = 0x834c,
  [0x0670] = 0x8353, [0x0671] = 0x831e, [0x0672] = 0x832c, [0x0673] = 0x834b,
  [0x0674] = 0x8327, [0x0675] = 0x8348, [0x0676] = 0x8653, [0x0677] = 0x8652,
  [0x0678] = 0x86a2, [0x0679] = 0x86a8, [0x067a] = 0x8696, [0x067b] = 0x868d,
  [0x067c] = 0x8691, [0x067d] = 0x869e, [0x067e] = 0x8687, [0x067f] = 0x8697,
  [0x0680] = 0x8686, [0x0681] = 0x868b, [0x0682] = 0x869a, [0x0683] = 0x8685,
  [0x0684] = 0x86a5, [0x0685] = 0x8699, [0x0686] = 0x86a1, [0x0687] = 0x86a7,
  [0x0688] = 0x8695, [0x0689] = 0x8698, [0x068a] = 0x868e, [0x068b] = 0x869d,
  [0x068c] = 0x8690, [0x068d] = 0x8694, [0x068e] = 0x8843, [0x068f] = 0x8844,
  [0x0690] = 0x886d, [0x0691] = 0x8875, [0x0692] = 0x8876, [0x0693] = 0x8872,
  [0x0694] = 0x8880, [0x0695] = 0x8871, [0x0696] = 0x887f, [0x0697] = 0x886f,
  [0x0698] = 0x8883, [0x0699] = 0x887e, [0x069a] = 0x8874, [0x069b] = 0x887c,
  [0x069c] = 0x8a12, [0x069d] = 0x8c47, [0x069e] = 0x8c57, [0x069f] = 0x8c7b,
  [0x06a0] = 0x8ca4, [0x06a1] = 0x8ca3, [0x06a2] = 0x8d76, [0x06a3] = 0x8d78,
  [0x06a4] = 0x8db5, [0x06a5] = 0x8db7, [0x06a6] = 0x8db6, [0x06a7] = 0x8ed1,
  [0x06a8] = 0x8ed3, [0x06a9] = 0x8ffe, [0x06aa] = 0x8ff5, [0x06ab] = 0x9002,
  [0x06ac] = 0x8fff, [0x06ad] = 0x8ffb, [0x06ae] = 0x9004, [0x06af] = 0x8ffc,
  [0x06b0] = 0x8ff6, [0x06b1] = 0x90d6, [0x06b2] = 0x90e0, [0x06b3] = 0x90d9,
  [0x06b4] = 0x90da, [0x06b5] = 0x90e3, [0x06b6] = 0x90df, [0x06b7] = 0x90e5,
  [0x06b8] = 0x90d8, [0x06b9] = 0x90db, [0x06ba] = 0x90d7, [0x06bb] = 0x90dc,
  [0x06bc] = 0x90e4, [0x06bd] = 0x9150, [0x06be] = 0x914e, [0x06bf] = 0x914f,
  [0x06c0] = 0x91d5, [0x06c1] = 0x91e2, [0x06c2] = 0x91da, [0x06c3] = 0x965c,
  [0x06c4] = 0x965f, [0x06c5] = 0x96bc, [0x06c6] = 0x98e3, [0x06c7] = 0x9adf,
  [0x06c8] = 0x9b2f, [0x06c9] = 0x4e7f, [0x06ca] = 0x5070, [0x06cb] = 0x506a,
  [0x06cc] = 0x5061, [0x06cd] = 0x505e, [0x06ce] = 0x5060, [0x06cf] = 0x5053,
  [0x06d0] = 0x504b, [0x06d1] = 0x505d, [0x06d2] = 0x5072, [0x06d3] = 0x5048,
  [0x06d4] = 0x504d, [0x06d5] = 0x5041, [0x06d6] = 0x505b, [0x06d7] = 0x504a,
  [0x06d8] = 0x5062, [0x06d9] = 0x5015, [0x06da] = 0x5045, [0x06db] = 0x505f,
  [0x06dc] = 0x5069, [0x06dd] = 0x506b, [0x06de] = 0x5063, [0x06df] = 0x5064,
  [0x06e0] = 0x5046, [0x06e1] = 0x5040, [0x06e2] = 0x506e, [0x06e3] = 0x5073,
  [0x06e4] = 0x5057, [0x06e5] = 0x5051, [0x06e6] = 0x51d0, [0x06e7] = 0x526b,
  [0x06e8] = 0x526d, [0x06e9] = 0x526c, [0x06ea] = 0x526e, [0x06eb] = 0x52d6,
  [0x06ec] = 0x52d3, [0x06ed] = 0x532d, [0x06ee] = 0x539c, [0x06ef] = 0x5575,
  [0x06f0] = 0x5576, [0x06f1] = 0x553c, [0x06f2] = 0x554d, [0x06f3] = 0x5550,
  [0x06f4] = 0x5534, [0x06f5] = 0x552a, [0x06f6] = 0x5551, [0x06f7] = 0x5562,
  [0x06f8] = 0x5536, [0x06f9] = 0x5535, [0x06fa] = 0x5530, [0x06fb] = 0x5552,
  [0x06fc] = 0x5545, [0x06fd] = 0x550c, [0x06fe] = 0x5532, [0x06ff] = 0x5565,
  [0x0700] = 0x554e, [0x0701] = 0x5539, [0x0702] = 0x5548, [0x0703] = 0x552d,
  [0x0704] = 0x553b, [0x0705] = 0x5540, [0x0706] = 0x554b, [0x0707] = 0x570a,
  [0x0708] = 0x5707, [0x0709] = 0x57fb, [0x070a] = 0x5814, [0x070b] = 0x57e2,
  [0x070c] = 0x57f6, [0x070d] = 0x57dc, [0x070e] = 0x57f4, [0x070f] = 0x5800,
  [0x0710] = 0x57ed, [0x0711] = 0x57fd, [0x0712] = 0x5808, [0x0713] = 0x57f8,
  [0x0714] = 0x580b, [0x0715] = 0x57f3, [0x0716] = 0x57cf, [0x0717] = 0x5807,
  [0x0718] = 0x57ee, [0x0719] = 0x57e3, [0x071a] = 0x57f2, [0x071b] = 0x57e5,
  [0x071c] = 0x57ec, [0x071d] = 0x57e1, [0x071e] = 0x580e, [0x071f] = 0x57fc,
  [0x0720] = 0x5810, [0x0721] = 0x57e7, [0x0722] = 0x5801, [0x0723] = 0x580c,
  [0x0724] = 0x57f1, [0x0725] = 0x57e9, [0x0726] = 0x57f0, [0x0727] = 0x580d,
  [0x0728] = 0x5804, [0x0729] = 0x595c, [0x072a] = 0x5a60, [0x072b] = 0x5a58,
  [0x072c] = 0x5a55, [0x072d] = 0x5a67, [0x072e] = 0x5a5e, [0x072f] = 0x5a38,
  [0x0730] = 0x5a35, [0x0731] = 0x5a6d, [0x0732] = 0x5a50, [0x0733] = 0x5a5f,
  [0x0734] = 0x5a65, [0x0735] = 0x5a6c, [0x0736] = 0x5a53, [0x0737] = 0x5a64,
  [0x0738] = 0x5a57, [0x0739] = 0x5a43, [0x073a] = 0x5a5d, [0x073b] = 0x5a52,
  [0x073c] = 0x5a44, [0x073d] = 0x5a5b, [0x073e] = 0x5a48, [0x073f] = 0x5a8e,
  [0x0740] = 0x5a3e, [0x0741] = 0x5a4d, [0x0742] = 0x5a39, [0x0743] = 0x5a4c,
  [0x0744] = 0x5a70, [0x0745] = 0x5a69, [0x0746] = 0x5a47, [0x0747] = 0x5a51,
  [0x0748] = 0x5a56, [0x0749] = 0x5a42, [0x074a] = 0x5a5c, [0x074b] = 0x5b72,
  [0x074c] = 0x5b6e, [0x074d] = 0x5bc1, [0x074e] = 0x5bc0, [0x074f] = 0x5c59,
  [0x0750] = 0x5d1e, [0x0751] = 0x5d0b, [0x0752] = 0x5d1d, [0x0753] = 0x5d1a,
  [0x0754] = 0x5d20, [0x0755] = 0x5d0c, [0x0756] = 0x5d28, [0x0757] = 0x5d0d,
  [0x0758] = 0x5d26, [0x0759] = 0x5d25, [0x075a] = 0x5d0f, [0x075b] = 0x5d30,
  [0x075c] = 0x5d12, [0x075d] = 0x5d23, [0x075e] = 0x5d1f, [0x075f] = 0x5d2e,
  [0x0760] = 0x5e3e, [0x0761] = 0x5e34, [0x0762] = 0x5eb1, [0x0763] = 0x5eb4,
  [0x0764] = 0x5eb9, [0x0765] = 0x5eb2, [0x0766] = 0x5eb3, [0x0767] = 0x5f36,
  [0x0768] = 0x5f38, [0x0769] = 0x5f9b, [0x076a] = 0x5f96, [0x076b] = 0x5f9f,
  [0x076c] = 0x608a, [0x076d] = 0x6090, [0x076e] = 0x6086, [0x076f] = 0x60be,
  [0x0770] = 0x60b0, [0x0771] = 0x60ba, [0x0772] = 0x60d3, [0x0773] = 0x60d4,
  [0x0774] = 0x60cf, [0x0775] = 0x60e4, [0x0776] = 0x60d9, [0x0777] = 0x60dd,
  [0x0778] = 0x60c8, [0x0779] = 0x60b1, [0x077a] = 0x60db, [0x077b] = 0x60b7,
  [0x077c] = 0x60ca, [0x077d] = 0x60bf, [0x077e] = 0x60c3, [0x077f] = 0x60cd,
  [0x0780] = 0x60c0, [0x0781] = 0x6332, [0x0782] = 0x6365, [0x0783] = 0x638a,
  [0x0784] = 0x6382, [0x0785] = 0x637d, [0x0786] = 0x63bd, [0x0787] = 0x639e,
  [0x0788] = 0x63ad, [0x0789] = 0x639d, [0x078a] = 0x6397, [0x078b] = 0x63ab,
  [0x078c] = 0x638e, [0x078d] = 0x636f, [0x078e] = 0x6387, [0x078f] = 0x6390,
  [0x0790] = 0x636e, [0x0791] = 0x63af, [0x0792] = 0x6375, [0x0793] = 0x639c,
  [0x0794] = 0x636d, [0x0795] = 0x63ae, [0x0796] = 0x637c, [0x0797] = 0x63a4,
  [0x0798] = 0x633b, [0x0799] = 0x639f, [0x079a] = 0x6378, [0x079b] = 0x6385,
  [0x079c] = 0x6381, [0x079d] = 0x6391, [0x079e] = 0x638d, [0x079f] = 0x6370,
  [0x07a0] = 0x6553, [0x07a1] = 0x65cd, [0x07a2] = 0x6665, [0x07a3] = 0x6661,
  [0x07a4] = 0x665b, [0x07a5] = 0x6659, [0x07a6] = 0x665c, [0x07a7] = 0x6662,
  [0x07a8] = 0x6718, [0x07a9] = 0x6879, [0x07aa] = 0x6887, [0x07ab] = 0x6890,
  [0x07ac] = 0x689c, [0x07ad] = 0x686d, [0x07ae] = 0x686e, [0x07af] = 0x68ae,
  [0x07b0] = 0x68ab, [0x07b1] = 0x6956, [0x07b2] = 0x686f, [0x07b3] = 0x68a3,
  [0x07b4] = 0x68ac, [0x07b5] = 0x68a9, [0x07b6] = 0x6875, [0x07b7] = 0x6874,
  [0x07b8] = 0x68b2, [0x07b9] = 0x688f, [0x07ba] = 0x6877, [0x07bb] = 0x6892,
  [0x07bc] = 0x687c, [0x07bd] = 0x686b, [0x07be] = 0x6872, [0x07bf] = 0x68aa,
  [0x07c0] = 0x6880, [0x07c1] = 0x6871, [0x07c2] = 0x687e, [0x07c3] = 0x689b,
  [0x07c4] = 0x6896, [0x07c5] = 0x688b, [0x07c6] = 0x68a0, [0x07c7] = 0x6889,
  [0x07c8] = 0x68a4, [0x07c9] = 0x6878, [0x07ca] = 0x687b, [0x07cb] = 0x6891,
  [0x07cc] = 0x688c, [0x07cd] = 0x688a, [0x07ce] = 0x687d, [0x07cf] = 0x6b36,
  [0x07d0] = 0x6b33, [0x07d1] = 0x6b37, [0x07d2] = 0x6b38, [0x07d3] = 0x6b91,
  [0x07d4] = 0x6b8f, [0x07d5] = 0x6b8d, [0x07d6] = 0x6b8e, [0x07d7] = 0x6b8c,
  [0x07d8] = 0x6c2a, [0x07d9] = 0x6dc0, [0x07da] = 0x6dab, [0x07db] = 0x6db4,
  [0x07dc] = 0x6db3, [0x07dd] = 0x6e74, [0x07de] = 0x6dac, [0x07df] = 0x6de9,
  [0x07e0] = 0x6de2, [0x07e1] = 0x6db7, [0x07e2] = 0x6df6, [0x07e3] = 0x6dd4,
  [0x07e4] = 0x6e00, [0x07e5] = 0x6dc8, [0x07e6] = 0x6de0, [0x07e7] = 0x6ddf,
  [0x07e8] = 0x6dd6, [0x07e9] = 0x6dbe, [0x07ea] = 0x6de5, [0x07eb] = 0x6ddc,
  [0x07ec] = 0x6ddd, [0x07ed] = 0x6ddb, [0x07ee] = 0x6df4, [0x07ef] = 0x6dca,
  [0x07f0] = 0x6dbd, [0x07f1] = 0x6ded, [0x07f2] = 0x6df0, [0x07f3] = 0x6dba,
  [0x07f4] = 0x6dd5, [0x07f5] = 0x6dc2, [0x07f6] = 0x6dcf, [0x07f7] = 0x6dc9,
  [0x07f8] = 0x6dd0, [0x07f9] = 0x6df2, [0x07fa] = 0x6dd3, [0x07fb] = 0x6dfd,
  [0x07fc] = 0x6dd7, [0x07fd] = 0x6dcd, [0x07fe] = 0x6de3, [0x07ff] = 0x6dbb,
  [0x0800] = 0x70fa, [0x0801] = 0x710d, [0x0802] = 0x70f7, [0x0803] = 0x7117,
  [0x0804] = 0x70f4, [0x0805] = 0x710c, [0x0806] = 0x70f0, [0x0807] = 0x7104,
  [0x0808] = 0x70f3, [0x0809] = 0x7110, [0x080a] = 0x70fc, [0x080b] = 0x70ff,
  [0x080c] = 0x7106, [0x080d] = 0x7113, [0x080e] = 0x7100, [0x080f] = 0x70f8,
  [0x0810] = 0x70f6, [0x0811] = 0x710b, [0x0812] = 0x7102, [0x0813] = 0x710e,
  [0x0814] = 0x727e, [0x0815] = 0x727b, [0x0816] = 0x727c, [0x0817] = 0x727f,
  [0x0818] = 0x731d, [0x0819] = 0x7317, [0x081a] = 0x7307, [0x081b] = 0x7311,
  [0x081c] = 0x7318, [0x081d] = 0x730a, [0x081e] = 0x7308, [0x081f] = 0x72ff,
  [0x0820] = 0x730f, [0x0821] = 0x731e, [0x0822] = 0x7388, [0x0823] = 0x73f6,
  [0x0824] = 0x73f8, [0x0825] = 0x73f5, [0x0826] = 0x7404, [0x0827] = 0x7401,
  [0x0828] = 0x73fd, [0x0829] = 0x7407, [0x082a] = 0x7400, [0x082b] = 0x73fa,
  [0x082c] = 0x73fc, [0x082d] = 0x73ff, [0x082e] = 0x740c, [0x082f] = 0x740b,
  [0x0830] = 0x73f4, [0x0831] = 0x7408, [0x0832] = 0x7564, [0x0833] = 0x7563,
  [0x0834] = 0x75ce, [0x0835] = 0x75d2, [0x0836] = 0x75cf, [0x0837] = 0x75cb,
  [0x0838] = 0x75cc, [0x0839] = 0x75d1, [0x083a] = 0x75d0, [0x083b] = 0x768f,
  [0x083c] = 0x7689, [0x083d] = 0x76d3, [0x083e] = 0x7739, [0x083f] = 0x772f,
  [0x0840] = 0x772d, [0x0841] = 0x7731, [0x0842] = 0x7732, [0x0843] = 0x7734,
  [0x0844] = 0x7733, [0x0845] = 0x773d, [0x0846] = 0x7725, [0x0847] = 0x773b,
  [0x0848] = 0x7735, [0x0849] = 0x7848, [0x084a] = 0x7852, [0x084b] = 0x7849,
  [0x084c] = 0x784d, [0x084d] = 0x784a, [0x084e] = 0x784c, [0x084f] = 0x7826,
  [0x0850] = 0x7845, [0x0851] = 0x7850, [0x0852] = 0x7964, [0x0853] = 0x7967,
  [0x0854] = 0x7969, [0x0855] = 0x796a, [0x0856] = 0x7963, [0x0857] = 0x796b,
  [0x0858] = 0x7961, [0x0859] = 0x79bb, [0x085a] = 0x79fa, [0x085b] = 0x79f8,
  [0x085c] = 0x79f6, [0x085d] = 0x79f7, [0x085e] = 0x7a8f, [0x085f] = 0x7a94,
  [0x0860] = 0x7a90, [0x0861] = 0x7b35, [0x0862] = 0x7b3b, [0x0863] = 0x7b34,
  [0x0864] = 0x7b25, [0x0865] = 0x7b30, [0x0866] = 0x7b22, [0x0867] = 0x7b24,
  [0x0868] = 0x7b33, [0x0869] = 0x7b18, [0x086a] = 0x7b2a, [0x086b] = 0x7b1d,
  [0x086c] = 0x7b31, [0x086d] = 0x7b2b, [0x086e] = 0x7b2d, [0x086f] = 0x7b2f,
  [0x0870] = 0x7b32, [0x0871] = 0x7b38, [0x0872] = 0x7b1a, [0x0873] = 0x7b23,
  [0x0874] = 0x7c94, [0x0875] = 0x7c98, [0x0876] = 0x7c96, [0x0877] = 0x7ca3,
  [0x0878] = 0x7d35, [0x0879] = 0x7d3d, [0x087a] = 0x7d38, [0x087b] = 0x7d36,
  [0x087c] = 0x7d3a, [0x087d] = 0x7d45, [0x087e] = 0x7d2c, [0x087f] = 0x7d29,
  [0x0880] = 0x7d41, [0x0881] = 0x7d47, [0x0882] = 0x7d3e, [0x0883] = 0x7d3f,
  [0x0884] = 0x7d4a, [0x0885] = 0x7d3b, [0x0886] = 0x7d28, [0x0887] = 0x7f63,
  [0x0888] = 0x7f95, [0x0889] = 0x7f9c, [0x088a] = 0x7f9d, [0x088b] = 0x7f9b,
  [0x088c] = 0x7fca, [0x088d] = 0x7fcb, [0x088e] = 0x7fcd, [0x088f] = 0x7fd0,
  [0x0890] = 0x7fd1, [0x0891] = 0x7fc7, [0x0892] = 0x7fcf, [0x0893] = 0x7fc9,
  [0x0894] = 0x801f, [0x0895] = 0x801e, [0x0896] = 0x801b, [0x0897] = 0x8047,
  [0x0898] = 0x8043, [0x0899] = 0x8048, [0x089a] = 0x8118, [0x089b] = 0x8125,
  [0x089c] = 0x8119, [0x089d] = 0x811b, [0x089e] = 0x812d, [0x089f] = 0x811f,
  [0x08a0] = 0x812c, [0x08a1] = 0x811e, [0x08a2] = 0x8121, [0x08a3] = 0x8115,
  [0x08a4] = 0x8127, [0x08a5] = 0x811d, [0x08a6] = 0x8122, [0x08a7] = 0x8211,
  [0x08a8] = 0x8238, [0x08a9] = 0x8233, [0x08aa] = 0x823a, [0x08ab] = 0x8234,
  [0x08ac] = 0x8232, [0x08ad] = 0x8274, [0x08ae] = 0x8390, [0x08af] = 0x83a3,
  [0x08b0] = 0x83a8, [0x08b1] = 0x838d, [0x08b2] = 0x837a, [0x08b3] = 0x8373,
  [0x08b4] = 0x83a4, [0x08b5] = 0x8374, [0x08b6] = 0x838f, [0x08b7] = 0x8381,
  [0x08b8] = 0x8395, [0x08b9] = 0x8399, [0x08ba] = 0x8375, [0x08bb] = 0x8394,
  [0x08bc] = 0x83a9, [0x08bd] = 0x837d, [0x08be] = 0x8383, [0x08bf] = 0x838c,
  [0x08c0] = 0x839d, [0x08c1] = 0x839b, [0x08c2] = 0x83aa, [0x08c3] = 0x838b,
  [0x08c4] = 0x837e, [0x08c5] = 0x83a5, [0x08c6] = 0x83af, [0x08c7] = 0x8388,
  [0x08c8] = 0x8397, [0x08c9] = 0x83b0, [0x08ca] = 0x837f, [0x08cb] = 0x83a6,
  [0x08cc] = 0x8387, [0x08cd] = 0x83ae, [0x08ce] = 0x8376, [0x08cf] = 0x8659,
  [0x08d0] = 0x8656, [0x08d1] = 0x86bf, [0x08d2] = 0x86b7, [0x08d3] = 0x86c2,
  [0x08d4] = 0x86c1, [0x08d5] = 0x86c5, [0x08d6] = 0x86ba, [0x08d7] = 0x86b0,
  [0x08d8] = 0x86c8, [0x08d9] = 0x86b9, [0x08da] = 0x86b3, [0x08db] = 0x86b8,
  [0x08dc] = 0x86cc, [0x08dd] = 0x86b4, [0x08de] = 0x86bb, [0x08df] = 0x86bc,
  [0x08e0] = 0x86c3, [0x08e1] = 0x86bd, [0x08e2] = 0x86be, [0x08e3] = 0x8852,
  [0x08e4] = 0x8889, [0x08e5] = 0x8895, [0x08e6] = 0x88a8, [0x08e7] = 0x88a2,
  [0x08e8] = 0x88aa, [0x08e9] = 0x889a, [0x08ea] = 0x8891, [0x08eb] = 0x88a1,
  [0x08ec] = 0x889f, [0x08ed] = 0x8898, [0x08ee] = 0x88a7, [0x08ef] = 0x8899,
  [0x08f0] = 0x889b, [0x08f1] = 0x8897, [0x08f2] = 0x88a4, [0x08f3] = 0x88ac,
  [0x08f4] = 0x888c, [0x08f5] = 0x8893, [0x08f6] = 0x888e, [0x08f7] = 0x8982,
  [0x08f8] = 0x89d6, [0x08f9] = 0x89d9, [0x08fa] = 0x89d5, [0x08fb] = 0x8a30,
  [0x08fc] = 0x8a27, [0x08fd] = 0x8a2c, [0x08fe] = 0x8a1e, [0x08ff] = 0x8c39,
  [0x0900] = 0x8c3b, [0x0901] = 0x8c5c, [0x0902] = 0x8c5d, [0x0903] = 0x8c7d,
  [0x0904] = 0x8ca5, [0x0905] = 0x8d7d, [0x0906] = 0x8d7b, [0x0907] = 0x8d79,
  [0x0908] = 0x8dbc, [0x0909] = 0x8dc2, [0x090a] = 0x8db9, [0x090b] = 0x8dbf,
  [0x090c] = 0x8dc1, [0x090d] = 0x8ed8, [0x090e] = 0x8ede, [0x090f] = 0x8edd,
  [0x0910] = 0x8edc, [0x0911] = 0x8ed7, [0x0912] = 0x8ee0, [0x0913] = 0x8ee1,
  [0x0914] = 0x9024, [0x0915] = 0x900b, [0x0916] = 0x9011, [0x0917] = 0x901c,
  [0x0918] = 0x900c, [0x0919] = 0x9021, [0x091a] = 0x90ef, [0x091b] = 0x90ea,
  [0x091c] = 0x90f0, [0x091d] = 0x90f4, [0x091e] = 0x90f2, [0x091f] = 0x90f3,
  [0x0920] = 0x90d4, [0x0921] = 0x90eb, [0x0922] = 0x90ec, [0x0923] = 0x90e9,
  [0x0924] = 0x9156, [0x0925] = 0x9158, [0x0926] = 0x915a, [0x0927] = 0x9153,
  [0x0928] = 0x9155, [0x0929] = 0x91ec, [0x092a] = 0x91f4, [0x092b] = 0x91f1,
  [0x092c] = 0x91f3, [0x092d] = 0x91f8, [0x092e] = 0x91e4, [0x092f] = 0x91f9,
  [0x0930] = 0x91ea, [0x0931] = 0x91eb, [0x0932] = 0x91f7, [0x0933] = 0x91e8,
  [0x0934] = 0x91ee, [0x0935] = 0x957a, [0x0936] = 0x9586, [0x0937] = 0x9588,
  [0x0938] = 0x967c, [0x0939] = 0x966d, [0x093a] = 0x966b, [0x093b] = 0x9671,
  [0x093c] = 0x966f, [0x093d] = 0x96bf, [0x093e] = 0x976a, [0x093f] = 0x9804,
  [0x0940] = 0x98e5, [0x0941] = 0x9997, [0x0942] = 0x509b, [0x0943] = 0x5095,
  [0x0944] = 0x5094, [0x0945] = 0x509e, [0x0946] = 0x508b, [0x0947] = 0x50a3,
  [0x0948] = 0x5083, [0x0949] = 0x508c, [0x094a] = 0x508e, [0x094b] = 0x509d,
  [0x094c] = 0x5068, [0x094d] = 0x509c, [0x094e] = 0x5092, [0x094f] = 0x5082,
  [0x0950] = 0x5087, [0x0951] = 0x515f, [0x0952] = 0x51d4, [0x0953] = 0x5312,
  [0x0954] = 0x5311, [0x0955] = 0x53a4, [0x0956] = 0x53a7, [0x0957] = 0x5591,
  [0x0958] = 0x55a8, [0x0959] = 0x55a5, [0x095a] = 0x55ad, [0x095b] = 0x5577,
  [0x095c] = 0x5645, [0x095d] = 0x55a2, [0x095e] = 0x5593, [0x095f] = 0x5588,
  [0x0960] = 0x558f, [0x0961] = 0x55b5, [0x0962] = 0x5581, [0x0963] = 0x55a3,
  [0x0964] = 0x5592, [0x0965] = 0x55a4, [0x0966] = 0x557d, [0x0967] = 0x558c,
  [0x0968] = 0x55a6, [0x0969] = 0x557f, [0x096a] = 0x5595, [0x096b] = 0x55a1,
  [0x096c] = 0x558e, [0x096d] = 0x570c, [0x096e] = 0x5829, [0x096f] = 0x5837,
  [0x0970] = 0x5819, [0x0971] = 0x581e, [0x0972] = 0x5827, [0x0973] = 0x5823,
  [0x0974] = 0x5828, [0x0975] = 0x57f5, [0x0976] = 0x5848, [0x0977] = 0x5825,
  [0x0978] = 0x581c, [0x0979] = 0x581b, [0x097a] = 0x5833, [0x097b] = 0x583f,
  [0x097c] = 0x5836, [0x097d] = 0x582e, [0x097e] = 0x5839, [0x097f] = 0x5838,
  [0x0980] = 0x582d, [0x0981] = 0x582c, [0x0982] = 0x583b, [0x0983] = 0x5961,
  [0x0984] = 0x5aaf, [0x0985] = 0x5a94, [0x0986] = 0x5a9f, [0x0987] = 0x5a7a,
  [0x0988] = 0x5aa2, [0x0989] = 0x5a9e, [0x098a] = 0x5a78, [0x098b] = 0x5aa6,
  [0x098c] = 0x5a7c, [0x098d] = 0x5aa5, [0x098e] = 0x5aac, [0x098f] = 0x5a95,
  [0x0990] = 0x5aae, [0x0991] = 0x5a37, [0x0992] = 0x5a84, [0x0993] = 0x5a8a,
  [0x0994] = 0x5a97, [0x0995] = 0x5a83, [0x0996] = 0x5a8b, [0x0997] = 0x5aa9,
  [0x0998] = 0x5a7b, [0x0999] = 0x5a7d, [0x099a] = 0x5a8c, [0x099b] = 0x5a9c,
  [0x099c] = 0x5a8f, [0x099d] = 0x5a93, [0x099e] = 0x5a9d, [0x099f] = 0x5bea,
  [0x09a0] = 0x5bcd, [0x09a1] = 0x5bcb, [0x09a2] = 0x5bd4, [0x09a3] = 0x5bd1,
  [0x09a4] = 0x5bca, [0x09a5] = 0x5bce, [0x09a6] = 0x5c0c, [0x09a7] = 0x5c30,
  [0x09a8] = 0x5d37, [0x09a9] = 0x5d43, [0x09aa] = 0x5d6b, [0x09ab] = 0x5d41,
  [0x09ac] = 0x5d4b, [0x09ad] = 0x5d3f, [0x09ae] = 0x5d35, [0x09af] = 0x5d51,
  [0x09b0] = 0x5d4e, [0x09b1] = 0x5d55, [0x09b2] = 0x5d33, [0x09b3] = 0x5d3a,
  [0x09b4] = 0x5d52, [0x09b5] = 0x5d3d, [0x09b6] = 0x5d31, [0x09b7] = 0x5d59,
  [0x09b8] = 0x5d42, [0x09b9] = 0x5d39, [0x09ba] = 0x5d49, [0x09bb] = 0x5d38,
  [0x09bc] = 0x5d3c, [0x09bd] = 0x5d32, [0x09be] = 0x5d36, [0x09bf] = 0x5d40,
  [0x09c0] = 0x5d45, [0x09c1] = 0x5e44, [0x09c2] = 0x5e41, [0x09c3] = 0x5f58,
  [0x09c4] = 0x5fa6, [0x09c5] = 0x5fa5, [0x09c6] = 0x5fab, [0x09c7] = 0x60c9,
  [0x09c8] = 0x60b9, [0x09c9] = 0x60cc, [0x09ca] = 0x60e2, [0x09cb] = 0x60ce,
  [0x09cc] = 0x60c4, [0x09cd] = 0x6114, [0x09ce] = 0x60f2, [0x09cf] = 0x610a,
  [0x09d0] = 0x6116, [0x09d1] = 0x6105, [0x09d2] = 0x60f5, [0x09d3] = 0x6113,
  [0x09d4] = 0x60f8, [0x09d5] = 0x60fc, [0x09d6] = 0x60fe, [0x09d7] = 0x60c1,
  [0x09d8] = 0x6103, [0x09d9] = 0x6118, [0x09da] = 0x611d, [0x09db] = 0x6110,
  [0x09dc] = 0x60ff, [0x09dd] = 0x6104, [0x09de] = 0x610b, [0x09df] = 0x624a,
  [0x09e0] = 0x6394, [0x09e1] = 0x63b1, [0x09e2] = 0x63b0, [0x09e3] = 0x63ce,
  [0x09e4] = 0x63e5, [0x09e5] = 0x63e8, [0x09e6] = 0x63ef, [0x09e7] = 0x63c3,
  [0x09e8] = 0x649d, [0x09e9] = 0x63f3, [0x09ea] = 0x63ca, [0x09eb] = 0x63e0,
  [0x09ec] = 0x63f6, [0x09ed] = 0x63d5, [0x09ee] = 0x63f2, [0x09ef] = 0x63f5,
  [0x09f0] = 0x6461, [0x09f1] = 0x63df, [0x09f2] = 0x63be, [0x09f3] = 0x63dd,
  [0x09f4] = 0x63dc, [0x09f5] = 0x63c4, [0x09f6] = 0x63d8, [0x09f7] = 0x63d3,
  [0x09f8] = 0x63c2, [0x09f9] = 0x63c7, [0x09fa] = 0x63cc, [0x09fb] = 0x63cb,
  [0x09fc] = 0x63c8, [0x09fd] = 0x63f0, [0x09fe] = 0x63d7, [0x09ff] = 0x63d9,
  [0x0a00] = 0x6532, [0x0a01] = 0x6567, [0x0a02] = 0x656a, [0x0a03] = 0x6564,
  [0x0a04] = 0x655c, [0x0a05] = 0x6568, [0x0a06] = 0x6565, [0x0a07] = 0x658c,
  [0x0a08] = 0x659d, [0x0a09] = 0x659e, [0x0a0a] = 0x65ae, [0x0a0b] = 0x65d0,
  [0x0a0c] = 0x65d2, [0x0a0d] = 0x667c, [0x0a0e] = 0x666c, [0x0a0f] = 0x667b,
  [0x0a10] = 0x6680, [0x0a11] = 0x6671, [0x0a12] = 0x6679, [0x0a13] = 0x666a,
  [0x0a14] = 0x6672, [0x0a15] = 0x6701, [0x0a16] = 0x690c, [0x0a17] = 0x68d3,
  [0x0a18] = 0x6904, [0x0a19] = 0x68dc, [0x0a1a] = 0x692a, [0x0a1b] = 0x68ec,
  [0x0a1c] = 0x68ea, [0x0a1d] = 0x68f1, [0x0a1e] = 0x690f, [0x0a1f] = 0x68d6,
  [0x0a20] = 0x68f7, [0x0a21] = 0x68eb, [0x0a22] = 0x68e4, [0x0a23] = 0x68f6,
  [0x0a24] = 0x6913, [0x0a25] = 0x6910, [0x0a26] = 0x68f3, [0x0a27] = 0x68e1,
  [0x0a28] = 0x6907, [0x0a29] = 0x68cc, [0x0a2a] = 0x6908, [0x0a2b] = 0x6970,
  [0x0a2c] = 0x68b4, [0x0a2d] = 0x6911, [0x0a2e] = 0x68ef, [0x0a2f] = 0x68c6,
  [0x0a30] = 0x6914, [0x0a31] = 0x68f8, [0x0a32] = 0x68d0, [0x0a33] = 0x68fd,
  [0x0a34] = 0x68fc, [0x0a35] = 0x68e8, [0x0a36] = 0x690b, [0x0a37] = 0x690a,
  [0x0a38] = 0x6917, [0x0a39] = 0x68ce, [0x0a3a] = 0x68c8, [0x0a3b] = 0x68dd,
  [0x0a3c] = 0x68de, [0x0a3d] = 0x68e6, [0x0a3e] = 0x68f4, [0x0a3f] = 0x68d1,
  [0x0a40] = 0x6906, [0x0a41] = 0x68d4, [0x0a42] = 0x68e9, [0x0a43] = 0x6915,
  [0x0a44] = 0x6925, [0x0a45] = 0x68c7, [0x0a46] = 0x6b39, [0x0a47] = 0x6b3b,
  [0x0a48] = 0x6b3f, [0x0a49] = 0x6b3c, [0x0a4a] = 0x6b94, [0x0a4b] = 0x6b97,
  [0x0a4c] = 0x6b99, [0x0a4d] = 0x6b95, [0x0a4e] = 0x6bbd, [0x0a4f] = 0x6bf0,
  [0x0a50] = 0x6bf2, [0x0a51] = 0x6bf3, [0x0a52] = 0x6c30, [0x0a53] = 0x6dfc,
  [0x0a54] = 0x6e46, [0x0a55] = 0x6e47, [0x0a56] = 0x6e1f, [0x0a57] = 0x6e49,
  [0x0a58] = 0x6e88, [0x0a59] = 0x6e3c, [0x0a5a] = 0x6e3d, [0x0a5b] = 0x6e45,
  [0x0a5c] = 0x6e62, [0x0a5d] = 0x6e2b, [0x0a5e] = 0x6e3f, [0x0a5f] = 0x6e41,
  [0x0a60] = 0x6e5d, [0x0a61] = 0x6e73, [0x0a62] = 0x6e1c, [0x0a63] = 0x6e33,
  [0x0a64] = 0x6e4b, [0x0a65] = 0x6e40, [0x0a66] = 0x6e51, [0x0a67] = 0x6e3b,
  [0x0a68] = 0x6e03, [0x0a69] = 0x6e2e, [0x0a6a] = 0x6e5e, [0x0a6b] = 0x6e68,
  [0x0a6c] = 0x6e5c, [0x0a6d] = 0x6e61, [0x0a6e] = 0x6e31, [0x0a6f] = 0x6e28,
  [0x0a70] = 0x6e60, [0x0a71] = 0x6e71, [0x0a72] = 0x6e6b, [0x0a73] = 0x6e39,
  [0x0a74] = 0x6e22, [0x0a75] = 0x6e30, [0x0a76] = 0x6e53, [0x0a77] = 0x6e65,
  [0x0a78] = 0x6e27, [0x0a79] = 0x6e78, [0x0a7a] = 0x6e64, [0x0a7b] = 0x6e77,
  [0x0a7c] = 0x6e55, [0x0a7d] = 0x6e79, [0x0a7e] = 0x6e52, [0x0a7f] = 0x6e66,
  [0x0a80] = 0x6e35, [0x0a81] = 0x6e36, [0x0a82] = 0x6e5a, [0x0a83] = 0x7120,
  [0x0a84] = 0x711e, [0x0a85] = 0x712f, [0x0a86] = 0x70fb, [0x0a87] = 0x712e,
  [0x0a88] = 0x7131, [0x0a89] = 0x7123, [0x0a8a] = 0x7125, [0x0a8b] = 0x7122,
  [0x0a8c] = 0x7132, [0x0a8d] = 0x711f, [0x0a8e] = 0x7128, [0x0a8f] = 0x713a,
  [0x0a90] = 0x711b, [0x0a91] = 0x724b, [0x0a92] = 0x725a, [0x0a93] = 0x7288,
  [0x0a94] = 0x7289, [0x0a95] = 0x7286, [0x0a96] = 0x7285, [0x0a97] = 0x728b,
  [0x0a98] = 0x7312, [0x0a99] = 0x730b, [0x0a9a] = 0x7330, [0x0a9b] = 0x7322,
  [0x0a9c] = 0x7331, [0x0a9d] = 0x7333, [0x0a9e] = 0x7327, [0x0a9f] = 0x7332,
  [0x0aa0] = 0x732d, [0x0aa1] = 0x7326, [0x0aa2] = 0x7323, [0x0aa3] = 0x7335,
  [0x0aa4] = 0x730c, [0x0aa5] = 0x742e, [0x0aa6] = 0x742c, [0x0aa7] = 0x7430,
  [0x0aa8] = 0x742b, [0x0aa9] = 0x7416, [0x0aaa] = 0x741a, [0x0aab] = 0x7421,
  [0x0aac] = 0x742d, [0x0aad] = 0x7431, [0x0aae] = 0x7424, [0x0aaf] = 0x7423,
  [0x0ab0] = 0x741d, [0x0ab1] = 0x7429, [0x0ab2] = 0x7420, [0x0ab3] = 0x7432,
  [0x0ab4] = 0x74fb, [0x0ab5] = 0x752f, [0x0ab6] = 0x756f, [0x0ab7] = 0x756c,
  [0x0ab8] = 0x75e7, [0x0ab9] = 0x75da, [0x0aba] = 0x75e1, [0x0abb] = 0x75e6,
  [0x0abc] = 0x75dd, [0x0abd] = 0x75df, [0x0abe] = 0x75e4, [0x0abf] = 0x75d7,
  [0x0ac0] = 0x7695, [0x0ac1] = 0x7692, [0x0ac2] = 0x76da, [0x0ac3] = 0x7746,
  [0x0ac4] = 0x7747, [0x0ac5] = 0x7744, [0x0ac6] = 0x774d, [0x0ac7] = 0x7745,
  [0x0ac8] = 0x774a, [0x0ac9] = 0x774e, [0x0aca] = 0x774b, [0x0acb] = 0x774c,
  [0x0acc] = 0x77de, [0x0acd] = 0x77ec, [0x0ace] = 0x7860, [0x0acf] = 0x7864,
  [0x0ad0] = 0x7865, [0x0ad1] = 0x785c, [0x0ad2] = 0x786d, [0x0ad3] = 0x7871,
  [0x0ad4] = 0x786a, [0x0ad5] = 0x786e, [0x0ad6] = 0x7870, [0x0ad7] = 0x7869,
  [0x0ad8] = 0x7868, [0x0ad9] = 0x785e, [0x0ada] = 0x7862, [0x0adb] = 0x7974,
  [0x0adc] = 0x7973, [0x0add] = 0x7972, [0x0ade] = 0x7970, [0x0adf] = 0x7a02,
  [0x0ae0] = 0x7a0a, [0x0ae1] = 0x7a03, [0x0ae2] = 0x7a0c, [0x0ae3] = 0x7a04,
  [0x0ae4] = 0x7a99, [0x0ae5] = 0x7ae6, [0x0ae6] = 0x7ae4, [0x0ae7] = 0x7b4a,
  [0x0ae8] = 0x7b47, [0x0ae9] = 0x7b44, [0x0aea] = 0x7b48, [0x0aeb] = 0x7b4c,
  [0x0aec] = 0x7b4e, [0x0aed] = 0x7b40, [0x0aee] = 0x7b58, [0x0aef] = 0x7b45,
  [0x0af0] = 0x7ca2, [0x0af1] = 0x7c9e, [0x0af2] = 0x7ca8, [0x0af3] = 0x7ca1,
  [0x0af4] = 0x7d58, [0x0af5] = 0x7d6f, [0x0af6] = 0x7d63, [0x0af7] = 0x7d53,
  [0x0af8] = 0x7d56, [0x0af9] = 0x7d67, [0x0afa] = 0x7d6a, [0x0afb] = 0x7d4f,
  [0x0afc] = 0x7d6d, [0x0afd] = 0x7d5c, [0x0afe] = 0x7d6b, [0x0aff] = 0x7d52,
  [0x0b00] = 0x7d54, [0x0b01] = 0x7d69, [0x0b02] = 0x7d51, [0x0b03] = 0x7d5f,
  [0x0b04] = 0x7d4e, [0x0b05] = 0x7f3e, [0x0b06] = 0x7f3f, [0x0b07] = 0x7f65,
  [0x0b08] = 0x7f66, [0x0b09] = 0x7fa2, [0x0b0a] = 0x7fa0, [0x0b0b] = 0x7fa1,
  [0x0b0c] = 0x7fd7, [0x0b0d] = 0x8051, [0x0b0e] = 0x804f, [0x0b0f] = 0x8050,
  [0x0b10] = 0x80fe, [0x0b11] = 0x80d4, [0x0b12] = 0x8143, [0x0b13] = 0x814a,
  [0x0b14] = 0x8152, [0x0b15] = 0x814f, [0x0b16] = 0x8147, [0x0b17] = 0x813d,
  [0x0b18] = 0x814d, [0x0b19] = 0x813a, [0x0b1a] = 0x81e6, [0x0b1b] = 0x81ee,
  [0x0b1c] = 0x81f7, [0x0b1d] = 0x81f8, [0x0b1e] = 0x81f9, [0x0b1f] = 0x8204,
  [0x0b20] = 0x823c, [0x0b21] = 0x823d, [0x0b22] = 0x823f, [0x0b23] = 0x8275,
  [0x0b24] = 0x833b, [0x0b25] = 0x83cf, [0x0b26] = 0x83f9, [0x0b27] = 0x8423,
  [0x0b28] = 0x83c0, [0x0b29] = 0x83e8, [0x0b2a] = 0x8412, [0x0b2b] = 0x83e7,
  [0x0b2c] = 0x83e4, [0x0b2d] = 0x83fc, [0x0b2e] = 0x83f6, [0x0b2f] = 0x8410,
  [0x0b30] = 0x83c6, [0x0b31] = 0x83c8, [0x0b32] = 0x83eb, [0x0b33] = 0x83e3,
  [0x0b34] = 0x83bf, [0x0b35] = 0x8401, [0x0b36] = 0x83dd, [0x0b37] = 0x83e5,
  [0x0b38] = 0x83d8, [0x0b39] = 0x83ff, [0x0b3a] = 0x83e1, [0x0b3b] = 0x83cb,
  [0x0b3c] = 0x83ce, [0x0b3d] = 0x83d6, [0x0b3e] = 0x83f5, [0x0b3f] = 0x83c9,
  [0x0b40] = 0x8409, [0x0b41] = 0x840f, [0x0b42] = 0x83de, [0x0b43] = 0x8411,
  [0x0b44] = 0x8406, [0x0b45] = 0x83c2, [0x0b46] = 0x83f3, [0x0b47] = 0x83d5,
  [0x0b48] = 0x83fa, [0x0b49] = 0x83c7, [0x0b4a] = 0x83d1, [0x0b4b] = 0x83ea,
  [0x0b4c] = 0x8413, [0x0b4d] = 0x839a, [0x0b4e] = 0x83c3, [0x0b4f] = 0x83ec,
  [0x0b50] = 0x83ee, [0x0b51] = 0x83c4, [0x0b52] = 0x83fb, [0x0b53] = 0x83d7,
  [0x0b54] = 0x83e2, [0x0b55] = 0x841b, [0x0b56] = 0x83db, [0x0b57] = 0x83fe,
  [0x0b58] = 0x86d8, [0x0b59] = 0x86e2, [0x0b5a] = 0x86e6, [0x0b5b] = 0x86d3,
  [0x0b5c] = 0x86e3, [0x0b5d] = 0x86da, [0x0b5e] = 0x86ea, [0x0b5f] = 0x86dd,
  [0x0b60] = 0x86eb, [0x0b61] = 0x86dc, [0x0b62] = 0x86ec, [0x0b63] = 0x86e9,
  [0x0b64] = 0x86d7, [0x0b65] = 0x86e8, [0x0b66] = 0x86d1, [0x0b67] = 0x8848,
  [0x0b68] = 0x8856, [0x0b69] = 0x8855, [0x0b6a] = 0x88ba, [0x0b6b] = 0x88d7,
  [0x0b6c] = 0x88b9, [0x0b6d] = 0x88b8, [0x0b6e] = 0x88c0, [0x0b6f] = 0x88be,
  [0x0b70] = 0x88b6, [0x0b71] = 0x88bc, [0x0b72] = 0x88b7, [0x0b73] = 0x88bd,
  [0x0b74] = 0x88b2, [0x0b75] = 0x8901, [0x0b76] = 0x88c9, [0x0b77] = 0x8995,
  [0x0b78] = 0x8998, [0x0b79] = 0x8997, [0x0b7a] = 0x89dd, [0x0b7b] = 0x89da,
  [0x0b7c] = 0x89db, [0x0b7d] = 0x8a4e, [0x0b7e] = 0x8a4d, [0x0b7f] = 0x8a39,
  [0x0b80] = 0x8a59, [0x0b81] = 0x8a40, [0x0b82] = 0x8a57, [0x0b83] = 0x8a58,
  [0x0b84] = 0x8a44, [0x0b85] = 0x8a45, [0x0b86] = 0x8a52, [0x0b87] = 0x8a48,
  [0x0b88] = 0x8a51, [0x0b89] = 0x8a4a, [0x0b8a] = 0x8a4c, [0x0b8b] = 0x8a4f,
  [0x0b8c] = 0x8c5f, [0x0b8d] = 0x8c81, [0x0b8e] = 0x8c80, [0x0b8f] = 0x8cba,
  [0x0b90] = 0x8cbe, [0x0b91] = 0x8cb0, [0x0b92] = 0x8cb9, [0x0b93] = 0x8cb5,
  [0x0b94] = 0x8d84, [0x0b95] = 0x8d80, [0x0b96] = 0x8d89, [0x0b97] = 0x8dd8,
  [0x0b98] = 0x8dd3, [0x0b99] = 0x8dcd, [0x0b9a] = 0x8dc7, [0x0b9b] = 0x8dd6,
  [0x0b9c] = 0x8ddc, [0x0b9d] = 0x8dcf, [0x0b9e] = 0x8dd5, [0x0b9f] = 0x8dd9,
  [0x0ba0] = 0x8dc8, [0x0ba1] = 0x8dd7, [0x0ba2] = 0x8dc5, [0x0ba3] = 0x8eef,
  [0x0ba4] = 0x8ef7, [0x0ba5] = 0x8efa, [0x0ba6] = 0x8ef9, [0x0ba7] = 0x8ee6,
  [0x0ba8] = 0x8eee, [0x0ba9] = 0x8ee5, [0x0baa] = 0x8ef5, [0x0bab] = 0x8ee7,
  [0x0bac] = 0x8ee8, [0x0bad] = 0x8ef6, [0x0bae] = 0x8eeb, [0x0baf] = 0x8ef1,
  [0x0bb0] = 0x8eec, [0x0bb1] = 0x8ef4, [0x0bb2] = 0x8ee9, [0x0bb3] = 0x902d,
  [0x0bb4] = 0x9034, [0x0bb5] = 0x902f, [0x0bb6] = 0x9106, [0x0bb7] = 0x912c,
  [0x0bb8] = 0x9104, [0x0bb9] = 0x90ff, [0x0bba] = 0x90fc, [0x0bbb] = 0x9108,
  [0x0bbc] = 0x90f9, [0x0bbd] = 0x90fb, [0x0bbe] = 0x9101, [0x0bbf] = 0x9100,
  [0x0bc0] = 0x9107, [0x0bc1] = 0x9105, [0x0bc2] = 0x9103, [0x0bc3] = 0x9161,
  [0x0bc4] = 0x9164, [0x0bc5] = 0x915f, [0x0bc6] = 0x9162, [0x0bc7] = 0x9160,
  [0x0bc8] = 0x9201, [0x0bc9] = 0x920a, [0x0bca] = 0x9225, [0x0bcb] = 0x9203,
  [0x0bcc] = 0x921a, [0x0bcd] = 0x9226, [0x0bce] = 0x920f, [0x0bcf] = 0x920c,
  [0x0bd0] = 0x9200, [0x0bd1] = 0x9212, [0x0bd2] = 0x91ff, [0x0bd3] = 0x91fd,
  [0x0bd4] = 0x9206, [0x0bd5] = 0x9204, [0x0bd6] = 0x9227, [0x0bd7] = 0x9202,
  [0x0bd8] = 0x921c, [0x0bd9] = 0x9224, [0x0bda] = 0x9219, [0x0bdb] = 0x9217,
  [0x0bdc] = 0x9205, [0x0bdd] = 0x9216, [0x0bde] = 0x957b, [0x0bdf] = 0x958d,
  [0x0be0] = 0x958c, [0x0be1] = 0x9590, [0x0be2] = 0x9687, [0x0be3] = 0x967e,
  [0x0be4] = 0x9688, [0x0be5] = 0x9689, [0x0be6] = 0x9683, [0x0be7] = 0x9680,
  [0x0be8] = 0x96c2, [0x0be9] = 0x96c8, [0x0bea] = 0x96c3, [0x0beb] = 0x96f1,
  [0x0bec] = 0x96f0, [0x0bed] = 0x976c, [0x0bee] = 0x9770, [0x0bef] = 0x976e,
  [0x0bf0] = 0x9807, [0x0bf1] = 0x98a9, [0x0bf2] = 0x98eb, [0x0bf3] = 0x9ce6,
  [0x0bf4] = 0x9ef9, [0x0bf5] = 0x4e83, [0x0bf6] = 0x4e84, [0x0bf7] = 0x4eb6,
  [0x0bf8] = 0x50bd, [0x0bf9] = 0x50bf, [0x0bfa] = 0x50c6, [0x0bfb] = 0x50ae,
  [0x0bfc] = 0x50c4, [0x0bfd] = 0x50ca, [0x0bfe] = 0x50b4, [0x0bff] = 0x50c8,
  [0x0c00] = 0x50c2, [0x0c01] = 0x50b0, [0x0c02] = 0x50c1, [0x0c03] = 0x50ba,
  [0x0c04] = 0x50b1, [0x0c05] = 0x50cb, [0x0c06] = 0x50c9, [0x0c07] = 0x50b6,
  [0x0c08] = 0x50b8, [0x0c09] = 0x51d7, [0x0c0a] = 0x527a, [0x0c0b] = 0x5278,
  [0x0c0c] = 0x527b, [0x0c0d] = 0x527c, [0x0c0e] = 0x55c3, [0x0c0f] = 0x55db,
  [0x0c10] = 0x55cc, [0x0c11] = 0x55d0, [0x0c12] = 0x55cb, [0x0c13] = 0x55ca,
  [0x0c14] = 0x55dd, [0x0c15] = 0x55c0, [0x0c16] = 0x55d4, [0x0c17] = 0x55c4,
  [0x0c18] = 0x55e9, [0x0c19] = 0x55bf, [0x0c1a] = 0x55d2, [0x0c1b] = 0x558d,
  [0x0c1c] = 0x55cf, [0x0c1d] = 0x55d5, [0x0c1e] = 0x55e2, [0x0c1f] = 0x55d6,
  [0x0c20] = 0x55c8, [0x0c21] = 0x55f2, [0x0c22] = 0x55cd, [0x0c23] = 0x55d9,
  [0x0c24] = 0x55c2, [0x0c25] = 0x5714, [0x0c26] = 0x5853, [0x0c27] = 0x5868,
  [0x0c28] = 0x5864, [0x0c29] = 0x584f, [0x0c2a] = 0x584d, [0x0c2b] = 0x5849,
  [0x0c2c] = 0x586f, [0x0c2d] = 0x5855, [0x0c2e] = 0x584e, [0x0c2f] = 0x585d,
  [0x0c30] = 0x5859, [0x0c31] = 0x5865, [0x0c32] = 0x585b, [0x0c33] = 0x583d,
  [0x0c34] = 0x5863, [0x0c35] = 0x5871, [0x0c36] = 0x58fc, [0x0c37] = 0x5ac7,
  [0x0c38] = 0x5ac4, [0x0c39] = 0x5acb, [0x0c3a] = 0x5aba, [0x0c3b] = 0x5ab8,
  [0x0c3c] = 0x5ab1, [0x0c3d] = 0x5ab5, [0x0c3e] = 0x5ab0, [0x0c3f] = 0x5abf,
  [0x0c40] = 0x5ac8, [0x0c41] = 0x5abb, [0x0c42] = 0x5ac6, [0x0c43] = 0x5ab7,
  [0x0c44] = 0x5ac0, [0x0c45] = 0x5aca, [0x0c46] = 0x5ab4, [0x0c47] = 0x5ab6,
  [0x0c48] = 0x5acd, [0x0c49] = 0x5ab9, [0x0c4a] = 0x5a90, [0x0c4b] = 0x5bd6,
  [0x0c4c] = 0x5bd8, [0x0c4d] = 0x5bd9, [0x0c4e] = 0x5c1f, [0x0c4f] = 0x5c33,
  [0x0c50] = 0x5d71, [0x0c51] = 0x5d63, [0x0c52] = 0x5d4a, [0x0c53] = 0x5d65,
  [0x0c54] = 0x5d72, [0x0c55] = 0x5d6c, [0x0c56] = 0x5d5e, [0x0c57] = 0x5d68,
  [0x0c58] = 0x5d67, [0x0c59] = 0x5d62, [0x0c5a] = 0x5df0, [0x0c5b] = 0x5e4f,
  [0x0c5c] = 0x5e4e, [0x0c5d] = 0x5e4a, [0x0c5e] = 0x5e4d, [0x0c5f] = 0x5e4b,
  [0x0c60] = 0x5ec5, [0x0c61] = 0x5ecc, [0x0c62] = 0x5ec6, [0x0c63] = 0x5ecb,
  [0x0c64] = 0x5ec7, [0x0c65] = 0x5f40, [0x0c66] = 0x5faf, [0x0c67] = 0x5fad,
  [0x0c68] = 0x60f7, [0x0c69] = 0x6149, [0x0c6a] = 0x614a, [0x0c6b] = 0x612b,
  [0x0c6c] = 0x6145, [0x0c6d] = 0x6136, [0x0c6e] = 0x6132, [0x0c6f] = 0x612e,
  [0x0c70] = 0x6146, [0x0c71] = 0x612f, [0x0c72] = 0x614f, [0x0c73] = 0x6129,
  [0x0c74] = 0x6140, [0x0c75] = 0x6220, [0x0c76] = 0x9168, [0x0c77] = 0x6223,
  [0x0c78] = 0x6225, [0x0c79] = 0x6224, [0x0c7a] = 0x63c5, [0x0c7b] = 0x63f1,
  [0x0c7c] = 0x63eb, [0x0c7d] = 0x6410, [0x0c7e] = 0x6412, [0x0c7f] = 0x6409,
  [0x0c80] = 0x6420, [0x0c81] = 0x6424, [0x0c82] = 0x6433, [0x0c83] = 0x6443,
  [0x0c84] = 0x641f, [0x0c85] = 0x6415, [0x0c86] = 0x6418, [0x0c87] = 0x6439,
  [0x0c88] = 0x6437, [0x0c89] = 0x6422, [0x0c8a] = 0x6423, [0x0c8b] = 0x640c,
  [0x0c8c] = 0x6426, [0x0c8d] = 0x6430, [0x0c8e] = 0x6428, [0x0c8f] = 0x6441,
  [0x0c90] = 0x6435, [0x0c91] = 0x642f, [0x0c92] = 0x640a, [0x0c93] = 0x641a,
  [0x0c94] = 0x6440, [0x0c95] = 0x6425, [0x0c96] = 0x6427, [0x0c97] = 0x640b,
  [0x0c98] = 0x63e7, [0x0c99] = 0x641b, [0x0c9a] = 0x642e, [0x0c9b] = 0x6421,
  [0x0c9c] = 0x640e, [0x0c9d] = 0x656f, [0x0c9e] = 0x6592, [0x0c9f] = 0x65d3,
  [0x0ca0] = 0x6686, [0x0ca1] = 0x668c, [0x0ca2] = 0x6695, [0x0ca3] = 0x6690,
  [0x0ca4] = 0x668b, [0x0ca5] = 0x668a, [0x0ca6] = 0x6699, [0x0ca7] = 0x6694,
  [0x0ca8] = 0x6678, [0x0ca9] = 0x6720, [0x0caa] = 0x6966, [0x0cab] = 0x695f,
  [0x0cac] = 0x6938, [0x0cad] = 0x694e, [0x0cae] = 0x6962, [0x0caf] = 0x6971,
  [0x0cb0] = 0x693f, [0x0cb1] = 0x6945, [0x0cb2] = 0x696a, [0x0cb3] = 0x6939,
  [0x0cb4] = 0x6942, [0x0cb5] = 0x6957, [0x0cb6] = 0x6959, [0x0cb7] = 0x697a,
  [0x0cb8] = 0x6948, [0x0cb9] = 0x6949, [0x0cba] = 0x6935, [0x0cbb] = 0x696c,
  [0x0cbc] = 0x6933, [0x0cbd] = 0x693d, [0x0cbe] = 0x6965, [0x0cbf] = 0x68f0,
  [0x0cc0] = 0x6978, [0x0cc1] = 0x6934, [0x0cc2] = 0x6969, [0x0cc3] = 0x6940,
  [0x0cc4] = 0x696f, [0x0cc5] = 0x6944, [0x0cc6] = 0x6976, [0x0cc7] = 0x6958,
  [0x0cc8] = 0x6941, [0x0cc9] = 0x6974, [0x0cca] = 0x694c, [0x0ccb] = 0x693b,
  [0x0ccc] = 0x694b, [0x0ccd] = 0x6937, [0x0cce] = 0x695c, [0x0ccf] = 0x694f,
  [0x0cd0] = 0x6951, [0x0cd1] = 0x6932, [0x0cd2] = 0x6952, [0x0cd3] = 0x692f,
  [0x0cd4] = 0x697b, [0x0cd5] = 0x693c, [0x0cd6] = 0x6b46, [0x0cd7] = 0x6b45,
  [0x0cd8] = 0x6b43, [0x0cd9] = 0x6b42, [0x0cda] = 0x6b48, [0x0cdb] = 0x6b41,
  [0x0cdc] = 0x6b9b, [0x0cdd] = 0x6bfb, [0x0cde] = 0x6bfc, [0x0cdf] = 0x6bf9,
  [0x0ce0] = 0x6bf7, [0x0ce1] = 0x6bf8, [0x0ce2] = 0x6e9b, [0x0ce3] = 0x6ed6,
  [0x0ce4] = 0x6ec8, [0x0ce5] = 0x6e8f, [0x0ce6] = 0x6ec0, [0x0ce7] = 0x6e9f,
  [0x0ce8] = 0x6e93, [0x0ce9] = 0x6e94, [0x0cea] = 0x6ea0, [0x0ceb] = 0x6eb1,
  [0x0cec] = 0x6eb9, [0x0ced] = 0x6ec6, [0x0cee] = 0x6ed2, [0x0cef] = 0x6ebd,
  [0x0cf0] = 0x6ec1, [0x0cf1] = 0x6e9e, [0x0cf2] = 0x6ec9, [0x0cf3] = 0x6eb7,
  [0x0cf4] = 0x6eb0, [0x0cf5] = 0x6ecd, [0x0cf6] = 0x6ea6, [0x0cf7] = 0x6ecf,
  [0x0cf8] = 0x6eb2, [0x0cf9] = 0x6ebe, [0x0cfa] = 0x6ec3, [0x0cfb] = 0x6edc,
  [0x0cfc] = 0x6ed8, [0x0cfd] = 0x6e99, [0x0cfe] = 0x6e92, [0x0cff] = 0x6e8e,
  [0x0d00] = 0x6e8d, [0x0d01] = 0x6ea4, [0x0d02] = 0x6ea1, [0x0d03] = 0x6ebf,
  [0x0d04] = 0x6eb3, [0x0d05] = 0x6ed0, [0x0d06] = 0x6eca, [0x0d07] = 0x6e97,
  [0x0d08] = 0x6eae, [0x0d09] = 0x6ea3, [0x0d0a] = 0x7147, [0x0d0b] = 0x7154,
  [0x0d0c] = 0x7152, [0x0d0d] = 0x7163, [0x0d0e] = 0x7160, [0x0d0f] = 0x7141,
  [0x0d10] = 0x715d, [0x0d11] = 0x7162, [0x0d12] = 0x7172, [0x0d13] = 0x7178,
  [0x0d14] = 0x716a, [0x0d15] = 0x7161, [0x0d16] = 0x7142, [0x0d17] = 0x7158,
  [0x0d18] = 0x7143, [0x0d19] = 0x714b, [0x0d1a] = 0x7170, [0x0d1b] = 0x715f,
  [0x0d1c] = 0x7150, [0x0d1d] = 0x7153, [0x0d1e] = 0x7144, [0x0d1f] = 0x714d,
  [0x0d20] = 0x715a, [0x0d21] = 0x724f, [0x0d22] = 0x728d, [0x0d23] = 0x728c,
  [0x0d24] = 0x7291, [0x0d25] = 0x7290, [0x0d26] = 0x728e, [0x0d27] = 0x733c,
  [0x0d28] = 0x7342, [0x0d29] = 0x733b, [0x0d2a] = 0x733a, [0x0d2b] = 0x7340,
  [0x0d2c] = 0x734a, [0x0d2d] = 0x7349, [0x0d2e] = 0x7444, [0x0d2f] = 0x744a,
  [0x0d30] = 0x744b, [0x0d31] = 0x7452, [0x0d32] = 0x7451, [0x0d33] = 0x7457,
  [0x0d34] = 0x7440, [0x0d35] = 0x744f, [0x0d36] = 0x7450, [0x0d37] = 0x744e,
  [0x0d38] = 0x7442, [0x0d39] = 0x7446, [0x0d3a] = 0x744d, [0x0d3b] = 0x7454,
  [0x0d3c] = 0x74e1, [0x0d3d] = 0x74ff, [0x0d3e] = 0x74fe, [0x0d3f] = 0x74fd,
  [0x0d40] = 0x751d, [0x0d41] = 0x7579, [0x0d42] = 0x7577, [0x0d43] = 0x6983,
  [0x0d44] = 0x75ef, [0x0d45] = 0x760f, [0x0d46] = 0x7603, [0x0d47] = 0x75f7,
  [0x0d48] = 0x75fe, [0x0d49] = 0x75fc, [0x0d4a] = 0x75f9, [0x0d4b] = 0x75f8,
  [0x0d4c] = 0x7610, [0x0d4d] = 0x75fb, [0x0d4e] = 0x75f6, [0x0d4f] = 0x75ed,
  [0x0d50] = 0x75f5, [0x0d51] = 0x75fd, [0x0d52] = 0x7699, [0x0d53] = 0x76b5,
  [0x0d54] = 0x76dd, [0x0d55] = 0x7755, [0x0d56] = 0x775f, [0x0d57] = 0x7760,
  [0x0d58] = 0x7752, [0x0d59] = 0x7756, [0x0d5a] = 0x775a, [0x0d5b] = 0x7769,
  [0x0d5c] = 0x7767, [0x0d5d] = 0x7754, [0x0d5e] = 0x7759, [0x0d5f] = 0x776d,
  [0x0d60] = 0x77e0, [0x0d61] = 0x7887, [0x0d62] = 0x789a, [0x0d63] = 0x7894,
  [0x0d64] = 0x788f, [0x0d65] = 0x7884, [0x0d66] = 0x7895, [0x0d67] = 0x7885,
  [0x0d68] = 0x7886, [0x0d69] = 0x78a1, [0x0d6a] = 0x7883, [0x0d6b] = 0x7879,
  [0x0d6c] = 0x7899, [0x0d6d] = 0x7880, [0x0d6e] = 0x7896, [0x0d6f] = 0x787b,
  [0x0d70] = 0x797c, [0x0d71] = 0x7982, [0x0d72] = 0x797d, [0x0d73] = 0x7979,
  [0x0d74] = 0x7a11, [0x0d75] = 0x7a18, [0x0d76] = 0x7a19, [0x0d77] = 0x7a12,
  [0x0d78] = 0x7a17, [0x0d79] = 0x7a15, [0x0d7a] = 0x7a22, [0x0d7b] = 0x7a13,
  [0x0d7c] = 0x7a1b, [0x0d7d] = 0x7a10, [0x0d7e] = 0x7aa3, [0x0d7f] = 0x7aa2,
  [0x0d80] = 0x7a9e, [0x0d81] = 0x7aeb, [0x0d82] = 0x7b66, [0x0d83] = 0x7b64,
  [0x0d84] = 0x7b6d, [0x0d85] = 0x7b74, [0x0d86] = 0x7b69, [0x0d87] = 0x7b72,
  [0x0d88] = 0x7b65, [0x0d89] = 0x7b73, [0x0d8a] = 0x7b71, [0x0d8b] = 0x7b70,
  [0x0d8c] = 0x7b61, [0x0d8d] = 0x7b78, [0x0d8e] = 0x7b76, [0x0d8f] = 0x7b63,
  [0x0d90] = 0x7cb2, [0x0d91] = 0x7cb4, [0x0d92] = 0x7caf, [0x0d93] = 0x7d88,
  [0x0d94] = 0x7d86, [0x0d95] = 0x7d80, [0x0d96] = 0x7d8d, [0x0d97] = 0x7d7f,
  [0x0d98] = 0x7d85, [0x0d99] = 0x7d7a, [0x0d9a] = 0x7d8e, [0x0d9b] = 0x7d7b,
  [0x0d9c] = 0x7d83, [0x0d9d] = 0x7d7c, [0x0d9e] = 0x7d8c, [0x0d9f] = 0x7d94,
  [0x0da0] = 0x7d84, [0x0da1] = 0x7d7d, [0x0da2] = 0x7d92, [0x0da3] = 0x7f6d,
  [0x0da4] = 0x7f6b, [0x0da5] = 0x7f67, [0x0da6] = 0x7f68, [0x0da7] = 0x7f6c,
  [0x0da8] = 0x7fa6, [0x0da9] = 0x7fa5, [0x0daa] = 0x7fa7, [0x0dab] = 0x7fdb,
  [0x0dac] = 0x7fdc, [0x0dad] = 0x8021, [0x0dae] = 0x8164, [0x0daf] = 0x8160,
  [0x0db0] = 0x8177, [0x0db1] = 0x815c, [0x0db2] = 0x8169, [0x0db3] = 0x815b,
  [0x0db4] = 0x8162, [0x0db5] = 0x8172, [0x0db6] = 0x6721, [0x0db7] = 0x815e,
  [0x0db8] = 0x8176, [0x0db9] = 0x8167, [0x0dba] = 0x816f, [0x0dbb] = 0x8144,
  [0x0dbc] = 0x8161, [0x0dbd] = 0x821d, [0x0dbe] = 0x8249, [0x0dbf] = 0x8244,
  [0x0dc0] = 0x8240, [0x0dc1] = 0x8242, [0x0dc2] = 0x8245, [0x0dc3] = 0x84f1,
  [0x0dc4] = 0x843f, [0x0dc5] = 0x8456, [0x0dc6] = 0x8476, [0x0dc7] = 0x8479,
  [0x0dc8] = 0x848f, [0x0dc9] = 0x848d, [0x0dca] = 0x8465, [0x0dcb] = 0x8451,
  [0x0dcc] = 0x8440, [0x0dcd] = 0x8486, [0x0dce] = 0x8467, [0x0dcf] = 0x8430,
  [0x0dd0] = 0x844d, [0x0dd1] = 0x847d, [0x0dd2] = 0x845a, [0x0dd3] = 0x8459,
  [0x0dd4] = 0x8474, [0x0dd5] = 0x8473, [0x0dd6] = 0x845d, [0x0dd7] = 0x8507,
  [0x0dd8] = 0x845e, [0x0dd9] = 0x8437, [0x0dda] = 0x843a, [0x0ddb] = 0x8434,
  [0x0ddc] = 0x847a, [0x0ddd] = 0x8443, [0x0dde] = 0x8478, [0x0ddf] = 0x8432,
  [0x0de0] = 0x8445, [0x0de1] = 0x8429, [0x0de2] = 0x83d9, [0x0de3] = 0x844b,
  [0x0de4] = 0x842f, [0x0de5] = 0x8442, [0x0de6] = 0x842d, [0x0de7] = 0x845f,
  [0x0de8] = 0x8470, [0x0de9] = 0x8439, [0x0dea] = 0x844e, [0x0deb] = 0x844c,
  [0x0dec] = 0x8452, [0x0ded] = 0x846f, [0x0dee] = 0x84c5, [0x0def] = 0x848e,
  [0x0df0] = 0x843b, [0x0df1] = 0x8447, [0x0df2] = 0x8436, [0x0df3] = 0x8433,
  [0x0df4] = 0x8468, [0x0df5] = 0x847e, [0x0df6] = 0x8444, [0x0df7] = 0x842b,
  [0x0df8] = 0x8460, [0x0df9] = 0x8454, [0x0dfa] = 0x846e, [0x0dfb] = 0x8450,
  [0x0dfc] = 0x870b, [0x0dfd] = 0x8704, [0x0dfe] = 0x86f7, [0x0dff] = 0x870c,
  [0x0e00] = 0x86fa, [0x0e01] = 0x86d6, [0x0e02] = 0x86f5, [0x0e03] = 0x874d,
  [0x0e04] = 0x86f8, [0x0e05] = 0x870e, [0x0e06] = 0x8709, [0x0e07] = 0x8701,
  [0x0e08] = 0x86f6, [0x0e09] = 0x870d, [0x0e0a] = 0x8705, [0x0e0b] = 0x88d6,
  [0x0e0c] = 0x88cb, [0x0e0d] = 0x88cd, [0x0e0e] = 0x88ce, [0x0e0f] = 0x88de,
  [0x0e10] = 0x88db, [0x0e11] = 0x88da, [0x0e12] = 0x88cc, [0x0e13] = 0x88d0,
  [0x0e14] = 0x8985, [0x0e15] = 0x899b, [0x0e16] = 0x89df, [0x0e17] = 0x89e5,
  [0x0e18] = 0x89e4, [0x0e19] = 0x89e1, [0x0e1a] = 0x89e0, [0x0e1b] = 0x89e2,
  [0x0e1c] = 0x89dc, [0x0e1d] = 0x89e6, [0x0e1e] = 0x8a76, [0x0e1f] = 0x8a86,
  [0x0e20] = 0x8a7f, [0x0e21] = 0x8a61, [0x0e22] = 0x8a3f, [0x0e23] = 0x8a77,
  [0x0e24] = 0x8a82, [0x0e25] = 0x8a84, [0x0e26] = 0x8a75, [0x0e27] = 0x8a83,
  [0x0e28] = 0x8a81, [0x0e29] = 0x8a74, [0x0e2a] = 0x8a7a, [0x0e2b] = 0x8c3c,
  [0x0e2c] = 0x8c4b, [0x0e2d] = 0x8c4a, [0x0e2e] = 0x8c65, [0x0e2f] = 0x8c64,
  [0x0e30] = 0x8c66, [0x0e31] = 0x8c86, [0x0e32] = 0x8c84, [0x0e33] = 0x8c85,
  [0x0e34] = 0x8ccc, [0x0e35] = 0x8d68, [0x0e36] = 0x8d69, [0x0e37] = 0x8d91,
  [0x0e38] = 0x8d8c, [0x0e39] = 0x8d8e, [0x0e3a] = 0x8d8f, [0x0e3b] = 0x8d8d,
  [0x0e3c] = 0x8d93, [0x0e3d] = 0x8d94, [0x0e3e] = 0x8d90, [0x0e3f] = 0x8d92,
  [0x0e40] = 0x8df0, [0x0e41] = 0x8de0, [0x0e42] = 0x8dec, [0x0e43] = 0x8df1,
  [0x0e44] = 0x8dee, [0x0e45] = 0x8dd0, [0x0e46] = 0x8de9, [0x0e47] = 0x8de3,
  [0x0e48] = 0x8de2, [0x0e49] = 0x8de7, [0x0e4a] = 0x8df2, [0x0e4b] = 0x8deb,
  [0x0e4c] = 0x8df4, [0x0e4d] = 0x8f06, [0x0e4e] = 0x8eff, [0x0e4f] = 0x8f01,
  [0x0e50] = 0x8f00, [0x0e51] = 0x8f05, [0x0e52] = 0x8f07, [0x0e53] = 0x8f08,
  [0x0e54] = 0x8f02, [0x0e55] = 0x8f0b, [0x0e56] = 0x9052, [0x0e57] = 0x903f,
  [0x0e58] = 0x9044, [0x0e59] = 0x9049, [0x0e5a] = 0x903d, [0x0e5b] = 0x9110,
  [0x0e5c] = 0x910d, [0x0e5d] = 0x910f, [0x0e5e] = 0x9111, [0x0e5f] = 0x9116,
  [0x0e60] = 0x9114, [0x0e61] = 0x910b, [0x0e62] = 0x910e, [0x0e63] = 0x916e,
  [0x0e64] = 0x916f, [0x0e65] = 0x9248, [0x0e66] = 0x9252, [0x0e67] = 0x9230,
  [0x0e68] = 0x923a, [0x0e69] = 0x9266, [0x0e6a] = 0x9233, [0x0e6b] = 0x9265,
  [0x0e6c] = 0x925e, [0x0e6d] = 0x9283, [0x0e6e] = 0x922e, [0x0e6f] = 0x924a,
  [0x0e70] = 0x9246, [0x0e71] = 0x926d, [0x0e72] = 0x926c, [0x0e73] = 0x924f,
  [0x0e74] = 0x9260, [0x0e75] = 0x9267, [0x0e76] = 0x926f, [0x0e77] = 0x9236,
  [0x0e78] = 0x9261, [0x0e79] = 0x9270, [0x0e7a] = 0x9231, [0x0e7b] = 0x9254,
  [0x0e7c] = 0x9263, [0x0e7d] = 0x9250, [0x0e7e] = 0x9272, [0x0e7f] = 0x924e,
  [0x0e80] = 0x9253, [0x0e81] = 0x924c, [0x0e82] = 0x9256, [0x0e83] = 0x9232,
  [0x0e84] = 0x959f, [0x0e85] = 0x959c, [0x0e86] = 0x959e, [0x0e87] = 0x959b,
  [0x0e88] = 0x9692, [0x0e89] = 0x9693, [0x0e8a] = 0x9691, [0x0e8b] = 0x9697,
  [0x0e8c] = 0x96ce, [0x0e8d] = 0x96fa, [0x0e8e] = 0x96fd, [0x0e8f] = 0x96f8,
  [0x0e90] = 0x96f5, [0x0e91] = 0x9773, [0x0e92] = 0x9777, [0x0e93] = 0x9778,
  [0x0e94] = 0x9772, [0x0e95] = 0x980f, [0x0e96] = 0x980d, [0x0e97] = 0x980e,
  [0x0e98] = 0x98ac, [0x0e99] = 0x98f6, [0x0e9a] = 0x98f9, [0x0e9b] = 0x99af,
  [0x0e9c] = 0x99b2, [0x0e9d] = 0x99b0, [0x0e9e] = 0x99b5, [0x0e9f] = 0x9aad,
  [0x0ea0] = 0x9aab, [0x0ea1] = 0x9b5b, [0x0ea2] = 0x9cea, [0x0ea3] = 0x9ced,
  [0x0ea4] = 0x9ce7, [0x0ea5] = 0x9e80, [0x0ea6] = 0x9efd, [0x0ea7] = 0x50e6,
  [0x0ea8] = 0x50d4, [0x0ea9] = 0x50d7, [0x0eaa] = 0x50e8, [0x0eab] = 0x50f3,
  [0x0eac] = 0x50db, [0x0ead] = 0x50ea, [0x0eae] = 0x50dd, [0x0eaf] = 0x50e4,
  [0x0eb0] = 0x50d3, [0x0eb1] = 0x50ec, [0x0eb2] = 0x50f0, [0x0eb3] = 0x50ef,
  [0x0eb4] = 0x50e3, [0x0eb5] = 0x50e0, [0x0eb6] = 0x51d8, [0x0eb7] = 0x5280,
  [0x0eb8] = 0x5281, [0x0eb9] = 0x52e9, [0x0eba] = 0x52eb, [0x0ebb] = 0x5330,
  [0x0ebc] = 0x53ac, [0x0ebd] = 0x5627, [0x0ebe] = 0x5615, [0x0ebf] = 0x560c,
  [0x0ec0] = 0x5612, [0x0ec1] = 0x55fc, [0x0ec2] = 0x560f, [0x0ec3] = 0x561c,
  [0x0ec4] = 0x5601, [0x0ec5] = 0x5613, [0x0ec6] = 0x5602, [0x0ec7] = 0x55fa,
  [0x0ec8] = 0x561d, [0x0ec9] = 0x5604, [0x0eca] = 0x55ff, [0x0ecb] = 0x55f9,
  [0x0ecc] = 0x5889, [0x0ecd] = 0x587c, [0x0ece] = 0x5890, [0x0ecf] = 0x5898,
  [0x0ed0] = 0x5886, [0x0ed1] = 0x5881, [0x0ed2] = 0x587f, [0x0ed3] = 0x5874,
  [0x0ed4] = 0x588b, [0x0ed5] = 0x587a, [0x0ed6] = 0x5887, [0x0ed7] = 0x5891,
  [0x0ed8] = 0x588e, [0x0ed9] = 0x5876, [0x0eda] = 0x5882, [0x0edb] = 0x5888,
  [0x0edc] = 0x587b, [0x0edd] = 0x5894, [0x0ede] = 0x588f, [0x0edf] = 0x58fe,
  [0x0ee0] = 0x596b, [0x0ee1] = 0x5adc, [0x0ee2] = 0x5aee, [0x0ee3] = 0x5ae5,
  [0x0ee4] = 0x5ad5, [0x0ee5] = 0x5aea, [0x0ee6] = 0x5ada, [0x0ee7] = 0x5aed,
  [0x0ee8] = 0x5aeb, [0x0ee9] = 0x5af3, [0x0eea] = 0x5ae2, [0x0eeb] = 0x5ae0,
  [0x0eec] = 0x5adb, [0x0eed] = 0x5aec, [0x0eee] = 0x5ade, [0x0eef] = 0x5add,
  [0x0ef0] = 0x5ad9, [0x0ef1] = 0x5ae8, [0x0ef2] = 0x5adf, [0x0ef3] = 0x5b77,
  [0x0ef4] = 0x5be0, [0x0ef5] = 0x5be3, [0x0ef6] = 0x5c63, [0x0ef7] = 0x5d82,
  [0x0ef8] = 0x5d80, [0x0ef9] = 0x5d7d, [0x0efa] = 0x5d86, [0x0efb] = 0x5d7a,
  [0x0efc] = 0x5d81, [0x0efd] = 0x5d77, [0x0efe] = 0x5d8a, [0x0eff] = 0x5d89,
  [0x0f00] = 0x5d88, [0x0f01] = 0x5d7e, [0x0f02] = 0x5d7c, [0x0f03] = 0x5d8d,
  [0x0f04] = 0x5d79, [0x0f05] = 0x5d7f, [0x0f06] = 0x5e58, [0x0f07] = 0x5e59,
  [0x0f08] = 0x5e53, [0x0f09] = 0x5ed8, [0x0f0a] = 0x5ed1, [0x0f0b] = 0x5ed7,
  [0x0f0c] = 0x5ece, [0x0f0d] = 0x5edc, [0x0f0e] = 0x5ed5, [0x0f0f] = 0x5ed9,
  [0x0f10] = 0x5ed2, [0x0f11] = 0x5ed4, [0x0f12] = 0x5f44, [0x0f13] = 0x5f43,
  [0x0f14] = 0x5f6f, [0x0f15] = 0x5fb6, [0x0f16] = 0x612c, [0x0f17] = 0x6128,
  [0x0f18] = 0x6141, [0x0f19] = 0x615e, [0x0f1a] = 0x6171, [0x0f1b] = 0x6173,
  [0x0f1c] = 0x6152, [0x0f1d] = 0x6153, [0x0f1e] = 0x6172, [0x0f1f] = 0x616c,
  [0x0f20] = 0x6180, [0x0f21] = 0x6174, [0x0f22] = 0x6154, [0x0f23] = 0x617a,
  [0x0f24] = 0x615b, [0x0f25] = 0x6165, [0x0f26] = 0x613b, [0x0f27] = 0x616a,
  [0x0f28] = 0x6161, [0x0f29] = 0x6156, [0x0f2a] = 0x6229, [0x0f2b] = 0x6227,
  [0x0f2c] = 0x622b, [0x0f2d] = 0x642b, [0x0f2e] = 0x644d, [0x0f2f] = 0x645b,
  [0x0f30] = 0x645d, [0x0f31] = 0x6474, [0x0f32] = 0x6476, [0x0f33] = 0x6472,
  [0x0f34] = 0x6473, [0x0f35] = 0x647d, [0x0f36] = 0x6475, [0x0f37] = 0x6466,
  [0x0f38] = 0x64a6, [0x0f39] = 0x644e, [0x0f3a] = 0x6482, [0x0f3b] = 0x645e,
  [0x0f3c] = 0x645c, [0x0f3d] = 0x644b, [0x0f3e] = 0x6453, [0x0f3f] = 0x6460,
  [0x0f40] = 0x6450, [0x0f41] = 0x647f, [0x0f42] = 0x643f, [0x0f43] = 0x646c,
  [0x0f44] = 0x646b, [0x0f45] = 0x6459, [0x0f46] = 0x6465, [0x0f47] = 0x6477,
  [0x0f48] = 0x6573, [0x0f49] = 0x65a0, [0x0f4a] = 0x66a1, [0x0f4b] = 0x66a0,
  [0x0f4c] = 0x669f, [0x0f4d] = 0x6705, [0x0f4e] = 0x6704, [0x0f4f] = 0x6722,
  [0x0f50] = 0x69b1, [0x0f51] = 0x69b6, [0x0f52] = 0x69c9, [0x0f53] = 0x69a0,
  [0x0f54] = 0x69ce, [0x0f55] = 0x6996, [0x0f56] = 0x69b0, [0x0f57] = 0x69ac,
  [0x0f58] = 0x69bc, [0x0f59] = 0x6991, [0x0f5a] = 0x6999, [0x0f5b] = 0x698e,
  [0x0f5c] = 0x69a7, [0x0f5d] = 0x698d, [0x0f5e] = 0x69a9, [0x0f5f] = 0x69be,
  [0x0f60] = 0x69af, [0x0f61] = 0x69bf, [0x0f62] = 0x69c4, [0x0f63] = 0x69bd,
  [0x0f64] = 0x69a4, [0x0f65] = 0x69d4, [0x0f66] = 0x69b9, [0x0f67] = 0x69ca,
  [0x0f68] = 0x699a, [0x0f69] = 0x69cf, [0x0f6a] = 0x69b3, [0x0f6b] = 0x6993,
  [0x0f6c] = 0x69aa, [0x0f6d] = 0x69a1, [0x0f6e] = 0x699e, [0x0f6f] = 0x69d9,
  [0x0f70] = 0x6997, [0x0f71] = 0x6990, [0x0f72] = 0x69c2, [0x0f73] = 0x69b5,
  [0x0f74] = 0x69a5, [0x0f75] = 0x69c6, [0x0f76] = 0x6b4a, [0x0f77] = 0x6b4d,
  [0x0f78] = 0x6b4b, [0x0f79] = 0x6b9e, [0x0f7a] = 0x6b9f, [0x0f7b] = 0x6ba0,
  [0x0f7c] = 0x6bc3, [0x0f7d] = 0x6bc4, [0x0f7e] = 0x6bfe, [0x0f7f] = 0x6ece,
  [0x0f80] = 0x6ef5, [0x0f81] = 0x6ef1, [0x0f82] = 0x6f03, [0x0f83] = 0x6f25,
  [0x0f84] = 0x6ef8, [0x0f85] = 0x6f37, [0x0f86] = 0x6efb, [0x0f87] = 0x6f2e,
  [0x0f88] = 0x6f09, [0x0f89] = 0x6f4e, [0x0f8a] = 0x6f19, [0x0f8b] = 0x6f1a,
  [0x0f8c] = 0x6f27, [0x0f8d] = 0x6f18, [0x0f8e] = 0x6f3b, [0x0f8f] = 0x6f12,
  [0x0f90] = 0x6eed, [0x0f91] = 0x6f0a, [0x0f92] = 0x6f36, [0x0f93] = 0x6f73,
  [0x0f94] = 0x6ef9, [0x0f95] = 0x6eee, [0x0f96] = 0x6f2d, [0x0f97] = 0x6f40,
  [0x0f98] = 0x6f30, [0x0f99] = 0x6f3c, [0x0f9a] = 0x6f35, [0x0f9b] = 0x6eeb,
  [0x0f9c] = 0x6f07, [0x0f9d] = 0x6f0e, [0x0f9e] = 0x6f43, [0x0f9f] = 0x6f05,
  [0x0fa0] = 0x6efd, [0x0fa1] = 0x6ef6, [0x0fa2] = 0x6f39, [0x0fa3] = 0x6f1c,
  [0x0fa4] = 0x6efc, [0x0fa5] = 0x6f3a, [0x0fa6] = 0x6f1f, [0x0fa7] = 0x6f0d,
  [0x0fa8] = 0x6f1e, [0x0fa9] = 0x6f08, [0x0faa] = 0x6f21, [0x0fab] = 0x7187,
  [0x0fac] = 0x7190, [0x0fad] = 0x7189, [0x0fae] = 0x7180, [0x0faf] = 0x7185,
  [0x0fb0] = 0x7182, [0x0fb1] = 0x718f, [0x0fb2] = 0x717b, [0x0fb3] = 0x7186,
  [0x0fb4] = 0x7181, [0x0fb5] = 0x7197, [0x0fb6] = 0x7244, [0x0fb7] = 0x7253,
  [0x0fb8] = 0x7297, [0x0fb9] = 0x7295, [0x0fba] = 0x7293, [0x0fbb] = 0x7343,
  [0x0fbc] = 0x734d, [0x0fbd] = 0x7351, [0x0fbe] = 0x734c, [0x0fbf] = 0x7462,
  [0x0fc0] = 0x7473, [0x0fc1] = 0x7471, [0x0fc2] = 0x7475, [0x0fc3] = 0x7472,
  [0x0fc4] = 0x7467, [0x0fc5] = 0x746e, [0x0fc6] = 0x7500, [0x0fc7] = 0x7502,
  [0x0fc8] = 0x7503, [0x0fc9] = 0x757d, [0x0fca] = 0x7590, [0x0fcb] = 0x7616,
  [0x0fcc] = 0x7608, [0x0fcd] = 0x760c, [0x0fce] = 0x7615, [0x0fcf] = 0x7611,
  [0x0fd0] = 0x760a, [0x0fd1] = 0x7614, [0x0fd2] = 0x76b8, [0x0fd3] = 0x7781,
  [0x0fd4] = 0x777c, [0x0fd5] = 0x7785, [0x0fd6] = 0x7782, [0x0fd7] = 0x776e,
  [0x0fd8] = 0x7780, [0x0fd9] = 0x776f, [0x0fda] = 0x777e, [0x0fdb] = 0x7783,
  [0x0fdc] = 0x78b2, [0x0fdd] = 0x78aa, [0x0fde] = 0x78b4, [0x0fdf] = 0x78ad,
  [0x0fe0] = 0x78a8, [0x0fe1] = 0x787e, [0x0fe2] = 0x78ab, [0x0fe3] = 0x789e,
  [0x0fe4] = 0x78a5, [0x0fe5] = 0x78a0, [0x0fe6] = 0x78ac, [0x0fe7] = 0x78a2,
  [0x0fe8] = 0x78a4, [0x0fe9] = 0x7998, [0x0fea] = 0x798a, [0x0feb] = 0x798b,
  [0x0fec] = 0x7996, [0x0fed] = 0x7995, [0x0fee] = 0x7994, [0x0fef] = 0x7993,
  [0x0ff0] = 0x7997, [0x0ff1] = 0x7988, [0x0ff2] = 0x7992, [0x0ff3] = 0x7990,
  [0x0ff4] = 0x7a2b, [0x0ff5] = 0x7a4a, [0x0ff6] = 0x7a30, [0x0ff7] = 0x7a2f,
  [0x0ff8] = 0x7a28, [0x0ff9] = 0x7a26, [0x0ffa] = 0x7aa8, [0x0ffb] = 0x7aab,
  [0x0ffc] = 0x7aac, [0x0ffd] = 0x7aee, [0x0ffe] = 0x7b88, [0x0fff] = 0x7b9c,
  [0x1000] = 0x7b8a, [0x1001] = 0x7b91, [0x1002] = 0x7b90, [0x1003] = 0x7b96,
  [0x1004] = 0x7b8d, [0x1005] = 0x7b8c, [0x1006] = 0x7b9b, [0x1007] = 0x7b8e,
  [0x1008] = 0x7b85, [0x1009] = 0x7b98, [0x100a] = 0x5284, [0x100b] = 0x7b99,
  [0x100c] = 0x7ba4, [0x100d] = 0x7b82, [0x100e] = 0x7cbb, [0x100f] = 0x7cbf,
  [0x1010] = 0x7cbc, [0x1011] = 0x7cba, [0x1012] = 0x7da7, [0x1013] = 0x7db7,
  [0x1014] = 0x7dc2, [0x1015] = 0x7da3, [0x1016] = 0x7daa, [0x1017] = 0x7dc1,
  [0x1018] = 0x7dc0, [0x1019] = 0x7dc5, [0x101a] = 0x7d9d, [0x101b] = 0x7dce,
  [0x101c] = 0x7dc4, [0x101d] = 0x7dc6, [0x101e] = 0x7dcb, [0x101f] = 0x7dcc,
  [0x1020] = 0x7daf, [0x1021] = 0x7db9, [0x1022] = 0x7d96, [0x1023] = 0x7dbc,
  [0x1024] = 0x7d9f, [0x1025] = 0x7da6, [0x1026] = 0x7dae, [0x1027] = 0x7da9,
  [0x1028] = 0x7da1, [0x1029] = 0x7dc9, [0x102a] = 0x7f73, [0x102b] = 0x7fe2,
  [0x102c] = 0x7fe3, [0x102d] = 0x7fe5, [0x102e] = 0x7fde, [0x102f] = 0x8024,
  [0x1030] = 0x805d, [0x1031] = 0x805c, [0x1032] = 0x8189, [0x1033] = 0x8186,
  [0x1034] = 0x8183, [0x1035] = 0x8187, [0x1036] = 0x818d, [0x1037] = 0x818c,
  [0x1038] = 0x818b, [0x1039] = 0x8215, [0x103a] = 0x8497, [0x103b] = 0x84a4,
  [0x103c] = 0x84a1, [0x103d] = 0x849f, [0x103e] = 0x84ba, [0x103f] = 0x84ce,
  [0x1040] = 0x84c2, [0x1041] = 0x84ac, [0x1042] = 0x84ae, [0x1043] = 0x84ab,
  [0x1044] = 0x84b9, [0x1045] = 0x84b4, [0x1046] = 0x84c1, [0x1047] = 0x84cd,
  [0x1048] = 0x84aa, [0x1049] = 0x849a, [0x104a] = 0x84b1, [0x104b] = 0x84d0,
  [0x104c] = 0x849d, [0x104d] = 0x84a7, [0x104e] = 0x84bb, [0x104f] = 0x84a2,
  [0x1050] = 0x8494, [0x1051] = 0x84c7, [0x1052] = 0x84cc, [0x1053] = 0x849b,
  [0x1054] = 0x84a9, [0x1055] = 0x84af, [0x1056] = 0x84a8, [0x1057] = 0x84d6,
  [0x1058] = 0x8498, [0x1059] = 0x84b6, [0x105a] = 0x84cf, [0x105b] = 0x84a0,
  [0x105c] = 0x84d7, [0x105d] = 0x84d4, [0x105e] = 0x84d2, [0x105f] = 0x84db,
  [0x1060] = 0x84b0, [0x1061] = 0x8491, [0x1062] = 0x8661, [0x1063] = 0x8733,
  [0x1064] = 0x8723, [0x1065] = 0x8728, [0x1066] = 0x876b, [0x1067] = 0x8740,
  [0x1068] = 0x872e, [0x1069] = 0x871e, [0x106a] = 0x8721, [0x106b] = 0x8719,
  [0x106c] = 0x871b, [0x106d] = 0x8743, [0x106e] = 0x872c, [0x106f] = 0x8741,
  [0x1070] = 0x873e, [0x1071] = 0x8746, [0x1072] = 0x8720, [0x1073] = 0x8732,
  [0x1074] = 0x872a, [0x1075] = 0x872d, [0x1076] = 0x873c, [0x1077] = 0x8712,
  [0x1078] = 0x873a, [0x1079] = 0x8731, [0x107a] = 0x8735, [0x107b] = 0x8742,
  [0x107c] = 0x8726, [0x107d] = 0x8727, [0x107e] = 0x8738, [0x107f] = 0x8724,
  [0x1080] = 0x871a, [0x1081] = 0x8730, [0x1082] = 0x8711, [0x1083] = 0x88f7,
  [0x1084] = 0x88e7, [0x1085] = 0x88f1, [0x1086] = 0x88f2, [0x1087] = 0x88fa,
  [0x1088] = 0x88fe, [0x1089] = 0x88ee, [0x108a] = 0x88fc, [0x108b] = 0x88f6,
  [0x108c] = 0x88fb, [0x108d] = 0x88f0, [0x108e] = 0x88ec, [0x108f] = 0x88eb,
  [0x1090] = 0x899d, [0x1091] = 0x89a1, [0x1092] = 0x899f, [0x1093] = 0x899e,
  [0x1094] = 0x89e9, [0x1095] = 0x89eb, [0x1096] = 0x89e8, [0x1097] = 0x8aab,
  [0x1098] = 0x8a99, [0x1099] = 0x8a8b, [0x109a] = 0x8a92, [0x109b] = 0x8a8f,
  [0x109c] = 0x8a96, [0x109d] = 0x8c3d, [0x109e] = 0x8c68, [0x109f] = 0x8c69,
  [0x10a0] = 0x8cd5, [0x10a1] = 0x8ccf, [0x10a2] = 0x8cd7, [0x10a3] = 0x8d96,
  [0x10a4] = 0x8e09, [0x10a5] = 0x8e02, [0x10a6] = 0x8dff, [0x10a7] = 0x8e0d,
  [0x10a8] = 0x8dfd, [0x10a9] = 0x8e0a, [0x10aa] = 0x8e03, [0x10ab] = 0x8e07,
  [0x10ac] = 0x8e06, [0x10ad] = 0x8e05, [0x10ae] = 0x8dfe, [0x10af] = 0x8e00,
  [0x10b0] = 0x8e04, [0x10b1] = 0x8f10, [0x10b2] = 0x8f11, [0x10b3] = 0x8f0e,
  [0x10b4] = 0x8f0d, [0x10b5] = 0x9123, [0x10b6] = 0x911c, [0x10b7] = 0x9120,
  [0x10b8] = 0x9122, [0x10b9] = 0x911f, [0x10ba] = 0x911d, [0x10bb] = 0x911a,
  [0x10bc] = 0x9124, [0x10bd] = 0x9121, [0x10be] = 0x911b, [0x10bf] = 0x917a,
  [0x10c0] = 0x9172, [0x10c1] = 0x9179, [0x10c2] = 0x9173, [0x10c3] = 0x92a5,
  [0x10c4] = 0x92a4, [0x10c5] = 0x9276, [0x10c6] = 0x929b, [0x10c7] = 0x927a,
  [0x10c8] = 0x92a0, [0x10c9] = 0x9294, [0x10ca] = 0x92aa, [0x10cb] = 0x928d,
  [0x10cc] = 0x92a6, [0x10cd] = 0x929a, [0x10ce] = 0x92ab, [0x10cf] = 0x9279,
  [0x10d0] = 0x9297, [0x10d1] = 0x927f, [0x10d2] = 0x92a3, [0x10d3] = 0x92ee,
  [0x10d4] = 0x928e, [0x10d5] = 0x9282, [0x10d6] = 0x9295, [0x10d7] = 0x92a2,
  [0x10d8] = 0x927d, [0x10d9] = 0x9288, [0x10da] = 0x92a1, [0x10db] = 0x928a,
  [0x10dc] = 0x9286, [0x10dd] = 0x928c, [0x10de] = 0x9299, [0x10df] = 0x92a7,
  [0x10e0] = 0x927e, [0x10e1] = 0x9287, [0x10e2] = 0x92a9, [0x10e3] = 0x929d,
  [0x10e4] = 0x928b, [0x10e5] = 0x922d, [0x10e6] = 0x969e, [0x10e7] = 0x96a1,
  [0x10e8] = 0x96ff, [0x10e9] = 0x9758, [0x10ea] = 0x977d, [0x10eb] = 0x977a,
  [0x10ec] = 0x977e, [0x10ed] = 0x9783, [0x10ee] = 0x9780, [0x10ef] = 0x9782,
  [0x10f0] = 0x977b, [0x10f1] = 0x9784, [0x10f2] = 0x9781, [0x10f3] = 0x977f,
  [0x10f4] = 0x97ce, [0x10f5] = 0x97cd, [0x10f6] = 0x9816, [0x10f7] = 0x98ad,
  [0x10f8] = 0x98ae, [0x10f9] = 0x9902, [0x10fa] = 0x9900, [0x10fb] = 0x9907,
  [0x10fc] = 0x999d, [0x10fd] = 0x999c, [0x10fe] = 0x99c3, [0x10ff] = 0x99b9,
  [0x1100] = 0x99bb, [0x1101] = 0x99ba, [0x1102] = 0x99c2, [0x1103] = 0x99bd,
  [0x1104] = 0x99c7, [0x1105] = 0x9ab1, [0x1106] = 0x9ae3, [0x1107] = 0x9ae7,
  [0x1108] = 0x9b3e, [0x1109] = 0x9b3f, [0x110a] = 0x9b60, [0x110b] = 0x9b61,
  [0x110c] = 0x9b5f, [0x110d] = 0x9cf1, [0x110e] = 0x9cf2, [0x110f] = 0x9cf5,
  [0x1110] = 0x9ea7, [0x1111] = 0x50ff, [0x1112] = 0x5103, [0x1113] = 0x5130,
  [0x1114] = 0x50f8, [0x1115] = 0x5106, [0x1116] = 0x5107, [0x1117] = 0x50f6,
  [0x1118] = 0x50fe, [0x1119] = 0x510b, [0x111a] = 0x510c, [0x111b] = 0x50fd,
  [0x111c] = 0x510a, [0x111d] = 0x528b, [0x111e] = 0x528c, [0x111f] = 0x52f1,
  [0x1120] = 0x52ef, [0x1121] = 0x5648, [0x1122] = 0x5642, [0x1123] = 0x564c,
  [0x1124] = 0x5635, [0x1125] = 0x5641, [0x1126] = 0x564a, [0x1127] = 0x5649,
  [0x1128] = 0x5646, [0x1129] = 0x5658, [0x112a] = 0x565a, [0x112b] = 0x5640,
  [0x112c] = 0x5633, [0x112d] = 0x563d, [0x112e] = 0x562c, [0x112f] = 0x563e,
  [0x1130] = 0x5638, [0x1131] = 0x562a, [0x1132] = 0x563a, [0x1133] = 0x571a,
  [0x1134] = 0x58ab, [0x1135] = 0x589d, [0x1136] = 0x58b1, [0x1137] = 0x58a0,
  [0x1138] = 0x58a3, [0x1139] = 0x58af, [0x113a] = 0x58ac, [0x113b] = 0x58a5,
  [0x113c] = 0x58a1, [0x113d] = 0x58ff, [0x113e] = 0x5aff, [0x113f] = 0x5af4,
  [0x1140] = 0x5afd, [0x1141] = 0x5af7, [0x1142] = 0x5af6, [0x1143] = 0x5b03,
  [0x1144] = 0x5af8, [0x1145] = 0x5b02, [0x1146] = 0x5af9, [0x1147] = 0x5b01,
  [0x1148] = 0x5b07, [0x1149] = 0x5b05, [0x114a] = 0x5b0f, [0x114b] = 0x5c67,
  [0x114c] = 0x5d99, [0x114d] = 0x5d97, [0x114e] = 0x5d9f, [0x114f] = 0x5d92,
  [0x1150] = 0x5da2, [0x1151] = 0x5d93, [0x1152] = 0x5d95, [0x1153] = 0x5da0,
  [0x1154] = 0x5d9c, [0x1155] = 0x5da1, [0x1156] = 0x5d9a, [0x1157] = 0x5d9e,
  [0x1158] = 0x5e69, [0x1159] = 0x5e5d, [0x115a] = 0x5e60, [0x115b] = 0x5e5c,
  [0x115c] = 0x7df3, [0x115d] = 0x5edb, [0x115e] = 0x5ede, [0x115f] = 0x5ee1,
  [0x1160] = 0x5f49, [0x1161] = 0x5fb2, [0x1162] = 0x618b, [0x1163] = 0x6183,
  [0x1164] = 0x6179, [0x1165] = 0x61b1, [0x1166] = 0x61b0, [0x1167] = 0x61a2,
  [0x1168] = 0x6189, [0x1169] = 0x619b, [0x116a] = 0x6193, [0x116b] = 0x61af,
  [0x116c] = 0x61ad, [0x116d] = 0x619f, [0x116e] = 0x6192, [0x116f] = 0x61aa,
  [0x1170] = 0x61a1, [0x1171] = 0x618d, [0x1172] = 0x6166, [0x1173] = 0x61b3,
  [0x1174] = 0x622d, [0x1175] = 0x646e, [0x1176] = 0x6470, [0x1177] = 0x6496,
  [0x1178] = 0x64a0, [0x1179] = 0x6485, [0x117a] = 0x6497, [0x117b] = 0x649c,
  [0x117c] = 0x648f, [0x117d] = 0x648b, [0x117e] = 0x648a, [0x117f] = 0x648c,
  [0x1180] = 0x64a3, [0x1181] = 0x649f, [0x1182] = 0x6468, [0x1183] = 0x64b1,
  [0x1184] = 0x6498, [0x1185] = 0x6576, [0x1186] = 0x657a, [0x1187] = 0x6579,
  [0x1188] = 0x657b, [0x1189] = 0x65b2, [0x118a] = 0x65b3, [0x118b] = 0x66b5,
  [0x118c] = 0x66b0, [0x118d] = 0x66a9, [0x118e] = 0x66b2, [0x118f] = 0x66b7,
  [0x1190] = 0x66aa, [0x1191] = 0x66af, [0x1192] = 0x6a00, [0x1193] = 0x6a06,
  [0x1194] = 0x6a17, [0x1195] = 0x69e5, [0x1196] = 0x69f8, [0x1197] = 0x6a15,
  [0x1198] = 0x69f1, [0x1199] = 0x69e4, [0x119a] = 0x6a20, [0x119b] = 0x69ff,
  [0x119c] = 0x69ec, [0x119d] = 0x69e2, [0x119e] = 0x6a1b, [0x119f] = 0x6a1d,
  [0x11a0] = 0x69fe, [0x11a1] = 0x6a27, [0x11a2] = 0x69f2, [0x11a3] = 0x69ee,
  [0x11a4] = 0x6a14, [0x11a5] = 0x69f7, [0x11a6] = 0x69e7, [0x11a7] = 0x6a40,
  [0x11a8] = 0x6a08, [0x11a9] = 0x69e6, [0x11aa] = 0x69fb, [0x11ab] = 0x6a0d,
  [0x11ac] = 0x69fc, [0x11ad] = 0x69eb, [0x11ae] = 0x6a09, [0x11af] = 0x6a04,
  [0x11b0] = 0x6a18, [0x11b1] = 0x6a25, [0x11b2] = 0x6a0f, [0x11b3] = 0x69f6,
  [0x11b4] = 0x6a26, [0x11b5] = 0x6a07, [0x11b6] = 0x69f4, [0x11b7] = 0x6a16,
  [0x11b8] = 0x6b51, [0x11b9] = 0x6ba5, [0x11ba] = 0x6ba3, [0x11bb] = 0x6ba2,
  [0x11bc] = 0x6ba6, [0x11bd] = 0x6c01, [0x11be] = 0x6c00, [0x11bf] = 0x6bff,
  [0x11c0] = 0x6c02, [0x11c1] = 0x6f41, [0x11c2] = 0x6f26, [0x11c3] = 0x6f7e,
  [0x11c4] = 0x6f87, [0x11c5] = 0x6fc6, [0x11c6] = 0x6f92, [0x11c7] = 0x6f8d,
  [0x11c8] = 0x6f89, [0x11c9] = 0x6f8c, [0x11ca] = 0x6f62, [0x11cb] = 0x6f4f,
  [0x11cc] = 0x6f85, [0x11cd] = 0x6f5a, [0x11ce] = 0x6f96, [0x11cf] = 0x6f76,
  [0x11d0] = 0x6f6c, [0x11d1] = 0x6f82, [0x11d2] = 0x6f55, [0x11d3] = 0x6f72,
  [0x11d4] = 0x6f52, [0x11d5] = 0x6f50, [0x11d6] = 0x6f57, [0x11d7] = 0x6f94,
  [0x11d8] = 0x6f93, [0x11d9] = 0x6f5d, [0x11da] = 0x6f00, [0x11db] = 0x6f61,
  [0x11dc] = 0x6f6b, [0x11dd] = 0x6f7d, [0x11de] = 0x6f67, [0x11df] = 0x6f90,
  [0x11e0] = 0x6f53, [0x11e1] = 0x6f8b, [0x11e2] = 0x6f69, [0x11e3] = 0x6f7f,
  [0x11e4] = 0x6f95, [0x11e5] = 0x6f63, [0x11e6] = 0x6f77, [0x11e7] = 0x6f6a,
  [0x11e8] = 0x6f7b, [0x11e9] = 0x71b2, [0x11ea] = 0x71af, [0x11eb] = 0x719b,
  [0x11ec] = 0x71b0, [0x11ed] = 0x71a0, [0x11ee] = 0x719a, [0x11ef] = 0x71a9,
  [0x11f0] = 0x71b5, [0x11f1] = 0x719d, [0x11f2] = 0x71a5, [0x11f3] = 0x719e,
  [0x11f4] = 0x71a4, [0x11f5] = 0x71a1, [0x11f6] = 0x71aa, [0x11f7] = 0x719c,
  [0x11f8] = 0x71a7, [0x11f9] = 0x71b3, [0x11fa] = 0x7298, [0x11fb] = 0x729a,
  [0x11fc] = 0x7358, [0x11fd] = 0x7352, [0x11fe] = 0x735e, [0x11ff] = 0x735f,
  [0x1200] = 0x7360, [0x1201] = 0x735d, [0x1202] = 0x735b, [0x1203] = 0x7361,
  [0x1204] = 0x735a, [0x1205] = 0x7359, [0x1206] = 0x7362, [0x1207] = 0x7487,
  [0x1208] = 0x7489, [0x1209] = 0x748a, [0x120a] = 0x7486, [0x120b] = 0x7481,
  [0x120c] = 0x747d, [0x120d] = 0x7485, [0x120e] = 0x7488, [0x120f] = 0x747c,
  [0x1210] = 0x7479, [0x1211] = 0x7508, [0x1212] = 0x7507, [0x1213] = 0x757e,
  [0x1214] = 0x7625, [0x1215] = 0x761e, [0x1216] = 0x7619, [0x1217] = 0x761d,
  [0x1218] = 0x761c, [0x1219] = 0x7623, [0x121a] = 0x761a, [0x121b] = 0x7628,
  [0x121c] = 0x761b, [0x121d] = 0x769c, [0x121e] = 0x769d, [0x121f] = 0x769e,
  [0x1220] = 0x769b, [0x1221] = 0x778d, [0x1222] = 0x778f, [0x1223] = 0x7789,
  [0x1224] = 0x7788, [0x1225] = 0x78cd, [0x1226] = 0x78bb, [0x1227] = 0x78cf,
  [0x1228] = 0x78cc, [0x1229] = 0x78d1, [0x122a] = 0x78ce, [0x122b] = 0x78d4,
  [0x122c] = 0x78c8, [0x122d] = 0x78c3, [0x122e] = 0x78c4, [0x122f] = 0x78c9,
  [0x1230] = 0x799a, [0x1231] = 0x79a1, [0x1232] = 0x79a0, [0x1233] = 0x799c,
  [0x1234] = 0x79a2, [0x1235] = 0x799b, [0x1236] = 0x6b76, [0x1237] = 0x7a39,
  [0x1238] = 0x7ab2, [0x1239] = 0x7ab4, [0x123a] = 0x7ab3, [0x123b] = 0x7bb7,
  [0x123c] = 0x7bcb, [0x123d] = 0x7bbe, [0x123e] = 0x7bac, [0x123f] = 0x7bce,
  [0x1240] = 0x7baf, [0x1241] = 0x7bb9, [0x1242] = 0x7bca, [0x1243] = 0x7bb5,
  [0x1244] = 0x7cc5, [0x1245] = 0x7cc8, [0x1246] = 0x7ccc, [0x1247] = 0x7ccb,
  [0x1248] = 0x7df7, [0x1249] = 0x7ddb, [0x124a] = 0x7dea, [0x124b] = 0x7de7,
  [0x124c] = 0x7dd7, [0x124d] = 0x7de1, [0x124e] = 0x7e03, [0x124f] = 0x7dfa,
  [0x1250] = 0x7de6, [0x1251] = 0x7df6, [0x1252] = 0x7df1, [0x1253] = 0x7df0,
  [0x1254] = 0x7dee, [0x1255] = 0x7ddf, [0x1256] = 0x7f76, [0x1257] = 0x7fac,
  [0x1258] = 0x7fb0, [0x1259] = 0x7fad, [0x125a] = 0x7fed, [0x125b] = 0x7feb,
  [0x125c] = 0x7fea, [0x125d] = 0x7fec, [0x125e] = 0x7fe6, [0x125f] = 0x7fe8,
  [0x1260] = 0x8064, [0x1261] = 0x8067, [0x1262] = 0x81a3, [0x1263] = 0x819f,
  [0x1264] = 0x819e, [0x1265] = 0x8195, [0x1266] = 0x81a2, [0x1267] = 0x8199,
  [0x1268] = 0x8197, [0x1269] = 0x8216, [0x126a] = 0x824f, [0x126b] = 0x8253,
  [0x126c] = 0x8252, [0x126d] = 0x8250, [0x126e] = 0x824e, [0x126f] = 0x8251,
  [0x1270] = 0x8524, [0x1271] = 0x853b, [0x1272] = 0x850f, [0x1273] = 0x8500,
  [0x1274] = 0x8529, [0x1275] = 0x850e, [0x1276] = 0x8509, [0x1277] = 0x850d,
  [0x1278] = 0x851f, [0x1279] = 0x850a, [0x127a] = 0x8527, [0x127b] = 0x851c,
  [0x127c] = 0x84fb, [0x127d] = 0x852b, [0x127e] = 0x84fa, [0x127f] = 0x8508,
  [0x1280] = 0x850c, [0x1281] = 0x84f4, [0x1282] = 0x852a, [0x1283] = 0x84f2,
  [0x1284] = 0x8515, [0x1285] = 0x84f7, [0x1286] = 0x84eb, [0x1287] = 0x84f3,
  [0x1288] = 0x84fc, [0x1289] = 0x8512, [0x128a] = 0x84ea, [0x128b] = 0x84e9,
  [0x128c] = 0x8516, [0x128d] = 0x84fe, [0x128e] = 0x8528, [0x128f] = 0x851d,
  [0x1290] = 0x852e, [0x1291] = 0x8502, [0x1292] = 0x84fd, [0x1293] = 0x851e,
  [0x1294] = 0x84f6, [0x1295] = 0x8531, [0x1296] = 0x8526, [0x1297] = 0x84e7,
  [0x1298] = 0x84e8, [0x1299] = 0x84f0, [0x129a] = 0x84ef, [0x129b] = 0x84f9,
  [0x129c] = 0x8518, [0x129d] = 0x8520, [0x129e] = 0x8530, [0x129f] = 0x850b,
  [0x12a0] = 0x8519, [0x12a1] = 0x852f, [0x12a2] = 0x8662, [0x12a3] = 0x8756,
  [0x12a4] = 0x8763, [0x12a5] = 0x8764, [0x12a6] = 0x8777, [0x12a7] = 0x87e1,
  [0x12a8] = 0x8773, [0x12a9] = 0x8758, [0x12aa] = 0x8754, [0x12ab] = 0x875b,
  [0x12ac] = 0x8752, [0x12ad] = 0x8761, [0x12ae] = 0x875a, [0x12af] = 0x8751,
  [0x12b0] = 0x875e, [0x12b1] = 0x876d, [0x12b2] = 0x876a, [0x12b3] = 0x8750,
  [0x12b4] = 0x874e, [0x12b5] = 0x875f, [0x12b6] = 0x875d, [0x12b7] = 0x876f,
  [0x12b8] = 0x876c, [0x12b9] = 0x877a, [0x12ba] = 0x876e, [0x12bb] = 0x875c,
  [0x12bc] = 0x8765, [0x12bd] = 0x874f, [0x12be] = 0x877b, [0x12bf] = 0x8775,
  [0x12c0] = 0x8762, [0x12c1] = 0x8767, [0x12c2] = 0x8769, [0x12c3] = 0x885a,
  [0x12c4] = 0x8905, [0x12c5] = 0x890c, [0x12c6] = 0x8914, [0x12c7] = 0x890b,
  [0x12c8] = 0x8917, [0x12c9] = 0x8918, [0x12ca] = 0x8919, [0x12cb] = 0x8906,
  [0x12cc] = 0x8916, [0x12cd] = 0x8911, [0x12ce] = 0x890e, [0x12cf] = 0x8909,
  [0x12d0] = 0x89a2, [0x12d1] = 0x89a4, [0x12d2] = 0x89a3, [0x12d3] = 0x89ed,
  [0x12d4] = 0x89f0, [0x12d5] = 0x89ec, [0x12d6] = 0x8acf, [0x12d7] = 0x8ac6,
  [0x12d8] = 0x8ab8, [0x12d9] = 0x8ad3, [0x12da] = 0x8ad1, [0x12db] = 0x8ad4,
  [0x12dc] = 0x8ad5, [0x12dd] = 0x8abb, [0x12de] = 0x8ad7, [0x12df] = 0x8abe,
  [0x12e0] = 0x8ac0, [0x12e1] = 0x8ac5, [0x12e2] = 0x8ad8, [0x12e3] = 0x8ac3,
  [0x12e4] = 0x8aba, [0x12e5] = 0x8abd, [0x12e6] = 0x8ad9, [0x12e7] = 0x8c3e,
  [0x12e8] = 0x8c4d, [0x12e9] = 0x8c8f, [0x12ea] = 0x8ce5, [0x12eb] = 0x8cdf,
  [0x12ec] = 0x8cd9, [0x12ed] = 0x8ce8, [0x12ee] = 0x8cda, [0x12ef] = 0x8cdd,
  [0x12f0] = 0x8ce7, [0x12f1] = 0x8da0, [0x12f2] = 0x8d9c, [0x12f3] = 0x8da1,
  [0x12f4] = 0x8d9b, [0x12f5] = 0x8e20, [0x12f6] = 0x8e23, [0x12f7] = 0x8e25,
  [0x12f8] = 0x8e24, [0x12f9] = 0x8e2e, [0x12fa] = 0x8e15, [0x12fb] = 0x8e1b,
  [0x12fc] = 0x8e16, [0x12fd] = 0x8e11, [0x12fe] = 0x8e19, [0x12ff] = 0x8e26,
  [0x1300] = 0x8e27, [0x1301] = 0x8e14, [0x1302] = 0x8e12, [0x1303] = 0x8e18,
  [0x1304] = 0x8e13, [0x1305] = 0x8e1c, [0x1306] = 0x8e17, [0x1307] = 0x8e1a,
  [0x1308] = 0x8f2c, [0x1309] = 0x8f24, [0x130a] = 0x8f18, [0x130b] = 0x8f1a,
  [0x130c] = 0x8f20, [0x130d] = 0x8f23, [0x130e] = 0x8f16, [0x130f] = 0x8f17,
  [0x1310] = 0x9073, [0x1311] = 0x9070, [0x1312] = 0x906f, [0x1313] = 0x9067,
  [0x1314] = 0x906b, [0x1315] = 0x912f, [0x1316] = 0x912b, [0x1317] = 0x9129,
  [0x1318] = 0x912a, [0x1319] = 0x9132, [0x131a] = 0x9126, [0x131b] = 0x912e,
  [0x131c] = 0x9185, [0x131d] = 0x9186, [0x131e] = 0x918a, [0x131f] = 0x9181,
  [0x1320] = 0x9182, [0x1321] = 0x9184, [0x1322] = 0x9180, [0x1323] = 0x92d0,
  [0x1324] = 0x92c3, [0x1325] = 0x92c4, [0x1326] = 0x92c0, [0x1327] = 0x92d9,
  [0x1328] = 0x92b6, [0x1329] = 0x92cf, [0x132a] = 0x92f1, [0x132b] = 0x92df,
  [0x132c] = 0x92d8, [0x132d] = 0x92e9, [0x132e] = 0x92d7, [0x132f] = 0x92dd,
  [0x1330] = 0x92cc, [0x1331] = 0x92ef, [0x1332] = 0x92c2, [0x1333] = 0x92e8,
  [0x1334] = 0x92ca, [0x1335] = 0x92c8, [0x1336] = 0x92ce, [0x1337] = 0x92e6,
  [0x1338] = 0x92cd, [0x1339] = 0x92d5, [0x133a] = 0x92c9, [0x133b] = 0x92e0,
  [0x133c] = 0x92de, [0x133d] = 0x92e7, [0x133e] = 0x92d1, [0x133f] = 0x92d3,
  [0x1340] = 0x92b5, [0x1341] = 0x92e1, [0x1342] = 0x9325, [0x1343] = 0x92c6,
  [0x1344] = 0x92b4, [0x1345] = 0x957c, [0x1346] = 0x95ac, [0x1347] = 0x95ab,
  [0x1348] = 0x95ae, [0x1349] = 0x95b0, [0x134a] = 0x96a4, [0x134b] = 0x96a2,
  [0x134c] = 0x96d3, [0x134d] = 0x9705, [0x134e] = 0x9708, [0x134f] = 0x9702,
  [0x1350] = 0x975a, [0x1351] = 0x978a, [0x1352] = 0x978e, [0x1353] = 0x9788,
  [0x1354] = 0x97d0, [0x1355] = 0x97cf, [0x1356] = 0x981e, [0x1357] = 0x981d,
  [0x1358] = 0x9826, [0x1359] = 0x9829, [0x135a] = 0x9828, [0x135b] = 0x9820,
  [0x135c] = 0x981b, [0x135d] = 0x9827, [0x135e] = 0x98b2, [0x135f] = 0x9908,
  [0x1360] = 0x98fa, [0x1361] = 0x9911, [0x1362] = 0x9914, [0x1363] = 0x9916,
  [0x1364] = 0x9917, [0x1365] = 0x9915, [0x1366] = 0x99dc, [0x1367] = 0x99cd,
  [0x1368] = 0x99cf, [0x1369] = 0x99d3, [0x136a] = 0x99d4, [0x136b] = 0x99ce,
  [0x136c] = 0x99c9, [0x136d] = 0x99d6, [0x136e] = 0x99d8, [0x136f] = 0x99cb,
  [0x1370] = 0x99d7, [0x1371] = 0x99cc, [0x1372] = 0x9ab3, [0x1373] = 0x9aec,
  [0x1374] = 0x9aeb, [0x1375] = 0x9af3, [0x1376] = 0x9af2, [0x1377] = 0x9af1,
  [0x1378] = 0x9b46, [0x1379] = 0x9b43, [0x137a] = 0x9b67, [0x137b] = 0x9b74,
  [0x137c] = 0x9b71, [0x137d] = 0x9b66, [0x137e] = 0x9b76, [0x137f] = 0x9b75,
  [0x1380] = 0x9b70, [0x1381] = 0x9b68, [0x1382] = 0x9b64, [0x1383] = 0x9b6c,
  [0x1384] = 0x9cfc, [0x1385] = 0x9cfa, [0x1386] = 0x9cfd, [0x1387] = 0x9cff,
  [0x1388] = 0x9cf7, [0x1389] = 0x9d07, [0x138a] = 0x9d00, [0x138b] = 0x9cf9,
  [0x138c] = 0x9cfb, [0x138d] = 0x9d08, [0x138e] = 0x9d05, [0x138f] = 0x9d04,
  [0x1390] = 0x9e83, [0x1391] = 0x9ed3, [0x1392] = 0x9f0f, [0x1393] = 0x9f10,
  [0x1394] = 0x511c, [0x1395] = 0x5113, [0x1396] = 0x5117, [0x1397] = 0x511a,
  [0x1398] = 0x5111, [0x1399] = 0x51de, [0x139a] = 0x5334, [0x139b] = 0x53e1,
  [0x139c] = 0x5670, [0x139d] = 0x5660, [0x139e] = 0x566e, [0x139f] = 0x5673,
  [0x13a0] = 0x5666, [0x13a1] = 0x5663, [0x13a2] = 0x566d, [0x13a3] = 0x5672,
  [0x13a4] = 0x565e, [0x13a5] = 0x5677, [0x13a6] = 0x571c, [0x13a7] = 0x571b,
  [0x13a8] = 0x58c8, [0x13a9] = 0x58bd, [0x13aa] = 0x58c9, [0x13ab] = 0x58bf,
  [0x13ac] = 0x58ba, [0x13ad] = 0x58c2, [0x13ae] = 0x58bc, [0x13af] = 0x58c6,
  [0x13b0] = 0x5b17, [0x13b1] = 0x5b19, [0x13b2] = 0x5b1b, [0x13b3] = 0x5b21,
  [0x13b4] = 0x5b14, [0x13b5] = 0x5b13, [0x13b6] = 0x5b10, [0x13b7] = 0x5b16,
  [0x13b8] = 0x5b28, [0x13b9] = 0x5b1a, [0x13ba] = 0x5b20, [0x13bb] = 0x5b1e,
  [0x13bc] = 0x5bef, [0x13bd] = 0x5dac, [0x13be] = 0x5db1, [0x13bf] = 0x5da9,
  [0x13c0] = 0x5da7, [0x13c1] = 0x5db5, [0x13c2] = 0x5db0, [0x13c3] = 0x5dae,
  [0x13c4] = 0x5daa, [0x13c5] = 0x5da8, [0x13c6] = 0x5db2, [0x13c7] = 0x5dad,
  [0x13c8] = 0x5daf, [0x13c9] = 0x5db4, [0x13ca] = 0x5e67, [0x13cb] = 0x5e68,
  [0x13cc] = 0x5e66, [0x13cd] = 0x5e6f, [0x13ce] = 0x5ee9, [0x13cf] = 0x5ee7,
  [0x13d0] = 0x5ee6, [0x13d1] = 0x5ee8, [0x13d2] = 0x5ee5, [0x13d3] = 0x5f4b,
  [0x13d4] = 0x5fbc, [0x13d5] = 0x5fbb, [0x13d6] = 0x619d, [0x13d7] = 0x61a8,
  [0x13d8] = 0x6196, [0x13d9] = 0x61c5, [0x13da] = 0x61b4, [0x13db] = 0x61c6,
  [0x13dc] = 0x61c1, [0x13dd] = 0x61cc, [0x13de] = 0x61ba, [0x13df] = 0x61bf,
  [0x13e0] = 0x61b8, [0x13e1] = 0x618c, [0x13e2] = 0x64d7, [0x13e3] = 0x64d6,
  [0x13e4] = 0x64d0, [0x13e5] = 0x64cf, [0x13e6] = 0x64c9, [0x13e7] = 0x64bd,
  [0x13e8] = 0x6489, [0x13e9] = 0x64c3, [0x13ea] = 0x64db, [0x13eb] = 0x64f3,
  [0x13ec] = 0x64d9, [0x13ed] = 0x6533, [0x13ee] = 0x657f, [0x13ef] = 0x657c,
  [0x13f0] = 0x65a2, [0x13f1] = 0x66c8, [0x13f2] = 0x66be, [0x13f3] = 0x66c0,
  [0x13f4] = 0x66ca, [0x13f5] = 0x66cb, [0x13f6] = 0x66cf, [0x13f7] = 0x66bd,
  [0x13f8] = 0x66bb, [0x13f9] = 0x66ba, [0x13fa] = 0x66cc, [0x13fb] = 0x6723,
  [0x13fc] = 0x6a34, [0x13fd] = 0x6a66, [0x13fe] = 0x6a49, [0x13ff] = 0x6a67,
  [0x1400] = 0x6a32, [0x1401] = 0x6a68, [0x1402] = 0x6a3e, [0x1403] = 0x6a5d,
  [0x1404] = 0x6a6d, [0x1405] = 0x6a76, [0x1406] = 0x6a5b, [0x1407] = 0x6a51,
  [0x1408] = 0x6a28, [0x1409] = 0x6a5a, [0x140a] = 0x6a3b, [0x140b] = 0x6a3f,
  [0x140c] = 0x6a41, [0x140d] = 0x6a6a, [0x140e] = 0x6a64, [0x140f] = 0x6a50,
  [0x1410] = 0x6a4f, [0x1411] = 0x6a54, [0x1412] = 0x6a6f, [0x1413] = 0x6a69,
  [0x1414] = 0x6a60, [0x1415] = 0x6a3c, [0x1416] = 0x6a5e, [0x1417] = 0x6a56,
  [0x1418] = 0x6a55, [0x1419] = 0x6a4d, [0x141a] = 0x6a4e, [0x141b] = 0x6a46,
  [0x141c] = 0x6b55, [0x141d] = 0x6b54, [0x141e] = 0x6b56, [0x141f] = 0x6ba7,
  [0x1420] = 0x6baa, [0x1421] = 0x6bab, [0x1422] = 0x6bc8, [0x1423] = 0x6bc7,
  [0x1424] = 0x6c04, [0x1425] = 0x6c03, [0x1426] = 0x6c06, [0x1427] = 0x6fad,
  [0x1428] = 0x6fcb, [0x1429] = 0x6fa3, [0x142a] = 0x6fc7, [0x142b] = 0x6fbc,
  [0x142c] = 0x6fce, [0x142d] = 0x6fc8, [0x142e] = 0x6f5e, [0x142f] = 0x6fc4,
  [0x1430] = 0x6fbd, [0x1431] = 0x6f9e, [0x1432] = 0x6fca, [0x1433] = 0x6fa8,
  [0x1434] = 0x7004, [0x1435] = 0x6fa5, [0x1436] = 0x6fae, [0x1437] = 0x6fba,
  [0x1438] = 0x6fac, [0x1439] = 0x6faa, [0x143a] = 0x6fcf, [0x143b] = 0x6fbf,
  [0x143c] = 0x6fb8, [0x143d] = 0x6fa2, [0x143e] = 0x6fc9, [0x143f] = 0x6fab,
  [0x1440] = 0x6fcd, [0x1441] = 0x6faf, [0x1442] = 0x6fb2, [0x1443] = 0x6fb0,
  [0x1444] = 0x71c5, [0x1445] = 0x71c2, [0x1446] = 0x71bf, [0x1447] = 0x71b8,
  [0x1448] = 0x71d6, [0x1449] = 0x71c0, [0x144a] = 0x71c1, [0x144b] = 0x71cb,
  [0x144c] = 0x71d4, [0x144d] = 0x71ca, [0x144e] = 0x71c7, [0x144f] = 0x71cf,
  [0x1450] = 0x71bd, [0x1451] = 0x71d8, [0x1452] = 0x71bc, [0x1453] = 0x71c6,
  [0x1454] = 0x71da, [0x1455] = 0x71db, [0x1456] = 0x729d, [0x1457] = 0x729e,
  [0x1458] = 0x7369, [0x1459] = 0x7366, [0x145a] = 0x7367, [0x145b] = 0x736c,
  [0x145c] = 0x7365, [0x145d] = 0x736b, [0x145e] = 0x736a, [0x145f] = 0x747f,
  [0x1460] = 0x749a, [0x1461] = 0x74a0, [0x1462] = 0x7494, [0x1463] = 0x7492,
  [0x1464] = 0x7495, [0x1465] = 0x74a1, [0x1466] = 0x750b, [0x1467] = 0x7580,
  [0x1468] = 0x762f, [0x1469] = 0x762d, [0x146a] = 0x7631, [0x146b] = 0x763d,
  [0x146c] = 0x7633, [0x146d] = 0x763c, [0x146e] = 0x7635, [0x146f] = 0x7632,
  [0x1470] = 0x7630, [0x1471] = 0x76bb, [0x1472] = 0x76e6, [0x1473] = 0x779a,
  [0x1474] = 0x779d, [0x1475] = 0x77a1, [0x1476] = 0x779c, [0x1477] = 0x779b,
  [0x1478] = 0x77a2, [0x1479] = 0x77a3, [0x147a] = 0x7795, [0x147b] = 0x7799,
  [0x147c] = 0x7797, [0x147d] = 0x78dd, [0x147e] = 0x78e9, [0x147f] = 0x78e5,
  [0x1480] = 0x78ea, [0x1481] = 0x78de, [0x1482] = 0x78e3, [0x1483] = 0x78db,
  [0x1484] = 0x78e1, [0x1485] = 0x78e2, [0x1486] = 0x78ed, [0x1487] = 0x78df,
  [0x1488] = 0x78e0, [0x1489] = 0x79a4, [0x148a] = 0x7a44, [0x148b] = 0x7a48,
  [0x148c] = 0x7a47, [0x148d] = 0x7ab6, [0x148e] = 0x7ab8, [0x148f] = 0x7ab5,
  [0x1490] = 0x7ab1, [0x1491] = 0x7ab7, [0x1492] = 0x7bde, [0x1493] = 0x7be3,
  [0x1494] = 0x7be7, [0x1495] = 0x7bdd, [0x1496] = 0x7bd5, [0x1497] = 0x7be5,
  [0x1498] = 0x7bda, [0x1499] = 0x7be8, [0x149a] = 0x7bf9, [0x149b] = 0x7bd4,
  [0x149c] = 0x7bea, [0x149d] = 0x7be2, [0x149e] = 0x7bdc, [0x149f] = 0x7beb,
  [0x14a0] = 0x7bd8, [0x14a1] = 0x7bdf, [0x14a2] = 0x7cd2, [0x14a3] = 0x7cd4,
  [0x14a4] = 0x7cd7, [0x14a5] = 0x7cd0, [0x14a6] = 0x7cd1, [0x14a7] = 0x7e12,
  [0x14a8] = 0x7e21, [0x14a9] = 0x7e17, [0x14aa] = 0x7e0c, [0x14ab] = 0x7e1f,
  [0x14ac] = 0x7e20, [0x14ad] = 0x7e13, [0x14ae] = 0x7e0e, [0x14af] = 0x7e1c,
  [0x14b0] = 0x7e15, [0x14b1] = 0x7e1a, [0x14b2] = 0x7e22, [0x14b3] = 0x7e0b,
  [0x14b4] = 0x7e0f, [0x14b5] = 0x7e16, [0x14b6] = 0x7e0d, [0x14b7] = 0x7e14,
  [0x14b8] = 0x7e25, [0x14b9] = 0x7e24, [0x14ba] = 0x7f43, [0x14bb] = 0x7f7b,
  [0x14bc] = 0x7f7c, [0x14bd] = 0x7f7a, [0x14be] = 0x7fb1, [0x14bf] = 0x7fef,
  [0x14c0] = 0x802a, [0x14c1] = 0x8029, [0x14c2] = 0x806c, [0x14c3] = 0x81b1,
  [0x14c4] = 0x81a6, [0x14c5] = 0x81ae, [0x14c6] = 0x81b9, [0x14c7] = 0x81b5,
  [0x14c8] = 0x81ab, [0x14c9] = 0x81b0, [0x14ca] = 0x81ac, [0x14cb] = 0x81b4,
  [0x14cc] = 0x81b2, [0x14cd] = 0x81b7, [0x14ce] = 0x81a7, [0x14cf] = 0x81f2,
  [0x14d0] = 0x8255, [0x14d1] = 0x8256, [0x14d2] = 0x8257, [0x14d3] = 0x8556,
  [0x14d4] = 0x8545, [0x14d5] = 0x856b, [0x14d6] = 0x854d, [0x14d7] = 0x8553,
  [0x14d8] = 0x8561, [0x14d9] = 0x8558, [0x14da] = 0x8540, [0x14db] = 0x8546,
  [0x14dc] = 0x8564, [0x14dd] = 0x8541, [0x14de] = 0x8562, [0x14df] = 0x8544,
  [0x14e0] = 0x8551, [0x14e1] = 0x8547, [0x14e2] = 0x8563, [0x14e3] = 0x853e,
  [0x14e4] = 0x855b, [0x14e5] = 0x8571, [0x14e6] = 0x854e, [0x14e7] = 0x856e,
  [0x14e8] = 0x8575, [0x14e9] = 0x8555, [0x14ea] = 0x8567, [0x14eb] = 0x8560,
  [0x14ec] = 0x858c, [0x14ed] = 0x8566, [0x14ee] = 0x855d, [0x14ef] = 0x8554,
  [0x14f0] = 0x8565, [0x14f1] = 0x856c, [0x14f2] = 0x8663, [0x14f3] = 0x8665,
  [0x14f4] = 0x8664, [0x14f5] = 0x87a4, [0x14f6] = 0x879b, [0x14f7] = 0x878f,
  [0x14f8] = 0x8797, [0x14f9] = 0x8793, [0x14fa] = 0x8792, [0x14fb] = 0x8788,
  [0x14fc] = 0x8781, [0x14fd] = 0x8796, [0x14fe] = 0x8798, [0x14ff] = 0x8779,
  [0x1500] = 0x8787, [0x1501] = 0x87a3, [0x1502] = 0x8785, [0x1503] = 0x8790,
  [0x1504] = 0x8791, [0x1505] = 0x879d, [0x1506] = 0x8784, [0x1507] = 0x8794,
  [0x1508] = 0x879c, [0x1509] = 0x879a, [0x150a] = 0x8789, [0x150b] = 0x891e,
  [0x150c] = 0x8926, [0x150d] = 0x8930, [0x150e] = 0x892d, [0x150f] = 0x892e,
  [0x1510] = 0x8927, [0x1511] = 0x8931, [0x1512] = 0x8922, [0x1513] = 0x8929,
  [0x1514] = 0x8923, [0x1515] = 0x892f, [0x1516] = 0x892c, [0x1517] = 0x891f,
  [0x1518] = 0x89f1, [0x1519] = 0x8ae0, [0x151a] = 0x8ae2, [0x151b] = 0x8af2,
  [0x151c] = 0x8af4, [0x151d] = 0x8af5, [0x151e] = 0x8add, [0x151f] = 0x8b14,
  [0x1520] = 0x8ae4, [0x1521] = 0x8adf, [0x1522] = 0x8af0, [0x1523] = 0x8ac8,
  [0x1524] = 0x8ade, [0x1525] = 0x8ae1, [0x1526] = 0x8ae8, [0x1527] = 0x8aff,
  [0x1528] = 0x8aef, [0x1529] = 0x8afb, [0x152a] = 0x8c91, [0x152b] = 0x8c92,
  [0x152c] = 0x8c90, [0x152d] = 0x8cf5, [0x152e] = 0x8cee, [0x152f] = 0x8cf1,
  [0x1530] = 0x8cf0, [0x1531] = 0x8cf3, [0x1532] = 0x8d6c, [0x1533] = 0x8d6e,
  [0x1534] = 0x8da5, [0x1535] = 0x8da7, [0x1536] = 0x8e33, [0x1537] = 0x8e3e,
  [0x1538] = 0x8e38, [0x1539] = 0x8e40, [0x153a] = 0x8e45, [0x153b] = 0x8e36,
  [0x153c] = 0x8e3c, [0x153d] = 0x8e3d, [0x153e] = 0x8e41, [0x153f] = 0x8e30,
  [0x1540] = 0x8e3f, [0x1541] = 0x8ebd, [0x1542] = 0x8f36, [0x1543] = 0x8f2e,
  [0x1544] = 0x8f35, [0x1545] = 0x8f32, [0x1546] = 0x8f39, [0x1547] = 0x8f37,
  [0x1548] = 0x8f34, [0x1549] = 0x9076, [0x154a] = 0x9079, [0x154b] = 0x907b,
  [0x154c] = 0x9086, [0x154d] = 0x90fa, [0x154e] = 0x9133, [0x154f] = 0x9135,
  [0x1550] = 0x9136, [0x1551] = 0x9193, [0x1552] = 0x9190, [0x1553] = 0x9191,
  [0x1554] = 0x918d, [0x1555] = 0x918f, [0x1556] = 0x9327, [0x1557] = 0x931e,
  [0x1558] = 0x9308, [0x1559] = 0x931f, [0x155a] = 0x9306, [0x155b] = 0x930f,
  [0x155c] = 0x937a, [0x155d] = 0x9338, [0x155e] = 0x933c, [0x155f] = 0x931b,
  [0x1560] = 0x9323, [0x1561] = 0x9312, [0x1562] = 0x9301, [0x1563] = 0x9346,
  [0x1564] = 0x932d, [0x1565] = 0x930e, [0x1566] = 0x930d, [0x1567] = 0x92cb,
  [0x1568] = 0x931d, [0x1569] = 0x92fa, [0x156a] = 0x9313, [0x156b] = 0x92f9,
  [0x156c] = 0x92f7, [0x156d] = 0x9334, [0x156e] = 0x9302, [0x156f] = 0x9324,
  [0x1570] = 0x92ff, [0x1571] = 0x9329, [0x1572] = 0x9339, [0x1573] = 0x9335,
  [0x1574] = 0x932a, [0x1575] = 0x9314, [0x1576] = 0x930c, [0x1577] = 0x930b,
  [0x1578] = 0x92fe, [0x1579] = 0x9309, [0x157a] = 0x9300, [0x157b] = 0x92fb,
  [0x157c] = 0x9316, [0x157d] = 0x95bc, [0x157e] = 0x95cd, [0x157f] = 0x95be,
  [0x1580] = 0x95b9, [0x1581] = 0x95ba, [0x1582] = 0x95b6, [0x1583] = 0x95bf,
  [0x1584] = 0x95b5, [0x1585] = 0x95bd, [0x1586] = 0x96a9, [0x1587] = 0x96d4,
  [0x1588] = 0x970b, [0x1589] = 0x9712, [0x158a] = 0x9710, [0x158b] = 0x9799,
  [0x158c] = 0x9797, [0x158d] = 0x9794, [0x158e] = 0x97f0, [0x158f] = 0x97f8,
  [0x1590] = 0x9835, [0x1591] = 0x982f, [0x1592] = 0x9832, [0x1593] = 0x9924,
  [0x1594] = 0x991f, [0x1595] = 0x9927, [0x1596] = 0x9929, [0x1597] = 0x999e,
  [0x1598] = 0x99ee, [0x1599] = 0x99ec, [0x159a] = 0x99e5, [0x159b] = 0x99e4,
  [0x159c] = 0x99f0, [0x159d] = 0x99e3, [0x159e] = 0x99ea, [0x159f] = 0x99e9,
  [0x15a0] = 0x99e7, [0x15a1] = 0x9ab9, [0x15a2] = 0x9abf, [0x15a3] = 0x9ab4,
  [0x15a4] = 0x9abb, [0x15a5] = 0x9af6, [0x15a6] = 0x9afa, [0x15a7] = 0x9af9,
  [0x15a8] = 0x9af7, [0x15a9] = 0x9b33, [0x15aa] = 0x9b80, [0x15ab] = 0x9b85,
  [0x15ac] = 0x9b87, [0x15ad] = 0x9b7c, [0x15ae] = 0x9b7e, [0x15af] = 0x9b7b,
  [0x15b0] = 0x9b82, [0x15b1] = 0x9b93, [0x15b2] = 0x9b92, [0x15b3] = 0x9b90,
  [0x15b4] = 0x9b7a, [0x15b5] = 0x9b95, [0x15b6] = 0x9b7d, [0x15b7] = 0x9b88,
  [0x15b8] = 0x9d25, [0x15b9] = 0x9d17, [0x15ba] = 0x9d20, [0x15bb] = 0x9d1e,
  [0x15bc] = 0x9d14, [0x15bd] = 0x9d29, [0x15be] = 0x9d1d, [0x15bf] = 0x9d18,
  [0x15c0] = 0x9d22, [0x15c1] = 0x9d10, [0x15c2] = 0x9d19, [0x15c3] = 0x9d1f,
  [0x15c4] = 0x9e88, [0x15c5] = 0x9e86, [0x15c6] = 0x9e87, [0x15c7] = 0x9eae,
  [0x15c8] = 0x9ead, [0x15c9] = 0x9ed5, [0x15ca] = 0x9ed6, [0x15cb] = 0x9efa,
  [0x15cc] = 0x9f12, [0x15cd] = 0x9f3d, [0x15ce] = 0x5126, [0x15cf] = 0x5125,
  [0x15d0] = 0x5122, [0x15d1] = 0x5124, [0x15d2] = 0x5120, [0x15d3] = 0x5129,
  [0x15d4] = 0x52f4, [0x15d5] = 0x5693, [0x15d6] = 0x568c, [0x15d7] = 0x568d,
  [0x15d8] = 0x5686, [0x15d9] = 0x5684, [0x15da] = 0x5683, [0x15db] = 0x567e,
  [0x15dc] = 0x5682, [0x15dd] = 0x567f, [0x15de] = 0x5681, [0x15df] = 0x58d6,
  [0x15e0] = 0x58d4, [0x15e1] = 0x58cf, [0x15e2] = 0x58d2, [0x15e3] = 0x5b2d,
  [0x15e4] = 0x5b25, [0x15e5] = 0x5b32, [0x15e6] = 0x5b23, [0x15e7] = 0x5b2c,
  [0x15e8] = 0x5b27, [0x15e9] = 0x5b26, [0x15ea] = 0x5b2f, [0x15eb] = 0x5b2e,
  [0x15ec] = 0x5b7b, [0x15ed] = 0x5bf1, [0x15ee] = 0x5bf2, [0x15ef] = 0x5db7,
  [0x15f0] = 0x5e6c, [0x15f1] = 0x5e6a, [0x15f2] = 0x5fbe, [0x15f3] = 0x61c3,
  [0x15f4] = 0x61b5, [0x15f5] = 0x61bc, [0x15f6] = 0x61e7, [0x15f7] = 0x61e0,
  [0x15f8] = 0x61e5, [0x15f9] = 0x61e4, [0x15fa] = 0x61e8, [0x15fb] = 0x61de,
  [0x15fc] = 0x64ef, [0x15fd] = 0x64e9, [0x15fe] = 0x64e3, [0x15ff] = 0x64eb,
  [0x1600] = 0x64e4, [0x1601] = 0x64e8, [0x1602] = 0x6581, [0x1603] = 0x6580,
  [0x1604] = 0x65b6, [0x1605] = 0x65da, [0x1606] = 0x66d2, [0x1607] = 0x6a8d,
  [0x1608] = 0x6a96, [0x1609] = 0x6a81, [0x160a] = 0x6aa5, [0x160b] = 0x6a89,
  [0x160c] = 0x6a9f, [0x160d] = 0x6a9b, [0x160e] = 0x6aa1, [0x160f] = 0x6a9e,
  [0x1610] = 0x6a87, [0x1611] = 0x6a93, [0x1612] = 0x6a8e, [0x1613] = 0x6a95,
  [0x1614] = 0x6a83, [0x1615] = 0x6aa8, [0x1616] = 0x6aa4, [0x1617] = 0x6a91,
  [0x1618] = 0x6a7f, [0x1619] = 0x6aa6, [0x161a] = 0x6a9a, [0x161b] = 0x6a85,
  [0x161c] = 0x6a8c, [0x161d] = 0x6a92, [0x161e] = 0x6b5b, [0x161f] = 0x6bad,
  [0x1620] = 0x6c09, [0x1621] = 0x6fcc, [0x1622] = 0x6fa9, [0x1623] = 0x6ff4,
  [0x1624] = 0x6fd4, [0x1625] = 0x6fe3, [0x1626] = 0x6fdc, [0x1627] = 0x6fed,
  [0x1628] = 0x6fe7, [0x1629] = 0x6fe6, [0x162a] = 0x6fde, [0x162b] = 0x6ff2,
  [0x162c] = 0x6fdd, [0x162d] = 0x6fe2, [0x162e] = 0x6fe8, [0x162f] = 0x71e1,
  [0x1630] = 0x71f1, [0x1631] = 0x71e8, [0x1632] = 0x71f2, [0x1633] = 0x71e4,
  [0x1634] = 0x71f0, [0x1635] = 0x71e2, [0x1636] = 0x7373, [0x1637] = 0x736e,
  [0x1638] = 0x736f, [0x1639] = 0x7497, [0x163a] = 0x74b2, [0x163b] = 0x74ab,
  [0x163c] = 0x7490, [0x163d] = 0x74aa, [0x163e] = 0x74ad, [0x163f] = 0x74b1,
  [0x1640] = 0x74a5, [0x1641] = 0x74af, [0x1642] = 0x7510, [0x1643] = 0x7511,
  [0x1644] = 0x7512, [0x1645] = 0x750f, [0x1646] = 0x7584, [0x1647] = 0x7643,
  [0x1648] = 0x7648, [0x1649] = 0x7649, [0x164a] = 0x7647, [0x164b] = 0x76a4,
  [0x164c] = 0x76e9, [0x164d] = 0x77b5, [0x164e] = 0x77ab, [0x164f] = 0x77b2,
  [0x1650] = 0x77b7, [0x1651] = 0x77b6, [0x1652] = 0x77b4, [0x1653] = 0x77b1,
  [0x1654] = 0x77a8, [0x1655] = 0x77f0, [0x1656] = 0x78f3, [0x1657] = 0x78fd,
  [0x1658] = 0x7902, [0x1659] = 0x78fb, [0x165a] = 0x78fc, [0x165b] = 0x78ff,
  [0x165c] = 0x78f2, [0x165d] = 0x7905, [0x165e] = 0x78f9, [0x165f] = 0x78fe,
  [0x1660] = 0x7904, [0x1661] = 0x79ab, [0x1662] = 0x79a8, [0x1663] = 0x7a5c,
  [0x1664] = 0x7a5b, [0x1665] = 0x7a56, [0x1666] = 0x7a58, [0x1667] = 0x7a54,
  [0x1668] = 0x7a5a, [0x1669] = 0x7abe, [0x166a] = 0x7ac0, [0x166b] = 0x7ac1,
  [0x166c] = 0x7c05, [0x166d] = 0x7c0f, [0x166e] = 0x7bf2, [0x166f] = 0x7c00,
  [0x1670] = 0x7bff, [0x1671] = 0x7bfb, [0x1672] = 0x7c0e, [0x1673] = 0x7bf4,
  [0x1674] = 0x7c0b, [0x1675] = 0x7bf3, [0x1676] = 0x7c02, [0x1677] = 0x7c09,
  [0x1678] = 0x7c03, [0x1679] = 0x7c01, [0x167a] = 0x7bf8, [0x167b] = 0x7bfd,
  [0x167c] = 0x7c06, [0x167d] = 0x7bf0, [0x167e] = 0x7bf1, [0x167f] = 0x7c10,
  [0x1680] = 0x7c0a, [0x1681] = 0x7ce8, [0x1682] = 0x7e2d, [0x1683] = 0x7e3c,
  [0x1684] = 0x7e42, [0x1685] = 0x7e33, [0x1686] = 0x9848, [0x1687] = 0x7e38,
  [0x1688] = 0x7e2a, [0x1689] = 0x7e49, [0x168a] = 0x7e40, [0x168b] = 0x7e47,
  [0x168c] = 0x7e29, [0x168d] = 0x7e4c, [0x168e] = 0x7e30, [0x168f] = 0x7e3b,
  [0x1690] = 0x7e36, [0x1691] = 0x7e44, [0x1692] = 0x7e3a, [0x1693] = 0x7f45,
  [0x1694] = 0x7f7f, [0x1695] = 0x7f7e, [0x1696] = 0x7f7d, [0x1697] = 0x7ff4,
  [0x1698] = 0x7ff2, [0x1699] = 0x802c, [0x169a] = 0x81bb, [0x169b] = 0x81c4,
  [0x169c] = 0x81cc, [0x169d] = 0x81ca, [0x169e] = 0x81c5, [0x169f] = 0x81c7,
  [0x16a0] = 0x81bc, [0x16a1] = 0x81e9, [0x16a2] = 0x825b, [0x16a3] = 0x825a,
  [0x16a4] = 0x825c, [0x16a5] = 0x8583, [0x16a6] = 0x8580, [0x16a7] = 0x858f,
  [0x16a8] = 0x85a7, [0x16a9] = 0x8595, [0x16aa] = 0x85a0, [0x16ab] = 0x858b,
  [0x16ac] = 0x85a3, [0x16ad] = 0x857b, [0x16ae] = 0x85a4, [0x16af] = 0x859a,
  [0x16b0] = 0x859e, [0x16b1] = 0x8577, [0x16b2] = 0x857c, [0x16b3] = 0x8589,
  [0x16b4] = 0x85a1, [0x16b5] = 0x857a, [0x16b6] = 0x8578, [0x16b7] = 0x8557,
  [0x16b8] = 0x858e, [0x16b9] = 0x8596, [0x16ba] = 0x8586, [0x16bb] = 0x858d,
  [0x16bc] = 0x8599, [0x16bd] = 0x859d, [0x16be] = 0x8581, [0x16bf] = 0x85a2,
  [0x16c0] = 0x8582, [0x16c1] = 0x8588, [0x16c2] = 0x8585, [0x16c3] = 0x8579,
  [0x16c4] = 0x8576, [0x16c5] = 0x8598, [0x16c6] = 0x8590, [0x16c7] = 0x859f,
  [0x16c8] = 0x8668, [0x16c9] = 0x87be, [0x16ca] = 0x87aa, [0x16cb] = 0x87ad,
  [0x16cc] = 0x87c5, [0x16cd] = 0x87b0, [0x16ce] = 0x87ac, [0x16cf] = 0x87b9,
  [0x16d0] = 0x87b5, [0x16d1] = 0x87bc, [0x16d2] = 0x87ae, [0x16d3] = 0x87c9,
  [0x16d4] = 0x87c3, [0x16d5] = 0x87c2, [0x16d6] = 0x87cc, [0x16d7] = 0x87b7,
  [0x16d8] = 0x87af, [0x16d9] = 0x87c4, [0x16da] = 0x87ca, [0x16db] = 0x87b4,
  [0x16dc] = 0x87b6, [0x16dd] = 0x87bf, [0x16de] = 0x87b8, [0x16df] = 0x87bd,
  [0x16e0] = 0x87de, [0x16e1] = 0x87b2, [0x16e2] = 0x8935, [0x16e3] = 0x8933,
  [0x16e4] = 0x893c, [0x16e5] = 0x893e, [0x16e6] = 0x8941, [0x16e7] = 0x8952,
  [0x16e8] = 0x8937, [0x16e9] = 0x8942, [0x16ea] = 0x89ad, [0x16eb] = 0x89af,
  [0x16ec] = 0x89ae, [0x16ed] = 0x89f2, [0x16ee] = 0x89f3, [0x16ef] = 0x8b1e,
  [0x16f0] = 0x8b18, [0x16f1] = 0x8b16, [0x16f2] = 0x8b11, [0x16f3] = 0x8b05,
  [0x16f4] = 0x8b0b, [0x16f5] = 0x8b22, [0x16f6] = 0x8b0f, [0x16f7] = 0x8b12,
  [0x16f8] = 0x8b15, [0x16f9] = 0x8b07, [0x16fa] = 0x8b0d, [0x16fb] = 0x8b08,
  [0x16fc] = 0x8b06, [0x16fd] = 0x8b1c, [0x16fe] = 0x8b13, [0x16ff] = 0x8b1a,
  [0x1700] = 0x8c4f, [0x1701] = 0x8c70, [0x1702] = 0x8c72, [0x1703] = 0x8c71,
  [0x1704] = 0x8c6f, [0x1705] = 0x8c95, [0x1706] = 0x8c94, [0x1707] = 0x8cf9,
  [0x1708] = 0x8d6f, [0x1709] = 0x8e4e, [0x170a] = 0x8e4d, [0x170b] = 0x8e53,
  [0x170c] = 0x8e50, [0x170d] = 0x8e4c, [0x170e] = 0x8e47, [0x170f] = 0x8f43,
  [0x1710] = 0x8f40, [0x1711] = 0x9085, [0x1712] = 0x907e, [0x1713] = 0x9138,
  [0x1714] = 0x919a, [0x1715] = 0x91a2, [0x1716] = 0x919b, [0x1717] = 0x9199,
  [0x1718] = 0x919f, [0x1719] = 0x91a1, [0x171a] = 0x919d, [0x171b] = 0x91a0,
  [0x171c] = 0x93a1, [0x171d] = 0x9383, [0x171e] = 0x93af, [0x171f] = 0x9364,
  [0x1720] = 0x9356, [0x1721] = 0x9347, [0x1722] = 0x937c, [0x1723] = 0x9358,
  [0x1724] = 0x935c, [0x1725] = 0x9376, [0x1726] = 0x9349, [0x1727] = 0x9350,
  [0x1728] = 0x9351, [0x1729] = 0x9360, [0x172a] = 0x936d, [0x172b] = 0x938f,
  [0x172c] = 0x934c, [0x172d] = 0x936a, [0x172e] = 0x9379, [0x172f] = 0x9357,
  [0x1730] = 0x9355, [0x1731] = 0x9352, [0x1732] = 0x934f, [0x1733] = 0x9371,
  [0x1734] = 0x9377, [0x1735] = 0x937b, [0x1736] = 0x9361, [0x1737] = 0x935e,
  [0x1738] = 0x9363, [0x1739] = 0x9367, [0x173a] = 0x934e, [0x173b] = 0x9359,
  [0x173c] = 0x95c7, [0x173d] = 0x95c0, [0x173e] = 0x95c9, [0x173f] = 0x95c3,
  [0x1740] = 0x95c5, [0x1741] = 0x95b7, [0x1742] = 0x96ae, [0x1743] = 0x96b0,
  [0x1744] = 0x96ac, [0x1745] = 0x9720, [0x1746] = 0x971f, [0x1747] = 0x9718,
  [0x1748] = 0x971d, [0x1749] = 0x9719, [0x174a] = 0x979a, [0x174b] = 0x97a1,
  [0x174c] = 0x979c, [0x174d] = 0x979e, [0x174e] = 0x979d, [0x174f] = 0x97d5,
  [0x1750] = 0x97d4, [0x1751] = 0x97f1, [0x1752] = 0x9841, [0x1753] = 0x9844,
  [0x1754] = 0x984a, [0x1755] = 0x9849, [0x1756] = 0x9845, [0x1757] = 0x9843,
  [0x1758] = 0x9925, [0x1759] = 0x992b, [0x175a] = 0x992c, [0x175b] = 0x992a,
  [0x175c] = 0x9933, [0x175d] = 0x9932, [0x175e] = 0x992f, [0x175f] = 0x992d,
  [0x1760] = 0x9931, [0x1761] = 0x9930, [0x1762] = 0x9998, [0x1763] = 0x99a3,
  [0x1764] = 0x99a1, [0x1765] = 0x9a02, [0x1766] = 0x99fa, [0x1767] = 0x99f4,
  [0x1768] = 0x99f7, [0x1769] = 0x99f9, [0x176a] = 0x99f8, [0x176b] = 0x99f6,
  [0x176c] = 0x99fb, [0x176d] = 0x99fd, [0x176e] = 0x99fe, [0x176f] = 0x99fc,
  [0x1770] = 0x9a03, [0x1771] = 0x9abe, [0x1772] = 0x9afe, [0x1773] = 0x9afd,
  [0x1774] = 0x9b01, [0x1775] = 0x9afc, [0x1776] = 0x9b48, [0x1777] = 0x9b9a,
  [0x1778] = 0x9ba8, [0x1779] = 0x9b9e, [0x177a] = 0x9b9b, [0x177b] = 0x9ba6,
  [0x177c] = 0x9ba1, [0x177d] = 0x9ba5, [0x177e] = 0x9ba4, [0x177f] = 0x9b86,
  [0x1780] = 0x9ba2, [0x1781] = 0x9ba0, [0x1782] = 0x9baf, [0x1783] = 0x9d33,
  [0x1784] = 0x9d41, [0x1785] = 0x9d67, [0x1786] = 0x9d36, [0x1787] = 0x9d2e,
  [0x1788] = 0x9d2f, [0x1789] = 0x9d31, [0x178a] = 0x9d38, [0x178b] = 0x9d30,
  [0x178c] = 0x9d45, [0x178d] = 0x9d42, [0x178e] = 0x9d43, [0x178f] = 0x9d3e,
  [0x1790] = 0x9d37, [0x1791] = 0x9d40, [0x1792] = 0x9d3d, [0x1793] = 0x7ff5,
  [0x1794] = 0x9d2d, [0x1795] = 0x9e8a, [0x1796] = 0x9e89, [0x1797] = 0x9e8d,
  [0x1798] = 0x9eb0, [0x1799] = 0x9ec8, [0x179a] = 0x9eda, [0x179b] = 0x9efb,
  [0x179c] = 0x9eff, [0x179d] = 0x9f24, [0x179e] = 0x9f23, [0x179f] = 0x9f22,
  [0x17a0] = 0x9f54, [0x17a1] = 0x9fa0, [0x17a2] = 0x5131, [0x17a3] = 0x512d,
  [0x17a4] = 0x512e, [0x17a5] = 0x5698, [0x17a6] = 0x569c, [0x17a7] = 0x5697,
  [0x17a8] = 0x569a, [0x17a9] = 0x569d, [0x17aa] = 0x5699, [0x17ab] = 0x5970,
  [0x17ac] = 0x5b3c, [0x17ad] = 0x5c69, [0x17ae] = 0x5c6a, [0x17af] = 0x5dc0,
  [0x17b0] = 0x5e6d, [0x17b1] = 0x5e6e, [0x17b2] = 0x61d8, [0x17b3] = 0x61df,
  [0x17b4] = 0x61ed, [0x17b5] = 0x61ee, [0x17b6] = 0x61f1, [0x17b7] = 0x61ea,
  [0x17b8] = 0x61f0, [0x17b9] = 0x61eb, [0x17ba] = 0x61d6, [0x17bb] = 0x61e9,
  [0x17bc] = 0x64ff, [0x17bd] = 0x6504, [0x17be] = 0x64fd, [0x17bf] = 0x64f8,
  [0x17c0] = 0x6501, [0x17c1] = 0x6503, [0x17c2] = 0x64fc, [0x17c3] = 0x6594,
  [0x17c4] = 0x65db, [0x17c5] = 0x66da, [0x17c6] = 0x66db, [0x17c7] = 0x66d8,
  [0x17c8] = 0x6ac5, [0x17c9] = 0x6ab9, [0x17ca] = 0x6abd, [0x17cb] = 0x6ae1,
  [0x17cc] = 0x6ac6, [0x17cd] = 0x6aba, [0x17ce] = 0x6ab6, [0x17cf] = 0x6ab7,
  [0x17d0] = 0x6ac7, [0x17d1] = 0x6ab4, [0x17d2] = 0x6aad, [0x17d3] = 0x6b5e,
  [0x17d4] = 0x6bc9, [0x17d5] = 0x6c0b, [0x17d6] = 0x7007, [0x17d7] = 0x700c,
  [0x17d8] = 0x700d, [0x17d9] = 0x7001, [0x17da] = 0x7005, [0x17db] = 0x7014,
  [0x17dc] = 0x700e, [0x17dd] = 0x6fff, [0x17de] = 0x7000, [0x17df] = 0x6ffb,
  [0x17e0] = 0x7026, [0x17e1] = 0x6ffc, [0x17e2] = 0x6ff7, [0x17e3] = 0x700a,
  [0x17e4] = 0x7201, [0x17e5] = 0x71ff, [0x17e6] = 0x71f9, [0x17e7] = 0x7203,
  [0x17e8] = 0x71fd, [0x17e9] = 0x7376, [0x17ea] = 0x74b8, [0x17eb] = 0x74c0,
  [0x17ec] = 0x74b5, [0x17ed] = 0x74c1, [0x17ee] = 0x74be, [0x17ef] = 0x74b6,
  [0x17f0] = 0x74bb, [0x17f1] = 0x74c2, [0x17f2] = 0x7514, [0x17f3] = 0x7513,
  [0x17f4] = 0x765c, [0x17f5] = 0x7664, [0x17f6] = 0x7659, [0x17f7] = 0x7650,
  [0x17f8] = 0x7653, [0x17f9] = 0x7657, [0x17fa] = 0x765a, [0x17fb] = 0x76a6,
  [0x17fc] = 0x76bd, [0x17fd] = 0x76ec, [0x17fe] = 0x77c2, [0x17ff] = 0x77ba,
  [0x1800] = 0x790c, [0x1801] = 0x7913, [0x1802] = 0x7914, [0x1803] = 0x7909,
  [0x1804] = 0x7910, [0x1805] = 0x7912, [0x1806] = 0x7911, [0x1807] = 0x79ad,
  [0x1808] = 0x79ac, [0x1809] = 0x7a5f, [0x180a] = 0x7c1c, [0x180b] = 0x7c29,
  [0x180c] = 0x7c19, [0x180d] = 0x7c20, [0x180e] = 0x7c1f, [0x180f] = 0x7c2d,
  [0x1810] = 0x7c1d, [0x1811] = 0x7c26, [0x1812] = 0x7c28, [0x1813] = 0x7c22,
  [0x1814] = 0x7c25, [0x1815] = 0x7c30, [0x1816] = 0x7e5c, [0x1817] = 0x7e50,
  [0x1818] = 0x7e56, [0x1819] = 0x7e63, [0x181a] = 0x7e58, [0x181b] = 0x7e62,
  [0x181c] = 0x7e5f, [0x181d] = 0x7e51, [0x181e] = 0x7e60, [0x181f] = 0x7e57,
  [0x1820] = 0x7e53, [0x1821] = 0x7fb5, [0x1822] = 0x7fb3, [0x1823] = 0x7ff7,
  [0x1824] = 0x7ff8, [0x1825] = 0x8075, [0x1826] = 0x81d1, [0x1827] = 0x81d2,
  [0x1828] = 0x81d0, [0x1829] = 0x825f, [0x182a] = 0x825e, [0x182b] = 0x85b4,
  [0x182c] = 0x85c6, [0x182d] = 0x85c0, [0x182e] = 0x85c3, [0x182f] = 0x85c2,
  [0x1830] = 0x85b3, [0x1831] = 0x85b5, [0x1832] = 0x85bd, [0x1833] = 0x85c7,
  [0x1834] = 0x85c4, [0x1835] = 0x85bf, [0x1836] = 0x85cb, [0x1837] = 0x85ce,
  [0x1838] = 0x85c8, [0x1839] = 0x85c5, [0x183a] = 0x85b1, [0x183b] = 0x85b6,
  [0x183c] = 0x85d2, [0x183d] = 0x8624, [0x183e] = 0x85b8, [0x183f] = 0x85b7,
  [0x1840] = 0x85be, [0x1841] = 0x8669, [0x1842] = 0x87e7, [0x1843] = 0x87e6,
  [0x1844] = 0x87e2, [0x1845] = 0x87db, [0x1846] = 0x87eb, [0x1847] = 0x87ea,
  [0x1848] = 0x87e5, [0x1849] = 0x87df, [0x184a] = 0x87f3, [0x184b] = 0x87e4,
  [0x184c] = 0x87d4, [0x184d] = 0x87dc, [0x184e] = 0x87d3, [0x184f] = 0x87ed,
  [0x1850] = 0x87d8, [0x1851] = 0x87e3, [0x1852] = 0x87d7, [0x1853] = 0x87d9,
  [0x1854] = 0x8801, [0x1855] = 0x87f4, [0x1856] = 0x87e8, [0x1857] = 0x87dd,
  [0x1858] = 0x8953, [0x1859] = 0x894b, [0x185a] = 0x894f, [0x185b] = 0x894c,
  [0x185c] = 0x8946, [0x185d] = 0x8950, [0x185e] = 0x8951, [0x185f] = 0x8949,
  [0x1860] = 0x8b2a, [0x1861] = 0x8b27, [0x1862] = 0x8b23, [0x1863] = 0x8b33,
  [0x1864] = 0x8b30, [0x1865] = 0x8b35, [0x1866] = 0x8b47, [0x1867] = 0x8b2f,
  [0x1868] = 0x8b3c, [0x1869] = 0x8b3e, [0x186a] = 0x8b31, [0x186b] = 0x8b25,
  [0x186c] = 0x8b37, [0x186d] = 0x8b26, [0x186e] = 0x8b36, [0x186f] = 0x8b2e,
  [0x1870] = 0x8b24, [0x1871] = 0x8b3b, [0x1872] = 0x8b3d, [0x1873] = 0x8b3a,
  [0x1874] = 0x8c42, [0x1875] = 0x8c75, [0x1876] = 0x8c99, [0x1877] = 0x8c98,
  [0x1878] = 0x8c97, [0x1879] = 0x8cfe, [0x187a] = 0x8d04, [0x187b] = 0x8d02,
  [0x187c] = 0x8d00, [0x187d] = 0x8e5c, [0x187e] = 0x8e62, [0x187f] = 0x8e60,
  [0x1880] = 0x8e57, [0x1881] = 0x8e56, [0x1882] = 0x8e5e, [0x1883] = 0x8e65,
  [0x1884] = 0x8e67, [0x1885] = 0x8e5b, [0x1886] = 0x8e5a, [0x1887] = 0x8e61,
  [0x1888] = 0x8e5d, [0x1889] = 0x8e69, [0x188a] = 0x8e54, [0x188b] = 0x8f46,
  [0x188c] = 0x8f47, [0x188d] = 0x8f48, [0x188e] = 0x8f4b, [0x188f] = 0x9128,
  [0x1890] = 0x913a, [0x1891] = 0x913b, [0x1892] = 0x913e, [0x1893] = 0x91a8,
  [0x1894] = 0x91a5, [0x1895] = 0x91a7, [0x1896] = 0x91af, [0x1897] = 0x91aa,
  [0x1898] = 0x93b5, [0x1899] = 0x938c, [0x189a] = 0x9392, [0x189b] = 0x93b7,
  [0x189c] = 0x939b, [0x189d] = 0x939d, [0x189e] = 0x9389, [0x189f] = 0x93a7,
  [0x18a0] = 0x938e, [0x18a1] = 0x93aa, [0x18a2] = 0x939e, [0x18a3] = 0x93a6,
  [0x18a4] = 0x9395, [0x18a5] = 0x9388, [0x18a6] = 0x9399, [0x18a7] = 0x939f,
  [0x18a8] = 0x9380, [0x18a9] = 0x938d, [0x18aa] = 0x93b1, [0x18ab] = 0x9391,
  [0x18ac] = 0x93b2, [0x18ad] = 0x93a4, [0x18ae] = 0x93a8, [0x18af] = 0x93b4,
  [0x18b0] = 0x93a3, [0x18b1] = 0x95d2, [0x18b2] = 0x95d3, [0x18b3] = 0x95d1,
  [0x18b4] = 0x96b3, [0x18b5] = 0x96d7, [0x18b6] = 0x96da, [0x18b7] = 0x5dc2,
  [0x18b8] = 0x96df, [0x18b9] = 0x96d8, [0x18ba] = 0x96dd, [0x18bb] = 0x9723,
  [0x18bc] = 0x9722, [0x18bd] = 0x9725, [0x18be] = 0x97ac, [0x18bf] = 0x97ae,
  [0x18c0] = 0x97a8, [0x18c1] = 0x97ab, [0x18c2] = 0x97a4, [0x18c3] = 0x97aa,
  [0x18c4] = 0x97a2, [0x18c5] = 0x97a5, [0x18c6] = 0x97d7, [0x18c7] = 0x97d9,
  [0x18c8] = 0x97d6, [0x18c9] = 0x97d8, [0x18ca] = 0x97fa, [0x18cb] = 0x9850,
  [0x18cc] = 0x9851, [0x18cd] = 0x9852, [0x18ce] = 0x98b8, [0x18cf] = 0x9941,
  [0x18d0] = 0x993c, [0x18d1] = 0x993a, [0x18d2] = 0x9a0f, [0x18d3] = 0x9a0b,
  [0x18d4] = 0x9a09, [0x18d5] = 0x9a0d, [0x18d6] = 0x9a04, [0x18d7] = 0x9a11,
  [0x18d8] = 0x9a0a, [0x18d9] = 0x9a05, [0x18da] = 0x9a07, [0x18db] = 0x9a06,
  [0x18dc] = 0x9ac0, [0x18dd] = 0x9adc, [0x18de] = 0x9b08, [0x18df] = 0x9b04,
  [0x18e0] = 0x9b05, [0x18e1] = 0x9b29, [0x18e2] = 0x9b35, [0x18e3] = 0x9b4a,
  [0x18e4] = 0x9b4c, [0x18e5] = 0x9b4b, [0x18e6] = 0x9bc7, [0x18e7] = 0x9bc6,
  [0x18e8] = 0x9bc3, [0x18e9] = 0x9bbf, [0x18ea] = 0x9bc1, [0x18eb] = 0x9bb5,
  [0x18ec] = 0x9bb8, [0x18ed] = 0x9bd3, [0x18ee] = 0x9bb6, [0x18ef] = 0x9bc4,
  [0x18f0] = 0x9bb9, [0x18f1] = 0x9bbd, [0x18f2] = 0x9d5c, [0x18f3] = 0x9d53,
  [0x18f4] = 0x9d4f, [0x18f5] = 0x9d4a, [0x18f6] = 0x9d5b, [0x18f7] = 0x9d4b,
  [0x18f8] = 0x9d59, [0x18f9] = 0x9d56, [0x18fa] = 0x9d4c, [0x18fb] = 0x9d57,
  [0x18fc] = 0x9d52, [0x18fd] = 0x9d54, [0x18fe] = 0x9d5f, [0x18ff] = 0x9d58,
  [0x1900] = 0x9d5a, [0x1901] = 0x9e8e, [0x1902] = 0x9e8c, [0x1903] = 0x9edf,
  [0x1904] = 0x9f01, [0x1905] = 0x9f00, [0x1906] = 0x9f16, [0x1907] = 0x9f25,
  [0x1908] = 0x9f2b, [0x1909] = 0x9f2a, [0x190a] = 0x9f29, [0x190b] = 0x9f28,
  [0x190c] = 0x9f4c, [0x190d] = 0x9f55, [0x190e] = 0x5134, [0x190f] = 0x5135,
  [0x1910] = 0x5296, [0x1911] = 0x52f7, [0x1912] = 0x53b4, [0x1913] = 0x56ab,
  [0x1914] = 0x56ad, [0x1915] = 0x56a6, [0x1916] = 0x56a7, [0x1917] = 0x56aa,
  [0x1918] = 0x56ac, [0x1919] = 0x58da, [0x191a] = 0x58dd, [0x191b] = 0x58db,
  [0x191c] = 0x5912, [0x191d] = 0x5b3d, [0x191e] = 0x5b3e, [0x191f] = 0x5b3f,
  [0x1920] = 0x5dc3, [0x1921] = 0x5e70, [0x1922] = 0x5fbf, [0x1923] = 0x61fb,
  [0x1924] = 0x6507, [0x1925] = 0x6510, [0x1926] = 0x650d, [0x1927] = 0x6509,
  [0x1928] = 0x650c, [0x1929] = 0x650e, [0x192a] = 0x6584, [0x192b] = 0x65de,
  [0x192c] = 0x65dd, [0x192d] = 0x66de, [0x192e] = 0x6ae7, [0x192f] = 0x6ae0,
  [0x1930] = 0x6acc, [0x1931] = 0x6ad1, [0x1932] = 0x6ad9, [0x1933] = 0x6acb,
  [0x1934] = 0x6adf, [0x1935] = 0x6adc, [0x1936] = 0x6ad0, [0x1937] = 0x6aeb,
  [0x1938] = 0x6acf, [0x1939] = 0x6acd, [0x193a] = 0x6ade, [0x193b] = 0x6b60,
  [0x193c] = 0x6bb0, [0x193d] = 0x6c0c, [0x193e] = 0x7019, [0x193f] = 0x7027,
  [0x1940] = 0x7020, [0x1941] = 0x7016, [0x1942] = 0x702b, [0x1943] = 0x7021,
  [0x1944] = 0x7022, [0x1945] = 0x7023, [0x1946] = 0x7029, [0x1947] = 0x7017,
  [0x1948] = 0x7024, [0x1949] = 0x701c, [0x194a] = 0x720c, [0x194b] = 0x720a,
  [0x194c] = 0x7207, [0x194d] = 0x7202, [0x194e] = 0x7205, [0x194f] = 0x72a5,
  [0x1950] = 0x72a6, [0x1951] = 0x72a4, [0x1952] = 0x72a3, [0x1953] = 0x72a1,
  [0x1954] = 0x74cb, [0x1955] = 0x74c5, [0x1956] = 0x74b7, [0x1957] = 0x74c3,
  [0x1958] = 0x7516, [0x1959] = 0x7660, [0x195a] = 0x77c9, [0x195b] = 0x77ca,
  [0x195c] = 0x77c4, [0x195d] = 0x77f1, [0x195e] = 0x791d, [0x195f] = 0x791b,
  [0x1960] = 0x7921, [0x1961] = 0x791c, [0x1962] = 0x7917, [0x1963] = 0x791e,
  [0x1964] = 0x79b0, [0x1965] = 0x7a67, [0x1966] = 0x7a68, [0x1967] = 0x7c33,
  [0x1968] = 0x7c3c, [0x1969] = 0x7c39, [0x196a] = 0x7c2c, [0x196b] = 0x7c3b,
  [0x196c] = 0x7cec, [0x196d] = 0x7cea, [0x196e] = 0x7e76, [0x196f] = 0x7e75,
  [0x1970] = 0x7e78, [0x1971] = 0x7e70, [0x1972] = 0x7e77, [0x1973] = 0x7e6f,
  [0x1974] = 0x7e7a, [0x1975] = 0x7e72, [0x1976] = 0x7e74, [0x1977] = 0x7e68,
  [0x1978] = 0x7f4b, [0x1979] = 0x7f4a, [0x197a] = 0x7f83, [0x197b] = 0x7f86,
  [0x197c] = 0x7fb7, [0x197d] = 0x7ffd, [0x197e] = 0x7ffe, [0x197f] = 0x8078,
  [0x1980] = 0x81d7, [0x1981] = 0x81d5, [0x1982] = 0x820b, [0x1983] = 0x8264,
  [0x1984] = 0x8261, [0x1985] = 0x8263, [0x1986] = 0x85eb, [0x1987] = 0x85f1,
  [0x1988] = 0x85ed, [0x1989] = 0x85d9, [0x198a] = 0x85e1, [0x198b] = 0x85e8,
  [0x198c] = 0x85da, [0x198d] = 0x85d7, [0x198e] = 0x85ec, [0x198f] = 0x85f2,
  [0x1990] = 0x85f8, [0x1991] = 0x85d8, [0x1992] = 0x85df, [0x1993] = 0x85e3,
  [0x1994] = 0x85dc, [0x1995] = 0x85d1, [0x1996] = 0x85f0, [0x1997] = 0x85e6,
  [0x1998] = 0x85ef, [0x1999] = 0x85de, [0x199a] = 0x85e2, [0x199b] = 0x8800,
  [0x199c] = 0x87fa, [0x199d] = 0x8803, [0x199e] = 0x87f6, [0x199f] = 0x87f7,
  [0x19a0] = 0x8809, [0x19a1] = 0x880c, [0x19a2] = 0x880b, [0x19a3] = 0x8806,
  [0x19a4] = 0x87fc, [0x19a5] = 0x8808, [0x19a6] = 0x87ff, [0x19a7] = 0x880a,
  [0x19a8] = 0x8802, [0x19a9] = 0x8962, [0x19aa] = 0x895a, [0x19ab] = 0x895b,
  [0x19ac] = 0x8957, [0x19ad] = 0x8961, [0x19ae] = 0x895c, [0x19af] = 0x8958,
  [0x19b0] = 0x895d, [0x19b1] = 0x8959, [0x19b2] = 0x8988, [0x19b3] = 0x89b7,
  [0x19b4] = 0x89b6, [0x19b5] = 0x89f6, [0x19b6] = 0x8b50, [0x19b7] = 0x8b48,
  [0x19b8] = 0x8b4a, [0x19b9] = 0x8b40, [0x19ba] = 0x8b53, [0x19bb] = 0x8b56,
  [0x19bc] = 0x8b54, [0x19bd] = 0x8b4b, [0x19be] = 0x8b55, [0x19bf] = 0x8b51,
  [0x19c0] = 0x8b42, [0x19c1] = 0x8b52, [0x19c2] = 0x8b57, [0x19c3] = 0x8c43,
  [0x19c4] = 0x8c77, [0x19c5] = 0x8c76, [0x19c6] = 0x8c9a, [0x19c7] = 0x8d06,
  [0x19c8] = 0x8d07, [0x19c9] = 0x8d09, [0x19ca] = 0x8dac, [0x19cb] = 0x8daa,
  [0x19cc] = 0x8dad, [0x19cd] = 0x8dab, [0x19ce] = 0x8e6d, [0x19cf] = 0x8e78,
  [0x19d0] = 0x8e73, [0x19d1] = 0x8e6a, [0x19d2] = 0x8e6f, [0x19d3] = 0x8e7b,
  [0x19d4] = 0x8ec2, [0x19d5] = 0x8f52, [0x19d6] = 0x8f51, [0x19d7] = 0x8f4f,
  [0x19d8] = 0x8f50, [0x19d9] = 0x8f53, [0x19da] = 0x8fb4, [0x19db] = 0x9140,
  [0x19dc] = 0x913f, [0x19dd] = 0x91b0, [0x19de] = 0x91ad, [0x19df] = 0x93de,
  [0x19e0] = 0x93c7, [0x19e1] = 0x93cf, [0x19e2] = 0x93c2, [0x19e3] = 0x93da,
  [0x19e4] = 0x93d0, [0x19e5] = 0x93f9, [0x19e6] = 0x93ec, [0x19e7] = 0x93cc,
  [0x19e8] = 0x93d9, [0x19e9] = 0x93a9, [0x19ea] = 0x93e6, [0x19eb] = 0x93ca,
  [0x19ec] = 0x93d4, [0x19ed] = 0x93ee, [0x19ee] = 0x93e3, [0x19ef] = 0x93d5,
  [0x19f0] = 0x93c4, [0x19f1] = 0x93ce, [0x19f2] = 0x93c0, [0x19f3] = 0x93d2,
  [0x19f4] = 0x93a5, [0x19f5] = 0x93e7, [0x19f6] = 0x957d, [0x19f7] = 0x95da,
  [0x19f8] = 0x95db, [0x19f9] = 0x96e1, [0x19fa] = 0x9729, [0x19fb] = 0x972b,
  [0x19fc] = 0x972c, [0x19fd] = 0x9728, [0x19fe] = 0x9726, [0x19ff] = 0x97b3,
  [0x1a00] = 0x97b7, [0x1a01] = 0x97b6, [0x1a02] = 0x97dd, [0x1a03] = 0x97de,
  [0x1a04] = 0x97df, [0x1a05] = 0x985c, [0x1a06] = 0x9859, [0x1a07] = 0x985d,
  [0x1a08] = 0x9857, [0x1a09] = 0x98bf, [0x1a0a] = 0x98bd, [0x1a0b] = 0x98bb,
  [0x1a0c] = 0x98be, [0x1a0d] = 0x9948, [0x1a0e] = 0x9947, [0x1a0f] = 0x9943,
  [0x1a10] = 0x99a6, [0x1a11] = 0x99a7, [0x1a12] = 0x9a1a, [0x1a13] = 0x9a15,
  [0x1a14] = 0x9a25, [0x1a15] = 0x9a1d, [0x1a16] = 0x9a24, [0x1a17] = 0x9a1b,
  [0x1a18] = 0x9a22, [0x1a19] = 0x9a20, [0x1a1a] = 0x9a27, [0x1a1b] = 0x9a23,
  [0x1a1c] = 0x9a1e, [0x1a1d] = 0x9a1c, [0x1a1e] = 0x9a14, [0x1a1f] = 0x9ac2,
  [0x1a20] = 0x9b0b, [0x1a21] = 0x9b0a, [0x1a22] = 0x9b0e, [0x1a23] = 0x9b0c,
  [0x1a24] = 0x9b37, [0x1a25] = 0x9bea, [0x1a26] = 0x9beb, [0x1a27] = 0x9be0,
  [0x1a28] = 0x9bde, [0x1a29] = 0x9be4, [0x1a2a] = 0x9be6, [0x1a2b] = 0x9be2,
  [0x1a2c] = 0x9bf0, [0x1a2d] = 0x9bd4, [0x1a2e] = 0x9bd7, [0x1a2f] = 0x9bec,
  [0x1a30] = 0x9bdc, [0x1a31] = 0x9bd9, [0x1a32] = 0x9be5, [0x1a33] = 0x9bd5,
  [0x1a34] = 0x9be1, [0x1a35] = 0x9bda, [0x1a36] = 0x9d77, [0x1a37] = 0x9d81,
  [0x1a38] = 0x9d8a, [0x1a39] = 0x9d84, [0x1a3a] = 0x9d88, [0x1a3b] = 0x9d71,
  [0x1a3c] = 0x9d80, [0x1a3d] = 0x9d78, [0x1a3e] = 0x9d86, [0x1a3f] = 0x9d8b,
  [0x1a40] = 0x9d8c, [0x1a41] = 0x9d7d, [0x1a42] = 0x9d6b, [0x1a43] = 0x9d74,
  [0x1a44] = 0x9d75, [0x1a45] = 0x9d70, [0x1a46] = 0x9d69, [0x1a47] = 0x9d85,
  [0x1a48] = 0x9d73, [0x1a49] = 0x9d7b, [0x1a4a] = 0x9d82, [0x1a4b] = 0x9d6f,
  [0x1a4c] = 0x9d79, [0x1a4d] = 0x9d7f, [0x1a4e] = 0x9d87, [0x1a4f] = 0x9d68,
  [0x1a50] = 0x9e94, [0x1a51] = 0x9e91, [0x1a52] = 0x9ec0, [0x1a53] = 0x9efc,
  [0x1a54] = 0x9f2d, [0x1a55] = 0x9f40, [0x1a56] = 0x9f41, [0x1a57] = 0x9f4d,
  [0x1a58] = 0x9f56, [0x1a59] = 0x9f57, [0x1a5a] = 0x9f58, [0x1a5b] = 0x5337,
  [0x1a5c] = 0x56b2, [0x1a5d] = 0x56b5, [0x1a5e] = 0x56b3, [0x1a5f] = 0x58e3,
  [0x1a60] = 0x5b45, [0x1a61] = 0x5dc6, [0x1a62] = 0x5dc7, [0x1a63] = 0x5eee,
  [0x1a64] = 0x5eef, [0x1a65] = 0x5fc0, [0x1a66] = 0x5fc1, [0x1a67] = 0x61f9,
  [0x1a68] = 0x6517, [0x1a69] = 0x6516, [0x1a6a] = 0x6515, [0x1a6b] = 0x6513,
  [0x1a6c] = 0x65df, [0x1a6d] = 0x66e8, [0x1a6e] = 0x66e3, [0x1a6f] = 0x66e4,
  [0x1a70] = 0x6af3, [0x1a71] = 0x6af0, [0x1a72] = 0x6aea, [0x1a73] = 0x6ae8,
  [0x1a74] = 0x6af9, [0x1a75] = 0x6af1, [0x1a76] = 0x6aee, [0x1a77] = 0x6aef,
  [0x1a78] = 0x703c, [0x1a79] = 0x7035, [0x1a7a] = 0x702f, [0x1a7b] = 0x7037,
  [0x1a7c] = 0x7034, [0x1a7d] = 0x7031, [0x1a7e] = 0x7042, [0x1a7f] = 0x7038,
  [0x1a80] = 0x703f, [0x1a81] = 0x703a, [0x1a82] = 0x7039, [0x1a83] = 0x702a,
  [0x1a84] = 0x7040, [0x1a85] = 0x703b, [0x1a86] = 0x7033, [0x1a87] = 0x7041,
  [0x1a88] = 0x7213, [0x1a89] = 0x7214, [0x1a8a] = 0x72a8, [0x1a8b] = 0x737d,
  [0x1a8c] = 0x737c, [0x1a8d] = 0x74ba, [0x1a8e] = 0x76ab, [0x1a8f] = 0x76aa,
  [0x1a90] = 0x76be, [0x1a91] = 0x76ed, [0x1a92] = 0x77cc, [0x1a93] = 0x77ce,
  [0x1a94] = 0x77cf, [0x1a95] = 0x77cd, [0x1a96] = 0x77f2, [0x1a97] = 0x7925,
  [0x1a98] = 0x7923, [0x1a99] = 0x7927, [0x1a9a] = 0x7928, [0x1a9b] = 0x7924,
  [0x1a9c] = 0x7929, [0x1a9d] = 0x79b2, [0x1a9e] = 0x7a6e, [0x1a9f] = 0x7a6c,
  [0x1aa0] = 0x7a6d, [0x1aa1] = 0x7af7, [0x1aa2] = 0x7c49, [0x1aa3] = 0x7c48,
  [0x1aa4] = 0x7c4a, [0x1aa5] = 0x7c47, [0x1aa6] = 0x7c45, [0x1aa7] = 0x7cee,
  [0x1aa8] = 0x7e7b, [0x1aa9] = 0x7e7e, [0x1aaa] = 0x7e81, [0x1aab] = 0x7e80,
  [0x1aac] = 0x7fba, [0x1aad] = 0x7fff, [0x1aae] = 0x8079, [0x1aaf] = 0x81db,
  [0x1ab0] = 0x81d9, [0x1ab1] = 0x8268, [0x1ab2] = 0x8269, [0x1ab3] = 0x8622,
  [0x1ab4] = 0x85ff, [0x1ab5] = 0x8601, [0x1ab6] = 0x85fe, [0x1ab7] = 0x861b,
  [0x1ab8] = 0x8600, [0x1ab9] = 0x85f6, [0x1aba] = 0x8604, [0x1abb] = 0x8609,
  [0x1abc] = 0x8605, [0x1abd] = 0x860c, [0x1abe] = 0x85fd, [0x1abf] = 0x8819,
  [0x1ac0] = 0x8810, [0x1ac1] = 0x8811, [0x1ac2] = 0x8817, [0x1ac3] = 0x8813,
  [0x1ac4] = 0x8816, [0x1ac5] = 0x8963, [0x1ac6] = 0x8966, [0x1ac7] = 0x89b9,
  [0x1ac8] = 0x89f7, [0x1ac9] = 0x8b60, [0x1aca] = 0x8b6a, [0x1acb] = 0x8b5d,
  [0x1acc] = 0x8b68, [0x1acd] = 0x8b63, [0x1ace] = 0x8b65, [0x1acf] = 0x8b67,
  [0x1ad0] = 0x8b6d, [0x1ad1] = 0x8dae, [0x1ad2] = 0x8e86, [0x1ad3] = 0x8e88,
  [0x1ad4] = 0x8e84, [0x1ad5] = 0x8f59, [0x1ad6] = 0x8f56, [0x1ad7] = 0x8f57,
  [0x1ad8] = 0x8f55, [0x1ad9] = 0x8f58, [0x1ada] = 0x8f5a, [0x1adb] = 0x908d,
  [0x1adc] = 0x9143, [0x1add] = 0x9141, [0x1ade] = 0x91b7, [0x1adf] = 0x91b5,
  [0x1ae0] = 0x91b2, [0x1ae1] = 0x91b3, [0x1ae2] = 0x940b, [0x1ae3] = 0x9413,
  [0x1ae4] = 0x93fb, [0x1ae5] = 0x9420, [0x1ae6] = 0x940f, [0x1ae7] = 0x9414,
  [0x1ae8] = 0x93fe, [0x1ae9] = 0x9415, [0x1aea] = 0x9410, [0x1aeb] = 0x9428,
  [0x1aec] = 0x9419, [0x1aed] = 0x940d, [0x1aee] = 0x93f5, [0x1aef] = 0x9400,
  [0x1af0] = 0x93f7, [0x1af1] = 0x9407, [0x1af2] = 0x940e, [0x1af3] = 0x9416,
  [0x1af4] = 0x9412, [0x1af5] = 0x93fa, [0x1af6] = 0x9409, [0x1af7] = 0x93f8,
  [0x1af8] = 0x943c, [0x1af9] = 0x940a, [0x1afa] = 0x93ff, [0x1afb] = 0x93fc,
  [0x1afc] = 0x940c, [0x1afd] = 0x93f6, [0x1afe] = 0x9411, [0x1aff] = 0x9406,
  [0x1b00] = 0x95de, [0x1b01] = 0x95e0, [0x1b02] = 0x95df, [0x1b03] = 0x972e,
  [0x1b04] = 0x972f, [0x1b05] = 0x97b9, [0x1b06] = 0x97bb, [0x1b07] = 0x97fd,
  [0x1b08] = 0x97fe, [0x1b09] = 0x9860, [0x1b0a] = 0x9862, [0x1b0b] = 0x9863,
  [0x1b0c] = 0x985f, [0x1b0d] = 0x98c1, [0x1b0e] = 0x98c2, [0x1b0f] = 0x9950,
  [0x1b10] = 0x994e, [0x1b11] = 0x9959, [0x1b12] = 0x994c, [0x1b13] = 0x994b,
  [0x1b14] = 0x9953, [0x1b15] = 0x9a32, [0x1b16] = 0x9a34, [0x1b17] = 0x9a31,
  [0x1b18] = 0x9a2c, [0x1b19] = 0x9a2a, [0x1b1a] = 0x9a36, [0x1b1b] = 0x9a29,
  [0x1b1c] = 0x9a2e, [0x1b1d] = 0x9a38, [0x1b1e] = 0x9a2d, [0x1b1f] = 0x9ac7,
  [0x1b20] = 0x9aca, [0x1b21] = 0x9ac6, [0x1b22] = 0x9b10, [0x1b23] = 0x9b12,
  [0x1b24] = 0x9b11, [0x1b25] = 0x9c0b, [0x1b26] = 0x9c08, [0x1b27] = 0x9bf7,
  [0x1b28] = 0x9c05, [0x1b29] = 0x9c12, [0x1b2a] = 0x9bf8, [0x1b2b] = 0x9c40,
  [0x1b2c] = 0x9c07, [0x1b2d] = 0x9c0e, [0x1b2e] = 0x9c06, [0x1b2f] = 0x9c17,
  [0x1b30] = 0x9c14, [0x1b31] = 0x9c09, [0x1b32] = 0x9d9f, [0x1b33] = 0x9d99,
  [0x1b34] = 0x9da4, [0x1b35] = 0x9d9d, [0x1b36] = 0x9d92, [0x1b37] = 0x9d98,
  [0x1b38] = 0x9d90, [0x1b39] = 0x9d9b, [0x1b3a] = 0x9da0, [0x1b3b] = 0x9d94,
  [0x1b3c] = 0x9d9c, [0x1b3d] = 0x9daa, [0x1b3e] = 0x9d97, [0x1b3f] = 0x9da1,
  [0x1b40] = 0x9d9a, [0x1b41] = 0x9da2, [0x1b42] = 0x9da8, [0x1b43] = 0x9d9e,
  [0x1b44] = 0x9da3, [0x1b45] = 0x9dbf, [0x1b46] = 0x9da9, [0x1b47] = 0x9d96,
  [0x1b48] = 0x9da6, [0x1b49] = 0x9da7, [0x1b4a] = 0x9e99, [0x1b4b] = 0x9e9b,
  [0x1b4c] = 0x9e9a, [0x1b4d] = 0x9ee5, [0x1b4e] = 0x9ee4, [0x1b4f] = 0x9ee7,
  [0x1b50] = 0x9ee6, [0x1b51] = 0x9f30, [0x1b52] = 0x9f2e, [0x1b53] = 0x9f5b,
  [0x1b54] = 0x9f60, [0x1b55] = 0x9f5e, [0x1b56] = 0x9f5d, [0x1b57] = 0x9f59,
  [0x1b58] = 0x9f91, [0x1b59] = 0x513a, [0x1b5a] = 0x5139, [0x1b5b] = 0x5298,
  [0x1b5c] = 0x5297, [0x1b5d] = 0x56c3, [0x1b5e] = 0x56bd, [0x1b5f] = 0x56be,
  [0x1b60] = 0x5b48, [0x1b61] = 0x5b47, [0x1b62] = 0x5dcb, [0x1b63] = 0x5dcf,
  [0x1b64] = 0x5ef1, [0x1b65] = 0x61fd, [0x1b66] = 0x651b, [0x1b67] = 0x6b02,
  [0x1b68] = 0x6afc, [0x1b69] = 0x6b03, [0x1b6a] = 0x6af8, [0x1b6b] = 0x6b00,
  [0x1b6c] = 0x7043, [0x1b6d] = 0x7044, [0x1b6e] = 0x704a, [0x1b6f] = 0x7048,
  [0x1b70] = 0x7049, [0x1b71] = 0x7045, [0x1b72] = 0x7046, [0x1b73] = 0x721d,
  [0x1b74] = 0x721a, [0x1b75] = 0x7219, [0x1b76] = 0x737e, [0x1b77] = 0x7517,
  [0x1b78] = 0x766a, [0x1b79] = 0x77d0, [0x1b7a] = 0x792d, [0x1b7b] = 0x7931,
  [0x1b7c] = 0x792f, [0x1b7d] = 0x7c54, [0x1b7e] = 0x7c53, [0x1b7f] = 0x7cf2,
  [0x1b80] = 0x7e8a, [0x1b81] = 0x7e87, [0x1b82] = 0x7e88, [0x1b83] = 0x7e8b,
  [0x1b84] = 0x7e86, [0x1b85] = 0x7e8d, [0x1b86] = 0x7f4d, [0x1b87] = 0x7fbb,
  [0x1b88] = 0x8030, [0x1b89] = 0x81dd, [0x1b8a] = 0x8618, [0x1b8b] = 0x862a,
  [0x1b8c] = 0x8626, [0x1b8d] = 0x861f, [0x1b8e] = 0x8623, [0x1b8f] = 0x861c,
  [0x1b90] = 0x8619, [0x1b91] = 0x8627, [0x1b92] = 0x862e, [0x1b93] = 0x8621,
  [0x1b94] = 0x8620, [0x1b95] = 0x8629, [0x1b96] = 0x861e, [0x1b97] = 0x8625,
  [0x1b98] = 0x8829, [0x1b99] = 0x881d, [0x1b9a] = 0x881b, [0x1b9b] = 0x8820,
  [0x1b9c] = 0x8824, [0x1b9d] = 0x881c, [0x1b9e] = 0x882b, [0x1b9f] = 0x884a,
  [0x1ba0] = 0x896d, [0x1ba1] = 0x8969, [0x1ba2] = 0x896e, [0x1ba3] = 0x896b,
  [0x1ba4] = 0x89fa, [0x1ba5] = 0x8b79, [0x1ba6] = 0x8b78, [0x1ba7] = 0x8b45,
  [0x1ba8] = 0x8b7a, [0x1ba9] = 0x8b7b, [0x1baa] = 0x8d10, [0x1bab] = 0x8d14,
  [0x1bac] = 0x8daf, [0x1bad] = 0x8e8e, [0x1bae] = 0x8e8c, [0x1baf] = 0x8f5e,
  [0x1bb0] = 0x8f5b, [0x1bb1] = 0x8f5d, [0x1bb2] = 0x9146, [0x1bb3] = 0x9144,
  [0x1bb4] = 0x9145, [0x1bb5] = 0x91b9, [0x1bb6] = 0x943f, [0x1bb7] = 0x943b,
  [0x1bb8] = 0x9436, [0x1bb9] = 0x9429, [0x1bba] = 0x943d, [0x1bbb] = 0x9430,
  [0x1bbc] = 0x9439, [0x1bbd] = 0x942a, [0x1bbe] = 0x9437, [0x1bbf] = 0x942c,
  [0x1bc0] = 0x9440, [0x1bc1] = 0x9431, [0x1bc2] = 0x95e5, [0x1bc3] = 0x95e4,
  [0x1bc4] = 0x95e3, [0x1bc5] = 0x9735, [0x1bc6] = 0x973a, [0x1bc7] = 0x97bf,
  [0x1bc8] = 0x97e1, [0x1bc9] = 0x9864, [0x1bca] = 0x98c9, [0x1bcb] = 0x98c6,
  [0x1bcc] = 0x98c0, [0x1bcd] = 0x9958, [0x1bce] = 0x9956, [0x1bcf] = 0x9a39,
  [0x1bd0] = 0x9a3d, [0x1bd1] = 0x9a46, [0x1bd2] = 0x9a44, [0x1bd3] = 0x9a42,
  [0x1bd4] = 0x9a41, [0x1bd5] = 0x9a3a, [0x1bd6] = 0x9a3f, [0x1bd7] = 0x9acd,
  [0x1bd8] = 0x9b15, [0x1bd9] = 0x9b17, [0x1bda] = 0x9b18, [0x1bdb] = 0x9b16,
  [0x1bdc] = 0x9b3a, [0x1bdd] = 0x9b52, [0x1bde] = 0x9c2b, [0x1bdf] = 0x9c1d,
  [0x1be0] = 0x9c1c, [0x1be1] = 0x9c2c, [0x1be2] = 0x9c23, [0x1be3] = 0x9c28,
  [0x1be4] = 0x9c29, [0x1be5] = 0x9c24, [0x1be6] = 0x9c21, [0x1be7] = 0x9db7,
  [0x1be8] = 0x9db6, [0x1be9] = 0x9dbc, [0x1bea] = 0x9dc1, [0x1beb] = 0x9dc7,
  [0x1bec] = 0x9dca, [0x1bed] = 0x9dcf, [0x1bee] = 0x9dbe, [0x1bef] = 0x9dc5,
  [0x1bf0] = 0x9dc3, [0x1bf1] = 0x9dbb, [0x1bf2] = 0x9db5, [0x1bf3] = 0x9dce,
  [0x1bf4] = 0x9db9, [0x1bf5] = 0x9dba, [0x1bf6] = 0x9dac, [0x1bf7] = 0x9dc8,
  [0x1bf8] = 0x9db1, [0x1bf9] = 0x9dad, [0x1bfa] = 0x9dcc, [0x1bfb] = 0x9db3,
  [0x1bfc] = 0x9dcd, [0x1bfd] = 0x9db2, [0x1bfe] = 0x9e7a, [0x1bff] = 0x9e9c,
  [0x1c00] = 0x9eeb, [0x1c01] = 0x9eee, [0x1c02] = 0x9eed, [0x1c03] = 0x9f1b,
  [0x1c04] = 0x9f18, [0x1c05] = 0x9f1a, [0x1c06] = 0x9f31, [0x1c07] = 0x9f4e,
  [0x1c08] = 0x9f65, [0x1c09] = 0x9f64, [0x1c0a] = 0x9f92, [0x1c0b] = 0x4eb9,
  [0x1c0c] = 0x56c6, [0x1c0d] = 0x56c5, [0x1c0e] = 0x56cb, [0x1c0f] = 0x5971,
  [0x1c10] = 0x5b4b, [0x1c11] = 0x5b4c, [0x1c12] = 0x5dd5, [0x1c13] = 0x5dd1,
  [0x1c14] = 0x5ef2, [0x1c15] = 0x6521, [0x1c16] = 0x6520, [0x1c17] = 0x6526,
  [0x1c18] = 0x6522, [0x1c19] = 0x6b0b, [0x1c1a] = 0x6b08, [0x1c1b] = 0x6b09,
  [0x1c1c] = 0x6c0d, [0x1c1d] = 0x7055, [0x1c1e] = 0x7056, [0x1c1f] = 0x7057,
  [0x1c20] = 0x7052, [0x1c21] = 0x721e, [0x1c22] = 0x721f, [0x1c23] = 0x72a9,
  [0x1c24] = 0x737f, [0x1c25] = 0x74d8, [0x1c26] = 0x74d5, [0x1c27] = 0x74d9,
  [0x1c28] = 0x74d7, [0x1c29] = 0x766d, [0x1c2a] = 0x76ad, [0x1c2b] = 0x7935,
  [0x1c2c] = 0x79b4, [0x1c2d] = 0x7a70, [0x1c2e] = 0x7a71, [0x1c2f] = 0x7c57,
  [0x1c30] = 0x7c5c, [0x1c31] = 0x7c59, [0x1c32] = 0x7c5b, [0x1c33] = 0x7c5a,
  [0x1c34] = 0x7cf4, [0x1c35] = 0x7cf1, [0x1c36] = 0x7e91, [0x1c37] = 0x7f4f,
  [0x1c38] = 0x7f87, [0x1c39] = 0x81de, [0x1c3a] = 0x826b, [0x1c3b] = 0x8634,
  [0x1c3c] = 0x8635, [0x1c3d] = 0x8633, [0x1c3e] = 0x862c, [0x1c3f] = 0x8632,
  [0x1c40] = 0x8636, [0x1c41] = 0x882c, [0x1c42] = 0x8828, [0x1c43] = 0x8826,
  [0x1c44] = 0x882a, [0x1c45] = 0x8825, [0x1c46] = 0x8971, [0x1c47] = 0x89bf,
  [0x1c48] = 0x89be, [0x1c49] = 0x89fb, [0x1c4a] = 0x8b7e, [0x1c4b] = 0x8b84,
  [0x1c4c] = 0x8b82, [0x1c4d] = 0x8b86, [0x1c4e] = 0x8b85, [0x1c4f] = 0x8b7f,
  [0x1c50] = 0x8d15, [0x1c51] = 0x8e95, [0x1c52] = 0x8e94, [0x1c53] = 0x8e9a,
  [0x1c54] = 0x8e92, [0x1c55] = 0x8e90, [0x1c56] = 0x8e96, [0x1c57] = 0x8e97,
  [0x1c58] = 0x8f60, [0x1c59] = 0x8f62, [0x1c5a] = 0x9147, [0x1c5b] = 0x944c,
  [0x1c5c] = 0x9450, [0x1c5d] = 0x944a, [0x1c5e] = 0x944b, [0x1c5f] = 0x944f,
  [0x1c60] = 0x9447, [0x1c61] = 0x9445, [0x1c62] = 0x9448, [0x1c63] = 0x9449,
  [0x1c64] = 0x9446, [0x1c65] = 0x973f, [0x1c66] = 0x97e3, [0x1c67] = 0x986a,
  [0x1c68] = 0x9869, [0x1c69] = 0x98cb, [0x1c6a] = 0x9954, [0x1c6b] = 0x995b,
  [0x1c6c] = 0x9a4e, [0x1c6d] = 0x9a53, [0x1c6e] = 0x9a54, [0x1c6f] = 0x9a4c,
  [0x1c70] = 0x9a4f, [0x1c71] = 0x9a48, [0x1c72] = 0x9a4a, [0x1c73] = 0x9a49,
  [0x1c74] = 0x9a52, [0x1c75] = 0x9a50, [0x1c76] = 0x9ad0, [0x1c77] = 0x9b19,
  [0x1c78] = 0x9b2b, [0x1c79] = 0x9b3b, [0x1c7a] = 0x9b56, [0x1c7b] = 0x9b55,
  [0x1c7c] = 0x9c46, [0x1c7d] = 0x9c48, [0x1c7e] = 0x9c3f, [0x1c7f] = 0x9c44,
  [0x1c80] = 0x9c39, [0x1c81] = 0x9c33, [0x1c82] = 0x9c41, [0x1c83] = 0x9c3c,
  [0x1c84] = 0x9c37, [0x1c85] = 0x9c34, [0x1c86] = 0x9c32, [0x1c87] = 0x9c3d,
  [0x1c88] = 0x9c36, [0x1c89] = 0x9ddb, [0x1c8a] = 0x9dd2, [0x1c8b] = 0x9dde,
  [0x1c8c] = 0x9dda, [0x1c8d] = 0x9dcb, [0x1c8e] = 0x9dd0, [0x1c8f] = 0x9ddc,
  [0x1c90] = 0x9dd1, [0x1c91] = 0x9ddf, [0x1c92] = 0x9de9, [0x1c93] = 0x9dd9,
  [0x1c94] = 0x9dd8, [0x1c95] = 0x9dd6, [0x1c96] = 0x9df5, [0x1c97] = 0x9dd5,
  [0x1c98] = 0x9ddd, [0x1c99] = 0x9eb6, [0x1c9a] = 0x9ef0, [0x1c9b] = 0x9f35,
  [0x1c9c] = 0x9f33, [0x1c9d] = 0x9f32, [0x1c9e] = 0x9f42, [0x1c9f] = 0x9f6b,
  [0x1ca0] = 0x9f95, [0x1ca1] = 0x9fa2, [0x1ca2] = 0x513d, [0x1ca3] = 0x5299,
  [0x1ca4] = 0x58e8, [0x1ca5] = 0x58e7, [0x1ca6] = 0x5972, [0x1ca7] = 0x5b4d,
  [0x1ca8] = 0x5dd8, [0x1ca9] = 0x882f, [0x1caa] = 0x5f4f, [0x1cab] = 0x6201,
  [0x1cac] = 0x6203, [0x1cad] = 0x6204, [0x1cae] = 0x6529, [0x1caf] = 0x6525,
  [0x1cb0] = 0x6596, [0x1cb1] = 0x66eb, [0x1cb2] = 0x6b11, [0x1cb3] = 0x6b12,
  [0x1cb4] = 0x6b0f, [0x1cb5] = 0x6bca, [0x1cb6] = 0x705b, [0x1cb7] = 0x705a,
  [0x1cb8] = 0x7222, [0x1cb9] = 0x7382, [0x1cba] = 0x7381, [0x1cbb] = 0x7383,
  [0x1cbc] = 0x7670, [0x1cbd] = 0x77d4, [0x1cbe] = 0x7c67, [0x1cbf] = 0x7c66,
  [0x1cc0] = 0x7e95, [0x1cc1] = 0x826c, [0x1cc2] = 0x863a, [0x1cc3] = 0x8640,
  [0x1cc4] = 0x8639, [0x1cc5] = 0x863c, [0x1cc6] = 0x8631, [0x1cc7] = 0x863b,
  [0x1cc8] = 0x863e, [0x1cc9] = 0x8830, [0x1cca] = 0x8832, [0x1ccb] = 0x882e,
  [0x1ccc] = 0x8833, [0x1ccd] = 0x8976, [0x1cce] = 0x8974, [0x1ccf] = 0x8973,
  [0x1cd0] = 0x89fe, [0x1cd1] = 0x8b8c, [0x1cd2] = 0x8b8e, [0x1cd3] = 0x8b8b,
  [0x1cd4] = 0x8b88, [0x1cd5] = 0x8c45, [0x1cd6] = 0x8d19, [0x1cd7] = 0x8e98,
  [0x1cd8] = 0x8f64, [0x1cd9] = 0x8f63, [0x1cda] = 0x91bc, [0x1cdb] = 0x9462,
  [0x1cdc] = 0x9455, [0x1cdd] = 0x945d, [0x1cde] = 0x9457, [0x1cdf] = 0x945e,
  [0x1ce0] = 0x97c4, [0x1ce1] = 0x97c5, [0x1ce2] = 0x9800, [0x1ce3] = 0x9a56,
  [0x1ce4] = 0x9a59, [0x1ce5] = 0x9b1e, [0x1ce6] = 0x9b1f, [0x1ce7] = 0x9b20,
  [0x1ce8] = 0x9c52, [0x1ce9] = 0x9c58, [0x1cea] = 0x9c50, [0x1ceb] = 0x9c4a,
  [0x1cec] = 0x9c4d, [0x1ced] = 0x9c4b, [0x1cee] = 0x9c55, [0x1cef] = 0x9c59,
  [0x1cf0] = 0x9c4c, [0x1cf1] = 0x9c4e, [0x1cf2] = 0x9dfb, [0x1cf3] = 0x9df7,
  [0x1cf4] = 0x9def, [0x1cf5] = 0x9de3, [0x1cf6] = 0x9deb, [0x1cf7] = 0x9df8,
  [0x1cf8] = 0x9de4, [0x1cf9] = 0x9df6, [0x1cfa] = 0x9de1, [0x1cfb] = 0x9dee,
  [0x1cfc] = 0x9de6, [0x1cfd] = 0x9df2, [0x1cfe] = 0x9df0, [0x1cff] = 0x9de2,
  [0x1d00] = 0x9dec, [0x1d01] = 0x9df4, [0x1d02] = 0x9df3, [0x1d03] = 0x9de8,
  [0x1d04] = 0x9ded, [0x1d05] = 0x9ec2, [0x1d06] = 0x9ed0, [0x1d07] = 0x9ef2,
  [0x1d08] = 0x9ef3, [0x1d09] = 0x9f06, [0x1d0a] = 0x9f1c, [0x1d0b] = 0x9f38,
  [0x1d0c] = 0x9f37, [0x1d0d] = 0x9f36, [0x1d0e] = 0x9f43, [0x1d0f] = 0x9f4f,
  [0x1d10] = 0x9f71, [0x1d11] = 0x9f70, [0x1d12] = 0x9f6e, [0x1d13] = 0x9f6f,
  [0x1d14] = 0x56d3, [0x1d15] = 0x56cd, [0x1d16] = 0x5b4e, [0x1d17] = 0x5c6d,
  [0x1d18] = 0x652d, [0x1d19] = 0x66ed, [0x1d1a] = 0x66ee, [0x1d1b] = 0x6b13,
  [0x1d1c] = 0x705f, [0x1d1d] = 0x7061, [0x1d1e] = 0x705d, [0x1d1f] = 0x7060,
  [0x1d20] = 0x7223, [0x1d21] = 0x74db, [0x1d22] = 0x74e5, [0x1d23] = 0x77d5,
  [0x1d24] = 0x7938, [0x1d25] = 0x79b7, [0x1d26] = 0x79b6, [0x1d27] = 0x7c6a,
  [0x1d28] = 0x7e97, [0x1d29] = 0x7f89, [0x1d2a] = 0x826d, [0x1d2b] = 0x8643,
  [0x1d2c] = 0x8838, [0x1d2d] = 0x8837, [0x1d2e] = 0x8835, [0x1d2f] = 0x884b,
  [0x1d30] = 0x8b94, [0x1d31] = 0x8b95, [0x1d32] = 0x8e9e, [0x1d33] = 0x8e9f,
  [0x1d34] = 0x8ea0, [0x1d35] = 0x8e9d, [0x1d36] = 0x91be, [0x1d37] = 0x91bd,
  [0x1d38] = 0x91c2, [0x1d39] = 0x946b, [0x1d3a] = 0x9468, [0x1d3b] = 0x9469,
  [0x1d3c] = 0x96e5, [0x1d3d] = 0x9746, [0x1d3e] = 0x9743, [0x1d3f] = 0x9747,
  [0x1d40] = 0x97c7, [0x1d41] = 0x97e5, [0x1d42] = 0x9a5e, [0x1d43] = 0x9ad5,
  [0x1d44] = 0x9b59, [0x1d45] = 0x9c63, [0x1d46] = 0x9c67, [0x1d47] = 0x9c66,
  [0x1d48] = 0x9c62, [0x1d49] = 0x9c5e, [0x1d4a] = 0x9c60, [0x1d4b] = 0x9e02,
  [0x1d4c] = 0x9dfe, [0x1d4d] = 0x9e07, [0x1d4e] = 0x9e03, [0x1d4f] = 0x9e06,
  [0x1d50] = 0x9e05, [0x1d51] = 0x9e00, [0x1d52] = 0x9e01, [0x1d53] = 0x9e09,
  [0x1d54] = 0x9dff, [0x1d55] = 0x9dfd, [0x1d56] = 0x9e04, [0x1d57] = 0x9ea0,
  [0x1d58] = 0x9f1e, [0x1d59] = 0x9f46, [0x1d5a] = 0x9f74, [0x1d5b] = 0x9f75,
  [0x1d5c] = 0x9f76, [0x1d5d] = 0x56d4, [0x1d5e] = 0x652e, [0x1d5f] = 0x65b8,
  [0x1d60] = 0x6b18, [0x1d61] = 0x6b19, [0x1d62] = 0x6b17, [0x1d63] = 0x6b1a,
  [0x1d64] = 0x7062, [0x1d65] = 0x7226, [0x1d66] = 0x72aa, [0x1d67] = 0x77d8,
  [0x1d68] = 0x77d9, [0x1d69] = 0x7939, [0x1d6a] = 0x7c69, [0x1d6b] = 0x7c6b,
  [0x1d6c] = 0x7cf6, [0x1d6d] = 0x7e9a, [0x1d6e] = 0x7e98, [0x1d6f] = 0x7e9b,
  [0x1d70] = 0x7e99, [0x1d71] = 0x81e0, [0x1d72] = 0x81e1, [0x1d73] = 0x8646,
  [0x1d74] = 0x8647, [0x1d75] = 0x8648, [0x1d76] = 0x8979, [0x1d77] = 0x897a,
  [0x1d78] = 0x897c, [0x1d79] = 0x897b, [0x1d7a] = 0x89ff, [0x1d7b] = 0x8b98,
  [0x1d7c] = 0x8b99, [0x1d7d] = 0x8ea5, [0x1d7e] = 0x8ea4, [0x1d7f] = 0x8ea3,
  [0x1d80] = 0x946e, [0x1d81] = 0x946d, [0x1d82] = 0x946f, [0x1d83] = 0x9471,
  [0x1d84] = 0x9473, [0x1d85] = 0x9749, [0x1d86] = 0x9872, [0x1d87] = 0x995f,
  [0x1d88] = 0x9c68, [0x1d89] = 0x9c6e, [0x1d8a] = 0x9c6d, [0x1d8b] = 0x9e0b,
  [0x1d8c] = 0x9e0d, [0x1d8d] = 0x9e10, [0x1d8e] = 0x9e0f, [0x1d8f] = 0x9e12,
  [0x1d90] = 0x9e11, [0x1d91] = 0x9ea1, [0x1d92] = 0x9ef5, [0x1d93] = 0x9f09,
  [0x1d94] = 0x9f47, [0x1d95] = 0x9f78, [0x1d96] = 0x9f7b, [0x1d97] = 0x9f7a,
  [0x1d98] = 0x9f79, [0x1d99] = 0x571e, [0x1d9a] = 0x7066, [0x1d9b] = 0x7c6f,
  [0x1d9c] = 0x883c, [0x1d9d] = 0x8db2, [0x1d9e] = 0x8ea6, [0x1d9f] = 0x91c3,
  [0x1da0] = 0x9474, [0x1da1] = 0x9478, [0x1da2] = 0x9476, [0x1da3] = 0x9475,
  [0x1da4] = 0x9a60, [0x1da5] = 0x9b2e, [0x1da6] = 0x9c74, [0x1da7] = 0x9c73,
  [0x1da8] = 0x9c71, [0x1da9] = 0x9c75, [0x1daa] = 0x9e14, [0x1dab] = 0x9e13,
  [0x1dac] = 0x9ef6, [0x1dad] = 0x9f0a, [0x1dae] = 0x9fa4, [0x1daf] = 0x7068,
  [0x1db0] = 0x7065, [0x1db1] = 0x7cf7, [0x1db2] = 0x866a, [0x1db3] = 0x883e,
  [0x1db4] = 0x883d, [0x1db5] = 0x883f, [0x1db6] = 0x8b9e, [0x1db7] = 0x8c9c,
  [0x1db8] = 0x8ea9, [0x1db9] = 0x8ec9, [0x1dba] = 0x974b, [0x1dbb] = 0x9873,
  [0x1dbc] = 0x9874, [0x1dbd] = 0x98cc, [0x1dbe] = 0x9961, [0x1dbf] = 0x99ab,
  [0x1dc0] = 0x9a64, [0x1dc1] = 0x9a66, [0x1dc2] = 0x9a67, [0x1dc3] = 0x9b24,
  [0x1dc4] = 0x9e15, [0x1dc5] = 0x9e17, [0x1dc6] = 0x9f48, [0x1dc7] = 0x6207,
  [0x1dc8] = 0x6b1e, [0x1dc9] = 0x7227, [0x1dca] = 0x864c, [0x1dcb] = 0x8ea8,
  [0x1dcc] = 0x9482, [0x1dcd] = 0x9480, [0x1dce] = 0x9481, [0x1dcf] = 0x9a69,
  [0x1dd0] = 0x9a68, [0x1dd1] = 0x9e19, [0x1dd2] = 0x864b, [0x1dd3] = 0x8b9f,
  [0x1dd4] = 0x9483, [0x1dd5] = 0x9c79, [0x1dd6] = 0x9eb7, [0x1dd7] = 0x7675,
  [0x1dd8] = 0x9a6b, [0x1dd9] = 0x9c7a, [0x1dda] = 0x9e1d, [0x1ddb] = 0x7069,
  [0x1ddc] = 0x706a, [0x1ddd] = 0x7229, [0x1dde] = 0x9ea4, [0x1ddf] = 0x9f7e,
  [0x1de0] = 0x9f49, [0x1de1] = 0x9f98,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x3' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x32121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l3_to_ucs4_tab[] =
{
  [0x0000] = 0x4e28, [0x0001] = 0x4e36, [0x0002] = 0x4e3f, [0x0003] = 0x4e85,
  [0x0004] = 0x4e05, [0x0005] = 0x4e04, [0x0006] = 0x5182, [0x0007] = 0x5196,
  [0x0008] = 0x5338, [0x0009] = 0x5369, [0x000a] = 0x53b6, [0x000b] = 0x4e2a,
  [0x000c] = 0x4e87, [0x000d] = 0x4e49, [0x000e] = 0x51e2, [0x000f] = 0x4e46,
  [0x0010] = 0x4e8f, [0x0011] = 0x4ebc, [0x0012] = 0x4ebe, [0x0013] = 0x5166,
  [0x0014] = 0x51e3, [0x0015] = 0x5204, [0x0016] = 0x529c, [0x0017] = 0x5344,
  [0x0018] = 0x5902, [0x0019] = 0x590a, [0x001a] = 0x5b80, [0x001b] = 0x5ddb,
  [0x001c] = 0x5e7a, [0x001d] = 0x5e7f, [0x001e] = 0x5ef4, [0x001f] = 0x5f50,
  [0x0020] = 0x5f51, [0x0021] = 0x5f61, [0x0022] = 0x961d, [0x0023] = 0x2000b,
  [0x0024] = 0x4e63, [0x0025] = 0x4e62, [0x0026] = 0x4ea3, [0x0027] = 0x5185,
  [0x0028] = 0x4ec5, [0x0029] = 0x4ecf, [0x002a] = 0x4ece, [0x002b] = 0x4ecc,
  [0x002c] = 0x5184, [0x002d] = 0x5186, [0x002e] = 0x2f817, [0x002f] = 0x34c5,
  [0x0030] = 0x51e4, [0x0031] = 0x5205, [0x0032] = 0x529e, [0x0033] = 0x529d,
  [0x0034] = 0x52fd, [0x0035] = 0x5300, [0x0036] = 0x533a, [0x0037] = 0x3539,
  [0x0038] = 0x5346, [0x0039] = 0x535d, [0x003a] = 0x5386, [0x003b] = 0x53b7,
  [0x003c] = 0x3555, [0x003d] = 0x53cc, [0x003e] = 0x355b, [0x003f] = 0x53ce,
  [0x0040] = 0x5721, [0x0041] = 0x37a2, [0x0042] = 0x5e00, [0x0043] = 0x5f0c,
  [0x0044] = 0x6237, [0x0045] = 0x6238, [0x0046] = 0x6534, [0x0047] = 0x6535,
  [0x0048] = 0x65e0, [0x0049] = 0x3e26, [0x004a] = 0x738d, [0x004b] = 0x4e97,
  [0x004c] = 0x4ee0, [0x004d] = 0x3432, [0x004e] = 0x201a9, [0x004f] = 0x4ee7,
  [0x0050] = 0x3433, [0x0051] = 0x4ee6, [0x0052] = 0x3434, [0x0053] = 0x36a2,
  [0x0054] = 0x3431, [0x0055] = 0x34b0, [0x0056] = 0x56d8, [0x0057] = 0x518b,
  [0x0058] = 0x518c, [0x0059] = 0x5199, [0x005a] = 0x51e5, [0x005b] = 0x2f850,
  [0x005c] = 0x520b, [0x005d] = 0x34dc, [0x005e] = 0x361e, [0x005f] = 0x5304,
  [0x0060] = 0x5303, [0x0061] = 0x5307, [0x0062] = 0x2f82a, [0x0063] = 0x531e,
  [0x0064] = 0x535f, [0x0065] = 0x536d, [0x0066] = 0x5389, [0x0067] = 0x53ba,
  [0x0068] = 0x53d0, [0x0069] = 0x3565, [0x006a] = 0x53f6, [0x006b] = 0x53f7,
  [0x006c] = 0x53f9, [0x006d] = 0x3564, [0x006e] = 0x53f4, [0x006f] = 0x361d,
  [0x0070] = 0x3626, [0x0071] = 0x5724, [0x0072] = 0x5904, [0x0073] = 0x5918,
  [0x0074] = 0x5932, [0x0075] = 0x5930, [0x0076] = 0x5934, [0x0077] = 0x368e,
  [0x0078] = 0x5975, [0x0079] = 0x374a, [0x007a] = 0x5b82, [0x007b] = 0x5bf9,
  [0x007c] = 0x5c14, [0x007d] = 0x378b, [0x007e] = 0x37a6, [0x007f] = 0x37a4,
  [0x0080] = 0x37a5, [0x0081] = 0x37a7, [0x0082] = 0x382f, [0x0083] = 0x3832,
  [0x0084] = 0x5e81, [0x0085] = 0x5e83, [0x0086] = 0x5f0d, [0x0087] = 0x5f52,
  [0x0088] = 0x38d4, [0x0089] = 0x5fca, [0x008a] = 0x5fc7, [0x008b] = 0x6239,
  [0x008c] = 0x39c5, [0x008d] = 0x624f, [0x008e] = 0x65e7, [0x008f] = 0x672f,
  [0x0090] = 0x6b7a, [0x0091] = 0x6c39, [0x0092] = 0x3cba, [0x0093] = 0x3cb9,
  [0x0094] = 0x6c37, [0x0095] = 0x6c44, [0x0096] = 0x6c45, [0x0097] = 0x738c,
  [0x0098] = 0x7592, [0x0099] = 0x7676, [0x009a] = 0x9093, [0x009b] = 0x9092,
  [0x009c] = 0x48b3, [0x009d] = 0x49ba, [0x009e] = 0x4e21, [0x009f] = 0x4e20,
  [0x00a0] = 0x4e22, [0x00a1] = 0x4e68, [0x00a2] = 0x4e89, [0x00a3] = 0x4e98,
  [0x00a4] = 0x4ef9, [0x00a5] = 0x4eef, [0x00a6] = 0x343b, [0x00a7] = 0x343c,
  [0x00a8] = 0x4ef8, [0x00a9] = 0x4f06, [0x00aa] = 0x4f03, [0x00ab] = 0x4efc,
  [0x00ac] = 0x4eee, [0x00ad] = 0x4f16, [0x00ae] = 0x3439, [0x00af] = 0x4f28,
  [0x00b0] = 0x4f1c, [0x00b1] = 0x4f07, [0x00b2] = 0x4f1a, [0x00b3] = 0x4efa,
  [0x00b4] = 0x4f17, [0x00b5] = 0x514a, [0x00b6] = 0x34b2, [0x00b7] = 0x5172,
  [0x00b8] = 0x2f815, [0x00b9] = 0x51b4, [0x00ba] = 0x51b3, [0x00bb] = 0x51b2,
  [0x00bc] = 0x34c7, [0x00bd] = 0x51e8, [0x00be] = 0x342b, [0x00bf] = 0x5214,
  [0x00c0] = 0x520f, [0x00c1] = 0x5215, [0x00c2] = 0x5218, [0x00c3] = 0x52a8,
  [0x00c4] = 0x2f82c, [0x00c5] = 0x534b, [0x00c6] = 0x534f, [0x00c7] = 0x353b,
  [0x00c8] = 0x5350, [0x00c9] = 0x3544, [0x00ca] = 0x538b, [0x00cb] = 0x3542,
  [0x00cc] = 0x53be, [0x00cd] = 0x355c, [0x00ce] = 0x53d2, [0x00cf] = 0x5416,
  [0x00d0] = 0x53ff, [0x00d1] = 0x3567, [0x00d2] = 0x5400, [0x00d3] = 0x3566,
  [0x00d4] = 0x5405, [0x00d5] = 0x5413, [0x00d6] = 0x5415, [0x00d7] = 0x2f83b,
  [0x00d8] = 0x361f, [0x00d9] = 0x56e3, [0x00da] = 0x5735, [0x00db] = 0x5736,
  [0x00dc] = 0x5731, [0x00dd] = 0x5732, [0x00de] = 0x58ee, [0x00df] = 0x5905,
  [0x00e0] = 0x4e54, [0x00e1] = 0x368f, [0x00e2] = 0x5936, [0x00e3] = 0x3690,
  [0x00e4] = 0x36a8, [0x00e5] = 0x36a4, [0x00e6] = 0x597a, [0x00e7] = 0x36a3,
  [0x00e8] = 0x5986, [0x00e9] = 0x373d, [0x00ea] = 0x374c, [0x00eb] = 0x5b86,
  [0x00ec] = 0x5f53, [0x00ed] = 0x5c18, [0x00ee] = 0x378c, [0x00ef] = 0x5c3d,
  [0x00f0] = 0x5c78, [0x00f1] = 0x37a8, [0x00f2] = 0x37ad, [0x00f3] = 0x37af,
  [0x00f4] = 0x21d46, [0x00f5] = 0x5c80, [0x00f6] = 0x3829, [0x00f7] = 0x5e08,
  [0x00f8] = 0x3836, [0x00f9] = 0x3871, [0x00fa] = 0x3870, [0x00fb] = 0x386f,
  [0x00fc] = 0x5ef5, [0x00fd] = 0x5f0e, [0x00fe] = 0x38a9, [0x00ff] = 0x38aa,
  [0x0100] = 0x38fb, [0x0101] = 0x5fd3, [0x0102] = 0x5fda, [0x0103] = 0x38fc,
  [0x0104] = 0x5fdb, [0x0105] = 0x39ae, [0x0106] = 0x620f, [0x0107] = 0x625d,
  [0x0108] = 0x625f, [0x0109] = 0x6267, [0x010a] = 0x6257, [0x010b] = 0x9f50,
  [0x010c] = 0x3ac3, [0x010d] = 0x65eb, [0x010e] = 0x65ea, [0x010f] = 0x3b30,
  [0x0110] = 0x6737, [0x0111] = 0x3b41, [0x0112] = 0x6732, [0x0113] = 0x6736,
  [0x0114] = 0x6b22, [0x0115] = 0x6bce, [0x0116] = 0x3c8c, [0x0117] = 0x6c58,
  [0x0118] = 0x6c51, [0x0119] = 0x6c77, [0x011a] = 0x6c3c, [0x011b] = 0x3cbb,
  [0x011c] = 0x6c5a, [0x011d] = 0x23c86, [0x011e] = 0x6c53, [0x011f] = 0x706f,
  [0x0120] = 0x7072, [0x0121] = 0x706e, [0x0122] = 0x2f835, [0x0123] = 0x3da1,
  [0x0124] = 0x7073, [0x0125] = 0x72b1, [0x0126] = 0x72b2, [0x0127] = 0x3ea8,
  [0x0128] = 0x738f, [0x0129] = 0x3eaa, [0x012a] = 0x3eab, [0x012b] = 0x4096,
  [0x012c] = 0x793c, [0x012d] = 0x41c2, [0x012e] = 0x808d, [0x012f] = 0x808e,
  [0x0130] = 0x4493, [0x0131] = 0x827b, [0x0132] = 0x4494, [0x0133] = 0x8d71,
  [0x0134] = 0x8fb9, [0x0135] = 0x9096, [0x0136] = 0x909a, [0x0137] = 0x49bb,
  [0x0138] = 0x4e24, [0x0139] = 0x4e71, [0x013a] = 0x2f81b, [0x013b] = 0x4e9c,
  [0x013c] = 0x4f45, [0x013d] = 0x4f4a, [0x013e] = 0x4f39, [0x013f] = 0x4f37,
  [0x0140] = 0x3443, [0x0141] = 0x4f32, [0x0142] = 0x4f42, [0x0143] = 0x3442,
  [0x0144] = 0x4f44, [0x0145] = 0x4f4b, [0x0146] = 0x3444, [0x0147] = 0x4f40,
  [0x0148] = 0x4f35, [0x0149] = 0x4f31, [0x014a] = 0x5151, [0x014b] = 0x2f80e,
  [0x014c] = 0x5150, [0x014d] = 0x514e, [0x014e] = 0x34b3, [0x014f] = 0x34b7,
  [0x0150] = 0x519d, [0x0151] = 0x34c8, [0x0152] = 0x51b5, [0x0153] = 0x51b8,
  [0x0154] = 0x51ec, [0x0155] = 0x5223, [0x0156] = 0x5227, [0x0157] = 0x5226,
  [0x0158] = 0x521f, [0x0159] = 0x522b, [0x015a] = 0x5220, [0x015b] = 0x52b4,
  [0x015c] = 0x52b3, [0x015d] = 0x3518, [0x015e] = 0x5325, [0x015f] = 0x533b,
  [0x0160] = 0x5374, [0x0161] = 0x3547, [0x0162] = 0x3546, [0x0163] = 0x3545,
  [0x0164] = 0x356b, [0x0165] = 0x3569, [0x0166] = 0x544d, [0x0167] = 0x3572,
  [0x0168] = 0x3571, [0x0169] = 0x543a, [0x016a] = 0x356c, [0x016b] = 0x356f,
  [0x016c] = 0x5444, [0x016d] = 0x544c, [0x016e] = 0x5423, [0x016f] = 0x541a,
  [0x0170] = 0x5432, [0x0171] = 0x544b, [0x0172] = 0x5421, [0x0173] = 0x3573,
  [0x0174] = 0x5434, [0x0175] = 0x5449, [0x0176] = 0x5450, [0x0177] = 0x5422,
  [0x0178] = 0x543f, [0x0179] = 0x5451, [0x017a] = 0x545a, [0x017b] = 0x542f,
  [0x017c] = 0x3576, [0x017d] = 0x56e9, [0x017e] = 0x56f2, [0x017f] = 0x56f3,
  [0x0180] = 0x56ef, [0x0181] = 0x56ed, [0x0182] = 0x56ec, [0x0183] = 0x56e6,
  [0x0184] = 0x5748, [0x0185] = 0x3627, [0x0186] = 0x5744, [0x0187] = 0x573f,
  [0x0188] = 0x573c, [0x0189] = 0x5753, [0x018a] = 0x5756, [0x018b] = 0x3630,
  [0x018c] = 0x575f, [0x018d] = 0x5743, [0x018e] = 0x5758, [0x018f] = 0x5757,
  [0x0190] = 0x3629, [0x0191] = 0x362a, [0x0192] = 0x362f, [0x0193] = 0x5746,
  [0x0194] = 0x362c, [0x0195] = 0x573d, [0x0196] = 0x362d, [0x0197] = 0x5742,
  [0x0198] = 0x5754, [0x0199] = 0x5755, [0x019a] = 0x58f1, [0x019b] = 0x58f2,
  [0x019c] = 0x58f0, [0x019d] = 0x590b, [0x019e] = 0x9ea6, [0x019f] = 0x56f1,
  [0x01a0] = 0x593d, [0x01a1] = 0x3693, [0x01a2] = 0x5994, [0x01a3] = 0x598c,
  [0x01a4] = 0x36ad, [0x01a5] = 0x599c, [0x01a6] = 0x36ac, [0x01a7] = 0x36ab,
  [0x01a8] = 0x599f, [0x01a9] = 0x36a9, [0x01aa] = 0x599b, [0x01ab] = 0x36ae,
  [0x01ac] = 0x5989, [0x01ad] = 0x599a, [0x01ae] = 0x36aa, [0x01af] = 0x6588,
  [0x01b0] = 0x374e, [0x01b1] = 0x5b8d, [0x01b2] = 0x3750, [0x01b3] = 0x5bfe,
  [0x01b4] = 0x5bff, [0x01b5] = 0x5bfd, [0x01b6] = 0x5c2b, [0x01b7] = 0x37b2,
  [0x01b8] = 0x5c84, [0x01b9] = 0x5c8e, [0x01ba] = 0x5c9c, [0x01bb] = 0x37b5,
  [0x01bc] = 0x37b6, [0x01bd] = 0x5c85, [0x01be] = 0x5df5, [0x01bf] = 0x5e09,
  [0x01c0] = 0x3839, [0x01c1] = 0x383b, [0x01c2] = 0x5e0b, [0x01c3] = 0x3872,
  [0x01c4] = 0x5e92, [0x01c5] = 0x5e90, [0x01c6] = 0x5f03, [0x01c7] = 0x38ac,
  [0x01c8] = 0x5f1e, [0x01c9] = 0x5f63, [0x01ca] = 0x3908, [0x01cb] = 0x5fe7,
  [0x01cc] = 0x5ffe, [0x01cd] = 0x5fe6, [0x01ce] = 0x5fdc, [0x01cf] = 0x5fce,
  [0x01d0] = 0x3903, [0x01d1] = 0x5ffc, [0x01d2] = 0x5fdf, [0x01d3] = 0x5fec,
  [0x01d4] = 0x5ff6, [0x01d5] = 0x225d7, [0x01d6] = 0x5ff2, [0x01d7] = 0x5ff0,
  [0x01d8] = 0x5ff9, [0x01d9] = 0x390b, [0x01da] = 0x6213, [0x01db] = 0x39af,
  [0x01dc] = 0x2f8b2, [0x01dd] = 0x623b, [0x01de] = 0x623c, [0x01df] = 0x6282,
  [0x01e0] = 0x39ce, [0x01e1] = 0x39cb, [0x01e2] = 0x39cc, [0x01e3] = 0x6278,
  [0x01e4] = 0x628b, [0x01e5] = 0x39cd, [0x01e6] = 0x629e, [0x01e7] = 0x62a5,
  [0x01e8] = 0x629b, [0x01e9] = 0x629c, [0x01ea] = 0x6299, [0x01eb] = 0x628d,
  [0x01ec] = 0x6285, [0x01ed] = 0x629d, [0x01ee] = 0x6275, [0x01ef] = 0x3a80,
  [0x01f0] = 0x3aaf, [0x01f1] = 0x3ad3, [0x01f2] = 0x65f6, [0x01f3] = 0x3ad5,
  [0x01f4] = 0x3ad4, [0x01f5] = 0x3ad7, [0x01f6] = 0x66f5, [0x01f7] = 0x675b,
  [0x01f8] = 0x3b42, [0x01f9] = 0x6754, [0x01fa] = 0x6752, [0x01fb] = 0x3b44,
  [0x01fc] = 0x6758, [0x01fd] = 0x6744, [0x01fe] = 0x674a, [0x01ff] = 0x6761,
  [0x0200] = 0x3cc6, [0x0201] = 0x6c7f, [0x0202] = 0x6c91, [0x0203] = 0x6c9e,
  [0x0204] = 0x3cc0, [0x0205] = 0x6c6e, [0x0206] = 0x6c7c, [0x0207] = 0x6c9f,
  [0x0208] = 0x6c75, [0x0209] = 0x3cbe, [0x020a] = 0x6c56, [0x020b] = 0x6ca2,
  [0x020c] = 0x6c79, [0x020d] = 0x3cca, [0x020e] = 0x6ca1, [0x020f] = 0x3cc4,
  [0x0210] = 0x6caa, [0x0211] = 0x6ca0, [0x0212] = 0x3cc2, [0x0213] = 0x7079,
  [0x0214] = 0x7077, [0x0215] = 0x707e, [0x0216] = 0x3da4, [0x0217] = 0x7075,
  [0x0218] = 0x707b, [0x0219] = 0x7264, [0x021a] = 0x3e29, [0x021b] = 0x72bb,
  [0x021c] = 0x72bc, [0x021d] = 0x72c7, [0x021e] = 0x72b9, [0x021f] = 0x72be,
  [0x0220] = 0x72b6, [0x0221] = 0x3e60, [0x0222] = 0x3e5e, [0x0223] = 0x7398,
  [0x0224] = 0x3ead, [0x0225] = 0x3eae, [0x0226] = 0x3eac, [0x0227] = 0x3f57,
  [0x0228] = 0x7593, [0x0229] = 0x7680, [0x022a] = 0x3fdd, [0x022b] = 0x7683,
  [0x022c] = 0x76c0, [0x022d] = 0x76c1, [0x022e] = 0x400e, [0x022f] = 0x4097,
  [0x0230] = 0x77f4, [0x0231] = 0x77f5, [0x0232] = 0x4127, [0x0233] = 0x7acc,
  [0x0234] = 0x7acd, [0x0235] = 0x7cfa, [0x0236] = 0x809f, [0x0237] = 0x8091,
  [0x0238] = 0x8097, [0x0239] = 0x8094, [0x023a] = 0x4495, [0x023b] = 0x8286,
  [0x023c] = 0x828c, [0x023d] = 0x2f98f, [0x023e] = 0x8295, [0x023f] = 0x4498,
  [0x0240] = 0x866c, [0x0241] = 0x459d, [0x0242] = 0x8fb5, [0x0243] = 0x8fbe,
  [0x0244] = 0x8fc7, [0x0245] = 0x488a, [0x0246] = 0x8fc1, [0x0247] = 0x90a9,
  [0x0248] = 0x90a4, [0x0249] = 0x48b5, [0x024a] = 0x48b6, [0x024b] = 0x48b7,
  [0x024c] = 0x90a8, [0x024d] = 0x9627, [0x024e] = 0x9626, [0x024f] = 0x962b,
  [0x0250] = 0x9633, [0x0251] = 0x9634, [0x0252] = 0x9629, [0x0253] = 0x4e3d,
  [0x0254] = 0x3428, [0x0255] = 0x4e9d, [0x0256] = 0x4f93, [0x0257] = 0x4f8a,
  [0x0258] = 0x344d, [0x0259] = 0x3449, [0x025a] = 0x4f6d, [0x025b] = 0x4f8e,
  [0x025c] = 0x4fa0, [0x025d] = 0x4fa2, [0x025e] = 0x4fa1, [0x025f] = 0x4f9f,
  [0x0260] = 0x4fa3, [0x0261] = 0x20209, [0x0262] = 0x4f72, [0x0263] = 0x3451,
  [0x0264] = 0x4f8c, [0x0265] = 0x5156, [0x0266] = 0x2f80f, [0x0267] = 0x2f811,
  [0x0268] = 0x5190, [0x0269] = 0x34cb, [0x026a] = 0x34ca, [0x026b] = 0x34cc,
  [0x026c] = 0x51ed, [0x026d] = 0x51fe, [0x026e] = 0x522f, [0x026f] = 0x206ec,
  [0x0270] = 0x523c, [0x0271] = 0x5234, [0x0272] = 0x5239, [0x0273] = 0x52b9,
  [0x0274] = 0x52b5, [0x0275] = 0x52bf, [0x0276] = 0x5355, [0x0277] = 0x353d,
  [0x0278] = 0x5376, [0x0279] = 0x537a, [0x027a] = 0x5393, [0x027b] = 0x3548,
  [0x027c] = 0x53c1, [0x027d] = 0x53c2, [0x027e] = 0x53d5, [0x027f] = 0x5485,
  [0x0280] = 0x3578, [0x0281] = 0x545f, [0x0282] = 0x5493, [0x0283] = 0x5489,
  [0x0284] = 0x5479, [0x0285] = 0x9efe, [0x0286] = 0x548f, [0x0287] = 0x5469,
  [0x0288] = 0x546d, [0x0289] = 0x357a, [0x028a] = 0x5494, [0x028b] = 0x546a,
  [0x028c] = 0x548a, [0x028d] = 0x3577, [0x028e] = 0x56fd, [0x028f] = 0x56fb,
  [0x0290] = 0x56f8, [0x0291] = 0x3621, [0x0292] = 0x56fc, [0x0293] = 0x56f6,
  [0x0294] = 0x5765, [0x0295] = 0x5781, [0x0296] = 0x5763, [0x0297] = 0x5767,
  [0x0298] = 0x3631, [0x0299] = 0x576e, [0x029a] = 0x5778, [0x029b] = 0x577f,
  [0x029c] = 0x3633, [0x029d] = 0x3634, [0x029e] = 0x58f3, [0x029f] = 0x594b,
  [0x02a0] = 0x594c, [0x02a1] = 0x36c1, [0x02a2] = 0x36b0, [0x02a3] = 0x36b4,
  [0x02a4] = 0x59ad, [0x02a5] = 0x36b8, [0x02a6] = 0x59c4, [0x02a7] = 0x36bc,
  [0x02a8] = 0x59c2, [0x02a9] = 0x59b0, [0x02aa] = 0x36bf, [0x02ab] = 0x36b5,
  [0x02ac] = 0x36b1, [0x02ad] = 0x36bd, [0x02ae] = 0x59bf, [0x02af] = 0x36bb,
  [0x02b0] = 0x59c9, [0x02b1] = 0x59b8, [0x02b2] = 0x59ac, [0x02b3] = 0x36b3,
  [0x02b4] = 0x36b6, [0x02b5] = 0x36ba, [0x02b6] = 0x59b7, [0x02b7] = 0x59d7,
  [0x02b8] = 0x36b7, [0x02b9] = 0x5b60, [0x02ba] = 0x3740, [0x02bb] = 0x5b96,
  [0x02bc] = 0x5b9e, [0x02bd] = 0x5b94, [0x02be] = 0x5b9f, [0x02bf] = 0x5b9d,
  [0x02c0] = 0x3752, [0x02c1] = 0x5c00, [0x02c2] = 0x5c19, [0x02c3] = 0x3790,
  [0x02c4] = 0x3791, [0x02c5] = 0x5c49, [0x02c6] = 0x5c4a, [0x02c7] = 0x37be,
  [0x02c8] = 0x5cbb, [0x02c9] = 0x5cc1, [0x02ca] = 0x37c0, [0x02cb] = 0x37c1,
  [0x02cc] = 0x37b9, [0x02cd] = 0x5cb9, [0x02ce] = 0x5c9e, [0x02cf] = 0x5cb4,
  [0x02d0] = 0x5cba, [0x02d1] = 0x5df6, [0x02d2] = 0x5e13, [0x02d3] = 0x5e12,
  [0x02d4] = 0x5e77, [0x02d5] = 0x3879, [0x02d6] = 0x5e98, [0x02d7] = 0x387b,
  [0x02d8] = 0x5e99, [0x02d9] = 0x5e9d, [0x02da] = 0x5ef8, [0x02db] = 0x38a0,
  [0x02dc] = 0x5ef9, [0x02dd] = 0x3429, [0x02de] = 0x5f06, [0x02df] = 0x5f21,
  [0x02e0] = 0x38ae, [0x02e1] = 0x5f25, [0x02e2] = 0x5f55, [0x02e3] = 0x38cd,
  [0x02e4] = 0x38cb, [0x02e5] = 0x38d9, [0x02e6] = 0x5f84, [0x02e7] = 0x5f83,
  [0x02e8] = 0x6030, [0x02e9] = 0x6007, [0x02ea] = 0x390c, [0x02eb] = 0x6036,
  [0x02ec] = 0x3901, [0x02ed] = 0x3905, [0x02ee] = 0x3902, [0x02ef] = 0x5fe9,
  [0x02f0] = 0x603d, [0x02f1] = 0x6008, [0x02f2] = 0x3913, [0x02f3] = 0x3911,
  [0x02f4] = 0x62ba, [0x02f5] = 0x62b2, [0x02f6] = 0x39e4, [0x02f7] = 0x62b7,
  [0x02f8] = 0x62e4, [0x02f9] = 0x62a7, [0x02fa] = 0x39da, [0x02fb] = 0x39d5,
  [0x02fc] = 0x39d3, [0x02fd] = 0x62d5, [0x02fe] = 0x62e1, [0x02ff] = 0x62dd,
  [0x0300] = 0x62a6, [0x0301] = 0x62c1, [0x0302] = 0x62c5, [0x0303] = 0x62c0,
  [0x0304] = 0x62df, [0x0305] = 0x62e0, [0x0306] = 0x62de, [0x0307] = 0x39d6,
  [0x0308] = 0x6589, [0x0309] = 0x3ab4, [0x030a] = 0x65a6, [0x030b] = 0x65ba,
  [0x030c] = 0x3ad9, [0x030d] = 0x65ff, [0x030e] = 0x3ad8, [0x030f] = 0x6617,
  [0x0310] = 0x6618, [0x0311] = 0x6601, [0x0312] = 0x65fe, [0x0313] = 0x3b33,
  [0x0314] = 0x670c, [0x0315] = 0x3b48, [0x0316] = 0x676b, [0x0317] = 0x6796,
  [0x0318] = 0x6782, [0x0319] = 0x678a, [0x031a] = 0x3b47, [0x031b] = 0x67a3,
  [0x031c] = 0x3b4b, [0x031d] = 0x67a2, [0x031e] = 0x678f, [0x031f] = 0x3b4a,
  [0x0320] = 0x67f9, [0x0321] = 0x6780, [0x0322] = 0x6b26, [0x0323] = 0x6b27,
  [0x0324] = 0x6b68, [0x0325] = 0x6b69, [0x0326] = 0x3c5a, [0x0327] = 0x6b81,
  [0x0328] = 0x6bb4, [0x0329] = 0x6bd1, [0x032a] = 0x3c8e, [0x032b] = 0x3cb4,
  [0x032c] = 0x6c1c, [0x032d] = 0x3ccd, [0x032e] = 0x3ccc, [0x032f] = 0x3ccf,
  [0x0330] = 0x3ccb, [0x0331] = 0x3cce, [0x0332] = 0x6c97, [0x0333] = 0x6c6c,
  [0x0334] = 0x6cdf, [0x0335] = 0x3cd2, [0x0336] = 0x6cea, [0x0337] = 0x3cd1,
  [0x0338] = 0x6ce4, [0x0339] = 0x6cd8, [0x033a] = 0x6cb2, [0x033b] = 0x6cce,
  [0x033c] = 0x6cc8, [0x033d] = 0x3da6, [0x033e] = 0x708b, [0x033f] = 0x7088,
  [0x0340] = 0x7090, [0x0341] = 0x708f, [0x0342] = 0x3daa, [0x0343] = 0x7087,
  [0x0344] = 0x7089, [0x0345] = 0x708d, [0x0346] = 0x7081, [0x0347] = 0x3da8,
  [0x0348] = 0x708c, [0x0349] = 0x3e13, [0x034a] = 0x3e1a, [0x034b] = 0x7240,
  [0x034c] = 0x3e1d, [0x034d] = 0x3e1e, [0x034e] = 0x7265, [0x034f] = 0x7266,
  [0x0350] = 0x7268, [0x0351] = 0x3e65, [0x0352] = 0x3e66, [0x0353] = 0x72cd,
  [0x0354] = 0x72d3, [0x0355] = 0x72db, [0x0356] = 0x3e64, [0x0357] = 0x72cf,
  [0x0358] = 0x73a7, [0x0359] = 0x73a3, [0x035a] = 0x739e, [0x035b] = 0x3eb0,
  [0x035c] = 0x73af, [0x035d] = 0x3eb3, [0x035e] = 0x3eb5, [0x035f] = 0x73aa,
  [0x0360] = 0x739c, [0x0361] = 0x3f19, [0x0362] = 0x7542, [0x0363] = 0x7544,
  [0x0364] = 0x753b, [0x0365] = 0x7541, [0x0366] = 0x2233f, [0x0367] = 0x759b,
  [0x0368] = 0x759e, [0x0369] = 0x3f75, [0x036a] = 0x79c4, [0x036b] = 0x79c3,
  [0x036c] = 0x79c6, [0x036d] = 0x412b, [0x036e] = 0x412c, [0x036f] = 0x79c7,
  [0x0370] = 0x412d, [0x0371] = 0x79ca, [0x0372] = 0x2592e, [0x0373] = 0x41c3,
  [0x0374] = 0x7acf, [0x0375] = 0x7c76, [0x0376] = 0x7c74, [0x0377] = 0x7cff,
  [0x0378] = 0x7cfc, [0x0379] = 0x34ba, [0x037a] = 0x4350, [0x037b] = 0x7f59,
  [0x037c] = 0x80a8, [0x037d] = 0x43d3, [0x037e] = 0x43d0, [0x037f] = 0x80b0,
  [0x0380] = 0x43dc, [0x0381] = 0x80b3, [0x0382] = 0x43d2, [0x0383] = 0x80a4,
  [0x0384] = 0x80b6, [0x0385] = 0x80a7, [0x0386] = 0x80ac, [0x0387] = 0x43db,
  [0x0388] = 0x80a6, [0x0389] = 0x5367, [0x038a] = 0x820e, [0x038b] = 0x82c4,
  [0x038c] = 0x833e, [0x038d] = 0x829c, [0x038e] = 0x44a5, [0x038f] = 0x449f,
  [0x0390] = 0x449a, [0x0391] = 0x449c, [0x0392] = 0x44a2, [0x0393] = 0x82aa,
  [0x0394] = 0x449b, [0x0395] = 0x82c9, [0x0396] = 0x44a3, [0x0397] = 0x449d,
  [0x0398] = 0x82a6, [0x0399] = 0x82b2, [0x039a] = 0x4588, [0x039b] = 0x461a,
  [0x039c] = 0x488d, [0x039d] = 0x8fcc, [0x039e] = 0x8fd9, [0x039f] = 0x8fca,
  [0x03a0] = 0x8fd8, [0x03a1] = 0x8fcf, [0x03a2] = 0x90b7, [0x03a3] = 0x48b8,
  [0x03a4] = 0x90ad, [0x03a5] = 0x90b9, [0x03a6] = 0x9637, [0x03a7] = 0x49c3,
  [0x03a8] = 0x9641, [0x03a9] = 0x963e, [0x03aa] = 0x96b6, [0x03ab] = 0x9751,
  [0x03ac] = 0x9763, [0x03ad] = 0x4e57, [0x03ae] = 0x4e79, [0x03af] = 0x4eb2,
  [0x03b0] = 0x4eb0, [0x03b1] = 0x4eaf, [0x03b2] = 0x4eb1, [0x03b3] = 0x4fd2,
  [0x03b4] = 0x4fd5, [0x03b5] = 0x345d, [0x03b6] = 0x4fbe, [0x03b7] = 0x4fb8,
  [0x03b8] = 0x4fb0, [0x03b9] = 0x4fb1, [0x03ba] = 0x4fc8, [0x03bb] = 0x345a,
  [0x03bc] = 0x3457, [0x03bd] = 0x4fc6, [0x03be] = 0x4fcc, [0x03bf] = 0x4fe5,
  [0x03c0] = 0x4fe3, [0x03c1] = 0x4fb4, [0x03c2] = 0x516a, [0x03c3] = 0x34b8,
  [0x03c4] = 0x519f, [0x03c5] = 0x34c2, [0x03c6] = 0x51c1, [0x03c7] = 0x34cf,
  [0x03c8] = 0x51c2, [0x03c9] = 0x51c3, [0x03ca] = 0x5245, [0x03cb] = 0x5248,
  [0x03cc] = 0x34e7, [0x03cd] = 0x34e9, [0x03ce] = 0x524f, [0x03cf] = 0x4452,
  [0x03d0] = 0x34e8, [0x03d1] = 0x52c5, [0x03d2] = 0x52ca, [0x03d3] = 0x52c4,
  [0x03d4] = 0x5327, [0x03d5] = 0x5358, [0x03d6] = 0x537d, [0x03d7] = 0x354a,
  [0x03d8] = 0x53dd, [0x03d9] = 0x53dc, [0x03da] = 0x53da, [0x03db] = 0x53d9,
  [0x03dc] = 0x54b9, [0x03dd] = 0x3580, [0x03de] = 0x54d0, [0x03df] = 0x54b4,
  [0x03e0] = 0x54ca, [0x03e1] = 0x3587, [0x03e2] = 0x54a3, [0x03e3] = 0x54da,
  [0x03e4] = 0x54a4, [0x03e5] = 0x3584, [0x03e6] = 0x54b2, [0x03e7] = 0x549e,
  [0x03e8] = 0x549f, [0x03e9] = 0x54b5, [0x03ea] = 0x3582, [0x03eb] = 0x3581,
  [0x03ec] = 0x54cd, [0x03ed] = 0x3583, [0x03ee] = 0x54cc, [0x03ef] = 0x3622,
  [0x03f0] = 0x5700, [0x03f1] = 0x57ac, [0x03f2] = 0x5791, [0x03f3] = 0x578e,
  [0x03f4] = 0x578d, [0x03f5] = 0x5792, [0x03f6] = 0x57a1, [0x03f7] = 0x5790,
  [0x03f8] = 0x57a6, [0x03f9] = 0x57a8, [0x03fa] = 0x363b, [0x03fb] = 0x579c,
  [0x03fc] = 0x5796, [0x03fd] = 0x57a7, [0x03fe] = 0x363a, [0x03ff] = 0x3638,
  [0x0400] = 0x3639, [0x0401] = 0x3636, [0x0402] = 0x58f5, [0x0403] = 0x3685,
  [0x0404] = 0x5909, [0x0405] = 0x5908, [0x0406] = 0x3c54, [0x0407] = 0x5952,
  [0x0408] = 0x369a, [0x0409] = 0x36c4, [0x040a] = 0x59df, [0x040b] = 0x36c5,
  [0x040c] = 0x59eb, [0x040d] = 0x59ef, [0x040e] = 0x59f0, [0x040f] = 0x59d5,
  [0x0410] = 0x5a0d, [0x0411] = 0x5a04, [0x0412] = 0x59f9, [0x0413] = 0x5a02,
  [0x0414] = 0x59f8, [0x0415] = 0x59e2, [0x0416] = 0x59d9, [0x0417] = 0x59e7,
  [0x0418] = 0x5b6a, [0x0419] = 0x3754, [0x041a] = 0x3755, [0x041b] = 0x5bab,
  [0x041c] = 0x3756, [0x041d] = 0x5c1b, [0x041e] = 0x5c2f, [0x041f] = 0x3796,
  [0x0420] = 0x663c, [0x0421] = 0x3795, [0x0422] = 0x3794, [0x0423] = 0x37c4,
  [0x0424] = 0x5cd1, [0x0425] = 0x5cdc, [0x0426] = 0x5ce6, [0x0427] = 0x5ce1,
  [0x0428] = 0x5ccd, [0x0429] = 0x2f87a, [0x042a] = 0x5ce2, [0x042b] = 0x5cdd,
  [0x042c] = 0x5ce5, [0x042d] = 0x5dfb, [0x042e] = 0x5dfa, [0x042f] = 0x5e1e,
  [0x0430] = 0x3844, [0x0431] = 0x5ea1, [0x0432] = 0x387d, [0x0433] = 0x387e,
  [0x0434] = 0x5efc, [0x0435] = 0x5efb, [0x0436] = 0x5f2f, [0x0437] = 0x38b2,
  [0x0438] = 0x38b6, [0x0439] = 0x5f66, [0x043a] = 0x2f899, [0x043b] = 0x38dc,
  [0x043c] = 0x38df, [0x043d] = 0x605c, [0x043e] = 0x3928, [0x043f] = 0x604e,
  [0x0440] = 0x6051, [0x0441] = 0x3919, [0x0442] = 0x3910, [0x0443] = 0x6023,
  [0x0444] = 0x6031, [0x0445] = 0x607c, [0x0446] = 0x6052, [0x0447] = 0x392c,
  [0x0448] = 0x6060, [0x0449] = 0x604a, [0x044a] = 0x6061, [0x044b] = 0x391b,
  [0x044c] = 0x6218, [0x044d] = 0x39c2, [0x044e] = 0x39ef, [0x044f] = 0x39e3,
  [0x0450] = 0x39e5, [0x0451] = 0x39ea, [0x0452] = 0x39e6, [0x0453] = 0x39ee,
  [0x0454] = 0x631f, [0x0455] = 0x6317, [0x0456] = 0x62ea, [0x0457] = 0x6321,
  [0x0458] = 0x6304, [0x0459] = 0x6305, [0x045a] = 0x39e8, [0x045b] = 0x6531,
  [0x045c] = 0x6544, [0x045d] = 0x6540, [0x045e] = 0x3a85, [0x045f] = 0x6542,
  [0x0460] = 0x65be, [0x0461] = 0x3ae0, [0x0462] = 0x6629, [0x0463] = 0x661b,
  [0x0464] = 0x3add, [0x0465] = 0x6623, [0x0466] = 0x662c, [0x0467] = 0x661a,
  [0x0468] = 0x6630, [0x0469] = 0x663b, [0x046a] = 0x661e, [0x046b] = 0x6637,
  [0x046c] = 0x6638, [0x046d] = 0x3ae1, [0x046e] = 0x670e, [0x046f] = 0x3b51,
  [0x0470] = 0x3b55, [0x0471] = 0x67e8, [0x0472] = 0x67d6, [0x0473] = 0x3b52,
  [0x0474] = 0x67c7, [0x0475] = 0x67bc, [0x0476] = 0x6852, [0x0477] = 0x67bf,
  [0x0478] = 0x67d5, [0x0479] = 0x67fe, [0x047a] = 0x8363, [0x047b] = 0x67fb,
  [0x047c] = 0x2f8df, [0x047d] = 0x67b1, [0x047e] = 0x6801, [0x047f] = 0x6805,
  [0x0480] = 0x6800, [0x0481] = 0x67d7, [0x0482] = 0x409e, [0x0483] = 0x6b2a,
  [0x0484] = 0x6b6b, [0x0485] = 0x3c52, [0x0486] = 0x3c5e, [0x0487] = 0x3c60,
  [0x0488] = 0x3c5f, [0x0489] = 0x6be1, [0x048a] = 0x3c92, [0x048b] = 0x3cd6,
  [0x048c] = 0x6d23, [0x048d] = 0x6cff, [0x048e] = 0x6d14, [0x048f] = 0x6d05,
  [0x0490] = 0x6d13, [0x0491] = 0x6d06, [0x0492] = 0x6d21, [0x0493] = 0x3cde,
  [0x0494] = 0x6d15, [0x0495] = 0x6caf, [0x0496] = 0x6cf4, [0x0497] = 0x6d02,
  [0x0498] = 0x6d45, [0x0499] = 0x2f8fe, [0x049a] = 0x6d26, [0x049b] = 0x3cd9,
  [0x049c] = 0x6d44, [0x049d] = 0x3cdd, [0x049e] = 0x6d24, [0x049f] = 0x70a5,
  [0x04a0] = 0x3dac, [0x04a1] = 0x70a3, [0x04a2] = 0x3db0, [0x04a3] = 0x70a2,
  [0x04a4] = 0x70bb, [0x04a5] = 0x70a0, [0x04a6] = 0x70aa, [0x04a7] = 0x3daf,
  [0x04a8] = 0x3dae, [0x04a9] = 0x70a8, [0x04aa] = 0x70b6, [0x04ab] = 0x70b2,
  [0x04ac] = 0x70a7, [0x04ad] = 0x3dad, [0x04ae] = 0x3dab, [0x04af] = 0x70b9,
  [0x04b0] = 0x722e, [0x04b1] = 0x3e16, [0x04b2] = 0x723c, [0x04b3] = 0x3e30,
  [0x04b4] = 0x726d, [0x04b5] = 0x3e33, [0x04b6] = 0x3e31, [0x04b7] = 0x72e7,
  [0x04b8] = 0x72ed, [0x04b9] = 0x3e6e, [0x04ba] = 0x72ec, [0x04bb] = 0x72e5,
  [0x04bc] = 0x72e2, [0x04bd] = 0x3eb1, [0x04be] = 0x73c4, [0x04bf] = 0x73bd,
  [0x04c0] = 0x73cf, [0x04c1] = 0x73c9, [0x04c2] = 0x73c1, [0x04c3] = 0x73d0,
  [0x04c4] = 0x3eb7, [0x04c5] = 0x73ce, [0x04c6] = 0x74ed, [0x04c7] = 0x74eb,
  [0x04c8] = 0x3f1a, [0x04c9] = 0x74ef, [0x04ca] = 0x7549, [0x04cb] = 0x7550,
  [0x04cc] = 0x7546, [0x04cd] = 0x754a, [0x04ce] = 0x3f59, [0x04cf] = 0x754d,
  [0x04d0] = 0x75a6, [0x04d1] = 0x3f7a, [0x04d2] = 0x3f78, [0x04d3] = 0x3f7b,
  [0x04d4] = 0x75a8, [0x04d5] = 0x3fde, [0x04d6] = 0x3fec, [0x04d7] = 0x76c7,
  [0x04d8] = 0x76ff, [0x04d9] = 0x401e, [0x04da] = 0x76fd, [0x04db] = 0x77e6,
  [0x04dc] = 0x780a, [0x04dd] = 0x409b, [0x04de] = 0x7804, [0x04df] = 0x780b,
  [0x04e0] = 0x7807, [0x04e1] = 0x409d, [0x04e2] = 0x7815, [0x04e3] = 0x7808,
  [0x04e4] = 0x40fd, [0x04e5] = 0x79d3, [0x04e6] = 0x79d4, [0x04e7] = 0x79d0,
  [0x04e8] = 0x79d7, [0x04e9] = 0x7a7c, [0x04ea] = 0x4194, [0x04eb] = 0x4193,
  [0x04ec] = 0x7a7d, [0x04ed] = 0x7a83, [0x04ee] = 0x7a82, [0x04ef] = 0x41c6,
  [0x04f0] = 0x7ad4, [0x04f1] = 0x7ad5, [0x04f2] = 0x7ad3, [0x04f3] = 0x7ad0,
  [0x04f4] = 0x7ad2, [0x04f5] = 0x7afe, [0x04f6] = 0x7afc, [0x04f7] = 0x7c77,
  [0x04f8] = 0x7c7c, [0x04f9] = 0x7c7b, [0x04fa] = 0x42b8, [0x04fb] = 0x2f96a,
  [0x04fc] = 0x42b7, [0x04fd] = 0x42b9, [0x04fe] = 0x4353, [0x04ff] = 0x25133,
  [0x0500] = 0x4352, [0x0501] = 0x4351, [0x0502] = 0x7f8f, [0x0503] = 0x80d3,
  [0x0504] = 0x43e3, [0x0505] = 0x80cb, [0x0506] = 0x80d2, [0x0507] = 0x43e2,
  [0x0508] = 0x8109, [0x0509] = 0x80e2, [0x050a] = 0x80df, [0x050b] = 0x80c6,
  [0x050c] = 0x4463, [0x050d] = 0x8224, [0x050e] = 0x82f7, [0x050f] = 0x82d8,
  [0x0510] = 0x82dd, [0x0511] = 0x44aa, [0x0512] = 0x44a6, [0x0513] = 0x82f8,
  [0x0514] = 0x82fc, [0x0515] = 0x44a8, [0x0516] = 0x44a9, [0x0517] = 0x82e9,
  [0x0518] = 0x44ab, [0x0519] = 0x82ee, [0x051a] = 0x44ac, [0x051b] = 0x82d0,
  [0x051c] = 0x830e, [0x051d] = 0x82e2, [0x051e] = 0x830b, [0x051f] = 0x82fd,
  [0x0520] = 0x5179, [0x0521] = 0x8676, [0x0522] = 0x459e, [0x0523] = 0x8678,
  [0x0524] = 0x459f, [0x0525] = 0x45a0, [0x0526] = 0x8675, [0x0527] = 0x867d,
  [0x0528] = 0x460f, [0x0529] = 0x8842, [0x052a] = 0x8866, [0x052b] = 0x461c,
  [0x052c] = 0x898c, [0x052d] = 0x8a05, [0x052e] = 0x46ae, [0x052f] = 0x8a06,
  [0x0530] = 0x46b0, [0x0531] = 0x8c9f, [0x0532] = 0x47d4, [0x0533] = 0x8ff1,
  [0x0534] = 0x8fe7, [0x0535] = 0x8fe9, [0x0536] = 0x8fef, [0x0537] = 0x90c2,
  [0x0538] = 0x90bc, [0x0539] = 0x48bb, [0x053a] = 0x90c6, [0x053b] = 0x90c0,
  [0x053c] = 0x48c1, [0x053d] = 0x48c2, [0x053e] = 0x90cd, [0x053f] = 0x90c9,
  [0x0540] = 0x48be, [0x0541] = 0x90c4, [0x0542] = 0x48e5, [0x0543] = 0x9581,
  [0x0544] = 0x49c6, [0x0545] = 0x9cec, [0x0546] = 0x5032, [0x0547] = 0x4ff9,
  [0x0548] = 0x501d, [0x0549] = 0x4fff, [0x054a] = 0x5004, [0x054b] = 0x4ff0,
  [0x054c] = 0x5003, [0x054d] = 0x462e, [0x054e] = 0x5002, [0x054f] = 0x4ffc,
  [0x0550] = 0x4ff2, [0x0551] = 0x5024, [0x0552] = 0x5008, [0x0553] = 0x5036,
  [0x0554] = 0x502e, [0x0555] = 0x3465, [0x0556] = 0x5010, [0x0557] = 0x5038,
  [0x0558] = 0x5039, [0x0559] = 0x4ffd, [0x055a] = 0x5056, [0x055b] = 0x4ffb,
  [0x055c] = 0x51a3, [0x055d] = 0x51a6, [0x055e] = 0x51a1, [0x055f] = 0x34d1,
  [0x0560] = 0x34d0, [0x0561] = 0x51c7, [0x0562] = 0x51c9, [0x0563] = 0x5260,
  [0x0564] = 0x5264, [0x0565] = 0x5259, [0x0566] = 0x5265, [0x0567] = 0x5267,
  [0x0568] = 0x5257, [0x0569] = 0x5263, [0x056a] = 0x34ee, [0x056b] = 0x5253,
  [0x056c] = 0x34ef, [0x056d] = 0x52cf, [0x056e] = 0x351e, [0x056f] = 0x52ce,
  [0x0570] = 0x52d0, [0x0571] = 0x52d1, [0x0572] = 0x52cc, [0x0573] = 0x354b,
  [0x0574] = 0x354d, [0x0575] = 0x3556, [0x0576] = 0x550d, [0x0577] = 0x54f4,
  [0x0578] = 0x3592, [0x0579] = 0x5513, [0x057a] = 0x54ef, [0x057b] = 0x54f5,
  [0x057c] = 0x54f9, [0x057d] = 0x5502, [0x057e] = 0x5500, [0x057f] = 0x3593,
  [0x0580] = 0x3590, [0x0581] = 0x5518, [0x0582] = 0x54f0, [0x0583] = 0x54f6,
  [0x0584] = 0x2f841, [0x0585] = 0x3597, [0x0586] = 0x5519, [0x0587] = 0x3623,
  [0x0588] = 0x5705, [0x0589] = 0x57c9, [0x058a] = 0x363f, [0x058b] = 0x57b7,
  [0x058c] = 0x57cd, [0x058d] = 0x3643, [0x058e] = 0x3642, [0x058f] = 0x3644,
  [0x0590] = 0x57be, [0x0591] = 0x57bb, [0x0592] = 0x3645, [0x0593] = 0x57db,
  [0x0594] = 0x57c8, [0x0595] = 0x57c4, [0x0596] = 0x57c5, [0x0597] = 0x57d1,
  [0x0598] = 0x57ca, [0x0599] = 0x57c0, [0x059a] = 0x36d9, [0x059b] = 0x36de,
  [0x059c] = 0x5a21, [0x059d] = 0x5a2a, [0x059e] = 0x36cf, [0x059f] = 0x5a1d,
  [0x05a0] = 0x36cd, [0x05a1] = 0x5a0b, [0x05a2] = 0x36dd, [0x05a3] = 0x36ce,
  [0x05a4] = 0x36d3, [0x05a5] = 0x36d6, [0x05a6] = 0x5a22, [0x05a7] = 0x36dc,
  [0x05a8] = 0x36d1, [0x05a9] = 0x5a24, [0x05aa] = 0x36d0, [0x05ab] = 0x5a14,
  [0x05ac] = 0x5a31, [0x05ad] = 0x36d5, [0x05ae] = 0x5a2f, [0x05af] = 0x5a1a,
  [0x05b0] = 0x5a12, [0x05b1] = 0x36d4, [0x05b2] = 0x36db, [0x05b3] = 0x5a26,
  [0x05b4] = 0x2172e, [0x05b5] = 0x3743, [0x05b6] = 0x5bbc, [0x05b7] = 0x5bbb,
  [0x05b8] = 0x5bb7, [0x05b9] = 0x5c05, [0x05ba] = 0x5c06, [0x05bb] = 0x5c52,
  [0x05bc] = 0x5c53, [0x05bd] = 0x37cd, [0x05be] = 0x37d1, [0x05bf] = 0x5cfa,
  [0x05c0] = 0x5ceb, [0x05c1] = 0x37ca, [0x05c2] = 0x5cf3, [0x05c3] = 0x5cf5,
  [0x05c4] = 0x5ce9, [0x05c5] = 0x5cef, [0x05c6] = 0x37d4, [0x05c7] = 0x5e2a,
  [0x05c8] = 0x5e30, [0x05c9] = 0x5e2e, [0x05ca] = 0x5e2c, [0x05cb] = 0x5e2f,
  [0x05cc] = 0x5eaf, [0x05cd] = 0x5ea9, [0x05ce] = 0x3886, [0x05cf] = 0x5efd,
  [0x05d0] = 0x5f32, [0x05d1] = 0x5f8e, [0x05d2] = 0x5f93, [0x05d3] = 0x5f8f,
  [0x05d4] = 0x604f, [0x05d5] = 0x6099, [0x05d6] = 0x3933, [0x05d7] = 0x607e,
  [0x05d8] = 0x3937, [0x05d9] = 0x6074, [0x05da] = 0x604b, [0x05db] = 0x6073,
  [0x05dc] = 0x6075, [0x05dd] = 0x392a, [0x05de] = 0x391f, [0x05df] = 0x6056,
  [0x05e0] = 0x60a9, [0x05e1] = 0x608b, [0x05e2] = 0x60a6, [0x05e3] = 0x3939,
  [0x05e4] = 0x6093, [0x05e5] = 0x60ae, [0x05e6] = 0x609e, [0x05e7] = 0x60a7,
  [0x05e8] = 0x6245, [0x05e9] = 0x39f2, [0x05ea] = 0x39f8, [0x05eb] = 0x632e,
  [0x05ec] = 0x39f7, [0x05ed] = 0x6352, [0x05ee] = 0x6330, [0x05ef] = 0x635b,
  [0x05f0] = 0x39f4, [0x05f1] = 0x6319, [0x05f2] = 0x631b, [0x05f3] = 0x39f1,
  [0x05f4] = 0x6331, [0x05f5] = 0x635d, [0x05f6] = 0x6337, [0x05f7] = 0x6335,
  [0x05f8] = 0x6353, [0x05f9] = 0x39f5, [0x05fa] = 0x635c, [0x05fb] = 0x633f,
  [0x05fc] = 0x654b, [0x05fd] = 0x3a87, [0x05fe] = 0x4369, [0x05ff] = 0x658b,
  [0x0600] = 0x3ab6, [0x0601] = 0x659a, [0x0602] = 0x6650, [0x0603] = 0x6646,
  [0x0604] = 0x664e, [0x0605] = 0x6640, [0x0606] = 0x3ae9, [0x0607] = 0x664b,
  [0x0608] = 0x6648, [0x0609] = 0x3aeb, [0x060a] = 0x6660, [0x060b] = 0x6644,
  [0x060c] = 0x664d, [0x060d] = 0x3b34, [0x060e] = 0x6837, [0x060f] = 0x6824,
  [0x0610] = 0x3b62, [0x0611] = 0x3b5c, [0x0612] = 0x681b, [0x0613] = 0x6836,
  [0x0614] = 0x3b60, [0x0615] = 0x682c, [0x0616] = 0x6819, [0x0617] = 0x6856,
  [0x0618] = 0x6847, [0x0619] = 0x683e, [0x061a] = 0x681e, [0x061b] = 0x2f8e1,
  [0x061c] = 0x6815, [0x061d] = 0x6822, [0x061e] = 0x6827, [0x061f] = 0x6859,
  [0x0620] = 0x6858, [0x0621] = 0x6855, [0x0622] = 0x6830, [0x0623] = 0x6823,
  [0x0624] = 0x6b2e, [0x0625] = 0x6b2b, [0x0626] = 0x6b30, [0x0627] = 0x6b6c,
  [0x0628] = 0x3c61, [0x0629] = 0x6b8b, [0x062a] = 0x3c7f, [0x062b] = 0x6be9,
  [0x062c] = 0x6bea, [0x062d] = 0x6be5, [0x062e] = 0x6d6b, [0x062f] = 0x3ce5,
  [0x0630] = 0x3ce6, [0x0631] = 0x6d73, [0x0632] = 0x6d57, [0x0633] = 0x3ce9,
  [0x0634] = 0x3cf3, [0x0635] = 0x6d5d, [0x0636] = 0x6d56, [0x0637] = 0x6d8f,
  [0x0638] = 0x6d5b, [0x0639] = 0x6d1c, [0x063a] = 0x6d9a, [0x063b] = 0x6d9b,
  [0x063c] = 0x6d99, [0x063d] = 0x3cee, [0x063e] = 0x6d81, [0x063f] = 0x6d71,
  [0x0640] = 0x3ced, [0x0641] = 0x3cec, [0x0642] = 0x6d72, [0x0643] = 0x6d5c,
  [0x0644] = 0x6d96, [0x0645] = 0x70c4, [0x0646] = 0x70db, [0x0647] = 0x70cc,
  [0x0648] = 0x70d0, [0x0649] = 0x70e3, [0x064a] = 0x70df, [0x064b] = 0x3db3,
  [0x064c] = 0x70d6, [0x064d] = 0x70ee, [0x064e] = 0x70d5, [0x064f] = 0x3db5,
  [0x0650] = 0x3e27, [0x0651] = 0x3e35, [0x0652] = 0x3e36, [0x0653] = 0x727a,
  [0x0654] = 0x3e71, [0x0655] = 0x72f5, [0x0656] = 0x7302, [0x0657] = 0x3eb8,
  [0x0658] = 0x3ec2, [0x0659] = 0x73e2, [0x065a] = 0x73ec, [0x065b] = 0x73d5,
  [0x065c] = 0x73f9, [0x065d] = 0x73df, [0x065e] = 0x73e6, [0x065f] = 0x3ec8,
  [0x0660] = 0x3ec0, [0x0661] = 0x3ec1, [0x0662] = 0x3ec4, [0x0663] = 0x73e4,
  [0x0664] = 0x73e1, [0x0665] = 0x74f3, [0x0666] = 0x3f1f, [0x0667] = 0x3f1c,
  [0x0668] = 0x3f1d, [0x0669] = 0x3f4d, [0x066a] = 0x7556, [0x066b] = 0x7555,
  [0x066c] = 0x7558, [0x066d] = 0x7557, [0x066e] = 0x755e, [0x066f] = 0x75c3,
  [0x0670] = 0x3f87, [0x0671] = 0x3f82, [0x0672] = 0x75b4, [0x0673] = 0x3f7d,
  [0x0674] = 0x75b1, [0x0675] = 0x3fdf, [0x0676] = 0x4000, [0x0677] = 0x76cb,
  [0x0678] = 0x76cc, [0x0679] = 0x772a, [0x067a] = 0x4020, [0x067b] = 0x7716,
  [0x067c] = 0x770f, [0x067d] = 0x4022, [0x067e] = 0x4024, [0x067f] = 0x773f,
  [0x0680] = 0x772b, [0x0681] = 0x770e, [0x0682] = 0x7724, [0x0683] = 0x4021,
  [0x0684] = 0x7721, [0x0685] = 0x7718, [0x0686] = 0x77dd, [0x0687] = 0x40a4,
  [0x0688] = 0x40a5, [0x0689] = 0x7824, [0x068a] = 0x7836, [0x068b] = 0x4101,
  [0x068c] = 0x7958, [0x068d] = 0x7959, [0x068e] = 0x4103, [0x068f] = 0x7962,
  [0x0690] = 0x79da, [0x0691] = 0x79d9, [0x0692] = 0x4137, [0x0693] = 0x79e1,
  [0x0694] = 0x79e5, [0x0695] = 0x79e8, [0x0696] = 0x79db, [0x0697] = 0x4138,
  [0x0698] = 0x79e2, [0x0699] = 0x79f0, [0x069a] = 0x4199, [0x069b] = 0x4198,
  [0x069c] = 0x4197, [0x069d] = 0x41c9, [0x069e] = 0x7ada, [0x069f] = 0x7add,
  [0x06a0] = 0x41c7, [0x06a1] = 0x7adb, [0x06a2] = 0x7adc, [0x06a3] = 0x41d9,
  [0x06a4] = 0x41db, [0x06a5] = 0x7b0d, [0x06a6] = 0x7b0b, [0x06a7] = 0x7b14,
  [0x06a8] = 0x7c8e, [0x06a9] = 0x7c86, [0x06aa] = 0x427b, [0x06ab] = 0x7c87,
  [0x06ac] = 0x7c83, [0x06ad] = 0x7c8b, [0x06ae] = 0x427c, [0x06af] = 0x42bd,
  [0x06b0] = 0x42bc, [0x06b1] = 0x42c3, [0x06b2] = 0x7d24, [0x06b3] = 0x42c1,
  [0x06b4] = 0x42bf, [0x06b5] = 0x42c4, [0x06b6] = 0x7d25, [0x06b7] = 0x7f62,
  [0x06b8] = 0x7f93, [0x06b9] = 0x7f99, [0x06ba] = 0x7f97, [0x06bb] = 0x437e,
  [0x06bc] = 0x437f, [0x06bd] = 0x7fc4, [0x06be] = 0x7fc6, [0x06bf] = 0x800a,
  [0x06c0] = 0x43b4, [0x06c1] = 0x43b3, [0x06c2] = 0x8040, [0x06c3] = 0x803c,
  [0x06c4] = 0x803b, [0x06c5] = 0x80f6, [0x06c6] = 0x80ff, [0x06c7] = 0x80ee,
  [0x06c8] = 0x8104, [0x06c9] = 0x8103, [0x06ca] = 0x8107, [0x06cb] = 0x2f983,
  [0x06cc] = 0x43e6, [0x06cd] = 0x80f7, [0x06ce] = 0x4459, [0x06cf] = 0x445a,
  [0x06d0] = 0x822d, [0x06d1] = 0x4464, [0x06d2] = 0x8227, [0x06d3] = 0x8229,
  [0x06d4] = 0x831f, [0x06d5] = 0x8357, [0x06d6] = 0x44b4, [0x06d7] = 0x44b9,
  [0x06d8] = 0x44b7, [0x06d9] = 0x44b5, [0x06da] = 0x8321, [0x06db] = 0x44c1,
  [0x06dc] = 0x44b1, [0x06dd] = 0x8318, [0x06de] = 0x8358, [0x06df] = 0x44b3,
  [0x06e0] = 0x44ba, [0x06e1] = 0x458c, [0x06e2] = 0x458b, [0x06e3] = 0x458d,
  [0x06e4] = 0x8684, [0x06e5] = 0x869f, [0x06e6] = 0x869b, [0x06e7] = 0x8689,
  [0x06e8] = 0x86a6, [0x06e9] = 0x8692, [0x06ea] = 0x868f, [0x06eb] = 0x86a0,
  [0x06ec] = 0x884f, [0x06ed] = 0x8878, [0x06ee] = 0x887a, [0x06ef] = 0x886e,
  [0x06f0] = 0x887b, [0x06f1] = 0x8884, [0x06f2] = 0x8873, [0x06f3] = 0x4678,
  [0x06f4] = 0x4677, [0x06f5] = 0x8a0d, [0x06f6] = 0x8a0b, [0x06f7] = 0x8a19,
  [0x06f8] = 0x46b2, [0x06f9] = 0x47d6, [0x06fa] = 0x8ed0, [0x06fb] = 0x4845,
  [0x06fc] = 0x4892, [0x06fd] = 0x4895, [0x06fe] = 0x8ff9, [0x06ff] = 0x9009,
  [0x0700] = 0x9008, [0x0701] = 0x48c6, [0x0702] = 0x90de, [0x0703] = 0x9151,
  [0x0704] = 0x48e7, [0x0705] = 0x48e8, [0x0706] = 0x91db, [0x0707] = 0x91df,
  [0x0708] = 0x91de, [0x0709] = 0x91d6, [0x070a] = 0x91e0, [0x070b] = 0x9585,
  [0x070c] = 0x9660, [0x070d] = 0x9659, [0x070e] = 0x49cb, [0x070f] = 0x9656,
  [0x0710] = 0x49cd, [0x0711] = 0x49f1, [0x0712] = 0x96bd, [0x0713] = 0x4b22,
  [0x0714] = 0x3421, [0x0715] = 0x5042, [0x0716] = 0x5059, [0x0717] = 0x346f,
  [0x0718] = 0x5044, [0x0719] = 0x5066, [0x071a] = 0x5052, [0x071b] = 0x5054,
  [0x071c] = 0x5071, [0x071d] = 0x5050, [0x071e] = 0x507b, [0x071f] = 0x507c,
  [0x0720] = 0x5058, [0x0721] = 0x3470, [0x0722] = 0x3464, [0x0723] = 0x5079,
  [0x0724] = 0x506c, [0x0725] = 0x5078, [0x0726] = 0x51a8, [0x0727] = 0x51d1,
  [0x0728] = 0x51cf, [0x0729] = 0x5268, [0x072a] = 0x5276, [0x072b] = 0x52d4,
  [0x072c] = 0x352d, [0x072d] = 0x53a0, [0x072e] = 0x53c4, [0x072f] = 0x3558,
  [0x0730] = 0x5558, [0x0731] = 0x554c, [0x0732] = 0x5568, [0x0733] = 0x35a6,
  [0x0734] = 0x5549, [0x0735] = 0x35a4, [0x0736] = 0x359f, [0x0737] = 0x555d,
  [0x0738] = 0x5529, [0x0739] = 0x20dae, [0x073a] = 0x5554, [0x073b] = 0x5553,
  [0x073c] = 0x35a3, [0x073d] = 0x555a, [0x073e] = 0x35a0, [0x073f] = 0x553a,
  [0x0740] = 0x553f, [0x0741] = 0x552b, [0x0742] = 0x57ea, [0x0743] = 0x364a,
  [0x0744] = 0x57ef, [0x0745] = 0x3647, [0x0746] = 0x3648, [0x0747] = 0x57dd,
  [0x0748] = 0x57fe, [0x0749] = 0x2f855, [0x074a] = 0x57de, [0x074b] = 0x57e6,
  [0x074c] = 0x3649, [0x074d] = 0x57e8, [0x074e] = 0x57ff, [0x074f] = 0x5803,
  [0x0750] = 0x58f7, [0x0751] = 0x68a6, [0x0752] = 0x591f, [0x0753] = 0x369e,
  [0x0754] = 0x595b, [0x0755] = 0x595d, [0x0756] = 0x595e, [0x0757] = 0x21637,
  [0x0758] = 0x36e8, [0x0759] = 0x5a2b, [0x075a] = 0x36ec, [0x075b] = 0x5a3b,
  [0x075c] = 0x36ed, [0x075d] = 0x36e6, [0x075e] = 0x5a61, [0x075f] = 0x5a3a,
  [0x0760] = 0x5a6e, [0x0761] = 0x5a4b, [0x0762] = 0x5a6b, [0x0763] = 0x36eb,
  [0x0764] = 0x36e7, [0x0765] = 0x5a45, [0x0766] = 0x5a4e, [0x0767] = 0x5a68,
  [0x0768] = 0x5a3d, [0x0769] = 0x5a71, [0x076a] = 0x5a3f, [0x076b] = 0x5a6f,
  [0x076c] = 0x5a75, [0x076d] = 0x36e9, [0x076e] = 0x5a73, [0x076f] = 0x5a2c,
  [0x0770] = 0x5a59, [0x0771] = 0x5a54, [0x0772] = 0x5a4f, [0x0773] = 0x5a63,
  [0x0774] = 0x375c, [0x0775] = 0x375d, [0x0776] = 0x5bc8, [0x0777] = 0x3760,
  [0x0778] = 0x5bc3, [0x0779] = 0x375b, [0x077a] = 0x5c5b, [0x077b] = 0x5c61,
  [0x077c] = 0x3799, [0x077d] = 0x5d21, [0x077e] = 0x5d0a, [0x077f] = 0x5d09,
  [0x0780] = 0x37d8, [0x0781] = 0x5d2c, [0x0782] = 0x5d08, [0x0783] = 0x37da,
  [0x0784] = 0x37dd, [0x0785] = 0x5d2a, [0x0786] = 0x5d15, [0x0787] = 0x37e0,
  [0x0788] = 0x5d10, [0x0789] = 0x5d13, [0x078a] = 0x37e5, [0x078b] = 0x5d2f,
  [0x078c] = 0x5d18, [0x078d] = 0x37d7, [0x078e] = 0x5de3, [0x078f] = 0x5e39,
  [0x0790] = 0x5e35, [0x0791] = 0x5e3a, [0x0792] = 0x5e32, [0x0793] = 0x384e,
  [0x0794] = 0x388c, [0x0795] = 0x3888, [0x0796] = 0x2f88d, [0x0797] = 0x5ebb,
  [0x0798] = 0x5eba, [0x0799] = 0x5f34, [0x079a] = 0x5f39, [0x079b] = 0x38ce,
  [0x079c] = 0x2f89c, [0x079d] = 0x38e5, [0x079e] = 0x38e6, [0x079f] = 0x6098,
  [0x07a0] = 0x3932, [0x07a1] = 0x60d0, [0x07a2] = 0x3940, [0x07a3] = 0x3947,
  [0x07a4] = 0x394c, [0x07a5] = 0x60d7, [0x07a6] = 0x60aa, [0x07a7] = 0x3935,
  [0x07a8] = 0x60a1, [0x07a9] = 0x60a4, [0x07aa] = 0x3930, [0x07ab] = 0x60ee,
  [0x07ac] = 0x3943, [0x07ad] = 0x60e7, [0x07ae] = 0x394d, [0x07af] = 0x60e8,
  [0x07b0] = 0x60de, [0x07b1] = 0x39b7, [0x07b2] = 0x39f3, [0x07b3] = 0x637e,
  [0x07b4] = 0x638b, [0x07b5] = 0x3a02, [0x07b6] = 0x3a0b, [0x07b7] = 0x6379,
  [0x07b8] = 0x6386, [0x07b9] = 0x6393, [0x07ba] = 0x3a04, [0x07bb] = 0x6373,
  [0x07bc] = 0x636a, [0x07bd] = 0x2f8ba, [0x07be] = 0x636c, [0x07bf] = 0x3a08,
  [0x07c0] = 0x637f, [0x07c1] = 0x39fc, [0x07c2] = 0x63b2, [0x07c3] = 0x63ba,
  [0x07c4] = 0x39ff, [0x07c5] = 0x3a00, [0x07c6] = 0x6366, [0x07c7] = 0x6374,
  [0x07c8] = 0x3a8b, [0x07c9] = 0x655a, [0x07ca] = 0x3a8d, [0x07cb] = 0x654e,
  [0x07cc] = 0x654d, [0x07cd] = 0x658d, [0x07ce] = 0x658e, [0x07cf] = 0x65ad,
  [0x07d0] = 0x3aca, [0x07d1] = 0x65c7, [0x07d2] = 0x65ca, [0x07d3] = 0x3acb,
  [0x07d4] = 0x65c9, [0x07d5] = 0x2f8cb, [0x07d6] = 0x65e3, [0x07d7] = 0x6657,
  [0x07d8] = 0x3af3, [0x07d9] = 0x6663, [0x07da] = 0x6667, [0x07db] = 0x671a,
  [0x07dc] = 0x6719, [0x07dd] = 0x6716, [0x07de] = 0x3b36, [0x07df] = 0x3b6a,
  [0x07e0] = 0x689e, [0x07e1] = 0x68b6, [0x07e2] = 0x6898, [0x07e3] = 0x6873,
  [0x07e4] = 0x3b6b, [0x07e5] = 0x689a, [0x07e6] = 0x688e, [0x07e7] = 0x68b7,
  [0x07e8] = 0x68db, [0x07e9] = 0x68a5, [0x07ea] = 0x686c, [0x07eb] = 0x68c1,
  [0x07ec] = 0x6884, [0x07ed] = 0x3b71, [0x07ee] = 0x3b68, [0x07ef] = 0x6895,
  [0x07f0] = 0x687a, [0x07f1] = 0x6899, [0x07f2] = 0x3b72, [0x07f3] = 0x68b8,
  [0x07f4] = 0x68b9, [0x07f5] = 0x6870, [0x07f6] = 0x3c2e, [0x07f7] = 0x6b35,
  [0x07f8] = 0x3c62, [0x07f9] = 0x6b90, [0x07fa] = 0x6bbb, [0x07fb] = 0x6bed,
  [0x07fc] = 0x3c98, [0x07fd] = 0x3cb5, [0x07fe] = 0x3ceb, [0x07ff] = 0x6dc1,
  [0x0800] = 0x6dc3, [0x0801] = 0x6dce, [0x0802] = 0x3cfb, [0x0803] = 0x3cf8,
  [0x0804] = 0x6dad, [0x0805] = 0x6e04, [0x0806] = 0x3cf5, [0x0807] = 0x6db9,
  [0x0808] = 0x3d08, [0x0809] = 0x6de7, [0x080a] = 0x2f907, [0x080b] = 0x6e08,
  [0x080c] = 0x6e06, [0x080d] = 0x3d0a, [0x080e] = 0x6e0a, [0x080f] = 0x6db0,
  [0x0810] = 0x3d06, [0x0811] = 0x6df8, [0x0812] = 0x6e0c, [0x0813] = 0x3cfd,
  [0x0814] = 0x6db1, [0x0815] = 0x3cfa, [0x0816] = 0x6e02, [0x0817] = 0x6e07,
  [0x0818] = 0x6e09, [0x0819] = 0x6e01, [0x081a] = 0x6e17, [0x081b] = 0x6dff,
  [0x081c] = 0x6e12, [0x081d] = 0x3dba, [0x081e] = 0x3db9, [0x081f] = 0x7103,
  [0x0820] = 0x7107, [0x0821] = 0x7101, [0x0822] = 0x70f5, [0x0823] = 0x70f1,
  [0x0824] = 0x7108, [0x0825] = 0x70f2, [0x0826] = 0x710f, [0x0827] = 0x3dbb,
  [0x0828] = 0x70fe, [0x0829] = 0x3e18, [0x082a] = 0x3e40, [0x082b] = 0x3e3d,
  [0x082c] = 0x731a, [0x082d] = 0x7310, [0x082e] = 0x730e, [0x082f] = 0x7402,
  [0x0830] = 0x73f3, [0x0831] = 0x3ecd, [0x0832] = 0x3ec9, [0x0833] = 0x73fb,
  [0x0834] = 0x3ecb, [0x0835] = 0x3eca, [0x0836] = 0x3ece, [0x0837] = 0x751b,
  [0x0838] = 0x7523, [0x0839] = 0x7561, [0x083a] = 0x7568, [0x083b] = 0x3f5e,
  [0x083c] = 0x7567, [0x083d] = 0x75d3, [0x083e] = 0x3f91, [0x083f] = 0x3f8c,
  [0x0840] = 0x7690, [0x0841] = 0x3fe1, [0x0842] = 0x4002, [0x0843] = 0x76d5,
  [0x0844] = 0x76d7, [0x0845] = 0x76d6, [0x0846] = 0x7730, [0x0847] = 0x402b,
  [0x0848] = 0x7726, [0x0849] = 0x402a, [0x084a] = 0x7740, [0x084b] = 0x3e14,
  [0x084c] = 0x771e, [0x084d] = 0x40ad, [0x084e] = 0x40a3, [0x084f] = 0x40ab,
  [0x0850] = 0x7847, [0x0851] = 0x40af, [0x0852] = 0x784b, [0x0853] = 0x7851,
  [0x0854] = 0x784f, [0x0855] = 0x7842, [0x0856] = 0x7846, [0x0857] = 0x4104,
  [0x0858] = 0x796e, [0x0859] = 0x796c, [0x085a] = 0x79f2, [0x085b] = 0x4144,
  [0x085c] = 0x79f1, [0x085d] = 0x79f5, [0x085e] = 0x79f3, [0x085f] = 0x79f9,
  [0x0860] = 0x413d, [0x0861] = 0x4147, [0x0862] = 0x419c, [0x0863] = 0x7a9a,
  [0x0864] = 0x7a93, [0x0865] = 0x7a91, [0x0866] = 0x7ae1, [0x0867] = 0x41e0,
  [0x0868] = 0x41e4, [0x0869] = 0x7b21, [0x086a] = 0x7b1c, [0x086b] = 0x7b16,
  [0x086c] = 0x7b17, [0x086d] = 0x7b36, [0x086e] = 0x7b1f, [0x086f] = 0x4280,
  [0x0870] = 0x7c93, [0x0871] = 0x7c99, [0x0872] = 0x7c9a, [0x0873] = 0x7c9c,
  [0x0874] = 0x42ca, [0x0875] = 0x7d49, [0x0876] = 0x42d4, [0x0877] = 0x7d34,
  [0x0878] = 0x7d37, [0x0879] = 0x42d2, [0x087a] = 0x7d2d, [0x087b] = 0x42cb,
  [0x087c] = 0x7d4c, [0x087d] = 0x42ce, [0x087e] = 0x42d3, [0x087f] = 0x7d48,
  [0x0880] = 0x4344, [0x0881] = 0x4348, [0x0882] = 0x7f3b, [0x0883] = 0x4345,
  [0x0884] = 0x4381, [0x0885] = 0x4386, [0x0886] = 0x4385, [0x0887] = 0x8008,
  [0x0888] = 0x801a, [0x0889] = 0x43a3, [0x088a] = 0x801d, [0x088b] = 0x43b5,
  [0x088c] = 0x8049, [0x088d] = 0x8045, [0x088e] = 0x8044, [0x088f] = 0x7c9b,
  [0x0890] = 0x43fa, [0x0891] = 0x43f9, [0x0892] = 0x812a, [0x0893] = 0x812e,
  [0x0894] = 0x43fb, [0x0895] = 0x43f2, [0x0896] = 0x8131, [0x0897] = 0x43ef,
  [0x0898] = 0x811a, [0x0899] = 0x8134, [0x089a] = 0x8117, [0x089b] = 0x445b,
  [0x089c] = 0x4466, [0x089d] = 0x44ce, [0x089e] = 0x831d, [0x089f] = 0x8371,
  [0x08a0] = 0x8384, [0x08a1] = 0x8380, [0x08a2] = 0x8372, [0x08a3] = 0x83a1,
  [0x08a4] = 0x35b4, [0x08a5] = 0x8379, [0x08a6] = 0x8391, [0x08a7] = 0x44c8,
  [0x08a8] = 0x839f, [0x08a9] = 0x83ad, [0x08aa] = 0x44d1, [0x08ab] = 0x44c5,
  [0x08ac] = 0x8323, [0x08ad] = 0x44d2, [0x08ae] = 0x8385, [0x08af] = 0x839c,
  [0x08b0] = 0x83b7, [0x08b1] = 0x8658, [0x08b2] = 0x865a, [0x08b3] = 0x458f,
  [0x08b4] = 0x8657, [0x08b5] = 0x86b2, [0x08b6] = 0x45a7, [0x08b7] = 0x86ae,
  [0x08b8] = 0x45a5, [0x08b9] = 0x45a4, [0x08ba] = 0x4611, [0x08bb] = 0x8845,
  [0x08bc] = 0x889c, [0x08bd] = 0x8894, [0x08be] = 0x88a3, [0x08bf] = 0x888f,
  [0x08c0] = 0x88a5, [0x08c1] = 0x88a9, [0x08c2] = 0x88a6, [0x08c3] = 0x888a,
  [0x08c4] = 0x88a0, [0x08c5] = 0x8890, [0x08c6] = 0x8992, [0x08c7] = 0x8991,
  [0x08c8] = 0x8994, [0x08c9] = 0x46b5, [0x08ca] = 0x8a26, [0x08cb] = 0x8a32,
  [0x08cc] = 0x8a28, [0x08cd] = 0x46b4, [0x08ce] = 0x46bd, [0x08cf] = 0x8a1c,
  [0x08d0] = 0x46bb, [0x08d1] = 0x8a2b, [0x08d2] = 0x8a20, [0x08d3] = 0x46b9,
  [0x08d4] = 0x8a29, [0x08d5] = 0x46c2, [0x08d6] = 0x46be, [0x08d7] = 0x46ba,
  [0x08d8] = 0x8a21, [0x08d9] = 0x8c3a, [0x08da] = 0x3ab7, [0x08db] = 0x8c5b,
  [0x08dc] = 0x8c58, [0x08dd] = 0x8c7c, [0x08de] = 0x4758, [0x08df] = 0x8ca6,
  [0x08e0] = 0x8cae, [0x08e1] = 0x8cad, [0x08e2] = 0x8d65, [0x08e3] = 0x479b,
  [0x08e4] = 0x8d7e, [0x08e5] = 0x479c, [0x08e6] = 0x8d7c, [0x08e7] = 0x8d7f,
  [0x08e8] = 0x8d7a, [0x08e9] = 0x8dbd, [0x08ea] = 0x47da, [0x08eb] = 0x47de,
  [0x08ec] = 0x8dc0, [0x08ed] = 0x8dbb, [0x08ee] = 0x8ead, [0x08ef] = 0x8eaf,
  [0x08f0] = 0x8ed6, [0x08f1] = 0x484d, [0x08f2] = 0x4846, [0x08f3] = 0x4847,
  [0x08f4] = 0x484b, [0x08f5] = 0x484c, [0x08f6] = 0x8ed9, [0x08f7] = 0x4848,
  [0x08f8] = 0x4899, [0x08f9] = 0x9012, [0x08fa] = 0x900e, [0x08fb] = 0x9025,
  [0x08fc] = 0x489b, [0x08fd] = 0x9013, [0x08fe] = 0x90ee, [0x08ff] = 0x48ce,
  [0x0900] = 0x90ab, [0x0901] = 0x90f7, [0x0902] = 0x48eb, [0x0903] = 0x9159,
  [0x0904] = 0x9154, [0x0905] = 0x91f2, [0x0906] = 0x91f0, [0x0907] = 0x91e5,
  [0x0908] = 0x91f6, [0x0909] = 0x491c, [0x090a] = 0x498c, [0x090b] = 0x9587,
  [0x090c] = 0x49d1, [0x090d] = 0x965a, [0x090e] = 0x49d6, [0x090f] = 0x49d3,
  [0x0910] = 0x966e, [0x0911] = 0x49d4, [0x0912] = 0x49d0, [0x0913] = 0x49d5,
  [0x0914] = 0x9679, [0x0915] = 0x4a0b, [0x0916] = 0x98e1, [0x0917] = 0x98e6,
  [0x0918] = 0x4bc6, [0x0919] = 0x9ec4, [0x091a] = 0x9ed2, [0x091b] = 0x4e80,
  [0x091c] = 0x3424, [0x091d] = 0x4e81, [0x091e] = 0x508f, [0x091f] = 0x5097,
  [0x0920] = 0x5088, [0x0921] = 0x5089, [0x0922] = 0x3474, [0x0923] = 0x347a,
  [0x0924] = 0x5081, [0x0925] = 0x5160, [0x0926] = 0x20564, [0x0927] = 0x34c3,
  [0x0928] = 0x5e42, [0x0929] = 0x51d3, [0x092a] = 0x34d4, [0x092b] = 0x34d5,
  [0x092c] = 0x51d2, [0x092d] = 0x51d6, [0x092e] = 0x5273, [0x092f] = 0x34fb,
  [0x0930] = 0x5270, [0x0931] = 0x34f7, [0x0932] = 0x3532, [0x0933] = 0x2f833,
  [0x0934] = 0x53a8, [0x0935] = 0x53a6, [0x0936] = 0x53c5, [0x0937] = 0x5597,
  [0x0938] = 0x55de, [0x0939] = 0x35ba, [0x093a] = 0x35bf, [0x093b] = 0x5596,
  [0x093c] = 0x55b4, [0x093d] = 0x35c7, [0x093e] = 0x5585, [0x093f] = 0x35b7,
  [0x0940] = 0x559b, [0x0941] = 0x55a0, [0x0942] = 0x35b9, [0x0943] = 0x5559,
  [0x0944] = 0x35c3, [0x0945] = 0x5586, [0x0946] = 0x35bd, [0x0947] = 0x35d0,
  [0x0948] = 0x55af, [0x0949] = 0x557a, [0x094a] = 0x35c1, [0x094b] = 0x35be,
  [0x094c] = 0x35cd, [0x094d] = 0x559e, [0x094e] = 0x35cb, [0x094f] = 0x55a9,
  [0x0950] = 0x570f, [0x0951] = 0x570e, [0x0952] = 0x581a, [0x0953] = 0x364f,
  [0x0954] = 0x581f, [0x0955] = 0x3653, [0x0956] = 0x583c, [0x0957] = 0x5818,
  [0x0958] = 0x583e, [0x0959] = 0x5826, [0x095a] = 0x3655, [0x095b] = 0x583a,
  [0x095c] = 0x21364, [0x095d] = 0x5822, [0x095e] = 0x3651, [0x095f] = 0x58fb,
  [0x0960] = 0x5963, [0x0961] = 0x5964, [0x0962] = 0x369f, [0x0963] = 0x5aa8,
  [0x0964] = 0x5aa3, [0x0965] = 0x5a82, [0x0966] = 0x5a88, [0x0967] = 0x5aa1,
  [0x0968] = 0x5a85, [0x0969] = 0x5a98, [0x096a] = 0x36fe, [0x096b] = 0x5a99,
  [0x096c] = 0x36fb, [0x096d] = 0x5a89, [0x096e] = 0x5a81, [0x096f] = 0x5a96,
  [0x0970] = 0x5a80, [0x0971] = 0x36f1, [0x0972] = 0x36f5, [0x0973] = 0x5a91,
  [0x0974] = 0x36ef, [0x0975] = 0x3704, [0x0976] = 0x3703, [0x0977] = 0x36f4,
  [0x0978] = 0x5acf, [0x0979] = 0x36f3, [0x097a] = 0x3702, [0x097b] = 0x36f7,
  [0x097c] = 0x36fa, [0x097d] = 0x36fd, [0x097e] = 0x36ee, [0x097f] = 0x5a87,
  [0x0980] = 0x5aa0, [0x0981] = 0x36f0, [0x0982] = 0x5a79, [0x0983] = 0x36f2,
  [0x0984] = 0x5a86, [0x0985] = 0x5aab, [0x0986] = 0x5aaa, [0x0987] = 0x5aa4,
  [0x0988] = 0x5a8d, [0x0989] = 0x5a7e, [0x098a] = 0x3744, [0x098b] = 0x5bd5,
  [0x098c] = 0x3762, [0x098d] = 0x3777, [0x098e] = 0x3dc9, [0x098f] = 0x5c1e,
  [0x0990] = 0x5c5f, [0x0991] = 0x5c5e, [0x0992] = 0x5d44, [0x0993] = 0x5d3e,
  [0x0994] = 0x37e8, [0x0995] = 0x5d48, [0x0996] = 0x5d1c, [0x0997] = 0x37ef,
  [0x0998] = 0x5d5b, [0x0999] = 0x5d4d, [0x099a] = 0x37e6, [0x099b] = 0x37ed,
  [0x099c] = 0x5d57, [0x099d] = 0x37e7, [0x099e] = 0x5d53, [0x099f] = 0x5d4f,
  [0x09a0] = 0x37eb, [0x09a1] = 0x5d3b, [0x09a2] = 0x5d46, [0x09a3] = 0x382d,
  [0x09a4] = 0x3855, [0x09a5] = 0x5e46, [0x09a6] = 0x5e47, [0x09a7] = 0x3853,
  [0x09a8] = 0x5e48, [0x09a9] = 0x5ec0, [0x09aa] = 0x5ebd, [0x09ab] = 0x5ebf,
  [0x09ac] = 0x3890, [0x09ad] = 0x5f11, [0x09ae] = 0x38be, [0x09af] = 0x5f3e,
  [0x09b0] = 0x5f3b, [0x09b1] = 0x38bd, [0x09b2] = 0x5f3a, [0x09b3] = 0x38cf,
  [0x09b4] = 0x38d0, [0x09b5] = 0x38ec, [0x09b6] = 0x5fa7, [0x09b7] = 0x394b,
  [0x09b8] = 0x60ea, [0x09b9] = 0x3948, [0x09ba] = 0x6107, [0x09bb] = 0x6122,
  [0x09bc] = 0x610c, [0x09bd] = 0x3955, [0x09be] = 0x3951, [0x09bf] = 0x60b3,
  [0x09c0] = 0x60d6, [0x09c1] = 0x60d2, [0x09c2] = 0x394e, [0x09c3] = 0x60e3,
  [0x09c4] = 0x60e5, [0x09c5] = 0x60e9, [0x09c6] = 0x396b, [0x09c7] = 0x395e,
  [0x09c8] = 0x6111, [0x09c9] = 0x60fd, [0x09ca] = 0x3960, [0x09cb] = 0x3967,
  [0x09cc] = 0x611e, [0x09cd] = 0x6120, [0x09ce] = 0x6121, [0x09cf] = 0x621e,
  [0x09d0] = 0x39b8, [0x09d1] = 0x63e2, [0x09d2] = 0x63de, [0x09d3] = 0x63e6,
  [0x09d4] = 0x3a14, [0x09d5] = 0x3a0f, [0x09d6] = 0x3a07, [0x09d7] = 0x3a13,
  [0x09d8] = 0x63f8, [0x09d9] = 0x3a17, [0x09da] = 0x63fe, [0x09db] = 0x63c1,
  [0x09dc] = 0x63bf, [0x09dd] = 0x63f7, [0x09de] = 0x63d1, [0x09df] = 0x655f,
  [0x09e0] = 0x6560, [0x09e1] = 0x6561, [0x09e2] = 0x3a9a, [0x09e3] = 0x3ab8,
  [0x09e4] = 0x65d1, [0x09e5] = 0x3af7, [0x09e6] = 0x3af8, [0x09e7] = 0x667d,
  [0x09e8] = 0x666b, [0x09e9] = 0x667f, [0x09ea] = 0x3afd, [0x09eb] = 0x3af5,
  [0x09ec] = 0x6673, [0x09ed] = 0x6681, [0x09ee] = 0x666d, [0x09ef] = 0x6669,
  [0x09f0] = 0x3afa, [0x09f1] = 0x3b38, [0x09f2] = 0x671e, [0x09f3] = 0x68ed,
  [0x09f4] = 0x3b87, [0x09f5] = 0x3b80, [0x09f6] = 0x3b88, [0x09f7] = 0x3b79,
  [0x09f8] = 0x6903, [0x09f9] = 0x3b7c, [0x09fa] = 0x68fe, [0x09fb] = 0x68e5,
  [0x09fc] = 0x691e, [0x09fd] = 0x6902, [0x09fe] = 0x3b83, [0x09ff] = 0x3b85,
  [0x0a00] = 0x6909, [0x0a01] = 0x68ca, [0x0a02] = 0x6900, [0x0a03] = 0x2f8e5,
  [0x0a04] = 0x6901, [0x0a05] = 0x6918, [0x0a06] = 0x68e2, [0x0a07] = 0x68cf,
  [0x0a08] = 0x3b7b, [0x0a09] = 0x692e, [0x0a0a] = 0x68c5, [0x0a0b] = 0x68ff,
  [0x0a0c] = 0x3b86, [0x0a0d] = 0x691c, [0x0a0e] = 0x68c3, [0x0a0f] = 0x3c34,
  [0x0a10] = 0x6b6f, [0x0a11] = 0x3c55, [0x0a12] = 0x6b6e, [0x0a13] = 0x3c68,
  [0x0a14] = 0x6bbe, [0x0a15] = 0x3c9c, [0x0a16] = 0x6bf4, [0x0a17] = 0x6c2d,
  [0x0a18] = 0x3cfc, [0x0a19] = 0x6db6, [0x0a1a] = 0x6e75, [0x0a1b] = 0x6e1e,
  [0x0a1c] = 0x3d1a, [0x0a1d] = 0x6e18, [0x0a1e] = 0x3d17, [0x0a1f] = 0x6e48,
  [0x0a20] = 0x3d1b, [0x0a21] = 0x6e4f, [0x0a22] = 0x3d13, [0x0a23] = 0x6e42,
  [0x0a24] = 0x6e6a, [0x0a25] = 0x6e70, [0x0a26] = 0x6dfe, [0x0a27] = 0x3d05,
  [0x0a28] = 0x3d07, [0x0a29] = 0x6e6d, [0x0a2a] = 0x3d1c, [0x0a2b] = 0x6e7b,
  [0x0a2c] = 0x6e7e, [0x0a2d] = 0x6e59, [0x0a2e] = 0x3d11, [0x0a2f] = 0x6e57,
  [0x0a30] = 0x3d16, [0x0a31] = 0x6e80, [0x0a32] = 0x6e50, [0x0a33] = 0x3d15,
  [0x0a34] = 0x6e29, [0x0a35] = 0x6e76, [0x0a36] = 0x6e2a, [0x0a37] = 0x6e4c,
  [0x0a38] = 0x712a, [0x0a39] = 0x3dcb, [0x0a3a] = 0x7135, [0x0a3b] = 0x712c,
  [0x0a3c] = 0x7137, [0x0a3d] = 0x711d, [0x0a3e] = 0x3dc5, [0x0a3f] = 0x3dc2,
  [0x0a40] = 0x7138, [0x0a41] = 0x3dcd, [0x0a42] = 0x7134, [0x0a43] = 0x712b,
  [0x0a44] = 0x7133, [0x0a45] = 0x7127, [0x0a46] = 0x7124, [0x0a47] = 0x3dca,
  [0x0a48] = 0x712d, [0x0a49] = 0x7232, [0x0a4a] = 0x7283, [0x0a4b] = 0x7282,
  [0x0a4c] = 0x7287, [0x0a4d] = 0x7306, [0x0a4e] = 0x7324, [0x0a4f] = 0x7338,
  [0x0a50] = 0x732a, [0x0a51] = 0x732c, [0x0a52] = 0x732b, [0x0a53] = 0x3e83,
  [0x0a54] = 0x732f, [0x0a55] = 0x7328, [0x0a56] = 0x7417, [0x0a57] = 0x3ed6,
  [0x0a58] = 0x3ed5, [0x0a59] = 0x7419, [0x0a5a] = 0x7438, [0x0a5b] = 0x3ed1,
  [0x0a5c] = 0x741f, [0x0a5d] = 0x7414, [0x0a5e] = 0x743c, [0x0a5f] = 0x73f7,
  [0x0a60] = 0x741c, [0x0a61] = 0x7415, [0x0a62] = 0x7418, [0x0a63] = 0x7439,
  [0x0a64] = 0x74f9, [0x0a65] = 0x7524, [0x0a66] = 0x2f934, [0x0a67] = 0x3f52,
  [0x0a68] = 0x3f5f, [0x0a69] = 0x756e, [0x0a6a] = 0x756d, [0x0a6b] = 0x7571,
  [0x0a6c] = 0x758e, [0x0a6d] = 0x3f95, [0x0a6e] = 0x75e5, [0x0a6f] = 0x3f9d,
  [0x0a70] = 0x3f98, [0x0a71] = 0x3f9e, [0x0a72] = 0x3f96, [0x0a73] = 0x7694,
  [0x0a74] = 0x76b3, [0x0a75] = 0x4003, [0x0a76] = 0x76d9, [0x0a77] = 0x402f,
  [0x0a78] = 0x7748, [0x0a79] = 0x7749, [0x0a7a] = 0x7743, [0x0a7b] = 0x4031,
  [0x0a7c] = 0x4033, [0x0a7d] = 0x7742, [0x0a7e] = 0x77df, [0x0a7f] = 0x40b4,
  [0x0a80] = 0x7863, [0x0a81] = 0x7876, [0x0a82] = 0x40b0, [0x0a83] = 0x785f,
  [0x0a84] = 0x7866, [0x0a85] = 0x7966, [0x0a86] = 0x7971, [0x0a87] = 0x4108,
  [0x0a88] = 0x4107, [0x0a89] = 0x7976, [0x0a8a] = 0x7984, [0x0a8b] = 0x7975,
  [0x0a8c] = 0x79ff, [0x0a8d] = 0x7a07, [0x0a8e] = 0x414e, [0x0a8f] = 0x7a0e,
  [0x0a90] = 0x7a09, [0x0a91] = 0x4150, [0x0a92] = 0x4152, [0x0a93] = 0x41a1,
  [0x0a94] = 0x41a3, [0x0a95] = 0x41a5, [0x0a96] = 0x41cc, [0x0a97] = 0x7ae7,
  [0x0a98] = 0x7ae2, [0x0a99] = 0x7b55, [0x0a9a] = 0x41ef, [0x0a9b] = 0x41ea,
  [0x0a9c] = 0x7b43, [0x0a9d] = 0x7b57, [0x0a9e] = 0x7b6c, [0x0a9f] = 0x7b42,
  [0x0aa0] = 0x7b53, [0x0aa1] = 0x41ed, [0x0aa2] = 0x7b41, [0x0aa3] = 0x4285,
  [0x0aa4] = 0x4284, [0x0aa5] = 0x7ca7, [0x0aa6] = 0x7ca0, [0x0aa7] = 0x7ca6,
  [0x0aa8] = 0x7ca4, [0x0aa9] = 0x7d74, [0x0aaa] = 0x42db, [0x0aab] = 0x7d59,
  [0x0aac] = 0x42d9, [0x0aad] = 0x7d60, [0x0aae] = 0x7d57, [0x0aaf] = 0x7d6c,
  [0x0ab0] = 0x7d7e, [0x0ab1] = 0x7d64, [0x0ab2] = 0x42d7, [0x0ab3] = 0x7d5a,
  [0x0ab4] = 0x7d5d, [0x0ab5] = 0x42da, [0x0ab6] = 0x42de, [0x0ab7] = 0x42d8,
  [0x0ab8] = 0x7d76, [0x0ab9] = 0x7d4d, [0x0aba] = 0x7d75, [0x0abb] = 0x42d5,
  [0x0abc] = 0x7fd3, [0x0abd] = 0x7fd6, [0x0abe] = 0x439c, [0x0abf] = 0x439d,
  [0x0ac0] = 0x8060, [0x0ac1] = 0x804e, [0x0ac2] = 0x8145, [0x0ac3] = 0x813b,
  [0x0ac4] = 0x43fe, [0x0ac5] = 0x8148, [0x0ac6] = 0x8142, [0x0ac7] = 0x8149,
  [0x0ac8] = 0x8140, [0x0ac9] = 0x8114, [0x0aca] = 0x8141, [0x0acb] = 0x4407,
  [0x0acc] = 0x81ef, [0x0acd] = 0x81f6, [0x0ace] = 0x8203, [0x0acf] = 0x446a,
  [0x0ad0] = 0x83ed, [0x0ad1] = 0x44e7, [0x0ad2] = 0x83da, [0x0ad3] = 0x8418,
  [0x0ad4] = 0x83d2, [0x0ad5] = 0x8408, [0x0ad6] = 0x44e2, [0x0ad7] = 0x8400,
  [0x0ad8] = 0x44df, [0x0ad9] = 0x44e1, [0x0ada] = 0x44e5, [0x0adb] = 0x8417,
  [0x0adc] = 0x8346, [0x0add] = 0x8414, [0x0ade] = 0x83d3, [0x0adf] = 0x8405,
  [0x0ae0] = 0x841f, [0x0ae1] = 0x8402, [0x0ae2] = 0x8416, [0x0ae3] = 0x83cd,
  [0x0ae4] = 0x83e6, [0x0ae5] = 0x4591, [0x0ae6] = 0x865d, [0x0ae7] = 0x86d5,
  [0x0ae8] = 0x86e1, [0x0ae9] = 0x45b4, [0x0aea] = 0x45b0, [0x0aeb] = 0x45b5,
  [0x0aec] = 0x45ae, [0x0aed] = 0x86ee, [0x0aee] = 0x8847, [0x0aef] = 0x8846,
  [0x0af0] = 0x462d, [0x0af1] = 0x462c, [0x0af2] = 0x88bb, [0x0af3] = 0x462b,
  [0x0af4] = 0x88bf, [0x0af5] = 0x88b4, [0x0af6] = 0x4629, [0x0af7] = 0x88b5,
  [0x0af8] = 0x467f, [0x0af9] = 0x899a, [0x0afa] = 0x8a43, [0x0afb] = 0x46c9,
  [0x0afc] = 0x46cb, [0x0afd] = 0x8a5a, [0x0afe] = 0x46c5, [0x0aff] = 0x46c6,
  [0x0b00] = 0x46ca, [0x0b01] = 0x8a35, [0x0b02] = 0x8a38, [0x0b03] = 0x8a42,
  [0x0b04] = 0x8a49, [0x0b05] = 0x8a5d, [0x0b06] = 0x8a4b, [0x0b07] = 0x8a3d,
  [0x0b08] = 0x46d2, [0x0b09] = 0x46d0, [0x0b0a] = 0x472d, [0x0b0b] = 0x4735,
  [0x0b0c] = 0x8c60, [0x0b0d] = 0x8c5e, [0x0b0e] = 0x8c7f, [0x0b0f] = 0x8c7e,
  [0x0b10] = 0x8c83, [0x0b11] = 0x476c, [0x0b12] = 0x8cb1, [0x0b13] = 0x8d87,
  [0x0b14] = 0x479d, [0x0b15] = 0x47a0, [0x0b16] = 0x8d88, [0x0b17] = 0x8d83,
  [0x0b18] = 0x47a2, [0x0b19] = 0x479f, [0x0b1a] = 0x8d86, [0x0b1b] = 0x8d8b,
  [0x0b1c] = 0x8d82, [0x0b1d] = 0x8dca, [0x0b1e] = 0x8dd2, [0x0b1f] = 0x47eb,
  [0x0b20] = 0x47e2, [0x0b21] = 0x8dd4, [0x0b22] = 0x8dc9, [0x0b23] = 0x8eb0,
  [0x0b24] = 0x4836, [0x0b25] = 0x4832, [0x0b26] = 0x4850, [0x0b27] = 0x8ef2,
  [0x0b28] = 0x8ee4, [0x0b29] = 0x8ef3, [0x0b2a] = 0x8eea, [0x0b2b] = 0x484f,
  [0x0b2c] = 0x8efd, [0x0b2d] = 0x4852, [0x0b2e] = 0x8f9d, [0x0b2f] = 0x902b,
  [0x0b30] = 0x902a, [0x0b31] = 0x489e, [0x0b32] = 0x9028, [0x0b33] = 0x9029,
  [0x0b34] = 0x902c, [0x0b35] = 0x48a0, [0x0b36] = 0x489c, [0x0b37] = 0x903a,
  [0x0b38] = 0x9030, [0x0b39] = 0x9037, [0x0b3a] = 0x903b, [0x0b3b] = 0x48d1,
  [0x0b3c] = 0x910a, [0x0b3d] = 0x48ef, [0x0b3e] = 0x48f0, [0x0b3f] = 0x48f1,
  [0x0b40] = 0x91fe, [0x0b41] = 0x9220, [0x0b42] = 0x491d, [0x0b43] = 0x920b,
  [0x0b44] = 0x491f, [0x0b45] = 0x9218, [0x0b46] = 0x9222, [0x0b47] = 0x491e,
  [0x0b48] = 0x921b, [0x0b49] = 0x9208, [0x0b4a] = 0x4920, [0x0b4b] = 0x920e,
  [0x0b4c] = 0x9213, [0x0b4d] = 0x498e, [0x0b4e] = 0x4991, [0x0b4f] = 0x9595,
  [0x0b50] = 0x28cdd, [0x0b51] = 0x4990, [0x0b52] = 0x49d7, [0x0b53] = 0x968c,
  [0x0b54] = 0x967b, [0x0b55] = 0x967f, [0x0b56] = 0x9681, [0x0b57] = 0x49d9,
  [0x0b58] = 0x9682, [0x0b59] = 0x49f4, [0x0b5a] = 0x49f6, [0x0b5b] = 0x3560,
  [0x0b5c] = 0x49f5, [0x0b5d] = 0x49f3, [0x0b5e] = 0x96ee, [0x0b5f] = 0x96ed,
  [0x0b60] = 0x4a0c, [0x0b61] = 0x96ec, [0x0b62] = 0x975f, [0x0b63] = 0x976f,
  [0x0b64] = 0x4a51, [0x0b65] = 0x976d, [0x0b66] = 0x4aa6, [0x0b67] = 0x4aa7,
  [0x0b68] = 0x4aa8, [0x0b69] = 0x4b27, [0x0b6a] = 0x4b24, [0x0b6b] = 0x4b25,
  [0x0b6c] = 0x98f0, [0x0b6d] = 0x4b2a, [0x0b6e] = 0x4b74, [0x0b6f] = 0x4bc7,
  [0x0b70] = 0x9aa9, [0x0b71] = 0x4be7, [0x0b72] = 0x4bed, [0x0b73] = 0x9ae0,
  [0x0b74] = 0x4eb7, [0x0b75] = 0x342e, [0x0b76] = 0x347b, [0x0b77] = 0x50cc,
  [0x0b78] = 0x50bc, [0x0b79] = 0x347c, [0x0b7a] = 0x50aa, [0x0b7b] = 0x50b9,
  [0x0b7c] = 0x347d, [0x0b7d] = 0x50ab, [0x0b7e] = 0x50c3, [0x0b7f] = 0x50cd,
  [0x0b80] = 0x517e, [0x0b81] = 0x527e, [0x0b82] = 0x5279, [0x0b83] = 0x34fd,
  [0x0b84] = 0x2f823, [0x0b85] = 0x52e1, [0x0b86] = 0x52e0, [0x0b87] = 0x52e7,
  [0x0b88] = 0x5380, [0x0b89] = 0x53ab, [0x0b8a] = 0x53aa, [0x0b8b] = 0x53a9,
  [0x0b8c] = 0x53e0, [0x0b8d] = 0x55ea, [0x0b8e] = 0x35da, [0x0b8f] = 0x55d7,
  [0x0b90] = 0x35d6, [0x0b91] = 0x35db, [0x0b92] = 0x55c1, [0x0b93] = 0x5715,
  [0x0b94] = 0x365b, [0x0b95] = 0x586c, [0x0b96] = 0x365c, [0x0b97] = 0x585c,
  [0x0b98] = 0x5850, [0x0b99] = 0x5861, [0x0b9a] = 0x586a, [0x0b9b] = 0x5869,
  [0x0b9c] = 0x5856, [0x0b9d] = 0x5860, [0x0b9e] = 0x5866, [0x0b9f] = 0x585f,
  [0x0ba0] = 0x5923, [0x0ba1] = 0x5966, [0x0ba2] = 0x5968, [0x0ba3] = 0x3706,
  [0x0ba4] = 0x370b, [0x0ba5] = 0x5ace, [0x0ba6] = 0x370d, [0x0ba7] = 0x5ac5,
  [0x0ba8] = 0x5ac3, [0x0ba9] = 0x370a, [0x0baa] = 0x3713, [0x0bab] = 0x5ad0,
  [0x0bac] = 0x3710, [0x0bad] = 0x3712, [0x0bae] = 0x3709, [0x0baf] = 0x3708,
  [0x0bb0] = 0x3711, [0x0bb1] = 0x370f, [0x0bb2] = 0x5b74, [0x0bb3] = 0x5b76,
  [0x0bb4] = 0x5bdc, [0x0bb5] = 0x5bd7, [0x0bb6] = 0x5bda, [0x0bb7] = 0x5bdb,
  [0x0bb8] = 0x3767, [0x0bb9] = 0x5c20, [0x0bba] = 0x5d6d, [0x0bbb] = 0x5d66,
  [0x0bbc] = 0x37f6, [0x0bbd] = 0x5d64, [0x0bbe] = 0x5d6e, [0x0bbf] = 0x2f87e,
  [0x0bc0] = 0x5d60, [0x0bc1] = 0x5f42, [0x0bc2] = 0x5f5a, [0x0bc3] = 0x5f6e,
  [0x0bc4] = 0x3964, [0x0bc5] = 0x396c, [0x0bc6] = 0x6130, [0x0bc7] = 0x613a,
  [0x0bc8] = 0x612a, [0x0bc9] = 0x6143, [0x0bca] = 0x6119, [0x0bcb] = 0x6131,
  [0x0bcc] = 0x396d, [0x0bcd] = 0x613d, [0x0bce] = 0x397a, [0x0bcf] = 0x3975,
  [0x0bd0] = 0x3a0d, [0x0bd1] = 0x6408, [0x0bd2] = 0x6432, [0x0bd3] = 0x6438,
  [0x0bd4] = 0x3a1e, [0x0bd5] = 0x6431, [0x0bd6] = 0x3a1b, [0x0bd7] = 0x6419,
  [0x0bd8] = 0x3a2a, [0x0bd9] = 0x6411, [0x0bda] = 0x3a1f, [0x0bdb] = 0x3a22,
  [0x0bdc] = 0x6429, [0x0bdd] = 0x641d, [0x0bde] = 0x3a25, [0x0bdf] = 0x3a27,
  [0x0be0] = 0x3a29, [0x0be1] = 0x643c, [0x0be2] = 0x3a24, [0x0be3] = 0x6446,
  [0x0be4] = 0x6447, [0x0be5] = 0x3a28, [0x0be6] = 0x3a26, [0x0be7] = 0x643a,
  [0x0be8] = 0x6407, [0x0be9] = 0x3a23, [0x0bea] = 0x656b, [0x0beb] = 0x3a9f,
  [0x0bec] = 0x6570, [0x0bed] = 0x656d, [0x0bee] = 0x3ab1, [0x0bef] = 0x65e4,
  [0x0bf0] = 0x6693, [0x0bf1] = 0x3b03, [0x0bf2] = 0x3b07, [0x0bf3] = 0x3b0c,
  [0x0bf4] = 0x3b06, [0x0bf5] = 0x668f, [0x0bf6] = 0x3b04, [0x0bf7] = 0x3b09,
  [0x0bf8] = 0x6692, [0x0bf9] = 0x3b05, [0x0bfa] = 0x668e, [0x0bfb] = 0x3b08,
  [0x0bfc] = 0x6946, [0x0bfd] = 0x3b96, [0x0bfe] = 0x3b9c, [0x0bff] = 0x3b9f,
  [0x0c00] = 0x3b9b, [0x0c01] = 0x3b98, [0x0c02] = 0x3b99, [0x0c03] = 0x3b94,
  [0x0c04] = 0x6931, [0x0c05] = 0x3b8d, [0x0c06] = 0x3ba3, [0x0c07] = 0x693e,
  [0x0c08] = 0x3b93, [0x0c09] = 0x697c, [0x0c0a] = 0x6943, [0x0c0b] = 0x3b92,
  [0x0c0c] = 0x6973, [0x0c0d] = 0x2f8e8, [0x0c0e] = 0x6955, [0x0c0f] = 0x3b8e,
  [0x0c10] = 0x3b8c, [0x0c11] = 0x6985, [0x0c12] = 0x694d, [0x0c13] = 0x6950,
  [0x0c14] = 0x6947, [0x0c15] = 0x6967, [0x0c16] = 0x6936, [0x0c17] = 0x6964,
  [0x0c18] = 0x6961, [0x0c19] = 0x3b9a, [0x0c1a] = 0x697d, [0x0c1b] = 0x6b44,
  [0x0c1c] = 0x6b40, [0x0c1d] = 0x6b71, [0x0c1e] = 0x6b73, [0x0c1f] = 0x6b9c,
  [0x0c20] = 0x3c6a, [0x0c21] = 0x3c6d, [0x0c22] = 0x3c84, [0x0c23] = 0x6bc1,
  [0x0c24] = 0x3ca0, [0x0c25] = 0x6bfa, [0x0c26] = 0x6c31, [0x0c27] = 0x6c32,
  [0x0c28] = 0x3d1d, [0x0c29] = 0x3d26, [0x0c2a] = 0x6eb8, [0x0c2b] = 0x6ea8,
  [0x0c2c] = 0x3d33, [0x0c2d] = 0x6e91, [0x0c2e] = 0x6ebb, [0x0c2f] = 0x3d38,
  [0x0c30] = 0x6e9a, [0x0c31] = 0x3d30, [0x0c32] = 0x3d28, [0x0c33] = 0x6ea9,
  [0x0c34] = 0x3d27, [0x0c35] = 0x3d2a, [0x0c36] = 0x6eb5, [0x0c37] = 0x6e6c,
  [0x0c38] = 0x6ee8, [0x0c39] = 0x3d31, [0x0c3a] = 0x6edd, [0x0c3b] = 0x6eda,
  [0x0c3c] = 0x6ee6, [0x0c3d] = 0x6eac, [0x0c3e] = 0x3d34, [0x0c3f] = 0x3d2e,
  [0x0c40] = 0x3d3b, [0x0c41] = 0x6ed9, [0x0c42] = 0x6ee3, [0x0c43] = 0x6ee9,
  [0x0c44] = 0x6edb, [0x0c45] = 0x3d29, [0x0c46] = 0x716f, [0x0c47] = 0x3dd2,
  [0x0c48] = 0x3dd8, [0x0c49] = 0x7148, [0x0c4a] = 0x3dcf, [0x0c4b] = 0x714a,
  [0x0c4c] = 0x716b, [0x0c4d] = 0x3dd9, [0x0c4e] = 0x714f, [0x0c4f] = 0x7157,
  [0x0c50] = 0x7174, [0x0c51] = 0x3dce, [0x0c52] = 0x3dd3, [0x0c53] = 0x3dd0,
  [0x0c54] = 0x7145, [0x0c55] = 0x7151, [0x0c56] = 0x716d, [0x0c57] = 0x3ba1,
  [0x0c58] = 0x7251, [0x0c59] = 0x7250, [0x0c5a] = 0x724e, [0x0c5b] = 0x3e47,
  [0x0c5c] = 0x7341, [0x0c5d] = 0x3e8b, [0x0c5e] = 0x732e, [0x0c5f] = 0x7346,
  [0x0c60] = 0x3ed4, [0x0c61] = 0x7427, [0x0c62] = 0x3ede, [0x0c63] = 0x7448,
  [0x0c64] = 0x7453, [0x0c65] = 0x743d, [0x0c66] = 0x3edf, [0x0c67] = 0x745d,
  [0x0c68] = 0x7456, [0x0c69] = 0x3ed7, [0x0c6a] = 0x741e, [0x0c6b] = 0x7447,
  [0x0c6c] = 0x7443, [0x0c6d] = 0x7458, [0x0c6e] = 0x7449, [0x0c6f] = 0x3ee1,
  [0x0c70] = 0x744c, [0x0c71] = 0x7445, [0x0c72] = 0x743e, [0x0c73] = 0x3f2f,
  [0x0c74] = 0x7501, [0x0c75] = 0x751e, [0x0c76] = 0x3f62, [0x0c77] = 0x3f63,
  [0x0c78] = 0x757a, [0x0c79] = 0x75ee, [0x0c7a] = 0x7602, [0x0c7b] = 0x7697,
  [0x0c7c] = 0x7698, [0x0c7d] = 0x3fe2, [0x0c7e] = 0x4004, [0x0c7f] = 0x4043,
  [0x0c80] = 0x775d, [0x0c81] = 0x7764, [0x0c82] = 0x7753, [0x0c83] = 0x7758,
  [0x0c84] = 0x7882, [0x0c85] = 0x7890, [0x0c86] = 0x788a, [0x0c87] = 0x40be,
  [0x0c88] = 0x787a, [0x0c89] = 0x787d, [0x0c8a] = 0x40ba, [0x0c8b] = 0x788b,
  [0x0c8c] = 0x7878, [0x0c8d] = 0x40bc, [0x0c8e] = 0x2f94e, [0x0c8f] = 0x788d,
  [0x0c90] = 0x7888, [0x0c91] = 0x7892, [0x0c92] = 0x7881, [0x0c93] = 0x797e,
  [0x0c94] = 0x7983, [0x0c95] = 0x410d, [0x0c96] = 0x410e, [0x0c97] = 0x4111,
  [0x0c98] = 0x7980, [0x0c99] = 0x410f, [0x0c9a] = 0x4112, [0x0c9b] = 0x4155,
  [0x0c9c] = 0x7a0f, [0x0c9d] = 0x4159, [0x0c9e] = 0x415b, [0x0c9f] = 0x7a1d,
  [0x0ca0] = 0x4157, [0x0ca1] = 0x7aa1, [0x0ca2] = 0x7aa4, [0x0ca3] = 0x41ce,
  [0x0ca4] = 0x7ae9, [0x0ca5] = 0x7aea, [0x0ca6] = 0x41fe, [0x0ca7] = 0x7b62,
  [0x0ca8] = 0x7b6b, [0x0ca9] = 0x41fc, [0x0caa] = 0x7b5e, [0x0cab] = 0x41f5,
  [0x0cac] = 0x7b79, [0x0cad] = 0x41f9, [0x0cae] = 0x41fa, [0x0caf] = 0x7b6f,
  [0x0cb0] = 0x7b68, [0x0cb1] = 0x4288, [0x0cb2] = 0x4289, [0x0cb3] = 0x7cae,
  [0x0cb4] = 0x428a, [0x0cb5] = 0x4287, [0x0cb6] = 0x428b, [0x0cb7] = 0x7cb0,
  [0x0cb8] = 0x42e6, [0x0cb9] = 0x7d90, [0x0cba] = 0x42ed, [0x0cbb] = 0x7d8a,
  [0x0cbc] = 0x42e5, [0x0cbd] = 0x7d8b, [0x0cbe] = 0x7d99, [0x0cbf] = 0x7d95,
  [0x0cc0] = 0x42e0, [0x0cc1] = 0x7d87, [0x0cc2] = 0x7d78, [0x0cc3] = 0x7d97,
  [0x0cc4] = 0x7d89, [0x0cc5] = 0x7d98, [0x0cc6] = 0x42e1, [0x0cc7] = 0x435b,
  [0x0cc8] = 0x435c, [0x0cc9] = 0x7fa3, [0x0cca] = 0x438f, [0x0ccb] = 0x438b,
  [0x0ccc] = 0x438d, [0x0ccd] = 0x7fdd, [0x0cce] = 0x8057, [0x0ccf] = 0x43b9,
  [0x0cd0] = 0x8163, [0x0cd1] = 0x816a, [0x0cd2] = 0x816c, [0x0cd3] = 0x440f,
  [0x0cd4] = 0x4419, [0x0cd5] = 0x4413, [0x0cd6] = 0x815d, [0x0cd7] = 0x8175,
  [0x0cd8] = 0x4418, [0x0cd9] = 0x815f, [0x0cda] = 0x4416, [0x0cdb] = 0x817d,
  [0x0cdc] = 0x816d, [0x0cdd] = 0x4453, [0x0cde] = 0x2f98d, [0x0cdf] = 0x8241,
  [0x0ce0] = 0x844f, [0x0ce1] = 0x8484, [0x0ce2] = 0x44f6, [0x0ce3] = 0x847f,
  [0x0ce4] = 0x44f5, [0x0ce5] = 0x8448, [0x0ce6] = 0x842a, [0x0ce7] = 0x847b,
  [0x0ce8] = 0x8472, [0x0ce9] = 0x8464, [0x0cea] = 0x842e, [0x0ceb] = 0x845c,
  [0x0cec] = 0x8453, [0x0ced] = 0x44f7, [0x0cee] = 0x8441, [0x0cef] = 0x84c8,
  [0x0cf0] = 0x44f0, [0x0cf1] = 0x8462, [0x0cf2] = 0x8480, [0x0cf3] = 0x843e,
  [0x0cf4] = 0x8483, [0x0cf5] = 0x8471, [0x0cf6] = 0x44f9, [0x0cf7] = 0x844a,
  [0x0cf8] = 0x8455, [0x0cf9] = 0x8458, [0x0cfa] = 0x4592, [0x0cfb] = 0x4595,
  [0x0cfc] = 0x4596, [0x0cfd] = 0x86fc, [0x0cfe] = 0x86fd, [0x0cff] = 0x8715,
  [0x0d00] = 0x45b9, [0x0d01] = 0x8716, [0x0d02] = 0x86ff, [0x0d03] = 0x45bd,
  [0x0d04] = 0x45b8, [0x0d05] = 0x4612, [0x0d06] = 0x8858, [0x0d07] = 0x88cf,
  [0x0d08] = 0x88e0, [0x0d09] = 0x4680, [0x0d0a] = 0x4681, [0x0d0b] = 0x469a,
  [0x0d0c] = 0x4698, [0x0d0d] = 0x89e7, [0x0d0e] = 0x8a6a, [0x0d0f] = 0x8a80,
  [0x0d10] = 0x46d4, [0x0d11] = 0x8a6f, [0x0d12] = 0x8a65, [0x0d13] = 0x46da,
  [0x0d14] = 0x8a78, [0x0d15] = 0x8a7d, [0x0d16] = 0x8a88, [0x0d17] = 0x46d6,
  [0x0d18] = 0x46db, [0x0d19] = 0x8a64, [0x0d1a] = 0x8a7e, [0x0d1b] = 0x46dc,
  [0x0d1c] = 0x8a67, [0x0d1d] = 0x8c63, [0x0d1e] = 0x8c88, [0x0d1f] = 0x4771,
  [0x0d20] = 0x8ccd, [0x0d21] = 0x4772, [0x0d22] = 0x8cc9, [0x0d23] = 0x47a8,
  [0x0d24] = 0x8ded, [0x0d25] = 0x47f0, [0x0d26] = 0x2f9db, [0x0d27] = 0x47f1,
  [0x0d28] = 0x47fd, [0x0d29] = 0x4838, [0x0d2a] = 0x4837, [0x0d2b] = 0x4839,
  [0x0d2c] = 0x8eb1, [0x0d2d] = 0x4855, [0x0d2e] = 0x4853, [0x0d2f] = 0x8f04,
  [0x0d30] = 0x8f9e, [0x0d31] = 0x8fa0, [0x0d32] = 0x9043, [0x0d33] = 0x9046,
  [0x0d34] = 0x9048, [0x0d35] = 0x9045, [0x0d36] = 0x9040, [0x0d37] = 0x904c,
  [0x0d38] = 0x48d5, [0x0d39] = 0x48bd, [0x0d3a] = 0x910c, [0x0d3b] = 0x9113,
  [0x0d3c] = 0x9115, [0x0d3d] = 0x48f5, [0x0d3e] = 0x916b, [0x0d3f] = 0x9167,
  [0x0d40] = 0x925d, [0x0d41] = 0x9255, [0x0d42] = 0x9235, [0x0d43] = 0x4921,
  [0x0d44] = 0x9259, [0x0d45] = 0x922f, [0x0d46] = 0x923c, [0x0d47] = 0x928f,
  [0x0d48] = 0x925c, [0x0d49] = 0x926a, [0x0d4a] = 0x9262, [0x0d4b] = 0x925f,
  [0x0d4c] = 0x926b, [0x0d4d] = 0x926e, [0x0d4e] = 0x923b, [0x0d4f] = 0x9244,
  [0x0d50] = 0x9241, [0x0d51] = 0x959a, [0x0d52] = 0x4992, [0x0d53] = 0x9599,
  [0x0d54] = 0x49de, [0x0d55] = 0x49db, [0x0d56] = 0x49da, [0x0d57] = 0x968f,
  [0x0d58] = 0x49df, [0x0d59] = 0x9696, [0x0d5a] = 0x49f9, [0x0d5b] = 0x49f8,
  [0x0d5c] = 0x49fa, [0x0d5d] = 0x96f4, [0x0d5e] = 0x96fc, [0x0d5f] = 0x4a0e,
  [0x0d60] = 0x9755, [0x0d61] = 0x4a43, [0x0d62] = 0x9779, [0x0d63] = 0x4a56,
  [0x0d64] = 0x4a53, [0x0d65] = 0x4a9e, [0x0d66] = 0x97ee, [0x0d67] = 0x97f5,
  [0x0d68] = 0x4aa9, [0x0d69] = 0x980b, [0x0d6a] = 0x4afa, [0x0d6b] = 0x98f3,
  [0x0d6c] = 0x4b31, [0x0d6d] = 0x4b30, [0x0d6e] = 0x98f7, [0x0d6f] = 0x98ff,
  [0x0d70] = 0x98f5, [0x0d71] = 0x4b32, [0x0d72] = 0x98ec, [0x0d73] = 0x98f1,
  [0x0d74] = 0x4b29, [0x0d75] = 0x4b2e, [0x0d76] = 0x999a, [0x0d77] = 0x4b76,
  [0x0d78] = 0x9ae2, [0x0d79] = 0x9b3d, [0x0d7a] = 0x9b5d, [0x0d7b] = 0x9ce8,
  [0x0d7c] = 0x4ca5, [0x0d7d] = 0x9ceb, [0x0d7e] = 0x9cef, [0x0d7f] = 0x9cee,
  [0x0d80] = 0x9e81, [0x0d81] = 0x9f14, [0x0d82] = 0x50d0, [0x0d83] = 0x50d9,
  [0x0d84] = 0x50dc, [0x0d85] = 0x50d8, [0x0d86] = 0x348c, [0x0d87] = 0x50e1,
  [0x0d88] = 0x50eb, [0x0d89] = 0x348b, [0x0d8a] = 0x3489, [0x0d8b] = 0x50f4,
  [0x0d8c] = 0x50e2, [0x0d8d] = 0x50de, [0x0d8e] = 0x348d, [0x0d8f] = 0x3486,
  [0x0d90] = 0x34d7, [0x0d91] = 0x51f4, [0x0d92] = 0x3504, [0x0d93] = 0x3507,
  [0x0d94] = 0x3503, [0x0d95] = 0x52ed, [0x0d96] = 0x52ea, [0x0d97] = 0x3522,
  [0x0d98] = 0x5332, [0x0d99] = 0x3551, [0x0d9a] = 0x53ae, [0x0d9b] = 0x53b0,
  [0x0d9c] = 0x3561, [0x0d9d] = 0x55fb, [0x0d9e] = 0x5603, [0x0d9f] = 0x560b,
  [0x0da0] = 0x35e9, [0x0da1] = 0x5607, [0x0da2] = 0x35e5, [0x0da3] = 0x55f8,
  [0x0da4] = 0x35e4, [0x0da5] = 0x5628, [0x0da6] = 0x561e, [0x0da7] = 0x35e3,
  [0x0da8] = 0x5618, [0x0da9] = 0x5611, [0x0daa] = 0x5651, [0x0dab] = 0x5605,
  [0x0dac] = 0x5717, [0x0dad] = 0x5892, [0x0dae] = 0x3665, [0x0daf] = 0x588c,
  [0x0db0] = 0x3663, [0x0db1] = 0x5878, [0x0db2] = 0x5884, [0x0db3] = 0x5873,
  [0x0db4] = 0x58ad, [0x0db5] = 0x5897, [0x0db6] = 0x5895, [0x0db7] = 0x5877,
  [0x0db8] = 0x5872, [0x0db9] = 0x5896, [0x0dba] = 0x588d, [0x0dbb] = 0x5910,
  [0x0dbc] = 0x368c, [0x0dbd] = 0x596c, [0x0dbe] = 0x371a, [0x0dbf] = 0x5ae7,
  [0x0dc0] = 0x3715, [0x0dc1] = 0x5ae4, [0x0dc2] = 0x3720, [0x0dc3] = 0x3721,
  [0x0dc4] = 0x5aef, [0x0dc5] = 0x5626, [0x0dc6] = 0x371c, [0x0dc7] = 0x371b,
  [0x0dc8] = 0x5af0, [0x0dc9] = 0x5d7b, [0x0dca] = 0x37fe, [0x0dcb] = 0x5d83,
  [0x0dcc] = 0x3804, [0x0dcd] = 0x3801, [0x0dce] = 0x5d8b, [0x0dcf] = 0x5d8c,
  [0x0dd0] = 0x3800, [0x0dd1] = 0x5d78, [0x0dd2] = 0x5e52, [0x0dd3] = 0x386d,
  [0x0dd4] = 0x3893, [0x0dd5] = 0x5ed0, [0x0dd6] = 0x5ecf, [0x0dd7] = 0x38a1,
  [0x0dd8] = 0x5fb3, [0x0dd9] = 0x5fb4, [0x0dda] = 0x3976, [0x0ddb] = 0x3979,
  [0x0ddc] = 0x3972, [0x0ddd] = 0x617b, [0x0dde] = 0x3983, [0x0ddf] = 0x616f,
  [0x0de0] = 0x6181, [0x0de1] = 0x613c, [0x0de2] = 0x6142, [0x0de3] = 0x6138,
  [0x0de4] = 0x6133, [0x0de5] = 0x2f8a6, [0x0de6] = 0x6160, [0x0de7] = 0x6169,
  [0x0de8] = 0x617d, [0x0de9] = 0x6186, [0x0dea] = 0x622c, [0x0deb] = 0x6228,
  [0x0dec] = 0x3a38, [0x0ded] = 0x644c, [0x0dee] = 0x3a30, [0x0def] = 0x6457,
  [0x0df0] = 0x647c, [0x0df1] = 0x3a34, [0x0df2] = 0x3a3a, [0x0df3] = 0x6455,
  [0x0df4] = 0x6462, [0x0df5] = 0x6471, [0x0df6] = 0x646a, [0x0df7] = 0x6456,
  [0x0df8] = 0x643b, [0x0df9] = 0x6481, [0x0dfa] = 0x3a35, [0x0dfb] = 0x644f,
  [0x0dfc] = 0x647e, [0x0dfd] = 0x6464, [0x0dfe] = 0x3a3f, [0x0dff] = 0x3a40,
  [0x0e00] = 0x3a32, [0x0e01] = 0x3a31, [0x0e02] = 0x3a36, [0x0e03] = 0x6571,
  [0x0e04] = 0x230ba, [0x0e05] = 0x3b0f, [0x0e06] = 0x66a5, [0x0e07] = 0x669a,
  [0x0e08] = 0x669c, [0x0e09] = 0x3b10, [0x0e0a] = 0x66a6, [0x0e0b] = 0x3b0d,
  [0x0e0c] = 0x66a4, [0x0e0d] = 0x698f, [0x0e0e] = 0x69c5, [0x0e0f] = 0x69c8,
  [0x0e10] = 0x6992, [0x0e11] = 0x69b2, [0x0e12] = 0x3ba9, [0x0e13] = 0x3bb4,
  [0x0e14] = 0x3bac, [0x0e15] = 0x69e3, [0x0e16] = 0x69c0, [0x0e17] = 0x69d6,
  [0x0e18] = 0x69d1, [0x0e19] = 0x699f, [0x0e1a] = 0x69a2, [0x0e1b] = 0x69d2,
  [0x0e1c] = 0x3bb8, [0x0e1d] = 0x3bae, [0x0e1e] = 0x235f3, [0x0e1f] = 0x69e1,
  [0x0e20] = 0x69d5, [0x0e21] = 0x699d, [0x0e22] = 0x3bb3, [0x0e23] = 0x3bba,
  [0x0e24] = 0x6998, [0x0e25] = 0x3c3f, [0x0e26] = 0x6b74, [0x0e27] = 0x6ba1,
  [0x0e28] = 0x3d3c, [0x0e29] = 0x6ef0, [0x0e2a] = 0x6ef3, [0x0e2b] = 0x3d42,
  [0x0e2c] = 0x3d40, [0x0e2d] = 0x6f1b, [0x0e2e] = 0x6f0c, [0x0e2f] = 0x6f1d,
  [0x0e30] = 0x6f34, [0x0e31] = 0x6f28, [0x0e32] = 0x6f17, [0x0e33] = 0x3d3e,
  [0x0e34] = 0x6f44, [0x0e35] = 0x6f42, [0x0e36] = 0x6f04, [0x0e37] = 0x6f11,
  [0x0e38] = 0x6efa, [0x0e39] = 0x6f4a, [0x0e3a] = 0x7191, [0x0e3b] = 0x718e,
  [0x0e3c] = 0x3de1, [0x0e3d] = 0x718b, [0x0e3e] = 0x718d, [0x0e3f] = 0x717f,
  [0x0e40] = 0x718c, [0x0e41] = 0x717e, [0x0e42] = 0x717c, [0x0e43] = 0x7183,
  [0x0e44] = 0x3de6, [0x0e45] = 0x7188, [0x0e46] = 0x3de0, [0x0e47] = 0x3e15,
  [0x0e48] = 0x7294, [0x0e49] = 0x3e93, [0x0e4a] = 0x7355, [0x0e4b] = 0x7353,
  [0x0e4c] = 0x734f, [0x0e4d] = 0x7354, [0x0e4e] = 0x746c, [0x0e4f] = 0x7465,
  [0x0e50] = 0x7466, [0x0e51] = 0x7461, [0x0e52] = 0x746b, [0x0e53] = 0x7468,
  [0x0e54] = 0x7476, [0x0e55] = 0x3ee7, [0x0e56] = 0x7460, [0x0e57] = 0x24a0f,
  [0x0e58] = 0x7474, [0x0e59] = 0x7506, [0x0e5a] = 0x760e, [0x0e5b] = 0x3fad,
  [0x0e5c] = 0x7607, [0x0e5d] = 0x3fae, [0x0e5e] = 0x3fe3, [0x0e5f] = 0x76b9,
  [0x0e60] = 0x3ff5, [0x0e61] = 0x76b7, [0x0e62] = 0x76e2, [0x0e63] = 0x4006,
  [0x0e64] = 0x7774, [0x0e65] = 0x7777, [0x0e66] = 0x7776, [0x0e67] = 0x7775,
  [0x0e68] = 0x404f, [0x0e69] = 0x7778, [0x0e6a] = 0x7771, [0x0e6b] = 0x4054,
  [0x0e6c] = 0x777a, [0x0e6d] = 0x715b, [0x0e6e] = 0x777b, [0x0e6f] = 0x78a6,
  [0x0e70] = 0x78ae, [0x0e71] = 0x78b8, [0x0e72] = 0x40cb, [0x0e73] = 0x40e3,
  [0x0e74] = 0x40c9, [0x0e75] = 0x78b1, [0x0e76] = 0x78af, [0x0e77] = 0x4113,
  [0x0e78] = 0x7989, [0x0e79] = 0x7987, [0x0e7a] = 0x4115, [0x0e7b] = 0x4161,
  [0x0e7c] = 0x7a29, [0x0e7d] = 0x4166, [0x0e7e] = 0x7a2a, [0x0e7f] = 0x4164,
  [0x0e80] = 0x7a2d, [0x0e81] = 0x7a2c, [0x0e82] = 0x4160, [0x0e83] = 0x7a32,
  [0x0e84] = 0x4163, [0x0e85] = 0x7aec, [0x0e86] = 0x7af0, [0x0e87] = 0x7b81,
  [0x0e88] = 0x7b9e, [0x0e89] = 0x7b83, [0x0e8a] = 0x420a, [0x0e8b] = 0x7b92,
  [0x0e8c] = 0x4204, [0x0e8d] = 0x7ba3, [0x0e8e] = 0x7b9f, [0x0e8f] = 0x7b93,
  [0x0e90] = 0x4207, [0x0e91] = 0x7b86, [0x0e92] = 0x7cb8, [0x0e93] = 0x7cb7,
  [0x0e94] = 0x428d, [0x0e95] = 0x428f, [0x0e96] = 0x4290, [0x0e97] = 0x4292,
  [0x0e98] = 0x42ec, [0x0e99] = 0x7dc8, [0x0e9a] = 0x7db6, [0x0e9b] = 0x2f96c,
  [0x0e9c] = 0x7dd1, [0x0e9d] = 0x42e7, [0x0e9e] = 0x7da8, [0x0e9f] = 0x7dab,
  [0x0ea0] = 0x42f2, [0x0ea1] = 0x7db3, [0x0ea2] = 0x7dcd, [0x0ea3] = 0x42ee,
  [0x0ea4] = 0x7dcf, [0x0ea5] = 0x7da4, [0x0ea6] = 0x42ef, [0x0ea7] = 0x434c,
  [0x0ea8] = 0x7f41, [0x0ea9] = 0x7f6f, [0x0eaa] = 0x7f71, [0x0eab] = 0x435e,
  [0x0eac] = 0x435f, [0x0ead] = 0x4376, [0x0eae] = 0x4374, [0x0eaf] = 0x4372,
  [0x0eb0] = 0x4390, [0x0eb1] = 0x8023, [0x0eb2] = 0x805b, [0x0eb3] = 0x43be,
  [0x0eb4] = 0x8061, [0x0eb5] = 0x805f, [0x0eb6] = 0x8181, [0x0eb7] = 0x4426,
  [0x0eb8] = 0x4425, [0x0eb9] = 0x8184, [0x0eba] = 0x8213, [0x0ebb] = 0x4474,
  [0x0ebc] = 0x824a, [0x0ebd] = 0x824c, [0x0ebe] = 0x44fd, [0x0ebf] = 0x4505,
  [0x0ec0] = 0x4501, [0x0ec1] = 0x84bd, [0x0ec2] = 0x8495, [0x0ec3] = 0x4509,
  [0x0ec4] = 0x8492, [0x0ec5] = 0x84c3, [0x0ec6] = 0x450c, [0x0ec7] = 0x8496,
  [0x0ec8] = 0x84a5, [0x0ec9] = 0x84b5, [0x0eca] = 0x84b3, [0x0ecb] = 0x84a3,
  [0x0ecc] = 0x84e4, [0x0ecd] = 0x84d8, [0x0ece] = 0x84d5, [0x0ecf] = 0x450d,
  [0x0ed0] = 0x84b7, [0x0ed1] = 0x84ad, [0x0ed2] = 0x84da, [0x0ed3] = 0x8493,
  [0x0ed4] = 0x8736, [0x0ed5] = 0x45c0, [0x0ed6] = 0x45c5, [0x0ed7] = 0x45c9,
  [0x0ed8] = 0x873d, [0x0ed9] = 0x872b, [0x0eda] = 0x8747, [0x0edb] = 0x8739,
  [0x0edc] = 0x45d5, [0x0edd] = 0x8745, [0x0ede] = 0x871d, [0x0edf] = 0x4641,
  [0x0ee0] = 0x88ff, [0x0ee1] = 0x88ea, [0x0ee2] = 0x4633, [0x0ee3] = 0x88f5,
  [0x0ee4] = 0x463a, [0x0ee5] = 0x8900, [0x0ee6] = 0x88ed, [0x0ee7] = 0x8903,
  [0x0ee8] = 0x88e9, [0x0ee9] = 0x4640, [0x0eea] = 0x4642, [0x0eeb] = 0x89ea,
  [0x0eec] = 0x46e8, [0x0eed] = 0x8a9b, [0x0eee] = 0x8a8e, [0x0eef] = 0x8aa2,
  [0x0ef0] = 0x46e4, [0x0ef1] = 0x8a9c, [0x0ef2] = 0x8a94, [0x0ef3] = 0x8a90,
  [0x0ef4] = 0x8aa9, [0x0ef5] = 0x8aac, [0x0ef6] = 0x46e7, [0x0ef7] = 0x8a9f,
  [0x0ef8] = 0x46e6, [0x0ef9] = 0x46e1, [0x0efa] = 0x8a9d, [0x0efb] = 0x4739,
  [0x0efc] = 0x8c67, [0x0efd] = 0x475c, [0x0efe] = 0x4775, [0x0eff] = 0x8cd0,
  [0x0f00] = 0x8cd6, [0x0f01] = 0x8cd4, [0x0f02] = 0x8d98, [0x0f03] = 0x8d9a,
  [0x0f04] = 0x8d97, [0x0f05] = 0x47ae, [0x0f06] = 0x47b0, [0x0f07] = 0x47fa,
  [0x0f08] = 0x8e0b, [0x0f09] = 0x8e08, [0x0f0a] = 0x8e01, [0x0f0b] = 0x8eb4,
  [0x0f0c] = 0x8eb3, [0x0f0d] = 0x485b, [0x0f0e] = 0x8fa1, [0x0f0f] = 0x8fa2,
  [0x0f10] = 0x48a5, [0x0f11] = 0x905a, [0x0f12] = 0x48a2, [0x0f13] = 0x9061,
  [0x0f14] = 0x905f, [0x0f15] = 0x48db, [0x0f16] = 0x48da, [0x0f17] = 0x9125,
  [0x0f18] = 0x917b, [0x0f19] = 0x9176, [0x0f1a] = 0x917c, [0x0f1b] = 0x4924,
  [0x0f1c] = 0x9289, [0x0f1d] = 0x92f6, [0x0f1e] = 0x92b1, [0x0f1f] = 0x92ad,
  [0x0f20] = 0x9292, [0x0f21] = 0x9281, [0x0f22] = 0x9284, [0x0f23] = 0x4926,
  [0x0f24] = 0x92ae, [0x0f25] = 0x9290, [0x0f26] = 0x929e, [0x0f27] = 0x4998,
  [0x0f28] = 0x4996, [0x0f29] = 0x499a, [0x0f2a] = 0x95a2, [0x0f2b] = 0x95a7,
  [0x0f2c] = 0x4997, [0x0f2d] = 0x49e1, [0x0f2e] = 0x49e0, [0x0f2f] = 0x49e3,
  [0x0f30] = 0x49e2, [0x0f31] = 0x96a0, [0x0f32] = 0x969d, [0x0f33] = 0x969f,
  [0x0f34] = 0x96d0, [0x0f35] = 0x49fb, [0x0f36] = 0x96d1, [0x0f37] = 0x4a12,
  [0x0f38] = 0x4a14, [0x0f39] = 0x9759, [0x0f3a] = 0x4a45, [0x0f3b] = 0x9764,
  [0x0f3c] = 0x4a5c, [0x0f3d] = 0x4a5d, [0x0f3e] = 0x4ab8, [0x0f3f] = 0x9819,
  [0x0f40] = 0x4aba, [0x0f41] = 0x9814, [0x0f42] = 0x9815, [0x0f43] = 0x981a,
  [0x0f44] = 0x4b03, [0x0f45] = 0x4b35, [0x0f46] = 0x4b36, [0x0f47] = 0x4b39,
  [0x0f48] = 0x9906, [0x0f49] = 0x4b2d, [0x0f4a] = 0x98f8, [0x0f4b] = 0x9901,
  [0x0f4c] = 0x4b7a, [0x0f4d] = 0x99be, [0x0f4e] = 0x99bc, [0x0f4f] = 0x99b7,
  [0x0f50] = 0x99b6, [0x0f51] = 0x99c0, [0x0f52] = 0x4b78, [0x0f53] = 0x99b8,
  [0x0f54] = 0x4b7b, [0x0f55] = 0x4b7c, [0x0f56] = 0x4b7e, [0x0f57] = 0x99c4,
  [0x0f58] = 0x4b7d, [0x0f59] = 0x99bf, [0x0f5a] = 0x4bc9, [0x0f5b] = 0x9ada,
  [0x0f5c] = 0x9ae4, [0x0f5d] = 0x9ae9, [0x0f5e] = 0x9ae8, [0x0f5f] = 0x9aea,
  [0x0f60] = 0x9ae5, [0x0f61] = 0x4bf3, [0x0f62] = 0x9b26, [0x0f63] = 0x4c1a,
  [0x0f64] = 0x4c19, [0x0f65] = 0x9b40, [0x0f66] = 0x4c1f, [0x0f67] = 0x4ca6,
  [0x0f68] = 0x4ca7, [0x0f69] = 0x4ca8, [0x0f6a] = 0x4cab, [0x0f6b] = 0x4ca9,
  [0x0f6c] = 0x4d2e, [0x0f6d] = 0x9ebd, [0x0f6e] = 0x4d5e, [0x0f6f] = 0x3495,
  [0x0f70] = 0x3493, [0x0f71] = 0x3492, [0x0f72] = 0x510e, [0x0f73] = 0x3496,
  [0x0f74] = 0x50f7, [0x0f75] = 0x3497, [0x0f76] = 0x50fc, [0x0f77] = 0x510d,
  [0x0f78] = 0x5101, [0x0f79] = 0x51da, [0x0f7a] = 0x51d9, [0x0f7b] = 0x51db,
  [0x0f7c] = 0x5286, [0x0f7d] = 0x528e, [0x0f7e] = 0x52ee, [0x0f7f] = 0x5333,
  [0x0f80] = 0x53b1, [0x0f81] = 0x35f5, [0x0f82] = 0x5647, [0x0f83] = 0x562d,
  [0x0f84] = 0x5654, [0x0f85] = 0x35ea, [0x0f86] = 0x564b, [0x0f87] = 0x5652,
  [0x0f88] = 0x5631, [0x0f89] = 0x5644, [0x0f8a] = 0x5656, [0x0f8b] = 0x5650,
  [0x0f8c] = 0x562b, [0x0f8d] = 0x35f3, [0x0f8e] = 0x564d, [0x0f8f] = 0x5637,
  [0x0f90] = 0x564f, [0x0f91] = 0x58a2, [0x0f92] = 0x58b7, [0x0f93] = 0x3669,
  [0x0f94] = 0x58b2, [0x0f95] = 0x366b, [0x0f96] = 0x58aa, [0x0f97] = 0x58b5,
  [0x0f98] = 0x58b0, [0x0f99] = 0x366c, [0x0f9a] = 0x58b4, [0x0f9b] = 0x58a4,
  [0x0f9c] = 0x58a7, [0x0f9d] = 0x3668, [0x0f9e] = 0x5926, [0x0f9f] = 0x5afe,
  [0x0fa0] = 0x3728, [0x0fa1] = 0x5b04, [0x0fa2] = 0x3726, [0x0fa3] = 0x5afc,
  [0x0fa4] = 0x3725, [0x0fa5] = 0x5b06, [0x0fa6] = 0x5b0a, [0x0fa7] = 0x5afa,
  [0x0fa8] = 0x5b0d, [0x0fa9] = 0x5b00, [0x0faa] = 0x5b0e, [0x0fab] = 0x376b,
  [0x0fac] = 0x380f, [0x0fad] = 0x3808, [0x0fae] = 0x5d91, [0x0faf] = 0x380c,
  [0x0fb0] = 0x5d8f, [0x0fb1] = 0x5d90, [0x0fb2] = 0x5d98, [0x0fb3] = 0x5da4,
  [0x0fb4] = 0x5d9b, [0x0fb5] = 0x5da3, [0x0fb6] = 0x5d96, [0x0fb7] = 0x5de4,
  [0x0fb8] = 0x5e5a, [0x0fb9] = 0x3860, [0x0fba] = 0x3862, [0x0fbb] = 0x5e5e,
  [0x0fbc] = 0x3898, [0x0fbd] = 0x5fb8, [0x0fbe] = 0x6157, [0x0fbf] = 0x615c,
  [0x0fc0] = 0x61a6, [0x0fc1] = 0x6195, [0x0fc2] = 0x6188, [0x0fc3] = 0x398a,
  [0x0fc4] = 0x61a3, [0x0fc5] = 0x618f, [0x0fc6] = 0x3984, [0x0fc7] = 0x6164,
  [0x0fc8] = 0x397f, [0x0fc9] = 0x6159, [0x0fca] = 0x6178, [0x0fcb] = 0x3982,
  [0x0fcc] = 0x6185, [0x0fcd] = 0x6187, [0x0fce] = 0x619e, [0x0fcf] = 0x3996,
  [0x0fd0] = 0x3989, [0x0fd1] = 0x6198, [0x0fd2] = 0x619c, [0x0fd3] = 0x398d,
  [0x0fd4] = 0x39bc, [0x0fd5] = 0x622f, [0x0fd6] = 0x6480, [0x0fd7] = 0x649b,
  [0x0fd8] = 0x648e, [0x0fd9] = 0x648d, [0x0fda] = 0x6494, [0x0fdb] = 0x64c6,
  [0x0fdc] = 0x3a44, [0x0fdd] = 0x64a8, [0x0fde] = 0x6483, [0x0fdf] = 0x3a3c,
  [0x0fe0] = 0x64b9, [0x0fe1] = 0x6486, [0x0fe2] = 0x64b4, [0x0fe3] = 0x64af,
  [0x0fe4] = 0x6491, [0x0fe5] = 0x3a4e, [0x0fe6] = 0x64aa, [0x0fe7] = 0x64a1,
  [0x0fe8] = 0x64a7, [0x0fe9] = 0x66b6, [0x0fea] = 0x66b3, [0x0feb] = 0x3b14,
  [0x0fec] = 0x66bc, [0x0fed] = 0x66ac, [0x0fee] = 0x3b15, [0x0fef] = 0x66ad,
  [0x0ff0] = 0x6a0e, [0x0ff1] = 0x3bce, [0x0ff2] = 0x6a1c, [0x0ff3] = 0x6a1a,
  [0x0ff4] = 0x3be0, [0x0ff5] = 0x3bc2, [0x0ff6] = 0x6a0b, [0x0ff7] = 0x3bbf,
  [0x0ff8] = 0x69ef, [0x0ff9] = 0x6a0c, [0x0ffa] = 0x69f0, [0x0ffb] = 0x6a22,
  [0x0ffc] = 0x3bc4, [0x0ffd] = 0x69d8, [0x0ffe] = 0x3bcf, [0x0fff] = 0x6a12,
  [0x1000] = 0x69fa, [0x1001] = 0x3bc8, [0x1002] = 0x6a2a, [0x1003] = 0x3bcc,
  [0x1004] = 0x6a10, [0x1005] = 0x3bcd, [0x1006] = 0x3bc7, [0x1007] = 0x6a29,
  [0x1008] = 0x69f9, [0x1009] = 0x69ea, [0x100a] = 0x6a2c, [0x100b] = 0x6a24,
  [0x100c] = 0x4cb7, [0x100d] = 0x69e9, [0x100e] = 0x6b52, [0x100f] = 0x6b4f,
  [0x1010] = 0x6b53, [0x1011] = 0x3c43, [0x1012] = 0x3cb6, [0x1013] = 0x6f10,
  [0x1014] = 0x6f65, [0x1015] = 0x6f75, [0x1016] = 0x3d51, [0x1017] = 0x3d4a,
  [0x1018] = 0x3d4d, [0x1019] = 0x3d56, [0x101a] = 0x6fd0, [0x101b] = 0x3d53,
  [0x101c] = 0x6f5c, [0x101d] = 0x6f3d, [0x101e] = 0x6f71, [0x101f] = 0x3d59,
  [0x1020] = 0x6f91, [0x1021] = 0x6f0b, [0x1022] = 0x6f79, [0x1023] = 0x6f81,
  [0x1024] = 0x6f8f, [0x1025] = 0x3d4e, [0x1026] = 0x6f59, [0x1027] = 0x6f74,
  [0x1028] = 0x3dee, [0x1029] = 0x71ae, [0x102a] = 0x3dec, [0x102b] = 0x71a3,
  [0x102c] = 0x71ad, [0x102d] = 0x3deb, [0x102e] = 0x3def, [0x102f] = 0x71ab,
  [0x1030] = 0x71a6, [0x1031] = 0x71a2, [0x1032] = 0x3ded, [0x1033] = 0x52f2,
  [0x1034] = 0x7257, [0x1035] = 0x7255, [0x1036] = 0x7299, [0x1037] = 0x734b,
  [0x1038] = 0x747a, [0x1039] = 0x3ef2, [0x103a] = 0x3eef, [0x103b] = 0x3ef1,
  [0x103c] = 0x748c, [0x103d] = 0x7484, [0x103e] = 0x3eed, [0x103f] = 0x3ef0,
  [0x1040] = 0x7482, [0x1041] = 0x7493, [0x1042] = 0x747b, [0x1043] = 0x3eee,
  [0x1044] = 0x7509, [0x1045] = 0x4c1b, [0x1046] = 0x3f50, [0x1047] = 0x3f66,
  [0x1048] = 0x3684, [0x1049] = 0x3fb8, [0x104a] = 0x3ff6, [0x104b] = 0x778a,
  [0x104c] = 0x4057, [0x104d] = 0x7790, [0x104e] = 0x405e, [0x104f] = 0x78c6,
  [0x1050] = 0x78d3, [0x1051] = 0x78c0, [0x1052] = 0x78d2, [0x1053] = 0x78c7,
  [0x1054] = 0x78c2, [0x1055] = 0x4119, [0x1056] = 0x799f, [0x1057] = 0x799d,
  [0x1058] = 0x799e, [0x1059] = 0x4170, [0x105a] = 0x7a41, [0x105b] = 0x416e,
  [0x105c] = 0x7a38, [0x105d] = 0x7a3a, [0x105e] = 0x7a42, [0x105f] = 0x4172,
  [0x1060] = 0x4176, [0x1061] = 0x7a3e, [0x1062] = 0x7ab0, [0x1063] = 0x7bae,
  [0x1064] = 0x7bb3, [0x1065] = 0x4212, [0x1066] = 0x421f, [0x1067] = 0x7bbf,
  [0x1068] = 0x4211, [0x1069] = 0x4216, [0x106a] = 0x7bcd, [0x106b] = 0x4219,
  [0x106c] = 0x7bb2, [0x106d] = 0x4224, [0x106e] = 0x4214, [0x106f] = 0x4225,
  [0x1070] = 0x4295, [0x1071] = 0x4296, [0x1072] = 0x4293, [0x1073] = 0x4294,
  [0x1074] = 0x7cc4, [0x1075] = 0x7ccd, [0x1076] = 0x7cc2, [0x1077] = 0x7cc6,
  [0x1078] = 0x7cc3, [0x1079] = 0x7cc9, [0x107a] = 0x7cc7, [0x107b] = 0x42a0,
  [0x107c] = 0x7df8, [0x107d] = 0x42fb, [0x107e] = 0x7ded, [0x107f] = 0x7de2,
  [0x1080] = 0x42fc, [0x1081] = 0x4300, [0x1082] = 0x42f8, [0x1083] = 0x7ddc,
  [0x1084] = 0x7e02, [0x1085] = 0x7e01, [0x1086] = 0x42f9, [0x1087] = 0x7dd6,
  [0x1088] = 0x4304, [0x1089] = 0x7de4, [0x108a] = 0x7dfe, [0x108b] = 0x4303,
  [0x108c] = 0x7e00, [0x108d] = 0x7dfc, [0x108e] = 0x7dfd, [0x108f] = 0x42f3,
  [0x1090] = 0x7df5, [0x1091] = 0x7dff, [0x1092] = 0x42fa, [0x1093] = 0x7deb,
  [0x1094] = 0x7de5, [0x1095] = 0x7f78, [0x1096] = 0x7fae, [0x1097] = 0x7fe7,
  [0x1098] = 0x43bf, [0x1099] = 0x8065, [0x109a] = 0x806a, [0x109b] = 0x8066,
  [0x109c] = 0x8068, [0x109d] = 0x806b, [0x109e] = 0x8194, [0x109f] = 0x81a1,
  [0x10a0] = 0x8192, [0x10a1] = 0x8196, [0x10a2] = 0x8193, [0x10a3] = 0x4479,
  [0x10a4] = 0x4510, [0x10a5] = 0x8501, [0x10a6] = 0x4514, [0x10a7] = 0x84f8,
  [0x10a8] = 0x450e, [0x10a9] = 0x84f5, [0x10aa] = 0x451a, [0x10ab] = 0x8504,
  [0x10ac] = 0x4519, [0x10ad] = 0x4521, [0x10ae] = 0x4523, [0x10af] = 0x451f,
  [0x10b0] = 0x851b, [0x10b1] = 0x8503, [0x10b2] = 0x8533, [0x10b3] = 0x8534,
  [0x10b4] = 0x84ed, [0x10b5] = 0x4525, [0x10b6] = 0x452b, [0x10b7] = 0x8535,
  [0x10b8] = 0x4516, [0x10b9] = 0x8505, [0x10ba] = 0x4522, [0x10bb] = 0x451b,
  [0x10bc] = 0x45ce, [0x10bd] = 0x45cf, [0x10be] = 0x877d, [0x10bf] = 0x45cb,
  [0x10c0] = 0x45d1, [0x10c1] = 0x45cc, [0x10c2] = 0x8771, [0x10c3] = 0x4617,
  [0x10c4] = 0x885c, [0x10c5] = 0x88e6, [0x10c6] = 0x890f, [0x10c7] = 0x891b,
  [0x10c8] = 0x4651, [0x10c9] = 0x89a9, [0x10ca] = 0x89a5, [0x10cb] = 0x89ee,
  [0x10cc] = 0x8ab1, [0x10cd] = 0x46ed, [0x10ce] = 0x8acc, [0x10cf] = 0x8ace,
  [0x10d0] = 0x46f4, [0x10d1] = 0x8ab7, [0x10d2] = 0x46f1, [0x10d3] = 0x8ab5,
  [0x10d4] = 0x8ae9, [0x10d5] = 0x8ab4, [0x10d6] = 0x46f8, [0x10d7] = 0x8ab3,
  [0x10d8] = 0x8ac1, [0x10d9] = 0x8aaf, [0x10da] = 0x8aca, [0x10db] = 0x8ad0,
  [0x10dc] = 0x472f, [0x10dd] = 0x475e, [0x10de] = 0x475d, [0x10df] = 0x8c8e,
  [0x10e0] = 0x4776, [0x10e1] = 0x4777, [0x10e2] = 0x8ce9, [0x10e3] = 0x8cdb,
  [0x10e4] = 0x477e, [0x10e5] = 0x8ceb, [0x10e6] = 0x8da4, [0x10e7] = 0x47b6,
  [0x10e8] = 0x8da2, [0x10e9] = 0x8d9d, [0x10ea] = 0x47b3, [0x10eb] = 0x47fc,
  [0x10ec] = 0x4803, [0x10ed] = 0x4800, [0x10ee] = 0x8e2a, [0x10ef] = 0x8e28,
  [0x10f0] = 0x480a, [0x10f1] = 0x4802, [0x10f2] = 0x8eb8, [0x10f3] = 0x8eb6,
  [0x10f4] = 0x8eb9, [0x10f5] = 0x8eb7, [0x10f6] = 0x8f22, [0x10f7] = 0x8f2b,
  [0x10f8] = 0x8f27, [0x10f9] = 0x8f19, [0x10fa] = 0x8fa4, [0x10fb] = 0x4887,
  [0x10fc] = 0x8fb3, [0x10fd] = 0x48a6, [0x10fe] = 0x9071, [0x10ff] = 0x906a,
  [0x1100] = 0x48a9, [0x1101] = 0x48de, [0x1102] = 0x9188, [0x1103] = 0x918c,
  [0x1104] = 0x92bf, [0x1105] = 0x92b8, [0x1106] = 0x92be, [0x1107] = 0x92dc,
  [0x1108] = 0x92e5, [0x1109] = 0x492e, [0x110a] = 0x492d, [0x110b] = 0x92d4,
  [0x110c] = 0x92d6, [0x110d] = 0x4930, [0x110e] = 0x92da, [0x110f] = 0x92ed,
  [0x1110] = 0x92f3, [0x1111] = 0x92db, [0x1112] = 0x492b, [0x1113] = 0x92b9,
  [0x1114] = 0x92e2, [0x1115] = 0x92eb, [0x1116] = 0x95af, [0x1117] = 0x499e,
  [0x1118] = 0x95b2, [0x1119] = 0x95b3, [0x111a] = 0x499f, [0x111b] = 0x49e5,
  [0x111c] = 0x49e4, [0x111d] = 0x96a3, [0x111e] = 0x96a5, [0x111f] = 0x49fd,
  [0x1120] = 0x49fc, [0x1121] = 0x4a17, [0x1122] = 0x4a19, [0x1123] = 0x970a,
  [0x1124] = 0x4a18, [0x1125] = 0x9787, [0x1126] = 0x9789, [0x1127] = 0x978c,
  [0x1128] = 0x97ef, [0x1129] = 0x982a, [0x112a] = 0x9822, [0x112b] = 0x4abf,
  [0x112c] = 0x981f, [0x112d] = 0x4b3c, [0x112e] = 0x9919, [0x112f] = 0x4b6b,
  [0x1130] = 0x99ca, [0x1131] = 0x99da, [0x1132] = 0x4b83, [0x1133] = 0x4b81,
  [0x1134] = 0x4b80, [0x1135] = 0x99de, [0x1136] = 0x99c8, [0x1137] = 0x99e0,
  [0x1138] = 0x4bca, [0x1139] = 0x9ab6, [0x113a] = 0x9ab5, [0x113b] = 0x4bce,
  [0x113c] = 0x9af4, [0x113d] = 0x4bf6, [0x113e] = 0x9b6b, [0x113f] = 0x9b69,
  [0x1140] = 0x9b72, [0x1141] = 0x9b63, [0x1142] = 0x4c39, [0x1143] = 0x9d0d,
  [0x1144] = 0x4cae, [0x1145] = 0x9d01, [0x1146] = 0x9d0c, [0x1147] = 0x4cb5,
  [0x1148] = 0x9cf8, [0x1149] = 0x4cb3, [0x114a] = 0x4cb4, [0x114b] = 0x9cfe,
  [0x114c] = 0x9d02, [0x114d] = 0x9e84, [0x114e] = 0x4d22, [0x114f] = 0x9eab,
  [0x1150] = 0x9eaa, [0x1151] = 0x511d, [0x1152] = 0x5116, [0x1153] = 0x3499,
  [0x1154] = 0x512b, [0x1155] = 0x511e, [0x1156] = 0x511b, [0x1157] = 0x5290,
  [0x1158] = 0x5294, [0x1159] = 0x5314, [0x115a] = 0x20b89, [0x115b] = 0x3602,
  [0x115c] = 0x5667, [0x115d] = 0x3601, [0x115e] = 0x567b, [0x115f] = 0x36a1,
  [0x1160] = 0x565f, [0x1161] = 0x5661, [0x1162] = 0x35fd, [0x1163] = 0x3673,
  [0x1164] = 0x3674, [0x1165] = 0x3670, [0x1166] = 0x3676, [0x1167] = 0x3675,
  [0x1168] = 0x3672, [0x1169] = 0x58c3, [0x116a] = 0x58ca, [0x116b] = 0x58bb,
  [0x116c] = 0x58c0, [0x116d] = 0x58c4, [0x116e] = 0x5901, [0x116f] = 0x5b1f,
  [0x1170] = 0x5b18, [0x1171] = 0x5b11, [0x1172] = 0x5b15, [0x1173] = 0x3729,
  [0x1174] = 0x5b12, [0x1175] = 0x5b1c, [0x1176] = 0x372a, [0x1177] = 0x5b22,
  [0x1178] = 0x5b79, [0x1179] = 0x5da6, [0x117a] = 0x3816, [0x117b] = 0x5db3,
  [0x117c] = 0x5dab, [0x117d] = 0x5eea, [0x117e] = 0x3899, [0x117f] = 0x5f5b,
  [0x1180] = 0x38d3, [0x1181] = 0x38f5, [0x1182] = 0x61b7, [0x1183] = 0x61ce,
  [0x1184] = 0x61b9, [0x1185] = 0x61bd, [0x1186] = 0x61cf, [0x1187] = 0x61c0,
  [0x1188] = 0x6199, [0x1189] = 0x6197, [0x118a] = 0x3994, [0x118b] = 0x61bb,
  [0x118c] = 0x61d0, [0x118d] = 0x61c4, [0x118e] = 0x6231, [0x118f] = 0x3a56,
  [0x1190] = 0x64d3, [0x1191] = 0x64c0, [0x1192] = 0x3a59, [0x1193] = 0x3a58,
  [0x1194] = 0x3a55, [0x1195] = 0x3a52, [0x1196] = 0x64dc, [0x1197] = 0x64d1,
  [0x1198] = 0x64c8, [0x1199] = 0x3a57, [0x119a] = 0x64d5, [0x119b] = 0x66c3,
  [0x119c] = 0x3b1b, [0x119d] = 0x3b1c, [0x119e] = 0x66bf, [0x119f] = 0x66c5,
  [0x11a0] = 0x3b19, [0x11a1] = 0x66cd, [0x11a2] = 0x66c1, [0x11a3] = 0x6706,
  [0x11a4] = 0x3b3f, [0x11a5] = 0x6724, [0x11a6] = 0x6a63, [0x11a7] = 0x6a42,
  [0x11a8] = 0x6a52, [0x11a9] = 0x3bdb, [0x11aa] = 0x6a43, [0x11ab] = 0x6a33,
  [0x11ac] = 0x3be2, [0x11ad] = 0x6a6c, [0x11ae] = 0x6a57, [0x11af] = 0x3bd7,
  [0x11b0] = 0x6a4c, [0x11b1] = 0x6a6e, [0x11b2] = 0x3bde, [0x11b3] = 0x3be5,
  [0x11b4] = 0x3be4, [0x11b5] = 0x3be6, [0x11b6] = 0x3bd6, [0x11b7] = 0x6a37,
  [0x11b8] = 0x3bdf, [0x11b9] = 0x6a71, [0x11ba] = 0x6a4a, [0x11bb] = 0x6a36,
  [0x11bc] = 0x3bdc, [0x11bd] = 0x6a53, [0x11be] = 0x3bda, [0x11bf] = 0x6a45,
  [0x11c0] = 0x6a70, [0x11c1] = 0x3bd3, [0x11c2] = 0x3bd0, [0x11c3] = 0x6a5c,
  [0x11c4] = 0x6b58, [0x11c5] = 0x6b57, [0x11c6] = 0x3c86, [0x11c7] = 0x3c87,
  [0x11c8] = 0x3cad, [0x11c9] = 0x3cb7, [0x11ca] = 0x3d58, [0x11cb] = 0x3d6a,
  [0x11cc] = 0x6fbb, [0x11cd] = 0x3d62, [0x11ce] = 0x3d61, [0x11cf] = 0x6fbe,
  [0x11d0] = 0x3d69, [0x11d1] = 0x3d6c, [0x11d2] = 0x3d65, [0x11d3] = 0x6fb5,
  [0x11d4] = 0x6fd3, [0x11d5] = 0x6f9f, [0x11d6] = 0x3d66, [0x11d7] = 0x6fb7,
  [0x11d8] = 0x6ff5, [0x11d9] = 0x71b7, [0x11da] = 0x3df5, [0x11db] = 0x71bb,
  [0x11dc] = 0x3df4, [0x11dd] = 0x71d1, [0x11de] = 0x3df7, [0x11df] = 0x71ba,
  [0x11e0] = 0x3df8, [0x11e1] = 0x71b6, [0x11e2] = 0x71cc, [0x11e3] = 0x3dfb,
  [0x11e4] = 0x3dfc, [0x11e5] = 0x71d3, [0x11e6] = 0x749b, [0x11e7] = 0x3ef5,
  [0x11e8] = 0x3ef8, [0x11e9] = 0x7496, [0x11ea] = 0x74a2, [0x11eb] = 0x749d,
  [0x11ec] = 0x750a, [0x11ed] = 0x750e, [0x11ee] = 0x3f3c, [0x11ef] = 0x7581,
  [0x11f0] = 0x762c, [0x11f1] = 0x7637, [0x11f2] = 0x7636, [0x11f3] = 0x763b,
  [0x11f4] = 0x3fc5, [0x11f5] = 0x76a1, [0x11f6] = 0x4062, [0x11f7] = 0x4063,
  [0x11f8] = 0x7798, [0x11f9] = 0x4067, [0x11fa] = 0x7796, [0x11fb] = 0x4066,
  [0x11fc] = 0x40d9, [0x11fd] = 0x40db, [0x11fe] = 0x78d6, [0x11ff] = 0x78eb,
  [0x1200] = 0x40d8, [0x1201] = 0x78dc, [0x1202] = 0x411b, [0x1203] = 0x79a5,
  [0x1204] = 0x79a9, [0x1205] = 0x9834, [0x1206] = 0x7a53, [0x1207] = 0x7a45,
  [0x1208] = 0x4179, [0x1209] = 0x7a4f, [0x120a] = 0x417d, [0x120b] = 0x7abd,
  [0x120c] = 0x7abb, [0x120d] = 0x7af1, [0x120e] = 0x422c, [0x120f] = 0x4237,
  [0x1210] = 0x7bec, [0x1211] = 0x7bed, [0x1212] = 0x4230, [0x1213] = 0x429a,
  [0x1214] = 0x7cd3, [0x1215] = 0x4a00, [0x1216] = 0x7ce1, [0x1217] = 0x4305,
  [0x1218] = 0x7e19, [0x1219] = 0x4307, [0x121a] = 0x4309, [0x121b] = 0x430a,
  [0x121c] = 0x7e27, [0x121d] = 0x7e26, [0x121e] = 0x4379, [0x121f] = 0x43c2,
  [0x1220] = 0x806e, [0x1221] = 0x81af, [0x1222] = 0x4438, [0x1223] = 0x4437,
  [0x1224] = 0x81ad, [0x1225] = 0x4421, [0x1226] = 0x81aa, [0x1227] = 0x8218,
  [0x1228] = 0x445e, [0x1229] = 0x453d, [0x122a] = 0x4537, [0x122b] = 0x4540,
  [0x122c] = 0x856f, [0x122d] = 0x854c, [0x122e] = 0x451d, [0x122f] = 0x8542,
  [0x1230] = 0x4533, [0x1231] = 0x855c, [0x1232] = 0x8570, [0x1233] = 0x855f,
  [0x1234] = 0x4535, [0x1235] = 0x855a, [0x1236] = 0x854b, [0x1237] = 0x853f,
  [0x1238] = 0x878a, [0x1239] = 0x45d8, [0x123a] = 0x878b, [0x123b] = 0x87a1,
  [0x123c] = 0x878e, [0x123d] = 0x45dc, [0x123e] = 0x45de, [0x123f] = 0x8799,
  [0x1240] = 0x885e, [0x1241] = 0x885f, [0x1242] = 0x8924, [0x1243] = 0x89a7,
  [0x1244] = 0x8aea, [0x1245] = 0x8afd, [0x1246] = 0x8af9, [0x1247] = 0x8ae3,
  [0x1248] = 0x8ae5, [0x1249] = 0x46fa, [0x124a] = 0x46fb, [0x124b] = 0x8aec,
  [0x124c] = 0x473d, [0x124d] = 0x473b, [0x124e] = 0x473f, [0x124f] = 0x475f,
  [0x1250] = 0x8cf2, [0x1251] = 0x477f, [0x1252] = 0x8cef, [0x1253] = 0x4784,
  [0x1254] = 0x8da6, [0x1255] = 0x47bc, [0x1256] = 0x4814, [0x1257] = 0x480f,
  [0x1258] = 0x8e3b, [0x1259] = 0x8e43, [0x125a] = 0x480e, [0x125b] = 0x8e32,
  [0x125c] = 0x8f31, [0x125d] = 0x8f30, [0x125e] = 0x4860, [0x125f] = 0x8f2d,
  [0x1260] = 0x8f3c, [0x1261] = 0x8fa7, [0x1262] = 0x8fa5, [0x1263] = 0x48ab,
  [0x1264] = 0x48ac, [0x1265] = 0x48aa, [0x1266] = 0x9137, [0x1267] = 0x9195,
  [0x1268] = 0x918e, [0x1269] = 0x4904, [0x126a] = 0x9196, [0x126b] = 0x4908,
  [0x126c] = 0x9345, [0x126d] = 0x930a, [0x126e] = 0x4933, [0x126f] = 0x4934,
  [0x1270] = 0x92fd, [0x1271] = 0x9317, [0x1272] = 0x931c, [0x1273] = 0x9307,
  [0x1274] = 0x9331, [0x1275] = 0x9332, [0x1276] = 0x932c, [0x1277] = 0x9330,
  [0x1278] = 0x9303, [0x1279] = 0x9305, [0x127a] = 0x49a2, [0x127b] = 0x95c2,
  [0x127c] = 0x49a4, [0x127d] = 0x95b8, [0x127e] = 0x49a5, [0x127f] = 0x95c1,
  [0x1280] = 0x49a7, [0x1281] = 0x49a6, [0x1282] = 0x49e7, [0x1283] = 0x96ab,
  [0x1284] = 0x96b7, [0x1285] = 0x49ff, [0x1286] = 0x49fe, [0x1287] = 0x9715,
  [0x1288] = 0x9714, [0x1289] = 0x4a1d, [0x128a] = 0x4a1c, [0x128b] = 0x970c,
  [0x128c] = 0x9717, [0x128d] = 0x4a67, [0x128e] = 0x9793, [0x128f] = 0x4a94,
  [0x1290] = 0x97d2, [0x1291] = 0x4ac5, [0x1292] = 0x4ac8, [0x1293] = 0x9836,
  [0x1294] = 0x9831, [0x1295] = 0x9833, [0x1296] = 0x983c, [0x1297] = 0x982e,
  [0x1298] = 0x983a, [0x1299] = 0x4ac9, [0x129a] = 0x983d, [0x129b] = 0x4ac7,
  [0x129c] = 0x98b5, [0x129d] = 0x9922, [0x129e] = 0x9923, [0x129f] = 0x9920,
  [0x12a0] = 0x991c, [0x12a1] = 0x991d, [0x12a2] = 0x4b6c, [0x12a3] = 0x99a0,
  [0x12a4] = 0x4b8a, [0x12a5] = 0x99ef, [0x12a6] = 0x99e8, [0x12a7] = 0x99eb,
  [0x12a8] = 0x4b88, [0x12a9] = 0x4b87, [0x12aa] = 0x4b86, [0x12ab] = 0x99e1,
  [0x12ac] = 0x99e6, [0x12ad] = 0x4bcf, [0x12ae] = 0x4bd0, [0x12af] = 0x9af8,
  [0x12b0] = 0x9af5, [0x12b1] = 0x4c1c, [0x12b2] = 0x4c23, [0x12b3] = 0x9b83,
  [0x12b4] = 0x9b94, [0x12b5] = 0x9b84, [0x12b6] = 0x4c49, [0x12b7] = 0x9b8b,
  [0x12b8] = 0x9b8f, [0x12b9] = 0x4c43, [0x12ba] = 0x9b8c, [0x12bb] = 0x4c48,
  [0x12bc] = 0x9b89, [0x12bd] = 0x4c47, [0x12be] = 0x9b8e, [0x12bf] = 0x4c46,
  [0x12c0] = 0x4c3f, [0x12c1] = 0x4c44, [0x12c2] = 0x9d24, [0x12c3] = 0x9d0f,
  [0x12c4] = 0x4cbe, [0x12c5] = 0x9d13, [0x12c6] = 0x9d0a, [0x12c7] = 0x4cc2,
  [0x12c8] = 0x4cba, [0x12c9] = 0x4cbc, [0x12ca] = 0x4cc6, [0x12cb] = 0x9d2a,
  [0x12cc] = 0x9d1a, [0x12cd] = 0x4cc8, [0x12ce] = 0x9d27, [0x12cf] = 0x9d16,
  [0x12d0] = 0x9d21, [0x12d1] = 0x4d23, [0x12d2] = 0x9e85, [0x12d3] = 0x9eac,
  [0x12d4] = 0x9ec6, [0x12d5] = 0x9ec5, [0x12d6] = 0x9ed7, [0x12d7] = 0x9f53,
  [0x12d8] = 0x349d, [0x12d9] = 0x5128, [0x12da] = 0x5127, [0x12db] = 0x51df,
  [0x12dc] = 0x3524, [0x12dd] = 0x5335, [0x12de] = 0x53b3, [0x12df] = 0x3607,
  [0x12e0] = 0x568a, [0x12e1] = 0x567d, [0x12e2] = 0x5689, [0x12e3] = 0x3679,
  [0x12e4] = 0x58cd, [0x12e5] = 0x58d0, [0x12e6] = 0x3678, [0x12e7] = 0x5b2b,
  [0x12e8] = 0x5b33, [0x12e9] = 0x5b29, [0x12ea] = 0x5b35, [0x12eb] = 0x5b31,
  [0x12ec] = 0x5b37, [0x12ed] = 0x5c36, [0x12ee] = 0x5dbe, [0x12ef] = 0x3819,
  [0x12f0] = 0x5db9, [0x12f1] = 0x381c, [0x12f2] = 0x5dbb, [0x12f3] = 0x3818,
  [0x12f4] = 0x61e2, [0x12f5] = 0x61db, [0x12f6] = 0x61dd, [0x12f7] = 0x61dc,
  [0x12f8] = 0x61da, [0x12f9] = 0x2f8af, [0x12fa] = 0x61d9, [0x12fb] = 0x39bd,
  [0x12fc] = 0x3a5d, [0x12fd] = 0x64df, [0x12fe] = 0x3a5a, [0x12ff] = 0x3a5e,
  [0x1300] = 0x64e1, [0x1301] = 0x3a5c, [0x1302] = 0x64ee, [0x1303] = 0x3a5b,
  [0x1304] = 0x65b5, [0x1305] = 0x66d4, [0x1306] = 0x66d5, [0x1307] = 0x3b21,
  [0x1308] = 0x66d0, [0x1309] = 0x66d1, [0x130a] = 0x66ce, [0x130b] = 0x66d7,
  [0x130c] = 0x3b20, [0x130d] = 0x3b32, [0x130e] = 0x6a7d, [0x130f] = 0x6a8a,
  [0x1310] = 0x3bf2, [0x1311] = 0x6aa7, [0x1312] = 0x3bf5, [0x1313] = 0x6a99,
  [0x1314] = 0x6a82, [0x1315] = 0x6a88, [0x1316] = 0x3bee, [0x1317] = 0x3bec,
  [0x1318] = 0x6a86, [0x1319] = 0x3bea, [0x131a] = 0x6a98, [0x131b] = 0x6a9d,
  [0x131c] = 0x3bed, [0x131d] = 0x3bf3, [0x131e] = 0x6a8f, [0x131f] = 0x3bf6,
  [0x1320] = 0x6aaa, [0x1321] = 0x3c48, [0x1322] = 0x6b5d, [0x1323] = 0x3c49,
  [0x1324] = 0x6c0a, [0x1325] = 0x3d75, [0x1326] = 0x6fd7, [0x1327] = 0x6fd6,
  [0x1328] = 0x6fe5, [0x1329] = 0x3d6f, [0x132a] = 0x3d7b, [0x132b] = 0x3d73,
  [0x132c] = 0x6fd9, [0x132d] = 0x6fda, [0x132e] = 0x6fea, [0x132f] = 0x3d70,
  [0x1330] = 0x6ff6, [0x1331] = 0x24039, [0x1332] = 0x3d78, [0x1333] = 0x71e3,
  [0x1334] = 0x3dfe, [0x1335] = 0x71e9, [0x1336] = 0x3e00, [0x1337] = 0x71eb,
  [0x1338] = 0x71ef, [0x1339] = 0x71f3, [0x133a] = 0x71ea, [0x133b] = 0x3e01,
  [0x133c] = 0x2f921, [0x133d] = 0x3e55, [0x133e] = 0x3e56, [0x133f] = 0x3e9d,
  [0x1340] = 0x7371, [0x1341] = 0x3ef9, [0x1342] = 0x74ae, [0x1343] = 0x3eff,
  [0x1344] = 0x74b3, [0x1345] = 0x3efd, [0x1346] = 0x74ac, [0x1347] = 0x3f43,
  [0x1348] = 0x3f41, [0x1349] = 0x7583, [0x134a] = 0x7645, [0x134b] = 0x764e,
  [0x134c] = 0x7644, [0x134d] = 0x76a3, [0x134e] = 0x76a5, [0x134f] = 0x77a6,
  [0x1350] = 0x77a4, [0x1351] = 0x406f, [0x1352] = 0x77a9, [0x1353] = 0x77af,
  [0x1354] = 0x408a, [0x1355] = 0x40e5, [0x1356] = 0x40e6, [0x1357] = 0x78f0,
  [0x1358] = 0x78f8, [0x1359] = 0x78f1, [0x135a] = 0x417f, [0x135b] = 0x7a49,
  [0x135c] = 0x41b5, [0x135d] = 0x41b6, [0x135e] = 0x41bb, [0x135f] = 0x7ac2,
  [0x1360] = 0x7af2, [0x1361] = 0x7af3, [0x1362] = 0x7bfa, [0x1363] = 0x4240,
  [0x1364] = 0x7bf6, [0x1365] = 0x7bfc, [0x1366] = 0x7c18, [0x1367] = 0x7c08,
  [0x1368] = 0x7c12, [0x1369] = 0x429d, [0x136a] = 0x429c, [0x136b] = 0x7cdb,
  [0x136c] = 0x7cda, [0x136d] = 0x430f, [0x136e] = 0x4311, [0x136f] = 0x430d,
  [0x1370] = 0x7e2c, [0x1371] = 0x7e4d, [0x1372] = 0x4314, [0x1373] = 0x4313,
  [0x1374] = 0x7f46, [0x1375] = 0x7ff6, [0x1376] = 0x802b, [0x1377] = 0x8074,
  [0x1378] = 0x81b8, [0x1379] = 0x81c8, [0x137a] = 0x4482, [0x137b] = 0x4483,
  [0x137c] = 0x454d, [0x137d] = 0x8592, [0x137e] = 0x8593, [0x137f] = 0x454f,
  [0x1380] = 0x857f, [0x1381] = 0x85ab, [0x1382] = 0x8597, [0x1383] = 0x454c,
  [0x1384] = 0x4551, [0x1385] = 0x85ac, [0x1386] = 0x45ee, [0x1387] = 0x45e8,
  [0x1388] = 0x4ccb, [0x1389] = 0x87ce, [0x138a] = 0x45eb, [0x138b] = 0x87cd,
  [0x138c] = 0x45e2, [0x138d] = 0x45e6, [0x138e] = 0x87c1, [0x138f] = 0x87b1,
  [0x1390] = 0x87c7, [0x1391] = 0x45ec, [0x1392] = 0x8940, [0x1393] = 0x4659,
  [0x1394] = 0x893f, [0x1395] = 0x8939, [0x1396] = 0x465d, [0x1397] = 0x8943,
  [0x1398] = 0x4657, [0x1399] = 0x465b, [0x139a] = 0x4656, [0x139b] = 0x89ab,
  [0x139c] = 0x46fe, [0x139d] = 0x8b1f, [0x139e] = 0x8b09, [0x139f] = 0x8b0c,
  [0x13a0] = 0x4700, [0x13a1] = 0x4701, [0x13a2] = 0x8c40, [0x13a3] = 0x4742,
  [0x13a4] = 0x8c96, [0x13a5] = 0x4760, [0x13a6] = 0x8cf6, [0x13a7] = 0x8cf7,
  [0x13a8] = 0x481d, [0x13a9] = 0x8e46, [0x13aa] = 0x8e4f, [0x13ab] = 0x483e,
  [0x13ac] = 0x4869, [0x13ad] = 0x4865, [0x13ae] = 0x8f3d, [0x13af] = 0x8f41,
  [0x13b0] = 0x9366, [0x13b1] = 0x9378, [0x13b2] = 0x935d, [0x13b3] = 0x9369,
  [0x13b4] = 0x9374, [0x13b5] = 0x937d, [0x13b6] = 0x936e, [0x13b7] = 0x9372,
  [0x13b8] = 0x9373, [0x13b9] = 0x9362, [0x13ba] = 0x9348, [0x13bb] = 0x9353,
  [0x13bc] = 0x935f, [0x13bd] = 0x9368, [0x13be] = 0x4938, [0x13bf] = 0x937f,
  [0x13c0] = 0x936b, [0x13c1] = 0x49ae, [0x13c2] = 0x95c4, [0x13c3] = 0x49ad,
  [0x13c4] = 0x96af, [0x13c5] = 0x96ad, [0x13c6] = 0x96b2, [0x13c7] = 0x4a02,
  [0x13c8] = 0x4a1f, [0x13c9] = 0x971a, [0x13ca] = 0x971b, [0x13cb] = 0x4a22,
  [0x13cc] = 0x4a20, [0x13cd] = 0x2f9f5, [0x13ce] = 0x4a6c, [0x13cf] = 0x979b,
  [0x13d0] = 0x979f, [0x13d1] = 0x4a68, [0x13d2] = 0x4a6d, [0x13d3] = 0x4a6e,
  [0x13d4] = 0x4aa0, [0x13d5] = 0x4ace, [0x13d6] = 0x4ad0, [0x13d7] = 0x4ad1,
  [0x13d8] = 0x4acb, [0x13d9] = 0x9840, [0x13da] = 0x4ad2, [0x13db] = 0x9847,
  [0x13dc] = 0x4ad3, [0x13dd] = 0x98b7, [0x13de] = 0x4b20, [0x13df] = 0x4b4e,
  [0x13e0] = 0x4b4b, [0x13e1] = 0x4b72, [0x13e2] = 0x4b70, [0x13e3] = 0x99a2,
  [0x13e4] = 0x4b92, [0x13e5] = 0x4b8f, [0x13e6] = 0x9a00, [0x13e7] = 0x99f3,
  [0x13e8] = 0x4b90, [0x13e9] = 0x29937, [0x13ea] = 0x99f5, [0x13eb] = 0x4bd9,
  [0x13ec] = 0x4bd5, [0x13ed] = 0x9abd, [0x13ee] = 0x9b00, [0x13ef] = 0x9b02,
  [0x13f0] = 0x4bfa, [0x13f1] = 0x9b34, [0x13f2] = 0x9b49, [0x13f3] = 0x9b9f,
  [0x13f4] = 0x4c4b, [0x13f5] = 0x9ba3, [0x13f6] = 0x9bcd, [0x13f7] = 0x9b99,
  [0x13f8] = 0x9b9d, [0x13f9] = 0x4cd0, [0x13fa] = 0x4cce, [0x13fb] = 0x9d39,
  [0x13fc] = 0x4ccf, [0x13fd] = 0x9d44, [0x13fe] = 0x4cc4, [0x13ff] = 0x4ccc,
  [0x1400] = 0x9d35, [0x1401] = 0x4cd2, [0x1402] = 0x4d35, [0x1403] = 0x9eaf,
  [0x1404] = 0x3e03, [0x1405] = 0x512f, [0x1406] = 0x349e, [0x1407] = 0x34af,
  [0x1408] = 0x9f8e, [0x1409] = 0x360c, [0x140a] = 0x569f, [0x140b] = 0x569b,
  [0x140c] = 0x569e, [0x140d] = 0x5696, [0x140e] = 0x5694, [0x140f] = 0x56a0,
  [0x1410] = 0x367c, [0x1411] = 0x5b3b, [0x1412] = 0x3730, [0x1413] = 0x3731,
  [0x1414] = 0x5b3a, [0x1415] = 0x5dc1, [0x1416] = 0x5f4d, [0x1417] = 0x5f5d,
  [0x1418] = 0x61f3, [0x1419] = 0x39a1, [0x141a] = 0x399e, [0x141b] = 0x3a68,
  [0x141c] = 0x3a61, [0x141d] = 0x64f6, [0x141e] = 0x64e5, [0x141f] = 0x64ea,
  [0x1420] = 0x64e7, [0x1421] = 0x6505, [0x1422] = 0x3a65, [0x1423] = 0x64f9,
  [0x1424] = 0x3a66, [0x1425] = 0x3a6a, [0x1426] = 0x3aab, [0x1427] = 0x6aab,
  [0x1428] = 0x6aed, [0x1429] = 0x6ab2, [0x142a] = 0x6ab0, [0x142b] = 0x6ab5,
  [0x142c] = 0x6abe, [0x142d] = 0x6ac1, [0x142e] = 0x6ac8, [0x142f] = 0x3bf9,
  [0x1430] = 0x6ac0, [0x1431] = 0x6abc, [0x1432] = 0x6ab1, [0x1433] = 0x6ac4,
  [0x1434] = 0x6abf, [0x1435] = 0x3c58, [0x1436] = 0x3c8a, [0x1437] = 0x7008,
  [0x1438] = 0x7003, [0x1439] = 0x6ffd, [0x143a] = 0x7010, [0x143b] = 0x7002,
  [0x143c] = 0x7013, [0x143d] = 0x3e04, [0x143e] = 0x71fa, [0x143f] = 0x7200,
  [0x1440] = 0x74b9, [0x1441] = 0x74bc, [0x1442] = 0x3f02, [0x1443] = 0x765b,
  [0x1444] = 0x7651, [0x1445] = 0x764f, [0x1446] = 0x76eb, [0x1447] = 0x77b8,
  [0x1448] = 0x4079, [0x1449] = 0x77b9, [0x144a] = 0x77c1, [0x144b] = 0x77c0,
  [0x144c] = 0x77be, [0x144d] = 0x790b, [0x144e] = 0x40eb, [0x144f] = 0x7907,
  [0x1450] = 0x790a, [0x1451] = 0x7908, [0x1452] = 0x40e9, [0x1453] = 0x790d,
  [0x1454] = 0x7906, [0x1455] = 0x7915, [0x1456] = 0x79af, [0x1457] = 0x4120,
  [0x1458] = 0x4121, [0x1459] = 0x4181, [0x145a] = 0x7af5, [0x145b] = 0x424d,
  [0x145c] = 0x4259, [0x145d] = 0x7c2e, [0x145e] = 0x4258, [0x145f] = 0x7c1b,
  [0x1460] = 0x25cd1, [0x1461] = 0x7c1a, [0x1462] = 0x7c24, [0x1463] = 0x42a5,
  [0x1464] = 0x42a9, [0x1465] = 0x7ce6, [0x1466] = 0x7ce3, [0x1467] = 0x431a,
  [0x1468] = 0x4319, [0x1469] = 0x7e5d, [0x146a] = 0x7e4f, [0x146b] = 0x7e66,
  [0x146c] = 0x7e5b, [0x146d] = 0x7f47, [0x146e] = 0x7fb4, [0x146f] = 0x4396,
  [0x1470] = 0x4398, [0x1471] = 0x4397, [0x1472] = 0x7ffa, [0x1473] = 0x802e,
  [0x1474] = 0x265df, [0x1475] = 0x43c8, [0x1476] = 0x81ce, [0x1477] = 0x4443,
  [0x1478] = 0x4445, [0x1479] = 0x8219, [0x147a] = 0x4552, [0x147b] = 0x4557,
  [0x147c] = 0x85cc, [0x147d] = 0x85b2, [0x147e] = 0x4555, [0x147f] = 0x85bb,
  [0x1480] = 0x85c1, [0x1481] = 0x4556, [0x1482] = 0x4558, [0x1483] = 0x45f2,
  [0x1484] = 0x87e9, [0x1485] = 0x87ee, [0x1486] = 0x87f0, [0x1487] = 0x87d6,
  [0x1488] = 0x880e, [0x1489] = 0x87da, [0x148a] = 0x8948, [0x148b] = 0x894a,
  [0x148c] = 0x894e, [0x148d] = 0x894d, [0x148e] = 0x89b1, [0x148f] = 0x89b0,
  [0x1490] = 0x89b3, [0x1491] = 0x4707, [0x1492] = 0x8b38, [0x1493] = 0x8b32,
  [0x1494] = 0x4708, [0x1495] = 0x8b2d, [0x1496] = 0x470a, [0x1497] = 0x8b34,
  [0x1498] = 0x431b, [0x1499] = 0x8b29, [0x149a] = 0x8c74, [0x149b] = 0x4761,
  [0x149c] = 0x4762, [0x149d] = 0x8d03, [0x149e] = 0x47c2, [0x149f] = 0x47c6,
  [0x14a0] = 0x8da9, [0x14a1] = 0x8e58, [0x14a2] = 0x481e, [0x14a3] = 0x4825,
  [0x14a4] = 0x8ebf, [0x14a5] = 0x8ec1, [0x14a6] = 0x8f4a, [0x14a7] = 0x8fac,
  [0x14a8] = 0x48b0, [0x14a9] = 0x9089, [0x14aa] = 0x913d, [0x14ab] = 0x913c,
  [0x14ac] = 0x91a9, [0x14ad] = 0x93a0, [0x14ae] = 0x493d, [0x14af] = 0x9390,
  [0x14b0] = 0x493e, [0x14b1] = 0x9393, [0x14b2] = 0x938b, [0x14b3] = 0x93ad,
  [0x14b4] = 0x93bb, [0x14b5] = 0x93b8, [0x14b6] = 0x4946, [0x14b7] = 0x4945,
  [0x14b8] = 0x939c, [0x14b9] = 0x95d8, [0x14ba] = 0x95d7, [0x14bb] = 0x4a03,
  [0x14bc] = 0x4a26, [0x14bd] = 0x4a27, [0x14be] = 0x975d, [0x14bf] = 0x97a9,
  [0x14c0] = 0x97da, [0x14c1] = 0x4a98, [0x14c2] = 0x4aad, [0x14c3] = 0x4ad5,
  [0x14c4] = 0x4ada, [0x14c5] = 0x9854, [0x14c6] = 0x4ad9, [0x14c7] = 0x9855,
  [0x14c8] = 0x984b, [0x14c9] = 0x4add, [0x14ca] = 0x983f, [0x14cb] = 0x98b9,
  [0x14cc] = 0x4b15, [0x14cd] = 0x4b16, [0x14ce] = 0x4b17, [0x14cf] = 0x4b21,
  [0x14d0] = 0x9938, [0x14d1] = 0x9936, [0x14d2] = 0x9940, [0x14d3] = 0x4b4c,
  [0x14d4] = 0x993b, [0x14d5] = 0x9939, [0x14d6] = 0x99a4, [0x14d7] = 0x4b96,
  [0x14d8] = 0x4b98, [0x14d9] = 0x9a08, [0x14da] = 0x9a0c, [0x14db] = 0x4b9b,
  [0x14dc] = 0x9a10, [0x14dd] = 0x4bff, [0x14de] = 0x9b07, [0x14df] = 0x4c25,
  [0x14e0] = 0x9bd2, [0x14e1] = 0x4c4f, [0x14e2] = 0x9bc2, [0x14e3] = 0x9bbb,
  [0x14e4] = 0x9bcc, [0x14e5] = 0x9bcb, [0x14e6] = 0x4c56, [0x14e7] = 0x4c54,
  [0x14e8] = 0x9d4d, [0x14e9] = 0x9d63, [0x14ea] = 0x9d4e, [0x14eb] = 0x4cd8,
  [0x14ec] = 0x9d50, [0x14ed] = 0x9d55, [0x14ee] = 0x4cd7, [0x14ef] = 0x9d5e,
  [0x14f0] = 0x4d26, [0x14f1] = 0x9e90, [0x14f2] = 0x9eb2, [0x14f3] = 0x9eb1,
  [0x14f4] = 0x4d38, [0x14f5] = 0x9eca, [0x14f6] = 0x9f02, [0x14f7] = 0x9f27,
  [0x14f8] = 0x9f26, [0x14f9] = 0x4d8a, [0x14fa] = 0x56af, [0x14fb] = 0x58e0,
  [0x14fc] = 0x58dc, [0x14fd] = 0x3734, [0x14fe] = 0x5b39, [0x14ff] = 0x3735,
  [0x1500] = 0x2f86a, [0x1501] = 0x5b7c, [0x1502] = 0x5bf3, [0x1503] = 0x2f870,
  [0x1504] = 0x37a1, [0x1505] = 0x5c6b, [0x1506] = 0x5dc4, [0x1507] = 0x650b,
  [0x1508] = 0x6508, [0x1509] = 0x650a, [0x150a] = 0x3a6c, [0x150b] = 0x3a6d,
  [0x150c] = 0x65dc, [0x150d] = 0x3b29, [0x150e] = 0x3b2a, [0x150f] = 0x66e1,
  [0x1510] = 0x66df, [0x1511] = 0x6ace, [0x1512] = 0x6ad4, [0x1513] = 0x6ae3,
  [0x1514] = 0x6ad7, [0x1515] = 0x6ae2, [0x1516] = 0x3c00, [0x1517] = 0x3c08,
  [0x1518] = 0x3c06, [0x1519] = 0x3c05, [0x151a] = 0x6ad8, [0x151b] = 0x6ad5,
  [0x151c] = 0x6ad2, [0x151d] = 0x3cb1, [0x151e] = 0x3d88, [0x151f] = 0x701e,
  [0x1520] = 0x702c, [0x1521] = 0x7025, [0x1522] = 0x6ff3, [0x1523] = 0x7204,
  [0x1524] = 0x7208, [0x1525] = 0x7215, [0x1526] = 0x3e09, [0x1527] = 0x74c4,
  [0x1528] = 0x74c9, [0x1529] = 0x74c7, [0x152a] = 0x74c8, [0x152b] = 0x76a9,
  [0x152c] = 0x77c6, [0x152d] = 0x77c5, [0x152e] = 0x7918, [0x152f] = 0x791a,
  [0x1530] = 0x7920, [0x1531] = 0x4122, [0x1532] = 0x7a66, [0x1533] = 0x7a64,
  [0x1534] = 0x7a6a, [0x1535] = 0x41d5, [0x1536] = 0x4261, [0x1537] = 0x425d,
  [0x1538] = 0x4262, [0x1539] = 0x424f, [0x153a] = 0x4260, [0x153b] = 0x7c35,
  [0x153c] = 0x7c34, [0x153d] = 0x42aa, [0x153e] = 0x4322, [0x153f] = 0x7e6c,
  [0x1540] = 0x4321, [0x1541] = 0x7e6e, [0x1542] = 0x7e71, [0x1543] = 0x4446,
  [0x1544] = 0x81d4, [0x1545] = 0x81d6, [0x1546] = 0x821a, [0x1547] = 0x8262,
  [0x1548] = 0x8265, [0x1549] = 0x8276, [0x154a] = 0x85db, [0x154b] = 0x85d6,
  [0x154c] = 0x4562, [0x154d] = 0x85e7, [0x154e] = 0x4560, [0x154f] = 0x4564,
  [0x1550] = 0x85f4, [0x1551] = 0x2f9b6, [0x1552] = 0x87fd, [0x1553] = 0x87d5,
  [0x1554] = 0x8807, [0x1555] = 0x45f6, [0x1556] = 0x880f, [0x1557] = 0x87f8,
  [0x1558] = 0x2f9c1, [0x1559] = 0x4619, [0x155a] = 0x8987, [0x155b] = 0x4691,
  [0x155c] = 0x89b5, [0x155d] = 0x89f5, [0x155e] = 0x470d, [0x155f] = 0x8b3f,
  [0x1560] = 0x8b43, [0x1561] = 0x8b4c, [0x1562] = 0x4765, [0x1563] = 0x8d0b,
  [0x1564] = 0x8e6b, [0x1565] = 0x8e68, [0x1566] = 0x8e70, [0x1567] = 0x8e75,
  [0x1568] = 0x8e77, [0x1569] = 0x483f, [0x156a] = 0x8ec3, [0x156b] = 0x494b,
  [0x156c] = 0x93e9, [0x156d] = 0x93ea, [0x156e] = 0x93cb, [0x156f] = 0x93c5,
  [0x1570] = 0x93c6, [0x1571] = 0x4948, [0x1572] = 0x93ed, [0x1573] = 0x93d3,
  [0x1574] = 0x4952, [0x1575] = 0x93e5, [0x1576] = 0x494a, [0x1577] = 0x4951,
  [0x1578] = 0x93db, [0x1579] = 0x93eb, [0x157a] = 0x93e0, [0x157b] = 0x93c1,
  [0x157c] = 0x4950, [0x157d] = 0x494c, [0x157e] = 0x95dd, [0x157f] = 0x49ee,
  [0x1580] = 0x4a04, [0x1581] = 0x4a06, [0x1582] = 0x4a2d, [0x1583] = 0x4a2e,
  [0x1584] = 0x4a2f, [0x1585] = 0x4a7b, [0x1586] = 0x4a78, [0x1587] = 0x4a77,
  [0x1588] = 0x97b2, [0x1589] = 0x97b4, [0x158a] = 0x97b1, [0x158b] = 0x97b5,
  [0x158c] = 0x97f2, [0x158d] = 0x4aa2, [0x158e] = 0x4aa1, [0x158f] = 0x4ae3,
  [0x1590] = 0x9856, [0x1591] = 0x4b1a, [0x1592] = 0x4b19, [0x1593] = 0x4b57,
  [0x1594] = 0x9944, [0x1595] = 0x4b9e, [0x1596] = 0x9a26, [0x1597] = 0x9a1f,
  [0x1598] = 0x9a18, [0x1599] = 0x9a21, [0x159a] = 0x9a17, [0x159b] = 0x4bdd,
  [0x159c] = 0x9b09, [0x159d] = 0x4c05, [0x159e] = 0x4c28, [0x159f] = 0x9bc5,
  [0x15a0] = 0x9bdf, [0x15a1] = 0x4c60, [0x15a2] = 0x9be3, [0x15a3] = 0x4c66,
  [0x15a4] = 0x9be9, [0x15a5] = 0x9bee, [0x15a6] = 0x4c67, [0x15a7] = 0x4c68,
  [0x15a8] = 0x9d66, [0x15a9] = 0x9d7a, [0x15aa] = 0x4cde, [0x15ab] = 0x9d6e,
  [0x15ac] = 0x9d91, [0x15ad] = 0x9d83, [0x15ae] = 0x9d76, [0x15af] = 0x9d7e,
  [0x15b0] = 0x9d6d, [0x15b1] = 0x4ce1, [0x15b2] = 0x9e95, [0x15b3] = 0x9ee3,
  [0x15b4] = 0x4d69, [0x15b5] = 0x4d77, [0x15b6] = 0x9f03, [0x15b7] = 0x9f04,
  [0x15b8] = 0x2fa19, [0x15b9] = 0x9f17, [0x15ba] = 0x34a6, [0x15bb] = 0x5136,
  [0x15bc] = 0x34a5, [0x15bd] = 0x5336, [0x15be] = 0x3614, [0x15bf] = 0x5b42,
  [0x15c0] = 0x3736, [0x15c1] = 0x3738, [0x15c2] = 0x5b44, [0x15c3] = 0x5b46,
  [0x15c4] = 0x5b7e, [0x15c5] = 0x5dca, [0x15c6] = 0x5dc8, [0x15c7] = 0x5dcc,
  [0x15c8] = 0x5ef0, [0x15c9] = 0x3a70, [0x15ca] = 0x6585, [0x15cb] = 0x66e5,
  [0x15cc] = 0x66e7, [0x15cd] = 0x3b2b, [0x15ce] = 0x3c11, [0x15cf] = 0x3c0a,
  [0x15d0] = 0x6af4, [0x15d1] = 0x3c0d, [0x15d2] = 0x6ae9, [0x15d3] = 0x3c16,
  [0x15d4] = 0x3c10, [0x15d5] = 0x3c09, [0x15d6] = 0x3c0e, [0x15d7] = 0x3c7a,
  [0x15d8] = 0x703d, [0x15d9] = 0x3d8c, [0x15da] = 0x7036, [0x15db] = 0x3d91,
  [0x15dc] = 0x7216, [0x15dd] = 0x3e0a, [0x15de] = 0x7212, [0x15df] = 0x720f,
  [0x15e0] = 0x7217, [0x15e1] = 0x7211, [0x15e2] = 0x720b, [0x15e3] = 0x3e08,
  [0x15e4] = 0x3e0b, [0x15e5] = 0x74cd, [0x15e6] = 0x74d0, [0x15e7] = 0x74cc,
  [0x15e8] = 0x74ce, [0x15e9] = 0x74d1, [0x15ea] = 0x3f07, [0x15eb] = 0x7589,
  [0x15ec] = 0x40f2, [0x15ed] = 0x7a6f, [0x15ee] = 0x7c4b, [0x15ef] = 0x7c44,
  [0x15f0] = 0x7c55, [0x15f1] = 0x42ae, [0x15f2] = 0x4324, [0x15f3] = 0x4326,
  [0x15f4] = 0x4327, [0x15f5] = 0x7e7f, [0x15f6] = 0x8b71, [0x15f7] = 0x4399,
  [0x15f8] = 0x802f, [0x15f9] = 0x807a, [0x15fa] = 0x807b, [0x15fb] = 0x807c,
  [0x15fc] = 0x455f, [0x15fd] = 0x456a, [0x15fe] = 0x4571, [0x15ff] = 0x85fc,
  [0x1600] = 0x8610, [0x1601] = 0x8602, [0x1602] = 0x456c, [0x1603] = 0x456f,
  [0x1604] = 0x85ee, [0x1605] = 0x8603, [0x1606] = 0x4568, [0x1607] = 0x860d,
  [0x1608] = 0x8613, [0x1609] = 0x8608, [0x160a] = 0x860f, [0x160b] = 0x8818,
  [0x160c] = 0x8812, [0x160d] = 0x4601, [0x160e] = 0x4668, [0x160f] = 0x8967,
  [0x1610] = 0x8965, [0x1611] = 0x89bb, [0x1612] = 0x8b69, [0x1613] = 0x8b62,
  [0x1614] = 0x4713, [0x1615] = 0x8b6e, [0x1616] = 0x4716, [0x1617] = 0x8b61,
  [0x1618] = 0x4718, [0x1619] = 0x8b64, [0x161a] = 0x8b4d, [0x161b] = 0x8c51,
  [0x161c] = 0x4789, [0x161d] = 0x47c8, [0x161e] = 0x8e83, [0x161f] = 0x8ec6,
  [0x1620] = 0x4884, [0x1621] = 0x941f, [0x1622] = 0x4954, [0x1623] = 0x9404,
  [0x1624] = 0x9417, [0x1625] = 0x9408, [0x1626] = 0x9405, [0x1627] = 0x4956,
  [0x1628] = 0x93f3, [0x1629] = 0x941e, [0x162a] = 0x9402, [0x162b] = 0x941a,
  [0x162c] = 0x941b, [0x162d] = 0x9427, [0x162e] = 0x941c, [0x162f] = 0x495a,
  [0x1630] = 0x96b5, [0x1631] = 0x4a05, [0x1632] = 0x4a07, [0x1633] = 0x9733,
  [0x1634] = 0x4a31, [0x1635] = 0x9734, [0x1636] = 0x9731, [0x1637] = 0x97b8,
  [0x1638] = 0x97ba, [0x1639] = 0x4aa3, [0x163a] = 0x97fc, [0x163b] = 0x4aeb,
  [0x163c] = 0x4b1c, [0x163d] = 0x98c3, [0x163e] = 0x4b5a, [0x163f] = 0x994d,
  [0x1640] = 0x4b5b, [0x1641] = 0x9a2f, [0x1642] = 0x4ba6, [0x1643] = 0x4baa,
  [0x1644] = 0x4ba5, [0x1645] = 0x9ac9, [0x1646] = 0x4be1, [0x1647] = 0x9ac8,
  [0x1648] = 0x9ac4, [0x1649] = 0x9b2a, [0x164a] = 0x9b38, [0x164b] = 0x9b50,
  [0x164c] = 0x4c2a, [0x164d] = 0x9c0a, [0x164e] = 0x9bfb, [0x164f] = 0x9c04,
  [0x1650] = 0x9bfc, [0x1651] = 0x9bfe, [0x1652] = 0x4c72, [0x1653] = 0x4c6f,
  [0x1654] = 0x4c73, [0x1655] = 0x9c02, [0x1656] = 0x9bf6, [0x1657] = 0x9c1b,
  [0x1658] = 0x9bf9, [0x1659] = 0x9c15, [0x165a] = 0x9c10, [0x165b] = 0x9bff,
  [0x165c] = 0x9c00, [0x165d] = 0x9c0c, [0x165e] = 0x4c6b, [0x165f] = 0x4ce6,
  [0x1660] = 0x9d95, [0x1661] = 0x9da5, [0x1662] = 0x4ce9, [0x1663] = 0x4cec,
  [0x1664] = 0x4ce8, [0x1665] = 0x4cf0, [0x1666] = 0x9e98, [0x1667] = 0x9ec1,
  [0x1668] = 0x4d8c, [0x1669] = 0x9f5a, [0x166a] = 0x5164, [0x166b] = 0x56bb,
  [0x166c] = 0x3615, [0x166d] = 0x58e6, [0x166e] = 0x5b49, [0x166f] = 0x5bf7,
  [0x1670] = 0x3771, [0x1671] = 0x3826, [0x1672] = 0x5dd0, [0x1673] = 0x38c6,
  [0x1674] = 0x5fc2, [0x1675] = 0x39a8, [0x1676] = 0x6511, [0x1677] = 0x3a73,
  [0x1678] = 0x6aff, [0x1679] = 0x6afe, [0x167a] = 0x6afd, [0x167b] = 0x3c15,
  [0x167c] = 0x6b01, [0x167d] = 0x3d98, [0x167e] = 0x3d97, [0x167f] = 0x704b,
  [0x1680] = 0x704d, [0x1681] = 0x7047, [0x1682] = 0x74d3, [0x1683] = 0x7668,
  [0x1684] = 0x7667, [0x1685] = 0x3fd7, [0x1686] = 0x4080, [0x1687] = 0x77d1,
  [0x1688] = 0x7930, [0x1689] = 0x7932, [0x168a] = 0x792e, [0x168b] = 0x4188,
  [0x168c] = 0x9f9d, [0x168d] = 0x7ac9, [0x168e] = 0x7ac8, [0x168f] = 0x4269,
  [0x1690] = 0x7c56, [0x1691] = 0x7c51, [0x1692] = 0x426b, [0x1693] = 0x4329,
  [0x1694] = 0x4328, [0x1695] = 0x7e85, [0x1696] = 0x7e89, [0x1697] = 0x7e8e,
  [0x1698] = 0x7e84, [0x1699] = 0x445f, [0x169a] = 0x826a, [0x169b] = 0x862b,
  [0x169c] = 0x862f, [0x169d] = 0x8628, [0x169e] = 0x4574, [0x169f] = 0x8616,
  [0x16a0] = 0x8615, [0x16a1] = 0x861d, [0x16a2] = 0x881a, [0x16a3] = 0x4602,
  [0x16a4] = 0x466a, [0x16a5] = 0x4694, [0x16a6] = 0x89bc, [0x16a7] = 0x8b75,
  [0x16a8] = 0x8b7c, [0x16a9] = 0x478a, [0x16aa] = 0x8d11, [0x16ab] = 0x8d12,
  [0x16ac] = 0x8f5c, [0x16ad] = 0x91bb, [0x16ae] = 0x4964, [0x16af] = 0x93f4,
  [0x16b0] = 0x495e, [0x16b1] = 0x4961, [0x16b2] = 0x942d, [0x16b3] = 0x4965,
  [0x16b4] = 0x4966, [0x16b5] = 0x96e4, [0x16b6] = 0x9737, [0x16b7] = 0x9736,
  [0x16b8] = 0x9767, [0x16b9] = 0x97be, [0x16ba] = 0x97bd, [0x16bb] = 0x97e2,
  [0x16bc] = 0x9868, [0x16bd] = 0x9866, [0x16be] = 0x98c8, [0x16bf] = 0x98ca,
  [0x16c0] = 0x98c7, [0x16c1] = 0x98dc, [0x16c2] = 0x4b5f, [0x16c3] = 0x994f,
  [0x16c4] = 0x99a9, [0x16c5] = 0x9a3c, [0x16c6] = 0x4baf, [0x16c7] = 0x9a3b,
  [0x16c8] = 0x9ace, [0x16c9] = 0x4c0d, [0x16ca] = 0x9b14, [0x16cb] = 0x9b53,
  [0x16cc] = 0x4c7c, [0x16cd] = 0x9c2e, [0x16ce] = 0x4c7a, [0x16cf] = 0x9c1f,
  [0x16d0] = 0x4c76, [0x16d1] = 0x4c79, [0x16d2] = 0x4c7d, [0x16d3] = 0x4c77,
  [0x16d4] = 0x9db0, [0x16d5] = 0x9dbd, [0x16d6] = 0x4cf6, [0x16d7] = 0x4cf1,
  [0x16d8] = 0x9dae, [0x16d9] = 0x9dc4, [0x16da] = 0x9e7b, [0x16db] = 0x400b,
  [0x16dc] = 0x4d29, [0x16dd] = 0x9e9e, [0x16de] = 0x4d6f, [0x16df] = 0x9f05,
  [0x16e0] = 0x4d9a, [0x16e1] = 0x9f69, [0x16e2] = 0x9fa1, [0x16e3] = 0x56c7,
  [0x16e4] = 0x571d, [0x16e5] = 0x5b4a, [0x16e6] = 0x5dd3, [0x16e7] = 0x3869,
  [0x16e8] = 0x5f72, [0x16e9] = 0x6202, [0x16ea] = 0x39ab, [0x16eb] = 0x6235,
  [0x16ec] = 0x6527, [0x16ed] = 0x651e, [0x16ee] = 0x651f, [0x16ef] = 0x3b2c,
  [0x16f0] = 0x3b2d, [0x16f1] = 0x6b07, [0x16f2] = 0x6b06, [0x16f3] = 0x3c17,
  [0x16f4] = 0x3d9a, [0x16f5] = 0x7054, [0x16f6] = 0x721c, [0x16f7] = 0x7220,
  [0x16f8] = 0x7af8, [0x16f9] = 0x426e, [0x16fa] = 0x7c5d, [0x16fb] = 0x7c58,
  [0x16fc] = 0x432c, [0x16fd] = 0x7e92, [0x16fe] = 0x7f4e, [0x16ff] = 0x43ca,
  [0x1700] = 0x4578, [0x1701] = 0x4606, [0x1702] = 0x8827, [0x1703] = 0x4607,
  [0x1704] = 0x8b81, [0x1705] = 0x8b83, [0x1706] = 0x4720, [0x1707] = 0x8c44,
  [0x1708] = 0x4753, [0x1709] = 0x47ce, [0x170a] = 0x487a, [0x170b] = 0x4879,
  [0x170c] = 0x9442, [0x170d] = 0x944d, [0x170e] = 0x9454, [0x170f] = 0x944e,
  [0x1710] = 0x496b, [0x1711] = 0x9443, [0x1712] = 0x4967, [0x1713] = 0x496d,
  [0x1714] = 0x973c, [0x1715] = 0x9740, [0x1716] = 0x97c0, [0x1717] = 0x4a85,
  [0x1718] = 0x4ab0, [0x1719] = 0x4af3, [0x171a] = 0x4b63, [0x171b] = 0x995a,
  [0x171c] = 0x9a51, [0x171d] = 0x4bb6, [0x171e] = 0x9add, [0x171f] = 0x4c82,
  [0x1720] = 0x4c7f, [0x1721] = 0x9c38, [0x1722] = 0x4c86, [0x1723] = 0x9c45,
  [0x1724] = 0x9c3a, [0x1725] = 0x4c84, [0x1726] = 0x9c35, [0x1727] = 0x4cfc,
  [0x1728] = 0x4cfd, [0x1729] = 0x4cfa, [0x172a] = 0x9ef1, [0x172b] = 0x4d87,
  [0x172c] = 0x9f93, [0x172d] = 0x529a, [0x172e] = 0x361a, [0x172f] = 0x3619,
  [0x1730] = 0x8641, [0x1731] = 0x5dd7, [0x1732] = 0x3a75, [0x1733] = 0x6528,
  [0x1734] = 0x3c1a, [0x1735] = 0x3c1b, [0x1736] = 0x3c19, [0x1737] = 0x7053,
  [0x1738] = 0x7059, [0x1739] = 0x3d9c, [0x173a] = 0x7221, [0x173b] = 0x3e10,
  [0x173c] = 0x766f, [0x173d] = 0x7937, [0x173e] = 0x79b5, [0x173f] = 0x7c62,
  [0x1740] = 0x7c5e, [0x1741] = 0x7cf5, [0x1742] = 0x457b, [0x1743] = 0x457c,
  [0x1744] = 0x863d, [0x1745] = 0x4608, [0x1746] = 0x882d, [0x1747] = 0x8989,
  [0x1748] = 0x8b8d, [0x1749] = 0x8b87, [0x174a] = 0x8b90, [0x174b] = 0x8d1a,
  [0x174c] = 0x8e99, [0x174d] = 0x4841, [0x174e] = 0x48e3, [0x174f] = 0x4972,
  [0x1750] = 0x945f, [0x1751] = 0x4973, [0x1752] = 0x4968, [0x1753] = 0x9456,
  [0x1754] = 0x9461, [0x1755] = 0x945b, [0x1756] = 0x945a, [0x1757] = 0x945c,
  [0x1758] = 0x9465, [0x1759] = 0x4a35, [0x175a] = 0x9741, [0x175b] = 0x4a88,
  [0x175c] = 0x4a9d, [0x175d] = 0x986e, [0x175e] = 0x986c, [0x175f] = 0x986d,
  [0x1760] = 0x4275, [0x1761] = 0x99aa, [0x1762] = 0x9a5c, [0x1763] = 0x9a58,
  [0x1764] = 0x9ade, [0x1765] = 0x4c8f, [0x1766] = 0x9c4f, [0x1767] = 0x9c51,
  [0x1768] = 0x4c8e, [0x1769] = 0x9c53, [0x176a] = 0x4d05, [0x176b] = 0x4d04,
  [0x176c] = 0x4cff, [0x176d] = 0x9dfc, [0x176e] = 0x9f39, [0x176f] = 0x4d9e,
  [0x1770] = 0x513e, [0x1771] = 0x3554, [0x1772] = 0x56d2, [0x1773] = 0x3681,
  [0x1774] = 0x5b4f, [0x1775] = 0x6b14, [0x1776] = 0x40fa, [0x1777] = 0x7a72,
  [0x1778] = 0x7a73, [0x1779] = 0x4332, [0x177a] = 0x4670, [0x177b] = 0x466e,
  [0x177c] = 0x8b91, [0x177d] = 0x2f9d6, [0x177e] = 0x487c, [0x177f] = 0x91bf,
  [0x1780] = 0x4975, [0x1781] = 0x946c, [0x1782] = 0x4974, [0x1783] = 0x4977,
  [0x1784] = 0x96e6, [0x1785] = 0x9745, [0x1786] = 0x4a37, [0x1787] = 0x97c8,
  [0x1788] = 0x97e4, [0x1789] = 0x995d, [0x178a] = 0x4bba, [0x178b] = 0x9b21,
  [0x178c] = 0x4c11, [0x178d] = 0x9b2c, [0x178e] = 0x9b57, [0x178f] = 0x4c92,
  [0x1790] = 0x4c99, [0x1791] = 0x9c5d, [0x1792] = 0x9c61, [0x1793] = 0x9c65,
  [0x1794] = 0x9e08, [0x1795] = 0x4d0a, [0x1796] = 0x4d2a, [0x1797] = 0x4d2b,
  [0x1798] = 0x4d44, [0x1799] = 0x4d79, [0x179a] = 0x9f45, [0x179b] = 0x34aa,
  [0x179c] = 0x3748, [0x179d] = 0x6205, [0x179e] = 0x66ef, [0x179f] = 0x6b1b,
  [0x17a0] = 0x6b1d, [0x17a1] = 0x7225, [0x17a2] = 0x7224, [0x17a3] = 0x7c6d,
  [0x17a4] = 0x42b4, [0x17a5] = 0x8642, [0x17a6] = 0x8649, [0x17a7] = 0x460d,
  [0x17a8] = 0x8978, [0x17a9] = 0x898a, [0x17aa] = 0x8b97, [0x17ab] = 0x4754,
  [0x17ac] = 0x8c9b, [0x17ad] = 0x8d1c, [0x17ae] = 0x4830, [0x17af] = 0x8ea2,
  [0x17b0] = 0x4a09, [0x17b1] = 0x4a38, [0x17b2] = 0x4a36, [0x17b3] = 0x4a8b,
  [0x17b4] = 0x4af7, [0x17b5] = 0x4b66, [0x17b6] = 0x4bbd, [0x17b7] = 0x4c1e,
  [0x17b8] = 0x9c6c, [0x17b9] = 0x4c96, [0x17ba] = 0x9c6f, [0x17bb] = 0x4d0d,
  [0x17bc] = 0x9e0e, [0x17bd] = 0x4d73, [0x17be] = 0x9f08, [0x17bf] = 0x9f1d,
  [0x17c0] = 0x9fa3, [0x17c1] = 0x373b, [0x17c2] = 0x373c, [0x17c3] = 0x5f60,
  [0x17c4] = 0x6b1c, [0x17c5] = 0x3da0, [0x17c6] = 0x40fb, [0x17c8] = 0x7cf3,
  [0x17c9] = 0x4581, [0x17ca] = 0x8b9b, [0x17cb] = 0x8ea7, [0x17cc] = 0x91c4,
  [0x17cd] = 0x4978, [0x17ce] = 0x947a, [0x17cf] = 0x4a8d, [0x17d0] = 0x4b73,
  [0x17d1] = 0x9a61, [0x17d2] = 0x9a63, [0x17d3] = 0x9ad7, [0x17d4] = 0x9c76,
  [0x17d5] = 0x4da6, [0x17d6] = 0x9fa5, [0x17d7] = 0x39ad, [0x17d8] = 0x7067,
  [0x17d9] = 0x3e11, [0x17da] = 0x72ab, [0x17db] = 0x864a, [0x17dc] = 0x897d,
  [0x17dd] = 0x8b9d, [0x17de] = 0x8c53, [0x17df] = 0x8f65, [0x17e0] = 0x947b,
  [0x17e1] = 0x4a39, [0x17e2] = 0x98cd, [0x17e3] = 0x98dd, [0x17e4] = 0x4bbf,
  [0x17e5] = 0x9b30, [0x17e6] = 0x9e16, [0x17e7] = 0x4d0f, [0x17e8] = 0x4da7,
  [0x17e9] = 0x4db5, [0x17ea] = 0x3fdc, [0x17eb] = 0x4831, [0x17ec] = 0x96e7,
  [0x17ed] = 0x9e18, [0x17ee] = 0x9ea2, [0x17ef] = 0x4da8, [0x17f0] = 0x9f7c,
  [0x17f1] = 0x4125, [0x17f2] = 0x7e9e, [0x17f3] = 0x9484, [0x17f4] = 0x4bc1,
  [0x17f5] = 0x9e1c, [0x17f6] = 0x4190, [0x17f7] = 0x7c71, [0x17f8] = 0x97ca,
  [0x17f9] = 0x4696, [0x17fa] = 0x487f, [0x17fb] = 0x4d10, [0x17fc] = 0x9ea3,
  [0x17fd] = 0x4a0a, [0x17fe] = 0x9c7b, [0x17ff] = 0x9f97, [0x1800] = 0x4d12,
  [0x1801] = 0x4a3a, [0x1802] = 0x9750, [0x1803] = 0x4a3b, [0x18c1] = 0x4f66,
  [0x18c2] = 0x4f68, [0x18c3] = 0x4fe7, [0x18c4] = 0x503f, [0x18c6] = 0x50a6,
  [0x18c7] = 0x510f, [0x18c8] = 0x523e, [0x18c9] = 0x5324, [0x18ca] = 0x5365,
  [0x18cb] = 0x539b, [0x18cc] = 0x517f, [0x18cd] = 0x54cb, [0x18ce] = 0x5573,
  [0x18cf] = 0x5571, [0x18d0] = 0x556b, [0x18d1] = 0x55f4, [0x18d2] = 0x5622,
  [0x18d3] = 0x5620, [0x18d4] = 0x5692, [0x18d5] = 0x56ba, [0x18d6] = 0x5691,
  [0x18d7] = 0x56b0, [0x18d8] = 0x5759, [0x18d9] = 0x578a, [0x18da] = 0x580f,
  [0x18db] = 0x5812, [0x18dc] = 0x5813, [0x18dd] = 0x5847, [0x18de] = 0x589b,
  [0x18df] = 0x5900, [0x18e0] = 0x594d, [0x18e1] = 0x5ad1, [0x18e2] = 0x5ad3,
  [0x18e3] = 0x5b67, [0x18e4] = 0x5c57, [0x18e5] = 0x5c77, [0x18e6] = 0x5cd5,
  [0x18e7] = 0x5d75, [0x18e8] = 0x5d8e, [0x18e9] = 0x5da5, [0x18ea] = 0x5db6,
  [0x18eb] = 0x5dbf, [0x18ec] = 0x5e65, [0x18ed] = 0x5ecd, [0x18ee] = 0x5eed,
  [0x18ef] = 0x5f94, [0x18f0] = 0x5f9a, [0x18f1] = 0x5fba, [0x18f2] = 0x6125,
  [0x18f3] = 0x6150, [0x18f4] = 0x62a3, [0x18f5] = 0x6360, [0x18f6] = 0x6364,
  [0x18f7] = 0x63b6, [0x18f8] = 0x6403, [0x18f9] = 0x64b6, [0x18fa] = 0x651a,
  [0x18fb] = 0x7a25, [0x18fc] = 0x5c21, [0x18fd] = 0x66e2, [0x18fe] = 0x6702,
  [0x18ff] = 0x67a4, [0x1900] = 0x67ac, [0x1901] = 0x6810, [0x1902] = 0x6806,
  [0x1903] = 0x685e, [0x1904] = 0x685a, [0x1905] = 0x692c, [0x1906] = 0x6929,
  [0x1907] = 0x6a2d, [0x1908] = 0x6a77, [0x1909] = 0x6a7a, [0x190a] = 0x6aca,
  [0x190b] = 0x6ae6, [0x190c] = 0x6af5, [0x190d] = 0x6b0d, [0x190e] = 0x6b0e,
  [0x190f] = 0x6bdc, [0x1910] = 0x6bdd, [0x1911] = 0x6bf6, [0x1912] = 0x6c1e,
  [0x1913] = 0x6c63, [0x1914] = 0x6da5, [0x1915] = 0x6e0f, [0x1916] = 0x6e8a,
  [0x1917] = 0x6e84, [0x1918] = 0x6e8b, [0x1919] = 0x6e7c, [0x191a] = 0x6f4c,
  [0x191b] = 0x6f48, [0x191c] = 0x6f49, [0x191d] = 0x6f9d, [0x191e] = 0x6f99,
  [0x191f] = 0x6ff8, [0x1920] = 0x702e, [0x1921] = 0x702d, [0x1922] = 0x705c,
  [0x1923] = 0x79cc, [0x1924] = 0x70bf, [0x1925] = 0x70ea, [0x1926] = 0x70e5,
  [0x1927] = 0x7111, [0x1928] = 0x7112, [0x1929] = 0x713f, [0x192a] = 0x7139,
  [0x192b] = 0x713b, [0x192c] = 0x713d, [0x192d] = 0x7177, [0x192e] = 0x7175,
  [0x192f] = 0x7176, [0x1930] = 0x7171, [0x1931] = 0x7196, [0x1932] = 0x7193,
  [0x1933] = 0x71b4, [0x1934] = 0x71dd, [0x1935] = 0x71de, [0x1936] = 0x720e,
  [0x1937] = 0x5911, [0x1938] = 0x7218, [0x1939] = 0x7347, [0x193a] = 0x7348,
  [0x193b] = 0x73ef, [0x193c] = 0x7412, [0x193d] = 0x743b, [0x193e] = 0x74a4,
  [0x193f] = 0x748d, [0x1940] = 0x74b4, [0x1941] = 0x7673, [0x1942] = 0x7677,
  [0x1943] = 0x76bc, [0x1944] = 0x7819, [0x1945] = 0x781b, [0x1946] = 0x783d,
  [0x1947] = 0x7853, [0x1948] = 0x7854, [0x1949] = 0x7858, [0x194a] = 0x78b7,
  [0x194b] = 0x78d8, [0x194c] = 0x78ee, [0x194d] = 0x7922, [0x194e] = 0x794d,
  [0x194f] = 0x7986, [0x1950] = 0x7999, [0x1951] = 0x79a3, [0x1952] = 0x79bc,
  [0x1953] = 0x7aa7, [0x1954] = 0x7b37, [0x1955] = 0x7b59, [0x1956] = 0x7bd0,
  [0x1957] = 0x7c2f, [0x1958] = 0x7c32, [0x1959] = 0x7c42, [0x195a] = 0x7c4e,
  [0x195b] = 0x7c68, [0x195c] = 0x7ca9, [0x195d] = 0x7ced, [0x195e] = 0x7dd0,
  [0x195f] = 0x7e07, [0x1960] = 0x7dd3, [0x1961] = 0x7e64, [0x1962] = 0x7f40,
  [0x1964] = 0x8041, [0x1965] = 0x8063, [0x1966] = 0x80bb, [0x1967] = 0x6711,
  [0x1968] = 0x6725, [0x1969] = 0x8248, [0x196a] = 0x8310, [0x196b] = 0x8362,
  [0x196c] = 0x8312, [0x196d] = 0x8421, [0x196e] = 0x841e, [0x196f] = 0x84e2,
  [0x1970] = 0x84de, [0x1971] = 0x84e1, [0x1972] = 0x8573, [0x1973] = 0x85d4,
  [0x1974] = 0x85f5, [0x1975] = 0x8637, [0x1976] = 0x8645, [0x1977] = 0x8672,
  [0x1978] = 0x874a, [0x1979] = 0x87a9, [0x197a] = 0x87a5, [0x197b] = 0x87f5,
  [0x197c] = 0x8834, [0x197d] = 0x8850, [0x197e] = 0x8887, [0x197f] = 0x8954,
  [0x1980] = 0x8984, [0x1981] = 0x8b03, [0x1982] = 0x8c52, [0x1983] = 0x8cd8,
  [0x1984] = 0x8d0c, [0x1985] = 0x8d18, [0x1986] = 0x8db0, [0x1987] = 0x8ebc,
  [0x1988] = 0x8ed5, [0x1989] = 0x8faa, [0x198a] = 0x909c, [0x198c] = 0x915c,
  [0x198d] = 0x922b, [0x198e] = 0x9221, [0x198f] = 0x9273, [0x1990] = 0x92f4,
  [0x1991] = 0x92f5, [0x1992] = 0x933f, [0x1993] = 0x9342, [0x1994] = 0x9386,
  [0x1995] = 0x93be, [0x1996] = 0x93bc, [0x1997] = 0x93bd, [0x1998] = 0x93f1,
  [0x1999] = 0x93f2, [0x199a] = 0x93ef, [0x199b] = 0x9422, [0x199c] = 0x9423,
  [0x199d] = 0x9424, [0x199e] = 0x9467, [0x199f] = 0x9466, [0x19a0] = 0x9597,
  [0x19a1] = 0x95ce, [0x19a2] = 0x95e7, [0x19a3] = 0x973b, [0x19a4] = 0x974d,
  [0x19a5] = 0x98e4, [0x19a6] = 0x9942, [0x19a7] = 0x9b1d, [0x19a8] = 0x9b98,
  [0x19aa] = 0x9d49, [0x19ab] = 0x6449, [0x19ac] = 0x5e71, [0x19ad] = 0x5e85,
  [0x19ae] = 0x61d3, [0x19af] = 0x990e, [0x19b0] = 0x8002, [0x19b1] = 0x781e,
  [0x19b4] = 0x5528, [0x19b5] = 0x5572, [0x19b6] = 0x55ba, [0x19b7] = 0x55f0,
  [0x19b8] = 0x55ee, [0x19b9] = 0x56b8, [0x19ba] = 0x56b9, [0x19bb] = 0x56c4,
  [0x19bc] = 0x8053, [0x19bd] = 0x92b0,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x4' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x42121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l4_to_ucs4_tab[] =
{
  [0x0000] = 0x20086, [0x0001] = 0x4e40, [0x0002] = 0x4e41, [0x0003] = 0x4e5a,
  [0x0004] = 0x21fe8, [0x0005] = 0x4e02, [0x0006] = 0x4e29, [0x0007] = 0x2010e,
  [0x0008] = 0x20627, [0x0009] = 0x5202, [0x000a] = 0x353e, [0x000b] = 0x5ddc,
  [0x000c] = 0x2053c, [0x000d] = 0x20675, [0x000e] = 0x2f828, [0x000f] = 0x5342,
  [0x0010] = 0x536a, [0x0011] = 0x5b52, [0x0012] = 0x2193c, [0x0013] = 0x2f875,
  [0x0014] = 0x2f878, [0x0015] = 0x5fc4, [0x0016] = 0x624c, [0x0017] = 0x72ad,
  [0x0018] = 0x4e12, [0x0019] = 0x4e2f, [0x001a] = 0x4e96, [0x001b] = 0x4ed0,
  [0x001c] = 0x5142, [0x001d] = 0x5183, [0x001e] = 0x2f819, [0x001f] = 0x206a5,
  [0x0020] = 0x206aa, [0x0021] = 0x2090f, [0x0022] = 0x20a0e, [0x0023] = 0x5383,
  [0x0024] = 0x53b8, [0x0025] = 0x20b9c, [0x0026] = 0x20b9b, [0x0027] = 0x2123c,
  [0x0028] = 0x5928, [0x0029] = 0x21bc2, [0x002a] = 0x5c23, [0x002b] = 0x5e01,
  [0x002c] = 0x5f00, [0x002d] = 0x233b4, [0x002e] = 0x3cb8, [0x002f] = 0x706c,
  [0x0030] = 0x722b, [0x0031] = 0x5188, [0x0032] = 0x8279, [0x0033] = 0x8fb6,
  [0x0034] = 0x4e17, [0x0035] = 0x20065, [0x0036] = 0x340c, [0x0037] = 0x201b2,
  [0x0038] = 0x3430, [0x0039] = 0x4ee2, [0x003a] = 0x4edb, [0x003b] = 0x20477,
  [0x003c] = 0x20542, [0x003d] = 0x51ad, [0x003e] = 0x20633, [0x003f] = 0x51f7,
  [0x0040] = 0x34da, [0x0041] = 0x206af, [0x0042] = 0x20836, [0x0043] = 0x3513,
  [0x0044] = 0x531b, [0x0045] = 0x5388, [0x0046] = 0x5387, [0x0047] = 0x20b22,
  [0x0048] = 0x53cf, [0x0049] = 0x53fd, [0x004a] = 0x3563, [0x004b] = 0x53e7,
  [0x004c] = 0x56dc, [0x004d] = 0x211a3, [0x004e] = 0x56d9, [0x004f] = 0x5725,
  [0x0050] = 0x5727, [0x0051] = 0x5933, [0x0052] = 0x5c13, [0x0053] = 0x21c25,
  [0x0054] = 0x21c24, [0x0055] = 0x5c75, [0x0056] = 0x22052, [0x0057] = 0x22189,
  [0x0058] = 0x224bf, [0x0059] = 0x39c4, [0x005a] = 0x39c3, [0x005b] = 0x66f1,
  [0x005c] = 0x233b5, [0x005d] = 0x2574c, [0x005e] = 0x7f52, [0x005f] = 0x28e14,
  [0x0060] = 0x28e12, [0x0061] = 0x3401, [0x0062] = 0x200a3, [0x0063] = 0x200a2,
  [0x0064] = 0x4e51, [0x0065] = 0x4e6a, [0x0066] = 0x201b6, [0x0067] = 0x4f0c,
  [0x0068] = 0x201b5, [0x0069] = 0x201b4, [0x006a] = 0x4efe, [0x006b] = 0x4f1b,
  [0x006c] = 0x201c2, [0x006d] = 0x201b8, [0x006e] = 0x343a, [0x006f] = 0x20479,
  [0x0070] = 0x34ab, [0x0071] = 0x5173, [0x0072] = 0x20508, [0x0073] = 0x518e,
  [0x0074] = 0x2057a, [0x0075] = 0x205b6, [0x0076] = 0x206bd, [0x0077] = 0x206b7,
  [0x0078] = 0x206bc, [0x0079] = 0x34dd, [0x007a] = 0x206c4, [0x007b] = 0x206c9,
  [0x007c] = 0x52a5, [0x007d] = 0x3515, [0x007e] = 0x52a7, [0x007f] = 0x52a4,
  [0x0080] = 0x20930, [0x0081] = 0x20983, [0x0082] = 0x20985, [0x0083] = 0x20a12,
  [0x0084] = 0x20a34, [0x0085] = 0x53bd, [0x0086] = 0x20ae4, [0x0087] = 0x20ade,
  [0x0088] = 0x20b2a, [0x0089] = 0x5402, [0x008a] = 0x20baf, [0x008b] = 0x20bb4,
  [0x008c] = 0x20bb3, [0x008d] = 0x20bb1, [0x008e] = 0x572b, [0x008f] = 0x591b,
  [0x0090] = 0x5935, [0x0091] = 0x215de, [0x0092] = 0x36a7, [0x0093] = 0x36a5,
  [0x0094] = 0x216b8, [0x0095] = 0x216b0, [0x0096] = 0x36a6, [0x0097] = 0x219c2,
  [0x0099] = 0x5c17, [0x009a] = 0x377c, [0x009b] = 0x21c2a, [0x009c] = 0x5c70,
  [0x009d] = 0x5c7d, [0x009e] = 0x37a9, [0x009f] = 0x21fe9, [0x00a0] = 0x5de9,
  [0x00a1] = 0x22057, [0x00a2] = 0x3834, [0x00a3] = 0x3835, [0x00a4] = 0x221b6,
  [0x00a5] = 0x221d7, [0x00a6] = 0x221de, [0x00a7] = 0x38a8, [0x00a8] = 0x5f19,
  [0x00a9] = 0x5f1c, [0x00aa] = 0x5f75, [0x00ab] = 0x224c2, [0x00ac] = 0x225ab,
  [0x00ad] = 0x38ff, [0x00ae] = 0x5fc8, [0x00af] = 0x2298f, [0x00b0] = 0x39c7,
  [0x00b1] = 0x39c6, [0x00b2] = 0x39c8, [0x00b3] = 0x22a6d, [0x00b4] = 0x22eb5,
  [0x00b5] = 0x3ad0, [0x00b6] = 0x3ad1, [0x00b7] = 0x233c0, [0x00b8] = 0x233bf,
  [0x00b9] = 0x3c59, [0x00ba] = 0x239b9, [0x00bb] = 0x23b1c, [0x00bc] = 0x6c12,
  [0x00bd] = 0x3cbd, [0x00be] = 0x23c80, [0x00bf] = 0x24184, [0x00c0] = 0x24185,
  [0x00c1] = 0x3e28, [0x00c2] = 0x72b3, [0x00c3] = 0x24724, [0x00c4] = 0x3ea9,
  [0x00c5] = 0x7390, [0x00c6] = 0x7536, [0x00c7] = 0x2634c, [0x00c8] = 0x43cc,
  [0x00c9] = 0x26af6, [0x00ca] = 0x8281, [0x00cb] = 0x8fb8, [0x00cc] = 0x28670,
  [0x00cd] = 0x48b4, [0x00ce] = 0x28675, [0x00cf] = 0x28e17, [0x00d0] = 0x4e23,
  [0x00d1] = 0x3416, [0x00d2] = 0x342c, [0x00d3] = 0x201f1, [0x00d4] = 0x4f2e,
  [0x00d5] = 0x201da, [0x00d6] = 0x514f, [0x00d7] = 0x205b9, [0x00d8] = 0x51ba,
  [0x00d9] = 0x34df, [0x00da] = 0x34e0, [0x00db] = 0x5222, [0x00dc] = 0x206ce,
  [0x00dd] = 0x206d2, [0x00de] = 0x206d0, [0x00df] = 0x3517, [0x00e0] = 0x20843,
  [0x00e1] = 0x52af, [0x00e2] = 0x52b0, [0x00e3] = 0x52b1, [0x00e4] = 0x2084d,
  [0x00e5] = 0x20934, [0x00e6] = 0x352f, [0x00e7] = 0x209d9, [0x00e8] = 0x5364,
  [0x00e9] = 0x20a18, [0x00ea] = 0x53d3, [0x00eb] = 0x20b38, [0x00ec] = 0x356a,
  [0x00ed] = 0x20bd7, [0x00ee] = 0x3570, [0x00ef] = 0x356d, [0x00f0] = 0x20bdc,
  [0x00f1] = 0x20bcb, [0x00f2] = 0x20bd6, [0x00f3] = 0x20bd8, [0x00f4] = 0x356e,
  [0x00f5] = 0x20bd1, [0x00f6] = 0x2f83e, [0x00f7] = 0x2f83c, [0x00f8] = 0x20bd2,
  [0x00f9] = 0x362b, [0x00fa] = 0x3628, [0x00fb] = 0x21259, [0x00fd] = 0x593f,
  [0x00fe] = 0x215e5, [0x00ff] = 0x215e6, [0x0100] = 0x3692, [0x0101] = 0x215e8,
  [0x0102] = 0x598b, [0x0103] = 0x216bd, [0x0104] = 0x5991, [0x0105] = 0x5995,
  [0x0106] = 0x216be, [0x0107] = 0x21949, [0x0108] = 0x373f, [0x0109] = 0x219cd,
  [0x010a] = 0x5b8a, [0x010b] = 0x374f, [0x010c] = 0x3774, [0x010d] = 0x21b5e,
  [0x010e] = 0x21bd0, [0x010f] = 0x377d, [0x0110] = 0x21d06, [0x0111] = 0x37b7,
  [0x0112] = 0x37a3, [0x0113] = 0x37b0, [0x0114] = 0x37b1, [0x0115] = 0x5c87,
  [0x0116] = 0x37ab, [0x0117] = 0x21d7c, [0x0118] = 0x21d71, [0x0119] = 0x21fef,
  [0x011a] = 0x383a, [0x011b] = 0x3837, [0x011c] = 0x5e0d, [0x011d] = 0x3838,
  [0x011e] = 0x3840, [0x011f] = 0x2218d, [0x0120] = 0x221b7, [0x0121] = 0x5e8e,
  [0x0122] = 0x389f, [0x0123] = 0x22333, [0x0124] = 0x2237f, [0x0125] = 0x5f7a,
  [0x0126] = 0x225f2, [0x0127] = 0x3904, [0x0128] = 0x3909, [0x0129] = 0x3906,
  [0x012a] = 0x38fd, [0x012b] = 0x390a, [0x012c] = 0x3907, [0x012d] = 0x22a2a,
  [0x012e] = 0x22a8b, [0x012f] = 0x39ca, [0x0130] = 0x22a83, [0x0131] = 0x22a8a,
  [0x0132] = 0x6290, [0x0133] = 0x39c9, [0x0134] = 0x22a8e, [0x0135] = 0x629a,
  [0x0136] = 0x22eb6, [0x0137] = 0x653c, [0x0138] = 0x653a, [0x0139] = 0x3a7f,
  [0x013a] = 0x6598, [0x013b] = 0x230d8, [0x013c] = 0x3ad2, [0x013d] = 0x23156,
  [0x013e] = 0x233d2, [0x013f] = 0x6765, [0x0140] = 0x2f8db, [0x0141] = 0x3b43,
  [0x0142] = 0x233c2, [0x0143] = 0x23887, [0x0144] = 0x23881, [0x0145] = 0x239c2,
  [0x0146] = 0x239bc, [0x0147] = 0x23b23, [0x0148] = 0x3cc1, [0x0149] = 0x23c93,
  [0x014a] = 0x3cc5, [0x014b] = 0x3da3, [0x014c] = 0x2418d, [0x014d] = 0x2418e,
  [0x014e] = 0x3e2a, [0x014f] = 0x3e5f, [0x0150] = 0x2473b, [0x0151] = 0x2474d,
  [0x0152] = 0x3e5d, [0x0153] = 0x24734, [0x0154] = 0x24731, [0x0155] = 0x248f2,
  [0x0156] = 0x248f0, [0x0157] = 0x3f17, [0x0158] = 0x24c15, [0x0159] = 0x24d28,
  [0x015a] = 0x3f71, [0x015b] = 0x3f72, [0x015c] = 0x24f25, [0x015d] = 0x250e7,
  [0x015e] = 0x400f, [0x015f] = 0x250e9, [0x0160] = 0x25417, [0x0161] = 0x79c2,
  [0x0162] = 0x4191, [0x0163] = 0x2626c, [0x0164] = 0x26280, [0x0165] = 0x2634b,
  [0x0166] = 0x43b2, [0x0167] = 0x43cf, [0x0168] = 0x43ce, [0x0169] = 0x809e,
  [0x016a] = 0x268de, [0x016b] = 0x268fb, [0x016c] = 0x81eb, [0x016d] = 0x26951,
  [0x016e] = 0x8289, [0x016f] = 0x4496, [0x0170] = 0x26b01, [0x0171] = 0x26b02,
  [0x0172] = 0x8296, [0x0173] = 0x26b05, [0x0174] = 0x8287, [0x0175] = 0x26b03,
  [0x0176] = 0x4497, [0x0177] = 0x28451, [0x0178] = 0x8fc0, [0x0179] = 0x488b,
  [0x017a] = 0x8fc3, [0x017b] = 0x28453, [0x017c] = 0x2867a, [0x017d] = 0x28678,
  [0x017e] = 0x28676, [0x017f] = 0x28679, [0x0180] = 0x2868f, [0x0181] = 0x2867c,
  [0x0182] = 0x9578, [0x0183] = 0x28e1b, [0x0184] = 0x28e1c, [0x0185] = 0x9625,
  [0x0186] = 0x23d92, [0x0187] = 0x4e75, [0x0188] = 0x4e74, [0x0189] = 0x2012e,
  [0x018a] = 0x2012c, [0x018b] = 0x342d, [0x018c] = 0x2020c, [0x018d] = 0x4f99,
  [0x018e] = 0x2020b, [0x018f] = 0x3450, [0x0190] = 0x344b, [0x0191] = 0x20205,
  [0x0192] = 0x344f, [0x0193] = 0x344c, [0x0194] = 0x2f805, [0x0195] = 0x4f71,
  [0x0196] = 0x5153, [0x0197] = 0x51bf, [0x0198] = 0x205c2, [0x0199] = 0x205c3,
  [0x019a] = 0x51c0, [0x019b] = 0x20648, [0x019c] = 0x51ee, [0x019d] = 0x20646,
  [0x019e] = 0x34e4, [0x019f] = 0x34e3, [0x01a0] = 0x206ea, [0x01a1] = 0x34e1,
  [0x01a2] = 0x206eb, [0x01a3] = 0x34e2, [0x01a4] = 0x206ee, [0x01a5] = 0x523d,
  [0x01a6] = 0x3519, [0x01a7] = 0x52bd, [0x01a8] = 0x530c, [0x01a9] = 0x208d8,
  [0x01aa] = 0x24c1e, [0x01ab] = 0x209df, [0x01ac] = 0x209de, [0x01ad] = 0x3541,
  [0x01ae] = 0x7f37, [0x01af] = 0x20a4f, [0x01b0] = 0x53c0, [0x01b1] = 0x355e,
  [0x01b2] = 0x20c0d, [0x01b3] = 0x20c08, [0x01b4] = 0x3579, [0x01b5] = 0x20c09,
  [0x01b6] = 0x546e, [0x01b7] = 0x5483, [0x01b8] = 0x20c12, [0x01b9] = 0x20c65,
  [0x01ba] = 0x545e, [0x01bb] = 0x545d, [0x01bc] = 0x577e, [0x01bd] = 0x5779,
  [0x01be] = 0x21289, [0x01bf] = 0x577a, [0x01c0] = 0x576c, [0x01c1] = 0x21284,
  [0x01c2] = 0x2129e, [0x01c3] = 0x2128b, [0x01c4] = 0x3632, [0x01c5] = 0x5787,
  [0x01c6] = 0x21562, [0x01c7] = 0x591d, [0x01c8] = 0x3694, [0x01c9] = 0x5946,
  [0x01ca] = 0x3697, [0x01cb] = 0x215f9, [0x01cc] = 0x5943, [0x01cd] = 0x215f7,
  [0x01ce] = 0x3696, [0x01cf] = 0x3698, [0x01d0] = 0x21606, [0x01d1] = 0x216dd,
  [0x01d2] = 0x36b2, [0x01d3] = 0x216d9, [0x01d4] = 0x216df, [0x01d5] = 0x216e2,
  [0x01d6] = 0x36b9, [0x01d7] = 0x5b61, [0x01d8] = 0x5b66, [0x01d9] = 0x200f1,
  [0x01da] = 0x5b90, [0x01db] = 0x3775, [0x01dc] = 0x377f, [0x01dd] = 0x377e,
  [0x01de] = 0x5c29, [0x01df] = 0x378f, [0x01e0] = 0x21c3f, [0x01e1] = 0x21c3e,
  [0x01e2] = 0x21d0b, [0x01e3] = 0x37bd, [0x01e4] = 0x5cb2, [0x01e5] = 0x21d83,
  [0x01e6] = 0x37bb, [0x01e7] = 0x37bc, [0x01e8] = 0x21d86, [0x01e9] = 0x5cc0,
  [0x01ea] = 0x2207c, [0x01eb] = 0x383d, [0x01ec] = 0x383e, [0x01ed] = 0x3874,
  [0x01ee] = 0x221f7, [0x01ef] = 0x387a, [0x01f0] = 0x3876, [0x01f1] = 0x3878,
  [0x01f2] = 0x3875, [0x01f3] = 0x221fb, [0x01f4] = 0x2233b, [0x01f5] = 0x38af,
  [0x01f6] = 0x38b0, [0x01f7] = 0x38c7, [0x01f8] = 0x38cc, [0x01f9] = 0x225f9,
  [0x01fa] = 0x225ca, [0x01fb] = 0x3916, [0x01fc] = 0x225ff, [0x01fd] = 0x3912,
  [0x01fe] = 0x391d, [0x01ff] = 0x22609, [0x0200] = 0x22603, [0x0201] = 0x3915,
  [0x0202] = 0x390f, [0x0203] = 0x3914, [0x0204] = 0x601f, [0x0205] = 0x5fe2,
  [0x0206] = 0x22610, [0x0207] = 0x2299a, [0x0208] = 0x2299f, [0x0209] = 0x39b0,
  [0x020a] = 0x39bf, [0x020b] = 0x39c0, [0x020c] = 0x22a87, [0x020d] = 0x39d2,
  [0x020e] = 0x39d9, [0x020f] = 0x22eb7, [0x0210] = 0x3a7a, [0x0211] = 0x22f08,
  [0x0212] = 0x22ef9, [0x0213] = 0x3a81, [0x0214] = 0x3a82, [0x0215] = 0x22ef4,
  [0x0216] = 0x6616, [0x0217] = 0x65f9, [0x0218] = 0x3ada, [0x0219] = 0x26657,
  [0x021a] = 0x6788, [0x021b] = 0x233d6, [0x021c] = 0x679b, [0x021d] = 0x2f8de,
  [0x021e] = 0x676e, [0x021f] = 0x679e, [0x0220] = 0x233d4, [0x0221] = 0x3c22,
  [0x0222] = 0x3c1f, [0x0223] = 0x23890, [0x0224] = 0x2388d, [0x0225] = 0x3c21,
  [0x0226] = 0x6b24, [0x0227] = 0x23960, [0x0228] = 0x239ce, [0x0229] = 0x239cf,
  [0x022a] = 0x3c5c, [0x022b] = 0x6b7d, [0x022c] = 0x239d1, [0x022d] = 0x3c7d,
  [0x022e] = 0x3c8d, [0x022f] = 0x3c8f, [0x0230] = 0x6ce6, [0x0231] = 0x23cca,
  [0x0232] = 0x6ccb, [0x0233] = 0x3cd0, [0x0234] = 0x23cb7, [0x0235] = 0x3cd8,
  [0x0236] = 0x23cb2, [0x0237] = 0x23cbb, [0x0238] = 0x23cb9, [0x0239] = 0x6cb5,
  [0x023a] = 0x3da7, [0x023b] = 0x2419e, [0x023c] = 0x7097, [0x023d] = 0x241b2,
  [0x023e] = 0x709b, [0x023f] = 0x3e12, [0x0240] = 0x2456f, [0x0241] = 0x245ac,
  [0x0242] = 0x3e2f, [0x0243] = 0x24623, [0x0244] = 0x726b, [0x0245] = 0x3e2e,
  [0x0246] = 0x3e2c, [0x0247] = 0x3e5c, [0x0248] = 0x2476c, [0x0249] = 0x72d5,
  [0x024a] = 0x24732, [0x024b] = 0x2475e, [0x024c] = 0x3e62, [0x024d] = 0x3e67,
  [0x024e] = 0x3eb4, [0x024f] = 0x248fb, [0x0250] = 0x24b29, [0x0251] = 0x24bbd,
  [0x0252] = 0x24c1f, [0x0253] = 0x7543, [0x0254] = 0x22341, [0x0255] = 0x24c1d,
  [0x0256] = 0x759c, [0x0257] = 0x24d31, [0x0258] = 0x24d2f, [0x0259] = 0x3fea,
  [0x025a] = 0x25043, [0x025b] = 0x3ffb, [0x025c] = 0x25041, [0x025d] = 0x4014,
  [0x025e] = 0x250f5, [0x025f] = 0x4013, [0x0260] = 0x4012, [0x0261] = 0x4010,
  [0x0262] = 0x4011, [0x0263] = 0x250f2, [0x0264] = 0x4086, [0x0265] = 0x77e4,
  [0x0266] = 0x4098, [0x0267] = 0x2541e, [0x0268] = 0x25754, [0x0269] = 0x412a,
  [0x026a] = 0x25756, [0x026b] = 0x25929, [0x026c] = 0x25927, [0x026d] = 0x25928,
  [0x026e] = 0x7ace, [0x026f] = 0x42b5, [0x0270] = 0x26279, [0x0271] = 0x2627e,
  [0x0272] = 0x264b6, [0x0273] = 0x264b7, [0x0274] = 0x8013, [0x0275] = 0x43d6,
  [0x0276] = 0x43d8, [0x0277] = 0x80b7, [0x0278] = 0x43d9, [0x0279] = 0x43d4,
  [0x027a] = 0x43d7, [0x027b] = 0x2f981, [0x027c] = 0x80b9, [0x027e] = 0x2664d,
  [0x027f] = 0x81e4, [0x0280] = 0x81fd, [0x0281] = 0x820f, [0x0282] = 0x4460,
  [0x0283] = 0x26a09, [0x0284] = 0x26a08, [0x0285] = 0x449e, [0x0286] = 0x44a1,
  [0x0287] = 0x26b1e, [0x0288] = 0x26b1a, [0x0289] = 0x26b23, [0x028a] = 0x26b15,
  [0x028b] = 0x26b19, [0x028c] = 0x26b16, [0x028d] = 0x82bf, [0x028e] = 0x82ca,
  [0x028f] = 0x26b14, [0x0290] = 0x26b18, [0x0291] = 0x82c1, [0x0292] = 0x44a0,
  [0x0293] = 0x2721f, [0x0294] = 0x275dd, [0x0295] = 0x275df, [0x0296] = 0x28469,
  [0x0297] = 0x8fd0, [0x0299] = 0x2846c, [0x029a] = 0x48b9, [0x029b] = 0x28693,
  [0x029c] = 0x90ae, [0x029d] = 0x28695, [0x029e] = 0x28694, [0x029f] = 0x28691,
  [0x02a0] = 0x28697, [0x02a1] = 0x28e30, [0x02a2] = 0x49c1, [0x02a3] = 0x49c2,
  [0x02a4] = 0x9638, [0x02a5] = 0x2fa18, [0x02a6] = 0x341c, [0x02a7] = 0x20231,
  [0x02a8] = 0x345e, [0x02a9] = 0x4fbc, [0x02aa] = 0x3459, [0x02ab] = 0x345c,
  [0x02ac] = 0x20236, [0x02ad] = 0x345f, [0x02ae] = 0x4fe9, [0x02af] = 0x4fbd,
  [0x02b0] = 0x4fe2, [0x02b1] = 0x5158, [0x02b2] = 0x205c9, [0x02b3] = 0x205ca,
  [0x02b4] = 0x34ce, [0x02b5] = 0x20718, [0x02b6] = 0x20712, [0x02b7] = 0x20719,
  [0x02b8] = 0x20710, [0x02b9] = 0x20716, [0x02ba] = 0x52c6, [0x02bb] = 0x2085f,
  [0x02bc] = 0x52c8, [0x02bd] = 0x208df, [0x02be] = 0x20917, [0x02bf] = 0x5328,
  [0x02c0] = 0x2093c, [0x02c1] = 0x5329, [0x02c2] = 0x209ea, [0x02c3] = 0x209e9,
  [0x02c4] = 0x20a1c, [0x02c5] = 0x20a58, [0x02c6] = 0x20a64, [0x02c7] = 0x20af3,
  [0x02c8] = 0x355f, [0x02c9] = 0x20c97, [0x02ca] = 0x3585, [0x02cb] = 0x20c5a,
  [0x02cc] = 0x20c59, [0x02cd] = 0x3586, [0x02ce] = 0x20c8e, [0x02cf] = 0x57b4,
  [0x02d0] = 0x212bc, [0x02d1] = 0x57a9, [0x02d2] = 0x3687, [0x02d3] = 0x2160d,
  [0x02d4] = 0x215fc, [0x02d5] = 0x2160c, [0x02d6] = 0x21707, [0x02d7] = 0x2170a,
  [0x02d8] = 0x36ca, [0x02d9] = 0x21702, [0x02da] = 0x36c3, [0x02db] = 0x21726,
  [0x02dc] = 0x21708, [0x02dd] = 0x2171d, [0x02de] = 0x36c2, [0x02df] = 0x5b68,
  [0x02e0] = 0x2194e, [0x02e1] = 0x3741, [0x02e2] = 0x219ed, [0x02e3] = 0x219ee,
  [0x02e4] = 0x219ef, [0x02e5] = 0x3780, [0x02e6] = 0x3781, [0x02e7] = 0x21c50,
  [0x02e8] = 0x21c4c, [0x02e9] = 0x3793, [0x02ea] = 0x3792, [0x02eb] = 0x21c4f,
  [0x02ec] = 0x37c5, [0x02ed] = 0x21dd4, [0x02ef] = 0x21db0, [0x02f0] = 0x3846,
  [0x02f1] = 0x3841, [0x02f2] = 0x3845, [0x02f3] = 0x3842, [0x02f4] = 0x383f,
  [0x02f5] = 0x220a5, [0x02f6] = 0x22209, [0x02f7] = 0x3882, [0x02f8] = 0x3881,
  [0x02f9] = 0x387f, [0x02fa] = 0x38a5, [0x02fb] = 0x5f2b, [0x02fc] = 0x38b3,
  [0x02fd] = 0x38b5, [0x02fe] = 0x223f3, [0x02ff] = 0x223d7, [0x0300] = 0x5f8d,
  [0x0301] = 0x38da, [0x0302] = 0x224dc, [0x0303] = 0x38db, [0x0304] = 0x390d,
  [0x0305] = 0x6018, [0x0306] = 0x390e, [0x0307] = 0x2260b, [0x0308] = 0x391e,
  [0x0309] = 0x3925, [0x030a] = 0x3926, [0x030b] = 0x391c, [0x030c] = 0x3921,
  [0x030d] = 0x6057, [0x030e] = 0x6048, [0x030f] = 0x3927, [0x0310] = 0x391a,
  [0x0311] = 0x2263a, [0x0312] = 0x2260d, [0x0313] = 0x22611, [0x0314] = 0x6038,
  [0x0315] = 0x22620, [0x0316] = 0x22601, [0x0317] = 0x3924, [0x0318] = 0x22637,
  [0x0319] = 0x2f8a3, [0x031a] = 0x6071, [0x031b] = 0x229aa, [0x031c] = 0x39c1,
  [0x031d] = 0x39e1, [0x031e] = 0x22ae7, [0x031f] = 0x22ae8, [0x0320] = 0x6312,
  [0x0321] = 0x39eb, [0x0322] = 0x22af5, [0x0323] = 0x39e2, [0x0324] = 0x39d7,
  [0x0325] = 0x39e9, [0x0326] = 0x22b35, [0x0327] = 0x22ae6, [0x0328] = 0x630a,
  [0x0329] = 0x22af8, [0x032a] = 0x6323, [0x032b] = 0x22af1, [0x032c] = 0x3a84,
  [0x032d] = 0x22f14, [0x032e] = 0x22f15, [0x032f] = 0x22f09, [0x0330] = 0x3ab5,
  [0x0331] = 0x3abc, [0x0332] = 0x3adc, [0x0333] = 0x3ade, [0x0334] = 0x3adf,
  [0x0335] = 0x2317b, [0x0336] = 0x662a, [0x0337] = 0x2317e, [0x0338] = 0x23366,
  [0x0339] = 0x23367, [0x033a] = 0x2340a, [0x033b] = 0x3b54, [0x033c] = 0x67e0,
  [0x033d] = 0x67be, [0x033e] = 0x3b53, [0x033f] = 0x3c24, [0x0340] = 0x3c25,
  [0x0341] = 0x6b29, [0x0342] = 0x3c28, [0x0343] = 0x3c27, [0x0344] = 0x2389c,
  [0x0345] = 0x23962, [0x0346] = 0x239e1, [0x0347] = 0x239de, [0x0348] = 0x239df,
  [0x0349] = 0x239e9, [0x034a] = 0x3c8b, [0x034b] = 0x23ce4, [0x034c] = 0x23cf7,
  [0x034d] = 0x3cdc, [0x034e] = 0x6d43, [0x034f] = 0x23cea, [0x0350] = 0x23cf6,
  [0x0351] = 0x23d12, [0x0352] = 0x2f901, [0x0353] = 0x70a6, [0x0354] = 0x3db2,
  [0x0355] = 0x70c0, [0x0356] = 0x241d9, [0x0357] = 0x722f, [0x0358] = 0x2455f,
  [0x0359] = 0x3e1b, [0x035a] = 0x24573, [0x035b] = 0x3e32, [0x035c] = 0x7271,
  [0x035d] = 0x2463f, [0x035e] = 0x24778, [0x035f] = 0x3e6b, [0x0360] = 0x2477d,
  [0x0361] = 0x3e6c, [0x0362] = 0x3e6d, [0x0363] = 0x24912, [0x0364] = 0x3eb9,
  [0x0365] = 0x3eba, [0x0366] = 0x3f09, [0x0367] = 0x3f0a, [0x0368] = 0x24b2f,
  [0x0369] = 0x74ea, [0x036a] = 0x3f1b, [0x036b] = 0x24b30, [0x036c] = 0x24bdb,
  [0x036d] = 0x7520, [0x036e] = 0x3f58, [0x036f] = 0x24c29, [0x0370] = 0x3f5a,
  [0x0371] = 0x2f936, [0x0372] = 0x24d40, [0x0373] = 0x3f77, [0x0374] = 0x24d3d,
  [0x0375] = 0x24d3e, [0x0376] = 0x3f79, [0x0377] = 0x75a9, [0x0378] = 0x24d38,
  [0x0379] = 0x7685, [0x037a] = 0x24f48, [0x037b] = 0x3feb, [0x037c] = 0x24fd0,
  [0x037d] = 0x3ffd, [0x037e] = 0x3ffc, [0x037f] = 0x7706, [0x0380] = 0x4015,
  [0x0381] = 0x4018, [0x0382] = 0x76f6, [0x0383] = 0x4016, [0x0384] = 0x4017,
  [0x0385] = 0x4019, [0x0386] = 0x7700, [0x0387] = 0x401b, [0x0388] = 0x2510e,
  [0x0389] = 0x25109, [0x038a] = 0x7702, [0x038b] = 0x25107, [0x038c] = 0x4087,
  [0x038d] = 0x2535f, [0x038e] = 0x25360, [0x038f] = 0x409c, [0x0390] = 0x2542a,
  [0x0391] = 0x409a, [0x0392] = 0x25429, [0x0393] = 0x40ff, [0x0394] = 0x40fe,
  [0x0395] = 0x2564d, [0x0396] = 0x4131, [0x0397] = 0x412e, [0x0398] = 0x4130,
  [0x0399] = 0x4132, [0x039a] = 0x2576c, [0x039c] = 0x412f, [0x039d] = 0x25767,
  [0x039e] = 0x4195, [0x039f] = 0x4196, [0x03a0] = 0x25939, [0x03a1] = 0x25934,
  [0x03a2] = 0x25a59, [0x03a3] = 0x41c5, [0x03a4] = 0x427a, [0x03a5] = 0x26223,
  [0x03a6] = 0x4342, [0x03a7] = 0x26282, [0x03a8] = 0x4354, [0x03a9] = 0x26283,
  [0x03aa] = 0x262a8, [0x03ab] = 0x26355, [0x03ac] = 0x263f9, [0x03ad] = 0x263f8,
  [0x03ae] = 0x263fc, [0x03af] = 0x263fa, [0x03b0] = 0x8009, [0x03b1] = 0x2f97a,
  [0x03b2] = 0x439f, [0x03b3] = 0x43a0, [0x03b4] = 0x43a2, [0x03b5] = 0x43e0,
  [0x03b6] = 0x26674, [0x03b7] = 0x43e1, [0x03b8] = 0x2666b, [0x03b9] = 0x26671,
  [0x03ba] = 0x43df, [0x03bb] = 0x26676, [0x03bc] = 0x26a0e, [0x03bd] = 0x4462,
  [0x03be] = 0x4461, [0x03bf] = 0x26a10, [0x03c0] = 0x26a0f, [0x03c1] = 0x44a7,
  [0x03c2] = 0x26b38, [0x03c3] = 0x26b39, [0x03c4] = 0x26b3a, [0x03c5] = 0x82da,
  [0x03c6] = 0x26b37, [0x03c7] = 0x26b3e, [0x03c8] = 0x830a, [0x03c9] = 0x4589,
  [0x03ca] = 0x2719c, [0x03cb] = 0x2722a, [0x03cc] = 0x2723a, [0x03cd] = 0x2722b,
  [0x03ce] = 0x27228, [0x03cf] = 0x275aa, [0x03d0] = 0x2760f, [0x03d1] = 0x461d,
  [0x03d2] = 0x27610, [0x03d3] = 0x277e6, [0x03d4] = 0x472a, [0x03d5] = 0x27d25,
  [0x03d6] = 0x27e8e, [0x03d7] = 0x47d3, [0x03d8] = 0x4842, [0x03d9] = 0x4843,
  [0x03da] = 0x2848b, [0x03db] = 0x4891, [0x03dc] = 0x2848c, [0x03dd] = 0x2849a,
  [0x03de] = 0x4890, [0x03df] = 0x48bf, [0x03e0] = 0x286b3, [0x03e1] = 0x48bc,
  [0x03e2] = 0x286b2, [0x03e3] = 0x48c0, [0x03e4] = 0x49c7, [0x03e5] = 0x49c5,
  [0x03e6] = 0x9655, [0x03e7] = 0x28e44, [0x03e8] = 0x9652, [0x03e9] = 0x4e35,
  [0x03ea] = 0x20158, [0x03eb] = 0x202a7, [0x03ec] = 0x5034, [0x03ed] = 0x5001,
  [0x03ee] = 0x20263, [0x03ef] = 0x500a, [0x03f0] = 0x3466, [0x03f1] = 0x2026a,
  [0x03f2] = 0x20265, [0x03f3] = 0x2026b, [0x03f4] = 0x34ad, [0x03f5] = 0x204ed,
  [0x03f6] = 0x205d9, [0x03f7] = 0x2073b, [0x03f8] = 0x20732, [0x03f9] = 0x5258,
  [0x03fa] = 0x20735, [0x03fb] = 0x20731, [0x03fc] = 0x2073e, [0x03fd] = 0x20743,
  [0x03fe] = 0x20736, [0x03ff] = 0x2086d, [0x0400] = 0x351d, [0x0401] = 0x208e4,
  [0x0402] = 0x3531, [0x0403] = 0x532b, [0x0404] = 0x209f4, [0x0405] = 0x354c,
  [0x0406] = 0x20a69, [0x0407] = 0x20a6b, [0x0408] = 0x3595, [0x0409] = 0x3591,
  [0x040a] = 0x3594, [0x040b] = 0x20cbf, [0x040c] = 0x20cc3, [0x040d] = 0x20cae,
  [0x040e] = 0x358f, [0x040f] = 0x54ec, [0x0410] = 0x20cb1, [0x0411] = 0x5515,
  [0x0412] = 0x54fe, [0x0413] = 0x20cbb, [0x0414] = 0x20caf, [0x0415] = 0x20cb0,
  [0x0416] = 0x54e3, [0x0417] = 0x5516, [0x0418] = 0x3640, [0x0419] = 0x212ef,
  [0x041a] = 0x3641, [0x041b] = 0x212ed, [0x041c] = 0x57d3, [0x041d] = 0x212f0,
  [0x041e] = 0x21593, [0x041f] = 0x21634, [0x0420] = 0x2162b, [0x0421] = 0x5959,
  [0x0422] = 0x5a27, [0x0423] = 0x36d8, [0x0424] = 0x36d2, [0x0425] = 0x36da,
  [0x0426] = 0x5a28, [0x0427] = 0x5a10, [0x0428] = 0x36d7, [0x0429] = 0x5a0e,
  [0x042a] = 0x3742, [0x042b] = 0x3757, [0x042c] = 0x21a04, [0x042d] = 0x21a03,
  [0x042e] = 0x21a02, [0x042f] = 0x5baf, [0x0430] = 0x21a05, [0x0431] = 0x5bba,
  [0x0432] = 0x5bb1, [0x0433] = 0x21a00, [0x0434] = 0x21b74, [0x0435] = 0x3778,
  [0x0436] = 0x3782, [0x0437] = 0x3797, [0x0438] = 0x21c65, [0x0439] = 0x21c63,
  [0x043a] = 0x37c9, [0x043b] = 0x21dd7, [0x043c] = 0x37c8, [0x043d] = 0x37d6,
  [0x043e] = 0x37cc, [0x043f] = 0x21de8, [0x0440] = 0x37d0, [0x0441] = 0x21ddb,
  [0x0442] = 0x21dd8, [0x0443] = 0x37ce, [0x0444] = 0x37c7, [0x0445] = 0x5cfc,
  [0x0446] = 0x21e09, [0x0447] = 0x37cf, [0x0448] = 0x37cb, [0x0449] = 0x5cf2,
  [0x044a] = 0x5cfe, [0x044b] = 0x21de5, [0x044c] = 0x21de7, [0x044d] = 0x5df8,
  [0x044e] = 0x2203f, [0x044f] = 0x3847, [0x0450] = 0x220c0, [0x0451] = 0x3848,
  [0x0452] = 0x220b9, [0x0453] = 0x220ba, [0x0454] = 0x3883, [0x0455] = 0x3885,
  [0x0456] = 0x3884, [0x0457] = 0x22220, [0x0458] = 0x2234f, [0x0459] = 0x5f2c,
  [0x045a] = 0x38b8, [0x045b] = 0x38bc, [0x045c] = 0x22491, [0x045d] = 0x22490,
  [0x045e] = 0x224ee, [0x045f] = 0x224f3, [0x0460] = 0x224ec, [0x0461] = 0x224f1,
  [0x0462] = 0x224eb, [0x0463] = 0x224f2, [0x0464] = 0x224f4, [0x0465] = 0x224ed,
  [0x0466] = 0x22641, [0x0467] = 0x22648, [0x0468] = 0x22645, [0x0469] = 0x3931,
  [0x046a] = 0x3934, [0x046b] = 0x3936, [0x046c] = 0x6082, [0x046d] = 0x2267f,
  [0x046e] = 0x2268b, [0x046f] = 0x3923, [0x0470] = 0x2263b, [0x0471] = 0x2264e,
  [0x0472] = 0x393a, [0x0473] = 0x6091, [0x0474] = 0x608f, [0x0475] = 0x2267c,
  [0x0476] = 0x39b4, [0x0477] = 0x39b5, [0x0478] = 0x39ed, [0x0479] = 0x39ec,
  [0x047a] = 0x39d8, [0x047b] = 0x22af0, [0x047c] = 0x22b36, [0x047d] = 0x22b3f,
  [0x047e] = 0x22b43, [0x047f] = 0x22b31, [0x0480] = 0x22b34, [0x0481] = 0x22b46,
  [0x0482] = 0x39f6, [0x0483] = 0x39e7, [0x0484] = 0x22b2f, [0x0485] = 0x22ec3,
  [0x0486] = 0x3a7c, [0x0487] = 0x3a7b, [0x0488] = 0x22f36, [0x0489] = 0x3a88,
  [0x048a] = 0x6547, [0x048b] = 0x654c, [0x048c] = 0x22f21, [0x048d] = 0x2f8c8,
  [0x048e] = 0x658a, [0x048f] = 0x23073, [0x0490] = 0x3abe, [0x0491] = 0x230e6,
  [0x0492] = 0x3ae7, [0x0493] = 0x3ae5, [0x0494] = 0x3aee, [0x0495] = 0x67e1,
  [0x0496] = 0x684a, [0x0497] = 0x2343f, [0x0498] = 0x3b59, [0x0499] = 0x3b5e,
  [0x049a] = 0x2346b, [0x049b] = 0x3b5a, [0x049c] = 0x683f, [0x049d] = 0x3b61,
  [0x049e] = 0x3b58, [0x049f] = 0x3b5b, [0x04a0] = 0x67bd, [0x04a1] = 0x23471,
  [0x04a2] = 0x3b5f, [0x04a3] = 0x2f8e2, [0x04a4] = 0x3c2a, [0x04a5] = 0x3c2d,
  [0x04a6] = 0x238b0, [0x04a7] = 0x3c23, [0x04a8] = 0x3c2b, [0x04a9] = 0x3c2c,
  [0x04aa] = 0x239f3, [0x04ab] = 0x3c7e, [0x04ac] = 0x2f8f5, [0x04ad] = 0x3c93,
  [0x04ae] = 0x3c99, [0x04af] = 0x23b61, [0x04b0] = 0x23b5f, [0x04b1] = 0x3cb3,
  [0x04b2] = 0x3ce7, [0x04b3] = 0x3cea, [0x04b4] = 0x23d28, [0x04b5] = 0x23d1e,
  [0x04b6] = 0x23d29, [0x04b7] = 0x2f902, [0x04b8] = 0x23ceb, [0x04b9] = 0x23d66,
  [0x04ba] = 0x3db4, [0x04bb] = 0x241f6, [0x04bc] = 0x70c9, [0x04bd] = 0x3e17,
  [0x04be] = 0x24560, [0x04bf] = 0x24577, [0x04c0] = 0x245bb, [0x04c1] = 0x3e21,
  [0x04c2] = 0x245bc, [0x04c3] = 0x3e38, [0x04c4] = 0x3e37, [0x04c5] = 0x3e74,
  [0x04c6] = 0x3e73, [0x04c7] = 0x3e75, [0x04c8] = 0x247a2, [0x04c9] = 0x247b2,
  [0x04ca] = 0x2479e, [0x04cb] = 0x3e76, [0x04cc] = 0x3e78, [0x04cd] = 0x2493a,
  [0x04ce] = 0x24938, [0x04cf] = 0x73ba, [0x04d0] = 0x3f0c, [0x04d1] = 0x3f20,
  [0x04d2] = 0x3f1e, [0x04d3] = 0x24c4d, [0x04d4] = 0x3f5b, [0x04d5] = 0x24c5e,
  [0x04d6] = 0x3f5c, [0x04d7] = 0x24d17, [0x04d8] = 0x3f83, [0x04d9] = 0x75c6,
  [0x04da] = 0x3f80, [0x04db] = 0x3f81, [0x04dc] = 0x3f7e, [0x04dd] = 0x3f88,
  [0x04de] = 0x3f85, [0x04df] = 0x3f89, [0x04e0] = 0x3f7f, [0x04e1] = 0x3f8e,
  [0x04e2] = 0x24d64, [0x04e3] = 0x24d67, [0x04e4] = 0x3f84, [0x04e5] = 0x75b7,
  [0x04e6] = 0x768c, [0x04e7] = 0x24f59, [0x04e8] = 0x768d, [0x04e9] = 0x24fda,
  [0x04ea] = 0x3fee, [0x04eb] = 0x3fed, [0x04ec] = 0x3ffe, [0x04ed] = 0x25051,
  [0x04ee] = 0x3fff, [0x04ef] = 0x25055, [0x04f0] = 0x2513b, [0x04f1] = 0x25134,
  [0x04f2] = 0x4023, [0x04f3] = 0x25137, [0x04f4] = 0x7717, [0x04f5] = 0x25131,
  [0x04f6] = 0x771c, [0x04f7] = 0x401f, [0x04f8] = 0x2513f, [0x04f9] = 0x25159,
  [0x04fa] = 0x7714, [0x04fb] = 0x25361, [0x04fc] = 0x408f, [0x04fd] = 0x4090,
  [0x04fe] = 0x25458, [0x04ff] = 0x25462, [0x0500] = 0x25450, [0x0501] = 0x40a0,
  [0x0502] = 0x25451, [0x0503] = 0x40a6, [0x0504] = 0x409f, [0x0505] = 0x25446,
  [0x0506] = 0x40a7, [0x0507] = 0x40a1, [0x0508] = 0x2562d, [0x0509] = 0x4102,
  [0x050a] = 0x4136, [0x050b] = 0x25948, [0x050c] = 0x25951, [0x050d] = 0x2594a,
  [0x050e] = 0x2594f, [0x050f] = 0x25950, [0x0510] = 0x25949, [0x0511] = 0x25a63,
  [0x0512] = 0x41c8, [0x0513] = 0x25a66, [0x0514] = 0x25a60, [0x0515] = 0x41d8,
  [0x0516] = 0x7b0c, [0x0517] = 0x41dd, [0x0518] = 0x41dc, [0x0519] = 0x25af1,
  [0x051a] = 0x41d7, [0x051b] = 0x25b07, [0x051c] = 0x41da, [0x051d] = 0x42ba,
  [0x051e] = 0x25f9b, [0x051f] = 0x25fa3, [0x0520] = 0x25f9d, [0x0521] = 0x42be,
  [0x0522] = 0x42c2, [0x0523] = 0x42bb, [0x0524] = 0x42c0, [0x0525] = 0x25fa9,
  [0x0526] = 0x25fa7, [0x0527] = 0x25fa4, [0x0528] = 0x7d23, [0x0529] = 0x25fba,
  [0x052a] = 0x221c1, [0x052b] = 0x26226, [0x052c] = 0x4343, [0x052d] = 0x4355,
  [0x052e] = 0x26294, [0x052f] = 0x26293, [0x0530] = 0x4357, [0x0531] = 0x4368,
  [0x0532] = 0x7f98, [0x0533] = 0x7f90, [0x0534] = 0x26407, [0x0535] = 0x2640d,
  [0x0536] = 0x26402, [0x0537] = 0x2640c, [0x0538] = 0x26409, [0x0539] = 0x26408,
  [0x053a] = 0x26413, [0x053b] = 0x43a1, [0x053c] = 0x264d2, [0x053d] = 0x803a,
  [0x053e] = 0x2653e, [0x053f] = 0x2653c, [0x0540] = 0x26667, [0x0541] = 0x2669e,
  [0x0542] = 0x43ea, [0x0543] = 0x266ad, [0x0544] = 0x43e7, [0x0545] = 0x266d9,
  [0x0546] = 0x26642, [0x0547] = 0x266a3, [0x0548] = 0x266a0, [0x0549] = 0x43e8,
  [0x054a] = 0x43e9, [0x054b] = 0x2667c, [0x054c] = 0x26904, [0x054d] = 0x4454,
  [0x054e] = 0x26958, [0x054f] = 0x26960, [0x0550] = 0x269c9, [0x0551] = 0x269c8,
  [0x0552] = 0x8226, [0x0553] = 0x4465, [0x0554] = 0x26a1c, [0x0555] = 0x26a1a,
  [0x0556] = 0x448a, [0x0557] = 0x26b74, [0x0558] = 0x44b0, [0x0559] = 0x26b6e,
  [0x055a] = 0x26b6f, [0x055b] = 0x26b79, [0x055c] = 0x44bc, [0x055d] = 0x832e,
  [0x055e] = 0x26b77, [0x055f] = 0x8355, [0x0560] = 0x831a, [0x0561] = 0x44b8,
  [0x0562] = 0x833d, [0x0563] = 0x26b7c, [0x0564] = 0x44b2, [0x0565] = 0x26b70,
  [0x0566] = 0x8330, [0x0567] = 0x44bd, [0x0568] = 0x26b7a, [0x0569] = 0x26b82,
  [0x056a] = 0x458a, [0x056b] = 0x8651, [0x056c] = 0x45a1, [0x056d] = 0x27244,
  [0x056e] = 0x45a2, [0x056f] = 0x27243, [0x0570] = 0x8688, [0x0571] = 0x27245,
  [0x0572] = 0x4615, [0x0573] = 0x27625, [0x0574] = 0x2761f, [0x0575] = 0x4620,
  [0x0576] = 0x4673, [0x0577] = 0x277e8, [0x0578] = 0x898e, [0x0579] = 0x898d,
  [0x057a] = 0x27809, [0x057b] = 0x278b7, [0x057c] = 0x278b9, [0x057d] = 0x27964,
  [0x057e] = 0x27963, [0x057f] = 0x8a09, [0x0580] = 0x8a14, [0x0581] = 0x46b1,
  [0x0582] = 0x27962, [0x0583] = 0x27965, [0x0584] = 0x472b, [0x0585] = 0x27bac,
  [0x0586] = 0x4745, [0x0587] = 0x27d2a, [0x0588] = 0x4797, [0x0589] = 0x4798,
  [0x058a] = 0x27e96, [0x058b] = 0x47d5, [0x058c] = 0x27fc2, [0x058d] = 0x2f9de,
  [0x058e] = 0x4893, [0x058f] = 0x4896, [0x0590] = 0x9007, [0x0591] = 0x4894,
  [0x0592] = 0x284aa, [0x0593] = 0x284b0, [0x0594] = 0x48c7, [0x0595] = 0x48c5,
  [0x0596] = 0x286d4, [0x0597] = 0x286cd, [0x0598] = 0x286d6, [0x0599] = 0x48c4,
  [0x059a] = 0x287f0, [0x059b] = 0x9579, [0x059c] = 0x9584, [0x059d] = 0x49ce,
  [0x059e] = 0x49ca, [0x059f] = 0x49cc, [0x05a0] = 0x9657, [0x05a1] = 0x49c9,
  [0x05a2] = 0x96ba, [0x05a4] = 0x29d4b, [0x05a5] = 0x20162, [0x05a6] = 0x346e,
  [0x05a7] = 0x202b7, [0x05a8] = 0x5067, [0x05a9] = 0x202ac, [0x05aa] = 0x202ab,
  [0x05ab] = 0x3471, [0x05ac] = 0x202c1, [0x05ad] = 0x34bb, [0x05ae] = 0x34d3,
  [0x05af] = 0x205e8, [0x05b0] = 0x20769, [0x05b1] = 0x2075d, [0x05b2] = 0x2075f,
  [0x05b3] = 0x34f3, [0x05b4] = 0x34ed, [0x05b5] = 0x34f5, [0x05b6] = 0x20764,
  [0x05b7] = 0x20762, [0x05b8] = 0x34f1, [0x05b9] = 0x34f2, [0x05ba] = 0x34f6,
  [0x05bb] = 0x3520, [0x05bc] = 0x208f9, [0x05bd] = 0x3528, [0x05be] = 0x208ee,
  [0x05bf] = 0x5318, [0x05c0] = 0x532c, [0x05c1] = 0x5359, [0x05c2] = 0x209ab,
  [0x05c3] = 0x209ac, [0x05c4] = 0x209aa, [0x05c5] = 0x5368, [0x05c6] = 0x537e,
  [0x05c7] = 0x20a86, [0x05c8] = 0x20a7a, [0x05c9] = 0x53a1, [0x05ca] = 0x20d28,
  [0x05cb] = 0x35a1, [0x05cc] = 0x555b, [0x05cd] = 0x35aa, [0x05ce] = 0x35a9,
  [0x05cf] = 0x20d31, [0x05d0] = 0x35b5, [0x05d1] = 0x35a5, [0x05d2] = 0x35a8,
  [0x05d3] = 0x5542, [0x05d4] = 0x35a7, [0x05d5] = 0x5547, [0x05d6] = 0x20d2d,
  [0x05d7] = 0x2f843, [0x05d8] = 0x553d, [0x05d9] = 0x20d2b, [0x05da] = 0x20d32,
  [0x05db] = 0x5560, [0x05dc] = 0x57eb, [0x05dd] = 0x21329, [0x05de] = 0x21326,
  [0x05df] = 0x21353, [0x05e0] = 0x364d, [0x05e1] = 0x21333, [0x05e2] = 0x21334,
  [0x05e3] = 0x2159d, [0x05e4] = 0x369c, [0x05e5] = 0x595f, [0x05e6] = 0x21641,
  [0x05e7] = 0x36ea, [0x05e8] = 0x36e5, [0x05e9] = 0x21791, [0x05ea] = 0x21769,
  [0x05eb] = 0x5b6f, [0x05ec] = 0x21a22, [0x05ed] = 0x375e, [0x05ee] = 0x21a34,
  [0x05ef] = 0x2f86d, [0x05f0] = 0x21b2e, [0x05f1] = 0x3786, [0x05f2] = 0x3784,
  [0x05f3] = 0x21bf4, [0x05f4] = 0x5c5a, [0x05f5] = 0x21e28, [0x05f6] = 0x37d9,
  [0x05f7] = 0x21e1c, [0x05f8] = 0x37de, [0x05f9] = 0x21e11, [0x05fa] = 0x21e24,
  [0x05fb] = 0x21e15, [0x05fc] = 0x37db, [0x05fd] = 0x21e16, [0x05fe] = 0x21e54,
  [0x05ff] = 0x21e23, [0x0600] = 0x21fff, [0x0601] = 0x3831, [0x0602] = 0x220cf,
  [0x0603] = 0x220d0, [0x0604] = 0x384b, [0x0605] = 0x220ce, [0x0606] = 0x3849,
  [0x0607] = 0x384a, [0x0608] = 0x384c, [0x0609] = 0x221c7, [0x060a] = 0x2225c,
  [0x060b] = 0x388a, [0x060c] = 0x22236, [0x060d] = 0x3889, [0x060e] = 0x388b,
  [0x060f] = 0x22243, [0x0610] = 0x38bb, [0x0611] = 0x5fa2, [0x0612] = 0x5f9d,
  [0x0613] = 0x38e4, [0x0614] = 0x2252a, [0x0615] = 0x22501, [0x0616] = 0x5fa3,
  [0x0617] = 0x2250b, [0x0618] = 0x2250f, [0x0619] = 0x393b, [0x061a] = 0x392e,
  [0x061b] = 0x393e, [0x061c] = 0x3946, [0x061d] = 0x3953, [0x061e] = 0x226df,
  [0x061f] = 0x3944, [0x0620] = 0x226d2, [0x0621] = 0x393f, [0x0622] = 0x3942,
  [0x0623] = 0x394f, [0x0624] = 0x226cd, [0x0625] = 0x3952, [0x0626] = 0x394a,
  [0x0627] = 0x60c2, [0x0628] = 0x226db, [0x0629] = 0x395a, [0x062a] = 0x60a5,
  [0x062b] = 0x3949, [0x062c] = 0x226cf, [0x062d] = 0x621c, [0x062e] = 0x229c0,
  [0x062f] = 0x621d, [0x0630] = 0x3a03, [0x0631] = 0x22b81, [0x0632] = 0x22b82,
  [0x0633] = 0x22b8a, [0x0634] = 0x22ba8, [0x0635] = 0x22b8c, [0x0636] = 0x6395,
  [0x0637] = 0x639a, [0x0638] = 0x3a01, [0x0639] = 0x3a06, [0x063a] = 0x39fb,
  [0x063b] = 0x39f9, [0x063c] = 0x22b8e, [0x063d] = 0x3a05, [0x063e] = 0x39fa,
  [0x063f] = 0x22b8b, [0x0640] = 0x63a6, [0x0641] = 0x22b96, [0x0642] = 0x39fe,
  [0x0643] = 0x22bcc, [0x0644] = 0x3a7d, [0x0645] = 0x3a8c, [0x0646] = 0x22f4f,
  [0x0647] = 0x22f4a, [0x0648] = 0x6550, [0x0649] = 0x3a8e, [0x064a] = 0x22f4b,
  [0x064b] = 0x22f3d, [0x064c] = 0x6552, [0x064d] = 0x22f44, [0x064e] = 0x22f49,
  [0x064f] = 0x2309e, [0x0650] = 0x230f3, [0x0651] = 0x65c8, [0x0652] = 0x230f5,
  [0x0653] = 0x3af0, [0x0654] = 0x3af2, [0x0655] = 0x231b3, [0x0656] = 0x6658,
  [0x0657] = 0x3af1, [0x0658] = 0x3ae6, [0x0659] = 0x231e5, [0x065a] = 0x231b6,
  [0x065b] = 0x2f8d8, [0x065c] = 0x6888, [0x065d] = 0x23486, [0x065e] = 0x3b6f,
  [0x065f] = 0x3b6d, [0x0660] = 0x3b69, [0x0661] = 0x234b6, [0x0662] = 0x23485,
  [0x0663] = 0x3b6e, [0x0664] = 0x234ab, [0x0665] = 0x3c2f, [0x0666] = 0x3c30,
  [0x0667] = 0x3c63, [0x0668] = 0x23a0d, [0x0669] = 0x23a8b, [0x066a] = 0x23a8c,
  [0x066b] = 0x6bb8, [0x066c] = 0x3c80, [0x066d] = 0x6bb9, [0x066e] = 0x3c9a,
  [0x066f] = 0x3c94, [0x0670] = 0x3c96, [0x0671] = 0x3c95, [0x0672] = 0x3c97,
  [0x0673] = 0x23b72, [0x0674] = 0x3cf4, [0x0675] = 0x3cfe, [0x0676] = 0x23d8f,
  [0x0677] = 0x3d01, [0x0678] = 0x23d79, [0x0679] = 0x3d02, [0x067a] = 0x23d77,
  [0x067b] = 0x3cf9, [0x067c] = 0x23d90, [0x067d] = 0x23d88, [0x067e] = 0x23d80,
  [0x067f] = 0x23d9e, [0x0680] = 0x3cf6, [0x0681] = 0x3cf7, [0x0682] = 0x23d82,
  [0x0683] = 0x23d34, [0x0684] = 0x23d89, [0x0685] = 0x3cff, [0x0686] = 0x23d85,
  [0x0687] = 0x23d7e, [0x0688] = 0x23d7a, [0x0689] = 0x23da6, [0x068a] = 0x6e0b,
  [0x068b] = 0x23db5, [0x068c] = 0x241f4, [0x068d] = 0x3dbf, [0x068e] = 0x3dbc,
  [0x068f] = 0x7105, [0x0690] = 0x24229, [0x0691] = 0x24226, [0x0692] = 0x2422a,
  [0x0693] = 0x3dbe, [0x0694] = 0x3dc0, [0x0695] = 0x2457e, [0x0696] = 0x3e3b,
  [0x0697] = 0x3e39, [0x0698] = 0x2465b, [0x0699] = 0x24661, [0x069a] = 0x24660,
  [0x069b] = 0x3e3c, [0x069c] = 0x2479b, [0x069d] = 0x7314, [0x069e] = 0x7304,
  [0x069f] = 0x247c3, [0x06a0] = 0x3e7d, [0x06a1] = 0x247cd, [0x06a2] = 0x3e7f,
  [0x06a3] = 0x3e7a, [0x06a4] = 0x3e7c, [0x06a5] = 0x7305, [0x06a6] = 0x3e7e,
  [0x06a7] = 0x7315, [0x06a8] = 0x730d, [0x06a9] = 0x3e80, [0x06aa] = 0x3ebf,
  [0x06ab] = 0x3ec3, [0x06ac] = 0x24954, [0x06ad] = 0x24959, [0x06ae] = 0x3ecc,
  [0x06af] = 0x3f0e, [0x06b0] = 0x3f0d, [0x06b1] = 0x24af5, [0x06b2] = 0x3f26,
  [0x06b3] = 0x3f24, [0x06b4] = 0x3f25, [0x06b5] = 0x3f23, [0x06b6] = 0x3f21,
  [0x06b7] = 0x3f29, [0x06b8] = 0x24c69, [0x06b9] = 0x24c65, [0x06ba] = 0x3f8f,
  [0x06bb] = 0x3f8d, [0x06bc] = 0x24d7c, [0x06bd] = 0x3f8b, [0x06be] = 0x3f92,
  [0x06bf] = 0x24d80, [0x06c0] = 0x3f90, [0x06c1] = 0x24f65, [0x06c2] = 0x3fef,
  [0x06c3] = 0x3ff0, [0x06c4] = 0x24fdf, [0x06c5] = 0x25060, [0x06c6] = 0x4001,
  [0x06c7] = 0x2505e, [0x06c8] = 0x402e, [0x06c9] = 0x402d, [0x06ca] = 0x772e,
  [0x06cb] = 0x4028, [0x06cc] = 0x4029, [0x06cd] = 0x402c, [0x06ce] = 0x2518f,
  [0x06cf] = 0x25161, [0x06d0] = 0x2515a, [0x06d1] = 0x7741, [0x06d2] = 0x4088,
  [0x06d3] = 0x25367, [0x06d4] = 0x4092, [0x06d5] = 0x4091, [0x06d6] = 0x77ea,
  [0x06d7] = 0x253bb, [0x06d8] = 0x7844, [0x06d9] = 0x40a9, [0x06da] = 0x40ac,
  [0x06db] = 0x40ae, [0x06dc] = 0x40aa, [0x06dd] = 0x4106, [0x06de] = 0x4105,
  [0x06df] = 0x414a, [0x06e0] = 0x413e, [0x06e1] = 0x413c, [0x06e2] = 0x413b,
  [0x06e3] = 0x2579b, [0x06e4] = 0x4142, [0x06e5] = 0x4141, [0x06e6] = 0x4143,
  [0x06e7] = 0x257a9, [0x06e8] = 0x4145, [0x06e9] = 0x25966, [0x06ea] = 0x419a,
  [0x06eb] = 0x419b, [0x06ec] = 0x419f, [0x06ed] = 0x419e, [0x06ee] = 0x25a72,
  [0x06ef] = 0x25a76, [0x06f0] = 0x25b14, [0x06f1] = 0x41de, [0x06f2] = 0x41e2,
  [0x06f3] = 0x41e6, [0x06f4] = 0x25b0f, [0x06f5] = 0x7b29, [0x06f6] = 0x41e3,
  [0x06f7] = 0x7b27, [0x06f8] = 0x41df, [0x06f9] = 0x25e41, [0x06fa] = 0x25e46,
  [0x06fb] = 0x7c9d, [0x06fc] = 0x25e4b, [0x06fd] = 0x25e43, [0x06fe] = 0x427e,
  [0x06ff] = 0x42c9, [0x0700] = 0x42cc, [0x0701] = 0x25fc6, [0x0702] = 0x42d1,
  [0x0703] = 0x25fc7, [0x0704] = 0x42d0, [0x0705] = 0x25fce, [0x0706] = 0x42cf,
  [0x0707] = 0x42c8, [0x0708] = 0x25fc9, [0x0709] = 0x42cd, [0x070a] = 0x25fcb,
  [0x070b] = 0x25fc5, [0x070c] = 0x4349, [0x070d] = 0x4347, [0x070e] = 0x4358,
  [0x070f] = 0x436b, [0x0710] = 0x436c, [0x0711] = 0x436a, [0x0712] = 0x26369,
  [0x0713] = 0x4380, [0x0714] = 0x4382, [0x0715] = 0x4384, [0x0716] = 0x7fc8,
  [0x0717] = 0x4383, [0x0718] = 0x2641e, [0x0719] = 0x264ba, [0x071a] = 0x43b6,
  [0x071b] = 0x26552, [0x071c] = 0x26551, [0x071d] = 0x26553, [0x071e] = 0x2669f,
  [0x071f] = 0x266db, [0x0720] = 0x8126, [0x0721] = 0x43f1, [0x0722] = 0x266e3,
  [0x0723] = 0x43f6, [0x0724] = 0x43f3, [0x0725] = 0x43f0, [0x0726] = 0x811c,
  [0x0727] = 0x266d7, [0x0728] = 0x266e9, [0x0729] = 0x8128, [0x072a] = 0x43f5,
  [0x072b] = 0x43f4, [0x072c] = 0x43f7, [0x072d] = 0x266de, [0x072e] = 0x266dc,
  [0x072f] = 0x2693c, [0x0730] = 0x2f98b, [0x0731] = 0x4455, [0x0732] = 0x26a22,
  [0x0733] = 0x26a32, [0x0734] = 0x26a23, [0x0735] = 0x448b, [0x0736] = 0x44cb,
  [0x0737] = 0x44c2, [0x0738] = 0x26bd4, [0x0739] = 0x26bc8, [0x073a] = 0x26bbc,
  [0x073b] = 0x44ca, [0x073c] = 0x44cc, [0x073d] = 0x26bcd, [0x073e] = 0x44c7,
  [0x073f] = 0x44c9, [0x0740] = 0x8370, [0x0741] = 0x26bbe, [0x0742] = 0x44c6,
  [0x0743] = 0x26c40, [0x0744] = 0x26bd6, [0x0745] = 0x44c3, [0x0746] = 0x8382,
  [0x0747] = 0x26bbd, [0x0748] = 0x83ac, [0x0749] = 0x26bba, [0x074a] = 0x44c4,
  [0x074b] = 0x26bc0, [0x074c] = 0x2f99b, [0x074d] = 0x2f99d, [0x074e] = 0x26bc1,
  [0x074f] = 0x271a6, [0x0750] = 0x271a5, [0x0751] = 0x271a8, [0x0752] = 0x45a9,
  [0x0753] = 0x86ad, [0x0754] = 0x45a8, [0x0755] = 0x45a6, [0x0756] = 0x2725f,
  [0x0757] = 0x27270, [0x0758] = 0x2726a, [0x0759] = 0x27265, [0x075a] = 0x27267,
  [0x075b] = 0x2726f, [0x075c] = 0x27287, [0x075d] = 0x86ca, [0x075e] = 0x8851,
  [0x075f] = 0x2763d, [0x0760] = 0x27643, [0x0761] = 0x4622, [0x0762] = 0x27640,
  [0x0763] = 0x4626, [0x0764] = 0x4624, [0x0765] = 0x4625, [0x0766] = 0x889d,
  [0x0767] = 0x462a, [0x0768] = 0x4674, [0x0769] = 0x27818, [0x076a] = 0x4679,
  [0x076b] = 0x8990, [0x076c] = 0x467a, [0x076d] = 0x27819, [0x076e] = 0x278c1,
  [0x076f] = 0x89d8, [0x0770] = 0x89d7, [0x0771] = 0x4697, [0x0772] = 0x278bc,
  [0x0773] = 0x278c8, [0x0774] = 0x8a2e, [0x0775] = 0x27971, [0x0776] = 0x46bc,
  [0x0777] = 0x2796e, [0x0778] = 0x46b3, [0x0779] = 0x27978, [0x077a] = 0x46bf,
  [0x077b] = 0x46b7, [0x077c] = 0x2f9cd, [0x077d] = 0x2797a, [0x077e] = 0x27bb1,
  [0x077f] = 0x27baf, [0x0780] = 0x27bb0, [0x0781] = 0x4734, [0x0782] = 0x4746,
  [0x0783] = 0x4748, [0x0784] = 0x8c59, [0x0785] = 0x27ca8, [0x0786] = 0x27ca6,
  [0x0787] = 0x4756, [0x0788] = 0x4767, [0x0789] = 0x27d48, [0x078a] = 0x27d45,
  [0x078b] = 0x4768, [0x078c] = 0x4799, [0x078d] = 0x479a, [0x078e] = 0x27ea0,
  [0x078f] = 0x27ea4, [0x0790] = 0x47d8, [0x0791] = 0x27fd4, [0x0792] = 0x47db,
  [0x0793] = 0x47dc, [0x0794] = 0x47dd, [0x0795] = 0x47d7, [0x0796] = 0x27fd2,
  [0x0797] = 0x28222, [0x0798] = 0x4849, [0x0799] = 0x484a, [0x079a] = 0x8eda,
  [0x079b] = 0x282b7, [0x079c] = 0x282cc, [0x079d] = 0x9033, [0x079e] = 0x284e6,
  [0x079f] = 0x9018, [0x07a0] = 0x489a, [0x07a1] = 0x284c8, [0x07a2] = 0x48cd,
  [0x07a3] = 0x48ca, [0x07a4] = 0x286f4, [0x07a5] = 0x48cb, [0x07a6] = 0x286ed,
  [0x07a7] = 0x48cf, [0x07a8] = 0x2867e, [0x07a9] = 0x2f9e3, [0x07aa] = 0x48cc,
  [0x07ab] = 0x48ea, [0x07ac] = 0x48ed, [0x07ad] = 0x287f9, [0x07ae] = 0x287fd,
  [0x07af] = 0x48e9, [0x07b0] = 0x28807, [0x07b1] = 0x491a, [0x07b2] = 0x91ef,
  [0x07b3] = 0x498d, [0x07b4] = 0x49d2, [0x07b5] = 0x28e78, [0x07b6] = 0x28e7a,
  [0x07b7] = 0x28e79, [0x07b8] = 0x28f8a, [0x07b9] = 0x49f2, [0x07ba] = 0x2907b,
  [0x07bb] = 0x291d5, [0x07bc] = 0x291e9, [0x07bd] = 0x4a3d, [0x07be] = 0x291eb,
  [0x07bf] = 0x4a3e, [0x07c0] = 0x4af8, [0x07c1] = 0x4b23, [0x07c2] = 0x9ad9,
  [0x07c3] = 0x4eb4, [0x07c4] = 0x2016c, [0x07c5] = 0x50a0, [0x07c6] = 0x5090,
  [0x07c7] = 0x3475, [0x07c8] = 0x5086, [0x07c9] = 0x5084, [0x07ca] = 0x202fa,
  [0x07cb] = 0x508a, [0x07cc] = 0x3476, [0x07cd] = 0x3473, [0x07ce] = 0x509f,
  [0x07cf] = 0x50a1, [0x07d0] = 0x20318, [0x07d1] = 0x5093, [0x07d2] = 0x34bd,
  [0x07d3] = 0x205f6, [0x07d4] = 0x51d5, [0x07d5] = 0x2078a, [0x07d6] = 0x20789,
  [0x07d7] = 0x34f9, [0x07d8] = 0x2077f, [0x07d9] = 0x20788, [0x07da] = 0x34fa,
  [0x07db] = 0x20783, [0x07dc] = 0x20784, [0x07dd] = 0x20798, [0x07de] = 0x20763,
  [0x07df] = 0x20787, [0x07e0] = 0x3521, [0x07e1] = 0x2f827, [0x07e2] = 0x3529,
  [0x07e3] = 0x208f5, [0x07e4] = 0x20950, [0x07e5] = 0x3538, [0x07e6] = 0x354e,
  [0x07e7] = 0x20a87, [0x07e8] = 0x20a8a, [0x07e9] = 0x20b65, [0x07ea] = 0x20db7,
  [0x07eb] = 0x20dc1, [0x07ec] = 0x20dc7, [0x07ed] = 0x35c8, [0x07ee] = 0x35bc,
  [0x07ef] = 0x20dc5, [0x07f0] = 0x20dcb, [0x07f1] = 0x5590, [0x07f2] = 0x20dd1,
  [0x07f3] = 0x35bb, [0x07f4] = 0x35c2, [0x07f5] = 0x35c0, [0x07f6] = 0x20db8,
  [0x07f7] = 0x20dca, [0x07f8] = 0x35ca, [0x07f9] = 0x20dde, [0x07fa] = 0x35c9,
  [0x07fb] = 0x20dce, [0x07fc] = 0x35b8, [0x07fd] = 0x5710, [0x07fe] = 0x5817,
  [0x07ff] = 0x21368, [0x0800] = 0x364e, [0x0801] = 0x2136b, [0x0802] = 0x5844,
  [0x0803] = 0x3650, [0x0804] = 0x582b, [0x0805] = 0x2136a, [0x0806] = 0x5845,
  [0x0807] = 0x21366, [0x0808] = 0x21377, [0x0809] = 0x21396, [0x080a] = 0x2136e,
  [0x080b] = 0x2f856, [0x080c] = 0x368a, [0x080d] = 0x21657, [0x080e] = 0x5965,
  [0x080f] = 0x21656, [0x0810] = 0x2179c, [0x0811] = 0x2179e, [0x0812] = 0x36fc,
  [0x0813] = 0x36f9, [0x0814] = 0x217a1, [0x0815] = 0x3763, [0x0816] = 0x21a42,
  [0x0817] = 0x5bcf, [0x0818] = 0x21a46, [0x0819] = 0x21a3e, [0x081a] = 0x21b33,
  [0x081b] = 0x3787, [0x081c] = 0x3788, [0x081d] = 0x21bfa, [0x081e] = 0x379a,
  [0x081f] = 0x21c97, [0x0820] = 0x21c9b, [0x0821] = 0x21caa, [0x0822] = 0x5d56,
  [0x0823] = 0x21e73, [0x0824] = 0x21e7c, [0x0825] = 0x37e9, [0x0826] = 0x21e86,
  [0x0827] = 0x37ea, [0x0828] = 0x5d54, [0x0829] = 0x3850, [0x082a] = 0x220f3,
  [0x082b] = 0x220f0, [0x082c] = 0x3856, [0x082d] = 0x3852, [0x082e] = 0x384f,
  [0x082f] = 0x3854, [0x0830] = 0x3851, [0x0831] = 0x220ec, [0x0832] = 0x222af,
  [0x0833] = 0x388e, [0x0834] = 0x388f, [0x0835] = 0x22264, [0x0836] = 0x22268,
  [0x0837] = 0x2235a, [0x0838] = 0x5f3d, [0x0839] = 0x2251f, [0x083a] = 0x22525,
  [0x083b] = 0x38ed, [0x083c] = 0x38eb, [0x083d] = 0x5fa4, [0x083e] = 0x226c6,
  [0x083f] = 0x226d6, [0x0840] = 0x226c3, [0x0841] = 0x3962, [0x0842] = 0x2272c,
  [0x0843] = 0x395d, [0x0844] = 0x2272e, [0x0845] = 0x2275e, [0x0846] = 0x22733,
  [0x0847] = 0x3961, [0x0848] = 0x3965, [0x0849] = 0x395c, [0x084a] = 0x2272d,
  [0x084b] = 0x22746, [0x084c] = 0x395f, [0x084d] = 0x226c1, [0x084e] = 0x2273a,
  [0x084f] = 0x229cc, [0x0850] = 0x22b9d, [0x0851] = 0x22b83, [0x0852] = 0x22bf6,
  [0x0853] = 0x63ec, [0x0854] = 0x3a16, [0x0855] = 0x3a0a, [0x0856] = 0x22bf8,
  [0x0857] = 0x3a0e, [0x0858] = 0x3a12, [0x0859] = 0x22bfe, [0x085a] = 0x22bf3,
  [0x085b] = 0x3a11, [0x085c] = 0x22beb, [0x085d] = 0x22bfa, [0x085e] = 0x3a10,
  [0x085f] = 0x22c07, [0x0860] = 0x22bfc, [0x0861] = 0x3a09, [0x0862] = 0x63fa,
  [0x0863] = 0x3a15, [0x0864] = 0x63d4, [0x0865] = 0x22f72, [0x0866] = 0x22f73,
  [0x0867] = 0x22f74, [0x0868] = 0x3a91, [0x0869] = 0x3a95, [0x086a] = 0x3a93,
  [0x086b] = 0x3a92, [0x086c] = 0x3a8f, [0x086d] = 0x22f5f, [0x086e] = 0x22f60,
  [0x086f] = 0x230aa, [0x0870] = 0x23134, [0x0871] = 0x3af6, [0x0872] = 0x231b4,
  [0x0873] = 0x3afb, [0x0874] = 0x231f0, [0x0875] = 0x6675, [0x0876] = 0x3af9,
  [0x0877] = 0x2320d, [0x0878] = 0x231f3, [0x0879] = 0x2320f, [0x087a] = 0x671c,
  [0x087b] = 0x3b7d, [0x087c] = 0x234c9, [0x087d] = 0x3b7a, [0x087e] = 0x3b7f,
  [0x087f] = 0x234c5, [0x0880] = 0x3b78, [0x0881] = 0x234d7, [0x0882] = 0x234cc,
  [0x0883] = 0x68d9, [0x0884] = 0x234e7, [0x0885] = 0x3b70, [0x0886] = 0x3b82,
  [0x0887] = 0x234c1, [0x0888] = 0x3b84, [0x0889] = 0x234e8, [0x088a] = 0x3c33,
  [0x088b] = 0x238cb, [0x088c] = 0x3c32, [0x088d] = 0x3c36, [0x088e] = 0x238c8,
  [0x088f] = 0x2397a, [0x0890] = 0x3c56, [0x0891] = 0x23979, [0x0892] = 0x2397e,
  [0x0893] = 0x3c67, [0x0894] = 0x23a1b, [0x0895] = 0x3c65, [0x0896] = 0x3c64,
  [0x0897] = 0x3c66, [0x0898] = 0x23a1f, [0x0899] = 0x23a19, [0x089a] = 0x3c81,
  [0x089b] = 0x3c82, [0x089c] = 0x3c83, [0x089d] = 0x3c9e, [0x089e] = 0x23b89,
  [0x089f] = 0x6bf1, [0x08a0] = 0x3c9d, [0x08a1] = 0x23b8b, [0x08a2] = 0x3d0f,
  [0x08a3] = 0x3d12, [0x08a4] = 0x23e09, [0x08a5] = 0x23dfe, [0x08a6] = 0x23e00,
  [0x08a7] = 0x3d10, [0x08a8] = 0x3d18, [0x08a9] = 0x23e0b, [0x08aa] = 0x3d14,
  [0x08ab] = 0x3d19, [0x08ac] = 0x6e37, [0x08ae] = 0x23e4a, [0x08af] = 0x6e7d,
  [0x08b0] = 0x6e86, [0x08b1] = 0x3dc8, [0x08b2] = 0x3dc4, [0x08b3] = 0x3dc6,
  [0x08b4] = 0x24279, [0x08b5] = 0x3dc7, [0x08b6] = 0x3dc3, [0x08b7] = 0x2426c,
  [0x08b8] = 0x24299, [0x08b9] = 0x2f91b, [0x08ba] = 0x24262, [0x08bb] = 0x24514,
  [0x08bc] = 0x3e19, [0x08bd] = 0x3e1c, [0x08be] = 0x245c8, [0x08bf] = 0x245d7,
  [0x08c0] = 0x2460c, [0x08c1] = 0x3e41, [0x08c2] = 0x24670, [0x08c3] = 0x3e42,
  [0x08c4] = 0x3e43, [0x08c5] = 0x24671, [0x08c6] = 0x247df, [0x08c7] = 0x247e4,
  [0x08c8] = 0x3e82, [0x08c9] = 0x247e7, [0x08ca] = 0x3e81, [0x08cb] = 0x3e94,
  [0x08cc] = 0x3e84, [0x08cd] = 0x24990, [0x08ce] = 0x3ed2, [0x08cf] = 0x3f0f,
  [0x08d0] = 0x3f22, [0x08d1] = 0x24b4c, [0x08d2] = 0x3f27, [0x08d3] = 0x3f2a,
  [0x08d4] = 0x74fa, [0x08d5] = 0x3f28, [0x08d6] = 0x24b4f, [0x08d7] = 0x3f60,
  [0x08d8] = 0x24c7d, [0x08d9] = 0x24c7e, [0x08da] = 0x24c7c, [0x08db] = 0x7572,
  [0x08dc] = 0x24c8d, [0x08dd] = 0x24d19, [0x08de] = 0x3f9b, [0x08df] = 0x3f9c,
  [0x08e0] = 0x24dc0, [0x08e1] = 0x3f93, [0x08e2] = 0x3f94, [0x08e3] = 0x75dc,
  [0x08e4] = 0x3fa0, [0x08e5] = 0x3f99, [0x08e6] = 0x24db6, [0x08e7] = 0x3fa1,
  [0x08e8] = 0x24da8, [0x08e9] = 0x3ff1, [0x08ea] = 0x24feb, [0x08eb] = 0x24fe7,
  [0x08ec] = 0x2506f, [0x08ed] = 0x251c7, [0x08ee] = 0x4036, [0x08ef] = 0x2519f,
  [0x08f0] = 0x4037, [0x08f1] = 0x403f, [0x08f2] = 0x403c, [0x08f3] = 0x251c4,
  [0x08f4] = 0x4034, [0x08f5] = 0x4039, [0x08f6] = 0x403b, [0x08f7] = 0x4035,
  [0x08f8] = 0x4030, [0x08f9] = 0x4032, [0x08fa] = 0x4038, [0x08fb] = 0x403e,
  [0x08fc] = 0x403a, [0x08fd] = 0x2519b, [0x08fe] = 0x25197, [0x08ff] = 0x251c5,
  [0x0900] = 0x2536b, [0x0901] = 0x253ce, [0x0902] = 0x40b6, [0x0903] = 0x7867,
  [0x0904] = 0x254b1, [0x0905] = 0x254ae, [0x0906] = 0x254b0, [0x0907] = 0x40b3,
  [0x0908] = 0x4109, [0x0909] = 0x7977, [0x090a] = 0x2573e, [0x090b] = 0x257b5,
  [0x090c] = 0x257ba, [0x090d] = 0x414c, [0x090e] = 0x257b2, [0x090f] = 0x257b4,
  [0x0910] = 0x4153, [0x0911] = 0x414d, [0x0912] = 0x4151, [0x0913] = 0x414f,
  [0x0914] = 0x7a9b, [0x0915] = 0x25977, [0x0916] = 0x41a2, [0x0917] = 0x25986,
  [0x0918] = 0x2597b, [0x0919] = 0x41cd, [0x091a] = 0x25a7e, [0x091b] = 0x25b2e,
  [0x091c] = 0x41e7, [0x091d] = 0x25b2f, [0x091e] = 0x41f0, [0x091f] = 0x25b37,
  [0x0920] = 0x41e9, [0x0921] = 0x41ec, [0x0922] = 0x25b32, [0x0923] = 0x41e8,
  [0x0924] = 0x41ee, [0x0925] = 0x4202, [0x0926] = 0x25b36, [0x0927] = 0x25b39,
  [0x0928] = 0x25b35, [0x0929] = 0x25e5c, [0x092a] = 0x4282, [0x092b] = 0x4283,
  [0x092c] = 0x4286, [0x092d] = 0x25e7b, [0x092e] = 0x25e61, [0x092f] = 0x25fee,
  [0x0930] = 0x25feb, [0x0931] = 0x25fef, [0x0932] = 0x26020, [0x0933] = 0x7d2a,
  [0x0934] = 0x7d65, [0x0935] = 0x434a, [0x0936] = 0x26230, [0x0937] = 0x435a,
  [0x0938] = 0x7f64, [0x0939] = 0x262be, [0x093a] = 0x436e, [0x093b] = 0x26367,
  [0x093c] = 0x2637c, [0x093d] = 0x4370, [0x093e] = 0x436f, [0x093f] = 0x26424,
  [0x0940] = 0x438a, [0x0941] = 0x4387, [0x0942] = 0x4388, [0x0943] = 0x26429,
  [0x0944] = 0x264ef, [0x0945] = 0x8020, [0x0946] = 0x43b7, [0x0947] = 0x43fd,
  [0x0948] = 0x8120, [0x0949] = 0x2670d, [0x094a] = 0x4405, [0x094b] = 0x813c,
  [0x094c] = 0x4408, [0x094d] = 0x4403, [0x094e] = 0x4402, [0x094f] = 0x4404,
  [0x0950] = 0x3b39, [0x0951] = 0x4409, [0x0952] = 0x43ff, [0x0953] = 0x26764,
  [0x0954] = 0x813f, [0x0955] = 0x26715, [0x0956] = 0x43fc, [0x0957] = 0x4401,
  [0x0958] = 0x440a, [0x0959] = 0x81f0, [0x095a] = 0x81f5, [0x095b] = 0x446b,
  [0x095c] = 0x446c, [0x095d] = 0x26a35, [0x095e] = 0x26a36, [0x095f] = 0x26a34,
  [0x0960] = 0x26a37, [0x0961] = 0x26c47, [0x0962] = 0x26c2b, [0x0963] = 0x44de,
  [0x0964] = 0x26c41, [0x0965] = 0x26c43, [0x0966] = 0x44db, [0x0967] = 0x26c42,
  [0x0968] = 0x44dd, [0x0969] = 0x44e3, [0x096a] = 0x26c32, [0x096b] = 0x44e0,
  [0x096c] = 0x44d9, [0x096d] = 0x44d8, [0x096e] = 0x44e4, [0x096f] = 0x26c44,
  [0x0970] = 0x26c4a, [0x0971] = 0x44da, [0x0972] = 0x44ef, [0x0973] = 0x2f9a0,
  [0x0974] = 0x8415, [0x0975] = 0x83be, [0x0976] = 0x26c54, [0x0977] = 0x26c6e,
  [0x0978] = 0x26c52, [0x0979] = 0x44d7, [0x097a] = 0x45b3, [0x097b] = 0x45bb,
  [0x097c] = 0x86e5, [0x097d] = 0x45b2, [0x097e] = 0x86d2, [0x097f] = 0x45ad,
  [0x0980] = 0x27292, [0x0981] = 0x45af, [0x0982] = 0x2f9b8, [0x0983] = 0x86e0,
  [0x0984] = 0x272bf, [0x0985] = 0x4616, [0x0986] = 0x4628, [0x0987] = 0x4623,
  [0x0988] = 0x88b3, [0x0989] = 0x27641, [0x098a] = 0x27669, [0x098b] = 0x27665,
  [0x098c] = 0x4675, [0x098d] = 0x467e, [0x098e] = 0x467c, [0x098f] = 0x27822,
  [0x0990] = 0x2781d, [0x0991] = 0x278d2, [0x0992] = 0x278da, [0x0993] = 0x278db,
  [0x0994] = 0x46ce, [0x0995] = 0x46cd, [0x0996] = 0x46cf, [0x0997] = 0x8a53,
  [0x0998] = 0x279a4, [0x0999] = 0x2799e, [0x099a] = 0x8a37, [0x099b] = 0x8a47,
  [0x099c] = 0x8a5c, [0x099d] = 0x2799d, [0x099e] = 0x46c4, [0x099f] = 0x46cc,
  [0x09a0] = 0x46c8, [0x09a1] = 0x46c7, [0x09a2] = 0x279ad, [0x09a3] = 0x279a6,
  [0x09a4] = 0x279a7, [0x09a5] = 0x27bb3, [0x09a6] = 0x27be0, [0x09a7] = 0x27c35,
  [0x09a8] = 0x27c37, [0x09a9] = 0x475a, [0x09aa] = 0x476a, [0x09ab] = 0x27d4a,
  [0x09ac] = 0x476b, [0x09ad] = 0x476d, [0x09ae] = 0x476f, [0x09af] = 0x27e5e,
  [0x09b0] = 0x479e, [0x09b1] = 0x27ec0, [0x09b2] = 0x47a4, [0x09b3] = 0x47a3,
  [0x09b4] = 0x47e4, [0x09b5] = 0x47e8, [0x09b6] = 0x47e9, [0x09b7] = 0x47e0,
  [0x09b8] = 0x47e3, [0x09b9] = 0x27ff9, [0x09ba] = 0x47ea, [0x09bb] = 0x47e1,
  [0x09bc] = 0x47ed, [0x09bd] = 0x4834, [0x09be] = 0x4835, [0x09bf] = 0x4851,
  [0x09c0] = 0x8ef0, [0x09c1] = 0x284f0, [0x09c2] = 0x489d, [0x09c3] = 0x284f3,
  [0x09c4] = 0x2851b, [0x09c5] = 0x284f2, [0x09c6] = 0x284fb, [0x09c7] = 0x2871c,
  [0x09c8] = 0x28713, [0x09c9] = 0x48d0, [0x09ca] = 0x2870f, [0x09cb] = 0x48ee,
  [0x09cc] = 0x28816, [0x09cd] = 0x28811, [0x09ce] = 0x28812, [0x09cf] = 0x48f2,
  [0x09d0] = 0x2880e, [0x09d1] = 0x921d, [0x09d2] = 0x4988, [0x09d3] = 0x28c67,
  [0x09d4] = 0x28cf2, [0x09d5] = 0x28cda, [0x09d6] = 0x498f, [0x09d7] = 0x28cdc,
  [0x09d8] = 0x49d8, [0x09d9] = 0x28f94, [0x09da] = 0x28f9b, [0x09db] = 0x28f92,
  [0x09dc] = 0x28f9a, [0x09dd] = 0x29088, [0x09de] = 0x29086, [0x09df] = 0x4a3f,
  [0x09e0] = 0x291ef, [0x09e1] = 0x4a52, [0x09e2] = 0x976b, [0x09e3] = 0x2925e,
  [0x09e4] = 0x4a50, [0x09e5] = 0x29458, [0x09e6] = 0x4ab1, [0x09e7] = 0x29456,
  [0x09e8] = 0x29454, [0x09e9] = 0x4af9, [0x09ea] = 0x2959b, [0x09eb] = 0x29696,
  [0x09ec] = 0x296a4, [0x09ed] = 0x4b26, [0x09ee] = 0x4b28, [0x09ef] = 0x2969a,
  [0x09f0] = 0x29a12, [0x09f1] = 0x3480, [0x09f2] = 0x50c0, [0x09f3] = 0x3481,
  [0x09f4] = 0x2036b, [0x09f5] = 0x347e, [0x09f6] = 0x347f, [0x09f7] = 0x20337,
  [0x09f8] = 0x2033c, [0x09f9] = 0x2f80a, [0x09fa] = 0x204f7, [0x09fb] = 0x34be,
  [0x09fc] = 0x20566, [0x09fd] = 0x34d6, [0x09fe] = 0x207a9, [0x09ff] = 0x207ae,
  [0x0a00] = 0x207ad, [0x0a01] = 0x207c8, [0x0a02] = 0x207a5, [0x0a03] = 0x207b0,
  [0x0a04] = 0x20895, [0x0a05] = 0x52e5, [0x0a06] = 0x20925, [0x0a07] = 0x3534,
  [0x0a08] = 0x20a99, [0x0a09] = 0x53af, [0x0a0a] = 0x20b74, [0x0a0b] = 0x20b70,
  [0x0a0c] = 0x20b6f, [0x0a0d] = 0x20e41, [0x0a0e] = 0x20e54, [0x0a0f] = 0x35d5,
  [0x0a10] = 0x35d8, [0x0a11] = 0x20e40, [0x0a12] = 0x20e38, [0x0a13] = 0x35d4,
  [0x0a14] = 0x55d8, [0x0a15] = 0x35d9, [0x0a16] = 0x20e52, [0x0a17] = 0x20e3a,
  [0x0a18] = 0x20e57, [0x0a19] = 0x2f84a, [0x0a1a] = 0x20e59, [0x0a1b] = 0x5711,
  [0x0a1c] = 0x5867, [0x0a1d] = 0x213b4, [0x0a1e] = 0x213c0, [0x0a1f] = 0x365d,
  [0x0a20] = 0x5843, [0x0a21] = 0x365e, [0x0a22] = 0x3659, [0x0a23] = 0x21375,
  [0x0a24] = 0x365a, [0x0a25] = 0x36a0, [0x0a26] = 0x21660, [0x0a27] = 0x21647,
  [0x0a28] = 0x3705, [0x0a29] = 0x217f0, [0x0a2a] = 0x3707, [0x0a2b] = 0x217ef,
  [0x0a2c] = 0x370e, [0x0a2d] = 0x370c, [0x0a2e] = 0x217ec, [0x0a2f] = 0x21983,
  [0x0a30] = 0x3745, [0x0a31] = 0x21984, [0x0a32] = 0x2198f, [0x0a33] = 0x3764,
  [0x0a34] = 0x21a61, [0x0a35] = 0x3765, [0x0a36] = 0x21a69, [0x0a37] = 0x5bdd,
  [0x0a38] = 0x3766, [0x0a39] = 0x21a62, [0x0a3a] = 0x3789, [0x0a3b] = 0x37ec,
  [0x0a3c] = 0x37f1, [0x0a3d] = 0x5d70, [0x0a3e] = 0x5d6a, [0x0a3f] = 0x37f0,
  [0x0a40] = 0x37f8, [0x0a41] = 0x5d74, [0x0a42] = 0x5d5f, [0x0a43] = 0x21eae,
  [0x0a44] = 0x5d61, [0x0a45] = 0x5d73, [0x0a46] = 0x21eb2, [0x0a47] = 0x37f2,
  [0x0a48] = 0x37f4, [0x0a49] = 0x2210b, [0x0a4a] = 0x3858, [0x0a4b] = 0x22110,
  [0x0a4c] = 0x2210d, [0x0a4d] = 0x385a, [0x0a4e] = 0x3859, [0x0a4f] = 0x3857,
  [0x0a50] = 0x385b, [0x0a51] = 0x5e50, [0x0a52] = 0x2227f, [0x0a53] = 0x22281,
  [0x0a54] = 0x38a6, [0x0a55] = 0x38c2, [0x0a56] = 0x38c1, [0x0a57] = 0x5f3f,
  [0x0a58] = 0x2240b, [0x0a59] = 0x2254e, [0x0a5a] = 0x38ef, [0x0a5b] = 0x5fb0,
  [0x0a5c] = 0x227ea, [0x0a5d] = 0x22745, [0x0a5e] = 0x3968, [0x0a5f] = 0x6135,
  [0x0a60] = 0x612d, [0x0a61] = 0x3973, [0x0a62] = 0x396e, [0x0a63] = 0x3974,
  [0x0a64] = 0x6102, [0x0a65] = 0x3966, [0x0a66] = 0x22728, [0x0a67] = 0x2275d,
  [0x0a68] = 0x229dc, [0x0a69] = 0x39b9, [0x0a6a] = 0x6226, [0x0a6b] = 0x229d4,
  [0x0a6c] = 0x3a0c, [0x0a6d] = 0x22c64, [0x0a6e] = 0x22c68, [0x0a6f] = 0x3a20,
  [0x0a70] = 0x22c62, [0x0a71] = 0x3a1d, [0x0a72] = 0x22c61, [0x0a73] = 0x3a1c,
  [0x0a74] = 0x22c66, [0x0a75] = 0x3a21, [0x0a76] = 0x3a1a, [0x0a77] = 0x3a19,
  [0x0a78] = 0x22bf2, [0x0a79] = 0x22c69, [0x0a7a] = 0x22c67, [0x0a7b] = 0x3a7e,
  [0x0a7c] = 0x22f9a, [0x0a7d] = 0x22f9b, [0x0a7e] = 0x22f85, [0x0a7f] = 0x22f9c,
  [0x0a80] = 0x3a9d, [0x0a81] = 0x22fa4, [0x0a82] = 0x3a9e, [0x0a83] = 0x3a9c,
  [0x0a84] = 0x656e, [0x0a85] = 0x22f99, [0x0a86] = 0x22f86, [0x0a87] = 0x22f90,
  [0x0a88] = 0x23081, [0x0a89] = 0x230ae, [0x0a8a] = 0x65b1, [0x0a8b] = 0x65d4,
  [0x0a8c] = 0x3acd, [0x0a8d] = 0x23138, [0x0a8e] = 0x3b0b, [0x0a8f] = 0x3b0a,
  [0x0a90] = 0x6685, [0x0a91] = 0x2353b, [0x0a92] = 0x3b8f, [0x0a93] = 0x6972,
  [0x0a94] = 0x3b95, [0x0a95] = 0x3b90, [0x0a96] = 0x3b91, [0x0a97] = 0x2354c,
  [0x0a98] = 0x693a, [0x0a99] = 0x3bb9, [0x0a9a] = 0x23547, [0x0a9b] = 0x23535,
  [0x0a9c] = 0x3b97, [0x0a9d] = 0x3b9e, [0x0a9e] = 0x23533, [0x0a9f] = 0x3b8b,
  [0x0aa0] = 0x23582, [0x0aa1] = 0x23540, [0x0aa2] = 0x3c3b, [0x0aa3] = 0x3c3a,
  [0x0aa4] = 0x3c3c, [0x0aa5] = 0x3c3d, [0x0aa6] = 0x3c39, [0x0aa7] = 0x3c3e,
  [0x0aa8] = 0x3c6b, [0x0aa9] = 0x3c6c, [0x0aaa] = 0x23a36, [0x0aab] = 0x23b0e,
  [0x0aac] = 0x3ca2, [0x0aad] = 0x3ca1, [0x0aae] = 0x3c9f, [0x0aaf] = 0x23bad,
  [0x0ab0] = 0x23baa, [0x0ab1] = 0x23e17, [0x0ab2] = 0x3d2d, [0x0ab3] = 0x23e67,
  [0x0ab4] = 0x3d36, [0x0ab5] = 0x3d2b, [0x0ab6] = 0x23e72, [0x0ab7] = 0x3d37,
  [0x0ab8] = 0x2f90a, [0x0ab9] = 0x23e61, [0x0aba] = 0x23eb0, [0x0abb] = 0x6ead,
  [0x0abc] = 0x3d25, [0x0abd] = 0x3d2f, [0x0abe] = 0x3d2c, [0x0abf] = 0x23e6e,
  [0x0ac0] = 0x23e64, [0x0ac1] = 0x3d32, [0x0ac2] = 0x23e8c, [0x0ac3] = 0x23e66,
  [0x0ac4] = 0x6e95, [0x0ac5] = 0x23e6b, [0x0ac6] = 0x23e5f, [0x0ac7] = 0x23e74,
  [0x0ac8] = 0x23e65, [0x0ac9] = 0x242bb, [0x0aca] = 0x242be, [0x0acb] = 0x3dd5,
  [0x0acc] = 0x242b9, [0x0acd] = 0x3dd4, [0x0ace] = 0x3dd6, [0x0acf] = 0x242ef,
  [0x0ad0] = 0x3dd1, [0x0ad1] = 0x7243, [0x0ad2] = 0x245da, [0x0ad3] = 0x3e46,
  [0x0ad4] = 0x728f, [0x0ad5] = 0x24690, [0x0ad6] = 0x247e9, [0x0ad7] = 0x3e8c,
  [0x0ad8] = 0x3e8a, [0x0ad9] = 0x3e88, [0x0ada] = 0x24811, [0x0adb] = 0x2480d,
  [0x0adc] = 0x247ed, [0x0add] = 0x24821, [0x0ade] = 0x3edd, [0x0adf] = 0x24981,
  [0x0ae0] = 0x249b1, [0x0ae1] = 0x24b01, [0x0ae2] = 0x3f2d, [0x0ae3] = 0x24b5e,
  [0x0ae4] = 0x24b62, [0x0ae5] = 0x3f2e, [0x0ae6] = 0x3f2c, [0x0ae7] = 0x3f2b,
  [0x0ae8] = 0x3f30, [0x0ae9] = 0x24b5b, [0x0aea] = 0x3f4e, [0x0aeb] = 0x24c96,
  [0x0aec] = 0x24c93, [0x0aed] = 0x3f64, [0x0aee] = 0x3f61, [0x0aef] = 0x24c92,
  [0x0af0] = 0x7575, [0x0af1] = 0x2f939, [0x0af2] = 0x3f70, [0x0af3] = 0x3fa6,
  [0x0af4] = 0x3fa4, [0x0af5] = 0x24dc4, [0x0af6] = 0x24dc7, [0x0af7] = 0x24dc3,
  [0x0af8] = 0x3fa8, [0x0af9] = 0x3fa2, [0x0afa] = 0x24dc8, [0x0afb] = 0x3fa7,
  [0x0afc] = 0x75ec, [0x0afd] = 0x3fa5, [0x0afe] = 0x24dca, [0x0aff] = 0x3fa9,
  [0x0b00] = 0x24dc5, [0x0b01] = 0x24dcf, [0x0b02] = 0x24ddc, [0x0b03] = 0x24f7c,
  [0x0b04] = 0x25001, [0x0b05] = 0x403d, [0x0b06] = 0x251d3, [0x0b07] = 0x251dc,
  [0x0b08] = 0x4044, [0x0b09] = 0x4045, [0x0b0a] = 0x4046, [0x0b0b] = 0x251d4,
  [0x0b0c] = 0x7757, [0x0b0d] = 0x251cc, [0x0b0e] = 0x4047, [0x0b0f] = 0x4048,
  [0x0b10] = 0x4042, [0x0b11] = 0x251d6, [0x0b12] = 0x251db, [0x0b13] = 0x4041,
  [0x0b14] = 0x251d5, [0x0b15] = 0x253d9, [0x0b16] = 0x4094, [0x0b17] = 0x253dd,
  [0x0b18] = 0x253dc, [0x0b19] = 0x253e0, [0x0b1a] = 0x254cc, [0x0b1b] = 0x40c0,
  [0x0b1c] = 0x40b8, [0x0b1d] = 0x40c1, [0x0b1e] = 0x40c2, [0x0b1f] = 0x40bb,
  [0x0b20] = 0x40bd, [0x0b21] = 0x40bf, [0x0b22] = 0x40b9, [0x0b23] = 0x40b7,
  [0x0b24] = 0x254d2, [0x0b25] = 0x40c7, [0x0b26] = 0x254d3, [0x0b27] = 0x410c,
  [0x0b28] = 0x410b, [0x0b29] = 0x797b, [0x0b2a] = 0x4110, [0x0b2b] = 0x2568d,
  [0x0b2c] = 0x415d, [0x0b2d] = 0x7a21, [0x0b2e] = 0x415a, [0x0b2f] = 0x4158,
  [0x0b30] = 0x4156, [0x0b31] = 0x257d8, [0x0b32] = 0x4154, [0x0b33] = 0x7a16,
  [0x0b34] = 0x257bc, [0x0b35] = 0x41a8, [0x0b36] = 0x41a7, [0x0b37] = 0x41cf,
  [0x0b38] = 0x41d0, [0x0b39] = 0x25a8a, [0x0b3a] = 0x7ae8, [0x0b3b] = 0x25a8b,
  [0x0b3c] = 0x25a8d, [0x0b3d] = 0x41d1, [0x0b3e] = 0x41eb, [0x0b3f] = 0x25b53,
  [0x0b40] = 0x41fb, [0x0b41] = 0x7b6a, [0x0b42] = 0x25b59, [0x0b43] = 0x41fd,
  [0x0b44] = 0x41f8, [0x0b45] = 0x41f7, [0x0b46] = 0x4200, [0x0b47] = 0x25b56,
  [0x0b48] = 0x25b57, [0x0b49] = 0x41f6, [0x0b4a] = 0x7b5f, [0x0b4b] = 0x25e73,
  [0x0b4c] = 0x2601b, [0x0b4d] = 0x42df, [0x0b4e] = 0x26021, [0x0b4f] = 0x26016,
  [0x0b50] = 0x26018, [0x0b51] = 0x42e2, [0x0b52] = 0x42e4, [0x0b53] = 0x26044,
  [0x0b54] = 0x7d82, [0x0b55] = 0x26026, [0x0b56] = 0x42e3, [0x0b57] = 0x26236,
  [0x0b58] = 0x4359, [0x0b59] = 0x4371, [0x0b5a] = 0x438e, [0x0b5b] = 0x438c,
  [0x0b5c] = 0x2643a, [0x0b5d] = 0x43a4, [0x0b5e] = 0x264f4, [0x0b5f] = 0x8055,
  [0x0b60] = 0x4414, [0x0b61] = 0x26762, [0x0b62] = 0x2675b, [0x0b63] = 0x2676a,
  [0x0b64] = 0x4411, [0x0b65] = 0x26768, [0x0b66] = 0x441b, [0x0b67] = 0x4412,
  [0x0b68] = 0x440e, [0x0b69] = 0x4415, [0x0b6a] = 0x8168, [0x0b6b] = 0x4410,
  [0x0b6c] = 0x26781, [0x0b6d] = 0x4417, [0x0b6e] = 0x8246, [0x0b6f] = 0x8243,
  [0x0b70] = 0x4470, [0x0b71] = 0x44ed, [0x0b72] = 0x26ce7, [0x0b73] = 0x44ee,
  [0x0b74] = 0x26ce6, [0x0b75] = 0x26cd1, [0x0b76] = 0x26d11, [0x0b77] = 0x8481,
  [0x0b78] = 0x26cd0, [0x0b79] = 0x26cd9, [0x0b7a] = 0x26d0a, [0x0b7b] = 0x44f4,
  [0x0b7c] = 0x26cda, [0x0b7d] = 0x26cdd, [0x0b7e] = 0x26cc8, [0x0b7f] = 0x26ce2,
  [0x0b80] = 0x26ce9, [0x0b81] = 0x26ccb, [0x0b82] = 0x847c, [0x0b83] = 0x26ccc,
  [0x0b84] = 0x26cc1, [0x0b85] = 0x846a, [0x0b86] = 0x26cde, [0x0b87] = 0x8488,
  [0x0b88] = 0x44f2, [0x0b89] = 0x44f8, [0x0b8a] = 0x44f3, [0x0b8b] = 0x26d51,
  [0x0b8c] = 0x26d12, [0x0b8d] = 0x44fa, [0x0b8e] = 0x26ce0, [0x0b8f] = 0x26d2b,
  [0x0b90] = 0x2f9b4, [0x0b91] = 0x272d8, [0x0b92] = 0x272d1, [0x0b93] = 0x8710,
  [0x0b94] = 0x272d2, [0x0b95] = 0x272cd, [0x0b96] = 0x871f, [0x0b97] = 0x45b6,
  [0x0b98] = 0x45b7, [0x0b99] = 0x2733d, [0x0b9a] = 0x870f, [0x0b9b] = 0x272ca,
  [0x0b9c] = 0x45ba, [0x0b9d] = 0x272cb, [0x0b9e] = 0x45bc, [0x0b9f] = 0x272d7,
  [0x0ba0] = 0x275f4, [0x0ba1] = 0x463b, [0x0ba2] = 0x88d3, [0x0ba3] = 0x462f,
  [0x0ba4] = 0x27687, [0x0ba5] = 0x2768b, [0x0ba6] = 0x27683, [0x0ba7] = 0x4637,
  [0x0ba8] = 0x4699, [0x0ba9] = 0x279c5, [0x0baa] = 0x279d2, [0x0bab] = 0x279c6,
  [0x0bac] = 0x279d3, [0x0bad] = 0x46d9, [0x0bae] = 0x46d8, [0x0baf] = 0x46d7,
  [0x0bb0] = 0x279f0, [0x0bb1] = 0x279cd, [0x0bb2] = 0x279cc, [0x0bb3] = 0x279dc,
  [0x0bb4] = 0x279d6, [0x0bb5] = 0x4736, [0x0bb6] = 0x27be6, [0x0bb7] = 0x27c41,
  [0x0bb8] = 0x27c3f, [0x0bb9] = 0x8c87, [0x0bba] = 0x27cc5, [0x0bbb] = 0x27cc0,
  [0x0bbc] = 0x27cc6, [0x0bbd] = 0x27cbe, [0x0bbe] = 0x27cbf, [0x0bbf] = 0x8cc6,
  [0x0bc0] = 0x4770, [0x0bc1] = 0x27d63, [0x0bc2] = 0x27ed3, [0x0bc3] = 0x27eda,
  [0x0bc4] = 0x27ed5, [0x0bc5] = 0x47a5, [0x0bc6] = 0x47a6, [0x0bc7] = 0x47a9,
  [0x0bc8] = 0x47ee, [0x0bc9] = 0x4854, [0x0bca] = 0x282ee, [0x0bcb] = 0x4857,
  [0x0bcc] = 0x282fc, [0x0bcd] = 0x28414, [0x0bce] = 0x48a1, [0x0bcf] = 0x28523,
  [0x0bd0] = 0x28522, [0x0bd1] = 0x28530, [0x0bd2] = 0x286b5, [0x0bd3] = 0x48d3,
  [0x0bd4] = 0x28738, [0x0bd5] = 0x48d4, [0x0bd6] = 0x2873f, [0x0bd7] = 0x48d7,
  [0x0bd8] = 0x90cc, [0x0bd9] = 0x916d, [0x0bda] = 0x9170, [0x0bdb] = 0x48f7,
  [0x0bdc] = 0x48f6, [0x0bdd] = 0x48f9, [0x0bde] = 0x48f8, [0x0bdf] = 0x9258,
  [0x0be0] = 0x9242, [0x0be1] = 0x9268, [0x0be2] = 0x9269, [0x0be3] = 0x28968,
  [0x0be4] = 0x28967, [0x0be5] = 0x9243, [0x0be6] = 0x28965, [0x0be7] = 0x9247,
  [0x0be8] = 0x498a, [0x0be9] = 0x28c75, [0x0bea] = 0x28cf7, [0x0beb] = 0x28cf3,
  [0x0bec] = 0x28cf6, [0x0bed] = 0x4994, [0x0bee] = 0x28d00, [0x0bef] = 0x4993,
  [0x0bf0] = 0x28cff, [0x0bf1] = 0x28cf5, [0x0bf2] = 0x959d, [0x0bf3] = 0x49dd,
  [0x0bf4] = 0x49dc, [0x0bf5] = 0x49f7, [0x0bf6] = 0x96cf, [0x0bf7] = 0x29092,
  [0x0bf8] = 0x29206, [0x0bf9] = 0x4a42, [0x0bfa] = 0x29207, [0x0bfb] = 0x2926b,
  [0x0bfc] = 0x4a54, [0x0bfd] = 0x2926c, [0x0bfe] = 0x4a55, [0x0bff] = 0x29393,
  [0x0c00] = 0x4a8f, [0x0c01] = 0x29394, [0x0c02] = 0x97f4, [0x0c03] = 0x4ab4,
  [0x0c04] = 0x4ab3, [0x0c05] = 0x29462, [0x0c06] = 0x29463, [0x0c07] = 0x9809,
  [0x0c08] = 0x2945f, [0x0c09] = 0x29464, [0x0c0a] = 0x2945b, [0x0c0b] = 0x29459,
  [0x0c0c] = 0x4afb, [0x0c0d] = 0x4afd, [0x0c0e] = 0x295a4, [0x0c0f] = 0x295ac,
  [0x0c10] = 0x98ab, [0x0c11] = 0x4afc, [0x0c12] = 0x296b5, [0x0c13] = 0x4b2c,
  [0x0c14] = 0x4b2f, [0x0c15] = 0x296ae, [0x0c16] = 0x4b2b, [0x0c17] = 0x296c5,
  [0x0c18] = 0x4b33, [0x0c19] = 0x4b34, [0x0c1a] = 0x98fb, [0x0c1b] = 0x2983b,
  [0x0c1c] = 0x9aac, [0x0c1d] = 0x9aae, [0x0c1e] = 0x9aaa, [0x0c1f] = 0x4be8,
  [0x0c20] = 0x29abf, [0x0c21] = 0x29c2b, [0x0c22] = 0x29c84, [0x0c23] = 0x29c80,
  [0x0c24] = 0x29d4d, [0x0c25] = 0x29d4f, [0x0c26] = 0x9b5c, [0x0c27] = 0x29d4c,
  [0x0c28] = 0x29f92, [0x0c29] = 0x4d5d, [0x0c2a] = 0x50d2, [0x0c2b] = 0x3485,
  [0x0c2c] = 0x3488, [0x0c2d] = 0x203ae, [0x0c2e] = 0x20379, [0x0c2f] = 0x348e,
  [0x0c30] = 0x3484, [0x0c31] = 0x20371, [0x0c32] = 0x50df, [0x0c33] = 0x203b3,
  [0x0c34] = 0x3483, [0x0c35] = 0x2032c, [0x0c36] = 0x20377, [0x0c37] = 0x2052f,
  [0x0c38] = 0x207c3, [0x0c39] = 0x3502, [0x0c3a] = 0x207cd, [0x0c3b] = 0x3506,
  [0x0c3c] = 0x3505, [0x0c3d] = 0x207c9, [0x0c3e] = 0x34fe, [0x0c3f] = 0x3501,
  [0x0c40] = 0x3500, [0x0c41] = 0x207ce, [0x0c42] = 0x208a5, [0x0c43] = 0x208a0,
  [0x0c44] = 0x208fe, [0x0c45] = 0x20959, [0x0c46] = 0x20a9a, [0x0c47] = 0x35e7,
  [0x0c48] = 0x5619, [0x0c49] = 0x20ee4, [0x0c4a] = 0x20ed7, [0x0c4b] = 0x20edc,
  [0x0c4c] = 0x35e6, [0x0c4d] = 0x20ee7, [0x0c4e] = 0x35ed, [0x0c4f] = 0x35e2,
  [0x0c50] = 0x35eb, [0x0c51] = 0x20ee5, [0x0c52] = 0x35e8, [0x0c53] = 0x35ec,
  [0x0c54] = 0x560a, [0x0c55] = 0x3624, [0x0c56] = 0x589a, [0x0c57] = 0x21414,
  [0x0c58] = 0x3662, [0x0c59] = 0x213f1, [0x0c5a] = 0x3661, [0x0c5b] = 0x3660,
  [0x0c5c] = 0x3664, [0x0c5d] = 0x368b, [0x0c5e] = 0x215b6, [0x0c5f] = 0x2166e,
  [0x0c60] = 0x2166c, [0x0c61] = 0x21681, [0x0c62] = 0x21842, [0x0c63] = 0x3719,
  [0x0c64] = 0x3716, [0x0c65] = 0x3718, [0x0c66] = 0x3722, [0x0c67] = 0x2181a,
  [0x0c68] = 0x371d, [0x0c69] = 0x3717, [0x0c6a] = 0x371e, [0x0c6b] = 0x21827,
  [0x0c6c] = 0x21825, [0x0c6d] = 0x21817, [0x0c6e] = 0x2181c, [0x0c6f] = 0x2183d,
  [0x0c70] = 0x21820, [0x0c71] = 0x3769, [0x0c72] = 0x376a, [0x0c73] = 0x21a81,
  [0x0c74] = 0x376c, [0x0c75] = 0x377a, [0x0c76] = 0x378a, [0x0c77] = 0x21c09,
  [0x0c78] = 0x21cc6, [0x0c79] = 0x21cc4, [0x0c7a] = 0x379c, [0x0c7b] = 0x21cc5,
  [0x0c7c] = 0x37fd, [0x0c7d] = 0x37f9, [0x0c7e] = 0x21eef, [0x0c7f] = 0x37ff,
  [0x0c80] = 0x21edf, [0x0c81] = 0x21ede, [0x0c82] = 0x21eee, [0x0c83] = 0x21ef5,
  [0x0c84] = 0x21eec, [0x0c85] = 0x37fc, [0x0c86] = 0x21edd, [0x0c87] = 0x5d85,
  [0x0c88] = 0x37fb, [0x0c89] = 0x3802, [0x0c8a] = 0x385f, [0x0c8b] = 0x5e56,
  [0x0c8c] = 0x385e, [0x0c8d] = 0x385d, [0x0c8e] = 0x385c, [0x0c8f] = 0x22122,
  [0x0c90] = 0x5e51, [0x0c91] = 0x3892, [0x0c92] = 0x22297, [0x0c93] = 0x22295,
  [0x0c94] = 0x3894, [0x0c95] = 0x3895, [0x0c96] = 0x38d1, [0x0c97] = 0x224a9,
  [0x0c98] = 0x38f1, [0x0c99] = 0x22553, [0x0c9a] = 0x5fb1, [0x0c9b] = 0x22556,
  [0x0c9c] = 0x22555, [0x0c9d] = 0x3977, [0x0c9e] = 0x396f, [0x0c9f] = 0x227e3,
  [0x0ca0] = 0x227e2, [0x0ca1] = 0x3987, [0x0ca2] = 0x397d, [0x0ca3] = 0x397c,
  [0x0ca4] = 0x397e, [0x0ca5] = 0x3985, [0x0ca6] = 0x398b, [0x0ca7] = 0x3986,
  [0x0ca8] = 0x3980, [0x0caa] = 0x227a9, [0x0cab] = 0x3978, [0x0cac] = 0x2f8ab,
  [0x0cad] = 0x22833, [0x0cae] = 0x2280a, [0x0caf] = 0x39ba, [0x0cb0] = 0x22a59,
  [0x0cb1] = 0x3a33, [0x0cb2] = 0x22cbc, [0x0cb3] = 0x3a2d, [0x0cb4] = 0x22cc4,
  [0x0cb5] = 0x22cc7, [0x0cb6] = 0x22cc6, [0x0cb7] = 0x3a37, [0x0cb8] = 0x645a,
  [0x0cb9] = 0x6463, [0x0cba] = 0x22cb8, [0x0cbb] = 0x22cda, [0x0cbc] = 0x3a2e,
  [0x0cbd] = 0x22cb7, [0x0cbe] = 0x22cc0, [0x0cbf] = 0x3a3d, [0x0cc0] = 0x22ccd,
  [0x0cc1] = 0x3aa0, [0x0cc2] = 0x22fbc, [0x0cc3] = 0x22fab, [0x0cc4] = 0x3aa3,
  [0x0cc5] = 0x23086, [0x0cc6] = 0x669b, [0x0cc7] = 0x2325a, [0x0cc8] = 0x66a3,
  [0x0cc9] = 0x3b0e, [0x0cca] = 0x669e, [0x0ccb] = 0x23343, [0x0ccc] = 0x3bb6,
  [0x0ccd] = 0x235ac, [0x0cce] = 0x3bab, [0x0ccf] = 0x3bad, [0x0cd0] = 0x3ba6,
  [0x0cd1] = 0x235be, [0x0cd2] = 0x69b8, [0x0cd3] = 0x3baa, [0x0cd4] = 0x69ba,
  [0x0cd5] = 0x3bb1, [0x0cd6] = 0x235ab, [0x0cd7] = 0x3ba8, [0x0cd8] = 0x3baf,
  [0x0cd9] = 0x3bb0, [0x0cda] = 0x3ba7, [0x0cdb] = 0x3bb2, [0x0cdc] = 0x3b9d,
  [0x0cdd] = 0x3ba5, [0x0cde] = 0x3bb5, [0x0cdf] = 0x21b3e, [0x0ce0] = 0x69c7,
  [0x0ce1] = 0x69d7, [0x0ce2] = 0x238f9, [0x0ce3] = 0x3c41, [0x0ce4] = 0x238f3,
  [0x0ce5] = 0x238f7, [0x0ce6] = 0x6b70, [0x0ce7] = 0x23a45, [0x0ce8] = 0x23a48,
  [0x0ce9] = 0x3c72, [0x0cea] = 0x6b9d, [0x0ceb] = 0x3c6f, [0x0cec] = 0x3c71,
  [0x0ced] = 0x23a44, [0x0cee] = 0x3c85, [0x0cef] = 0x23aaf, [0x0cf0] = 0x23ab1,
  [0x0cf1] = 0x3ca4, [0x0cf2] = 0x3ca5, [0x0cf3] = 0x3ca6, [0x0cf4] = 0x23bcf,
  [0x0cf5] = 0x3ca8, [0x0cf6] = 0x23bcb, [0x0cf7] = 0x23bcd, [0x0cf8] = 0x3ca3,
  [0x0cf9] = 0x23c50, [0x0cfa] = 0x23eec, [0x0cfb] = 0x23f1a, [0x0cfc] = 0x6f16,
  [0x0cfd] = 0x6f24, [0x0cfe] = 0x23ed8, [0x0cff] = 0x3d43, [0x0d00] = 0x23ed0,
  [0x0d01] = 0x23ed1, [0x0d02] = 0x3d3d, [0x0d03] = 0x3d45, [0x0d04] = 0x23e7b,
  [0x0d05] = 0x23ed2, [0x0d06] = 0x3d44, [0x0d07] = 0x23ed4, [0x0d08] = 0x23ee7,
  [0x0d09] = 0x23edf, [0x0d0a] = 0x23ede, [0x0d0b] = 0x6f45, [0x0d0c] = 0x23ed9,
  [0x0d0d] = 0x23ecf, [0x0d0e] = 0x2430d, [0x0d0f] = 0x3de3, [0x0d10] = 0x7179,
  [0x0d12] = 0x3ddf, [0x0d13] = 0x3de4, [0x0d14] = 0x717a, [0x0d15] = 0x3de5,
  [0x0d16] = 0x24338, [0x0d17] = 0x7254, [0x0d18] = 0x3e22, [0x0d19] = 0x3e4a,
  [0x0d1a] = 0x246a9, [0x0d1b] = 0x3e49, [0x0d1c] = 0x3e44, [0x0d1d] = 0x3e4b,
  [0x0d1e] = 0x3e87, [0x0d1f] = 0x3e89, [0x0d20] = 0x3e92, [0x0d21] = 0x3e91,
  [0x0d22] = 0x3e90, [0x0d23] = 0x3e8e, [0x0d24] = 0x2483e, [0x0d25] = 0x24846,
  [0x0d26] = 0x249ed, [0x0d27] = 0x2f930, [0x0d28] = 0x3f12, [0x0d29] = 0x3f10,
  [0x0d2a] = 0x3f11, [0x0d2b] = 0x24b6c, [0x0d2c] = 0x3f32, [0x0d2d] = 0x3f34,
  [0x0d2e] = 0x3f37, [0x0d2f] = 0x3f33, [0x0d30] = 0x3f36, [0x0d31] = 0x3f35,
  [0x0d32] = 0x3f65, [0x0d33] = 0x24cac, [0x0d34] = 0x757c, [0x0d35] = 0x757b,
  [0x0d36] = 0x24dc9, [0x0d37] = 0x7612, [0x0d38] = 0x3fb0, [0x0d39] = 0x24e12,
  [0x0d3a] = 0x3faf, [0x0d3b] = 0x3faa, [0x0d3c] = 0x24dfd, [0x0d3d] = 0x24e01,
  [0x0d3e] = 0x24e11, [0x0d3f] = 0x3fab, [0x0d40] = 0x24e89, [0x0d41] = 0x24e05,
  [0x0d42] = 0x24dfe, [0x0d43] = 0x24e0b, [0x0d44] = 0x24e20, [0x0d45] = 0x24e04,
  [0x0d46] = 0x24f88, [0x0d47] = 0x3ff3, [0x0d48] = 0x25002, [0x0d49] = 0x3ff4,
  [0x0d4a] = 0x25003, [0x0d4b] = 0x76b6, [0x0d4c] = 0x76e0, [0x0d4d] = 0x4008,
  [0x0d4e] = 0x404e, [0x0d4f] = 0x4055, [0x0d50] = 0x404b, [0x0d51] = 0x25229,
  [0x0d52] = 0x2523b, [0x0d53] = 0x404d, [0x0d54] = 0x7773, [0x0d55] = 0x25206,
  [0x0d56] = 0x4052, [0x0d57] = 0x7772, [0x0d58] = 0x404c, [0x0d59] = 0x7770,
  [0x0d5a] = 0x4050, [0x0d5b] = 0x4053, [0x0d5c] = 0x25203, [0x0d5d] = 0x4051,
  [0x0d5e] = 0x25378, [0x0d5f] = 0x25379, [0x0d60] = 0x2537d, [0x0d61] = 0x4089,
  [0x0d62] = 0x2537f, [0x0d63] = 0x4095, [0x0d64] = 0x40cc, [0x0d65] = 0x40c8,
  [0x0d66] = 0x40ce, [0x0d67] = 0x25532, [0x0d68] = 0x40ca, [0x0d69] = 0x25500,
  [0x0d6a] = 0x789d, [0x0d6b] = 0x25522, [0x0d6c] = 0x40cd, [0x0d6d] = 0x256a9,
  [0x0d6e] = 0x415c, [0x0d6f] = 0x4167, [0x0d70] = 0x4169, [0x0d71] = 0x4165,
  [0x0d72] = 0x4162, [0x0d73] = 0x25804, [0x0d74] = 0x7a27, [0x0d75] = 0x7a35,
  [0x0d76] = 0x2f959, [0x0d77] = 0x41aa, [0x0d78] = 0x259b8, [0x0d79] = 0x25a9b,
  [0x0d7a] = 0x41d2, [0x0d7b] = 0x7ba2, [0x0d7c] = 0x4203, [0x0d7d] = 0x420c,
  [0x0d7e] = 0x25b92, [0x0d7f] = 0x4209, [0x0d80] = 0x4206, [0x0d81] = 0x4205,
  [0x0d82] = 0x7b89, [0x0d83] = 0x25b8f, [0x0d84] = 0x420b, [0x0d85] = 0x4208,
  [0x0d86] = 0x25b98, [0x0d87] = 0x7ba5, [0x0d88] = 0x25bae, [0x0d89] = 0x428e,
  [0x0d8a] = 0x25e9d, [0x0d8b] = 0x7cb6, [0x0d8c] = 0x42e8, [0x0d8d] = 0x26049,
  [0x0d8e] = 0x26050, [0x0d8f] = 0x26046, [0x0d90] = 0x42ea, [0x0d91] = 0x2604e,
  [0x0d92] = 0x7da5, [0x0d93] = 0x7dc3, [0x0d94] = 0x26055, [0x0d95] = 0x42e9,
  [0x0d96] = 0x42eb, [0x0d97] = 0x26064, [0x0d98] = 0x42f0, [0x0d99] = 0x26238,
  [0x0d9a] = 0x434b, [0x0d9b] = 0x7fab, [0x0d9c] = 0x4373, [0x0d9d] = 0x4375,
  [0x0d9e] = 0x4392, [0x0d9f] = 0x4391, [0x0da0] = 0x4393, [0x0da1] = 0x8025,
  [0x0da2] = 0x43a7, [0x0da3] = 0x43a6, [0x0da4] = 0x43a8, [0x0da5] = 0x43aa,
  [0x0da6] = 0x264fe, [0x0da7] = 0x43a9, [0x0da8] = 0x8059, [0x0da9] = 0x43bb,
  [0x0daa] = 0x43bc, [0x0dab] = 0x43ba, [0x0dac] = 0x43bd, [0x0dad] = 0x4427,
  [0x0dae] = 0x8185, [0x0daf] = 0x267a6, [0x0db0] = 0x4424, [0x0db1] = 0x441e,
  [0x0db2] = 0x441f, [0x0db3] = 0x441d, [0x0db4] = 0x4420, [0x0db5] = 0x4423,
  [0x0db6] = 0x4429, [0x0db7] = 0x4422, [0x0db8] = 0x2679c, [0x0db9] = 0x26799,
  [0x0dba] = 0x441c, [0x0dbb] = 0x818e, [0x0dbc] = 0x4428, [0x0dbd] = 0x267b5,
  [0x0dbe] = 0x267a3, [0x0dbf] = 0x267a0, [0x0dc0] = 0x267a7, [0x0dc1] = 0x2679b,
  [0x0dc2] = 0x269df, [0x0dc3] = 0x269e1, [0x0dc4] = 0x4471, [0x0dc5] = 0x4473,
  [0x0dc6] = 0x4472, [0x0dc7] = 0x26a4d, [0x0dc8] = 0x4502, [0x0dc9] = 0x26df3,
  [0x0dca] = 0x26d6f, [0x0dcb] = 0x26d69, [0x0dcc] = 0x44fe, [0x0dcd] = 0x84be,
  [0x0dce] = 0x26d6b, [0x0dcf] = 0x26d78, [0x0dd0] = 0x26d87, [0x0dd1] = 0x4508,
  [0x0dd2] = 0x26d85, [0x0dd3] = 0x26d82, [0x0dd4] = 0x26d90, [0x0dd5] = 0x4507,
  [0x0dd6] = 0x4504, [0x0dd7] = 0x26d80, [0x0dd8] = 0x26d67, [0x0dd9] = 0x4500,
  [0x0dda] = 0x44fc, [0x0ddb] = 0x26d61, [0x0ddc] = 0x4544, [0x0ddd] = 0x26d93,
  [0x0dde] = 0x44f1, [0x0ddf] = 0x26df2, [0x0de0] = 0x26d86, [0x0de1] = 0x84a6,
  [0x0de2] = 0x4506, [0x0de3] = 0x271cd, [0x0de4] = 0x271c4, [0x0de5] = 0x45c6,
  [0x0de6] = 0x45c3, [0x0de7] = 0x45c1, [0x0de8] = 0x45c2, [0x0de9] = 0x2730f,
  [0x0dea] = 0x45c4, [0x0deb] = 0x45c7, [0x0dec] = 0x2730d, [0x0ded] = 0x45bf,
  [0x0dee] = 0x45d2, [0x0def] = 0x27373, [0x0df0] = 0x45ca, [0x0df1] = 0x2f9ba,
  [0x0df2] = 0x2730a, [0x0df3] = 0x872f, [0x0df4] = 0x27308, [0x0df5] = 0x27313,
  [0x0df6] = 0x4613, [0x0df7] = 0x275f8, [0x0df8] = 0x4630, [0x0df9] = 0x463e,
  [0x0dfa] = 0x4639, [0x0dfb] = 0x276ab, [0x0dfc] = 0x276a8, [0x0dfd] = 0x463c,
  [0x0dfe] = 0x463f, [0x0dff] = 0x276a5, [0x0e00] = 0x4634, [0x0e01] = 0x463d,
  [0x0e02] = 0x276c3, [0x0e03] = 0x276a4, [0x0e04] = 0x4638, [0x0e05] = 0x276d4,
  [0x0e06] = 0x276ba, [0x0e07] = 0x277f1, [0x0e08] = 0x89a0, [0x0e09] = 0x4682,
  [0x0e0a] = 0x2783f, [0x0e0b] = 0x4683, [0x0e0c] = 0x278ea, [0x0e0d] = 0x469b,
  [0x0e0e] = 0x279f7, [0x0e0f] = 0x46e0, [0x0e10] = 0x46dd, [0x0e11] = 0x279fa,
  [0x0e12] = 0x279f5, [0x0e13] = 0x46de, [0x0e14] = 0x279fe, [0x0e15] = 0x46e3,
  [0x0e16] = 0x46e5, [0x0e17] = 0x27a00, [0x0e18] = 0x8a97, [0x0e19] = 0x46e2,
  [0x0e1a] = 0x27a02, [0x0e1b] = 0x279fb, [0x0e1c] = 0x27a07, [0x0e1d] = 0x46df,
  [0x0e1e] = 0x27a1a, [0x0e1f] = 0x472e, [0x0e20] = 0x27bb8, [0x0e21] = 0x4737,
  [0x0e22] = 0x4738, [0x0e23] = 0x27be9, [0x0e24] = 0x27beb, [0x0e25] = 0x27c50,
  [0x0e26] = 0x27c4f, [0x0e27] = 0x8c8b, [0x0e28] = 0x27d86, [0x0e29] = 0x3562,
  [0x0e2a] = 0x27d8e, [0x0e2b] = 0x4794, [0x0e2c] = 0x4793, [0x0e2d] = 0x47ab,
  [0x0e2e] = 0x47ad, [0x0e2f] = 0x27ef0, [0x0e30] = 0x27efb, [0x0e31] = 0x47f5,
  [0x0e32] = 0x47f7, [0x0e33] = 0x28042, [0x0e34] = 0x47f6, [0x0e35] = 0x47f8,
  [0x0e36] = 0x28085, [0x0e37] = 0x47fb, [0x0e38] = 0x47f9, [0x0e39] = 0x4858,
  [0x0e3a] = 0x485a, [0x0e3b] = 0x28305, [0x0e3c] = 0x4859, [0x0e3d] = 0x8f0f,
  [0x0e3e] = 0x4885, [0x0e3f] = 0x48a4, [0x0e40] = 0x48d8, [0x0e41] = 0x48d9,
  [0x0e42] = 0x2874b, [0x0e43] = 0x48dd, [0x0e44] = 0x48c8, [0x0e45] = 0x28763,
  [0x0e46] = 0x48fa, [0x0e47] = 0x48fb, [0x0e48] = 0x9275, [0x0e49] = 0x4927,
  [0x0e4a] = 0x929f, [0x0e4b] = 0x492a, [0x0e4c] = 0x4925, [0x0e4d] = 0x28999,
  [0x0e4e] = 0x4928, [0x0e4f] = 0x28c81, [0x0e50] = 0x28d17, [0x0e51] = 0x28d10,
  [0x0e52] = 0x28d12, [0x0e53] = 0x95a6, [0x0e54] = 0x4995, [0x0e55] = 0x969a,
  [0x0e56] = 0x28efa, [0x0e57] = 0x28ef3, [0x0e58] = 0x28ef2, [0x0e59] = 0x28ef5,
  [0x0e5a] = 0x28ef6, [0x0e5b] = 0x28fbb, [0x0e5c] = 0x28fc2, [0x0e5d] = 0x290a7,
  [0x0e5e] = 0x4a0f, [0x0e5f] = 0x4a11, [0x0e60] = 0x290a8, [0x0e61] = 0x290a3,
  [0x0e62] = 0x4a10, [0x0e63] = 0x290aa, [0x0e64] = 0x4a15, [0x0e65] = 0x4a13,
  [0x0e66] = 0x9757, [0x0e67] = 0x291dc, [0x0e68] = 0x4a47, [0x0e69] = 0x4a46,
  [0x0e6a] = 0x2920f, [0x0e6b] = 0x4a59, [0x0e6c] = 0x4a5b, [0x0e6d] = 0x29279,
  [0x0e6e] = 0x4a5e, [0x0e6f] = 0x2927f, [0x0e70] = 0x29285, [0x0e71] = 0x4a5a,
  [0x0e72] = 0x4a91, [0x0e73] = 0x4a92, [0x0e74] = 0x4a90, [0x0e75] = 0x4a93,
  [0x0e76] = 0x2941b, [0x0e77] = 0x97f7, [0x0e78] = 0x4abe, [0x0e79] = 0x29477,
  [0x0e7a] = 0x29476, [0x0e7b] = 0x29498, [0x0e7c] = 0x4abc, [0x0e7d] = 0x4abb,
  [0x0e7e] = 0x4ab7, [0x0e7f] = 0x4ab9, [0x0e80] = 0x2947a, [0x0e81] = 0x4b01,
  [0x0e82] = 0x4afe, [0x0e83] = 0x295bc, [0x0e84] = 0x295ba, [0x0e85] = 0x4b02,
  [0x0e86] = 0x295b6, [0x0e87] = 0x4aff, [0x0e88] = 0x98b0, [0x0e89] = 0x295b4,
  [0x0e8a] = 0x4b00, [0x0e8b] = 0x296cf, [0x0e8c] = 0x4b37, [0x0e8d] = 0x4b3a,
  [0x0e8e] = 0x4b6f, [0x0e8f] = 0x4b77, [0x0e90] = 0x4b79, [0x0e91] = 0x99c6,
  [0x0e92] = 0x2987b, [0x0e93] = 0x4bc8, [0x0e94] = 0x29a1c, [0x0e95] = 0x29a21,
  [0x0e96] = 0x29a1d, [0x0e97] = 0x29ac0, [0x0e98] = 0x4bf2, [0x0e99] = 0x29aff,
  [0x0e9a] = 0x4bf1, [0x0e9b] = 0x4bf0, [0x0e9c] = 0x9b62, [0x0e9d] = 0x29d56,
  [0x0e9e] = 0x4c34, [0x0e9f] = 0x29fa1, [0x0ea0] = 0x29fa2, [0x0ea1] = 0x29fa6,
  [0x0ea2] = 0x2a256, [0x0ea3] = 0x2a257, [0x0ea4] = 0x4d2c, [0x0ea5] = 0x4d2d,
  [0x0ea6] = 0x2a301, [0x0ea7] = 0x2a3ed, [0x0ea8] = 0x2fa1c, [0x0ea9] = 0x2a5f2,
  [0x0eaa] = 0x50fa, [0x0eab] = 0x203b8, [0x0eac] = 0x3491, [0x0ead] = 0x203c0,
  [0x0eae] = 0x3494, [0x0eaf] = 0x203b7, [0x0eb0] = 0x203e1, [0x0eb1] = 0x34c4,
  [0x0eb2] = 0x350a, [0x0eb3] = 0x207e8, [0x0eb4] = 0x5285, [0x0eb5] = 0x208b2,
  [0x0eb6] = 0x3552, [0x0eb7] = 0x20aae, [0x0eb8] = 0x3559, [0x0eb9] = 0x366f,
  [0x0eba] = 0x20f7e, [0x0ebb] = 0x35f2, [0x0ebc] = 0x35f4, [0x0ebd] = 0x5643,
  [0x0ebe] = 0x20f76, [0x0ebf] = 0x35f1, [0x0ec0] = 0x563c, [0x0ec1] = 0x20f96,
  [0x0ec2] = 0x366a, [0x0ec3] = 0x21420, [0x0ec4] = 0x21421, [0x0ec5] = 0x21423,
  [0x0ec6] = 0x21429, [0x0ec7] = 0x21677, [0x0ec8] = 0x21851, [0x0ec9] = 0x3724,
  [0x0eca] = 0x21856, [0x0ecb] = 0x3723, [0x0ecc] = 0x21888, [0x0ecd] = 0x21859,
  [0x0ece] = 0x21855, [0x0ecf] = 0x3727, [0x0ed0] = 0x21997, [0x0ed1] = 0x21998,
  [0x0ed2] = 0x376d, [0x0ed3] = 0x5bed, [0x0ed4] = 0x376e, [0x0ed5] = 0x376f,
  [0x0ed6] = 0x21aa1, [0x0ed7] = 0x21aa3, [0x0ed8] = 0x5c35, [0x0ed9] = 0x379f,
  [0x0eda] = 0x380a, [0x0edb] = 0x3806, [0x0edc] = 0x380e, [0x0edd] = 0x21f0a,
  [0x0ede] = 0x380d, [0x0edf] = 0x3805, [0x0ee0] = 0x21f3d, [0x0ee1] = 0x21f0c,
  [0x0ee2] = 0x380b, [0x0ee3] = 0x3810, [0x0ee4] = 0x382e, [0x0ee5] = 0x2213d,
  [0x0ee6] = 0x22139, [0x0ee7] = 0x222b0, [0x0ee8] = 0x222b2, [0x0ee9] = 0x222ae,
  [0x0eea] = 0x3896, [0x0eeb] = 0x3897, [0x0eec] = 0x38c4, [0x0eed] = 0x5f47,
  [0x0eee] = 0x38c5, [0x0eef] = 0x22420, [0x0ef0] = 0x38d2, [0x0ef1] = 0x2256e,
  [0x0ef2] = 0x2256d, [0x0ef3] = 0x2256a, [0x0ef4] = 0x3981, [0x0ef5] = 0x22832,
  [0x0ef6] = 0x398e, [0x0ef7] = 0x3990, [0x0ef8] = 0x398f, [0x0ef9] = 0x22839,
  [0x0efa] = 0x3991, [0x0efb] = 0x3995, [0x0efc] = 0x3993, [0x0efd] = 0x227a3,
  [0x0efe] = 0x616d, [0x0eff] = 0x2287f, [0x0f00] = 0x22835, [0x0f01] = 0x2283d,
  [0x0f02] = 0x229f4, [0x0f03] = 0x229f5, [0x0f04] = 0x3a3b, [0x0f05] = 0x3a48,
  [0x0f06] = 0x22d48, [0x0f07] = 0x22d28, [0x0f08] = 0x3a46, [0x0f09] = 0x3a47,
  [0x0f0a] = 0x22d27, [0x0f0b] = 0x22d32, [0x0f0c] = 0x22d2c, [0x0f0d] = 0x3a4c,
  [0x0f0e] = 0x22d2e, [0x0f0f] = 0x3a4a, [0x0f10] = 0x3a50, [0x0f11] = 0x3a43,
  [0x0f12] = 0x22d23, [0x0f13] = 0x22d31, [0x0f14] = 0x2f8c5, [0x0f15] = 0x3a49,
  [0x0f16] = 0x3aa6, [0x0f17] = 0x3aa5, [0x0f18] = 0x3aa4, [0x0f19] = 0x22fc9,
  [0x0f1a] = 0x23089, [0x0f1b] = 0x3ab9, [0x0f1c] = 0x230bb, [0x0f1d] = 0x3ace,
  [0x0f1e] = 0x3acf, [0x0f1f] = 0x3b13, [0x0f20] = 0x23397, [0x0f21] = 0x23393,
  [0x0f22] = 0x23628, [0x0f23] = 0x3bc6, [0x0f24] = 0x3bc5, [0x0f25] = 0x3bca,
  [0x0f26] = 0x3bd9, [0x0f27] = 0x3bc1, [0x0f28] = 0x23618, [0x0f29] = 0x2363b,
  [0x0f2a] = 0x69f5, [0x0f2b] = 0x23627, [0x0f2c] = 0x23624, [0x0f2d] = 0x2361b,
  [0x0f2e] = 0x23631, [0x0f2f] = 0x3bcb, [0x0f30] = 0x23626, [0x0f31] = 0x236a3,
  [0x0f32] = 0x2363f, [0x0f33] = 0x23622, [0x0f34] = 0x23619, [0x0f35] = 0x6a03,
  [0x0f36] = 0x23641, [0x0f37] = 0x2362b, [0x0f38] = 0x6a65, [0x0f39] = 0x3c42,
  [0x0f3a] = 0x2390c, [0x0f3b] = 0x6b75, [0x0f3c] = 0x3c74, [0x0f3d] = 0x3c73,
  [0x0f3e] = 0x23a4e, [0x0f3f] = 0x23ab9, [0x0f40] = 0x23afa, [0x0f41] = 0x23be9,
  [0x0f42] = 0x23be8, [0x0f43] = 0x23be4, [0x0f44] = 0x6f8a, [0x0f45] = 0x6f56,
  [0x0f46] = 0x3d52, [0x0f47] = 0x23f5f, [0x0f48] = 0x23f5d, [0x0f49] = 0x23f52,
  [0x0f4a] = 0x3d50, [0x0f4b] = 0x23f74, [0x0f4c] = 0x3d4b, [0x0f4d] = 0x23f46,
  [0x0f4e] = 0x3d4c, [0x0f4f] = 0x3d6d, [0x0f50] = 0x23faa, [0x0f51] = 0x6f98,
  [0x0f52] = 0x23f4a, [0x0f53] = 0x23f59, [0x0f54] = 0x23f4b, [0x0f55] = 0x3d4f,
  [0x0f56] = 0x6f68, [0x0f57] = 0x3df0, [0x0f58] = 0x24350, [0x0f59] = 0x7234,
  [0x0f5a] = 0x7245, [0x0f5b] = 0x3e4d, [0x0f5c] = 0x3e4c, [0x0f5d] = 0x246d0,
  [0x0f5e] = 0x3e4f, [0x0f5f] = 0x3e4e, [0x0f60] = 0x3e50, [0x0f61] = 0x246cc,
  [0x0f62] = 0x735c, [0x0f63] = 0x3e96, [0x0f64] = 0x7356, [0x0f65] = 0x24864,
  [0x0f66] = 0x24865, [0x0f67] = 0x3e97, [0x0f68] = 0x3e95, [0x0f69] = 0x3e98,
  [0x0f6a] = 0x2485c, [0x0f6b] = 0x24a15, [0x0f6c] = 0x3eec, [0x0f6d] = 0x3eeb,
  [0x0f6e] = 0x3f13, [0x0f6f] = 0x3f14, [0x0f70] = 0x3f38, [0x0f71] = 0x3f3a,
  [0x0f72] = 0x3f39, [0x0f73] = 0x24b79, [0x0f74] = 0x3f68, [0x0f75] = 0x3f67,
  [0x0f76] = 0x24cc5, [0x0f77] = 0x24cb8, [0x0f78] = 0x24e2c, [0x0f79] = 0x3fbe,
  [0x0f7a] = 0x3fbc, [0x0f7b] = 0x24e37, [0x0f7c] = 0x24e35, [0x0f7d] = 0x24e31,
  [0x0f7e] = 0x3fbb, [0x0f7f] = 0x24e2f, [0x0f80] = 0x3fba, [0x0f81] = 0x24e2b,
  [0x0f82] = 0x3fb9, [0x0f83] = 0x3fb7, [0x0f84] = 0x24e2d, [0x0f85] = 0x24e2a,
  [0x0f86] = 0x3fc1, [0x0f87] = 0x24f95, [0x0f88] = 0x3ff7, [0x0f89] = 0x2523e,
  [0x0f8a] = 0x4060, [0x0f8b] = 0x25247, [0x0f8c] = 0x25245, [0x0f8d] = 0x4059,
  [0x0f8e] = 0x405c, [0x0f8f] = 0x405a, [0x0f90] = 0x4058, [0x0f91] = 0x25252,
  [0x0f92] = 0x405b, [0x0f93] = 0x25270, [0x0f94] = 0x25250, [0x0f95] = 0x25258,
  [0x0f96] = 0x25251, [0x0f97] = 0x2523d, [0x0f98] = 0x405d, [0x0f99] = 0x25241,
  [0x0f9a] = 0x2520c, [0x0f9b] = 0x2523c, [0x0f9c] = 0x25386, [0x0f9d] = 0x25383,
  [0x0f9e] = 0x25389, [0x0f9f] = 0x253f3, [0x0fa0] = 0x40d4, [0x0fa1] = 0x40d3,
  [0x0fa2] = 0x78bf, [0x0fa3] = 0x40d2, [0x0fa4] = 0x78bd, [0x0fa5] = 0x2552d,
  [0x0fa6] = 0x40d7, [0x0fa7] = 0x40d1, [0x0fa8] = 0x78e4, [0x0fa9] = 0x40d5,
  [0x0faa] = 0x256c5, [0x0fab] = 0x2f955, [0x0fac] = 0x416d, [0x0fad] = 0x416f,
  [0x0fae] = 0x7a34, [0x0faf] = 0x4175, [0x0fb0] = 0x416c, [0x0fb1] = 0x4174,
  [0x0fb2] = 0x25843, [0x0fb3] = 0x4173, [0x0fb4] = 0x25837, [0x0fb5] = 0x25845,
  [0x0fb6] = 0x7a36, [0x0fb7] = 0x41ac, [0x0fb8] = 0x259e5, [0x0fb9] = 0x25aa6,
  [0x0fba] = 0x25bdb, [0x0fbb] = 0x4210, [0x0fbc] = 0x25bda, [0x0fbd] = 0x25be6,
  [0x0fbe] = 0x7bba, [0x0fbf] = 0x7bbc, [0x0fc0] = 0x420f, [0x0fc1] = 0x7bc8,
  [0x0fc2] = 0x4223, [0x0fc3] = 0x7bc3, [0x0fc4] = 0x421d, [0x0fc5] = 0x7bb6,
  [0x0fc6] = 0x420e, [0x0fc7] = 0x25bf8, [0x0fc8] = 0x25be9, [0x0fc9] = 0x4215,
  [0x0fca] = 0x7bc2, [0x0fcb] = 0x4213, [0x0fcc] = 0x25be8, [0x0fcd] = 0x25bf6,
  [0x0fce] = 0x421b, [0x0fcf] = 0x7bc5, [0x0fd0] = 0x4222, [0x0fd1] = 0x4226,
  [0x0fd2] = 0x25be7, [0x0fd3] = 0x7bbd, [0x0fd4] = 0x7bb0, [0x0fd5] = 0x4221,
  [0x0fd6] = 0x421c, [0x0fd7] = 0x4217, [0x0fd8] = 0x25bd5, [0x0fd9] = 0x421a,
  [0x0fda] = 0x7bbb, [0x0fdb] = 0x25ed3, [0x0fdc] = 0x25ec7, [0x0fdd] = 0x25ed1,
  [0x0fde] = 0x4299, [0x0fdf] = 0x25ec3, [0x0fe0] = 0x4297, [0x0fe1] = 0x26080,
  [0x0fe2] = 0x26098, [0x0fe3] = 0x42fd, [0x0fe4] = 0x26084, [0x0fe5] = 0x42f6,
  [0x0fe6] = 0x42fe, [0x0fe7] = 0x42f5, [0x0fe8] = 0x42ff, [0x0fe9] = 0x42f7,
  [0x0fea] = 0x26097, [0x0feb] = 0x26083, [0x0fec] = 0x26081, [0x0fed] = 0x4301,
  [0x0fee] = 0x7e04, [0x0fef] = 0x2608c, [0x0ff0] = 0x26242, [0x0ff1] = 0x263ab,
  [0x0ff2] = 0x4377, [0x0ff3] = 0x263a3, [0x0ff4] = 0x263a6, [0x0ff5] = 0x2645c,
  [0x0ff6] = 0x26469, [0x0ff7] = 0x26467, [0x0ff8] = 0x43ab, [0x0ff9] = 0x2658b,
  [0x0ffa] = 0x265a8, [0x0ffb] = 0x43c0, [0x0ffc] = 0x267d8, [0x0ffd] = 0x4431,
  [0x0ffe] = 0x442e, [0x0fff] = 0x2f989, [0x1000] = 0x267dc, [0x1001] = 0x442c,
  [0x1002] = 0x267e0, [0x1003] = 0x267e5, [0x1004] = 0x4432, [0x1005] = 0x442f,
  [0x1006] = 0x442b, [0x1007] = 0x442d, [0x1008] = 0x4433, [0x1009] = 0x26918,
  [0x100a] = 0x26919, [0x100b] = 0x81f1, [0x100c] = 0x4457, [0x100d] = 0x445c,
  [0x100e] = 0x447b, [0x100f] = 0x26a5f, [0x1010] = 0x26a5e, [0x1011] = 0x447a,
  [0x1012] = 0x26e02, [0x1013] = 0x8522, [0x1014] = 0x4513, [0x1015] = 0x451e,
  [0x1016] = 0x4517, [0x1017] = 0x4520, [0x1018] = 0x452a, [0x1019] = 0x4511,
  [0x101a] = 0x4515, [0x101b] = 0x450f, [0x101c] = 0x4518, [0x101d] = 0x8538,
  [0x101e] = 0x26e12, [0x101f] = 0x26e36, [0x1020] = 0x452c, [0x1021] = 0x8532,
  [0x1022] = 0x2f9a8, [0x1023] = 0x8510, [0x1024] = 0x26e23, [0x1025] = 0x26e03,
  [0x1026] = 0x451c, [0x1027] = 0x26e00, [0x1028] = 0x4529, [0x1029] = 0x26e46,
  [0x102a] = 0x26e61, [0x102b] = 0x4512, [0x102c] = 0x854f, [0x102d] = 0x4597,
  [0x102e] = 0x27384, [0x102f] = 0x8772, [0x1030] = 0x2736b, [0x1031] = 0x27362,
  [0x1032] = 0x27356, [0x1033] = 0x2736a, [0x1034] = 0x27352, [0x1035] = 0x27355,
  [0x1036] = 0x45d4, [0x1037] = 0x45d0, [0x1038] = 0x877c, [0x1039] = 0x27361,
  [0x103a] = 0x2f9bb, [0x103b] = 0x27358, [0x103c] = 0x27377, [0x103d] = 0x45d3,
  [0x103e] = 0x4614, [0x103f] = 0x276d7, [0x1040] = 0x4646, [0x1041] = 0x4645,
  [0x1042] = 0x276de, [0x1043] = 0x4643, [0x1044] = 0x276df, [0x1045] = 0x890d,
  [0x1046] = 0x4644, [0x1047] = 0x4648, [0x1048] = 0x276d1, [0x1049] = 0x4647,
  [0x104a] = 0x276e2, [0x104b] = 0x276e1, [0x104c] = 0x276dd, [0x104d] = 0x8908,
  [0x104e] = 0x4649, [0x104f] = 0x4685, [0x1050] = 0x2784b, [0x1051] = 0x4684,
  [0x1052] = 0x2784e, [0x1053] = 0x469d, [0x1054] = 0x278fc, [0x1055] = 0x469e,
  [0x1056] = 0x46a0, [0x1057] = 0x278fa, [0x1058] = 0x469c, [0x1059] = 0x278fb,
  [0x105a] = 0x469f, [0x105b] = 0x278fe, [0x105c] = 0x46f7, [0x105d] = 0x46ea,
  [0x105e] = 0x27a31, [0x105f] = 0x46ef, [0x1060] = 0x46e9, [0x1061] = 0x46f3,
  [0x1062] = 0x46f0, [0x1063] = 0x46eb, [0x1064] = 0x27a38, [0x1065] = 0x46ec,
  [0x1066] = 0x46f2, [0x1067] = 0x46f5, [0x1068] = 0x46ee, [0x1069] = 0x27a3a,
  [0x106a] = 0x27bbb, [0x106b] = 0x473a, [0x106c] = 0x474b, [0x106d] = 0x27c59,
  [0x106e] = 0x474a, [0x106f] = 0x474c, [0x1070] = 0x27cdb, [0x1071] = 0x27cdf,
  [0x1072] = 0x27ce2, [0x1073] = 0x4779, [0x1074] = 0x477b, [0x1075] = 0x4778,
  [0x1076] = 0x27d9e, [0x1077] = 0x27da1, [0x1078] = 0x47b5, [0x1079] = 0x27f10,
  [0x107a] = 0x47b4, [0x107b] = 0x47b7, [0x107c] = 0x8d9e, [0x107d] = 0x4809,
  [0x107e] = 0x47fe, [0x107f] = 0x4808, [0x1080] = 0x4807, [0x1081] = 0x28076,
  [0x1082] = 0x2807f, [0x1083] = 0x2807d, [0x1084] = 0x4806, [0x1085] = 0x4804,
  [0x1086] = 0x4805, [0x1087] = 0x47ff, [0x1088] = 0x480b, [0x1089] = 0x28082,
  [0x108a] = 0x28257, [0x108b] = 0x483b, [0x108c] = 0x485d, [0x108d] = 0x485c,
  [0x108e] = 0x485f, [0x108f] = 0x485e, [0x1090] = 0x8f28, [0x1091] = 0x2832e,
  [0x1092] = 0x8f21, [0x1093] = 0x4883, [0x1094] = 0x2858b, [0x1095] = 0x2858d,
  [0x1096] = 0x48a7, [0x1097] = 0x9066, [0x1098] = 0x906c, [0x1099] = 0x28590,
  [0x109a] = 0x48a8, [0x109b] = 0x286ec, [0x109c] = 0x2876f, [0x109d] = 0x2876b,
  [0x109e] = 0x28771, [0x109f] = 0x28778, [0x10a0] = 0x90f6, [0x10a1] = 0x48e0,
  [0x10a2] = 0x48df, [0x10a3] = 0x48fe, [0x10a4] = 0x48fc, [0x10a5] = 0x48ff,
  [0x10a6] = 0x48fd, [0x10a7] = 0x289ad, [0x10a8] = 0x492c, [0x10a9] = 0x92ec,
  [0x10aa] = 0x92ba, [0x10ab] = 0x92e3, [0x10ac] = 0x92bd, [0x10ad] = 0x499d,
  [0x10ae] = 0x28d2f, [0x10af] = 0x95b4, [0x10b0] = 0x28f78, [0x10b1] = 0x4a40,
  [0x10b2] = 0x291f8, [0x10b3] = 0x29219, [0x10b4] = 0x4a5f, [0x10b5] = 0x2929b,
  [0x10b6] = 0x29294, [0x10b7] = 0x29297, [0x10b8] = 0x29299, [0x10b9] = 0x293a6,
  [0x10ba] = 0x293a4, [0x10bb] = 0x293a7, [0x10bc] = 0x97d1, [0x10bd] = 0x29495,
  [0x10be] = 0x4ac0, [0x10bf] = 0x9823, [0x10c0] = 0x29490, [0x10c1] = 0x4ac1,
  [0x10c2] = 0x4ac6, [0x10c3] = 0x2949b, [0x10c4] = 0x295c4, [0x10c5] = 0x4b04,
  [0x10c6] = 0x4b05, [0x10c7] = 0x295c6, [0x10c8] = 0x990b, [0x10c9] = 0x4b3e,
  [0x10ca] = 0x4b3d, [0x10cb] = 0x4b40, [0x10cc] = 0x4b3f, [0x10cd] = 0x296e3,
  [0x10ce] = 0x4b42, [0x10cf] = 0x296df, [0x10d0] = 0x296dd, [0x10d1] = 0x296e7,
  [0x10d2] = 0x4b84, [0x10d3] = 0x4b82, [0x10d4] = 0x4b7f, [0x10d5] = 0x4b85,
  [0x10d6] = 0x29a2d, [0x10d7] = 0x29a2c, [0x10d8] = 0x4bcc, [0x10d9] = 0x9ab2,
  [0x10da] = 0x4bcb, [0x10db] = 0x4bcd, [0x10dc] = 0x29a34, [0x10dd] = 0x29a38,
  [0x10de] = 0x9adb, [0x10df] = 0x29b1d, [0x10e0] = 0x4bf5, [0x10e1] = 0x29b1a,
  [0x10e2] = 0x29b1b, [0x10e3] = 0x29b14, [0x10e4] = 0x9af0, [0x10e5] = 0x29b17,
  [0x10e6] = 0x29c21, [0x10e7] = 0x4c20, [0x10e8] = 0x4c21, [0x10e9] = 0x29caa,
  [0x10ea] = 0x29ca1, [0x10eb] = 0x4c37, [0x10ec] = 0x4c3e, [0x10ed] = 0x9b73,
  [0x10ee] = 0x4c3d, [0x10ef] = 0x9b6e, [0x10f0] = 0x29d63, [0x10f1] = 0x29d79,
  [0x10f2] = 0x29d60, [0x10f3] = 0x9b65, [0x10f4] = 0x29d62, [0x10f5] = 0x4c3c,
  [0x10f6] = 0x29d61, [0x10f7] = 0x4c38, [0x10f8] = 0x9b6a, [0x10f9] = 0x29d70,
  [0x10fa] = 0x9b6d, [0x10fb] = 0x29d6a, [0x10fc] = 0x4c3b, [0x10fd] = 0x29fc8,
  [0x10fe] = 0x4cb0, [0x10ff] = 0x29fc5, [0x1100] = 0x29fbe, [0x1101] = 0x29fc2,
  [0x1102] = 0x4cad, [0x1103] = 0x4cb2, [0x1104] = 0x4cb8, [0x1105] = 0x9d0b,
  [0x1106] = 0x29fc7, [0x1107] = 0x4caf, [0x1108] = 0x29fb0, [0x1109] = 0x29fca,
  [0x110a] = 0x4d1a, [0x110b] = 0x9e76, [0x110c] = 0x4d20, [0x110d] = 0x4d21,
  [0x110e] = 0x4d30, [0x110f] = 0x9ea8, [0x1110] = 0x4d2f, [0x1111] = 0x2a30d,
  [0x1112] = 0x2a307, [0x1113] = 0x2a396, [0x1114] = 0x2a3ef, [0x1115] = 0x4d5f,
  [0x1116] = 0x4d60, [0x1117] = 0x2a41e, [0x1118] = 0x2a41d, [0x1119] = 0x9f11,
  [0x111a] = 0x2a590, [0x111b] = 0x203eb, [0x111c] = 0x348a, [0x111d] = 0x5119,
  [0x111e] = 0x349c, [0x111f] = 0x203ee, [0x1120] = 0x349a, [0x1121] = 0x203ef,
  [0x1122] = 0x203ec, [0x1123] = 0x207fa, [0x1124] = 0x350c, [0x1125] = 0x350b,
  [0x1126] = 0x350d, [0x1127] = 0x5292, [0x1128] = 0x20904, [0x1129] = 0x20fd3,
  [0x112a] = 0x35fe, [0x112b] = 0x20fd1, [0x112c] = 0x20ffa, [0x112d] = 0x35ff,
  [0x112e] = 0x35fb, [0x112f] = 0x35fc, [0x1130] = 0x3609, [0x1131] = 0x20fc8,
  [0x1132] = 0x3600, [0x1133] = 0x20fd5, [0x1134] = 0x5675, [0x1135] = 0x20fcd,
  [0x1136] = 0x20fd2, [0x1137] = 0x20ffb, [0x1138] = 0x2122a, [0x1139] = 0x21488,
  [0x113a] = 0x21461, [0x113b] = 0x3671, [0x113c] = 0x21463, [0x113d] = 0x21462,
  [0x113e] = 0x596f, [0x113f] = 0x20057, [0x1140] = 0x218c8, [0x1141] = 0x21898,
  [0x1142] = 0x372b, [0x1143] = 0x21ac1, [0x1144] = 0x21f3f, [0x1145] = 0x21f41,
  [0x1146] = 0x3814, [0x1147] = 0x3811, [0x1148] = 0x3812, [0x1149] = 0x22152,
  [0x114a] = 0x3863, [0x114b] = 0x2215e, [0x114c] = 0x386e, [0x114d] = 0x389a,
  [0x114e] = 0x222c7, [0x114f] = 0x389b, [0x1150] = 0x22370, [0x1151] = 0x22427,
  [0x1152] = 0x38c8, [0x1153] = 0x22471, [0x1154] = 0x224b0, [0x1155] = 0x38f6,
  [0x1156] = 0x22842, [0x1157] = 0x22843, [0x1158] = 0x61a5, [0x1159] = 0x398c,
  [0x115a] = 0x3997, [0x115b] = 0x39a2, [0x115c] = 0x61a0, [0x115d] = 0x22857,
  [0x115e] = 0x2289d, [0x115f] = 0x22d89, [0x1160] = 0x22d8d, [0x1161] = 0x22d8b,
  [0x1162] = 0x22d80, [0x1163] = 0x22d92, [0x1164] = 0x22d8a, [0x1165] = 0x22dc8,
  [0x1166] = 0x3a54, [0x1167] = 0x22d8f, [0x1168] = 0x22d93, [0x1169] = 0x22d91,
  [0x116a] = 0x3aa8, [0x116b] = 0x22ff2, [0x116c] = 0x22fde, [0x116d] = 0x3aa9,
  [0x116e] = 0x230c8, [0x116f] = 0x65b4, [0x1170] = 0x65d8, [0x1171] = 0x66c2,
  [0x1172] = 0x3b18, [0x1173] = 0x3b17, [0x1174] = 0x3b1d, [0x1175] = 0x232a7,
  [0x1176] = 0x232a1, [0x1177] = 0x3b31, [0x1178] = 0x2368c, [0x1179] = 0x2367f,
  [0x117a] = 0x3bd8, [0x117b] = 0x3bd5, [0x117c] = 0x2367b, [0x117d] = 0x23695,
  [0x117e] = 0x23699, [0x117f] = 0x3be1, [0x1180] = 0x2368e, [0x1181] = 0x3bd4,
  [0x1182] = 0x236da, [0x1183] = 0x2368a, [0x1184] = 0x2369c, [0x1185] = 0x3be3,
  [0x1186] = 0x2367e, [0x1187] = 0x3c44, [0x1188] = 0x3c45, [0x1189] = 0x2391a,
  [0x118a] = 0x23a55, [0x118b] = 0x3c76, [0x118c] = 0x3c75, [0x118d] = 0x23a60,
  [0x118e] = 0x6ba8, [0x118f] = 0x3c88, [0x1190] = 0x3caa, [0x1191] = 0x23bfb,
  [0x1192] = 0x3cab, [0x1193] = 0x3cac, [0x1194] = 0x23bfd, [0x1195] = 0x3d57,
  [0x1196] = 0x6f83, [0x1197] = 0x3d60, [0x1198] = 0x3d5d, [0x1199] = 0x3d6b,
  [0x119a] = 0x23fd0, [0x119b] = 0x3d63, [0x119c] = 0x3d67, [0x119d] = 0x23fc5,
  [0x119e] = 0x3d5e, [0x119f] = 0x23fd2, [0x11a0] = 0x24011, [0x11a1] = 0x6fc5,
  [0x11a2] = 0x71cd, [0x11a3] = 0x3df9, [0x11a4] = 0x3df3, [0x11a5] = 0x243ad,
  [0x11a6] = 0x243a4, [0x11a7] = 0x243a9, [0x11a8] = 0x243b0, [0x11a9] = 0x2439d,
  [0x11aa] = 0x3df6, [0x11ab] = 0x24598, [0x11ac] = 0x729c, [0x11ad] = 0x3e51,
  [0x11ae] = 0x3e53, [0x11af] = 0x3e52, [0x11b0] = 0x24892, [0x11b1] = 0x3e9b,
  [0x11b2] = 0x2488a, [0x11b3] = 0x3e9c, [0x11b4] = 0x2488d, [0x11b5] = 0x24896,
  [0x11b6] = 0x3ef7, [0x11b7] = 0x7499, [0x11b8] = 0x748f, [0x11b9] = 0x7491,
  [0x11ba] = 0x24a45, [0x11bb] = 0x24a4b, [0x11bc] = 0x24b0f, [0x11bd] = 0x24b16,
  [0x11be] = 0x3f3b, [0x11bf] = 0x24b8a, [0x11c0] = 0x3f3e, [0x11c1] = 0x3f3d,
  [0x11c2] = 0x24cc9, [0x11c3] = 0x3f69, [0x11c4] = 0x24cc8, [0x11c5] = 0x24cca,
  [0x11c6] = 0x24e64, [0x11c7] = 0x3fc3, [0x11c8] = 0x3fc4, [0x11c9] = 0x3fc7,
  [0x11ca] = 0x24e5d, [0x11cb] = 0x24e63, [0x11cc] = 0x7639, [0x11cd] = 0x3fc6,
  [0x11ce] = 0x762e, [0x11cf] = 0x3fc8, [0x11d0] = 0x24e88, [0x11d1] = 0x24f3a,
  [0x11d2] = 0x24f39, [0x11d3] = 0x769f, [0x11d4] = 0x76a0, [0x11d5] = 0x3fe6,
  [0x11d6] = 0x3ff8, [0x11d7] = 0x25017, [0x11d8] = 0x25093, [0x11d9] = 0x4007,
  [0x11da] = 0x25095, [0x11db] = 0x4064, [0x11dc] = 0x4068, [0x11dd] = 0x25276,
  [0x11de] = 0x7794, [0x11df] = 0x4065, [0x11e0] = 0x77ae, [0x11e1] = 0x25280,
  [0x11e2] = 0x2527b, [0x11e3] = 0x4069, [0x11e4] = 0x25248, [0x11e5] = 0x252a8,
  [0x11e6] = 0x25288, [0x11e7] = 0x2538b, [0x11e8] = 0x2538a, [0x11e9] = 0x2538c,
  [0x11ea] = 0x253fc, [0x11eb] = 0x40da, [0x11ec] = 0x40e0, [0x11ed] = 0x78e6,
  [0x11ee] = 0x25558, [0x11ef] = 0x40de, [0x11f0] = 0x25551, [0x11f1] = 0x25555,
  [0x11f2] = 0x25553, [0x11f3] = 0x411c, [0x11f4] = 0x411d, [0x11f5] = 0x411a,
  [0x11f6] = 0x256dc, [0x11f7] = 0x417b, [0x11f8] = 0x417a, [0x11f9] = 0x417c,
  [0x11fa] = 0x2585c, [0x11fb] = 0x4178, [0x11fc] = 0x4177, [0x11fd] = 0x25865,
  [0x11fe] = 0x2f95a, [0x11ff] = 0x41b1, [0x1200] = 0x259ee, [0x1201] = 0x41b2,
  [0x1202] = 0x41b0, [0x1203] = 0x259f0, [0x1204] = 0x7abc, [0x1205] = 0x259f2,
  [0x1206] = 0x4236, [0x1207] = 0x25c35, [0x1208] = 0x422e, [0x1209] = 0x25c22,
  [0x120a] = 0x7bd6, [0x120b] = 0x25c20, [0x120c] = 0x4234, [0x120d] = 0x25c2d,
  [0x120e] = 0x25c28, [0x120f] = 0x25c26, [0x1210] = 0x25c3c, [0x1211] = 0x422a,
  [0x1212] = 0x25c38, [0x1213] = 0x4233, [0x1214] = 0x422d, [0x1215] = 0x422f,
  [0x1216] = 0x4231, [0x1217] = 0x422b, [0x1218] = 0x4232, [0x1219] = 0x25c2a,
  [0x121a] = 0x4235, [0x121b] = 0x25c1a, [0x121c] = 0x25c30, [0x121d] = 0x25be3,
  [0x121e] = 0x25c19, [0x121f] = 0x25ee9, [0x1220] = 0x25ee7, [0x1221] = 0x7ccf,
  [0x1222] = 0x260d6, [0x1223] = 0x260c4, [0x1224] = 0x4308, [0x1225] = 0x26087,
  [0x1226] = 0x4306, [0x1227] = 0x260c7, [0x1228] = 0x260d9, [0x1229] = 0x7e18,
  [0x122a] = 0x260df, [0x122b] = 0x434d, [0x122c] = 0x4361, [0x122d] = 0x2630a,
  [0x122e] = 0x26301, [0x122f] = 0x263b1, [0x1230] = 0x4378, [0x1231] = 0x263c1,
  [0x1232] = 0x4394, [0x1233] = 0x4395, [0x1234] = 0x26476, [0x1235] = 0x2650d,
  [0x1236] = 0x2650e, [0x1237] = 0x43c1, [0x1238] = 0x43c3, [0x1239] = 0x806d,
  [0x123a] = 0x26807, [0x123b] = 0x4439, [0x123c] = 0x26809, [0x123d] = 0x26816,
  [0x123e] = 0x443a, [0x123f] = 0x443b, [0x1240] = 0x4435, [0x1241] = 0x4436,
  [0x1242] = 0x2680e, [0x1243] = 0x443c, [0x1244] = 0x8190, [0x1245] = 0x26806,
  [0x1246] = 0x2691e, [0x1247] = 0x2694a, [0x1248] = 0x4458, [0x1249] = 0x447c,
  [0x124a] = 0x447d, [0x124b] = 0x448d, [0x124c] = 0x448c, [0x124d] = 0x26ea6,
  [0x124e] = 0x26e91, [0x124f] = 0x453b, [0x1250] = 0x453f, [0x1251] = 0x26eb8,
  [0x1252] = 0x26e96, [0x1253] = 0x4532, [0x1254] = 0x452d, [0x1255] = 0x26ec5,
  [0x1256] = 0x452f, [0x1257] = 0x4539, [0x1258] = 0x452e, [0x1259] = 0x453a,
  [0x125a] = 0x26f0a, [0x125b] = 0x4536, [0x125c] = 0x4531, [0x125d] = 0x453e,
  [0x125e] = 0x4538, [0x125f] = 0x8552, [0x1260] = 0x4534, [0x1261] = 0x26eb2,
  [0x1262] = 0x4541, [0x1263] = 0x26eb7, [0x1264] = 0x26eb4, [0x1265] = 0x26e89,
  [0x1266] = 0x26e8d, [0x1267] = 0x4530, [0x1268] = 0x26e87, [0x1269] = 0x26ec2,
  [0x126a] = 0x26e9c, [0x126b] = 0x26e92, [0x126c] = 0x4543, [0x126d] = 0x26eca,
  [0x126e] = 0x8550, [0x126f] = 0x26e8a, [0x1270] = 0x271e0, [0x1271] = 0x4598,
  [0x1272] = 0x271e6, [0x1273] = 0x271e5, [0x1274] = 0x87a0, [0x1275] = 0x273a5,
  [0x1276] = 0x273b3, [0x1277] = 0x8786, [0x1278] = 0x45da, [0x1279] = 0x45d7,
  [0x127a] = 0x273b5, [0x127b] = 0x273af, [0x127c] = 0x8795, [0x127d] = 0x273b0,
  [0x127e] = 0x273cc, [0x127f] = 0x878c, [0x1280] = 0x273a2, [0x1281] = 0x273be,
  [0x1282] = 0x273c6, [0x1283] = 0x273ac, [0x1284] = 0x273ae, [0x1285] = 0x4618,
  [0x1286] = 0x8860, [0x1287] = 0x2f9c3, [0x1288] = 0x27708, [0x1289] = 0x27705,
  [0x128a] = 0x4652, [0x128b] = 0x8928, [0x128c] = 0x276fe, [0x128d] = 0x464e,
  [0x128e] = 0x8920, [0x128f] = 0x27700, [0x1290] = 0x464f, [0x1291] = 0x4650,
  [0x1292] = 0x27864, [0x1293] = 0x27868, [0x1294] = 0x89a8, [0x1295] = 0x4686,
  [0x1296] = 0x4687, [0x1297] = 0x4689, [0x1298] = 0x2786a, [0x1299] = 0x27869,
  [0x129a] = 0x2790d, [0x129b] = 0x27912, [0x129c] = 0x46a2, [0x129d] = 0x46a3,
  [0x129e] = 0x2790f, [0x129f] = 0x46a1, [0x12a0] = 0x27a67, [0x12a1] = 0x27a79,
  [0x12a2] = 0x27a72, [0x12a3] = 0x27a66, [0x12a4] = 0x27a7c, [0x12a5] = 0x46f9,
  [0x12a6] = 0x46fd, [0x12a7] = 0x27a68, [0x12a8] = 0x27a85, [0x12a9] = 0x27a76,
  [0x12aa] = 0x27a74, [0x12ab] = 0x27a71, [0x12ac] = 0x27a64, [0x12ad] = 0x473e,
  [0x12ae] = 0x473c, [0x12af] = 0x27bf8, [0x12b0] = 0x474d, [0x12b1] = 0x27c6c,
  [0x12b2] = 0x474e, [0x12b3] = 0x27ceb, [0x12b4] = 0x4781, [0x12b5] = 0x4783,
  [0x12b6] = 0x4782, [0x12b7] = 0x27db8, [0x12b8] = 0x4780, [0x12b9] = 0x4788,
  [0x12ba] = 0x27f2d, [0x12bb] = 0x27f2e, [0x12bc] = 0x27f28, [0x12bd] = 0x27f29,
  [0x12be] = 0x27f31, [0x12bf] = 0x47b9, [0x12c0] = 0x27f2f, [0x12c1] = 0x27f2a,
  [0x12c2] = 0x8e3a, [0x12c3] = 0x280a4, [0x12c4] = 0x280b7, [0x12c5] = 0x280bf,
  [0x12c6] = 0x4811, [0x12c7] = 0x480d, [0x12c8] = 0x4810, [0x12c9] = 0x4813,
  [0x12ca] = 0x28263, [0x12cb] = 0x483c, [0x12cc] = 0x4862, [0x12cd] = 0x4863,
  [0x12ce] = 0x28367, [0x12cf] = 0x285be, [0x12d0] = 0x2f9e0, [0x12d1] = 0x28795,
  [0x12d2] = 0x2879c, [0x12d3] = 0x4903, [0x12d4] = 0x4906, [0x12d5] = 0x4902,
  [0x12d6] = 0x4901, [0x12d7] = 0x28871, [0x12d8] = 0x28872, [0x12d9] = 0x4905,
  [0x12da] = 0x2886f, [0x12db] = 0x9194, [0x12dc] = 0x9311, [0x12dd] = 0x289f1,
  [0x12de] = 0x9337, [0x12df] = 0x4936, [0x12e0] = 0x289e8, [0x12e1] = 0x4935,
  [0x12e2] = 0x9343, [0x12e3] = 0x2f9ea, [0x12e4] = 0x49a1, [0x12e5] = 0x49a3,
  [0x12e6] = 0x28d49, [0x12e7] = 0x28d64, [0x12e8] = 0x49a0, [0x12e9] = 0x28d48,
  [0x12ea] = 0x49ea, [0x12eb] = 0x96a6, [0x12ec] = 0x49e8, [0x12ed] = 0x28f2c,
  [0x12ee] = 0x28fe1, [0x12ef] = 0x28fef, [0x12f0] = 0x290c0, [0x12f1] = 0x4a21,
  [0x12f2] = 0x4a1b, [0x12f3] = 0x290d7, [0x12f4] = 0x290d4, [0x12f5] = 0x4a49,
  [0x12f6] = 0x4a48, [0x12f7] = 0x292ad, [0x12f8] = 0x9795, [0x12f9] = 0x4a62,
  [0x12fa] = 0x4a61, [0x12fb] = 0x4a64, [0x12fc] = 0x4a60, [0x12fd] = 0x4a63,
  [0x12fe] = 0x292ae, [0x12ff] = 0x292ac, [0x1300] = 0x9796, [0x1301] = 0x4a66,
  [0x1302] = 0x4aac, [0x1303] = 0x4aab, [0x1304] = 0x29427, [0x1305] = 0x294ae,
  [0x1306] = 0x4ac3, [0x1307] = 0x294b0, [0x1308] = 0x4ac4, [0x1309] = 0x9825,
  [0x130a] = 0x294c0, [0x130b] = 0x294c4, [0x130c] = 0x4b08, [0x130d] = 0x4b09,
  [0x130e] = 0x4b0a, [0x130f] = 0x4b06, [0x1310] = 0x4b07, [0x1311] = 0x4b41,
  [0x1312] = 0x29707, [0x1313] = 0x4b45, [0x1314] = 0x296fd, [0x1315] = 0x4b43,
  [0x1316] = 0x4b44, [0x1317] = 0x9926, [0x1318] = 0x9934, [0x1319] = 0x4b47,
  [0x131a] = 0x2fa04, [0x131b] = 0x2981b, [0x131c] = 0x2981c, [0x131d] = 0x4b71,
  [0x131e] = 0x298b2, [0x131f] = 0x298b4, [0x1320] = 0x298b7, [0x1321] = 0x298b1,
  [0x1322] = 0x298b3, [0x1323] = 0x298ae, [0x1324] = 0x4b8b, [0x1325] = 0x4bd4,
  [0x1326] = 0x4bd3, [0x1327] = 0x4bd1, [0x1328] = 0x9aba, [0x1329] = 0x4bd2,
  [0x132a] = 0x29a3f, [0x132b] = 0x29b36, [0x132c] = 0x4bf7, [0x132d] = 0x4bf8,
  [0x132e] = 0x29b31, [0x132f] = 0x29b3b, [0x1330] = 0x29b35, [0x1331] = 0x29b3a,
  [0x1332] = 0x29b37, [0x1333] = 0x29c22, [0x1334] = 0x29c36, [0x1335] = 0x29c32,
  [0x1336] = 0x4c22, [0x1337] = 0x29cc0, [0x1338] = 0x4c45, [0x1339] = 0x4c41,
  [0x133a] = 0x9b81, [0x133b] = 0x4c40, [0x133c] = 0x9b8a, [0x133d] = 0x29d85,
  [0x133e] = 0x9b7f, [0x133f] = 0x4c42, [0x1340] = 0x29d89, [0x1341] = 0x4cc1,
  [0x1342] = 0x4cc5, [0x1343] = 0x29fe1, [0x1344] = 0x4cbb, [0x1345] = 0x4cb9,
  [0x1346] = 0x4cbd, [0x1347] = 0x4cc9, [0x1348] = 0x2a00a, [0x1349] = 0x9d11,
  [0x134a] = 0x2a009, [0x134b] = 0x4cbf, [0x134c] = 0x4cc7, [0x134d] = 0x29fe8,
  [0x134e] = 0x4cc3, [0x134f] = 0x4d24, [0x1350] = 0x4d31, [0x1351] = 0x4d33,
  [0x1352] = 0x2a318, [0x1353] = 0x4d32, [0x1354] = 0x4d34, [0x1355] = 0x4d52,
  [0x1356] = 0x4d61, [0x1357] = 0x9ed9, [0x1358] = 0x4d7a, [0x1359] = 0x4d82,
  [0x135a] = 0x9f3c, [0x135b] = 0x2a5c5, [0x135c] = 0x5123, [0x135d] = 0x2040d,
  [0x135e] = 0x2040c, [0x135f] = 0x20421, [0x1360] = 0x512c, [0x1361] = 0x20616,
  [0x1362] = 0x2080e, [0x1363] = 0x350f, [0x1364] = 0x5295, [0x1365] = 0x20817,
  [0x1366] = 0x3523, [0x1367] = 0x3525, [0x1368] = 0x21048, [0x1369] = 0x3606,
  [0x136a] = 0x3608, [0x136b] = 0x5688, [0x136c] = 0x21055, [0x136d] = 0x21049,
  [0x136e] = 0x2104c, [0x136f] = 0x568b, [0x1370] = 0x2104f, [0x1371] = 0x2103d,
  [0x1372] = 0x367a, [0x1373] = 0x3677, [0x1374] = 0x2148a, [0x1375] = 0x214a8,
  [0x1376] = 0x2168a, [0x1377] = 0x372e, [0x1378] = 0x218cb, [0x1379] = 0x372f,
  [0x137a] = 0x218d5, [0x137b] = 0x221d4, [0x137c] = 0x381b, [0x137d] = 0x3813,
  [0x137e] = 0x21f5c, [0x137f] = 0x21f75, [0x1380] = 0x3866, [0x1381] = 0x22161,
  [0x1382] = 0x3865, [0x1383] = 0x21696, [0x1384] = 0x38f7, [0x1385] = 0x2258a,
  [0x1386] = 0x228d0, [0x1387] = 0x61e1, [0x1388] = 0x61d7, [0x1389] = 0x22849,
  [0x138a] = 0x228cf, [0x138b] = 0x399c, [0x138c] = 0x22dd6, [0x138d] = 0x22d7f,
  [0x138e] = 0x3a53, [0x138f] = 0x22dd2, [0x1390] = 0x22dcf, [0x1391] = 0x23006,
  [0x1392] = 0x23009, [0x1393] = 0x3aba, [0x1394] = 0x65a3, [0x1395] = 0x3b22,
  [0x1396] = 0x66d3, [0x1397] = 0x23706, [0x1398] = 0x236f1, [0x1399] = 0x23704,
  [0x139a] = 0x6a8b, [0x139b] = 0x236fa, [0x139c] = 0x236f4, [0x139d] = 0x3beb,
  [0x139e] = 0x3bdd, [0x139f] = 0x236f9, [0x13a0] = 0x3bef, [0x13a1] = 0x2368b,
  [0x13a2] = 0x23703, [0x13a3] = 0x3c47, [0x13a4] = 0x3c46, [0x13a5] = 0x239a2,
  [0x13a6] = 0x3c78, [0x13a7] = 0x6bac, [0x13a8] = 0x3c89, [0x13a9] = 0x23c0c,
  [0x13aa] = 0x23c0b, [0x13ab] = 0x3d68, [0x13ac] = 0x3d76, [0x13ad] = 0x3d74,
  [0x13ae] = 0x3d79, [0x13af] = 0x24024, [0x13b0] = 0x3d7a, [0x13b1] = 0x3d77,
  [0x13b2] = 0x2f914, [0x13b3] = 0x3d71, [0x13b4] = 0x24025, [0x13b5] = 0x3d72,
  [0x13b6] = 0x243f6, [0x13b7] = 0x3dff, [0x13b8] = 0x243fb, [0x13b9] = 0x3e05,
  [0x13ba] = 0x24532, [0x13bb] = 0x245fb, [0x13bc] = 0x3e54, [0x13bd] = 0x246f2,
  [0x13be] = 0x246f3, [0x13bf] = 0x3e9e, [0x13c0] = 0x3e9f, [0x13c1] = 0x7374,
  [0x13c2] = 0x2489c, [0x13c3] = 0x24a72, [0x13c4] = 0x3efa, [0x13c5] = 0x24a74,
  [0x13c6] = 0x3f44, [0x13c7] = 0x3f3f, [0x13c8] = 0x3f40, [0x13c9] = 0x24b86,
  [0x13ca] = 0x3f42, [0x13cb] = 0x24b8e, [0x13cc] = 0x24b90, [0x13cd] = 0x3f51,
  [0x13ce] = 0x24bcd, [0x13cf] = 0x24bf7, [0x13d0] = 0x24cd6, [0x13d1] = 0x24e84,
  [0x13d2] = 0x24e95, [0x13d3] = 0x24e8a, [0x13d4] = 0x7640, [0x13d5] = 0x3fca,
  [0x13d6] = 0x24e97, [0x13d7] = 0x7641, [0x13d8] = 0x3fce, [0x13d9] = 0x3fc9,
  [0x13da] = 0x24fa0, [0x13db] = 0x24fa1, [0x13dc] = 0x25022, [0x13dd] = 0x250a6,
  [0x13de] = 0x250a4, [0x13df] = 0x4009, [0x13e0] = 0x76e8, [0x13e1] = 0x406c,
  [0x13e2] = 0x406e, [0x13e3] = 0x4070, [0x13e4] = 0x406d, [0x13e5] = 0x406b,
  [0x13e6] = 0x4071, [0x13e7] = 0x4072, [0x13e8] = 0x252af, [0x13e9] = 0x252b0,
  [0x13ea] = 0x252bd, [0x13eb] = 0x408c, [0x13ec] = 0x40e4, [0x13ed] = 0x25576,
  [0x13ee] = 0x40e1, [0x13ef] = 0x2557b, [0x13f0] = 0x25579, [0x13f1] = 0x78f6,
  [0x13f2] = 0x40e7, [0x13f3] = 0x7900, [0x13f4] = 0x40e2, [0x13f5] = 0x411f,
  [0x13f6] = 0x256ee, [0x13f7] = 0x256f1, [0x13f8] = 0x417e, [0x13f9] = 0x25894,
  [0x13fa] = 0x4180, [0x13fb] = 0x7a59, [0x13fc] = 0x7a55, [0x13fd] = 0x25891,
  [0x13fe] = 0x41b9, [0x13ff] = 0x41b7, [0x1400] = 0x41b8, [0x1401] = 0x25a10,
  [0x1402] = 0x41ba, [0x1403] = 0x7af4, [0x1404] = 0x25aaf, [0x1405] = 0x41d3,
  [0x1406] = 0x423f, [0x1407] = 0x7c04, [0x1408] = 0x4245, [0x1409] = 0x4241,
  [0x140a] = 0x7c15, [0x140b] = 0x4242, [0x140c] = 0x4243, [0x140d] = 0x423b,
  [0x140e] = 0x4238, [0x140f] = 0x25c7b, [0x1410] = 0x25c77, [0x1411] = 0x423a,
  [0x1412] = 0x7bf5, [0x1413] = 0x25c80, [0x1414] = 0x25cc6, [0x1415] = 0x423c,
  [0x1416] = 0x25c7c, [0x1417] = 0x25c90, [0x1418] = 0x423e, [0x1419] = 0x25ca3,
  [0x141a] = 0x25c7d, [0x141b] = 0x25cbd, [0x141c] = 0x429e, [0x141d] = 0x429f,
  [0x141e] = 0x42a1, [0x141f] = 0x25f13, [0x1420] = 0x429b, [0x1421] = 0x4312,
  [0x1422] = 0x26111, [0x1423] = 0x26144, [0x1424] = 0x2610d, [0x1425] = 0x4318,
  [0x1426] = 0x430c, [0x1427] = 0x26314, [0x1428] = 0x4362, [0x1429] = 0x263b8,
  [0x142a] = 0x437a, [0x142b] = 0x263b7, [0x142c] = 0x26483, [0x142d] = 0x43ae,
  [0x142e] = 0x43af, [0x142f] = 0x26514, [0x1430] = 0x43ad, [0x1431] = 0x2651c,
  [0x1432] = 0x43c4, [0x1433] = 0x43c7, [0x1434] = 0x43c6, [0x1435] = 0x43c5,
  [0x1436] = 0x265d4, [0x1437] = 0x265d5, [0x1438] = 0x81c1, [0x1439] = 0x4440,
  [0x143a] = 0x26843, [0x143b] = 0x26842, [0x143c] = 0x443f, [0x143d] = 0x4441,
  [0x143e] = 0x2f98a, [0x143f] = 0x26841, [0x1440] = 0x2694e, [0x1441] = 0x447f,
  [0x1442] = 0x26a87, [0x1443] = 0x4486, [0x1444] = 0x4481, [0x1445] = 0x4480,
  [0x1446] = 0x448e, [0x1447] = 0x454a, [0x1448] = 0x26f39, [0x1449] = 0x4547,
  [0x144a] = 0x26f8f, [0x144b] = 0x26f2a, [0x144c] = 0x454b, [0x144d] = 0x4546,
  [0x144e] = 0x454e, [0x144f] = 0x857d, [0x1450] = 0x26f2b, [0x1451] = 0x85a5,
  [0x1452] = 0x26f50, [0x1453] = 0x4548, [0x1454] = 0x26f6e, [0x1455] = 0x4545,
  [0x1456] = 0x26f3b, [0x1457] = 0x26f53, [0x1458] = 0x26f5f, [0x1459] = 0x26f2f,
  [0x145a] = 0x26f30, [0x145b] = 0x26f38, [0x145c] = 0x26f4c, [0x145d] = 0x271f1,
  [0x145e] = 0x45db, [0x145f] = 0x45e7, [0x1460] = 0x45e4, [0x1461] = 0x27407,
  [0x1462] = 0x27416, [0x1463] = 0x45e1, [0x1464] = 0x27414, [0x1465] = 0x45e9,
  [0x1466] = 0x273fb, [0x1467] = 0x45e5, [0x1468] = 0x45e0, [0x1469] = 0x45e3,
  [0x146a] = 0x273f8, [0x146b] = 0x27410, [0x146c] = 0x2741d, [0x146d] = 0x273ff,
  [0x146e] = 0x2740b, [0x146f] = 0x27404, [0x1470] = 0x45ea, [0x1471] = 0x273fe,
  [0x1472] = 0x275ff, [0x1473] = 0x893a, [0x1474] = 0x4654, [0x1475] = 0x4658,
  [0x1476] = 0x465c, [0x1477] = 0x27723, [0x1478] = 0x4655, [0x1479] = 0x468b,
  [0x147a] = 0x468c, [0x147b] = 0x46a6, [0x147c] = 0x46a5, [0x147d] = 0x2792a,
  [0x147e] = 0x27aa0, [0x147f] = 0x27a98, [0x1480] = 0x27a9c, [0x1481] = 0x46ff,
  [0x1482] = 0x27a9e, [0x1483] = 0x27aa6, [0x1484] = 0x27ab5, [0x1485] = 0x27ab0,
  [0x1486] = 0x4730, [0x1487] = 0x4740, [0x1488] = 0x4741, [0x1489] = 0x27cf9,
  [0x148a] = 0x27cf5, [0x148b] = 0x4786, [0x148c] = 0x27dd2, [0x148d] = 0x47bf,
  [0x148e] = 0x47bd, [0x148f] = 0x27f50, [0x1490] = 0x27f4e, [0x1491] = 0x27f4b,
  [0x1492] = 0x27f52, [0x1493] = 0x27f4d, [0x1494] = 0x47be, [0x1495] = 0x4819,
  [0x1496] = 0x280e4, [0x1497] = 0x481c, [0x1498] = 0x280da, [0x1499] = 0x481b,
  [0x149a] = 0x4817, [0x149b] = 0x4818, [0x149c] = 0x8e51, [0x149d] = 0x280df,
  [0x149e] = 0x280e8, [0x149f] = 0x483d, [0x14a0] = 0x486a, [0x14a1] = 0x4866,
  [0x14a2] = 0x28370, [0x14a3] = 0x28372, [0x14a4] = 0x4867, [0x14a5] = 0x28377,
  [0x14a6] = 0x4868, [0x14a7] = 0x48ad, [0x14a8] = 0x48ae, [0x14a9] = 0x285e6,
  [0x14aa] = 0x287aa, [0x14ab] = 0x48d6, [0x14ac] = 0x4909, [0x14ad] = 0x2888c,
  [0x14ae] = 0x9198, [0x14af] = 0x28889, [0x14b0] = 0x490c, [0x14b1] = 0x490a,
  [0x14b2] = 0x28a32, [0x14b3] = 0x493b, [0x14b4] = 0x493a, [0x14b5] = 0x9384,
  [0x14b6] = 0x9381, [0x14b7] = 0x28a7a, [0x14b8] = 0x936f, [0x14b9] = 0x28c9e,
  [0x14ba] = 0x28ca0, [0x14bb] = 0x28d70, [0x14bc] = 0x49af, [0x14bd] = 0x49aa,
  [0x14be] = 0x49ab, [0x14bf] = 0x28d6e, [0x14c0] = 0x28d66, [0x14c1] = 0x49b1,
  [0x14c2] = 0x28d65, [0x14c3] = 0x49ac, [0x14c4] = 0x49ec, [0x14c5] = 0x28f7f,
  [0x14c6] = 0x28fe0, [0x14c7] = 0x4a01, [0x14c8] = 0x28fff, [0x14c9] = 0x290f6,
  [0x14ca] = 0x290f7, [0x14cb] = 0x290f5, [0x14cc] = 0x4a23, [0x14cd] = 0x290fc,
  [0x14ce] = 0x4a24, [0x14cf] = 0x4a1e, [0x14d0] = 0x290f9, [0x14d1] = 0x4a4a,
  [0x14d2] = 0x4a65, [0x14d3] = 0x4a6a, [0x14d4] = 0x292ca, [0x14d5] = 0x292c3,
  [0x14d6] = 0x292c6, [0x14d7] = 0x4a69, [0x14d8] = 0x293b8, [0x14d9] = 0x293bd,
  [0x14da] = 0x4a95, [0x14db] = 0x293bc, [0x14dc] = 0x29405, [0x14dd] = 0x294e0,
  [0x14de] = 0x294e9, [0x14df] = 0x9842, [0x14e0] = 0x294df, [0x14e1] = 0x294ec,
  [0x14e2] = 0x4acc, [0x14e3] = 0x294e5, [0x14e4] = 0x294de, [0x14e5] = 0x2fa00,
  [0x14e6] = 0x4acf, [0x14e7] = 0x294f0, [0x14e8] = 0x294e3, [0x14e9] = 0x4b0f,
  [0x14ea] = 0x295ec, [0x14eb] = 0x4b0e, [0x14ec] = 0x4b0b, [0x14ed] = 0x4b10,
  [0x14ee] = 0x4b0d, [0x14ef] = 0x4b0c, [0x14f0] = 0x295f2, [0x14f1] = 0x295ef,
  [0x14f2] = 0x295e9, [0x14f3] = 0x296fb, [0x14f4] = 0x4b46, [0x14f5] = 0x4b48,
  [0x14f6] = 0x9937, [0x14f7] = 0x4b49, [0x14f8] = 0x29737, [0x14f9] = 0x298de,
  [0x14fa] = 0x4b91, [0x14fb] = 0x4b8e, [0x14fc] = 0x298da, [0x14fd] = 0x4bd8,
  [0x14fe] = 0x4bd6, [0x14ff] = 0x29a4b, [0x1500] = 0x4bda, [0x1501] = 0x29a49,
  [0x1502] = 0x4bd7, [0x1503] = 0x29ad5, [0x1504] = 0x9aff, [0x1505] = 0x4bf9,
  [0x1506] = 0x29b52, [0x1507] = 0x29b47, [0x1508] = 0x4bfc, [0x1509] = 0x29b48,
  [0x150a] = 0x29ccc, [0x150b] = 0x29cd0, [0x150c] = 0x9ba9, [0x150d] = 0x4c4a,
  [0x150e] = 0x9ba7, [0x150f] = 0x4c4e, [0x1510] = 0x9bb3, [0x1511] = 0x9bac,
  [0x1512] = 0x9bb0, [0x1513] = 0x29d86, [0x1514] = 0x29da7, [0x1515] = 0x29da3,
  [0x1516] = 0x9b9c, [0x1517] = 0x29db6, [0x1518] = 0x29dad, [0x1519] = 0x2a013,
  [0x151a] = 0x9d3c, [0x151b] = 0x9d1c, [0x151c] = 0x9d3a, [0x151d] = 0x4cd3,
  [0x151e] = 0x4ccd, [0x151f] = 0x4cd1, [0x1520] = 0x2a017, [0x1521] = 0x2a022,
  [0x1522] = 0x9d32, [0x1523] = 0x9d34, [0x1524] = 0x2a049, [0x1525] = 0x2a026,
  [0x1526] = 0x2fa0c, [0x1527] = 0x2a03c, [0x1528] = 0x2a028, [0x1529] = 0x2a2a8,
  [0x152a] = 0x9ec7, [0x152b] = 0x2a3fb, [0x152c] = 0x4d62, [0x152d] = 0x2a432,
  [0x152e] = 0x2a4d6, [0x152f] = 0x2a548, [0x1530] = 0x4d83, [0x1531] = 0x9f3f,
  [0x1532] = 0x2a5c6, [0x1533] = 0x4d92, [0x1534] = 0x349f, [0x1535] = 0x34a0,
  [0x1536] = 0x20425, [0x1537] = 0x20426, [0x1538] = 0x20424, [0x1539] = 0x20433,
  [0x153a] = 0x20570, [0x153b] = 0x3527, [0x153c] = 0x20b90, [0x153d] = 0x360b,
  [0x153e] = 0x21095, [0x153f] = 0x210a1, [0x1540] = 0x21092, [0x1541] = 0x2108f,
  [0x1542] = 0x2109f, [0x1543] = 0x21096, [0x1544] = 0x21098, [0x1545] = 0x2109d,
  [0x1546] = 0x210a0, [0x1547] = 0x368d, [0x1548] = 0x21697, [0x1549] = 0x218eb,
  [0x154a] = 0x3770, [0x154b] = 0x21f87, [0x154c] = 0x5eeb, [0x154d] = 0x22432,
  [0x154e] = 0x399a, [0x154f] = 0x399f, [0x1550] = 0x399d, [0x1551] = 0x228d8,
  [0x1552] = 0x228fb, [0x1553] = 0x22906, [0x1554] = 0x399b, [0x1555] = 0x228d1,
  [0x1556] = 0x61d5, [0x1557] = 0x2f8b0, [0x1558] = 0x3a60, [0x1559] = 0x3a64,
  [0x155a] = 0x3a69, [0x155b] = 0x3a63, [0x155c] = 0x3a67, [0x155d] = 0x3a62,
  [0x155e] = 0x22df6, [0x155f] = 0x22e04, [0x1560] = 0x22dfe, [0x1561] = 0x6502,
  [0x1562] = 0x22dff, [0x1563] = 0x22df7, [0x1564] = 0x23018, [0x1565] = 0x3aac,
  [0x1566] = 0x23014, [0x1567] = 0x230cd, [0x1568] = 0x23120, [0x1569] = 0x232de,
  [0x156a] = 0x3b26, [0x156b] = 0x3b23, [0x156c] = 0x3b25, [0x156d] = 0x23745,
  [0x156e] = 0x23753, [0x156f] = 0x3bf8, [0x1570] = 0x2374b, [0x1571] = 0x23755,
  [0x1572] = 0x23741, [0x1573] = 0x3bf7, [0x1574] = 0x3bfb, [0x1575] = 0x3bfa,
  [0x1576] = 0x2375c, [0x1577] = 0x23754, [0x1578] = 0x23a71, [0x1579] = 0x23ad0,
  [0x157a] = 0x3cb0, [0x157b] = 0x3caf, [0x157c] = 0x23c53, [0x157d] = 0x24029,
  [0x157e] = 0x2407e, [0x157f] = 0x3d7e, [0x1580] = 0x24079, [0x1581] = 0x3d7d,
  [0x1582] = 0x3d80, [0x1583] = 0x24070, [0x1584] = 0x2406a, [0x1585] = 0x3d7f,
  [0x1586] = 0x3d86, [0x1587] = 0x24085, [0x1588] = 0x24064, [0x1589] = 0x7012,
  [0x158a] = 0x24078, [0x158b] = 0x3d81, [0x158c] = 0x24432, [0x158d] = 0x24427,
  [0x158e] = 0x2442f, [0x158f] = 0x3e24, [0x1590] = 0x3e58, [0x1591] = 0x3e57,
  [0x1592] = 0x3ea0, [0x1593] = 0x248ba, [0x1594] = 0x3efe, [0x1595] = 0x24a71,
  [0x1596] = 0x24a8c, [0x1597] = 0x3f15, [0x1598] = 0x24b1a, [0x1599] = 0x3f47,
  [0x159a] = 0x3f46, [0x159b] = 0x24b98, [0x159c] = 0x24ce4, [0x159d] = 0x3f6b,
  [0x159e] = 0x3f6c, [0x159f] = 0x7585, [0x15a0] = 0x7654, [0x15a1] = 0x24ec2,
  [0x15a2] = 0x3fcc, [0x15a3] = 0x24eba, [0x15a4] = 0x7655, [0x15a5] = 0x24ec8,
  [0x15a6] = 0x3fcb, [0x15a7] = 0x76a7, [0x15a8] = 0x76a8, [0x15a9] = 0x3ff9,
  [0x15aa] = 0x250b9, [0x15ab] = 0x250b8, [0x15ac] = 0x250a5, [0x15ad] = 0x252e2,
  [0x15ae] = 0x4078, [0x15af] = 0x407a, [0x15b0] = 0x4075, [0x15b1] = 0x252d9,
  [0x15b2] = 0x4076, [0x15b3] = 0x4077, [0x15b4] = 0x252ac, [0x15b5] = 0x252dd,
  [0x15b6] = 0x40ea, [0x15b7] = 0x40ee, [0x15b8] = 0x40ed, [0x15b9] = 0x2559d,
  [0x15ba] = 0x40ec, [0x15bb] = 0x790f, [0x15bc] = 0x25711, [0x15bd] = 0x25703,
  [0x15be] = 0x4184, [0x15bf] = 0x4185, [0x15c0] = 0x4183, [0x15c1] = 0x258ee,
  [0x15c2] = 0x41bc, [0x15c3] = 0x41bd, [0x15c4] = 0x41d4, [0x15c5] = 0x25ca4,
  [0x15c6] = 0x25cd8, [0x15c7] = 0x25cdd, [0x15c8] = 0x25cde, [0x15c9] = 0x4255,
  [0x15ca] = 0x25ce7, [0x15cb] = 0x4250, [0x15cc] = 0x424c, [0x15cd] = 0x4248,
  [0x15ce] = 0x25cd4, [0x15cf] = 0x4253, [0x15d0] = 0x25cce, [0x15d1] = 0x4257,
  [0x15d2] = 0x4254, [0x15d3] = 0x424e, [0x15d4] = 0x424a, [0x15d5] = 0x4251,
  [0x15d6] = 0x25cf1, [0x15d7] = 0x25cd3, [0x15d8] = 0x4249, [0x15d9] = 0x424b,
  [0x15da] = 0x4263, [0x15db] = 0x25cca, [0x15dc] = 0x25ce9, [0x15dd] = 0x42a7,
  [0x15de] = 0x42a6, [0x15df] = 0x42a4, [0x15e0] = 0x25f1a, [0x15e1] = 0x25f41,
  [0x15e2] = 0x2f968, [0x15e3] = 0x7ce4, [0x15e4] = 0x7ce5, [0x15e5] = 0x2613d,
  [0x15e6] = 0x26140, [0x15e7] = 0x7e65, [0x15e8] = 0x7e4e, [0x15e9] = 0x4317,
  [0x15ea] = 0x26143, [0x15eb] = 0x4316, [0x15ec] = 0x2613f, [0x15ed] = 0x2613c,
  [0x15ee] = 0x4363, [0x15ef] = 0x26321, [0x15f0] = 0x26320, [0x15f1] = 0x7f82,
  [0x15f2] = 0x263c6, [0x15f3] = 0x437b, [0x15f4] = 0x437c, [0x15f5] = 0x263d1,
  [0x15f6] = 0x263ca, [0x15f7] = 0x2648e, [0x15f8] = 0x26491, [0x15f9] = 0x43b0,
  [0x15fa] = 0x802d, [0x15fb] = 0x265e3, [0x15fc] = 0x26888, [0x15fd] = 0x4442,
  [0x15fe] = 0x26870, [0x15ff] = 0x4444, [0x1600] = 0x2699d, [0x1601] = 0x26a9d,
  [0x1602] = 0x26a91, [0x1603] = 0x26a98, [0x1604] = 0x26a99, [0x1605] = 0x4488,
  [0x1606] = 0x448f, [0x1607] = 0x4553, [0x1608] = 0x455b, [0x1609] = 0x26fbf,
  [0x160a] = 0x4559, [0x160b] = 0x26fae, [0x160c] = 0x26fb1, [0x160d] = 0x26fcc,
  [0x160e] = 0x27004, [0x160f] = 0x85ca, [0x1610] = 0x27016, [0x1611] = 0x26fcd,
  [0x1612] = 0x4554, [0x1613] = 0x85bc, [0x1614] = 0x26fe0, [0x1615] = 0x26fcb,
  [0x1616] = 0x26fd4, [0x1617] = 0x26fc9, [0x1618] = 0x271ff, [0x1619] = 0x4599,
  [0x161a] = 0x271fd, [0x161b] = 0x27457, [0x161c] = 0x27452, [0x161d] = 0x27450,
  [0x161e] = 0x27445, [0x161f] = 0x2744d, [0x1620] = 0x45f1, [0x1621] = 0x27453,
  [0x1622] = 0x45ef, [0x1623] = 0x27482, [0x1624] = 0x27444, [0x1625] = 0x275ce,
  [0x1626] = 0x275cf, [0x1627] = 0x275d2, [0x1628] = 0x27602, [0x1629] = 0x2774f,
  [0x162a] = 0x27758, [0x162b] = 0x4662, [0x162c] = 0x27743, [0x162d] = 0x4663,
  [0x162e] = 0x27752, [0x162f] = 0x4660, [0x1630] = 0x4661, [0x1631] = 0x465f,
  [0x1632] = 0x27749, [0x1633] = 0x27753, [0x1634] = 0x2774d, [0x1635] = 0x468d,
  [0x1636] = 0x27884, [0x1637] = 0x468e, [0x1638] = 0x27883, [0x1639] = 0x27932,
  [0x163a] = 0x27ae2, [0x163b] = 0x4709, [0x163c] = 0x27ae4, [0x163d] = 0x27ad3,
  [0x163e] = 0x4705, [0x163f] = 0x27ad5, [0x1640] = 0x27add, [0x1641] = 0x4703,
  [0x1642] = 0x4706, [0x1643] = 0x27aec, [0x1644] = 0x27ae6, [0x1645] = 0x27ad2,
  [0x1646] = 0x27afa, [0x1647] = 0x27ada, [0x1648] = 0x4731, [0x1649] = 0x27c06,
  [0x164a] = 0x27c04, [0x164b] = 0x474f, [0x164c] = 0x27c7b, [0x164d] = 0x27d04,
  [0x164e] = 0x27d02, [0x164f] = 0x4766, [0x1650] = 0x8cff, [0x1651] = 0x47c4,
  [0x1652] = 0x27f60, [0x1653] = 0x47c3, [0x1654] = 0x47c1, [0x1655] = 0x47c5,
  [0x1656] = 0x28107, [0x1657] = 0x28105, [0x1658] = 0x2810c, [0x1659] = 0x4821,
  [0x165a] = 0x2815a, [0x165b] = 0x481f, [0x165c] = 0x4822, [0x165d] = 0x2811a,
  [0x165e] = 0x4827, [0x165f] = 0x4820, [0x1660] = 0x2838a, [0x1661] = 0x486d,
  [0x1662] = 0x486c, [0x1663] = 0x486b, [0x1664] = 0x486f, [0x1665] = 0x4870,
  [0x1666] = 0x2838c, [0x1667] = 0x28609, [0x1668] = 0x288a6, [0x1669] = 0x91a6,
  [0x166a] = 0x288ac, [0x166b] = 0x288a9, [0x166c] = 0x4942, [0x166d] = 0x28a8b,
  [0x166e] = 0x93b6, [0x166f] = 0x28a8c, [0x1670] = 0x4944, [0x1671] = 0x4940,
  [0x1672] = 0x28a88, [0x1673] = 0x28a89, [0x1674] = 0x493f, [0x1675] = 0x28a93,
  [0x1676] = 0x93ab, [0x1677] = 0x498b, [0x1678] = 0x28d83, [0x1679] = 0x28f4f,
  [0x167a] = 0x4a25, [0x167b] = 0x4a28, [0x167c] = 0x29120, [0x167d] = 0x9721,
  [0x167e] = 0x29236, [0x167f] = 0x292e2, [0x1680] = 0x4a75, [0x1681] = 0x4a72,
  [0x1682] = 0x292ee, [0x1683] = 0x4a6f, [0x1684] = 0x292e7, [0x1685] = 0x292e9,
  [0x1686] = 0x4a76, [0x1687] = 0x4a71, [0x1688] = 0x97a7, [0x1689] = 0x292df,
  [0x168a] = 0x4a97, [0x168b] = 0x293c7, [0x168c] = 0x4ad7, [0x168d] = 0x29509,
  [0x168e] = 0x4ad6, [0x168f] = 0x29501, [0x1690] = 0x4ad8, [0x1691] = 0x4adc,
  [0x1692] = 0x4adb, [0x1693] = 0x4ad4, [0x1694] = 0x983e, [0x1695] = 0x4b13,
  [0x1696] = 0x4b11, [0x1697] = 0x4b14, [0x1698] = 0x29605, [0x1699] = 0x2960c,
  [0x169a] = 0x29778, [0x169b] = 0x2975d, [0x169c] = 0x4b51, [0x169d] = 0x4b50,
  [0x169e] = 0x4b53, [0x169f] = 0x4b54, [0x16a0] = 0x4b52, [0x16a1] = 0x2975e,
  [0x16a2] = 0x29760, [0x16a3] = 0x29767, [0x16a4] = 0x4b6d, [0x16a5] = 0x2fa05,
  [0x16a6] = 0x298f4, [0x16a7] = 0x4b95, [0x16a8] = 0x4b99, [0x16a9] = 0x298f1,
  [0x16aa] = 0x4b9a, [0x16ab] = 0x298fa, [0x16ac] = 0x4b93, [0x16ad] = 0x4b97,
  [0x16ae] = 0x298f8, [0x16af] = 0x298f9, [0x16b0] = 0x29909, [0x16b1] = 0x298fd,
  [0x16b2] = 0x298f7, [0x16b3] = 0x4bdc, [0x16b4] = 0x29a59, [0x16b5] = 0x4bfd,
  [0x16b6] = 0x29b60, [0x16b7] = 0x29b68, [0x16b8] = 0x4bfe, [0x16b9] = 0x4c00,
  [0x16ba] = 0x4c02, [0x16bb] = 0x4c01, [0x16bc] = 0x4c03, [0x16bd] = 0x29b6a,
  [0x16be] = 0x29c14, [0x16bf] = 0x29c3e, [0x16c0] = 0x29ce4, [0x16c1] = 0x4c27,
  [0x16c2] = 0x4c26, [0x16c3] = 0x4c24, [0x16c4] = 0x4c4c, [0x16c5] = 0x9bbc,
  [0x16c6] = 0x4c50, [0x16c7] = 0x4c55, [0x16c8] = 0x4c53, [0x16c9] = 0x9bb7,
  [0x16ca] = 0x4c52, [0x16cb] = 0x29dd2, [0x16cc] = 0x4c57, [0x16cd] = 0x9bbe,
  [0x16ce] = 0x4c58, [0x16cf] = 0x4cd6, [0x16d0] = 0x2a058, [0x16d1] = 0x2a050,
  [0x16d2] = 0x4cd4, [0x16d3] = 0x2a05c, [0x16d4] = 0x4cda, [0x16d5] = 0x4cd9,
  [0x16d6] = 0x28119, [0x16d7] = 0x9d62, [0x16d8] = 0x4cd5, [0x16d9] = 0x4ce4,
  [0x16da] = 0x2fa0e, [0x16db] = 0x4cdc, [0x16dc] = 0x4d1b, [0x16dd] = 0x9e8f,
  [0x16de] = 0x4d37, [0x16df] = 0x4d36, [0x16e0] = 0x4d4b, [0x16e1] = 0x9ecb,
  [0x16e2] = 0x4d66, [0x16e3] = 0x4d76, [0x16e4] = 0x2a4df, [0x16e5] = 0x4d7e,
  [0x16e6] = 0x4d7d, [0x16e7] = 0x4d7f, [0x16e8] = 0x4d84, [0x16e9] = 0x4d8b,
  [0x16ea] = 0x2a5d9, [0x16eb] = 0x4d94, [0x16ec] = 0x34a1, [0x16ed] = 0x3511,
  [0x16ee] = 0x20966, [0x16ef] = 0x3610, [0x16f0] = 0x56a9, [0x16f1] = 0x210e1,
  [0x16f2] = 0x210ef, [0x16f3] = 0x210e8, [0x16f4] = 0x21233, [0x16f5] = 0x5913,
  [0x16f6] = 0x218fe, [0x16f7] = 0x3732, [0x16f8] = 0x5bf4, [0x16f9] = 0x21aef,
  [0x16fa] = 0x21aec, [0x16fb] = 0x21cec, [0x16fc] = 0x21f9c, [0x16fd] = 0x3820,
  [0x16fe] = 0x21f99, [0x16ff] = 0x22170, [0x1700] = 0x2216e, [0x1701] = 0x389d,
  [0x1702] = 0x22441, [0x1703] = 0x61ec, [0x1704] = 0x61ef, [0x1705] = 0x22902,
  [0x1706] = 0x22901, [0x1707] = 0x39a5, [0x1708] = 0x22a1b, [0x1709] = 0x22e23,
  [0x170a] = 0x22e25, [0x170b] = 0x22e24, [0x170c] = 0x22e26, [0x170d] = 0x22e33,
  [0x170e] = 0x22e2f, [0x170f] = 0x2308f, [0x1710] = 0x23356, [0x1711] = 0x3c02,
  [0x1712] = 0x3bfe, [0x1713] = 0x3c01, [0x1714] = 0x6ad6, [0x1715] = 0x3c03,
  [0x1716] = 0x3bff, [0x1717] = 0x23799, [0x1718] = 0x3c04, [0x1719] = 0x2f8ed,
  [0x171a] = 0x3c4a, [0x171b] = 0x23ad9, [0x171c] = 0x3d87, [0x171d] = 0x240b7,
  [0x171e] = 0x3d84, [0x171f] = 0x240b6, [0x1720] = 0x3d85, [0x1721] = 0x7209,
  [0x1722] = 0x24454, [0x1723] = 0x24457, [0x1724] = 0x2447a, [0x1725] = 0x3e59,
  [0x1726] = 0x7379, [0x1727] = 0x24a8e, [0x1728] = 0x74c6, [0x1729] = 0x24a90,
  [0x172a] = 0x24ab9, [0x172b] = 0x3f04, [0x172c] = 0x3f49, [0x172d] = 0x3f48,
  [0x172e] = 0x24cef, [0x172f] = 0x3f6d, [0x1730] = 0x3fd2, [0x1731] = 0x3fd3,
  [0x1732] = 0x24ee2, [0x1733] = 0x3fd1, [0x1734] = 0x24ed6, [0x1735] = 0x24ed8,
  [0x1736] = 0x24eda, [0x1737] = 0x24ede, [0x1738] = 0x3fe7, [0x1739] = 0x400a,
  [0x173a] = 0x77c3, [0x173b] = 0x25308, [0x173c] = 0x25304, [0x173d] = 0x2530a,
  [0x173e] = 0x2530b, [0x173f] = 0x25302, [0x1740] = 0x40f0, [0x1741] = 0x255aa,
  [0x1742] = 0x255c1, [0x1743] = 0x791f, [0x1744] = 0x258d7, [0x1745] = 0x258d9,
  [0x1746] = 0x7a65, [0x1747] = 0x258de, [0x1748] = 0x258da, [0x1749] = 0x41be,
  [0x174a] = 0x41bf, [0x174b] = 0x25a2a, [0x174c] = 0x7ac6, [0x174d] = 0x7c3a,
  [0x174e] = 0x25d31, [0x174f] = 0x7c36, [0x1750] = 0x25d2b, [0x1751] = 0x25d2c,
  [0x1752] = 0x25d29, [0x1753] = 0x25d2e, [0x1754] = 0x425e, [0x1755] = 0x25d27,
  [0x1756] = 0x25d28, [0x1757] = 0x425b, [0x1758] = 0x25d5f, [0x1759] = 0x25d30,
  [0x175a] = 0x25d24, [0x175b] = 0x7ceb, [0x175c] = 0x42ab, [0x175d] = 0x25f3a,
  [0x175e] = 0x42ac, [0x175f] = 0x25f39, [0x1760] = 0x25f40, [0x1761] = 0x26180,
  [0x1762] = 0x26184, [0x1763] = 0x431f, [0x1764] = 0x431d, [0x1765] = 0x26175,
  [0x1766] = 0x26176, [0x1767] = 0x431c, [0x1768] = 0x2617c, [0x1769] = 0x431e,
  [0x176a] = 0x26178, [0x176b] = 0x2619b, [0x176c] = 0x2617e, [0x176d] = 0x2625a,
  [0x176e] = 0x4364, [0x176f] = 0x2632c, [0x1770] = 0x7f84, [0x1771] = 0x2649c,
  [0x1772] = 0x26847, [0x1773] = 0x2688a, [0x1774] = 0x4448, [0x1775] = 0x4447,
  [0x1776] = 0x26927, [0x1777] = 0x2704a, [0x1778] = 0x455e, [0x1779] = 0x4561,
  [0x177a] = 0x27027, [0x177b] = 0x85e0, [0x177c] = 0x85f3, [0x177d] = 0x2702e,
  [0x177e] = 0x27026, [0x177f] = 0x27208, [0x1780] = 0x45f7, [0x1781] = 0x274bd,
  [0x1782] = 0x27496, [0x1783] = 0x45f4, [0x1784] = 0x45f8, [0x1785] = 0x881e,
  [0x1786] = 0x274be, [0x1787] = 0x2748e, [0x1788] = 0x27774, [0x1789] = 0x4664,
  [0x178a] = 0x27780, [0x178b] = 0x4692, [0x178c] = 0x468f, [0x178d] = 0x4690,
  [0x178e] = 0x89b4, [0x178f] = 0x4693, [0x1790] = 0x46a8, [0x1791] = 0x2793c,
  [0x1792] = 0x2793d, [0x1793] = 0x46a9, [0x1794] = 0x2793a, [0x1795] = 0x27942,
  [0x1796] = 0x89f9, [0x1797] = 0x27b24, [0x1798] = 0x27b06, [0x1799] = 0x8b44,
  [0x179a] = 0x470e, [0x179b] = 0x27b15, [0x179c] = 0x470f, [0x179d] = 0x27b02,
  [0x179e] = 0x27b0c, [0x179f] = 0x470b, [0x17a0] = 0x27b08, [0x17a1] = 0x27b0a,
  [0x17a2] = 0x27b05, [0x17a3] = 0x27b1c, [0x17a4] = 0x4710, [0x17a5] = 0x4751,
  [0x17a6] = 0x27c82, [0x17a7] = 0x4750, [0x17a8] = 0x27c86, [0x17a9] = 0x4763,
  [0x17aa] = 0x27e03, [0x17ab] = 0x27f7b, [0x17ac] = 0x47c7, [0x17ad] = 0x27f7a,
  [0x17ae] = 0x8e71, [0x17af] = 0x4824, [0x17b0] = 0x4826, [0x17b1] = 0x8e6e,
  [0x17b2] = 0x28180, [0x17b3] = 0x8e79, [0x17b4] = 0x28158, [0x17b5] = 0x8ec4,
  [0x17b6] = 0x4874, [0x17b7] = 0x4873, [0x17b8] = 0x4872, [0x17b9] = 0x283aa,
  [0x17ba] = 0x283ab, [0x17bb] = 0x28436, [0x17bc] = 0x2844a, [0x17bd] = 0x48b1,
  [0x17be] = 0x908c, [0x17bf] = 0x287d6, [0x17c0] = 0x490e, [0x17c1] = 0x4911,
  [0x17c2] = 0x4910, [0x17c3] = 0x490f, [0x17c4] = 0x4912, [0x17c5] = 0x4949,
  [0x17c6] = 0x93c9, [0x17c7] = 0x494f, [0x17c8] = 0x494d, [0x17c9] = 0x28b03,
  [0x17ca] = 0x28acf, [0x17cb] = 0x4955, [0x17cc] = 0x28d9f, [0x17cd] = 0x28da2,
  [0x17ce] = 0x2902a, [0x17cf] = 0x2902f, [0x17d0] = 0x29144, [0x17d1] = 0x29140,
  [0x17d2] = 0x4a2c, [0x17d3] = 0x4a2b, [0x17d4] = 0x29311, [0x17d5] = 0x2930f,
  [0x17d6] = 0x4a79, [0x17d7] = 0x2930d, [0x17d8] = 0x29307, [0x17d9] = 0x29303,
  [0x17da] = 0x4a7a, [0x17db] = 0x97b0, [0x17dc] = 0x4a99, [0x17dd] = 0x4a9a,
  [0x17de] = 0x29435, [0x17df] = 0x4aae, [0x17e0] = 0x4aaf, [0x17e1] = 0x29504,
  [0x17e2] = 0x4ae4, [0x17e3] = 0x4ae1, [0x17e4] = 0x4ade, [0x17e5] = 0x4ae6,
  [0x17e6] = 0x4adf, [0x17e7] = 0x29526, [0x17e8] = 0x4ae7, [0x17e9] = 0x4ae2,
  [0x17ea] = 0x4ae0, [0x17eb] = 0x2951e, [0x17ec] = 0x4ae5, [0x17ed] = 0x985a,
  [0x17ee] = 0x2960e, [0x17ef] = 0x4b18, [0x17f0] = 0x2961d, [0x17f1] = 0x2961e,
  [0x17f2] = 0x2961f, [0x17f3] = 0x4b56, [0x17f4] = 0x29788, [0x17f5] = 0x9946,
  [0x17f6] = 0x2978d, [0x17f7] = 0x29791, [0x17f8] = 0x29780, [0x17f9] = 0x4b6e,
  [0x17fa] = 0x29854, [0x17fb] = 0x29855, [0x17fc] = 0x4b9d, [0x17fd] = 0x4ba0,
  [0x17fe] = 0x4b9c, [0x17ff] = 0x29925, [0x1800] = 0x2991a, [0x1801] = 0x4ba1,
  [0x1802] = 0x4ba2, [0x1803] = 0x4b9f, [0x1804] = 0x4bdf, [0x1805] = 0x4bde,
  [0x1806] = 0x9ac3, [0x1807] = 0x4bea, [0x1808] = 0x29b88, [0x1809] = 0x4c06,
  [0x180a] = 0x29b7a, [0x180b] = 0x4c04, [0x180c] = 0x9b0f, [0x180d] = 0x29b80,
  [0x180e] = 0x29d1e, [0x180f] = 0x29dfc, [0x1810] = 0x29e25, [0x1811] = 0x4c5f,
  [0x1812] = 0x9bf4, [0x1813] = 0x9bfa, [0x1814] = 0x4c5c, [0x1815] = 0x29e0b,
  [0x1816] = 0x4c5e, [0x1817] = 0x29e06, [0x1818] = 0x29e04, [0x1819] = 0x9bdd,
  [0x181a] = 0x4c59, [0x181b] = 0x29df9, [0x181c] = 0x29e00, [0x181d] = 0x4c64,
  [0x181e] = 0x4c5d, [0x181f] = 0x4c62, [0x1820] = 0x29e02, [0x1821] = 0x4c65,
  [0x1822] = 0x29e07, [0x1823] = 0x9bed, [0x1824] = 0x4c5b, [0x1825] = 0x9bef,
  [0x1826] = 0x2a0b5, [0x1827] = 0x4cdd, [0x1828] = 0x2a087, [0x1829] = 0x4cdf,
  [0x182a] = 0x2a093, [0x182b] = 0x2fa0f, [0x182c] = 0x4ce2, [0x182d] = 0x2a0be,
  [0x182e] = 0x2a266, [0x182f] = 0x4d27, [0x1830] = 0x2a2c7, [0x1831] = 0x2a2cf,
  [0x1832] = 0x9e96, [0x1833] = 0x4d3a, [0x1834] = 0x4d3c, [0x1835] = 0x2a33d,
  [0x1836] = 0x4d39, [0x1837] = 0x2a33c, [0x1838] = 0x2a347, [0x1839] = 0x4d3d,
  [0x183a] = 0x4d3b, [0x183b] = 0x9eb3, [0x183c] = 0x4d4c, [0x183d] = 0x2a3c3,
  [0x183e] = 0x4d68, [0x183f] = 0x9ee2, [0x1840] = 0x2a51b, [0x1841] = 0x4d80,
  [0x1842] = 0x4d85, [0x1843] = 0x2a5c9, [0x1844] = 0x4d95, [0x1845] = 0x2a5dd,
  [0x1846] = 0x4d96, [0x1847] = 0x2a693, [0x1848] = 0x9f8f, [0x1849] = 0x2043d,
  [0x184a] = 0x34a4, [0x184b] = 0x3512, [0x184c] = 0x56b1, [0x184d] = 0x3625,
  [0x184e] = 0x214e6, [0x184f] = 0x5b41, [0x1850] = 0x3737, [0x1851] = 0x21af7,
  [0x1852] = 0x21fb0, [0x1853] = 0x21fae, [0x1855] = 0x3868, [0x1856] = 0x3867,
  [0x1857] = 0x389e, [0x1858] = 0x2259f, [0x1859] = 0x2259e, [0x185a] = 0x22930,
  [0x185b] = 0x39aa, [0x185c] = 0x2294f, [0x185d] = 0x39a9, [0x185e] = 0x39a4,
  [0x185f] = 0x22927, [0x1860] = 0x22951, [0x1861] = 0x3a71, [0x1862] = 0x3a6f,
  [0x1863] = 0x22e51, [0x1864] = 0x22e54, [0x1865] = 0x22e56, [0x1866] = 0x23027,
  [0x1867] = 0x3aad, [0x1868] = 0x23024, [0x1869] = 0x6af6, [0x186a] = 0x3c0c,
  [0x186b] = 0x6af2, [0x186c] = 0x3c0b, [0x186d] = 0x237ec, [0x186e] = 0x237c4,
  [0x186f] = 0x3c0f, [0x1870] = 0x3c79, [0x1871] = 0x240f8, [0x1872] = 0x240f6,
  [0x1873] = 0x240f7, [0x1874] = 0x240ed, [0x1875] = 0x3d8d, [0x1876] = 0x3d8f,
  [0x1877] = 0x240f4, [0x1878] = 0x240ef, [0x1879] = 0x3d8e, [0x187a] = 0x3e0c,
  [0x187b] = 0x2447f, [0x187c] = 0x244a2, [0x187d] = 0x2447e, [0x187e] = 0x3ea6,
  [0x187f] = 0x248c5, [0x1880] = 0x3ea3, [0x1881] = 0x3ea4, [0x1882] = 0x3ea5,
  [0x1883] = 0x7588, [0x1884] = 0x3f6e, [0x1885] = 0x24ef1, [0x1886] = 0x24ef2,
  [0x1887] = 0x3ffa, [0x1888] = 0x2502f, [0x1889] = 0x407c, [0x188a] = 0x407e,
  [0x188b] = 0x407b, [0x188c] = 0x407d, [0x188d] = 0x25323, [0x188e] = 0x25329,
  [0x188f] = 0x408d, [0x1890] = 0x40f4, [0x1891] = 0x40f3, [0x1892] = 0x2571b,
  [0x1893] = 0x258eb, [0x1894] = 0x4189, [0x1895] = 0x258ea, [0x1896] = 0x25a33,
  [0x1897] = 0x41c0, [0x1898] = 0x25d63, [0x1899] = 0x4265, [0x189a] = 0x25d92,
  [0x189b] = 0x25d65, [0x189c] = 0x42ad, [0x189d] = 0x4325, [0x189e] = 0x261a0,
  [0x189f] = 0x2f970, [0x18a0] = 0x2625e, [0x18a1] = 0x43c9, [0x18a2] = 0x268aa,
  [0x18a3] = 0x444a, [0x18a4] = 0x268a9, [0x18a5] = 0x8267, [0x18a6] = 0x4489,
  [0x18a7] = 0x2709c, [0x18a8] = 0x4566, [0x18a9] = 0x4570, [0x18aa] = 0x27092,
  [0x18ab] = 0x456d, [0x18ac] = 0x4569, [0x18ad] = 0x4567, [0x18ae] = 0x27086,
  [0x18af] = 0x4572, [0x18b0] = 0x860e, [0x18b1] = 0x456e, [0x18b2] = 0x27083,
  [0x18b3] = 0x459c, [0x18b4] = 0x45fc, [0x18b5] = 0x45fd, [0x18b6] = 0x4604,
  [0x18b7] = 0x45ff, [0x18b8] = 0x2f9c2, [0x18b9] = 0x45fe, [0x18ba] = 0x4600,
  [0x18bb] = 0x274ce, [0x18bc] = 0x4666, [0x18bd] = 0x4669, [0x18be] = 0x27793,
  [0x18bf] = 0x46aa, [0x18c0] = 0x46ab, [0x18c1] = 0x4717, [0x18c2] = 0x27b2e,
  [0x18c3] = 0x27b27, [0x18c4] = 0x27b28, [0x18c5] = 0x4715, [0x18c6] = 0x8b5e,
  [0x18c7] = 0x4712, [0x18c8] = 0x8d0e, [0x18c9] = 0x27e18, [0x18ca] = 0x27e16,
  [0x18cb] = 0x27f8d, [0x18cc] = 0x47ca, [0x18cd] = 0x27f8e, [0x18ce] = 0x47c9,
  [0x18cf] = 0x47cb, [0x18d0] = 0x27f90, [0x18d1] = 0x27f8f, [0x18d2] = 0x28181,
  [0x18d3] = 0x4829, [0x18d4] = 0x4828, [0x18d5] = 0x2818a, [0x18d6] = 0x2818c,
  [0x18d7] = 0x2828d, [0x18d8] = 0x4840, [0x18d9] = 0x4875, [0x18da] = 0x4876,
  [0x18db] = 0x283b2, [0x18dc] = 0x4888, [0x18dd] = 0x287d9, [0x18de] = 0x91b6,
  [0x18df] = 0x4957, [0x18e0] = 0x9401, [0x18e1] = 0x28b0d, [0x18e2] = 0x495f,
  [0x18e3] = 0x28b13, [0x18e4] = 0x941d, [0x18e5] = 0x4958, [0x18e6] = 0x495b,
  [0x18e7] = 0x28b1b, [0x18e8] = 0x942f, [0x18e9] = 0x28dac, [0x18ea] = 0x49b3,
  [0x18eb] = 0x28db3, [0x18ec] = 0x49ef, [0x18ed] = 0x2915e, [0x18ee] = 0x4a30,
  [0x18ef] = 0x29160, [0x18f0] = 0x29168, [0x18f1] = 0x29163, [0x18f2] = 0x29169,
  [0x18f3] = 0x29167, [0x18f4] = 0x4a41, [0x18f5] = 0x4a4b, [0x18f6] = 0x29328,
  [0x18f7] = 0x4a7d, [0x18f8] = 0x2932e, [0x18f9] = 0x29330, [0x18fa] = 0x4a7c,
  [0x18fb] = 0x29326, [0x18fc] = 0x29331, [0x18fd] = 0x29341, [0x18fe] = 0x97e0,
  [0x18ff] = 0x293da, [0x1900] = 0x97db, [0x1901] = 0x2f9fa, [0x1902] = 0x2940b,
  [0x1903] = 0x9861, [0x1904] = 0x29534, [0x1905] = 0x29533, [0x1906] = 0x4ae8,
  [0x1907] = 0x4aea, [0x1908] = 0x4ae9, [0x1909] = 0x29539, [0x190a] = 0x2953b,
  [0x190b] = 0x29540, [0x190c] = 0x29630, [0x190d] = 0x4b1b, [0x190e] = 0x29632,
  [0x190f] = 0x29637, [0x1910] = 0x4b55, [0x1911] = 0x994a, [0x1912] = 0x4b59,
  [0x1913] = 0x4b58, [0x1914] = 0x29781, [0x1915] = 0x2979f, [0x1916] = 0x297a7,
  [0x1917] = 0x4ba4, [0x1918] = 0x4ba3, [0x1919] = 0x29944, [0x191a] = 0x29947,
  [0x191b] = 0x29948, [0x191c] = 0x2993d, [0x191d] = 0x9a33, [0x191e] = 0x4ba7,
  [0x191f] = 0x29949, [0x1920] = 0x4be0, [0x1921] = 0x29a80, [0x1922] = 0x29ba0,
  [0x1923] = 0x29b9d, [0x1924] = 0x4c08, [0x1925] = 0x4c0a, [0x1926] = 0x4c09,
  [0x1927] = 0x29b9c, [0x1928] = 0x29c47, [0x1929] = 0x29d07, [0x192a] = 0x4c71,
  [0x192b] = 0x9c0f, [0x192c] = 0x4c6c, [0x192d] = 0x29e49, [0x192e] = 0x9c11,
  [0x192f] = 0x29e44, [0x1930] = 0x9c03, [0x1931] = 0x9c01, [0x1932] = 0x4c6e,
  [0x1933] = 0x29edf, [0x1934] = 0x9c16, [0x1935] = 0x29e4c, [0x1937] = 0x29e4f,
  [0x1938] = 0x4ce0, [0x1939] = 0x4cee, [0x193a] = 0x2a0c1, [0x193b] = 0x4ceb,
  [0x193c] = 0x2a0b9, [0x193d] = 0x2a0cb, [0x193e] = 0x2a0cf, [0x193f] = 0x2a0c4,
  [0x1940] = 0x9d93, [0x1941] = 0x4cea, [0x1942] = 0x4cef, [0x1943] = 0x4ce7,
  [0x1944] = 0x2a0ca, [0x1945] = 0x2a0c3, [0x1946] = 0x2a2d0, [0x1947] = 0x2a351,
  [0x1948] = 0x4d48, [0x1949] = 0x4d49, [0x194a] = 0x2a3a8, [0x194b] = 0x2a3c7,
  [0x194c] = 0x2a3c6, [0x194d] = 0x4d4d, [0x194e] = 0x2a3ca, [0x194f] = 0x2a402,
  [0x1950] = 0x4d55, [0x1951] = 0x2a45d, [0x1952] = 0x2a45a, [0x1953] = 0x4d6a,
  [0x1954] = 0x4d6c, [0x1955] = 0x2a459, [0x1956] = 0x4d6b, [0x1957] = 0x2a4cc,
  [0x1958] = 0x2a51c, [0x1959] = 0x2a5cc, [0x195a] = 0x4d98, [0x195b] = 0x4d99,
  [0x195c] = 0x4d97, [0x195d] = 0x2a5e8, [0x195e] = 0x2a5ec, [0x195f] = 0x2a5ea,
  [0x1960] = 0x2a6ad, [0x1961] = 0x2a6b0, [0x1962] = 0x2045f, [0x1963] = 0x20458,
  [0x1964] = 0x20457, [0x1965] = 0x535b, [0x1966] = 0x20b93, [0x1967] = 0x3616,
  [0x1968] = 0x2113b, [0x1969] = 0x56bf, [0x196a] = 0x21134, [0x196b] = 0x214f2,
  [0x196c] = 0x3739, [0x196d] = 0x21afd, [0x196e] = 0x21c1d, [0x196f] = 0x3825,
  [0x1970] = 0x5dce, [0x1971] = 0x22304, [0x1972] = 0x22448, [0x1973] = 0x22958,
  [0x1974] = 0x22e4f, [0x1975] = 0x3a74, [0x1976] = 0x22e6e, [0x1977] = 0x22e72,
  [0x1978] = 0x3aae, [0x1979] = 0x2302e, [0x197a] = 0x237fc, [0x197b] = 0x237f4,
  [0x197c] = 0x23c36, [0x197d] = 0x2410e, [0x197e] = 0x3d92, [0x197f] = 0x3d94,
  [0x1980] = 0x24114, [0x1981] = 0x3d95, [0x1982] = 0x24119, [0x1983] = 0x3e0d,
  [0x1984] = 0x244a6, [0x1985] = 0x3e25, [0x1986] = 0x24ac9, [0x1987] = 0x24ac0,
  [0x1988] = 0x24acc, [0x1989] = 0x24b1b, [0x198a] = 0x24baa, [0x198b] = 0x24ba8,
  [0x198c] = 0x24f03, [0x198d] = 0x3fd5, [0x198e] = 0x3fd6, [0x198f] = 0x76ac,
  [0x1990] = 0x3fe8, [0x1991] = 0x25035, [0x1992] = 0x407f, [0x1993] = 0x77d2,
  [0x1994] = 0x40f5, [0x1995] = 0x40f6, [0x1996] = 0x40f7, [0x1997] = 0x255d9,
  [0x1998] = 0x4124, [0x1999] = 0x418d, [0x199a] = 0x418a, [0x199b] = 0x25a3f,
  [0x199c] = 0x25a3d, [0x199d] = 0x426c, [0x199e] = 0x4266, [0x199f] = 0x426a,
  [0x19a0] = 0x25d8b, [0x19a1] = 0x4267, [0x19a2] = 0x426d, [0x19a3] = 0x4268,
  [0x19a4] = 0x7c52, [0x19a5] = 0x25d68, [0x19a6] = 0x25d8a, [0x19a7] = 0x25f58,
  [0x19a8] = 0x25f57, [0x19a9] = 0x261ce, [0x19aa] = 0x261bc, [0x19ab] = 0x261c0,
  [0x19ac] = 0x261c1, [0x19ad] = 0x261bf, [0x19ae] = 0x261ab, [0x19af] = 0x2625f,
  [0x19b0] = 0x4365, [0x19b1] = 0x264a6, [0x19b2] = 0x439a, [0x19b3] = 0x26529,
  [0x19b4] = 0x43b1, [0x19b5] = 0x444b, [0x19b6] = 0x444d, [0x19b7] = 0x444c,
  [0x19b8] = 0x444e, [0x19b9] = 0x268b8, [0x19ba] = 0x4573, [0x19bb] = 0x4575,
  [0x19bc] = 0x270dd, [0x19bd] = 0x270d6, [0x19be] = 0x2f9b1, [0x19bf] = 0x270d5,
  [0x19c0] = 0x270e7, [0x19c1] = 0x270d8, [0x19c2] = 0x274ec, [0x19c3] = 0x4603,
  [0x19c4] = 0x27500, [0x19c5] = 0x27507, [0x19c6] = 0x274fd, [0x19c7] = 0x274f1,
  [0x19c8] = 0x274ff, [0x19c9] = 0x277aa, [0x19ca] = 0x277b0, [0x19cb] = 0x27b48,
  [0x19cc] = 0x471e, [0x19cd] = 0x27b53, [0x19ce] = 0x8b73, [0x19cf] = 0x27b4d,
  [0x19d0] = 0x4719, [0x19d1] = 0x471c, [0x19d2] = 0x471a, [0x19d3] = 0x471d,
  [0x19d4] = 0x8b76, [0x19d5] = 0x27b43, [0x19d6] = 0x27b50, [0x19d7] = 0x4743,
  [0x19d8] = 0x4752, [0x19d9] = 0x27c8c, [0x19da] = 0x27e27, [0x19db] = 0x4795,
  [0x19dc] = 0x27f99, [0x19dd] = 0x47cc, [0x19de] = 0x281b1, [0x19df] = 0x482b,
  [0x19e0] = 0x281b0, [0x19e1] = 0x281aa, [0x19e2] = 0x281ac, [0x19e3] = 0x482a,
  [0x19e4] = 0x8ec7, [0x19e5] = 0x4877, [0x19e6] = 0x283c8, [0x19e7] = 0x283ca,
  [0x19e8] = 0x28642, [0x19e9] = 0x287e0, [0x19ea] = 0x288e7, [0x19eb] = 0x288e8,
  [0x19ec] = 0x288e6, [0x19ed] = 0x4913, [0x19ee] = 0x4914, [0x19ef] = 0x9434,
  [0x19f0] = 0x28b5b, [0x19f1] = 0x28b56, [0x19f2] = 0x495d, [0x19f3] = 0x28b5a,
  [0x19f4] = 0x4960, [0x19f5] = 0x943e, [0x19f6] = 0x4962, [0x19f7] = 0x28cb8,
  [0x19f8] = 0x28dc5, [0x19f9] = 0x49b2, [0x19fa] = 0x49f0, [0x19fb] = 0x29048,
  [0x19fc] = 0x2917d, [0x19fd] = 0x2917c, [0x19fe] = 0x29181, [0x19ff] = 0x29182,
  [0x1a00] = 0x29162, [0x1a01] = 0x4a4c, [0x1a02] = 0x29345, [0x1a03] = 0x4a82,
  [0x1a04] = 0x97bc, [0x1a05] = 0x4a81, [0x1a06] = 0x4a9b, [0x1a07] = 0x293e3,
  [0x1a08] = 0x4aa4, [0x1a09] = 0x4aee, [0x1a0a] = 0x4aec, [0x1a0b] = 0x29550,
  [0x1a0c] = 0x4aed, [0x1a0d] = 0x2954f, [0x1a0e] = 0x4af0, [0x1a0f] = 0x4aef,
  [0x1a10] = 0x29639, [0x1a11] = 0x4b1d, [0x1a12] = 0x2963a, [0x1a13] = 0x4b60,
  [0x1a14] = 0x4b5e, [0x1a15] = 0x4b5d, [0x1a16] = 0x297c1, [0x1a17] = 0x2994e,
  [0x1a18] = 0x2996e, [0x1a19] = 0x4bb1, [0x1a1a] = 0x4bab, [0x1a1b] = 0x4bac,
  [0x1a1c] = 0x4bad, [0x1a1d] = 0x29971, [0x1a1e] = 0x4bae, [0x1a1f] = 0x29a8c,
  [0x1a20] = 0x29a89, [0x1a21] = 0x4be2, [0x1a22] = 0x29ae5, [0x1a23] = 0x29bb3,
  [0x1a24] = 0x29bb6, [0x1a25] = 0x29bb4, [0x1a26] = 0x29c4d, [0x1a27] = 0x9b39,
  [0x1a28] = 0x29d13, [0x1a29] = 0x29e78, [0x1a2a] = 0x9c2a, [0x1a2b] = 0x4c7b,
  [0x1a2c] = 0x9c26, [0x1a2d] = 0x4c78, [0x1a2e] = 0x4c75, [0x1a2f] = 0x9c27,
  [0x1a30] = 0x29e72, [0x1a31] = 0x4cf2, [0x1a32] = 0x4cf4, [0x1a33] = 0x4cf3,
  [0x1a34] = 0x9dc0, [0x1a35] = 0x9dc9, [0x1a36] = 0x2a275, [0x1a37] = 0x2a2da,
  [0x1a38] = 0x2a2d7, [0x1a39] = 0x4d3f, [0x1a3a] = 0x4d3e, [0x1a3b] = 0x4d40,
  [0x1a3c] = 0x4d4e, [0x1a3d] = 0x4d57, [0x1a3e] = 0x4d59, [0x1a3f] = 0x4d58,
  [0x1a40] = 0x2fa16, [0x1a41] = 0x2a469, [0x1a42] = 0x2a467, [0x1a43] = 0x4d6e,
  [0x1a44] = 0x2a466, [0x1a45] = 0x2a46f, [0x1a46] = 0x2a471, [0x1a47] = 0x9eec,
  [0x1a48] = 0x2a4cf, [0x1a49] = 0x2a523, [0x1a4a] = 0x4d81, [0x1a4b] = 0x4d86,
  [0x1a4c] = 0x2a5a5, [0x1a4d] = 0x4d8f, [0x1a4e] = 0x2a5fe, [0x1a4f] = 0x2a5fb,
  [0x1a50] = 0x2a5fd, [0x1a51] = 0x9f68, [0x1a52] = 0x4d9b, [0x1a53] = 0x4db1,
  [0x1a54] = 0x4db3, [0x1a55] = 0x2116d, [0x1a56] = 0x373a, [0x1a57] = 0x21b05,
  [0x1a58] = 0x21c20, [0x1a59] = 0x3827, [0x1a5a] = 0x21fc8, [0x1a5b] = 0x21fc9,
  [0x1a5c] = 0x386a, [0x1a5d] = 0x39ac, [0x1a5e] = 0x23127, [0x1a5f] = 0x3c18,
  [0x1a60] = 0x2381e, [0x1a61] = 0x23936, [0x1a62] = 0x3c4c, [0x1a63] = 0x2413d,
  [0x1a64] = 0x3d96, [0x1a65] = 0x248d7, [0x1a66] = 0x3f4a, [0x1a67] = 0x24bad,
  [0x1a68] = 0x24d00, [0x1a69] = 0x4081, [0x1a6a] = 0x2533a, [0x1a6b] = 0x4083,
  [0x1a6c] = 0x40f9, [0x1a6d] = 0x40f8, [0x1a6e] = 0x25726, [0x1a6f] = 0x418e,
  [0x1a70] = 0x418f, [0x1a71] = 0x41c1, [0x1a72] = 0x25db7, [0x1a73] = 0x25db6,
  [0x1a74] = 0x25dc0, [0x1a75] = 0x4270, [0x1a76] = 0x25d9b, [0x1a77] = 0x4271,
  [0x1a78] = 0x261d8, [0x1a79] = 0x432a, [0x1a7a] = 0x432d, [0x1a7b] = 0x437d,
  [0x1a7c] = 0x8032, [0x1a7d] = 0x8031, [0x1a7e] = 0x268c0, [0x1a7f] = 0x444f,
  [0x1a80] = 0x268bf, [0x1a81] = 0x26abf, [0x1a82] = 0x4490, [0x1a83] = 0x27120,
  [0x1a84] = 0x270d0, [0x1a85] = 0x2710e, [0x1a86] = 0x4579, [0x1a87] = 0x2711d,
  [0x1a88] = 0x2711e, [0x1a89] = 0x27115, [0x1a8a] = 0x27114, [0x1a8b] = 0x2751e,
  [0x1a8c] = 0x2752c, [0x1a8d] = 0x4605, [0x1a8e] = 0x277b9, [0x1a8f] = 0x277b8,
  [0x1a90] = 0x277b6, [0x1a91] = 0x2789e, [0x1a92] = 0x2789c, [0x1a93] = 0x2794d,
  [0x1a94] = 0x89fd, [0x1a95] = 0x27b6e, [0x1a96] = 0x27b60, [0x1a97] = 0x4721,
  [0x1a98] = 0x27b64, [0x1a99] = 0x27b62, [0x1a9a] = 0x261e5, [0x1a9b] = 0x4732,
  [0x1a9c] = 0x27fa3, [0x1a9d] = 0x47cd, [0x1a9e] = 0x47cf, [0x1a9f] = 0x283d2,
  [0x1aa0] = 0x283d5, [0x1aa1] = 0x908e, [0x1aa2] = 0x4916, [0x1aa3] = 0x4915,
  [0x1aa4] = 0x49b5, [0x1aa5] = 0x4a08, [0x1aa6] = 0x29055, [0x1aa7] = 0x4a32,
  [0x1aa8] = 0x29193, [0x1aa9] = 0x4a33, [0x1aaa] = 0x4a34, [0x1aab] = 0x4a3c,
  [0x1aac] = 0x29356, [0x1aad] = 0x97c2, [0x1aae] = 0x293e9, [0x1aaf] = 0x4a9c,
  [0x1ab0] = 0x29445, [0x1ab1] = 0x4af4, [0x1ab2] = 0x4af2, [0x1ab3] = 0x2967c,
  [0x1ab4] = 0x4b62, [0x1ab5] = 0x297d3, [0x1ab6] = 0x4b61, [0x1ab7] = 0x4b64,
  [0x1ab8] = 0x4bb5, [0x1ab9] = 0x9a4b, [0x1aba] = 0x4bb4, [0x1abb] = 0x2998e,
  [0x1abc] = 0x29a97, [0x1abd] = 0x4be3, [0x1abe] = 0x29a9b, [0x1abf] = 0x29a99,
  [0x1ac0] = 0x9b1c, [0x1ac1] = 0x4c0e, [0x1ac2] = 0x29bcf, [0x1ac3] = 0x9b1b,
  [0x1ac4] = 0x29c59, [0x1ac5] = 0x4c2c, [0x1ac6] = 0x4c2b, [0x1ac7] = 0x29d20,
  [0x1ac8] = 0x29d23, [0x1ac9] = 0x29d2a, [0x1aca] = 0x4c85, [0x1acb] = 0x4c81,
  [0x1acc] = 0x4c7e, [0x1acd] = 0x4c83, [0x1ace] = 0x4c80, [0x1acf] = 0x29eb0,
  [0x1ad0] = 0x9c42, [0x1ad1] = 0x2a12f, [0x1ad2] = 0x9dd4, [0x1ad3] = 0x4cfb,
  [0x1ad4] = 0x4cf7, [0x1ad5] = 0x2a132, [0x1ad6] = 0x2a143, [0x1ad7] = 0x2a13f,
  [0x1ad8] = 0x2a139, [0x1ad9] = 0x4cf8, [0x1ada] = 0x2a130, [0x1adb] = 0x2a2dd,
  [0x1adc] = 0x2a3da, [0x1add] = 0x2a3db, [0x1ade] = 0x4d5a, [0x1adf] = 0x2a484,
  [0x1ae0] = 0x2a47f, [0x1ae1] = 0x2a472, [0x1ae2] = 0x2a480, [0x1ae3] = 0x2a4ee,
  [0x1ae4] = 0x4d78, [0x1ae5] = 0x2a52a, [0x1ae6] = 0x2a522, [0x1ae7] = 0x2a571,
  [0x1ae8] = 0x2a5cd, [0x1ae9] = 0x4d9d, [0x1aea] = 0x4d9c, [0x1aeb] = 0x2a60f,
  [0x1aec] = 0x2a618, [0x1aed] = 0x2046a, [0x1aee] = 0x34a9, [0x1aef] = 0x34bf,
  [0x1af0] = 0x56d0, [0x1af1] = 0x56cf, [0x1af2] = 0x21b0c, [0x1af3] = 0x5dda,
  [0x1af4] = 0x225a6, [0x1af5] = 0x3a77, [0x1af6] = 0x3a76, [0x1af7] = 0x23037,
  [0x1af8] = 0x3abb, [0x1af9] = 0x66ea, [0x1afa] = 0x23ae2, [0x1afb] = 0x3d9b,
  [0x1afc] = 0x244bc, [0x1afd] = 0x3e0f, [0x1afe] = 0x3e5b, [0x1aff] = 0x24ad5,
  [0x1b00] = 0x3f4c, [0x1b01] = 0x3f6f, [0x1b02] = 0x3fd9, [0x1b03] = 0x24f12,
  [0x1b04] = 0x4082, [0x1b05] = 0x2534b, [0x1b06] = 0x25341, [0x1b07] = 0x253a1,
  [0x1b08] = 0x255ec, [0x1b09] = 0x4274, [0x1b0a] = 0x4272, [0x1b0b] = 0x25dd4,
  [0x1b0c] = 0x25dd8, [0x1b0d] = 0x25dd9, [0x1b0e] = 0x4273, [0x1b0f] = 0x25dda,
  [0x1b10] = 0x25f6c, [0x1b11] = 0x25f6d, [0x1b12] = 0x42b1, [0x1b13] = 0x432e,
  [0x1b14] = 0x261e7, [0x1b15] = 0x261eb, [0x1b16] = 0x261ec, [0x1b17] = 0x26262,
  [0x1b18] = 0x434e, [0x1b19] = 0x2652b, [0x1b1a] = 0x2660d, [0x1b1b] = 0x268cd,
  [0x1b1c] = 0x26ac3, [0x1b1d] = 0x2713f, [0x1b1e] = 0x2713c, [0x1b1f] = 0x2713e,
  [0x1b20] = 0x2713d, [0x1b21] = 0x2713a, [0x1b22] = 0x27138, [0x1b23] = 0x27544,
  [0x1b24] = 0x27545, [0x1b25] = 0x460b, [0x1b26] = 0x27548, [0x1b27] = 0x27550,
  [0x1b28] = 0x466c, [0x1b29] = 0x8b89, [0x1b2a] = 0x27b78, [0x1b2b] = 0x27b79,
  [0x1b2c] = 0x478b, [0x1b2d] = 0x27e3e, [0x1b2e] = 0x47d0, [0x1b2f] = 0x482d,
  [0x1b30] = 0x28654, [0x1b31] = 0x48e4, [0x1b32] = 0x4971, [0x1b33] = 0x28bb9,
  [0x1b34] = 0x9458, [0x1b35] = 0x496f, [0x1b36] = 0x2905f, [0x1b37] = 0x4a87,
  [0x1b38] = 0x4aa5, [0x1b39] = 0x29572, [0x1b3a] = 0x29575, [0x1b3b] = 0x4b1e,
  [0x1b3c] = 0x4b65, [0x1b3d] = 0x4bb9, [0x1b3e] = 0x4bb7, [0x1b3f] = 0x4bb8,
  [0x1b40] = 0x4be4, [0x1b41] = 0x29aa3, [0x1b42] = 0x29aa5, [0x1b43] = 0x29bdc,
  [0x1b44] = 0x29bdd, [0x1b45] = 0x29c5a, [0x1b46] = 0x4c8c, [0x1b47] = 0x4c89,
  [0x1b48] = 0x4c8a, [0x1b49] = 0x29edb, [0x1b4a] = 0x29edc, [0x1b4b] = 0x4c8b,
  [0x1b4c] = 0x2a1ab, [0x1b4d] = 0x2a184, [0x1b4e] = 0x2a176, [0x1b4f] = 0x4d01,
  [0x1b50] = 0x4cfe, [0x1b51] = 0x9de7, [0x1b52] = 0x4d03, [0x1b53] = 0x4d06,
  [0x1b54] = 0x2a183, [0x1b55] = 0x9dea, [0x1b56] = 0x9df1, [0x1b57] = 0x2a27f,
  [0x1b58] = 0x4d1d, [0x1b59] = 0x4d43, [0x1b5a] = 0x2a373, [0x1b5b] = 0x2a3ad,
  [0x1b5c] = 0x2a3b0, [0x1b5d] = 0x4d4f, [0x1b5e] = 0x2a40f, [0x1b5f] = 0x2a40c,
  [0x1b60] = 0x4d5b, [0x1b61] = 0x4d70, [0x1b62] = 0x2a579, [0x1b63] = 0x4d88,
  [0x1b64] = 0x2a577, [0x1b65] = 0x2a57a, [0x1b66] = 0x4d89, [0x1b67] = 0x9f44,
  [0x1b68] = 0x2a632, [0x1b69] = 0x2a627, [0x1b6a] = 0x2a62a, [0x1b6b] = 0x2a62c,
  [0x1b6c] = 0x9f6d, [0x1b6d] = 0x2a628, [0x1b6e] = 0x2a629, [0x1b6f] = 0x2a638,
  [0x1b70] = 0x2082f, [0x1b71] = 0x2117b, [0x1b72] = 0x21b0d, [0x1b73] = 0x5dd9,
  [0x1b74] = 0x21fd6, [0x1b75] = 0x21fd5, [0x1b76] = 0x22ea1, [0x1b77] = 0x2384c,
  [0x1b78] = 0x3d9e, [0x1b79] = 0x3d9f, [0x1b7a] = 0x3ea7, [0x1b7b] = 0x3f4b,
  [0x1b7c] = 0x3fdb, [0x1b7d] = 0x3fda, [0x1b7e] = 0x24fc0, [0x1b7f] = 0x77d6,
  [0x1b80] = 0x408e, [0x1b81] = 0x4276, [0x1b82] = 0x25df4, [0x1b83] = 0x4330,
  [0x1b84] = 0x432f, [0x1b85] = 0x261f0, [0x1b86] = 0x4366, [0x1b87] = 0x2633f,
  [0x1b88] = 0x457e, [0x1b89] = 0x2755d, [0x1b8a] = 0x27572, [0x1b8b] = 0x27562,
  [0x1b8c] = 0x883a, [0x1b8d] = 0x27566, [0x1b8e] = 0x8975, [0x1b8f] = 0x466f,
  [0x1b90] = 0x27b88, [0x1b91] = 0x47d1, [0x1b92] = 0x482f, [0x1b93] = 0x281e8,
  [0x1b94] = 0x281e4, [0x1b95] = 0x48b2, [0x1b96] = 0x4918, [0x1b97] = 0x4917,
  [0x1b98] = 0x288ff, [0x1b99] = 0x4976, [0x1b9a] = 0x291af, [0x1b9b] = 0x291ae,
  [0x1b9c] = 0x4a4f, [0x1b9d] = 0x4a89, [0x1b9e] = 0x293f2, [0x1b9f] = 0x29448,
  [0x1ba0] = 0x29581, [0x1ba1] = 0x2957e, [0x1ba2] = 0x4af5, [0x1ba3] = 0x4b1f,
  [0x1ba4] = 0x29652, [0x1ba5] = 0x297ef, [0x1ba6] = 0x9a5d, [0x1ba7] = 0x4be5,
  [0x1ba8] = 0x29aad, [0x1ba9] = 0x29be6, [0x1baa] = 0x4c10, [0x1bab] = 0x29bed,
  [0x1bac] = 0x4c0f, [0x1bad] = 0x29be9, [0x1bae] = 0x29c61, [0x1baf] = 0x29c60,
  [0x1bb0] = 0x29d33, [0x1bb1] = 0x4c2f, [0x1bb2] = 0x4c30, [0x1bb3] = 0x9c64,
  [0x1bb4] = 0x29f0b, [0x1bb5] = 0x29f08, [0x1bb6] = 0x4c93, [0x1bb7] = 0x4c94,
  [0x1bb8] = 0x29f07, [0x1bb9] = 0x4d07, [0x1bba] = 0x4d09, [0x1bbb] = 0x4d08,
  [0x1bbc] = 0x2a1ca, [0x1bbd] = 0x4d0b, [0x1bbe] = 0x2a1c6, [0x1bbf] = 0x9e0a,
  [0x1bc0] = 0x2a284, [0x1bc1] = 0x2a2eb, [0x1bc2] = 0x2a37d, [0x1bc3] = 0x4d50,
  [0x1bc4] = 0x4d71, [0x1bc5] = 0x2a49b, [0x1bc6] = 0x2a4a2, [0x1bc7] = 0x2a4a1,
  [0x1bc8] = 0x2a4a0, [0x1bc9] = 0x2a49c, [0x1bca] = 0x4d7b, [0x1bcb] = 0x4d7c,
  [0x1bcc] = 0x2a580, [0x1bcd] = 0x9f73, [0x1bce] = 0x2a640, [0x1bcf] = 0x4da1,
  [0x1bd0] = 0x2a639, [0x1bd1] = 0x2a63c, [0x1bd2] = 0x4da0, [0x1bd3] = 0x4da2,
  [0x1bd4] = 0x208ca, [0x1bd5] = 0x361b, [0x1bd6] = 0x21189, [0x1bd7] = 0x3682,
  [0x1bd8] = 0x2303c, [0x1bd9] = 0x23940, [0x1bda] = 0x24163, [0x1bdb] = 0x24169,
  [0x1bdc] = 0x3fe9, [0x1bdd] = 0x25353, [0x1bde] = 0x4084, [0x1bdf] = 0x77e1,
  [0x1be0] = 0x25917, [0x1be1] = 0x2591a, [0x1be2] = 0x25e00, [0x1be3] = 0x42b3,
  [0x1be4] = 0x4334, [0x1be5] = 0x4333, [0x1be6] = 0x4580, [0x1be7] = 0x2756f,
  [0x1be8] = 0x278ab, [0x1be9] = 0x46ad, [0x1bea] = 0x27b91, [0x1beb] = 0x4744,
  [0x1bec] = 0x4755, [0x1bed] = 0x27fb1, [0x1bee] = 0x47d2, [0x1bef] = 0x281ef,
  [0x1bf0] = 0x28df1, [0x1bf1] = 0x28df2, [0x1bf2] = 0x291b7, [0x1bf3] = 0x291b5,
  [0x1bf4] = 0x4a8a, [0x1bf5] = 0x29586, [0x1bf6] = 0x2965a, [0x1bf7] = 0x4b67,
  [0x1bf8] = 0x299c6, [0x1bf9] = 0x299cb, [0x1bfa] = 0x4be6, [0x1bfb] = 0x29ab2,
  [0x1bfc] = 0x4c13, [0x1bfd] = 0x29bf3, [0x1bfe] = 0x9b2d, [0x1bff] = 0x29f27,
  [0x1c00] = 0x4c97, [0x1c01] = 0x9e0c, [0x1c02] = 0x2a1d5, [0x1c03] = 0x2a1d8,
  [0x1c04] = 0x4d0c, [0x1c05] = 0x2a1ec, [0x1c06] = 0x2a287, [0x1c07] = 0x2a2f2,
  [0x1c08] = 0x4d46, [0x1c09] = 0x4d5c, [0x1c0a] = 0x4d74, [0x1c0b] = 0x4d72,
  [0x1c0c] = 0x2a4ad, [0x1c0d] = 0x2a4b0, [0x1c0e] = 0x2a4fd, [0x1c0f] = 0x9f1f,
  [0x1c10] = 0x2a587, [0x1c11] = 0x2a64a, [0x1c12] = 0x4da4, [0x1c13] = 0x4da3,
  [0x1c14] = 0x2a64e, [0x1c15] = 0x2a649, [0x1c16] = 0x2a651, [0x1c17] = 0x2a64d,
  [0x1c18] = 0x4db4, [0x1c19] = 0x20472, [0x1c1a] = 0x3536, [0x1c1b] = 0x21b16,
  [0x1c1c] = 0x2386d, [0x1c1d] = 0x23941, [0x1c1e] = 0x3cb2, [0x1c1f] = 0x24171,
  [0x1c20] = 0x24174, [0x1c21] = 0x3f16, [0x1c22] = 0x7c70, [0x1c23] = 0x4277,
  [0x1c24] = 0x25f7c, [0x1c25] = 0x457f, [0x1c26] = 0x2718a, [0x1c27] = 0x27956,
  [0x1c28] = 0x487d, [0x1c29] = 0x9479, [0x1c2a] = 0x28bfa, [0x1c2b] = 0x974a,
  [0x1c2c] = 0x4a8c, [0x1c2d] = 0x2965b, [0x1c2e] = 0x4b68, [0x1c2f] = 0x4bbe,
  [0x1c30] = 0x4c15, [0x1c31] = 0x2a1f5, [0x1c32] = 0x2a1f0, [0x1c33] = 0x2a2f3,
  [0x1c34] = 0x2a37f, [0x1c35] = 0x2a413, [0x1c36] = 0x4d75, [0x1c37] = 0x4da5,
  [0x1c38] = 0x21b19, [0x1c39] = 0x22985, [0x1c3a] = 0x22eb0, [0x1c3b] = 0x24177,
  [0x1c3c] = 0x255ff, [0x1c3d] = 0x4278, [0x1c3e] = 0x4335, [0x1c3f] = 0x7e9d,
  [0x1c40] = 0x4582, [0x1c41] = 0x27187, [0x1c42] = 0x4583, [0x1c43] = 0x2718b,
  [0x1c44] = 0x4671, [0x1c45] = 0x27b9e, [0x1c46] = 0x487e, [0x1c47] = 0x4a8e,
  [0x1c48] = 0x2958f, [0x1c49] = 0x9960, [0x1c4a] = 0x4b69, [0x1c4b] = 0x29839,
  [0x1c4c] = 0x299d4, [0x1c4d] = 0x29af1, [0x1c4e] = 0x29c02, [0x1c4f] = 0x29c6b,
  [0x1c50] = 0x29d40, [0x1c51] = 0x4c9a, [0x1c52] = 0x4c9b, [0x1c53] = 0x2a210,
  [0x1c54] = 0x2a4be, [0x1c55] = 0x2a4b9, [0x1c56] = 0x4d90, [0x1c57] = 0x2a664,
  [0x1c58] = 0x9f9e, [0x1c59] = 0x24ae9, [0x1c5a] = 0x27190, [0x1c5b] = 0x4586,
  [0x1c5c] = 0x4585, [0x1c5d] = 0x2721c, [0x1c5e] = 0x460e, [0x1c5f] = 0x27592,
  [0x1c60] = 0x278ae, [0x1c61] = 0x27ba3, [0x1c62] = 0x4919, [0x1c63] = 0x291c9,
  [0x1c64] = 0x4bc0, [0x1c65] = 0x299d8, [0x1c66] = 0x29d44, [0x1c67] = 0x2a224,
  [0x1c68] = 0x9ef8, [0x1c69] = 0x9f3a, [0x1c6a] = 0x9f7d, [0x1c6b] = 0x2a670,
  [0x1c6c] = 0x2a6d3, [0x1c6d] = 0x400d, [0x1c6e] = 0x4c16, [0x1c6f] = 0x2a4c3,
  [0x1c70] = 0x4da9, [0x1c71] = 0x4daa, [0x1c72] = 0x4085, [0x1c73] = 0x25e21,
  [0x1c74] = 0x26aca, [0x1c75] = 0x2759c, [0x1c76] = 0x29c73, [0x1c77] = 0x2a386,
  [0x1c78] = 0x2a5c1, [0x1c79] = 0x29c09, [0x1c7a] = 0x9f96, [0x1c7b] = 0x2a6d5,
  [0x1c7c] = 0x4bc2, [0x1c7d] = 0x4c31, [0x1c7e] = 0x4d11, [0x1c7f] = 0x4dab,
  [0x1c80] = 0x4c9c, [0x1c81] = 0x291d4,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x5' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x52121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l5_to_ucs4_tab[] =
{
  [0x0000] = 0x200d1, [0x0001] = 0x200cb, [0x0002] = 0x200c9, [0x0003] = 0x2010c,
  [0x0004] = 0x20000, [0x0005] = 0x20087, [0x0006] = 0x2010d, [0x0007] = 0x2010f,
  [0x0008] = 0x2f81d, [0x0009] = 0x22398, [0x000a] = 0x200d2, [0x000b] = 0x20111,
  [0x000c] = 0x2007e, [0x000d] = 0x20ad3, [0x000e] = 0x21552, [0x000f] = 0x21cfe,
  [0x0010] = 0x225a9, [0x0011] = 0x23942, [0x0012] = 0x20009, [0x0013] = 0x200d6,
  [0x0014] = 0x204dd, [0x0015] = 0x204dc, [0x0016] = 0x20502, [0x0017] = 0x20573,
  [0x0018] = 0x20676, [0x0019] = 0x206a7, [0x001a] = 0x206a8, [0x001b] = 0x20833,
  [0x001c] = 0x208cc, [0x001d] = 0x20a0d, [0x001e] = 0x20a2c, [0x001f] = 0x20b1b,
  [0x0020] = 0x355a, [0x0021] = 0x20b1d, [0x0022] = 0x20b99, [0x0023] = 0x2123e,
  [0x0024] = 0x21d2d, [0x0025] = 0x21d2f, [0x0026] = 0x22399, [0x0027] = 0x233b3,
  [0x0028] = 0x239b6, [0x0029] = 0x23b1b, [0x002a] = 0x244ef, [0x002b] = 0x248e9,
  [0x002c] = 0x20014, [0x002d] = 0x20094, [0x002e] = 0x200e1, [0x002f] = 0x20506,
  [0x0030] = 0x2054a, [0x0031] = 0x20544, [0x0032] = 0x20631, [0x0033] = 0x2067a,
  [0x0034] = 0x206ad, [0x0035] = 0x206ae, [0x0036] = 0x206b1, [0x0037] = 0x20838,
  [0x0038] = 0x20837, [0x0039] = 0x2097d, [0x003a] = 0x2097e, [0x003b] = 0x20a2d,
  [0x003c] = 0x20a32, [0x003d] = 0x20b23, [0x003e] = 0x20b26, [0x003f] = 0x2f83a,
  [0x0040] = 0x20ba0, [0x0041] = 0x216a8, [0x0042] = 0x2193f, [0x0043] = 0x219bc,
  [0x0044] = 0x21bc4, [0x0045] = 0x21d3b, [0x0046] = 0x22019, [0x0047] = 0x22018,
  [0x0048] = 0x221b3, [0x0049] = 0x223aa, [0x004a] = 0x224be, [0x004b] = 0x225ac,
  [0x004c] = 0x22a25, [0x004d] = 0x22a66, [0x004e] = 0x22a6b, [0x004f] = 0x23943,
  [0x0050] = 0x239b5, [0x0051] = 0x23aec, [0x0052] = 0x23c76, [0x0053] = 0x3cbc,
  [0x0054] = 0x24614, [0x0055] = 0x215dc, [0x0056] = 0x2471c, [0x0057] = 0x2471d,
  [0x0058] = 0x264d0, [0x0059] = 0x26612, [0x005a] = 0x26af3, [0x005b] = 0x28669,
  [0x005c] = 0x49b9, [0x005d] = 0x28e11, [0x005e] = 0x28e15, [0x005f] = 0x28e10,
  [0x0060] = 0x28e13, [0x0061] = 0x200a4, [0x0062] = 0x200a5, [0x0063] = 0x201d0,
  [0x0064] = 0x201ba, [0x0065] = 0x201cb, [0x0066] = 0x2057b, [0x0067] = 0x20686,
  [0x0068] = 0x206bb, [0x0069] = 0x206ba, [0x006a] = 0x206c3, [0x006b] = 0x206b9,
  [0x006c] = 0x206c0, [0x006d] = 0x34de, [0x006e] = 0x20839, [0x006f] = 0x2083b,
  [0x0070] = 0x20841, [0x0071] = 0x208d1, [0x0072] = 0x208cf, [0x0073] = 0x208d2,
  [0x0074] = 0x2092e, [0x0075] = 0x2096e, [0x0076] = 0x2098c, [0x0077] = 0x20988,
  [0x0078] = 0x20a35, [0x0079] = 0x3543, [0x007a] = 0x20a3a, [0x007b] = 0x20a3b,
  [0x007c] = 0x20a38, [0x007d] = 0x20a39, [0x007e] = 0x20b2e, [0x007f] = 0x20b2c,
  [0x0080] = 0x20bc8, [0x0081] = 0x20bb5, [0x0082] = 0x20bad, [0x0083] = 0x20bb0,
  [0x0084] = 0x20bab, [0x0085] = 0x211b3, [0x0086] = 0x211a9, [0x0087] = 0x216bb,
  [0x0088] = 0x216b9, [0x0089] = 0x21b5a, [0x008a] = 0x21bcd, [0x008b] = 0x21c2b,
  [0x008c] = 0x21d05, [0x008d] = 0x37ac, [0x008e] = 0x37aa, [0x008f] = 0x21d51,
  [0x0090] = 0x21d4b, [0x0091] = 0x21fea, [0x0092] = 0x21fed, [0x0093] = 0x22036,
  [0x0094] = 0x22062, [0x0095] = 0x22056, [0x0096] = 0x22055, [0x0097] = 0x22064,
  [0x0098] = 0x22058, [0x0099] = 0x5e07, [0x009a] = 0x221da, [0x009b] = 0x221d8,
  [0x009c] = 0x223ad, [0x009d] = 0x223ab, [0x009e] = 0x224c0, [0x009f] = 0x224c3,
  [0x00a0] = 0x225b3, [0x00a1] = 0x225b7, [0x00a2] = 0x225ba, [0x00a3] = 0x5fcb,
  [0x00a4] = 0x38fe, [0x00a5] = 0x225c7, [0x00a6] = 0x225b5, [0x00a7] = 0x2298e,
  [0x00a8] = 0x22a73, [0x00a9] = 0x22a6c, [0x00aa] = 0x22a74, [0x00ab] = 0x2f8b4,
  [0x00ac] = 0x22a72, [0x00ad] = 0x22a71, [0x00ae] = 0x22a82, [0x00af] = 0x22a70,
  [0x00b0] = 0x2313f, [0x00b1] = 0x233ba, [0x00b2] = 0x233b8, [0x00b3] = 0x239b7,
  [0x00b4] = 0x24723, [0x00b5] = 0x24721, [0x00b6] = 0x2472b, [0x00b7] = 0x24726,
  [0x00b8] = 0x24722, [0x00b9] = 0x24d25, [0x00ba] = 0x250e4, [0x00bb] = 0x25415,
  [0x00bc] = 0x264e4, [0x00bd] = 0x26633, [0x00be] = 0x2662a, [0x00bf] = 0x26632,
  [0x00c0] = 0x26629, [0x00c1] = 0x26afb, [0x00c2] = 0x2795b, [0x00c3] = 0x2795c,
  [0x00c4] = 0x27c28, [0x00c5] = 0x2866f, [0x00c6] = 0x2866d, [0x00c7] = 0x2866b,
  [0x00c8] = 0x2866e, [0x00c9] = 0x28e16, [0x00ca] = 0x20022, [0x00cb] = 0x20116,
  [0x00cc] = 0x20148, [0x00cd] = 0x201d8, [0x00ce] = 0x20200, [0x00cf] = 0x3441,
  [0x00d0] = 0x20480, [0x00d1] = 0x204e4, [0x00d2] = 0x34b4, [0x00d3] = 0x204e7,
  [0x00d4] = 0x2050c, [0x00d5] = 0x20553, [0x00d6] = 0x205b7, [0x00d7] = 0x20ae8,
  [0x00d8] = 0x206d1, [0x00d9] = 0x206e5, [0x00da] = 0x206d8, [0x00db] = 0x206d6,
  [0x00dc] = 0x206da, [0x00dd] = 0x206d9, [0x00de] = 0x206d5, [0x00df] = 0x206e6,
  [0x00e0] = 0x206d4, [0x00e1] = 0x206d3, [0x00e2] = 0x20844, [0x00e3] = 0x208d6,
  [0x00e4] = 0x208d7, [0x00e5] = 0x20913, [0x00e7] = 0x20970, [0x00e8] = 0x20992,
  [0x00e9] = 0x209db, [0x00ea] = 0x209da, [0x00eb] = 0x20a41, [0x00ec] = 0x20a43,
  [0x00ed] = 0x20b33, [0x00ee] = 0x20bf6, [0x00ef] = 0x20bd4, [0x00f0] = 0x20bdd,
  [0x00f1] = 0x20bd0, [0x00f2] = 0x20c04, [0x00f3] = 0x20bd9, [0x00f4] = 0x20bdb,
  [0x00f5] = 0x20c05, [0x00f6] = 0x20bd3, [0x00f7] = 0x20be1, [0x00f8] = 0x21280,
  [0x00f9] = 0x21277, [0x00fa] = 0x21281, [0x00fb] = 0x21282, [0x00fc] = 0x21283,
  [0x00fd] = 0x2f85c, [0x00fe] = 0x21561, [0x00ff] = 0x21560, [0x0100] = 0x216d8,
  [0x0101] = 0x216d6, [0x0102] = 0x216d7, [0x0103] = 0x24605, [0x0104] = 0x2194d,
  [0x0105] = 0x21948, [0x0106] = 0x21bd6, [0x0107] = 0x21bcf, [0x0108] = 0x21bd9,
  [0x0109] = 0x21bd8, [0x010a] = 0x21c3d, [0x010b] = 0x37b3, [0x010c] = 0x21d54,
  [0x010d] = 0x21d5b, [0x010e] = 0x21d5c, [0x010f] = 0x37b4, [0x0110] = 0x21d59,
  [0x0111] = 0x21d58, [0x0112] = 0x21d5a, [0x0113] = 0x21d55, [0x0114] = 0x21d56,
  [0x0115] = 0x21d82, [0x0116] = 0x21ff0, [0x0117] = 0x2201c, [0x0118] = 0x2206a,
  [0x0119] = 0x22069, [0x011a] = 0x22071, [0x011b] = 0x22067, [0x011c] = 0x2206c,
  [0x011d] = 0x2206e, [0x011e] = 0x22079, [0x011f] = 0x2218b, [0x0120] = 0x221b8,
  [0x0121] = 0x221e7, [0x0122] = 0x221e8, [0x0123] = 0x221e4, [0x0124] = 0x221e6,
  [0x0125] = 0x221e5, [0x0126] = 0x221f0, [0x0127] = 0x22383, [0x0128] = 0x223b7,
  [0x0129] = 0x223bb, [0x012a] = 0x223b9, [0x012b] = 0x223c5, [0x012c] = 0x22453,
  [0x012d] = 0x224c4, [0x012e] = 0x224c6, [0x012f] = 0x224cb, [0x0130] = 0x225c8,
  [0x0131] = 0x2f89f, [0x0132] = 0x225c9, [0x0133] = 0x225db, [0x0134] = 0x22994,
  [0x0135] = 0x22a95, [0x0136] = 0x22a88, [0x0137] = 0x22a8d, [0x0138] = 0x22eec,
  [0x0139] = 0x22eee, [0x013a] = 0x22ef1, [0x013b] = 0x2312e, [0x013c] = 0x23149,
  [0x013d] = 0x2335f, [0x013e] = 0x233d1, [0x013f] = 0x233cb, [0x0140] = 0x23885,
  [0x0141] = 0x3c1d, [0x0142] = 0x2394a, [0x0143] = 0x23947, [0x0144] = 0x2394b,
  [0x0145] = 0x239bb, [0x0146] = 0x239c4, [0x0147] = 0x3c7c, [0x0148] = 0x23b02,
  [0x0149] = 0x23c82, [0x014a] = 0x23c91, [0x014b] = 0x23c8f, [0x014c] = 0x23c92,
  [0x014d] = 0x2418f, [0x014e] = 0x24191, [0x014f] = 0x241b4, [0x0150] = 0x24192,
  [0x0151] = 0x244f4, [0x0152] = 0x245aa, [0x0153] = 0x24735, [0x0154] = 0x24737,
  [0x0155] = 0x2473d, [0x0156] = 0x2473c, [0x0157] = 0x2472f, [0x0158] = 0x24730,
  [0x0159] = 0x248f8, [0x015a] = 0x248ef, [0x015b] = 0x24b26, [0x015c] = 0x3f55,
  [0x015d] = 0x24d26, [0x015e] = 0x24d27, [0x015f] = 0x24f41, [0x0160] = 0x24fc6,
  [0x0161] = 0x24fc7, [0x0162] = 0x250eb, [0x0163] = 0x250e8, [0x0164] = 0x250ea,
  [0x0165] = 0x2541a, [0x0166] = 0x25419, [0x0167] = 0x4126, [0x0168] = 0x25923,
  [0x0169] = 0x25924, [0x016a] = 0x25e25, [0x016b] = 0x26635, [0x016c] = 0x26634,
  [0x016d] = 0x26a07, [0x016e] = 0x2721d, [0x016f] = 0x277e1, [0x0170] = 0x27bab,
  [0x0171] = 0x28460, [0x0172] = 0x28677, [0x0173] = 0x2867d, [0x0174] = 0x49bc,
  [0x0175] = 0x28e1e, [0x0176] = 0x28e1a, [0x0177] = 0x28e1d, [0x0178] = 0x2006b,
  [0x0179] = 0x2012d, [0x017a] = 0x2014c, [0x017b] = 0x2022a, [0x017c] = 0x344a,
  [0x017d] = 0x20215, [0x017e] = 0x2022c, [0x017f] = 0x20206, [0x0180] = 0x20208,
  [0x0181] = 0x2020a, [0x0182] = 0x344e, [0x0183] = 0x20556, [0x0184] = 0x20584,
  [0x0185] = 0x34c9, [0x0186] = 0x205c6, [0x0187] = 0x206f5, [0x0188] = 0x20709,
  [0x0189] = 0x206f8, [0x018a] = 0x206e8, [0x018b] = 0x206fb, [0x018c] = 0x206e9,
  [0x018d] = 0x206f6, [0x018e] = 0x52b7, [0x018f] = 0x52b8, [0x0190] = 0x20852,
  [0x0191] = 0x52b6, [0x0192] = 0x52ba, [0x0193] = 0x20854, [0x0194] = 0x20851,
  [0x0195] = 0x20939, [0x0196] = 0x2093a, [0x0197] = 0x20998, [0x0198] = 0x20a4e,
  [0x0199] = 0x357b, [0x019a] = 0x20c36, [0x019b] = 0x20c0c, [0x019c] = 0x20c10,
  [0x019d] = 0x20c32, [0x019e] = 0x20c11, [0x019f] = 0x20c18, [0x01a0] = 0x20c1c,
  [0x01a1] = 0x20c1a, [0x01a2] = 0x20c19, [0x01a3] = 0x20c0b, [0x01a4] = 0x20c0f,
  [0x01a5] = 0x20c1d, [0x01a6] = 0x20c21, [0x01a7] = 0x20c13, [0x01a8] = 0x20c1b,
  [0x01a9] = 0x20c15, [0x01aa] = 0x3620, [0x01ab] = 0x211d1, [0x01ac] = 0x211c8,
  [0x01ad] = 0x2128d, [0x01ae] = 0x212b5, [0x01af] = 0x3689, [0x01b0] = 0x3695,
  [0x01b1] = 0x215f8, [0x01b2] = 0x216e0, [0x01b3] = 0x216e1, [0x01b4] = 0x36be,
  [0x01b5] = 0x216de, [0x01b6] = 0x216fe, [0x01b7] = 0x216dc, [0x01b8] = 0x21701,
  [0x01b9] = 0x216fd, [0x01ba] = 0x21700, [0x01bb] = 0x219d7, [0x01bc] = 0x219d6,
  [0x01bd] = 0x219d9, [0x01be] = 0x21b65, [0x01bf] = 0x21bd2, [0x01c0] = 0x21bdc,
  [0x01c1] = 0x21be1, [0x01c2] = 0x21bde, [0x01c3] = 0x21c42, [0x01c4] = 0x21c4b,
  [0x01c5] = 0x37c3, [0x01c6] = 0x21d8e, [0x01c7] = 0x21d91, [0x01c8] = 0x21d8b,
  [0x01c9] = 0x21d90, [0x01ca] = 0x21d88, [0x01cb] = 0x21d89, [0x01cc] = 0x21d84,
  [0x01cd] = 0x21daa, [0x01ce] = 0x2208d, [0x01cf] = 0x2208f, [0x01d0] = 0x2207d,
  [0x01d1] = 0x2207f, [0x01d2] = 0x2207b, [0x01d3] = 0x22080, [0x01d4] = 0x2207e,
  [0x01d5] = 0x22083, [0x01d6] = 0x22081, [0x01d7] = 0x22086, [0x01d8] = 0x2208a,
  [0x01d9] = 0x221bd, [0x01da] = 0x221be, [0x01db] = 0x3877, [0x01dc] = 0x221f4,
  [0x01dd] = 0x221f3, [0x01de] = 0x22202, [0x01df] = 0x221f6, [0x01e0] = 0x221f5,
  [0x01e1] = 0x221f2, [0x01e2] = 0x22204, [0x01e3] = 0x2233d, [0x01e4] = 0x223c6,
  [0x01e5] = 0x22489, [0x01e6] = 0x224d4, [0x01e7] = 0x224d6, [0x01e8] = 0x224d2,
  [0x01e9] = 0x225fa, [0x01ea] = 0x22606, [0x01eb] = 0x2260c, [0x01ec] = 0x2260a,
  [0x01ed] = 0x2260f, [0x01ee] = 0x22a2f, [0x01ef] = 0x39d4, [0x01f0] = 0x22ab7,
  [0x01f1] = 0x22abe, [0x01f2] = 0x22abc, [0x01f3] = 0x22ae2, [0x01f4] = 0x22f06,
  [0x01f5] = 0x22ef7, [0x01f6] = 0x22f07, [0x01f7] = 0x22ef6, [0x01f8] = 0x23b35,
  [0x01f9] = 0x2315f, [0x01fa] = 0x23165, [0x01fb] = 0x23164, [0x01fc] = 0x23161,
  [0x01fd] = 0x2315a, [0x01fe] = 0x2315d, [0x01ff] = 0x2f8d7, [0x0200] = 0x233d7,
  [0x0201] = 0x233d5, [0x0202] = 0x233de, [0x0203] = 0x233e1, [0x0204] = 0x233e0,
  [0x0205] = 0x233d9, [0x0206] = 0x3b4c, [0x0207] = 0x233da, [0x0208] = 0x233df,
  [0x0209] = 0x233f6, [0x020a] = 0x23403, [0x020b] = 0x3c20, [0x020c] = 0x23893,
  [0x020d] = 0x2388b, [0x020e] = 0x23891, [0x020f] = 0x23892, [0x0210] = 0x2388a,
  [0x0211] = 0x23889, [0x0212] = 0x2388f, [0x0213] = 0x23956, [0x0214] = 0x23959,
  [0x0215] = 0x239ca, [0x0216] = 0x239cc, [0x0217] = 0x3c5b, [0x0218] = 0x239d0,
  [0x0219] = 0x239cd, [0x021a] = 0x239d6, [0x021b] = 0x239cb, [0x021c] = 0x23a83,
  [0x021d] = 0x23a82, [0x021e] = 0x23b06, [0x021f] = 0x23b2a, [0x0220] = 0x23b2c,
  [0x0221] = 0x23b2b, [0x0222] = 0x23b29, [0x0223] = 0x23cbd, [0x0224] = 0x23cb3,
  [0x0225] = 0x23cbe, [0x0226] = 0x23cc0, [0x0227] = 0x23cb5, [0x0228] = 0x23cb6,
  [0x0229] = 0x2419d, [0x022a] = 0x2419f, [0x022b] = 0x24620, [0x022c] = 0x3e2d,
  [0x022d] = 0x2461e, [0x022e] = 0x2461c, [0x022f] = 0x24632, [0x0230] = 0x2461f,
  [0x0231] = 0x2462b, [0x0232] = 0x24621, [0x0233] = 0x24626, [0x0234] = 0x24624,
  [0x0235] = 0x2461d, [0x0236] = 0x24627, [0x0237] = 0x2475b, [0x0238] = 0x24759,
  [0x0239] = 0x24780, [0x023a] = 0x24755, [0x023b] = 0x24750, [0x023c] = 0x2475a,
  [0x023d] = 0x24756, [0x023e] = 0x24763, [0x023f] = 0x24752, [0x0240] = 0x24758,
  [0x0241] = 0x3e63, [0x0242] = 0x24739, [0x0243] = 0x248f9, [0x0244] = 0x24b28,
  [0x0245] = 0x3f18, [0x0246] = 0x24bbc, [0x0247] = 0x24bd6, [0x0248] = 0x24c05,
  [0x0249] = 0x24c20, [0x024a] = 0x3f74, [0x024b] = 0x24d33, [0x024c] = 0x24f46,
  [0x024d] = 0x24f42, [0x024e] = 0x24f43, [0x024f] = 0x24f47, [0x0250] = 0x24fcb,
  [0x0251] = 0x25042, [0x0252] = 0x25045, [0x0253] = 0x250fd, [0x0254] = 0x250f4,
  [0x0255] = 0x250f9, [0x0256] = 0x250fa, [0x0257] = 0x2535e, [0x0258] = 0x2541d,
  [0x0259] = 0x4128, [0x025a] = 0x25758, [0x025b] = 0x25755, [0x025c] = 0x25a57,
  [0x025d] = 0x25a58, [0x025e] = 0x25ad9, [0x025f] = 0x25adc, [0x0260] = 0x25adb,
  [0x0261] = 0x25e27, [0x0262] = 0x25f85, [0x0263] = 0x25f87, [0x0264] = 0x25f8a,
  [0x0265] = 0x2628b, [0x0266] = 0x2634f, [0x0267] = 0x26530, [0x0268] = 0x2652f,
  [0x0269] = 0x26668, [0x026a] = 0x26649, [0x026b] = 0x43da, [0x026c] = 0x26648,
  [0x026d] = 0x2663f, [0x026e] = 0x2664b, [0x026f] = 0x2665c, [0x0270] = 0x26646,
  [0x0271] = 0x26640, [0x0272] = 0x26a0d, [0x0273] = 0x26b1b, [0x0274] = 0x26b1d,
  [0x0275] = 0x26b17, [0x0276] = 0x44a4, [0x0277] = 0x26b22, [0x0278] = 0x275a7,
  [0x0279] = 0x27608, [0x027a] = 0x27806, [0x027b] = 0x2847d, [0x027c] = 0x488e,
  [0x027d] = 0x28463, [0x027e] = 0x28464, [0x027f] = 0x2847f, [0x0280] = 0x2846b,
  [0x0281] = 0x2847c, [0x0282] = 0x28468, [0x0283] = 0x28465, [0x0284] = 0x2846a,
  [0x0285] = 0x28698, [0x0286] = 0x28699, [0x0287] = 0x286ae, [0x0288] = 0x28e2b,
  [0x0289] = 0x28e2d, [0x028a] = 0x28e31, [0x028b] = 0x28e2e, [0x028c] = 0x28e2f,
  [0x028d] = 0x2006d, [0x028e] = 0x20131, [0x028f] = 0x2022d, [0x0290] = 0x20239,
  [0x0291] = 0x20238, [0x0292] = 0x345b, [0x0293] = 0x2023a, [0x0294] = 0x20235,
  [0x0295] = 0x20262, [0x0296] = 0x204ea, [0x0297] = 0x204eb, [0x0298] = 0x20515,
  [0x0299] = 0x205cc, [0x029a] = 0x205c8, [0x029b] = 0x205cd, [0x029c] = 0x205db,
  [0x029d] = 0x205cb, [0x029e] = 0x20711, [0x029f] = 0x20726, [0x02a0] = 0x20862,
  [0x02a1] = 0x2085e, [0x02a2] = 0x20861, [0x02a3] = 0x20941, [0x02a4] = 0x2093f,
  [0x02a5] = 0x20975, [0x02a6] = 0x23965, [0x02a7] = 0x20b49, [0x02a8] = 0x209eb,
  [0x02a9] = 0x20a1a, [0x02aa] = 0x20a57, [0x02ab] = 0x20c6b, [0x02ac] = 0x20ca9,
  [0x02ad] = 0x20c54, [0x02ae] = 0x20c62, [0x02af] = 0x20caa, [0x02b0] = 0x20c58,
  [0x02b1] = 0x20c72, [0x02b2] = 0x20c63, [0x02b3] = 0x20c73, [0x02b4] = 0x20c8b,
  [0x02b5] = 0x20c8c, [0x02b6] = 0x20c5c, [0x02b7] = 0x20c51, [0x02b8] = 0x20c16,
  [0x02b9] = 0x20c60, [0x02ba] = 0x20c61, [0x02bb] = 0x20c5e, [0x02bc] = 0x20c5d,
  [0x02bd] = 0x20c64, [0x02be] = 0x20c53, [0x02bf] = 0x20c5f, [0x02c0] = 0x20c5b,
  [0x02c1] = 0x20c6e, [0x02c2] = 0x20c55, [0x02c3] = 0x20c52, [0x02c4] = 0x20c68,
  [0x02c5] = 0x211d2, [0x02c6] = 0x211d6, [0x02c7] = 0x211d3, [0x02c8] = 0x212b8,
  [0x02c9] = 0x212bb, [0x02ca] = 0x212eb, [0x02cb] = 0x212e8, [0x02cc] = 0x212b6,
  [0x02cd] = 0x212ea, [0x02ce] = 0x212e7, [0x02cf] = 0x212e9, [0x02d0] = 0x21591,
  [0x02d1] = 0x21610, [0x02d2] = 0x2160e, [0x02d3] = 0x2160f, [0x02d4] = 0x2170b,
  [0x02d5] = 0x21703, [0x02d6] = 0x2172b, [0x02d7] = 0x21709, [0x02d8] = 0x21728,
  [0x02d9] = 0x2172c, [0x02da] = 0x21725, [0x02db] = 0x21727, [0x02dc] = 0x2170f,
  [0x02dd] = 0x219e9, [0x02de] = 0x219ec, [0x02df] = 0x3753, [0x02e0] = 0x21b27,
  [0x02e1] = 0x21b73, [0x02e2] = 0x21be2, [0x02e3] = 0x21be6, [0x02e4] = 0x2f876,
  [0x02e5] = 0x21c61, [0x02e6] = 0x21c51, [0x02e7] = 0x21d12, [0x02e8] = 0x21d0e,
  [0x02e9] = 0x21dad, [0x02ea] = 0x21daf, [0x02eb] = 0x21db1, [0x02ec] = 0x21dd3,
  [0x02ed] = 0x22094, [0x02ee] = 0x22092, [0x02ef] = 0x22090, [0x02f0] = 0x2209d,
  [0x02f1] = 0x22093, [0x02f2] = 0x22095, [0x02f3] = 0x22098, [0x02f4] = 0x22097,
  [0x02f5] = 0x3880, [0x02f6] = 0x2f88a, [0x02f7] = 0x22207, [0x02f8] = 0x2220b,
  [0x02f9] = 0x22208, [0x02fa] = 0x223d9, [0x02fb] = 0x223d5, [0x02fc] = 0x224d7,
  [0x02fd] = 0x224df, [0x02fe] = 0x38dd, [0x02ff] = 0x224de, [0x0300] = 0x224e3,
  [0x0301] = 0x224e2, [0x0302] = 0x38de, [0x0303] = 0x22647, [0x0304] = 0x3922,
  [0x0305] = 0x2263d, [0x0306] = 0x22638, [0x0307] = 0x22a44, [0x0308] = 0x22a3a,
  [0x0309] = 0x22ab8, [0x030a] = 0x22b2a, [0x030b] = 0x22b00, [0x030c] = 0x22af3,
  [0x030d] = 0x22af2, [0x030e] = 0x22aeb, [0x030f] = 0x22afa, [0x0310] = 0x22aed,
  [0x0311] = 0x22aec, [0x0312] = 0x22aff, [0x0313] = 0x22afb, [0x0314] = 0x6306,
  [0x0315] = 0x22b4e, [0x0316] = 0x22aef, [0x0317] = 0x22af7, [0x0318] = 0x22f10,
  [0x0319] = 0x22f11, [0x031a] = 0x22f0c, [0x031b] = 0x22f0b, [0x031c] = 0x230dd,
  [0x031d] = 0x23130, [0x031e] = 0x3adb, [0x031f] = 0x2317a, [0x0320] = 0x23177,
  [0x0321] = 0x23328, [0x0322] = 0x23327, [0x0323] = 0x23365, [0x0324] = 0x2340e,
  [0x0325] = 0x2340b, [0x0326] = 0x23441, [0x0327] = 0x23413, [0x0328] = 0x2340f,
  [0x0329] = 0x23412, [0x032a] = 0x23430, [0x032b] = 0x238a0, [0x032c] = 0x238a3,
  [0x032d] = 0x2389e, [0x032e] = 0x23897, [0x032f] = 0x238a1, [0x0330] = 0x2389d,
  [0x0331] = 0x2389b, [0x0332] = 0x23963, [0x0333] = 0x239e3, [0x0334] = 0x239e4,
  [0x0335] = 0x239e6, [0x0336] = 0x239e7, [0x0337] = 0x239dd, [0x0338] = 0x6b85,
  [0x0339] = 0x239ec, [0x033a] = 0x239e5, [0x033b] = 0x239f1, [0x033c] = 0x23b3b,
  [0x033d] = 0x23b39, [0x033e] = 0x23b3a, [0x033f] = 0x23b3c, [0x0340] = 0x23b3d,
  [0x0341] = 0x23b47, [0x0342] = 0x23b3f, [0x0343] = 0x23b44, [0x0344] = 0x23c48,
  [0x0345] = 0x23c49, [0x0346] = 0x23c60, [0x0347] = 0x23ce3, [0x0348] = 0x23ce9,
  [0x0349] = 0x23cf0, [0x034a] = 0x3cda, [0x034b] = 0x3cdb, [0x034c] = 0x23cf2,
  [0x034d] = 0x23ced, [0x034e] = 0x23cec, [0x034f] = 0x23ce6, [0x0350] = 0x23d16,
  [0x0351] = 0x3cd7, [0x0352] = 0x241bc, [0x0353] = 0x241be, [0x0354] = 0x241c0,
  [0x0355] = 0x241e0, [0x0356] = 0x24500, [0x0357] = 0x24502, [0x0358] = 0x244fe,
  [0x0359] = 0x2455e, [0x035a] = 0x245b3, [0x035b] = 0x245b7, [0x035c] = 0x2463a,
  [0x035d] = 0x24639, [0x035e] = 0x2463e, [0x035f] = 0x24638, [0x0360] = 0x2463d,
  [0x0361] = 0x2464f, [0x0362] = 0x2476e, [0x0363] = 0x2475f, [0x0364] = 0x24781,
  [0x0365] = 0x24782, [0x0366] = 0x24771, [0x0367] = 0x2477b, [0x0368] = 0x24779,
  [0x0369] = 0x24777, [0x036a] = 0x24773, [0x036b] = 0x2476f, [0x036c] = 0x24797,
  [0x036d] = 0x2477e, [0x036e] = 0x248fc, [0x036f] = 0x24911, [0x0370] = 0x24932,
  [0x0371] = 0x24931, [0x0372] = 0x24910, [0x0373] = 0x24aec, [0x0374] = 0x24aeb,
  [0x0375] = 0x24b2c, [0x0376] = 0x24b2d, [0x0377] = 0x24bda, [0x0378] = 0x24c2e,
  [0x0379] = 0x24c2c, [0x037a] = 0x24c2a, [0x037b] = 0x24c33, [0x037c] = 0x24c4b,
  [0x037d] = 0x24d18, [0x037e] = 0x24d42, [0x037f] = 0x24d3b, [0x0380] = 0x24d3f,
  [0x0381] = 0x24d41, [0x0382] = 0x24d43, [0x0383] = 0x24d3c, [0x0384] = 0x24f4c,
  [0x0385] = 0x24f4a, [0x0386] = 0x24f49, [0x0387] = 0x24f56, [0x0388] = 0x24fcf,
  [0x0389] = 0x25047, [0x038a] = 0x25048, [0x038b] = 0x25046, [0x038c] = 0x2511b,
  [0x038d] = 0x25111, [0x038e] = 0x25114, [0x038f] = 0x2511d, [0x0390] = 0x25117,
  [0x0391] = 0x2511e, [0x0392] = 0x2510b, [0x0393] = 0x2511c, [0x0394] = 0x25112,
  [0x0395] = 0x25116, [0x0396] = 0x2510d, [0x0397] = 0x25115, [0x0398] = 0x25113,
  [0x0399] = 0x25118, [0x039a] = 0x401a, [0x039b] = 0x253ac, [0x039c] = 0x25444,
  [0x039d] = 0x2542c, [0x039e] = 0x25445, [0x039f] = 0x25766, [0x03a0] = 0x2576e,
  [0x03a1] = 0x25765, [0x03a2] = 0x25768, [0x03a3] = 0x25784, [0x03a4] = 0x25938,
  [0x03a5] = 0x2593b, [0x03a6] = 0x2593a, [0x03a7] = 0x2593f, [0x03a8] = 0x25adf,
  [0x03a9] = 0x25add, [0x03aa] = 0x25ade, [0x03ab] = 0x25ae2, [0x03ac] = 0x41d6,
  [0x03ad] = 0x25e2c, [0x03ae] = 0x25e2f, [0x03af] = 0x25f8f, [0x03b0] = 0x25f90,
  [0x03b1] = 0x25f8c, [0x03b2] = 0x25f93, [0x03b3] = 0x26225, [0x03b4] = 0x26288,
  [0x03b5] = 0x262a7, [0x03b6] = 0x263f7, [0x03b7] = 0x264bb, [0x03b8] = 0x26538,
  [0x03b9] = 0x26539, [0x03ba] = 0x26537, [0x03bb] = 0x26614, [0x03bc] = 0x2666e,
  [0x03bd] = 0x2667b, [0x03be] = 0x43de, [0x03bf] = 0x2666c, [0x03c0] = 0x2667f,
  [0x03c1] = 0x26678, [0x03c2] = 0x2667a, [0x03c3] = 0x2666f, [0x03c4] = 0x43e5,
  [0x03c5] = 0x268fe, [0x03c6] = 0x26b48, [0x03c7] = 0x26b4b, [0x03c8] = 0x26b41,
  [0x03c9] = 0x26b49, [0x03ca] = 0x26b3b, [0x03cb] = 0x2722d, [0x03cc] = 0x275e3,
  [0x03cd] = 0x2761a, [0x03ce] = 0x2760d, [0x03cf] = 0x27613, [0x03d0] = 0x27611,
  [0x03d1] = 0x2761c, [0x03d2] = 0x4672, [0x03d3] = 0x278b5, [0x03d4] = 0x46af,
  [0x03d5] = 0x27c29, [0x03d6] = 0x27ca1, [0x03d7] = 0x27d24, [0x03d8] = 0x27fbb,
  [0x03d9] = 0x282a1, [0x03da] = 0x28483, [0x03db] = 0x2848d, [0x03dc] = 0x28487,
  [0x03dd] = 0x28485, [0x03de] = 0x2848a, [0x03df] = 0x284a3, [0x03e0] = 0x28489,
  [0x03e1] = 0x28484, [0x03e2] = 0x284a2, [0x03e3] = 0x286b1, [0x03e4] = 0x286b0,
  [0x03e5] = 0x286af, [0x03e6] = 0x286b6, [0x03e7] = 0x286c9, [0x03e8] = 0x286ca,
  [0x03e9] = 0x286c8, [0x03ea] = 0x286b4, [0x03eb] = 0x28c59, [0x03ec] = 0x28cc9,
  [0x03ed] = 0x28e41, [0x03ee] = 0x28e4c, [0x03ef] = 0x49c4, [0x03f0] = 0x28e45,
  [0x03f1] = 0x28e5a, [0x03f2] = 0x200b9, [0x03f3] = 0x20294, [0x03f4] = 0x2026e,
  [0x03f5] = 0x202a2, [0x03f6] = 0x20267, [0x03f7] = 0x3463, [0x03f8] = 0x202a3,
  [0x03f9] = 0x20276, [0x03fa] = 0x2026c, [0x03fb] = 0x20264, [0x03fc] = 0x202a5,
  [0x03fd] = 0x20230, [0x03fe] = 0x202a6, [0x03ff] = 0x20269, [0x0400] = 0x202a4,
  [0x0401] = 0x20490, [0x0402] = 0x2055f, [0x0403] = 0x20560, [0x0404] = 0x205da,
  [0x0405] = 0x205d8, [0x0406] = 0x205dd, [0x0407] = 0x34ec, [0x0408] = 0x20759,
  [0x0409] = 0x2075b, [0x040a] = 0x2075a, [0x040b] = 0x20739, [0x040c] = 0x20734,
  [0x040d] = 0x20744, [0x040e] = 0x20733, [0x040f] = 0x2073c, [0x0410] = 0x20758,
  [0x0411] = 0x2076a, [0x0412] = 0x2086f, [0x0413] = 0x20872, [0x0414] = 0x20870,
  [0x0415] = 0x20871, [0x0416] = 0x2086e, [0x0417] = 0x208f0, [0x0418] = 0x208eb,
  [0x0419] = 0x20949, [0x041a] = 0x2094c, [0x041b] = 0x209f5, [0x041c] = 0x20a20,
  [0x041d] = 0x20a6a, [0x041e] = 0x20a72, [0x041f] = 0x20b4b, [0x0420] = 0x20b5d,
  [0x0421] = 0x20c50, [0x0422] = 0x20cf9, [0x0423] = 0x20cfa, [0x0424] = 0x20cfb,
  [0x0425] = 0x20d23, [0x0426] = 0x20cb7, [0x0427] = 0x20d22, [0x0428] = 0x20cb5,
  [0x0429] = 0x20cc4, [0x042a] = 0x20cb4, [0x042b] = 0x20cef, [0x042c] = 0x3598,
  [0x042d] = 0x20ccd, [0x042e] = 0x20cf0, [0x042f] = 0x20cb8, [0x0430] = 0x20ccb,
  [0x0431] = 0x20cf1, [0x0432] = 0x20cba, [0x0433] = 0x20d03, [0x0434] = 0x20cf8,
  [0x0435] = 0x20cb6, [0x0436] = 0x20cbc, [0x0437] = 0x20cb3, [0x0438] = 0x20d21,
  [0x0439] = 0x20cf7, [0x043a] = 0x20cc2, [0x043b] = 0x20cbe, [0x043c] = 0x20cc9,
  [0x043d] = 0x20cbd, [0x043e] = 0x20cce, [0x043f] = 0x20cb9, [0x0440] = 0x20cc7,
  [0x0441] = 0x20cca, [0x0442] = 0x20ccc, [0x0443] = 0x20d26, [0x0444] = 0x211e4,
  [0x0445] = 0x211e0, [0x0446] = 0x211f0, [0x0447] = 0x212ec, [0x0448] = 0x21324,
  [0x0449] = 0x21318, [0x044a] = 0x2132f, [0x044b] = 0x21325, [0x044c] = 0x21528,
  [0x044d] = 0x21522, [0x044e] = 0x2156a, [0x044f] = 0x20157, [0x0450] = 0x21594,
  [0x0451] = 0x21592, [0x0452] = 0x2159c, [0x0453] = 0x21622, [0x0454] = 0x21630,
  [0x0455] = 0x2161d, [0x0456] = 0x21623, [0x0457] = 0x21735, [0x0458] = 0x21733,
  [0x0459] = 0x2172f, [0x045a] = 0x2174d, [0x045b] = 0x21731, [0x045c] = 0x2175b,
  [0x045d] = 0x2175a, [0x045e] = 0x21732, [0x045f] = 0x2175c, [0x0460] = 0x21759,
  [0x0461] = 0x2175d, [0x0462] = 0x21969, [0x0463] = 0x2196d, [0x0464] = 0x22232,
  [0x0465] = 0x3758, [0x0466] = 0x21a0a, [0x0467] = 0x21a20, [0x0468] = 0x21a1f,
  [0x0469] = 0x21a0d, [0x046a] = 0x21be8, [0x046b] = 0x21bec, [0x046c] = 0x21c77,
  [0x046d] = 0x21c70, [0x046e] = 0x21c71, [0x046f] = 0x21df8, [0x0470] = 0x21dd6,
  [0x0471] = 0x21dd9, [0x0472] = 0x21dde, [0x0473] = 0x21dd5, [0x0474] = 0x21de1,
  [0x0475] = 0x37d3, [0x0476] = 0x21de2, [0x0477] = 0x21ddd, [0x0478] = 0x21de0,
  [0x0479] = 0x21e0e, [0x047a] = 0x21e08, [0x047b] = 0x21ddc, [0x047c] = 0x21e0a,
  [0x047d] = 0x21e0c, [0x047e] = 0x220b4, [0x047f] = 0x220b1, [0x0480] = 0x220b6,
  [0x0481] = 0x220bc, [0x0482] = 0x2221d, [0x0483] = 0x22224, [0x0484] = 0x22219,
  [0x0485] = 0x2221b, [0x0486] = 0x22222, [0x0487] = 0x2221a, [0x0488] = 0x2221c,
  [0x0489] = 0x22221, [0x048a] = 0x223e6, [0x048b] = 0x223e4, [0x048c] = 0x223e7,
  [0x048d] = 0x223ea, [0x048e] = 0x38e2, [0x048f] = 0x2f89b, [0x0490] = 0x224f0,
  [0x0491] = 0x224f5, [0x0492] = 0x224ef, [0x0493] = 0x22642, [0x0494] = 0x2266e,
  [0x0495] = 0x2263c, [0x0496] = 0x22697, [0x0497] = 0x2268c, [0x0498] = 0x22681,
  [0x0499] = 0x22671, [0x049a] = 0x226ce, [0x049b] = 0x226ba, [0x049c] = 0x2267a,
  [0x049d] = 0x2267e, [0x049e] = 0x3954, [0x049f] = 0x22672, [0x04a0] = 0x226bb,
  [0x04a1] = 0x392f, [0x04a2] = 0x22673, [0x04a3] = 0x22682, [0x04a4] = 0x39b6,
  [0x04a5] = 0x229ba, [0x04a6] = 0x229b8, [0x04a7] = 0x22a4b, [0x04a8] = 0x22a48,
  [0x04a9] = 0x22b2b, [0x04aa] = 0x22af4, [0x04ab] = 0x22b2c, [0x04ac] = 0x22b42,
  [0x04ad] = 0x22b48, [0x04ae] = 0x22b44, [0x04af] = 0x22b45, [0x04b0] = 0x22b3c,
  [0x04b1] = 0x22b37, [0x04b2] = 0x22b33, [0x04b3] = 0x22b41, [0x04b4] = 0x22b32,
  [0x04b5] = 0x22b87, [0x04b6] = 0x22f27, [0x04b7] = 0x22f23, [0x04b8] = 0x22f2d,
  [0x04b9] = 0x22f1f, [0x04ba] = 0x22f2c, [0x04bb] = 0x22f28, [0x04bc] = 0x23075,
  [0x04bd] = 0x23074, [0x04be] = 0x230e7, [0x04bf] = 0x23132, [0x04c0] = 0x23131,
  [0x04c1] = 0x23199, [0x04c2] = 0x23196, [0x04c3] = 0x23198, [0x04c4] = 0x2319d,
  [0x04c5] = 0x23192, [0x04c6] = 0x23194, [0x04c7] = 0x23195, [0x04c8] = 0x23197,
  [0x04c9] = 0x266b9, [0x04ca] = 0x3b35, [0x04cb] = 0x23411, [0x04cc] = 0x2343b,
  [0x04cd] = 0x23479, [0x04ce] = 0x23442, [0x04cf] = 0x23443, [0x04d0] = 0x3b5d,
  [0x04d1] = 0x23478, [0x04d2] = 0x238ac, [0x04d3] = 0x3c29, [0x04d4] = 0x238b1,
  [0x04d5] = 0x238b4, [0x04d6] = 0x238b3, [0x04d7] = 0x238af, [0x04d8] = 0x238aa,
  [0x04d9] = 0x238b2, [0x04da] = 0x23968, [0x04db] = 0x239f2, [0x04dc] = 0x239fe,
  [0x04dd] = 0x239f8, [0x04de] = 0x239f9, [0x04df] = 0x239ff, [0x04e0] = 0x239f5,
  [0x04e1] = 0x239f7, [0x04e2] = 0x239fd, [0x04e3] = 0x23a00, [0x04e4] = 0x23a88,
  [0x04e5] = 0x23b6e, [0x04e6] = 0x23b5d, [0x04e7] = 0x23b63, [0x04e8] = 0x23b60,
  [0x04e9] = 0x23c61, [0x04ea] = 0x23d37, [0x04eb] = 0x23d71, [0x04ec] = 0x23d23,
  [0x04ed] = 0x23d22, [0x04ee] = 0x23d1f, [0x04ef] = 0x23d25, [0x04f0] = 0x241fd,
  [0x04f1] = 0x24211, [0x04f2] = 0x24223, [0x04f3] = 0x241f0, [0x04f4] = 0x241ef,
  [0x04f5] = 0x2450e, [0x04f6] = 0x2450c, [0x04f7] = 0x245ba, [0x04f8] = 0x3e1f,
  [0x04f9] = 0x24652, [0x04fa] = 0x24655, [0x04fb] = 0x24653, [0x04fc] = 0x24654,
  [0x04fd] = 0x24651, [0x04fe] = 0x24658, [0x04ff] = 0x247b0, [0x0500] = 0x2479f,
  [0x0501] = 0x3e72, [0x0502] = 0x247a1, [0x0503] = 0x2479a, [0x0504] = 0x247be,
  [0x0505] = 0x247a0, [0x0506] = 0x24937, [0x0507] = 0x3f0b, [0x0508] = 0x24af1,
  [0x0509] = 0x24af2, [0x050a] = 0x24b34, [0x050b] = 0x24b37, [0x050c] = 0x24b35,
  [0x050d] = 0x2238c, [0x050e] = 0x24bdf, [0x050f] = 0x24c4c, [0x0510] = 0x24c4e,
  [0x0511] = 0x24c64, [0x0512] = 0x24c63, [0x0513] = 0x24c36, [0x0514] = 0x24d58,
  [0x0515] = 0x24d68, [0x0516] = 0x24d5b, [0x0517] = 0x24d5f, [0x0518] = 0x24d59,
  [0x0519] = 0x24d65, [0x051a] = 0x24d60, [0x051b] = 0x24d5e, [0x051c] = 0x24fd5,
  [0x051d] = 0x24fd6, [0x051e] = 0x2504e, [0x051f] = 0x25052, [0x0520] = 0x25135,
  [0x0521] = 0x25139, [0x0522] = 0x25158, [0x0523] = 0x25141, [0x0524] = 0x25157,
  [0x0525] = 0x25138, [0x0526] = 0x2513d, [0x0527] = 0x25132, [0x0528] = 0x25144,
  [0x0529] = 0x2f944, [0x052a] = 0x25363, [0x052b] = 0x25362, [0x052c] = 0x253b8,
  [0x052d] = 0x253b0, [0x052e] = 0x253b1, [0x052f] = 0x2544d, [0x0530] = 0x25463,
  [0x0531] = 0x25470, [0x0532] = 0x2544b, [0x0533] = 0x2544f, [0x0534] = 0x2544e,
  [0x0535] = 0x25453, [0x0536] = 0x25447, [0x0537] = 0x40a2, [0x0538] = 0x25454,
  [0x0539] = 0x25452, [0x053a] = 0x25459, [0x053b] = 0x2547e, [0x053c] = 0x2562b,
  [0x053d] = 0x2562e, [0x053e] = 0x2577f, [0x053f] = 0x2577e, [0x0540] = 0x25781,
  [0x0541] = 0x25783, [0x0542] = 0x25947, [0x0543] = 0x25954, [0x0544] = 0x2594d,
  [0x0545] = 0x2594c, [0x0546] = 0x2594b, [0x0547] = 0x25957, [0x0548] = 0x25a65,
  [0x0549] = 0x25a64, [0x054a] = 0x25a61, [0x054b] = 0x25a62, [0x054c] = 0x25af0,
  [0x054d] = 0x25af3, [0x054e] = 0x25af9, [0x054f] = 0x25af4, [0x0550] = 0x25af5,
  [0x0551] = 0x25aef, [0x0552] = 0x25af8, [0x0553] = 0x25afc, [0x0554] = 0x25af7,
  [0x0555] = 0x25afd, [0x0556] = 0x25b17, [0x0557] = 0x25e34, [0x0558] = 0x25e36,
  [0x0559] = 0x25e38, [0x055a] = 0x25faf, [0x055b] = 0x25fae, [0x055c] = 0x25fac,
  [0x055d] = 0x25fa2, [0x055e] = 0x26228, [0x055f] = 0x22ec2, [0x0560] = 0x2629c,
  [0x0561] = 0x262b3, [0x0562] = 0x26290, [0x0563] = 0x26406, [0x0564] = 0x26404,
  [0x0565] = 0x2640b, [0x0566] = 0x26405, [0x0567] = 0x26540, [0x0568] = 0x2654f,
  [0x0569] = 0x266ab, [0x056a] = 0x2669d, [0x056b] = 0x266b8, [0x056c] = 0x266b2,
  [0x056d] = 0x266d2, [0x056e] = 0x266cf, [0x056f] = 0x266a2, [0x0570] = 0x43ec,
  [0x0571] = 0x266a5, [0x0572] = 0x266a9, [0x0573] = 0x266a7, [0x0574] = 0x266d0,
  [0x0575] = 0x26902, [0x0576] = 0x269cf, [0x0577] = 0x269d0, [0x0578] = 0x26a16,
  [0x0579] = 0x26a19, [0x057a] = 0x26b75, [0x057b] = 0x26b71, [0x057c] = 0x26b91,
  [0x057d] = 0x26b72, [0x057e] = 0x26b8e, [0x057f] = 0x26b6d, [0x0580] = 0x26b88,
  [0x0581] = 0x26b73, [0x0582] = 0x44bb, [0x0583] = 0x26b81, [0x0584] = 0x26b76,
  [0x0585] = 0x26b7b, [0x0586] = 0x26b7f, [0x0587] = 0x26b90, [0x0588] = 0x26b78,
  [0x0589] = 0x2724b, [0x058a] = 0x27248, [0x058b] = 0x2725b, [0x058c] = 0x27251,
  [0x058d] = 0x2723d, [0x058e] = 0x27240, [0x058f] = 0x27246, [0x0590] = 0x2724d,
  [0x0591] = 0x2723b, [0x0592] = 0x2724a, [0x0593] = 0x27241, [0x0594] = 0x2723e,
  [0x0595] = 0x2724e, [0x0596] = 0x275ac, [0x0597] = 0x275e6, [0x0598] = 0x275e9,
  [0x0599] = 0x27629, [0x059a] = 0x4621, [0x059b] = 0x27633, [0x059c] = 0x2762e,
  [0x059d] = 0x2761e, [0x059e] = 0x461f, [0x059f] = 0x2f9c5, [0x05a0] = 0x27622,
  [0x05a1] = 0x27627, [0x05a2] = 0x277e7, [0x05a3] = 0x2780b, [0x05a4] = 0x27966,
  [0x05a5] = 0x27bad, [0x05a6] = 0x27bae, [0x05a7] = 0x8c38, [0x05a8] = 0x27c2b,
  [0x05a9] = 0x27c2a, [0x05aa] = 0x27d2c, [0x05ab] = 0x27d2d, [0x05ac] = 0x4791,
  [0x05ad] = 0x4796, [0x05ae] = 0x27e8f, [0x05af] = 0x27e93, [0x05b0] = 0x27fc6,
  [0x05b1] = 0x27fc5, [0x05b2] = 0x28215, [0x05b3] = 0x28213, [0x05b4] = 0x282b1,
  [0x05b5] = 0x282a9, [0x05b6] = 0x8ece, [0x05b7] = 0x284a8, [0x05b8] = 0x284ab,
  [0x05b9] = 0x284be, [0x05ba] = 0x284ac, [0x05bb] = 0x284a9, [0x05bc] = 0x284a6,
  [0x05bd] = 0x284c1, [0x05be] = 0x286cc, [0x05bf] = 0x286d1, [0x05c0] = 0x286d3,
  [0x05c1] = 0x286e4, [0x05c2] = 0x286cb, [0x05c3] = 0x286e1, [0x05c4] = 0x286d2,
  [0x05c5] = 0x286e3, [0x05c6] = 0x286cf, [0x05c7] = 0x286d0, [0x05c8] = 0x286e5,
  [0x05c9] = 0x2870e, [0x05ca] = 0x28ccc, [0x05cb] = 0x28cca, [0x05cc] = 0x28ccb,
  [0x05cd] = 0x28e5c, [0x05ce] = 0x28e61, [0x05cf] = 0x28e5d, [0x05d0] = 0x28e59,
  [0x05d1] = 0x29077, [0x05d2] = 0x29684, [0x05d3] = 0x20041, [0x05d4] = 0x200fd,
  [0x05d5] = 0x2011a, [0x05d6] = 0x20163, [0x05d7] = 0x202b6, [0x05d8] = 0x202b2,
  [0x05d9] = 0x202e9, [0x05da] = 0x202e7, [0x05db] = 0x202d6, [0x05dc] = 0x20330,
  [0x05dd] = 0x202ec, [0x05de] = 0x202dd, [0x05df] = 0x202e8, [0x05e0] = 0x202b0,
  [0x05e1] = 0x202b5, [0x05e2] = 0x202e1, [0x05e3] = 0x20498, [0x05e4] = 0x20499,
  [0x05e5] = 0x34bc, [0x05e6] = 0x20593, [0x05e7] = 0x205e5, [0x05e8] = 0x34d8,
  [0x05e9] = 0x20760, [0x05ea] = 0x2075e, [0x05eb] = 0x34f4, [0x05ec] = 0x2087b,
  [0x05ed] = 0x208f1, [0x05ee] = 0x208f3, [0x05ef] = 0x20a7f, [0x05f0] = 0x2f838,
  [0x05f1] = 0x20b57, [0x05f2] = 0x20b58, [0x05f3] = 0x20d39, [0x05f4] = 0x20da7,
  [0x05f5] = 0x20d35, [0x05f6] = 0x20d60, [0x05f7] = 0x20d51, [0x05f8] = 0x20d62,
  [0x05f9] = 0x20d42, [0x05fa] = 0x20d3c, [0x05fb] = 0x20d43, [0x05fc] = 0x20dab,
  [0x05fd] = 0x20da9, [0x05fe] = 0x20db4, [0x05ff] = 0x20d79, [0x0600] = 0x20d6a,
  [0x0601] = 0x20d3b, [0x0602] = 0x20daa, [0x0603] = 0x20d33, [0x0604] = 0x20d37,
  [0x0605] = 0x20d27, [0x0606] = 0x20da8, [0x0607] = 0x20d70, [0x0608] = 0x20d2f,
  [0x0609] = 0x20d36, [0x060a] = 0x20db5, [0x060b] = 0x20d30, [0x060c] = 0x20d3a,
  [0x060d] = 0x20d38, [0x060e] = 0x20db1, [0x060f] = 0x20dac, [0x0610] = 0x20d2e,
  [0x0611] = 0x20d75, [0x0612] = 0x20db0, [0x0613] = 0x211f7, [0x0614] = 0x21200,
  [0x0615] = 0x211f1, [0x0616] = 0x211f3, [0x0617] = 0x21360, [0x0618] = 0x21361,
  [0x0619] = 0x21332, [0x061a] = 0x2132d, [0x061b] = 0x2134a, [0x061c] = 0x2132a,
  [0x061d] = 0x21567, [0x061e] = 0x215a2, [0x061f] = 0x215a3, [0x0620] = 0x2159e,
  [0x0621] = 0x215a4, [0x0622] = 0x21644, [0x0623] = 0x21632, [0x0624] = 0x2179b,
  [0x0625] = 0x21798, [0x0626] = 0x21799, [0x0627] = 0x2179a, [0x0628] = 0x21766,
  [0x0629] = 0x21762, [0x062a] = 0x2176b, [0x062b] = 0x24562, [0x062c] = 0x21767,
  [0x062d] = 0x21a3b, [0x062e] = 0x21a23, [0x062f] = 0x21a29, [0x0630] = 0x21b7e,
  [0x0631] = 0x21bf3, [0x0632] = 0x3785, [0x0633] = 0x21bf5, [0x0634] = 0x3783,
  [0x0635] = 0x21c80, [0x0636] = 0x21c7e, [0x0637] = 0x21c7c, [0x0638] = 0x21c7f,
  [0x0639] = 0x21c7d, [0x063a] = 0x21c81, [0x063b] = 0x21e25, [0x063c] = 0x21e29,
  [0x063d] = 0x21e17, [0x063e] = 0x21e63, [0x063f] = 0x21e67, [0x0640] = 0x21e65,
  [0x0641] = 0x21e2a, [0x0642] = 0x21e68, [0x0643] = 0x21e26, [0x0644] = 0x21e64,
  [0x0645] = 0x21e1b, [0x0646] = 0x21e62, [0x0647] = 0x21e69, [0x0648] = 0x21ffe,
  [0x0649] = 0x22044, [0x064a] = 0x22042, [0x064b] = 0x220b7, [0x064c] = 0x220d2,
  [0x064d] = 0x220d4, [0x064e] = 0x220d7, [0x064f] = 0x220dc, [0x0650] = 0x220d1,
  [0x0651] = 0x220d5, [0x0652] = 0x220cd, [0x0653] = 0x220d8, [0x0654] = 0x2219a,
  [0x0655] = 0x22242, [0x0656] = 0x22235, [0x0657] = 0x22258, [0x0658] = 0x22234,
  [0x0659] = 0x2223c, [0x065a] = 0x2223b, [0x065b] = 0x2223e, [0x065c] = 0x2223d,
  [0x065d] = 0x2225a, [0x065e] = 0x22241, [0x065f] = 0x22238, [0x0660] = 0x22245,
  [0x0661] = 0x22233, [0x0662] = 0x22326, [0x0663] = 0x38ba, [0x0664] = 0x223f7,
  [0x0665] = 0x223ef, [0x0666] = 0x223f0, [0x0667] = 0x223ee, [0x0668] = 0x2249d,
  [0x0669] = 0x22506, [0x066a] = 0x22511, [0x066b] = 0x22508, [0x066c] = 0x22502,
  [0x066d] = 0x2250a, [0x066e] = 0x22507, [0x066f] = 0x22691, [0x0670] = 0x2268e,
  [0x0671] = 0x22685, [0x0672] = 0x226e8, [0x0673] = 0x22717, [0x0674] = 0x226d4,
  [0x0675] = 0x3969, [0x0676] = 0x226c4, [0x0677] = 0x226d5, [0x0678] = 0x3945,
  [0x0679] = 0x226d8, [0x067a] = 0x22680, [0x067b] = 0x226de, [0x067c] = 0x22742,
  [0x067d] = 0x229c7, [0x067e] = 0x229bf, [0x067f] = 0x22b8d, [0x0680] = 0x22b89,
  [0x0681] = 0x22b90, [0x0682] = 0x22b9a, [0x0683] = 0x22bc8, [0x0684] = 0x22b92,
  [0x0685] = 0x22ba1, [0x0686] = 0x22b84, [0x0687] = 0x22f57, [0x0688] = 0x22f47,
  [0x0689] = 0x22f77, [0x068a] = 0x22f3a, [0x068b] = 0x22f50, [0x068c] = 0x22f42,
  [0x068d] = 0x22f43, [0x068e] = 0x22fa3, [0x068f] = 0x23077, [0x0690] = 0x231b2,
  [0x0691] = 0x231e2, [0x0692] = 0x231bd, [0x0693] = 0x23374, [0x0694] = 0x23482,
  [0x0695] = 0x234b8, [0x0696] = 0x3b6c, [0x0697] = 0x234ba, [0x0698] = 0x234b9,
  [0x0699] = 0x234bb, [0x069a] = 0x23487, [0x069b] = 0x2f8e4, [0x069c] = 0x23483,
  [0x069d] = 0x234b7, [0x069e] = 0x20043, [0x069f] = 0x238ce, [0x06a0] = 0x238ba,
  [0x06a1] = 0x23973, [0x06a2] = 0x23a09, [0x06a3] = 0x23a0a, [0x06a4] = 0x23a0b,
  [0x06a5] = 0x23a07, [0x06a6] = 0x23a05, [0x06a7] = 0x23a0e, [0x06a8] = 0x23a97,
  [0x06a9] = 0x23b77, [0x06aa] = 0x23b7b, [0x06ab] = 0x23b7a, [0x06ac] = 0x23b74,
  [0x06ad] = 0x23b79, [0x06ae] = 0x23d86, [0x06af] = 0x23df5, [0x06b0] = 0x23da5,
  [0x06b1] = 0x23d9d, [0x06b2] = 0x23dbe, [0x06b3] = 0x23d96, [0x06b4] = 0x23de1,
  [0x06b5] = 0x23da4, [0x06b6] = 0x23da1, [0x06b7] = 0x23df6, [0x06b8] = 0x3d04,
  [0x06b9] = 0x23d8d, [0x06ba] = 0x23d8b, [0x06bb] = 0x23d78, [0x06bc] = 0x23d98,
  [0x06bd] = 0x23e0a, [0x06be] = 0x2425d, [0x06bf] = 0x24227, [0x06c0] = 0x2425f,
  [0x06c1] = 0x245c0, [0x06c2] = 0x245c1, [0x06c3] = 0x2466d, [0x06c4] = 0x2465e,
  [0x06c5] = 0x2468a, [0x06c6] = 0x24662, [0x06c7] = 0x24664, [0x06c8] = 0x3e3a,
  [0x06c9] = 0x247c0, [0x06ca] = 0x247ca, [0x06cb] = 0x247d1, [0x06cc] = 0x3e79,
  [0x06cd] = 0x247c7, [0x06ce] = 0x247ce, [0x06cf] = 0x7309, [0x06d0] = 0x247c5,
  [0x06d1] = 0x24936, [0x06d2] = 0x24957, [0x06d3] = 0x24977, [0x06d4] = 0x24af6,
  [0x06d5] = 0x24af7, [0x06d6] = 0x24b55, [0x06d7] = 0x24b44, [0x06d8] = 0x24b3e,
  [0x06d9] = 0x24b42, [0x06da] = 0x24b52, [0x06db] = 0x24be1, [0x06dc] = 0x24c08,
  [0x06dd] = 0x24c68, [0x06de] = 0x24c67, [0x06df] = 0x3f5d, [0x06e0] = 0x2f937,
  [0x06e1] = 0x24d7e, [0x06e2] = 0x24d93, [0x06e3] = 0x24d79, [0x06e4] = 0x24d81,
  [0x06e5] = 0x24d7d, [0x06e6] = 0x24d7b, [0x06e7] = 0x24d94, [0x06e8] = 0x3f8a,
  [0x06e9] = 0x24f66, [0x06ea] = 0x24fd9, [0x06eb] = 0x24fe0, [0x06ec] = 0x2f93e,
  [0x06ed] = 0x2505f, [0x06ee] = 0x2515f, [0x06ef] = 0x2515e, [0x06f0] = 0x25168,
  [0x06f1] = 0x4027, [0x06f2] = 0x25188, [0x06f3] = 0x2516a, [0x06f4] = 0x2516c,
  [0x06f5] = 0x25166, [0x06f6] = 0x25167, [0x06f7] = 0x25189, [0x06f8] = 0x25160,
  [0x06f9] = 0x25185, [0x06fa] = 0x2032a, [0x06fb] = 0x25368, [0x06fc] = 0x253b9,
  [0x06fd] = 0x253ba, [0x06fe] = 0x253be, [0x06ff] = 0x2546f, [0x0700] = 0x25480,
  [0x0701] = 0x25474, [0x0702] = 0x25481, [0x0703] = 0x2547a, [0x0704] = 0x2549c,
  [0x0705] = 0x25473, [0x0706] = 0x25482, [0x0707] = 0x2547f, [0x0708] = 0x254a7,
  [0x0709] = 0x25479, [0x070a] = 0x25478, [0x070b] = 0x2547d, [0x070c] = 0x254a8,
  [0x070d] = 0x2547c, [0x070e] = 0x25668, [0x070f] = 0x2564e, [0x0710] = 0x2579e,
  [0x0711] = 0x4146, [0x0712] = 0x4140, [0x0713] = 0x25798, [0x0714] = 0x413f,
  [0x0715] = 0x25785, [0x0716] = 0x2579c, [0x0717] = 0x257c5, [0x0718] = 0x2579a,
  [0x0719] = 0x25968, [0x071a] = 0x25965, [0x071b] = 0x25967, [0x071c] = 0x25961,
  [0x071d] = 0x25960, [0x071e] = 0x419d, [0x071f] = 0x41cb, [0x0720] = 0x25a7b,
  [0x0721] = 0x41e1, [0x0722] = 0x25b19, [0x0723] = 0x25b0e, [0x0724] = 0x25b31,
  [0x0725] = 0x25b12, [0x0726] = 0x25b10, [0x0727] = 0x25b15, [0x0728] = 0x25b3f,
  [0x0729] = 0x25b1d, [0x072a] = 0x25b1e, [0x072b] = 0x25e4d, [0x072c] = 0x25e48,
  [0x072d] = 0x25e45, [0x072e] = 0x25e42, [0x072f] = 0x25e49, [0x0730] = 0x25e4a,
  [0x0731] = 0x25e47, [0x0732] = 0x427f, [0x0733] = 0x25e4c, [0x0734] = 0x25fcd,
  [0x0735] = 0x25fd5, [0x0736] = 0x25fc4, [0x0737] = 0x25fca, [0x0738] = 0x25fc3,
  [0x0739] = 0x4346, [0x073a] = 0x26229, [0x073b] = 0x262b4, [0x073c] = 0x2636d,
  [0x073d] = 0x26366, [0x073e] = 0x2636a, [0x073f] = 0x26415, [0x0740] = 0x264d3,
  [0x0741] = 0x264ec, [0x0742] = 0x26555, [0x0743] = 0x441a, [0x0744] = 0x266d6,
  [0x0745] = 0x266d8, [0x0746] = 0x26751, [0x0747] = 0x266e0, [0x0748] = 0x26703,
  [0x0749] = 0x26704, [0x074a] = 0x266da, [0x074b] = 0x266ea, [0x074c] = 0x26702,
  [0x074d] = 0x266d3, [0x074e] = 0x266e4, [0x074f] = 0x266e5, [0x0750] = 0x2693b,
  [0x0751] = 0x26966, [0x0752] = 0x26a2c, [0x0753] = 0x26a21, [0x0754] = 0x26a26,
  [0x0755] = 0x26ad4, [0x0756] = 0x26ad5, [0x0757] = 0x26ad9, [0x0758] = 0x26bd5,
  [0x0759] = 0x26beb, [0x075a] = 0x26bc5, [0x075b] = 0x44d3, [0x075c] = 0x26c27,
  [0x075d] = 0x26bf0, [0x075e] = 0x26bb9, [0x075f] = 0x26c18, [0x0760] = 0x26bee,
  [0x0761] = 0x44d0, [0x0762] = 0x26bb7, [0x0763] = 0x26bbb, [0x0764] = 0x26bef,
  [0x0765] = 0x26bb8, [0x0766] = 0x26bdf, [0x0767] = 0x26bda, [0x0768] = 0x26be3,
  [0x0769] = 0x26bc9, [0x076a] = 0x26bec, [0x076b] = 0x271a4, [0x076c] = 0x458e,
  [0x076d] = 0x271a3, [0x076e] = 0x2725e, [0x076f] = 0x27261, [0x0770] = 0x27262,
  [0x0771] = 0x27263, [0x0772] = 0x2728e, [0x0773] = 0x27269, [0x0774] = 0x2726b,
  [0x0775] = 0x27285, [0x0776] = 0x2728d, [0x0777] = 0x27264, [0x0778] = 0x2726d,
  [0x0779] = 0x2764c, [0x077a] = 0x27649, [0x077b] = 0x27647, [0x077c] = 0x27648,
  [0x077d] = 0x27642, [0x077e] = 0x27645, [0x077f] = 0x2765b, [0x0780] = 0x2765d,
  [0x0781] = 0x27646, [0x0782] = 0x27813, [0x0783] = 0x2781a, [0x0784] = 0x2780e,
  [0x0785] = 0x2780f, [0x0786] = 0x2781b, [0x0787] = 0x27812, [0x0788] = 0x278c3,
  [0x0789] = 0x278be, [0x078a] = 0x278bd, [0x078b] = 0x27979, [0x078c] = 0x2797c,
  [0x078d] = 0x46c3, [0x078e] = 0x46b6, [0x078f] = 0x27980, [0x0790] = 0x27975,
  [0x0791] = 0x2799b, [0x0792] = 0x8a2f, [0x0793] = 0x27974, [0x0794] = 0x46c0,
  [0x0795] = 0x46b8, [0x0796] = 0x22041, [0x0797] = 0x22494, [0x0798] = 0x27c30,
  [0x0799] = 0x27ca7, [0x079a] = 0x27ca5, [0x079b] = 0x27ca4, [0x079c] = 0x27d3a,
  [0x079d] = 0x27e5b, [0x079e] = 0x27e9f, [0x079f] = 0x27ea8, [0x07a0] = 0x27eb3,
  [0x07a1] = 0x27ea1, [0x07a2] = 0x27ea7, [0x07a3] = 0x27eb2, [0x07a4] = 0x27e9d,
  [0x07a5] = 0x27ea3, [0x07a6] = 0x27ea2, [0x07a7] = 0x27fef, [0x07a8] = 0x27fd9,
  [0x07a9] = 0x47d9, [0x07aa] = 0x27fd5, [0x07ab] = 0x27fdd, [0x07ac] = 0x27fda,
  [0x07ad] = 0x27fee, [0x07ae] = 0x27fd7, [0x07af] = 0x2821a, [0x07b0] = 0x28219,
  [0x07b1] = 0x28218, [0x07b2] = 0x282b3, [0x07b3] = 0x282bf, [0x07b4] = 0x282bb,
  [0x07b5] = 0x282c0, [0x07b6] = 0x282b9, [0x07b7] = 0x282b8, [0x07b8] = 0x282d5,
  [0x07b9] = 0x28411, [0x07ba] = 0x28410, [0x07bb] = 0x284c3, [0x07bc] = 0x284c2,
  [0x07bd] = 0x284e9, [0x07be] = 0x284cd, [0x07bf] = 0x284e5, [0x07c0] = 0x284ca,
  [0x07c1] = 0x284c7, [0x07c2] = 0x284e8, [0x07c3] = 0x286f1, [0x07c4] = 0x2870c,
  [0x07c5] = 0x286ef, [0x07c6] = 0x286ee, [0x07c7] = 0x286f3, [0x07c8] = 0x2870d,
  [0x07c9] = 0x286f6, [0x07ca] = 0x286f0, [0x07cb] = 0x48ec, [0x07cc] = 0x287f5,
  [0x07cd] = 0x287f4, [0x07ce] = 0x287f8, [0x07cf] = 0x287fe, [0x07d0] = 0x28911,
  [0x07d1] = 0x28922, [0x07d2] = 0x28942, [0x07d3] = 0x28c5c, [0x07d4] = 0x28c5a,
  [0x07d5] = 0x28c5d, [0x07d6] = 0x28cd1, [0x07d7] = 0x28cd0, [0x07d8] = 0x28e75,
  [0x07d9] = 0x28e7b, [0x07da] = 0x28f89, [0x07db] = 0x2944d, [0x07dc] = 0x29689,
  [0x07dd] = 0x29692, [0x07de] = 0x2968c, [0x07df] = 0x29688, [0x07e0] = 0x29867,
  [0x07e1] = 0x29868, [0x07e2] = 0x20321, [0x07e3] = 0x202f2, [0x07e4] = 0x20309,
  [0x07e5] = 0x202f8, [0x07e6] = 0x202f0, [0x07e7] = 0x202f3, [0x07e8] = 0x202f5,
  [0x07e9] = 0x202fb, [0x07ea] = 0x202f7, [0x07eb] = 0x202ef, [0x07ec] = 0x2030b,
  [0x07ed] = 0x204a2, [0x07ee] = 0x205f5, [0x07ef] = 0x205f3, [0x07f0] = 0x205f4,
  [0x07f1] = 0x205f2, [0x07f2] = 0x20698, [0x07f3] = 0x20768, [0x07f4] = 0x20780,
  [0x07f5] = 0x20785, [0x07f6] = 0x2078e, [0x07f7] = 0x2078f, [0x07f8] = 0x202f4,
  [0x07f9] = 0x20786, [0x07fa] = 0x207a4, [0x07fb] = 0x20886, [0x07fc] = 0x20889,
  [0x07fd] = 0x20887, [0x07fe] = 0x20885, [0x07ff] = 0x52dc, [0x0800] = 0x20888,
  [0x0801] = 0x20aff, [0x0802] = 0x20b66, [0x0803] = 0x35cc, [0x0804] = 0x20dc3,
  [0x0805] = 0x35a2, [0x0806] = 0x20dc4, [0x0807] = 0x20e3b, [0x0808] = 0x20de7,
  [0x0809] = 0x20df8, [0x080a] = 0x20dfb, [0x080b] = 0x35b6, [0x080c] = 0x20dbe,
  [0x080d] = 0x20dc6, [0x080e] = 0x35c5, [0x080f] = 0x35c6, [0x0810] = 0x20dfc,
  [0x0811] = 0x20dc0, [0x0812] = 0x20e33, [0x0813] = 0x20dc9, [0x0814] = 0x20dfe,
  [0x0815] = 0x20dda, [0x0816] = 0x20dcc, [0x0817] = 0x20dbb, [0x0818] = 0x20dfd,
  [0x0819] = 0x20ddf, [0x081a] = 0x20dcd, [0x081b] = 0x20dc2, [0x081c] = 0x20dc8,
  [0x081d] = 0x20e32, [0x081e] = 0x20e2d, [0x081f] = 0x20dd2, [0x0820] = 0x20e31,
  [0x0821] = 0x20dd3, [0x0822] = 0x20e2e, [0x0823] = 0x20dcf, [0x0824] = 0x211ff,
  [0x0825] = 0x21209, [0x0826] = 0x211fc, [0x0827] = 0x21372, [0x0828] = 0x213aa,
  [0x0829] = 0x213ab, [0x082a] = 0x213a7, [0x082b] = 0x2137b, [0x082c] = 0x21376,
  [0x082d] = 0x213a8, [0x082e] = 0x213ac, [0x082f] = 0x3683, [0x0830] = 0x5921,
  [0x0831] = 0x215aa, [0x0832] = 0x20a03, [0x0833] = 0x215ac, [0x0834] = 0x215ad,
  [0x0835] = 0x215ab, [0x0836] = 0x21650, [0x0837] = 0x21658, [0x0838] = 0x217dd,
  [0x0839] = 0x36f8, [0x083a] = 0x217ae, [0x083b] = 0x217a2, [0x083c] = 0x217ab,
  [0x083d] = 0x217e3, [0x083e] = 0x217af, [0x083f] = 0x217e0, [0x0840] = 0x217e4,
  [0x0841] = 0x36f6, [0x0842] = 0x217a7, [0x0843] = 0x217aa, [0x0844] = 0x217e2,
  [0x0845] = 0x217e1, [0x0846] = 0x217df, [0x0847] = 0x217ad, [0x0848] = 0x217e5,
  [0x0849] = 0x217d1, [0x084a] = 0x217ac, [0x084b] = 0x217d5, [0x084c] = 0x217a3,
  [0x084d] = 0x2179f, [0x084e] = 0x21979, [0x084f] = 0x21976, [0x0850] = 0x21a43,
  [0x0851] = 0x21a3d, [0x0852] = 0x21a47, [0x0853] = 0x21a44, [0x0854] = 0x21a3f,
  [0x0855] = 0x21a5b, [0x0856] = 0x21b7f, [0x0857] = 0x21c9a, [0x0858] = 0x21c9c,
  [0x0859] = 0x379b, [0x085a] = 0x21d1e, [0x085b] = 0x21e88, [0x085c] = 0x21e6a,
  [0x085d] = 0x21e76, [0x085e] = 0x21e72, [0x085f] = 0x21e70, [0x0860] = 0x21e7e,
  [0x0861] = 0x21e6c, [0x0862] = 0x21e6f, [0x0863] = 0x21e75, [0x0864] = 0x21e77,
  [0x0865] = 0x21e78, [0x0866] = 0x21e8a, [0x0867] = 0x21e79, [0x0868] = 0x21e94,
  [0x0869] = 0x22001, [0x086a] = 0x22105, [0x086b] = 0x220f1, [0x086c] = 0x220ee,
  [0x086d] = 0x220ef, [0x086e] = 0x22104, [0x086f] = 0x5e3f, [0x0870] = 0x220fa,
  [0x0871] = 0x22107, [0x0872] = 0x220f4, [0x0873] = 0x220f5, [0x0874] = 0x2219c,
  [0x0875] = 0x2226d, [0x0876] = 0x22269, [0x0877] = 0x22266, [0x0878] = 0x22262,
  [0x0879] = 0x22246, [0x087a] = 0x388d, [0x087b] = 0x22265, [0x087c] = 0x2225d,
  [0x087d] = 0x2225e, [0x087e] = 0x2225f, [0x087f] = 0x2238e, [0x0880] = 0x22402,
  [0x0881] = 0x22406, [0x0882] = 0x22404, [0x0883] = 0x22403, [0x0884] = 0x22462,
  [0x0885] = 0x22527, [0x0886] = 0x22521, [0x0887] = 0x22520, [0x0888] = 0x22524,
  [0x0889] = 0x22529, [0x088a] = 0x22531, [0x088b] = 0x22523, [0x088c] = 0x22522,
  [0x088d] = 0x226dc, [0x088e] = 0x2274d, [0x088f] = 0x226e5, [0x0890] = 0x226d3,
  [0x0891] = 0x226ee, [0x0892] = 0x226e6, [0x0893] = 0x3956, [0x0894] = 0x22736,
  [0x0895] = 0x22740, [0x0896] = 0x2283f, [0x0897] = 0x2273d, [0x0898] = 0x22744,
  [0x0899] = 0x395b, [0x089a] = 0x2275f, [0x089b] = 0x22729, [0x089c] = 0x22749,
  [0x089d] = 0x2278a, [0x089e] = 0x2272a, [0x089f] = 0x22787, [0x08a0] = 0x2274c,
  [0x08a1] = 0x22731, [0x08a2] = 0x22748, [0x08a3] = 0x2272b, [0x08a4] = 0x2273b,
  [0x08a5] = 0x22741, [0x08a6] = 0x226c9, [0x08a7] = 0x22734, [0x08a8] = 0x22753,
  [0x08a9] = 0x22735, [0x08aa] = 0x22747, [0x08ab] = 0x22738, [0x08ac] = 0x229d1,
  [0x08ad] = 0x229d2, [0x08ae] = 0x22a51, [0x08af] = 0x22a55, [0x08b0] = 0x22a52,
  [0x08b1] = 0x22a53, [0x08b2] = 0x22b8f, [0x08b3] = 0x22bf4, [0x08b4] = 0x22c47,
  [0x08b5] = 0x22c0d, [0x08b6] = 0x22c1c, [0x08b7] = 0x22bfb, [0x08b8] = 0x22c19,
  [0x08b9] = 0x22bf7, [0x08ba] = 0x22bf9, [0x08bb] = 0x22bf5, [0x08bc] = 0x22be9,
  [0x08bd] = 0x22c0a, [0x08be] = 0x22bee, [0x08bf] = 0x22c0b, [0x08c0] = 0x22bfd,
  [0x08c1] = 0x22c06, [0x08c2] = 0x22c02, [0x08c3] = 0x22c16, [0x08c4] = 0x22c18,
  [0x08c5] = 0x22bf0, [0x08c6] = 0x22ed4, [0x08c7] = 0x22f66, [0x08c8] = 0x22f6a,
  [0x08c9] = 0x22f75, [0x08ca] = 0x22f76, [0x08cb] = 0x22f80, [0x08cc] = 0x22f6d,
  [0x08cd] = 0x3a96, [0x08ce] = 0x22f69, [0x08cf] = 0x22f67, [0x08d0] = 0x22f68,
  [0x08d1] = 0x22f5d, [0x08d2] = 0x230a4, [0x08d3] = 0x230fe, [0x08d4] = 0x230fd,
  [0x08d5] = 0x231ec, [0x08d6] = 0x231e9, [0x08d7] = 0x23221, [0x08d8] = 0x231eb,
  [0x08d9] = 0x231e7, [0x08da] = 0x231f2, [0x08db] = 0x23220, [0x08dc] = 0x2337a,
  [0x08dd] = 0x2352d, [0x08de] = 0x2352e, [0x08df] = 0x234ca, [0x08e0] = 0x2352f,
  [0x08e1] = 0x234c8, [0x08e2] = 0x3b7e, [0x08e3] = 0x234cb, [0x08e4] = 0x234c3,
  [0x08e5] = 0x23484, [0x08e6] = 0x234d2, [0x08e7] = 0x234c6, [0x08e8] = 0x3b81,
  [0x08e9] = 0x234cf, [0x08ea] = 0x234d5, [0x08eb] = 0x234d4, [0x08ec] = 0x234dd,
  [0x08ed] = 0x2352b, [0x08ee] = 0x235a3, [0x08ef] = 0x234db, [0x08f0] = 0x2352c,
  [0x08f1] = 0x23513, [0x08f2] = 0x3c35, [0x08f3] = 0x238d1, [0x08f4] = 0x238cc,
  [0x08f5] = 0x238d5, [0x08f6] = 0x238c9, [0x08f7] = 0x238cf, [0x08f8] = 0x238d2,
  [0x08f9] = 0x2397b, [0x08fa] = 0x2397c, [0x08fb] = 0x23a23, [0x08fc] = 0x23a1d,
  [0x08fd] = 0x23a1c, [0x08fe] = 0x23a1e, [0x08ff] = 0x23a22, [0x0900] = 0x23a24,
  [0x0901] = 0x23a20, [0x0902] = 0x23a18, [0x0903] = 0x23a21, [0x0904] = 0x23b88,
  [0x0905] = 0x23b8a, [0x0906] = 0x23b94, [0x0907] = 0x23b8f, [0x0908] = 0x23b8e,
  [0x0909] = 0x23b90, [0x090a] = 0x3c9b, [0x090b] = 0x23ba7, [0x090c] = 0x23c64,
  [0x090d] = 0x3d00, [0x090e] = 0x23e18, [0x090f] = 0x23e14, [0x0910] = 0x23e1d,
  [0x0911] = 0x23e05, [0x0912] = 0x23dff, [0x0913] = 0x23e1b, [0x0914] = 0x23e1a,
  [0x0915] = 0x23e19, [0x0916] = 0x23e03, [0x0917] = 0x23e0e, [0x0918] = 0x23e47,
  [0x0919] = 0x23e04, [0x091a] = 0x23e5a, [0x091b] = 0x23e07, [0x091c] = 0x23e20,
  [0x091d] = 0x2426a, [0x091e] = 0x24266, [0x091f] = 0x2427a, [0x0920] = 0x24265,
  [0x0921] = 0x242af, [0x0922] = 0x245cd, [0x0923] = 0x245cf, [0x0924] = 0x245d1,
  [0x0925] = 0x245ce, [0x0926] = 0x2465f, [0x0927] = 0x24677, [0x0928] = 0x24674,
  [0x0929] = 0x24688, [0x092a] = 0x24675, [0x092b] = 0x24689, [0x092c] = 0x24672,
  [0x092d] = 0x247c6, [0x092e] = 0x247ea, [0x092f] = 0x247e0, [0x0930] = 0x24807,
  [0x0931] = 0x247e6, [0x0932] = 0x247e5, [0x0933] = 0x247dd, [0x0934] = 0x247e3,
  [0x0935] = 0x247da, [0x0936] = 0x249ae, [0x0937] = 0x2497e, [0x0938] = 0x2497d,
  [0x0939] = 0x2497f, [0x093a] = 0x3ed3, [0x093b] = 0x249d9, [0x093c] = 0x24afa,
  [0x093d] = 0x24afb, [0x093e] = 0x24b4d, [0x093f] = 0x24b54, [0x0940] = 0x24b51,
  [0x0941] = 0x24b50, [0x0942] = 0x24d98, [0x0943] = 0x24d95, [0x0944] = 0x24d9d,
  [0x0945] = 0x24dc1, [0x0946] = 0x24d9a, [0x0947] = 0x24d9b, [0x0948] = 0x3f9f,
  [0x0949] = 0x24d9c, [0x094a] = 0x24dd7, [0x094b] = 0x24da4, [0x094c] = 0x24d96,
  [0x094d] = 0x24da0, [0x094e] = 0x24f31, [0x094f] = 0x24fe9, [0x0950] = 0x2506e,
  [0x0951] = 0x2519a, [0x0952] = 0x251a5, [0x0953] = 0x25194, [0x0954] = 0x251a6,
  [0x0955] = 0x25199, [0x0956] = 0x2518b, [0x0957] = 0x25198, [0x0958] = 0x25191,
  [0x0959] = 0x2518c, [0x095a] = 0x251a1, [0x095b] = 0x251a3, [0x095c] = 0x251c6,
  [0x095d] = 0x2536f, [0x095e] = 0x2536d, [0x095f] = 0x2536a, [0x0960] = 0x2536c,
  [0x0961] = 0x25371, [0x0962] = 0x2536e, [0x0963] = 0x253df, [0x0964] = 0x254ac,
  [0x0965] = 0x254aa, [0x0966] = 0x40b1, [0x0967] = 0x254b5, [0x0968] = 0x254ad,
  [0x0969] = 0x254b2, [0x096a] = 0x254b3, [0x096b] = 0x254ab, [0x096c] = 0x254b6,
  [0x096d] = 0x254c7, [0x096e] = 0x254c6, [0x096f] = 0x2566c, [0x0970] = 0x25670,
  [0x0971] = 0x257b3, [0x0972] = 0x414b, [0x0973] = 0x2598c, [0x0974] = 0x25981,
  [0x0975] = 0x25983, [0x0976] = 0x2597d, [0x0977] = 0x25985, [0x0978] = 0x25990,
  [0x0979] = 0x259a2, [0x097a] = 0x2598d, [0x097b] = 0x2598f, [0x097c] = 0x2597e,
  [0x097d] = 0x2598a, [0x097e] = 0x2597f, [0x097f] = 0x25976, [0x0980] = 0x25987,
  [0x0981] = 0x25978, [0x0982] = 0x25a80, [0x0983] = 0x25a82, [0x0984] = 0x25a81,
  [0x0985] = 0x25a83, [0x0986] = 0x25b5a, [0x0987] = 0x25b34, [0x0988] = 0x25b2c,
  [0x0989] = 0x25b2a, [0x098a] = 0x25b40, [0x098b] = 0x25b2d, [0x098c] = 0x25b33,
  [0x098d] = 0x25b2b, [0x098e] = 0x25b4b, [0x098f] = 0x7b3f, [0x0990] = 0x25b3c,
  [0x0991] = 0x25e5a, [0x0992] = 0x25e77, [0x0993] = 0x25e60, [0x0994] = 0x25ffa,
  [0x0995] = 0x25ff9, [0x0996] = 0x25ffc, [0x0997] = 0x42d6, [0x0998] = 0x25ff5,
  [0x0999] = 0x25fe8, [0x099a] = 0x26009, [0x099b] = 0x25fe7, [0x099c] = 0x25fe6,
  [0x099d] = 0x25fe9, [0x099e] = 0x26232, [0x099f] = 0x2622c, [0x09a0] = 0x262bd,
  [0x09a1] = 0x262bc, [0x09a2] = 0x2637a, [0x09a3] = 0x2637e, [0x09a4] = 0x26379,
  [0x09a5] = 0x4389, [0x09a6] = 0x26423, [0x09a7] = 0x26430, [0x09a8] = 0x26428,
  [0x09a9] = 0x264d4, [0x09aa] = 0x26ace, [0x09ab] = 0x264f1, [0x09ac] = 0x264f0,
  [0x09ad] = 0x26560, [0x09ae] = 0x26565, [0x09af] = 0x266a8, [0x09b0] = 0x2670a,
  [0x09b1] = 0x2671b, [0x09b2] = 0x26712, [0x09b3] = 0x26707, [0x09b4] = 0x2674f,
  [0x09b5] = 0x2670e, [0x09b6] = 0x26752, [0x09b7] = 0x26716, [0x09b8] = 0x4400,
  [0x09b9] = 0x26753, [0x09ba] = 0x26754, [0x09bb] = 0x2f984, [0x09bc] = 0x26756,
  [0x09bd] = 0x2696d, [0x09be] = 0x26972, [0x09bf] = 0x26a2f, [0x09c0] = 0x26c2a,
  [0x09c1] = 0x44dc, [0x09c2] = 0x26c63, [0x09c3] = 0x26c84, [0x09c4] = 0x26caf,
  [0x09c5] = 0x26c45, [0x09c6] = 0x26c70, [0x09c7] = 0x26c56, [0x09c8] = 0x26c16,
  [0x09c9] = 0x26c3d, [0x09ca] = 0x26c38, [0x09cb] = 0x26c6f, [0x09cc] = 0x26c77,
  [0x09cd] = 0x26c2e, [0x09ce] = 0x26c31, [0x09cf] = 0x26c4c, [0x09d0] = 0x26c5f,
  [0x09d1] = 0x26c75, [0x09d2] = 0x26c39, [0x09d3] = 0x26c3a, [0x09d4] = 0x26c2c,
  [0x09d5] = 0x26c2d, [0x09d6] = 0x26c3f, [0x09d7] = 0x26c2f, [0x09d8] = 0x26c30,
  [0x09d9] = 0x26c3e, [0x09da] = 0x83d0, [0x09db] = 0x4590, [0x09dc] = 0x271b2,
  [0x09dd] = 0x272b7, [0x09de] = 0x2728f, [0x09df] = 0x45b1, [0x09e0] = 0x27296,
  [0x09e1] = 0x2729e, [0x09e2] = 0x27297, [0x09e3] = 0x27295, [0x09e4] = 0x272a5,
  [0x09e5] = 0x27298, [0x09e6] = 0x86e7, [0x09e7] = 0x272a3, [0x09e8] = 0x2729a,
  [0x09e9] = 0x272ad, [0x09ea] = 0x272a6, [0x09eb] = 0x272b6, [0x09ec] = 0x27299,
  [0x09ed] = 0x272af, [0x09ee] = 0x272ac, [0x09ef] = 0x45aa, [0x09f0] = 0x2729f,
  [0x09f1] = 0x27294, [0x09f2] = 0x275eb, [0x09f3] = 0x2766f, [0x09f4] = 0x2766d,
  [0x09f5] = 0x2767e, [0x09f6] = 0x2766c, [0x09f7] = 0x27661, [0x09f8] = 0x27663,
  [0x09f9] = 0x2766b, [0x09fa] = 0x2765e, [0x09fb] = 0x2763e, [0x09fc] = 0x27664,
  [0x09fd] = 0x276ad, [0x09fe] = 0x467d, [0x09ff] = 0x27820, [0x0a00] = 0x278d1,
  [0x0a01] = 0x278d5, [0x0a02] = 0x279ae, [0x0a03] = 0x279af, [0x0a04] = 0x2799f,
  [0x0a05] = 0x279a8, [0x0a06] = 0x279ca, [0x0a07] = 0x27be1, [0x0a08] = 0x27c39,
  [0x0a09] = 0x27c38, [0x0a0a] = 0x27c3b, [0x0a0b] = 0x27cad, [0x0a0c] = 0x27cb5,
  [0x0a0d] = 0x27caf, [0x0a0e] = 0x27cb1, [0x0a0f] = 0x27cae, [0x0a10] = 0x27cb4,
  [0x0a11] = 0x27cb0, [0x0a12] = 0x27d4b, [0x0a13] = 0x27d51, [0x0a14] = 0x27d4c,
  [0x0a15] = 0x4769, [0x0a16] = 0x27d60, [0x0a17] = 0x27ebe, [0x0a18] = 0x27eb9,
  [0x0a19] = 0x27eb7, [0x0a1a] = 0x27eb6, [0x0a1b] = 0x47a1, [0x0a1c] = 0x27ebd,
  [0x0a1d] = 0x27eb8, [0x0a1e] = 0x27ec1, [0x0a1f] = 0x27ff7, [0x0a20] = 0x27ff3,
  [0x0a21] = 0x27ff4, [0x0a22] = 0x27ffb, [0x0a23] = 0x27ffd, [0x0a24] = 0x27ff5,
  [0x0a25] = 0x47e7, [0x0a26] = 0x27ffa, [0x0a27] = 0x28014, [0x0a28] = 0x47ec,
  [0x0a29] = 0x27ff2, [0x0a2a] = 0x27ffc, [0x0a2b] = 0x28000, [0x0a2c] = 0x47df,
  [0x0a2d] = 0x2822b, [0x0a2e] = 0x28229, [0x0a2f] = 0x2822e, [0x0a30] = 0x4833,
  [0x0a31] = 0x282d7, [0x0a32] = 0x282db, [0x0a33] = 0x282d9, [0x0a34] = 0x282d8,
  [0x0a35] = 0x282e6, [0x0a36] = 0x282d6, [0x0a37] = 0x282d1, [0x0a38] = 0x282e7,
  [0x0a39] = 0x282d0, [0x0a3a] = 0x282dc, [0x0a3b] = 0x284ed, [0x0a3c] = 0x2851a,
  [0x0a3d] = 0x28519, [0x0a3e] = 0x28516, [0x0a3f] = 0x284ec, [0x0a40] = 0x284ef,
  [0x0a41] = 0x284ee, [0x0a42] = 0x284f4, [0x0a43] = 0x2852f, [0x0a44] = 0x28730,
  [0x0a45] = 0x28719, [0x0a46] = 0x28718, [0x0a47] = 0x28711, [0x0a48] = 0x28716,
  [0x0a49] = 0x28712, [0x0a4a] = 0x28710, [0x0a4b] = 0x28714, [0x0a4c] = 0x2880c,
  [0x0a4d] = 0x2880d, [0x0a4e] = 0x28910, [0x0a4f] = 0x28929, [0x0a50] = 0x28928,
  [0x0a51] = 0x28c65, [0x0a52] = 0x28c66, [0x0a53] = 0x28cde, [0x0a54] = 0x28cdf,
  [0x0a55] = 0x28ce0, [0x0a56] = 0x28cd9, [0x0a57] = 0x28cdb, [0x0a58] = 0x28cd8,
  [0x0a59] = 0x28ce3, [0x0a5a] = 0x28cd7, [0x0a5b] = 0x28ea6, [0x0a5c] = 0x28ea7,
  [0x0a5d] = 0x28ebf, [0x0a5e] = 0x28ea8, [0x0a5f] = 0x28e9f, [0x0a60] = 0x28ea1,
  [0x0a61] = 0x28ea2, [0x0a62] = 0x28ead, [0x0a63] = 0x28ea3, [0x0a64] = 0x28ea0,
  [0x0a65] = 0x29083, [0x0a66] = 0x29085, [0x0a67] = 0x29082, [0x0a68] = 0x29084,
  [0x0a69] = 0x291ee, [0x0a6a] = 0x2925d, [0x0a6b] = 0x29392, [0x0a6c] = 0x29455,
  [0x0a6d] = 0x29453, [0x0a6e] = 0x2f9fc, [0x0a6f] = 0x29451, [0x0a70] = 0x29452,
  [0x0a71] = 0x29450, [0x0a72] = 0x2959d, [0x0a73] = 0x29687, [0x0a74] = 0x29697,
  [0x0a75] = 0x2969b, [0x0a76] = 0x2969d, [0x0a77] = 0x29869, [0x0a78] = 0x29a11,
  [0x0a79] = 0x4bee, [0x0a7a] = 0x29c7a, [0x0a7b] = 0x4c32, [0x0a7c] = 0x23244,
  [0x0a7d] = 0x200c6, [0x0a7e] = 0x20340, [0x0a7f] = 0x2032e, [0x0a80] = 0x20368,
  [0x0a81] = 0x2032d, [0x0a82] = 0x20334, [0x0a83] = 0x20332, [0x0a84] = 0x20369,
  [0x0a85] = 0x20331, [0x0a86] = 0x2036a, [0x0a87] = 0x2033e, [0x0a88] = 0x2036f,
  [0x0a89] = 0x20375, [0x0a8a] = 0x204f6, [0x0a8b] = 0x20520, [0x0a8c] = 0x20528,
  [0x0a8d] = 0x20567, [0x0a8e] = 0x205fb, [0x0a8f] = 0x205fe, [0x0a90] = 0x205fc,
  [0x0a91] = 0x205fd, [0x0a92] = 0x205ff, [0x0a93] = 0x205fa, [0x0a94] = 0x207ac,
  [0x0a95] = 0x207af, [0x0a96] = 0x207b1, [0x0a97] = 0x34fc, [0x0a98] = 0x207bf,
  [0x0a99] = 0x207be, [0x0a9a] = 0x207a7, [0x0a9b] = 0x207aa, [0x0a9c] = 0x20894,
  [0x0a9d] = 0x20893, [0x0a9e] = 0x20899, [0x0a9f] = 0x352a, [0x0aa0] = 0x208fa,
  [0x0aa1] = 0x20956, [0x0aa2] = 0x20a22, [0x0aa3] = 0x20a91, [0x0aa4] = 0x20a92,
  [0x0aa5] = 0x20b08, [0x0aa6] = 0x20b06, [0x0aa7] = 0x20dd6, [0x0aa8] = 0x20e37,
  [0x0aa9] = 0x20e4b, [0x0aaa] = 0x20ece, [0x0aab] = 0x20ed0, [0x0aac] = 0x20e39,
  [0x0aad] = 0x20e64, [0x0aae] = 0x20ecf, [0x0aaf] = 0x20e46, [0x0ab0] = 0x20e66,
  [0x0ab1] = 0x20e56, [0x0ab2] = 0x20e43, [0x0ab3] = 0x20e4a, [0x0ab4] = 0x20e58,
  [0x0ab5] = 0x20e65, [0x0ab6] = 0x20e5a, [0x0ab7] = 0x20e5b, [0x0ab8] = 0x20e55,
  [0x0ab9] = 0x20e50, [0x0aba] = 0x20e4c, [0x0abb] = 0x20ecd, [0x0abc] = 0x20e51,
  [0x0abd] = 0x20e47, [0x0abe] = 0x35d3, [0x0abf] = 0x20e53, [0x0ac0] = 0x20e62,
  [0x0ac1] = 0x20ecc, [0x0ac2] = 0x35d7, [0x0ac3] = 0x20e7f, [0x0ac4] = 0x20e4d,
  [0x0ac5] = 0x2120a, [0x0ac6] = 0x2120c, [0x0ac7] = 0x2120f, [0x0ac8] = 0x213bb,
  [0x0ac9] = 0x213b9, [0x0aca] = 0x213eb, [0x0acb] = 0x213ae, [0x0acc] = 0x213c1,
  [0x0acd] = 0x213ea, [0x0ace] = 0x213be, [0x0acf] = 0x213bf, [0x0ad0] = 0x213ba,
  [0x0ad1] = 0x213b7, [0x0ad2] = 0x213b3, [0x0ad3] = 0x213e9, [0x0ad4] = 0x2156e,
  [0x0ad5] = 0x215af, [0x0ad6] = 0x215b3, [0x0ad7] = 0x215b2, [0x0ad8] = 0x21668,
  [0x0ad9] = 0x21669, [0x0ada] = 0x2166a, [0x0adb] = 0x217de, [0x0adc] = 0x217eb,
  [0x0add] = 0x21811, [0x0ade] = 0x217ed, [0x0adf] = 0x217e9, [0x0ae0] = 0x217f7,
  [0x0ae1] = 0x217f4, [0x0ae2] = 0x217ea, [0x0ae3] = 0x21812, [0x0ae4] = 0x217e8,
  [0x0ae5] = 0x21813, [0x0ae6] = 0x2198e, [0x0ae7] = 0x21986, [0x0ae8] = 0x21a63,
  [0x0ae9] = 0x21a65, [0x0aea] = 0x21b99, [0x0aeb] = 0x21cc2, [0x0aec] = 0x21cae,
  [0x0aed] = 0x21cb0, [0x0aee] = 0x21cad, [0x0aef] = 0x21eb4, [0x0af0] = 0x21ed1,
  [0x0af1] = 0x21ead, [0x0af2] = 0x37f3, [0x0af3] = 0x21ebd, [0x0af4] = 0x21eb7,
  [0x0af5] = 0x21eab, [0x0af6] = 0x21eaf, [0x0af7] = 0x21ec1, [0x0af8] = 0x21eb5,
  [0x0af9] = 0x2210f, [0x0afa] = 0x2210e, [0x0afb] = 0x2211c, [0x0afc] = 0x22113,
  [0x0afd] = 0x2210a, [0x0afe] = 0x2211e, [0x0aff] = 0x22114, [0x0b00] = 0x2211a,
  [0x0b01] = 0x221cb, [0x0b02] = 0x22287, [0x0b03] = 0x22293, [0x0b04] = 0x3891,
  [0x0b05] = 0x22284, [0x0b06] = 0x22285, [0x0b07] = 0x22292, [0x0b08] = 0x22280,
  [0x0b09] = 0x2232b, [0x0b0a] = 0x22363, [0x0b0b] = 0x2240a, [0x0b0c] = 0x38c0,
  [0x0b0d] = 0x22466, [0x0b0e] = 0x22534, [0x0b0f] = 0x22533, [0x0b10] = 0x2254d,
  [0x0b11] = 0x22789, [0x0b12] = 0x22757, [0x0b13] = 0x6117, [0x0b14] = 0x2274a,
  [0x0b15] = 0x3963, [0x0b16] = 0x22739, [0x0b17] = 0x2273f, [0x0b18] = 0x22790,
  [0x0b19] = 0x2278f, [0x0b1a] = 0x227b2, [0x0b1b] = 0x2279f, [0x0b1c] = 0x22795,
  [0x0b1d] = 0x227a1, [0x0b1e] = 0x2279c, [0x0b1f] = 0x2278e, [0x0b20] = 0x227a0,
  [0x0b21] = 0x227ae, [0x0b22] = 0x22796, [0x0b23] = 0x227ca, [0x0b24] = 0x227ac,
  [0x0b25] = 0x3970, [0x0b26] = 0x227a6, [0x0b27] = 0x22797, [0x0b28] = 0x22793,
  [0x0b29] = 0x203b6, [0x0b2a] = 0x229d6, [0x0b2b] = 0x229e0, [0x0b2c] = 0x229d8,
  [0x0b2d] = 0x22a56, [0x0b2e] = 0x22c6e, [0x0b2f] = 0x22c7e, [0x0b30] = 0x22c74,
  [0x0b31] = 0x22c77, [0x0b32] = 0x22c80, [0x0b33] = 0x22c7a, [0x0b34] = 0x22c6b,
  [0x0b35] = 0x22c5f, [0x0b36] = 0x22c76, [0x0b37] = 0x22cb0, [0x0b38] = 0x22c63,
  [0x0b39] = 0x22f92, [0x0b3a] = 0x22f82, [0x0b3b] = 0x22f9d, [0x0b3c] = 0x22f94,
  [0x0b3d] = 0x22f83, [0x0b3e] = 0x22f8b, [0x0b3f] = 0x22f84, [0x0b40] = 0x23057,
  [0x0b41] = 0x23084, [0x0b42] = 0x230b5, [0x0b43] = 0x23109, [0x0b44] = 0x231e6,
  [0x0b45] = 0x3b02, [0x0b46] = 0x23225, [0x0b47] = 0x23251, [0x0b48] = 0x2333e,
  [0x0b49] = 0x2f8da, [0x0b4a] = 0x23385, [0x0b4b] = 0x2353f, [0x0b4c] = 0x23542,
  [0x0b4d] = 0x23538, [0x0b4e] = 0x2353a, [0x0b4f] = 0x2354e, [0x0b50] = 0x23545,
  [0x0b51] = 0x23539, [0x0b52] = 0x23537, [0x0b53] = 0x23544, [0x0b54] = 0x2f8e7,
  [0x0b55] = 0x235a4, [0x0b56] = 0x2353d, [0x0b57] = 0x23541, [0x0b58] = 0x238df,
  [0x0b59] = 0x238de, [0x0b5a] = 0x238ca, [0x0b5b] = 0x238e9, [0x0b5c] = 0x238e2,
  [0x0b5d] = 0x238e0, [0x0b5e] = 0x23989, [0x0b5f] = 0x23987, [0x0b60] = 0x23a32,
  [0x0b61] = 0x23a33, [0x0b62] = 0x23aac, [0x0b63] = 0x23af9, [0x0b64] = 0x23b0d,
  [0x0b65] = 0x23bb9, [0x0b66] = 0x23e5e, [0x0b67] = 0x23e6a, [0x0b68] = 0x3d35,
  [0x0b69] = 0x23e81, [0x0b6a] = 0x23e69, [0x0b6b] = 0x23eac, [0x0b6c] = 0x23e80,
  [0x0b6d] = 0x23e75, [0x0b6e] = 0x23e71, [0x0b6f] = 0x23e76, [0x0b70] = 0x23e6f,
  [0x0b71] = 0x242b6, [0x0b72] = 0x24303, [0x0b73] = 0x242bf, [0x0b74] = 0x24301,
  [0x0b75] = 0x242b2, [0x0b76] = 0x242c4, [0x0b77] = 0x245db, [0x0b78] = 0x245dc,
  [0x0b79] = 0x245d8, [0x0b7a] = 0x29672, [0x0b7b] = 0x246b5, [0x0b7c] = 0x246a8,
  [0x0b7d] = 0x2468f, [0x0b7e] = 0x2469e, [0x0b7f] = 0x2468e, [0x0b80] = 0x247de,
  [0x0b81] = 0x247e2, [0x0b82] = 0x24820, [0x0b83] = 0x2481d, [0x0b84] = 0x2481a,
  [0x0b85] = 0x24819, [0x0b86] = 0x2482b, [0x0b87] = 0x24818, [0x0b88] = 0x24810,
  [0x0b89] = 0x2480e, [0x0b8a] = 0x24824, [0x0b8b] = 0x2481f, [0x0b8c] = 0x24813,
  [0x0b8d] = 0x24835, [0x0b8e] = 0x24983, [0x0b8f] = 0x3ee0, [0x0b90] = 0x249d7,
  [0x0b91] = 0x24b02, [0x0b92] = 0x24b03, [0x0b93] = 0x24b5c, [0x0b94] = 0x24b5f,
  [0x0b95] = 0x24bc7, [0x0b96] = 0x24c9a, [0x0b97] = 0x24c97, [0x0b98] = 0x24ca3,
  [0x0b99] = 0x24df7, [0x0b9a] = 0x24dc6, [0x0b9b] = 0x24dcc, [0x0b9c] = 0x24dd6,
  [0x0b9d] = 0x24dcd, [0x0b9e] = 0x24df9, [0x0b9f] = 0x24dd9, [0x0ba0] = 0x24dd8,
  [0x0ba1] = 0x3f9a, [0x0ba2] = 0x24dd1, [0x0ba3] = 0x24dce, [0x0ba4] = 0x24dd5,
  [0x0ba5] = 0x3fa3, [0x0ba6] = 0x24dd4, [0x0ba7] = 0x24df8, [0x0ba8] = 0x24f7a,
  [0x0ba9] = 0x24f79, [0x0baa] = 0x24f7f, [0x0bab] = 0x24ff3, [0x0bac] = 0x24ff5,
  [0x0bad] = 0x24ff6, [0x0bae] = 0x24ff4, [0x0baf] = 0x24ff7, [0x0bb0] = 0x4005,
  [0x0bb1] = 0x25079, [0x0bb2] = 0x2519c, [0x0bb3] = 0x251df, [0x0bb4] = 0x251f5,
  [0x0bb5] = 0x251ff, [0x0bb6] = 0x251d1, [0x0bb7] = 0x251da, [0x0bb8] = 0x251cf,
  [0x0bb9] = 0x251cd, [0x0bba] = 0x251de, [0x0bbb] = 0x251d8, [0x0bbc] = 0x23aad,
  [0x0bbd] = 0x251e0, [0x0bbe] = 0x251fe, [0x0bbf] = 0x2519e, [0x0bc0] = 0x25375,
  [0x0bc1] = 0x25373, [0x0bc2] = 0x25372, [0x0bc3] = 0x253d8, [0x0bc4] = 0x254d0,
  [0x0bc5] = 0x254d6, [0x0bc6] = 0x254d7, [0x0bc7] = 0x254ca, [0x0bc8] = 0x254cd,
  [0x0bc9] = 0x2568a, [0x0bca] = 0x25689, [0x0bcb] = 0x2568b, [0x0bcc] = 0x257d3,
  [0x0bcd] = 0x257da, [0x0bce] = 0x257ce, [0x0bcf] = 0x257d2, [0x0bd0] = 0x415e,
  [0x0bd1] = 0x257d4, [0x0bd2] = 0x257d7, [0x0bd3] = 0x257d5, [0x0bd4] = 0x257fb,
  [0x0bd5] = 0x257cd, [0x0bd6] = 0x257fa, [0x0bd7] = 0x257d6, [0x0bd8] = 0x41a6,
  [0x0bd9] = 0x2599b, [0x0bda] = 0x2599e, [0x0bdb] = 0x2599d, [0x0bdc] = 0x259a4,
  [0x0bdd] = 0x2599f, [0x0bde] = 0x259a5, [0x0bdf] = 0x259a8, [0x0be0] = 0x25a8f,
  [0x0be1] = 0x25a8c, [0x0be2] = 0x25a95, [0x0be3] = 0x25b60, [0x0be4] = 0x25b55,
  [0x0be5] = 0x25b5c, [0x0be6] = 0x25b6f, [0x0be7] = 0x25b51, [0x0be8] = 0x25b61,
  [0x0be9] = 0x25ba7, [0x0bea] = 0x25b5e, [0x0beb] = 0x25b62, [0x0bec] = 0x25b54,
  [0x0bed] = 0x25b6e, [0x0bee] = 0x25b52, [0x0bef] = 0x25b58, [0x0bf0] = 0x25b6d,
  [0x0bf1] = 0x25b5f, [0x0bf2] = 0x25b6c, [0x0bf3] = 0x25b50, [0x0bf4] = 0x25b5d,
  [0x0bf5] = 0x25b8e, [0x0bf6] = 0x25e72, [0x0bf7] = 0x25e78, [0x0bf8] = 0x25e75,
  [0x0bf9] = 0x25e76, [0x0bfa] = 0x25e7c, [0x0bfb] = 0x25e7e, [0x0bfc] = 0x25e7d,
  [0x0bfd] = 0x26015, [0x0bfe] = 0x26014, [0x0bff] = 0x2601d, [0x0c00] = 0x2601c,
  [0x0c01] = 0x26023, [0x0c02] = 0x26057, [0x0c03] = 0x262c8, [0x0c04] = 0x435d,
  [0x0c05] = 0x262c9, [0x0c06] = 0x262c5, [0x0c07] = 0x26387, [0x0c08] = 0x2638a,
  [0x0c09] = 0x26394, [0x0c0a] = 0x26439, [0x0c0b] = 0x26440, [0x0c0c] = 0x2643e,
  [0x0c0d] = 0x26438, [0x0c0e] = 0x264d6, [0x0c0f] = 0x43b8, [0x0c10] = 0x26577,
  [0x0c11] = 0x26578, [0x0c12] = 0x2657d, [0x0c13] = 0x266dd, [0x0c14] = 0x2675e,
  [0x0c15] = 0x26777, [0x0c16] = 0x267d4, [0x0c17] = 0x26766, [0x0c18] = 0x2675f,
  [0x0c19] = 0x26773, [0x0c1a] = 0x26772, [0x0c1b] = 0x26797, [0x0c1c] = 0x2675a,
  [0x0c1d] = 0x26796, [0x0c1e] = 0x2675d, [0x0c1f] = 0x26765, [0x0c20] = 0x26770,
  [0x0c21] = 0x26775, [0x0c22] = 0x26774, [0x0c23] = 0x2675c, [0x0c24] = 0x26760,
  [0x0c25] = 0x4434, [0x0c26] = 0x2676d, [0x0c27] = 0x26941, [0x0c28] = 0x26a3c,
  [0x0c29] = 0x26a3e, [0x0c2a] = 0x446f, [0x0c2b] = 0x26a49, [0x0c2c] = 0x26be4,
  [0x0c2d] = 0x26d0c, [0x0c2e] = 0x26d15, [0x0c2f] = 0x26d0e, [0x0c30] = 0x26cd4,
  [0x0c31] = 0x26cc7, [0x0c32] = 0x26d14, [0x0c33] = 0x26cdb, [0x0c34] = 0x26d5f,
  [0x0c35] = 0x26ced, [0x0c36] = 0x26d09, [0x0c37] = 0x26ccf, [0x0c38] = 0x26cff,
  [0x0c39] = 0x26d1a, [0x0c3a] = 0x26cf5, [0x0c3b] = 0x26cee, [0x0c3c] = 0x26ce5,
  [0x0c3d] = 0x26d0d, [0x0c3e] = 0x26cdf, [0x0c3f] = 0x26cec, [0x0c40] = 0x26cd7,
  [0x0c41] = 0x26d2c, [0x0c42] = 0x26ccd, [0x0c43] = 0x26ce3, [0x0c44] = 0x26d00,
  [0x0c45] = 0x26ceb, [0x0c46] = 0x26cfd, [0x0c47] = 0x26d0f, [0x0c48] = 0x26d2d,
  [0x0c49] = 0x26d50, [0x0c4a] = 0x26cfe, [0x0c4b] = 0x26cc4, [0x0c4c] = 0x26ce1,
  [0x0c4d] = 0x26cf9, [0x0c4e] = 0x271bb, [0x0c4f] = 0x271bc, [0x0c50] = 0x4594,
  [0x0c51] = 0x4593, [0x0c52] = 0x271ba, [0x0c53] = 0x272d0, [0x0c54] = 0x272d6,
  [0x0c55] = 0x272ce, [0x0c56] = 0x272e1, [0x0c57] = 0x272d5, [0x0c58] = 0x272cf,
  [0x0c59] = 0x272d3, [0x0c5a] = 0x8714, [0x0c5b] = 0x272d4, [0x0c5c] = 0x275b5,
  [0x0c5d] = 0x27690, [0x0c5e] = 0x276a1, [0x0c5f] = 0x27681, [0x0c60] = 0x27685,
  [0x0c61] = 0x2766e, [0x0c62] = 0x27688, [0x0c63] = 0x27682, [0x0c64] = 0x27686,
  [0x0c65] = 0x2768f, [0x0c66] = 0x27689, [0x0c67] = 0x27680, [0x0c68] = 0x27684,
  [0x0c69] = 0x88d1, [0x0c6a] = 0x27829, [0x0c6b] = 0x2782b, [0x0c6c] = 0x2782a,
  [0x0c6d] = 0x27830, [0x0c6e] = 0x278e2, [0x0c6f] = 0x278e3, [0x0c70] = 0x278e1,
  [0x0c71] = 0x279d5, [0x0c72] = 0x279c4, [0x0c73] = 0x279c9, [0x0c74] = 0x279e0,
  [0x0c75] = 0x279df, [0x0c76] = 0x279cb, [0x0c77] = 0x279dd, [0x0c78] = 0x279e8,
  [0x0c79] = 0x279d4, [0x0c7a] = 0x27a1c, [0x0c7b] = 0x27bb5, [0x0c7c] = 0x27bb6,
  [0x0c7d] = 0x27be5, [0x0c7e] = 0x27c45, [0x0c7f] = 0x27c49, [0x0c80] = 0x27c40,
  [0x0c81] = 0x27c46, [0x0c82] = 0x27c44, [0x0c83] = 0x27c4a, [0x0c84] = 0x27cc9,
  [0x0c85] = 0x27cc4, [0x0c86] = 0x27cc2, [0x0c87] = 0x27d68, [0x0c88] = 0x8ccb,
  [0x0c89] = 0x27d82, [0x0c8a] = 0x27d6b, [0x0c8b] = 0x27d67, [0x0c8c] = 0x27d6c,
  [0x0c8d] = 0x27d6d, [0x0c8e] = 0x27d71, [0x0c8f] = 0x4792, [0x0c90] = 0x27ed7,
  [0x0c91] = 0x27ed2, [0x0c92] = 0x27ed9, [0x0c93] = 0x47aa, [0x0c94] = 0x27ebc,
  [0x0c95] = 0x27edd, [0x0c96] = 0x27ed6, [0x0c97] = 0x47a7, [0x0c98] = 0x27ed8,
  [0x0c99] = 0x28020, [0x0c9a] = 0x28016, [0x0c9b] = 0x28018, [0x0c9c] = 0x28015,
  [0x0c9d] = 0x28019, [0x0c9e] = 0x28027, [0x0c9f] = 0x28050, [0x0ca0] = 0x2801d,
  [0x0ca1] = 0x2802c, [0x0ca2] = 0x2801c, [0x0ca3] = 0x28029, [0x0ca4] = 0x2802b,
  [0x0ca5] = 0x28024, [0x0ca6] = 0x47ef, [0x0ca7] = 0x28028, [0x0ca8] = 0x28239,
  [0x0ca9] = 0x282f2, [0x0caa] = 0x282f0, [0x0cab] = 0x8eed, [0x0cac] = 0x282ed,
  [0x0cad] = 0x282ef, [0x0cae] = 0x282ea, [0x0caf] = 0x28301, [0x0cb0] = 0x284bc,
  [0x0cb1] = 0x2855c, [0x0cb2] = 0x2855b, [0x0cb3] = 0x2855a, [0x0cb4] = 0x28526,
  [0x0cb5] = 0x28743, [0x0cb6] = 0x28733, [0x0cb7] = 0x2873a, [0x0cb8] = 0x28737,
  [0x0cb9] = 0x28736, [0x0cba] = 0x2873c, [0x0cbb] = 0x28734, [0x0cbc] = 0x2874a,
  [0x0cbd] = 0x2873b, [0x0cbe] = 0x28735, [0x0cbf] = 0x2873d, [0x0cc0] = 0x28740,
  [0x0cc1] = 0x2873e, [0x0cc2] = 0x28717, [0x0cc3] = 0x28826, [0x0cc4] = 0x28824,
  [0x0cc5] = 0x28810, [0x0cc6] = 0x28836, [0x0cc7] = 0x28825, [0x0cc8] = 0x4922,
  [0x0cc9] = 0x28966, [0x0cca] = 0x28c7f, [0x0ccb] = 0x28c7a, [0x0ccc] = 0x28cfb,
  [0x0ccd] = 0x28cfd, [0x0cce] = 0x28cfc, [0x0ccf] = 0x28cfa, [0x0cd0] = 0x28ed6,
  [0x0cd1] = 0x28ed2, [0x0cd2] = 0x28ec8, [0x0cd3] = 0x28ec2, [0x0cd4] = 0x28ec7,
  [0x0cd5] = 0x28ecd, [0x0cd6] = 0x28ec1, [0x0cd7] = 0x28ec6, [0x0cd8] = 0x28ecc,
  [0x0cd9] = 0x28fa1, [0x0cda] = 0x29095, [0x0cdb] = 0x29094, [0x0cdc] = 0x29097,
  [0x0cdd] = 0x29096, [0x0cde] = 0x29093, [0x0cdf] = 0x290a1, [0x0ce0] = 0x29209,
  [0x0ce1] = 0x29208, [0x0ce2] = 0x29267, [0x0ce3] = 0x29265, [0x0ce4] = 0x29262,
  [0x0ce5] = 0x29261, [0x0ce6] = 0x29397, [0x0ce7] = 0x29401, [0x0ce8] = 0x29419,
  [0x0ce9] = 0x29418, [0x0cea] = 0x2945c, [0x0ceb] = 0x29467, [0x0cec] = 0x2946c,
  [0x0ced] = 0x4ab5, [0x0cee] = 0x2f9fd, [0x0cef] = 0x29466, [0x0cf0] = 0x29461,
  [0x0cf1] = 0x2946e, [0x0cf2] = 0x2945d, [0x0cf3] = 0x2945a, [0x0cf4] = 0x295a2,
  [0x0cf5] = 0x295a9, [0x0cf6] = 0x295a5, [0x0cf7] = 0x295a6, [0x0cf8] = 0x295b2,
  [0x0cf9] = 0x295a3, [0x0cfa] = 0x296ac, [0x0cfb] = 0x296aa, [0x0cfc] = 0x296af,
  [0x0cfd] = 0x296ab, [0x0cfe] = 0x296b1, [0x0cff] = 0x296c6, [0x0d00] = 0x296ad,
  [0x0d01] = 0x296b9, [0x0d02] = 0x29870, [0x0d03] = 0x4b75, [0x0d04] = 0x29c0c,
  [0x0d05] = 0x29c81, [0x0d06] = 0x29c83, [0x0d07] = 0x29c7e, [0x0d08] = 0x29d51,
  [0x0d09] = 0x29f99, [0x0d0a] = 0x29f94, [0x0d0b] = 0x29f95, [0x0d0c] = 0x2a2fe,
  [0x0d0d] = 0x2a418, [0x0d0e] = 0x20105, [0x0d0f] = 0x20372, [0x0d10] = 0x3482,
  [0x0d11] = 0x20399, [0x0d12] = 0x204ad, [0x0d13] = 0x20606, [0x0d14] = 0x207c2,
  [0x0d15] = 0x207cb, [0x0d16] = 0x207d5, [0x0d17] = 0x207d2, [0x0d18] = 0x207cc,
  [0x0d19] = 0x207d7, [0x0d1a] = 0x207c5, [0x0d1b] = 0x207ca, [0x0d1c] = 0x52ec,
  [0x0d1d] = 0x208a2, [0x0d1e] = 0x208a3, [0x0d1f] = 0x208a1, [0x0d20] = 0x52e8,
  [0x0d21] = 0x208a6, [0x0d22] = 0x208a4, [0x0d23] = 0x208ff, [0x0d24] = 0x20900,
  [0x0d25] = 0x3535, [0x0d26] = 0x20a06, [0x0d27] = 0x20a25, [0x0d28] = 0x20a9f,
  [0x0d29] = 0x20b7f, [0x0d2a] = 0x20eea, [0x0d2b] = 0x20eeb, [0x0d2c] = 0x20ede,
  [0x0d2d] = 0x20f18, [0x0d2e] = 0x20ee0, [0x0d2f] = 0x20ee8, [0x0d30] = 0x20edf,
  [0x0d31] = 0x20ef1, [0x0d32] = 0x20eec, [0x0d33] = 0x20f1a, [0x0d34] = 0x20ee6,
  [0x0d35] = 0x20ee1, [0x0d36] = 0x20f5d, [0x0d37] = 0x20077, [0x0d38] = 0x20f1e,
  [0x0d39] = 0x35f0, [0x0d3a] = 0x20f80, [0x0d3b] = 0x21217, [0x0d3c] = 0x2141d,
  [0x0d3d] = 0x213f7, [0x0d3e] = 0x213ef, [0x0d3f] = 0x213ee, [0x0d40] = 0x213ed,
  [0x0d41] = 0x213f5, [0x0d42] = 0x2141e, [0x0d43] = 0x215bd, [0x0d44] = 0x257ff,
  [0x0d45] = 0x215be, [0x0d46] = 0x2166d, [0x0d47] = 0x21676, [0x0d48] = 0x2184c,
  [0x0d49] = 0x21819, [0x0d4a] = 0x21848, [0x0d4b] = 0x2181e, [0x0d4c] = 0x2184a,
  [0x0d4d] = 0x21849, [0x0d4e] = 0x21826, [0x0d4f] = 0x21995, [0x0d50] = 0x21a85,
  [0x0d51] = 0x21a7e, [0x0d52] = 0x21a97, [0x0d53] = 0x21a7d, [0x0d54] = 0x21b42,
  [0x0d55] = 0x21b9e, [0x0d56] = 0x21c0c, [0x0d57] = 0x21c0b, [0x0d58] = 0x21c0e,
  [0x0d59] = 0x21ccf, [0x0d5a] = 0x21ed8, [0x0d5b] = 0x21ee0, [0x0d5c] = 0x21ee8,
  [0x0d5d] = 0x21ee9, [0x0d5e] = 0x21ee2, [0x0d5f] = 0x21ee1, [0x0d60] = 0x21eda,
  [0x0d61] = 0x21ee3, [0x0d62] = 0x21ed9, [0x0d63] = 0x21ef0, [0x0d64] = 0x21ee7,
  [0x0d65] = 0x21ee4, [0x0d66] = 0x22123, [0x0d67] = 0x22131, [0x0d68] = 0x22133,
  [0x0d69] = 0x2210c, [0x0d6a] = 0x269e0, [0x0d6b] = 0x22126, [0x0d6c] = 0x22124,
  [0x0d6d] = 0x22132, [0x0d6e] = 0x22127, [0x0d6f] = 0x22121, [0x0d70] = 0x22130,
  [0x0d71] = 0x221a2, [0x0d72] = 0x221cc, [0x0d73] = 0x22296, [0x0d74] = 0x2f88f,
  [0x0d75] = 0x22298, [0x0d76] = 0x2229c, [0x0d77] = 0x2229b, [0x0d78] = 0x2236b,
  [0x0d79] = 0x22412, [0x0d7a] = 0x22414, [0x0d7b] = 0x2255a, [0x0d7c] = 0x22554,
  [0x0d7d] = 0x22551, [0x0d7e] = 0x2255e, [0x0d7f] = 0x22558, [0x0d80] = 0x38f0,
  [0x0d81] = 0x22559, [0x0d82] = 0x2279a, [0x0d83] = 0x227f2, [0x0d84] = 0x227ad,
  [0x0d85] = 0x22820, [0x0d86] = 0x227f1, [0x0d87] = 0x227f0, [0x0d88] = 0x227f3,
  [0x0d89] = 0x227f4, [0x0d8a] = 0x227e9, [0x0d8b] = 0x227fe, [0x0d8c] = 0x2282e,
  [0x0d8d] = 0x227e8, [0x0d8e] = 0x227e7, [0x0d8f] = 0x227ff, [0x0d90] = 0x227f5,
  [0x0d91] = 0x229e5, [0x0d92] = 0x229e7, [0x0d93] = 0x229e6, [0x0d94] = 0x229e9,
  [0x0d95] = 0x22a58, [0x0d96] = 0x22c6c, [0x0d97] = 0x22cd9, [0x0d98] = 0x3a3e,
  [0x0d99] = 0x22ccb, [0x0d9a] = 0x3a39, [0x0d9b] = 0x22cbe, [0x0d9c] = 0x22cb5,
  [0x0d9d] = 0x22cd1, [0x0d9e] = 0x22cc2, [0x0d9f] = 0x22cb6, [0x0da0] = 0x22cd3,
  [0x0da1] = 0x2f8c2, [0x0da2] = 0x22eda, [0x0da3] = 0x3aa2, [0x0da4] = 0x22fae,
  [0x0da5] = 0x3aa1, [0x0da6] = 0x22fac, [0x0da7] = 0x22fa9, [0x0da8] = 0x22fba,
  [0x0da9] = 0x22fa7, [0x0daa] = 0x22fb3, [0x0dab] = 0x22fbd, [0x0dac] = 0x22fb1,
  [0x0dad] = 0x2310e, [0x0dae] = 0x23111, [0x0daf] = 0x2310f, [0x0db0] = 0x23252,
  [0x0db1] = 0x2326a, [0x0db2] = 0x23254, [0x0db3] = 0x23255, [0x0db4] = 0x23253,
  [0x0db5] = 0x23257, [0x0db6] = 0x2f8d0, [0x0db7] = 0x2325b, [0x0db8] = 0x23258,
  [0x0db9] = 0x23259, [0x0dba] = 0x3b3b, [0x0dbb] = 0x235b8, [0x0dbc] = 0x235b5,
  [0x0dbd] = 0x235bf, [0x0dbe] = 0x235c1, [0x0dbf] = 0x235bb, [0x0dc0] = 0x3bb7,
  [0x0dc1] = 0x235c3, [0x0dc2] = 0x235ad, [0x0dc3] = 0x235a8, [0x0dc4] = 0x235ae,
  [0x0dc5] = 0x235aa, [0x0dc6] = 0x235d0, [0x0dc7] = 0x235b1, [0x0dc8] = 0x235bc,
  [0x0dc9] = 0x235b7, [0x0dca] = 0x235b0, [0x0dcb] = 0x235fa, [0x0dcc] = 0x235fb,
  [0x0dcd] = 0x235b9, [0x0dce] = 0x23907, [0x0dcf] = 0x238fc, [0x0dd0] = 0x238ff,
  [0x0dd1] = 0x238f2, [0x0dd2] = 0x238f8, [0x0dd3] = 0x238f4, [0x0dd4] = 0x3c57,
  [0x0dd5] = 0x3c70, [0x0dd6] = 0x23aae, [0x0dd7] = 0x23b0f, [0x0dd8] = 0x23b10,
  [0x0dd9] = 0x29ac1, [0x0dda] = 0x23bc7, [0x0ddb] = 0x23bc6, [0x0ddc] = 0x23bce,
  [0x0ddd] = 0x3ca7, [0x0dde] = 0x23bdc, [0x0ddf] = 0x23ed5, [0x0de0] = 0x23ed3,
  [0x0de1] = 0x23f20, [0x0de2] = 0x23ee2, [0x0de3] = 0x23ef1, [0x0de4] = 0x23edb,
  [0x0de5] = 0x23eea, [0x0de6] = 0x23eeb, [0x0de7] = 0x23ee1, [0x0de8] = 0x23eff,
  [0x0de9] = 0x23ed6, [0x0dea] = 0x23ee0, [0x0deb] = 0x23ed7, [0x0dec] = 0x23ee5,
  [0x0ded] = 0x3d54, [0x0dee] = 0x2431b, [0x0def] = 0x24310, [0x0df0] = 0x2430f,
  [0x0df1] = 0x2430e, [0x0df2] = 0x24339, [0x0df3] = 0x2430a, [0x0df4] = 0x24314,
  [0x0df5] = 0x2430c, [0x0df6] = 0x24523, [0x0df7] = 0x2455a, [0x0df8] = 0x245e2,
  [0x0df9] = 0x24693, [0x0dfa] = 0x246b0, [0x0dfb] = 0x246ab, [0x0dfc] = 0x246bd,
  [0x0dfd] = 0x246b1, [0x0dfe] = 0x2468d, [0x0dff] = 0x246ad, [0x0e00] = 0x246aa,
  [0x0e01] = 0x246ac, [0x0e02] = 0x246bc, [0x0e03] = 0x246c9, [0x0e04] = 0x24847,
  [0x0e05] = 0x24862, [0x0e06] = 0x24840, [0x0e07] = 0x24844, [0x0e08] = 0x2483f,
  [0x0e09] = 0x24843, [0x0e0a] = 0x2483d, [0x0e0b] = 0x24860, [0x0e0c] = 0x2485f,
  [0x0e0d] = 0x2483a, [0x0e0e] = 0x24842, [0x0e0f] = 0x2485e, [0x0e10] = 0x2485d,
  [0x0e11] = 0x249bc, [0x0e12] = 0x249e0, [0x0e13] = 0x249ee, [0x0e14] = 0x249eb,
  [0x0e15] = 0x24a0d, [0x0e16] = 0x24b09, [0x0e17] = 0x24b0c, [0x0e18] = 0x24b69,
  [0x0e19] = 0x24b74, [0x0e1a] = 0x24b67, [0x0e1b] = 0x3f31, [0x0e1c] = 0x7527,
  [0x0e1d] = 0x24cad, [0x0e1e] = 0x24dff, [0x0e1f] = 0x24e16, [0x0e20] = 0x24e08,
  [0x0e21] = 0x24e09, [0x0e22] = 0x24e00, [0x0e23] = 0x24dfc, [0x0e24] = 0x24e13,
  [0x0e25] = 0x24e14, [0x0e26] = 0x24e0a, [0x0e27] = 0x2983e, [0x0e28] = 0x24e29,
  [0x0e29] = 0x24e17, [0x0e2a] = 0x24e3b, [0x0e2b] = 0x24dfb, [0x0e2c] = 0x3fbf,
  [0x0e2d] = 0x24e0e, [0x0e2e] = 0x3fe4, [0x0e2f] = 0x25008, [0x0e30] = 0x2521b,
  [0x0e31] = 0x2521a, [0x0e32] = 0x25219, [0x0e33] = 0x25202, [0x0e34] = 0x404a,
  [0x0e35] = 0x25217, [0x0e36] = 0x25207, [0x0e37] = 0x25213, [0x0e38] = 0x25209,
  [0x0e39] = 0x2520a, [0x0e3a] = 0x25212, [0x0e3b] = 0x252aa, [0x0e3c] = 0x25238,
  [0x0e3d] = 0x2537b, [0x0e3e] = 0x2537c, [0x0e3f] = 0x2537a, [0x0e40] = 0x253ea,
  [0x0e41] = 0x253eb, [0x0e42] = 0x253ef, [0x0e43] = 0x253f0, [0x0e44] = 0x25509,
  [0x0e45] = 0x25502, [0x0e46] = 0x25501, [0x0e47] = 0x2552b, [0x0e48] = 0x25507,
  [0x0e49] = 0x40cf, [0x0e4a] = 0x25528, [0x0e4b] = 0x254fb, [0x0e4c] = 0x2552a,
  [0x0e4d] = 0x254fe, [0x0e4e] = 0x25504, [0x0e4f] = 0x25529, [0x0e50] = 0x2550a,
  [0x0e51] = 0x798c, [0x0e52] = 0x256c2, [0x0e53] = 0x7991, [0x0e54] = 0x256a6,
  [0x0e55] = 0x4114, [0x0e56] = 0x22fc6, [0x0e57] = 0x21c07, [0x0e58] = 0x2580a,
  [0x0e59] = 0x257fd, [0x0e5a] = 0x25806, [0x0e5b] = 0x2580d, [0x0e5c] = 0x25809,
  [0x0e5d] = 0x2580b, [0x0e5e] = 0x25800, [0x0e5f] = 0x25805, [0x0e60] = 0x2583d,
  [0x0e61] = 0x259d5, [0x0e62] = 0x259bf, [0x0e63] = 0x259ba, [0x0e64] = 0x259c7,
  [0x0e65] = 0x25a98, [0x0e66] = 0x25a9a, [0x0e67] = 0x420d, [0x0e68] = 0x25ba4,
  [0x0e69] = 0x25b9c, [0x0e6a] = 0x25b9d, [0x0e6b] = 0x25ba2, [0x0e6c] = 0x25b96,
  [0x0e6d] = 0x25b95, [0x0e6e] = 0x25ba5, [0x0e6f] = 0x25baa, [0x0e70] = 0x25bad,
  [0x0e71] = 0x25b9a, [0x0e72] = 0x25ba3, [0x0e73] = 0x25b97, [0x0e74] = 0x25b90,
  [0x0e75] = 0x25baf, [0x0e76] = 0x25b8d, [0x0e77] = 0x4201, [0x0e78] = 0x25ba8,
  [0x0e79] = 0x25bee, [0x0e7a] = 0x25ea3, [0x0e7b] = 0x25e99, [0x0e7c] = 0x25ea1,
  [0x0e7d] = 0x25e9c, [0x0e7e] = 0x25ea4, [0x0e7f] = 0x25ecb, [0x0e80] = 0x26056,
  [0x0e81] = 0x26048, [0x0e82] = 0x26059, [0x0e83] = 0x2604a, [0x0e84] = 0x2605c,
  [0x0e85] = 0x42f1, [0x0e86] = 0x2604f, [0x0e87] = 0x2604d, [0x0e88] = 0x2605d,
  [0x0e89] = 0x2623a, [0x0e8a] = 0x262e3, [0x0e8b] = 0x262e1, [0x0e8c] = 0x262de,
  [0x0e8d] = 0x26397, [0x0e8e] = 0x2639a, [0x0e8f] = 0x26448, [0x0e90] = 0x2644a,
  [0x0e91] = 0x2644c, [0x0e92] = 0x2644e, [0x0e93] = 0x2644f, [0x0e94] = 0x2644b,
  [0x0e95] = 0x2644d, [0x0e96] = 0x26447, [0x0e97] = 0x26451, [0x0e98] = 0x264da,
  [0x0e99] = 0x264f9, [0x0e9a] = 0x264fc, [0x0e9b] = 0x264fb, [0x0e9c] = 0x264fd,
  [0x0e9d] = 0x264f8, [0x0e9e] = 0x264fa, [0x0e9f] = 0x2658a, [0x0ea0] = 0x2658e,
  [0x0ea1] = 0x26588, [0x0ea2] = 0x26589, [0x0ea3] = 0x2658d, [0x0ea4] = 0x26590,
  [0x0ea5] = 0x2f97e, [0x0ea6] = 0x2671c, [0x0ea7] = 0x267ab, [0x0ea8] = 0x267a4,
  [0x0ea9] = 0x267a8, [0x0eaa] = 0x267ad, [0x0eab] = 0x2679f, [0x0eac] = 0x2679a,
  [0x0ead] = 0x267b0, [0x0eae] = 0x267a5, [0x0eaf] = 0x267d5, [0x0eb0] = 0x267a2,
  [0x0eb1] = 0x267b2, [0x0eb2] = 0x2679d, [0x0eb3] = 0x267a1, [0x0eb4] = 0x26915,
  [0x0eb5] = 0x26942, [0x0eb6] = 0x26973, [0x0eb7] = 0x26984, [0x0eb8] = 0x269dd,
  [0x0eb9] = 0x26a5c, [0x0eba] = 0x26a4c, [0x0ebb] = 0x4476, [0x0ebc] = 0x26a4b,
  [0x0ebd] = 0x26c67, [0x0ebe] = 0x26da8, [0x0ebf] = 0x26d7f, [0x0ec0] = 0x26d8d,
  [0x0ec1] = 0x26d8b, [0x0ec2] = 0x26df7, [0x0ec3] = 0x26da9, [0x0ec4] = 0x26df4,
  [0x0ec5] = 0x26d9a, [0x0ec6] = 0x26d8c, [0x0ec7] = 0x26d95, [0x0ec8] = 0x26d7e,
  [0x0ec9] = 0x26d77, [0x0eca] = 0x450a, [0x0ecb] = 0x4503, [0x0ecc] = 0x26d71,
  [0x0ecd] = 0x26dfd, [0x0ece] = 0x26d88, [0x0ecf] = 0x26d99, [0x0ed0] = 0x26d6d,
  [0x0ed1] = 0x26dd4, [0x0ed2] = 0x26d91, [0x0ed3] = 0x26d7d, [0x0ed4] = 0x26d63,
  [0x0ed5] = 0x26d75, [0x0ed6] = 0x26df9, [0x0ed7] = 0x26d8e, [0x0ed8] = 0x26d74,
  [0x0ed9] = 0x26dd9, [0x0eda] = 0x26d66, [0x0edb] = 0x26dfa, [0x0edc] = 0x26df5,
  [0x0edd] = 0x8660, [0x0ede] = 0x271c1, [0x0edf] = 0x271c9, [0x0ee0] = 0x27303,
  [0x0ee1] = 0x2732c, [0x0ee2] = 0x27334, [0x0ee3] = 0x27317, [0x0ee4] = 0x27309,
  [0x0ee5] = 0x27304, [0x0ee6] = 0x27306, [0x0ee7] = 0x2731a, [0x0ee8] = 0x2730e,
  [0x0ee9] = 0x27327, [0x0eea] = 0x27305, [0x0eeb] = 0x27307, [0x0eec] = 0x27319,
  [0x0eed] = 0x27314, [0x0eee] = 0x2730c, [0x0eef] = 0x2731d, [0x0ef0] = 0x27322,
  [0x0ef1] = 0x27323, [0x0ef2] = 0x275bb, [0x0ef3] = 0x275bc, [0x0ef4] = 0x275ba,
  [0x0ef5] = 0x276a9, [0x0ef6] = 0x276b5, [0x0ef7] = 0x276ac, [0x0ef8] = 0x276aa,
  [0x0ef9] = 0x276af, [0x0efa] = 0x276b7, [0x0efb] = 0x4635, [0x0efc] = 0x4636,
  [0x0efd] = 0x2783d, [0x0efe] = 0x27845, [0x0eff] = 0x27848, [0x0f00] = 0x2783c,
  [0x0f01] = 0x2783e, [0x0f02] = 0x278ec, [0x0f03] = 0x278ed, [0x0f04] = 0x278e9,
  [0x0f05] = 0x27a0a, [0x0f06] = 0x27a0b, [0x0f07] = 0x279f8, [0x0f08] = 0x27a04,
  [0x0f09] = 0x279f6, [0x0f0a] = 0x279f9, [0x0f0b] = 0x27c53, [0x0f0c] = 0x27c52,
  [0x0f0d] = 0x27c60, [0x0f0e] = 0x27c63, [0x0f0f] = 0x27cce, [0x0f10] = 0x27ccd,
  [0x0f11] = 0x27cd0, [0x0f12] = 0x27ccc, [0x0f13] = 0x27d87, [0x0f14] = 0x27d8d,
  [0x0f15] = 0x27d9d, [0x0f16] = 0x4773, [0x0f17] = 0x27e63, [0x0f18] = 0x27ef4,
  [0x0f19] = 0x27ef1, [0x0f1a] = 0x27eff, [0x0f1b] = 0x27ef5, [0x0f1c] = 0x27efc,
  [0x0f1d] = 0x27ef2, [0x0f1e] = 0x27ef6, [0x0f1f] = 0x47b1, [0x0f20] = 0x27efa,
  [0x0f21] = 0x47af, [0x0f22] = 0x2802e, [0x0f23] = 0x28045, [0x0f24] = 0x28043,
  [0x0f25] = 0x28051, [0x0f26] = 0x2804c, [0x0f27] = 0x28073, [0x0f28] = 0x28047,
  [0x0f29] = 0x2804b, [0x0f2a] = 0x2804f, [0x0f2b] = 0x47f4, [0x0f2c] = 0x28044,
  [0x0f2d] = 0x2804d, [0x0f2e] = 0x2804e, [0x0f2f] = 0x2804a, [0x0f30] = 0x28041,
  [0x0f31] = 0x28052, [0x0f32] = 0x2809f, [0x0f33] = 0x28054, [0x0f34] = 0x28059,
  [0x0f35] = 0x280a2, [0x0f36] = 0x2824b, [0x0f37] = 0x28248, [0x0f38] = 0x2830d,
  [0x0f39] = 0x2831f, [0x0f3a] = 0x28308, [0x0f3b] = 0x2830e, [0x0f3c] = 0x2831d,
  [0x0f3d] = 0x28304, [0x0f3e] = 0x28418, [0x0f3f] = 0x2841b, [0x0f40] = 0x28566,
  [0x0f41] = 0x28562, [0x0f42] = 0x28583, [0x0f43] = 0x28567, [0x0f44] = 0x28762,
  [0x0f45] = 0x2874f, [0x0f46] = 0x2874e, [0x0f47] = 0x28767, [0x0f48] = 0x28753,
  [0x0f49] = 0x2874d, [0x0f4a] = 0x2874c, [0x0f4b] = 0x28768, [0x0f4c] = 0x28751,
  [0x0f4d] = 0x48dc, [0x0f4e] = 0x28758, [0x0f4f] = 0x2875b, [0x0f50] = 0x28750,
  [0x0f51] = 0x2876a, [0x0f52] = 0x28839, [0x0f53] = 0x28838, [0x0f54] = 0x28982,
  [0x0f55] = 0x28984, [0x0f56] = 0x28986, [0x0f57] = 0x28c80, [0x0f58] = 0x28c86,
  [0x0f59] = 0x28d25, [0x0f5a] = 0x28d11, [0x0f5b] = 0x4999, [0x0f5c] = 0x28d0e,
  [0x0f5d] = 0x28d15, [0x0f5e] = 0x28d0f, [0x0f5f] = 0x28d18, [0x0f60] = 0x28d0d,
  [0x0f61] = 0x28d13, [0x0f62] = 0x28d16, [0x0f63] = 0x28d19, [0x0f64] = 0x28f08,
  [0x0f65] = 0x28ef7, [0x0f66] = 0x28efb, [0x0f67] = 0x28efc, [0x0f68] = 0x28fb4,
  [0x0f69] = 0x28fbe, [0x0f6a] = 0x28fc1, [0x0f6b] = 0x290a6, [0x0f6c] = 0x290a5,
  [0x0f6d] = 0x290a2, [0x0f6e] = 0x290a4, [0x0f6f] = 0x29212, [0x0f70] = 0x2920e,
  [0x0f71] = 0x2927a, [0x0f72] = 0x29278, [0x0f73] = 0x2927d, [0x0f74] = 0x2927e,
  [0x0f75] = 0x29281, [0x0f76] = 0x29283, [0x0f77] = 0x29280, [0x0f78] = 0x29282,
  [0x0f79] = 0x2939b, [0x0f7a] = 0x29399, [0x0f7b] = 0x2939c, [0x0f7c] = 0x2939a,
  [0x0f7d] = 0x293a1, [0x0f7e] = 0x2941a, [0x0f7f] = 0x2941c, [0x0f80] = 0x2948a,
  [0x0f81] = 0x29479, [0x0f82] = 0x29473, [0x0f83] = 0x29475, [0x0f84] = 0x2947c,
  [0x0f85] = 0x29478, [0x0f86] = 0x29460, [0x0f87] = 0x29483, [0x0f88] = 0x29470,
  [0x0f89] = 0x4ab6, [0x0f8a] = 0x4abd, [0x0f8b] = 0x29487, [0x0f8c] = 0x29484,
  [0x0f8d] = 0x2947b, [0x0f8e] = 0x2947e, [0x0f8f] = 0x29474, [0x0f90] = 0x295b9,
  [0x0f91] = 0x295b8, [0x0f92] = 0x295b7, [0x0f93] = 0x295b5, [0x0f94] = 0x296cc,
  [0x0f95] = 0x296ce, [0x0f96] = 0x296d1, [0x0f97] = 0x9904, [0x0f98] = 0x296cb,
  [0x0f99] = 0x999b, [0x0f9a] = 0x2987e, [0x0f9b] = 0x2987a, [0x0f9c] = 0x29877,
  [0x0f9d] = 0x2988c, [0x0f9e] = 0x298e1, [0x0f9f] = 0x29a20, [0x0fa0] = 0x29a1e,
  [0x0fa1] = 0x29ac2, [0x0fa2] = 0x29b0f, [0x0fa3] = 0x29b00, [0x0fa4] = 0x4bf4,
  [0x0fa5] = 0x29b10, [0x0fa6] = 0x29c0d, [0x0fa7] = 0x29c0e, [0x0fa8] = 0x29c8e,
  [0x0fa9] = 0x29c8d, [0x0faa] = 0x29c8a, [0x0fab] = 0x29c8b, [0x0fac] = 0x29c8c,
  [0x0fad] = 0x29c8f, [0x0fae] = 0x29d57, [0x0faf] = 0x4c35, [0x0fb0] = 0x29d55,
  [0x0fb1] = 0x29d5b, [0x0fb2] = 0x29fac, [0x0fb3] = 0x29fa7, [0x0fb4] = 0x29fa0,
  [0x0fb5] = 0x29f9e, [0x0fb6] = 0x4caa, [0x0fb7] = 0x2a28d, [0x0fb8] = 0x4d1f,
  [0x0fb9] = 0x2a393, [0x0fba] = 0x2a394, [0x0fbb] = 0x2a392, [0x0fbc] = 0x20181,
  [0x0fbd] = 0x203d6, [0x0fbe] = 0x203c3, [0x0fbf] = 0x203e4, [0x0fc0] = 0x203e2,
  [0x0fc1] = 0x20409, [0x0fc2] = 0x203bf, [0x0fc3] = 0x203c8, [0x0fc4] = 0x203c7,
  [0x0fc5] = 0x204b5, [0x0fc6] = 0x204b2, [0x0fc7] = 0x2060c, [0x0fc8] = 0x2060b,
  [0x0fc9] = 0x2060a, [0x0fca] = 0x20605, [0x0fcb] = 0x207d3, [0x0fcc] = 0x207e6,
  [0x0fcd] = 0x207e9, [0x0fce] = 0x207f0, [0x0fcf] = 0x207ea, [0x0fd0] = 0x207e7,
  [0x0fd1] = 0x208b1, [0x0fd2] = 0x208b3, [0x0fd3] = 0x208b4, [0x0fd4] = 0x208b0,
  [0x0fd5] = 0x2095c, [0x0fd6] = 0x2095d, [0x0fd7] = 0x2095e, [0x0fd8] = 0x209c0,
  [0x0fd9] = 0x20aaf, [0x0fda] = 0x3550, [0x0fdb] = 0x20b86, [0x0fdc] = 0x20b83,
  [0x0fdd] = 0x20b84, [0x0fde] = 0x20f5c, [0x0fdf] = 0x20f70, [0x0fe0] = 0x20f8c,
  [0x0fe1] = 0x20f7b, [0x0fe2] = 0x20f66, [0x0fe3] = 0x20f79, [0x0fe4] = 0x20f63,
  [0x0fe5] = 0x20fc5, [0x0fe6] = 0x20f6b, [0x0fe7] = 0x20f6d, [0x0fe8] = 0x20f72,
  [0x0fe9] = 0x20f69, [0x0fea] = 0x20f75, [0x0feb] = 0x20f89, [0x0fec] = 0x20fc6,
  [0x0fed] = 0x20fa5, [0x0fee] = 0x203e7, [0x0fef] = 0x20f6a, [0x0ff0] = 0x20f97,
  [0x0ff1] = 0x2145a, [0x0ff2] = 0x2145c, [0x0ff3] = 0x21459, [0x0ff4] = 0x21424,
  [0x0ff5] = 0x21425, [0x0ff6] = 0x21430, [0x0ff7] = 0x21458, [0x0ff8] = 0x21431,
  [0x0ff9] = 0x21576, [0x0ffa] = 0x215c1, [0x0ffb] = 0x21685, [0x0ffc] = 0x2167d,
  [0x0ffd] = 0x21865, [0x0ffe] = 0x21887, [0x0fff] = 0x21852, [0x1000] = 0x21854,
  [0x1001] = 0x2188a, [0x1002] = 0x21850, [0x1003] = 0x21886, [0x1004] = 0x2184f,
  [0x1005] = 0x21868, [0x1006] = 0x2199d, [0x1007] = 0x21aa0, [0x1008] = 0x21aba,
  [0x1009] = 0x21abd, [0x100a] = 0x21ab8, [0x100b] = 0x21ba6, [0x100c] = 0x377b,
  [0x100d] = 0x21cde, [0x100e] = 0x21cd8, [0x100f] = 0x21cd1, [0x1010] = 0x21f13,
  [0x1011] = 0x3809, [0x1012] = 0x21f0e, [0x1013] = 0x21f1b, [0x1014] = 0x21f3a,
  [0x1015] = 0x3807, [0x1016] = 0x21f1c, [0x1017] = 0x21f12, [0x1018] = 0x21f16,
  [0x1019] = 0x21f1a, [0x101a] = 0x21448, [0x101b] = 0x2214b, [0x101c] = 0x22137,
  [0x101d] = 0x22136, [0x101e] = 0x22138, [0x101f] = 0x2213a, [0x1020] = 0x22149,
  [0x1021] = 0x2213c, [0x1022] = 0x2214a, [0x1023] = 0x222b1, [0x1024] = 0x222c2,
  [0x1025] = 0x222b5, [0x1026] = 0x222c4, [0x1027] = 0x222b6, [0x1028] = 0x2241e,
  [0x1029] = 0x2241f, [0x102a] = 0x224af, [0x102b] = 0x2256b, [0x102c] = 0x2256f,
  [0x102d] = 0x22792, [0x102e] = 0x227e4, [0x102f] = 0x2282c, [0x1030] = 0x227ef,
  [0x1031] = 0x6184, [0x1032] = 0x2284f, [0x1033] = 0x22852, [0x1034] = 0x22850,
  [0x1035] = 0x2283a, [0x1036] = 0x22837, [0x1037] = 0x22847, [0x1038] = 0x22864,
  [0x1039] = 0x22840, [0x103a] = 0x2283c, [0x103b] = 0x22845, [0x103c] = 0x22841,
  [0x103d] = 0x229f3, [0x103e] = 0x39bb, [0x103f] = 0x22cc1, [0x1040] = 0x22cb4,
  [0x1041] = 0x3a4b, [0x1042] = 0x22d2a, [0x1043] = 0x22d22, [0x1044] = 0x22d29,
  [0x1045] = 0x3a4d, [0x1046] = 0x22d3e, [0x1047] = 0x22d3c, [0x1048] = 0x22d30,
  [0x1049] = 0x22fc7, [0x104a] = 0x22fd5, [0x104b] = 0x22fd6, [0x104c] = 0x22fd3,
  [0x104d] = 0x22fce, [0x104e] = 0x22fc8, [0x104f] = 0x2305c, [0x1050] = 0x2305f,
  [0x1051] = 0x23062, [0x1052] = 0x3ac2, [0x1053] = 0x230bd, [0x1054] = 0x230bf,
  [0x1055] = 0x2325c, [0x1056] = 0x232b6, [0x1057] = 0x3b1a, [0x1058] = 0x2329d,
  [0x1059] = 0x2327f, [0x105a] = 0x3b12, [0x105b] = 0x23294, [0x105c] = 0x23281,
  [0x105d] = 0x23347, [0x105e] = 0x23346, [0x105f] = 0x3b3c, [0x1060] = 0x23394,
  [0x1061] = 0x23675, [0x1062] = 0x2361a, [0x1063] = 0x2362a, [0x1064] = 0x23632,
  [0x1065] = 0x23617, [0x1066] = 0x23623, [0x1067] = 0x23674, [0x1068] = 0x3bc3,
  [0x1069] = 0x23676, [0x106a] = 0x2362e, [0x106b] = 0x23625, [0x106c] = 0x23620,
  [0x106d] = 0x23671, [0x106e] = 0x23616, [0x106f] = 0x23670, [0x1070] = 0x2362c,
  [0x1071] = 0x2362f, [0x1072] = 0x2361f, [0x1073] = 0x23664, [0x1074] = 0x3bc0,
  [0x1075] = 0x23908, [0x1076] = 0x23914, [0x1077] = 0x2390a, [0x1078] = 0x2390b,
  [0x1079] = 0x23a4f, [0x107a] = 0x23a59, [0x107b] = 0x23a54, [0x107c] = 0x23a51,
  [0x107d] = 0x23ab6, [0x107e] = 0x23bec, [0x107f] = 0x23bed, [0x1080] = 0x23bea,
  [0x1081] = 0x23eee, [0x1082] = 0x23f5a, [0x1083] = 0x23f73, [0x1084] = 0x23f65,
  [0x1085] = 0x23f61, [0x1086] = 0x23f55, [0x1087] = 0x23f6b, [0x1088] = 0x23f64,
  [0x1089] = 0x23f5b, [0x108a] = 0x23f4c, [0x108b] = 0x23f6f, [0x108c] = 0x23f84,
  [0x108d] = 0x23f70, [0x108e] = 0x2435d, [0x108f] = 0x24357, [0x1090] = 0x24366,
  [0x1091] = 0x24353, [0x1092] = 0x24398, [0x1093] = 0x24397, [0x1094] = 0x24528,
  [0x1095] = 0x245eb, [0x1096] = 0x245ea, [0x1097] = 0x245e8, [0x1098] = 0x245ec,
  [0x1099] = 0x245ef, [0x109a] = 0x246d1, [0x109b] = 0x246ca, [0x109c] = 0x729f,
  [0x109d] = 0x246cf, [0x109e] = 0x246cd, [0x109f] = 0x246ce, [0x10a0] = 0x24870,
  [0x10a1] = 0x24867, [0x10a2] = 0x24873, [0x10a3] = 0x2486d, [0x10a4] = 0x24876,
  [0x10a5] = 0x24879, [0x10a6] = 0x2486a, [0x10a7] = 0x2488b, [0x10a8] = 0x24872,
  [0x10a9] = 0x24871, [0x10aa] = 0x2486e, [0x10ab] = 0x2487a, [0x10ac] = 0x24a16,
  [0x10ad] = 0x24a39, [0x10ae] = 0x24a3b, [0x10af] = 0x24b10, [0x10b0] = 0x24b96,
  [0x10b1] = 0x24b7b, [0x10b2] = 0x24b7c, [0x10b3] = 0x24b7d, [0x10b4] = 0x24cbe,
  [0x10b5] = 0x24e5c, [0x10b6] = 0x24e38, [0x10b7] = 0x24e39, [0x10b8] = 0x24e2e,
  [0x10b9] = 0x24e34, [0x10ba] = 0x24e32, [0x10bb] = 0x24e5b, [0x10bc] = 0x24e33,
  [0x10bd] = 0x24e3c, [0x10be] = 0x3fe5, [0x10bf] = 0x2500e, [0x10c0] = 0x2500f,
  [0x10c1] = 0x25010, [0x10c2] = 0x25087, [0x10c3] = 0x2525b, [0x10c4] = 0x25253,
  [0x10c5] = 0x405f, [0x10c6] = 0x2525c, [0x10c7] = 0x2523f, [0x10c8] = 0x25259,
  [0x10c9] = 0x2524a, [0x10ca] = 0x25244, [0x10cb] = 0x2524c, [0x10cc] = 0x25240,
  [0x10cd] = 0x2525f, [0x10ce] = 0x2525e, [0x10cf] = 0x2524e, [0x10d0] = 0x25254,
  [0x10d1] = 0x25243, [0x10d2] = 0x25287, [0x10d3] = 0x25382, [0x10d4] = 0x25385,
  [0x10d5] = 0x25531, [0x10d6] = 0x25547, [0x10d7] = 0x2552e, [0x10d8] = 0x2552f,
  [0x10d9] = 0x25548, [0x10da] = 0x25534, [0x10db] = 0x4118, [0x10dc] = 0x2059f,
  [0x10dd] = 0x25832, [0x10de] = 0x25836, [0x10df] = 0x25833, [0x10e0] = 0x25831,
  [0x10e1] = 0x25840, [0x10e2] = 0x25841, [0x10e3] = 0x259dd, [0x10e4] = 0x259d2,
  [0x10e5] = 0x259d9, [0x10e6] = 0x41ad, [0x10e7] = 0x259df, [0x10e8] = 0x259db,
  [0x10e9] = 0x259d8, [0x10ea] = 0x259d3, [0x10eb] = 0x259de, [0x10ec] = 0x259e0,
  [0x10ed] = 0x259d4, [0x10ee] = 0x259d7, [0x10ef] = 0x259da, [0x10f0] = 0x25aa7,
  [0x10f1] = 0x25bfe, [0x10f2] = 0x25bdd, [0x10f3] = 0x25c40, [0x10f4] = 0x25be2,
  [0x10f5] = 0x421e, [0x10f6] = 0x25bd6, [0x10f7] = 0x25bde, [0x10f8] = 0x25bef,
  [0x10f9] = 0x4227, [0x10fa] = 0x25beb, [0x10fb] = 0x25bea, [0x10fc] = 0x25be4,
  [0x10fd] = 0x25bd1, [0x10fe] = 0x4218, [0x10ff] = 0x25bec, [0x1100] = 0x25bfa,
  [0x1101] = 0x25bd9, [0x1102] = 0x25bf3, [0x1103] = 0x25be1, [0x1104] = 0x25bdc,
  [0x1105] = 0x25be5, [0x1106] = 0x25bdf, [0x1107] = 0x25bd4, [0x1108] = 0x4220,
  [0x1109] = 0x25c05, [0x110a] = 0x25ed6, [0x110b] = 0x25ed5, [0x110c] = 0x25ed0,
  [0x110d] = 0x25ec8, [0x110e] = 0x25ec4, [0x110f] = 0x25ec9, [0x1110] = 0x25ecd,
  [0x1111] = 0x25ed2, [0x1112] = 0x25ecc, [0x1113] = 0x25eca, [0x1114] = 0x260a0,
  [0x1115] = 0x26092, [0x1116] = 0x42f4, [0x1117] = 0x4302, [0x1118] = 0x2608b,
  [0x1119] = 0x260a1, [0x111a] = 0x26095, [0x111b] = 0x26088, [0x111c] = 0x26086,
  [0x111d] = 0x2608d, [0x111e] = 0x26085, [0x111f] = 0x26091, [0x1120] = 0x26089,
  [0x1121] = 0x260a2, [0x1122] = 0x262f0, [0x1123] = 0x262f3, [0x1124] = 0x262f4,
  [0x1125] = 0x262ef, [0x1126] = 0x26306, [0x1127] = 0x263a2, [0x1128] = 0x2645e,
  [0x1129] = 0x26463, [0x112a] = 0x26460, [0x112b] = 0x2645b, [0x112c] = 0x2647b,
  [0x112d] = 0x26458, [0x112e] = 0x26459, [0x112f] = 0x2645d, [0x1130] = 0x26505,
  [0x1131] = 0x26506, [0x1132] = 0x26502, [0x1133] = 0x26503, [0x1134] = 0x26504,
  [0x1135] = 0x265a2, [0x1136] = 0x265a7, [0x1137] = 0x265a6, [0x1138] = 0x265a4,
  [0x1139] = 0x265ac, [0x113a] = 0x265a9, [0x113b] = 0x267e7, [0x113c] = 0x26801,
  [0x113d] = 0x267df, [0x113e] = 0x267d9, [0x113f] = 0x267e3, [0x1140] = 0x2680f,
  [0x1141] = 0x2691a, [0x1142] = 0x26945, [0x1143] = 0x26948, [0x1144] = 0x2697b,
  [0x1145] = 0x26985, [0x1146] = 0x26986, [0x1147] = 0x269fe, [0x1148] = 0x269ff,
  [0x1149] = 0x26a64, [0x114a] = 0x26a71, [0x114b] = 0x26a72, [0x114c] = 0x26a61,
  [0x114d] = 0x26a62, [0x114e] = 0x26a6c, [0x114f] = 0x26a60, [0x1150] = 0x26a5d,
  [0x1151] = 0x26a63, [0x1152] = 0x26a67, [0x1153] = 0x26e01, [0x1154] = 0x26e2a,
  [0x1155] = 0x26e30, [0x1156] = 0x26e06, [0x1157] = 0x26e7d, [0x1158] = 0x26e22,
  [0x1159] = 0x26eb9, [0x115a] = 0x26e0a, [0x115b] = 0x26e10, [0x115c] = 0x26d8f,
  [0x115d] = 0x26e80, [0x115e] = 0x26e13, [0x115f] = 0x26e2e, [0x1160] = 0x26e18,
  [0x1161] = 0x26e17, [0x1162] = 0x26e1a, [0x1163] = 0x26e14, [0x1164] = 0x26e47,
  [0x1165] = 0x26e63, [0x1166] = 0x26e38, [0x1167] = 0x4524, [0x1168] = 0x26e7f,
  [0x1169] = 0x26e21, [0x116a] = 0x26e37, [0x116b] = 0x26e31, [0x116c] = 0x26e1b,
  [0x116d] = 0x26ea5, [0x116e] = 0x26f6c, [0x116f] = 0x271d3, [0x1170] = 0x27388,
  [0x1171] = 0x2736e, [0x1172] = 0x2735c, [0x1173] = 0x27374, [0x1174] = 0x27354,
  [0x1175] = 0x27383, [0x1176] = 0x8770, [0x1177] = 0x27365, [0x1178] = 0x27360,
  [0x1179] = 0x27370, [0x117a] = 0x2736f, [0x117b] = 0x2736d, [0x117c] = 0x27372,
  [0x117d] = 0x2736c, [0x117e] = 0x27376, [0x117f] = 0x27375, [0x1180] = 0x27359,
  [0x1181] = 0x273c7, [0x1182] = 0x275c1, [0x1183] = 0x275f9, [0x1184] = 0x276e3,
  [0x1185] = 0x276e7, [0x1186] = 0x276d6, [0x1187] = 0x276cf, [0x1188] = 0x276da,
  [0x1189] = 0x276e9, [0x118a] = 0x276d2, [0x118b] = 0x27855, [0x118c] = 0x2784f,
  [0x118d] = 0x2784d, [0x118e] = 0x27851, [0x118f] = 0x27856, [0x1190] = 0x278f9,
  [0x1191] = 0x278f8, [0x1192] = 0x27906, [0x1193] = 0x27903, [0x1194] = 0x27909,
  [0x1195] = 0x278f7, [0x1196] = 0x278f5, [0x1197] = 0x27a47, [0x1198] = 0x27a45,
  [0x1199] = 0x27a3e, [0x119a] = 0x27a46, [0x119b] = 0x27a29, [0x119c] = 0x27a34,
  [0x119d] = 0x27a3f, [0x119e] = 0x27a44, [0x119f] = 0x27a35, [0x11a0] = 0x27a5a,
  [0x11a1] = 0x27a2a, [0x11a2] = 0x27a33, [0x11a3] = 0x27a37, [0x11a4] = 0x27a43,
  [0x11a5] = 0x27a39, [0x11a6] = 0x27a30, [0x11a7] = 0x27a5d, [0x11a8] = 0x27a32,
  [0x11a9] = 0x27a27, [0x11aa] = 0x27bbd, [0x11ab] = 0x27bba, [0x11ac] = 0x27bbc,
  [0x11ad] = 0x27bf1, [0x11ae] = 0x27bf0, [0x11af] = 0x27c61, [0x11b0] = 0x27c5c,
  [0x11b1] = 0x27c58, [0x11b2] = 0x27c5d, [0x11b3] = 0x27cda, [0x11b4] = 0x27cd9,
  [0x11b5] = 0x27cdd, [0x11b6] = 0x27cdc, [0x11b7] = 0x27cde, [0x11b8] = 0x477d,
  [0x11b9] = 0x27da0, [0x11ba] = 0x27da2, [0x11bb] = 0x27f0f, [0x11bc] = 0x27f13,
  [0x11bd] = 0x27f12, [0x11be] = 0x27f11, [0x11bf] = 0x27f14, [0x11c0] = 0x27f19,
  [0x11c1] = 0x27f0e, [0x11c2] = 0x27f17, [0x11c3] = 0x27f21, [0x11c4] = 0x27f20,
  [0x11c5] = 0x27f16, [0x11c6] = 0x28078, [0x11c7] = 0x8e37, [0x11c8] = 0x28087,
  [0x11c9] = 0x28079, [0x11ca] = 0x28080, [0x11cb] = 0x28077, [0x11cc] = 0x28081,
  [0x11cd] = 0x28046, [0x11ce] = 0x2807a, [0x11cf] = 0x2809c, [0x11d0] = 0x28083,
  [0x11d1] = 0x28084, [0x11d2] = 0x280ad, [0x11d3] = 0x2809d, [0x11d4] = 0x2809e,
  [0x11d5] = 0x28255, [0x11d6] = 0x28336, [0x11d7] = 0x2832b, [0x11d8] = 0x28327,
  [0x11d9] = 0x28346, [0x11da] = 0x2832c, [0x11db] = 0x28345, [0x11dc] = 0x28333,
  [0x11dd] = 0x2832d, [0x11de] = 0x28334, [0x11df] = 0x28322, [0x11e0] = 0x4864,
  [0x11e1] = 0x2841f, [0x11e2] = 0x285b3, [0x11e3] = 0x2858c, [0x11e4] = 0x2858f,
  [0x11e5] = 0x285af, [0x11e6] = 0x285ad, [0x11e7] = 0x2858e, [0x11e8] = 0x285ac,
  [0x11e9] = 0x285b0, [0x11ea] = 0x285b1, [0x11eb] = 0x285ae, [0x11ec] = 0x28599,
  [0x11ed] = 0x286eb, [0x11ee] = 0x28774, [0x11ef] = 0x28775, [0x11f0] = 0x2878e,
  [0x11f1] = 0x2876d, [0x11f2] = 0x28770, [0x11f3] = 0x2878c, [0x11f4] = 0x2884f,
  [0x11f5] = 0x28851, [0x11f6] = 0x28858, [0x11f7] = 0x2884c, [0x11f8] = 0x2884e,
  [0x11f9] = 0x28915, [0x11fa] = 0x289be, [0x11fb] = 0x289db, [0x11fc] = 0x289b3,
  [0x11fd] = 0x289ae, [0x11fe] = 0x28c87, [0x11ff] = 0x28c8a, [0x1200] = 0x28c88,
  [0x1201] = 0x28c8b, [0x1202] = 0x28c8c, [0x1203] = 0x28d44, [0x1204] = 0x28d2d,
  [0x1205] = 0x28d2a, [0x1206] = 0x28d31, [0x1207] = 0x28d2c, [0x1208] = 0x28d45,
  [0x1209] = 0x28d30, [0x120a] = 0x28d29, [0x120b] = 0x28d46, [0x120c] = 0x28ef4,
  [0x120d] = 0x28f14, [0x120e] = 0x28f10, [0x120f] = 0x28f0f, [0x1210] = 0x28f12,
  [0x1211] = 0x28f0b, [0x1212] = 0x28f0c, [0x1213] = 0x28f0a, [0x1214] = 0x28f13,
  [0x1215] = 0x28f0e, [0x1216] = 0x28fd9, [0x1217] = 0x28fd0, [0x1218] = 0x9703,
  [0x1219] = 0x290c1, [0x121a] = 0x290bf, [0x121b] = 0x290bd, [0x121c] = 0x290bc,
  [0x121d] = 0x290ba, [0x121e] = 0x290bb, [0x121f] = 0x290d1, [0x1220] = 0x290be,
  [0x1221] = 0x290d0, [0x1222] = 0x290b9, [0x1223] = 0x2921a, [0x1224] = 0x2921c,
  [0x1225] = 0x2921b, [0x1226] = 0x29291, [0x1227] = 0x29296, [0x1228] = 0x2929f,
  [0x1229] = 0x2929c, [0x122a] = 0x2929a, [0x122b] = 0x2929d, [0x122c] = 0x293ad,
  [0x122d] = 0x293a5, [0x122e] = 0x293ae, [0x122f] = 0x29403, [0x1230] = 0x29426,
  [0x1231] = 0x29420, [0x1232] = 0x29423, [0x1233] = 0x29424, [0x1234] = 0x29421,
  [0x1235] = 0x29428, [0x1236] = 0x29425, [0x1237] = 0x2941e, [0x1238] = 0x29494,
  [0x1239] = 0x29493, [0x123a] = 0x2948f, [0x123b] = 0x2949a, [0x123c] = 0x294ad,
  [0x123d] = 0x4ac2, [0x123e] = 0x295ca, [0x123f] = 0x295c5, [0x1240] = 0x296bb,
  [0x1241] = 0x296e1, [0x1242] = 0x296ea, [0x1243] = 0x296e4, [0x1244] = 0x296ed,
  [0x1245] = 0x296e6, [0x1246] = 0x296e0, [0x1247] = 0x296e8, [0x1248] = 0x296e5,
  [0x1249] = 0x2981a, [0x124a] = 0x29894, [0x124b] = 0x29896, [0x124c] = 0x2989e,
  [0x124d] = 0x29895, [0x124e] = 0x298a1, [0x124f] = 0x2988e, [0x1250] = 0x2989b,
  [0x1251] = 0x29892, [0x1252] = 0x29897, [0x1253] = 0x29899, [0x1254] = 0x29893,
  [0x1255] = 0x29a32, [0x1256] = 0x29a2f, [0x1257] = 0x29a2e, [0x1258] = 0x29a33,
  [0x1259] = 0x29b1c, [0x125a] = 0x29b1e, [0x125b] = 0x29b11, [0x125c] = 0x29b20,
  [0x125d] = 0x29b1f, [0x125e] = 0x29b19, [0x125f] = 0x29b16, [0x1260] = 0x29cac,
  [0x1261] = 0x29cb9, [0x1262] = 0x29cb3, [0x1263] = 0x29ca6, [0x1264] = 0x29ca2,
  [0x1265] = 0x29ca9, [0x1266] = 0x29ca7, [0x1267] = 0x29d6c, [0x1268] = 0x29d6d,
  [0x1269] = 0x29d69, [0x126a] = 0x29d80, [0x126b] = 0x29d66, [0x126c] = 0x29d65,
  [0x126d] = 0x29d71, [0x126e] = 0x29d6b, [0x126f] = 0x29fbd, [0x1270] = 0x29fc1,
  [0x1271] = 0x29fd3, [0x1272] = 0x29fb6, [0x1273] = 0x4cac, [0x1274] = 0x29fb9,
  [0x1275] = 0x29fd4, [0x1276] = 0x29fb7, [0x1277] = 0x29fdb, [0x1278] = 0x29fb8,
  [0x1279] = 0x29fc0, [0x127a] = 0x29fbc, [0x127b] = 0x29fd5, [0x127c] = 0x29fbf,
  [0x127d] = 0x29fc3, [0x127e] = 0x29fc9, [0x127f] = 0x2a258, [0x1280] = 0x2a290,
  [0x1281] = 0x2a295, [0x1282] = 0x2a297, [0x1283] = 0x2a309, [0x1284] = 0x2a308,
  [0x1285] = 0x2a306, [0x1286] = 0x2a305, [0x1287] = 0x2a310, [0x1288] = 0x2a397,
  [0x1289] = 0x2a395, [0x128a] = 0x2a3ee, [0x128b] = 0x2a41f, [0x128c] = 0x2fa1a,
  [0x128d] = 0x21b4b, [0x128e] = 0x2018e, [0x128f] = 0x203e8, [0x1290] = 0x203f0,
  [0x1291] = 0x203f4, [0x1292] = 0x20406, [0x1293] = 0x203ed, [0x1294] = 0x349b,
  [0x1295] = 0x203e9, [0x1296] = 0x20400, [0x1297] = 0x204b8, [0x1298] = 0x2056d,
  [0x1299] = 0x20804, [0x129a] = 0x20801, [0x129b] = 0x20803, [0x129c] = 0x20802,
  [0x129d] = 0x52f6, [0x129e] = 0x3526, [0x129f] = 0x20962, [0x12a0] = 0x20aba,
  [0x12a1] = 0x20b13, [0x12a2] = 0x20fde, [0x12a3] = 0x20fd7, [0x12a4] = 0x20fe4,
  [0x12a5] = 0x20fce, [0x12a6] = 0x20fe3, [0x12a7] = 0x20fdd, [0x12a8] = 0x2103a,
  [0x12a9] = 0x20fdb, [0x12aa] = 0x20fd6, [0x12ab] = 0x20fe0, [0x12ac] = 0x20fd4,
  [0x12ad] = 0x20fcb, [0x12ae] = 0x20fe1, [0x12af] = 0x20fc9, [0x12b0] = 0x20fdf,
  [0x12b1] = 0x2103c, [0x12b2] = 0x20ffc, [0x12b3] = 0x20fcf, [0x12b4] = 0x21479,
  [0x12b5] = 0x21476, [0x12b6] = 0x21478, [0x12b7] = 0x21687, [0x12b8] = 0x2189c,
  [0x12b9] = 0x218b1, [0x12ba] = 0x218c1, [0x12bb] = 0x2189a, [0x12bc] = 0x21892,
  [0x12bd] = 0x41af, [0x12be] = 0x218c2, [0x12bf] = 0x21896, [0x12c0] = 0x218c0,
  [0x12c1] = 0x21891, [0x12c2] = 0x21895, [0x12c3] = 0x219a2, [0x12c4] = 0x21ac3,
  [0x12c5] = 0x21ac0, [0x12c6] = 0x21ac2, [0x12c7] = 0x20b8a, [0x12c8] = 0x21bb0,
  [0x12c9] = 0x21c16, [0x12ca] = 0x21f46, [0x12cb] = 0x21f4a, [0x12cc] = 0x21f3e,
  [0x12cd] = 0x21f45, [0x12ce] = 0x21f42, [0x12cf] = 0x3815, [0x12d0] = 0x21f5b,
  [0x12d1] = 0x21f44, [0x12d2] = 0x22004, [0x12d3] = 0x24022, [0x12d4] = 0x22155,
  [0x12d5] = 0x22157, [0x12d6] = 0x22151, [0x12d7] = 0x2214e, [0x12d8] = 0x2215a,
  [0x12d9] = 0x222c6, [0x12da] = 0x222c3, [0x12db] = 0x389c, [0x12dc] = 0x222c5,
  [0x12dd] = 0x222cc, [0x12de] = 0x22371, [0x12df] = 0x22426, [0x12e0] = 0x224b2,
  [0x12e1] = 0x2257a, [0x12e2] = 0x22584, [0x12e3] = 0x2257b, [0x12e4] = 0x22874,
  [0x12e5] = 0x2288c, [0x12e6] = 0x22851, [0x12e7] = 0x22848, [0x12e8] = 0x2288d,
  [0x12e9] = 0x22892, [0x12ea] = 0x22898, [0x12eb] = 0x22893, [0x12ec] = 0x228b0,
  [0x12ed] = 0x2284e, [0x12ee] = 0x22896, [0x12ef] = 0x22897, [0x12f0] = 0x2289c,
  [0x12f1] = 0x228ca, [0x12f2] = 0x22d33, [0x12f3] = 0x22d83, [0x12f4] = 0x22d84,
  [0x12f5] = 0x22d9a, [0x12f6] = 0x22d8c, [0x12f7] = 0x64cc, [0x12f8] = 0x22d99,
  [0x12f9] = 0x22ee0, [0x12fa] = 0x22fe4, [0x12fb] = 0x22ff8, [0x12fc] = 0x22fed,
  [0x12fd] = 0x22ff3, [0x12fe] = 0x22ff4, [0x12ff] = 0x22ff5, [0x1300] = 0x22ffd,
  [0x1301] = 0x23119, [0x1302] = 0x2329e, [0x1303] = 0x232c4, [0x1304] = 0x2329f,
  [0x1305] = 0x2339b, [0x1306] = 0x2339f, [0x1307] = 0x2339a, [0x1308] = 0x236aa,
  [0x1309] = 0x2369d, [0x130a] = 0x23692, [0x130b] = 0x236a2, [0x130c] = 0x236af,
  [0x130d] = 0x236eb, [0x130e] = 0x236a0, [0x130f] = 0x236a1, [0x1310] = 0x23694,
  [0x1311] = 0x23698, [0x1312] = 0x2368f, [0x1313] = 0x23687, [0x1314] = 0x23684,
  [0x1315] = 0x236a9, [0x1316] = 0x2367c, [0x1317] = 0x23918, [0x1318] = 0x23a5f,
  [0x1319] = 0x23a62, [0x131a] = 0x23a61, [0x131b] = 0x23ac0, [0x131c] = 0x23b15,
  [0x131d] = 0x23bfc, [0x131e] = 0x23bf9, [0x131f] = 0x23fc8, [0x1320] = 0x23fc9,
  [0x1321] = 0x23fde, [0x1322] = 0x23fca, [0x1323] = 0x23fe2, [0x1324] = 0x3d64,
  [0x1325] = 0x3d6e, [0x1326] = 0x24004, [0x1327] = 0x23fcc, [0x1328] = 0x23fdd,
  [0x1329] = 0x23fe4, [0x132a] = 0x23fd3, [0x132b] = 0x23fc7, [0x132c] = 0x23fc6,
  [0x132d] = 0x3d5f, [0x132e] = 0x24037, [0x132f] = 0x243d8, [0x1330] = 0x243ee,
  [0x1331] = 0x243b2, [0x1332] = 0x243a3, [0x1333] = 0x243b3, [0x1334] = 0x243ed,
  [0x1335] = 0x243f8, [0x1336] = 0x24531, [0x1337] = 0x2455b, [0x1338] = 0x24566,
  [0x1339] = 0x24569, [0x133a] = 0x24596, [0x133b] = 0x2459b, [0x133c] = 0x245f7,
  [0x133d] = 0x245f3, [0x133e] = 0x245f4, [0x133f] = 0x245f5, [0x1340] = 0x246e2,
  [0x1341] = 0x246e7, [0x1342] = 0x246e5, [0x1343] = 0x246e9, [0x1344] = 0x246e6,
  [0x1345] = 0x246e3, [0x1346] = 0x24874, [0x1347] = 0x2487f, [0x1348] = 0x2488f,
  [0x1349] = 0x24890, [0x134a] = 0x24897, [0x134b] = 0x248a3, [0x134c] = 0x2488e,
  [0x134d] = 0x24898, [0x134e] = 0x2488c, [0x134f] = 0x24a42, [0x1350] = 0x24a44,
  [0x1351] = 0x24a69, [0x1352] = 0x24a43, [0x1353] = 0x24a68, [0x1354] = 0x3ef6,
  [0x1355] = 0x24b8d, [0x1356] = 0x24b88, [0x1357] = 0x24b8b, [0x1358] = 0x24b89,
  [0x1359] = 0x24ccc, [0x135a] = 0x24d1f, [0x135b] = 0x24e80, [0x135c] = 0x24e5e,
  [0x135d] = 0x24e67, [0x135e] = 0x24e68, [0x135f] = 0x24e65, [0x1360] = 0x764a,
  [0x1361] = 0x24e74, [0x1362] = 0x24e69, [0x1363] = 0x24e61, [0x1364] = 0x24e62,
  [0x1365] = 0x24e6c, [0x1366] = 0x24e93, [0x1367] = 0x24e86, [0x1368] = 0x24f9d,
  [0x1369] = 0x24f9b, [0x136a] = 0x2501b, [0x136b] = 0x25016, [0x136c] = 0x25019,
  [0x136d] = 0x25014, [0x136e] = 0x25018, [0x136f] = 0x25015, [0x1370] = 0x25099,
  [0x1371] = 0x25098, [0x1372] = 0x2525d, [0x1373] = 0x25289, [0x1374] = 0x2527a,
  [0x1375] = 0x2527d, [0x1376] = 0x2524b, [0x1377] = 0x4073, [0x1378] = 0x25278,
  [0x1379] = 0x2527f, [0x137a] = 0x25277, [0x137b] = 0x2527e, [0x137c] = 0x25279,
  [0x137d] = 0x252ab, [0x137e] = 0x2527c, [0x137f] = 0x25274, [0x1380] = 0x25275,
  [0x1381] = 0x252a7, [0x1382] = 0x2538d, [0x1383] = 0x2554e, [0x1384] = 0x25566,
  [0x1385] = 0x25561, [0x1386] = 0x2554d, [0x1387] = 0x2554f, [0x1388] = 0x40dd,
  [0x1389] = 0x25554, [0x138a] = 0x2557c, [0x138b] = 0x256da, [0x138c] = 0x256de,
  [0x138d] = 0x256d8, [0x138e] = 0x256dd, [0x138f] = 0x256df, [0x1390] = 0x25866,
  [0x1391] = 0x25862, [0x1392] = 0x2585f, [0x1393] = 0x25864, [0x1394] = 0x25863,
  [0x1395] = 0x25860, [0x1396] = 0x25888, [0x1397] = 0x2586a, [0x1398] = 0x25867,
  [0x1399] = 0x25887, [0x139a] = 0x2583f, [0x139b] = 0x2586c, [0x139c] = 0x2586e,
  [0x139d] = 0x258ad, [0x139e] = 0x259f1, [0x139f] = 0x259f4, [0x13a0] = 0x259f6,
  [0x13a1] = 0x259f5, [0x13a2] = 0x259f8, [0x13a3] = 0x259fb, [0x13a4] = 0x259ec,
  [0x13a5] = 0x259ef, [0x13a6] = 0x259ed, [0x13a7] = 0x7ab9, [0x13a8] = 0x259f7,
  [0x13a9] = 0x259f9, [0x13aa] = 0x259fd, [0x13ab] = 0x25ab1, [0x13ac] = 0x25c36,
  [0x13ad] = 0x25c1b, [0x13ae] = 0x25c32, [0x13af] = 0x25c42, [0x13b0] = 0x25c4d,
  [0x13b1] = 0x25c1f, [0x13b2] = 0x25c21, [0x13b3] = 0x25c1c, [0x13b4] = 0x25c31,
  [0x13b5] = 0x25c2e, [0x13b6] = 0x25c47, [0x13b7] = 0x25c3b, [0x13b8] = 0x25c41,
  [0x13b9] = 0x25c18, [0x13ba] = 0x25c39, [0x13bb] = 0x25c1d, [0x13bc] = 0x25c27,
  [0x13bd] = 0x25c23, [0x13be] = 0x7bd7, [0x13bf] = 0x25eee, [0x13c0] = 0x25ee8,
  [0x13c1] = 0x25ee5, [0x13c2] = 0x25eef, [0x13c3] = 0x25ee4, [0x13c4] = 0x25eec,
  [0x13c5] = 0x25ef0, [0x13c6] = 0x260d7, [0x13c7] = 0x260d8, [0x13c8] = 0x260d4,
  [0x13c9] = 0x260ca, [0x13ca] = 0x260d2, [0x13cb] = 0x260cb, [0x13cc] = 0x260d3,
  [0x13cd] = 0x260e6, [0x13ce] = 0x260e2, [0x13cf] = 0x26249, [0x13d0] = 0x26248,
  [0x13d1] = 0x262ff, [0x13d2] = 0x26309, [0x13d3] = 0x263b0, [0x13d4] = 0x263af,
  [0x13d5] = 0x2647c, [0x13d6] = 0x26478, [0x13d7] = 0x2647a, [0x13d8] = 0x26472,
  [0x13d9] = 0x26479, [0x13da] = 0x2647e, [0x13db] = 0x2650c, [0x13dc] = 0x2650b,
  [0x13dd] = 0x265cd, [0x13de] = 0x265be, [0x13df] = 0x265bc, [0x13e0] = 0x265bf,
  [0x13e1] = 0x265c0, [0x13e2] = 0x265bd, [0x13e3] = 0x26838, [0x13e4] = 0x26808,
  [0x13e5] = 0x26805, [0x13e6] = 0x2683b, [0x13e7] = 0x26810, [0x13e8] = 0x2680c,
  [0x13e9] = 0x2680d, [0x13ea] = 0x26804, [0x13eb] = 0x2683a, [0x13ec] = 0x26813,
  [0x13ed] = 0x26837, [0x13ee] = 0x26839, [0x13ef] = 0x2691f, [0x13f0] = 0x2694b,
  [0x13f1] = 0x269ee, [0x13f2] = 0x26a75, [0x13f3] = 0x26a78, [0x13f4] = 0x26a7c,
  [0x13f5] = 0x26a74, [0x13f6] = 0x26a76, [0x13f7] = 0x26aea, [0x13f8] = 0x26aeb,
  [0x13f9] = 0x26dfb, [0x13fa] = 0x26e19, [0x13fb] = 0x26ea7, [0x13fc] = 0x26e8c,
  [0x13fd] = 0x26edc, [0x13fe] = 0x26e98, [0x13ff] = 0x26ebe, [0x1400] = 0x26e9e,
  [0x1401] = 0x26f0f, [0x1402] = 0x26e9f, [0x1403] = 0x26edd, [0x1404] = 0x26e93,
  [0x1405] = 0x26ebb, [0x1406] = 0x26eb6, [0x1407] = 0x26e90, [0x1408] = 0x26ea1,
  [0x1409] = 0x26ebd, [0x140a] = 0x26ede, [0x140b] = 0x26e3a, [0x140c] = 0x26f22,
  [0x140d] = 0x26e97, [0x140e] = 0x26e94, [0x140f] = 0x26ec3, [0x1410] = 0x26e8e,
  [0x1411] = 0x26ea8, [0x1412] = 0x26e99, [0x1413] = 0x26ead, [0x1414] = 0x26e9b,
  [0x1415] = 0x26ea2, [0x1416] = 0x26f21, [0x1417] = 0x26eac, [0x1418] = 0x26f0e,
  [0x1419] = 0x26f31, [0x141a] = 0x271e1, [0x141b] = 0x271de, [0x141c] = 0x271df,
  [0x141d] = 0x271dc, [0x141e] = 0x271dd, [0x141f] = 0x271ec, [0x1420] = 0x271e7,
  [0x1421] = 0x27369, [0x1422] = 0x273b8, [0x1423] = 0x273a1, [0x1424] = 0x273a8,
  [0x1425] = 0x273ba, [0x1426] = 0x273c2, [0x1427] = 0x273a6, [0x1428] = 0x273a4,
  [0x1429] = 0x273a3, [0x142a] = 0x273ab, [0x142b] = 0x273bc, [0x142c] = 0x273b7,
  [0x142d] = 0x273bf, [0x142e] = 0x273ad, [0x142f] = 0x273b1, [0x1430] = 0x273ca,
  [0x1431] = 0x273c4, [0x1432] = 0x273b9, [0x1433] = 0x275c8, [0x1434] = 0x275c6,
  [0x1435] = 0x275c7, [0x1436] = 0x2770a, [0x1437] = 0x2770d, [0x1438] = 0x276fb,
  [0x1439] = 0x27703, [0x143a] = 0x27702, [0x143b] = 0x276fc, [0x143c] = 0x276f9,
  [0x143d] = 0x276f8, [0x143e] = 0x2786e, [0x143f] = 0x27863, [0x1440] = 0x27862,
  [0x1441] = 0x27861, [0x1442] = 0x2786b, [0x1443] = 0x4688, [0x1444] = 0x2786f,
  [0x1445] = 0x27866, [0x1446] = 0x2786c, [0x1447] = 0x2790e, [0x1448] = 0x27915,
  [0x1449] = 0x27916, [0x144a] = 0x27910, [0x144b] = 0x27917, [0x144c] = 0x27911,
  [0x144d] = 0x27a6e, [0x144e] = 0x27a6c, [0x144f] = 0x27a87, [0x1450] = 0x27a83,
  [0x1451] = 0x27a63, [0x1452] = 0x27adc, [0x1453] = 0x27bc0, [0x1454] = 0x27bf7,
  [0x1455] = 0x27bfa, [0x1456] = 0x27c70, [0x1457] = 0x27c6a, [0x1458] = 0x27c68,
  [0x1459] = 0x27c69, [0x145a] = 0x27c84, [0x145b] = 0x27cec, [0x145c] = 0x27ce7,
  [0x145d] = 0x27cee, [0x145e] = 0x27dba, [0x145f] = 0x27db2, [0x1460] = 0x27db5,
  [0x1461] = 0x27dcb, [0x1462] = 0x2080b, [0x1463] = 0x27dd0, [0x1464] = 0x27e6d,
  [0x1465] = 0x27e6c, [0x1466] = 0x47bb, [0x1467] = 0x27f45, [0x1468] = 0x27f46,
  [0x1469] = 0x27f34, [0x146a] = 0x27f2c, [0x146b] = 0x27f35, [0x146c] = 0x27f44,
  [0x146d] = 0x47ba, [0x146e] = 0x27f76, [0x146f] = 0x280b1, [0x1470] = 0x280aa,
  [0x1471] = 0x280a1, [0x1472] = 0x280b2, [0x1473] = 0x4812, [0x1474] = 0x280a6,
  [0x1475] = 0x280b5, [0x1476] = 0x280b4, [0x1477] = 0x280b8, [0x1478] = 0x280af,
  [0x1479] = 0x280b0, [0x147a] = 0x280a3, [0x147b] = 0x28262, [0x147c] = 0x28264,
  [0x147d] = 0x28356, [0x147e] = 0x28351, [0x147f] = 0x2834f, [0x1480] = 0x28368,
  [0x1481] = 0x2834c, [0x1482] = 0x28350, [0x1483] = 0x4861, [0x1484] = 0x28348,
  [0x1485] = 0x2834a, [0x1486] = 0x28421, [0x1487] = 0x285e3, [0x1488] = 0x285b9,
  [0x1489] = 0x285de, [0x148a] = 0x285b7, [0x148b] = 0x285e1, [0x148c] = 0x285b6,
  [0x148d] = 0x285b5, [0x148e] = 0x285df, [0x148f] = 0x48e1, [0x1490] = 0x28797,
  [0x1491] = 0x2879a, [0x1492] = 0x2879b, [0x1493] = 0x28798, [0x1494] = 0x28792,
  [0x1495] = 0x28793, [0x1496] = 0x287d7, [0x1497] = 0x28773, [0x1498] = 0x2886b,
  [0x1499] = 0x28874, [0x149a] = 0x28878, [0x149b] = 0x2886d, [0x149c] = 0x28918,
  [0x149d] = 0x289e9, [0x149e] = 0x289f5, [0x149f] = 0x289ea, [0x14a0] = 0x28a2e,
  [0x14a1] = 0x289e7, [0x14a2] = 0x289fe, [0x14a3] = 0x289e5, [0x14a4] = 0x28a36,
  [0x14a5] = 0x289f0, [0x14a6] = 0x289e6, [0x14a7] = 0x28a2c, [0x14a8] = 0x28c89,
  [0x14a9] = 0x28c95, [0x14aa] = 0x28c93, [0x14ab] = 0x28d4d, [0x14ac] = 0x28d4a,
  [0x14ad] = 0x28d4f, [0x14ae] = 0x28d50, [0x14af] = 0x28d4b, [0x14b0] = 0x28f2a,
  [0x14b1] = 0x28f2b, [0x14b2] = 0x28f2f, [0x14b3] = 0x28f2e, [0x14b4] = 0x28f7c,
  [0x14b5] = 0x28fed, [0x14b6] = 0x28fe2, [0x14b7] = 0x290e0, [0x14b8] = 0x290dc,
  [0x14b9] = 0x290da, [0x14ba] = 0x290d6, [0x14bb] = 0x290f4, [0x14bc] = 0x290d9,
  [0x14bd] = 0x290d5, [0x14be] = 0x29222, [0x14bf] = 0x29221, [0x14c0] = 0x29224,
  [0x14c1] = 0x29225, [0x14c2] = 0x29226, [0x14c3] = 0x29223, [0x14c4] = 0x292aa,
  [0x14c5] = 0x292af, [0x14c6] = 0x292b0, [0x14c7] = 0x292ab, [0x14c8] = 0x293af,
  [0x14c9] = 0x293b7, [0x14ca] = 0x293b5, [0x14cb] = 0x293b2, [0x14cc] = 0x293b3,
  [0x14cd] = 0x2942b, [0x14ce] = 0x294d8, [0x14cf] = 0x294c2, [0x14d0] = 0x294af,
  [0x14d1] = 0x294bc, [0x14d2] = 0x294b8, [0x14d3] = 0x294be, [0x14d4] = 0x294b7,
  [0x14d5] = 0x294b4, [0x14d6] = 0x294bf, [0x14d7] = 0x294b3, [0x14d8] = 0x294b1,
  [0x14d9] = 0x294bb, [0x14da] = 0x294bd, [0x14db] = 0x294d6, [0x14dc] = 0x294dd,
  [0x14dd] = 0x295d8, [0x14de] = 0x295d3, [0x14df] = 0x295d5, [0x14e0] = 0x295e3,
  [0x14e1] = 0x295e2, [0x14e2] = 0x295d9, [0x14e3] = 0x295de, [0x14e4] = 0x295df,
  [0x14e5] = 0x295da, [0x14e6] = 0x295d4, [0x14e7] = 0x296f3, [0x14e8] = 0x296e2,
  [0x14e9] = 0x2970d, [0x14ea] = 0x29701, [0x14eb] = 0x29705, [0x14ec] = 0x2971a,
  [0x14ed] = 0x29703, [0x14ee] = 0x2971f, [0x14ef] = 0x29716, [0x14f0] = 0x296fa,
  [0x14f1] = 0x296fc, [0x14f2] = 0x2970a, [0x14f3] = 0x999f, [0x14f4] = 0x298bc,
  [0x14f5] = 0x298ca, [0x14f6] = 0x298b6, [0x14f7] = 0x298c7, [0x14f8] = 0x298bf,
  [0x14f9] = 0x4b89, [0x14fa] = 0x298b9, [0x14fb] = 0x4b8c, [0x14fc] = 0x298b0,
  [0x14fd] = 0x298b8, [0x14fe] = 0x298bd, [0x14ff] = 0x29891, [0x1500] = 0x298bb,
  [0x1501] = 0x298be, [0x1502] = 0x29a3e, [0x1503] = 0x29a3d, [0x1504] = 0x29b38,
  [0x1505] = 0x29b3d, [0x1506] = 0x29b39, [0x1507] = 0x29b33, [0x1508] = 0x29c33,
  [0x1509] = 0x29cbb, [0x150a] = 0x29cc6, [0x150b] = 0x29cc5, [0x150c] = 0x29cc7,
  [0x150d] = 0x29ccb, [0x150e] = 0x29ca8, [0x150f] = 0x29cc8, [0x1510] = 0x29cbe,
  [0x1511] = 0x29cc1, [0x1512] = 0x29cbd, [0x1513] = 0x29d82, [0x1514] = 0x29d9e,
  [0x1515] = 0x29d81, [0x1516] = 0x29d84, [0x1517] = 0x9b8d, [0x1518] = 0x29d96,
  [0x1519] = 0x29d8e, [0x151a] = 0x29d88, [0x151b] = 0x29d87, [0x151c] = 0x29fe0,
  [0x151d] = 0x2a00d, [0x151e] = 0x29fdf, [0x151f] = 0x29fe4, [0x1520] = 0x29fe2,
  [0x1521] = 0x29fdd, [0x1522] = 0x29fec, [0x1523] = 0x29fde, [0x1524] = 0x29fe7,
  [0x1525] = 0x29fea, [0x1526] = 0x29fe3, [0x1527] = 0x2a25c, [0x1528] = 0x9e77,
  [0x1529] = 0x2a25d, [0x152a] = 0x2a29c, [0x152b] = 0x2a31d, [0x152c] = 0x2a39d,
  [0x152d] = 0x2a39e, [0x152e] = 0x2a39b, [0x152f] = 0x2a3b5, [0x1530] = 0x2a3b9,
  [0x1531] = 0x2a3b6, [0x1532] = 0x2a3f3, [0x1533] = 0x2a3f2, [0x1534] = 0x2a3f4,
  [0x1535] = 0x2a426, [0x1536] = 0x2a427, [0x1537] = 0x2a425, [0x1538] = 0x2a506,
  [0x1539] = 0x2a50b, [0x153a] = 0x2a53a, [0x153b] = 0x2a53c, [0x153c] = 0x2040a,
  [0x153d] = 0x2040b, [0x153e] = 0x204bf, [0x153f] = 0x204fd, [0x1540] = 0x20618,
  [0x1541] = 0x291df, [0x1542] = 0x25390, [0x1543] = 0x2069e, [0x1544] = 0x207fc,
  [0x1545] = 0x20810, [0x1546] = 0x2080f, [0x1547] = 0x2080d, [0x1548] = 0x208b9,
  [0x1549] = 0x208b7, [0x154a] = 0x208ba, [0x154b] = 0x2090a, [0x154c] = 0x21041,
  [0x154d] = 0x2108b, [0x154e] = 0x21046, [0x154f] = 0x21053, [0x1550] = 0x210e2,
  [0x1551] = 0x2103f, [0x1552] = 0x20f7c, [0x1553] = 0x2104b, [0x1554] = 0x2104e,
  [0x1555] = 0x2108a, [0x1556] = 0x21047, [0x1557] = 0x21493, [0x1558] = 0x27dce,
  [0x1559] = 0x2148c, [0x155a] = 0x214af, [0x155b] = 0x214c9, [0x155c] = 0x215c6,
  [0x155d] = 0x215c8, [0x155e] = 0x21691, [0x155f] = 0x218cf, [0x1560] = 0x218d4,
  [0x1561] = 0x218ce, [0x1562] = 0x21add, [0x1563] = 0x21ad4, [0x1564] = 0x21f49,
  [0x1565] = 0x21f63, [0x1566] = 0x21f5d, [0x1567] = 0x381a, [0x1568] = 0x21f67,
  [0x1569] = 0x21fbb, [0x156a] = 0x21f60, [0x156b] = 0x21f80, [0x156c] = 0x2215f,
  [0x156d] = 0x22160, [0x156e] = 0x222da, [0x156f] = 0x222d2, [0x1570] = 0x222dd,
  [0x1571] = 0x2258e, [0x1572] = 0x22588, [0x1573] = 0x2256c, [0x1574] = 0x2289e,
  [0x1575] = 0x228c9, [0x1576] = 0x228a4, [0x1577] = 0x3998, [0x1578] = 0x228d9,
  [0x1579] = 0x228d2, [0x157a] = 0x228da, [0x157b] = 0x228dd, [0x157c] = 0x228ce,
  [0x157d] = 0x228fc, [0x157e] = 0x22a14, [0x157f] = 0x22a60, [0x1580] = 0x22dd7,
  [0x1581] = 0x22dcb, [0x1582] = 0x22dcd, [0x1583] = 0x22dd5, [0x1584] = 0x22ee7,
  [0x1585] = 0x23000, [0x1586] = 0x2300a, [0x1587] = 0x2300f, [0x1588] = 0x23002,
  [0x1589] = 0x23001, [0x158a] = 0x2311d, [0x158b] = 0x232ca, [0x158c] = 0x232cb,
  [0x158d] = 0x232cd, [0x158e] = 0x2334f, [0x158f] = 0x2339c, [0x1590] = 0x23680,
  [0x1591] = 0x2370d, [0x1592] = 0x23702, [0x1593] = 0x3bf1, [0x1594] = 0x23707,
  [0x1595] = 0x236f7, [0x1596] = 0x236f8, [0x1597] = 0x236fd, [0x1598] = 0x23724,
  [0x1599] = 0x236fb, [0x159a] = 0x23739, [0x159b] = 0x2373a, [0x159c] = 0x23922,
  [0x159d] = 0x6ba9, [0x159e] = 0x23a6a, [0x159f] = 0x23a6d, [0x15a0] = 0x23a74,
  [0x15a1] = 0x23c0e, [0x15a2] = 0x23fdf, [0x15a3] = 0x2402b, [0x15a4] = 0x23fe3,
  [0x15a5] = 0x24026, [0x15a6] = 0x2402a, [0x15a7] = 0x24023, [0x15a8] = 0x24035,
  [0x15a9] = 0x2404a, [0x15aa] = 0x243fd, [0x15ab] = 0x24400, [0x15ac] = 0x2441e,
  [0x15ad] = 0x243ff, [0x15ae] = 0x2459e, [0x15af] = 0x245fa, [0x15b0] = 0x246f1,
  [0x15b1] = 0x24895, [0x15b2] = 0x248a8, [0x15b3] = 0x248a6, [0x15b4] = 0x24a6e,
  [0x15b5] = 0x24a83, [0x15b6] = 0x24a6d, [0x15b7] = 0x24b8f, [0x15b8] = 0x3f6a,
  [0x15b9] = 0x24cdf, [0x15ba] = 0x24cd5, [0x15bb] = 0x24ce0, [0x15bc] = 0x24cd3,
  [0x15bd] = 0x24cd8, [0x15be] = 0x24e8c, [0x15bf] = 0x764b, [0x15c0] = 0x24e94,
  [0x15c1] = 0x24e96, [0x15c2] = 0x24e85, [0x15c3] = 0x24e8f, [0x15c4] = 0x24ea9,
  [0x15c5] = 0x24e8e, [0x15c6] = 0x24e90, [0x15c7] = 0x24eb8, [0x15c8] = 0x24ec3,
  [0x15c9] = 0x24ebb, [0x15ca] = 0x24fa5, [0x15cb] = 0x24fa2, [0x15cc] = 0x76a2,
  [0x15cd] = 0x24fa3, [0x15ce] = 0x250b5, [0x15cf] = 0x250ac, [0x15d0] = 0x250a8,
  [0x15d1] = 0x252ad, [0x15d2] = 0x252b8, [0x15d3] = 0x252b4, [0x15d4] = 0x252ae,
  [0x15d5] = 0x252b6, [0x15d6] = 0x4485, [0x15d7] = 0x252c1, [0x15d8] = 0x252bf,
  [0x15d9] = 0x25392, [0x15da] = 0x25400, [0x15db] = 0x25401, [0x15dc] = 0x25575,
  [0x15dd] = 0x25572, [0x15de] = 0x25578, [0x15df] = 0x25570, [0x15e0] = 0x2557e,
  [0x15e1] = 0x2557d, [0x15e2] = 0x2557f, [0x15e3] = 0x256ef, [0x15e4] = 0x2586d,
  [0x15e5] = 0x2588e, [0x15e6] = 0x2588f, [0x15e7] = 0x2588a, [0x15e8] = 0x2588d,
  [0x15e9] = 0x25895, [0x15ea] = 0x2588b, [0x15eb] = 0x2588c, [0x15ec] = 0x258b0,
  [0x15ed] = 0x25893, [0x15ee] = 0x259fc, [0x15ef] = 0x25a15, [0x15f0] = 0x25a0e,
  [0x15f1] = 0x25a18, [0x15f2] = 0x25a11, [0x15f3] = 0x25a0d, [0x15f4] = 0x25abb,
  [0x15f5] = 0x25ab3, [0x15f6] = 0x4239, [0x15f7] = 0x25ca1, [0x15f8] = 0x25c9a,
  [0x15f9] = 0x25c84, [0x15fa] = 0x25ca0, [0x15fb] = 0x25c86, [0x15fc] = 0x25c9d,
  [0x15fd] = 0x25caa, [0x15fe] = 0x25c78, [0x15ff] = 0x25c8d, [0x1600] = 0x25c8a,
  [0x1601] = 0x25ca6, [0x1602] = 0x25c7a, [0x1603] = 0x25c97, [0x1604] = 0x25c88,
  [0x1605] = 0x25c8e, [0x1606] = 0x423d, [0x1607] = 0x4244, [0x1608] = 0x25c76,
  [0x1609] = 0x25c81, [0x160a] = 0x25c85, [0x160b] = 0x25c75, [0x160c] = 0x25ca8,
  [0x160d] = 0x25c8f, [0x160e] = 0x25c91, [0x160f] = 0x25ca2, [0x1610] = 0x25c9c,
  [0x1611] = 0x25c89, [0x1612] = 0x25c7f, [0x1613] = 0x25c96, [0x1614] = 0x25c79,
  [0x1615] = 0x25c9f, [0x1616] = 0x25ca7, [0x1617] = 0x4246, [0x1618] = 0x25c87,
  [0x1619] = 0x25c9b, [0x161a] = 0x25ca5, [0x161b] = 0x25c8b, [0x161c] = 0x25cc9,
  [0x161d] = 0x25eff, [0x161e] = 0x25f03, [0x161f] = 0x25f00, [0x1620] = 0x25f02,
  [0x1621] = 0x25f04, [0x1622] = 0x25f05, [0x1623] = 0x25ee6, [0x1624] = 0x25f1f,
  [0x1625] = 0x26114, [0x1626] = 0x430e, [0x1627] = 0x2610c, [0x1628] = 0x2610f,
  [0x1629] = 0x26119, [0x162a] = 0x2610b, [0x162b] = 0x26113, [0x162c] = 0x260d5,
  [0x162d] = 0x4310, [0x162e] = 0x2611c, [0x162f] = 0x26250, [0x1630] = 0x2624e,
  [0x1631] = 0x26312, [0x1632] = 0x263b9, [0x1633] = 0x26485, [0x1634] = 0x26488,
  [0x1635] = 0x26490, [0x1636] = 0x26513, [0x1637] = 0x265d0, [0x1638] = 0x265d1,
  [0x1639] = 0x265d2, [0x163a] = 0x265d7, [0x163b] = 0x265d6, [0x163c] = 0x265e5,
  [0x163d] = 0x26846, [0x163e] = 0x2686b, [0x163f] = 0x26845, [0x1640] = 0x2683f,
  [0x1641] = 0x2683e, [0x1642] = 0x443e, [0x1643] = 0x2686a, [0x1644] = 0x26868,
  [0x1645] = 0x2684c, [0x1646] = 0x26923, [0x1647] = 0x26922, [0x1648] = 0x26997,
  [0x1649] = 0x26991, [0x164a] = 0x25ab5, [0x164b] = 0x26998, [0x164c] = 0x2699c,
  [0x164d] = 0x26a89, [0x164e] = 0x4484, [0x164f] = 0x26a8b, [0x1650] = 0x26a8a,
  [0x1651] = 0x26a8d, [0x1652] = 0x26a8e, [0x1653] = 0x26a88, [0x1654] = 0x26f4e,
  [0x1655] = 0x26f44, [0x1656] = 0x26f37, [0x1657] = 0x26f75, [0x1658] = 0x26f54,
  [0x1659] = 0x26f76, [0x165a] = 0x26f34, [0x165b] = 0x26f6b, [0x165c] = 0x26f32,
  [0x165d] = 0x26f57, [0x165e] = 0x26f52, [0x165f] = 0x26f45, [0x1660] = 0x4549,
  [0x1661] = 0x26eb1, [0x1662] = 0x26f4b, [0x1663] = 0x26f47, [0x1664] = 0x26f33,
  [0x1665] = 0x26f40, [0x1666] = 0x26f3c, [0x1667] = 0x26f43, [0x1668] = 0x26f4f,
  [0x1669] = 0x26f55, [0x166a] = 0x26f41, [0x166b] = 0x26fb2, [0x166c] = 0x26fc0,
  [0x166d] = 0x27434, [0x166e] = 0x273f9, [0x166f] = 0x27419, [0x1670] = 0x2740d,
  [0x1671] = 0x273fa, [0x1672] = 0x2741e, [0x1673] = 0x2741f, [0x1674] = 0x2740e,
  [0x1675] = 0x27440, [0x1676] = 0x27408, [0x1677] = 0x45df, [0x1678] = 0x27413,
  [0x1679] = 0x2744c, [0x167a] = 0x45f0, [0x167b] = 0x275cc, [0x167c] = 0x275cb,
  [0x167d] = 0x27724, [0x167e] = 0x27725, [0x167f] = 0x2773d, [0x1680] = 0x27720,
  [0x1681] = 0x27727, [0x1682] = 0x27726, [0x1683] = 0x2771d, [0x1684] = 0x2771e,
  [0x1685] = 0x27732, [0x1686] = 0x2776c, [0x1687] = 0x27759, [0x1688] = 0x468a,
  [0x1689] = 0x2787a, [0x168a] = 0x27879, [0x168b] = 0x2791b, [0x168c] = 0x2792e,
  [0x168d] = 0x27923, [0x168e] = 0x27920, [0x168f] = 0x2791f, [0x1690] = 0x27ab6,
  [0x1691] = 0x27ab9, [0x1692] = 0x27aa1, [0x1693] = 0x27aa3, [0x1694] = 0x27aa8,
  [0x1695] = 0x27aaf, [0x1696] = 0x27a9a, [0x1697] = 0x27a99, [0x1698] = 0x27aa2,
  [0x1699] = 0x27a9d, [0x169a] = 0x4702, [0x169b] = 0x27aab, [0x169c] = 0x27bc9,
  [0x169d] = 0x27bfe, [0x169e] = 0x27c00, [0x169f] = 0x27bfc, [0x16a0] = 0x27c07,
  [0x16a1] = 0x27c75, [0x16a2] = 0x27c72, [0x16a3] = 0x27c73, [0x16a4] = 0x27c74,
  [0x16a5] = 0x27cfa, [0x16a6] = 0x27cfc, [0x16a7] = 0x27cf8, [0x16a8] = 0x27cf6,
  [0x16a9] = 0x27cfb, [0x16aa] = 0x27dcd, [0x16ab] = 0x27dd1, [0x16ac] = 0x27dcf,
  [0x16ad] = 0x27e74, [0x16ae] = 0x27e72, [0x16af] = 0x27e73, [0x16b0] = 0x27f54,
  [0x16b1] = 0x27f51, [0x16b2] = 0x47c0, [0x16b3] = 0x27f53, [0x16b4] = 0x27f49,
  [0x16b5] = 0x27f4c, [0x16b6] = 0x27f4a, [0x16b7] = 0x27f4f, [0x16b8] = 0x27f56,
  [0x16b9] = 0x28100, [0x16ba] = 0x280d5, [0x16bb] = 0x280d7, [0x16bc] = 0x280ff,
  [0x16bd] = 0x280d9, [0x16be] = 0x280e3, [0x16bf] = 0x280d3, [0x16c0] = 0x4815,
  [0x16c1] = 0x280d8, [0x16c2] = 0x280d4, [0x16c3] = 0x280de, [0x16c4] = 0x28271,
  [0x16c5] = 0x28374, [0x16c6] = 0x28388, [0x16c7] = 0x2837f, [0x16c8] = 0x2836b,
  [0x16c9] = 0x28387, [0x16ca] = 0x28379, [0x16cb] = 0x28378, [0x16cc] = 0x28389,
  [0x16cd] = 0x28380, [0x16ce] = 0x28376, [0x16cf] = 0x28429, [0x16d0] = 0x28428,
  [0x16d1] = 0x2842f, [0x16d2] = 0x285e5, [0x16d3] = 0x28604, [0x16d4] = 0x28603,
  [0x16d5] = 0x285f0, [0x16d6] = 0x285e8, [0x16d7] = 0x285ea, [0x16d8] = 0x285f1,
  [0x16d9] = 0x28601, [0x16da] = 0x28602, [0x16db] = 0x287a9, [0x16dc] = 0x287ab,
  [0x16dd] = 0x287b7, [0x16de] = 0x287b6, [0x16df] = 0x48e2, [0x16e0] = 0x288a0,
  [0x16e1] = 0x2888e, [0x16e2] = 0x28886, [0x16e3] = 0x28887, [0x16e4] = 0x28885,
  [0x16e5] = 0x2888b, [0x16e6] = 0x28888, [0x16e7] = 0x28890, [0x16e8] = 0x2891a,
  [0x16e9] = 0x28934, [0x16ea] = 0x28a37, [0x16eb] = 0x28a2f, [0x16ec] = 0x28a30,
  [0x16ed] = 0x28a39, [0x16ee] = 0x28a34, [0x16ef] = 0x28a33, [0x16f0] = 0x28a85,
  [0x16f1] = 0x28a84, [0x16f2] = 0x28a3a, [0x16f3] = 0x28c9f, [0x16f4] = 0x28d69,
  [0x16f5] = 0x28d6c, [0x16f6] = 0x28d6a, [0x16f7] = 0x28d67, [0x16f8] = 0x28d6b,
  [0x16f9] = 0x28f3f, [0x16fa] = 0x28f40, [0x16fb] = 0x2900b, [0x16fc] = 0x290d2,
  [0x16fd] = 0x290f8, [0x16fe] = 0x29101, [0x16ff] = 0x290fa, [0x1700] = 0x29116,
  [0x1701] = 0x2f9f6, [0x1702] = 0x2922c, [0x1703] = 0x2922d, [0x1704] = 0x2922f,
  [0x1705] = 0x2922e, [0x1706] = 0x292c1, [0x1707] = 0x292f5, [0x1708] = 0x292c4,
  [0x1709] = 0x292de, [0x170a] = 0x292cc, [0x170b] = 0x292d2, [0x170c] = 0x292c2,
  [0x170d] = 0x292cd, [0x170e] = 0x292cf, [0x170f] = 0x292dd, [0x1710] = 0x292c8,
  [0x1711] = 0x293ba, [0x1712] = 0x293be, [0x1713] = 0x293bf, [0x1714] = 0x2942e,
  [0x1715] = 0x29430, [0x1716] = 0x294e7, [0x1717] = 0x294ee, [0x1718] = 0x294e8,
  [0x1719] = 0x294e6, [0x171a] = 0x294e2, [0x171b] = 0x294e4, [0x171c] = 0x294fb,
  [0x171d] = 0x294fc, [0x171e] = 0x294ea, [0x171f] = 0x295ed, [0x1720] = 0x295f6,
  [0x1721] = 0x295f3, [0x1722] = 0x295f4, [0x1723] = 0x295f1, [0x1724] = 0x295f7,
  [0x1725] = 0x295f5, [0x1726] = 0x295ea, [0x1727] = 0x295eb, [0x1728] = 0x29700,
  [0x1729] = 0x2972c, [0x172a] = 0x29712, [0x172b] = 0x2973d, [0x172c] = 0x29733,
  [0x172d] = 0x4b4d, [0x172e] = 0x29730, [0x172f] = 0x29740, [0x1730] = 0x29731,
  [0x1731] = 0x29757, [0x1732] = 0x4b4a, [0x1733] = 0x29735, [0x1734] = 0x29732,
  [0x1735] = 0x2972d, [0x1736] = 0x29736, [0x1737] = 0x29738, [0x1738] = 0x29762,
  [0x1739] = 0x2975f, [0x173a] = 0x2978a, [0x173b] = 0x298d6, [0x173c] = 0x298d8,
  [0x173d] = 0x298dd, [0x173e] = 0x298e4, [0x173f] = 0x298e3, [0x1740] = 0x29a4d,
  [0x1741] = 0x29ad7, [0x1742] = 0x29b4a, [0x1743] = 0x29b4c, [0x1744] = 0x29b50,
  [0x1745] = 0x29b4b, [0x1746] = 0x29b4e, [0x1747] = 0x29b4f, [0x1748] = 0x29c39,
  [0x1749] = 0x29cd2, [0x174a] = 0x29ccd, [0x174b] = 0x29cd3, [0x174c] = 0x29ce1,
  [0x174d] = 0x29cce, [0x174e] = 0x29cd5, [0x174f] = 0x29cdd, [0x1750] = 0x29cd4,
  [0x1751] = 0x29ccf, [0x1752] = 0x29dcb, [0x1753] = 0x29da6, [0x1754] = 0x29dc2,
  [0x1755] = 0x29da5, [0x1756] = 0x29da9, [0x1757] = 0x29da2, [0x1758] = 0x29da4,
  [0x1759] = 0x29db0, [0x175a] = 0x29dcc, [0x175b] = 0x29daf, [0x175c] = 0x29dbf,
  [0x175d] = 0x2a024, [0x175e] = 0x2a01a, [0x175f] = 0x2a014, [0x1760] = 0x2a03a,
  [0x1761] = 0x2a020, [0x1762] = 0x2a01b, [0x1763] = 0x2a021, [0x1764] = 0x2a025,
  [0x1765] = 0x2a01e, [0x1766] = 0x2a03f, [0x1767] = 0x2a040, [0x1768] = 0x2a018,
  [0x1769] = 0x2a02c, [0x176a] = 0x2a015, [0x176b] = 0x2a02d, [0x176c] = 0x2a01f,
  [0x176d] = 0x2a029, [0x176e] = 0x2a04b, [0x176f] = 0x2a2b3, [0x1770] = 0x2a2a5,
  [0x1771] = 0x2a2a7, [0x1772] = 0x2a2ab, [0x1773] = 0x2a2a6, [0x1774] = 0x2a2aa,
  [0x1775] = 0x2a322, [0x1776] = 0x2a323, [0x1777] = 0x2a3a1, [0x1778] = 0x2a3a3,
  [0x1779] = 0x2a3a0, [0x177a] = 0x2a3bd, [0x177b] = 0x2a3ba, [0x177c] = 0x2a3fa,
  [0x177d] = 0x2a3f8, [0x177e] = 0x2a3fc, [0x177f] = 0x2a3f6, [0x1780] = 0x2a434,
  [0x1781] = 0x2a443, [0x1782] = 0x4d63, [0x1783] = 0x2a4d4, [0x1784] = 0x2a50d,
  [0x1785] = 0x2a543, [0x1786] = 0x2a541, [0x1787] = 0x2a53d, [0x1788] = 0x2a540,
  [0x1789] = 0x2a53e, [0x178a] = 0x2a53f, [0x178b] = 0x2a546, [0x178c] = 0x4d93,
  [0x178d] = 0x2a5d4, [0x178e] = 0x20435, [0x178f] = 0x20436, [0x1790] = 0x20432,
  [0x1791] = 0x2043a, [0x1792] = 0x204c8, [0x1793] = 0x20536, [0x1794] = 0x2061e,
  [0x1795] = 0x2061d, [0x1796] = 0x2061f, [0x1797] = 0x2081c, [0x1798] = 0x2081d,
  [0x1799] = 0x20820, [0x179a] = 0x350e, [0x179b] = 0x208c0, [0x179c] = 0x2109a,
  [0x179d] = 0x21093, [0x179e] = 0x210dd, [0x179f] = 0x360a, [0x17a0] = 0x210a3,
  [0x17a1] = 0x210a9, [0x17a2] = 0x2109c, [0x17a3] = 0x2109b, [0x17a4] = 0x360d,
  [0x17a5] = 0x21097, [0x17a6] = 0x214b1, [0x17a7] = 0x214ca, [0x17a8] = 0x214b3,
  [0x17a9] = 0x214cd, [0x17aa] = 0x218ea, [0x17ab] = 0x218ee, [0x17ac] = 0x218ef,
  [0x17ad] = 0x21adf, [0x17ae] = 0x21b50, [0x17af] = 0x21bbb, [0x17b0] = 0x21f88,
  [0x17b1] = 0x21f89, [0x17b2] = 0x21f8c, [0x17b3] = 0x21f85, [0x17b4] = 0x21f5e,
  [0x17b5] = 0x21f94, [0x17b6] = 0x21f95, [0x17b7] = 0x2216b, [0x17b8] = 0x2216a,
  [0x17b9] = 0x22169, [0x17ba] = 0x222e3, [0x17bb] = 0x222f1, [0x17bc] = 0x22437,
  [0x17bd] = 0x22433, [0x17be] = 0x22591, [0x17bf] = 0x2258f, [0x17c0] = 0x22597,
  [0x17c1] = 0x228d4, [0x17c2] = 0x228de, [0x17c3] = 0x228d3, [0x17c4] = 0x228e0,
  [0x17c5] = 0x22943, [0x17c6] = 0x2290b, [0x17c7] = 0x228ff, [0x17c8] = 0x22904,
  [0x17c9] = 0x22907, [0x17ca] = 0x22df9, [0x17cb] = 0x22dfa, [0x17cc] = 0x22dfb,
  [0x17cd] = 0x22df8, [0x17ce] = 0x23012, [0x17cf] = 0x23010, [0x17d0] = 0x2311f,
  [0x17d1] = 0x232df, [0x17d2] = 0x232e3, [0x17d3] = 0x23352, [0x17d4] = 0x233a9,
  [0x17d5] = 0x23765, [0x17d6] = 0x23787, [0x17d7] = 0x23742, [0x17d8] = 0x23752,
  [0x17d9] = 0x2374c, [0x17da] = 0x2369f, [0x17db] = 0x23701, [0x17dc] = 0x23748,
  [0x17dd] = 0x2374f, [0x17de] = 0x2377e, [0x17df] = 0x2374d, [0x17e0] = 0x23758,
  [0x17e1] = 0x23747, [0x17e2] = 0x2375e, [0x17e3] = 0x23749, [0x17e4] = 0x2374e,
  [0x17e5] = 0x2375d, [0x17e6] = 0x2375a, [0x17e7] = 0x23786, [0x17e8] = 0x23751,
  [0x17e9] = 0x23929, [0x17ea] = 0x239a6, [0x17eb] = 0x239a7, [0x17ec] = 0x23a70,
  [0x17ed] = 0x23a6f, [0x17ee] = 0x23ad3, [0x17ef] = 0x23ad2, [0x17f0] = 0x23c28,
  [0x17f1] = 0x23c1b, [0x17f2] = 0x23c1a, [0x17f3] = 0x23c1c, [0x17f4] = 0x23c21,
  [0x17f5] = 0x24032, [0x17f6] = 0x24066, [0x17f7] = 0x2407d, [0x17f8] = 0x24073,
  [0x17f9] = 0x2407f, [0x17fa] = 0x24065, [0x17fb] = 0x24080, [0x17fc] = 0x24061,
  [0x17fd] = 0x24075, [0x17fe] = 0x2406e, [0x17ff] = 0x24067, [0x1800] = 0x24071,
  [0x1801] = 0x2406c, [0x1802] = 0x24063, [0x1803] = 0x24062, [0x1804] = 0x24083,
  [0x1805] = 0x240b2, [0x1806] = 0x24081, [0x1807] = 0x2406f, [0x1808] = 0x2406b,
  [0x1809] = 0x24082, [0x180a] = 0x2408a, [0x180b] = 0x24429, [0x180c] = 0x24430,
  [0x180d] = 0x24431, [0x180e] = 0x24597, [0x180f] = 0x24600, [0x1810] = 0x245ff,
  [0x1811] = 0x248b6, [0x1812] = 0x248b5, [0x1813] = 0x248c3, [0x1814] = 0x24a8d,
  [0x1815] = 0x24b18, [0x1816] = 0x24b9c, [0x1817] = 0x24b9a, [0x1818] = 0x24b99,
  [0x1819] = 0x24eb7, [0x181a] = 0x24ec4, [0x181b] = 0x24ec6, [0x181c] = 0x24ec7,
  [0x181d] = 0x24ebc, [0x181e] = 0x24ec0, [0x181f] = 0x24ec5, [0x1820] = 0x24ecd,
  [0x1821] = 0x24ec1, [0x1822] = 0x24ebe, [0x1823] = 0x24fa7, [0x1824] = 0x24fb8,
  [0x1825] = 0x25023, [0x1826] = 0x4074, [0x1827] = 0x252e8, [0x1828] = 0x252de,
  [0x1829] = 0x252e3, [0x182a] = 0x252ef, [0x182b] = 0x252dc, [0x182c] = 0x252e4,
  [0x182d] = 0x252e1, [0x182e] = 0x252e5, [0x182f] = 0x25395, [0x1830] = 0x25394,
  [0x1831] = 0x25393, [0x1832] = 0x2538e, [0x1833] = 0x25598, [0x1834] = 0x2559c,
  [0x1835] = 0x25599, [0x1836] = 0x255a0, [0x1837] = 0x2559e, [0x1838] = 0x25704,
  [0x1839] = 0x258d3, [0x183a] = 0x258b4, [0x183b] = 0x258bb, [0x183c] = 0x258b7,
  [0x183d] = 0x258b8, [0x183e] = 0x258bd, [0x183f] = 0x258b6, [0x1840] = 0x258b9,
  [0x1841] = 0x258b5, [0x1842] = 0x25a22, [0x1843] = 0x25a21, [0x1844] = 0x25ac3,
  [0x1845] = 0x25abc, [0x1846] = 0x25cff, [0x1847] = 0x25ce5, [0x1848] = 0x25cf7,
  [0x1849] = 0x25cd2, [0x184a] = 0x25d00, [0x184b] = 0x25cdb, [0x184c] = 0x25cf0,
  [0x184d] = 0x25ce2, [0x184e] = 0x25ccd, [0x184f] = 0x4256, [0x1850] = 0x25ce1,
  [0x1851] = 0x25cf2, [0x1852] = 0x25cdc, [0x1853] = 0x25ccc, [0x1854] = 0x25cd6,
  [0x1855] = 0x25cf3, [0x1856] = 0x25cfa, [0x1857] = 0x25cf6, [0x1858] = 0x25cec,
  [0x1859] = 0x25cea, [0x185a] = 0x25ce3, [0x185b] = 0x25cd0, [0x185c] = 0x425a,
  [0x185d] = 0x25c95, [0x185e] = 0x25cd5, [0x185f] = 0x25cf4, [0x1860] = 0x25ccb,
  [0x1861] = 0x25cda, [0x1862] = 0x25cc8, [0x1863] = 0x25cdf, [0x1864] = 0x25cf5,
  [0x1865] = 0x25ccf, [0x1866] = 0x25cc7, [0x1867] = 0x25cd7, [0x1868] = 0x25f24,
  [0x1869] = 0x42a3, [0x186a] = 0x25f1c, [0x186b] = 0x25f21, [0x186c] = 0x25f1e,
  [0x186d] = 0x25f18, [0x186e] = 0x25f1b, [0x186f] = 0x42a8, [0x1870] = 0x26146,
  [0x1871] = 0x2614b, [0x1872] = 0x26148, [0x1873] = 0x26147, [0x1874] = 0x26167,
  [0x1875] = 0x26154, [0x1876] = 0x26315, [0x1877] = 0x26322, [0x1878] = 0x263c5,
  [0x1879] = 0x26329, [0x187a] = 0x263c7, [0x187b] = 0x2648d, [0x187c] = 0x2651b,
  [0x187d] = 0x26520, [0x187e] = 0x265e2, [0x187f] = 0x265e7, [0x1880] = 0x265e8,
  [0x1881] = 0x265e1, [0x1882] = 0x26872, [0x1883] = 0x2687b, [0x1884] = 0x26874,
  [0x1885] = 0x26871, [0x1886] = 0x26879, [0x1887] = 0x26875, [0x1888] = 0x26890,
  [0x1889] = 0x26877, [0x188a] = 0x2687d, [0x188b] = 0x2694f, [0x188c] = 0x26950,
  [0x188d] = 0x269a3, [0x188e] = 0x269a2, [0x188f] = 0x269f4, [0x1890] = 0x26a94,
  [0x1891] = 0x26a9a, [0x1892] = 0x26a9b, [0x1893] = 0x26aa7, [0x1894] = 0x26a97,
  [0x1895] = 0x26a95, [0x1896] = 0x26a92, [0x1897] = 0x26a9c, [0x1898] = 0x26a96,
  [0x1899] = 0x26fb6, [0x189a] = 0x26fb8, [0x189b] = 0x26fb0, [0x189c] = 0x27018,
  [0x189d] = 0x26fc5, [0x189e] = 0x26fb5, [0x189f] = 0x26fc2, [0x18a0] = 0x27006,
  [0x18a1] = 0x27019, [0x18a2] = 0x26fb9, [0x18a3] = 0x27015, [0x18a4] = 0x26fd6,
  [0x18a5] = 0x26fac, [0x18a6] = 0x453c, [0x18a7] = 0x26fc6, [0x18a8] = 0x26fb3,
  [0x18a9] = 0x26fc3, [0x18aa] = 0x26fca, [0x18ab] = 0x26fcf, [0x18ac] = 0x26fbd,
  [0x18ad] = 0x26fce, [0x18ae] = 0x27014, [0x18af] = 0x26fba, [0x18b0] = 0x2701a,
  [0x18b1] = 0x26fc1, [0x18b2] = 0x26fbb, [0x18b3] = 0x459b, [0x18b4] = 0x459a,
  [0x18b5] = 0x27201, [0x18b6] = 0x271fc, [0x18b7] = 0x2745a, [0x18b8] = 0x27454,
  [0x18b9] = 0x27461, [0x18ba] = 0x2745c, [0x18bb] = 0x27455, [0x18bc] = 0x2744a,
  [0x18bd] = 0x2744b, [0x18be] = 0x27451, [0x18bf] = 0x27469, [0x18c0] = 0x2746b,
  [0x18c1] = 0x87f1, [0x18c2] = 0x27466, [0x18c3] = 0x27458, [0x18c4] = 0x2745d,
  [0x18c5] = 0x27467, [0x18c6] = 0x27456, [0x18c7] = 0x27488, [0x18c8] = 0x27464,
  [0x18c9] = 0x2744e, [0x18ca] = 0x27757, [0x18cb] = 0x2775a, [0x18cc] = 0x27751,
  [0x18cd] = 0x2774a, [0x18ce] = 0x2774b, [0x18cf] = 0x465e, [0x18d0] = 0x27747,
  [0x18d1] = 0x2776f, [0x18d2] = 0x2776a, [0x18d3] = 0x2776b, [0x18d4] = 0x27746,
  [0x18d5] = 0x2776d, [0x18d6] = 0x27754, [0x18d7] = 0x2776e, [0x18d8] = 0x2774c,
  [0x18d9] = 0x27878, [0x18da] = 0x27886, [0x18db] = 0x27882, [0x18dc] = 0x46a7,
  [0x18dd] = 0x27ae1, [0x18de] = 0x27ae5, [0x18df] = 0x27adb, [0x18e0] = 0x27ade,
  [0x18e1] = 0x27ad7, [0x18e2] = 0x27c03, [0x18e3] = 0x27c7c, [0x18e4] = 0x27c7e,
  [0x18e5] = 0x27d05, [0x18e6] = 0x27d07, [0x18e7] = 0x27de6, [0x18e8] = 0x27de1,
  [0x18e9] = 0x27dfb, [0x18ea] = 0x27de5, [0x18eb] = 0x27de7, [0x18ec] = 0x27ddf,
  [0x18ed] = 0x27dff, [0x18ee] = 0x27e76, [0x18ef] = 0x27f63, [0x18f0] = 0x27f66,
  [0x18f1] = 0x27f65, [0x18f2] = 0x27f5e, [0x18f3] = 0x27f64, [0x18f4] = 0x27f6b,
  [0x18f5] = 0x27f5f, [0x18f6] = 0x27f67, [0x18f7] = 0x27f68, [0x18f8] = 0x28108,
  [0x18f9] = 0x28109, [0x18fa] = 0x28117, [0x18fb] = 0x28115, [0x18fc] = 0x2811b,
  [0x18fd] = 0x2810b, [0x18fe] = 0x28128, [0x18ff] = 0x2810e, [0x1900] = 0x28118,
  [0x1901] = 0x28153, [0x1902] = 0x28145, [0x1903] = 0x2810d, [0x1904] = 0x2810a,
  [0x1905] = 0x28113, [0x1906] = 0x2814a, [0x1907] = 0x28279, [0x1908] = 0x283a1,
  [0x1909] = 0x2838d, [0x190a] = 0x283a2, [0x190b] = 0x28390, [0x190c] = 0x486e,
  [0x190d] = 0x28431, [0x190e] = 0x28430, [0x190f] = 0x2861f, [0x1910] = 0x28619,
  [0x1911] = 0x2860c, [0x1912] = 0x2861e, [0x1913] = 0x2861d, [0x1914] = 0x28607,
  [0x1915] = 0x28766, [0x1916] = 0x287c5, [0x1917] = 0x287ba, [0x1918] = 0x287bd,
  [0x1919] = 0x287c2, [0x191a] = 0x287c3, [0x191b] = 0x287bf, [0x191c] = 0x490d,
  [0x191d] = 0x288a1, [0x191e] = 0x288a2, [0x191f] = 0x288a8, [0x1920] = 0x288a3,
  [0x1921] = 0x288aa, [0x1922] = 0x288af, [0x1923] = 0x288b9, [0x1924] = 0x28937,
  [0x1925] = 0x28a8f, [0x1926] = 0x4943, [0x1927] = 0x28a8e, [0x1928] = 0x28a87,
  [0x1929] = 0x28a8a, [0x192a] = 0x28a92, [0x192b] = 0x28a97, [0x192c] = 0x28a9f,
  [0x192d] = 0x28b05, [0x192e] = 0x28ca9, [0x192f] = 0x28ca7, [0x1930] = 0x28d8a,
  [0x1931] = 0x28d82, [0x1932] = 0x28d85, [0x1933] = 0x28d8b, [0x1934] = 0x28d89,
  [0x1935] = 0x28d81, [0x1936] = 0x28d80, [0x1937] = 0x28d87, [0x1938] = 0x28d86,
  [0x1939] = 0x28f4d, [0x193a] = 0x2901c, [0x193b] = 0x2901f, [0x193c] = 0x29121,
  [0x193d] = 0x2911d, [0x193e] = 0x29122, [0x193f] = 0x290fe, [0x1940] = 0x2911b,
  [0x1941] = 0x2913a, [0x1942] = 0x29137, [0x1943] = 0x29117, [0x1944] = 0x29138,
  [0x1945] = 0x29126, [0x1946] = 0x29118, [0x1947] = 0x29234, [0x1948] = 0x29235,
  [0x1949] = 0x29232, [0x194a] = 0x4a73, [0x194b] = 0x292e1, [0x194c] = 0x292fd,
  [0x194d] = 0x292e3, [0x194e] = 0x292e8, [0x194f] = 0x292f9, [0x1950] = 0x292ff,
  [0x1951] = 0x292fe, [0x1952] = 0x4a74, [0x1953] = 0x292e0, [0x1954] = 0x29300,
  [0x1955] = 0x4a70, [0x1956] = 0x292ec, [0x1957] = 0x292e4, [0x1958] = 0x292ef,
  [0x1959] = 0x292fa, [0x195a] = 0x293ca, [0x195b] = 0x29431, [0x195c] = 0x29432,
  [0x195d] = 0x29434, [0x195e] = 0x29441, [0x195f] = 0x29500, [0x1960] = 0x29506,
  [0x1961] = 0x29508, [0x1962] = 0x29505, [0x1963] = 0x29503, [0x1964] = 0x2950b,
  [0x1965] = 0x29502, [0x1966] = 0x2950a, [0x1967] = 0x295f0, [0x1968] = 0x29613,
  [0x1969] = 0x2960a, [0x196a] = 0x2960f, [0x196b] = 0x29611, [0x196c] = 0x29608,
  [0x196d] = 0x2960b, [0x196e] = 0x29612, [0x196f] = 0x2960d, [0x1970] = 0x2975b,
  [0x1971] = 0x29763, [0x1972] = 0x29761, [0x1973] = 0x29768, [0x1974] = 0x2975a,
  [0x1975] = 0x2984e, [0x1976] = 0x2984d, [0x1977] = 0x29850, [0x1978] = 0x298fc,
  [0x1979] = 0x29912, [0x197a] = 0x298f5, [0x197b] = 0x2991e, [0x197c] = 0x298f0,
  [0x197d] = 0x298f3, [0x197e] = 0x298f2, [0x197f] = 0x29901, [0x1980] = 0x298ef,
  [0x1981] = 0x298ee, [0x1982] = 0x29916, [0x1983] = 0x298f6, [0x1984] = 0x298fb,
  [0x1985] = 0x2991c, [0x1986] = 0x4bdb, [0x1987] = 0x29a5d, [0x1988] = 0x29a60,
  [0x1989] = 0x29a66, [0x198a] = 0x29a5f, [0x198b] = 0x29a61, [0x198c] = 0x29a5b,
  [0x198d] = 0x29a62, [0x198e] = 0x29a57, [0x198f] = 0x29b69, [0x1990] = 0x29b6b,
  [0x1991] = 0x29b61, [0x1992] = 0x29b77, [0x1993] = 0x29b5f, [0x1994] = 0x29b63,
  [0x1995] = 0x29b62, [0x1996] = 0x29b65, [0x1997] = 0x29cef, [0x1998] = 0x29ce2,
  [0x1999] = 0x29d00, [0x199a] = 0x29ce8, [0x199b] = 0x29cf2, [0x199c] = 0x29ce7,
  [0x199d] = 0x4c29, [0x199e] = 0x29ce5, [0x199f] = 0x29dd5, [0x19a0] = 0x29dd0,
  [0x19a1] = 0x29dda, [0x19a2] = 0x29dd3, [0x19a3] = 0x4c51, [0x19a4] = 0x29dd9,
  [0x19a5] = 0x29dcf, [0x19a6] = 0x29dd6, [0x19a7] = 0x29dd8, [0x19a8] = 0x29df5,
  [0x19a9] = 0x29dce, [0x19aa] = 0x29dd7, [0x19ab] = 0x29df4, [0x19ac] = 0x29dcd,
  [0x19ad] = 0x29e01, [0x19ae] = 0x9bba, [0x19af] = 0x2a052, [0x19b0] = 0x2a055,
  [0x19b1] = 0x2a05a, [0x19b2] = 0x2a04d, [0x19b3] = 0x2a054, [0x19b4] = 0x2a053,
  [0x19b5] = 0x2a05e, [0x19b6] = 0x2a067, [0x19b7] = 0x2a065, [0x19b8] = 0x2a04e,
  [0x19b9] = 0x4cdb, [0x19ba] = 0x2a04f, [0x19bb] = 0x2a061, [0x19bc] = 0x2a06e,
  [0x19bd] = 0x2a051, [0x19be] = 0x2a05b, [0x19bf] = 0x2a263, [0x19c0] = 0x2a2b7,
  [0x19c1] = 0x2a2b6, [0x19c2] = 0x2a2c3, [0x19c3] = 0x2a2bb, [0x19c4] = 0x2a2b5,
  [0x19c5] = 0x2a32e, [0x19c6] = 0x2a330, [0x19c7] = 0x2a333, [0x19c8] = 0x2a331,
  [0x19c9] = 0x2a3a4, [0x19ca] = 0x2a3c1, [0x19cb] = 0x2a3bb, [0x19cc] = 0x2a3be,
  [0x19cd] = 0x2a400, [0x19ce] = 0x2a3ff, [0x19cf] = 0x2a440, [0x19d0] = 0x4d65,
  [0x19d1] = 0x2a43f, [0x19d2] = 0x2a444, [0x19d3] = 0x4d64, [0x19d4] = 0x4d67,
  [0x19d5] = 0x2a4db, [0x19d6] = 0x2a50f, [0x19d7] = 0x2a511, [0x19d8] = 0x2a54e,
  [0x19d9] = 0x2a54d, [0x19da] = 0x2a54b, [0x19db] = 0x2a54c, [0x19dc] = 0x2a595,
  [0x19dd] = 0x2043b, [0x19de] = 0x20445, [0x19df] = 0x20444, [0x19e0] = 0x2043e,
  [0x19e1] = 0x2043c, [0x19e2] = 0x2043f, [0x19e3] = 0x204cc, [0x19e4] = 0x204ce,
  [0x19e5] = 0x20622, [0x19e6] = 0x20623, [0x19e7] = 0x2069f, [0x19e8] = 0x208c5,
  [0x19e9] = 0x208c4, [0x19ea] = 0x210e9, [0x19eb] = 0x21099, [0x19ec] = 0x3611,
  [0x19ed] = 0x210e6, [0x19ee] = 0x210e7, [0x19ef] = 0x210f7, [0x19f0] = 0x214d2,
  [0x19f1] = 0x3733, [0x19f2] = 0x218fd, [0x19f3] = 0x2190c, [0x19f4] = 0x2190b,
  [0x19f5] = 0x21ced, [0x19f6] = 0x3821, [0x19f7] = 0x3822, [0x19f8] = 0x21f9b,
  [0x19f9] = 0x2200a, [0x19fa] = 0x2216f, [0x19fb] = 0x22175, [0x19fc] = 0x222f2,
  [0x19fd] = 0x222f8, [0x19fe] = 0x2243e, [0x19ff] = 0x22924, [0x1a00] = 0x2290e,
  [0x1a01] = 0x22916, [0x1a02] = 0x22918, [0x1a03] = 0x22910, [0x1a04] = 0x22931,
  [0x1a05] = 0x22944, [0x1a06] = 0x39a6, [0x1a07] = 0x22929, [0x1a08] = 0x2292f,
  [0x1a09] = 0x2294b, [0x1a0a] = 0x22936, [0x1a0b] = 0x39a3, [0x1a0c] = 0x22e34,
  [0x1a0d] = 0x22e00, [0x1a0e] = 0x22e2b, [0x1a0f] = 0x23020, [0x1a10] = 0x23021,
  [0x1a11] = 0x2301e, [0x1a12] = 0x2301d, [0x1a13] = 0x232f5, [0x1a14] = 0x232f2,
  [0x1a15] = 0x232f6, [0x1a16] = 0x23790, [0x1a17] = 0x2379d, [0x1a18] = 0x2379c,
  [0x1a19] = 0x23792, [0x1a1a] = 0x23794, [0x1a1b] = 0x237d1, [0x1a1c] = 0x23793,
  [0x1a1d] = 0x237b7, [0x1a1e] = 0x23797, [0x1a1f] = 0x237b0, [0x1a20] = 0x2379f,
  [0x1a21] = 0x237c9, [0x1a22] = 0x2392d, [0x1a23] = 0x2392c, [0x1a24] = 0x23a77,
  [0x1a25] = 0x23c2c, [0x1a26] = 0x240c0, [0x1a27] = 0x240b9, [0x1a28] = 0x24453,
  [0x1a29] = 0x24540, [0x1a2a] = 0x24702, [0x1a2b] = 0x246fb, [0x1a2c] = 0x3ea1,
  [0x1a2d] = 0x24a8b, [0x1a2e] = 0x24aae, [0x1a2f] = 0x24aab, [0x1a30] = 0x24ba1,
  [0x1a31] = 0x2a2cd, [0x1a32] = 0x24cea, [0x1a33] = 0x24edd, [0x1a34] = 0x24edc,
  [0x1a35] = 0x24ed9, [0x1a36] = 0x24fab, [0x1a37] = 0x24fac, [0x1a38] = 0x24fad,
  [0x1a39] = 0x24fb2, [0x1a3a] = 0x2502c, [0x1a3b] = 0x2502b, [0x1a3c] = 0x250c2,
  [0x1a3d] = 0x25300, [0x1a3e] = 0x2530d, [0x1a3f] = 0x25306, [0x1a40] = 0x252ff,
  [0x1a41] = 0x25303, [0x1a42] = 0x25301, [0x1a43] = 0x25310, [0x1a44] = 0x2530f,
  [0x1a45] = 0x25305, [0x1a46] = 0x25398, [0x1a47] = 0x25397, [0x1a48] = 0x25396,
  [0x1a49] = 0x25399, [0x1a4a] = 0x255ac, [0x1a4b] = 0x255ab, [0x1a4c] = 0x25712,
  [0x1a4d] = 0x258d8, [0x1a4e] = 0x258df, [0x1a4f] = 0x258d6, [0x1a50] = 0x25a2d,
  [0x1a51] = 0x25a32, [0x1a52] = 0x25d3f, [0x1a53] = 0x25d2f, [0x1a54] = 0x25d26,
  [0x1a55] = 0x25d3a, [0x1a56] = 0x25d39, [0x1a57] = 0x425f, [0x1a58] = 0x25d3b,
  [0x1a59] = 0x25d35, [0x1a5a] = 0x25d2a, [0x1a5b] = 0x25d21, [0x1a5c] = 0x25d38,
  [0x1a5d] = 0x25d37, [0x1a5e] = 0x25d34, [0x1a5f] = 0x425c, [0x1a60] = 0x25d22,
  [0x1a61] = 0x25d36, [0x1a62] = 0x25d44, [0x1a63] = 0x25f45, [0x1a64] = 0x25f3b,
  [0x1a65] = 0x25f36, [0x1a66] = 0x25f42, [0x1a67] = 0x2617a, [0x1a68] = 0x26186,
  [0x1a69] = 0x2618b, [0x1a6a] = 0x2617f, [0x1a6b] = 0x26181, [0x1a6c] = 0x2632a,
  [0x1a6d] = 0x263d5, [0x1a6e] = 0x2649f, [0x1a6f] = 0x2649d, [0x1a70] = 0x26526,
  [0x1a71] = 0x265f4, [0x1a72] = 0x265f5, [0x1a73] = 0x26815, [0x1a74] = 0x2688e,
  [0x1a75] = 0x2688f, [0x1a76] = 0x26926, [0x1a77] = 0x269a7, [0x1a78] = 0x269af,
  [0x1a79] = 0x26aad, [0x1a7a] = 0x26aac, [0x1a7b] = 0x26aab, [0x1a7c] = 0x26aaa,
  [0x1a7d] = 0x2702f, [0x1a7e] = 0x27021, [0x1a7f] = 0x27023, [0x1a80] = 0x270a3,
  [0x1a81] = 0x27049, [0x1a82] = 0x2703a, [0x1a83] = 0x27048, [0x1a84] = 0x2702d,
  [0x1a85] = 0x27025, [0x1a86] = 0x27029, [0x1a87] = 0x27032, [0x1a88] = 0x27034,
  [0x1a89] = 0x27024, [0x1a8a] = 0x2702c, [0x1a8b] = 0x2704b, [0x1a8c] = 0x2703b,
  [0x1a8d] = 0x27020, [0x1a8e] = 0x27028, [0x1a8f] = 0x27498, [0x1a90] = 0x45f5,
  [0x1a91] = 0x27497, [0x1a92] = 0x8804, [0x1a93] = 0x2749d, [0x1a94] = 0x274a8,
  [0x1a95] = 0x275d5, [0x1a96] = 0x27777, [0x1a97] = 0x27778, [0x1a98] = 0x27772,
  [0x1a99] = 0x27773, [0x1a9a] = 0x27802, [0x1a9b] = 0x2793b, [0x1a9c] = 0x27afe,
  [0x1a9d] = 0x27b0b, [0x1a9e] = 0x27aff, [0x1a9f] = 0x27b07, [0x1aa0] = 0x4711,
  [0x1aa1] = 0x470c, [0x1aa2] = 0x27b30, [0x1aa3] = 0x27bcd, [0x1aa4] = 0x27bcf,
  [0x1aa5] = 0x27c10, [0x1aa6] = 0x27c0a, [0x1aa7] = 0x27c83, [0x1aa8] = 0x27d15,
  [0x1aa9] = 0x27d0e, [0x1aaa] = 0x27d0c, [0x1aab] = 0x27e02, [0x1aac] = 0x27dfe,
  [0x1aad] = 0x27e05, [0x1aae] = 0x27e15, [0x1aaf] = 0x27e08, [0x1ab0] = 0x27f7f,
  [0x1ab1] = 0x27f77, [0x1ab2] = 0x27f7c, [0x1ab3] = 0x27f82, [0x1ab4] = 0x27f7e,
  [0x1ab5] = 0x27f78, [0x1ab6] = 0x27f7d, [0x1ab7] = 0x27f79, [0x1ab8] = 0x27f81,
  [0x1ab9] = 0x2814b, [0x1aba] = 0x28163, [0x1abb] = 0x28164, [0x1abc] = 0x28156,
  [0x1abd] = 0x28154, [0x1abe] = 0x2814e, [0x1abf] = 0x28110, [0x1ac0] = 0x2814f,
  [0x1ac1] = 0x28157, [0x1ac2] = 0x2815e, [0x1ac3] = 0x28151, [0x1ac4] = 0x2816a,
  [0x1ac5] = 0x28169, [0x1ac6] = 0x283ad, [0x1ac7] = 0x283a4, [0x1ac8] = 0x283ac,
  [0x1ac9] = 0x283a9, [0x1aca] = 0x283ae, [0x1acb] = 0x2044c, [0x1acc] = 0x28650,
  [0x1acd] = 0x28635, [0x1ace] = 0x28632, [0x1acf] = 0x287d8, [0x1ad0] = 0x287d1,
  [0x1ad1] = 0x287cf, [0x1ad2] = 0x287be, [0x1ad3] = 0x288d5, [0x1ad4] = 0x288c1,
  [0x1ad5] = 0x288c6, [0x1ad6] = 0x288c3, [0x1ad7] = 0x288c2, [0x1ad8] = 0x288c0,
  [0x1ad9] = 0x288c5, [0x1ada] = 0x288c7, [0x1adb] = 0x288bf, [0x1adc] = 0x288c4,
  [0x1add] = 0x288d4, [0x1ade] = 0x28a90, [0x1adf] = 0x28ad2, [0x1ae0] = 0x494e,
  [0x1ae1] = 0x28ad4, [0x1ae2] = 0x28caf, [0x1ae3] = 0x28cae, [0x1ae4] = 0x28cb2,
  [0x1ae5] = 0x28cad, [0x1ae6] = 0x28d9c, [0x1ae7] = 0x28da0, [0x1ae8] = 0x28db6,
  [0x1ae9] = 0x28f3d, [0x1aea] = 0x28f56, [0x1aeb] = 0x28f82, [0x1aec] = 0x29028,
  [0x1aed] = 0x2902b, [0x1aee] = 0x2913c, [0x1aef] = 0x2913e, [0x1af0] = 0x2913f,
  [0x1af1] = 0x29142, [0x1af2] = 0x2913d, [0x1af3] = 0x29141, [0x1af4] = 0x2913b,
  [0x1af5] = 0x29149, [0x1af6] = 0x29143, [0x1af7] = 0x29239, [0x1af8] = 0x29238,
  [0x1af9] = 0x29322, [0x1afa] = 0x29308, [0x1afb] = 0x2930c, [0x1afc] = 0x29306,
  [0x1afd] = 0x29313, [0x1afe] = 0x29304, [0x1aff] = 0x29320, [0x1b00] = 0x2931d,
  [0x1b01] = 0x29305, [0x1b02] = 0x2930a, [0x1b03] = 0x293d6, [0x1b04] = 0x293d7,
  [0x1b05] = 0x29436, [0x1b06] = 0x29437, [0x1b07] = 0x29523, [0x1b08] = 0x29522,
  [0x1b09] = 0x29520, [0x1b0a] = 0x2951a, [0x1b0b] = 0x2951d, [0x1b0c] = 0x2961c,
  [0x1b0d] = 0x29620, [0x1b0e] = 0x29677, [0x1b0f] = 0x2977f, [0x1b10] = 0x2978b,
  [0x1b11] = 0x2977d, [0x1b12] = 0x29799, [0x1b13] = 0x29784, [0x1b14] = 0x29789,
  [0x1b15] = 0x29785, [0x1b16] = 0x29783, [0x1b17] = 0x29786, [0x1b18] = 0x2979e,
  [0x1b19] = 0x29853, [0x1b1a] = 0x29917, [0x1b1b] = 0x4b94, [0x1b1c] = 0x29919,
  [0x1b1d] = 0x29920, [0x1b1e] = 0x2991f, [0x1b1f] = 0x29923, [0x1b20] = 0x29918,
  [0x1b21] = 0x29921, [0x1b22] = 0x29929, [0x1b23] = 0x29924, [0x1b24] = 0x29926,
  [0x1b25] = 0x29a5e, [0x1b26] = 0x29a6f, [0x1b27] = 0x29a6e, [0x1b28] = 0x29a74,
  [0x1b29] = 0x29a72, [0x1b2a] = 0x29a73, [0x1b2b] = 0x29b7d, [0x1b2c] = 0x29b7e,
  [0x1b2d] = 0x29b85, [0x1b2e] = 0x29b7f, [0x1b2f] = 0x29b84, [0x1b30] = 0x29c44,
  [0x1b31] = 0x29cf5, [0x1b32] = 0x29cf8, [0x1b33] = 0x29d03, [0x1b34] = 0x29cf6,
  [0x1b35] = 0x29e28, [0x1b36] = 0x4c63, [0x1b37] = 0x29e39, [0x1b38] = 0x29dfb,
  [0x1b39] = 0x29e0e, [0x1b3a] = 0x29dfd, [0x1b3b] = 0x29e1f, [0x1b3c] = 0x4c61,
  [0x1b3d] = 0x29e03, [0x1b3e] = 0x9bd8, [0x1b3f] = 0x29e10, [0x1b40] = 0x4c5a,
  [0x1b41] = 0x29e0d, [0x1b42] = 0x29e27, [0x1b43] = 0x29e41, [0x1b44] = 0x2a072,
  [0x1b45] = 0x2a078, [0x1b46] = 0x2a080, [0x1b47] = 0x4ce3, [0x1b48] = 0x2a085,
  [0x1b49] = 0x2a07b, [0x1b4a] = 0x9d7c, [0x1b4b] = 0x2a07d, [0x1b4c] = 0x2a091,
  [0x1b4d] = 0x2a088, [0x1b4e] = 0x2a0aa, [0x1b4f] = 0x2a08d, [0x1b50] = 0x2a089,
  [0x1b51] = 0x2a095, [0x1b52] = 0x2a09b, [0x1b53] = 0x2a08c, [0x1b54] = 0x2a09e,
  [0x1b55] = 0x2a07c, [0x1b56] = 0x2a086, [0x1b57] = 0x2a084, [0x1b58] = 0x2a268,
  [0x1b59] = 0x2a2c5, [0x1b5a] = 0x2a2c4, [0x1b5b] = 0x2a2c9, [0x1b5c] = 0x2a2c6,
  [0x1b5d] = 0x2a342, [0x1b5e] = 0x2a345, [0x1b5f] = 0x2a341, [0x1b60] = 0x2a344,
  [0x1b61] = 0x2a3a6, [0x1b62] = 0x2a3c2, [0x1b63] = 0x2a442, [0x1b64] = 0x2a449,
  [0x1b65] = 0x2a448, [0x1b66] = 0x2a44a, [0x1b67] = 0x2a44c, [0x1b68] = 0x2a44b,
  [0x1b69] = 0x2a517, [0x1b6a] = 0x2a518, [0x1b6b] = 0x2a515, [0x1b6c] = 0x2a552,
  [0x1b6d] = 0x2a554, [0x1b6e] = 0x2a553, [0x1b6f] = 0x2a59a, [0x1b70] = 0x2a59b,
  [0x1b71] = 0x2a5dc, [0x1b72] = 0x2a5e4, [0x1b73] = 0x2a691, [0x1b74] = 0x4daf,
  [0x1b75] = 0x2a6ba, [0x1b76] = 0x20451, [0x1b77] = 0x2045b, [0x1b78] = 0x204cf,
  [0x1b79] = 0x23305, [0x1b7a] = 0x21113, [0x1b7b] = 0x21115, [0x1b7c] = 0x21114,
  [0x1b7d] = 0x21123, [0x1b7e] = 0x3613, [0x1b7f] = 0x21111, [0x1b80] = 0x21112,
  [0x1b81] = 0x3680, [0x1b82] = 0x214f0, [0x1b83] = 0x2169b, [0x1b84] = 0x21912,
  [0x1b85] = 0x21916, [0x1b86] = 0x21917, [0x1b87] = 0x219b5, [0x1b88] = 0x21cf4,
  [0x1b89] = 0x21fb1, [0x1b8a] = 0x22179, [0x1b8b] = 0x222ff, [0x1b8c] = 0x222fc,
  [0x1b8d] = 0x222fb, [0x1b8e] = 0x2243f, [0x1b8f] = 0x22444, [0x1b90] = 0x2259d,
  [0x1b91] = 0x22932, [0x1b92] = 0x2294c, [0x1b93] = 0x2292b, [0x1b94] = 0x2295a,
  [0x1b95] = 0x22a1f, [0x1b96] = 0x22e01, [0x1b97] = 0x22e2d, [0x1b98] = 0x22e27,
  [0x1b99] = 0x22e59, [0x1b9a] = 0x22e5a, [0x1b9b] = 0x22e4d, [0x1b9c] = 0x22e58,
  [0x1b9d] = 0x23023, [0x1b9e] = 0x23025, [0x1b9f] = 0x2302b, [0x1ba0] = 0x232ff,
  [0x1ba1] = 0x233ae, [0x1ba2] = 0x237cb, [0x1ba3] = 0x237ca, [0x1ba4] = 0x237d0,
  [0x1ba5] = 0x237ce, [0x1ba6] = 0x237cc, [0x1ba7] = 0x237d8, [0x1ba8] = 0x237c6,
  [0x1ba9] = 0x237d2, [0x1baa] = 0x237cf, [0x1bab] = 0x237c8, [0x1bac] = 0x23b17,
  [0x1bad] = 0x4db0, [0x1bae] = 0x240c4, [0x1baf] = 0x29240, [0x1bb0] = 0x240e9,
  [0x1bb1] = 0x240f2, [0x1bb2] = 0x240fc, [0x1bb3] = 0x240ea, [0x1bb4] = 0x240eb,
  [0x1bb5] = 0x240fd, [0x1bb6] = 0x24478, [0x1bb7] = 0x24477, [0x1bb8] = 0x24473,
  [0x1bb9] = 0x2449e, [0x1bba] = 0x24479, [0x1bbb] = 0x245a2, [0x1bbc] = 0x24603,
  [0x1bbd] = 0x24704, [0x1bbe] = 0x24705, [0x1bbf] = 0x248ce, [0x1bc0] = 0x24abf,
  [0x1bc1] = 0x24ef6, [0x1bc2] = 0x24ef7, [0x1bc3] = 0x25031, [0x1bc4] = 0x25030,
  [0x1bc5] = 0x250c9, [0x1bc6] = 0x250c7, [0x1bc7] = 0x2531c, [0x1bc8] = 0x25328,
  [0x1bc9] = 0x2531a, [0x1bca] = 0x2531e, [0x1bcb] = 0x2531b, [0x1bcc] = 0x2531f,
  [0x1bcd] = 0x255bf, [0x1bce] = 0x255bb, [0x1bcf] = 0x255bc, [0x1bd0] = 0x255c0,
  [0x1bd1] = 0x2571a, [0x1bd2] = 0x258ef, [0x1bd3] = 0x258ec, [0x1bd4] = 0x258e9,
  [0x1bd5] = 0x258f0, [0x1bd6] = 0x258fe, [0x1bd7] = 0x25a34, [0x1bd8] = 0x25d6a,
  [0x1bd9] = 0x25d95, [0x1bda] = 0x25d6c, [0x1bdb] = 0x25d72, [0x1bdc] = 0x25d67,
  [0x1bdd] = 0x25d60, [0x1bde] = 0x25d6b, [0x1bdf] = 0x25d5e, [0x1be0] = 0x25d6f,
  [0x1be1] = 0x25d66, [0x1be2] = 0x7c46, [0x1be3] = 0x25d62, [0x1be4] = 0x25d5d,
  [0x1be5] = 0x25d5c, [0x1be6] = 0x2f965, [0x1be7] = 0x25f50, [0x1be8] = 0x2619f,
  [0x1be9] = 0x4323, [0x1bea] = 0x2619e, [0x1beb] = 0x261a6, [0x1bec] = 0x26335,
  [0x1bed] = 0x26338, [0x1bee] = 0x26336, [0x1bef] = 0x2633a, [0x1bf0] = 0x263dc,
  [0x1bf1] = 0x2687c, [0x1bf2] = 0x268ab, [0x1bf3] = 0x4449, [0x1bf4] = 0x268a8,
  [0x1bf5] = 0x268a7, [0x1bf6] = 0x2692b, [0x1bf7] = 0x2692c, [0x1bf8] = 0x26928,
  [0x1bf9] = 0x2092b, [0x1bfa] = 0x269a9, [0x1bfb] = 0x269aa, [0x1bfc] = 0x269ab,
  [0x1bfd] = 0x269f8, [0x1bfe] = 0x26ab1, [0x1bff] = 0x26af0, [0x1c00] = 0x26aef,
  [0x1c01] = 0x270a8, [0x1c02] = 0x2708b, [0x1c03] = 0x27094, [0x1c04] = 0x2709e,
  [0x1c05] = 0x2708f, [0x1c06] = 0x27088, [0x1c07] = 0x2707e, [0x1c08] = 0x27081,
  [0x1c09] = 0x27030, [0x1c0a] = 0x2709b, [0x1c0b] = 0x27082, [0x1c0c] = 0x27090,
  [0x1c0d] = 0x27085, [0x1c0e] = 0x2707f, [0x1c0f] = 0x270a9, [0x1c10] = 0x270de,
  [0x1c11] = 0x2720d, [0x1c12] = 0x45fa, [0x1c13] = 0x274cf, [0x1c14] = 0x274cb,
  [0x1c15] = 0x274d8, [0x1c16] = 0x274dd, [0x1c17] = 0x274d3, [0x1c18] = 0x274d0,
  [0x1c19] = 0x274d5, [0x1c1a] = 0x274d6, [0x1c1b] = 0x275d6, [0x1c1c] = 0x27792,
  [0x1c1d] = 0x27795, [0x1c1e] = 0x2a5cb, [0x1c1f] = 0x2778d, [0x1c20] = 0x2779b,
  [0x1c21] = 0x2779d, [0x1c22] = 0x2778f, [0x1c23] = 0x2779e, [0x1c24] = 0x277a6,
  [0x1c25] = 0x27896, [0x1c26] = 0x27892, [0x1c27] = 0x27b16, [0x1c28] = 0x27b2a,
  [0x1c29] = 0x27b29, [0x1c2a] = 0x27b2c, [0x1c2b] = 0x27c15, [0x1c2c] = 0x27c12,
  [0x1c2d] = 0x27c11, [0x1c2e] = 0x27c13, [0x1c2f] = 0x27c88, [0x1c30] = 0x27c8b,
  [0x1c31] = 0x27c8a, [0x1c32] = 0x27c87, [0x1c33] = 0x27d17, [0x1c34] = 0x27d16,
  [0x1c35] = 0x27d1a, [0x1c36] = 0x27e19, [0x1c37] = 0x27e17, [0x1c38] = 0x27f91,
  [0x1c39] = 0x27f94, [0x1c3a] = 0x2818b, [0x1c3b] = 0x28190, [0x1c3c] = 0x2818f,
  [0x1c3d] = 0x28186, [0x1c3e] = 0x28183, [0x1c3f] = 0x2818e, [0x1c40] = 0x28187,
  [0x1c41] = 0x281a8, [0x1c42] = 0x28185, [0x1c43] = 0x281a6, [0x1c44] = 0x28182,
  [0x1c45] = 0x281a7, [0x1c46] = 0x283b9, [0x1c47] = 0x283b7, [0x1c48] = 0x283b4,
  [0x1c49] = 0x283b6, [0x1c4a] = 0x283b3, [0x1c4b] = 0x283c6, [0x1c4c] = 0x2863c,
  [0x1c4d] = 0x28640, [0x1c4e] = 0x28638, [0x1c4f] = 0x28791, [0x1c50] = 0x287a6,
  [0x1c51] = 0x287da, [0x1c52] = 0x288da, [0x1c53] = 0x288d8, [0x1c54] = 0x288d9,
  [0x1c55] = 0x288db, [0x1c56] = 0x288d7, [0x1c57] = 0x28b16, [0x1c58] = 0x28b12,
  [0x1c59] = 0x28b1f, [0x1c5a] = 0x4959, [0x1c5b] = 0x28b14, [0x1c5c] = 0x28b1a,
  [0x1c5d] = 0x28b10, [0x1c5e] = 0x28cb3, [0x1c5f] = 0x28dae, [0x1c60] = 0x28dc1,
  [0x1c61] = 0x28db0, [0x1c62] = 0x28daf, [0x1c63] = 0x28db1, [0x1c64] = 0x28dad,
  [0x1c65] = 0x28db2, [0x1c66] = 0x28dc4, [0x1c67] = 0x2903c, [0x1c68] = 0x2903f,
  [0x1c69] = 0x29161, [0x1c6a] = 0x29166, [0x1c6b] = 0x2923c, [0x1c6c] = 0x2923b,
  [0x1c6d] = 0x2932c, [0x1c6e] = 0x4a7e, [0x1c6f] = 0x2932a, [0x1c70] = 0x2933e,
  [0x1c71] = 0x2932f, [0x1c72] = 0x29332, [0x1c73] = 0x29327, [0x1c74] = 0x29329,
  [0x1c75] = 0x29340, [0x1c76] = 0x293df, [0x1c77] = 0x293de, [0x1c78] = 0x2943c,
  [0x1c79] = 0x2943b, [0x1c7a] = 0x2943e, [0x1c7b] = 0x29521, [0x1c7c] = 0x29546,
  [0x1c7d] = 0x2953c, [0x1c7e] = 0x29536, [0x1c7f] = 0x29538, [0x1c80] = 0x29535,
  [0x1c81] = 0x29631, [0x1c82] = 0x29636, [0x1c83] = 0x2962d, [0x1c84] = 0x29633,
  [0x1c85] = 0x2962f, [0x1c86] = 0x2962e, [0x1c87] = 0x29635, [0x1c88] = 0x297ac,
  [0x1c89] = 0x297a9, [0x1c8a] = 0x297a6, [0x1c8b] = 0x4ba8, [0x1c8c] = 0x2994c,
  [0x1c8d] = 0x29943, [0x1c8e] = 0x29941, [0x1c8f] = 0x2994f, [0x1c90] = 0x29942,
  [0x1c91] = 0x29951, [0x1c92] = 0x4ba9, [0x1c93] = 0x29940, [0x1c94] = 0x29950,
  [0x1c95] = 0x29945, [0x1c96] = 0x2994a, [0x1c97] = 0x2994b, [0x1c98] = 0x29a83,
  [0x1c99] = 0x29a82, [0x1c9a] = 0x29a81, [0x1c9b] = 0x29ae0, [0x1c9c] = 0x29b98,
  [0x1c9d] = 0x29b9f, [0x1c9e] = 0x29b9b, [0x1c9f] = 0x29b9a, [0x1ca0] = 0x29b99,
  [0x1ca1] = 0x29b96, [0x1ca2] = 0x29bae, [0x1ca3] = 0x29b9e, [0x1ca4] = 0x29d09,
  [0x1ca5] = 0x29d0d, [0x1ca6] = 0x29e4e, [0x1ca7] = 0x29e4a, [0x1ca8] = 0x29e4d,
  [0x1ca9] = 0x29e40, [0x1caa] = 0x29e3e, [0x1cab] = 0x29e48, [0x1cac] = 0x29e42,
  [0x1cad] = 0x29e62, [0x1cae] = 0x29e45, [0x1caf] = 0x29e51, [0x1cb0] = 0x2a0c6,
  [0x1cb1] = 0x2a0d0, [0x1cb2] = 0x2a0c0, [0x1cb3] = 0x2a0b7, [0x1cb4] = 0x2a0c2,
  [0x1cb5] = 0x2a0bc, [0x1cb6] = 0x2a0c5, [0x1cb7] = 0x2a0dc, [0x1cb8] = 0x2a0db,
  [0x1cb9] = 0x2a0d2, [0x1cba] = 0x2a0c7, [0x1cbb] = 0x2a0b6, [0x1cbc] = 0x2a0c9,
  [0x1cbd] = 0x2a0cc, [0x1cbe] = 0x2a0d1, [0x1cbf] = 0x2a0cd, [0x1cc0] = 0x2a0da,
  [0x1cc1] = 0x2a0ba, [0x1cc2] = 0x2a0d3, [0x1cc3] = 0x2a0ce, [0x1cc4] = 0x2a0f6,
  [0x1cc5] = 0x2a0bd, [0x1cc6] = 0x2a0dd, [0x1cc7] = 0x2a0c8, [0x1cc8] = 0x2a10d,
  [0x1cc9] = 0x2a135, [0x1cca] = 0x2a271, [0x1ccb] = 0x2a26e, [0x1ccc] = 0x2a26f,
  [0x1ccd] = 0x2a2d6, [0x1cce] = 0x2a34c, [0x1ccf] = 0x2a34f, [0x1cd0] = 0x2a354,
  [0x1cd1] = 0x2a353, [0x1cd2] = 0x2a3c9, [0x1cd3] = 0x2a3c8, [0x1cd4] = 0x2a403,
  [0x1cd5] = 0x4d54, [0x1cd6] = 0x2a404, [0x1cd7] = 0x4d56, [0x1cd8] = 0x2a457,
  [0x1cd9] = 0x2a452, [0x1cda] = 0x2a453, [0x1cdb] = 0x4d6d, [0x1cdc] = 0x2a456,
  [0x1cdd] = 0x2a45c, [0x1cde] = 0x2a455, [0x1cdf] = 0x2a564, [0x1ce0] = 0x2a55d,
  [0x1ce1] = 0x2a55e, [0x1ce2] = 0x2a5a2, [0x1ce3] = 0x2a5a3, [0x1ce4] = 0x4d8e,
  [0x1ce5] = 0x2a5e7, [0x1ce6] = 0x2a5e6, [0x1ce7] = 0x2a698, [0x1ce8] = 0x2a6ac,
  [0x1ce9] = 0x2a6af, [0x1cea] = 0x2a6ae, [0x1ceb] = 0x20459, [0x1cec] = 0x205a9,
  [0x1ced] = 0x2082a, [0x1cee] = 0x208c7, [0x1cef] = 0x21141, [0x1cf0] = 0x21137,
  [0x1cf1] = 0x21135, [0x1cf2] = 0x21133, [0x1cf3] = 0x21139, [0x1cf4] = 0x21132,
  [0x1cf5] = 0x214ff, [0x1cf6] = 0x21501, [0x1cf7] = 0x214f8, [0x1cf8] = 0x2191b,
  [0x1cf9] = 0x21919, [0x1cfa] = 0x21bbf, [0x1cfb] = 0x21fbc, [0x1cfc] = 0x21fbe,
  [0x1cfd] = 0x21fbd, [0x1cfe] = 0x2217d, [0x1cff] = 0x22446, [0x1d00] = 0x22447,
  [0x1d01] = 0x225a4, [0x1d02] = 0x22a21, [0x1d03] = 0x22a62, [0x1d04] = 0x22e86,
  [0x1d05] = 0x3a72, [0x1d06] = 0x2302f, [0x1d07] = 0x23031, [0x1d08] = 0x3c14,
  [0x1d09] = 0x237f8, [0x1d0a] = 0x237f5, [0x1d0b] = 0x237f9, [0x1d0c] = 0x237f2,
  [0x1d0d] = 0x237fa, [0x1d0e] = 0x237f3, [0x1d0f] = 0x23814, [0x1d10] = 0x237fd,
  [0x1d11] = 0x2380f, [0x1d12] = 0x2380e, [0x1d13] = 0x23801, [0x1d14] = 0x23937,
  [0x1d15] = 0x23935, [0x1d16] = 0x23934, [0x1d17] = 0x23931, [0x1d18] = 0x23a7a,
  [0x1d19] = 0x23a7b, [0x1d1a] = 0x23c37, [0x1d1b] = 0x2412b, [0x1d1c] = 0x240fb,
  [0x1d1d] = 0x24116, [0x1d1e] = 0x24113, [0x1d1f] = 0x3d93, [0x1d20] = 0x24111,
  [0x1d21] = 0x2410f, [0x1d22] = 0x2411b, [0x1d23] = 0x24138, [0x1d24] = 0x244a4,
  [0x1d25] = 0x24709, [0x1d26] = 0x24707, [0x1d27] = 0x2470b, [0x1d28] = 0x248d3,
  [0x1d29] = 0x248d1, [0x1d2a] = 0x248d8, [0x1d2b] = 0x24b1d, [0x1d2c] = 0x24ba9,
  [0x1d2d] = 0x24bd0, [0x1d2e] = 0x27215, [0x1d2f] = 0x24f02, [0x1d30] = 0x24f05,
  [0x1d31] = 0x24f01, [0x1d32] = 0x24f00, [0x1d33] = 0x2532e, [0x1d34] = 0x25330,
  [0x1d35] = 0x2532f, [0x1d36] = 0x25331, [0x1d37] = 0x255d2, [0x1d38] = 0x255d3,
  [0x1d39] = 0x25902, [0x1d3a] = 0x25a40, [0x1d3b] = 0x25a42, [0x1d3c] = 0x25a3b,
  [0x1d3d] = 0x25ace, [0x1d3e] = 0x25d98, [0x1d3f] = 0x25d8f, [0x1d40] = 0x25d94,
  [0x1d41] = 0x25d91, [0x1d42] = 0x426f, [0x1d43] = 0x25dba, [0x1d44] = 0x25d90,
  [0x1d45] = 0x25d86, [0x1d46] = 0x25d9a, [0x1d47] = 0x25d8c, [0x1d48] = 0x25d93,
  [0x1d49] = 0x25d87, [0x1d4a] = 0x25d88, [0x1d4b] = 0x25d97, [0x1d4c] = 0x25d8d,
  [0x1d4d] = 0x25d9c, [0x1d4e] = 0x25dbd, [0x1d4f] = 0x25f3c, [0x1d50] = 0x25f59,
  [0x1d51] = 0x42b0, [0x1d52] = 0x261d1, [0x1d53] = 0x261bb, [0x1d54] = 0x261be,
  [0x1d55] = 0x2625d, [0x1d56] = 0x263e2, [0x1d57] = 0x26605, [0x1d58] = 0x268ba,
  [0x1d59] = 0x352e, [0x1d5a] = 0x268f4, [0x1d5b] = 0x269b2, [0x1d5c] = 0x269f9,
  [0x1d5d] = 0x26ab7, [0x1d5e] = 0x26ab6, [0x1d5f] = 0x27089, [0x1d60] = 0x270f9,
  [0x1d61] = 0x270d9, [0x1d62] = 0x270e8, [0x1d63] = 0x270d4, [0x1d64] = 0x270db,
  [0x1d65] = 0x270e2, [0x1d66] = 0x270df, [0x1d67] = 0x270d1, [0x1d68] = 0x270e9,
  [0x1d69] = 0x270ea, [0x1d6a] = 0x27213, [0x1d6b] = 0x27211, [0x1d6c] = 0x274fa,
  [0x1d6d] = 0x274f8, [0x1d6e] = 0x274f4, [0x1d6f] = 0x274fb, [0x1d70] = 0x2750e,
  [0x1d71] = 0x27502, [0x1d72] = 0x2750f, [0x1d73] = 0x27790, [0x1d74] = 0x277ad,
  [0x1d75] = 0x277a9, [0x1d76] = 0x27948, [0x1d77] = 0x27b5a, [0x1d78] = 0x27b4f,
  [0x1d79] = 0x27b4e, [0x1d7a] = 0x27b55, [0x1d7b] = 0x27b54, [0x1d7c] = 0x27b4a,
  [0x1d7d] = 0x27bd5, [0x1d7e] = 0x27c18, [0x1d7f] = 0x27c8d, [0x1d80] = 0x27d1d,
  [0x1d81] = 0x27d19, [0x1d82] = 0x27e26, [0x1d83] = 0x27e28, [0x1d84] = 0x27e2b,
  [0x1d85] = 0x27e7d, [0x1d86] = 0x27fa0, [0x1d87] = 0x27f9a, [0x1d88] = 0x27f9b,
  [0x1d89] = 0x281b5, [0x1d8a] = 0x281ad, [0x1d8b] = 0x281b2, [0x1d8c] = 0x28294,
  [0x1d8d] = 0x283c9, [0x1d8e] = 0x2864e, [0x1d8f] = 0x2864f, [0x1d90] = 0x28644,
  [0x1d91] = 0x28652, [0x1d92] = 0x288e9, [0x1d93] = 0x28939, [0x1d94] = 0x4969,
  [0x1d95] = 0x957e, [0x1d96] = 0x28dc7, [0x1d97] = 0x28dd3, [0x1d98] = 0x28dc6,
  [0x1d99] = 0x28dc3, [0x1d9a] = 0x49b4, [0x1d9b] = 0x28dd2, [0x1d9c] = 0x28f66,
  [0x1d9d] = 0x2917f, [0x1d9e] = 0x29180, [0x1d9f] = 0x29184, [0x1da0] = 0x29185,
  [0x1da1] = 0x291e3, [0x1da2] = 0x4a4d, [0x1da3] = 0x29241, [0x1da4] = 0x29244,
  [0x1da5] = 0x29243, [0x1da6] = 0x2934e, [0x1da7] = 0x2934f, [0x1da8] = 0x4a83,
  [0x1da9] = 0x29349, [0x1daa] = 0x2934a, [0x1dab] = 0x2934b, [0x1dac] = 0x29343,
  [0x1dad] = 0x293e0, [0x1dae] = 0x293e5, [0x1daf] = 0x293e1, [0x1db0] = 0x293e6,
  [0x1db1] = 0x293e2, [0x1db2] = 0x2940c, [0x1db3] = 0x29440, [0x1db4] = 0x29549,
  [0x1db5] = 0x2954a, [0x1db6] = 0x29554, [0x1db7] = 0x2954c, [0x1db8] = 0x29555,
  [0x1db9] = 0x29556, [0x1dba] = 0x2963b, [0x1dbb] = 0x2963d, [0x1dbc] = 0x297a4,
  [0x1dbd] = 0x297a8, [0x1dbe] = 0x297c3, [0x1dbf] = 0x297bf, [0x1dc0] = 0x297c8,
  [0x1dc1] = 0x297c2, [0x1dc2] = 0x297ca, [0x1dc3] = 0x297cc, [0x1dc4] = 0x297c9,
  [0x1dc5] = 0x297be, [0x1dc6] = 0x297cd, [0x1dc7] = 0x297c7, [0x1dc8] = 0x297c5,
  [0x1dc9] = 0x2985d, [0x1dca] = 0x4bb0, [0x1dcb] = 0x2996c, [0x1dcc] = 0x2996b,
  [0x1dcd] = 0x29970, [0x1dce] = 0x2996d, [0x1dcf] = 0x2996f, [0x1dd0] = 0x29989,
  [0x1dd1] = 0x29984, [0x1dd2] = 0x29a8d, [0x1dd3] = 0x29a8a, [0x1dd4] = 0x29a8e,
  [0x1dd5] = 0x29a91, [0x1dd6] = 0x29bb5, [0x1dd7] = 0x29bb1, [0x1dd8] = 0x29baf,
  [0x1dd9] = 0x29bb9, [0x1dda] = 0x29bb7, [0x1ddb] = 0x29bb0, [0x1ddc] = 0x4c0c,
  [0x1ddd] = 0x29c17, [0x1dde] = 0x4c18, [0x1ddf] = 0x29c4f, [0x1de0] = 0x29d19,
  [0x1de1] = 0x29d10, [0x1de2] = 0x29d18, [0x1de3] = 0x29d11, [0x1de4] = 0x29d1c,
  [0x1de5] = 0x29d12, [0x1de6] = 0x29e76, [0x1de7] = 0x29e71, [0x1de8] = 0x9c20,
  [0x1de9] = 0x29e7a, [0x1dea] = 0x29e7f, [0x1deb] = 0x9c22, [0x1dec] = 0x29e73,
  [0x1ded] = 0x29eab, [0x1dee] = 0x9c1e, [0x1def] = 0x29e77, [0x1df0] = 0x29e74,
  [0x1df1] = 0x29e7e, [0x1df2] = 0x29e9b, [0x1df3] = 0x29e84, [0x1df4] = 0x29e7c,
  [0x1df5] = 0x2a129, [0x1df6] = 0x2a12b, [0x1df7] = 0x2a10e, [0x1df8] = 0x2a100,
  [0x1df9] = 0x2a10b, [0x1dfa] = 0x2a0fe, [0x1dfb] = 0x2a0fa, [0x1dfc] = 0x2a117,
  [0x1dfd] = 0x2a0ff, [0x1dfe] = 0x2a10c, [0x1dff] = 0x2a10f, [0x1e00] = 0x4cf5,
  [0x1e01] = 0x2a102, [0x1e02] = 0x2a101, [0x1e03] = 0x2a0fc, [0x1e04] = 0x2a149,
  [0x1e05] = 0x2a106, [0x1e06] = 0x2a112, [0x1e07] = 0x2a113, [0x1e08] = 0x2a278,
  [0x1e09] = 0x2a35b, [0x1e0a] = 0x2a362, [0x1e0b] = 0x2a35f, [0x1e0c] = 0x2a35d,
  [0x1e0d] = 0x2a3ab, [0x1e0e] = 0x2a3d5, [0x1e0f] = 0x2a3d4, [0x1e10] = 0x2a3d3,
  [0x1e11] = 0x2a407, [0x1e12] = 0x2a46c, [0x1e13] = 0x2a470, [0x1e14] = 0x2a46e,
  [0x1e15] = 0x2a468, [0x1e16] = 0x2a46d, [0x1e17] = 0x2a477, [0x1e18] = 0x2a46a,
  [0x1e19] = 0x2a4ce, [0x1e1a] = 0x2a4ec, [0x1e1b] = 0x2a569, [0x1e1c] = 0x2a568,
  [0x1e1d] = 0x2a5a6, [0x1e1e] = 0x2a5a9, [0x1e1f] = 0x2a5aa, [0x1e20] = 0x2a600,
  [0x1e21] = 0x2a60d, [0x1e22] = 0x2a5f8, [0x1e23] = 0x2a5fc, [0x1e24] = 0x2a60a,
  [0x1e25] = 0x2a5f7, [0x1e26] = 0x2a601, [0x1e27] = 0x2a6b6, [0x1e28] = 0x2a6bb,
  [0x1e29] = 0x2a6b7, [0x1e2a] = 0x2a6b9, [0x1e2b] = 0x2a6ca, [0x1e2c] = 0x20469,
  [0x1e2d] = 0x20625, [0x1e2e] = 0x21159, [0x1e2f] = 0x21507, [0x1e30] = 0x21509,
  [0x1e31] = 0x21922, [0x1e32] = 0x21b07, [0x1e33] = 0x21b04, [0x1e34] = 0x2330f,
  [0x1e35] = 0x21cf8, [0x1e36] = 0x21fc7, [0x1e37] = 0x21fd1, [0x1e38] = 0x2217e,
  [0x1e39] = 0x22308, [0x1e3a] = 0x2244a, [0x1e3b] = 0x2f898, [0x1e3c] = 0x224b9,
  [0x1e3d] = 0x22e88, [0x1e3e] = 0x22e91, [0x1e3f] = 0x22e84, [0x1e40] = 0x22e73,
  [0x1e41] = 0x22e89, [0x1e42] = 0x22e85, [0x1e43] = 0x23033, [0x1e44] = 0x23313,
  [0x1e45] = 0x2381d, [0x1e46] = 0x2381f, [0x1e47] = 0x2381c, [0x1e48] = 0x23820,
  [0x1e49] = 0x2381a, [0x1e4a] = 0x2381b, [0x1e4b] = 0x23939, [0x1e4c] = 0x239af,
  [0x1e4d] = 0x23ae5, [0x1e4e] = 0x23c3c, [0x1e4f] = 0x24137, [0x1e50] = 0x2413a,
  [0x1e51] = 0x244bb, [0x1e52] = 0x3e0e, [0x1e53] = 0x24710, [0x1e54] = 0x2470d,
  [0x1e55] = 0x24baf, [0x1e56] = 0x24c11, [0x1e57] = 0x3fd8, [0x1e58] = 0x24f0d,
  [0x1e59] = 0x24f0c, [0x1e5a] = 0x24f0b, [0x1e5b] = 0x250d4, [0x1e5c] = 0x2533d,
  [0x1e5d] = 0x2533e, [0x1e5e] = 0x2533b, [0x1e5f] = 0x25343, [0x1e60] = 0x25340,
  [0x1e61] = 0x25346, [0x1e62] = 0x25411, [0x1e63] = 0x255dd, [0x1e64] = 0x255df,
  [0x1e65] = 0x255ea, [0x1e66] = 0x2574a, [0x1e67] = 0x25906, [0x1e68] = 0x25dc1,
  [0x1e69] = 0x25db9, [0x1e6a] = 0x25dc6, [0x1e6b] = 0x25db8, [0x1e6c] = 0x25dbb,
  [0x1e6d] = 0x25dc8, [0x1e6e] = 0x25dc5, [0x1e6f] = 0x25dbf, [0x1e70] = 0x25dc7,
  [0x1e71] = 0x25dc4, [0x1e72] = 0x25f65, [0x1e73] = 0x25f67, [0x1e74] = 0x261d7,
  [0x1e75] = 0x261db, [0x1e76] = 0x261d4, [0x1e77] = 0x261d6, [0x1e78] = 0x263e8,
  [0x1e79] = 0x2660a, [0x1e7a] = 0x26abd, [0x1e7b] = 0x26abe, [0x1e7c] = 0x2711a,
  [0x1e7d] = 0x2710d, [0x1e7e] = 0x2710f, [0x1e7f] = 0x2711b, [0x1e80] = 0x27110,
  [0x1e81] = 0x27111, [0x1e82] = 0x27113, [0x1e83] = 0x27218, [0x1e84] = 0x27520,
  [0x1e85] = 0x2751f, [0x1e86] = 0x27523, [0x1e87] = 0x2751d, [0x1e88] = 0x27537,
  [0x1e89] = 0x27525, [0x1e8a] = 0x27524, [0x1e8b] = 0x2752a, [0x1e8c] = 0x27527,
  [0x1e8d] = 0x27533, [0x1e8e] = 0x27528, [0x1e8f] = 0x27534, [0x1e90] = 0x277ba,
  [0x1e91] = 0x8970, [0x1e92] = 0x2789d, [0x1e93] = 0x2794c, [0x1e94] = 0x27b5c,
  [0x1e95] = 0x27b6a, [0x1e96] = 0x27b5d, [0x1e97] = 0x27b65, [0x1e98] = 0x27b63,
  [0x1e99] = 0x27b5e, [0x1e9a] = 0x27c19, [0x1e9b] = 0x27c97, [0x1e9c] = 0x27e3f,
  [0x1e9d] = 0x27e33, [0x1e9e] = 0x27e32, [0x1e9f] = 0x27fa1, [0x1ea0] = 0x27fa5,
  [0x1ea1] = 0x27fa4, [0x1ea2] = 0x27fa2, [0x1ea3] = 0x281c1, [0x1ea4] = 0x482c,
  [0x1ea5] = 0x281c7, [0x1ea6] = 0x281c4, [0x1ea7] = 0x281c6, [0x1ea8] = 0x281c5,
  [0x1ea9] = 0x281d4, [0x1eaa] = 0x281ca, [0x1eab] = 0x28653, [0x1eac] = 0x288f1,
  [0x1ead] = 0x28921, [0x1eae] = 0x496e, [0x1eaf] = 0x28bb7, [0x1eb0] = 0x28b92,
  [0x1eb1] = 0x28dd4, [0x1eb2] = 0x28f44, [0x1eb3] = 0x29198, [0x1eb4] = 0x2919b,
  [0x1eb5] = 0x29191, [0x1eb6] = 0x29195, [0x1eb7] = 0x2919a, [0x1eb8] = 0x29192,
  [0x1eb9] = 0x29353, [0x1eba] = 0x29357, [0x1ebb] = 0x4a86, [0x1ebc] = 0x2935c,
  [0x1ebd] = 0x2935d, [0x1ebe] = 0x29364, [0x1ebf] = 0x293ea, [0x1ec0] = 0x293ed,
  [0x1ec1] = 0x29442, [0x1ec2] = 0x29443, [0x1ec3] = 0x29564, [0x1ec4] = 0x29561,
  [0x1ec5] = 0x29560, [0x1ec6] = 0x2967d, [0x1ec7] = 0x297d4, [0x1ec8] = 0x297d5,
  [0x1ec9] = 0x297d9, [0x1eca] = 0x29987, [0x1ecb] = 0x29999, [0x1ecc] = 0x2998c,
  [0x1ecd] = 0x2998a, [0x1ece] = 0x2998f, [0x1ecf] = 0x4bb3, [0x1ed0] = 0x2998b,
  [0x1ed1] = 0x29982, [0x1ed2] = 0x2999b, [0x1ed3] = 0x4beb, [0x1ed4] = 0x29bc8,
  [0x1ed5] = 0x29bc4, [0x1ed6] = 0x29bcc, [0x1ed7] = 0x29bc7, [0x1ed8] = 0x29bc3,
  [0x1ed9] = 0x29bc6, [0x1eda] = 0x29bcb, [0x1edb] = 0x29bca, [0x1edc] = 0x29bc9,
  [0x1edd] = 0x29bcd, [0x1ede] = 0x29c53, [0x1edf] = 0x29d21, [0x1ee0] = 0x29d29,
  [0x1ee1] = 0x29d1d, [0x1ee2] = 0x29d24, [0x1ee3] = 0x29d28, [0x1ee4] = 0x9c43,
  [0x1ee5] = 0x29eb1, [0x1ee6] = 0x29eb2, [0x1ee7] = 0x29eb5, [0x1ee8] = 0x29ed6,
  [0x1ee9] = 0x29eaf, [0x1eea] = 0x29eca, [0x1eeb] = 0x29eb8, [0x1eec] = 0x2fa12,
  [0x1eed] = 0x2fa11, [0x1eee] = 0x2a145, [0x1eef] = 0x2a147, [0x1ef0] = 0x2a134,
  [0x1ef1] = 0x2a16e, [0x1ef2] = 0x2a142, [0x1ef3] = 0x2a131, [0x1ef4] = 0x2a12e,
  [0x1ef5] = 0x2a156, [0x1ef6] = 0x2a138, [0x1ef7] = 0x2a137, [0x1ef8] = 0x2a14b,
  [0x1ef9] = 0x2a12d, [0x1efa] = 0x2a133, [0x1efb] = 0x2a136, [0x1efc] = 0x9de0,
  [0x1efd] = 0x2a148, [0x1efe] = 0x2a2dc, [0x1eff] = 0x4d42, [0x1f00] = 0x2a366,
  [0x1f01] = 0x2a36d, [0x1f02] = 0x2a363, [0x1f03] = 0x2a364, [0x1f04] = 0x4d41,
  [0x1f05] = 0x2a367, [0x1f06] = 0x2a3d9, [0x1f07] = 0x2a40b, [0x1f08] = 0x2a47e,
  [0x1f09] = 0x2a48b, [0x1f0a] = 0x2a526, [0x1f0b] = 0x2a52c, [0x1f0c] = 0x2a529,
  [0x1f0d] = 0x2a56f, [0x1f0e] = 0x2a56b, [0x1f0f] = 0x2a56d, [0x1f10] = 0x2a56e,
  [0x1f11] = 0x2a61c, [0x1f12] = 0x2a611, [0x1f13] = 0x2a610, [0x1f14] = 0x2a624,
  [0x1f15] = 0x2a612, [0x1f16] = 0x2a615, [0x1f17] = 0x2a617, [0x1f18] = 0x4dac,
  [0x1f19] = 0x2046f, [0x1f1a] = 0x205ab, [0x1f1b] = 0x2082e, [0x1f1c] = 0x208c9,
  [0x1f1d] = 0x20b96, [0x1f1e] = 0x21185, [0x1f1f] = 0x216a0, [0x1f20] = 0x2192b,
  [0x1f21] = 0x2230d, [0x1f22] = 0x23036, [0x1f24] = 0x3b2f, [0x1f25] = 0x3b2e,
  [0x1f26] = 0x233b1, [0x1f27] = 0x2384a, [0x1f28] = 0x23837, [0x1f29] = 0x2383c,
  [0x1f2a] = 0x23838, [0x1f2b] = 0x2383a, [0x1f2c] = 0x2383e, [0x1f2d] = 0x23849,
  [0x1f2e] = 0x3c4d, [0x1f2f] = 0x3c7b, [0x1f30] = 0x23a80, [0x1f31] = 0x23a7f,
  [0x1f32] = 0x23ae3, [0x1f33] = 0x23c3f, [0x1f34] = 0x24152, [0x1f35] = 0x2414e,
  [0x1f36] = 0x2414a, [0x1f37] = 0x2414b, [0x1f38] = 0x244d5, [0x1f39] = 0x24ad8,
  [0x1f3a] = 0x24b20, [0x1f3b] = 0x24bb3, [0x1f3c] = 0x24bb1, [0x1f3d] = 0x24bb0,
  [0x1f3e] = 0x24f17, [0x1f3f] = 0x250d9, [0x1f40] = 0x25349, [0x1f41] = 0x25413,
  [0x1f42] = 0x255ed, [0x1f43] = 0x255eb, [0x1f44] = 0x255ee, [0x1f45] = 0x2590a,
  [0x1f46] = 0x2590b, [0x1f47] = 0x25a4a, [0x1f48] = 0x25dd5, [0x1f49] = 0x25dd7,
  [0x1f4a] = 0x25dde, [0x1f4b] = 0x25ddc, [0x1f4c] = 0x25dee, [0x1f4d] = 0x25f70,
  [0x1f4e] = 0x42b2, [0x1f4f] = 0x261d9, [0x1f50] = 0x263d7, [0x1f51] = 0x268cb,
  [0x1f52] = 0x268c7, [0x1f53] = 0x269fc, [0x1f54] = 0x2713b, [0x1f55] = 0x27139,
  [0x1f56] = 0x2714b, [0x1f57] = 0x27143, [0x1f58] = 0x27140, [0x1f59] = 0x27146,
  [0x1f5a] = 0x2754d, [0x1f5b] = 0x27543, [0x1f5c] = 0x27547, [0x1f5d] = 0x2754b,
  [0x1f5e] = 0x27555, [0x1f5f] = 0x27552, [0x1f60] = 0x27b5f, [0x1f61] = 0x4722,
  [0x1f62] = 0x27b7c, [0x1f63] = 0x27b7b, [0x1f64] = 0x27fa8, [0x1f65] = 0x27fa9,
  [0x1f66] = 0x281de, [0x1f67] = 0x281d7, [0x1f68] = 0x281dd, [0x1f69] = 0x281d6,
  [0x1f6a] = 0x281d8, [0x1f6b] = 0x2829b, [0x1f6c] = 0x283e0, [0x1f6d] = 0x283e8,
  [0x1f6e] = 0x283e6, [0x1f6f] = 0x288f8, [0x1f70] = 0x288fb, [0x1f71] = 0x28bbb,
  [0x1f72] = 0x9459, [0x1f73] = 0x28bb8, [0x1f74] = 0x4970, [0x1f75] = 0x28cc1,
  [0x1f76] = 0x28cc0, [0x1f77] = 0x291a4, [0x1f78] = 0x291ab, [0x1f79] = 0x2924d,
  [0x1f7a] = 0x29365, [0x1f7b] = 0x29367, [0x1f7c] = 0x2936a, [0x1f7d] = 0x29366,
  [0x1f7e] = 0x29369, [0x1f7f] = 0x29573, [0x1f80] = 0x29580, [0x1f81] = 0x2956f,
  [0x1f82] = 0x29571, [0x1f83] = 0x297e4, [0x1f84] = 0x297e6, [0x1f85] = 0x297e7,
  [0x1f86] = 0x299a0, [0x1f87] = 0x299a4, [0x1f88] = 0x29aa2, [0x1f89] = 0x29aa7,
  [0x1f8a] = 0x29aa4, [0x1f8b] = 0x29bde, [0x1f8c] = 0x29bdb, [0x1f8d] = 0x29c58,
  [0x1f8e] = 0x29c5c, [0x1f8f] = 0x29d2f, [0x1f90] = 0x29d2e, [0x1f91] = 0x29edd,
  [0x1f92] = 0x29ee4, [0x1f93] = 0x29ed8, [0x1f94] = 0x29ee7, [0x1f95] = 0x29eda,
  [0x1f96] = 0x29e75, [0x1f97] = 0x4c95, [0x1f98] = 0x2a179, [0x1f99] = 0x2a180,
  [0x1f9a] = 0x2a17f, [0x1f9b] = 0x2a17c, [0x1f9c] = 0x2a175, [0x1f9d] = 0x2a17b,
  [0x1f9e] = 0x2a182, [0x1f9f] = 0x4d00, [0x1fa0] = 0x2a189, [0x1fa1] = 0x2a174,
  [0x1fa2] = 0x2a17d, [0x1fa3] = 0x2a17a, [0x1fa4] = 0x2a186, [0x1fa5] = 0x2a1a8,
  [0x1fa6] = 0x2a172, [0x1fa7] = 0x4d02, [0x1fa8] = 0x2a18b, [0x1fa9] = 0x2a191,
  [0x1faa] = 0x2a1b3, [0x1fab] = 0x2a181, [0x1fac] = 0x2a282, [0x1fad] = 0x2a2e1,
  [0x1fae] = 0x2a2e3, [0x1faf] = 0x2a2e2, [0x1fb0] = 0x2a376, [0x1fb1] = 0x2a374,
  [0x1fb2] = 0x2a372, [0x1fb3] = 0x2a375, [0x1fb4] = 0x2a40e, [0x1fb5] = 0x2a40d,
  [0x1fb6] = 0x2a494, [0x1fb7] = 0x2a492, [0x1fb8] = 0x2a493, [0x1fb9] = 0x2a491,
  [0x1fba] = 0x2a48f, [0x1fbb] = 0x2a495, [0x1fbc] = 0x2a4d0, [0x1fbd] = 0x2a4f7,
  [0x1fbe] = 0x2a576, [0x1fbf] = 0x2a5af, [0x1fc0] = 0x4d9f, [0x1fc1] = 0x2a626,
  [0x1fc2] = 0x2a625, [0x1fc3] = 0x2a62d, [0x1fc4] = 0x2a6a0, [0x1fc5] = 0x2a6c3,
  [0x1fc6] = 0x204d7, [0x1fc7] = 0x20acc, [0x1fc8] = 0x21179, [0x1fc9] = 0x2117a,
  [0x1fca] = 0x21515, [0x1fcb] = 0x21fdb, [0x1fcc] = 0x22185, [0x1fcd] = 0x22970,
  [0x1fce] = 0x2297b, [0x1fcf] = 0x22ea5, [0x1fd0] = 0x22e9f, [0x1fd1] = 0x23317,
  [0x1fd2] = 0x233b2, [0x1fd3] = 0x23839, [0x1fd4] = 0x23840, [0x1fd5] = 0x2384e,
  [0x1fd6] = 0x2393e, [0x1fd7] = 0x23ae8, [0x1fd8] = 0x23ae7, [0x1fd9] = 0x24166,
  [0x1fda] = 0x24161, [0x1fdb] = 0x244da, [0x1fdc] = 0x24714, [0x1fdd] = 0x248df,
  [0x1fde] = 0x24f18, [0x1fdf] = 0x25039, [0x1fe0] = 0x2503a, [0x1fe1] = 0x25730,
  [0x1fe2] = 0x25732, [0x1fe3] = 0x2590e, [0x1fe4] = 0x25a4c, [0x1fe5] = 0x25de8,
  [0x1fe6] = 0x25df1, [0x1fe7] = 0x25deb, [0x1fe8] = 0x25dec, [0x1fe9] = 0x25f74,
  [0x1fea] = 0x25f73, [0x1feb] = 0x261f1, [0x1fec] = 0x26342, [0x1fed] = 0x2633e,
  [0x1fee] = 0x26341, [0x1fef] = 0x2652c, [0x1ff0] = 0x268d2, [0x1ff1] = 0x269bb,
  [0x1ff2] = 0x27158, [0x1ff3] = 0x27157, [0x1ff4] = 0x27156, [0x1ff5] = 0x2715a,
  [0x1ff6] = 0x457d, [0x1ff7] = 0x27561, [0x1ff8] = 0x27568, [0x1ff9] = 0x27565,
  [0x1ffa] = 0x2755f, [0x1ffb] = 0x27564, [0x1ffc] = 0x2755e, [0x1ffd] = 0x2755b,
  [0x1ffe] = 0x27567, [0x1fff] = 0x277c3, [0x2000] = 0x278a2, [0x2001] = 0x27953,
  [0x2002] = 0x27b7d, [0x2003] = 0x27c20, [0x2004] = 0x27fae, [0x2005] = 0x27faf,
  [0x2006] = 0x27fb0, [0x2007] = 0x281e7, [0x2008] = 0x281e6, [0x2009] = 0x281e9,
  [0x200a] = 0x283f1, [0x200b] = 0x283eb, [0x200c] = 0x283ea, [0x200d] = 0x283e9,
  [0x200e] = 0x28663, [0x200f] = 0x28902, [0x2010] = 0x288fe, [0x2011] = 0x28bde,
  [0x2012] = 0x28cc2, [0x2013] = 0x28de6, [0x2014] = 0x28f5d, [0x2015] = 0x291ad,
  [0x2016] = 0x29375, [0x2017] = 0x29372, [0x2018] = 0x29377, [0x2019] = 0x4af6,
  [0x201a] = 0x29651, [0x201b] = 0x297e8, [0x201c] = 0x297ed, [0x201d] = 0x297ee,
  [0x201e] = 0x299b9, [0x201f] = 0x299a1, [0x2020] = 0x299b6, [0x2021] = 0x29aae,
  [0x2022] = 0x29be8, [0x2023] = 0x4c12, [0x2024] = 0x29c1e, [0x2025] = 0x29d31,
  [0x2026] = 0x29d32, [0x2027] = 0x4c91, [0x2028] = 0x29f0e, [0x2029] = 0x29f12,
  [0x202a] = 0x29f09, [0x202b] = 0x29f05, [0x202c] = 0x4c90, [0x202d] = 0x29f03,
  [0x202e] = 0x29f1f, [0x202f] = 0x29f0d, [0x2030] = 0x29f0c, [0x2031] = 0x29f04,
  [0x2032] = 0x29f0a, [0x2033] = 0x2a1c2, [0x2034] = 0x2a1bf, [0x2035] = 0x2a1c9,
  [0x2036] = 0x2a1b2, [0x2037] = 0x2a1c1, [0x2038] = 0x2a1af, [0x2039] = 0x2a1b4,
  [0x203a] = 0x2a1b0, [0x203b] = 0x2a1b6, [0x203c] = 0x2a1b7, [0x203d] = 0x2a1bb,
  [0x203e] = 0x2a1b1, [0x203f] = 0x2a2f0, [0x2040] = 0x2a378, [0x2041] = 0x2a37a,
  [0x2042] = 0x2a379, [0x2043] = 0x2a3e4, [0x2044] = 0x2a3e6, [0x2045] = 0x2a49f,
  [0x2046] = 0x2a49d, [0x2047] = 0x2a498, [0x2048] = 0x2a499, [0x2049] = 0x2a4f9,
  [0x204a] = 0x2a530, [0x204b] = 0x2a582, [0x204c] = 0x2a581, [0x204d] = 0x2a5b3,
  [0x204e] = 0x2a57f, [0x204f] = 0x2a63a, [0x2050] = 0x2a63e, [0x2051] = 0x2a648,
  [0x2052] = 0x21186, [0x2053] = 0x21936, [0x2054] = 0x21b13, [0x2055] = 0x21c22,
  [0x2056] = 0x21fdd, [0x2057] = 0x225a7, [0x2058] = 0x2297d, [0x2059] = 0x3a79,
  [0x205a] = 0x2331c, [0x205b] = 0x23865, [0x205c] = 0x23860, [0x205d] = 0x23867,
  [0x205e] = 0x3c4e, [0x205f] = 0x23b1a, [0x2060] = 0x24ae3, [0x2061] = 0x25734,
  [0x2062] = 0x25918, [0x2063] = 0x25a52, [0x2064] = 0x25dfc, [0x2065] = 0x25f79,
  [0x2066] = 0x25f78, [0x2067] = 0x25f76, [0x2068] = 0x261fa, [0x2069] = 0x261f8,
  [0x206a] = 0x2652d, [0x206b] = 0x268d6, [0x206c] = 0x269bd, [0x206d] = 0x269bf,
  [0x206e] = 0x269be, [0x206f] = 0x27144, [0x2070] = 0x27170, [0x2071] = 0x27162,
  [0x2072] = 0x2716e, [0x2073] = 0x2756e, [0x2074] = 0x2757c, [0x2075] = 0x27574,
  [0x2076] = 0x27578, [0x2077] = 0x27570, [0x2078] = 0x27579, [0x2079] = 0x27571,
  [0x207a] = 0x277cc, [0x207b] = 0x278a7, [0x207c] = 0x278a6, [0x207d] = 0x27b93,
  [0x207e] = 0x27c21, [0x207f] = 0x27c9c, [0x2080] = 0x2829d, [0x2081] = 0x283f4,
  [0x2082] = 0x283f3, [0x2083] = 0x28df0, [0x2084] = 0x291b8, [0x2085] = 0x291b6,
  [0x2086] = 0x291bd, [0x2087] = 0x29373, [0x2088] = 0x29382, [0x2089] = 0x29587,
  [0x208a] = 0x29656, [0x208b] = 0x29659, [0x208c] = 0x297f6, [0x208d] = 0x299c9,
  [0x208e] = 0x299c5, [0x208f] = 0x299c7, [0x2090] = 0x299ca, [0x2091] = 0x299c2,
  [0x2092] = 0x299c4, [0x2093] = 0x29bf2, [0x2094] = 0x29bf0, [0x2095] = 0x29d3b,
  [0x2096] = 0x29d3a, [0x2097] = 0x29f26, [0x2098] = 0x29f28, [0x2099] = 0x29f34,
  [0x209a] = 0x29f2d, [0x209b] = 0x2a1d7, [0x209c] = 0x2a1d2, [0x209d] = 0x2a1d6,
  [0x209e] = 0x2a1dc, [0x209f] = 0x2a1d3, [0x20a0] = 0x2a1d1, [0x20a1] = 0x2a286,
  [0x20a2] = 0x4d1e, [0x20a3] = 0x2a2ef, [0x20a4] = 0x2a2ee, [0x20a5] = 0x2a3e8,
  [0x20a6] = 0x2a4ac, [0x20a7] = 0x2a4a9, [0x20a8] = 0x2a4aa, [0x20a9] = 0x2a4ab,
  [0x20aa] = 0x2a4fb, [0x20ab] = 0x2a533, [0x20ac] = 0x2a588, [0x20ad] = 0x2a5b6,
  [0x20ae] = 0x2a5b7, [0x20af] = 0x2a5d0, [0x20b0] = 0x2a5cf, [0x20b1] = 0x2a64f,
  [0x20b2] = 0x2a659, [0x20b3] = 0x2a64c, [0x20b4] = 0x21b18, [0x20b5] = 0x21fe0,
  [0x20b6] = 0x23869, [0x20b7] = 0x24173, [0x20b8] = 0x24172, [0x20b9] = 0x24ae8,
  [0x20ba] = 0x255fb, [0x20bb] = 0x2591c, [0x20bc] = 0x25e09, [0x20bd] = 0x25e0a,
  [0x20be] = 0x25e08, [0x20bf] = 0x25f7d, [0x20c0] = 0x25f7f, [0x20c1] = 0x26267,
  [0x20c2] = 0x2717a, [0x20c3] = 0x2717b, [0x20c4] = 0x277d1, [0x20c5] = 0x2f9cb,
  [0x20c6] = 0x27b9d, [0x20c7] = 0x27c9d, [0x20c8] = 0x281fc, [0x20c9] = 0x281fb,
  [0x20ca] = 0x281fd, [0x20cb] = 0x283f5, [0x20cc] = 0x283f6, [0x20cd] = 0x28c0f,
  [0x20ce] = 0x28c1d, [0x20cf] = 0x291c7, [0x20d0] = 0x291c1, [0x20d1] = 0x291bf,
  [0x20d2] = 0x29254, [0x20d3] = 0x2937f, [0x20d4] = 0x293f5, [0x20d5] = 0x2958a,
  [0x20d6] = 0x2958c, [0x20d7] = 0x2965c, [0x20d8] = 0x29865, [0x20d9] = 0x299cf,
  [0x20da] = 0x299d0, [0x20db] = 0x29ab8, [0x20dc] = 0x29bfa, [0x20dd] = 0x29c66,
  [0x20de] = 0x29f40, [0x20df] = 0x29f41, [0x20e0] = 0x9c72, [0x20e1] = 0x2a1f7,
  [0x20e2] = 0x2a1f3, [0x20e3] = 0x2a1ef, [0x20e4] = 0x2a1f4, [0x20e5] = 0x2a1ed,
  [0x20e6] = 0x2a1f2, [0x20e7] = 0x2a1f1, [0x20e8] = 0x2a1f9, [0x20e9] = 0x2a4b4,
  [0x20ea] = 0x2a4fc, [0x20eb] = 0x2a5bc, [0x20ec] = 0x2a65e, [0x20ed] = 0x2a65b,
  [0x20ee] = 0x2a65f, [0x20ef] = 0x2a65d, [0x20f0] = 0x2a6cf, [0x20f1] = 0x2a6ce,
  [0x20f2] = 0x2193a, [0x20f3] = 0x21fe1, [0x20f4] = 0x22315, [0x20f5] = 0x22314,
  [0x20f6] = 0x24178, [0x20f7] = 0x24179, [0x20f8] = 0x244e6, [0x20f9] = 0x24bb8,
  [0x20fa] = 0x24f22, [0x20fb] = 0x255fe, [0x20fc] = 0x268db, [0x20fd] = 0x26a06,
  [0x20fe] = 0x26ac8, [0x20ff] = 0x27189, [0x2100] = 0x27586, [0x2101] = 0x278ad,
  [0x2102] = 0x27b99, [0x2103] = 0x27bd8, [0x2104] = 0x27c23, [0x2105] = 0x27d23,
  [0x2106] = 0x27fb5, [0x2107] = 0x283f9, [0x2108] = 0x28907, [0x2109] = 0x28c1e,
  [0x210a] = 0x28dfb, [0x210b] = 0x28f72, [0x210c] = 0x28f73, [0x210d] = 0x29387,
  [0x210e] = 0x29449, [0x210f] = 0x2965d, [0x2110] = 0x297f8, [0x2111] = 0x29c03,
  [0x2112] = 0x29f4e, [0x2113] = 0x29f4d, [0x2114] = 0x29f50, [0x2115] = 0x29f55,
  [0x2116] = 0x2a209, [0x2117] = 0x2a207, [0x2118] = 0x2a20c, [0x2119] = 0x2a203,
  [0x211a] = 0x2a206, [0x211b] = 0x2a20b, [0x211c] = 0x2a20a, [0x211d] = 0x2a289,
  [0x211e] = 0x2a4bb, [0x211f] = 0x2a4ff, [0x2120] = 0x2a536, [0x2121] = 0x2a58b,
  [0x2122] = 0x2a666, [0x2123] = 0x2a669, [0x2124] = 0x2a667, [0x2125] = 0x2a6c2,
  [0x2126] = 0x2a6d2, [0x2127] = 0x21fe2, [0x2128] = 0x3828, [0x2129] = 0x22988,
  [0x212a] = 0x2335b, [0x212b] = 0x23876, [0x212c] = 0x2417d, [0x212d] = 0x245a6,
  [0x212e] = 0x2535a, [0x212f] = 0x25e17, [0x2130] = 0x268dc, [0x2131] = 0x2718f,
  [0x2132] = 0x27594, [0x2133] = 0x27595, [0x2134] = 0x27fb6, [0x2135] = 0x28204,
  [0x2136] = 0x287ee, [0x2137] = 0x2938a, [0x2138] = 0x293f9, [0x2139] = 0x29c07,
  [0x213a] = 0x29c1f, [0x213b] = 0x29c2a, [0x213c] = 0x29d45, [0x213d] = 0x29f5c,
  [0x213e] = 0x29f5b, [0x213f] = 0x29f61, [0x2140] = 0x29f5d, [0x2141] = 0x2a218,
  [0x2142] = 0x2a21f, [0x2143] = 0x2a383, [0x2144] = 0x2a416, [0x2145] = 0x2a414,
  [0x2146] = 0x2a4bf, [0x2147] = 0x2a4c0, [0x2148] = 0x2a673, [0x2149] = 0x2a6c0,
  [0x214a] = 0x21517, [0x214b] = 0x22989, [0x214c] = 0x2591e, [0x214d] = 0x2591f,
  [0x214e] = 0x25a54, [0x214f] = 0x25e18, [0x2150] = 0x26205, [0x2151] = 0x27195,
  [0x2152] = 0x27598, [0x2153] = 0x2759b, [0x2154] = 0x27959, [0x2155] = 0x28208,
  [0x2156] = 0x28401, [0x2157] = 0x291ce, [0x2158] = 0x293fa, [0x2159] = 0x29abe,
  [0x215a] = 0x29d47, [0x215b] = 0x29f6a, [0x215c] = 0x29f69, [0x215d] = 0x29f68,
  [0x215e] = 0x29f67, [0x215f] = 0x2a225, [0x2160] = 0x2a228, [0x2161] = 0x2a4c4,
  [0x2162] = 0x2a537, [0x2163] = 0x2a58d, [0x2164] = 0x2a58c, [0x2165] = 0x4d91,
  [0x2166] = 0x2f920, [0x2167] = 0x25e22, [0x2168] = 0x25f82, [0x2169] = 0x277d9,
  [0x216a] = 0x28c38, [0x216b] = 0x293fb, [0x216c] = 0x299e1, [0x216d] = 0x29c72,
  [0x216e] = 0x29d48, [0x216f] = 0x29f6c, [0x2170] = 0x29f70, [0x2171] = 0x2a231,
  [0x2172] = 0x2a230, [0x2173] = 0x2a679, [0x2174] = 0x25e23, [0x2175] = 0x268f8,
  [0x2176] = 0x29c74, [0x2177] = 0x29f73, [0x2178] = 0x2a234, [0x2179] = 0x2a387,
  [0x217a] = 0x2a3b2, [0x217b] = 0x2a5c2, [0x217c] = 0x2a67d, [0x217d] = 0x21fe5,
  [0x217e] = 0x2938c, [0x217f] = 0x29f77, [0x2180] = 0x29f75, [0x2181] = 0x2a238,
  [0x2182] = 0x2a23a, [0x2183] = 0x2a683, [0x2184] = 0x2a681, [0x2185] = 0x24180,
  [0x2186] = 0x27199, [0x2187] = 0x4587, [0x2188] = 0x2944a, [0x2189] = 0x29d4a,
  [0x218a] = 0x2a23c, [0x218b] = 0x2a6c8, [0x218c] = 0x25604, [0x218d] = 0x278af,
  [0x218e] = 0x2a689, [0x218f] = 0x2a2fa, [0x2190] = 0x29661, [0x2191] = 0x2a23f,
  [0x2192] = 0x27193, [0x2193] = 0x2a4c9, [0x2194] = 0x277de, [0x2195] = 0x29391,
  [0x2196] = 0x2a68e, [0x2197] = 0x2a68d, [0x2198] = 0x27198, [0x2199] = 0x269c5,
  [0x219a] = 0x2a6a5,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x6' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x62121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l6_to_ucs4_tab[] =
{
  [0x0000] = 0x2f802, [0x0001] = 0x20062, [0x0002] = 0x3405, [0x0003] = 0x20088,
  [0x0004] = 0x200d0, [0x0005] = 0x200cf, [0x0006] = 0x2011e, [0x0007] = 0x2011f,
  [0x0008] = 0x20120, [0x0009] = 0x2090e, [0x000a] = 0x2092c, [0x000b] = 0x20061,
  [0x000c] = 0x20004, [0x000d] = 0x20005, [0x000e] = 0x20003, [0x000f] = 0x3404,
  [0x0010] = 0x2f801, [0x0011] = 0x2007d, [0x0012] = 0x2008d, [0x0013] = 0x201a3,
  [0x0014] = 0x201a4, [0x0015] = 0x204db, [0x0016] = 0x20500, [0x0017] = 0x2f81e,
  [0x0018] = 0x206a3, [0x0019] = 0x20832, [0x001a] = 0x353f, [0x001b] = 0x20ad4,
  [0x001c] = 0x20ad5, [0x001d] = 0x21c23, [0x001e] = 0x2232c, [0x001f] = 0x2237a,
  [0x0020] = 0x20503, [0x0021] = 0x20007, [0x0022] = 0x20008, [0x0023] = 0x2008f,
  [0x0024] = 0x2009c, [0x0025] = 0x200d5, [0x0026] = 0x20112, [0x0027] = 0x20113,
  [0x0028] = 0x20114, [0x0029] = 0x20121, [0x002a] = 0x201a8, [0x002b] = 0x201a6,
  [0x002c] = 0x20476, [0x002d] = 0x20501, [0x002e] = 0x2053d, [0x002f] = 0x2f814,
  [0x0030] = 0x2053e, [0x0031] = 0x20629, [0x0032] = 0x20677, [0x0033] = 0x20678,
  [0x0034] = 0x206a6, [0x0035] = 0x206a9, [0x0036] = 0x206ac, [0x0037] = 0x20834,
  [0x0038] = 0x20504, [0x0039] = 0x2097b, [0x003a] = 0x20a10, [0x003b] = 0x2062f,
  [0x003c] = 0x20ad7, [0x003d] = 0x20b1c, [0x003e] = 0x20b1f, [0x003f] = 0x20b1a,
  [0x0040] = 0x2f836, [0x0041] = 0x20b1e, [0x0042] = 0x20b9e, [0x0043] = 0x20b9a,
  [0x0044] = 0x211a0, [0x0045] = 0x211a1, [0x0046] = 0x215d2, [0x0047] = 0x215d3,
  [0x0048] = 0x219b9, [0x0049] = 0x21b1d, [0x004a] = 0x21b55, [0x004b] = 0x21cff,
  [0x004c] = 0x21fe7, [0x004d] = 0x382a, [0x004e] = 0x22011, [0x004f] = 0x22013,
  [0x0050] = 0x22012, [0x0051] = 0x221af, [0x0052] = 0x221b1, [0x0053] = 0x2f890,
  [0x0054] = 0x2232d, [0x0055] = 0x2237b, [0x0056] = 0x2239d, [0x0057] = 0x2239a,
  [0x0058] = 0x38a7, [0x0059] = 0x2239f, [0x005a] = 0x223a0, [0x005b] = 0x2239c,
  [0x005c] = 0x224bd, [0x005d] = 0x38fa, [0x005e] = 0x225aa, [0x005f] = 0x2306c,
  [0x0060] = 0x23091, [0x0061] = 0x23c74, [0x0062] = 0x2456a, [0x0063] = 0x2f929,
  [0x0064] = 0x25605, [0x0065] = 0x2626b, [0x0066] = 0x2626a, [0x0067] = 0x200e0,
  [0x0068] = 0x20012, [0x0069] = 0x3400, [0x006a] = 0x2007f, [0x006b] = 0x200a0,
  [0x006c] = 0x20099, [0x006d] = 0x20095, [0x006e] = 0x2009e, [0x006f] = 0x200df,
  [0x0070] = 0x200de, [0x0071] = 0x201af, [0x0072] = 0x201ad, [0x0073] = 0x201b0,
  [0x0074] = 0x204df, [0x0075] = 0x20543, [0x0076] = 0x20541, [0x0077] = 0x20548,
  [0x0078] = 0x20575, [0x0079] = 0x2f81a, [0x007a] = 0x20632, [0x007b] = 0x2067b,
  [0x007c] = 0x20679, [0x007d] = 0x206b3, [0x007e] = 0x34db, [0x007f] = 0x20ba7,
  [0x0080] = 0x206b2, [0x0081] = 0x206b0, [0x0082] = 0x206b4, [0x0083] = 0x2f829,
  [0x0084] = 0x208cd, [0x0085] = 0x208ce, [0x0086] = 0x2f82b, [0x0087] = 0x20910,
  [0x0088] = 0x20980, [0x0089] = 0x209d3, [0x008a] = 0x20a30, [0x008b] = 0x20098,
  [0x008c] = 0x20a2f, [0x008d] = 0x20a31, [0x008e] = 0x20ada, [0x008f] = 0x20ad9,
  [0x0090] = 0x20b27, [0x0091] = 0x20b28, [0x0092] = 0x20b21, [0x0093] = 0x20b25,
  [0x0094] = 0x20ba6, [0x0095] = 0x20ba2, [0x0096] = 0x20ba1, [0x0097] = 0x20ba4,
  [0x0098] = 0x2f839, [0x0099] = 0x20baa, [0x009a] = 0x20ba5, [0x009b] = 0x20ba3,
  [0x009c] = 0x20064, [0x009d] = 0x2123f, [0x009e] = 0x21244, [0x009f] = 0x2151b,
  [0x00a0] = 0x21553, [0x00a1] = 0x21584, [0x00a2] = 0x21586, [0x00a3] = 0x215d5,
  [0x00a4] = 0x2f860, [0x00a5] = 0x2193e, [0x00a6] = 0x21940, [0x00a7] = 0x21bca,
  [0x00a8] = 0x21bc9, [0x00a9] = 0x21c26, [0x00aa] = 0x21d00, [0x00ab] = 0x21d31,
  [0x00ac] = 0x21d3c, [0x00ad] = 0x21d30, [0x00ae] = 0x21d33, [0x00af] = 0x21d32,
  [0x00b0] = 0x21d38, [0x00b1] = 0x22034, [0x00b2] = 0x22053, [0x00b3] = 0x22054,
  [0x00b4] = 0x221b4, [0x00b5] = 0x221b2, [0x00b6] = 0x2232f, [0x00b7] = 0x223a5,
  [0x00b8] = 0x223a7, [0x00b9] = 0x223a4, [0x00ba] = 0x223a8, [0x00bb] = 0x223a9,
  [0x00bc] = 0x223a2, [0x00bd] = 0x223a3, [0x00be] = 0x223a6, [0x00bf] = 0x22481,
  [0x00c0] = 0x22480, [0x00c1] = 0x225ad, [0x00c2] = 0x22a24, [0x00c3] = 0x22a65,
  [0x00c4] = 0x22a68, [0x00c5] = 0x22a69, [0x00c6] = 0x2313c, [0x00c7] = 0x2313b,
  [0x00c8] = 0x2335d, [0x00c9] = 0x23944, [0x00ca] = 0x24182, [0x00cb] = 0x244f1,
  [0x00cc] = 0x2456b, [0x00cd] = 0x2471b, [0x00ce] = 0x248eb, [0x00cf] = 0x24bba,
  [0x00d0] = 0x24bd3, [0x00d1] = 0x24d13, [0x00d2] = 0x25922, [0x00d3] = 0x26af4,
  [0x00d4] = 0x2820f, [0x00d5] = 0x28210, [0x00d6] = 0x20507, [0x00d7] = 0x29c0b,
  [0x00d8] = 0x29c0a, [0x00d9] = 0x20019, [0x00da] = 0x2001a, [0x00db] = 0x20018,
  [0x00dc] = 0x20066, [0x00dd] = 0x20082, [0x00de] = 0x200a8, [0x00df] = 0x200e5,
  [0x00e0] = 0x200e7, [0x00e1] = 0x200e4, [0x00e2] = 0x20115, [0x00e3] = 0x20123,
  [0x00e4] = 0x20124, [0x00e5] = 0x20143, [0x00e6] = 0x201c3, [0x00e7] = 0x201bc,
  [0x00e8] = 0x201bb, [0x00e9] = 0x3438, [0x00ea] = 0x201b7, [0x00eb] = 0x201b9,
  [0x00ec] = 0x201cf, [0x00ed] = 0x2f80d, [0x00ee] = 0x2047a, [0x00ef] = 0x2047b,
  [0x00f0] = 0x204e1, [0x00f1] = 0x204e0, [0x00f2] = 0x20509, [0x00f3] = 0x2054d,
  [0x00f4] = 0x2f816, [0x00f5] = 0x2054b, [0x00f6] = 0x20579, [0x00f7] = 0x205b1,
  [0x00f8] = 0x20636, [0x00f9] = 0x20635, [0x00fa] = 0x20637, [0x00fb] = 0x20638,
  [0x00fc] = 0x2067f, [0x00fd] = 0x20680, [0x00fe] = 0x20681, [0x00ff] = 0x26953,
  [0x0100] = 0x2067e, [0x0101] = 0x206cd, [0x0102] = 0x206bf, [0x0103] = 0x206be,
  [0x0104] = 0x206c2, [0x0105] = 0x206b8, [0x0106] = 0x206c1, [0x0107] = 0x6530,
  [0x0108] = 0x20931, [0x0109] = 0x2092f, [0x010a] = 0x2096f, [0x010b] = 0x20982,
  [0x010c] = 0x353a, [0x010d] = 0x20987, [0x010e] = 0x20986, [0x010f] = 0x2098d,
  [0x0110] = 0x20994, [0x0111] = 0x209d7, [0x0112] = 0x209d4, [0x0113] = 0x209d8,
  [0x0114] = 0x20a16, [0x0115] = 0x20a14, [0x0116] = 0x20a2e, [0x0117] = 0x20a36,
  [0x0118] = 0x20a37, [0x0119] = 0x21fee, [0x011a] = 0x20ae0, [0x011b] = 0x20ae5,
  [0x011c] = 0x20adf, [0x011d] = 0x20b2b, [0x011e] = 0x20b29, [0x011f] = 0x20b2d,
  [0x0120] = 0x20b2f, [0x0121] = 0x20bb7, [0x0122] = 0x20bc9, [0x0123] = 0x20bc1,
  [0x0124] = 0x20bca, [0x0125] = 0x20bb2, [0x0126] = 0x20bac, [0x0127] = 0x20bae,
  [0x0128] = 0x211b2, [0x0129] = 0x211a8, [0x012a] = 0x211b0, [0x012b] = 0x211a7,
  [0x012c] = 0x2f851, [0x012d] = 0x21252, [0x012e] = 0x2124a, [0x012f] = 0x21556,
  [0x0130] = 0x21555, [0x0131] = 0x2155f, [0x0132] = 0x2155e, [0x0133] = 0x21588,
  [0x0134] = 0x2f85d, [0x0135] = 0x215dd, [0x0136] = 0x215e2, [0x0137] = 0x215e1,
  [0x0138] = 0x215df, [0x0139] = 0x215e0, [0x013a] = 0x216af, [0x013b] = 0x216b1,
  [0x013c] = 0x216ba, [0x013d] = 0x21945, [0x013e] = 0x21943, [0x013f] = 0x219bd,
  [0x0140] = 0x219c8, [0x0141] = 0x2f86c, [0x0142] = 0x219be, [0x0143] = 0x219bf,
  [0x0144] = 0x2f874, [0x0145] = 0x21b59, [0x0146] = 0x21c29, [0x0147] = 0x21c2c,
  [0x0148] = 0x21c2f, [0x0149] = 0x21c2d, [0x014a] = 0x21c2e, [0x014b] = 0x21d04,
  [0x014c] = 0x21d02, [0x014d] = 0x21d01, [0x014e] = 0x21d40, [0x014f] = 0x21d4a,
  [0x0150] = 0x21d3f, [0x0151] = 0x21d4f, [0x0152] = 0x21d41, [0x0153] = 0x21d4e,
  [0x0154] = 0x21feb, [0x0155] = 0x21fec, [0x0156] = 0x200ac, [0x0157] = 0x2205b,
  [0x0158] = 0x22063, [0x0159] = 0x2205a, [0x015a] = 0x22059, [0x015b] = 0x221dc,
  [0x015c] = 0x221db, [0x015d] = 0x221d9, [0x015e] = 0x221e0, [0x015f] = 0x221dd,
  [0x0160] = 0x20020, [0x0161] = 0x223b0, [0x0162] = 0x223b1, [0x0163] = 0x223b2,
  [0x0164] = 0x223b6, [0x0165] = 0x22451, [0x0166] = 0x22452, [0x0167] = 0x22450,
  [0x0168] = 0x22484, [0x0169] = 0x38c9, [0x016a] = 0x22483, [0x016b] = 0x225c3,
  [0x016c] = 0x225b8, [0x016d] = 0x225b9, [0x016e] = 0x225b2, [0x016f] = 0x22990,
  [0x0170] = 0x22a26, [0x0171] = 0x22a29, [0x0172] = 0x22a6e, [0x0173] = 0x23140,
  [0x0174] = 0x2313e, [0x0175] = 0x23141, [0x0176] = 0x2335e, [0x0177] = 0x233bb,
  [0x0178] = 0x233bc, [0x0179] = 0x233b9, [0x017a] = 0x2f8ef, [0x017b] = 0x23880,
  [0x017c] = 0x3c4f, [0x017d] = 0x23945, [0x017e] = 0x239ba, [0x017f] = 0x239bd,
  [0x0180] = 0x23c56, [0x0181] = 0x23c55, [0x0182] = 0x23c75, [0x0183] = 0x23c77,
  [0x0184] = 0x23c81, [0x0185] = 0x2f8fa, [0x0186] = 0x23c83, [0x0187] = 0x23c8b,
  [0x0188] = 0x23c84, [0x0189] = 0x24186, [0x018a] = 0x24183, [0x018b] = 0x2418b,
  [0x018c] = 0x2456d, [0x018d] = 0x24616, [0x018e] = 0x24728, [0x018f] = 0x248ea,
  [0x0190] = 0x24bd4, [0x0191] = 0x24c03, [0x0192] = 0x24d15, [0x0193] = 0x24f3d,
  [0x0194] = 0x24f3e, [0x0195] = 0x25416, [0x0196] = 0x2f94c, [0x0197] = 0x25606,
  [0x0198] = 0x2574d, [0x0199] = 0x26222, [0x019a] = 0x26273, [0x019b] = 0x2626d,
  [0x019c] = 0x2626e, [0x019d] = 0x264b1, [0x019e] = 0x2054e, [0x019f] = 0x2662d,
  [0x01a0] = 0x43cd, [0x01a1] = 0x268f9, [0x01a2] = 0x26952, [0x01a3] = 0x26b00,
  [0x01a4] = 0x26afc, [0x01a5] = 0x26afa, [0x01a6] = 0x4492, [0x01a7] = 0x26af7,
  [0x01a8] = 0x26af8, [0x01a9] = 0x2f990, [0x01aa] = 0x2f9c4, [0x01ab] = 0x27c27,
  [0x01ac] = 0x28211, [0x01ad] = 0x200ae, [0x01ae] = 0x28450, [0x01af] = 0x2844f,
  [0x01b0] = 0x2844d, [0x01b1] = 0x28671, [0x01b2] = 0x2f9e2, [0x01b3] = 0x28672,
  [0x01b4] = 0x28e18, [0x01b5] = 0x20a44, [0x01b6] = 0x20021, [0x01b7] = 0x20067,
  [0x01b8] = 0x200af, [0x01b9] = 0x200b0, [0x01ba] = 0x20029, [0x01bb] = 0x20117,
  [0x01bc] = 0x20122, [0x01bd] = 0x2f803, [0x01be] = 0x20025, [0x01bf] = 0x20128,
  [0x01c0] = 0x201ff, [0x01c1] = 0x201d9, [0x01c2] = 0x201db, [0x01c3] = 0x201de,
  [0x01c4] = 0x201f2, [0x01c5] = 0x201fe, [0x01c6] = 0x201f4, [0x01c7] = 0x201dd,
  [0x01c8] = 0x3445, [0x01c9] = 0x2f804, [0x01ca] = 0x20201, [0x01cb] = 0x201dc,
  [0x01cc] = 0x2047d, [0x01cd] = 0x20481, [0x01ce] = 0x2047e, [0x01cf] = 0x2047f,
  [0x01d0] = 0x204e5, [0x01d1] = 0x2050f, [0x01d2] = 0x34b5, [0x01d3] = 0x2050a,
  [0x01d4] = 0x34b6, [0x01d5] = 0x2050b, [0x01d6] = 0x20551, [0x01d7] = 0x20552,
  [0x01d8] = 0x20550, [0x01d9] = 0x205b8, [0x01da] = 0x2063d, [0x01db] = 0x2063e,
  [0x01dc] = 0x2063c, [0x01dd] = 0x20687, [0x01de] = 0x206d7, [0x01df] = 0x206db,
  [0x01e0] = 0x206cf, [0x01e1] = 0x206e0, [0x01e2] = 0x2084c, [0x01e3] = 0x20842,
  [0x01e4] = 0x208d4, [0x01e5] = 0x20911, [0x01e6] = 0x20914, [0x01e7] = 0x20915,
  [0x01e8] = 0x20912, [0x01e9] = 0x20933, [0x01ea] = 0x20935, [0x01eb] = 0x3537,
  [0x01ec] = 0x20971, [0x01ed] = 0x3530, [0x01ee] = 0x20991, [0x01ef] = 0x20993,
  [0x01f0] = 0x20995, [0x01f1] = 0x209dc, [0x01f2] = 0x20a42, [0x01f3] = 0x20a40,
  [0x01f4] = 0x253a6, [0x01f5] = 0x20ae6, [0x01f6] = 0x2128f, [0x01f7] = 0x20b32,
  [0x01f8] = 0x2f837, [0x01f9] = 0x20b31, [0x01fa] = 0x20bf5, [0x01fb] = 0x20bcd,
  [0x01fc] = 0x20bde, [0x01fd] = 0x20bd5, [0x01fe] = 0x20bda, [0x01ff] = 0x20bee,
  [0x0200] = 0x20bec, [0x0201] = 0x20bcf, [0x0202] = 0x20bce, [0x0203] = 0x2f83d,
  [0x0204] = 0x20bcc, [0x0205] = 0x20bf3, [0x0206] = 0x211c1, [0x0207] = 0x211c4,
  [0x0208] = 0x211c6, [0x0209] = 0x211b4, [0x020a] = 0x211c3, [0x020b] = 0x211b8,
  [0x020c] = 0x211bb, [0x020d] = 0x211b7, [0x020e] = 0x211bc, [0x020f] = 0x211bf,
  [0x0210] = 0x211c7, [0x0211] = 0x211be, [0x0212] = 0x21260, [0x0213] = 0x2125d,
  [0x0214] = 0x21276, [0x0215] = 0x2126c, [0x0216] = 0x21275, [0x0217] = 0x2125b,
  [0x0218] = 0x21271, [0x0219] = 0x21273, [0x021a] = 0x21261, [0x021b] = 0x2125a,
  [0x021c] = 0x21262, [0x021d] = 0x21272, [0x021e] = 0x2125f, [0x021f] = 0x21258,
  [0x0220] = 0x2151e, [0x0221] = 0x21558, [0x0222] = 0x2158b, [0x0223] = 0x2158a,
  [0x0224] = 0x215f4, [0x0225] = 0x215f0, [0x0226] = 0x216d2, [0x0227] = 0x216cd,
  [0x0228] = 0x216ce, [0x0229] = 0x216bc, [0x022a] = 0x216d0, [0x022b] = 0x21946,
  [0x022c] = 0x21947, [0x022d] = 0x373e, [0x022e] = 0x374d, [0x022f] = 0x3751,
  [0x0230] = 0x219ce, [0x0231] = 0x219cf, [0x0232] = 0x219cc, [0x0233] = 0x219cb,
  [0x0234] = 0x2f872, [0x0235] = 0x21b60, [0x0236] = 0x21b5f, [0x0237] = 0x21b5d,
  [0x0238] = 0x21bd4, [0x0239] = 0x21c3a, [0x023a] = 0x21c38, [0x023b] = 0x21c3c,
  [0x023c] = 0x21c36, [0x023d] = 0x21d08, [0x023e] = 0x21d7e, [0x023f] = 0x21d5e,
  [0x0240] = 0x21d6e, [0x0241] = 0x21d53, [0x0242] = 0x21d70, [0x0243] = 0x21d57,
  [0x0244] = 0x21d5d, [0x0245] = 0x21d63, [0x0246] = 0x21d7a, [0x0247] = 0x21d79,
  [0x0248] = 0x21d93, [0x0249] = 0x2f881, [0x024a] = 0x2201d, [0x024b] = 0x22068,
  [0x024c] = 0x22073, [0x024d] = 0x22074, [0x024e] = 0x22066, [0x024f] = 0x2206f,
  [0x0250] = 0x22075, [0x0251] = 0x2207a, [0x0252] = 0x22070, [0x0253] = 0x2206d,
  [0x0254] = 0x22065, [0x0255] = 0x2218a, [0x0256] = 0x221ed, [0x0257] = 0x221e9,
  [0x0258] = 0x221e3, [0x0259] = 0x221ef, [0x025a] = 0x3873, [0x025b] = 0x221ea,
  [0x025c] = 0x22331, [0x025d] = 0x2f891, [0x025e] = 0x22338, [0x025f] = 0x26b20,
  [0x0260] = 0x22335, [0x0261] = 0x2233a, [0x0262] = 0x22381, [0x0263] = 0x22380,
  [0x0264] = 0x223b8, [0x0265] = 0x223c1, [0x0266] = 0x223ba, [0x0267] = 0x22488,
  [0x0268] = 0x224cf, [0x0269] = 0x224ca, [0x026a] = 0x224c7, [0x026b] = 0x224c5,
  [0x026c] = 0x224c8, [0x026d] = 0x224d0, [0x026e] = 0x224c9, [0x026f] = 0x225bb,
  [0x0270] = 0x225b6, [0x0271] = 0x225bd, [0x0272] = 0x2f89d, [0x0273] = 0x225b4,
  [0x0274] = 0x225c1, [0x0275] = 0x225d1, [0x0276] = 0x225d2, [0x0277] = 0x225cc,
  [0x0278] = 0x225d4, [0x0279] = 0x225cb, [0x027a] = 0x225ce, [0x027b] = 0x225d5,
  [0x027c] = 0x225d8, [0x027d] = 0x22995, [0x027e] = 0x22993, [0x027f] = 0x22999,
  [0x0280] = 0x22996, [0x0281] = 0x39be, [0x0282] = 0x22a89, [0x0283] = 0x22a94,
  [0x0284] = 0x22a91, [0x0285] = 0x22a8f, [0x0286] = 0x22a84, [0x0287] = 0x22a97,
  [0x0288] = 0x22a86, [0x0289] = 0x22a85, [0x028a] = 0x22a8c, [0x028b] = 0x22ef2,
  [0x028c] = 0x22eed, [0x028d] = 0x22eef, [0x028e] = 0x230d9, [0x028f] = 0x23146,
  [0x0290] = 0x23147, [0x0291] = 0x2314b, [0x0292] = 0x23150, [0x0293] = 0x2314a,
  [0x0294] = 0x2314e, [0x0295] = 0x2314c, [0x0296] = 0x23148, [0x0297] = 0x23363,
  [0x0298] = 0x23361, [0x0299] = 0x23362, [0x029a] = 0x23360, [0x029b] = 0x233c3,
  [0x029c] = 0x233c5, [0x029d] = 0x233c1, [0x029e] = 0x233d0, [0x029f] = 0x2f8dc,
  [0x02a0] = 0x23882, [0x02a1] = 0x3c1e, [0x02a2] = 0x23886, [0x02a3] = 0x23884,
  [0x02a4] = 0x2394c, [0x02a5] = 0x23948, [0x02a6] = 0x239b8, [0x02a7] = 0x239bf,
  [0x02a8] = 0x239c1, [0x02a9] = 0x239c6, [0x02aa] = 0x239c5, [0x02ab] = 0x239c7,
  [0x02ac] = 0x239c0, [0x02ad] = 0x239c3, [0x02ae] = 0x23aed, [0x02af] = 0x23b03,
  [0x02b0] = 0x23b22, [0x02b1] = 0x23b21, [0x02b2] = 0x23c9f, [0x02b3] = 0x23c90,
  [0x02b4] = 0x23c9e, [0x02b5] = 0x3cc8, [0x02b6] = 0x23ca0, [0x02b7] = 0x23c94,
  [0x02b8] = 0x23c95, [0x02b9] = 0x23ca1, [0x02ba] = 0x3cc3, [0x02bb] = 0x3cc7,
  [0x02bc] = 0x2419a, [0x02bd] = 0x24190, [0x02be] = 0x244f2, [0x02bf] = 0x244f3,
  [0x02c0] = 0x2456e, [0x02c1] = 0x245ab, [0x02c2] = 0x245a9, [0x02c3] = 0x2461a,
  [0x02c4] = 0x24618, [0x02c5] = 0x2461b, [0x02c6] = 0x24760, [0x02c7] = 0x2474c,
  [0x02c8] = 0x24742, [0x02c9] = 0x24733, [0x02ca] = 0x2473f, [0x02cb] = 0x2473e,
  [0x02cc] = 0x24738, [0x02cd] = 0x24743, [0x02ce] = 0x24746, [0x02cf] = 0x24736,
  [0x02d0] = 0x248ee, [0x02d1] = 0x248f5, [0x02d2] = 0x248f4, [0x02d3] = 0x3f56,
  [0x02d4] = 0x24c1b, [0x02d5] = 0x3540, [0x02d6] = 0x24d2b, [0x02d7] = 0x24d29,
  [0x02d8] = 0x24d2a, [0x02d9] = 0x24d2e, [0x02da] = 0x2503f, [0x02db] = 0x250e6,
  [0x02dc] = 0x253a7, [0x02dd] = 0x253a8, [0x02de] = 0x25418, [0x02df] = 0x25608,
  [0x02e0] = 0x25750, [0x02e1] = 0x25753, [0x02e2] = 0x25a56, [0x02e3] = 0x25a55,
  [0x02e4] = 0x2626f, [0x02e5] = 0x2627f, [0x02e6] = 0x264b3, [0x02e7] = 0x2663a,
  [0x02e8] = 0x26638, [0x02e9] = 0x26637, [0x02ea] = 0x26636, [0x02eb] = 0x20554,
  [0x02ec] = 0x2f980, [0x02ed] = 0x268dd, [0x02ee] = 0x268fc, [0x02ef] = 0x268fa,
  [0x02f0] = 0x26933, [0x02f1] = 0x26b04, [0x02f2] = 0x26b07, [0x02f3] = 0x26b08,
  [0x02f4] = 0x26b06, [0x02f5] = 0x26b09, [0x02f6] = 0x2f991, [0x02f7] = 0x2f993,
  [0x02f8] = 0x2721e, [0x02f9] = 0x2f9d2, [0x02fa] = 0x2840b, [0x02fb] = 0x28443,
  [0x02fc] = 0x28452, [0x02fd] = 0x28458, [0x02fe] = 0x2845e, [0x02ff] = 0x28668,
  [0x0300] = 0x2868d, [0x0301] = 0x28682, [0x0302] = 0x2867b, [0x0303] = 0x28c58,
  [0x0304] = 0x49bd, [0x0305] = 0x2f800, [0x0306] = 0x2002c, [0x0307] = 0x2002e,
  [0x0308] = 0x2160b, [0x0309] = 0x2002d, [0x030a] = 0x20069, [0x030b] = 0x200b2,
  [0x030c] = 0x200b6, [0x030d] = 0x200a9, [0x030e] = 0x200ed, [0x030f] = 0x20119,
  [0x0310] = 0x2014f, [0x0311] = 0x20216, [0x0312] = 0x20203, [0x0313] = 0x2020e,
  [0x0314] = 0x20204, [0x0315] = 0x20207, [0x0316] = 0x20210, [0x0317] = 0x20227,
  [0x0318] = 0x20228, [0x0319] = 0x2020d, [0x031a] = 0x20229, [0x031b] = 0x2020f,
  [0x031c] = 0x20222, [0x031d] = 0x20211, [0x031e] = 0x20485, [0x031f] = 0x20484,
  [0x0320] = 0x20483, [0x0321] = 0x2048a, [0x0322] = 0x20486, [0x0323] = 0x20510,
  [0x0324] = 0x20558, [0x0325] = 0x2055a, [0x0326] = 0x2070c, [0x0327] = 0x205c4,
  [0x0328] = 0x205c5, [0x0329] = 0x20647, [0x032a] = 0x20649, [0x032b] = 0x2064a,
  [0x032c] = 0x2064b, [0x032d] = 0x221bb, [0x032e] = 0x2068d, [0x032f] = 0x2068b,
  [0x0330] = 0x20707, [0x0331] = 0x20706, [0x0332] = 0x2f820, [0x0333] = 0x206ef,
  [0x0334] = 0x206f0, [0x0335] = 0x206ed, [0x0336] = 0x206f9, [0x0337] = 0x20702,
  [0x0338] = 0x206e7, [0x0339] = 0x206f4, [0x033a] = 0x206f7, [0x033b] = 0x206f1,
  [0x033c] = 0x20853, [0x033d] = 0x20855, [0x033e] = 0x351a, [0x033f] = 0x208da,
  [0x0340] = 0x208d9, [0x0341] = 0x352c, [0x0342] = 0x20938, [0x0343] = 0x20999,
  [0x0344] = 0x209a2, [0x0345] = 0x2099c, [0x0346] = 0x2099a, [0x0347] = 0x2099b,
  [0x0348] = 0x209e5, [0x0349] = 0x209e3, [0x034a] = 0x209e1, [0x034b] = 0x209e0,
  [0x034c] = 0x209e2, [0x034d] = 0x209e4, [0x034f] = 0x209e8, [0x0350] = 0x2f82f,
  [0x0351] = 0x20a13, [0x0352] = 0x20a56, [0x0353] = 0x20a4b, [0x0354] = 0x20a4a,
  [0x0355] = 0x20a4d, [0x0356] = 0x20a4c, [0x0357] = 0x3549, [0x0358] = 0x20aea,
  [0x0359] = 0x20ae9, [0x035a] = 0x20aeb, [0x035b] = 0x20aec, [0x035c] = 0x20b3a,
  [0x035d] = 0x20b3c, [0x035e] = 0x20b39, [0x035f] = 0x20b3b, [0x0360] = 0x20c14,
  [0x0361] = 0x20c0e, [0x0362] = 0x20c35, [0x0363] = 0x20c0a, [0x0364] = 0x20c3f,
  [0x0365] = 0x20c38, [0x0366] = 0x357c, [0x0367] = 0x20c4e, [0x0368] = 0x20c17,
  [0x0369] = 0x2f83f, [0x036a] = 0x20c1e, [0x036b] = 0x211b6, [0x036c] = 0x211ce,
  [0x036d] = 0x211ca, [0x036e] = 0x211cb, [0x036f] = 0x211cc, [0x0370] = 0x211c9,
  [0x0371] = 0x212b4, [0x0372] = 0x212ac, [0x0373] = 0x21288, [0x0374] = 0x21286,
  [0x0375] = 0x2128c, [0x0376] = 0x21293, [0x0377] = 0x2128a, [0x0378] = 0x212a4,
  [0x0379] = 0x21287, [0x037a] = 0x212a5, [0x037b] = 0x212a6, [0x037c] = 0x21285,
  [0x037d] = 0x212a3, [0x037e] = 0x2128e, [0x037f] = 0x2151f, [0x0380] = 0x21563,
  [0x0381] = 0x2158d, [0x0382] = 0x215fa, [0x0383] = 0x215fb, [0x0384] = 0x21609,
  [0x0385] = 0x215fe, [0x0386] = 0x215ff, [0x0387] = 0x21602, [0x0388] = 0x21604,
  [0x0389] = 0x215fd, [0x038a] = 0x21605, [0x038b] = 0x216f9, [0x038c] = 0x216f7,
  [0x038d] = 0x216db, [0x038e] = 0x216da, [0x038f] = 0x21951, [0x0390] = 0x21950,
  [0x0391] = 0x21957, [0x0392] = 0x219e2, [0x0393] = 0x219d8, [0x0394] = 0x219da,
  [0x0395] = 0x219dc, [0x0396] = 0x21b20, [0x0397] = 0x21b1f, [0x0398] = 0x21b69,
  [0x0399] = 0x21bdd, [0x039a] = 0x21c40, [0x039b] = 0x21c41, [0x039c] = 0x21c43,
  [0x039d] = 0x21d0d, [0x039e] = 0x21d0c, [0x039f] = 0x21d87, [0x03a0] = 0x21da9,
  [0x03a1] = 0x21d92, [0x03a2] = 0x21d85, [0x03a3] = 0x37bf, [0x03a4] = 0x21d8c,
  [0x03a5] = 0x21da2, [0x03a6] = 0x21d8f, [0x03a7] = 0x21d8a, [0x03a8] = 0x37ba,
  [0x03a9] = 0x21d8d, [0x03aa] = 0x21ff1, [0x03ab] = 0x22037, [0x03ac] = 0x2208b,
  [0x03ad] = 0x22087, [0x03ae] = 0x22082, [0x03af] = 0x2218e, [0x03b0] = 0x221bc,
  [0x03b1] = 0x221f8, [0x03b2] = 0x221fa, [0x03b3] = 0x221eb, [0x03b4] = 0x221f9,
  [0x03b5] = 0x221fc, [0x03b6] = 0x22205, [0x03b7] = 0x2231b, [0x03b8] = 0x2231d,
  [0x03b9] = 0x2231e, [0x03ba] = 0x2f892, [0x03bb] = 0x2233c, [0x03bc] = 0x2233e,
  [0x03bd] = 0x22384, [0x03be] = 0x223c8, [0x03bf] = 0x223c9, [0x03c0] = 0x223d3,
  [0x03c1] = 0x223c7, [0x03c2] = 0x223d2, [0x03c3] = 0x22455, [0x03c4] = 0x2f896,
  [0x03c5] = 0x2248c, [0x03c6] = 0x2248a, [0x03c7] = 0x224d5, [0x03c8] = 0x224d3,
  [0x03c9] = 0x38d8, [0x03ca] = 0x224d8, [0x03cb] = 0x225d0, [0x03cc] = 0x225ee,
  [0x03cd] = 0x225f1, [0x03ce] = 0x225fb, [0x03cf] = 0x225d3, [0x03d0] = 0x225ec,
  [0x03d1] = 0x225cd, [0x03d2] = 0x22615, [0x03d3] = 0x22626, [0x03d4] = 0x2260e,
  [0x03d5] = 0x22627, [0x03d6] = 0x2262a, [0x03d7] = 0x22600, [0x03d8] = 0x22604,
  [0x03d9] = 0x22628, [0x03da] = 0x22608, [0x03db] = 0x22614, [0x03dc] = 0x2299b,
  [0x03dd] = 0x39b2, [0x03de] = 0x229a1, [0x03df] = 0x2299d, [0x03e0] = 0x229a0,
  [0x03e1] = 0x2299e, [0x03e2] = 0x229a2, [0x03e3] = 0x2299c, [0x03e4] = 0x22a32,
  [0x03e5] = 0x22a34, [0x03e6] = 0x22a93, [0x03e7] = 0x22a96, [0x03e8] = 0x22a90,
  [0x03e9] = 0x22ab4, [0x03ea] = 0x22ab5, [0x03eb] = 0x2f8b6, [0x03ec] = 0x22ab6,
  [0x03ed] = 0x22ac2, [0x03ee] = 0x2f8b5, [0x03ef] = 0x22aba, [0x03f0] = 0x22a92,
  [0x03f1] = 0x22efb, [0x03f2] = 0x22f05, [0x03f3] = 0x22ef3, [0x03f4] = 0x22efe,
  [0x03f5] = 0x22efd, [0x03f6] = 0x2455d, [0x03f7] = 0x22efa, [0x03f8] = 0x23043,
  [0x03f9] = 0x23095, [0x03fa] = 0x23094, [0x03fb] = 0x3ac4, [0x03fc] = 0x2316f,
  [0x03fd] = 0x23173, [0x03fe] = 0x2315c, [0x03ff] = 0x2315b, [0x0400] = 0x23166,
  [0x0401] = 0x23174, [0x0402] = 0x23163, [0x0403] = 0x23167, [0x0404] = 0x23160,
  [0x0405] = 0x23162, [0x0406] = 0x23171, [0x0407] = 0x23172, [0x0408] = 0x2315e,
  [0x0409] = 0x23323, [0x040a] = 0x23326, [0x040b] = 0x23364, [0x040c] = 0x2f8d6,
  [0x040d] = 0x233f2, [0x040e] = 0x23401, [0x040f] = 0x233dd, [0x0410] = 0x233dc,
  [0x0411] = 0x233db, [0x0412] = 0x233f3, [0x0413] = 0x233fd, [0x0414] = 0x233d8,
  [0x0415] = 0x233f8, [0x0416] = 0x233fb, [0x0417] = 0x23883, [0x0418] = 0x2388c,
  [0x0419] = 0x2388e, [0x041a] = 0x23949, [0x041b] = 0x23954, [0x041c] = 0x23952,
  [0x041d] = 0x2394e, [0x041e] = 0x23958, [0x041f] = 0x23951, [0x0420] = 0x23955,
  [0x0421] = 0x23953, [0x0422] = 0x23957, [0x0423] = 0x23950, [0x0424] = 0x2394f,
  [0x0425] = 0x2394d, [0x0426] = 0x239d2, [0x0427] = 0x239d3, [0x0428] = 0x239c9,
  [0x0429] = 0x239d4, [0x042a] = 0x23aef, [0x042b] = 0x2f8f8, [0x042c] = 0x200b1,
  [0x042d] = 0x23b05, [0x042e] = 0x23b04, [0x042f] = 0x23b2e, [0x0430] = 0x23b2d,
  [0x0431] = 0x23c45, [0x0432] = 0x23c5b, [0x0433] = 0x23c5a, [0x0434] = 0x23c59,
  [0x0435] = 0x23c9d, [0x0436] = 0x23c8e, [0x0437] = 0x23cbc, [0x0438] = 0x23cdd,
  [0x0439] = 0x23cba, [0x043a] = 0x2f8fd, [0x043b] = 0x23cd8, [0x043c] = 0x23ccb,
  [0x043d] = 0x23cd9, [0x043e] = 0x23cda, [0x043f] = 0x23cc4, [0x0440] = 0x23cb8,
  [0x0441] = 0x23cbf, [0x0442] = 0x23cdb, [0x0443] = 0x23cc1, [0x0444] = 0x2f8fc,
  [0x0445] = 0x23cc5, [0x0446] = 0x2f919, [0x0447] = 0x241b1, [0x0448] = 0x241af,
  [0x0449] = 0x2f918, [0x044a] = 0x244f7, [0x044b] = 0x244f6, [0x044c] = 0x244f8,
  [0x044d] = 0x24550, [0x044e] = 0x24551, [0x044f] = 0x24571, [0x0450] = 0x24570,
  [0x0451] = 0x245b0, [0x0452] = 0x245af, [0x0453] = 0x245ae, [0x0454] = 0x245ad,
  [0x0455] = 0x24625, [0x0456] = 0x2462c, [0x0457] = 0x24622, [0x0458] = 0x24633,
  [0x0459] = 0x24634, [0x045a] = 0x3e2b, [0x045b] = 0x2473a, [0x045c] = 0x24740,
  [0x045d] = 0x24768, [0x045e] = 0x3e61, [0x045f] = 0x24764, [0x0460] = 0x2475c,
  [0x0461] = 0x24757, [0x0462] = 0x24753, [0x0463] = 0x2474f, [0x0464] = 0x24751,
  [0x0465] = 0x24754, [0x0466] = 0x24766, [0x0467] = 0x72dd, [0x0468] = 0x3eb2,
  [0x0469] = 0x2490d, [0x046a] = 0x24908, [0x046b] = 0x2f92b, [0x046c] = 0x248fa,
  [0x046d] = 0x24aea, [0x046e] = 0x24b2a, [0x046f] = 0x24bbb, [0x0470] = 0x24bd5,
  [0x0471] = 0x24bd7, [0x0472] = 0x21da4, [0x0473] = 0x200f0, [0x0474] = 0x200b5,
  [0x0475] = 0x24c06, [0x0476] = 0x24c28, [0x0477] = 0x24c22, [0x0478] = 0x24c21,
  [0x0479] = 0x22456, [0x047a] = 0x24c23, [0x047b] = 0x211cd, [0x047c] = 0x24d30,
  [0x047d] = 0x24d34, [0x047e] = 0x24fc9, [0x047f] = 0x24fca, [0x0480] = 0x24fc8,
  [0x0481] = 0x2f93d, [0x0482] = 0x25044, [0x0483] = 0x250f3, [0x0484] = 0x2f941,
  [0x0485] = 0x250f8, [0x0486] = 0x250fc, [0x0487] = 0x250f6, [0x0488] = 0x250fb,
  [0x0489] = 0x250f7, [0x048a] = 0x25100, [0x048b] = 0x25102, [0x048c] = 0x2f940,
  [0x048d] = 0x2f94d, [0x048e] = 0x25424, [0x048f] = 0x2560d, [0x0490] = 0x4129,
  [0x0491] = 0x25762, [0x0492] = 0x25930, [0x0493] = 0x2592b, [0x0494] = 0x2592a,
  [0x0495] = 0x4192, [0x0496] = 0x2592d, [0x0497] = 0x25ada, [0x0498] = 0x25e28,
  [0x0499] = 0x25f86, [0x049a] = 0x26277, [0x049b] = 0x2627c, [0x049c] = 0x26350,
  [0x049d] = 0x2634e, [0x049e] = 0x263f2, [0x049f] = 0x264b2, [0x04a0] = 0x264ce,
  [0x04a1] = 0x264cf, [0x04a2] = 0x264e5, [0x04a3] = 0x23407, [0x04a4] = 0x26535,
  [0x04a5] = 0x26531, [0x04a6] = 0x26613, [0x04a7] = 0x2662b, [0x04a8] = 0x2662c,
  [0x04a9] = 0x26645, [0x04aa] = 0x26660, [0x04ab] = 0x26641, [0x04ac] = 0x26658,
  [0x04ad] = 0x26647, [0x04ae] = 0x2665b, [0x04af] = 0x26650, [0x04b0] = 0x2f982,
  [0x04b1] = 0x2665f, [0x04b2] = 0x2664a, [0x04b3] = 0x26656, [0x04b4] = 0x26643,
  [0x04b5] = 0x26644, [0x04b6] = 0x26651, [0x04b7] = 0x268fd, [0x04b8] = 0x26934,
  [0x04b9] = 0x26935, [0x04ba] = 0x26954, [0x04bb] = 0x26955, [0x04bc] = 0x26a0c,
  [0x04bd] = 0x26a0b, [0x04be] = 0x26a0a, [0x04bf] = 0x26af9, [0x04c0] = 0x26b35,
  [0x04c1] = 0x26b24, [0x04c2] = 0x26b13, [0x04c3] = 0x26b1c, [0x04c4] = 0x26b69,
  [0x04c5] = 0x26b25, [0x04c6] = 0x26b21, [0x04c7] = 0x2f995, [0x04c8] = 0x2f996,
  [0x04c9] = 0x2f998, [0x04ca] = 0x27220, [0x04cb] = 0x275a9, [0x04cc] = 0x275a8,
  [0x04cd] = 0x275de, [0x04ce] = 0x27609, [0x04cf] = 0x2760a, [0x04d0] = 0x2760b,
  [0x04d1] = 0x277e3, [0x04d2] = 0x27807, [0x04d3] = 0x278b2, [0x04d4] = 0x278b3,
  [0x04d5] = 0x2795d, [0x04d6] = 0x27e58, [0x04d7] = 0x27e86, [0x04d8] = 0x27e87,
  [0x04d9] = 0x27fb9, [0x04da] = 0x27fb8, [0x04db] = 0x2840c, [0x04dc] = 0x28444,
  [0x04dd] = 0x28466, [0x04de] = 0x28480, [0x04df] = 0x28475, [0x04e0] = 0x2846d,
  [0x04e1] = 0x2847e, [0x04e2] = 0x28467, [0x04e3] = 0x28481, [0x04e4] = 0x28477,
  [0x04e5] = 0x286a8, [0x04e6] = 0x286a2, [0x04e7] = 0x286a5, [0x04e8] = 0x2869b,
  [0x04e9] = 0x28c57, [0x04ea] = 0x28e0f, [0x04eb] = 0x28e32, [0x04ec] = 0x28e2c,
  [0x04ed] = 0x28e33, [0x04ee] = 0x291e7, [0x04ef] = 0x2006e, [0x04f0] = 0x200b7,
  [0x04f1] = 0x200f6, [0x04f2] = 0x200f2, [0x04f3] = 0x200f3, [0x04f4] = 0x20517,
  [0x04f5] = 0x20130, [0x04f6] = 0x20154, [0x04f7] = 0x20153, [0x04f8] = 0x20252,
  [0x04f9] = 0x2023d, [0x04fa] = 0x2023c, [0x04fb] = 0x20234, [0x04fc] = 0x20233,
  [0x04fd] = 0x20254, [0x04fe] = 0x2022f, [0x04ff] = 0x3458, [0x0500] = 0x20255,
  [0x0501] = 0x2023f, [0x0502] = 0x20237, [0x0503] = 0x20256, [0x0504] = 0x20261,
  [0x0505] = 0x20258, [0x0506] = 0x2f806, [0x0507] = 0x2023b, [0x0508] = 0x20232,
  [0x0509] = 0x2022e, [0x050a] = 0x2023e, [0x050b] = 0x2003b, [0x050c] = 0x2048c,
  [0x050d] = 0x2048d, [0x050e] = 0x20513, [0x050f] = 0x20516, [0x0510] = 0x20512,
  [0x0511] = 0x2055b, [0x0512] = 0x2058a, [0x0513] = 0x205d7, [0x0514] = 0x205ce,
  [0x0515] = 0x20650, [0x0516] = 0x20651, [0x0517] = 0x2064f, [0x0518] = 0x2068e,
  [0x0519] = 0x2068f, [0x051a] = 0x206f2, [0x051b] = 0x2f821, [0x051c] = 0x20728,
  [0x051d] = 0x2071a, [0x051e] = 0x20725, [0x051f] = 0x2071d, [0x0520] = 0x20714,
  [0x0521] = 0x20720, [0x0522] = 0x2071f, [0x0523] = 0x2071b, [0x0524] = 0x20717,
  [0x0525] = 0x20715, [0x0526] = 0x2071c, [0x0527] = 0x20713, [0x0528] = 0x2f825,
  [0x0529] = 0x20860, [0x052a] = 0x351c, [0x052b] = 0x20866, [0x052c] = 0x208e3,
  [0x052d] = 0x208de, [0x052e] = 0x208e0, [0x052f] = 0x208e1, [0x0530] = 0x2091a,
  [0x0531] = 0x2091b, [0x0532] = 0x20918, [0x0533] = 0x2091c, [0x0534] = 0x20919,
  [0x0535] = 0x20942, [0x0536] = 0x20940, [0x0537] = 0x2093e, [0x0538] = 0x20944,
  [0x0539] = 0x20974, [0x053a] = 0x20943, [0x053b] = 0x2f82d, [0x053c] = 0x209f2,
  [0x053d] = 0x20a1b, [0x053e] = 0x20a19, [0x053f] = 0x20a1e, [0x0540] = 0x2f830,
  [0x0541] = 0x20a5e, [0x0542] = 0x20a66, [0x0543] = 0x20a5d, [0x0544] = 0x20a5a,
  [0x0545] = 0x20a5f, [0x0546] = 0x20a60, [0x0547] = 0x20a5b, [0x0548] = 0x20a5c,
  [0x0549] = 0x20a59, [0x054a] = 0x20af0, [0x054b] = 0x20af1, [0x054c] = 0x20b43,
  [0x054d] = 0x20b40, [0x054e] = 0x20b42, [0x054f] = 0x20b41, [0x0550] = 0x20b3f,
  [0x0551] = 0x20c57, [0x0552] = 0x20c69, [0x0553] = 0x20c6f, [0x0554] = 0x20cab,
  [0x0555] = 0x20c71, [0x0556] = 0x20c93, [0x0557] = 0x20c56, [0x0558] = 0x20c90,
  [0x0559] = 0x20c6d, [0x055a] = 0x20c6c, [0x055b] = 0x20c70, [0x055c] = 0x20c66,
  [0x055d] = 0x20c67, [0x055e] = 0x211d7, [0x055f] = 0x211b5, [0x0560] = 0x211db,
  [0x0561] = 0x211d8, [0x0562] = 0x211d5, [0x0563] = 0x211d4, [0x0564] = 0x212ba,
  [0x0565] = 0x212b7, [0x0566] = 0x212be, [0x0567] = 0x212b9, [0x0568] = 0x3637,
  [0x0569] = 0x212e0, [0x056a] = 0x212dd, [0x056b] = 0x212de, [0x056c] = 0x24918,
  [0x056d] = 0x212d8, [0x056e] = 0x212bd, [0x056f] = 0x212db, [0x0570] = 0x2151d,
  [0x0571] = 0x21525, [0x0572] = 0x21721, [0x0573] = 0x2170c, [0x0574] = 0x21729,
  [0x0575] = 0x2172d, [0x0576] = 0x2171b, [0x0577] = 0x2170e, [0x0578] = 0x2171f,
  [0x0579] = 0x21704, [0x057a] = 0x2171c, [0x057b] = 0x21705, [0x057c] = 0x21706,
  [0x057d] = 0x21720, [0x057e] = 0x2170d, [0x057f] = 0x2172a, [0x0580] = 0x21723,
  [0x0581] = 0x21711, [0x0582] = 0x2195c, [0x0583] = 0x21966, [0x0584] = 0x2195b,
  [0x0585] = 0x2194f, [0x0586] = 0x2195e, [0x0587] = 0x2195d, [0x0588] = 0x219f1,
  [0x0589] = 0x219ea, [0x058a] = 0x219f4, [0x058b] = 0x219eb, [0x058c] = 0x219f0,
  [0x058d] = 0x219fa, [0x058e] = 0x219fb, [0x058f] = 0x21b28, [0x0590] = 0x21b2c,
  [0x0591] = 0x21b6b, [0x0592] = 0x21b2a, [0x0593] = 0x21b6a, [0x0594] = 0x21b6f,
  [0x0595] = 0x21bdb, [0x0596] = 0x21be0, [0x0597] = 0x24fd1, [0x0598] = 0x21be3,
  [0x0599] = 0x21be5, [0x059a] = 0x21be7, [0x059b] = 0x21c4d, [0x059c] = 0x21c55,
  [0x059d] = 0x21c54, [0x059e] = 0x21c53, [0x059f] = 0x21c52, [0x05a0] = 0x21c4e,
  [0x05a1] = 0x21c60, [0x05a2] = 0x221c0, [0x05a3] = 0x21d0f, [0x05a4] = 0x21d11,
  [0x05a5] = 0x21d13, [0x05a6] = 0x21dc6, [0x05a7] = 0x21db3, [0x05a8] = 0x21dc7,
  [0x05a9] = 0x21dd2, [0x05aa] = 0x21db8, [0x05ab] = 0x21dac, [0x05ac] = 0x21dae,
  [0x05ad] = 0x21dcf, [0x05ae] = 0x21dc5, [0x05af] = 0x2f87b, [0x05b0] = 0x21dcc,
  [0x05b1] = 0x21dab, [0x05b2] = 0x21dc9, [0x05b3] = 0x21db9, [0x05b4] = 0x21ffb,
  [0x05b5] = 0x21ff8, [0x05b6] = 0x21ff7, [0x05b7] = 0x21ff9, [0x05b8] = 0x22020,
  [0x05b9] = 0x2203b, [0x05ba] = 0x22039, [0x05bb] = 0x2209e, [0x05bc] = 0x2209a,
  [0x05bd] = 0x220aa, [0x05be] = 0x220ab, [0x05bf] = 0x220af, [0x05c0] = 0x22096,
  [0x05c1] = 0x220a9, [0x05c2] = 0x220a6, [0x05c3] = 0x22091, [0x05c4] = 0x3843,
  [0x05c5] = 0x220ae, [0x05c6] = 0x2209f, [0x05c7] = 0x220ac, [0x05c8] = 0x220a0,
  [0x05c9] = 0x22192, [0x05ca] = 0x22191, [0x05cb] = 0x221bf, [0x05cc] = 0x22217,
  [0x05cd] = 0x2220a, [0x05ce] = 0x2220c, [0x05cf] = 0x2234a, [0x05d0] = 0x22346,
  [0x05d1] = 0x22334, [0x05d2] = 0x22345, [0x05d3] = 0x22343, [0x05d4] = 0x22344,
  [0x05d5] = 0x2f893, [0x05d6] = 0x22387, [0x05d7] = 0x22386, [0x05d8] = 0x2238a,
  [0x05d9] = 0x223da, [0x05da] = 0x223d8, [0x05db] = 0x38b4, [0x05dc] = 0x20518,
  [0x05dd] = 0x20134, [0x05de] = 0x223d6, [0x05df] = 0x223d4, [0x05e0] = 0x22454,
  [0x05e1] = 0x22459, [0x05e2] = 0x2245a, [0x05e3] = 0x22457, [0x05e4] = 0x38e0,
  [0x05e5] = 0x224dd, [0x05e6] = 0x224e9, [0x05e7] = 0x224e0, [0x05e8] = 0x22605,
  [0x05e9] = 0x22612, [0x05ea] = 0x22613, [0x05eb] = 0x22607, [0x05ec] = 0x22616,
  [0x05ed] = 0x22623, [0x05ee] = 0x22602, [0x05ef] = 0x2f8a1, [0x05f0] = 0x2264a,
  [0x05f1] = 0x22636, [0x05f2] = 0x22640, [0x05f3] = 0x22656, [0x05f4] = 0x22643,
  [0x05f5] = 0x2f8a0, [0x05f6] = 0x2264b, [0x05f7] = 0x22646, [0x05f8] = 0x2263e,
  [0x05f9] = 0x22649, [0x05fa] = 0x229a7, [0x05fb] = 0x229b6, [0x05fc] = 0x229a6,
  [0x05fd] = 0x229a8, [0x05fe] = 0x229ac, [0x05ff] = 0x229a9, [0x0600] = 0x229ab,
  [0x0601] = 0x22a38, [0x0602] = 0x22a37, [0x0603] = 0x22a39, [0x0604] = 0x22a41,
  [0x0605] = 0x22a3e, [0x0606] = 0x22ac0, [0x0607] = 0x22abb, [0x0608] = 0x22abf,
  [0x0609] = 0x22abd, [0x060a] = 0x22afe, [0x060b] = 0x22b1e, [0x060c] = 0x22aee,
  [0x060d] = 0x22afc, [0x060e] = 0x2f8b7, [0x060f] = 0x22af9, [0x0610] = 0x22b06,
  [0x0611] = 0x22ae4, [0x0612] = 0x22ae9, [0x0613] = 0x22ae5, [0x0614] = 0x22b03,
  [0x0615] = 0x22afd, [0x0616] = 0x22b49, [0x0617] = 0x22ebe, [0x0618] = 0x22f0a,
  [0x0619] = 0x25626, [0x061a] = 0x22f18, [0x061b] = 0x22f0d, [0x061c] = 0x22f0f,
  [0x061d] = 0x22f0e, [0x061e] = 0x22f20, [0x061f] = 0x23071, [0x0620] = 0x23099,
  [0x0621] = 0x230e2, [0x0622] = 0x230df, [0x0623] = 0x230de, [0x0624] = 0x23178,
  [0x0625] = 0x23179, [0x0626] = 0x3ae4, [0x0627] = 0x2317c, [0x0628] = 0x2317d,
  [0x0629] = 0x23184, [0x062a] = 0x2318b, [0x062b] = 0x2318a, [0x062c] = 0x2f8d2,
  [0x062d] = 0x23189, [0x062e] = 0x2332c, [0x062f] = 0x23329, [0x0630] = 0x2336c,
  [0x0631] = 0x6804, [0x0632] = 0x23409, [0x0633] = 0x23408, [0x0634] = 0x2340c,
  [0x0635] = 0x2340d, [0x0636] = 0x23410, [0x0637] = 0x3c26, [0x0638] = 0x2389f,
  [0x0639] = 0x23898, [0x063a] = 0x238a2, [0x063b] = 0x2389a, [0x063c] = 0x239d5,
  [0x063d] = 0x239e2, [0x063e] = 0x239f0, [0x063f] = 0x239ea, [0x0640] = 0x239eb,
  [0x0641] = 0x239ed, [0x0642] = 0x239e8, [0x0643] = 0x239e0, [0x0644] = 0x23a85,
  [0x0645] = 0x23a86, [0x0646] = 0x23af0, [0x0647] = 0x21e46, [0x0648] = 0x23b45,
  [0x0649] = 0x23b38, [0x064a] = 0x23b3e, [0x064b] = 0x23b42, [0x064c] = 0x23b40,
  [0x064d] = 0x23c47, [0x064e] = 0x23c5c, [0x064f] = 0x23c5e, [0x0650] = 0x23cb4,
  [0x0651] = 0x23cc2, [0x0652] = 0x23d14, [0x0653] = 0x23ce5, [0x0654] = 0x23d15,
  [0x0655] = 0x23d11, [0x0656] = 0x2f8ff, [0x0657] = 0x23cee, [0x0658] = 0x23ce7,
  [0x0659] = 0x23cf5, [0x065a] = 0x23cf4, [0x065b] = 0x23ce8, [0x065c] = 0x23d01,
  [0x065d] = 0x23cef, [0x065e] = 0x23cf1, [0x065f] = 0x2f900, [0x0660] = 0x241db,
  [0x0661] = 0x241bf, [0x0662] = 0x241da, [0x0663] = 0x241c7, [0x0664] = 0x241dd,
  [0x0665] = 0x2f91a, [0x0666] = 0x241eb, [0x0667] = 0x241e1, [0x0668] = 0x241c1,
  [0x0669] = 0x241bd, [0x066a] = 0x24507, [0x066b] = 0x244fd, [0x066c] = 0x24501,
  [0x066d] = 0x2450a, [0x066e] = 0x24503, [0x066f] = 0x24572, [0x0670] = 0x24574,
  [0x0671] = 0x24575, [0x0672] = 0x245b2, [0x0673] = 0x245b1, [0x0674] = 0x245b4,
  [0x0675] = 0x2464c, [0x0676] = 0x24642, [0x0677] = 0x24640, [0x0678] = 0x24649,
  [0x0679] = 0x2463c, [0x067a] = 0x2464d, [0x067b] = 0x2464a, [0x067c] = 0x2463b,
  [0x067d] = 0x24761, [0x067e] = 0x24774, [0x067f] = 0x2f926, [0x0680] = 0x2477f,
  [0x0681] = 0x2477a, [0x0682] = 0x24788, [0x0683] = 0x2477c, [0x0684] = 0x3e6f,
  [0x0685] = 0x24770, [0x0686] = 0x2490f, [0x0687] = 0x24928, [0x0688] = 0x24913,
  [0x0689] = 0x2492a, [0x068a] = 0x24aed, [0x068b] = 0x24aef, [0x068c] = 0x24b2e,
  [0x068d] = 0x24bc1, [0x068e] = 0x24bdd, [0x068f] = 0x20b47, [0x0690] = 0x24c2d,
  [0x0691] = 0x24c2b, [0x0692] = 0x24c35, [0x0693] = 0x24c2f, [0x0694] = 0x24c31,
  [0x0695] = 0x24c34, [0x0696] = 0x2f935, [0x0697] = 0x24c30, [0x0698] = 0x24d3a,
  [0x0699] = 0x24d39, [0x069a] = 0x24d37, [0x069b] = 0x24d4b, [0x069c] = 0x24d54,
  [0x069d] = 0x24d4d, [0x069e] = 0x24d51, [0x069f] = 0x24d47, [0x06a0] = 0x24f27,
  [0x06a1] = 0x24f50, [0x06a2] = 0x24f4d, [0x06a3] = 0x24f4e, [0x06a4] = 0x24f54,
  [0x06a5] = 0x24fd2, [0x06a6] = 0x24fce, [0x06a7] = 0x2504b, [0x06a8] = 0x25049,
  [0x06a9] = 0x25105, [0x06aa] = 0x2510f, [0x06ab] = 0x25119, [0x06ac] = 0x2f943,
  [0x06ad] = 0x25106, [0x06ae] = 0x2510c, [0x06af] = 0x25129, [0x06b0] = 0x25104,
  [0x06b1] = 0x25108, [0x06b2] = 0x25125, [0x06b3] = 0x401c, [0x06b4] = 0x25103,
  [0x06b5] = 0x25127, [0x06b6] = 0x25110, [0x06b7] = 0x2510a, [0x06b8] = 0x2f942,
  [0x06b9] = 0x26b5d, [0x06ba] = 0x253aa, [0x06bb] = 0x253ab, [0x06bc] = 0x253a9,
  [0x06bd] = 0x25441, [0x06be] = 0x2543a, [0x06bf] = 0x2543c, [0x06c0] = 0x2542b,
  [0x06c1] = 0x25449, [0x06c2] = 0x25615, [0x06c3] = 0x4100, [0x06c4] = 0x25616,
  [0x06c5] = 0x2f953, [0x06c6] = 0x25631, [0x06c7] = 0x23c4a, [0x06c8] = 0x2573c,
  [0x06c9] = 0x2577c, [0x06ca] = 0x2576d, [0x06cb] = 0x2576a, [0x06cc] = 0x25763,
  [0x06cd] = 0x2576b, [0x06ce] = 0x2577b, [0x06cf] = 0x25764, [0x06d0] = 0x2577a,
  [0x06d1] = 0x25769, [0x06d2] = 0x2576f, [0x06d3] = 0x25937, [0x06d4] = 0x25935,
  [0x06d5] = 0x2593c, [0x06d6] = 0x25936, [0x06d7] = 0x2593d, [0x06d8] = 0x2593e,
  [0x06d9] = 0x25ae1, [0x06da] = 0x25ae0, [0x06db] = 0x25e2d, [0x06dc] = 0x25e2b,
  [0x06dd] = 0x25f8d, [0x06de] = 0x25f92, [0x06df] = 0x42b6, [0x06e0] = 0x25f8e,
  [0x06e1] = 0x2f96b, [0x06e2] = 0x25f91, [0x06e3] = 0x25f96, [0x06e4] = 0x2627b,
  [0x06e5] = 0x26278, [0x06e6] = 0x26281, [0x06e7] = 0x26296, [0x06e8] = 0x262a3,
  [0x06e9] = 0x26295, [0x06ea] = 0x26297, [0x06eb] = 0x26354, [0x06ec] = 0x26352,
  [0x06ed] = 0x263f5, [0x06ee] = 0x26400, [0x06ef] = 0x263fb, [0x06f0] = 0x263f4,
  [0x06f1] = 0x263f6, [0x06f2] = 0x264e8, [0x06f3] = 0x264ea, [0x06f4] = 0x2663e,
  [0x06f5] = 0x2663d, [0x06f6] = 0x2663b, [0x06f7] = 0x26680, [0x06f8] = 0x43e4,
  [0x06f9] = 0x26688, [0x06fa] = 0x26681, [0x06fb] = 0x26682, [0x06fc] = 0x266ce,
  [0x06fd] = 0x26683, [0x06fe] = 0x26677, [0x06ff] = 0x26679, [0x0700] = 0x26673,
  [0x0701] = 0x2666d, [0x0702] = 0x26670, [0x0703] = 0x2668d, [0x0704] = 0x26675,
  [0x0705] = 0x2668c, [0x0706] = 0x2666a, [0x0707] = 0x26691, [0x0708] = 0x26689,
  [0x0709] = 0x2668e, [0x070a] = 0x212dc, [0x070b] = 0x268ff, [0x070c] = 0x26959,
  [0x070d] = 0x269c7, [0x070e] = 0x26a12, [0x070f] = 0x26a14, [0x0710] = 0x26a13,
  [0x0711] = 0x26acb, [0x0712] = 0x26b42, [0x0713] = 0x44ad, [0x0714] = 0x26b3d,
  [0x0715] = 0x82ff, [0x0716] = 0x26b40, [0x0717] = 0x26b44, [0x0718] = 0x26b62,
  [0x0719] = 0x26b43, [0x071a] = 0x26b3f, [0x071b] = 0x26b45, [0x071c] = 0x26b3c,
  [0x071d] = 0x2f997, [0x071e] = 0x26b46, [0x071f] = 0x26b47, [0x0720] = 0x2f9b3,
  [0x0721] = 0x2719e, [0x0722] = 0x2719d, [0x0723] = 0x2722c, [0x0724] = 0x27229,
  [0x0725] = 0x2722f, [0x0726] = 0x2722e, [0x0727] = 0x27230, [0x0728] = 0x275e1,
  [0x0729] = 0x275e2, [0x072a] = 0x2760e, [0x072b] = 0x27619, [0x072c] = 0x27612,
  [0x072d] = 0x278b4, [0x072e] = 0x27bda, [0x072f] = 0x27d26, [0x0730] = 0x27e59,
  [0x0731] = 0x27e5a, [0x0732] = 0x27e88, [0x0733] = 0x27fc0, [0x0734] = 0x27fba,
  [0x0735] = 0x282a0, [0x0736] = 0x2f9dd, [0x0737] = 0x282a2, [0x0738] = 0x28486,
  [0x0739] = 0x8fec, [0x073a] = 0x2849d, [0x073b] = 0x28488, [0x073c] = 0x2848f,
  [0x073d] = 0x2848e, [0x073e] = 0x2849b, [0x073f] = 0x286c1, [0x0740] = 0x286c3,
  [0x0741] = 0x286c4, [0x0742] = 0x28696, [0x0743] = 0x286c7, [0x0744] = 0x286c6,
  [0x0745] = 0x286bf, [0x0746] = 0x48c3, [0x0747] = 0x2890f, [0x0748] = 0x28e55,
  [0x0749] = 0x28e42, [0x074a] = 0x28e46, [0x074b] = 0x28e4b, [0x074c] = 0x28e43,
  [0x074d] = 0x28e53, [0x074e] = 0x28e48, [0x074f] = 0x28e49, [0x0750] = 0x28e4a,
  [0x0751] = 0x28e4e, [0x0752] = 0x24bde, [0x0753] = 0x20691, [0x0754] = 0x2944b,
  [0x0755] = 0x29680, [0x0756] = 0x29681, [0x0757] = 0x29683, [0x0758] = 0x2028a,
  [0x0759] = 0x29682, [0x075a] = 0x22342, [0x075b] = 0x20922, [0x075c] = 0x20070,
  [0x075d] = 0x20071, [0x075e] = 0x200bc, [0x075f] = 0x21d18, [0x0760] = 0x200be,
  [0x0761] = 0x200ba, [0x0762] = 0x200f8, [0x0763] = 0x20137, [0x0764] = 0x20135,
  [0x0765] = 0x20afc, [0x0766] = 0x20156, [0x0767] = 0x20159, [0x0768] = 0x2015e,
  [0x0769] = 0x2026d, [0x076a] = 0x20291, [0x076b] = 0x20292, [0x076c] = 0x20268,
  [0x076d] = 0x20266, [0x076e] = 0x20273, [0x076f] = 0x3467, [0x0770] = 0x20296,
  [0x0771] = 0x2028b, [0x0772] = 0x2028c, [0x0773] = 0x20496, [0x0774] = 0x204ee,
  [0x0775] = 0x2051c, [0x0776] = 0x2051a, [0x0777] = 0x20519, [0x0778] = 0x2051b,
  [0x0779] = 0x2055d, [0x077a] = 0x2055e, [0x077b] = 0x2f818, [0x077c] = 0x205dc,
  [0x077d] = 0x205e2, [0x077e] = 0x20652, [0x077f] = 0x20692, [0x0780] = 0x20730,
  [0x0781] = 0x20752, [0x0782] = 0x20742, [0x0783] = 0x20741, [0x0784] = 0x20745,
  [0x0785] = 0x20737, [0x0786] = 0x20740, [0x0787] = 0x2073f, [0x0788] = 0x2073d,
  [0x0789] = 0x20738, [0x078a] = 0x2073a, [0x078b] = 0x20749, [0x078c] = 0x2086b,
  [0x078d] = 0x20878, [0x078e] = 0x20879, [0x078f] = 0x2f826, [0x0790] = 0x2086c,
  [0x0791] = 0x208e9, [0x0792] = 0x208e6, [0x0793] = 0x208e5, [0x0794] = 0x208ea,
  [0x0795] = 0x208e7, [0x0796] = 0x208e8, [0x0797] = 0x2091f, [0x0798] = 0x2094b,
  [0x0799] = 0x2094a, [0x079a] = 0x20953, [0x079b] = 0x20976, [0x079c] = 0x209a3,
  [0x079d] = 0x209a4, [0x079e] = 0x209f6, [0x079f] = 0x209f3, [0x07a0] = 0x209f9,
  [0x07a1] = 0x209f7, [0x07a2] = 0x209fc, [0x07a3] = 0x20a1d, [0x07a4] = 0x20a6d,
  [0x07a5] = 0x20a71, [0x07a6] = 0x20a6c, [0x07a7] = 0x20a6e, [0x07a8] = 0x20a70,
  [0x07a9] = 0x20a6f, [0x07aa] = 0x20a67, [0x07ab] = 0x20a68, [0x07ac] = 0x20afa,
  [0x07ad] = 0x20af9, [0x07ae] = 0x20b4e, [0x07af] = 0x53de, [0x07b0] = 0x20b4d,
  [0x07b1] = 0x20b4f, [0x07b2] = 0x20b4a, [0x07b3] = 0x20b4c, [0x07b4] = 0x3596,
  [0x07b5] = 0x2f842, [0x07b6] = 0x20cee, [0x07b7] = 0x20cb2, [0x07b8] = 0x20cc0,
  [0x07b9] = 0x20cc1, [0x07ba] = 0x20cf4, [0x07bb] = 0x20cc8, [0x07bc] = 0x20cc5,
  [0x07bd] = 0x20cc6, [0x07be] = 0x20cad, [0x07bf] = 0x211e2, [0x07c0] = 0x211ea,
  [0x07c1] = 0x211e3, [0x07c2] = 0x211e1, [0x07c3] = 0x212f7, [0x07c4] = 0x21301,
  [0x07c5] = 0x21312, [0x07c6] = 0x212f6, [0x07c7] = 0x212f1, [0x07c8] = 0x2131f,
  [0x07c9] = 0x212ee, [0x07ca] = 0x2f852, [0x07cb] = 0x212f3, [0x07cc] = 0x21315,
  [0x07cd] = 0x21316, [0x07ce] = 0x21317, [0x07cf] = 0x212f8, [0x07d0] = 0x21319,
  [0x07d1] = 0x212f2, [0x07d2] = 0x212f4, [0x07d3] = 0x212f5, [0x07d4] = 0x21313,
  [0x07d5] = 0x21306, [0x07d6] = 0x21526, [0x07d7] = 0x21524, [0x07d8] = 0x2155a,
  [0x07d9] = 0x22ec8, [0x07da] = 0x21597, [0x07db] = 0x21595, [0x07dc] = 0x2159a,
  [0x07dd] = 0x2161f, [0x07de] = 0x20afb, [0x07df] = 0x21629, [0x07e0] = 0x21620,
  [0x07e1] = 0x2f862, [0x07e2] = 0x2f863, [0x07e3] = 0x2174c, [0x07e4] = 0x21730,
  [0x07e5] = 0x21738, [0x07e6] = 0x2173d, [0x07e7] = 0x21751, [0x07e8] = 0x2174f,
  [0x07e9] = 0x2174a, [0x07ea] = 0x21734, [0x07eb] = 0x21736, [0x07ec] = 0x5a30,
  [0x07ed] = 0x2196a, [0x07ee] = 0x21968, [0x07ef] = 0x21a1c, [0x07f0] = 0x21a0e,
  [0x07f1] = 0x21a1e, [0x07f2] = 0x3759, [0x07f3] = 0x21a09, [0x07f4] = 0x21a08,
  [0x07f5] = 0x21a13, [0x07f6] = 0x21a01, [0x07f7] = 0x21a0f, [0x07f8] = 0x21a14,
  [0x07f9] = 0x21a06, [0x07fa] = 0x21a07, [0x07fb] = 0x5bb2, [0x07fc] = 0x2f873,
  [0x07fd] = 0x3776, [0x07fe] = 0x21b79, [0x07ff] = 0x21bea, [0x0800] = 0x21bed,
  [0x0801] = 0x21be9, [0x0802] = 0x21bee, [0x0803] = 0x21c68, [0x0804] = 0x21c64,
  [0x0805] = 0x21c67, [0x0806] = 0x21c72, [0x0807] = 0x21c62, [0x0808] = 0x21c74,
  [0x0809] = 0x21c79, [0x080a] = 0x21d19, [0x080b] = 0x21d17, [0x080c] = 0x21d15,
  [0x080d] = 0x21d16, [0x080e] = 0x21de6, [0x080f] = 0x262a4, [0x0810] = 0x21dee,
  [0x0811] = 0x37d2, [0x0812] = 0x21ddf, [0x0813] = 0x21de4, [0x0814] = 0x21dda,
  [0x0815] = 0x21dea, [0x0816] = 0x21ded, [0x0817] = 0x21de3, [0x0818] = 0x21de9,
  [0x0819] = 0x21ffd, [0x081a] = 0x20657, [0x081b] = 0x22021, [0x081c] = 0x220c6,
  [0x081d] = 0x220b8, [0x081e] = 0x220cb, [0x081f] = 0x2f885, [0x0820] = 0x220bd,
  [0x0821] = 0x220b5, [0x0822] = 0x220bb, [0x0823] = 0x220bf, [0x0824] = 0x220be,
  [0x0825] = 0x220b2, [0x0826] = 0x220c1, [0x0827] = 0x220c2, [0x0828] = 0x22199,
  [0x0829] = 0x221c6, [0x082a] = 0x2222c, [0x082b] = 0x2222d, [0x082c] = 0x22225,
  [0x082d] = 0x2221e, [0x082e] = 0x2221f, [0x082f] = 0x22223, [0x0830] = 0x22350,
  [0x0831] = 0x2234e, [0x0832] = 0x2234d, [0x0833] = 0x22352, [0x0834] = 0x223e9,
  [0x0835] = 0x223ec, [0x0836] = 0x223e8, [0x0837] = 0x22458, [0x0838] = 0x2245c,
  [0x0839] = 0x2245b, [0x083a] = 0x2248f, [0x083b] = 0x23972, [0x083c] = 0x224f6,
  [0x083d] = 0x22500, [0x083e] = 0x224fc, [0x083f] = 0x224f8, [0x0840] = 0x224ea,
  [0x0841] = 0x224fe, [0x0842] = 0x224f7, [0x0843] = 0x224fd, [0x0844] = 0x22670,
  [0x0845] = 0x22662, [0x0846] = 0x22644, [0x0847] = 0x3920, [0x0848] = 0x2264d,
  [0x0849] = 0x2264c, [0x084a] = 0x2263f, [0x084b] = 0x22666, [0x084c] = 0x22635,
  [0x084d] = 0x3929, [0x084e] = 0x22634, [0x084f] = 0x2268d, [0x0850] = 0x22684,
  [0x0851] = 0x3938, [0x0852] = 0x22686, [0x0853] = 0x22689, [0x0854] = 0x22687,
  [0x0855] = 0x22683, [0x0856] = 0x22675, [0x0857] = 0x22679, [0x0858] = 0x226af,
  [0x0859] = 0x226b0, [0x085a] = 0x229b7, [0x085b] = 0x229bb, [0x085c] = 0x229b9,
  [0x085d] = 0x22a46, [0x085e] = 0x22a47, [0x085f] = 0x22a45, [0x0860] = 0x22aea,
  [0x0861] = 0x22af6, [0x0862] = 0x22b68, [0x0863] = 0x22b39, [0x0864] = 0x2f8b9,
  [0x0865] = 0x22b3d, [0x0866] = 0x22b3b, [0x0867] = 0x22b4d, [0x0868] = 0x22b30,
  [0x0869] = 0x22b4a, [0x086a] = 0x22b3e, [0x086b] = 0x22b40, [0x086c] = 0x22b4c,
  [0x086d] = 0x22b47, [0x086e] = 0x22b38, [0x086f] = 0x22b52, [0x0870] = 0x22b3a,
  [0x0871] = 0x22b53, [0x0872] = 0x22ec4, [0x0873] = 0x22ec1, [0x0874] = 0x22f1c,
  [0x0875] = 0x22f1d, [0x0876] = 0x22f2a, [0x0877] = 0x22f1e, [0x0878] = 0x22f2f,
  [0x0879] = 0x22f22, [0x087a] = 0x22f2e, [0x087b] = 0x22f25, [0x087c] = 0x3a89,
  [0x087d] = 0x3ab0, [0x087e] = 0x2304a, [0x087f] = 0x2304b, [0x0880] = 0x23076,
  [0x0881] = 0x3abf, [0x0882] = 0x230e8, [0x0883] = 0x230ef, [0x0884] = 0x230e9,
  [0x0885] = 0x3ac5, [0x0886] = 0x230ea, [0x0887] = 0x2f8cc, [0x0888] = 0x3aea,
  [0x0889] = 0x2319b, [0x088a] = 0x2319e, [0x088b] = 0x23193, [0x088c] = 0x231a7,
  [0x088d] = 0x2319c, [0x088e] = 0x231a0, [0x088f] = 0x2319a, [0x0890] = 0x231ab,
  [0x0891] = 0x231be, [0x0892] = 0x231a9, [0x0893] = 0x2332d, [0x0894] = 0x2336e,
  [0x0895] = 0x23444, [0x0896] = 0x2343d, [0x0897] = 0x2343a, [0x0898] = 0x23468,
  [0x0899] = 0x2343c, [0x089a] = 0x2346a, [0x089b] = 0x2f8e0, [0x089c] = 0x23438,
  [0x089d] = 0x23465, [0x089e] = 0x23439, [0x089f] = 0x2346d, [0x08a0] = 0x23436,
  [0x08a1] = 0x2f8e3, [0x08a2] = 0x2343e, [0x08a3] = 0x2347e, [0x08a4] = 0x23437,
  [0x08a5] = 0x23899, [0x08a6] = 0x238a9, [0x08a7] = 0x238ad, [0x08a8] = 0x238a7,
  [0x08a9] = 0x238a8, [0x08aa] = 0x23966, [0x08ab] = 0x23969, [0x08ac] = 0x2396d,
  [0x08ad] = 0x23967, [0x08ae] = 0x2396b, [0x08af] = 0x2396a, [0x08b0] = 0x239ee,
  [0x08b1] = 0x23a01, [0x08b2] = 0x23a03, [0x08b3] = 0x239f4, [0x08b4] = 0x239fb,
  [0x08b5] = 0x3c37, [0x08b6] = 0x239f6, [0x08b7] = 0x239fc, [0x08b8] = 0x23af4,
  [0x08b9] = 0x23b08, [0x08ba] = 0x23b0a, [0x08bb] = 0x23b09, [0x08bc] = 0x23b6d,
  [0x08bd] = 0x23b62, [0x08be] = 0x23b41, [0x08bf] = 0x23b5e, [0x08c0] = 0x23b5c,
  [0x08c1] = 0x23cf3, [0x08c2] = 0x23d26, [0x08c3] = 0x3ce4, [0x08c4] = 0x23d39,
  [0x08c5] = 0x2f904, [0x08c6] = 0x23d6c, [0x08c7] = 0x23d2b, [0x08c8] = 0x23d2e,
  [0x08c9] = 0x23d3b, [0x08ca] = 0x23d5e, [0x08cb] = 0x23dfb, [0x08cc] = 0x23d27,
  [0x08cd] = 0x23d24, [0x08ce] = 0x23d69, [0x08cf] = 0x23d30, [0x08d0] = 0x2f905,
  [0x08d1] = 0x23d62, [0x08d2] = 0x23d38, [0x08d3] = 0x23d35, [0x08d4] = 0x23d2a,
  [0x08d5] = 0x23d2c, [0x08d6] = 0x23d68, [0x08d7] = 0x23d31, [0x08d8] = 0x23d2f,
  [0x08d9] = 0x23d2d, [0x08da] = 0x23d3a, [0x08db] = 0x23d36, [0x08dc] = 0x2f903,
  [0x08dd] = 0x23d21, [0x08de] = 0x23d3c, [0x08df] = 0x23d20, [0x08e0] = 0x23d64,
  [0x08e1] = 0x23d3e, [0x08e2] = 0x3ce8, [0x08e3] = 0x241f7, [0x08e4] = 0x24212,
  [0x08e5] = 0x241f1, [0x08e6] = 0x241f5, [0x08e7] = 0x24222, [0x08e8] = 0x241f2,
  [0x08e9] = 0x241df, [0x08ea] = 0x24215, [0x08eb] = 0x24216, [0x08ec] = 0x2457a,
  [0x08ed] = 0x24576, [0x08ee] = 0x245be, [0x08ef] = 0x3e20, [0x08f0] = 0x245bd,
  [0x08f1] = 0x24609, [0x08f2] = 0x24608, [0x08f3] = 0x24657, [0x08f4] = 0x247a3,
  [0x08f5] = 0x247bf, [0x08f6] = 0x247b8, [0x08f7] = 0x247af, [0x08f8] = 0x2479c,
  [0x08f9] = 0x247a5, [0x08fa] = 0x24772, [0x08fb] = 0x24775, [0x08fc] = 0x2479d,
  [0x08fd] = 0x24799, [0x08fe] = 0x247b9, [0x08ff] = 0x2494e, [0x0900] = 0x24939,
  [0x0901] = 0x2493b, [0x0902] = 0x24935, [0x0903] = 0x2493c, [0x0904] = 0x24955,
  [0x0905] = 0x24af0, [0x0906] = 0x24af3, [0x0907] = 0x24af4, [0x0908] = 0x24b3b,
  [0x0909] = 0x24b3c, [0x090a] = 0x24b3a, [0x090b] = 0x24b36, [0x090c] = 0x24c07,
  [0x090d] = 0x20ceb, [0x090e] = 0x24c55, [0x090f] = 0x24c50, [0x0910] = 0x24c4f,
  [0x0911] = 0x24c52, [0x0912] = 0x24c56, [0x0913] = 0x200bd, [0x0914] = 0x24c32,
  [0x0915] = 0x24d63, [0x0916] = 0x24d6b, [0x0917] = 0x24d66, [0x0918] = 0x24d57,
  [0x0919] = 0x24d5d, [0x091a] = 0x3f86, [0x091b] = 0x24d6d, [0x091c] = 0x24d61,
  [0x091d] = 0x24d69, [0x091e] = 0x24d5a, [0x091f] = 0x24d5c, [0x0920] = 0x24d62,
  [0x0921] = 0x24f2a, [0x0922] = 0x24f29, [0x0923] = 0x24f58, [0x0924] = 0x24f5a,
  [0x0925] = 0x24fd7, [0x0926] = 0x24fdb, [0x0927] = 0x24fdc, [0x0928] = 0x24fdd,
  [0x0929] = 0x24fd8, [0x092a] = 0x25054, [0x092b] = 0x2505b, [0x092c] = 0x2505c,
  [0x092d] = 0x25053, [0x092e] = 0x2504f, [0x092f] = 0x25056, [0x0930] = 0x25050,
  [0x0931] = 0x2505a, [0x0932] = 0x2506b, [0x0933] = 0x25136, [0x0934] = 0x25153,
  [0x0935] = 0x2513a, [0x0936] = 0x2513c, [0x0937] = 0x2513e, [0x0938] = 0x25149,
  [0x0939] = 0x25140, [0x093a] = 0x2f946, [0x093b] = 0x2f947, [0x093c] = 0x25364,
  [0x093d] = 0x25365, [0x093e] = 0x253b5, [0x093f] = 0x253b6, [0x0940] = 0x253b2,
  [0x0941] = 0x25448, [0x0942] = 0x2544a, [0x0943] = 0x25472, [0x0944] = 0x25469,
  [0x0945] = 0x2545a, [0x0946] = 0x2544c, [0x0947] = 0x2562c, [0x0948] = 0x25630,
  [0x0949] = 0x2564b, [0x094a] = 0x25649, [0x094b] = 0x25642, [0x094c] = 0x25644,
  [0x094d] = 0x2564a, [0x094e] = 0x2564f, [0x094f] = 0x25792, [0x0950] = 0x2f957,
  [0x0951] = 0x25797, [0x0952] = 0x25780, [0x0953] = 0x25782, [0x0954] = 0x25786,
  [0x0955] = 0x25953, [0x0956] = 0x2595e, [0x0957] = 0x25952, [0x0958] = 0x2595b,
  [0x0959] = 0x2594e, [0x095a] = 0x25a6d, [0x095b] = 0x25a6e, [0x095c] = 0x25afa,
  [0x095d] = 0x25af6, [0x095e] = 0x25afb, [0x095f] = 0x25e33, [0x0960] = 0x25e3d,
  [0x0961] = 0x25e37, [0x0962] = 0x25e3e, [0x0963] = 0x25e35, [0x0964] = 0x25f9a,
  [0x0965] = 0x25fab, [0x0966] = 0x25fa6, [0x0967] = 0x25fb0, [0x0968] = 0x25f99,
  [0x0969] = 0x25fa0, [0x096a] = 0x25f9e, [0x096b] = 0x25fa8, [0x096c] = 0x25fa1,
  [0x096d] = 0x25faa, [0x096e] = 0x25fad, [0x096f] = 0x25fbb, [0x0970] = 0x25f9c,
  [0x0971] = 0x25fa5, [0x0972] = 0x200b3, [0x0973] = 0x26227, [0x0974] = 0x2628d,
  [0x0975] = 0x2628e, [0x0976] = 0x2628f, [0x0977] = 0x26292, [0x0978] = 0x4356,
  [0x0979] = 0x26291, [0x097a] = 0x262ad, [0x097b] = 0x26357, [0x097c] = 0x26358,
  [0x097d] = 0x2635e, [0x097e] = 0x2635d, [0x097f] = 0x2635c, [0x0980] = 0x2635b,
  [0x0981] = 0x4367, [0x0982] = 0x2640a, [0x0983] = 0x26403, [0x0984] = 0x2640e,
  [0x0985] = 0x264b8, [0x0986] = 0x2654d, [0x0987] = 0x2653f, [0x0988] = 0x26547,
  [0x0989] = 0x2654b, [0x098a] = 0x2654a, [0x098b] = 0x2653d, [0x098c] = 0x8038,
  [0x098d] = 0x26541, [0x098e] = 0x2654c, [0x098f] = 0x8081, [0x0990] = 0x26662,
  [0x0991] = 0x26669, [0x0992] = 0x26661, [0x0993] = 0x43d1, [0x0994] = 0x266aa,
  [0x0995] = 0x266a6, [0x0996] = 0x266ac, [0x0997] = 0x266bd, [0x0998] = 0x266bb,
  [0x0999] = 0x266a4, [0x099a] = 0x266ba, [0x099b] = 0x2669a, [0x099c] = 0x43eb,
  [0x099d] = 0x266a1, [0x099e] = 0x266c1, [0x099f] = 0x268e0, [0x09a0] = 0x2690a,
  [0x09a1] = 0x26903, [0x09a2] = 0x26906, [0x09a3] = 0x26939, [0x09a4] = 0x2693a,
  [0x09a5] = 0x26936, [0x09a6] = 0x2695b, [0x09a7] = 0x2695f, [0x09a8] = 0x2695e,
  [0x09a9] = 0x26967, [0x09aa] = 0x26961, [0x09ab] = 0x26962, [0x09ac] = 0x2695d,
  [0x09ad] = 0x269ca, [0x09ae] = 0x269cc, [0x09af] = 0x269ce, [0x09b0] = 0x26a18,
  [0x09b1] = 0x26a1d, [0x09b2] = 0x26a1f, [0x09b3] = 0x26a20, [0x09b4] = 0x26a17,
  [0x09b5] = 0x26a15, [0x09b6] = 0x26b1f, [0x09b7] = 0x26b27, [0x09b8] = 0x26b26,
  [0x09b9] = 0x21e10, [0x09ba] = 0x26b8b, [0x09bb] = 0x26bae, [0x09bc] = 0x26b8a,
  [0x09bd] = 0x2f999, [0x09be] = 0x26b92, [0x09bf] = 0x26b89, [0x09c0] = 0x26b87,
  [0x09c1] = 0x44b6, [0x09c2] = 0x26b8f, [0x09c3] = 0x26b84, [0x09c4] = 0x26b83,
  [0x09c5] = 0x26b8c, [0x09c6] = 0x26b93, [0x09c7] = 0x26b8d, [0x09c8] = 0x26b98,
  [0x09c9] = 0x26b7d, [0x09ca] = 0x26b7e, [0x09cb] = 0x26bd2, [0x09cc] = 0x26b80,
  [0x09cd] = 0x26b99, [0x09ce] = 0x271ac, [0x09cf] = 0x27250, [0x09d0] = 0x27255,
  [0x09d1] = 0x27242, [0x09d2] = 0x2723f, [0x09d3] = 0x2723c, [0x09d4] = 0x45a3,
  [0x09d5] = 0x2724c, [0x09d6] = 0x27249, [0x09d7] = 0x27257, [0x09d8] = 0x27258,
  [0x09d9] = 0x2724f, [0x09da] = 0x2725c, [0x09db] = 0x27247, [0x09dc] = 0x2f9b7,
  [0x09dd] = 0x275ab, [0x09de] = 0x4610, [0x09df] = 0x275af, [0x09e0] = 0x275ad,
  [0x09e1] = 0x275e8, [0x09e2] = 0x275e7, [0x09e3] = 0x27630, [0x09e4] = 0x27626,
  [0x09e5] = 0x2762f, [0x09e6] = 0x27628, [0x09e7] = 0x2762b, [0x09e8] = 0x2761d,
  [0x09e9] = 0x2762d, [0x09ea] = 0x27620, [0x09eb] = 0x2762a, [0x09ec] = 0x2762c,
  [0x09ed] = 0x27635, [0x09ee] = 0x27621, [0x09ef] = 0x27623, [0x09f0] = 0x27624,
  [0x09f1] = 0x27636, [0x09f2] = 0x27637, [0x09f3] = 0x277e9, [0x09f4] = 0x278ba,
  [0x09f5] = 0x278b8, [0x09f6] = 0x2f9cc, [0x09f7] = 0x2796d, [0x09f8] = 0x2796a,
  [0x09f9] = 0x27968, [0x09fa] = 0x27969, [0x09fb] = 0x2796b, [0x09fc] = 0x27961,
  [0x09fd] = 0x27bdc, [0x09fe] = 0x27bdb, [0x09ff] = 0x27c2d, [0x0a00] = 0x27c2c,
  [0x0a01] = 0x27ca2, [0x0a02] = 0x27d2b, [0x0a03] = 0x27d32, [0x0a04] = 0x2f9d7,
  [0x0a05] = 0x27e94, [0x0a06] = 0x27e92, [0x0a07] = 0x27e90, [0x0a08] = 0x27fc9,
  [0x0a09] = 0x27fc4, [0x0a0a] = 0x27fc1, [0x0a0b] = 0x27fc3, [0x0a0c] = 0x27fcd,
  [0x0a0d] = 0x28214, [0x0a0e] = 0x282a7, [0x0a0f] = 0x282af, [0x0a10] = 0x282aa,
  [0x0a11] = 0x2840d, [0x0a12] = 0x28454, [0x0a13] = 0x2845b, [0x0a14] = 0x28461,
  [0x0a15] = 0x284ae, [0x0a16] = 0x284b3, [0x0a17] = 0x284c0, [0x0a18] = 0x284c4,
  [0x0a19] = 0x284bf, [0x0a1a] = 0x284cb, [0x0a1b] = 0x284ad, [0x0a1c] = 0x284a7,
  [0x0a1d] = 0x284a4, [0x0a1e] = 0x284bd, [0x0a1f] = 0x284af, [0x0a20] = 0x284b2,
  [0x0a21] = 0x284a5, [0x0a22] = 0x286e7, [0x0a23] = 0x286e0, [0x0a24] = 0x286ce,
  [0x0a25] = 0x286de, [0x0a26] = 0x286d5, [0x0a27] = 0x286df, [0x0a28] = 0x286d9,
  [0x0a29] = 0x287f2, [0x0a2a] = 0x28923, [0x0a2b] = 0x28940, [0x0a2c] = 0x491b,
  [0x0a2d] = 0x2893e, [0x0a2e] = 0x4987, [0x0a2f] = 0x28ccf, [0x0a30] = 0x20b54,
  [0x0a31] = 0x28e5b, [0x0a32] = 0x28e58, [0x0a33] = 0x28e62, [0x0a34] = 0x28e5f,
  [0x0a35] = 0x28e67, [0x0a36] = 0x28e63, [0x0a37] = 0x28e5e, [0x0a38] = 0x28e60,
  [0x0a39] = 0x28f85, [0x0a3a] = 0x28f86, [0x0a3b] = 0x28f87, [0x0a3c] = 0x291e8,
  [0x0a3d] = 0x291e6, [0x0a3e] = 0x29671, [0x0a3f] = 0x29685, [0x0a40] = 0x2fa02,
  [0x0a41] = 0x29686, [0x0a42] = 0x29810, [0x0a43] = 0x209f8, [0x0a44] = 0x200bf,
  [0x0a45] = 0x20b61, [0x0a46] = 0x200fe, [0x0a47] = 0x200fc, [0x0a48] = 0x20139,
  [0x0a49] = 0x20161, [0x0a4a] = 0x20160, [0x0a4b] = 0x202e2, [0x0a4c] = 0x202ea,
  [0x0a4d] = 0x202e3, [0x0a4e] = 0x202b4, [0x0a4f] = 0x202ae, [0x0a50] = 0x202be,
  [0x0a51] = 0x202b8, [0x0a52] = 0x202a8, [0x0a53] = 0x202aa, [0x0a54] = 0x202a9,
  [0x0a55] = 0x202b3, [0x0a56] = 0x202d5, [0x0a57] = 0x202ad, [0x0a58] = 0x202b9,
  [0x0a59] = 0x202bb, [0x0a5a] = 0x202b1, [0x0a5b] = 0x202c2, [0x0a5c] = 0x2f808,
  [0x0a5d] = 0x202eb, [0x0a5e] = 0x202ba, [0x0a5f] = 0x202d2, [0x0a60] = 0x202d4,
  [0x0a61] = 0x204f1, [0x0a62] = 0x2051d, [0x0a63] = 0x2f812, [0x0a64] = 0x20562,
  [0x0a65] = 0x2058c, [0x0a66] = 0x205e6, [0x0a67] = 0x205e7, [0x0a68] = 0x2065a,
  [0x0a69] = 0x20658, [0x0a6a] = 0x20659, [0x0a6b] = 0x20696, [0x0a6c] = 0x20697,
  [0x0a6d] = 0x20761, [0x0a6e] = 0x20767, [0x0a6f] = 0x20771, [0x0a70] = 0x20765,
  [0x0a71] = 0x2077d, [0x0a72] = 0x2077e, [0x0a73] = 0x2087d, [0x0a74] = 0x20884,
  [0x0a75] = 0x2087c, [0x0a76] = 0x2087e, [0x0a77] = 0x2087f, [0x0a78] = 0x20880,
  [0x0a79] = 0x208ef, [0x0a7a] = 0x208f4, [0x0a7b] = 0x2091e, [0x0a7c] = 0x2094e,
  [0x0a7d] = 0x209fb, [0x0a7e] = 0x209fa, [0x0a7f] = 0x209fd, [0x0a80] = 0x2f831,
  [0x0a81] = 0x2f832, [0x0a82] = 0x20a79, [0x0a83] = 0x20a7c, [0x0a84] = 0x20a7d,
  [0x0a85] = 0x20a84, [0x0a86] = 0x20a7b, [0x0a87] = 0x20a78, [0x0a88] = 0x3557,
  [0x0a89] = 0x20b5e, [0x0a8a] = 0x20b5a, [0x0a8b] = 0x20b5c, [0x0a8c] = 0x20b59,
  [0x0a8d] = 0x20b55, [0x0a8e] = 0x20b63, [0x0a8f] = 0x20b56, [0x0a90] = 0x20b5f,
  [0x0a91] = 0x20b60, [0x0a92] = 0x20b5b, [0x0a93] = 0x20d4a, [0x0a94] = 0x20d65,
  [0x0a95] = 0x20db3, [0x0a96] = 0x20d2c, [0x0a97] = 0x20d77, [0x0a98] = 0x20d3d,
  [0x0a99] = 0x20d52, [0x0a9a] = 0x20d61, [0x0a9b] = 0x20d2a, [0x0a9c] = 0x20d3e,
  [0x0a9d] = 0x20d34, [0x0a9e] = 0x20d29, [0x0a9f] = 0x20db2, [0x0aa0] = 0x20dad,
  [0x0aa1] = 0x20d40, [0x0aa2] = 0x20d53, [0x0aa3] = 0x2f844, [0x0aa4] = 0x20d3f,
  [0x0aa5] = 0x20d41, [0x0aa6] = 0x20d72, [0x0aa7] = 0x211f6, [0x0aa8] = 0x211f5,
  [0x0aa9] = 0x211f4, [0x0aaa] = 0x211f2, [0x0aab] = 0x211f9, [0x0aac] = 0x21327,
  [0x0aad] = 0x21354, [0x0aae] = 0x21355, [0x0aaf] = 0x2132e, [0x0ab0] = 0x2f854,
  [0x0ab1] = 0x2f853, [0x0ab2] = 0x2132c, [0x0ab3] = 0x21338, [0x0ab4] = 0x21339,
  [0x0ab5] = 0x21331, [0x0ab6] = 0x2134f, [0x0ab7] = 0x21373, [0x0ab8] = 0x21330,
  [0x0ab9] = 0x2132b, [0x0aba] = 0x21351, [0x0abb] = 0x2152c, [0x0abc] = 0x2155b,
  [0x0abd] = 0x2155c, [0x0abe] = 0x21568, [0x0abf] = 0x2156c, [0x0ac0] = 0x2156b,
  [0x0ac1] = 0x21569, [0x0ac2] = 0x2159f, [0x0ac3] = 0x21638, [0x0ac4] = 0x2163c,
  [0x0ac5] = 0x2163a, [0x0ac6] = 0x21635, [0x0ac7] = 0x369d, [0x0ac8] = 0x21639,
  [0x0ac9] = 0x21636, [0x0aca] = 0x2163b, [0x0acb] = 0x21760, [0x0acc] = 0x21761,
  [0x0acd] = 0x21763, [0x0ace] = 0x21764, [0x0acf] = 0x21794, [0x0ad0] = 0x21793,
  [0x0ad1] = 0x2175e, [0x0ad2] = 0x21768, [0x0ad3] = 0x2176a, [0x0ad4] = 0x21765,
  [0x0ad5] = 0x2f866, [0x0ad6] = 0x21790, [0x0ad7] = 0x2175f, [0x0ad8] = 0x21772,
  [0x0ad9] = 0x2f865, [0x0ada] = 0x21a3c, [0x0adb] = 0x21a27, [0x0adc] = 0x21a24,
  [0x0add] = 0x21a26, [0x0ade] = 0x21a25, [0x0adf] = 0x375f, [0x0ae0] = 0x21a28,
  [0x0ae1] = 0x21a36, [0x0ae2] = 0x21b31, [0x0ae3] = 0x21b30, [0x0ae4] = 0x21b34,
  [0x0ae5] = 0x21b81, [0x0ae6] = 0x21b7d, [0x0ae7] = 0x21b82, [0x0ae8] = 0x21b80,
  [0x0ae9] = 0x3779, [0x0aea] = 0x202d3, [0x0aeb] = 0x21bf2, [0x0aec] = 0x21c66,
  [0x0aed] = 0x21c8c, [0x0aee] = 0x21c7b, [0x0aef] = 0x21c83, [0x0af0] = 0x3798,
  [0x0af1] = 0x21c8e, [0x0af2] = 0x21c7a, [0x0af3] = 0x21c92, [0x0af4] = 0x21c91,
  [0x0af5] = 0x21c82, [0x0af6] = 0x21d1b, [0x0af7] = 0x21d1c, [0x0af8] = 0x21e27,
  [0x0af9] = 0x21e21, [0x0afa] = 0x37dc, [0x0afb] = 0x5d2b, [0x0afc] = 0x21e43,
  [0x0afd] = 0x37df, [0x0afe] = 0x21e18, [0x0aff] = 0x21e7b, [0x0b00] = 0x21e1a,
  [0x0b01] = 0x21e4b, [0x0b02] = 0x21e4a, [0x0b03] = 0x21e4d, [0x0b04] = 0x21e4f,
  [0x0b05] = 0x21e19, [0x0b06] = 0x21e35, [0x0b07] = 0x21e13, [0x0b08] = 0x21e52,
  [0x0b09] = 0x21e14, [0x0b0a] = 0x21e1e, [0x0b0b] = 0x21e2c, [0x0b0c] = 0x21e20,
  [0x0b0d] = 0x21e22, [0x0b0e] = 0x21e12, [0x0b0f] = 0x21e1f, [0x0b10] = 0x22000,
  [0x0b11] = 0x22023, [0x0b12] = 0x22040, [0x0b13] = 0x22043, [0x0b14] = 0x220e4,
  [0x0b15] = 0x220db, [0x0b16] = 0x220ea, [0x0b17] = 0x220dd, [0x0b18] = 0x220cc,
  [0x0b19] = 0x220d9, [0x0b1a] = 0x220e8, [0x0b1b] = 0x220f6, [0x0b1c] = 0x220e3,
  [0x0b1d] = 0x220d3, [0x0b1e] = 0x220da, [0x0b1f] = 0x220d6, [0x0b20] = 0x220e7,
  [0x0b21] = 0x2223a, [0x0b22] = 0x2223f, [0x0b23] = 0x22240, [0x0b24] = 0x22248,
  [0x0b25] = 0x22259, [0x0b26] = 0x22237, [0x0b27] = 0x22244, [0x0b28] = 0x2f88c,
  [0x0b29] = 0x2f88b, [0x0b2a] = 0x22255, [0x0b2b] = 0x22239, [0x0b2c] = 0x22354,
  [0x0b2d] = 0x22355, [0x0b2e] = 0x22356, [0x0b2f] = 0x22357, [0x0b30] = 0x22358,
  [0x0b31] = 0x22359, [0x0b32] = 0x2238d, [0x0b33] = 0x223f2, [0x0b34] = 0x223f8,
  [0x0b35] = 0x223f5, [0x0b36] = 0x223f6, [0x0b37] = 0x223fc, [0x0b38] = 0x223fe,
  [0x0b39] = 0x223f1, [0x0b3a] = 0x223fd, [0x0b3b] = 0x2245e, [0x0b3c] = 0x22496,
  [0x0b3d] = 0x22497, [0x0b3e] = 0x2249c, [0x0b3f] = 0x2249b, [0x0b40] = 0x22495,
  [0x0b41] = 0x2f89a, [0x0b42] = 0x2251a, [0x0b43] = 0x22509, [0x0b44] = 0x22504,
  [0x0b45] = 0x2250e, [0x0b46] = 0x2251c, [0x0b47] = 0x22518, [0x0b48] = 0x2250d,
  [0x0b49] = 0x22510, [0x0b4a] = 0x2250c, [0x0b4b] = 0x22503, [0x0b4c] = 0x2267b,
  [0x0b4d] = 0x226a6, [0x0b4e] = 0x22677, [0x0b4f] = 0x22688, [0x0b50] = 0x22674,
  [0x0b51] = 0x226da, [0x0b52] = 0x22676, [0x0b53] = 0x22678, [0x0b54] = 0x2268a,
  [0x0b55] = 0x2268f, [0x0b56] = 0x2267d, [0x0b57] = 0x22690, [0x0b58] = 0x226ed,
  [0x0b59] = 0x226d9, [0x0b5a] = 0x226d0, [0x0b5b] = 0x2271a, [0x0b5c] = 0x226d7,
  [0x0b5d] = 0x226e2, [0x0b5e] = 0x226e1, [0x0b5f] = 0x226c5, [0x0b60] = 0x226e0,
  [0x0b61] = 0x226ca, [0x0b62] = 0x22725, [0x0b63] = 0x226cc, [0x0b64] = 0x2f8b3,
  [0x0b65] = 0x229c6, [0x0b66] = 0x229c1, [0x0b67] = 0x22a4d, [0x0b68] = 0x22b4b,
  [0x0b69] = 0x22b64, [0x0b6a] = 0x22b95, [0x0b6b] = 0x22b99, [0x0b6c] = 0x2f8bc,
  [0x0b6d] = 0x22b94, [0x0b6e] = 0x22ba2, [0x0b6f] = 0x22bae, [0x0b70] = 0x22b9e,
  [0x0b71] = 0x22ba7, [0x0b72] = 0x22b86, [0x0b73] = 0x39fd, [0x0b74] = 0x22ba4,
  [0x0b75] = 0x22b91, [0x0b76] = 0x22b93, [0x0b77] = 0x2f8bb, [0x0b78] = 0x22b88,
  [0x0b79] = 0x22ecd, [0x0b7a] = 0x22eca, [0x0b7b] = 0x22f3f, [0x0b7c] = 0x22f40,
  [0x0b7d] = 0x22f46, [0x0b7e] = 0x22f41, [0x0b7f] = 0x22f45, [0x0b80] = 0x22f58,
  [0x0b81] = 0x22f3b, [0x0b82] = 0x22f48, [0x0b83] = 0x2304e, [0x0b84] = 0x23052,
  [0x0b85] = 0x2304f, [0x0b86] = 0x2307b, [0x0b87] = 0x2307a, [0x0b88] = 0x230a0,
  [0x0b89] = 0x2309f, [0x0b8a] = 0x230fb, [0x0b8b] = 0x230f7, [0x0b8c] = 0x231b8,
  [0x0b8d] = 0x231b9, [0x0b8e] = 0x231bb, [0x0b8f] = 0x231b7, [0x0b90] = 0x3af4,
  [0x0b91] = 0x231ba, [0x0b92] = 0x3aef, [0x0b93] = 0x231da, [0x0b94] = 0x231b5,
  [0x0b95] = 0x231bf, [0x0b96] = 0x231bc, [0x0b97] = 0x231c0, [0x0b98] = 0x2f8d3,
  [0x0b99] = 0x2f8d9, [0x0b9a] = 0x23375, [0x0b9b] = 0x23379, [0x0b9c] = 0x23376,
  [0x0b9d] = 0x23435, [0x0b9e] = 0x23440, [0x0b9f] = 0x234c0, [0x0ba0] = 0x23481,
  [0x0ba1] = 0x234ad, [0x0ba2] = 0x234af, [0x0ba3] = 0x234ac, [0x0ba4] = 0x2348f,
  [0x0ba5] = 0x234a8, [0x0ba6] = 0x234aa, [0x0ba7] = 0x234a9, [0x0ba8] = 0x23488,
  [0x0ba9] = 0x2347f, [0x0baa] = 0x23480, [0x0bab] = 0x234bc, [0x0bac] = 0x238ae,
  [0x0bad] = 0x238bb, [0x0bae] = 0x238bd, [0x0baf] = 0x3c31, [0x0bb0] = 0x23978,
  [0x0bb1] = 0x23974, [0x0bb2] = 0x23a0c, [0x0bb3] = 0x23a11, [0x0bb4] = 0x23a08,
  [0x0bb5] = 0x23a06, [0x0bb6] = 0x23a10, [0x0bb7] = 0x23a8f, [0x0bb8] = 0x23a90,
  [0x0bb9] = 0x23a8d, [0x0bba] = 0x23a8e, [0x0bbb] = 0x23a96, [0x0bbc] = 0x23a95,
  [0x0bbd] = 0x23b0b, [0x0bbe] = 0x23b0c, [0x0bbf] = 0x23b7c, [0x0bc0] = 0x23b73,
  [0x0bc1] = 0x23b75, [0x0bc2] = 0x23b76, [0x0bc3] = 0x23b7d, [0x0bc4] = 0x23b78,
  [0x0bc5] = 0x23b71, [0x0bc6] = 0x23c4b, [0x0bc7] = 0x23c4e, [0x0bc8] = 0x23d33,
  [0x0bc9] = 0x23d32, [0x0bca] = 0x3d03, [0x0bcb] = 0x23d91, [0x0bcc] = 0x23de7,
  [0x0bcd] = 0x23de9, [0x0bce] = 0x23da2, [0x0bcf] = 0x23d94, [0x0bd0] = 0x23d87,
  [0x0bd1] = 0x23da3, [0x0bd2] = 0x23ddd, [0x0bd3] = 0x23d7b, [0x0bd4] = 0x23d83,
  [0x0bd5] = 0x23d81, [0x0bd6] = 0x23ddf, [0x0bd7] = 0x23d7c, [0x0bd8] = 0x23de4,
  [0x0bd9] = 0x23de2, [0x0bda] = 0x23d93, [0x0bdb] = 0x23d7d, [0x0bdc] = 0x23dbf,
  [0x0bdd] = 0x23d9b, [0x0bde] = 0x23d8e, [0x0bdf] = 0x23d9f, [0x0be0] = 0x3d09,
  [0x0be1] = 0x23d8c, [0x0be2] = 0x23d7f, [0x0be3] = 0x23d9c, [0x0be4] = 0x23d84,
  [0x0be5] = 0x23d42, [0x0be6] = 0x23de6, [0x0be7] = 0x24251, [0x0be8] = 0x2424a,
  [0x0be9] = 0x24264, [0x0bea] = 0x24225, [0x0beb] = 0x2422f, [0x0bec] = 0x2422e,
  [0x0bed] = 0x2422b, [0x0bee] = 0x24228, [0x0bef] = 0x24232, [0x0bf0] = 0x2422d,
  [0x0bf1] = 0x24231, [0x0bf2] = 0x24239, [0x0bf3] = 0x2422c, [0x0bf4] = 0x24261,
  [0x0bf5] = 0x24511, [0x0bf6] = 0x24510, [0x0bf7] = 0x24512, [0x0bf8] = 0x24553,
  [0x0bf9] = 0x24555, [0x0bfa] = 0x2457b, [0x0bfb] = 0x24581, [0x0bfc] = 0x2457d,
  [0x0bfd] = 0x2457c, [0x0bfe] = 0x245c2, [0x0bff] = 0x245c5, [0x0c00] = 0x2f922,
  [0x0c01] = 0x245c4, [0x0c02] = 0x2f923, [0x0c03] = 0x2466b, [0x0c04] = 0x24668,
  [0x0c05] = 0x3e3e, [0x0c06] = 0x2465c, [0x0c07] = 0x2465d, [0x0c08] = 0x2466a,
  [0x0c09] = 0x2f924, [0x0c0a] = 0x24c76, [0x0c0b] = 0x24776, [0x0c0c] = 0x3e77,
  [0x0c0d] = 0x247c4, [0x0c0e] = 0x247cb, [0x0c0f] = 0x247c8, [0x0c10] = 0x247d4,
  [0x0c11] = 0x247d5, [0x0c12] = 0x247c9, [0x0c13] = 0x247d7, [0x0c14] = 0x3e7b,
  [0x0c15] = 0x24978, [0x0c16] = 0x2495a, [0x0c17] = 0x2495b, [0x0c18] = 0x2495c,
  [0x0c19] = 0x24956, [0x0c1a] = 0x24958, [0x0c1b] = 0x24971, [0x0c1c] = 0x269d4,
  [0x0c1d] = 0x24b40, [0x0c1e] = 0x2f933, [0x0c1f] = 0x24b3f, [0x0c20] = 0x24b43,
  [0x0c21] = 0x24b41, [0x0c22] = 0x24be2, [0x0c23] = 0x24be0, [0x0c24] = 0x24be3,
  [0x0c25] = 0x24c66, [0x0c26] = 0x24c73, [0x0c27] = 0x24c6c, [0x0c28] = 0x24c71,
  [0x0c29] = 0x24c6a, [0x0c2a] = 0x24c6d, [0x0c2b] = 0x24c6e, [0x0c2c] = 0x24c6b,
  [0x0c2d] = 0x24d8c, [0x0c2e] = 0x24d77, [0x0c2f] = 0x2f93a, [0x0c30] = 0x24d7f,
  [0x0c31] = 0x24d89, [0x0c32] = 0x24d7a, [0x0c33] = 0x24d85, [0x0c34] = 0x24d78,
  [0x0c35] = 0x24d8a, [0x0c36] = 0x24d86, [0x0c37] = 0x24f2c, [0x0c38] = 0x24f67,
  [0x0c39] = 0x24f5b, [0x0c3a] = 0x24fe5, [0x0c3b] = 0x24fe1, [0x0c3c] = 0x25061,
  [0x0c3d] = 0x25069, [0x0c3e] = 0x2506a, [0x0c3f] = 0x25165, [0x0c40] = 0x2516d,
  [0x0c41] = 0x25163, [0x0c42] = 0x25186, [0x0c43] = 0x2515c, [0x0c44] = 0x25162,
  [0x0c45] = 0x2f948, [0x0c46] = 0x25179, [0x0c47] = 0x25169, [0x0c48] = 0x25170,
  [0x0c49] = 0x25176, [0x0c4a] = 0x2515d, [0x0c4b] = 0x25187, [0x0c4c] = 0x2516e,
  [0x0c4d] = 0x25171, [0x0c4e] = 0x2517c, [0x0c4f] = 0x25173, [0x0c50] = 0x2515b,
  [0x0c51] = 0x2516b, [0x0c52] = 0x253bf, [0x0c53] = 0x253c1, [0x0c54] = 0x253bd,
  [0x0c55] = 0x253c9, [0x0c56] = 0x253bc, [0x0c57] = 0x253c2, [0x0c58] = 0x253c0,
  [0x0c59] = 0x25492, [0x0c5a] = 0x254a9, [0x0c5b] = 0x2548f, [0x0c5c] = 0x25476,
  [0x0c5d] = 0x2547b, [0x0c5e] = 0x25475, [0x0c5f] = 0x254a4, [0x0c60] = 0x25664,
  [0x0c61] = 0x2573d, [0x0c62] = 0x257af, [0x0c63] = 0x413a, [0x0c64] = 0x2f958,
  [0x0c65] = 0x2579d, [0x0c66] = 0x25799, [0x0c67] = 0x257b1, [0x0c68] = 0x25963,
  [0x0c69] = 0x25962, [0x0c6a] = 0x25964, [0x0c6b] = 0x25969, [0x0c6c] = 0x41a0,
  [0x0c6d] = 0x25a75, [0x0c6e] = 0x25a73, [0x0c6f] = 0x25a71, [0x0c70] = 0x25a74,
  [0x0c71] = 0x25b0c, [0x0c72] = 0x25b16, [0x0c73] = 0x41e5, [0x0c74] = 0x25b11,
  [0x0c75] = 0x25b1f, [0x0c76] = 0x25b1a, [0x0c77] = 0x25b0d, [0x0c78] = 0x25b1b,
  [0x0c79] = 0x25b13, [0x0c7a] = 0x25e4e, [0x0c7b] = 0x25e55, [0x0c7c] = 0x25e50,
  [0x0c7d] = 0x427d, [0x0c7e] = 0x25fd2, [0x0c7f] = 0x25fd3, [0x0c80] = 0x25fd1,
  [0x0c81] = 0x25ff1, [0x0c82] = 0x25fdc, [0x0c83] = 0x25fc8, [0x0c84] = 0x25fcc,
  [0x0c85] = 0x25fd0, [0x0c86] = 0x25fcf, [0x0c87] = 0x25fdf, [0x0c88] = 0x2622b,
  [0x0c89] = 0x2622e, [0x0c8a] = 0x2622d, [0x0c8b] = 0x2629d, [0x0c8c] = 0x2629e,
  [0x0c8d] = 0x2629f, [0x0c8e] = 0x262a9, [0x0c8f] = 0x262a0, [0x0c90] = 0x26298,
  [0x0c91] = 0x262a1, [0x0c92] = 0x262ab, [0x0c93] = 0x262af, [0x0c94] = 0x2636e,
  [0x0c95] = 0x2635f, [0x0c96] = 0x2635a, [0x0c97] = 0x436d, [0x0c98] = 0x26365,
  [0x0c99] = 0x26368, [0x0c9a] = 0x26372, [0x0c9b] = 0x26417, [0x0c9c] = 0x26416,
  [0x0c9d] = 0x26418, [0x0c9e] = 0x26419, [0x0c9f] = 0x2641a, [0x0ca0] = 0x26422,
  [0x0ca1] = 0x2641b, [0x0ca2] = 0x2641c, [0x0ca3] = 0x264be, [0x0ca4] = 0x264ee,
  [0x0ca5] = 0x2655a, [0x0ca6] = 0x26550, [0x0ca7] = 0x26558, [0x0ca8] = 0x26554,
  [0x0ca9] = 0x26557, [0x0caa] = 0x26556, [0x0cab] = 0x26615, [0x0cac] = 0x2669b,
  [0x0cad] = 0x26693, [0x0cae] = 0x26692, [0x0caf] = 0x26672, [0x0cb0] = 0x26698,
  [0x0cb1] = 0x26699, [0x0cb2] = 0x266df, [0x0cb3] = 0x43f8, [0x0cb4] = 0x266d5,
  [0x0cb5] = 0x8158, [0x0cb6] = 0x266d4, [0x0cb7] = 0x266f3, [0x0cb8] = 0x266f4,
  [0x0cb9] = 0x266e7, [0x0cba] = 0x266e1, [0x0cbb] = 0x266e6, [0x0cbc] = 0x266eb,
  [0x0cbd] = 0x266ec, [0x0cbe] = 0x202db, [0x0cbf] = 0x4450, [0x0cc0] = 0x2690d,
  [0x0cc1] = 0x2690c, [0x0cc2] = 0x21b2f, [0x0cc3] = 0x26968, [0x0cc4] = 0x26965,
  [0x0cc5] = 0x2696b, [0x0cc6] = 0x26969, [0x0cc7] = 0x269d1, [0x0cc8] = 0x269d3,
  [0x0cc9] = 0x26a27, [0x0cca] = 0x26a28, [0x0ccb] = 0x4468, [0x0ccc] = 0x4467,
  [0x0ccd] = 0x26a24, [0x0cce] = 0x26ad7, [0x0ccf] = 0x26bca, [0x0cd0] = 0x26bc7,
  [0x0cd1] = 0x2f99e, [0x0cd2] = 0x26bd8, [0x0cd3] = 0x26be1, [0x0cd4] = 0x26bc6,
  [0x0cd5] = 0x26bf8, [0x0cd6] = 0x26bc3, [0x0cd7] = 0x26bf4, [0x0cd8] = 0x26c17,
  [0x0cd9] = 0x26bea, [0x0cda] = 0x26bcb, [0x0cdb] = 0x26b86, [0x0cdc] = 0x26bc4,
  [0x0cdd] = 0x26bd9, [0x0cde] = 0x26c19, [0x0cdf] = 0x26bc2, [0x0ce0] = 0x26be2,
  [0x0ce1] = 0x44cf, [0x0ce2] = 0x26bde, [0x0ce3] = 0x26bed, [0x0ce4] = 0x2f99c,
  [0x0ce5] = 0x44cd, [0x0ce6] = 0x26c1d, [0x0ce7] = 0x26bdd, [0x0ce8] = 0x26bdb,
  [0x0ce9] = 0x26be8, [0x0cea] = 0x26be9, [0x0ceb] = 0x26bbf, [0x0cec] = 0x26be0,
  [0x0ced] = 0x26bd1, [0x0cee] = 0x26bdc, [0x0cef] = 0x26bce, [0x0cf0] = 0x26c1e,
  [0x0cf1] = 0x26bcc, [0x0cf2] = 0x26bf2, [0x0cf3] = 0x26bf3, [0x0cf4] = 0x2f99f,
  [0x0cf5] = 0x2f9a2, [0x0cf6] = 0x2f9a3, [0x0cf7] = 0x2f9a1, [0x0cf8] = 0x271ab,
  [0x0cf9] = 0x271aa, [0x0cfa] = 0x271a7, [0x0cfb] = 0x271a9, [0x0cfc] = 0x27288,
  [0x0cfd] = 0x27275, [0x0cfe] = 0x27260, [0x0cff] = 0x2726c, [0x0d00] = 0x27273,
  [0x0d01] = 0x45ab, [0x0d02] = 0x2726e, [0x0d03] = 0x27274, [0x0d04] = 0x27276,
  [0x0d05] = 0x27268, [0x0d06] = 0x27277, [0x0d07] = 0x27286, [0x0d08] = 0x275ea,
  [0x0d09] = 0x27651, [0x0d0a] = 0x27650, [0x0d0b] = 0x27658, [0x0d0c] = 0x2764d,
  [0x0d0d] = 0x2764f, [0x0d0e] = 0x2764e, [0x0d0f] = 0x2765c, [0x0d10] = 0x27652,
  [0x0d11] = 0x27644, [0x0d12] = 0x2764a, [0x0d13] = 0x2764b, [0x0d14] = 0x277ea,
  [0x0d15] = 0x27810, [0x0d16] = 0x27811, [0x0d17] = 0x278cd, [0x0d18] = 0x278bf,
  [0x0d19] = 0x278c4, [0x0d1a] = 0x278d0, [0x0d1b] = 0x278ce, [0x0d1c] = 0x278c0,
  [0x0d1d] = 0x278c2, [0x0d1e] = 0x278cf, [0x0d1f] = 0x278c9, [0x0d20] = 0x278bb,
  [0x0d21] = 0x27997, [0x0d22] = 0x27992, [0x0d23] = 0x2796f, [0x0d24] = 0x2797e,
  [0x0d25] = 0x2799a, [0x0d26] = 0x46c1, [0x0d27] = 0x27986, [0x0d28] = 0x27973,
  [0x0d29] = 0x8a24, [0x0d2a] = 0x27977, [0x0d2b] = 0x2798f, [0x0d2c] = 0x27970,
  [0x0d2d] = 0x27981, [0x0d2e] = 0x27982, [0x0d2f] = 0x27999, [0x0d30] = 0x2797d,
  [0x0d31] = 0x2797f, [0x0d32] = 0x2797b, [0x0d33] = 0x27987, [0x0d34] = 0x27bb2,
  [0x0d35] = 0x27c2f, [0x0d36] = 0x27c34, [0x0d37] = 0x27c2e, [0x0d38] = 0x27c32,
  [0x0d39] = 0x4747, [0x0d3a] = 0x27caa, [0x0d3b] = 0x4757, [0x0d3c] = 0x27ca9,
  [0x0d3d] = 0x27d38, [0x0d3e] = 0x27d36, [0x0d3f] = 0x27d37, [0x0d40] = 0x27d47,
  [0x0d41] = 0x27d33, [0x0d42] = 0x27d39, [0x0d43] = 0x27d35, [0x0d44] = 0x27d44,
  [0x0d45] = 0x27ea5, [0x0d46] = 0x27ea6, [0x0d47] = 0x27e9e, [0x0d48] = 0x27fe3,
  [0x0d49] = 0x27fdf, [0x0d4a] = 0x27fd3, [0x0d4b] = 0x27ff1, [0x0d4c] = 0x27fe4,
  [0x0d4d] = 0x27fe0, [0x0d4e] = 0x27fd6, [0x0d4f] = 0x8dc3, [0x0d50] = 0x27fe6,
  [0x0d51] = 0x27fd8, [0x0d52] = 0x27fde, [0x0d53] = 0x27fdb, [0x0d54] = 0x27fdc,
  [0x0d55] = 0x2821b, [0x0d56] = 0x2821f, [0x0d57] = 0x2821d, [0x0d58] = 0x2821c,
  [0x0d59] = 0x2821e, [0x0d5a] = 0x28220, [0x0d5b] = 0x28221, [0x0d5c] = 0x484e,
  [0x0d5d] = 0x282b6, [0x0d5e] = 0x282bc, [0x0d5f] = 0x282c6, [0x0d60] = 0x282c7,
  [0x0d61] = 0x282ba, [0x0d62] = 0x282be, [0x0d63] = 0x282bd, [0x0d64] = 0x282b5,
  [0x0d65] = 0x282b4, [0x0d66] = 0x2840f, [0x0d67] = 0x28462, [0x0d68] = 0x284eb,
  [0x0d69] = 0x284d9, [0x0d6a] = 0x284e4, [0x0d6b] = 0x284d7, [0x0d6c] = 0x284d8,
  [0x0d6d] = 0x284d6, [0x0d6e] = 0x284ce, [0x0d6f] = 0x284dd, [0x0d70] = 0x284e7,
  [0x0d71] = 0x284d2, [0x0d72] = 0x284c5, [0x0d73] = 0x284c9, [0x0d74] = 0x284db,
  [0x0d75] = 0x28692, [0x0d76] = 0x2868a, [0x0d77] = 0x2868b, [0x0d78] = 0x28689,
  [0x0d79] = 0x28708, [0x0d7a] = 0x28703, [0x0d7b] = 0x28706, [0x0d7c] = 0x28705,
  [0x0d7d] = 0x286f5, [0x0d7e] = 0x2870b, [0x0d7f] = 0x286fb, [0x0d80] = 0x287fc,
  [0x0d81] = 0x28801, [0x0d82] = 0x28802, [0x0d83] = 0x287fa, [0x0d84] = 0x28808,
  [0x0d85] = 0x287f7, [0x0d86] = 0x28800, [0x0d87] = 0x287ff, [0x0d88] = 0x28806,
  [0x0d89] = 0x287f6, [0x0d8a] = 0x287fb, [0x0d8b] = 0x2880a, [0x0d8c] = 0x28925,
  [0x0d8d] = 0x28943, [0x0d8e] = 0x28944, [0x0d8f] = 0x28c64, [0x0d90] = 0x28c62,
  [0x0d91] = 0x28c5e, [0x0d92] = 0x28c5b, [0x0d93] = 0x28cd6, [0x0d94] = 0x20072,
  [0x0d95] = 0x200c2, [0x0d96] = 0x28e77, [0x0d97] = 0x28e82, [0x0d98] = 0x28e7c,
  [0x0d99] = 0x28e7d, [0x0d9a] = 0x28e86, [0x0d9b] = 0x28e81, [0x0d9c] = 0x28e84,
  [0x0d9d] = 0x28e76, [0x0d9e] = 0x28e83, [0x0d9f] = 0x28e7f, [0x0da0] = 0x28e7e,
  [0x0da1] = 0x28f88, [0x0da2] = 0x28f8d, [0x0da3] = 0x28f8b, [0x0da4] = 0x28f91,
  [0x0da5] = 0x28f8f, [0x0da6] = 0x2907c, [0x0da7] = 0x29079, [0x0da8] = 0x2907a,
  [0x0da9] = 0x291ea, [0x0daa] = 0x2944c, [0x0dab] = 0x29599, [0x0dac] = 0x202d1,
  [0x0dad] = 0x2968b, [0x0dae] = 0x2968d, [0x0daf] = 0x2968a, [0x0db0] = 0x2968e,
  [0x0db1] = 0x29811, [0x0db2] = 0x22256, [0x0db3] = 0x2fa15, [0x0db4] = 0x20074,
  [0x0db5] = 0x200c3, [0x0db6] = 0x200c4, [0x0db7] = 0x2011b, [0x0db8] = 0x2015f,
  [0x0db9] = 0x2016a, [0x0dba] = 0x20169, [0x0dbb] = 0x2016b, [0x0dbc] = 0x2030c,
  [0x0dbd] = 0x202f6, [0x0dbe] = 0x202ed, [0x0dbf] = 0x20329, [0x0dc0] = 0x202fe,
  [0x0dc1] = 0x202f1, [0x0dc2] = 0x20317, [0x0dc3] = 0x202ff, [0x0dc4] = 0x202ee,
  [0x0dc5] = 0x202fd, [0x0dc6] = 0x2031c, [0x0dc7] = 0x202fc, [0x0dc8] = 0x20300,
  [0x0dc9] = 0x20320, [0x0dca] = 0x3477, [0x0dcb] = 0x202f9, [0x0dcc] = 0x20367,
  [0x0dcd] = 0x20308, [0x0dce] = 0x2049e, [0x0dcf] = 0x204f3, [0x0dd0] = 0x20525,
  [0x0dd1] = 0x20527, [0x0dd2] = 0x2051f, [0x0dd3] = 0x20565, [0x0dd4] = 0x20563,
  [0x0dd5] = 0x20594, [0x0dd6] = 0x20597, [0x0dd7] = 0x205f1, [0x0dd8] = 0x2065f,
  [0x0dd9] = 0x20662, [0x0dda] = 0x51f2, [0x0ddb] = 0x20660, [0x0ddc] = 0x2f822,
  [0x0ddd] = 0x2078c, [0x0dde] = 0x20782, [0x0ddf] = 0x20790, [0x0de0] = 0x2078b,
  [0x0de1] = 0x2078d, [0x0de2] = 0x20781, [0x0de3] = 0x2079d, [0x0de4] = 0x2088e,
  [0x0de5] = 0x2088f, [0x0de6] = 0x20892, [0x0de7] = 0x20923, [0x0de8] = 0x20952,
  [0x0de9] = 0x2f82e, [0x0dea] = 0x20a00, [0x0deb] = 0x20a01, [0x0dec] = 0x20a02,
  [0x0ded] = 0x20a1f, [0x0dee] = 0x20a8c, [0x0def] = 0x20a89, [0x0df0] = 0x20a8b,
  [0x0df1] = 0x20a88, [0x0df2] = 0x20a8d, [0x0df3] = 0x20a8f, [0x0df4] = 0x26385,
  [0x0df5] = 0x20b00, [0x0df6] = 0x20b05, [0x0df7] = 0x20b01, [0x0df8] = 0x20b68,
  [0x0df9] = 0x20b6e, [0x0dfa] = 0x20b67, [0x0dfb] = 0x20b75, [0x0dfc] = 0x55b6,
  [0x0dfd] = 0x2f845, [0x0dfe] = 0x20dd7, [0x0dff] = 0x2f846, [0x0e00] = 0x20048,
  [0x0e01] = 0x20dd4, [0x0e02] = 0x20dd8, [0x0e03] = 0x2f847, [0x0e04] = 0x2f848,
  [0x0e05] = 0x20dba, [0x0e06] = 0x2f849, [0x0e07] = 0x20ddb, [0x0e08] = 0x20dbf,
  [0x0e09] = 0x20e35, [0x0e0a] = 0x20dbc, [0x0e0b] = 0x20dd9, [0x0e0c] = 0x35c4,
  [0x0e0d] = 0x20ddd, [0x0e0e] = 0x20e00, [0x0e0f] = 0x20dd5, [0x0e10] = 0x20e30,
  [0x0e11] = 0x20dbd, [0x0e12] = 0x20ddc, [0x0e13] = 0x211fd, [0x0e14] = 0x211fe,
  [0x0e15] = 0x21207, [0x0e16] = 0x24517, [0x0e17] = 0x2136f, [0x0e18] = 0x21369,
  [0x0e19] = 0x21370, [0x0e1a] = 0x21367, [0x0e1b] = 0x213a9, [0x0e1c] = 0x21395,
  [0x0e1d] = 0x21390, [0x0e1e] = 0x2136c, [0x0e1f] = 0x21397, [0x0e20] = 0x21371,
  [0x0e21] = 0x3652, [0x0e22] = 0x21374, [0x0e23] = 0x2136d, [0x0e24] = 0x2138e,
  [0x0e25] = 0x2152f, [0x0e26] = 0x2a31b, [0x0e27] = 0x215a9, [0x0e28] = 0x2164e,
  [0x0e29] = 0x2f85f, [0x0e2a] = 0x2165a, [0x0e2b] = 0x21648, [0x0e2c] = 0x21655,
  [0x0e2d] = 0x2164c, [0x0e2e] = 0x21649, [0x0e2f] = 0x2164f, [0x0e30] = 0x2164a,
  [0x0e31] = 0x217d6, [0x0e32] = 0x217a0, [0x0e33] = 0x217a9, [0x0e34] = 0x2f867,
  [0x0e35] = 0x2179d, [0x0e36] = 0x217d4, [0x0e37] = 0x217a4, [0x0e38] = 0x217a8,
  [0x0e39] = 0x217a6, [0x0e3a] = 0x217e6, [0x0e3b] = 0x2197d, [0x0e3c] = 0x21977,
  [0x0e3d] = 0x2197a, [0x0e3e] = 0x21a41, [0x0e3f] = 0x21a49, [0x0e40] = 0x21a59,
  [0x0e41] = 0x21a45, [0x0e42] = 0x21a48, [0x0e43] = 0x21a40, [0x0e44] = 0x21b8e,
  [0x0e45] = 0x21b95, [0x0e46] = 0x21b90, [0x0e47] = 0x21bf7, [0x0e48] = 0x21bf8,
  [0x0e49] = 0x21bf6, [0x0e4a] = 0x21bfb, [0x0e4b] = 0x21c9e, [0x0e4c] = 0x21c9d,
  [0x0e4d] = 0x21c99, [0x0e4e] = 0x2f877, [0x0e4f] = 0x21ca3, [0x0e50] = 0x21ca9,
  [0x0e51] = 0x21c98, [0x0e52] = 0x21ca0, [0x0e53] = 0x21c96, [0x0e54] = 0x21c94,
  [0x0e55] = 0x21c95, [0x0e56] = 0x21c9f, [0x0e57] = 0x21ca1, [0x0e58] = 0x21d21,
  [0x0e59] = 0x21d1d, [0x0e5a] = 0x21d1f, [0x0e5b] = 0x21e6d, [0x0e5c] = 0x2f87c,
  [0x0e5d] = 0x21e9a, [0x0e5e] = 0x21e92, [0x0e5f] = 0x21e7a, [0x0e60] = 0x21e7d,
  [0x0e61] = 0x21ea1, [0x0e62] = 0x21e9d, [0x0e63] = 0x21e99, [0x0e64] = 0x21e6b,
  [0x0e65] = 0x21e6e, [0x0e66] = 0x2f87d, [0x0e67] = 0x22045, [0x0e68] = 0x2f884,
  [0x0e69] = 0x220fb, [0x0e6a] = 0x220eb, [0x0e6b] = 0x220f2, [0x0e6c] = 0x220f9,
  [0x0e6d] = 0x2f886, [0x0e6e] = 0x220f8, [0x0e6f] = 0x220ed, [0x0e70] = 0x22101,
  [0x0e71] = 0x220f7, [0x0e72] = 0x22106, [0x0e73] = 0x2219b, [0x0e74] = 0x221ca,
  [0x0e75] = 0x386b, [0x0e76] = 0x5ec3, [0x0e77] = 0x2226b, [0x0e78] = 0x2226c,
  [0x0e79] = 0x22274, [0x0e7a] = 0x22267, [0x0e7b] = 0x2225b, [0x0e7c] = 0x22260,
  [0x0e7d] = 0x22276, [0x0e7e] = 0x22263, [0x0e7f] = 0x22261, [0x0e80] = 0x22328,
  [0x0e81] = 0x2235b, [0x0e82] = 0x2235e, [0x0e83] = 0x22360, [0x0e84] = 0x2235d,
  [0x0e85] = 0x2235c, [0x0e86] = 0x223f4, [0x0e87] = 0x22400, [0x0e88] = 0x22408,
  [0x0e89] = 0x22407, [0x0e8a] = 0x22401, [0x0e8b] = 0x22405, [0x0e8c] = 0x22464,
  [0x0e8d] = 0x22463, [0x0e8e] = 0x2249e, [0x0e8f] = 0x224a0, [0x0e90] = 0x224a1,
  [0x0e91] = 0x2249f, [0x0e92] = 0x22526, [0x0e93] = 0x2252d, [0x0e94] = 0x22528,
  [0x0e95] = 0x2251d, [0x0e96] = 0x226ec, [0x0e97] = 0x226e3, [0x0e98] = 0x226eb,
  [0x0e99] = 0x22716, [0x0e9a] = 0x226c8, [0x0e9b] = 0x2661b, [0x0e9c] = 0x226e9,
  [0x0e9d] = 0x226e4, [0x0e9e] = 0x22724, [0x0e9f] = 0x226d1, [0x0ea0] = 0x3941,
  [0x0ea1] = 0x226dd, [0x0ea2] = 0x226c2, [0x0ea3] = 0x226cb, [0x0ea4] = 0x226c7,
  [0x0ea5] = 0x226e7, [0x0ea6] = 0x3950, [0x0ea7] = 0x226ea, [0x0ea8] = 0x2274b,
  [0x0ea9] = 0x2f8a5, [0x0eaa] = 0x22760, [0x0eab] = 0x2277d, [0x0eac] = 0x2273e,
  [0x0ead] = 0x2f8a4, [0x0eae] = 0x22752, [0x0eaf] = 0x2274e, [0x0eb0] = 0x2273c,
  [0x0eb1] = 0x22732, [0x0eb2] = 0x22730, [0x0eb3] = 0x22723, [0x0eb4] = 0x229ca,
  [0x0eb5] = 0x229cb, [0x0eb6] = 0x229c9, [0x0eb7] = 0x229c8, [0x0eb8] = 0x229cd,
  [0x0eb9] = 0x22b98, [0x0eba] = 0x22ba0, [0x0ebb] = 0x22b9f, [0x0ebc] = 0x22b9c,
  [0x0ebd] = 0x22ba3, [0x0ebe] = 0x22b97, [0x0ebf] = 0x2f8be, [0x0ec0] = 0x22bf1,
  [0x0ec1] = 0x22c09, [0x0ec2] = 0x22c03, [0x0ec3] = 0x22bea, [0x0ec4] = 0x22c45,
  [0x0ec5] = 0x22bed, [0x0ec6] = 0x22c05, [0x0ec7] = 0x22c1a, [0x0ec8] = 0x22c15,
  [0x0ec9] = 0x22c01, [0x0eca] = 0x22bec, [0x0ecb] = 0x22c0e, [0x0ecc] = 0x22c17,
  [0x0ecd] = 0x22c42, [0x0ece] = 0x22c12, [0x0ecf] = 0x22c10, [0x0ed0] = 0x22bef,
  [0x0ed1] = 0x22bff, [0x0ed2] = 0x22c00, [0x0ed3] = 0x22c0c, [0x0ed4] = 0x22c0f,
  [0x0ed5] = 0x22c04, [0x0ed6] = 0x22c08, [0x0ed7] = 0x22c14, [0x0ed8] = 0x22c43,
  [0x0ed9] = 0x2f8bd, [0x0eda] = 0x22c1b, [0x0edb] = 0x22c11, [0x0edc] = 0x22c13,
  [0x0edd] = 0x22ecf, [0x0ede] = 0x22ece, [0x0edf] = 0x22f6f, [0x0ee0] = 0x22f6e,
  [0x0ee1] = 0x22f7a, [0x0ee2] = 0x22f70, [0x0ee3] = 0x22f64, [0x0ee4] = 0x22f5e,
  [0x0ee5] = 0x22f6c, [0x0ee6] = 0x2f8c9, [0x0ee7] = 0x22f5b, [0x0ee8] = 0x22f61,
  [0x0ee9] = 0x22f65, [0x0eea] = 0x2307f, [0x0eeb] = 0x23080, [0x0eec] = 0x2307c,
  [0x0eed] = 0x230a7, [0x0eee] = 0x230a6, [0x0eef] = 0x230a1, [0x0ef0] = 0x3ac0,
  [0x0ef1] = 0x230a8, [0x0ef2] = 0x230a3, [0x0ef3] = 0x230a2, [0x0ef4] = 0x230ad,
  [0x0ef5] = 0x230a5, [0x0ef6] = 0x23101, [0x0ef7] = 0x230ff, [0x0ef8] = 0x230fc,
  [0x0ef9] = 0x23100, [0x0efa] = 0x23135, [0x0efb] = 0x231ee, [0x0efc] = 0x231ef,
  [0x0efd] = 0x231f6, [0x0efe] = 0x231e8, [0x0eff] = 0x231ea, [0x0f00] = 0x231e3,
  [0x0f01] = 0x2321f, [0x0f02] = 0x3afc, [0x0f03] = 0x231e4, [0x0f04] = 0x231fa,
  [0x0f05] = 0x231f1, [0x0f06] = 0x231fb, [0x0f07] = 0x2f8d4, [0x0f08] = 0x2333d,
  [0x0f09] = 0x2333c, [0x0f0a] = 0x3479, [0x0f0b] = 0x23378, [0x0f0c] = 0x23377,
  [0x0f0d] = 0x440b, [0x0f0e] = 0x234d1, [0x0f0f] = 0x234c7, [0x0f10] = 0x234df,
  [0x0f11] = 0x234d0, [0x0f12] = 0x234e0, [0x0f13] = 0x234d6, [0x0f14] = 0x234d8,
  [0x0f15] = 0x23516, [0x0f16] = 0x2350e, [0x0f17] = 0x234d9, [0x0f18] = 0x2350f,
  [0x0f19] = 0x23511, [0x0f1a] = 0x234cd, [0x0f1b] = 0x23489, [0x0f1c] = 0x234ce,
  [0x0f1d] = 0x23514, [0x0f1e] = 0x234da, [0x0f1f] = 0x23512, [0x0f20] = 0x234d3,
  [0x0f21] = 0x234c2, [0x0f22] = 0x234e1, [0x0f23] = 0x234e9, [0x0f24] = 0x234ea,
  [0x0f25] = 0x234de, [0x0f26] = 0x23515, [0x0f27] = 0x238d9, [0x0f28] = 0x238d6,
  [0x0f29] = 0x238cd, [0x0f2a] = 0x238d0, [0x0f2b] = 0x238d3, [0x0f2c] = 0x2f8f0,
  [0x0f2d] = 0x23982, [0x0f2e] = 0x23985, [0x0f2f] = 0x2397f, [0x0f30] = 0x2397d,
  [0x0f31] = 0x23981, [0x0f32] = 0x23983, [0x0f33] = 0x23984, [0x0f34] = 0x284a0,
  [0x0f35] = 0x23a28, [0x0f36] = 0x23a0f, [0x0f37] = 0x23a17, [0x0f38] = 0x23a1a,
  [0x0f39] = 0x23a25, [0x0f3a] = 0x2f8f7, [0x0f3b] = 0x23a9b, [0x0f3c] = 0x23a99,
  [0x0f3d] = 0x23b92, [0x0f3e] = 0x23b8c, [0x0f3f] = 0x23b95, [0x0f40] = 0x23b8d,
  [0x0f41] = 0x23ba3, [0x0f42] = 0x23b93, [0x0f43] = 0x23b91, [0x0f44] = 0x23ddb,
  [0x0f45] = 0x23d99, [0x0f46] = 0x23d9a, [0x0f47] = 0x23e08, [0x0f48] = 0x23e4c,
  [0x0f49] = 0x23e0d, [0x0f4a] = 0x23e01, [0x0f4b] = 0x23e4e, [0x0f4c] = 0x23e02,
  [0x0f4d] = 0x23e4d, [0x0f4e] = 0x23e21, [0x0f4f] = 0x23dfc, [0x0f50] = 0x2f909,
  [0x0f51] = 0x23d8a, [0x0f52] = 0x2f908, [0x0f53] = 0x23d95, [0x0f54] = 0x23e11,
  [0x0f55] = 0x23e12, [0x0f56] = 0x23e46, [0x0f57] = 0x23e1c, [0x0f58] = 0x23e49,
  [0x0f59] = 0x23e0c, [0x0f5a] = 0x3d1e, [0x0f5b] = 0x23e13, [0x0f5c] = 0x23e16,
  [0x0f5d] = 0x23dfd, [0x0f5e] = 0x23e0f, [0x0f5f] = 0x23e1f, [0x0f60] = 0x24230,
  [0x0f61] = 0x2426e, [0x0f62] = 0x2426b, [0x0f63] = 0x2429b, [0x0f64] = 0x2427b,
  [0x0f65] = 0x24263, [0x0f66] = 0x24297, [0x0f67] = 0x2426d, [0x0f68] = 0x2429c,
  [0x0f69] = 0x24298, [0x0f6a] = 0x2426f, [0x0f6b] = 0x24267, [0x0f6c] = 0x24269,
  [0x0f6d] = 0x24515, [0x0f6e] = 0x24563, [0x0f6f] = 0x24586, [0x0f70] = 0x2458a,
  [0x0f71] = 0x24587, [0x0f72] = 0x24588, [0x0f73] = 0x24585, [0x0f74] = 0x24589,
  [0x0f75] = 0x245ca, [0x0f76] = 0x245c7, [0x0f77] = 0x245cb, [0x0f78] = 0x245cc,
  [0x0f79] = 0x245c9, [0x0f7a] = 0x2460d, [0x0f7b] = 0x24683, [0x0f7c] = 0x24684,
  [0x0f7d] = 0x24678, [0x0f7e] = 0x24682, [0x0f7f] = 0x24673, [0x0f80] = 0x24679,
  [0x0f81] = 0x2468c, [0x0f82] = 0x247d0, [0x0f83] = 0x247cf, [0x0f84] = 0x247d2,
  [0x0f85] = 0x247d9, [0x0f86] = 0x247cc, [0x0f87] = 0x247eb, [0x0f88] = 0x247fd,
  [0x0f89] = 0x247ec, [0x0f8a] = 0x247e8, [0x0f8b] = 0x247f8, [0x0f8c] = 0x247fa,
  [0x0f8d] = 0x2f927, [0x0f8e] = 0x3e85, [0x0f8f] = 0x247e1, [0x0f90] = 0x247fb,
  [0x0f91] = 0x248e7, [0x0f92] = 0x2f92e, [0x0f93] = 0x249a5, [0x0f94] = 0x24991,
  [0x0f95] = 0x249a6, [0x0f96] = 0x2497c, [0x0f97] = 0x24992, [0x0f98] = 0x249a2,
  [0x0f99] = 0x249a0, [0x0f9a] = 0x24afc, [0x0f9b] = 0x24afe, [0x0f9c] = 0x24b57,
  [0x0f9d] = 0x24b53, [0x0f9e] = 0x24b58, [0x0f9f] = 0x24be7, [0x0fa0] = 0x24c8e,
  [0x0fa1] = 0x2f938, [0x0fa2] = 0x24c83, [0x0fa3] = 0x24c8b, [0x0fa4] = 0x24c84,
  [0x0fa5] = 0x3f97, [0x0fa6] = 0x24da2, [0x0fa7] = 0x24db7, [0x0fa8] = 0x24da9,
  [0x0fa9] = 0x24da5, [0x0faa] = 0x24d9f, [0x0fab] = 0x24daa, [0x0fac] = 0x24d97,
  [0x0fad] = 0x24da1, [0x0fae] = 0x24d9e, [0x0faf] = 0x24dab, [0x0fb0] = 0x24d99,
  [0x0fb1] = 0x24da3, [0x0fb2] = 0x24f30, [0x0fb3] = 0x24f32, [0x0fb4] = 0x24f2f,
  [0x0fb5] = 0x24f70, [0x0fb6] = 0x24f6c, [0x0fb7] = 0x24f6f, [0x0fb8] = 0x24fe8,
  [0x0fb9] = 0x24fee, [0x0fba] = 0x24fea, [0x0fbb] = 0x2506d, [0x0fbc] = 0x25076,
  [0x0fbd] = 0x25070, [0x0fbe] = 0x25071, [0x0fbf] = 0x2506c, [0x0fc0] = 0x251eb,
  [0x0fc1] = 0x251b5, [0x0fc2] = 0x25196, [0x0fc3] = 0x25190, [0x0fc4] = 0x2518d,
  [0x0fc5] = 0x2f949, [0x0fc6] = 0x251a2, [0x0fc7] = 0x251b0, [0x0fc8] = 0x25192,
  [0x0fc9] = 0x251a0, [0x0fca] = 0x25193, [0x0fcb] = 0x251c3, [0x0fcc] = 0x2518e,
  [0x0fcd] = 0x251b6, [0x0fce] = 0x2519d, [0x0fcf] = 0x25195, [0x0fd0] = 0x251b3,
  [0x0fd1] = 0x251a4, [0x0fd2] = 0x25370, [0x0fd3] = 0x253d4, [0x0fd4] = 0x4093,
  [0x0fd5] = 0x253cf, [0x0fd6] = 0x20170, [0x0fd7] = 0x25494, [0x0fd8] = 0x7861,
  [0x0fd9] = 0x254c0, [0x0fda] = 0x254b4, [0x0fdb] = 0x254c1, [0x0fdc] = 0x40b2,
  [0x0fdd] = 0x254bd, [0x0fde] = 0x254af, [0x0fdf] = 0x25677, [0x0fe0] = 0x25678,
  [0x0fe1] = 0x2566b, [0x0fe2] = 0x2566d, [0x0fe3] = 0x2566e, [0x0fe4] = 0x25672,
  [0x0fe5] = 0x2566f, [0x0fe6] = 0x25671, [0x0fe7] = 0x230ab, [0x0fe8] = 0x2568c,
  [0x0fe9] = 0x2573f, [0x0fea] = 0x257b6, [0x0feb] = 0x257b7, [0x0fec] = 0x7a06,
  [0x0fed] = 0x257bb, [0x0fee] = 0x257b8, [0x0fef] = 0x2079c, [0x0ff0] = 0x257b9,
  [0x0ff1] = 0x7a01, [0x0ff2] = 0x2598b, [0x0ff3] = 0x2597a, [0x0ff4] = 0x25984,
  [0x0ff5] = 0x25988, [0x0ff6] = 0x25991, [0x0ff7] = 0x25979, [0x0ff8] = 0x2598e,
  [0x0ff9] = 0x25980, [0x0ffa] = 0x25982, [0x0ffb] = 0x2597c, [0x0ffc] = 0x41a4,
  [0x0ffd] = 0x25a84, [0x0ffe] = 0x25a7f, [0x0fff] = 0x41f2, [0x1000] = 0x25b3b,
  [0x1001] = 0x25b71, [0x1002] = 0x25b3d, [0x1003] = 0x25b30, [0x1004] = 0x41f1,
  [0x1005] = 0x25b3e, [0x1006] = 0x25b38, [0x1007] = 0x25e5e, [0x1008] = 0x25e64,
  [0x1009] = 0x4281, [0x100a] = 0x25e5d, [0x100b] = 0x25e6d, [0x100c] = 0x25e4f,
  [0x100d] = 0x25e62, [0x100e] = 0x25e5f, [0x100f] = 0x25fec, [0x1010] = 0x25ff2,
  [0x1011] = 0x25ff4, [0x1012] = 0x25ff7, [0x1013] = 0x25ff6, [0x1014] = 0x26007,
  [0x1015] = 0x25fed, [0x1016] = 0x25fea, [0x1017] = 0x25ff0, [0x1018] = 0x25ff8,
  [0x1019] = 0x25ff3, [0x101a] = 0x42dc, [0x101b] = 0x2f972, [0x101c] = 0x26231,
  [0x101d] = 0x2622f, [0x101e] = 0x262b6, [0x101f] = 0x262ae, [0x1020] = 0x262aa,
  [0x1021] = 0x262bf, [0x1022] = 0x262cc, [0x1023] = 0x262c7, [0x1024] = 0x26c83,
  [0x1025] = 0x26380, [0x1026] = 0x2637b, [0x1027] = 0x2637f, [0x1028] = 0x2637d,
  [0x1029] = 0x26383, [0x102a] = 0x2f978, [0x102b] = 0x26446, [0x102c] = 0x2642d,
  [0x102d] = 0x26425, [0x102e] = 0x26426, [0x102f] = 0x2642c, [0x1030] = 0x26437,
  [0x1031] = 0x26431, [0x1032] = 0x26433, [0x1033] = 0x26432, [0x1034] = 0x26427,
  [0x1035] = 0x2642a, [0x1036] = 0x2642e, [0x1037] = 0x2642f, [0x1038] = 0x264c0,
  [0x1039] = 0x26571, [0x103a] = 0x26561, [0x103b] = 0x26562, [0x103c] = 0x26566,
  [0x103d] = 0x26618, [0x103e] = 0x266c5, [0x103f] = 0x266c3, [0x1040] = 0x266c4,
  [0x1041] = 0x266c2, [0x1042] = 0x266ae, [0x1043] = 0x26710, [0x1044] = 0x26708,
  [0x1045] = 0x2671f, [0x1046] = 0x2673a, [0x1047] = 0x2673b, [0x1048] = 0x26736,
  [0x1049] = 0x2670c, [0x104a] = 0x26706, [0x104b] = 0x2672a, [0x104c] = 0x26757,
  [0x104d] = 0x26750, [0x104e] = 0x26720, [0x104f] = 0x2f985, [0x1050] = 0x2672c,
  [0x1051] = 0x26721, [0x1052] = 0x2670b, [0x1053] = 0x26719, [0x1054] = 0x26735,
  [0x1055] = 0x26718, [0x1056] = 0x2670f, [0x1057] = 0x26713, [0x1058] = 0x26755,
  [0x1059] = 0x26739, [0x105a] = 0x2671a, [0x105b] = 0x440c, [0x105c] = 0x26717,
  [0x105d] = 0x268e4, [0x105e] = 0x268e9, [0x105f] = 0x2458b, [0x1060] = 0x26910,
  [0x1061] = 0x26912, [0x1062] = 0x2693f, [0x1063] = 0x2696e, [0x1064] = 0x24518,
  [0x1065] = 0x2f98c, [0x1066] = 0x269d6, [0x1067] = 0x269d5, [0x1068] = 0x269d7,
  [0x1069] = 0x20b02, [0x106a] = 0x446d, [0x106b] = 0x26a31, [0x106c] = 0x26a30,
  [0x106d] = 0x26a3a, [0x106e] = 0x26b85, [0x106f] = 0x26b6b, [0x1070] = 0x26c48,
  [0x1071] = 0x26c4b, [0x1072] = 0x26c37, [0x1073] = 0x26c7b, [0x1074] = 0x26c6c,
  [0x1075] = 0x26c85, [0x1076] = 0x26c65, [0x1077] = 0x26c36, [0x1078] = 0x26c86,
  [0x1079] = 0x26c34, [0x107a] = 0x26c68, [0x107b] = 0x26c5a, [0x107c] = 0x26c58,
  [0x107d] = 0x26c72, [0x107e] = 0x26c6a, [0x107f] = 0x26bd3, [0x1080] = 0x26c3c,
  [0x1081] = 0x26c33, [0x1082] = 0x26c3b, [0x1083] = 0x26c4d, [0x1084] = 0x26c4f,
  [0x1085] = 0x26c7c, [0x1086] = 0x26cb0, [0x1087] = 0x26c5b, [0x1088] = 0x26c55,
  [0x1089] = 0x26c64, [0x108a] = 0x26c6b, [0x108b] = 0x26c53, [0x108c] = 0x44e6,
  [0x108d] = 0x26c57, [0x108e] = 0x26c5e, [0x108f] = 0x26c6d, [0x1090] = 0x26c35,
  [0x1091] = 0x26c69, [0x1092] = 0x26c59, [0x1093] = 0x26c66, [0x1094] = 0x26c50,
  [0x1095] = 0x26c51, [0x1096] = 0x26c5c, [0x1097] = 0x26d5d, [0x1098] = 0x26c87,
  [0x1099] = 0x26c78, [0x109a] = 0x26c49, [0x109b] = 0x26c4e, [0x109c] = 0x26bf1,
  [0x109d] = 0x26c73, [0x109e] = 0x26c88, [0x109f] = 0x26bcf, [0x10a0] = 0x2f9a4,
  [0x10a1] = 0x271b1, [0x10a2] = 0x271b3, [0x10a3] = 0x2729c, [0x10a4] = 0x272a4,
  [0x10a5] = 0x27290, [0x10a6] = 0x272b8, [0x10a7] = 0x272a0, [0x10a8] = 0x2729d,
  [0x10a9] = 0x272a8, [0x10aa] = 0x272a9, [0x10ab] = 0x2f9b9, [0x10ac] = 0x272b1,
  [0x10ad] = 0x27293, [0x10ae] = 0x2729b, [0x10af] = 0x272a2, [0x10b0] = 0x272a1,
  [0x10b1] = 0x272b0, [0x10b2] = 0x272a7, [0x10b3] = 0x275b3, [0x10b4] = 0x8849,
  [0x10b5] = 0x275b4, [0x10b6] = 0x275f1, [0x10b7] = 0x275ed, [0x10b8] = 0x275ec,
  [0x10b9] = 0x27668, [0x10ba] = 0x27675, [0x10bb] = 0x2766a, [0x10bc] = 0x27662,
  [0x10bd] = 0x27667, [0x10be] = 0x27660, [0x10bf] = 0x27677, [0x10c0] = 0x2765f,
  [0x10c1] = 0x27679, [0x10c2] = 0x27823, [0x10c3] = 0x27821, [0x10c4] = 0x2781c,
  [0x10c5] = 0x8999, [0x10c6] = 0x2781f, [0x10c7] = 0x2781e, [0x10c8] = 0x278d6,
  [0x10c9] = 0x278d3, [0x10ca] = 0x278d9, [0x10cb] = 0x278d7, [0x10cc] = 0x278d4,
  [0x10cd] = 0x278dc, [0x10ce] = 0x278d8, [0x10cf] = 0x279a3, [0x10d0] = 0x46d1,
  [0x10d1] = 0x279be, [0x10d2] = 0x279a9, [0x10d3] = 0x279a1, [0x10d4] = 0x279a0,
  [0x10d5] = 0x279ab, [0x10d6] = 0x279a2, [0x10d7] = 0x279ba, [0x10d8] = 0x279c2,
  [0x10d9] = 0x2799c, [0x10da] = 0x279bb, [0x10db] = 0x279aa, [0x10dc] = 0x279ac,
  [0x10dd] = 0x279a5, [0x10de] = 0x279c1, [0x10df] = 0x472c, [0x10e0] = 0x27be2,
  [0x10e1] = 0x27c36, [0x10e2] = 0x27c3d, [0x10e3] = 0x27c3a, [0x10e4] = 0x22461,
  [0x10e5] = 0x27c48, [0x10e6] = 0x27c3c, [0x10e7] = 0x27cb7, [0x10e8] = 0x27cac,
  [0x10e9] = 0x27cb3, [0x10ea] = 0x27cb6, [0x10eb] = 0x27cb2, [0x10ec] = 0x476e,
  [0x10ed] = 0x27d5d, [0x10ee] = 0x27d49, [0x10ef] = 0x27d4e, [0x10f0] = 0x27d4f,
  [0x10f1] = 0x27d4d, [0x10f2] = 0x27d5c, [0x10f3] = 0x27e5d, [0x10f4] = 0x27ebb,
  [0x10f5] = 0x27ece, [0x10f6] = 0x27ebf, [0x10f7] = 0x27eba, [0x10f8] = 0x27ec3,
  [0x10f9] = 0x47e5, [0x10fa] = 0x28006, [0x10fb] = 0x27ff8, [0x10fc] = 0x27ffe,
  [0x10fd] = 0x28013, [0x10fe] = 0x27ff6, [0x10ff] = 0x47e6, [0x1100] = 0x28234,
  [0x1101] = 0x2822d, [0x1102] = 0x2822a, [0x1103] = 0x28235, [0x1104] = 0x2822c,
  [0x1105] = 0x282d4, [0x1106] = 0x282da, [0x1107] = 0x282d3, [0x1108] = 0x282d2,
  [0x1109] = 0x282ce, [0x110a] = 0x282cf, [0x110b] = 0x489f, [0x110c] = 0x284fd,
  [0x110d] = 0x2850b, [0x110e] = 0x284fe, [0x110f] = 0x284f8, [0x1110] = 0x284f7,
  [0x1111] = 0x28517, [0x1112] = 0x284fa, [0x1113] = 0x284f9, [0x1114] = 0x28500,
  [0x1115] = 0x284f5, [0x1116] = 0x284f1, [0x1117] = 0x28503, [0x1118] = 0x28505,
  [0x1119] = 0x2851c, [0x111a] = 0x286ac, [0x111b] = 0x2869a, [0x111c] = 0x286ad,
  [0x111d] = 0x286a0, [0x111e] = 0x2871b, [0x111f] = 0x28725, [0x1120] = 0x28724,
  [0x1121] = 0x28726, [0x1122] = 0x28727, [0x1123] = 0x28728, [0x1124] = 0x2872a,
  [0x1125] = 0x2871a, [0x1126] = 0x2872e, [0x1127] = 0x28715, [0x1128] = 0x28815,
  [0x1129] = 0x28814, [0x112a] = 0x28817, [0x112b] = 0x28818, [0x112c] = 0x2880f,
  [0x112d] = 0x28813, [0x112e] = 0x2880b, [0x112f] = 0x28822, [0x1130] = 0x2137a,
  [0x1131] = 0x28926, [0x1132] = 0x921f, [0x1133] = 0x28951, [0x1134] = 0x2894d,
  [0x1135] = 0x2894e, [0x1136] = 0x2894a, [0x1137] = 0x2894b, [0x1138] = 0x2894c,
  [0x1139] = 0x28950, [0x113a] = 0x28962, [0x113b] = 0x2894f, [0x113c] = 0x28952,
  [0x113d] = 0x28c68, [0x113e] = 0x28c69, [0x113f] = 0x28ce5, [0x1140] = 0x28cf0,
  [0x1141] = 0x28ce1, [0x1142] = 0x28ce2, [0x1143] = 0x2f9f2, [0x1144] = 0x28ea5,
  [0x1145] = 0x28ea9, [0x1146] = 0x28ea4, [0x1147] = 0x28eaf, [0x1148] = 0x28eac,
  [0x1149] = 0x28eae, [0x114a] = 0x28eaa, [0x114b] = 0x28f95, [0x114c] = 0x28f99,
  [0x114d] = 0x28f93, [0x114e] = 0x28f9d, [0x114f] = 0x28f98, [0x1150] = 0x28f9c,
  [0x1151] = 0x28f97, [0x1152] = 0x4a0d, [0x1153] = 0x29089, [0x1154] = 0x29087,
  [0x1155] = 0x291d7, [0x1156] = 0x291ed, [0x1157] = 0x291f1, [0x1158] = 0x291f0,
  [0x1159] = 0x29205, [0x115a] = 0x29203, [0x115b] = 0x29204, [0x115c] = 0x2925f,
  [0x115d] = 0x29457, [0x115e] = 0x2959c, [0x115f] = 0x295a1, [0x1160] = 0x2959a,
  [0x1161] = 0x2968f, [0x1162] = 0x29693, [0x1163] = 0x2969e, [0x1164] = 0x296a3,
  [0x1165] = 0x29698, [0x1166] = 0x29699, [0x1167] = 0x29695, [0x1168] = 0x2986e,
  [0x1169] = 0x2986a, [0x116a] = 0x29af4, [0x116b] = 0x29af7, [0x116c] = 0x29c79,
  [0x116d] = 0x29f8f, [0x116e] = 0x29f91, [0x116f] = 0x2a00b, [0x1170] = 0x2a502,
  [0x1171] = 0x20084, [0x1172] = 0x20104, [0x1173] = 0x20180, [0x1174] = 0x2032f,
  [0x1175] = 0x2033f, [0x1176] = 0x2033b, [0x1177] = 0x20362, [0x1178] = 0x20344,
  [0x1179] = 0x20333, [0x117a] = 0x2035f, [0x117b] = 0x2032b, [0x117c] = 0x2f809,
  [0x117d] = 0x20339, [0x117e] = 0x20336, [0x117f] = 0x20348, [0x1180] = 0x20335,
  [0x1181] = 0x2036c, [0x1182] = 0x20358, [0x1183] = 0x2033a, [0x1184] = 0x204a8,
  [0x1185] = 0x204f4, [0x1186] = 0x20529, [0x1187] = 0x2059a, [0x1188] = 0x20600,
  [0x1189] = 0x207bb, [0x118a] = 0x207bd, [0x118b] = 0x207b7, [0x118c] = 0x207b2,
  [0x118d] = 0x34f8, [0x118e] = 0x207a8, [0x118f] = 0x207ab, [0x1190] = 0x207a6,
  [0x1191] = 0x207bc, [0x1192] = 0x20897, [0x1193] = 0x20896, [0x1194] = 0x208f8,
  [0x1195] = 0x5313, [0x1196] = 0x20951, [0x1197] = 0x3533, [0x1198] = 0x209b5,
  [0x1199] = 0x209b4, [0x119a] = 0x209b6, [0x119b] = 0x353c, [0x119c] = 0x209b7,
  [0x119d] = 0x20a05, [0x119e] = 0x20a23, [0x119f] = 0x20a95, [0x11a0] = 0x20a98,
  [0x11a1] = 0x354f, [0x11a2] = 0x20a94, [0x11a3] = 0x20a93, [0x11a4] = 0x20b07,
  [0x11a5] = 0x20b73, [0x11a6] = 0x20b71, [0x11a7] = 0x20b72, [0x11a8] = 0x20b78,
  [0x11a9] = 0x20e5f, [0x11aa] = 0x20e6a, [0x11ab] = 0x20e67, [0x11ac] = 0x20e6b,
  [0x11ad] = 0x20e69, [0x11ae] = 0x20e8e, [0x11af] = 0x20e49, [0x11b0] = 0x20e80,
  [0x11b1] = 0x35dc, [0x11b2] = 0x20e44, [0x11b3] = 0x20e8f, [0x11b4] = 0x20e45,
  [0x11b5] = 0x2f84c, [0x11b6] = 0x20e4f, [0x11b7] = 0x20e63, [0x11b8] = 0x20e36,
  [0x11b9] = 0x20e48, [0x11ba] = 0x20e5c, [0x11bb] = 0x20e93, [0x11bc] = 0x20e61,
  [0x11bd] = 0x20e60, [0x11be] = 0x20e4e, [0x11bf] = 0x20e5e, [0x11c0] = 0x20e3f,
  [0x11c1] = 0x20ea4, [0x11c2] = 0x55e0, [0x11c3] = 0x20e68, [0x11c4] = 0x2120b,
  [0x11c5] = 0x21211, [0x11c6] = 0x2120d, [0x11c7] = 0x2120e, [0x11c8] = 0x213c2,
  [0x11c9] = 0x2f857, [0x11ca] = 0x213b0, [0x11cb] = 0x213c3, [0x11cc] = 0x213c8,
  [0x11cd] = 0x21365, [0x11ce] = 0x213d7, [0x11cf] = 0x213bd, [0x11d0] = 0x213b8,
  [0x11d1] = 0x213b6, [0x11d2] = 0x213c7, [0x11d3] = 0x213bc, [0x11d4] = 0x213b1,
  [0x11d5] = 0x213af, [0x11d6] = 0x2f85b, [0x11d7] = 0x21535, [0x11d8] = 0x21530,
  [0x11d9] = 0x2155d, [0x11da] = 0x20b76, [0x11db] = 0x2156f, [0x11dc] = 0x215b0,
  [0x11dd] = 0x2f85e, [0x11de] = 0x215b4, [0x11df] = 0x2165c, [0x11e0] = 0x2165d,
  [0x11e1] = 0x21808, [0x11e2] = 0x217ee, [0x11e3] = 0x2180b, [0x11e4] = 0x217f2,
  [0x11e5] = 0x217fd, [0x11e6] = 0x217f1, [0x11e7] = 0x21810, [0x11e8] = 0x2f868,
  [0x11e9] = 0x21814, [0x11ea] = 0x2198b, [0x11eb] = 0x2198c, [0x11ec] = 0x21985,
  [0x11ed] = 0x21a7a, [0x11ee] = 0x21a6c, [0x11ef] = 0x21a60, [0x11f0] = 0x21a67,
  [0x11f1] = 0x21a66, [0x11f2] = 0x21a6a, [0x11f3] = 0x21a5f, [0x11f4] = 0x21a6d,
  [0x11f5] = 0x21a68, [0x11f6] = 0x21a64, [0x11f7] = 0x21b3a, [0x11f8] = 0x21b3b,
  [0x11f9] = 0x21b39, [0x11fa] = 0x21c02, [0x11fb] = 0x21c04, [0x11fc] = 0x21c03,
  [0x11fd] = 0x21cb4, [0x11fe] = 0x21cb3, [0x11ff] = 0x21cbb, [0x1200] = 0x21cac,
  [0x1201] = 0x21cb6, [0x1202] = 0x21cb1, [0x1203] = 0x21caf, [0x1204] = 0x21cb5,
  [0x1205] = 0x21cbe, [0x1206] = 0x21cb2, [0x1207] = 0x21d24, [0x1208] = 0x21eb6,
  [0x1209] = 0x21eb9, [0x120a] = 0x21eac, [0x120b] = 0x21eb0, [0x120c] = 0x21ed7,
  [0x120d] = 0x21ebb, [0x120e] = 0x21efe, [0x120f] = 0x21ecb, [0x1210] = 0x2f87f,
  [0x1211] = 0x37f5, [0x1212] = 0x21eb3, [0x1213] = 0x21ebe, [0x1214] = 0x21ecd,
  [0x1215] = 0x21ebc, [0x1216] = 0x21eba, [0x1217] = 0x21ec7, [0x1218] = 0x22116,
  [0x1219] = 0x2211b, [0x121a] = 0x22117, [0x121b] = 0x22115, [0x121c] = 0x2219f,
  [0x121d] = 0x2f88e, [0x121e] = 0x22282, [0x121f] = 0x22283, [0x1220] = 0x2228e,
  [0x1221] = 0x2226a, [0x1222] = 0x22289, [0x1223] = 0x22294, [0x1224] = 0x22286,
  [0x1225] = 0x22290, [0x1226] = 0x22362, [0x1227] = 0x22390, [0x1228] = 0x2240c,
  [0x1229] = 0x2240f, [0x122a] = 0x38c3, [0x122b] = 0x224a5, [0x122c] = 0x224a7,
  [0x122d] = 0x224a6, [0x122e] = 0x224a4, [0x122f] = 0x22535, [0x1230] = 0x22538,
  [0x1231] = 0x22536, [0x1232] = 0x22543, [0x1233] = 0x22547, [0x1234] = 0x22537,
  [0x1235] = 0x22743, [0x1236] = 0x227a2, [0x1237] = 0x22751, [0x1238] = 0x22772,
  [0x1239] = 0x2276d, [0x123a] = 0x2272f, [0x123b] = 0x22754, [0x123c] = 0x2276e,
  [0x123d] = 0x22755, [0x123e] = 0x22737, [0x123f] = 0x2274f, [0x1240] = 0x22750,
  [0x1241] = 0x6139, [0x1242] = 0x2279e, [0x1243] = 0x2f8a8, [0x1244] = 0x227b3,
  [0x1245] = 0x227a7, [0x1246] = 0x3971, [0x1247] = 0x227f9, [0x1248] = 0x22791,
  [0x1249] = 0x2f8a7, [0x124a] = 0x227b6, [0x124b] = 0x227dd, [0x124c] = 0x22799,
  [0x124d] = 0x229df, [0x124e] = 0x229d5, [0x124f] = 0x247f5, [0x1250] = 0x22c28,
  [0x1251] = 0x22c40, [0x1252] = 0x22c71, [0x1253] = 0x22c98, [0x1254] = 0x22c95,
  [0x1255] = 0x22c65, [0x1256] = 0x22c78, [0x1257] = 0x22c7f, [0x1258] = 0x22c60,
  [0x1259] = 0x22c7c, [0x125a] = 0x22c96, [0x125b] = 0x22c6a, [0x125c] = 0x22c79,
  [0x125d] = 0x22c73, [0x125e] = 0x22c72, [0x125f] = 0x22c7b, [0x1260] = 0x22c70,
  [0x1261] = 0x22ed5, [0x1262] = 0x22ed7, [0x1263] = 0x22f8f, [0x1264] = 0x22f89,
  [0x1265] = 0x22f9e, [0x1266] = 0x22f87, [0x1267] = 0x22fa0, [0x1268] = 0x22f8d,
  [0x1269] = 0x22f88, [0x126a] = 0x22f7f, [0x126b] = 0x22f8c, [0x126c] = 0x22f93,
  [0x126d] = 0x23059, [0x126e] = 0x295b0, [0x126f] = 0x230b1, [0x1270] = 0x251f4,
  [0x1271] = 0x230af, [0x1272] = 0x230b2, [0x1273] = 0x23108, [0x1274] = 0x2310a,
  [0x1275] = 0x23136, [0x1276] = 0x23137, [0x1277] = 0x2f8cf, [0x1278] = 0x3b01,
  [0x1279] = 0x23236, [0x127a] = 0x23229, [0x127b] = 0x2324a, [0x127c] = 0x23226,
  [0x127d] = 0x23228, [0x127e] = 0x23224, [0x127f] = 0x2322a, [0x1280] = 0x23248,
  [0x1281] = 0x23344, [0x1282] = 0x23384, [0x1283] = 0x2338c, [0x1284] = 0x234c4,
  [0x1285] = 0x234dc, [0x1286] = 0x23587, [0x1287] = 0x23553, [0x1288] = 0x2357f,
  [0x1289] = 0x23531, [0x128a] = 0x23551, [0x128b] = 0x2354b, [0x128c] = 0x23552,
  [0x128d] = 0x23580, [0x128e] = 0x235a5, [0x128f] = 0x23581, [0x1290] = 0x23543,
  [0x1291] = 0x23534, [0x1292] = 0x23536, [0x1293] = 0x23532, [0x1294] = 0x23548,
  [0x1295] = 0x23549, [0x1296] = 0x2353c, [0x1297] = 0x2354d, [0x1298] = 0x2354a,
  [0x1299] = 0x2f8e6, [0x129a] = 0x2358a, [0x129b] = 0x23546, [0x129c] = 0x2353e,
  [0x129d] = 0x23583, [0x129e] = 0x23550, [0x129f] = 0x235b4, [0x12a0] = 0x238f0,
  [0x12a1] = 0x238e4, [0x12a2] = 0x238e3, [0x12a3] = 0x238e5, [0x12a4] = 0x238e6,
  [0x12a5] = 0x238e7, [0x12a6] = 0x238e1, [0x12a7] = 0x238ef, [0x12a8] = 0x238e8,
  [0x12a9] = 0x238dd, [0x12aa] = 0x23903, [0x12ab] = 0x23988, [0x12ac] = 0x23a26,
  [0x12ad] = 0x23a16, [0x12ae] = 0x23a3b, [0x12af] = 0x23a2f, [0x12b0] = 0x23a39,
  [0x12b1] = 0x23a34, [0x12b2] = 0x2f8f4, [0x12b3] = 0x23a35, [0x12b4] = 0x23a31,
  [0x12b5] = 0x23a38, [0x12b6] = 0x20b81, [0x12b7] = 0x23aaa, [0x12b8] = 0x23aa3,
  [0x12b9] = 0x23aa4, [0x12ba] = 0x23aa0, [0x12bb] = 0x23aa1, [0x12bc] = 0x23aa9,
  [0x12bd] = 0x23aa5, [0x12be] = 0x23baf, [0x12bf] = 0x23bb1, [0x12c0] = 0x23bab,
  [0x12c1] = 0x23bae, [0x12c2] = 0x23bb0, [0x12c3] = 0x23bb3, [0x12c4] = 0x23bac,
  [0x12c5] = 0x23ba9, [0x12c6] = 0x23bb2, [0x12c7] = 0x23ba8, [0x12c8] = 0x23bb4,
  [0x12c9] = 0x23bc2, [0x12ca] = 0x23c4f, [0x12cb] = 0x23c66, [0x12cc] = 0x23e1e,
  [0x12cd] = 0x23e15, [0x12ce] = 0x23e10, [0x12cf] = 0x23e7f, [0x12d0] = 0x23e7e,
  [0x12d1] = 0x23e60, [0x12d2] = 0x23ecc, [0x12d3] = 0x23eb2, [0x12d4] = 0x23e62,
  [0x12d5] = 0x23e8d, [0x12d6] = 0x23e8e, [0x12d7] = 0x23e77, [0x12d8] = 0x23e7c,
  [0x12d9] = 0x23e8f, [0x12da] = 0x23e5d, [0x12db] = 0x23e6d, [0x12dc] = 0x23e63,
  [0x12dd] = 0x23eaf, [0x12de] = 0x23e90, [0x12df] = 0x23e7d, [0x12e0] = 0x23e7a,
  [0x12e1] = 0x23e06, [0x12e2] = 0x2f90b, [0x12e3] = 0x23e68, [0x12e4] = 0x23eb4,
  [0x12e5] = 0x23e78, [0x12e6] = 0x23eb1, [0x12e7] = 0x6ed7, [0x12e8] = 0x2f90c,
  [0x12e9] = 0x3d41, [0x12ea] = 0x242e8, [0x12eb] = 0x242e9, [0x12ec] = 0x242c0,
  [0x12ed] = 0x3dd7, [0x12ee] = 0x242ea, [0x12ef] = 0x242b7, [0x12f0] = 0x242ba,
  [0x12f1] = 0x242b5, [0x12f2] = 0x2f91d, [0x12f3] = 0x242b4, [0x12f4] = 0x242bc,
  [0x12f5] = 0x242c6, [0x12f6] = 0x242b8, [0x12f7] = 0x242bd, [0x12f8] = 0x242c2,
  [0x12f9] = 0x2434d, [0x12fa] = 0x242f0, [0x12fb] = 0x242c7, [0x12fc] = 0x242c1,
  [0x12fd] = 0x242c3, [0x12fe] = 0x242f1, [0x12ff] = 0x242ec, [0x1300] = 0x3de2,
  [0x1301] = 0x2451c, [0x1302] = 0x24520, [0x1303] = 0x24558, [0x1304] = 0x24565,
  [0x1305] = 0x24564, [0x1306] = 0x2458c, [0x1307] = 0x2458d, [0x1308] = 0x245d9,
  [0x1309] = 0x245e0, [0x130a] = 0x24610, [0x130b] = 0x24694, [0x130c] = 0x24692,
  [0x130d] = 0x24696, [0x130e] = 0x24695, [0x130f] = 0x246bf, [0x1310] = 0x246a0,
  [0x1311] = 0x3e45, [0x1312] = 0x247f4, [0x1313] = 0x247f6, [0x1314] = 0x247dc,
  [0x1315] = 0x733d, [0x1316] = 0x24816, [0x1317] = 0x24815, [0x1318] = 0x2481c,
  [0x1319] = 0x2480f, [0x131a] = 0x2482c, [0x131b] = 0x24814, [0x131c] = 0x24825,
  [0x131d] = 0x24817, [0x131e] = 0x24812, [0x131f] = 0x2481e, [0x1320] = 0x24980,
  [0x1321] = 0x249a8, [0x1322] = 0x249af, [0x1323] = 0x249d6, [0x1324] = 0x249e2,
  [0x1325] = 0x249b4, [0x1326] = 0x249b3, [0x1327] = 0x249b0, [0x1328] = 0x249b2,
  [0x1329] = 0x249a1, [0x132a] = 0x24b60, [0x132b] = 0x24b66, [0x132c] = 0x24b61,
  [0x132d] = 0x24b4e, [0x132e] = 0x24b5d, [0x132f] = 0x24b63, [0x1330] = 0x24be6,
  [0x1331] = 0x24bef, [0x1332] = 0x24bec, [0x1333] = 0x24c98, [0x1334] = 0x24ca7,
  [0x1335] = 0x24c94, [0x1336] = 0x24c95, [0x1337] = 0x24c91, [0x1338] = 0x24c9d,
  [0x1339] = 0x24c99, [0x133a] = 0x24c9b, [0x133b] = 0x24c9c, [0x133c] = 0x24d1d,
  [0x133d] = 0x24d1c, [0x133e] = 0x24dd0, [0x133f] = 0x24de0, [0x1340] = 0x24dcb,
  [0x1341] = 0x24ddb, [0x1342] = 0x24dda, [0x1343] = 0x24dc2, [0x1344] = 0x24dd3,
  [0x1345] = 0x24de5, [0x1346] = 0x24f7d, [0x1347] = 0x24f7b, [0x1348] = 0x24fff,
  [0x1349] = 0x3ff2, [0x134a] = 0x24ff9, [0x134b] = 0x25077, [0x134c] = 0x2507c,
  [0x134d] = 0x25078, [0x134e] = 0x2507b, [0x134f] = 0x2507a, [0x1350] = 0x251d2,
  [0x1351] = 0x251cb, [0x1352] = 0x251c9, [0x1353] = 0x251ce, [0x1354] = 0x251e4,
  [0x1355] = 0x251ca, [0x1356] = 0x251d0, [0x1357] = 0x22fa5, [0x1358] = 0x4049,
  [0x1359] = 0x251d9, [0x135a] = 0x251ee, [0x135b] = 0x251dd, [0x135c] = 0x25200,
  [0x135d] = 0x251e1, [0x135e] = 0x253de, [0x135f] = 0x253e2, [0x1360] = 0x253da,
  [0x1361] = 0x254ce, [0x1362] = 0x2f94f, [0x1363] = 0x254cf, [0x1364] = 0x254da,
  [0x1365] = 0x254d1, [0x1366] = 0x254d4, [0x1367] = 0x254ed, [0x1368] = 0x254cb,
  [0x1369] = 0x254d5, [0x136a] = 0x254f1, [0x136b] = 0x2569c, [0x136c] = 0x25688,
  [0x136d] = 0x25741, [0x136e] = 0x257d0, [0x136f] = 0x257f7, [0x1370] = 0x257cf,
  [0x1371] = 0x257d1, [0x1372] = 0x257db, [0x1373] = 0x257de, [0x1374] = 0x257f8,
  [0x1375] = 0x257dc, [0x1376] = 0x257d9, [0x1377] = 0x259a0, [0x1378] = 0x259b2,
  [0x1379] = 0x259a1, [0x137a] = 0x259aa, [0x137b] = 0x259a9, [0x137c] = 0x41a9,
  [0x137d] = 0x259a6, [0x137e] = 0x2599c, [0x137f] = 0x259b5, [0x1380] = 0x259a7,
  [0x1381] = 0x25a8e, [0x1382] = 0x25a90, [0x1383] = 0x25a91, [0x1384] = 0x25b85,
  [0x1385] = 0x25b5b, [0x1386] = 0x25b70, [0x1387] = 0x25b64, [0x1388] = 0x25b67,
  [0x1389] = 0x25b63, [0x138a] = 0x25b77, [0x138b] = 0x25b68, [0x138c] = 0x25b65,
  [0x138d] = 0x25b6a, [0x138e] = 0x25b78, [0x138f] = 0x25b66, [0x1390] = 0x25e88,
  [0x1391] = 0x25e9e, [0x1392] = 0x25e74, [0x1393] = 0x25e7a, [0x1394] = 0x25e79,
  [0x1395] = 0x25e8b, [0x1396] = 0x25e7f, [0x1397] = 0x26013, [0x1398] = 0x2601e,
  [0x1399] = 0x26017, [0x139a] = 0x2601a, [0x139b] = 0x26022, [0x139c] = 0x26043,
  [0x139d] = 0x26019, [0x139e] = 0x2601f, [0x139f] = 0x26027, [0x13a0] = 0x26012,
  [0x13a1] = 0x26024, [0x13a2] = 0x26025, [0x13a3] = 0x2035e, [0x13a4] = 0x26235,
  [0x13a5] = 0x26234, [0x13a6] = 0x262d1, [0x13a7] = 0x262c4, [0x13a8] = 0x262ca,
  [0x13a9] = 0x262c6, [0x13aa] = 0x262cb, [0x13ab] = 0x262cd, [0x13ac] = 0x262e2,
  [0x13ad] = 0x26389, [0x13ae] = 0x2638b, [0x13af] = 0x26386, [0x13b0] = 0x26388,
  [0x13b1] = 0x2638d, [0x13b2] = 0x2643b, [0x13b3] = 0x2643c, [0x13b4] = 0x2643d,
  [0x13b5] = 0x264f5, [0x13b6] = 0x26579, [0x13b7] = 0x26575, [0x13b8] = 0x26582,
  [0x13b9] = 0x2657f, [0x13ba] = 0x26585, [0x13bb] = 0x26576, [0x13bc] = 0x2657c,
  [0x13bd] = 0x2657e, [0x13be] = 0x2657b, [0x13bf] = 0x26580, [0x13c0] = 0x2657a,
  [0x13c1] = 0x22548, [0x13c2] = 0x43cb, [0x13c3] = 0x2661d, [0x13c4] = 0x266f6,
  [0x13c5] = 0x266f7, [0x13c6] = 0x266f9, [0x13c7] = 0x26763, [0x13c8] = 0x2676c,
  [0x13c9] = 0x2676e, [0x13ca] = 0x26714, [0x13cb] = 0x440d, [0x13cc] = 0x26767,
  [0x13cd] = 0x2676f, [0x13ce] = 0x26769, [0x13cf] = 0x26776, [0x13d0] = 0x26795,
  [0x13d1] = 0x26771, [0x13d2] = 0x26761, [0x13d3] = 0x26778, [0x13d4] = 0x2676b,
  [0x13d5] = 0x26785, [0x13d6] = 0x26784, [0x13d7] = 0x26914, [0x13d8] = 0x26976,
  [0x13d9] = 0x259b6, [0x13da] = 0x26975, [0x13db] = 0x26974, [0x13dc] = 0x269e2,
  [0x13dd] = 0x26a3f, [0x13de] = 0x26a44, [0x13df] = 0x26a3d, [0x13e0] = 0x26a47,
  [0x13e1] = 0x26a48, [0x13e2] = 0x26adb, [0x13e3] = 0x26adc, [0x13e4] = 0x26add,
  [0x13e5] = 0x26be5, [0x13e6] = 0x26be6, [0x13e7] = 0x26cc2, [0x13e8] = 0x26d2e,
  [0x13e9] = 0x26d1d, [0x13ea] = 0x26cf8, [0x13eb] = 0x26cf0, [0x13ec] = 0x26cf6,
  [0x13ed] = 0x26cc5, [0x13ee] = 0x26cc6, [0x13ef] = 0x26cfc, [0x13f0] = 0x26d52,
  [0x13f1] = 0x26d2f, [0x13f2] = 0x26d10, [0x13f3] = 0x26cf3, [0x13f4] = 0x26cd2,
  [0x13f5] = 0x26cea, [0x13f6] = 0x26cdc, [0x13f7] = 0x26d1b, [0x13f8] = 0x26cfb,
  [0x13f9] = 0x26cc3, [0x13fa] = 0x26d16, [0x13fb] = 0x26d07, [0x13fc] = 0x26cc9,
  [0x13fd] = 0x26cd8, [0x13fe] = 0x26d30, [0x13ff] = 0x26d13, [0x1400] = 0x26d31,
  [0x1401] = 0x26cfa, [0x1402] = 0x26cf2, [0x1403] = 0x26de3, [0x1404] = 0x26cd5,
  [0x1405] = 0x26d01, [0x1406] = 0x26cf1, [0x1407] = 0x26d1c, [0x1408] = 0x26cd6,
  [0x1409] = 0x26d08, [0x140a] = 0x26d0b, [0x140b] = 0x26d17, [0x140c] = 0x26d20,
  [0x140d] = 0x26cca, [0x140e] = 0x26d32, [0x140f] = 0x26d05, [0x1410] = 0x26ce4,
  [0x1411] = 0x26cce, [0x1412] = 0x26d33, [0x1413] = 0x26d02, [0x1414] = 0x26d19,
  [0x1415] = 0x26d1e, [0x1416] = 0x26cd3, [0x1417] = 0x26cf7, [0x1418] = 0x26ce8,
  [0x1419] = 0x26d1f, [0x141a] = 0x26cf4, [0x141b] = 0x26dd8, [0x141c] = 0x271bf,
  [0x141d] = 0x271be, [0x141e] = 0x271bd, [0x141f] = 0x272dc, [0x1420] = 0x272dd,
  [0x1421] = 0x272ab, [0x1422] = 0x272c9, [0x1423] = 0x272c8, [0x1424] = 0x272df,
  [0x1425] = 0x272d9, [0x1426] = 0x272db, [0x1427] = 0x272cc, [0x1428] = 0x272e0,
  [0x1429] = 0x272ef, [0x142a] = 0x272f3, [0x142b] = 0x272ae, [0x142c] = 0x27301,
  [0x142d] = 0x275b7, [0x142e] = 0x275b9, [0x142f] = 0x275b6, [0x1430] = 0x275b8,
  [0x1431] = 0x275f6, [0x1432] = 0x275f3, [0x1433] = 0x275f5, [0x1434] = 0x275f2,
  [0x1435] = 0x27691, [0x1436] = 0x2769d, [0x1437] = 0x2769b, [0x1438] = 0x27692,
  [0x1439] = 0x2768d, [0x143a] = 0x2769e, [0x143b] = 0x2768c, [0x143c] = 0x4631,
  [0x143d] = 0x2f9c7, [0x143e] = 0x27695, [0x143f] = 0x2f9c6, [0x1440] = 0x2768a,
  [0x1441] = 0x2768e, [0x1442] = 0x2769c, [0x1443] = 0x277ef, [0x1444] = 0x2782d,
  [0x1445] = 0x27852, [0x1446] = 0x27835, [0x1447] = 0x27828, [0x1448] = 0x2782e,
  [0x1449] = 0x278e5, [0x144a] = 0x279ea, [0x144b] = 0x279f1, [0x144c] = 0x279eb,
  [0x144d] = 0x279d8, [0x144e] = 0x279d0, [0x144f] = 0x279f3, [0x1450] = 0x279db,
  [0x1451] = 0x279ce, [0x1452] = 0x46d5, [0x1453] = 0x279da, [0x1454] = 0x279d7,
  [0x1455] = 0x279e1, [0x1456] = 0x279f2, [0x1457] = 0x279c8, [0x1458] = 0x279d9,
  [0x1459] = 0x279de, [0x145a] = 0x279d1, [0x145b] = 0x279e7, [0x145c] = 0x279cf,
  [0x145d] = 0x27bb7, [0x145e] = 0x27c47, [0x145f] = 0x27c42, [0x1460] = 0x27c43,
  [0x1461] = 0x2f9d3, [0x1462] = 0x27cc3, [0x1463] = 0x27cc1, [0x1464] = 0x27cc7,
  [0x1465] = 0x27d64, [0x1466] = 0x27d6a, [0x1467] = 0x27d66, [0x1468] = 0x27d50,
  [0x1469] = 0x27d6e, [0x146a] = 0x27d65, [0x146b] = 0x238ec, [0x146c] = 0x27d7f,
  [0x146d] = 0x2f9d5, [0x146e] = 0x27d9a, [0x146f] = 0x27d69, [0x1470] = 0x27d72,
  [0x1471] = 0x27d6f, [0x1472] = 0x27d7d, [0x1473] = 0x27d70, [0x1474] = 0x27e60,
  [0x1475] = 0x27ec2, [0x1476] = 0x27ee7, [0x1477] = 0x27ed1, [0x1478] = 0x27eeb,
  [0x1479] = 0x27ed4, [0x147a] = 0x27edc, [0x147b] = 0x27edb, [0x147c] = 0x28037,
  [0x147d] = 0x28025, [0x147e] = 0x2801f, [0x147f] = 0x2801e, [0x1480] = 0x28021,
  [0x1481] = 0x2801b, [0x1482] = 0x28017, [0x1483] = 0x28022, [0x1484] = 0x2802a,
  [0x1485] = 0x2801a, [0x1486] = 0x2802d, [0x1487] = 0x28023, [0x1488] = 0x28026,
  [0x1489] = 0x28036, [0x148a] = 0x27fff, [0x148b] = 0x2823a, [0x148c] = 0x28240,
  [0x148d] = 0x28242, [0x148e] = 0x28238, [0x148f] = 0x2823b, [0x1490] = 0x2823c,
  [0x1491] = 0x28243, [0x1492] = 0x282e8, [0x1493] = 0x4856, [0x1494] = 0x282f9,
  [0x1495] = 0x282eb, [0x1496] = 0x282f1, [0x1497] = 0x282e9, [0x1498] = 0x282ec,
  [0x1499] = 0x28415, [0x149a] = 0x28447, [0x149b] = 0x28446, [0x149c] = 0x28445,
  [0x149d] = 0x28524, [0x149e] = 0x28556, [0x149f] = 0x28521, [0x14a0] = 0x28527,
  [0x14a1] = 0x2854d, [0x14a2] = 0x28531, [0x14a3] = 0x2851e, [0x14a4] = 0x2852c,
  [0x14a5] = 0x2854f, [0x14a6] = 0x2852b, [0x14a7] = 0x28553, [0x14a8] = 0x28551,
  [0x14a9] = 0x28554, [0x14aa] = 0x28529, [0x14ab] = 0x28550, [0x14ac] = 0x2851f,
  [0x14ad] = 0x28532, [0x14ae] = 0x2852a, [0x14af] = 0x2851d, [0x14b0] = 0x28528,
  [0x14b1] = 0x2852e, [0x14b2] = 0x2852d, [0x14b3] = 0x286bc, [0x14b4] = 0x286bb,
  [0x14b5] = 0x286bd, [0x14b6] = 0x2f9e4, [0x14b7] = 0x28747, [0x14b8] = 0x28741,
  [0x14b9] = 0x2f9e5, [0x14ba] = 0x28749, [0x14bb] = 0x48f4, [0x14bc] = 0x2882e,
  [0x14bd] = 0x28827, [0x14be] = 0x2896a, [0x14bf] = 0x2897b, [0x14c0] = 0x28973,
  [0x14c1] = 0x28975, [0x14c2] = 0x28969, [0x14c3] = 0x28979, [0x14c4] = 0x28972,
  [0x14c5] = 0x28c76, [0x14c6] = 0x28c77, [0x14c7] = 0x28c74, [0x14c8] = 0x28c73,
  [0x14c9] = 0x28d02, [0x14ca] = 0x28cfe, [0x14cb] = 0x28d01, [0x14cc] = 0x28cf9,
  [0x14cd] = 0x28cf4, [0x14ce] = 0x28ed3, [0x14cf] = 0x28ed5, [0x14d0] = 0x28ed8,
  [0x14d1] = 0x28ec3, [0x14d2] = 0x28eca, [0x14d3] = 0x28ed0, [0x14d4] = 0x28ecb,
  [0x14d5] = 0x28ece, [0x14d6] = 0x28ec5, [0x14d7] = 0x28ee6, [0x14d8] = 0x28ec4,
  [0x14d9] = 0x28ec0, [0x14da] = 0x28ed4, [0x14db] = 0x28ee8, [0x14dc] = 0x28f76,
  [0x14dd] = 0x28fa2, [0x14de] = 0x28fae, [0x14df] = 0x28fa8, [0x14e0] = 0x28fa3,
  [0x14e1] = 0x28fa7, [0x14e2] = 0x28f96, [0x14e3] = 0x28fa9, [0x14e4] = 0x28fa5,
  [0x14e5] = 0x28faf, [0x14e6] = 0x28fa4, [0x14e7] = 0x28fab, [0x14e8] = 0x28faa,
  [0x14e9] = 0x28fa6, [0x14ea] = 0x28fa0, [0x14eb] = 0x29098, [0x14ec] = 0x291db,
  [0x14ed] = 0x291f6, [0x14ee] = 0x291f5, [0x14ef] = 0x2920c, [0x14f0] = 0x2920a,
  [0x14f1] = 0x9775, [0x14f2] = 0x29268, [0x14f3] = 0x29263, [0x14f4] = 0x29266,
  [0x14f5] = 0x4a58, [0x14f6] = 0x29264, [0x14f7] = 0x2926a, [0x14f8] = 0x29269,
  [0x14f9] = 0x29395, [0x14fa] = 0x29402, [0x14fb] = 0x2946a, [0x14fc] = 0x2945e,
  [0x14fd] = 0x29468, [0x14fe] = 0x29469, [0x14ff] = 0x29465, [0x1500] = 0x2f9fe,
  [0x1501] = 0x295a7, [0x1502] = 0x295ae, [0x1503] = 0x295a8, [0x1504] = 0x295b3,
  [0x1505] = 0x2969c, [0x1506] = 0x296a9, [0x1507] = 0x296b6, [0x1508] = 0x296b3,
  [0x1509] = 0x296b2, [0x150a] = 0x2fa03, [0x150b] = 0x296b8, [0x150c] = 0x296c0,
  [0x150d] = 0x296bf, [0x150e] = 0x296ba, [0x150f] = 0x296a8, [0x1510] = 0x2983c,
  [0x1511] = 0x29872, [0x1512] = 0x29871, [0x1513] = 0x29875, [0x1514] = 0x29873,
  [0x1515] = 0x29a17, [0x1516] = 0x29a15, [0x1517] = 0x29a16, [0x1518] = 0x29a1b,
  [0x1519] = 0x29afa, [0x151a] = 0x29af9, [0x151b] = 0x29c2c, [0x151c] = 0x29c85,
  [0x151d] = 0x29c82, [0x151e] = 0x29c7f, [0x151f] = 0x29c7d, [0x1520] = 0x29c88,
  [0x1521] = 0x242f3, [0x1522] = 0x29d4e, [0x1523] = 0x29d50, [0x1524] = 0x29f93,
  [0x1525] = 0x29f97, [0x1526] = 0x29f98, [0x1527] = 0x29f9b, [0x1528] = 0x29f9c,
  [0x1529] = 0x29f96, [0x152a] = 0x2a28b, [0x152b] = 0x2a2fd, [0x152c] = 0x2a2fc,
  [0x152d] = 0x2fa17, [0x152e] = 0x2398b, [0x152f] = 0x20108, [0x1530] = 0x20107,
  [0x1531] = 0x20373, [0x1532] = 0x203a2, [0x1533] = 0x203af, [0x1534] = 0x20382,
  [0x1535] = 0x2037b, [0x1536] = 0x20374, [0x1537] = 0x203b0, [0x1538] = 0x20376,
  [0x1539] = 0x203b9, [0x153a] = 0x2039e, [0x153b] = 0x203b1, [0x153c] = 0x203a1,
  [0x153d] = 0x203b2, [0x153e] = 0x2036e, [0x153f] = 0x2f80b, [0x1540] = 0x3487,
  [0x1541] = 0x20378, [0x1542] = 0x2037a, [0x1543] = 0x20383, [0x1544] = 0x2039a,
  [0x1545] = 0x204f8, [0x1546] = 0x20531, [0x1547] = 0x20569, [0x1548] = 0x20568,
  [0x1549] = 0x2059c, [0x154a] = 0x20604, [0x154b] = 0x20699, [0x154c] = 0x207c7,
  [0x154d] = 0x207c6, [0x154e] = 0x207dc, [0x154f] = 0x207c4, [0x1550] = 0x207d8,
  [0x1551] = 0x207d4, [0x1552] = 0x207df, [0x1553] = 0x207d1, [0x1554] = 0x207d0,
  [0x1555] = 0x207d6, [0x1556] = 0x207cf, [0x1557] = 0x208ad, [0x1558] = 0x208af,
  [0x1559] = 0x208a7, [0x155a] = 0x208fd, [0x155b] = 0x2095b, [0x155c] = 0x2095a,
  [0x155d] = 0x20a07, [0x155e] = 0x20a9d, [0x155f] = 0x20a9b, [0x1560] = 0x20aab,
  [0x1561] = 0x20aa4, [0x1562] = 0x20a9c, [0x1563] = 0x20a9e, [0x1564] = 0x20aa5,
  [0x1565] = 0x20aaa, [0x1566] = 0x20aa6, [0x1567] = 0x20b0e, [0x1568] = 0x20b7e,
  [0x1569] = 0x20b7c, [0x156a] = 0x20eee, [0x156b] = 0x20ed5, [0x156c] = 0x20ee9,
  [0x156d] = 0x20e42, [0x156e] = 0x20ee2, [0x156f] = 0x20f23, [0x1570] = 0x20ed9,
  [0x1571] = 0x20ed4, [0x1572] = 0x20ee3, [0x1573] = 0x20f15, [0x1574] = 0x20eef,
  [0x1575] = 0x20ef0, [0x1576] = 0x20ed6, [0x1577] = 0x20edd, [0x1578] = 0x20ef6,
  [0x1579] = 0x20f1c, [0x157a] = 0x20ed8, [0x157b] = 0x20edb, [0x157c] = 0x20eda,
  [0x157d] = 0x20eed, [0x157e] = 0x21411, [0x157f] = 0x21215, [0x1580] = 0x21218,
  [0x1581] = 0x2121a, [0x1582] = 0x2121f, [0x1583] = 0x21216, [0x1584] = 0x2f84d,
  [0x1585] = 0x21219, [0x1586] = 0x2f84b, [0x1587] = 0x213f0, [0x1588] = 0x21409,
  [0x1589] = 0x2141b, [0x158a] = 0x27be7, [0x158b] = 0x213f6, [0x158c] = 0x213f4,
  [0x158d] = 0x213b5, [0x158e] = 0x21410, [0x158f] = 0x213f2, [0x1590] = 0x21415,
  [0x1591] = 0x213f3, [0x1592] = 0x213f8, [0x1593] = 0x21539, [0x1594] = 0x2153b,
  [0x1595] = 0x21536, [0x1596] = 0x2140e, [0x1597] = 0x21572, [0x1598] = 0x21574,
  [0x1599] = 0x215b9, [0x159a] = 0x215b7, [0x159b] = 0x215b8, [0x159c] = 0x21672,
  [0x159d] = 0x2166b, [0x159e] = 0x2181d, [0x159f] = 0x21837, [0x15a0] = 0x21822,
  [0x15a1] = 0x21843, [0x15a2] = 0x2184d, [0x15a3] = 0x21838, [0x15a4] = 0x2185b,
  [0x15a5] = 0x21879, [0x15a6] = 0x2181b, [0x15a7] = 0x217f3, [0x15a8] = 0x21991,
  [0x15a9] = 0x21a7b, [0x15aa] = 0x21a94, [0x15ab] = 0x2f86f, [0x15ac] = 0x21a96,
  [0x15ad] = 0x21a7f, [0x15ae] = 0x21a8f, [0x15af] = 0x21a84, [0x15b0] = 0x21a7c,
  [0x15b1] = 0x21a8e, [0x15b2] = 0x21a90, [0x15b3] = 0x21a98, [0x15b4] = 0x21a83,
  [0x15b5] = 0x21a80, [0x15b6] = 0x21a93, [0x15b7] = 0x21a82, [0x15b9] = 0x21b3d,
  [0x15ba] = 0x21b41, [0x15bb] = 0x21ba1, [0x15bc] = 0x21b9f, [0x15bd] = 0x21c0a,
  [0x15be] = 0x21c0d, [0x15bf] = 0x21cc8, [0x15c0] = 0x21cc9, [0x15c1] = 0x21cc7,
  [0x15c2] = 0x21ccd, [0x15c3] = 0x21d25, [0x15c4] = 0x21eb1, [0x15c5] = 0x21edc,
  [0x15c6] = 0x2f880, [0x15c7] = 0x21ee5, [0x15c8] = 0x21ef4, [0x15c9] = 0x21ebf,
  [0x15ca] = 0x21edb, [0x15cb] = 0x21eea, [0x15cc] = 0x21ef2, [0x15cd] = 0x37fa,
  [0x15ce] = 0x21ef1, [0x15cf] = 0x21eed, [0x15d0] = 0x21ee6, [0x15d1] = 0x22002,
  [0x15d2] = 0x2f882, [0x15d3] = 0x22125, [0x15d4] = 0x22118, [0x15d5] = 0x2211f,
  [0x15d6] = 0x22120, [0x15d7] = 0x221cf, [0x15d8] = 0x2229d, [0x15d9] = 0x22299,
  [0x15da] = 0x222a8, [0x15db] = 0x22368, [0x15dc] = 0x22366, [0x15dd] = 0x22367,
  [0x15de] = 0x22391, [0x15df] = 0x22413, [0x15e0] = 0x22415, [0x15e1] = 0x2241d,
  [0x15e2] = 0x22416, [0x15e3] = 0x22419, [0x15e4] = 0x2246b, [0x15e5] = 0x22468,
  [0x15e6] = 0x2246a, [0x15e7] = 0x2246d, [0x15e8] = 0x22469, [0x15e9] = 0x224aa,
  [0x15ea] = 0x22557, [0x15eb] = 0x22552, [0x15ec] = 0x22550, [0x15ed] = 0x2255f,
  [0x15ee] = 0x22567, [0x15ef] = 0x2254f, [0x15f0] = 0x38f2, [0x15f1] = 0x2255b,
  [0x15f2] = 0x2255c, [0x15f3] = 0x2255d, [0x15f4] = 0x2281f, [0x15f5] = 0x2279d,
  [0x15f6] = 0x227b1, [0x15f7] = 0x227b0, [0x15f8] = 0x22794, [0x15f9] = 0x227c3,
  [0x15fa] = 0x227af, [0x15fb] = 0x227a8, [0x15fc] = 0x227dc, [0x15fd] = 0x22798,
  [0x15fe] = 0x227c4, [0x15ff] = 0x227a4, [0x1600] = 0x227ab, [0x1601] = 0x227aa,
  [0x1602] = 0x227a5, [0x1603] = 0x22821, [0x1604] = 0x227eb, [0x1605] = 0x227e6,
  [0x1606] = 0x227f7, [0x1607] = 0x227f8, [0x1608] = 0x227fc, [0x1609] = 0x227fa,
  [0x160a] = 0x227e0, [0x160b] = 0x2f8aa, [0x160c] = 0x227f6, [0x160d] = 0x2f8a9,
  [0x160e] = 0x227e1, [0x160f] = 0x229ec, [0x1610] = 0x229e2, [0x1611] = 0x229e4,
  [0x1612] = 0x229f9, [0x1613] = 0x22c6f, [0x1614] = 0x23a4c, [0x1615] = 0x22cbb,
  [0x1616] = 0x22ce1, [0x1617] = 0x22d00, [0x1618] = 0x22cd8, [0x1619] = 0x3a2f,
  [0x161a] = 0x22cd6, [0x161b] = 0x22ce2, [0x161c] = 0x22cc3, [0x161d] = 0x22cb3,
  [0x161e] = 0x22cd2, [0x161f] = 0x2f8c1, [0x1620] = 0x22cce, [0x1621] = 0x22cd0,
  [0x1622] = 0x22cd5, [0x1623] = 0x22cb9, [0x1624] = 0x22cba, [0x1625] = 0x22ccf,
  [0x1626] = 0x22cbd, [0x1627] = 0x22edb, [0x1628] = 0x22faa, [0x1629] = 0x22fad,
  [0x162a] = 0x22fb8, [0x162b] = 0x22fb6, [0x162c] = 0x22fb5, [0x162d] = 0x22faf,
  [0x162e] = 0x22fb4, [0x162f] = 0x22fb7, [0x1630] = 0x22fa8, [0x1631] = 0x22fb9,
  [0x1632] = 0x22fbe, [0x1633] = 0x23082, [0x1634] = 0x230bc, [0x1635] = 0x230b8,
  [0x1636] = 0x230b6, [0x1637] = 0x230b9, [0x1638] = 0x3ac1, [0x1639] = 0x23110,
  [0x163a] = 0x23227, [0x163b] = 0x23269, [0x163c] = 0x23270, [0x163d] = 0x23256,
  [0x163e] = 0x2326b, [0x163f] = 0x2327a, [0x1640] = 0x2326c, [0x1641] = 0x2326d,
  [0x1642] = 0x2f8d5, [0x1643] = 0x267c1, [0x1644] = 0x2338d, [0x1645] = 0x23390,
  [0x1646] = 0x235b6, [0x1647] = 0x23610, [0x1648] = 0x23612, [0x1649] = 0x235ba,
  [0x164a] = 0x235bd, [0x164b] = 0x23605, [0x164c] = 0x235c2, [0x164d] = 0x23607,
  [0x164e] = 0x235f5, [0x164f] = 0x2f8e9, [0x1650] = 0x235af, [0x1651] = 0x235f4,
  [0x1652] = 0x235f7, [0x1653] = 0x235f8, [0x1654] = 0x23611, [0x1655] = 0x238f6,
  [0x1656] = 0x238f5, [0x1657] = 0x238fb, [0x1658] = 0x23901, [0x1659] = 0x23900,
  [0x165a] = 0x23902, [0x165b] = 0x238fe, [0x165c] = 0x238fa, [0x165d] = 0x238fd,
  [0x165e] = 0x3c40, [0x165f] = 0x23a37, [0x1660] = 0x23a49, [0x1661] = 0x23a4b,
  [0x1662] = 0x23a46, [0x1663] = 0x23a47, [0x1664] = 0x23ab3, [0x1665] = 0x23ab2,
  [0x1666] = 0x23ab0, [0x1667] = 0x23ab7, [0x1668] = 0x23b11, [0x1669] = 0x23bcc,
  [0x166a] = 0x23bdf, [0x166b] = 0x23bd3, [0x166c] = 0x23bd5, [0x166d] = 0x23bdb,
  [0x166e] = 0x23bc5, [0x166f] = 0x23bc8, [0x1670] = 0x23bc9, [0x1671] = 0x23be2,
  [0x1672] = 0x23bca, [0x1673] = 0x23bd1, [0x1674] = 0x23bd2, [0x1675] = 0x23bdd,
  [0x1676] = 0x23e6c, [0x1677] = 0x23e73, [0x1678] = 0x23f21, [0x1679] = 0x23ef0,
  [0x167a] = 0x23f1f, [0x167b] = 0x23f3b, [0x167c] = 0x23f22, [0x167d] = 0x23f23,
  [0x167e] = 0x23ee8, [0x167f] = 0x23edd, [0x1680] = 0x3d3f, [0x1681] = 0x23f1b,
  [0x1682] = 0x23eed, [0x1683] = 0x2f90e, [0x1684] = 0x23ef2, [0x1685] = 0x3d46,
  [0x1686] = 0x23edc, [0x1687] = 0x23ee9, [0x1688] = 0x23f1d, [0x1689] = 0x23eda,
  [0x168a] = 0x23ee6, [0x168b] = 0x24313, [0x168c] = 0x24315, [0x168d] = 0x24316,
  [0x168e] = 0x2433c, [0x168f] = 0x2430b, [0x1690] = 0x2431c, [0x1691] = 0x2433a,
  [0x1692] = 0x2433d, [0x1693] = 0x2439a, [0x1694] = 0x2431d, [0x1695] = 0x24309,
  [0x1696] = 0x24308, [0x1697] = 0x2433b, [0x1698] = 0x24522, [0x1699] = 0x24526,
  [0x169a] = 0x24525, [0x169b] = 0x24524, [0x169c] = 0x2039b, [0x169d] = 0x2458f,
  [0x169e] = 0x24590, [0x169f] = 0x245e6, [0x16a0] = 0x245e3, [0x16a1] = 0x245e5,
  [0x16a2] = 0x24611, [0x16a3] = 0x2f925, [0x16a4] = 0x246ae, [0x16a5] = 0x246be,
  [0x16a6] = 0x246b4, [0x16a7] = 0x246b3, [0x16a8] = 0x246af, [0x16a9] = 0x24691,
  [0x16aa] = 0x246c2, [0x16ab] = 0x246b6, [0x16ac] = 0x246b2, [0x16ad] = 0x24857,
  [0x16ae] = 0x2483b, [0x16af] = 0x24858, [0x16b0] = 0x24851, [0x16b1] = 0x24841,
  [0x16b2] = 0x24839, [0x16b3] = 0x3e8f, [0x16b4] = 0x24859, [0x16b5] = 0x24845,
  [0x16b6] = 0x24861, [0x16b7] = 0x248e8, [0x16b8] = 0x249fa, [0x16b9] = 0x249ea,
  [0x16ba] = 0x249ef, [0x16bb] = 0x249f2, [0x16bc] = 0x249f0, [0x16bd] = 0x24b08,
  [0x16be] = 0x24b70, [0x16bf] = 0x24b6a, [0x16c0] = 0x24b73, [0x16c1] = 0x24b68,
  [0x16c2] = 0x24bc8, [0x16c3] = 0x24bf2, [0x16c4] = 0x20b7b, [0x16c5] = 0x24cae,
  [0x16c6] = 0x24cab, [0x16c7] = 0x24cb5, [0x16c8] = 0x24caf, [0x16c9] = 0x24cb2,
  [0x16ca] = 0x24cb6, [0x16cb] = 0x24cb0, [0x16cc] = 0x24d1e, [0x16cd] = 0x24e03,
  [0x16ce] = 0x24e06, [0x16cf] = 0x24e1f, [0x16d0] = 0x3fac, [0x16d1] = 0x24e0f,
  [0x16d2] = 0x24e02, [0x16d3] = 0x24e19, [0x16d4] = 0x24e18, [0x16d5] = 0x24e22,
  [0x16d6] = 0x24e15, [0x16d7] = 0x24e07, [0x16d8] = 0x24e0d, [0x16d9] = 0x24e24,
  [0x16da] = 0x24e0c, [0x16db] = 0x24e1e, [0x16dc] = 0x24f89, [0x16dd] = 0x24f8a,
  [0x16de] = 0x2500a, [0x16df] = 0x2500b, [0x16e0] = 0x25007, [0x16e1] = 0x25004,
  [0x16e2] = 0x25009, [0x16e3] = 0x25084, [0x16e4] = 0x25083, [0x16e5] = 0x25218,
  [0x16e6] = 0x25214, [0x16e7] = 0x25205, [0x16e8] = 0x25216, [0x16e9] = 0x2520e,
  [0x16ea] = 0x25211, [0x16eb] = 0x25208, [0x16ec] = 0x2520b, [0x16ed] = 0x25215,
  [0x16ee] = 0x25085, [0x16ef] = 0x25237, [0x16f0] = 0x2522a, [0x16f1] = 0x2520d,
  [0x16f2] = 0x2520f, [0x16f3] = 0x2537e, [0x16f4] = 0x25376, [0x16f5] = 0x25377,
  [0x16f6] = 0x253ec, [0x16f7] = 0x254fc, [0x16f8] = 0x25508, [0x16f9] = 0x254ff,
  [0x16fa] = 0x25503, [0x16fb] = 0x25510, [0x16fc] = 0x25505, [0x16fd] = 0x25506,
  [0x16fe] = 0x254fa, [0x16ff] = 0x256c7, [0x1700] = 0x256c0, [0x1701] = 0x256c3,
  [0x1702] = 0x256a7, [0x1703] = 0x256a8, [0x1704] = 0x256ab, [0x1705] = 0x256c1,
  [0x1706] = 0x256aa, [0x1707] = 0x256c8, [0x1708] = 0x25743, [0x1709] = 0x25802,
  [0x170a] = 0x2580e, [0x170b] = 0x25801, [0x170c] = 0x257fe, [0x170d] = 0x25803,
  [0x170e] = 0x4168, [0x170f] = 0x25822, [0x1710] = 0x25821, [0x1711] = 0x25807,
  [0x1712] = 0x25808, [0x1713] = 0x2580c, [0x1714] = 0x259ca, [0x1715] = 0x259bc,
  [0x1716] = 0x259be, [0x1717] = 0x259bd, [0x1718] = 0x2f95c, [0x1719] = 0x259bb,
  [0x171a] = 0x259b9, [0x171b] = 0x41ab, [0x171c] = 0x259c5, [0x171d] = 0x25a99,
  [0x171e] = 0x25b6b, [0x171f] = 0x25b93, [0x1720] = 0x25b94, [0x1721] = 0x25ba9,
  [0x1722] = 0x25ba0, [0x1723] = 0x25ba6, [0x1724] = 0x2f961, [0x1725] = 0x25bab,
  [0x1726] = 0x25b9e, [0x1727] = 0x25b9b, [0x1728] = 0x25b91, [0x1729] = 0x25b99,
  [0x172a] = 0x25eb6, [0x172b] = 0x25eb8, [0x172c] = 0x25e9a, [0x172d] = 0x4291,
  [0x172e] = 0x25e98, [0x172f] = 0x25e9b, [0x1730] = 0x25eb3, [0x1731] = 0x25ea2,
  [0x1732] = 0x222aa, [0x1733] = 0x25ea0, [0x1734] = 0x25e9f, [0x1735] = 0x2605b,
  [0x1736] = 0x26070, [0x1737] = 0x26054, [0x1738] = 0x26071, [0x1739] = 0x26065,
  [0x173a] = 0x26051, [0x173b] = 0x2609d, [0x173c] = 0x26061, [0x173d] = 0x2605a,
  [0x173e] = 0x26074, [0x173f] = 0x2604c, [0x1740] = 0x2604b, [0x1741] = 0x2605e,
  [0x1742] = 0x26058, [0x1743] = 0x26053, [0x1744] = 0x26052, [0x1745] = 0x2623b,
  [0x1746] = 0x26239, [0x1747] = 0x262d6, [0x1748] = 0x262e7, [0x1749] = 0x262d7,
  [0x174a] = 0x262d8, [0x174b] = 0x262d9, [0x174c] = 0x262da, [0x174d] = 0x262db,
  [0x174e] = 0x262dc, [0x174f] = 0x262e0, [0x1750] = 0x262e4, [0x1751] = 0x262dd,
  [0x1752] = 0x262f5, [0x1753] = 0x262f1, [0x1754] = 0x26398, [0x1755] = 0x2639d,
  [0x1756] = 0x26399, [0x1757] = 0x26450, [0x1758] = 0x26449, [0x1759] = 0x7fe4,
  [0x175a] = 0x26462, [0x175b] = 0x264d7, [0x175c] = 0x26501, [0x175d] = 0x264f7,
  [0x175e] = 0x2f97d, [0x175f] = 0x2658c, [0x1760] = 0x2659c, [0x1761] = 0x8088,
  [0x1762] = 0x2661f, [0x1763] = 0x2661e, [0x1764] = 0x2673d, [0x1765] = 0x2673f,
  [0x1766] = 0x26711, [0x1767] = 0x26759, [0x1768] = 0x2673e, [0x1769] = 0x26758,
  [0x176a] = 0x26800, [0x176b] = 0x2679e, [0x176c] = 0x267b6, [0x176d] = 0x267aa,
  [0x176e] = 0x267af, [0x176f] = 0x267ac, [0x1770] = 0x4430, [0x1771] = 0x267c0,
  [0x1772] = 0x267a9, [0x1773] = 0x20b10, [0x1774] = 0x268ee, [0x1775] = 0x26977,
  [0x1776] = 0x26979, [0x1777] = 0x2697a, [0x1778] = 0x2697d, [0x1779] = 0x2697f,
  [0x177a] = 0x26983, [0x177b] = 0x26978, [0x177c] = 0x2697e, [0x177d] = 0x269e4,
  [0x177e] = 0x269e6, [0x177f] = 0x269e5, [0x1780] = 0x445d, [0x1781] = 0x26a4e,
  [0x1782] = 0x26a59, [0x1783] = 0x4475, [0x1784] = 0x2f98e, [0x1785] = 0x26a4f,
  [0x1786] = 0x26a4a, [0x1787] = 0x26ae3, [0x1788] = 0x26ade, [0x1789] = 0x26ae2,
  [0x178a] = 0x26c74, [0x178b] = 0x26cac, [0x178c] = 0x26c61, [0x178d] = 0x26c62,
  [0x178e] = 0x26c76, [0x178f] = 0x26c7a, [0x1790] = 0x26c79, [0x1791] = 0x26c60,
  [0x1792] = 0x26d64, [0x1793] = 0x26e81, [0x1794] = 0x26ddf, [0x1795] = 0x26d84,
  [0x1796] = 0x26d8a, [0x1797] = 0x26d92, [0x1798] = 0x26d79, [0x1799] = 0x26dde,
  [0x179a] = 0x26d98, [0x179b] = 0x26d6c, [0x179c] = 0x26de1, [0x179d] = 0x2f9a5,
  [0x179e] = 0x26d7c, [0x179f] = 0x26d72, [0x17a0] = 0x26d81, [0x17a1] = 0x26de0,
  [0x17a2] = 0x26d65, [0x17a3] = 0x44ff, [0x17a4] = 0x26d6a, [0x17a5] = 0x26d97,
  [0x17a6] = 0x26daa, [0x17a7] = 0x26dd3, [0x17a8] = 0x26dab, [0x17a9] = 0x26d6e,
  [0x17aa] = 0x26dac, [0x17ab] = 0x26d76, [0x17ac] = 0x26d7b, [0x17ad] = 0x26dad,
  [0x17ae] = 0x2f9a6, [0x17af] = 0x26d94, [0x17b0] = 0x26dd7, [0x17b1] = 0x26d70,
  [0x17b2] = 0x26dd5, [0x17b3] = 0x26df1, [0x17b4] = 0x26d7a, [0x17b5] = 0x26d68,
  [0x17b6] = 0x26d96, [0x17b7] = 0x450b, [0x17b8] = 0x26d73, [0x17b9] = 0x26dae,
  [0x17ba] = 0x26ddd, [0x17bb] = 0x26dda, [0x17bc] = 0x26daf, [0x17bd] = 0x26db0,
  [0x17be] = 0x26ddb, [0x17bf] = 0x26d62, [0x17c0] = 0x26df8, [0x17c1] = 0x271c2,
  [0x17c2] = 0x271c7, [0x17c3] = 0x271c8, [0x17c4] = 0x271c5, [0x17c5] = 0x271c3,
  [0x17c6] = 0x271c6, [0x17c7] = 0x272de, [0x17c8] = 0x45c8, [0x17c9] = 0x27311,
  [0x17ca] = 0x27315, [0x17cb] = 0x27328, [0x17cc] = 0x27321, [0x17cd] = 0x2732d,
  [0x17ce] = 0x27351, [0x17cf] = 0x2732b, [0x17d0] = 0x27316, [0x17d1] = 0x27324,
  [0x17d2] = 0x27335, [0x17d3] = 0x2731f, [0x17d4] = 0x27312, [0x17d5] = 0x27310,
  [0x17d6] = 0x27380, [0x17d7] = 0x2733b, [0x17d8] = 0x27329, [0x17d9] = 0x2732a,
  [0x17da] = 0x2731b, [0x17db] = 0x27318, [0x17dc] = 0x27320, [0x17dd] = 0x2733f,
  [0x17de] = 0x2731c, [0x17df] = 0x27326, [0x17e0] = 0x2730b, [0x17e1] = 0x275be,
  [0x17e2] = 0x275c4, [0x17e3] = 0x275bd, [0x17e4] = 0x275fa, [0x17e5] = 0x275fb,
  [0x17e6] = 0x276b1, [0x17e7] = 0x276b2, [0x17e8] = 0x276b0, [0x17e9] = 0x276b9,
  [0x17ea] = 0x276a6, [0x17eb] = 0x276bd, [0x17ec] = 0x276b6, [0x17ed] = 0x276b8,
  [0x17ee] = 0x276b4, [0x17ef] = 0x276b3, [0x17f0] = 0x276a7, [0x17f1] = 0x276ae,
  [0x17f2] = 0x276bc, [0x17f3] = 0x277f2, [0x17f4] = 0x277f3, [0x17f5] = 0x277f4,
  [0x17f6] = 0x2783b, [0x17f7] = 0x27840, [0x17f8] = 0x27846, [0x17f9] = 0x278f0,
  [0x17fa] = 0x278ee, [0x17fb] = 0x278e8, [0x17fc] = 0x278f1, [0x17fd] = 0x278eb,
  [0x17fe] = 0x278ef, [0x17ff] = 0x279fc, [0x1800] = 0x27a20, [0x1801] = 0x27a09,
  [0x1802] = 0x27a06, [0x1803] = 0x27a03, [0x1804] = 0x2f9cf, [0x1805] = 0x27a19,
  [0x1806] = 0x27a24, [0x1807] = 0x27a1b, [0x1808] = 0x27a1d, [0x1809] = 0x279fd,
  [0x180a] = 0x27a1e, [0x180b] = 0x279f4, [0x180c] = 0x27a01, [0x180d] = 0x27a08,
  [0x180e] = 0x27a05, [0x180f] = 0x27a23, [0x1810] = 0x279ff, [0x1811] = 0x27bea,
  [0x1812] = 0x27c4d, [0x1813] = 0x27c4e, [0x1814] = 0x27c56, [0x1815] = 0x27c57,
  [0x1816] = 0x27c51, [0x1817] = 0x27c55, [0x1818] = 0x27c54, [0x1819] = 0x27ccb,
  [0x181a] = 0x27cd4, [0x181b] = 0x27cd1, [0x181c] = 0x27ccf, [0x181d] = 0x27cd2,
  [0x181e] = 0x27cca, [0x181f] = 0x27cd6, [0x1820] = 0x27d8b, [0x1821] = 0x27d88,
  [0x1822] = 0x27d85, [0x1823] = 0x27d89, [0x1824] = 0x21a9b, [0x1825] = 0x27dbb,
  [0x1826] = 0x27d8c, [0x1827] = 0x4774, [0x1828] = 0x27d99, [0x1829] = 0x27d8a,
  [0x182a] = 0x27eec, [0x182b] = 0x27eef, [0x182c] = 0x27ef9, [0x182d] = 0x27f09,
  [0x182e] = 0x27ef8, [0x182f] = 0x27ef3, [0x1830] = 0x27f00, [0x1831] = 0x27f1d,
  [0x1832] = 0x47ac, [0x1833] = 0x27efd, [0x1834] = 0x28048, [0x1835] = 0x2805c,
  [0x1836] = 0x28055, [0x1837] = 0x2805e, [0x1838] = 0x28049, [0x1839] = 0x28063,
  [0x183a] = 0x28060, [0x183b] = 0x28053, [0x183c] = 0x28062, [0x183d] = 0x28040,
  [0x183e] = 0x28249, [0x183f] = 0x2824a, [0x1840] = 0x2824c, [0x1841] = 0x2824d,
  [0x1842] = 0x2830a, [0x1843] = 0x28306, [0x1844] = 0x2832f, [0x1845] = 0x28321,
  [0x1846] = 0x28307, [0x1847] = 0x28309, [0x1848] = 0x28302, [0x1849] = 0x28316,
  [0x184a] = 0x28303, [0x184b] = 0x2830b, [0x184c] = 0x2830f, [0x184d] = 0x28560,
  [0x184e] = 0x28568, [0x184f] = 0x2855e, [0x1850] = 0x2855d, [0x1851] = 0x28563,
  [0x1852] = 0x2855f, [0x1853] = 0x28564, [0x1854] = 0x28578, [0x1855] = 0x28561,
  [0x1856] = 0x28569, [0x1857] = 0x28565, [0x1858] = 0x286da, [0x1859] = 0x286e6,
  [0x185a] = 0x286db, [0x185b] = 0x286dc, [0x185c] = 0x28739, [0x185d] = 0x28757,
  [0x185e] = 0x28755, [0x185f] = 0x28765, [0x1860] = 0x28761, [0x1861] = 0x28754,
  [0x1862] = 0x28845, [0x1863] = 0x28841, [0x1864] = 0x2883e, [0x1865] = 0x28837,
  [0x1866] = 0x28912, [0x1867] = 0x28913, [0x1868] = 0x2892c, [0x1869] = 0x28996,
  [0x186a] = 0x2f9e9, [0x186b] = 0x2899c, [0x186c] = 0x2899d, [0x186d] = 0x28985,
  [0x186e] = 0x2f9e8, [0x186f] = 0x2899f, [0x1870] = 0x289a3, [0x1871] = 0x28c82,
  [0x1872] = 0x28c83, [0x1873] = 0x2f9ee, [0x1874] = 0x28d1d, [0x1875] = 0x28d14,
  [0x1876] = 0x28d1f, [0x1877] = 0x28d20, [0x1878] = 0x28e47, [0x1879] = 0x28e80,
  [0x187a] = 0x28ec9, [0x187b] = 0x28ef9, [0x187c] = 0x28f06, [0x187d] = 0x28ef0,
  [0x187e] = 0x28ef8, [0x187f] = 0x28eef, [0x1880] = 0x28efd, [0x1881] = 0x28ef1,
  [0x1882] = 0x28efe, [0x1883] = 0x28fb8, [0x1884] = 0x28fc0, [0x1885] = 0x28fc3,
  [0x1886] = 0x28fb5, [0x1887] = 0x28fb6, [0x1888] = 0x28fc9, [0x1889] = 0x2f9f3,
  [0x188a] = 0x28fbd, [0x188b] = 0x28fba, [0x188c] = 0x28fbf, [0x188d] = 0x28fb3,
  [0x188e] = 0x28fc6, [0x188f] = 0x28fb2, [0x1890] = 0x28fbc, [0x1891] = 0x28fb7,
  [0x1892] = 0x28fb9, [0x1893] = 0x28fc8, [0x1894] = 0x290b5, [0x1895] = 0x290b3,
  [0x1896] = 0x4a16, [0x1897] = 0x290ac, [0x1898] = 0x290a9, [0x1899] = 0x290ad,
  [0x189a] = 0x29211, [0x189b] = 0x2920d, [0x189c] = 0x29216, [0x189d] = 0x29289,
  [0x189e] = 0x2927c, [0x189f] = 0x2928b, [0x18a0] = 0x2927b, [0x18a1] = 0x29288,
  [0x18a2] = 0x29284, [0x18a3] = 0x2939d, [0x18a4] = 0x29398, [0x18a5] = 0x29488,
  [0x18a6] = 0x29486, [0x18a7] = 0x29482, [0x18a8] = 0x2948b, [0x18a9] = 0x2f9ff,
  [0x18aa] = 0x29471, [0x18ab] = 0x29472, [0x18ac] = 0x29481, [0x18ad] = 0x2948c,
  [0x18ae] = 0x29480, [0x18af] = 0x29489, [0x18b0] = 0x295bb, [0x18b1] = 0x295c1,
  [0x18b2] = 0x295be, [0x18b3] = 0x2fa01, [0x18b4] = 0x295bd, [0x18b5] = 0x296c1,
  [0x18b6] = 0x296b4, [0x18b7] = 0x296b7, [0x18b8] = 0x296c8, [0x18b9] = 0x4b3b,
  [0x18ba] = 0x296d3, [0x18bb] = 0x296d0, [0x18bc] = 0x296b0, [0x18bd] = 0x296ca,
  [0x18be] = 0x296cd, [0x18bf] = 0x29815, [0x18c0] = 0x29819, [0x18c1] = 0x29817,
  [0x18c2] = 0x2983f, [0x18c3] = 0x29840, [0x18c4] = 0x29844, [0x18c5] = 0x2987c,
  [0x18c6] = 0x29878, [0x18c7] = 0x29879, [0x18c8] = 0x29888, [0x18c9] = 0x2fa06,
  [0x18ca] = 0x29889, [0x18cb] = 0x2987d, [0x18cc] = 0x29a23, [0x18cd] = 0x29a24,
  [0x18ce] = 0x29a26, [0x18cf] = 0x29a22, [0x18d0] = 0x29a27, [0x18d1] = 0x29a1f,
  [0x18d2] = 0x29ac9, [0x18d3] = 0x29ac3, [0x18d4] = 0x29b0a, [0x18d5] = 0x29b0b,
  [0x18d6] = 0x29b04, [0x18d7] = 0x4bef, [0x18d8] = 0x29b03, [0x18d9] = 0x29b01,
  [0x18da] = 0x29b09, [0x18db] = 0x29c0f, [0x18dc] = 0x29c2e, [0x18dd] = 0x29c2d,
  [0x18de] = 0x29c91, [0x18df] = 0x29c99, [0x18e0] = 0x29c9e, [0x18e1] = 0x29c90,
  [0x18e2] = 0x29c94, [0x18e3] = 0x29c9d, [0x18e4] = 0x29c98, [0x18e5] = 0x29c9b,
  [0x18e6] = 0x29c9c, [0x18e7] = 0x29c9a, [0x18e8] = 0x29d54, [0x18e9] = 0x29f9d,
  [0x18ea] = 0x29f9f, [0x18eb] = 0x29fa3, [0x18ec] = 0x29fa4, [0x18ed] = 0x29fa5,
  [0x18ee] = 0x2a302, [0x18ef] = 0x2a417, [0x18f0] = 0x2a41b, [0x18f1] = 0x2a419,
  [0x18f2] = 0x2a4d1, [0x18f3] = 0x21537,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0x7' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x72121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l7_to_ucs4_tab[] =
{
  [0x0000] = 0x20055, [0x0001] = 0x20182, [0x0002] = 0x20189, [0x0003] = 0x20187,
  [0x0004] = 0x203be, [0x0005] = 0x203bd, [0x0006] = 0x203ca, [0x0007] = 0x203d4,
  [0x0008] = 0x203bc, [0x0009] = 0x203c4, [0x000a] = 0x203c1, [0x000b] = 0x203c2,
  [0x000c] = 0x203d7, [0x000d] = 0x20370, [0x000e] = 0x203ba, [0x000f] = 0x203e3,
  [0x0010] = 0x203bb, [0x0011] = 0x204b1, [0x0012] = 0x204b6, [0x0013] = 0x204b0,
  [0x0014] = 0x2056c, [0x0015] = 0x2060d, [0x0016] = 0x20607, [0x0017] = 0x207f1,
  [0x0018] = 0x207f5, [0x0019] = 0x207ed, [0x001a] = 0x207ef, [0x001b] = 0x207eb,
  [0x001c] = 0x207ec, [0x001d] = 0x207ee, [0x001e] = 0x20807, [0x001f] = 0x209c4,
  [0x0020] = 0x20a08, [0x0021] = 0x20ab3, [0x0022] = 0x20aad, [0x0023] = 0x20ab0,
  [0x0024] = 0x20b11, [0x0025] = 0x2018a, [0x0026] = 0x20f83, [0x0027] = 0x20f7a,
  [0x0028] = 0x20f82, [0x0029] = 0x20f68, [0x002a] = 0x20f71, [0x002b] = 0x20fb0,
  [0x002c] = 0x20faf, [0x002d] = 0x35f6, [0x002e] = 0x20f6c, [0x002f] = 0x20f6f,
  [0x0030] = 0x20f62, [0x0031] = 0x20f65, [0x0032] = 0x20f81, [0x0033] = 0x20f7f,
  [0x0034] = 0x20f64, [0x0035] = 0x20f73, [0x0036] = 0x20f6e, [0x0037] = 0x20f67,
  [0x0038] = 0x20f74, [0x0039] = 0x20f7d, [0x003a] = 0x20f78, [0x003b] = 0x20f85,
  [0x003c] = 0x20faa, [0x003d] = 0x20f88, [0x003e] = 0x20fa6, [0x003f] = 0x2f84e,
  [0x0040] = 0x20f8a, [0x0041] = 0x20f84, [0x0042] = 0x5655, [0x0043] = 0x21224,
  [0x0044] = 0x21222, [0x0045] = 0x21223, [0x0046] = 0x21432, [0x0047] = 0x2145b,
  [0x0048] = 0x2144f, [0x0049] = 0x3667, [0x004a] = 0x21426, [0x004b] = 0x2143e,
  [0x004c] = 0x21442, [0x004d] = 0x21447, [0x004e] = 0x21422, [0x004f] = 0x2069b,
  [0x0050] = 0x2144a, [0x0051] = 0x2141f, [0x0052] = 0x21449, [0x0053] = 0x2142b,
  [0x0054] = 0x21433, [0x0055] = 0x2f858, [0x0056] = 0x2142a, [0x0057] = 0x21428,
  [0x0058] = 0x2153d, [0x0059] = 0x2153c, [0x005a] = 0x215bf, [0x005b] = 0x21679,
  [0x005c] = 0x21680, [0x005d] = 0x2167c, [0x005e] = 0x2167a, [0x005f] = 0x21678,
  [0x0060] = 0x21882, [0x0061] = 0x2185c, [0x0062] = 0x2185a, [0x0063] = 0x2184b,
  [0x0064] = 0x2187c, [0x0065] = 0x2187f, [0x0066] = 0x2181f, [0x0067] = 0x21889,
  [0x0068] = 0x2188b, [0x0069] = 0x21857, [0x006a] = 0x2187e, [0x006b] = 0x2187d,
  [0x006c] = 0x21853, [0x006d] = 0x21a9f, [0x006e] = 0x21abc, [0x006f] = 0x21aa5,
  [0x0070] = 0x21a86, [0x0071] = 0x21adc, [0x0072] = 0x21aa4, [0x0073] = 0x21ab6,
  [0x0074] = 0x21aa2, [0x0075] = 0x21ab4, [0x0076] = 0x21b48, [0x0077] = 0x21b46,
  [0x0078] = 0x21ba7, [0x0079] = 0x21c14, [0x007a] = 0x21c12, [0x007b] = 0x21c11,
  [0x007c] = 0x21ce1, [0x007d] = 0x21cd0, [0x007e] = 0x21d26, [0x007f] = 0x21d28,
  [0x0080] = 0x21d29, [0x0081] = 0x21f2f, [0x0082] = 0x21f0d, [0x0083] = 0x21f25,
  [0x0084] = 0x21f14, [0x0085] = 0x21f3c, [0x0086] = 0x21f0b, [0x0087] = 0x21f11,
  [0x0088] = 0x21f30, [0x0089] = 0x21f18, [0x008a] = 0x21f19, [0x008b] = 0x21f36,
  [0x008c] = 0x21f17, [0x008d] = 0x21f10, [0x008e] = 0x21f15, [0x008f] = 0x22003,
  [0x0090] = 0x22029, [0x0091] = 0x2204c, [0x0092] = 0x2213f, [0x0093] = 0x2213e,
  [0x0094] = 0x22134, [0x0095] = 0x22142, [0x0096] = 0x22135, [0x0097] = 0x207f3,
  [0x0098] = 0x221cd, [0x0099] = 0x221d1, [0x009a] = 0x221d2, [0x009b] = 0x222ad,
  [0x009c] = 0x222ba, [0x009d] = 0x222c1, [0x009e] = 0x222bb, [0x009f] = 0x222b4,
  [0x00a0] = 0x222bc, [0x00a1] = 0x2236c, [0x00a2] = 0x22392, [0x00a3] = 0x22421,
  [0x00a4] = 0x22424, [0x00a5] = 0x23f19, [0x00a6] = 0x224ad, [0x00a7] = 0x224ae,
  [0x00a8] = 0x224ab, [0x00a9] = 0x224ac, [0x00aa] = 0x22570, [0x00ab] = 0x22576,
  [0x00ac] = 0x22571, [0x00ad] = 0x22577, [0x00ae] = 0x617f, [0x00af] = 0x227ec,
  [0x00b0] = 0x2281c, [0x00b1] = 0x227fb, [0x00b2] = 0x227ee, [0x00b3] = 0x228e2,
  [0x00b4] = 0x227fd, [0x00b5] = 0x2281e, [0x00b6] = 0x2283b, [0x00b7] = 0x2284a,
  [0x00b8] = 0x2f8ae, [0x00b9] = 0x3992, [0x00ba] = 0x2287d, [0x00bb] = 0x22856,
  [0x00bc] = 0x22844, [0x00bd] = 0x2284b, [0x00be] = 0x229fa, [0x00bf] = 0x229e8,
  [0x00c0] = 0x229f8, [0x00c1] = 0x229ff, [0x00c2] = 0x229f7, [0x00c3] = 0x229f6,
  [0x00c4] = 0x229fe, [0x00c5] = 0x22a07, [0x00c6] = 0x22a5d, [0x00c7] = 0x22cd4,
  [0x00c8] = 0x2f8c3, [0x00c9] = 0x22cc8, [0x00ca] = 0x22cc5, [0x00cb] = 0x2f8c0,
  [0x00cc] = 0x22cc9, [0x00cd] = 0x22d4d, [0x00ce] = 0x22d20, [0x00cf] = 0x22d36,
  [0x00d0] = 0x22d35, [0x00d1] = 0x22d88, [0x00d2] = 0x22d25, [0x00d3] = 0x22d21,
  [0x00d4] = 0x22d26, [0x00d5] = 0x3a45, [0x00d6] = 0x22d3f, [0x00d7] = 0x22d3b,
  [0x00d8] = 0x22d24, [0x00d9] = 0x22d43, [0x00da] = 0x22d4e, [0x00db] = 0x22d4f,
  [0x00dc] = 0x22d44, [0x00dd] = 0x22d40, [0x00de] = 0x22d41, [0x00df] = 0x22d39,
  [0x00e0] = 0x22d2b, [0x00e1] = 0x22edd, [0x00e2] = 0x22ede, [0x00e3] = 0x22fcd,
  [0x00e4] = 0x22fcb, [0x00e5] = 0x22fdb, [0x00e6] = 0x22fd8, [0x00e7] = 0x22fd9,
  [0x00e8] = 0x22fd2, [0x00e9] = 0x22fca, [0x00ea] = 0x22fd1, [0x00eb] = 0x22fd4,
  [0x00ec] = 0x22ff9, [0x00ed] = 0x22fd0, [0x00ee] = 0x2305e, [0x00ef] = 0x2305d,
  [0x00f0] = 0x2308a, [0x00f1] = 0x23114, [0x00f2] = 0x23283, [0x00f3] = 0x23284,
  [0x00f4] = 0x2329b, [0x00f5] = 0x23282, [0x00f6] = 0x2327d, [0x00f7] = 0x2327c,
  [0x00f8] = 0x23299, [0x00f9] = 0x2327e, [0x00fa] = 0x2329a, [0x00fb] = 0x23295,
  [0x00fc] = 0x2327b, [0x00fd] = 0x23286, [0x00fe] = 0x23348, [0x00ff] = 0x23392,
  [0x0100] = 0x235b2, [0x0101] = 0x23642, [0x0102] = 0x23672, [0x0103] = 0x23615,
  [0x0104] = 0x23630, [0x0105] = 0x23643, [0x0106] = 0x2365b, [0x0107] = 0x2361d,
  [0x0108] = 0x23673, [0x0109] = 0x2362d, [0x010a] = 0x3bc9, [0x010b] = 0x23621,
  [0x010c] = 0x2361c, [0x010d] = 0x2f8ea, [0x010e] = 0x23629, [0x010f] = 0x2361e,
  [0x0110] = 0x23633, [0x0111] = 0x23910, [0x0112] = 0x2390e, [0x0113] = 0x2f8f1,
  [0x0114] = 0x2390d, [0x0115] = 0x23911, [0x0116] = 0x23912, [0x0117] = 0x23917,
  [0x0118] = 0x23909, [0x0119] = 0x2390f, [0x011a] = 0x23998, [0x011b] = 0x23994,
  [0x011c] = 0x23996, [0x011d] = 0x23999, [0x011e] = 0x23995, [0x011f] = 0x23997,
  [0x0120] = 0x23a50, [0x0121] = 0x23a52, [0x0122] = 0x23a4d, [0x0123] = 0x23a57,
  [0x0124] = 0x23a53, [0x0125] = 0x23a56, [0x0126] = 0x23ab8, [0x0127] = 0x23b12,
  [0x0128] = 0x23bf1, [0x0129] = 0x23bf0, [0x012a] = 0x23bee, [0x012b] = 0x23bef,
  [0x012c] = 0x23be5, [0x012d] = 0x23beb, [0x012e] = 0x23be7, [0x012f] = 0x23bf2,
  [0x0130] = 0x23be6, [0x0131] = 0x23bf7, [0x0132] = 0x23c09, [0x0133] = 0x23c6b,
  [0x0134] = 0x23c6a, [0x0135] = 0x23e79, [0x0136] = 0x23ecd, [0x0137] = 0x23eef,
  [0x0138] = 0x23ee4, [0x0139] = 0x23ee3, [0x013a] = 0x23f6c, [0x013b] = 0x23f67,
  [0x013c] = 0x23f62, [0x013d] = 0x23f58, [0x013e] = 0x23f69, [0x013f] = 0x23fab,
  [0x0140] = 0x23f57, [0x0141] = 0x23fad, [0x0142] = 0x23f54, [0x0143] = 0x23fae,
  [0x0144] = 0x23f76, [0x0145] = 0x23f49, [0x0146] = 0x23f45, [0x0147] = 0x23f4f,
  [0x0148] = 0x2f90f, [0x0149] = 0x23f50, [0x014a] = 0x23f7d, [0x014b] = 0x23f44,
  [0x014c] = 0x23f66, [0x014d] = 0x23f48, [0x014e] = 0x23fa3, [0x014f] = 0x23f51,
  [0x0150] = 0x23f53, [0x0151] = 0x2f910, [0x0152] = 0x23f60, [0x0153] = 0x23f47,
  [0x0154] = 0x23f5e, [0x0155] = 0x23f85, [0x0156] = 0x23fac, [0x0157] = 0x23f6d,
  [0x0158] = 0x23f75, [0x0159] = 0x23fa8, [0x015a] = 0x3d55, [0x015b] = 0x23f43,
  [0x015c] = 0x23f4e, [0x015d] = 0x23f5c, [0x015e] = 0x23f56, [0x015f] = 0x23f6e,
  [0x0160] = 0x23f63, [0x0161] = 0x23f6a, [0x0162] = 0x24356, [0x0163] = 0x24312,
  [0x0164] = 0x2435c, [0x0165] = 0x24358, [0x0166] = 0x24355, [0x0167] = 0x2437e,
  [0x0168] = 0x243a8, [0x0169] = 0x24311, [0x016a] = 0x24359, [0x016b] = 0x2435a,
  [0x016c] = 0x24367, [0x016d] = 0x24354, [0x016e] = 0x2435b, [0x016f] = 0x243f0,
  [0x0170] = 0x24529, [0x0171] = 0x2452c, [0x0172] = 0x2452a, [0x0173] = 0x24527,
  [0x0174] = 0x24567, [0x0175] = 0x24568, [0x0176] = 0x24593, [0x0177] = 0x24595,
  [0x0178] = 0x2459a, [0x0179] = 0x245ee, [0x017a] = 0x245f1, [0x017b] = 0x245ed,
  [0x017c] = 0x245e9, [0x017d] = 0x246cb, [0x017e] = 0x246df, [0x017f] = 0x246db,
  [0x0180] = 0x2483c, [0x0181] = 0x2484e, [0x0182] = 0x2486c, [0x0183] = 0x2487b,
  [0x0184] = 0x2487c, [0x0185] = 0x24883, [0x0186] = 0x2486b, [0x0187] = 0x24866,
  [0x0188] = 0x24863, [0x0189] = 0x248a0, [0x018a] = 0x24885, [0x018b] = 0x24868,
  [0x018c] = 0x248a2, [0x018d] = 0x249ec, [0x018e] = 0x3ee8, [0x018f] = 0x249e7,
  [0x0190] = 0x24a10, [0x0191] = 0x24a18, [0x0192] = 0x24a3a, [0x0193] = 0x24a1a,
  [0x0194] = 0x24a38, [0x0195] = 0x24a41, [0x0196] = 0x24a1b, [0x0197] = 0x24a14,
  [0x0198] = 0x24a1c, [0x0199] = 0x24a1d, [0x019a] = 0x24b12, [0x019b] = 0x24b0d,
  [0x019c] = 0x24b07, [0x019d] = 0x24b7e, [0x019e] = 0x24b7f, [0x019f] = 0x20ab9,
  [0x01a0] = 0x24b80, [0x01a1] = 0x24b7a, [0x01a2] = 0x24bf3, [0x01a3] = 0x24cb9,
  [0x01a4] = 0x24cbb, [0x01a5] = 0x24cba, [0x01a6] = 0x24cbd, [0x01a7] = 0x24cb7,
  [0x01a8] = 0x24cbf, [0x01a9] = 0x24d21, [0x01aa] = 0x24e3a, [0x01ab] = 0x24e3d,
  [0x01ac] = 0x24e36, [0x01ad] = 0x24e49, [0x01ae] = 0x24e81, [0x01af] = 0x3fb6,
  [0x01b0] = 0x24e3f, [0x01b1] = 0x24e4d, [0x01b2] = 0x24e3e, [0x01b3] = 0x3fbd,
  [0x01b4] = 0x24e30, [0x01b5] = 0x24e4f, [0x01b6] = 0x24f37, [0x01b7] = 0x24f98,
  [0x01b8] = 0x24f99, [0x01b9] = 0x25011, [0x01ba] = 0x2500d, [0x01bb] = 0x25088,
  [0x01bc] = 0x2508a, [0x01bd] = 0x2521c, [0x01be] = 0x25255, [0x01bf] = 0x25256,
  [0x01c0] = 0x25262, [0x01c1] = 0x25257, [0x01c2] = 0x25242, [0x01c3] = 0x25210,
  [0x01c4] = 0x2524d, [0x01c5] = 0x2525a, [0x01c6] = 0x25266, [0x01c7] = 0x25246,
  [0x01c8] = 0x2287e, [0x01c9] = 0x2524f, [0x01ca] = 0x262f9, [0x01cb] = 0x25387,
  [0x01cc] = 0x25384, [0x01cd] = 0x253f4, [0x01ce] = 0x253f5, [0x01cf] = 0x253f2,
  [0x01d0] = 0x253f7, [0x01d1] = 0x25545, [0x01d2] = 0x2f950, [0x01d3] = 0x25549,
  [0x01d4] = 0x25535, [0x01d5] = 0x254fd, [0x01d6] = 0x25530, [0x01d7] = 0x25543,
  [0x01d8] = 0x25546, [0x01d9] = 0x40d6, [0x01da] = 0x25533, [0x01db] = 0x256c6,
  [0x01dc] = 0x2f956, [0x01dd] = 0x256d4, [0x01de] = 0x271d8, [0x01df] = 0x256e0,
  [0x01e0] = 0x20fa7, [0x01e1] = 0x25744, [0x01e2] = 0x20398, [0x01e3] = 0x25854,
  [0x01e4] = 0x2583e, [0x01e5] = 0x25835, [0x01e6] = 0x2583a, [0x01e7] = 0x4171,
  [0x01e8] = 0x25834, [0x01e9] = 0x25839, [0x01ea] = 0x25846, [0x01eb] = 0x2585a,
  [0x01ec] = 0x2585b, [0x01ed] = 0x2583c, [0x01ee] = 0x259dc, [0x01ef] = 0x259d6,
  [0x01f0] = 0x259d0, [0x01f1] = 0x2f95e, [0x01f2] = 0x25aac, [0x01f3] = 0x2f95d,
  [0x01f4] = 0x25aa5, [0x01f5] = 0x25bf4, [0x01f6] = 0x25c24, [0x01f7] = 0x25be0,
  [0x01f8] = 0x25bfd, [0x01f9] = 0x25bf2, [0x01fa] = 0x25bfc, [0x01fb] = 0x25c02,
  [0x01fc] = 0x25bf1, [0x01fd] = 0x25bff, [0x01fe] = 0x25bd2, [0x01ff] = 0x25bd8,
  [0x0200] = 0x25bd3, [0x0201] = 0x25bf9, [0x0202] = 0x2f963, [0x0203] = 0x25bed,
  [0x0204] = 0x25bf7, [0x0205] = 0x25bf0, [0x0206] = 0x25bd7, [0x0207] = 0x25c09,
  [0x0208] = 0x25bf5, [0x0209] = 0x25ec2, [0x020a] = 0x25ec5, [0x020b] = 0x25ecf,
  [0x020c] = 0x4298, [0x020d] = 0x25ec1, [0x020e] = 0x25ec6, [0x020f] = 0x25ed4,
  [0x0210] = 0x25ece, [0x0211] = 0x2609b, [0x0212] = 0x26099, [0x0213] = 0x2608f,
  [0x0214] = 0x260b7, [0x0215] = 0x26082, [0x0216] = 0x26093, [0x0217] = 0x260b4,
  [0x0218] = 0x260c1, [0x0219] = 0x26090, [0x021a] = 0x2609a, [0x021b] = 0x26094,
  [0x021c] = 0x2609c, [0x021d] = 0x260b5, [0x021e] = 0x2f96f, [0x021f] = 0x2f96e,
  [0x0220] = 0x2609f, [0x0221] = 0x2608e, [0x0222] = 0x2609e, [0x0223] = 0x2623c,
  [0x0224] = 0x26247, [0x0225] = 0x26241, [0x0226] = 0x262eb, [0x0227] = 0x262ec,
  [0x0228] = 0x262ed, [0x0229] = 0x262ee, [0x022a] = 0x262df, [0x022b] = 0x262f2,
  [0x022c] = 0x2630d, [0x022d] = 0x2630e, [0x022e] = 0x20186, [0x022f] = 0x263a9,
  [0x0230] = 0x263a4, [0x0231] = 0x263a5, [0x0232] = 0x2645f, [0x0233] = 0x2645a,
  [0x0234] = 0x26466, [0x0235] = 0x26468, [0x0236] = 0x2646a, [0x0237] = 0x26464,
  [0x0238] = 0x26465, [0x0239] = 0x264c1, [0x023a] = 0x264d8, [0x023b] = 0x2f97b,
  [0x023c] = 0x264d9, [0x023d] = 0x26509, [0x023e] = 0x265aa, [0x023f] = 0x265a3,
  [0x0240] = 0x265b9, [0x0241] = 0x259e2, [0x0242] = 0x265a5, [0x0243] = 0x265ab,
  [0x0244] = 0x26620, [0x0245] = 0x26789, [0x0246] = 0x2678b, [0x0247] = 0x267db,
  [0x0248] = 0x267e4, [0x0249] = 0x267ef, [0x024a] = 0x267e2, [0x024b] = 0x267e9,
  [0x024c] = 0x267f0, [0x024d] = 0x267f3, [0x024e] = 0x267dd, [0x024f] = 0x2681e,
  [0x0250] = 0x2691d, [0x0251] = 0x26946, [0x0252] = 0x2697c, [0x0253] = 0x26980,
  [0x0254] = 0x26981, [0x0255] = 0x26982, [0x0256] = 0x26987, [0x0257] = 0x26988,
  [0x0258] = 0x2698a, [0x0259] = 0x2698c, [0x025a] = 0x269e7, [0x025b] = 0x269eb,
  [0x025c] = 0x269ea, [0x025d] = 0x26a68, [0x025e] = 0x26a6b, [0x025f] = 0x26a69,
  [0x0260] = 0x26a66, [0x0261] = 0x26a6a, [0x0262] = 0x26a65, [0x0263] = 0x26ae5,
  [0x0264] = 0x26ae4, [0x0265] = 0x26ae6, [0x0266] = 0x26d18, [0x0267] = 0x26e24,
  [0x0268] = 0x26e48, [0x0269] = 0x26e2f, [0x026a] = 0x26e7e, [0x026b] = 0x26d89,
  [0x026c] = 0x26e1e, [0x026d] = 0x26e49, [0x026e] = 0x26e26, [0x026f] = 0x26e09,
  [0x0270] = 0x26e34, [0x0271] = 0x26e2d, [0x0272] = 0x26e4a, [0x0273] = 0x26e15,
  [0x0274] = 0x2f9a9, [0x0275] = 0x26e33, [0x0276] = 0x26e28, [0x0277] = 0x26e27,
  [0x0278] = 0x26e20, [0x0279] = 0x26e2b, [0x027a] = 0x26e29, [0x027b] = 0x26e6a,
  [0x027c] = 0x26e0b, [0x027d] = 0x26e0e, [0x027e] = 0x26e1c, [0x027f] = 0x26dff,
  [0x0280] = 0x2f9aa, [0x0281] = 0x26e35, [0x0282] = 0x26e11, [0x0283] = 0x26e4b,
  [0x0284] = 0x26e4c, [0x0285] = 0x26e1f, [0x0286] = 0x26e0f, [0x0287] = 0x26e39,
  [0x0288] = 0x26e68, [0x0289] = 0x26e1d, [0x028a] = 0x26e08, [0x028b] = 0x26e0c,
  [0x028c] = 0x26e0d, [0x028d] = 0x26e62, [0x028e] = 0x26eaf, [0x028f] = 0x26e32,
  [0x0290] = 0x26e2c, [0x0291] = 0x26e64, [0x0292] = 0x26e04, [0x0293] = 0x26e16,
  [0x0294] = 0x26e4d, [0x0295] = 0x26e07, [0x0296] = 0x26eae, [0x0297] = 0x271d4,
  [0x0298] = 0x271d1, [0x0299] = 0x271d5, [0x029a] = 0x271d0, [0x029b] = 0x27371,
  [0x029c] = 0x2735d, [0x029d] = 0x2735b, [0x029e] = 0x2737a, [0x029f] = 0x2f9bd,
  [0x02a0] = 0x2737f, [0x02a1] = 0x27379, [0x02a2] = 0x27367, [0x02a3] = 0x2739e,
  [0x02a4] = 0x27394, [0x02a5] = 0x27364, [0x02a6] = 0x2735e, [0x02a7] = 0x27381,
  [0x02a8] = 0x2735a, [0x02a9] = 0x27357, [0x02aa] = 0x2737c, [0x02ab] = 0x27398,
  [0x02ac] = 0x27366, [0x02ad] = 0x2737e, [0x02ae] = 0x27382, [0x02af] = 0x27368,
  [0x02b0] = 0x2735f, [0x02b1] = 0x27363, [0x02b2] = 0x27397, [0x02b3] = 0x45cd,
  [0x02b4] = 0x27353, [0x02b5] = 0x2737b, [0x02b6] = 0x2737d, [0x02b7] = 0x27378,
  [0x02b8] = 0x27406, [0x02b9] = 0x275c0, [0x02ba] = 0x275c2, [0x02bb] = 0x275c3,
  [0x02bc] = 0x276dc, [0x02bd] = 0x276e6, [0x02be] = 0x276ec, [0x02bf] = 0x276f2,
  [0x02c0] = 0x276e8, [0x02c1] = 0x276eb, [0x02c2] = 0x276ea, [0x02c3] = 0x276e5,
  [0x02c4] = 0x276e0, [0x02c5] = 0x276d0, [0x02c6] = 0x276d8, [0x02c7] = 0x276d5,
  [0x02c8] = 0x276d3, [0x02c9] = 0x276e4, [0x02ca] = 0x276f3, [0x02cb] = 0x276db,
  [0x02cc] = 0x2772f, [0x02cd] = 0x277f9, [0x02ce] = 0x2785d, [0x02cf] = 0x2785b,
  [0x02d0] = 0x2784c, [0x02d1] = 0x27858, [0x02d2] = 0x27860, [0x02d3] = 0x27853,
  [0x02d4] = 0x27850, [0x02d5] = 0x27907, [0x02d6] = 0x27908, [0x02d7] = 0x278f4,
  [0x02d8] = 0x278f6, [0x02d9] = 0x27904, [0x02da] = 0x278fd, [0x02db] = 0x27905,
  [0x02dc] = 0x27a28, [0x02dd] = 0x27a56, [0x02de] = 0x27a42, [0x02df] = 0x27a5c,
  [0x02e0] = 0x27a3d, [0x02e1] = 0x27a2f, [0x02e2] = 0x27a2b, [0x02e3] = 0x27a58,
  [0x02e4] = 0x27a61, [0x02e5] = 0x27a2c, [0x02e6] = 0x27a51, [0x02e7] = 0x27a50,
  [0x02e8] = 0x27a3c, [0x02e9] = 0x27a36, [0x02ea] = 0x46f6, [0x02eb] = 0x27a60,
  [0x02ec] = 0x27a5b, [0x02ed] = 0x27a2e, [0x02ee] = 0x27a40, [0x02ef] = 0x27a5f,
  [0x02f0] = 0x27a26, [0x02f1] = 0x27a2d, [0x02f2] = 0x27a5e, [0x02f3] = 0x27a3b,
  [0x02f4] = 0x27bef, [0x02f5] = 0x27bf3, [0x02f6] = 0x27bee, [0x02f7] = 0x27bed,
  [0x02f8] = 0x27c5a, [0x02f9] = 0x27c62, [0x02fa] = 0x27c5b, [0x02fb] = 0x27c5f,
  [0x02fc] = 0x27c5e, [0x02fd] = 0x27c64, [0x02fe] = 0x27ce0, [0x02ff] = 0x27ce1,
  [0x0300] = 0x27ce5, [0x0301] = 0x27da5, [0x0302] = 0x27d9b, [0x0303] = 0x27d9f,
  [0x0304] = 0x27d9c, [0x0305] = 0x477a, [0x0306] = 0x27daa, [0x0307] = 0x27da4,
  [0x0308] = 0x27da3, [0x0309] = 0x27e67, [0x030a] = 0x27f26, [0x030b] = 0x27f1a,
  [0x030c] = 0x47b8, [0x030d] = 0x27f1c, [0x030e] = 0x27f15, [0x030f] = 0x27f1b,
  [0x0310] = 0x27f18, [0x0311] = 0x27f1e, [0x0312] = 0x27f25, [0x0313] = 0x2f9dc,
  [0x0314] = 0x2807b, [0x0315] = 0x28075, [0x0316] = 0x28093, [0x0317] = 0x2808f,
  [0x0318] = 0x2807c, [0x0319] = 0x2809b, [0x031a] = 0x28088, [0x031b] = 0x28091,
  [0x031c] = 0x2807e, [0x031d] = 0x4801, [0x031e] = 0x28086, [0x031f] = 0x2825b,
  [0x0320] = 0x28254, [0x0321] = 0x28335, [0x0322] = 0x28340, [0x0323] = 0x28337,
  [0x0324] = 0x28329, [0x0325] = 0x28326, [0x0326] = 0x28323, [0x0327] = 0x28330,
  [0x0328] = 0x28320, [0x0329] = 0x28328, [0x032a] = 0x28332, [0x032b] = 0x2832a,
  [0x032c] = 0x28331, [0x032d] = 0x28324, [0x032e] = 0x28341, [0x032f] = 0x28342,
  [0x0330] = 0x28343, [0x0331] = 0x28325, [0x0332] = 0x28338, [0x0333] = 0x2841c,
  [0x0334] = 0x2841d, [0x0335] = 0x2841e, [0x0336] = 0x284f6, [0x0337] = 0x285a5,
  [0x0338] = 0x28589, [0x0339] = 0x2859d, [0x033a] = 0x2858a, [0x033b] = 0x28587,
  [0x033c] = 0x285ab, [0x033d] = 0x28586, [0x033e] = 0x2859e, [0x033f] = 0x28593,
  [0x0340] = 0x285a1, [0x0341] = 0x28588, [0x0342] = 0x285a3, [0x0343] = 0x2859f,
  [0x0344] = 0x285a0, [0x0345] = 0x286ff, [0x0346] = 0x28700, [0x0347] = 0x28702,
  [0x0348] = 0x286f8, [0x0349] = 0x286f7, [0x034a] = 0x28701, [0x034b] = 0x28787,
  [0x034c] = 0x28789, [0x034d] = 0x28776, [0x034e] = 0x2876e, [0x034f] = 0x28777,
  [0x0350] = 0x28782, [0x0351] = 0x28785, [0x0352] = 0x28783, [0x0353] = 0x28784,
  [0x0354] = 0x28857, [0x0355] = 0x2884b, [0x0356] = 0x28856, [0x0357] = 0x2885d,
  [0x0358] = 0x28860, [0x0359] = 0x2884d, [0x035a] = 0x28855, [0x035b] = 0x28854,
  [0x035c] = 0x28853, [0x035d] = 0x28850, [0x035e] = 0x28863, [0x035f] = 0x28862,
  [0x0360] = 0x28917, [0x0361] = 0x28916, [0x0362] = 0x2892d, [0x0363] = 0x289d9,
  [0x0364] = 0x289bf, [0x0365] = 0x492f, [0x0366] = 0x289b2, [0x0367] = 0x289ac,
  [0x0368] = 0x289b0, [0x0369] = 0x289cf, [0x036a] = 0x289b8, [0x036b] = 0x4931,
  [0x036c] = 0x289d3, [0x036d] = 0x289b1, [0x036e] = 0x289af, [0x036f] = 0x289ce,
  [0x0370] = 0x289b4, [0x0371] = 0x289b7, [0x0372] = 0x289d5, [0x0373] = 0x499c,
  [0x0374] = 0x28d38, [0x0375] = 0x28d28, [0x0376] = 0x28d3f, [0x0377] = 0x28d40,
  [0x0378] = 0x28d41, [0x0379] = 0x28d33, [0x037a] = 0x28d2b, [0x037b] = 0x28d2e,
  [0x037c] = 0x28f11, [0x037d] = 0x28f18, [0x037e] = 0x28f17, [0x037f] = 0x28f19,
  [0x0380] = 0x28f16, [0x0381] = 0x49e6, [0x0382] = 0x28f0d, [0x0383] = 0x28f26,
  [0x0384] = 0x28f79, [0x0385] = 0x28fd8, [0x0386] = 0x28ff3, [0x0387] = 0x28fd7,
  [0x0388] = 0x28fcc, [0x0389] = 0x28fcf, [0x038a] = 0x28fcb, [0x038b] = 0x28fd1,
  [0x038c] = 0x28fce, [0x038d] = 0x28fd4, [0x038e] = 0x28fd5, [0x038f] = 0x28fd3,
  [0x0390] = 0x28fd6, [0x0391] = 0x28fcd, [0x0392] = 0x28fda, [0x0393] = 0x28fd2,
  [0x0394] = 0x28fde, [0x0395] = 0x290cc, [0x0396] = 0x290b8, [0x0397] = 0x290b7,
  [0x0398] = 0x290c2, [0x0399] = 0x290c3, [0x039a] = 0x291fa, [0x039b] = 0x291f9,
  [0x039c] = 0x2921f, [0x039d] = 0x2921d, [0x039e] = 0x24382, [0x039f] = 0x292a6,
  [0x03a0] = 0x29290, [0x03a1] = 0x2929e, [0x03a2] = 0x29298, [0x03a3] = 0x29293,
  [0x03a4] = 0x29292, [0x03a5] = 0x292a5, [0x03a6] = 0x29295, [0x03a7] = 0x293a9,
  [0x03a8] = 0x293a8, [0x03a9] = 0x293aa, [0x03aa] = 0x293ab, [0x03ab] = 0x293ac,
  [0x03ac] = 0x2941f, [0x03ad] = 0x29422, [0x03ae] = 0x294aa, [0x03af] = 0x29496,
  [0x03b0] = 0x29491, [0x03b1] = 0x294ab, [0x03b2] = 0x294a2, [0x03b3] = 0x294a4,
  [0x03b4] = 0x2949d, [0x03b5] = 0x29497, [0x03b6] = 0x2947d, [0x03b7] = 0x2949f,
  [0x03b8] = 0x2949c, [0x03b9] = 0x29499, [0x03ba] = 0x294a6, [0x03bb] = 0x29492,
  [0x03bc] = 0x294a7, [0x03bd] = 0x295c9, [0x03be] = 0x295c8, [0x03bf] = 0x295c7,
  [0x03c0] = 0x295cb, [0x03c1] = 0x29674, [0x03c2] = 0x296c9, [0x03c3] = 0x4b38,
  [0x03c4] = 0x296d2, [0x03c5] = 0x296db, [0x03c6] = 0x296eb, [0x03c7] = 0x296e9,
  [0x03c8] = 0x296ef, [0x03c9] = 0x296ee, [0x03ca] = 0x296f9, [0x03cb] = 0x296de,
  [0x03cc] = 0x29816, [0x03cd] = 0x29845, [0x03ce] = 0x2987f, [0x03cf] = 0x29898,
  [0x03d0] = 0x2989f, [0x03d1] = 0x29886, [0x03d2] = 0x29890, [0x03d3] = 0x2989a,
  [0x03d4] = 0x2989d, [0x03d5] = 0x2988d, [0x03d6] = 0x2989c, [0x03d7] = 0x298a0,
  [0x03d8] = 0x298a7, [0x03d9] = 0x2988f, [0x03da] = 0x29a41, [0x03db] = 0x29a35,
  [0x03dc] = 0x29a37, [0x03dd] = 0x29a30, [0x03de] = 0x29a3a, [0x03df] = 0x29a36,
  [0x03e0] = 0x2fa08, [0x03e1] = 0x29aca, [0x03e2] = 0x29b12, [0x03e3] = 0x29b15,
  [0x03e4] = 0x29b22, [0x03e5] = 0x29b13, [0x03e6] = 0x29b18, [0x03e7] = 0x29b21,
  [0x03e8] = 0x29c10, [0x03e9] = 0x29c20, [0x03ea] = 0x29c2f, [0x03eb] = 0x29ca5,
  [0x03ec] = 0x29ca3, [0x03ed] = 0x29cb6, [0x03ee] = 0x29cab, [0x03ef] = 0x29d68,
  [0x03f0] = 0x29d67, [0x03f1] = 0x29d6f, [0x03f2] = 0x29d77, [0x03f3] = 0x29d64,
  [0x03f4] = 0x29d78, [0x03f5] = 0x4c3a, [0x03f6] = 0x29d76, [0x03f7] = 0x29d6e,
  [0x03f8] = 0x4cb1, [0x03f9] = 0x29fd0, [0x03fa] = 0x29fb3, [0x03fb] = 0x29fcb,
  [0x03fc] = 0x29fd6, [0x03fd] = 0x29fb1, [0x03fe] = 0x29fb4, [0x03ff] = 0x29fd7,
  [0x0400] = 0x29fc6, [0x0401] = 0x29fba, [0x0402] = 0x29fd1, [0x0403] = 0x29fd2,
  [0x0404] = 0x29fb5, [0x0405] = 0x29fc4, [0x0406] = 0x29fcc, [0x0407] = 0x29fbb,
  [0x0408] = 0x29fb2, [0x0409] = 0x2a25b, [0x040a] = 0x2a25a, [0x040b] = 0x2a259,
  [0x040c] = 0x2a299, [0x040d] = 0x2a294, [0x040e] = 0x2a296, [0x040f] = 0x2a293,
  [0x0410] = 0x2a291, [0x0411] = 0x2a28f, [0x0412] = 0x2a298, [0x0413] = 0x2a30c,
  [0x0414] = 0x2a30a, [0x0415] = 0x2a30e, [0x0416] = 0x2a312, [0x0417] = 0x2a311,
  [0x0418] = 0x2a30b, [0x0419] = 0x2a39a, [0x041a] = 0x4d47, [0x041b] = 0x2a3b4,
  [0x041c] = 0x4d51, [0x041d] = 0x2a41c, [0x041e] = 0x2a420, [0x041f] = 0x2a421,
  [0x0420] = 0x2a4d3, [0x0421] = 0x2a4d2, [0x0422] = 0x2a538, [0x0424] = 0x24ccf,
  [0x0425] = 0x2011d, [0x0426] = 0x20192, [0x0427] = 0x2018f, [0x0428] = 0x2018c,
  [0x0429] = 0x203ea, [0x042a] = 0x204b9, [0x042b] = 0x2056e, [0x042c] = 0x205a0,
  [0x042d] = 0x291dd, [0x042e] = 0x20612, [0x042f] = 0x20617, [0x0430] = 0x2066d,
  [0x0431] = 0x20805, [0x0432] = 0x207fb, [0x0433] = 0x207fd, [0x0434] = 0x20806,
  [0x0435] = 0x20800, [0x0436] = 0x207ff, [0x0437] = 0x207fe, [0x0438] = 0x208b6,
  [0x0439] = 0x20905, [0x043a] = 0x20907, [0x043b] = 0x20abc, [0x043c] = 0x20abb,
  [0x043d] = 0x20b8b, [0x043e] = 0x20b87, [0x043f] = 0x27bfb, [0x0440] = 0x20fe7,
  [0x0441] = 0x20fe6, [0x0442] = 0x20fe2, [0x0443] = 0x285d3, [0x0444] = 0x20fd9,
  [0x0445] = 0x20fca, [0x0446] = 0x21012, [0x0447] = 0x21010, [0x0448] = 0x20fdc,
  [0x0449] = 0x20fd0, [0x044a] = 0x20f8b, [0x044b] = 0x21013, [0x044c] = 0x22a10,
  [0x044d] = 0x20fda, [0x044e] = 0x2f84f, [0x044f] = 0x20ffd, [0x0450] = 0x20fe5,
  [0x0451] = 0x20fcc, [0x0452] = 0x21039, [0x0453] = 0x20fc7, [0x0454] = 0x20fd8,
  [0x0455] = 0x21460, [0x0456] = 0x21464, [0x0457] = 0x21486, [0x0458] = 0x2145e,
  [0x0459] = 0x21466, [0x045a] = 0x2145f, [0x045b] = 0x21477, [0x045c] = 0x21482,
  [0x045d] = 0x21496, [0x045e] = 0x2153e, [0x045f] = 0x21579, [0x0460] = 0x2157a,
  [0x0461] = 0x21688, [0x0462] = 0x21689, [0x0463] = 0x2168c, [0x0464] = 0x243e0,
  [0x0465] = 0x2167b, [0x0466] = 0x21686, [0x0467] = 0x21894, [0x0468] = 0x21897,
  [0x0469] = 0x218b7, [0x046a] = 0x218b5, [0x046b] = 0x218b8, [0x046c] = 0x21893,
  [0x046d] = 0x218b2, [0x046e] = 0x218b4, [0x046f] = 0x21899, [0x0470] = 0x218c3,
  [0x0471] = 0x2189d, [0x0472] = 0x2189b, [0x0473] = 0x218a3, [0x0474] = 0x3747,
  [0x0475] = 0x2199f, [0x0476] = 0x2199e, [0x0477] = 0x219a3, [0x0478] = 0x219a0,
  [0x0479] = 0x219a1, [0x047a] = 0x21ac5, [0x047b] = 0x21ac6, [0x047c] = 0x21abe,
  [0x047d] = 0x21ac4, [0x047e] = 0x21abf, [0x047f] = 0x21b4c, [0x0480] = 0x21b4a,
  [0x0481] = 0x21bb1, [0x0482] = 0x21c15, [0x0483] = 0x21c17, [0x0484] = 0x21c10,
  [0x0485] = 0x21c18, [0x0486] = 0x21ce0, [0x0487] = 0x21cdf, [0x0488] = 0x21d27,
  [0x0489] = 0x21f43, [0x048a] = 0x21f48, [0x048b] = 0x21f1d, [0x048c] = 0x21f50,
  [0x048d] = 0x21f58, [0x048e] = 0x21f47, [0x048f] = 0x21f54, [0x0490] = 0x21f53,
  [0x0491] = 0x21f40, [0x0492] = 0x21f55, [0x0493] = 0x3817, [0x0494] = 0x21f5a,
  [0x0495] = 0x22470, [0x0496] = 0x2215b, [0x0497] = 0x2214d, [0x0498] = 0x22156,
  [0x0499] = 0x2214f, [0x049a] = 0x2f887, [0x049b] = 0x22159, [0x049c] = 0x2f888,
  [0x049d] = 0x22154, [0x049e] = 0x22153, [0x049f] = 0x22150, [0x04a0] = 0x222b3,
  [0x04a1] = 0x2236e, [0x04a2] = 0x38a3, [0x04a3] = 0x2236f, [0x04a4] = 0x22393,
  [0x04a5] = 0x22429, [0x04a6] = 0x22425, [0x04a7] = 0x22428, [0x04a8] = 0x2246f,
  [0x04a9] = 0x232b8, [0x04aa] = 0x2f897, [0x04ab] = 0x224b3, [0x04ac] = 0x22579,
  [0x04ad] = 0x2257e, [0x04ae] = 0x2257c, [0x04af] = 0x22583, [0x04b0] = 0x2f8ac,
  [0x04b1] = 0x22800, [0x04b2] = 0x22834, [0x04b3] = 0x22865, [0x04b4] = 0x22836,
  [0x04b5] = 0x228a7, [0x04b6] = 0x22859, [0x04b7] = 0x22858, [0x04b8] = 0x22831,
  [0x04b9] = 0x22862, [0x04ba] = 0x22838, [0x04bb] = 0x22873, [0x04bc] = 0x2283e,
  [0x04bd] = 0x22855, [0x04be] = 0x22854, [0x04bf] = 0x22895, [0x04c0] = 0x2f8ad,
  [0x04c1] = 0x228a5, [0x04c2] = 0x22894, [0x04c3] = 0x2289a, [0x04c4] = 0x2289b,
  [0x04c5] = 0x22899, [0x04c6] = 0x2289f, [0x04c7] = 0x22853, [0x04c8] = 0x22a12,
  [0x04c9] = 0x22a0a, [0x04ca] = 0x22a09, [0x04cb] = 0x22a0c, [0x04cc] = 0x22a0b,
  [0x04cd] = 0x22a0e, [0x04ce] = 0x22a0d, [0x04cf] = 0x22a08, [0x04d0] = 0x22d38,
  [0x04d1] = 0x22d3a, [0x04d2] = 0x22d37, [0x04d3] = 0x22d3d, [0x04d4] = 0x22d2d,
  [0x04d5] = 0x22d82, [0x04d6] = 0x22d95, [0x04d7] = 0x22d87, [0x04d8] = 0x22d9d,
  [0x04d9] = 0x22db5, [0x04da] = 0x22d81, [0x04db] = 0x22dc7, [0x04dc] = 0x22d9b,
  [0x04dd] = 0x22d98, [0x04de] = 0x22d94, [0x04df] = 0x22d86, [0x04e0] = 0x22d90,
  [0x04e1] = 0x22d34, [0x04e2] = 0x22d8e, [0x04e3] = 0x22d85, [0x04e4] = 0x22da6,
  [0x04e5] = 0x22d96, [0x04e6] = 0x22fea, [0x04e7] = 0x22fdf, [0x04e8] = 0x232bd,
  [0x04e9] = 0x22ff7, [0x04ea] = 0x22feb, [0x04eb] = 0x22fe0, [0x04ec] = 0x22fe8,
  [0x04ed] = 0x22fe1, [0x04ee] = 0x22fe9, [0x04ef] = 0x22fee, [0x04f0] = 0x22fe5,
  [0x04f1] = 0x22fec, [0x04f2] = 0x23063, [0x04f3] = 0x230c6, [0x04f4] = 0x230c2,
  [0x04f5] = 0x230c1, [0x04f6] = 0x230c0, [0x04f7] = 0x230c3, [0x04f8] = 0x23118,
  [0x04f9] = 0x232ae, [0x04fa] = 0x232a3, [0x04fb] = 0x232c5, [0x04fc] = 0x232a0,
  [0x04fd] = 0x232b4, [0x04fe] = 0x232a5, [0x04ff] = 0x3b1e, [0x0500] = 0x232a4,
  [0x0501] = 0x232a8, [0x0502] = 0x232a6, [0x0503] = 0x236a8, [0x0504] = 0x2368d,
  [0x0505] = 0x23682, [0x0506] = 0x2369e, [0x0507] = 0x23689, [0x0508] = 0x23683,
  [0x0509] = 0x23691, [0x050a] = 0x236a6, [0x050b] = 0x2369a, [0x050c] = 0x23690,
  [0x050d] = 0x23685, [0x050e] = 0x2367d, [0x050f] = 0x2f8ec, [0x0510] = 0x2369b,
  [0x0511] = 0x236a5, [0x0512] = 0x236f0, [0x0513] = 0x23696, [0x0514] = 0x23688,
  [0x0515] = 0x236d6, [0x0516] = 0x236d8, [0x0517] = 0x23686, [0x0518] = 0x236a4,
  [0x0519] = 0x23697, [0x051a] = 0x2391c, [0x051b] = 0x23919, [0x051c] = 0x2399e,
  [0x051d] = 0x2399f, [0x051e] = 0x2399d, [0x051f] = 0x2399c, [0x0520] = 0x23a5d,
  [0x0521] = 0x23a5e, [0x0522] = 0x23a73, [0x0523] = 0x23a63, [0x0524] = 0x23abe,
  [0x0525] = 0x23ac1, [0x0526] = 0x23abc, [0x0527] = 0x23abf, [0x0528] = 0x23abb,
  [0x0529] = 0x23abd, [0x052a] = 0x23b13, [0x052b] = 0x23b14, [0x052c] = 0x23bf8,
  [0x052d] = 0x23bfa, [0x052e] = 0x23c06, [0x052f] = 0x23bfe, [0x0530] = 0x23c51,
  [0x0531] = 0x2400b, [0x0532] = 0x23ff1, [0x0533] = 0x23fa9, [0x0534] = 0x23fd5,
  [0x0535] = 0x23fdc, [0x0536] = 0x23fcb, [0x0537] = 0x23ff2, [0x0538] = 0x2405f,
  [0x0539] = 0x23fdb, [0x053a] = 0x23fd6, [0x053b] = 0x23fd9, [0x053c] = 0x23fd1,
  [0x053d] = 0x23fcf, [0x053e] = 0x23fd8, [0x053f] = 0x23fe0, [0x0540] = 0x23fd4,
  [0x0541] = 0x23ff3, [0x0542] = 0x24005, [0x0543] = 0x23fce, [0x0544] = 0x23ff4,
  [0x0545] = 0x2f912, [0x0546] = 0x23fda, [0x0547] = 0x2400e, [0x0548] = 0x24006,
  [0x0549] = 0x23fd7, [0x054a] = 0x23fcd, [0x054b] = 0x24007, [0x054c] = 0x23fe1,
  [0x054d] = 0x24008, [0x054e] = 0x243be, [0x054f] = 0x243ef, [0x0550] = 0x243d9,
  [0x0551] = 0x243da, [0x0552] = 0x71d7, [0x0553] = 0x243a0, [0x0554] = 0x243df,
  [0x0555] = 0x2439e, [0x0556] = 0x243bf, [0x0557] = 0x243af, [0x0558] = 0x243a7,
  [0x0559] = 0x243aa, [0x055a] = 0x243f4, [0x055b] = 0x243ae, [0x055c] = 0x243b5,
  [0x055d] = 0x243b8, [0x055e] = 0x243db, [0x055f] = 0x243c0, [0x0560] = 0x243b7,
  [0x0561] = 0x243a2, [0x0562] = 0x243dd, [0x0563] = 0x243a1, [0x0564] = 0x243a5,
  [0x0565] = 0x243b4, [0x0566] = 0x243a6, [0x0567] = 0x243ab, [0x0568] = 0x243ac,
  [0x0569] = 0x2439f, [0x056a] = 0x243b1, [0x056b] = 0x243fc, [0x056c] = 0x2452e,
  [0x056d] = 0x24530, [0x056e] = 0x24599, [0x056f] = 0x245f8, [0x0570] = 0x245f6,
  [0x0571] = 0x246f4, [0x0572] = 0x246eb, [0x0573] = 0x246e1, [0x0574] = 0x246e4,
  [0x0575] = 0x24875, [0x0576] = 0x24869, [0x0577] = 0x3e9a, [0x0578] = 0x24899,
  [0x0579] = 0x2489e, [0x057a] = 0x24894, [0x057b] = 0x24893, [0x057c] = 0x24891,
  [0x057d] = 0x24a19, [0x057e] = 0x24a17, [0x057f] = 0x24a49, [0x0580] = 0x24a47,
  [0x0581] = 0x24a48, [0x0582] = 0x24a52, [0x0583] = 0x24a40, [0x0584] = 0x24a63,
  [0x0585] = 0x24a46, [0x0586] = 0x24a5f, [0x0587] = 0x24b0e, [0x0588] = 0x24b85,
  [0x0589] = 0x24b87, [0x058a] = 0x24bcb, [0x058b] = 0x24bcc, [0x058c] = 0x24bc9,
  [0x058d] = 0x24ccb, [0x058e] = 0x21ce7, [0x058f] = 0x24d20, [0x0590] = 0x24e5f,
  [0x0591] = 0x24e60, [0x0592] = 0x24e6a, [0x0593] = 0x24e76, [0x0594] = 0x24e73,
  [0x0595] = 0x24e71, [0x0596] = 0x24e66, [0x0597] = 0x3fc2, [0x0598] = 0x24e6b,
  [0x0599] = 0x24e79, [0x059a] = 0x24e7b, [0x059b] = 0x24f3c, [0x059c] = 0x24f9e,
  [0x059d] = 0x24f9c, [0x059e] = 0x2501c, [0x059f] = 0x2501e, [0x05a0] = 0x25021,
  [0x05a1] = 0x25096, [0x05a2] = 0x250a1, [0x05a3] = 0x250b6, [0x05a4] = 0x250a0,
  [0x05a5] = 0x25094, [0x05a6] = 0x25097, [0x05a7] = 0x25281, [0x05a8] = 0x25286,
  [0x05a9] = 0x252c0, [0x05aa] = 0x2528a, [0x05ab] = 0x2f94b, [0x05ac] = 0x25285,
  [0x05ad] = 0x25295, [0x05ae] = 0x25249, [0x05af] = 0x25282, [0x05b0] = 0x25284,
  [0x05b1] = 0x22878, [0x05b2] = 0x25290, [0x05b3] = 0x252b1, [0x05b4] = 0x253fe,
  [0x05b5] = 0x253fb, [0x05b6] = 0x25550, [0x05b7] = 0x25556, [0x05b8] = 0x2555a,
  [0x05b9] = 0x2554c, [0x05ba] = 0x40dc, [0x05bb] = 0x25557, [0x05bc] = 0x256e1,
  [0x05bd] = 0x256d9, [0x05be] = 0x256db, [0x05bf] = 0x256f4, [0x05c0] = 0x25745,
  [0x05c1] = 0x25747, [0x05c2] = 0x25874, [0x05c3] = 0x2586b, [0x05c4] = 0x25868,
  [0x05c5] = 0x25869, [0x05c6] = 0x25879, [0x05c7] = 0x2f95b, [0x05c8] = 0x25880,
  [0x05c9] = 0x25881, [0x05ca] = 0x25861, [0x05cb] = 0x25870, [0x05cc] = 0x2586f,
  [0x05cd] = 0x25873, [0x05ce] = 0x2585d, [0x05cf] = 0x259fa, [0x05d0] = 0x41b3,
  [0x05d1] = 0x259f3, [0x05d2] = 0x25a0a, [0x05d3] = 0x25a17, [0x05d4] = 0x2f95f,
  [0x05d5] = 0x25ab0, [0x05d6] = 0x25c3e, [0x05d7] = 0x25c46, [0x05d8] = 0x25c4f,
  [0x05d9] = 0x25c44, [0x05da] = 0x25c69, [0x05db] = 0x25c33, [0x05dc] = 0x25c2f,
  [0x05dd] = 0x25c4e, [0x05de] = 0x25c3a, [0x05df] = 0x25c2b, [0x05e0] = 0x25ca9,
  [0x05e1] = 0x25c50, [0x05e2] = 0x25c45, [0x05e3] = 0x25c37, [0x05e4] = 0x25c25,
  [0x05e5] = 0x25c2c, [0x05e6] = 0x25c3f, [0x05e7] = 0x25c34, [0x05e8] = 0x25c29,
  [0x05e9] = 0x25c1e, [0x05ea] = 0x25c3d, [0x05eb] = 0x25eed, [0x05ec] = 0x2f966,
  [0x05ed] = 0x25ef3, [0x05ee] = 0x25eea, [0x05ef] = 0x25eeb, [0x05f0] = 0x260cc,
  [0x05f1] = 0x260ce, [0x05f2] = 0x260d0, [0x05f3] = 0x260e3, [0x05f4] = 0x260e0,
  [0x05f5] = 0x260d1, [0x05f6] = 0x260dc, [0x05f7] = 0x260dd, [0x05f8] = 0x260f0,
  [0x05f9] = 0x260db, [0x05fa] = 0x260e5, [0x05fb] = 0x260f1, [0x05fc] = 0x260c9,
  [0x05fd] = 0x260e8, [0x05fe] = 0x260e7, [0x05ff] = 0x260c8, [0x0600] = 0x260de,
  [0x0601] = 0x260cd, [0x0602] = 0x260c5, [0x0603] = 0x260c6, [0x0604] = 0x260e4,
  [0x0605] = 0x260c3, [0x0606] = 0x262fb, [0x0607] = 0x262fc, [0x0608] = 0x262fe,
  [0x0609] = 0x26303, [0x060a] = 0x26300, [0x060b] = 0x262fd, [0x060c] = 0x26305,
  [0x060d] = 0x26313, [0x060e] = 0x263a7, [0x060f] = 0x263b4, [0x0610] = 0x26470,
  [0x0611] = 0x26474, [0x0612] = 0x26471, [0x0613] = 0x26475, [0x0614] = 0x2647d,
  [0x0615] = 0x26473, [0x0616] = 0x264c4, [0x0617] = 0x264c3, [0x0618] = 0x264de,
  [0x0619] = 0x264dd, [0x061a] = 0x26511, [0x061b] = 0x2650f, [0x061c] = 0x26510,
  [0x061d] = 0x265c3, [0x061e] = 0x265c2, [0x061f] = 0x265ca, [0x0620] = 0x265c1,
  [0x0621] = 0x26623, [0x0622] = 0x267b1, [0x0623] = 0x26824, [0x0624] = 0x26811,
  [0x0625] = 0x26825, [0x0626] = 0x2680a, [0x0627] = 0x2681f, [0x0628] = 0x26820,
  [0x0629] = 0x26814, [0x062a] = 0x26828, [0x062b] = 0x26803, [0x062c] = 0x26812,
  [0x062d] = 0x2680b, [0x062e] = 0x26817, [0x062f] = 0x22a11, [0x0630] = 0x26949,
  [0x0631] = 0x26989, [0x0632] = 0x26990, [0x0633] = 0x2698f, [0x0634] = 0x26996,
  [0x0635] = 0x26995, [0x0636] = 0x26993, [0x0637] = 0x269ed, [0x0638] = 0x269f0,
  [0x0639] = 0x243de, [0x063a] = 0x26a85, [0x063b] = 0x26a77, [0x063c] = 0x26a7a,
  [0x063d] = 0x26a84, [0x063e] = 0x26a79, [0x063f] = 0x26a7d, [0x0640] = 0x26a7b,
  [0x0641] = 0x26aed, [0x0642] = 0x26aec, [0x0643] = 0x26df6, [0x0644] = 0x26d9b,
  [0x0645] = 0x26e9d, [0x0646] = 0x26f11, [0x0647] = 0x26ec4, [0x0648] = 0x26ec8,
  [0x0649] = 0x26edf, [0x064a] = 0x26ebf, [0x064b] = 0x26eb3, [0x064c] = 0x26eba,
  [0x064d] = 0x26f10, [0x064e] = 0x26f06, [0x064f] = 0x26e88, [0x0650] = 0x26e9a,
  [0x0651] = 0x26ee0, [0x0652] = 0x26ecc, [0x0653] = 0x26eb5, [0x0654] = 0x26ee1,
  [0x0655] = 0x26ebc, [0x0656] = 0x26ec6, [0x0657] = 0x26f0b, [0x0658] = 0x26ea4,
  [0x0659] = 0x26e95, [0x065a] = 0x26ea3, [0x065b] = 0x26ee2, [0x065c] = 0x26ecd,
  [0x065d] = 0x26ee3, [0x065e] = 0x26eab, [0x065f] = 0x26ecb, [0x0660] = 0x26e8f,
  [0x0661] = 0x2f9ab, [0x0662] = 0x26ea9, [0x0663] = 0x26f24, [0x0664] = 0x26ee4,
  [0x0665] = 0x26f12, [0x0666] = 0x26ee5, [0x0667] = 0x26f67, [0x0668] = 0x26eaa,
  [0x0669] = 0x26ea0, [0x066a] = 0x26ee6, [0x066b] = 0x26ec1, [0x066c] = 0x26ee7,
  [0x066d] = 0x26f0d, [0x066e] = 0x26e86, [0x066f] = 0x26eb0, [0x0670] = 0x26e8b,
  [0x0671] = 0x26ee8, [0x0672] = 0x26ec9, [0x0673] = 0x26f19, [0x0674] = 0x26ec0,
  [0x0675] = 0x26f0c, [0x0676] = 0x26ee9, [0x0677] = 0x26eea, [0x0678] = 0x271ed,
  [0x0679] = 0x271e3, [0x067a] = 0x271e4, [0x067b] = 0x271e8, [0x067c] = 0x271d2,
  [0x067d] = 0x271e2, [0x067e] = 0x8666, [0x067f] = 0x273cf, [0x0680] = 0x273d1,
  [0x0681] = 0x273c3, [0x0682] = 0x273c9, [0x0683] = 0x273ea, [0x0684] = 0x273b4,
  [0x0685] = 0x273dc, [0x0686] = 0x273bd, [0x0687] = 0x273e6, [0x0688] = 0x273c8,
  [0x0689] = 0x273ec, [0x068a] = 0x273b2, [0x068b] = 0x273a9, [0x068c] = 0x273d3,
  [0x068d] = 0x273c0, [0x068e] = 0x273e9, [0x068f] = 0x273d5, [0x0690] = 0x45d9,
  [0x0691] = 0x273c5, [0x0692] = 0x45dd, [0x0693] = 0x273cb, [0x0694] = 0x273d0,
  [0x0695] = 0x273d2, [0x0696] = 0x273e4, [0x0697] = 0x273e8, [0x0698] = 0x273cd,
  [0x0699] = 0x273b6, [0x069a] = 0x273aa, [0x069b] = 0x273d4, [0x069c] = 0x28449,
  [0x069d] = 0x273c1, [0x069e] = 0x273dd, [0x069f] = 0x273ce, [0x06a0] = 0x275c9,
  [0x06a1] = 0x275ca, [0x06a2] = 0x276fd, [0x06a3] = 0x276d9, [0x06a4] = 0x276ff,
  [0x06a5] = 0x27704, [0x06a6] = 0x276fa, [0x06a7] = 0x27706, [0x06a8] = 0x27715,
  [0x06a9] = 0x27711, [0x06aa] = 0x27707, [0x06ab] = 0x2770b, [0x06ac] = 0x277fc,
  [0x06ad] = 0x277fd, [0x06ae] = 0x2786d, [0x06af] = 0x27865, [0x06b0] = 0x27914,
  [0x06b1] = 0x27913, [0x06b2] = 0x2790c, [0x06b3] = 0x2790b, [0x06b4] = 0x27918,
  [0x06b5] = 0x27922, [0x06b6] = 0x27a7d, [0x06b7] = 0x27a6f, [0x06b8] = 0x27a75,
  [0x06b9] = 0x27a7b, [0x06ba] = 0x27a80, [0x06bb] = 0x27a7f, [0x06bc] = 0x27a96,
  [0x06bd] = 0x27a6d, [0x06be] = 0x27a6b, [0x06bf] = 0x27a86, [0x06c0] = 0x27a73,
  [0x06c1] = 0x27a62, [0x06c2] = 0x27a77, [0x06c3] = 0x27a81, [0x06c4] = 0x27a69,
  [0x06c5] = 0x27a82, [0x06c6] = 0x27a97, [0x06c7] = 0x27a84, [0x06c8] = 0x46fc,
  [0x06c9] = 0x27a78, [0x06ca] = 0x2f9d0, [0x06cb] = 0x27a7a, [0x06cc] = 0x27a6a,
  [0x06cd] = 0x27a65, [0x06ce] = 0x27a7e, [0x06cf] = 0x27a94, [0x06d0] = 0x27bc2,
  [0x06d1] = 0x27bc1, [0x06d2] = 0x27bf9, [0x06d3] = 0x27c71, [0x06d4] = 0x27c6b,
  [0x06d5] = 0x27c6d, [0x06d6] = 0x27c6f, [0x06d7] = 0x27c6e, [0x06d8] = 0x27cf2,
  [0x06d9] = 0x27ce8, [0x06da] = 0x27cef, [0x06db] = 0x27ce9, [0x06dc] = 0x27cea,
  [0x06dd] = 0x27ced, [0x06de] = 0x27cf3, [0x06df] = 0x27ce6, [0x06e0] = 0x27dc9,
  [0x06e1] = 0x27db4, [0x06e2] = 0x27db3, [0x06e3] = 0x27db9, [0x06e4] = 0x27dca,
  [0x06e5] = 0x27db1, [0x06e6] = 0x27db6, [0x06e7] = 0x27dc7, [0x06e8] = 0x27dc4,
  [0x06e9] = 0x27db7, [0x06ea] = 0x27e6f, [0x06eb] = 0x27f33, [0x06ec] = 0x27f32,
  [0x06ed] = 0x27f1f, [0x06ee] = 0x2f9d9, [0x06ef] = 0x27f2b, [0x06f0] = 0x27f30,
  [0x06f1] = 0x27f36, [0x06f2] = 0x27f42, [0x06f3] = 0x27f41, [0x06f4] = 0x2f9d8,
  [0x06f5] = 0x280a5, [0x06f6] = 0x280ab, [0x06f7] = 0x280a9, [0x06f8] = 0x280b3,
  [0x06f9] = 0x280a7, [0x06fa] = 0x280a0, [0x06fb] = 0x280d2, [0x06fc] = 0x280a8,
  [0x06fd] = 0x280b6, [0x06fe] = 0x280ac, [0x06ff] = 0x280ae, [0x0700] = 0x280e6,
  [0x0701] = 0x28265, [0x0702] = 0x28347, [0x0703] = 0x28363, [0x0704] = 0x2834d,
  [0x0705] = 0x2835f, [0x0706] = 0x2834b, [0x0707] = 0x28360, [0x0708] = 0x28349,
  [0x0709] = 0x28353, [0x070a] = 0x2f9df, [0x070b] = 0x8f3a, [0x070c] = 0x28357,
  [0x070d] = 0x2834e, [0x070e] = 0x28352, [0x070f] = 0x28354, [0x0710] = 0x28420,
  [0x0711] = 0x28422, [0x0712] = 0x285c1, [0x0713] = 0x285d7, [0x0714] = 0x285d5,
  [0x0715] = 0x285c4, [0x0716] = 0x285d0, [0x0717] = 0x285bb, [0x0718] = 0x285e2,
  [0x0719] = 0x285cb, [0x071a] = 0x285b8, [0x071b] = 0x285bf, [0x071c] = 0x285d8,
  [0x071d] = 0x285c0, [0x071e] = 0x285ba, [0x071f] = 0x285b4, [0x0720] = 0x285bc,
  [0x0721] = 0x285d4, [0x0722] = 0x285ed, [0x0723] = 0x285c2, [0x0724] = 0x285d9,
  [0x0725] = 0x285cc, [0x0726] = 0x2872f, [0x0727] = 0x28723, [0x0728] = 0x28796,
  [0x0729] = 0x2878d, [0x072a] = 0x2879e, [0x072b] = 0x28799, [0x072c] = 0x28883,
  [0x072d] = 0x4907, [0x072e] = 0x2886a, [0x072f] = 0x28869, [0x0730] = 0x28875,
  [0x0731] = 0x2886c, [0x0732] = 0x28880, [0x0733] = 0x2886e, [0x0734] = 0x28881,
  [0x0735] = 0x28877, [0x0736] = 0x28876, [0x0737] = 0x28873, [0x0738] = 0x28870,
  [0x0739] = 0x28884, [0x073a] = 0x28919, [0x073b] = 0x289f2, [0x073c] = 0x933d,
  [0x073d] = 0x289ec, [0x073e] = 0x289eb, [0x073f] = 0x289f6, [0x0740] = 0x289ef,
  [0x0741] = 0x289f4, [0x0742] = 0x289ee, [0x0743] = 0x289f3, [0x0744] = 0x289ed,
  [0x0745] = 0x28c9b, [0x0746] = 0x28c98, [0x0747] = 0x28c94, [0x0748] = 0x2147a,
  [0x0749] = 0x28c9a, [0x074a] = 0x2f9f0, [0x074b] = 0x28d4c, [0x074c] = 0x49a8,
  [0x074d] = 0x28d57, [0x074e] = 0x28d51, [0x074f] = 0x28d62, [0x0750] = 0x28d52,
  [0x0751] = 0x28d5c, [0x0752] = 0x28d53, [0x0753] = 0x2f9ef, [0x0754] = 0x28e85,
  [0x0755] = 0x28f2d, [0x0756] = 0x28f3e, [0x0757] = 0x28f30, [0x0758] = 0x28f7d,
  [0x0759] = 0x28fe6, [0x075a] = 0x28ff6, [0x075b] = 0x28fe4, [0x075c] = 0x28fe9,
  [0x075d] = 0x28fec, [0x075e] = 0x28ff2, [0x075f] = 0x28fe8, [0x0760] = 0x28fe3,
  [0x0761] = 0x28feb, [0x0762] = 0x28ff0, [0x0763] = 0x28fea, [0x0764] = 0x28fe7,
  [0x0765] = 0x28ffd, [0x0766] = 0x28fe5, [0x0767] = 0x28fee, [0x0768] = 0x28ff5,
  [0x0769] = 0x28ff9, [0x076a] = 0x28ff1, [0x076b] = 0x28ff4, [0x076c] = 0x2f9f4,
  [0x076d] = 0x29011, [0x076e] = 0x290de, [0x076f] = 0x290d8, [0x0770] = 0x290dd,
  [0x0771] = 0x290db, [0x0772] = 0x290d3, [0x0773] = 0x290e1, [0x0774] = 0x291de,
  [0x0775] = 0x291fc, [0x0776] = 0x29228, [0x0777] = 0x292b7, [0x0778] = 0x292b8,
  [0x0779] = 0x292c7, [0x077a] = 0x292b2, [0x077b] = 0x292a9, [0x077c] = 0x292b9,
  [0x077d] = 0x292bc, [0x077e] = 0x292b3, [0x077f] = 0x292d1, [0x0780] = 0x293b0,
  [0x0781] = 0x293b1, [0x0782] = 0x293b6, [0x0783] = 0x29404, [0x0784] = 0x2942c,
  [0x0785] = 0x294c1, [0x0786] = 0x2949e, [0x0787] = 0x294ba, [0x0788] = 0x4aca,
  [0x0789] = 0x294d1, [0x078a] = 0x294d3, [0x078b] = 0x294d4, [0x078c] = 0x294d5,
  [0x078d] = 0x294c5, [0x078e] = 0x294b6, [0x078f] = 0x294b2, [0x0790] = 0x294b5,
  [0x0791] = 0x294c3, [0x0792] = 0x294b9, [0x0793] = 0x294c6, [0x0794] = 0x295d6,
  [0x0795] = 0x295e0, [0x0796] = 0x295db, [0x0797] = 0x295d7, [0x0798] = 0x295dd,
  [0x0799] = 0x29715, [0x079a] = 0x2970c, [0x079b] = 0x2970f, [0x079c] = 0x2970e,
  [0x079d] = 0x296fe, [0x079e] = 0x29704, [0x079f] = 0x2970b, [0x07a0] = 0x29702,
  [0x07a1] = 0x296ff, [0x07a2] = 0x29708, [0x07a3] = 0x29710, [0x07a4] = 0x29717,
  [0x07a5] = 0x29713, [0x07a6] = 0x29706, [0x07a7] = 0x29709, [0x07a8] = 0x29824,
  [0x07a9] = 0x29826, [0x07aa] = 0x29825, [0x07ab] = 0x29848, [0x07ac] = 0x298b5,
  [0x07ad] = 0x298d2, [0x07ae] = 0x298d5, [0x07af] = 0x298c4, [0x07b0] = 0x298af,
  [0x07b1] = 0x298ad, [0x07b2] = 0x298c1, [0x07b3] = 0x298c0, [0x07b4] = 0x298cc,
  [0x07b5] = 0x298cd, [0x07b6] = 0x298c3, [0x07b7] = 0x298c8, [0x07b8] = 0x298c5,
  [0x07b9] = 0x298ba, [0x07ba] = 0x298d0, [0x07bb] = 0x298c2, [0x07bc] = 0x298ce,
  [0x07bd] = 0x4b8d, [0x07be] = 0x29a43, [0x07bf] = 0x29a42, [0x07c0] = 0x29a40,
  [0x07c1] = 0x29a31, [0x07c2] = 0x29acf, [0x07c3] = 0x29ace, [0x07c4] = 0x29b30,
  [0x07c5] = 0x29b34, [0x07c6] = 0x29b32, [0x07c7] = 0x29b43, [0x07c8] = 0x29b3c,
  [0x07c9] = 0x29c11, [0x07ca] = 0x29c35, [0x07cb] = 0x29c34, [0x07cc] = 0x29c37,
  [0x07cd] = 0x29c31, [0x07ce] = 0x29cbf, [0x07cf] = 0x29cbc, [0x07d0] = 0x29cc2,
  [0x07d1] = 0x29cc9, [0x07d2] = 0x29d83, [0x07d3] = 0x29d8b, [0x07d4] = 0x29da0,
  [0x07d5] = 0x29d8d, [0x07d6] = 0x29d8c, [0x07d7] = 0x29d9a, [0x07d8] = 0x29d8a,
  [0x07d9] = 0x29d91, [0x07da] = 0x29ff1, [0x07db] = 0x2a00f, [0x07dc] = 0x2a001,
  [0x07dd] = 0x2a007, [0x07de] = 0x2a00c, [0x07df] = 0x4cc0, [0x07e0] = 0x29fdc,
  [0x07e1] = 0x29fee, [0x07e2] = 0x29ff7, [0x07e3] = 0x29ff2, [0x07e4] = 0x29ff8,
  [0x07e5] = 0x29feb, [0x07e6] = 0x4cca, [0x07e7] = 0x29fe6, [0x07e8] = 0x29fed,
  [0x07e9] = 0x29fe9, [0x07ea] = 0x2a008, [0x07eb] = 0x2a000, [0x07ec] = 0x29fe5,
  [0x07ed] = 0x29ffc, [0x07ee] = 0x2a261, [0x07ef] = 0x2a25e, [0x07f0] = 0x2a25f,
  [0x07f1] = 0x4d25, [0x07f2] = 0x2a29b, [0x07f3] = 0x2a316, [0x07f4] = 0x2a315,
  [0x07f5] = 0x2a320, [0x07f6] = 0x2a314, [0x07f7] = 0x2a31a, [0x07f8] = 0x2a317,
  [0x07f9] = 0x2a39c, [0x07fa] = 0x4d4a, [0x07fb] = 0x2a3f1, [0x07fc] = 0x4d53,
  [0x07fd] = 0x2a3f0, [0x07fe] = 0x2a428, [0x07ff] = 0x2a424, [0x0800] = 0x2a42a,
  [0x0801] = 0x2a42b, [0x0802] = 0x2a4d5, [0x0803] = 0x2a504, [0x0804] = 0x2a505,
  [0x0805] = 0x2a591, [0x0806] = 0x2a592, [0x0807] = 0x2a6a6, [0x0808] = 0x20109,
  [0x0809] = 0x20422, [0x080a] = 0x2041a, [0x080b] = 0x2040e, [0x080c] = 0x2041b,
  [0x080d] = 0x20408, [0x080e] = 0x250b3, [0x080f] = 0x2f81c, [0x0810] = 0x2069d,
  [0x0811] = 0x20812, [0x0812] = 0x2081e, [0x0813] = 0x208bb, [0x0814] = 0x208bf,
  [0x0815] = 0x208bc, [0x0816] = 0x20908, [0x0817] = 0x20909, [0x0818] = 0x20963,
  [0x0819] = 0x209c8, [0x081a] = 0x23728, [0x081b] = 0x20b8e, [0x081c] = 0x20b8d,
  [0x081d] = 0x21043, [0x081e] = 0x2104d, [0x081f] = 0x2106d, [0x0820] = 0x21042,
  [0x0821] = 0x21052, [0x0822] = 0x21051, [0x0823] = 0x21069, [0x0824] = 0x21050,
  [0x0825] = 0x21056, [0x0826] = 0x2106c, [0x0827] = 0x21044, [0x0828] = 0x21045,
  [0x0829] = 0x2106b, [0x082a] = 0x3605, [0x082b] = 0x21068, [0x082c] = 0x21057,
  [0x082d] = 0x2122e, [0x082e] = 0x21231, [0x082f] = 0x2122d, [0x0830] = 0x2148b,
  [0x0831] = 0x214b2, [0x0832] = 0x2148d, [0x0833] = 0x214a3, [0x0834] = 0x214a4,
  [0x0835] = 0x21490, [0x0836] = 0x21489, [0x0837] = 0x2148e, [0x0838] = 0x21492,
  [0x0839] = 0x2157b, [0x083a] = 0x21690, [0x083b] = 0x2168f, [0x083c] = 0x218e9,
  [0x083d] = 0x218cd, [0x083e] = 0x218c5, [0x083f] = 0x218d0, [0x0840] = 0x218c9,
  [0x0841] = 0x218d1, [0x0842] = 0x218c7, [0x0843] = 0x218d2, [0x0844] = 0x2185f,
  [0x0845] = 0x218e1, [0x0846] = 0x218cc, [0x0847] = 0x218c6, [0x0848] = 0x218e4,
  [0x0849] = 0x219a9, [0x084a] = 0x21ad3, [0x084b] = 0x21ada, [0x084c] = 0x21ad2,
  [0x084d] = 0x21adb, [0x084e] = 0x21b4d, [0x084f] = 0x21b4e, [0x0850] = 0x21c1c,
  [0x0851] = 0x21ce8, [0x0852] = 0x21f6c, [0x0853] = 0x21f5f, [0x0854] = 0x21f66,
  [0x0855] = 0x21f64, [0x0856] = 0x21f78, [0x0857] = 0x21f65, [0x0858] = 0x21f6d,
  [0x0859] = 0x21f61, [0x085a] = 0x21f7a, [0x085b] = 0x22007, [0x085c] = 0x22166,
  [0x085d] = 0x22168, [0x085e] = 0x22162, [0x085f] = 0x222d3, [0x0860] = 0x222d4,
  [0x0861] = 0x222d1, [0x0862] = 0x222dc, [0x0863] = 0x22373, [0x0864] = 0x22372,
  [0x0865] = 0x2242d, [0x0866] = 0x2242c, [0x0867] = 0x2242f, [0x0868] = 0x22472,
  [0x0869] = 0x224b4, [0x086a] = 0x22589, [0x086b] = 0x228cd, [0x086c] = 0x228a6,
  [0x086d] = 0x228a2, [0x086e] = 0x228a0, [0x086f] = 0x22846, [0x0870] = 0x228a1,
  [0x0871] = 0x26857, [0x0872] = 0x228a3, [0x0873] = 0x228b1, [0x0874] = 0x228db,
  [0x0875] = 0x228f6, [0x0876] = 0x228dc, [0x0877] = 0x228d6, [0x0878] = 0x22a13,
  [0x0879] = 0x22d9c, [0x087a] = 0x22d97, [0x087b] = 0x22dd8, [0x087c] = 0x22de4,
  [0x087d] = 0x22dd4, [0x087e] = 0x22de5, [0x087f] = 0x22ddb, [0x0880] = 0x22dd0,
  [0x0881] = 0x22dda, [0x0882] = 0x22dcc, [0x0883] = 0x22ddc, [0x0884] = 0x22ded,
  [0x0885] = 0x22dd3, [0x0886] = 0x22dd1, [0x0887] = 0x22dce, [0x0888] = 0x22dd9,
  [0x0889] = 0x22ddd, [0x088a] = 0x2f8c6, [0x088b] = 0x22ee6, [0x088c] = 0x22ee5,
  [0x088d] = 0x22ee9, [0x088e] = 0x22fff, [0x088f] = 0x22ffe, [0x0890] = 0x2300e,
  [0x0891] = 0x2308c, [0x0892] = 0x230ca, [0x0893] = 0x230cb, [0x0894] = 0x2311a,
  [0x0895] = 0x232d7, [0x0896] = 0x232cf, [0x0897] = 0x232d0, [0x0898] = 0x232c9,
  [0x0899] = 0x232cc, [0x089a] = 0x232d3, [0x089b] = 0x233a4, [0x089c] = 0x233a3,
  [0x089d] = 0x23681, [0x089e] = 0x23726, [0x089f] = 0x236f3, [0x08a0] = 0x23727,
  [0x08a1] = 0x236f2, [0x08a2] = 0x236ff, [0x08a3] = 0x236f5, [0x08a4] = 0x236fc,
  [0x08a5] = 0x2370e, [0x08a6] = 0x3bf0, [0x08a7] = 0x236f6, [0x08a8] = 0x23700,
  [0x08a9] = 0x23723, [0x08aa] = 0x23705, [0x08ab] = 0x23725, [0x08ac] = 0x23708,
  [0x08ad] = 0x23750, [0x08ae] = 0x23925, [0x08af] = 0x23920, [0x08b0] = 0x23923,
  [0x08b1] = 0x23921, [0x08b2] = 0x239a1, [0x08b3] = 0x23a69, [0x08b4] = 0x23a6c,
  [0x08b5] = 0x23a68, [0x08b6] = 0x23a6b, [0x08b7] = 0x23acc, [0x08b8] = 0x23acd,
  [0x08b9] = 0x23c10, [0x08ba] = 0x23c0d, [0x08bb] = 0x23c0a, [0x08bc] = 0x23c16,
  [0x08bd] = 0x23c14, [0x08be] = 0x23c52, [0x08bf] = 0x24028, [0x08c0] = 0x2402e,
  [0x08c1] = 0x2402c, [0x08c2] = 0x2404d, [0x08c3] = 0x24049, [0x08c4] = 0x24031,
  [0x08c5] = 0x24030, [0x08c6] = 0x24033, [0x08c7] = 0x2402d, [0x08c8] = 0x24036,
  [0x08c9] = 0x2403e, [0x08ca] = 0x2402f, [0x08cb] = 0x24027, [0x08cc] = 0x24034,
  [0x08cd] = 0x2404c, [0x08ce] = 0x243fe, [0x08cf] = 0x24412, [0x08d0] = 0x2441f,
  [0x08d1] = 0x24417, [0x08d2] = 0x243f5, [0x08d3] = 0x24415, [0x08d4] = 0x243f7,
  [0x08d5] = 0x3e02, [0x08d6] = 0x21f7d, [0x08d7] = 0x243fa, [0x08d8] = 0x243f9,
  [0x08d9] = 0x2444b, [0x08da] = 0x2459c, [0x08db] = 0x2459d, [0x08dc] = 0x3e23,
  [0x08dd] = 0x246f0, [0x08de] = 0x246f6, [0x08df] = 0x246ef, [0x08e0] = 0x2489d,
  [0x08e1] = 0x2489a, [0x08e2] = 0x248a7, [0x08e3] = 0x248af, [0x08e4] = 0x248aa,
  [0x08e5] = 0x24a64, [0x08e6] = 0x24a86, [0x08e7] = 0x24a75, [0x08e8] = 0x24a70,
  [0x08e9] = 0x24a84, [0x08ea] = 0x24a6b, [0x08eb] = 0x24a85, [0x08ec] = 0x24a6c,
  [0x08ed] = 0x24b17, [0x08ee] = 0x24b94, [0x08ef] = 0x24b93, [0x08f0] = 0x24cd2,
  [0x08f1] = 0x24cd7, [0x08f2] = 0x24cd4, [0x08f3] = 0x24ea6, [0x08f4] = 0x24ea7,
  [0x08f5] = 0x24e9c, [0x08f6] = 0x24e8b, [0x08f7] = 0x24e8d, [0x08f8] = 0x24e98,
  [0x08f9] = 0x24eb9, [0x08fa] = 0x24e9b, [0x08fb] = 0x24e9d, [0x08fc] = 0x24e99,
  [0x08fd] = 0x24ea8, [0x08fe] = 0x24e91, [0x08ff] = 0x24e87, [0x0900] = 0x24e9a,
  [0x0901] = 0x24fa6, [0x0902] = 0x2501f, [0x0903] = 0x250a7, [0x0904] = 0x250b1,
  [0x0905] = 0x250b2, [0x0906] = 0x250b7, [0x0907] = 0x252d0, [0x0908] = 0x252b3,
  [0x0909] = 0x252b5, [0x090a] = 0x252c4, [0x090b] = 0x252c3, [0x090c] = 0x252bc,
  [0x090d] = 0x252b2, [0x090e] = 0x252ba, [0x090f] = 0x252bb, [0x0910] = 0x252c2,
  [0x0911] = 0x252cd, [0x0912] = 0x252be, [0x0913] = 0x252b7, [0x0914] = 0x2538f,
  [0x0915] = 0x25403, [0x0916] = 0x25404, [0x0917] = 0x25571, [0x0918] = 0x25577,
  [0x0919] = 0x25574, [0x091a] = 0x2558b, [0x091b] = 0x2557a, [0x091c] = 0x2f951,
  [0x091d] = 0x2558c, [0x091e] = 0x25573, [0x091f] = 0x256ff, [0x0920] = 0x256fb,
  [0x0921] = 0x256fd, [0x0922] = 0x256f0, [0x0923] = 0x256f3, [0x0924] = 0x256fc,
  [0x0925] = 0x256f2, [0x0926] = 0x25892, [0x0927] = 0x2589e, [0x0928] = 0x258ae,
  [0x0929] = 0x25896, [0x092a] = 0x25a14, [0x092b] = 0x25a12, [0x092c] = 0x25a13,
  [0x092d] = 0x25a16, [0x092e] = 0x25a0f, [0x092f] = 0x25ab6, [0x0930] = 0x25abd,
  [0x0931] = 0x25c8c, [0x0932] = 0x25cae, [0x0933] = 0x25cac, [0x0934] = 0x25cab,
  [0x0935] = 0x25c99, [0x0936] = 0x25c92, [0x0937] = 0x25cbb, [0x0938] = 0x25c9e,
  [0x0939] = 0x25c7e, [0x093a] = 0x25caf, [0x093b] = 0x25cbc, [0x093c] = 0x25c98,
  [0x093d] = 0x25f01, [0x093e] = 0x25f09, [0x093f] = 0x25f06, [0x0940] = 0x2f967,
  [0x0941] = 0x25f07, [0x0942] = 0x25f08, [0x0943] = 0x260cf, [0x0944] = 0x2610e,
  [0x0945] = 0x26132, [0x0946] = 0x4315, [0x0947] = 0x26112, [0x0948] = 0x26116,
  [0x0949] = 0x26117, [0x094a] = 0x2611b, [0x094b] = 0x26115, [0x094c] = 0x26131,
  [0x094d] = 0x26118, [0x094e] = 0x2611a, [0x094f] = 0x26110, [0x0950] = 0x2610a,
  [0x0951] = 0x26109, [0x0952] = 0x2624f, [0x0953] = 0x2f973, [0x0954] = 0x2630f,
  [0x0955] = 0x26310, [0x0956] = 0x2f975, [0x0957] = 0x26328, [0x0958] = 0x26311,
  [0x0959] = 0x26316, [0x095a] = 0x26317, [0x095b] = 0x26302, [0x095c] = 0x263bb,
  [0x095d] = 0x263ba, [0x095e] = 0x263c3, [0x095f] = 0x263bc, [0x0960] = 0x2648a,
  [0x0961] = 0x26484, [0x0962] = 0x26486, [0x0963] = 0x264e0, [0x0964] = 0x26517,
  [0x0965] = 0x26518, [0x0966] = 0x2651e, [0x0967] = 0x26515, [0x0968] = 0x265d3,
  [0x0969] = 0x265da, [0x096a] = 0x265d9, [0x096b] = 0x267e6, [0x096c] = 0x267f4,
  [0x096d] = 0x267e1, [0x096e] = 0x26869, [0x096f] = 0x26840, [0x0970] = 0x26858,
  [0x0971] = 0x2686c, [0x0972] = 0x2684d, [0x0973] = 0x26921, [0x0974] = 0x26999,
  [0x0975] = 0x269f1, [0x0976] = 0x23f68, [0x0977] = 0x26a86, [0x0978] = 0x26f5b,
  [0x0979] = 0x26f5c, [0x097a] = 0x26f77, [0x097b] = 0x26f2c, [0x097c] = 0x2f9ad,
  [0x097d] = 0x26f58, [0x097e] = 0x26f64, [0x097f] = 0x26f61, [0x0980] = 0x26f48,
  [0x0981] = 0x26f97, [0x0982] = 0x26f59, [0x0983] = 0x26f29, [0x0984] = 0x26f62,
  [0x0985] = 0x26f2e, [0x0986] = 0x26f68, [0x0987] = 0x26f90, [0x0988] = 0x26f3a,
  [0x0989] = 0x26f3d, [0x098a] = 0x26f5e, [0x098b] = 0x26f46, [0x098c] = 0x26f69,
  [0x098d] = 0x26f65, [0x098e] = 0x26f3e, [0x098f] = 0x26f49, [0x0990] = 0x26f56,
  [0x0991] = 0x26fe1, [0x0992] = 0x26f78, [0x0993] = 0x26f79, [0x0994] = 0x26f66,
  [0x0995] = 0x26f4a, [0x0996] = 0x26f35, [0x0997] = 0x26f7a, [0x0998] = 0x26f92,
  [0x0999] = 0x26f60, [0x099a] = 0x26f36, [0x099b] = 0x26f51, [0x099c] = 0x26f42,
  [0x099d] = 0x455d, [0x099e] = 0x26f3f, [0x099f] = 0x26f7b, [0x09a0] = 0x26f5d,
  [0x09a1] = 0x26f94, [0x09a2] = 0x26f6a, [0x09a3] = 0x2f9b5, [0x09a4] = 0x271f2,
  [0x09a5] = 0x273bb, [0x09a6] = 0x2741b, [0x09a7] = 0x2741a, [0x09a8] = 0x27433,
  [0x09a9] = 0x27417, [0x09aa] = 0x2740a, [0x09ab] = 0x27415, [0x09ac] = 0x27412,
  [0x09ad] = 0x27401, [0x09ae] = 0x2742d, [0x09af] = 0x273fd, [0x09b0] = 0x27423,
  [0x09b1] = 0x27405, [0x09b2] = 0x27411, [0x09b3] = 0x27400, [0x09b4] = 0x2741c,
  [0x09b5] = 0x27435, [0x09b6] = 0x2742e, [0x09b7] = 0x27436, [0x09b8] = 0x20abf,
  [0x09b9] = 0x2742f, [0x09ba] = 0x2740c, [0x09bb] = 0x27409, [0x09bc] = 0x27431,
  [0x09bd] = 0x273fc, [0x09be] = 0x2740f, [0x09bf] = 0x27418, [0x09c0] = 0x27402,
  [0x09c1] = 0x27600, [0x09c2] = 0x2771f, [0x09c3] = 0x27737, [0x09c4] = 0x465a,
  [0x09c5] = 0x27738, [0x09c6] = 0x2772b, [0x09c7] = 0x2772e, [0x09c8] = 0x27721,
  [0x09c9] = 0x27730, [0x09ca] = 0x27729, [0x09cb] = 0x27731, [0x09cc] = 0x2f9c9,
  [0x09cd] = 0x27701, [0x09ce] = 0x2772c, [0x09cf] = 0x27722, [0x09d0] = 0x277ff,
  [0x09d1] = 0x27877, [0x09d2] = 0x27867, [0x09d3] = 0x2787f, [0x09d4] = 0x2787d,
  [0x09d5] = 0x2787b, [0x09d6] = 0x2787e, [0x09d7] = 0x2791e, [0x09d8] = 0x2791c,
  [0x09d9] = 0x27921, [0x09da] = 0x27926, [0x09db] = 0x27927, [0x09dc] = 0x27929,
  [0x09dd] = 0x2792c, [0x09de] = 0x2791d, [0x09df] = 0x2792b, [0x09e0] = 0x27abf,
  [0x09e1] = 0x27aa4, [0x09e2] = 0x27aaa, [0x09e3] = 0x27aae, [0x09e4] = 0x27a9f,
  [0x09e5] = 0x27ad0, [0x09e6] = 0x27ab1, [0x09e7] = 0x27aad, [0x09e8] = 0x27a9b,
  [0x09e9] = 0x27ab2, [0x09ea] = 0x27aa9, [0x09eb] = 0x27ab3, [0x09ec] = 0x27ab4,
  [0x09ed] = 0x27aba, [0x09ee] = 0x27aa5, [0x09ef] = 0x27ab7, [0x09f0] = 0x27aac,
  [0x09f1] = 0x27acb, [0x09f2] = 0x27acf, [0x09f3] = 0x27bc6, [0x09f4] = 0x27c01,
  [0x09f5] = 0x27bff, [0x09f6] = 0x27bfd, [0x09f7] = 0x27c77, [0x09f8] = 0x27c78,
  [0x09f9] = 0x27c76, [0x09fa] = 0x27cf7, [0x09fb] = 0x27dcc, [0x09fc] = 0x4785,
  [0x09fd] = 0x27dd4, [0x09fe] = 0x27dd7, [0x09ff] = 0x27dd5, [0x0a00] = 0x27dd6,
  [0x0a01] = 0x27dd3, [0x0a02] = 0x27f5d, [0x0a03] = 0x27f55, [0x0a04] = 0x280e0,
  [0x0a05] = 0x28148, [0x0a06] = 0x280ee, [0x0a07] = 0x280db, [0x0a08] = 0x280e7,
  [0x0a09] = 0x280d6, [0x0a0a] = 0x280e5, [0x0a0b] = 0x280e1, [0x0a0c] = 0x280dd,
  [0x0a0d] = 0x280e2, [0x0a0e] = 0x28270, [0x0a0f] = 0x28266, [0x0a10] = 0x2826f,
  [0x0a11] = 0x2826e, [0x0a12] = 0x28381, [0x0a13] = 0x28369, [0x0a14] = 0x2836e,
  [0x0a15] = 0x2836d, [0x0a16] = 0x2836c, [0x0a17] = 0x28384, [0x0a18] = 0x28385,
  [0x0a19] = 0x28371, [0x0a1a] = 0x28373, [0x0a1b] = 0x2836a, [0x0a1c] = 0x2836f,
  [0x0a1d] = 0x2837b, [0x0a1e] = 0x2856a, [0x0a1f] = 0x2857c, [0x0a20] = 0x2857d,
  [0x0a21] = 0x28581, [0x0a22] = 0x285fa, [0x0a23] = 0x28605, [0x0a24] = 0x285eb,
  [0x0a25] = 0x285fb, [0x0a26] = 0x285e9, [0x0a27] = 0x285ef, [0x0a28] = 0x285fc,
  [0x0a29] = 0x285e7, [0x0a2a] = 0x2f9e1, [0x0a2b] = 0x285ee, [0x0a2c] = 0x285fd,
  [0x0a2d] = 0x28732, [0x0a2e] = 0x287a7, [0x0a2f] = 0x287b5, [0x0a30] = 0x287b1,
  [0x0a31] = 0x287b9, [0x0a32] = 0x287a8, [0x0a33] = 0x287b3, [0x0a34] = 0x2f9e6,
  [0x0a35] = 0x2888a, [0x0a36] = 0x28891, [0x0a37] = 0x2888d, [0x0a38] = 0x28899,
  [0x0a39] = 0x490b, [0x0a3a] = 0x2889a, [0x0a3b] = 0x2889b, [0x0a3c] = 0x28892,
  [0x0a3d] = 0x2888f, [0x0a3e] = 0x288ab, [0x0a3f] = 0x24cdb, [0x0a40] = 0x4939,
  [0x0a41] = 0x28a75, [0x0a42] = 0x28a31, [0x0a43] = 0x28a38, [0x0a44] = 0x4937,
  [0x0a45] = 0x28a35, [0x0a46] = 0x28a69, [0x0a47] = 0x28a3b, [0x0a48] = 0x28a3d,
  [0x0a49] = 0x28a6c, [0x0a4a] = 0x28a79, [0x0a4b] = 0x28a3c, [0x0a4c] = 0x28a3e,
  [0x0a4d] = 0x28c97, [0x0a4e] = 0x28ca5, [0x0a4f] = 0x28ca2, [0x0a50] = 0x28c9d,
  [0x0a51] = 0x28ca1, [0x0a52] = 0x28d68, [0x0a53] = 0x28d6f, [0x0a54] = 0x28d6d,
  [0x0a55] = 0x28d72, [0x0a56] = 0x28d75, [0x0a57] = 0x28d77, [0x0a58] = 0x28d79,
  [0x0a59] = 0x28eb5, [0x0a5a] = 0x28eea, [0x0a5b] = 0x28eab, [0x0a5c] = 0x28f43,
  [0x0a5d] = 0x28f41, [0x0a5e] = 0x28f42, [0x0a5f] = 0x29009, [0x0a60] = 0x29008,
  [0x0a61] = 0x29006, [0x0a62] = 0x29001, [0x0a63] = 0x29003, [0x0a64] = 0x29000,
  [0x0a65] = 0x29004, [0x0a66] = 0x2900a, [0x0a67] = 0x2900e, [0x0a68] = 0x2900d,
  [0x0a69] = 0x29007, [0x0a6a] = 0x2900f, [0x0a6b] = 0x29014, [0x0a6c] = 0x29002,
  [0x0a6d] = 0x29015, [0x0a6e] = 0x2900c, [0x0a6f] = 0x29010, [0x0a70] = 0x29005,
  [0x0a71] = 0x290fd, [0x0a72] = 0x290ff, [0x0a73] = 0x29104, [0x0a74] = 0x29100,
  [0x0a75] = 0x29109, [0x0a76] = 0x2922b, [0x0a77] = 0x2f9f7, [0x0a78] = 0x29231,
  [0x0a79] = 0x292db, [0x0a7a] = 0x292c5, [0x0a7b] = 0x292d3, [0x0a7c] = 0x292ce,
  [0x0a7d] = 0x4a6b, [0x0a7e] = 0x292c9, [0x0a7f] = 0x292bf, [0x0a80] = 0x292cb,
  [0x0a81] = 0x292c0, [0x0a82] = 0x292d0, [0x0a83] = 0x292d4, [0x0a84] = 0x293c1,
  [0x0a85] = 0x293b9, [0x0a86] = 0x293bb, [0x0a87] = 0x293c3, [0x0a88] = 0x293c9,
  [0x0a89] = 0x29407, [0x0a8a] = 0x2942d, [0x0a8b] = 0x294f8, [0x0a8c] = 0x294e1,
  [0x0a8d] = 0x294fa, [0x0a8e] = 0x294ef, [0x0a8f] = 0x294fd, [0x0a90] = 0x4acd,
  [0x0a91] = 0x294eb, [0x0a92] = 0x294f1, [0x0a93] = 0x294ed, [0x0a94] = 0x294fe,
  [0x0a95] = 0x295f8, [0x0a96] = 0x29603, [0x0a97] = 0x295ee, [0x0a98] = 0x295e8,
  [0x0a99] = 0x29601, [0x0a9a] = 0x296ec, [0x0a9b] = 0x29722, [0x0a9c] = 0x29714,
  [0x0a9d] = 0x29734, [0x0a9e] = 0x2972f, [0x0a9f] = 0x29739, [0x0aa0] = 0x29741,
  [0x0aa1] = 0x2973c, [0x0aa2] = 0x29749, [0x0aa3] = 0x29758, [0x0aa4] = 0x2973a,
  [0x0aa5] = 0x29742, [0x0aa6] = 0x2973f, [0x0aa7] = 0x29822, [0x0aa8] = 0x29823,
  [0x0aa9] = 0x2984a, [0x0aaa] = 0x298dc, [0x0aab] = 0x298d9, [0x0aac] = 0x298db,
  [0x0aad] = 0x298e2, [0x0aae] = 0x2fa07, [0x0aaf] = 0x298df, [0x0ab0] = 0x298e0,
  [0x0ab1] = 0x298d7, [0x0ab2] = 0x29a4f, [0x0ab3] = 0x29a46, [0x0ab4] = 0x29a53,
  [0x0ab5] = 0x29a55, [0x0ab6] = 0x29a4e, [0x0ab7] = 0x29a4a, [0x0ab8] = 0x29a4c,
  [0x0ab9] = 0x29a63, [0x0aba] = 0x29b51, [0x0abb] = 0x29b53, [0x0abc] = 0x29b58,
  [0x0abd] = 0x29b4d, [0x0abe] = 0x29b5a, [0x0abf] = 0x29b49, [0x0ac0] = 0x29b5d,
  [0x0ac1] = 0x29c12, [0x0ac2] = 0x29c3c, [0x0ac3] = 0x29cd1, [0x0ac4] = 0x29cdf,
  [0x0ac5] = 0x29cd6, [0x0ac6] = 0x29cd8, [0x0ac7] = 0x29ce0, [0x0ac8] = 0x29cd9,
  [0x0ac9] = 0x29db1, [0x0aca] = 0x29dac, [0x0acb] = 0x29daa, [0x0acc] = 0x29dee,
  [0x0acd] = 0x29dbd, [0x0ace] = 0x4c4d, [0x0acf] = 0x29dc3, [0x0ad0] = 0x29da8,
  [0x0ad1] = 0x29dae, [0x0ad2] = 0x29dab, [0x0ad3] = 0x2a01d, [0x0ad4] = 0x2a027,
  [0x0ad5] = 0x2a038, [0x0ad6] = 0x2a012, [0x0ad7] = 0x2a048, [0x0ad8] = 0x2a02b,
  [0x0ad9] = 0x2a016, [0x0ada] = 0x2a019, [0x0adb] = 0x2a03d, [0x0adc] = 0x2a023,
  [0x0add] = 0x2a02a, [0x0ade] = 0x2a264, [0x0adf] = 0x2a2ad, [0x0ae0] = 0x2a2ac,
  [0x0ae1] = 0x2fa14, [0x0ae2] = 0x2a2b1, [0x0ae3] = 0x2a2af, [0x0ae4] = 0x2a32c,
  [0x0ae5] = 0x2a324, [0x0ae6] = 0x2a325, [0x0ae7] = 0x2a328, [0x0ae8] = 0x2a3f9,
  [0x0ae9] = 0x2a3f7, [0x0aea] = 0x2a3fd, [0x0aeb] = 0x2a3fe, [0x0aec] = 0x2a439,
  [0x0aed] = 0x2a433, [0x0aee] = 0x2a4d7, [0x0aef] = 0x2a4d8, [0x0af0] = 0x2a4e4,
  [0x0af1] = 0x214a1, [0x0af2] = 0x2a50e, [0x0af3] = 0x2a53b, [0x0af4] = 0x2a544,
  [0x0af5] = 0x2a542, [0x0af6] = 0x2a594, [0x0af7] = 0x2a593, [0x0af8] = 0x2a5d5,
  [0x0af9] = 0x2a6a7, [0x0afa] = 0x20431, [0x0afb] = 0x20423, [0x0afc] = 0x20428,
  [0x0afd] = 0x20427, [0x0afe] = 0x204c6, [0x0aff] = 0x205a3, [0x0b00] = 0x20621,
  [0x0b01] = 0x2081b, [0x0b02] = 0x3510, [0x0b03] = 0x208c1, [0x0b04] = 0x208c3,
  [0x0b05] = 0x2092a, [0x0b06] = 0x20969, [0x0b07] = 0x20a27, [0x0b08] = 0x210b6,
  [0x0b09] = 0x210a7, [0x0b0a] = 0x210a4, [0x0b0b] = 0x210a6, [0x0b0c] = 0x21090,
  [0x0b0d] = 0x2109e, [0x0b0e] = 0x21094, [0x0b0f] = 0x210a8, [0x0b10] = 0x210a5,
  [0x0b11] = 0x210a2, [0x0b12] = 0x21091, [0x0b13] = 0x367b, [0x0b14] = 0x214bc,
  [0x0b15] = 0x214bd, [0x0b16] = 0x214b4, [0x0b17] = 0x214b0, [0x0b18] = 0x214e4,
  [0x0b19] = 0x21545, [0x0b1a] = 0x2154b, [0x0b1b] = 0x2157e, [0x0b1c] = 0x2157f,
  [0x0b1d] = 0x2157d, [0x0b1e] = 0x215c3, [0x0b1f] = 0x218fc, [0x0b20] = 0x218f7,
  [0x0b21] = 0x218f0, [0x0b22] = 0x218ed, [0x0b23] = 0x218f1, [0x0b24] = 0x218f8,
  [0x0b25] = 0x21ae9, [0x0b26] = 0x21ceb, [0x0b27] = 0x381d, [0x0b28] = 0x21f90,
  [0x0b29] = 0x21f8d, [0x0b2a] = 0x21f86, [0x0b2b] = 0x21f91, [0x0b2c] = 0x21f8a,
  [0x0b2d] = 0x22008, [0x0b2e] = 0x22050, [0x0b2f] = 0x222ea, [0x0b30] = 0x222e6,
  [0x0b31] = 0x222e2, [0x0b32] = 0x222e7, [0x0b33] = 0x222ed, [0x0b34] = 0x222e1,
  [0x0b35] = 0x22434, [0x0b36] = 0x22476, [0x0b37] = 0x22475, [0x0b38] = 0x22473,
  [0x0b39] = 0x224b5, [0x0b3a] = 0x22590, [0x0b3b] = 0x22592, [0x0b3c] = 0x228e1,
  [0x0b3d] = 0x228df, [0x0b3e] = 0x228d5, [0x0b3f] = 0x228f2, [0x0b40] = 0x228fe,
  [0x0b41] = 0x22913, [0x0b42] = 0x2292e, [0x0b43] = 0x22a19, [0x0b44] = 0x22e08,
  [0x0b45] = 0x64f5, [0x0b46] = 0x22e05, [0x0b47] = 0x22e09, [0x0b48] = 0x22e06,
  [0x0b49] = 0x22e03, [0x0b4a] = 0x22dfd, [0x0b4b] = 0x22dfc, [0x0b4c] = 0x22e02,
  [0x0b4d] = 0x22e42, [0x0b4e] = 0x2301a, [0x0b4f] = 0x23011, [0x0b50] = 0x23015,
  [0x0b51] = 0x23016, [0x0b52] = 0x230cc, [0x0b53] = 0x230cf, [0x0b54] = 0x230d0,
  [0x0b55] = 0x23122, [0x0b56] = 0x2311e, [0x0b57] = 0x23121, [0x0b58] = 0x232e5,
  [0x0b59] = 0x3b27, [0x0b5a] = 0x23354, [0x0b5b] = 0x232ef, [0x0b5c] = 0x23353,
  [0x0b5d] = 0x23351, [0x0b5e] = 0x233ad, [0x0b5f] = 0x23767, [0x0b60] = 0x23768,
  [0x0b61] = 0x237a4, [0x0b62] = 0x23777, [0x0b63] = 0x23789, [0x0b64] = 0x23744,
  [0x0b65] = 0x2378b, [0x0b66] = 0x23779, [0x0b67] = 0x2375b, [0x0b68] = 0x23743,
  [0x0b69] = 0x23757, [0x0b6a] = 0x2374a, [0x0b6b] = 0x2377c, [0x0b6c] = 0x23746,
  [0x0b6d] = 0x2377b, [0x0b6e] = 0x23756, [0x0b6f] = 0x239a8, [0x0b70] = 0x23a76,
  [0x0b71] = 0x23a72, [0x0b72] = 0x23ad6, [0x0b73] = 0x23ad8, [0x0b74] = 0x23ad1,
  [0x0b75] = 0x23c22, [0x0b76] = 0x23c20, [0x0b77] = 0x23c23, [0x0b78] = 0x23c1e,
  [0x0b79] = 0x23c6e, [0x0b7a] = 0x240a3, [0x0b7b] = 0x24077, [0x0b7c] = 0x240a6,
  [0x0b7d] = 0x2406d, [0x0b7e] = 0x240a2, [0x0b7f] = 0x2407c, [0x0b80] = 0x24084,
  [0x0b81] = 0x24068, [0x0b82] = 0x24074, [0x0b83] = 0x24086, [0x0b84] = 0x240a5,
  [0x0b85] = 0x2407b, [0x0b86] = 0x2407a, [0x0b87] = 0x24069, [0x0b88] = 0x24072,
  [0x0b89] = 0x24076, [0x0b8a] = 0x2444a, [0x0b8b] = 0x24437, [0x0b8c] = 0x2442a,
  [0x0b8d] = 0x2442d, [0x0b8e] = 0x24446, [0x0b8f] = 0x24428, [0x0b90] = 0x24426,
  [0x0b91] = 0x24442, [0x0b92] = 0x2442c, [0x0b93] = 0x24438, [0x0b94] = 0x2442b,
  [0x0b95] = 0x24433, [0x0b96] = 0x24445, [0x0b97] = 0x24539, [0x0b98] = 0x246f9,
  [0x0b99] = 0x246fa, [0x0b9a] = 0x248b8, [0x0b9b] = 0x248b7, [0x0b9c] = 0x248bb,
  [0x0b9d] = 0x248b9, [0x0b9e] = 0x248b4, [0x0b9f] = 0x24a6f, [0x0ba0] = 0x24a87,
  [0x0ba1] = 0x24a8f, [0x0ba2] = 0x24aa2, [0x0ba3] = 0x24aa3, [0x0ba4] = 0x2f931,
  [0x0ba5] = 0x24b9b, [0x0ba6] = 0x24b9d, [0x0ba7] = 0x24bce, [0x0ba8] = 0x3fcf,
  [0x0ba9] = 0x24ebd, [0x0baa] = 0x24ebf, [0x0bab] = 0x24e92, [0x0bac] = 0x3fcd,
  [0x0bad] = 0x24eef, [0x0bae] = 0x24ec9, [0x0baf] = 0x24fa4, [0x0bb0] = 0x24fa8,
  [0x0bb1] = 0x24faa, [0x0bb2] = 0x25028, [0x0bb3] = 0x25024, [0x0bb4] = 0x25025,
  [0x0bb5] = 0x25026, [0x0bb6] = 0x250a9, [0x0bb7] = 0x250ba, [0x0bb8] = 0x250be,
  [0x0bb9] = 0x250bc, [0x0bba] = 0x250c0, [0x0bbb] = 0x252f0, [0x0bbc] = 0x252df,
  [0x0bbd] = 0x252e0, [0x0bbe] = 0x252ed, [0x0bbf] = 0x252db, [0x0bc0] = 0x252fb,
  [0x0bc1] = 0x252b9, [0x0bc2] = 0x252da, [0x0bc3] = 0x252eb, [0x0bc4] = 0x252ec,
  [0x0bc5] = 0x2559a, [0x0bc6] = 0x2559f, [0x0bc7] = 0x2559b, [0x0bc8] = 0x25597,
  [0x0bc9] = 0x255a1, [0x0bca] = 0x2570f, [0x0bcb] = 0x25705, [0x0bcc] = 0x25748,
  [0x0bcd] = 0x4182, [0x0bce] = 0x258bc, [0x0bcf] = 0x258ba, [0x0bd0] = 0x25abf,
  [0x0bd1] = 0x25d01, [0x0bd2] = 0x25ce8, [0x0bd3] = 0x25cef, [0x0bd4] = 0x25ce4,
  [0x0bd5] = 0x25ce6, [0x0bd6] = 0x25d02, [0x0bd7] = 0x25ceb, [0x0bd8] = 0x25ce0,
  [0x0bd9] = 0x25ced, [0x0bda] = 0x25cd9, [0x0bdb] = 0x2f964, [0x0bdc] = 0x25d14,
  [0x0bdd] = 0x25cee, [0x0bde] = 0x4252, [0x0bdf] = 0x25d13, [0x0be0] = 0x25cf9,
  [0x0be1] = 0x25cf8, [0x0be2] = 0x25f25, [0x0be3] = 0x2f969, [0x0be4] = 0x25f19,
  [0x0be5] = 0x25f20, [0x0be6] = 0x25f43, [0x0be7] = 0x25f3f, [0x0be8] = 0x26145,
  [0x0be9] = 0x2614c, [0x0bea] = 0x26149, [0x0beb] = 0x2614f, [0x0bec] = 0x26141,
  [0x0bed] = 0x2613e, [0x0bee] = 0x2614d, [0x0bef] = 0x26152, [0x0bf0] = 0x2614a,
  [0x0bf1] = 0x2614e, [0x0bf2] = 0x26173, [0x0bf3] = 0x26142, [0x0bf4] = 0x26151,
  [0x0bf5] = 0x26155, [0x0bf6] = 0x26150, [0x0bf7] = 0x2616c, [0x0bf9] = 0x2616a,
  [0x0bfa] = 0x26153, [0x0bfb] = 0x26168, [0x0bfc] = 0x26255, [0x0bfd] = 0x26256,
  [0x0bfe] = 0x2631c, [0x0bff] = 0x2631d, [0x0c00] = 0x7f80, [0x0c01] = 0x2631e,
  [0x0c02] = 0x26323, [0x0c03] = 0x2631f, [0x0c04] = 0x263e1, [0x0c05] = 0x263cd,
  [0x0c06] = 0x263cb, [0x0c07] = 0x263cc, [0x0c08] = 0x263c8, [0x0c09] = 0x263c9,
  [0x0c0a] = 0x2649b, [0x0c0b] = 0x26494, [0x0c0c] = 0x26492, [0x0c0d] = 0x26496,
  [0x0c0e] = 0x26493, [0x0c0f] = 0x26495, [0x0c10] = 0x2648f, [0x0c11] = 0x2651d,
  [0x0c12] = 0x26522, [0x0c13] = 0x26521, [0x0c14] = 0x265e9, [0x0c15] = 0x265ef,
  [0x0c16] = 0x265e0, [0x0c17] = 0x265e6, [0x0c18] = 0x265e4, [0x0c19] = 0x26829,
  [0x0c1a] = 0x2682c, [0x0c1b] = 0x26876, [0x0c1c] = 0x26883, [0x0c1d] = 0x26878,
  [0x0c1e] = 0x2683c, [0x0c1f] = 0x24443, [0x0c20] = 0x2687a, [0x0c21] = 0x4451,
  [0x0c22] = 0x268f2, [0x0c23] = 0x2699e, [0x0c24] = 0x2699b, [0x0c25] = 0x2699a,
  [0x0c26] = 0x269f6, [0x0c27] = 0x269f5, [0x0c28] = 0x26aa5, [0x0c29] = 0x26a93,
  [0x0c2a] = 0x26aa4, [0x0c2b] = 0x26e82, [0x0c2c] = 0x26ec7, [0x0c2d] = 0x26fb7,
  [0x0c2e] = 0x2701d, [0x0c2f] = 0x26fe2, [0x0c30] = 0x26fd7, [0x0c31] = 0x26fe3,
  [0x0c32] = 0x26fe4, [0x0c33] = 0x26fbc, [0x0c34] = 0x26fd3, [0x0c35] = 0x455a,
  [0x0c36] = 0x26f5a, [0x0c37] = 0x26fd2, [0x0c38] = 0x26f2d, [0x0c39] = 0x2f9af,
  [0x0c3a] = 0x26fc4, [0x0c3b] = 0x26fd0, [0x0c3c] = 0x26fe5, [0x0c3d] = 0x27005,
  [0x0c3e] = 0x27007, [0x0c3f] = 0x26fe6, [0x0c40] = 0x2701b, [0x0c41] = 0x26fe7,
  [0x0c42] = 0x26fd8, [0x0c43] = 0x26fbe, [0x0c44] = 0x27017, [0x0c45] = 0x26fb4,
  [0x0c46] = 0x26fd9, [0x0c47] = 0x26fe8, [0x0c48] = 0x26fad, [0x0c49] = 0x26faf,
  [0x0c4a] = 0x26fc8, [0x0c4b] = 0x26fe9, [0x0c4c] = 0x26fea, [0x0c4d] = 0x271fe,
  [0x0c4e] = 0x271fb, [0x0c4f] = 0x27200, [0x0c50] = 0x27472, [0x0c51] = 0x27470,
  [0x0c52] = 0x27446, [0x0c53] = 0x27459, [0x0c54] = 0x2745e, [0x0c55] = 0x27448,
  [0x0c56] = 0x2744f, [0x0c57] = 0x27471, [0x0c58] = 0x27460, [0x0c59] = 0x2745f,
  [0x0c5a] = 0x2746e, [0x0c5b] = 0x27473, [0x0c5c] = 0x2f9c0, [0x0c5d] = 0x2f9bf,
  [0x0c5e] = 0x27447, [0x0c5f] = 0x2746d, [0x0c60] = 0x2746f, [0x0c61] = 0x27481,
  [0x0c62] = 0x2746c, [0x0c63] = 0x27478, [0x0c64] = 0x27483, [0x0c65] = 0x27449,
  [0x0c66] = 0x27468, [0x0c67] = 0x27474, [0x0c68] = 0x27463, [0x0c69] = 0x2746a,
  [0x0c6a] = 0x26885, [0x0c6b] = 0x27465, [0x0c6c] = 0x27462, [0x0c6d] = 0x274c8,
  [0x0c6e] = 0x275d0, [0x0c6f] = 0x275d4, [0x0c70] = 0x275d1, [0x0c71] = 0x27603,
  [0x0c72] = 0x27742, [0x0c73] = 0x27763, [0x0c74] = 0x27756, [0x0c75] = 0x2775b,
  [0x0c76] = 0x27755, [0x0c77] = 0x27750, [0x0c78] = 0x2772d, [0x0c79] = 0x27744,
  [0x0c7a] = 0x27748, [0x0c7b] = 0x27745, [0x0c7c] = 0x27782, [0x0c7d] = 0x4665,
  [0x0c7e] = 0x27762, [0x0c7f] = 0x27885, [0x0c80] = 0x2788d, [0x0c81] = 0x27936,
  [0x0c82] = 0x2792f, [0x0c83] = 0x27931, [0x0c84] = 0x27937, [0x0c85] = 0x27aa7,
  [0x0c86] = 0x27ad9, [0x0c87] = 0x27af0, [0x0c88] = 0x27af2, [0x0c89] = 0x27afd,
  [0x0c8a] = 0x27ae8, [0x0c8b] = 0x27aeb, [0x0c8c] = 0x27aee, [0x0c8d] = 0x27ae0,
  [0x0c8e] = 0x27ae9, [0x0c8f] = 0x27aed, [0x0c90] = 0x27ad6, [0x0c91] = 0x27af8,
  [0x0c92] = 0x27ad4, [0x0c93] = 0x27adf, [0x0c94] = 0x27ae7, [0x0c95] = 0x27ad8,
  [0x0c96] = 0x27ae3, [0x0c97] = 0x27aef, [0x0c98] = 0x27b0f, [0x0c99] = 0x27bca,
  [0x0c9a] = 0x21546, [0x0c9b] = 0x27c05, [0x0c9c] = 0x27c0c, [0x0c9d] = 0x27c0d,
  [0x0c9e] = 0x27c7f, [0x0c9f] = 0x27c80, [0x0ca0] = 0x27c81, [0x0ca1] = 0x27d01,
  [0x0ca2] = 0x27d03, [0x0ca3] = 0x27df2, [0x0ca4] = 0x27de2, [0x0ca5] = 0x27de3,
  [0x0ca6] = 0x27dde, [0x0ca7] = 0x27de9, [0x0ca8] = 0x27de8, [0x0ca9] = 0x27de0,
  [0x0caa] = 0x27e01, [0x0cab] = 0x27df5, [0x0cac] = 0x27de4, [0x0cad] = 0x8d01,
  [0x0cae] = 0x27e77, [0x0caf] = 0x27f74, [0x0cb0] = 0x27f6f, [0x0cb1] = 0x27f62,
  [0x0cb2] = 0x27f61, [0x0cb3] = 0x27f6d, [0x0cb4] = 0x27f73, [0x0cb5] = 0x27f6a,
  [0x0cb6] = 0x27f69, [0x0cb7] = 0x28112, [0x0cb8] = 0x2812d, [0x0cb9] = 0x28114,
  [0x0cba] = 0x2810f, [0x0cbb] = 0x28129, [0x0cbc] = 0x28116, [0x0cbd] = 0x28103,
  [0x0cbe] = 0x28146, [0x0cbf] = 0x2815c, [0x0cc0] = 0x28111, [0x0cc1] = 0x28106,
  [0x0cc2] = 0x280dc, [0x0cc3] = 0x2812b, [0x0cc4] = 0x2812a, [0x0cc5] = 0x2812c,
  [0x0cc6] = 0x28127, [0x0cc7] = 0x2827a, [0x0cc8] = 0x2839c, [0x0cc9] = 0x28399,
  [0x0cca] = 0x28395, [0x0ccb] = 0x2838b, [0x0ccc] = 0x28398, [0x0ccd] = 0x28396,
  [0x0cce] = 0x28432, [0x0ccf] = 0x285a4, [0x0cd0] = 0x285aa, [0x0cd1] = 0x2861b,
  [0x0cd2] = 0x48af, [0x0cd3] = 0x2860d, [0x0cd4] = 0x2861c, [0x0cd5] = 0x2860a,
  [0x0cd6] = 0x28620, [0x0cd7] = 0x28608, [0x0cd8] = 0x2861a, [0x0cd9] = 0x28613,
  [0x0cda] = 0x28611, [0x0cdb] = 0x2875d, [0x0cdc] = 0x2875f, [0x0cdd] = 0x2875e,
  [0x0cde] = 0x28760, [0x0cdf] = 0x287bb, [0x0ce0] = 0x287bc, [0x0ce1] = 0x287c1,
  [0x0ce2] = 0x287c0, [0x0ce3] = 0x287c8, [0x0ce4] = 0x287ce, [0x0ce5] = 0x288a7,
  [0x0ce6] = 0x288b2, [0x0ce7] = 0x288b6, [0x0ce8] = 0x288a5, [0x0ce9] = 0x288ba,
  [0x0cea] = 0x288b5, [0x0ceb] = 0x288ad, [0x0cec] = 0x288a4, [0x0ced] = 0x288d3,
  [0x0cee] = 0x288b0, [0x0cef] = 0x288b1, [0x0cf0] = 0x2891d, [0x0cf1] = 0x28a8d,
  [0x0cf2] = 0x4941, [0x0cf3] = 0x28a91, [0x0cf4] = 0x28ab6, [0x0cf5] = 0x28ab7,
  [0x0cf6] = 0x28abd, [0x0cf7] = 0x28abc, [0x0cf8] = 0x28a96, [0x0cf9] = 0x28a94,
  [0x0cfa] = 0x28aa0, [0x0cfb] = 0x28ca8, [0x0cfc] = 0x28ca6, [0x0cfd] = 0x28d84,
  [0x0cfe] = 0x28d96, [0x0cff] = 0x28d88, [0x0d00] = 0x28d9a, [0x0d01] = 0x28ed1,
  [0x0d02] = 0x28ecf, [0x0d03] = 0x28f50, [0x0d04] = 0x28f51, [0x0d05] = 0x28f4e,
  [0x0d06] = 0x28f80, [0x0d07] = 0x28f81, [0x0d08] = 0x2901b, [0x0d09] = 0x29017,
  [0x0d0a] = 0x29020, [0x0d0b] = 0x29019, [0x0d0c] = 0x2901a, [0x0d0d] = 0x29021,
  [0x0d0e] = 0x2901e, [0x0d0f] = 0x29018, [0x0d10] = 0x2901d, [0x0d11] = 0x4a29,
  [0x0d12] = 0x2912d, [0x0d13] = 0x29124, [0x0d14] = 0x29127, [0x0d15] = 0x2912e,
  [0x0d16] = 0x29125, [0x0d17] = 0x2911c, [0x0d18] = 0x29119, [0x0d19] = 0x4a2a,
  [0x0d1a] = 0x29123, [0x0d1b] = 0x2911f, [0x0d1c] = 0x2911a, [0x0d1d] = 0x2912b,
  [0x0d1e] = 0x2911e, [0x0d1f] = 0x291e0, [0x0d20] = 0x29233, [0x0d21] = 0x292e6,
  [0x0d22] = 0x292fc, [0x0d23] = 0x292e5, [0x0d24] = 0x292f8, [0x0d25] = 0x292f6,
  [0x0d26] = 0x292ea, [0x0d27] = 0x292f2, [0x0d28] = 0x292ed, [0x0d29] = 0x292eb,
  [0x0d2a] = 0x292f0, [0x0d2b] = 0x292f1, [0x0d2c] = 0x293c6, [0x0d2d] = 0x293c8,
  [0x0d2e] = 0x293ce, [0x0d2f] = 0x293c5, [0x0d30] = 0x4a96, [0x0d31] = 0x293cb,
  [0x0d32] = 0x29513, [0x0d33] = 0x29514, [0x0d34] = 0x29507, [0x0d35] = 0x2950c,
  [0x0d36] = 0x2961a, [0x0d37] = 0x4b12, [0x0d38] = 0x29617, [0x0d39] = 0x29606,
  [0x0d3a] = 0x29616, [0x0d3b] = 0x29607, [0x0d3c] = 0x29610, [0x0d3d] = 0x29609,
  [0x0d3e] = 0x29619, [0x0d3f] = 0x29615, [0x0d40] = 0x2976e, [0x0d41] = 0x2973b,
  [0x0d42] = 0x2973e, [0x0d43] = 0x2976c, [0x0d44] = 0x29765, [0x0d45] = 0x29764,
  [0x0d46] = 0x29759, [0x0d47] = 0x2977c, [0x0d48] = 0x29770, [0x0d49] = 0x29779,
  [0x0d4a] = 0x2982c, [0x0d4b] = 0x29852, [0x0d4c] = 0x29851, [0x0d4d] = 0x2984c,
  [0x0d4e] = 0x29900, [0x0d4f] = 0x29910, [0x0d50] = 0x29913, [0x0d51] = 0x298ff,
  [0x0d52] = 0x298fe, [0x0d53] = 0x298ed, [0x0d54] = 0x29a5a, [0x0d55] = 0x29a58,
  [0x0d56] = 0x29a5c, [0x0d57] = 0x29ada, [0x0d58] = 0x29b78, [0x0d59] = 0x29b5e,
  [0x0d5a] = 0x29b67, [0x0d5b] = 0x29b64, [0x0d5c] = 0x29c13, [0x0d5d] = 0x29c23,
  [0x0d5e] = 0x29c41, [0x0d5f] = 0x29c3f, [0x0d60] = 0x29c40, [0x0d61] = 0x29ced,
  [0x0d62] = 0x29ce3, [0x0d63] = 0x29cea, [0x0d64] = 0x29cf0, [0x0d65] = 0x29ce6,
  [0x0d66] = 0x29ce9, [0x0d67] = 0x29cf1, [0x0d68] = 0x29cee, [0x0d69] = 0x29dd4,
  [0x0d6a] = 0x29dd1, [0x0d6b] = 0x29ddc, [0x0d6c] = 0x29dec, [0x0d6d] = 0x2a069,
  [0x0d6e] = 0x2a06d, [0x0d6f] = 0x2a057, [0x0d70] = 0x2a066, [0x0d71] = 0x2a0f9,
  [0x0d72] = 0x2a04a, [0x0d73] = 0x2a060, [0x0d74] = 0x2a056, [0x0d75] = 0x2a059,
  [0x0d76] = 0x2a04c, [0x0d77] = 0x2a06a, [0x0d78] = 0x2a062, [0x0d79] = 0x2a063,
  [0x0d7a] = 0x2a070, [0x0d7b] = 0x2a05f, [0x0d7c] = 0x2fa0d, [0x0d7d] = 0x2a064,
  [0x0d7e] = 0x2a05d, [0x0d7f] = 0x2a068, [0x0d80] = 0x2a09f, [0x0d81] = 0x2a2ba,
  [0x0d82] = 0x2a2b8, [0x0d83] = 0x2a2bc, [0x0d84] = 0x2a2b9, [0x0d85] = 0x2a2b4,
  [0x0d86] = 0x2a33b, [0x0d87] = 0x2a32d, [0x0d88] = 0x2a338, [0x0d89] = 0x2a32f,
  [0x0d8a] = 0x2a332, [0x0d8b] = 0x2a441, [0x0d8c] = 0x2a4cb, [0x0d8d] = 0x2a4de,
  [0x0d8e] = 0x2a4dd, [0x0d8f] = 0x2a4da, [0x0d90] = 0x2a4dc, [0x0d91] = 0x2a510,
  [0x0d92] = 0x2a54f, [0x0d93] = 0x2a549, [0x0d94] = 0x2a598, [0x0d95] = 0x2a596,
  [0x0d96] = 0x2a597, [0x0d97] = 0x2a5c7, [0x0d98] = 0x2019c, [0x0d99] = 0x2a5da,
  [0x0d9a] = 0x2a5d8, [0x0d9b] = 0x2a6a8, [0x0d9c] = 0x2010a, [0x0d9d] = 0x2019d,
  [0x0d9e] = 0x204cb, [0x0d9f] = 0x20538, [0x0da0] = 0x20539, [0x0da1] = 0x205a6,
  [0x0da2] = 0x20823, [0x0da3] = 0x20822, [0x0da4] = 0x20821, [0x0da5] = 0x209ce,
  [0x0da6] = 0x20b92, [0x0da7] = 0x20b91, [0x0da8] = 0x210ec, [0x0da9] = 0x210e0,
  [0x0daa] = 0x210ed, [0x0dab] = 0x21108, [0x0dac] = 0x210e5, [0x0dad] = 0x210ee,
  [0x0dae] = 0x210e4, [0x0daf] = 0x210eb, [0x0db0] = 0x210e3, [0x0db1] = 0x210ea,
  [0x0db2] = 0x2110a, [0x0db3] = 0x2f859, [0x0db4] = 0x214d1, [0x0db5] = 0x214e3,
  [0x0db6] = 0x214d4, [0x0db7] = 0x214d0, [0x0db8] = 0x214d9, [0x0db9] = 0x367e,
  [0x0dba] = 0x58e1, [0x0dbb] = 0x214da, [0x0dbc] = 0x214d3, [0x0dbd] = 0x2154c,
  [0x0dbe] = 0x2154d, [0x0dbf] = 0x2157c, [0x0dc0] = 0x21580, [0x0dc1] = 0x215cc,
  [0x0dc2] = 0x218ff, [0x0dc3] = 0x21908, [0x0dc4] = 0x2a508, [0x0dc5] = 0x21901,
  [0x0dc6] = 0x2f86b, [0x0dc7] = 0x21900, [0x0dc8] = 0x21aed, [0x0dc9] = 0x21af3,
  [0x0dca] = 0x21aee, [0x0dcb] = 0x21af1, [0x0dcc] = 0x21af0, [0x0dcd] = 0x21ade,
  [0x0dce] = 0x21b51, [0x0dcf] = 0x21f82, [0x0dd0] = 0x21fa9, [0x0dd1] = 0x21f98,
  [0x0dd2] = 0x21f9d, [0x0dd3] = 0x21f9a, [0x0dd4] = 0x21f9e, [0x0dd5] = 0x21f9f,
  [0x0dd6] = 0x21fa6, [0x0dd7] = 0x21fa7, [0x0dd8] = 0x22009, [0x0dd9] = 0x2202f,
  [0x0dda] = 0x22171, [0x0ddb] = 0x2216d, [0x0ddc] = 0x22172, [0x0ddd] = 0x222ef,
  [0x0dde] = 0x222f0, [0x0ddf] = 0x2243b, [0x0de0] = 0x22439, [0x0de1] = 0x2243c,
  [0x0de2] = 0x22438, [0x0de3] = 0x24bfd, [0x0de4] = 0x2243a, [0x0de5] = 0x22478,
  [0x0de6] = 0x22479, [0x0de7] = 0x22477, [0x0de8] = 0x22598, [0x0de9] = 0x2259c,
  [0x0dea] = 0x22599, [0x0deb] = 0x2259a, [0x0dec] = 0x22911, [0x0ded] = 0x2290a,
  [0x0dee] = 0x228fd, [0x0def] = 0x2290f, [0x0df0] = 0x22919, [0x0df1] = 0x22903,
  [0x0df2] = 0x22915, [0x0df3] = 0x2290c, [0x0df4] = 0x22909, [0x0df5] = 0x22912,
  [0x0df6] = 0x22934, [0x0df7] = 0x2292a, [0x0df8] = 0x22908, [0x0df9] = 0x2292d,
  [0x0dfa] = 0x22928, [0x0dfb] = 0x2f8b1, [0x0dfc] = 0x2292c, [0x0dfd] = 0x22926,
  [0x0dfe] = 0x22933, [0x0dff] = 0x39a7, [0x0e00] = 0x22a1a, [0x0e01] = 0x22a1e,
  [0x0e02] = 0x22e07, [0x0e03] = 0x22e2c, [0x0e04] = 0x22e32, [0x0e05] = 0x22e28,
  [0x0e06] = 0x22e31, [0x0e07] = 0x22e29, [0x0e08] = 0x22e30, [0x0e09] = 0x22e2a,
  [0x0e0a] = 0x22e44, [0x0e0b] = 0x22e2e, [0x0e0c] = 0x230d1, [0x0e0d] = 0x23124,
  [0x0e0e] = 0x232f7, [0x0e0f] = 0x232f4, [0x0e10] = 0x232f3, [0x0e11] = 0x232f8,
  [0x0e12] = 0x237b5, [0x0e13] = 0x23796, [0x0e14] = 0x23798, [0x0e15] = 0x23795,
  [0x0e16] = 0x23791, [0x0e17] = 0x237b2, [0x0e18] = 0x2379e, [0x0e19] = 0x23759,
  [0x0e1a] = 0x237a3, [0x0e1b] = 0x2379a, [0x0e1c] = 0x2379b, [0x0e1d] = 0x4320,
  [0x0e1e] = 0x26183, [0x0e1f] = 0x23ada, [0x0e20] = 0x23adf, [0x0e21] = 0x23b16,
  [0x0e22] = 0x23c1f, [0x0e23] = 0x23c2d, [0x0e24] = 0x23c2e, [0x0e25] = 0x23c2b,
  [0x0e26] = 0x240b8, [0x0e27] = 0x240bb, [0x0e28] = 0x240bf, [0x0e29] = 0x240ba,
  [0x0e2a] = 0x240d5, [0x0e2b] = 0x240e3, [0x0e2c] = 0x240c1, [0x0e2d] = 0x240be,
  [0x0e2e] = 0x240bd, [0x0e2f] = 0x240b4, [0x0e30] = 0x240c2, [0x0e31] = 0x240a1,
  [0x0e32] = 0x24087, [0x0e33] = 0x240d7, [0x0e34] = 0x240ca, [0x0e35] = 0x240b5,
  [0x0e36] = 0x240da, [0x0e37] = 0x240d9, [0x0e38] = 0x240b3, [0x0e39] = 0x240d8,
  [0x0e3a] = 0x24467, [0x0e3b] = 0x24471, [0x0e3c] = 0x24462, [0x0e3d] = 0x2445c,
  [0x0e3e] = 0x24468, [0x0e3f] = 0x24452, [0x0e40] = 0x24456, [0x0e41] = 0x21109,
  [0x0e42] = 0x20442, [0x0e43] = 0x245a0, [0x0e44] = 0x24700, [0x0e45] = 0x246fe,
  [0x0e46] = 0x246ff, [0x0e47] = 0x248cb, [0x0e48] = 0x2f928, [0x0e49] = 0x248ca,
  [0x0e4a] = 0x248a9, [0x0e4b] = 0x248c8, [0x0e4c] = 0x24ab4, [0x0e4d] = 0x24aac,
  [0x0e4e] = 0x24aaa, [0x0e4f] = 0x24aa9, [0x0e50] = 0x24c0e, [0x0e51] = 0x24ce9,
  [0x0e52] = 0x24ced, [0x0e53] = 0x24cf2, [0x0e54] = 0x24ceb, [0x0e55] = 0x24cee,
  [0x0e56] = 0x24ee8, [0x0e57] = 0x24edb, [0x0e58] = 0x24ed7, [0x0e59] = 0x24ee3,
  [0x0e5a] = 0x24ee5, [0x0e5b] = 0x24eee, [0x0e5c] = 0x24ed5, [0x0e5d] = 0x24fb3,
  [0x0e5e] = 0x2502d, [0x0e5f] = 0x250c1, [0x0e60] = 0x250c3, [0x0e61] = 0x2530c,
  [0x0e62] = 0x2530e, [0x0e63] = 0x25307, [0x0e64] = 0x25317, [0x0e65] = 0x25309,
  [0x0e66] = 0x25316, [0x0e67] = 0x2539a, [0x0e68] = 0x2539c, [0x0e69] = 0x255b4,
  [0x0e6a] = 0x255b7, [0x0e6b] = 0x255b3, [0x0e6c] = 0x2154e, [0x0e6d] = 0x25713,
  [0x0e6e] = 0x25714, [0x0e6f] = 0x258e6, [0x0e70] = 0x258dc, [0x0e71] = 0x258e8,
  [0x0e72] = 0x258e5, [0x0e73] = 0x25a2e, [0x0e74] = 0x25a2c, [0x0e75] = 0x25a2b,
  [0x0e76] = 0x25acd, [0x0e77] = 0x25d3d, [0x0e78] = 0x25d32, [0x0e79] = 0x25d2d,
  [0x0e7a] = 0x25d45, [0x0e7b] = 0x25d3e, [0x0e7c] = 0x25d50, [0x0e7d] = 0x25d25,
  [0x0e7e] = 0x25d53, [0x0e7f] = 0x25d23, [0x0e80] = 0x25f37, [0x0e81] = 0x25f38,
  [0x0e82] = 0x25f47, [0x0e83] = 0x25f3d, [0x0e84] = 0x25f3e, [0x0e85] = 0x25f49,
  [0x0e86] = 0x25f4a, [0x0e87] = 0x25f1d, [0x0e88] = 0x7ce9, [0x0e89] = 0x261a5,
  [0x0e8a] = 0x2618c, [0x0e8b] = 0x2618d, [0x0e8c] = 0x26189, [0x0e8d] = 0x26196,
  [0x0e8e] = 0x26185, [0x0e8f] = 0x2618f, [0x0e90] = 0x26177, [0x0e91] = 0x2618e,
  [0x0e92] = 0x26182, [0x0e93] = 0x2618a, [0x0e94] = 0x26188, [0x0e95] = 0x2617b,
  [0x0e96] = 0x26197, [0x0e97] = 0x2617d, [0x0e98] = 0x26179, [0x0e99] = 0x26259,
  [0x0e9a] = 0x26324, [0x0e9b] = 0x2632d, [0x0e9c] = 0x2632e, [0x0e9d] = 0x2632b,
  [0x0e9e] = 0x2f976, [0x0e9f] = 0x263da, [0x0ea0] = 0x263d8, [0x0ea1] = 0x263d6,
  [0x0ea2] = 0x26487, [0x0ea3] = 0x264a0, [0x0ea4] = 0x26528, [0x0ea5] = 0x26525,
  [0x0ea6] = 0x2651f, [0x0ea7] = 0x265f3, [0x0ea8] = 0x265f7, [0x0ea9] = 0x265f6,
  [0x0eaa] = 0x2682b, [0x0eab] = 0x2685b, [0x0eac] = 0x26848, [0x0ead] = 0x81cb,
  [0x0eae] = 0x2685c, [0x0eaf] = 0x2686d, [0x0eb0] = 0x2689d, [0x0eb1] = 0x26899,
  [0x0eb2] = 0x2688c, [0x0eb3] = 0x26891, [0x0eb4] = 0x2689b, [0x0eb5] = 0x2689a,
  [0x0eb6] = 0x2689c, [0x0eb7] = 0x26895, [0x0eb8] = 0x2688d, [0x0eb9] = 0x26896,
  [0x0eba] = 0x268a5, [0x0ebb] = 0x2692a, [0x0ebc] = 0x269a1, [0x0ebd] = 0x269a4,
  [0x0ebe] = 0x269ad, [0x0ebf] = 0x26aa9, [0x0ec0] = 0x26aae, [0x0ec1] = 0x26ab0,
  [0x0ec2] = 0x2700d, [0x0ec3] = 0x26f63, [0x0ec4] = 0x26f71, [0x0ec5] = 0x27051,
  [0x0ec6] = 0x27054, [0x0ec7] = 0x2702a, [0x0ec8] = 0x27044, [0x0ec9] = 0x27055,
  [0x0eca] = 0x27099, [0x0ecb] = 0x27039, [0x0ecc] = 0x2703f, [0x0ecd] = 0x2703e,
  [0x0ece] = 0x2704f, [0x0ecf] = 0x2704d, [0x0ed0] = 0x27035, [0x0ed1] = 0x27040,
  [0x0ed2] = 0x27031, [0x0ed3] = 0x26fd5, [0x0ed4] = 0x2702b, [0x0ed5] = 0x27033,
  [0x0ed6] = 0x27041, [0x0ed7] = 0x27056, [0x0ed8] = 0x2704c, [0x0ed9] = 0x27046,
  [0x0eda] = 0x2703c, [0x0edb] = 0x27045, [0x0edc] = 0x27043, [0x0edd] = 0x2703d,
  [0x0ede] = 0x27070, [0x0edf] = 0x27057, [0x0ee0] = 0x27038, [0x0ee1] = 0x27058,
  [0x0ee2] = 0x4565, [0x0ee3] = 0x27037, [0x0ee4] = 0x27207, [0x0ee5] = 0x27206,
  [0x0ee6] = 0x27209, [0x0ee7] = 0x274ab, [0x0ee8] = 0x27490, [0x0ee9] = 0x27493,
  [0x0eea] = 0x274bc, [0x0eeb] = 0x274a9, [0x0eec] = 0x2749e, [0x0eed] = 0x274bf,
  [0x0eee] = 0x274aa, [0x0eef] = 0x27491, [0x0ef0] = 0x274a4, [0x0ef1] = 0x2749a,
  [0x0ef2] = 0x274a7, [0x0ef3] = 0x274a1, [0x0ef4] = 0x2749c, [0x0ef5] = 0x274a2,
  [0x0ef6] = 0x2749b, [0x0ef7] = 0x2749f, [0x0ef8] = 0x27494, [0x0ef9] = 0x2748f,
  [0x0efa] = 0x272f0, [0x0efb] = 0x27492, [0x0efc] = 0x27495, [0x0efd] = 0x274a5,
  [0x0efe] = 0x274a6, [0x0eff] = 0x27604, [0x0f00] = 0x2779c, [0x0f01] = 0x27779,
  [0x0f02] = 0x2777a, [0x0f03] = 0x2777e, [0x0f04] = 0x2777b, [0x0f05] = 0x27771,
  [0x0f06] = 0x27781, [0x0f07] = 0x2777f, [0x0f08] = 0x2777c, [0x0f09] = 0x2777d,
  [0x0f0a] = 0x27775, [0x0f0b] = 0x27776, [0x0f0c] = 0x2788e, [0x0f0d] = 0x2788f,
  [0x0f0e] = 0x2793e, [0x0f0f] = 0x2793f, [0x0f10] = 0x27940, [0x0f11] = 0x27941,
  [0x0f12] = 0x4704, [0x0f13] = 0x27b0d, [0x0f14] = 0x27b17, [0x0f15] = 0x27b10,
  [0x0f16] = 0x27b0e, [0x0f17] = 0x27aea, [0x0f18] = 0x27b1d, [0x0f19] = 0x27b03,
  [0x0f1a] = 0x27b22, [0x0f1b] = 0x27b04, [0x0f1c] = 0x27b00, [0x0f1d] = 0x27b20,
  [0x0f1e] = 0x27b21, [0x0f1f] = 0x27b23, [0x0f20] = 0x27b13, [0x0f21] = 0x27b09,
  [0x0f22] = 0x27b11, [0x0f23] = 0x27bcb, [0x0f24] = 0x27bce, [0x0f25] = 0x27bd0,
  [0x0f26] = 0x27bcc, [0x0f27] = 0x27bd4, [0x0f28] = 0x214db, [0x0f29] = 0x27c09,
  [0x0f2a] = 0x27c0b, [0x0f2b] = 0x27c85, [0x0f2c] = 0x27d06, [0x0f2d] = 0x27d0d,
  [0x0f2e] = 0x4764, [0x0f2f] = 0x27d14, [0x0f30] = 0x27d0f, [0x0f31] = 0x27e09,
  [0x0f32] = 0x27e14, [0x0f33] = 0x27e0b, [0x0f34] = 0x27dfc, [0x0f35] = 0x27e04,
  [0x0f36] = 0x27e0a, [0x0f37] = 0x27e00, [0x0f38] = 0x27dfd, [0x0f39] = 0x27e07,
  [0x0f3a] = 0x27e06, [0x0f3b] = 0x27e11, [0x0f3c] = 0x27e79, [0x0f3d] = 0x27e78,
  [0x0f3e] = 0x27f88, [0x0f3f] = 0x27f80, [0x0f40] = 0x27f8b, [0x0f41] = 0x28159,
  [0x0f42] = 0x28161, [0x0f43] = 0x28175, [0x0f44] = 0x4823, [0x0f45] = 0x28155,
  [0x0f46] = 0x2815b, [0x0f47] = 0x2815f, [0x0f48] = 0x28152, [0x0f49] = 0x28162,
  [0x0f4a] = 0x28172, [0x0f4b] = 0x2815d, [0x0f4c] = 0x28168, [0x0f4d] = 0x28171,
  [0x0f4e] = 0x28165, [0x0f4f] = 0x28166, [0x0f50] = 0x28167, [0x0f51] = 0x28176,
  [0x0f52] = 0x2814c, [0x0f53] = 0x28160, [0x0f54] = 0x28174, [0x0f55] = 0x28150,
  [0x0f56] = 0x2828a, [0x0f57] = 0x28281, [0x0f58] = 0x28286, [0x0f59] = 0x2827f,
  [0x0f5a] = 0x28280, [0x0f5b] = 0x283a5, [0x0f5c] = 0x283a7, [0x0f5d] = 0x283a8,
  [0x0f5e] = 0x283a6, [0x0f5f] = 0x283af, [0x0f60] = 0x26195, [0x0f61] = 0x283b1,
  [0x0f62] = 0x28435, [0x0f63] = 0x28439, [0x0f64] = 0x285c3, [0x0f65] = 0x28630,
  [0x0f66] = 0x2862a, [0x0f67] = 0x2862b, [0x0f68] = 0x2862d, [0x0f69] = 0x2862e,
  [0x0f6a] = 0x2862c, [0x0f6b] = 0x28623, [0x0f6c] = 0x28621, [0x0f6d] = 0x28622,
  [0x0f6e] = 0x2876c, [0x0f6f] = 0x28781, [0x0f70] = 0x2878f, [0x0f71] = 0x28780,
  [0x0f72] = 0x287d0, [0x0f73] = 0x287cd, [0x0f74] = 0x287d5, [0x0f75] = 0x287d4,
  [0x0f76] = 0x288d1, [0x0f77] = 0x288be, [0x0f78] = 0x288cb, [0x0f79] = 0x288ce,
  [0x0f7a] = 0x288bd, [0x0f7b] = 0x288d0, [0x0f7c] = 0x28b04, [0x0f7d] = 0x28ad5,
  [0x0f7e] = 0x28ad0, [0x0f7f] = 0x28ad3, [0x0f80] = 0x28afb, [0x0f81] = 0x28ad8,
  [0x0f82] = 0x28ad1, [0x0f83] = 0x28afd, [0x0f84] = 0x28ad9, [0x0f85] = 0x28ad6,
  [0x0f86] = 0x28ae6, [0x0f87] = 0x28af9, [0x0f88] = 0x28da1, [0x0f89] = 0x28d9d,
  [0x0f8a] = 0x28d9e, [0x0f8b] = 0x95d9, [0x0f8c] = 0x28eff, [0x0f8d] = 0x28f5f,
  [0x0f8e] = 0x28f57, [0x0f8f] = 0x28f60, [0x0f90] = 0x28f59, [0x0f91] = 0x2902c,
  [0x0f92] = 0x29025, [0x0f93] = 0x29027, [0x0f94] = 0x29030, [0x0f95] = 0x29024,
  [0x0f96] = 0x29026, [0x0f97] = 0x2902d, [0x0f98] = 0x2902e, [0x0f99] = 0x29029,
  [0x0f9a] = 0x29031, [0x0f9b] = 0x29145, [0x0f9c] = 0x29147, [0x0f9d] = 0x29152,
  [0x0f9e] = 0x2914a, [0x0f9f] = 0x29150, [0x0fa0] = 0x29146, [0x0fa1] = 0x2914f,
  [0x0fa2] = 0x2914e, [0x0fa3] = 0x29153, [0x0fa4] = 0x29321, [0x0fa5] = 0x29309,
  [0x0fa6] = 0x2931a, [0x0fa7] = 0x2931b, [0x0fa8] = 0x25315, [0x0fa9] = 0x29310,
  [0x0faa] = 0x2f9f9, [0x0fab] = 0x29314, [0x0fac] = 0x2930e, [0x0fad] = 0x29312,
  [0x0fae] = 0x2930b, [0x0faf] = 0x293cf, [0x0fb0] = 0x293d2, [0x0fb1] = 0x293d0,
  [0x0fb2] = 0x293d4, [0x0fb3] = 0x293d3, [0x0fb4] = 0x293d1, [0x0fb5] = 0x20b18,
  [0x0fb6] = 0x2f9fb, [0x0fb7] = 0x29409, [0x0fb8] = 0x2951c, [0x0fb9] = 0x29527,
  [0x0fba] = 0x29525, [0x0fbb] = 0x2951b, [0x0fbc] = 0x29529, [0x0fbd] = 0x2951f,
  [0x0fbe] = 0x29530, [0x0fbf] = 0x29524, [0x0fc0] = 0x29528, [0x0fc1] = 0x29519,
  [0x0fc2] = 0x2952f, [0x0fc3] = 0x29624, [0x0fc4] = 0x29621, [0x0fc5] = 0x29625,
  [0x0fc6] = 0x29626, [0x0fc7] = 0x29627, [0x0fc8] = 0x29676, [0x0fc9] = 0x29766,
  [0x0fca] = 0x29775, [0x0fcb] = 0x29769, [0x0fcc] = 0x2977e, [0x0fcd] = 0x2978f,
  [0x0fce] = 0x29774, [0x0fcf] = 0x27210, [0x0fd0] = 0x297ad, [0x0fd1] = 0x2982b,
  [0x0fd2] = 0x2982a, [0x0fd3] = 0x29858, [0x0fd4] = 0x29922, [0x0fd5] = 0x2991d,
  [0x0fd6] = 0x2992b, [0x0fd7] = 0x2992c, [0x0fd8] = 0x2992d, [0x0fd9] = 0x29933,
  [0x0fda] = 0x2991b, [0x0fdb] = 0x29927, [0x0fdc] = 0x2992a, [0x0fdd] = 0x29928,
  [0x0fde] = 0x2993b, [0x0fdf] = 0x29a7e, [0x0fe0] = 0x29a71, [0x0fe1] = 0x29a79,
  [0x0fe2] = 0x29a78, [0x0fe3] = 0x29a70, [0x0fe4] = 0x29a6d, [0x0fe5] = 0x29a7d,
  [0x0fe6] = 0x29a75, [0x0fe7] = 0x29a76, [0x0fe8] = 0x29ade, [0x0fe9] = 0x29b66,
  [0x0fea] = 0x29b83, [0x0feb] = 0x29b87, [0x0fec] = 0x29b7b, [0x0fed] = 0x29b89,
  [0x0fee] = 0x29b86, [0x0fef] = 0x29b82, [0x0ff0] = 0x29b7c, [0x0ff1] = 0x29b81,
  [0x0ff2] = 0x29c43, [0x0ff3] = 0x29c45, [0x0ff4] = 0x29cf7, [0x0ff5] = 0x29cf9,
  [0x0ff6] = 0x29cfa, [0x0ff7] = 0x29e05, [0x0ff8] = 0x29dfe, [0x0ff9] = 0x29e0f,
  [0x0ffa] = 0x29dff, [0x0ffb] = 0x29dfa, [0x0ffc] = 0x29e09, [0x0ffd] = 0x29e20,
  [0x0ffe] = 0x29e0c, [0x0fff] = 0x29e3c, [0x1000] = 0x29e22, [0x1001] = 0x29df8,
  [0x1002] = 0x29e0a, [0x1003] = 0x29e08, [0x1004] = 0x29df7, [0x1005] = 0x2a08e,
  [0x1006] = 0x2a077, [0x1007] = 0x2a08b, [0x1008] = 0x2a0b4, [0x1009] = 0x2a08a,
  [0x100a] = 0x2a09a, [0x100b] = 0x2a079, [0x100c] = 0x2a083, [0x100d] = 0x2a07f,
  [0x100e] = 0x2a0a1, [0x100f] = 0x2a08f, [0x1010] = 0x2a0a3, [0x1011] = 0x2a081,
  [0x1012] = 0x2a094, [0x1013] = 0x2a07e, [0x1014] = 0x2a082, [0x1015] = 0x2a090,
  [0x1016] = 0x2a0a5, [0x1017] = 0x2a0ad, [0x1018] = 0x2a09d, [0x1019] = 0x2a267,
  [0x101a] = 0x2a269, [0x101b] = 0x2a2cb, [0x101c] = 0x2a2c8, [0x101d] = 0x2a2d1,
  [0x101e] = 0x2a340, [0x101f] = 0x2a34b, [0x1020] = 0x2a349, [0x1021] = 0x2a346,
  [0x1022] = 0x2a33e, [0x1023] = 0x2a343, [0x1024] = 0x2a33f, [0x1025] = 0x2a3a5,
  [0x1026] = 0x2a3a7, [0x1027] = 0x2a44e, [0x1028] = 0x2a44d, [0x1029] = 0x2259b,
  [0x102a] = 0x2a4e5, [0x102b] = 0x2a4e1, [0x102c] = 0x2a4e2, [0x102d] = 0x2a516,
  [0x102e] = 0x2a514, [0x102f] = 0x2fa1b, [0x1030] = 0x2a559, [0x1031] = 0x2a551,
  [0x1032] = 0x2a55f, [0x1033] = 0x2a54a, [0x1034] = 0x2a557, [0x1035] = 0x2a558,
  [0x1036] = 0x2a5ca, [0x1037] = 0x2a5db, [0x1038] = 0x2a5de, [0x1039] = 0x2a5e0,
  [0x103a] = 0x2a5e1, [0x103b] = 0x2a5df, [0x103c] = 0x2a5e2, [0x103d] = 0x2a5e3,
  [0x103e] = 0x2a692, [0x103f] = 0x2a6bf, [0x1040] = 0x2a6be, [0x1041] = 0x2010b,
  [0x1042] = 0x20452, [0x1043] = 0x2044e, [0x1044] = 0x34b9, [0x1045] = 0x2f813,
  [0x1046] = 0x205a7, [0x1047] = 0x20826, [0x1048] = 0x20827, [0x1049] = 0x208c6,
  [0x104a] = 0x2090b, [0x104b] = 0x2096a, [0x104c] = 0x20978, [0x104d] = 0x2111a,
  [0x104e] = 0x21116, [0x104f] = 0x21119, [0x1050] = 0x21117, [0x1051] = 0x2111b,
  [0x1052] = 0x21118, [0x1053] = 0x21120, [0x1054] = 0x21237, [0x1055] = 0x214ec,
  [0x1056] = 0x21581, [0x1057] = 0x2923d, [0x1058] = 0x2190f, [0x1059] = 0x219ad,
  [0x105a] = 0x21afb, [0x105b] = 0x21b52, [0x105c] = 0x21faf, [0x105d] = 0x21fb7,
  [0x105e] = 0x21fb2, [0x105f] = 0x22178, [0x1060] = 0x221ac, [0x1061] = 0x22300,
  [0x1062] = 0x222fe, [0x1063] = 0x22302, [0x1064] = 0x222fd, [0x1065] = 0x22303,
  [0x1066] = 0x22440, [0x1067] = 0x22443, [0x1068] = 0x22442, [0x1069] = 0x224b7,
  [0x106a] = 0x225a2, [0x106b] = 0x22900, [0x106c] = 0x22935, [0x106d] = 0x22941,
  [0x106e] = 0x22905, [0x106f] = 0x20453, [0x1070] = 0x22950, [0x1071] = 0x2294e,
  [0x1072] = 0x22953, [0x1073] = 0x22e53, [0x1074] = 0x22e50, [0x1075] = 0x22e57,
  [0x1076] = 0x22e5f, [0x1077] = 0x22e55, [0x1078] = 0x22eea, [0x1079] = 0x23026,
  [0x107a] = 0x29830, [0x107b] = 0x2302a, [0x107c] = 0x23028, [0x107d] = 0x2302c,
  [0x107e] = 0x2302d, [0x107f] = 0x230d4, [0x1080] = 0x23307, [0x1081] = 0x23358,
  [0x1082] = 0x23359, [0x1083] = 0x237c5, [0x1084] = 0x237cd, [0x1085] = 0x237c7,
  [0x1086] = 0x237e8, [0x1087] = 0x3c4b, [0x1088] = 0x23932, [0x1089] = 0x2a697,
  [0x108a] = 0x23ade, [0x108b] = 0x23c32, [0x108c] = 0x20ac8, [0x108d] = 0x2f915,
  [0x108e] = 0x240f1, [0x108f] = 0x240f0, [0x1090] = 0x240ec, [0x1091] = 0x24109,
  [0x1092] = 0x240f9, [0x1093] = 0x240f5, [0x1094] = 0x240fe, [0x1095] = 0x24474,
  [0x1096] = 0x24481, [0x1097] = 0x2447c, [0x1098] = 0x24475, [0x1099] = 0x24489,
  [0x109a] = 0x24482, [0x109b] = 0x24497, [0x109c] = 0x24486, [0x109d] = 0x2447d,
  [0x109e] = 0x24493, [0x109f] = 0x2449c, [0x10a0] = 0x24476, [0x10a1] = 0x24480,
  [0x10a2] = 0x24545, [0x10a3] = 0x206a1, [0x10a4] = 0x24703, [0x10a5] = 0x248c9,
  [0x10a6] = 0x248cd, [0x10a7] = 0x248d0, [0x10a8] = 0x24aad, [0x10a9] = 0x24ac5,
  [0x10aa] = 0x24ba2, [0x10ab] = 0x24cec, [0x10ac] = 0x24cf6, [0x10ad] = 0x24cf3,
  [0x10ae] = 0x24ef3, [0x10af] = 0x24efa, [0x10b0] = 0x24ef9, [0x10b1] = 0x24ef5,
  [0x10b2] = 0x24ef4, [0x10b3] = 0x24ef8, [0x10b4] = 0x24fb6, [0x10b5] = 0x24fb4,
  [0x10b6] = 0x25032, [0x10b7] = 0x250cd, [0x10b8] = 0x250c8, [0x10b9] = 0x250ce,
  [0x10ba] = 0x250ca, [0x10bb] = 0x2532a, [0x10bc] = 0x25321, [0x10bd] = 0x2531d,
  [0x10be] = 0x255bd, [0x10bf] = 0x255be, [0x10c0] = 0x255c2, [0x10c1] = 0x40f1,
  [0x10c2] = 0x255c9, [0x10c3] = 0x2571f, [0x10c4] = 0x258f1, [0x10c5] = 0x258ed,
  [0x10c6] = 0x258f2, [0x10c7] = 0x258e0, [0x10c8] = 0x258f7, [0x10c9] = 0x25a30,
  [0x10ca] = 0x25a37, [0x10cb] = 0x25a31, [0x10cc] = 0x25a36, [0x10cd] = 0x25ac8,
  [0x10ce] = 0x25d6d, [0x10cf] = 0x25d69, [0x10d0] = 0x25d7d, [0x10d1] = 0x25d61,
  [0x10d2] = 0x25d70, [0x10d3] = 0x25d71, [0x10d4] = 0x25d73, [0x10d5] = 0x25d76,
  [0x10d6] = 0x25d75, [0x10d7] = 0x25d78, [0x10d8] = 0x25d79, [0x10d9] = 0x25d64,
  [0x10da] = 0x25d6e, [0x10db] = 0x25f51, [0x10dc] = 0x25f4f, [0x10dd] = 0x25f22,
  [0x10de] = 0x261af, [0x10df] = 0x261aa, [0x10e0] = 0x261a3, [0x10e1] = 0x2619d,
  [0x10e2] = 0x2619c, [0x10e3] = 0x261a1, [0x10e4] = 0x261b6, [0x10e5] = 0x261ac,
  [0x10e6] = 0x261a2, [0x10e7] = 0x261a7, [0x10e8] = 0x261b0, [0x10e9] = 0x261a9,
  [0x10ea] = 0x261c3, [0x10eb] = 0x26331, [0x10ec] = 0x26332, [0x10ed] = 0x26333,
  [0x10ee] = 0x26334, [0x10ef] = 0x26337, [0x10f0] = 0x2633c, [0x10f1] = 0x263d9,
  [0x10f2] = 0x263dd, [0x10f3] = 0x263de, [0x10f4] = 0x263df, [0x10f5] = 0x263e0,
  [0x10f6] = 0x264a5, [0x10f7] = 0x264aa, [0x10f8] = 0x264a2, [0x10f9] = 0x264a3,
  [0x10fa] = 0x26604, [0x10fb] = 0x26603, [0x10fc] = 0x265fe, [0x10fd] = 0x26628,
  [0x10fe] = 0x268af, [0x10ff] = 0x268ad, [0x1100] = 0x268a6, [0x1101] = 0x269ac,
  [0x1102] = 0x269a5, [0x1103] = 0x269b0, [0x1104] = 0x269b1, [0x1105] = 0x26a01,
  [0x1106] = 0x26ab2, [0x1107] = 0x26ad2, [0x1108] = 0x26af1, [0x1109] = 0x26fd1,
  [0x110a] = 0x27047, [0x110b] = 0x270c9, [0x110c] = 0x270a7, [0x110d] = 0x270c8,
  [0x110e] = 0x27095, [0x110f] = 0x2708e, [0x1110] = 0x27091, [0x1111] = 0x2707d,
  [0x1112] = 0x270ee, [0x1113] = 0x2708d, [0x1114] = 0x2708c, [0x1115] = 0x270b0,
  [0x1116] = 0x27096, [0x1117] = 0x27042, [0x1118] = 0x2707c, [0x1119] = 0x270b1,
  [0x111a] = 0x270b2, [0x111b] = 0x27084, [0x111c] = 0x2709d, [0x111d] = 0x270a1,
  [0x111e] = 0x27098, [0x111f] = 0x270b3, [0x1120] = 0x27022, [0x1121] = 0x2707b,
  [0x1122] = 0x2708a, [0x1123] = 0x270ce, [0x1124] = 0x27080, [0x1125] = 0x27097,
  [0x1126] = 0x270b4, [0x1127] = 0x270b5, [0x1128] = 0x2709a, [0x1129] = 0x2709f,
  [0x112a] = 0x27093, [0x112b] = 0x27212, [0x112c] = 0x2720b, [0x112d] = 0x2720e,
  [0x112e] = 0x274a3, [0x112f] = 0x274cc, [0x1130] = 0x274dc, [0x1131] = 0x274e1,
  [0x1132] = 0x274de, [0x1133] = 0x274d2, [0x1134] = 0x274db, [0x1135] = 0x274d9,
  [0x1136] = 0x274d7, [0x1137] = 0x274d4, [0x1138] = 0x274c9, [0x1139] = 0x274eb,
  [0x113a] = 0x274da, [0x113b] = 0x274d1, [0x113c] = 0x27504, [0x113d] = 0x274ca,
  [0x113e] = 0x274e2, [0x113f] = 0x275d7, [0x1140] = 0x2778c, [0x1141] = 0x27799,
  [0x1142] = 0x277a2, [0x1143] = 0x27796, [0x1144] = 0x27794, [0x1145] = 0x2779f,
  [0x1146] = 0x4667, [0x1147] = 0x2778e, [0x1148] = 0x27803, [0x1149] = 0x27894,
  [0x114a] = 0x27893, [0x114b] = 0x27944, [0x114c] = 0x27b2f, [0x114d] = 0x27b35,
  [0x114e] = 0x27b2b, [0x114f] = 0x27b32, [0x1150] = 0x27b2d, [0x1151] = 0x27b36,
  [0x1152] = 0x4714, [0x1153] = 0x27b31, [0x1154] = 0x27b12, [0x1155] = 0x27b33,
  [0x1156] = 0x27b1f, [0x1157] = 0x27b34, [0x1158] = 0x27b40, [0x1159] = 0x27b3f,
  [0x115a] = 0x27b41, [0x115b] = 0x27bd3, [0x115c] = 0x27c89, [0x115d] = 0x27d18,
  [0x115e] = 0x27d10, [0x115f] = 0x27e1a, [0x1160] = 0x27e25, [0x1161] = 0x27e1e,
  [0x1162] = 0x27f92, [0x1163] = 0x27f95, [0x1164] = 0x27f93, [0x1165] = 0x28184,
  [0x1166] = 0x2819a, [0x1167] = 0x28189, [0x1168] = 0x2818d, [0x1169] = 0x28188,
  [0x116a] = 0x28191, [0x116b] = 0x2819b, [0x116c] = 0x2819c, [0x116d] = 0x29548,
  [0x116e] = 0x2828e, [0x116f] = 0x28292, [0x1170] = 0x283c5, [0x1171] = 0x283c1,
  [0x1172] = 0x283b8, [0x1173] = 0x283be, [0x1174] = 0x283b5, [0x1175] = 0x283c7,
  [0x1176] = 0x2843c, [0x1177] = 0x4889, [0x1178] = 0x285ec, [0x1179] = 0x2863f,
  [0x117a] = 0x28639, [0x117b] = 0x28637, [0x117c] = 0x287a1, [0x117d] = 0x28794,
  [0x117e] = 0x287a0, [0x117f] = 0x287de, [0x1180] = 0x287db, [0x1181] = 0x287df,
  [0x1182] = 0x287dc, [0x1183] = 0x288d6, [0x1184] = 0x288dc, [0x1185] = 0x288dd,
  [0x1186] = 0x288e0, [0x1187] = 0x288e3, [0x1188] = 0x288e1, [0x1189] = 0x28b18,
  [0x118a] = 0x28b19, [0x118b] = 0x28b53, [0x118c] = 0x2f9ec, [0x118d] = 0x28b44,
  [0x118e] = 0x28b0e, [0x118f] = 0x28b0f, [0x1190] = 0x28b47, [0x1191] = 0x28b17,
  [0x1192] = 0x28b1d, [0x1193] = 0x28b11, [0x1194] = 0x28cb4, [0x1195] = 0x28cb6,
  [0x1196] = 0x28db7, [0x1197] = 0x28dbe, [0x1198] = 0x28dc2, [0x1199] = 0x28db4,
  [0x119a] = 0x28f31, [0x119b] = 0x28f15, [0x119c] = 0x28f83, [0x119d] = 0x2903b,
  [0x119e] = 0x29036, [0x119f] = 0x29042, [0x11a0] = 0x29050, [0x11a1] = 0x29040,
  [0x11a2] = 0x29034, [0x11a3] = 0x29038, [0x11a4] = 0x2903d, [0x11a5] = 0x2903e,
  [0x11a6] = 0x29035, [0x11a7] = 0x2903a, [0x11a8] = 0x29046, [0x11a9] = 0x29037,
  [0x11aa] = 0x29039, [0x11ab] = 0x29045, [0x11ac] = 0x29177, [0x11ad] = 0x2915d,
  [0x11ae] = 0x2916a, [0x11af] = 0x29176, [0x11b0] = 0x2916b, [0x11b1] = 0x2916c,
  [0x11b2] = 0x29165, [0x11b3] = 0x29164, [0x11b4] = 0x29171, [0x11b5] = 0x2915f,
  [0x11b6] = 0x29172, [0x11b7] = 0x291fe, [0x11b8] = 0x291ff, [0x11b9] = 0x2923e,
  [0x11ba] = 0x2932b, [0x11bb] = 0x29336, [0x11bc] = 0x2932d, [0x11bd] = 0x29339,
  [0x11be] = 0x2933f, [0x11bf] = 0x2933b, [0x11c0] = 0x29333, [0x11c1] = 0x29342,
  [0x11c2] = 0x2933a, [0x11c3] = 0x293d5, [0x11c4] = 0x293d8, [0x11c5] = 0x293d9,
  [0x11c6] = 0x2940d, [0x11c7] = 0x2940a, [0x11c8] = 0x29439, [0x11c9] = 0x2943a,
  [0x11ca] = 0x2953d, [0x11cb] = 0x29545, [0x11cc] = 0x2953a, [0x11cd] = 0x29537,
  [0x11ce] = 0x2953e, [0x11cf] = 0x29542, [0x11d0] = 0x29787, [0x11d1] = 0x2978c,
  [0x11d2] = 0x29782, [0x11d3] = 0x2976b, [0x11d4] = 0x297a0, [0x11d5] = 0x2979a,
  [0x11d6] = 0x29790, [0x11d7] = 0x2978e, [0x11d8] = 0x297a1, [0x11d9] = 0x297bd,
  [0x11da] = 0x297b2, [0x11db] = 0x297b5, [0x11dc] = 0x297b7, [0x11dd] = 0x297aa,
  [0x11de] = 0x297a2, [0x11df] = 0x297a5, [0x11e0] = 0x297ae, [0x11e1] = 0x297ab,
  [0x11e2] = 0x297bc, [0x11e3] = 0x29832, [0x11e4] = 0x2985a, [0x11e5] = 0x29964,
  [0x11e6] = 0x2995c, [0x11e7] = 0x2994d, [0x11e8] = 0x2993f, [0x11e9] = 0x2993e,
  [0x11ea] = 0x29952, [0x11eb] = 0x29958, [0x11ec] = 0x29957, [0x11ed] = 0x2995e,
  [0x11ee] = 0x29953, [0x11ef] = 0x29954, [0x11f0] = 0x29956, [0x11f1] = 0x28f65,
  [0x11f2] = 0x29a84, [0x11f3] = 0x29a85, [0x11f4] = 0x29a86, [0x11f5] = 0x29b97,
  [0x11f6] = 0x29ba1, [0x11f7] = 0x29ba2, [0x11f8] = 0x4c0b, [0x11f9] = 0x2fa0a,
  [0x11fa] = 0x29ba3, [0x11fb] = 0x29ba6, [0x11fc] = 0x4c17, [0x11fd] = 0x29c15,
  [0x11fe] = 0x29c24, [0x11ff] = 0x29c4a, [0x1200] = 0x29c49, [0x1201] = 0x29c48,
  [0x1202] = 0x29c4b, [0x1203] = 0x29d0e, [0x1204] = 0x29962, [0x1205] = 0x29d0b,
  [0x1206] = 0x29d0a, [0x1207] = 0x29d08, [0x1208] = 0x29d06, [0x1209] = 0x29e43,
  [0x120a] = 0x29e47, [0x120b] = 0x29e3f, [0x120c] = 0x29e46, [0x120d] = 0x29e50,
  [0x120e] = 0x4c6d, [0x120f] = 0x29e4b, [0x1210] = 0x4c70, [0x1211] = 0x29e52,
  [0x1212] = 0x2a0d7, [0x1213] = 0x2a0bf, [0x1214] = 0x2a0d8, [0x1215] = 0x2a0e0,
  [0x1216] = 0x2a0e7, [0x1217] = 0x2a0b8, [0x1218] = 0x2a0d5, [0x1219] = 0x2a0ef,
  [0x121a] = 0x2a0e6, [0x121b] = 0x2a0e4, [0x121c] = 0x2a0d4, [0x121d] = 0x2a0d6,
  [0x121e] = 0x2a0ea, [0x121f] = 0x4ced, [0x1220] = 0x2a0bb, [0x1221] = 0x2a0e9,
  [0x1222] = 0x2fa10, [0x1223] = 0x2a26d, [0x1224] = 0x2a270, [0x1225] = 0x2a273,
  [0x1226] = 0x2a272, [0x1227] = 0x2a2d4, [0x1228] = 0x2a2ce, [0x1229] = 0x2a2d5,
  [0x122a] = 0x2a35a, [0x122b] = 0x2a358, [0x122c] = 0x2a352, [0x122d] = 0x2a350,
  [0x122e] = 0x2a355, [0x122f] = 0x2a34e, [0x1230] = 0x2a34d, [0x1231] = 0x2a3cb,
  [0x1232] = 0x2a3cc, [0x1233] = 0x2a3cd, [0x1234] = 0x2a3d1, [0x1235] = 0x2a458,
  [0x1236] = 0x2a463, [0x1237] = 0x2a45e, [0x1238] = 0x2a454, [0x1239] = 0x2a45b,
  [0x123a] = 0x2a4e9, [0x123b] = 0x2a4e7, [0x123c] = 0x2a4e8, [0x123d] = 0x2a51d,
  [0x123e] = 0x2a567, [0x123f] = 0x2a55a, [0x1240] = 0x2a55c, [0x1241] = 0x2a55b,
  [0x1242] = 0x2a561, [0x1243] = 0x2a5a1, [0x1244] = 0x4d8d, [0x1245] = 0x2a5a4,
  [0x1246] = 0x2a5e9, [0x1247] = 0x2a5ef, [0x1248] = 0x2a5e5, [0x1249] = 0x2a5f5,
  [0x124a] = 0x2a5eb, [0x124b] = 0x2a5ed, [0x124c] = 0x2a696, [0x124d] = 0x2a695,
  [0x124e] = 0x2a6b3, [0x124f] = 0x2a6b5, [0x1250] = 0x2a6b1, [0x1251] = 0x34a7,
  [0x1252] = 0x2044d, [0x1253] = 0x20624, [0x1254] = 0x2082b, [0x1255] = 0x209cf,
  [0x1256] = 0x20ac9, [0x1257] = 0x21136, [0x1258] = 0x21131, [0x1259] = 0x21154,
  [0x125a] = 0x2113a, [0x125b] = 0x21138, [0x125c] = 0x21239, [0x125d] = 0x21238,
  [0x125e] = 0x214f4, [0x125f] = 0x214f3, [0x1260] = 0x214f6, [0x1261] = 0x214fc,
  [0x1262] = 0x214f5, [0x1263] = 0x214f1, [0x1264] = 0x2169c, [0x1265] = 0x21918,
  [0x1266] = 0x2191a, [0x1267] = 0x21afc, [0x1268] = 0x21afe, [0x1269] = 0x21b03,
  [0x126a] = 0x21b53, [0x126b] = 0x21d2b, [0x126c] = 0x21fc6, [0x126d] = 0x21fc1,
  [0x126e] = 0x2217b, [0x126f] = 0x22306, [0x1270] = 0x22449, [0x1271] = 0x224b8,
  [0x1272] = 0x225a3, [0x1273] = 0x22952, [0x1274] = 0x2294d, [0x1275] = 0x2295f,
  [0x1276] = 0x2295e, [0x1277] = 0x22961, [0x1278] = 0x22a23, [0x1279] = 0x22e8c,
  [0x127a] = 0x22e6f, [0x127b] = 0x22e75, [0x127c] = 0x22e74, [0x127d] = 0x22e71,
  [0x127e] = 0x22e70, [0x127f] = 0x22e6c, [0x1280] = 0x23126, [0x1281] = 0x23308,
  [0x1282] = 0x6707, [0x1283] = 0x237f0, [0x1284] = 0x237ef, [0x1285] = 0x237fb,
  [0x1286] = 0x23810, [0x1287] = 0x2380c, [0x1288] = 0x237f6, [0x1289] = 0x237fe,
  [0x128a] = 0x23a7c, [0x128b] = 0x23ae1, [0x128c] = 0x23c38, [0x128d] = 0x23c6f,
  [0x128e] = 0x24118, [0x128f] = 0x24115, [0x1290] = 0x2411c, [0x1291] = 0x24110,
  [0x1292] = 0x24135, [0x1293] = 0x2f917, [0x1294] = 0x24117, [0x1295] = 0x2411d,
  [0x1296] = 0x24126, [0x1297] = 0x24128, [0x1298] = 0x24129, [0x1299] = 0x2412a,
  [0x129a] = 0x2411a, [0x129b] = 0x2f916, [0x129c] = 0x22307, [0x129d] = 0x244ab,
  [0x129e] = 0x244ac, [0x129f] = 0x244a1, [0x12a0] = 0x244ae, [0x12a1] = 0x244a3,
  [0x12a2] = 0x244a7, [0x12a3] = 0x24548, [0x12a4] = 0x24604, [0x12a5] = 0x246fd,
  [0x12a6] = 0x3e5a, [0x12a7] = 0x24708, [0x12a8] = 0x248d2, [0x12a9] = 0x24ac6,
  [0x12aa] = 0x24abe, [0x12ab] = 0x24b1c, [0x12ac] = 0x24ba6, [0x12ad] = 0x24ba7,
  [0x12ae] = 0x24bab, [0x12af] = 0x24c00, [0x12b0] = 0x24cfb, [0x12b1] = 0x24cfc,
  [0x12b2] = 0x24cf9, [0x12b3] = 0x24d01, [0x12b4] = 0x24f06, [0x12b5] = 0x24f04,
  [0x12b6] = 0x2f93c, [0x12b7] = 0x25034, [0x12b8] = 0x25336, [0x12b9] = 0x25332,
  [0x12ba] = 0x25342, [0x12bb] = 0x2532d, [0x12bc] = 0x25335, [0x12bd] = 0x255d8,
  [0x12be] = 0x25723, [0x12bf] = 0x25720, [0x12c0] = 0x25901, [0x12c1] = 0x25900,
  [0x12c2] = 0x25903, [0x12c3] = 0x2a6bc, [0x12c4] = 0x25a3c, [0x12c5] = 0x25a41,
  [0x12c6] = 0x25a35, [0x12c7] = 0x25ac9, [0x12c8] = 0x25d8e, [0x12c9] = 0x25d9e,
  [0x12ca] = 0x25d99, [0x12cb] = 0x25db4, [0x12cc] = 0x25daa, [0x12cd] = 0x25d9f,
  [0x12ce] = 0x25d96, [0x12cf] = 0x25d9d, [0x12d0] = 0x25dc3, [0x12d1] = 0x25d74,
  [0x12d2] = 0x25dab, [0x12d3] = 0x42af, [0x12d4] = 0x25f63, [0x12d5] = 0x25f5b,
  [0x12d6] = 0x25f5a, [0x12d7] = 0x261c5, [0x12d8] = 0x261c4, [0x12d9] = 0x261cf,
  [0x12da] = 0x261c8, [0x12db] = 0x261a4, [0x12dc] = 0x261bd, [0x12dd] = 0x261d3,
  [0x12de] = 0x26260, [0x12df] = 0x2633b, [0x12e0] = 0x263e3, [0x12e1] = 0x263e7,
  [0x12e2] = 0x264a8, [0x12e3] = 0x264ac, [0x12e4] = 0x264a9, [0x12e5] = 0x2652a,
  [0x12e6] = 0x26608, [0x12e7] = 0x26609, [0x12e8] = 0x268b9, [0x12e9] = 0x26ac1,
  [0x12ea] = 0x26ac2, [0x12eb] = 0x26ab8, [0x12ec] = 0x270e1, [0x12ed] = 0x270eb,
  [0x12ee] = 0x270e5, [0x12ef] = 0x270fa, [0x12f0] = 0x270e4, [0x12f1] = 0x2710b,
  [0x12f2] = 0x270d7, [0x12f3] = 0x270ef, [0x12f4] = 0x270e0, [0x12f5] = 0x270ec,
  [0x12f6] = 0x270fb, [0x12f7] = 0x2f9b0, [0x12f8] = 0x270d3, [0x12f9] = 0x270e6,
  [0x12fa] = 0x270fc, [0x12fb] = 0x270e3, [0x12fc] = 0x270cf, [0x12fd] = 0x270da,
  [0x12fe] = 0x270dc, [0x12ff] = 0x270d2, [0x1300] = 0x270a4, [0x1301] = 0x456b,
  [0x1302] = 0x27217, [0x1303] = 0x27216, [0x1304] = 0x274f2, [0x1305] = 0x274fc,
  [0x1306] = 0x27518, [0x1307] = 0x274f6, [0x1308] = 0x274fe, [0x1309] = 0x274f3,
  [0x130a] = 0x274f7, [0x130b] = 0x27501, [0x130c] = 0x274f9, [0x130d] = 0x27506,
  [0x130e] = 0x274f5, [0x130f] = 0x27510, [0x1310] = 0x274df, [0x1311] = 0x27503,
  [0x1312] = 0x27508, [0x1313] = 0x275d8, [0x1314] = 0x27605, [0x1315] = 0x27797,
  [0x1316] = 0x277b3, [0x1317] = 0x277ae, [0x1318] = 0x277af, [0x1319] = 0x277a7,
  [0x131a] = 0x277b1, [0x131b] = 0x277a8, [0x131c] = 0x277ac, [0x131d] = 0x277ab,
  [0x131e] = 0x27804, [0x131f] = 0x2789a, [0x1320] = 0x2794a, [0x1321] = 0x27b42,
  [0x1322] = 0x27b58, [0x1323] = 0x27b4b, [0x1324] = 0x27b45, [0x1325] = 0x27b49,
  [0x1326] = 0x27b4c, [0x1327] = 0x27b59, [0x1328] = 0x27b56, [0x1329] = 0x471b,
  [0x132a] = 0x27b46, [0x132b] = 0x27b44, [0x132c] = 0x27b5b, [0x132d] = 0x27b69,
  [0x132e] = 0x27c8e, [0x132f] = 0x27c8f, [0x1330] = 0x27d1e, [0x1331] = 0x268bc,
  [0x1332] = 0x27e2f, [0x1333] = 0x27f9e, [0x1334] = 0x27f9d, [0x1335] = 0x27f9f,
  [0x1336] = 0x27f9c, [0x1337] = 0x281b4, [0x1338] = 0x281ae, [0x1339] = 0x281ab,
  [0x133a] = 0x281b3, [0x133b] = 0x281af, [0x133c] = 0x281c2, [0x133d] = 0x28293,
  [0x133e] = 0x28295, [0x133f] = 0x28296, [0x1340] = 0x28297, [0x1341] = 0x283cf,
  [0x1342] = 0x283ce, [0x1343] = 0x283cb, [0x1344] = 0x2844b, [0x1345] = 0x28646,
  [0x1346] = 0x28643, [0x1347] = 0x28645, [0x1348] = 0x28651, [0x1349] = 0x287ae,
  [0x134a] = 0x287af, [0x134b] = 0x287b0, [0x134c] = 0x287b8, [0x134d] = 0x287e2,
  [0x134e] = 0x287e3, [0x134f] = 0x287e6, [0x1350] = 0x288ed, [0x1351] = 0x288ea,
  [0x1352] = 0x2893a, [0x1353] = 0x28b59, [0x1354] = 0x28b84, [0x1355] = 0x28b5f,
  [0x1356] = 0x28b7c, [0x1357] = 0x28b5c, [0x1358] = 0x28b58, [0x1359] = 0x28b55,
  [0x135a] = 0x28b5d, [0x135b] = 0x28b7e, [0x135c] = 0x28b80, [0x135d] = 0x28b83,
  [0x135e] = 0x28b57, [0x135f] = 0x4963, [0x1360] = 0x28b5e, [0x1361] = 0x28cba,
  [0x1362] = 0x28dd5, [0x1363] = 0x28f58, [0x1364] = 0x28f68, [0x1365] = 0x28f67,
  [0x1366] = 0x2904a, [0x1367] = 0x2904c, [0x1368] = 0x29052, [0x1369] = 0x29049,
  [0x136a] = 0x2904e, [0x136b] = 0x29047, [0x136c] = 0x2904d, [0x136d] = 0x2904b,
  [0x136e] = 0x2904f, [0x136f] = 0x2917e, [0x1370] = 0x29187, [0x1371] = 0x29183,
  [0x1372] = 0x29189, [0x1373] = 0x24aca, [0x1374] = 0x29186, [0x1375] = 0x29188,
  [0x1376] = 0x29247, [0x1377] = 0x29242, [0x1378] = 0x29249, [0x1379] = 0x29248,
  [0x137a] = 0x4a80, [0x137b] = 0x4a84, [0x137c] = 0x4a7f, [0x137d] = 0x29344,
  [0x137e] = 0x29351, [0x137f] = 0x29346, [0x1380] = 0x29347, [0x1381] = 0x293e4,
  [0x1382] = 0x2940f, [0x1383] = 0x2943f, [0x1384] = 0x2954b, [0x1385] = 0x29557,
  [0x1386] = 0x29552, [0x1387] = 0x4af1, [0x1388] = 0x29551, [0x1389] = 0x29558,
  [0x138a] = 0x2955e, [0x138b] = 0x29553, [0x138c] = 0x2955d, [0x138d] = 0x2954d,
  [0x138e] = 0x2963c, [0x138f] = 0x2963f, [0x1390] = 0x29646, [0x1391] = 0x2963e,
  [0x1392] = 0x29644, [0x1393] = 0x29645, [0x1394] = 0x29641, [0x1395] = 0x29638,
  [0x1396] = 0x29642, [0x1397] = 0x29643, [0x1398] = 0x2967a, [0x1399] = 0x297a3,
  [0x139a] = 0x297ba, [0x139b] = 0x297c0, [0x139c] = 0x297c4, [0x139d] = 0x297c6,
  [0x139e] = 0x297cb, [0x139f] = 0x29861, [0x13a0] = 0x2997a, [0x13a1] = 0x29973,
  [0x13a2] = 0x29972, [0x13a3] = 0x29974, [0x13a4] = 0x29980, [0x13a5] = 0x29981,
  [0x13a6] = 0x9a47, [0x13a7] = 0x29a95, [0x13a8] = 0x29a8f, [0x13a9] = 0x29a90,
  [0x13aa] = 0x29a92, [0x13ab] = 0x29a94, [0x13ac] = 0x29a8b, [0x13ad] = 0x29ae6,
  [0x13ae] = 0x29bb2, [0x13af] = 0x29bb8, [0x13b0] = 0x29bbd, [0x13b1] = 0x29bbe,
  [0x13b2] = 0x29bce, [0x13b3] = 0x29bba, [0x13b4] = 0x29c16, [0x13b5] = 0x29c26,
  [0x13b6] = 0x29c25, [0x13b7] = 0x29c4c, [0x13b8] = 0x29c50, [0x13b9] = 0x29c4e,
  [0x13ba] = 0x29c51, [0x13bb] = 0x29c52, [0x13bc] = 0x29d14, [0x13bd] = 0x29d15,
  [0x13be] = 0x29d1b, [0x13bf] = 0x29e82, [0x13c0] = 0x29e99, [0x13c1] = 0x29e9a,
  [0x13c2] = 0x29e7d, [0x13c3] = 0x29e85, [0x13c4] = 0x29e86, [0x13c5] = 0x29e9c,
  [0x13c6] = 0x29e79, [0x13c7] = 0x29e7b, [0x13c8] = 0x29e80, [0x13c9] = 0x29e83,
  [0x13ca] = 0x29e81, [0x13cb] = 0x2a11e, [0x13cc] = 0x2a11b, [0x13cd] = 0x2a12a,
  [0x13ce] = 0x2a0fb, [0x13cf] = 0x2a105, [0x13d0] = 0x2a120, [0x13d1] = 0x2a111,
  [0x13d2] = 0x2a104, [0x13d3] = 0x2a0fd, [0x13d4] = 0x2a103, [0x13d5] = 0x2a110,
  [0x13d6] = 0x2a118, [0x13d7] = 0x2a10a, [0x13d8] = 0x2a14e, [0x13d9] = 0x2a109,
  [0x13da] = 0x2a107, [0x13db] = 0x2a11c, [0x13dc] = 0x4d1c, [0x13dd] = 0x2a277,
  [0x13de] = 0x2a276, [0x13df] = 0x2a2d8, [0x13e0] = 0x2a2d9, [0x13e1] = 0x2a361,
  [0x13e2] = 0x2a35c, [0x13e3] = 0x2a35e, [0x13e4] = 0x2a360, [0x13e5] = 0x2a3aa,
  [0x13e6] = 0x2a3d6, [0x13e7] = 0x2a3d8, [0x13e8] = 0x2a409, [0x13e9] = 0x2a408,
  [0x13ea] = 0x2a46b, [0x13eb] = 0x2a465, [0x13ec] = 0x2a473, [0x13ed] = 0x2a474,
  [0x13ee] = 0x2a4ed, [0x13ef] = 0x2a524, [0x13f0] = 0x2a525, [0x13f1] = 0x2a56a,
  [0x13f2] = 0x2a5a7, [0x13f3] = 0x2a5a8, [0x13f4] = 0x2a60b, [0x13f5] = 0x2a5fa,
  [0x13f6] = 0x2a5f9, [0x13f7] = 0x2a5ff, [0x13f8] = 0x2a604, [0x13f9] = 0x2a5f6,
  [0x13fa] = 0x2a605, [0x13fb] = 0x2a699, [0x13fc] = 0x2a6ab, [0x13fd] = 0x2a6bd,
  [0x13fe] = 0x2a6b8, [0x13ff] = 0x34a8, [0x1400] = 0x20464, [0x1401] = 0x2045a,
  [0x1402] = 0x20572, [0x1403] = 0x2097a, [0x1404] = 0x20b95, [0x1405] = 0x2115f,
  [0x1406] = 0x21161, [0x1407] = 0x2115e, [0x1408] = 0x2115a, [0x1409] = 0x2115c,
  [0x140a] = 0x2115d, [0x140b] = 0x2116e, [0x140c] = 0x21157, [0x140d] = 0x21158,
  [0x140e] = 0x21502, [0x140f] = 0x2150b, [0x1410] = 0x21508, [0x1411] = 0x21551,
  [0x1412] = 0x21925, [0x1413] = 0x21928, [0x1414] = 0x21923, [0x1415] = 0x21921,
  [0x1416] = 0x21924, [0x1417] = 0x21929, [0x1418] = 0x21b06, [0x1419] = 0x21b0a,
  [0x141a] = 0x21fca, [0x141b] = 0x21fcc, [0x141c] = 0x21fcb, [0x141d] = 0x21fcf,
  [0x141e] = 0x2217f, [0x141f] = 0x2217c, [0x1420] = 0x221d6, [0x1421] = 0x22309,
  [0x1422] = 0x2230b, [0x1423] = 0x22376, [0x1424] = 0x2247d, [0x1425] = 0x225a5,
  [0x1426] = 0x2295b, [0x1427] = 0x2295c, [0x1428] = 0x2295d, [0x1429] = 0x22965,
  [0x142a] = 0x22e6d, [0x142b] = 0x22e82, [0x142c] = 0x22e83, [0x142d] = 0x22e87,
  [0x142e] = 0x22e95, [0x142f] = 0x22e8a, [0x1430] = 0x230d6, [0x1431] = 0x23128,
  [0x1432] = 0x2330d, [0x1433] = 0x2382e, [0x1434] = 0x2f8ee, [0x1435] = 0x2382d,
  [0x1436] = 0x23821, [0x1437] = 0x23819, [0x1438] = 0x2393b, [0x1439] = 0x2393c,
  [0x143a] = 0x2393a, [0x143b] = 0x23a7e, [0x143c] = 0x23c3b, [0x143d] = 0x24147,
  [0x143e] = 0x24139, [0x143f] = 0x24134, [0x1440] = 0x24136, [0x1441] = 0x24146,
  [0x1442] = 0x2413b, [0x1443] = 0x24141, [0x1444] = 0x24145, [0x1445] = 0x244c0,
  [0x1446] = 0x244c4, [0x1447] = 0x244ba, [0x1448] = 0x244bd, [0x1449] = 0x244be,
  [0x144a] = 0x245a3, [0x144b] = 0x2470c, [0x144c] = 0x248d9, [0x144d] = 0x24acd,
  [0x144e] = 0x24bae, [0x144f] = 0x24cff, [0x1450] = 0x24d24, [0x1451] = 0x24fbb,
  [0x1452] = 0x24fbc, [0x1453] = 0x25036, [0x1454] = 0x250d5, [0x1455] = 0x250d3,
  [0x1456] = 0x250d6, [0x1457] = 0x2533c, [0x1458] = 0x2533f, [0x1459] = 0x255de,
  [0x145a] = 0x255e3, [0x145b] = 0x25727, [0x145c] = 0x25729, [0x145d] = 0x418b,
  [0x145e] = 0x25905, [0x145f] = 0x25907, [0x1460] = 0x2590c, [0x1461] = 0x25ad0,
  [0x1462] = 0x25dbe, [0x1463] = 0x25dbc, [0x1464] = 0x25dd0, [0x1465] = 0x25dc2,
  [0x1466] = 0x25db5, [0x1467] = 0x25dc9, [0x1468] = 0x25f66, [0x1469] = 0x432b,
  [0x146a] = 0x261d5, [0x146b] = 0x261e2, [0x146c] = 0x261dc, [0x146d] = 0x261e3,
  [0x146e] = 0x261da, [0x146f] = 0x261c2, [0x1470] = 0x261e8, [0x1471] = 0x263e9,
  [0x1472] = 0x264af, [0x1473] = 0x264ad, [0x1474] = 0x264ae, [0x1475] = 0x2660b,
  [0x1476] = 0x268c1, [0x1477] = 0x269b6, [0x1478] = 0x269b9, [0x1479] = 0x26ac0,
  [0x147a] = 0x270a5, [0x147b] = 0x27128, [0x147c] = 0x27122, [0x147d] = 0x27129,
  [0x147e] = 0x27118, [0x147f] = 0x2711f, [0x1480] = 0x2711c, [0x1481] = 0x27112,
  [0x1482] = 0x2712a, [0x1483] = 0x457a, [0x1484] = 0x27121, [0x1485] = 0x2712b,
  [0x1486] = 0x27117, [0x1487] = 0x270f0, [0x1488] = 0x27116, [0x1489] = 0x27123,
  [0x148a] = 0x2752b, [0x148b] = 0x27526, [0x148c] = 0x2753d, [0x148d] = 0x27522,
  [0x148e] = 0x2753a, [0x148f] = 0x27531, [0x1490] = 0x27532, [0x1491] = 0x27554,
  [0x1492] = 0x27521, [0x1493] = 0x27535, [0x1494] = 0x4609, [0x1495] = 0x2752e,
  [0x1496] = 0x27530, [0x1497] = 0x2752f, [0x1498] = 0x27536, [0x1499] = 0x275da,
  [0x149a] = 0x275d9, [0x149b] = 0x277bb, [0x149c] = 0x277bc, [0x149d] = 0x277b7,
  [0x149e] = 0x277c2, [0x149f] = 0x277bd, [0x14a0] = 0x277b2, [0x14a1] = 0x466d,
  [0x14a2] = 0x25344, [0x14a3] = 0x25dd1, [0x14a4] = 0x27b52, [0x14a5] = 0x27b6b,
  [0x14a6] = 0x27b67, [0x14a7] = 0x471f, [0x14a8] = 0x27b61, [0x14a9] = 0x27b6c,
  [0x14aa] = 0x27b51, [0x14ab] = 0x27b74, [0x14ac] = 0x27b77, [0x14ad] = 0x27b6f,
  [0x14ae] = 0x27b6d, [0x14af] = 0x27b68, [0x14b0] = 0x2f9d1, [0x14b1] = 0x27b84,
  [0x14b2] = 0x27c90, [0x14b3] = 0x27c92, [0x14b4] = 0x27c93, [0x14b5] = 0x27d1f,
  [0x14b6] = 0x27e31, [0x14b7] = 0x27e38, [0x14b8] = 0x27e39, [0x14b9] = 0x27e37,
  [0x14ba] = 0x27fab, [0x14bb] = 0x281c3, [0x14bc] = 0x281c8, [0x14bd] = 0x281cb,
  [0x14be] = 0x281cf, [0x14bf] = 0x28298, [0x14c0] = 0x283d4, [0x14c1] = 0x283d3,
  [0x14c2] = 0x283d8, [0x14c3] = 0x283d9, [0x14c4] = 0x283dd, [0x14c5] = 0x283d1,
  [0x14c6] = 0x283d6, [0x14c7] = 0x2843e, [0x14c8] = 0x28658, [0x14c9] = 0x28657,
  [0x14ca] = 0x28655, [0x14cb] = 0x287c4, [0x14cc] = 0x287e4, [0x14cd] = 0x288ee,
  [0x14ce] = 0x288ef, [0x14cf] = 0x288f3, [0x14d0] = 0x288f2, [0x14d1] = 0x288f0,
  [0x14d2] = 0x28bab, [0x14d3] = 0x28b9a, [0x14d4] = 0x28baf, [0x14d5] = 0x28b97,
  [0x14d6] = 0x496a, [0x14d7] = 0x496c, [0x14d8] = 0x28bbf, [0x14d9] = 0x28b94,
  [0x14da] = 0x28b93, [0x14db] = 0x28cbe, [0x14dc] = 0x28cbb, [0x14dd] = 0x28cbc,
  [0x14de] = 0x28dd9, [0x14df] = 0x28f6c, [0x14e0] = 0x29053, [0x14e1] = 0x29054,
  [0x14e2] = 0x2905b, [0x14e3] = 0x29058, [0x14e4] = 0x29056, [0x14e5] = 0x29057,
  [0x14e6] = 0x2919f, [0x14e7] = 0x29194, [0x14e8] = 0x29196, [0x14e9] = 0x29197,
  [0x14ea] = 0x2924a, [0x14eb] = 0x2924b, [0x14ec] = 0x29355, [0x14ed] = 0x2935a,
  [0x14ee] = 0x2935e, [0x14ef] = 0x2935f, [0x14f0] = 0x29359, [0x14f1] = 0x2935b,
  [0x14f2] = 0x29358, [0x14f3] = 0x29354, [0x14f4] = 0x293e8, [0x14f5] = 0x293eb,
  [0x14f6] = 0x293ec, [0x14f7] = 0x29413, [0x14f8] = 0x29566, [0x14f9] = 0x2956b,
  [0x14fa] = 0x29562, [0x14fb] = 0x29569, [0x14fc] = 0x29563, [0x14fd] = 0x2955f,
  [0x14fe] = 0x2954e, [0x14ff] = 0x29648, [0x1500] = 0x2964a, [0x1501] = 0x297e3,
  [0x1502] = 0x297db, [0x1503] = 0x297d8, [0x1504] = 0x297d6, [0x1505] = 0x29986,
  [0x1506] = 0x29990, [0x1507] = 0x29991, [0x1508] = 0x29988, [0x1509] = 0x29994,
  [0x150a] = 0x29983, [0x150b] = 0x4bb2, [0x150c] = 0x2999c, [0x150d] = 0x2998d,
  [0x150e] = 0x29985, [0x150f] = 0x29a98, [0x1510] = 0x29a9a, [0x1511] = 0x29a9c,
  [0x1512] = 0x29ae7, [0x1513] = 0x29bc5, [0x1514] = 0x29bd0, [0x1515] = 0x29bd1,
  [0x1516] = 0x29c19, [0x1517] = 0x29c27, [0x1518] = 0x4c1d, [0x1519] = 0x29c54,
  [0x151a] = 0x29d2d, [0x151b] = 0x29d22, [0x151c] = 0x4c2d, [0x151d] = 0x29d1f,
  [0x151e] = 0x29ebd, [0x151f] = 0x29eae, [0x1520] = 0x29ebb, [0x1521] = 0x29ead,
  [0x1522] = 0x29ebc, [0x1523] = 0x29eb9, [0x1524] = 0x29eb4, [0x1525] = 0x29ecb,
  [0x1526] = 0x29eb7, [0x1527] = 0x29eb3, [0x1528] = 0x29eba, [0x1529] = 0x29eb6,
  [0x152a] = 0x29ecd, [0x152b] = 0x29ebe, [0x152c] = 0x29ec9, [0x152d] = 0x2fa0b,
  [0x152e] = 0x2a15f, [0x152f] = 0x2a13b, [0x1530] = 0x2a161, [0x1531] = 0x2a15c,
  [0x1532] = 0x2a18a, [0x1533] = 0x2a15a, [0x1534] = 0x2a14d, [0x1535] = 0x2a146,
  [0x1536] = 0x2a144, [0x1537] = 0x2a13d, [0x1538] = 0x2a140, [0x1539] = 0x2a13c,
  [0x153a] = 0x2a18c, [0x153b] = 0x2a141, [0x153c] = 0x2a14c, [0x153d] = 0x2a13e,
  [0x153e] = 0x2a14a, [0x153f] = 0x2a27c, [0x1540] = 0x2a27a, [0x1541] = 0x2a365,
  [0x1542] = 0x2a36e, [0x1543] = 0x2a369, [0x1544] = 0x2a36a, [0x1545] = 0x2a36f,
  [0x1546] = 0x2a36c, [0x1547] = 0x2a370, [0x1548] = 0x2a368, [0x1549] = 0x2a36b,
  [0x154a] = 0x4d45, [0x154b] = 0x2a3ac, [0x154c] = 0x2a3de, [0x154d] = 0x2a3dd,
  [0x154e] = 0x2a3dc, [0x154f] = 0x244c5, [0x1550] = 0x2a48c, [0x1551] = 0x2a483,
  [0x1552] = 0x2a482, [0x1553] = 0x2a488, [0x1554] = 0x2a485, [0x1555] = 0x2a481,
  [0x1556] = 0x2a4f5, [0x1557] = 0x2a4ef, [0x1558] = 0x2a4f4, [0x1559] = 0x2a4f2,
  [0x155a] = 0x2a4f6, [0x155b] = 0x2a4f3, [0x155c] = 0x2a4f0, [0x155d] = 0x2a4f1,
  [0x155e] = 0x2a52b, [0x155f] = 0x2a527, [0x1560] = 0x2a528, [0x1561] = 0x2a56c,
  [0x1562] = 0x2a5ae, [0x1563] = 0x2a60e, [0x1564] = 0x2a61b, [0x1565] = 0x2a616,
  [0x1566] = 0x2a61f, [0x1567] = 0x2a622, [0x1568] = 0x2a620, [0x1569] = 0x2a621,
  [0x156a] = 0x2a614, [0x156b] = 0x2a613, [0x156c] = 0x2a69d, [0x156d] = 0x2a69c,
  [0x156e] = 0x2a69e, [0x156f] = 0x2a69f, [0x1570] = 0x2046b, [0x1571] = 0x208c8,
  [0x1572] = 0x21178, [0x1573] = 0x21176, [0x1574] = 0x21170, [0x1575] = 0x21171,
  [0x1576] = 0x2150a, [0x1577] = 0x2192c, [0x1578] = 0x22311, [0x1579] = 0x2247e,
  [0x157a] = 0x22957, [0x157b] = 0x22966, [0x157c] = 0x22969, [0x157d] = 0x22967,
  [0x157e] = 0x22968, [0x157f] = 0x22971, [0x1580] = 0x2296f, [0x1581] = 0x2a3ae,
  [0x1582] = 0x22e8b, [0x1583] = 0x22e96, [0x1584] = 0x23035, [0x1585] = 0x2303a,
  [0x1586] = 0x2306b, [0x1587] = 0x23316, [0x1588] = 0x23843, [0x1589] = 0x23846,
  [0x158a] = 0x2383f, [0x158b] = 0x2383b, [0x158c] = 0x2383d, [0x158d] = 0x239b1,
  [0x158e] = 0x239b2, [0x158f] = 0x23ae4, [0x1590] = 0x23c40, [0x1591] = 0x2415d,
  [0x1592] = 0x24151, [0x1593] = 0x2414d, [0x1594] = 0x2414c, [0x1595] = 0x2415b,
  [0x1596] = 0x244d4, [0x1597] = 0x244d2, [0x1598] = 0x244ca, [0x1599] = 0x244c8,
  [0x159a] = 0x244d0, [0x159b] = 0x244c9, [0x159c] = 0x24549, [0x159d] = 0x245a4,
  [0x159e] = 0x24712, [0x159f] = 0x2470f, [0x15a0] = 0x24711, [0x15a1] = 0x248db,
  [0x15a2] = 0x248dd, [0x15a3] = 0x248dc, [0x15a4] = 0x24ad4, [0x15a5] = 0x24b21,
  [0x15a6] = 0x24bb2, [0x15a7] = 0x24d04, [0x15a8] = 0x24d02, [0x15a9] = 0x24f11,
  [0x15aa] = 0x24f16, [0x15ab] = 0x24f10, [0x15ac] = 0x24fbe, [0x15ad] = 0x2721a,
  [0x15ae] = 0x2534c, [0x15af] = 0x2534a, [0x15b0] = 0x255f2, [0x15b1] = 0x255f1,
  [0x15b2] = 0x2572a, [0x15b3] = 0x2572c, [0x15b4] = 0x2572f, [0x15b5] = 0x25731,
  [0x15b6] = 0x25911, [0x15b7] = 0x25912, [0x15b8] = 0x25a4b, [0x15b9] = 0x25ddb,
  [0x15ba] = 0x25dd6, [0x15bb] = 0x25ddd, [0x15bc] = 0x25de2, [0x15bd] = 0x25de4,
  [0x15be] = 0x25de0, [0x15bf] = 0x25ddf, [0x15c0] = 0x25de3, [0x15c1] = 0x25f6e,
  [0x15c2] = 0x25f71, [0x15c3] = 0x261e9, [0x15c4] = 0x261ea, [0x15c5] = 0x26263,
  [0x15c6] = 0x263eb, [0x15c7] = 0x263ea, [0x15c8] = 0x268bd, [0x15c9] = 0x268bb,
  [0x15ca] = 0x268c6, [0x15cb] = 0x268cc, [0x15cc] = 0x268c8, [0x15cd] = 0x244cf,
  [0x15ce] = 0x268c9, [0x15cf] = 0x268ca, [0x15d0] = 0x268cf, [0x15d1] = 0x268d0,
  [0x15d2] = 0x269ba, [0x15d3] = 0x269fb, [0x15d4] = 0x26a03, [0x15d5] = 0x26ac4,
  [0x15d6] = 0x27149, [0x15d7] = 0x27153, [0x15d8] = 0x27136, [0x15d9] = 0x2714a,
  [0x15da] = 0x27141, [0x15db] = 0x2714e, [0x15dc] = 0x27119, [0x15dd] = 0x2714d,
  [0x15de] = 0x27145, [0x15df] = 0x2714c, [0x15e0] = 0x27147, [0x15e1] = 0x27148,
  [0x15e2] = 0x2714f, [0x15e3] = 0x27137, [0x15e4] = 0x27142, [0x15e5] = 0x2754a,
  [0x15e6] = 0x27546, [0x15e7] = 0x460a, [0x15e8] = 0x27549, [0x15e9] = 0x2754f,
  [0x15ea] = 0x27551, [0x15eb] = 0x2754c, [0x15ec] = 0x460c, [0x15ed] = 0x27606,
  [0x15ee] = 0x27951, [0x15ef] = 0x27b7a, [0x15f0] = 0x27b83, [0x15f1] = 0x27b7e,
  [0x15f2] = 0x27b7f, [0x15f3] = 0x27b80, [0x15f4] = 0x24bb5, [0x15f5] = 0x27c91,
  [0x15f6] = 0x27c94, [0x15f7] = 0x27c95, [0x15f8] = 0x27d21, [0x15f9] = 0x27d20,
  [0x15fa] = 0x27e3d, [0x15fb] = 0x27e40, [0x15fc] = 0x27e46, [0x15fd] = 0x27e84,
  [0x15fe] = 0x27fac, [0x15ff] = 0x27fad, [0x1600] = 0x482e, [0x1601] = 0x281da,
  [0x1602] = 0x281d9, [0x1603] = 0x283e2, [0x1604] = 0x283e1, [0x1605] = 0x283e3,
  [0x1606] = 0x283e4, [0x1607] = 0x283de, [0x1608] = 0x283df, [0x1609] = 0x28641,
  [0x160a] = 0x28659, [0x160b] = 0x2865c, [0x160c] = 0x2865a, [0x160d] = 0x287e8,
  [0x160e] = 0x288f6, [0x160f] = 0x288fc, [0x1610] = 0x288f7, [0x1611] = 0x288fa,
  [0x1612] = 0x288f9, [0x1613] = 0x28bc4, [0x1614] = 0x28bbe, [0x1615] = 0x28bd2,
  [0x1616] = 0x28bbd, [0x1617] = 0x28b95, [0x1618] = 0x28bd4, [0x1619] = 0x28de4,
  [0x161a] = 0x29061, [0x161b] = 0x29062, [0x161c] = 0x29063, [0x161d] = 0x29064,
  [0x161e] = 0x29060, [0x161f] = 0x2905c, [0x1620] = 0x2905d, [0x1621] = 0x2905e,
  [0x1622] = 0x291a5, [0x1623] = 0x291a6, [0x1624] = 0x2924c, [0x1625] = 0x29368,
  [0x1626] = 0x2936e, [0x1627] = 0x29371, [0x1628] = 0x2936b, [0x1629] = 0x2936f,
  [0x162a] = 0x293ee, [0x162b] = 0x293f1, [0x162c] = 0x293f0, [0x162d] = 0x293ef,
  [0x162e] = 0x29415, [0x162f] = 0x29414, [0x1630] = 0x28f6e, [0x1631] = 0x29447,
  [0x1632] = 0x2957c, [0x1633] = 0x2957a, [0x1634] = 0x29574, [0x1635] = 0x29576,
  [0x1636] = 0x2956e, [0x1637] = 0x29578, [0x1638] = 0x2956d, [0x1639] = 0x2956c,
  [0x163a] = 0x2964e, [0x163b] = 0x297d7, [0x163c] = 0x297ea, [0x163d] = 0x297e5,
  [0x163e] = 0x29864, [0x163f] = 0x299b3, [0x1640] = 0x299a3, [0x1641] = 0x299a5,
  [0x1642] = 0x299a7, [0x1643] = 0x299a2, [0x1644] = 0x2999f, [0x1645] = 0x299a6,
  [0x1646] = 0x2999e, [0x1647] = 0x299a8, [0x1648] = 0x29aa9, [0x1649] = 0x29aa6,
  [0x164a] = 0x29aaa, [0x164b] = 0x29aab, [0x164c] = 0x29aa0, [0x164d] = 0x29aa1,
  [0x164e] = 0x29aa8, [0x164f] = 0x29ae8, [0x1650] = 0x29ae9, [0x1651] = 0x29aea,
  [0x1652] = 0x29be4, [0x1653] = 0x29bdf, [0x1654] = 0x29be0, [0x1655] = 0x29c28,
  [0x1656] = 0x29c5d, [0x1657] = 0x29c5b, [0x1658] = 0x29c56, [0x1659] = 0x29c57,
  [0x165a] = 0x29c5f, [0x165b] = 0x29c62, [0x165c] = 0x29ee1, [0x165d] = 0x29ee3,
  [0x165e] = 0x29ede, [0x165f] = 0x29ed9, [0x1660] = 0x29ee8, [0x1661] = 0x29ef2,
  [0x1662] = 0x29ef6, [0x1663] = 0x29ee6, [0x1664] = 0x29ef4, [0x1665] = 0x29ef5,
  [0x1666] = 0x29ee5, [0x1667] = 0x29ee2, [0x1668] = 0x4c8d, [0x1669] = 0x2a196,
  [0x166a] = 0x2a1aa, [0x166b] = 0x2a197, [0x166c] = 0x2a170, [0x166d] = 0x2a1a1,
  [0x166e] = 0x2a19d, [0x166f] = 0x2a1a9, [0x1670] = 0x2a16f, [0x1671] = 0x2a17e,
  [0x1672] = 0x2a194, [0x1673] = 0x2a19a, [0x1674] = 0x2a173, [0x1675] = 0x2a187,
  [0x1676] = 0x2a171, [0x1677] = 0x2a177, [0x1678] = 0x2a188, [0x1679] = 0x2a18d,
  [0x167a] = 0x2a185, [0x167b] = 0x2a178, [0x167c] = 0x2a1ad, [0x167d] = 0x2a280,
  [0x167e] = 0x2a281, [0x167f] = 0x2a2e5, [0x1680] = 0x2a2e7, [0x1681] = 0x2a37c,
  [0x1682] = 0x2a3af, [0x1683] = 0x2a3e1, [0x1684] = 0x2a496, [0x1685] = 0x2a4a3,
  [0x1686] = 0x2a490, [0x1687] = 0x2a4f8, [0x1688] = 0x2a52e, [0x1689] = 0x2a575,
  [0x168a] = 0x2a57e, [0x168b] = 0x2a57d, [0x168c] = 0x2a57b, [0x168d] = 0x2a578,
  [0x168e] = 0x2a5b0, [0x168f] = 0x2a634, [0x1690] = 0x2a636, [0x1691] = 0x2a630,
  [0x1692] = 0x2fa1d, [0x1693] = 0x2a62e, [0x1694] = 0x2a637, [0x1695] = 0x20ace,
  [0x1696] = 0x20b97, [0x1697] = 0x20b98, [0x1698] = 0x2117c, [0x1699] = 0x2117e,
  [0x169a] = 0x2117d, [0x169b] = 0x2117f, [0x169c] = 0x2150f, [0x169d] = 0x216a4,
  [0x169e] = 0x21931, [0x169f] = 0x2192e, [0x16a0] = 0x2192f, [0x16a1] = 0x21932,
  [0x16a2] = 0x21d2c, [0x16a3] = 0x21fd4, [0x16a4] = 0x21fdc, [0x16a5] = 0x21fd8,
  [0x16a6] = 0x2200e, [0x16a7] = 0x22183, [0x16a8] = 0x22184, [0x16a9] = 0x22312,
  [0x16aa] = 0x22972, [0x16ab] = 0x2297c, [0x16ac] = 0x2297e, [0x16ad] = 0x22ea6,
  [0x16ae] = 0x22ea0, [0x16af] = 0x22e9e, [0x16b0] = 0x22ea2, [0x16b1] = 0x2312a,
  [0x16b2] = 0x23318, [0x16b3] = 0x2384d, [0x16b4] = 0x23858, [0x16b5] = 0x2385b,
  [0x16b6] = 0x25914, [0x16b7] = 0x2f8f2, [0x16b8] = 0x23ae6, [0x16b9] = 0x24164,
  [0x16ba] = 0x24168, [0x16bb] = 0x24160, [0x16bc] = 0x24162, [0x16bd] = 0x244d7,
  [0x16be] = 0x2454b, [0x16bf] = 0x248e0, [0x16c0] = 0x24b22, [0x16c1] = 0x24d05,
  [0x16c2] = 0x24f19, [0x16c3] = 0x24f1a, [0x16c4] = 0x24fc3, [0x16c5] = 0x250d8,
  [0x16c6] = 0x250dc, [0x16c7] = 0x2534f, [0x16c8] = 0x255f7, [0x16c9] = 0x255f4,
  [0x16ca] = 0x255f8, [0x16cb] = 0x25913, [0x16cc] = 0x25a50, [0x16cd] = 0x25df0,
  [0x16ce] = 0x25de9, [0x16cf] = 0x25def, [0x16d0] = 0x25ded, [0x16d1] = 0x25dea,
  [0x16d2] = 0x25df8, [0x16d3] = 0x25e05, [0x16d4] = 0x25df2, [0x16d5] = 0x25f72,
  [0x16d6] = 0x4331, [0x16d7] = 0x261f9, [0x16d8] = 0x261f3, [0x16d9] = 0x261f6,
  [0x16da] = 0x261f2, [0x16db] = 0x261f7, [0x16dc] = 0x26266, [0x16dd] = 0x26265,
  [0x16de] = 0x26340, [0x16df] = 0x2f977, [0x16e0] = 0x268d1, [0x16e1] = 0x268d3,
  [0x16e2] = 0x26a04, [0x16e3] = 0x27159, [0x16e4] = 0x27160, [0x16e5] = 0x2715b,
  [0x16e6] = 0x2715d, [0x16e7] = 0x2715e, [0x16e8] = 0x27169, [0x16e9] = 0x2715c,
  [0x16ea] = 0x27161, [0x16eb] = 0x2716a, [0x16ec] = 0x2715f, [0x16ed] = 0x2754e,
  [0x16ee] = 0x2755c, [0x16ef] = 0x27560, [0x16f0] = 0x27563, [0x16f1] = 0x275db,
  [0x16f2] = 0x277ca, [0x16f3] = 0x277c9, [0x16f4] = 0x277c8, [0x16f5] = 0x278a5,
  [0x16f6] = 0x278a3, [0x16f7] = 0x27b8e, [0x16f8] = 0x27b87, [0x16f9] = 0x27b89,
  [0x16fa] = 0x27b85, [0x16fb] = 0x27b86, [0x16fc] = 0x27b8f, [0x16fd] = 0x27b8a,
  [0x16fe] = 0x27b90, [0x16ff] = 0x27c98, [0x1700] = 0x27c9b, [0x1701] = 0x27e47,
  [0x1702] = 0x27e49, [0x1703] = 0x27e48, [0x1704] = 0x281e5, [0x1705] = 0x281ea,
  [0x1706] = 0x281ed, [0x1707] = 0x283f0, [0x1708] = 0x283ef, [0x1709] = 0x283f2,
  [0x170a] = 0x283ec, [0x170b] = 0x28440, [0x170c] = 0x28660, [0x170d] = 0x2865f,
  [0x170e] = 0x287eb, [0x170f] = 0x287ec, [0x1710] = 0x28900, [0x1711] = 0x28901,
  [0x1712] = 0x28be2, [0x1713] = 0x28bdf, [0x1714] = 0x28be0, [0x1715] = 0x28be1,
  [0x1716] = 0x28be3, [0x1717] = 0x28cc3, [0x1718] = 0x28deb, [0x1719] = 0x28dea,
  [0x171a] = 0x28f61, [0x171b] = 0x28f71, [0x171c] = 0x2906b, [0x171d] = 0x29068,
  [0x171e] = 0x29069, [0x171f] = 0x29067, [0x1720] = 0x291b0, [0x1721] = 0x291b1,
  [0x1722] = 0x291b2, [0x1723] = 0x29251, [0x1724] = 0x29374, [0x1725] = 0x2957f,
  [0x1726] = 0x29584, [0x1727] = 0x29653, [0x1728] = 0x29654, [0x1729] = 0x297f0,
  [0x172a] = 0x297f4, [0x172b] = 0x297f1, [0x172c] = 0x29837, [0x172d] = 0x299bf,
  [0x172e] = 0x4bbc, [0x172f] = 0x4bbb, [0x1730] = 0x299bd, [0x1731] = 0x299be,
  [0x1732] = 0x299b7, [0x1733] = 0x299c0, [0x1734] = 0x299ba, [0x1735] = 0x299b8,
  [0x1736] = 0x299bc, [0x1737] = 0x299bb, [0x1738] = 0x29aeb, [0x1739] = 0x29be7,
  [0x173a] = 0x29c1d, [0x173b] = 0x29c1c, [0x173c] = 0x29c63, [0x173d] = 0x2244b,
  [0x173e] = 0x29d38, [0x173f] = 0x29d36, [0x1740] = 0x29d34, [0x1741] = 0x29d37,
  [0x1742] = 0x29f06, [0x1743] = 0x29f1c, [0x1744] = 0x29f02, [0x1745] = 0x29f1d,
  [0x1746] = 0x29f1e, [0x1747] = 0x29ee0, [0x1748] = 0x29f11, [0x1749] = 0x29f18,
  [0x174a] = 0x29f21, [0x174b] = 0x29f20, [0x174c] = 0x29f10, [0x174d] = 0x2a1bd,
  [0x174e] = 0x2a1ae, [0x174f] = 0x2a1b5, [0x1750] = 0x2a1b8, [0x1751] = 0x2a1b9,
  [0x1752] = 0x2a1be, [0x1753] = 0x2a1c4, [0x1754] = 0x2a1bc, [0x1755] = 0x2a1ba,
  [0x1756] = 0x2a283, [0x1757] = 0x2a2ea, [0x1758] = 0x2a2ec, [0x1759] = 0x2a37e,
  [0x175a] = 0x2a37b, [0x175b] = 0x2a3e5, [0x175c] = 0x2a4a7, [0x175d] = 0x2a49e,
  [0x175e] = 0x2a49a, [0x175f] = 0x2a52f, [0x1760] = 0x2a531, [0x1761] = 0x2a583,
  [0x1762] = 0x2a5b5, [0x1763] = 0x2a646, [0x1764] = 0x2a641, [0x1765] = 0x2a643,
  [0x1766] = 0x2a63d, [0x1767] = 0x2a642, [0x1768] = 0x2a63b, [0x1769] = 0x2a647,
  [0x176a] = 0x2096b, [0x176b] = 0x209d0, [0x176c] = 0x2118e, [0x176d] = 0x21b11,
  [0x176e] = 0x2f871, [0x176f] = 0x2f889, [0x1770] = 0x2244c, [0x1771] = 0x2312b,
  [0x1772] = 0x2384f, [0x1773] = 0x2385e, [0x1774] = 0x23863, [0x1775] = 0x2386b,
  [0x1776] = 0x2393f, [0x1777] = 0x23ae9, [0x1778] = 0x2416d, [0x1779] = 0x2416b,
  [0x177a] = 0x2416a, [0x177b] = 0x29f2a, [0x177c] = 0x244e0, [0x177d] = 0x244dd,
  [0x177e] = 0x244e1, [0x177f] = 0x244de, [0x1780] = 0x244dc, [0x1781] = 0x2454d,
  [0x1782] = 0x24716, [0x1783] = 0x248e3, [0x1784] = 0x24ae5, [0x1785] = 0x24ae4,
  [0x1786] = 0x24f1d, [0x1787] = 0x2574b, [0x1788] = 0x2a6c1, [0x1789] = 0x25a4e,
  [0x178a] = 0x25ad6, [0x178b] = 0x25dfd, [0x178c] = 0x25e07, [0x178d] = 0x25dfe,
  [0x178e] = 0x25e03, [0x178f] = 0x25e0b, [0x1790] = 0x25dff, [0x1791] = 0x25f7a,
  [0x1792] = 0x25f77, [0x1793] = 0x261fb, [0x1794] = 0x26343, [0x1795] = 0x263ed,
  [0x1796] = 0x269bc, [0x1797] = 0x244df, [0x1798] = 0x26a05, [0x1799] = 0x26ac6,
  [0x179a] = 0x26ac5, [0x179b] = 0x27174, [0x179c] = 0x27173, [0x179d] = 0x27172,
  [0x179e] = 0x27178, [0x179f] = 0x27573, [0x17a0] = 0x2757a, [0x17a1] = 0x24f1c,
  [0x17a2] = 0x27576, [0x17a3] = 0x27575, [0x17a4] = 0x27577, [0x17a5] = 0x277cf,
  [0x17a6] = 0x277ce, [0x17a7] = 0x277cd, [0x17a8] = 0x278a8, [0x17a9] = 0x27b98,
  [0x17aa] = 0x27b92, [0x17ab] = 0x27b94, [0x17ac] = 0x27c9a, [0x17ad] = 0x27fb4,
  [0x17ae] = 0x281eb, [0x17af] = 0x281f4, [0x17b0] = 0x281f3, [0x17b1] = 0x281ee,
  [0x17b2] = 0x281f2, [0x17b3] = 0x281f0, [0x17b4] = 0x28664, [0x17b5] = 0x28c05,
  [0x17b6] = 0x28bfb, [0x17b7] = 0x28bfc, [0x17b8] = 0x28df3, [0x17b9] = 0x2906c,
  [0x17ba] = 0x291ba, [0x17bb] = 0x29379, [0x17bc] = 0x2937e, [0x17bd] = 0x29378,
  [0x17be] = 0x293f4, [0x17bf] = 0x29416, [0x17c0] = 0x29657, [0x17c1] = 0x299c8,
  [0x17c2] = 0x299c3, [0x17c3] = 0x299ce, [0x17c4] = 0x29ab3, [0x17c5] = 0x29aed,
  [0x17c6] = 0x29aee, [0x17c7] = 0x29bf1, [0x17c8] = 0x29bf8, [0x17c9] = 0x4c14,
  [0x17ca] = 0x29c64, [0x17cb] = 0x29c65, [0x17cc] = 0x29f35, [0x17cd] = 0x29f29,
  [0x17ce] = 0x29f2c, [0x17cf] = 0x29f31, [0x17d0] = 0x29f2b, [0x17d1] = 0x29f2e,
  [0x17d2] = 0x29f25, [0x17d3] = 0x2a1da, [0x17d4] = 0x2a1e0, [0x17d5] = 0x2a1d4,
  [0x17d6] = 0x2a1e1, [0x17d7] = 0x2a1dd, [0x17d8] = 0x2a3e7, [0x17d9] = 0x2a412,
  [0x17da] = 0x2a4ae, [0x17db] = 0x2a4af, [0x17dc] = 0x2a586, [0x17dd] = 0x2a585,
  [0x17de] = 0x2a5d1, [0x17df] = 0x2a658, [0x17e0] = 0x2a63f, [0x17e1] = 0x2a652,
  [0x17e2] = 0x2a64b, [0x17e3] = 0x2a653, [0x17e4] = 0x2a650, [0x17e5] = 0x2a656,
  [0x17e6] = 0x2a657, [0x17e7] = 0x2a6cd, [0x17e8] = 0x2a6cb, [0x17e9] = 0x20831,
  [0x17ea] = 0x20830, [0x17eb] = 0x209d1, [0x17ec] = 0x361c, [0x17ed] = 0x21192,
  [0x17ee] = 0x21190, [0x17ef] = 0x2118f, [0x17f0] = 0x21193, [0x17f1] = 0x21191,
  [0x17f2] = 0x3772, [0x17f3] = 0x22313, [0x17f4] = 0x2247f, [0x17f5] = 0x22eac,
  [0x17f6] = 0x24170, [0x17f7] = 0x244e5, [0x17f8] = 0x24b23, [0x17f9] = 0x24b24,
  [0x17fa] = 0x24d08, [0x17fb] = 0x24d07, [0x17fc] = 0x24f1f, [0x17fd] = 0x24f20,
  [0x17fe] = 0x24f21, [0x17ff] = 0x250e1, [0x1800] = 0x25354, [0x1801] = 0x25357,
  [0x1802] = 0x25355, [0x1803] = 0x255fa, [0x1804] = 0x25738, [0x1805] = 0x27186,
  [0x1806] = 0x25737, [0x1807] = 0x25a53, [0x1808] = 0x25f7e, [0x1809] = 0x25f7b,
  [0x180a] = 0x2f971, [0x180b] = 0x261fd, [0x180c] = 0x261fc, [0x180d] = 0x26346,
  [0x180e] = 0x26932, [0x180f] = 0x26ac7, [0x1810] = 0x27171, [0x1811] = 0x27183,
  [0x1812] = 0x2716f, [0x1813] = 0x2717e, [0x1814] = 0x2717d, [0x1815] = 0x27181,
  [0x1816] = 0x2717c, [0x1817] = 0x2758a, [0x1818] = 0x2757e, [0x1819] = 0x27580,
  [0x181a] = 0x2757d, [0x181b] = 0x2757f, [0x181c] = 0x27582, [0x181d] = 0x277d4,
  [0x181e] = 0x277d0, [0x181f] = 0x277d2, [0x1820] = 0x27955, [0x1821] = 0x27b9b,
  [0x1822] = 0x27b9a, [0x1823] = 0x27e4e, [0x1824] = 0x281f1, [0x1825] = 0x283f8,
  [0x1826] = 0x283fd, [0x1827] = 0x2865e, [0x1828] = 0x28666, [0x1829] = 0x28905,
  [0x182a] = 0x28c0a, [0x182b] = 0x28c0b, [0x182c] = 0x28c0e, [0x182d] = 0x28c0d,
  [0x182e] = 0x28c11, [0x182f] = 0x28c09, [0x1830] = 0x28c10, [0x1831] = 0x28c0c,
  [0x1832] = 0x28c12, [0x1833] = 0x28cc4, [0x1834] = 0x28df7, [0x1835] = 0x28df8,
  [0x1836] = 0x28f6a, [0x1837] = 0x28f6b, [0x1838] = 0x291c0, [0x1839] = 0x291c2,
  [0x183a] = 0x29385, [0x183b] = 0x29380, [0x183c] = 0x29384, [0x183d] = 0x29381,
  [0x183e] = 0x291c4, [0x183f] = 0x2958b, [0x1840] = 0x2958d, [0x1841] = 0x2958e,
  [0x1842] = 0x29ab7, [0x1843] = 0x29ab9, [0x1844] = 0x29aef, [0x1845] = 0x29bfb,
  [0x1846] = 0x29bff, [0x1847] = 0x29c67, [0x1848] = 0x29c68, [0x1849] = 0x29c69,
  [0x184a] = 0x29d3f, [0x184b] = 0x29f3c, [0x184c] = 0x29f4c, [0x184d] = 0x29f3d,
  [0x184e] = 0x29f3e, [0x184f] = 0x29f3f, [0x1850] = 0x29f3b, [0x1851] = 0x2a1ff,
  [0x1852] = 0x4d0e, [0x1853] = 0x2a1f6, [0x1854] = 0x2a1ee, [0x1855] = 0x2a1fc,
  [0x1856] = 0x2a1f8, [0x1857] = 0x2a201, [0x1858] = 0x2a1fa, [0x1859] = 0x2a288,
  [0x185a] = 0x2a380, [0x185b] = 0x2a411, [0x185c] = 0x2a4fe, [0x185d] = 0x2a500,
  [0x185e] = 0x2a535, [0x185f] = 0x2503c, [0x1860] = 0x2a5bd, [0x1861] = 0x2a5bb,
  [0x1862] = 0x2a65c, [0x1863] = 0x2a65a, [0x1864] = 0x2a6d1, [0x1865] = 0x2a6d0,
  [0x1866] = 0x20a0b, [0x1867] = 0x23873, [0x1868] = 0x21939, [0x1869] = 0x21b1a,
  [0x186a] = 0x21fe3, [0x186b] = 0x22187, [0x186c] = 0x22377, [0x186d] = 0x22378,
  [0x186e] = 0x22eb1, [0x186f] = 0x2386a, [0x1870] = 0x23874, [0x1871] = 0x23aeb,
  [0x1872] = 0x2417b, [0x1873] = 0x245a5, [0x1874] = 0x248e4, [0x1875] = 0x24d0b,
  [0x1876] = 0x24d0e, [0x1877] = 0x24d0c, [0x1878] = 0x25739, [0x1879] = 0x25e10,
  [0x187a] = 0x25e11, [0x187b] = 0x25e16, [0x187c] = 0x25f81, [0x187d] = 0x25f80,
  [0x187e] = 0x261fe, [0x187f] = 0x26201, [0x1880] = 0x26200, [0x1881] = 0x26347,
  [0x1882] = 0x263ef, [0x1883] = 0x27188, [0x1884] = 0x2758b, [0x1885] = 0x2758d,
  [0x1886] = 0x27587, [0x1887] = 0x27585, [0x1888] = 0x2758f, [0x1889] = 0x27584,
  [0x188a] = 0x27588, [0x188b] = 0x2758e, [0x188c] = 0x2758c, [0x188d] = 0x277d7,
  [0x188e] = 0x277d6, [0x188f] = 0x27b9f, [0x1890] = 0x22188, [0x1891] = 0x28203,
  [0x1892] = 0x283fe, [0x1893] = 0x287ef, [0x1894] = 0x28909, [0x1895] = 0x28908,
  [0x1896] = 0x28c20, [0x1897] = 0x28c24, [0x1898] = 0x28c1f, [0x1899] = 0x29070,
  [0x189a] = 0x29256, [0x189b] = 0x293f7, [0x189c] = 0x293f8, [0x189d] = 0x293f6,
  [0x189e] = 0x29590, [0x189f] = 0x2965e, [0x18a0] = 0x297f7, [0x18a1] = 0x299d6,
  [0x18a2] = 0x299d5, [0x18a3] = 0x29abb, [0x18a4] = 0x29af0, [0x18a5] = 0x29c01,
  [0x18a6] = 0x29c6c, [0x18a7] = 0x29d41, [0x18a8] = 0x29d42, [0x18a9] = 0x29f4f,
  [0x18aa] = 0x29f53, [0x18ab] = 0x29f58, [0x18ac] = 0x2a212, [0x18ad] = 0x2a204,
  [0x18ae] = 0x2a213, [0x18af] = 0x2a205, [0x18b0] = 0x2a20d, [0x18b1] = 0x2a382,
  [0x18b2] = 0x2a381, [0x18b3] = 0x2a4b5, [0x18b4] = 0x2a4ba, [0x18b5] = 0x2a5be,
  [0x18b6] = 0x2a665, [0x18b7] = 0x2a663, [0x18b8] = 0x2a66e, [0x18b9] = 0x2a66a,
  [0x18ba] = 0x2a66c, [0x18bb] = 0x2a6c4, [0x18bc] = 0x2096c, [0x18bd] = 0x27ba5,
  [0x18be] = 0x21b12, [0x18bf] = 0x2244d, [0x18c0] = 0x2331f, [0x18c1] = 0x23877,
  [0x18c2] = 0x23878, [0x18c3] = 0x23c44, [0x18c4] = 0x2417c, [0x18c5] = 0x244eb,
  [0x18c6] = 0x244ea, [0x18c7] = 0x244ec, [0x18c8] = 0x245a7, [0x18c9] = 0x24719,
  [0x18ca] = 0x24f23, [0x18cb] = 0x7674, [0x18cc] = 0x25601, [0x18cd] = 0x25e1c,
  [0x18ce] = 0x26203, [0x18cf] = 0x26348, [0x18d0] = 0x268da, [0x18d1] = 0x27191,
  [0x18d2] = 0x27192, [0x18d3] = 0x27596, [0x18d4] = 0x27597, [0x18d5] = 0x27591,
  [0x18d6] = 0x27593, [0x18d7] = 0x277d8, [0x18d8] = 0x277d5, [0x18d9] = 0x27957,
  [0x18da] = 0x27958, [0x18db] = 0x27ba2, [0x18dc] = 0x28205, [0x18dd] = 0x283fa,
  [0x18de] = 0x2890c, [0x18df] = 0x2890a, [0x18e0] = 0x28c2e, [0x18e1] = 0x28c29,
  [0x18e2] = 0x28c2f, [0x18e3] = 0x28cc5, [0x18e4] = 0x29072, [0x18e5] = 0x291c8,
  [0x18e6] = 0x2925a, [0x18e7] = 0x29259, [0x18e8] = 0x29389, [0x18e9] = 0x29388,
  [0x18ea] = 0x299db, [0x18eb] = 0x299d9, [0x18ec] = 0x299da, [0x18ed] = 0x29abc,
  [0x18ee] = 0x4bec, [0x18ef] = 0x29c06, [0x18f0] = 0x29c05, [0x18f1] = 0x29c6f,
  [0x18f2] = 0x29c6d, [0x18f3] = 0x29c70, [0x18f4] = 0x29f60, [0x18f5] = 0x29f5e,
  [0x18f6] = 0x29f63, [0x18f7] = 0x29f5a, [0x18f8] = 0x29f5f, [0x18f9] = 0x2a219,
  [0x18fa] = 0x2a21d, [0x18fb] = 0x2a21b, [0x18fc] = 0x2a222, [0x18fd] = 0x2a21c,
  [0x18fe] = 0x2a21e, [0x18ff] = 0x2a2f7, [0x1900] = 0x2a384, [0x1901] = 0x2a415,
  [0x1902] = 0x2a4b8, [0x1903] = 0x2a4c1, [0x1904] = 0x2a501, [0x1905] = 0x2a50a,
  [0x1906] = 0x2a5d2, [0x1907] = 0x2a675, [0x1908] = 0x2a674, [0x1909] = 0x2a672,
  [0x190a] = 0x20ad0, [0x190b] = 0x21b1b, [0x190c] = 0x22010, [0x190d] = 0x22379,
  [0x190e] = 0x244ed, [0x190f] = 0x24bb9, [0x1910] = 0x24f24, [0x1911] = 0x24fc5,
  [0x1912] = 0x25603, [0x1913] = 0x2634a, [0x1914] = 0x268f7, [0x1915] = 0x269c1,
  [0x1916] = 0x269c2, [0x1917] = 0x27194, [0x1918] = 0x27599, [0x1919] = 0x2759a,
  [0x191a] = 0x2795a, [0x191b] = 0x27ba8, [0x191c] = 0x27c25, [0x191d] = 0x27c9e,
  [0x191e] = 0x28441, [0x191f] = 0x28c32, [0x1920] = 0x28c33, [0x1921] = 0x291cb,
  [0x1922] = 0x291ca, [0x1923] = 0x291cc, [0x1924] = 0x29593, [0x1925] = 0x299e0,
  [0x1926] = 0x29c71, [0x1927] = 0x29f6b, [0x1928] = 0x2a22b, [0x1929] = 0x2a229,
  [0x192a] = 0x2a21a, [0x192b] = 0x2a226, [0x192c] = 0x2a227, [0x192d] = 0x2a22a,
  [0x192e] = 0x2a2f8, [0x192f] = 0x2503d, [0x1930] = 0x2a676, [0x1931] = 0x21518,
  [0x1932] = 0x2387a, [0x1933] = 0x2417f, [0x1934] = 0x4279, [0x1935] = 0x263f0,
  [0x1936] = 0x2719a, [0x1937] = 0x27196, [0x1938] = 0x2759e, [0x1939] = 0x2759d,
  [0x193a] = 0x275dc, [0x193b] = 0x277da, [0x193c] = 0x2820b, [0x193d] = 0x28402,
  [0x193e] = 0x28442, [0x193f] = 0x28667, [0x1940] = 0x2938b, [0x1941] = 0x299e2,
  [0x1942] = 0x29f71, [0x1943] = 0x2a22f, [0x1944] = 0x2a22e, [0x1945] = 0x2a4c6,
  [0x1946] = 0x2a58e, [0x1947] = 0x2a67b, [0x1948] = 0x2a677, [0x1949] = 0x2a678,
  [0x194a] = 0x2a67c, [0x194b] = 0x2387c, [0x194c] = 0x26207, [0x194d] = 0x27197,
  [0x194e] = 0x27bd9, [0x194f] = 0x2890d, [0x1950] = 0x2890e, [0x1951] = 0x299e3,
  [0x1952] = 0x299e4, [0x1953] = 0x29af3, [0x1954] = 0x29c75, [0x1955] = 0x2a237,
  [0x1956] = 0x2a235, [0x1957] = 0x2a58f, [0x1958] = 0x2a680, [0x1959] = 0x2a67f,
  [0x195a] = 0x2a6d4, [0x195b] = 0x22eb4, [0x195c] = 0x2a3ec, [0x195d] = 0x28f75,
  [0x195e] = 0x24d10, [0x195f] = 0x26269, [0x1960] = 0x275a4, [0x1961] = 0x277db,
  [0x1962] = 0x291d1, [0x1963] = 0x2938d, [0x1964] = 0x29f76, [0x1965] = 0x2a239,
  [0x1966] = 0x2a684, [0x1967] = 0x2a682, [0x1968] = 0x20ad2, [0x1969] = 0x244ee,
  [0x196a] = 0x24d11, [0x196b] = 0x25f84, [0x196c] = 0x28f74, [0x196d] = 0x2938f,
  [0x196e] = 0x2938e, [0x196f] = 0x2a23b, [0x1970] = 0x208cb, [0x1971] = 0x2a688,
  [0x1972] = 0x2a686, [0x1973] = 0x2335c, [0x1974] = 0x253a4, [0x1975] = 0x29075,
  [0x1976] = 0x2a68b, [0x1977] = 0x21519, [0x1978] = 0x27c9f, [0x1979] = 0x299e5,
  [0x197a] = 0x2a240, [0x197b] = 0x24d12, [0x197c] = 0x2a2fb, [0x197d] = 0x2a68c,
  [0x197e] = 0x253a5, [0x197f] = 0x29c77, [0x1980] = 0x29c78, [0x1981] = 0x2a6d6,
  [0x1982] = 0x277df, [0x1983] = 0x291d2, [0x1984] = 0x2151a, [0x1985] = 0x27ba9,
  [0x1986] = 0x291d3, [0x1987] = 0x2a4ca, [0x1988] = 0x269c4, [0x1989] = 0x278b1,
  [0x198a] = 0x29664,
};

/* The following table contains no hole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   grep -E '^0xF' CNS11643.TXT |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0xF2121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint32_t __cns11643l15_to_ucs4_tab[] =
{
  [0x0000] = 0x20002, [0x0001] = 0x2008c, [0x0002] = 0x200d4, [0x0003] = 0x201a5,
  [0x0004] = 0x20092, [0x0005] = 0x20093, [0x0006] = 0x200d8, [0x0007] = 0x200d9,
  [0x0008] = 0x201a7, [0x0009] = 0x2062b, [0x000a] = 0x2062c, [0x000b] = 0x2062d,
  [0x000c] = 0x20630, [0x000d] = 0x2062e, [0x000e] = 0x206ab, [0x000f] = 0x5301,
  [0x0010] = 0x2092d, [0x0011] = 0x2096d, [0x0012] = 0x2f834, [0x0013] = 0x20a0f,
  [0x0015] = 0x20b9d, [0x0016] = 0x211a2, [0x0017] = 0x215d4, [0x0018] = 0x21b56,
  [0x0019] = 0x21bc3, [0x001a] = 0x22014, [0x001c] = 0x2471a, [0x001d] = 0x2000f,
  [0x001e] = 0x20010, [0x001f] = 0x2009a, [0x0020] = 0x2009b, [0x0022] = 0x201ac,
  [0x0023] = 0x3436, [0x0024] = 0x20097, [0x0025] = 0x20546, [0x0026] = 0x434f,
  [0x0027] = 0x20576, [0x0028] = 0x20577, [0x0029] = 0x205ae, [0x002a] = 0x2063a,
  [0x002b] = 0x2067d, [0x002c] = 0x21bc5, [0x002d] = 0x20ba8, [0x002f] = 0x53fa,
  [0x0030] = 0x21241, [0x0031] = 0x21242, [0x0032] = 0x215d8, [0x0033] = 0x215d9,
  [0x0034] = 0x216aa, [0x0035] = 0x216ab, [0x0036] = 0x21942, [0x0037] = 0x21941,
  [0x0038] = 0x9f99, [0x0039] = 0x21bc6, [0x003b] = 0x21d35, [0x003c] = 0x21d36,
  [0x003d] = 0x21d37, [0x003e] = 0x2f883, [0x003f] = 0x2634d, [0x0041] = 0x221b5,
  [0x0042] = 0x22317, [0x0043] = 0x22482, [0x0044] = 0x233b6, [0x0045] = 0x6c49,
  [0x0046] = 0x23c72, [0x0047] = 0x2455c, [0x0048] = 0x245a8, [0x0049] = 0x8fb7,
  [0x004a] = 0x215e3, [0x004b] = 0x3406, [0x004d] = 0x200e6, [0x004e] = 0x20144,
  [0x004f] = 0x20145, [0x0050] = 0x4f29, [0x0051] = 0x343e, [0x0052] = 0x201c4,
  [0x0053] = 0x201c5, [0x0054] = 0x201c6, [0x0055] = 0x201c7, [0x0056] = 0x201c8,
  [0x0057] = 0x201c9, [0x0058] = 0x201ca, [0x0059] = 0x201d4, [0x005a] = 0x2057d,
  [0x005b] = 0x26272, [0x005c] = 0x205b2, [0x005d] = 0x205b4, [0x005e] = 0x20682,
  [0x005f] = 0x206c7, [0x0060] = 0x206c8, [0x0062] = 0x2083c, [0x0063] = 0x2083d,
  [0x0064] = 0x2083e, [0x0065] = 0x2083f, [0x0066] = 0x2f824, [0x0067] = 0x208d3,
  [0x0068] = 0x20147, [0x0069] = 0x534e, [0x006a] = 0x20a3d, [0x006b] = 0x20ae2,
  [0x006c] = 0x20bb8, [0x006d] = 0x20bb9, [0x006e] = 0x20bc2, [0x006f] = 0x20bba,
  [0x0070] = 0x20bc6, [0x0071] = 0x20bbb, [0x0072] = 0x211ad, [0x0073] = 0x2124c,
  [0x0074] = 0x2124d, [0x0076] = 0x216b2, [0x0077] = 0x216b3, [0x0078] = 0x216b4,
  [0x0079] = 0x216b5, [0x007a] = 0x219c3, [0x007b] = 0x219c4, [0x007c] = 0x219c5,
  [0x007d] = 0x219c6, [0x007e] = 0x219c7, [0x0080] = 0x21bce, [0x0081] = 0x21c30,
  [0x0082] = 0x21d47, [0x0083] = 0x21d48, [0x0084] = 0x5c81, [0x0085] = 0x2205c,
  [0x0087] = 0x2205f, [0x0088] = 0x22318, [0x0089] = 0x22319, [0x008a] = 0x2237e,
  [0x008b] = 0x5f10, [0x008c] = 0x225b0, [0x008d] = 0x22a75, [0x008e] = 0x6268,
  [0x008f] = 0x22a76, [0x0090] = 0x23041, [0x0091] = 0x23142, [0x0093] = 0x233bd,
  [0x0094] = 0x6742, [0x0095] = 0x6740, [0x0096] = 0x51ea, [0x0097] = 0x23b1e,
  [0x0098] = 0x23b1f, [0x0099] = 0x23c87, [0x009a] = 0x6c62, [0x009b] = 0x23c78,
  [0x009c] = 0x24188, [0x009d] = 0x24189, [0x009e] = 0x2418a, [0x009f] = 0x24617,
  [0x00a0] = 0x24615, [0x00a3] = 0x7391, [0x00a4] = 0x250e5, [0x00a5] = 0x25609,
  [0x00a6] = 0x2573b, [0x00a7] = 0x2574f, [0x00a9] = 0x2662e, [0x00aa] = 0x2662f,
  [0x00ab] = 0x26aff, [0x00ac] = 0x26afd, [0x00ad] = 0x26afe, [0x00ae] = 0x8fbb,
  [0x00af] = 0x8fbc, [0x00b1] = 0x20023, [0x00b2] = 0x200ea, [0x00b3] = 0x2012a,
  [0x00b4] = 0x20149, [0x00b5] = 0x2014a, [0x00b6] = 0x2014b, [0x00b7] = 0x201df,
  [0x00b8] = 0x201e0, [0x00b9] = 0x201e1, [0x00ba] = 0x201e2, [0x00bb] = 0x205c1,
  [0x00bc] = 0x201e3, [0x00bd] = 0x201e4, [0x00be] = 0x201e5, [0x00bf] = 0x205c0,
  [0x00c0] = 0x201e6, [0x00c1] = 0x201f7, [0x00c3] = 0x201e7, [0x00c4] = 0x201e8,
  [0x00c5] = 0x201e9, [0x00c6] = 0x201ea, [0x00c7] = 0x201eb, [0x00c8] = 0x2050e,
  [0x00c9] = 0x20580, [0x00ca] = 0x20581, [0x00cb] = 0x205ba, [0x00cc] = 0x205bb,
  [0x00cd] = 0x205bc, [0x00ce] = 0x205bd, [0x00cf] = 0x2063f, [0x00d0] = 0x20640,
  [0x00d1] = 0x20688, [0x00d2] = 0x2f81f, [0x00d3] = 0x20846, [0x00d5] = 0x20847,
  [0x00d6] = 0x20848, [0x00d7] = 0x20936, [0x00d8] = 0x20972, [0x00d9] = 0x20996,
  [0x00da] = 0x20a47, [0x00db] = 0x20ae7, [0x00dc] = 0x20b34, [0x00dd] = 0x20b35,
  [0x00de] = 0x20b36, [0x00df] = 0x20b37, [0x00e0] = 0x20be3, [0x00e1] = 0x20be4,
  [0x00e2] = 0x20be5, [0x00e3] = 0x20beb, [0x00e4] = 0x3575, [0x00e5] = 0x20be6,
  [0x00e6] = 0x20be7, [0x00e7] = 0x20be8, [0x00ea] = 0x211b9, [0x00eb] = 0x56e8,
  [0x00ec] = 0x211ba, [0x00ed] = 0x575b, [0x00ee] = 0x21263, [0x00ef] = 0x21264,
  [0x00f1] = 0x21265, [0x00f3] = 0x21266, [0x00f4] = 0x2127c, [0x00f5] = 0x21267,
  [0x00f6] = 0x2f85a, [0x00f7] = 0x21559, [0x00f8] = 0x215e9, [0x00f9] = 0x215ea,
  [0x00fa] = 0x215eb, [0x00fb] = 0x215ec, [0x00fc] = 0x215ed, [0x00fd] = 0x215ee,
  [0x00fe] = 0x215ef, [0x00ff] = 0x216bf, [0x0100] = 0x216cb, [0x0101] = 0x216c3,
  [0x0102] = 0x216d5, [0x0103] = 0x216c4, [0x0104] = 0x216c5, [0x0105] = 0x216d3,
  [0x0106] = 0x216c6, [0x0107] = 0x216c7, [0x0108] = 0x216c8, [0x0109] = 0x216c9,
  [0x010a] = 0x216ca, [0x010b] = 0x2194a, [0x010d] = 0x219d1, [0x010e] = 0x21b61,
  [0x010f] = 0x21c37, [0x0111] = 0x21d64, [0x0112] = 0x21d66, [0x0113] = 0x5c99,
  [0x0114] = 0x21d67, [0x0115] = 0x21d68, [0x0116] = 0x21d69, [0x0117] = 0x5c97,
  [0x0118] = 0x21d73, [0x0119] = 0x21d6a, [0x011a] = 0x21d6b, [0x011b] = 0x6762,
  [0x011d] = 0x22072, [0x011e] = 0x383c, [0x011f] = 0x221b9, [0x0120] = 0x2231c,
  [0x0121] = 0x22336, [0x0122] = 0x223bc, [0x0123] = 0x223be, [0x0124] = 0x223bf,
  [0x0125] = 0x223c0, [0x0126] = 0x22487, [0x0127] = 0x225dc, [0x0128] = 0x225dd,
  [0x0129] = 0x225f5, [0x012a] = 0x225de, [0x012b] = 0x225df, [0x012c] = 0x2f89e,
  [0x012d] = 0x225e0, [0x012e] = 0x225e1, [0x012f] = 0x225e2, [0x0130] = 0x22a2c,
  [0x0131] = 0x22a2d, [0x0132] = 0x62a4, [0x0133] = 0x22aa9, [0x0134] = 0x22a9c,
  [0x0135] = 0x22a9d, [0x0136] = 0x22a9e, [0x0137] = 0x22a9f, [0x0138] = 0x22aa0,
  [0x0139] = 0x22aa1, [0x013a] = 0x22aa2, [0x013c] = 0x22aa3, [0x013d] = 0x22aa4,
  [0x013e] = 0x22aa7, [0x013f] = 0x23042, [0x0140] = 0x23093, [0x0141] = 0x23151,
  [0x0143] = 0x23152, [0x0144] = 0x23144, [0x0145] = 0x23153, [0x0146] = 0x23322,
  [0x0147] = 0x233c6, [0x0148] = 0x233c7, [0x0149] = 0x233c9, [0x014a] = 0x2f8dd,
  [0x014b] = 0x233ca, [0x014c] = 0x6766, [0x014d] = 0x233c8, [0x014e] = 0x23b24,
  [0x014f] = 0x23b25, [0x0150] = 0x23b26, [0x0152] = 0x23b27, [0x0153] = 0x23b28,
  [0x0155] = 0x23c58, [0x0156] = 0x2f8fb, [0x0157] = 0x23ca7, [0x0158] = 0x23c88,
  [0x0159] = 0x23c89, [0x015a] = 0x23ca5, [0x015b] = 0x23ca8, [0x015c] = 0x23ca9,
  [0x015d] = 0x23caa, [0x015e] = 0x23cab, [0x015f] = 0x23cac, [0x0160] = 0x23cad,
  [0x0161] = 0x6ca3, [0x0162] = 0x23c79, [0x0164] = 0x24194, [0x0165] = 0x24195,
  [0x0166] = 0x24196, [0x0167] = 0x3da5, [0x0169] = 0x24197, [0x016a] = 0x707f,
  [0x016b] = 0x24198, [0x016c] = 0x2454e, [0x016d] = 0x2454f, [0x016e] = 0x24619,
  [0x0170] = 0x24741, [0x0171] = 0x2f92a, [0x0172] = 0x248f7, [0x0173] = 0x24c04,
  [0x0174] = 0x24c19, [0x0175] = 0x24c17, [0x0176] = 0x25040, [0x0177] = 0x250ec,
  [0x0178] = 0x2541b, [0x0179] = 0x2541c, [0x017a] = 0x77f6, [0x017b] = 0x2560e,
  [0x017c] = 0x25752, [0x017d] = 0x25ad8, [0x017e] = 0x25e26, [0x017f] = 0x26274,
  [0x0181] = 0x26275, [0x0182] = 0x269c6, [0x0183] = 0x26b0b, [0x0184] = 0x26b0c,
  [0x0185] = 0x26b0d, [0x0186] = 0x26b0e, [0x0187] = 0x26b0f, [0x0188] = 0x26b10,
  [0x0189] = 0x8fc8, [0x018a] = 0x28459, [0x018b] = 0x2845a, [0x018c] = 0x28683,
  [0x018d] = 0x28684, [0x018e] = 0x28685, [0x0190] = 0x95f5, [0x0191] = 0x28e20,
  [0x0192] = 0x28e21, [0x0193] = 0x28e22, [0x0194] = 0x28e23, [0x0195] = 0x28e24,
  [0x0196] = 0x28e29, [0x0197] = 0x200f4, [0x0199] = 0x2158f, [0x019a] = 0x20151,
  [0x019b] = 0x4fab, [0x019c] = 0x20217, [0x019d] = 0x20218, [0x019e] = 0x2025b,
  [0x019f] = 0x20219, [0x01a0] = 0x2021a, [0x01a1] = 0x3453, [0x01a2] = 0x2021b,
  [0x01a3] = 0x2021c, [0x01a4] = 0x2021d, [0x01a5] = 0x2021e, [0x01a6] = 0x2021f,
  [0x01a7] = 0x20220, [0x01a8] = 0x20221, [0x01a9] = 0x2048b, [0x01aa] = 0x20488,
  [0x01ab] = 0x5c2d, [0x01ac] = 0x20585, [0x01ad] = 0x26287, [0x01ae] = 0x205c7,
  [0x01af] = 0x2064c, [0x01b0] = 0x2064d, [0x01b1] = 0x206fc, [0x01b2] = 0x206fd,
  [0x01b3] = 0x206fe, [0x01b4] = 0x206ff, [0x01b7] = 0x20856, [0x01b8] = 0x20857,
  [0x01b9] = 0x20858, [0x01ba] = 0x20859, [0x01bb] = 0x208dc, [0x01bc] = 0x208dd,
  [0x01bd] = 0x20973, [0x01be] = 0x2099d, [0x01bf] = 0x209a1, [0x01c0] = 0x2099e,
  [0x01c1] = 0x2099f, [0x01c2] = 0x209a0, [0x01c3] = 0x209e6, [0x01c4] = 0x20a51,
  [0x01c5] = 0x20a52, [0x01c6] = 0x20b3d, [0x01c7] = 0x20c22, [0x01c8] = 0x549c,
  [0x01c9] = 0x20c23, [0x01ca] = 0x20c24, [0x01cb] = 0x20c25, [0x01cc] = 0x20c26,
  [0x01cd] = 0x20c27, [0x01ce] = 0x20c28, [0x01cf] = 0x20c29, [0x01d0] = 0x20c2a,
  [0x01d1] = 0x20c2b, [0x01d3] = 0x20c40, [0x01d4] = 0x20c2c, [0x01d5] = 0x20c2d,
  [0x01d6] = 0x20c2e, [0x01d7] = 0x20c2f, [0x01d8] = 0x20c30, [0x01d9] = 0x211d0,
  [0x01da] = 0x212a8, [0x01db] = 0x21294, [0x01dc] = 0x21295, [0x01dd] = 0x21296,
  [0x01de] = 0x21297, [0x01df] = 0x21298, [0x01e0] = 0x2129c, [0x01e1] = 0x2129b,
  [0x01e3] = 0x212a1, [0x01e4] = 0x212a0, [0x01e5] = 0x21299, [0x01e6] = 0x2129a,
  [0x01e7] = 0x212a7, [0x01e8] = 0x5788, [0x01e9] = 0x21564, [0x01eb] = 0x21600,
  [0x01ec] = 0x21601, [0x01ef] = 0x216e5, [0x01f0] = 0x216e9, [0x01f1] = 0x216fb,
  [0x01f2] = 0x216fc, [0x01f3] = 0x216ea, [0x01f4] = 0x216eb, [0x01f5] = 0x216ec,
  [0x01f6] = 0x216ed, [0x01f7] = 0x216ee, [0x01f8] = 0x216ef, [0x01f9] = 0x216fa,
  [0x01fa] = 0x216f0, [0x01fb] = 0x216f1, [0x01fc] = 0x216f2, [0x01fd] = 0x216f3,
  [0x01fe] = 0x216f4, [0x01ff] = 0x21952, [0x0200] = 0x21953, [0x0201] = 0x21954,
  [0x0202] = 0x21955, [0x0203] = 0x21958, [0x0204] = 0x21956, [0x0206] = 0x219dd,
  [0x0207] = 0x219de, [0x0208] = 0x219e3, [0x0209] = 0x219df, [0x020a] = 0x219e0,
  [0x020b] = 0x21b22, [0x020c] = 0x21b23, [0x020d] = 0x21b68, [0x020e] = 0x21bdf,
  [0x020f] = 0x34ac, [0x0210] = 0x21c44, [0x0211] = 0x21d94, [0x0212] = 0x21d95,
  [0x0213] = 0x21d96, [0x0214] = 0x21d97, [0x0215] = 0x21d98, [0x0216] = 0x21d99,
  [0x0217] = 0x2f879, [0x0218] = 0x21d9a, [0x0219] = 0x21d9b, [0x021a] = 0x21d9d,
  [0x021b] = 0x21d9e, [0x021c] = 0x21d9f, [0x021d] = 0x21da0, [0x021e] = 0x21ff3,
  [0x021f] = 0x21ff2, [0x0220] = 0x2201f, [0x0221] = 0x22088, [0x0222] = 0x2218f,
  [0x0223] = 0x22190, [0x0225] = 0x221fd, [0x0226] = 0x221fe, [0x0227] = 0x221ff,
  [0x0228] = 0x22200, [0x0229] = 0x22201, [0x022a] = 0x2261a, [0x022c] = 0x225e3,
  [0x022d] = 0x225e4, [0x022e] = 0x225e5, [0x022f] = 0x225e6, [0x0230] = 0x225e7,
  [0x0231] = 0x225eb, [0x0232] = 0x225e8, [0x0233] = 0x225e9, [0x0234] = 0x225ea,
  [0x0235] = 0x2261d, [0x0236] = 0x229a4, [0x0237] = 0x229a3, [0x0238] = 0x22ad4,
  [0x0239] = 0x22ac6, [0x023a] = 0x22ac7, [0x023b] = 0x22ac8, [0x023c] = 0x22ac9,
  [0x023d] = 0x22aca, [0x023e] = 0x22acb, [0x023f] = 0x22acc, [0x0240] = 0x22acd,
  [0x0241] = 0x39de, [0x0242] = 0x22ace, [0x0243] = 0x62c3, [0x0244] = 0x22eb8,
  [0x0245] = 0x22eb9, [0x0246] = 0x22eba, [0x0247] = 0x2306e, [0x0248] = 0x230db,
  [0x0249] = 0x230dc, [0x024a] = 0x23168, [0x024b] = 0x23169, [0x024c] = 0x2316a,
  [0x024d] = 0x2316b, [0x024e] = 0x2316c, [0x024f] = 0x2316d, [0x0250] = 0x2316e,
  [0x0251] = 0x6619, [0x0252] = 0x23324, [0x0253] = 0x23325, [0x0255] = 0x23400,
  [0x0256] = 0x233e5, [0x0257] = 0x233f9, [0x0258] = 0x233e7, [0x0259] = 0x233e8,
  [0x025a] = 0x233e9, [0x025b] = 0x233fe, [0x025c] = 0x233ea, [0x025d] = 0x233eb,
  [0x025e] = 0x233ec, [0x025f] = 0x3b49, [0x0260] = 0x233ed, [0x0261] = 0x67a1,
  [0x0262] = 0x233ee, [0x0263] = 0x67a6, [0x0264] = 0x233ef, [0x0266] = 0x23417,
  [0x0267] = 0x233f0, [0x0268] = 0x233f1, [0x0269] = 0x23894, [0x026a] = 0x2395b,
  [0x026b] = 0x2395c, [0x026c] = 0x2395d, [0x026d] = 0x23b2f, [0x026e] = 0x3c91,
  [0x026f] = 0x23b30, [0x0270] = 0x23b31, [0x0271] = 0x23b32, [0x0272] = 0x23b33,
  [0x0273] = 0x23ccc, [0x0274] = 0x23ccd, [0x0275] = 0x23cce, [0x0276] = 0x23cae,
  [0x0277] = 0x23caf, [0x0278] = 0x23cb0, [0x0279] = 0x3cd3, [0x027a] = 0x23ccf,
  [0x027b] = 0x23cd0, [0x027c] = 0x23cd1, [0x027d] = 0x23cd2, [0x027e] = 0x23cd3,
  [0x0280] = 0x23cd4, [0x0281] = 0x3da9, [0x0282] = 0x241a6, [0x0283] = 0x241a7,
  [0x0284] = 0x241a8, [0x0285] = 0x241b0, [0x0286] = 0x241a9, [0x0287] = 0x241aa,
  [0x0288] = 0x241ab, [0x0289] = 0x241ac, [0x028a] = 0x241ad, [0x028b] = 0x244fb,
  [0x028c] = 0x244fc, [0x028e] = 0x2462d, [0x028f] = 0x2462e, [0x0290] = 0x24650,
  [0x0291] = 0x2462f, [0x0292] = 0x24630, [0x0293] = 0x24765, [0x0294] = 0x248e6,
  [0x0295] = 0x2490c, [0x0296] = 0x24902, [0x0297] = 0x24903, [0x0298] = 0x24904,
  [0x0299] = 0x24905, [0x029a] = 0x24906, [0x029c] = 0x24bc0, [0x029d] = 0x24c24,
  [0x029e] = 0x24c25, [0x029f] = 0x24d16, [0x02a0] = 0x24d32, [0x02a1] = 0x25420,
  [0x02a2] = 0x25421, [0x02a3] = 0x25422, [0x02a4] = 0x25423, [0x02a5] = 0x77fe,
  [0x02a6] = 0x25425, [0x02a7] = 0x25619, [0x02a8] = 0x25624, [0x02a9] = 0x2561a,
  [0x02aa] = 0x2561b, [0x02ab] = 0x2561c, [0x02ac] = 0x2561d, [0x02ad] = 0x2f952,
  [0x02ae] = 0x2561e, [0x02af] = 0x2575a, [0x02b0] = 0x2575b, [0x02b1] = 0x2575c,
  [0x02b2] = 0x2575d, [0x02b3] = 0x2575e, [0x02b4] = 0x25931, [0x02b6] = 0x25e29,
  [0x02b8] = 0x25f88, [0x02b9] = 0x25f89, [0x02ba] = 0x26289, [0x02bb] = 0x7f57,
  [0x02bc] = 0x26356, [0x02bd] = 0x264d1, [0x02be] = 0x264e6, [0x02bf] = 0x26532,
  [0x02c0] = 0x26533, [0x02c1] = 0x26652, [0x02c2] = 0x26653, [0x02c3] = 0x43d5,
  [0x02c4] = 0x26654, [0x02c5] = 0x26655, [0x02c8] = 0x26b29, [0x02c9] = 0x26b2a,
  [0x02ca] = 0x26b2b, [0x02cb] = 0x26b2c, [0x02cc] = 0x26b2d, [0x02cd] = 0x2f992,
  [0x02ce] = 0x26b2e, [0x02cf] = 0x26b2f, [0x02d0] = 0x26b30, [0x02d1] = 0x26b31,
  [0x02d2] = 0x26b32, [0x02d3] = 0x26b33, [0x02d4] = 0x82c5, [0x02d5] = 0x26b34,
  [0x02d6] = 0x2719b, [0x02d7] = 0x27223, [0x02d8] = 0x27224, [0x02da] = 0x27225,
  [0x02db] = 0x27227, [0x02dc] = 0x27616, [0x02dd] = 0x27ca0, [0x02de] = 0x2846f,
  [0x02df] = 0x8fdf, [0x02e0] = 0x28470, [0x02e1] = 0x28471, [0x02e2] = 0x8fdc,
  [0x02e3] = 0x488c, [0x02e4] = 0x28472, [0x02e5] = 0x28473, [0x02e6] = 0x28474,
  [0x02e7] = 0x2869c, [0x02e8] = 0x286aa, [0x02e9] = 0x2869d, [0x02ea] = 0x2869e,
  [0x02eb] = 0x4099, [0x02ec] = 0x286ab, [0x02ed] = 0x2869f, [0x02ee] = 0x28e35,
  [0x02ef] = 0x28e38, [0x02f0] = 0x2a38d, [0x02f1] = 0x20035, [0x02f2] = 0x20036,
  [0x02f3] = 0x25aee, [0x02f4] = 0x200b4, [0x02f5] = 0x2a6c9, [0x02f6] = 0x20132,
  [0x02f7] = 0x20155, [0x02f8] = 0x20244, [0x02f9] = 0x20245, [0x02fa] = 0x20246,
  [0x02fb] = 0x20247, [0x02fc] = 0x4fe4, [0x02fd] = 0x20248, [0x02fe] = 0x20249,
  [0x0300] = 0x2024a, [0x0301] = 0x2024b, [0x0302] = 0x2024c, [0x0303] = 0x2024d,
  [0x0304] = 0x2024e, [0x0305] = 0x2024f, [0x0306] = 0x20250, [0x0307] = 0x20251,
  [0x0308] = 0x26299, [0x0309] = 0x205d1, [0x030b] = 0x205d2, [0x030c] = 0x205d3,
  [0x030d] = 0x205d4, [0x030e] = 0x205d5, [0x030f] = 0x20721, [0x0310] = 0x20722,
  [0x0311] = 0x20723, [0x0312] = 0x20868, [0x0313] = 0x20869, [0x0314] = 0x2086a,
  [0x0315] = 0x208e2, [0x0316] = 0x20945, [0x0317] = 0x209ec, [0x0318] = 0x209ed,
  [0x0319] = 0x209ee, [0x031a] = 0x209ef, [0x031c] = 0x20a62, [0x031d] = 0x20a63,
  [0x031e] = 0x20af4, [0x031f] = 0x2006f, [0x0320] = 0x20b44, [0x0321] = 0x20b45,
  [0x0322] = 0x20c79, [0x0323] = 0x551b, [0x0324] = 0x20c7a, [0x0325] = 0x20c7b,
  [0x0326] = 0x20c7c, [0x0327] = 0x20c7d, [0x0328] = 0x20c7e, [0x0329] = 0x20c7f,
  [0x032a] = 0x20c80, [0x032b] = 0x3588, [0x032d] = 0x20c81, [0x032e] = 0x20c82,
  [0x032f] = 0x20c83, [0x0330] = 0x20c84, [0x0331] = 0x20c85, [0x0332] = 0x20c86,
  [0x0333] = 0x20c87, [0x0334] = 0x20c8f, [0x0335] = 0x20c88, [0x0336] = 0x20c89,
  [0x0337] = 0x212c5, [0x0338] = 0x212c6, [0x0339] = 0x57aa, [0x033a] = 0x212c7,
  [0x033b] = 0x212c8, [0x033c] = 0x212c9, [0x033d] = 0x212ca, [0x033e] = 0x212cb,
  [0x033f] = 0x212cc, [0x0340] = 0x57ab, [0x0341] = 0x212cd, [0x0342] = 0x212ce,
  [0x0343] = 0x212e3, [0x0344] = 0x212cf, [0x0345] = 0x212d0, [0x0346] = 0x212d1,
  [0x0347] = 0x212d2, [0x0348] = 0x212d3, [0x0349] = 0x212c4, [0x034a] = 0x212d4,
  [0x034b] = 0x212d5, [0x034c] = 0x212d6, [0x034d] = 0x21520, [0x034e] = 0x21521,
  [0x034f] = 0x2161a, [0x0350] = 0x21612, [0x0351] = 0x21614, [0x0352] = 0x21615,
  [0x0353] = 0x21616, [0x0354] = 0x21619, [0x0355] = 0x21617, [0x0356] = 0x21618,
  [0x0357] = 0x36c9, [0x0358] = 0x2f861, [0x035a] = 0x21712, [0x035b] = 0x21713,
  [0x035c] = 0x21714, [0x035d] = 0x21715, [0x035e] = 0x21716, [0x035f] = 0x21717,
  [0x0360] = 0x21718, [0x0361] = 0x21719, [0x0362] = 0x21961, [0x0363] = 0x21962,
  [0x0364] = 0x21963, [0x0365] = 0x219f5, [0x0366] = 0x5ba9, [0x0367] = 0x219f6,
  [0x0368] = 0x219f7, [0x0369] = 0x219f8, [0x036a] = 0x21b29, [0x036b] = 0x3917,
  [0x036c] = 0x21c5d, [0x036d] = 0x21c57, [0x036e] = 0x21c58, [0x036f] = 0x21c59,
  [0x0370] = 0x21c5a, [0x0371] = 0x21c5b, [0x0372] = 0x21dbb, [0x0373] = 0x21dd1,
  [0x0374] = 0x21dbc, [0x0375] = 0x21dbd, [0x0376] = 0x268df, [0x0377] = 0x21dd0,
  [0x0378] = 0x21dbe, [0x0379] = 0x21dbf, [0x037a] = 0x21dc0, [0x037b] = 0x21dc1,
  [0x037c] = 0x21dc2, [0x037d] = 0x21dc3, [0x037e] = 0x21dc4, [0x037f] = 0x26b6c,
  [0x0380] = 0x220a1, [0x0381] = 0x220a2, [0x0382] = 0x220a3, [0x0383] = 0x22193,
  [0x0384] = 0x2220d, [0x0385] = 0x2220f, [0x0386] = 0x2220e, [0x0387] = 0x22210,
  [0x0388] = 0x22211, [0x0389] = 0x22212, [0x038a] = 0x2f895, [0x038b] = 0x224e5,
  [0x038c] = 0x2262f, [0x038e] = 0x22657, [0x038f] = 0x22630, [0x0390] = 0x22668,
  [0x0391] = 0x2262b, [0x0392] = 0x22631, [0x0393] = 0x22658, [0x0394] = 0x2f8a2,
  [0x0395] = 0x22632, [0x0396] = 0x22659, [0x0398] = 0x2265a, [0x0399] = 0x22633,
  [0x039a] = 0x2265b, [0x039b] = 0x2265c, [0x039c] = 0x2265d, [0x039d] = 0x2261e,
  [0x039e] = 0x229ae, [0x039f] = 0x229af, [0x03a0] = 0x22a3b, [0x03a1] = 0x22a3c,
  [0x03a2] = 0x22a3d, [0x03a3] = 0x22b07, [0x03a4] = 0x22b08, [0x03a5] = 0x22b09,
  [0x03a6] = 0x22b0a, [0x03a7] = 0x22b0b, [0x03a8] = 0x22b0c, [0x03a9] = 0x22b0d,
  [0x03aa] = 0x22b0e, [0x03ab] = 0x22b0f, [0x03ac] = 0x22b10, [0x03ad] = 0x22b22,
  [0x03ae] = 0x22b11, [0x03af] = 0x22b12, [0x03b0] = 0x22b13, [0x03b1] = 0x22b14,
  [0x03b2] = 0x22b15, [0x03b3] = 0x22b16, [0x03b4] = 0x22b17, [0x03b5] = 0x22b18,
  [0x03b6] = 0x2f8b8, [0x03b8] = 0x23046, [0x03b9] = 0x23047, [0x03ba] = 0x23048,
  [0x03bb] = 0x230e1, [0x03bc] = 0x23181, [0x03bd] = 0x23183, [0x03be] = 0x20c75,
  [0x03c0] = 0x23185, [0x03c1] = 0x23186, [0x03c2] = 0x2332a, [0x03c3] = 0x2f8d1,
  [0x03c4] = 0x2332b, [0x03c5] = 0x2336b, [0x03c6] = 0x23418, [0x03c7] = 0x23419,
  [0x03c8] = 0x2341a, [0x03ca] = 0x2341b, [0x03cb] = 0x2341c, [0x03cc] = 0x2341d,
  [0x03cd] = 0x2341e, [0x03ce] = 0x2341f, [0x03cf] = 0x23420, [0x03d0] = 0x23421,
  [0x03d1] = 0x23422, [0x03d2] = 0x23423, [0x03d3] = 0x6811, [0x03d4] = 0x23424,
  [0x03d5] = 0x23425, [0x03d6] = 0x23426, [0x03d7] = 0x23427, [0x03d8] = 0x23428,
  [0x03d9] = 0x23429, [0x03da] = 0x2342a, [0x03db] = 0x239ef, [0x03dc] = 0x23b49,
  [0x03dd] = 0x23b4a, [0x03de] = 0x23b4b, [0x03df] = 0x23b4c, [0x03e0] = 0x23b4d,
  [0x03e1] = 0x23b4e, [0x03e2] = 0x23b4f, [0x03e3] = 0x23b50, [0x03e4] = 0x23b51,
  [0x03e5] = 0x23b52, [0x03e6] = 0x23b53, [0x03e7] = 0x23b54, [0x03e8] = 0x23b5a,
  [0x03e9] = 0x23b55, [0x03ea] = 0x23c5f, [0x03eb] = 0x23d02, [0x03ec] = 0x23d03,
  [0x03ed] = 0x23d04, [0x03ee] = 0x23d05, [0x03ef] = 0x23cd5, [0x03f0] = 0x23d06,
  [0x03f1] = 0x23d07, [0x03f2] = 0x23cd6, [0x03f3] = 0x23d08, [0x03f4] = 0x23d09,
  [0x03f5] = 0x23d0a, [0x03f6] = 0x23d0b, [0x03f7] = 0x23cd7, [0x03f8] = 0x23d0c,
  [0x03f9] = 0x23d0d, [0x03fa] = 0x241c8, [0x03fb] = 0x241c9, [0x03fc] = 0x241ca,
  [0x03fd] = 0x241cb, [0x03fe] = 0x241cc, [0x03ff] = 0x241cd, [0x0400] = 0x241ce,
  [0x0401] = 0x241cf, [0x0402] = 0x241d0, [0x0403] = 0x241d1, [0x0404] = 0x241d2,
  [0x0405] = 0x7551, [0x0406] = 0x241d3, [0x0407] = 0x241d4, [0x0408] = 0x241d5,
  [0x0409] = 0x241d6, [0x040a] = 0x241d7, [0x040b] = 0x241e2, [0x040c] = 0x241d8,
  [0x040d] = 0x24504, [0x040e] = 0x24505, [0x040f] = 0x24506, [0x0411] = 0x24643,
  [0x0412] = 0x24644, [0x0413] = 0x24784, [0x0414] = 0x2478f, [0x0415] = 0x24785,
  [0x0416] = 0x24786, [0x0417] = 0x2492f, [0x0418] = 0x2491a, [0x0419] = 0x2491b,
  [0x041a] = 0x2491c, [0x041b] = 0x3ebd, [0x041c] = 0x2491d, [0x041d] = 0x2491e,
  [0x041e] = 0x2491f, [0x041f] = 0x24920, [0x0420] = 0x24921, [0x0421] = 0x24922,
  [0x0422] = 0x2f92c, [0x0423] = 0x24923, [0x0424] = 0x24924, [0x0425] = 0x2490e,
  [0x0426] = 0x24925, [0x0427] = 0x2492b, [0x0428] = 0x2492c, [0x0429] = 0x2490b,
  [0x042a] = 0x24aee, [0x042c] = 0x24bc3, [0x042d] = 0x24bd8, [0x042e] = 0x24bd9,
  [0x042f] = 0x24c38, [0x0430] = 0x24c39, [0x0431] = 0x24c3a, [0x0432] = 0x24c3b,
  [0x0433] = 0x7553, [0x0434] = 0x24c3c, [0x0435] = 0x24c3d, [0x0436] = 0x24c3e,
  [0x0437] = 0x24c3f, [0x0438] = 0x24c40, [0x0439] = 0x24d48, [0x043a] = 0x24d4c,
  [0x043b] = 0x24f28, [0x043c] = 0x24f51, [0x043d] = 0x24f52, [0x043e] = 0x2504c,
  [0x043f] = 0x25123, [0x0440] = 0x2f945, [0x0441] = 0x25124, [0x0442] = 0x25431,
  [0x0443] = 0x25432, [0x0444] = 0x25433, [0x0445] = 0x7818, [0x0446] = 0x25434,
  [0x0447] = 0x25435, [0x0448] = 0x25436, [0x0449] = 0x25440, [0x044a] = 0x25437,
  [0x044c] = 0x25438, [0x044d] = 0x2543d, [0x044e] = 0x25638, [0x044f] = 0x25639,
  [0x0450] = 0x2563a, [0x0451] = 0x2563b, [0x0452] = 0x2563c, [0x0453] = 0x2564c,
  [0x0454] = 0x25773, [0x0455] = 0x25774, [0x0456] = 0x4133, [0x0457] = 0x25775,
  [0x0458] = 0x25776, [0x0459] = 0x25779, [0x045a] = 0x25941, [0x045b] = 0x25942,
  [0x045c] = 0x25943, [0x045d] = 0x25a5b, [0x045f] = 0x7ad7, [0x0460] = 0x238a6,
  [0x0461] = 0x25a5c, [0x0462] = 0x25ae5, [0x0463] = 0x25ae6, [0x0464] = 0x25ae7,
  [0x0465] = 0x25ae8, [0x0466] = 0x25aed, [0x0467] = 0x25e30, [0x0468] = 0x7c7e,
  [0x0469] = 0x25e31, [0x046a] = 0x25f94, [0x046b] = 0x26351, [0x046c] = 0x263fd,
  [0x046d] = 0x263fe, [0x046e] = 0x264e9, [0x0470] = 0x266b3, [0x0471] = 0x26684,
  [0x0472] = 0x26694, [0x0474] = 0x2669c, [0x0476] = 0x26685, [0x0477] = 0x26900,
  [0x0478] = 0x26937, [0x0479] = 0x26957, [0x047a] = 0x26b54, [0x047b] = 0x26b68,
  [0x047c] = 0x26b55, [0x047d] = 0x26b5c, [0x047e] = 0x26b56, [0x047f] = 0x26b57,
  [0x0480] = 0x26b58, [0x0481] = 0x26b59, [0x0482] = 0x26b5a, [0x0483] = 0x26b5b,
  [0x0484] = 0x27234, [0x0485] = 0x867e, [0x0486] = 0x27235, [0x0488] = 0x27239,
  [0x0489] = 0x27236, [0x048a] = 0x27237, [0x048b] = 0x27238, [0x048c] = 0x275e5,
  [0x048d] = 0x27659, [0x048f] = 0x2763a, [0x0490] = 0x2763b, [0x0491] = 0x27d28,
  [0x0492] = 0x27e89, [0x0494] = 0x27e8a, [0x0495] = 0x27e8b, [0x0496] = 0x27e8c,
  [0x0497] = 0x27e8d, [0x0498] = 0x27fbc, [0x0499] = 0x282a3, [0x049a] = 0x4844,
  [0x049b] = 0x282a4, [0x049c] = 0x28493, [0x049d] = 0x28494, [0x049e] = 0x28495,
  [0x049f] = 0x28496, [0x04a0] = 0x28497, [0x04a1] = 0x286b9, [0x04a2] = 0x28e4f,
  [0x04a3] = 0x28e50, [0x04a4] = 0x28e51, [0x04a5] = 0x29076, [0x04a7] = 0x200fb,
  [0x04a8] = 0x20083, [0x04a9] = 0x20138, [0x04aa] = 0x2015d, [0x04ab] = 0x20271,
  [0x04ad] = 0x20277, [0x04ae] = 0x20278, [0x04af] = 0x2f807, [0x04b0] = 0x202c3,
  [0x04b1] = 0x20279, [0x04b2] = 0x2027a, [0x04b3] = 0x2027b, [0x04b4] = 0x2027c,
  [0x04b5] = 0x20272, [0x04b6] = 0x2027d, [0x04b7] = 0x202a0, [0x04b8] = 0x2027e,
  [0x04b9] = 0x2027f, [0x04ba] = 0x20280, [0x04bb] = 0x20281, [0x04bc] = 0x20282,
  [0x04bd] = 0x20283, [0x04be] = 0x2029e, [0x04bf] = 0x20284, [0x04c0] = 0x20285,
  [0x04c1] = 0x20286, [0x04c2] = 0x20287, [0x04c3] = 0x20288, [0x04c4] = 0x20493,
  [0x04c5] = 0x20494, [0x04c6] = 0x20495, [0x04c7] = 0x204ef, [0x04c8] = 0x204f0,
  [0x04ca] = 0x2058d, [0x04cb] = 0x2058f, [0x04cc] = 0x20590, [0x04cd] = 0x20591,
  [0x04ce] = 0x20592, [0x04cf] = 0x205de, [0x04d0] = 0x205e1, [0x04d1] = 0x20653,
  [0x04d2] = 0x5266, [0x04d3] = 0x2074a, [0x04d4] = 0x2074b, [0x04d5] = 0x2074c,
  [0x04d6] = 0x20751, [0x04d7] = 0x2074d, [0x04d8] = 0x20748, [0x04d9] = 0x20874,
  [0x04da] = 0x20875, [0x04db] = 0x20920, [0x04dc] = 0x20921, [0x04dd] = 0x209a5,
  [0x04de] = 0x209a6, [0x04df] = 0x20a73, [0x04e0] = 0x20a74, [0x04e1] = 0x20b50,
  [0x04e2] = 0x20b51, [0x04e3] = 0x20b52, [0x04e4] = 0x20b53, [0x04e5] = 0x20cdb,
  [0x04e6] = 0x20cdc, [0x04e7] = 0x5520, [0x04e8] = 0x20cdd, [0x04e9] = 0x20cde,
  [0x04ea] = 0x20cff, [0x04ed] = 0x20cdf, [0x04ee] = 0x20ce0, [0x04ef] = 0x5521,
  [0x04f0] = 0x20ce1, [0x04f1] = 0x20ce2, [0x04f2] = 0x20ce3, [0x04f3] = 0x20ce4,
  [0x04f4] = 0x20ce5, [0x04f5] = 0x20ce6, [0x04f6] = 0x20ce7, [0x04f7] = 0x20ce8,
  [0x04f8] = 0x20ce9, [0x04f9] = 0x20cea, [0x04fa] = 0x21307, [0x04fb] = 0x21308,
  [0x04fc] = 0x21309, [0x04fd] = 0x2130a, [0x04fe] = 0x2130b, [0x04ff] = 0x2130c,
  [0x0500] = 0x2130d, [0x0501] = 0x57d7, [0x0502] = 0x2131a, [0x0503] = 0x2130e,
  [0x0504] = 0x2130f, [0x0505] = 0x21310, [0x0506] = 0x2131c, [0x0507] = 0x21527,
  [0x0508] = 0x21566, [0x0509] = 0x21598, [0x050a] = 0x21599, [0x050b] = 0x21633,
  [0x050c] = 0x21624, [0x050d] = 0x21625, [0x050e] = 0x21626, [0x0510] = 0x36e1,
  [0x0511] = 0x2173e, [0x0512] = 0x2f864, [0x0513] = 0x2173f, [0x0514] = 0x21757,
  [0x0515] = 0x21740, [0x0516] = 0x21741, [0x0517] = 0x21758, [0x0518] = 0x21742,
  [0x0519] = 0x21743, [0x051a] = 0x21744, [0x051b] = 0x21745, [0x051c] = 0x21746,
  [0x051d] = 0x36e2, [0x051e] = 0x21747, [0x051f] = 0x2196b, [0x0520] = 0x2196c,
  [0x0521] = 0x21a10, [0x0522] = 0x21a11, [0x0523] = 0x5bbe, [0x0524] = 0x21b75,
  [0x0525] = 0x21bef, [0x0526] = 0x21c6a, [0x0527] = 0x21c6b, [0x0529] = 0x21c6c,
  [0x052a] = 0x21c6d, [0x052b] = 0x21c6e, [0x052c] = 0x21def, [0x052d] = 0x21df0,
  [0x052e] = 0x21df1, [0x052f] = 0x21df2, [0x0530] = 0x21df3, [0x0531] = 0x21df4,
  [0x0532] = 0x21df5, [0x0533] = 0x21dff, [0x0534] = 0x21df6, [0x0535] = 0x21e02,
  [0x0536] = 0x21df7, [0x0537] = 0x22022, [0x0538] = 0x220c4, [0x0539] = 0x220c3,
  [0x053a] = 0x22228, [0x053b] = 0x387c, [0x053c] = 0x22320, [0x053d] = 0x22325,
  [0x053e] = 0x22322, [0x053f] = 0x22323, [0x0540] = 0x223eb, [0x0541] = 0x38e3,
  [0x0542] = 0x224f9, [0x0543] = 0x224fa, [0x0544] = 0x22699, [0x0545] = 0x2269a,
  [0x0546] = 0x2269b, [0x0547] = 0x22663, [0x0549] = 0x2265e, [0x054a] = 0x2265f,
  [0x054b] = 0x22660, [0x054c] = 0x2269c, [0x054d] = 0x22661, [0x054e] = 0x2269d,
  [0x054f] = 0x22b54, [0x0550] = 0x22b55, [0x0551] = 0x22b56, [0x0552] = 0x22b57,
  [0x0553] = 0x22b58, [0x0554] = 0x22b59, [0x0555] = 0x22b5a, [0x0556] = 0x22b19,
  [0x0557] = 0x22b6a, [0x0558] = 0x22b5b, [0x0559] = 0x22b5c, [0x055a] = 0x22b5d,
  [0x055b] = 0x22b66, [0x055d] = 0x22b5e, [0x055e] = 0x22b5f, [0x055f] = 0x22b1a,
  [0x0560] = 0x22b60, [0x0561] = 0x22b61, [0x0563] = 0x22b62, [0x0564] = 0x22b1b,
  [0x0565] = 0x22b63, [0x0566] = 0x26410, [0x0567] = 0x26411, [0x0568] = 0x2304c,
  [0x0569] = 0x2309b, [0x056a] = 0x230eb, [0x056b] = 0x230ec, [0x056c] = 0x230ed,
  [0x056d] = 0x230ee, [0x056e] = 0x231a3, [0x056f] = 0x231a8, [0x0570] = 0x231a6,
  [0x0571] = 0x3aec, [0x0572] = 0x2332f, [0x0573] = 0x23330, [0x0574] = 0x23331,
  [0x0575] = 0x23332, [0x0576] = 0x23333, [0x0577] = 0x2344f, [0x0578] = 0x23451,
  [0x0579] = 0x23452, [0x057a] = 0x2347d, [0x057b] = 0x23453, [0x057c] = 0x23454,
  [0x057d] = 0x23455, [0x057e] = 0x23456, [0x057f] = 0x23457, [0x0580] = 0x23458,
  [0x0581] = 0x23459, [0x0582] = 0x2345a, [0x0583] = 0x2345b, [0x0584] = 0x2345c,
  [0x0585] = 0x6857, [0x0586] = 0x2345d, [0x0587] = 0x2345e, [0x0588] = 0x2345f,
  [0x0589] = 0x2346f, [0x058a] = 0x23460, [0x058b] = 0x23461, [0x058c] = 0x23462,
  [0x058d] = 0x23463, [0x058e] = 0x23472, [0x058f] = 0x23464, [0x0591] = 0x2346e,
  [0x0592] = 0x238b5, [0x0593] = 0x23b65, [0x0594] = 0x23b66, [0x0595] = 0x23b67,
  [0x0596] = 0x23b68, [0x0597] = 0x23b69, [0x0598] = 0x23b6a, [0x0599] = 0x23b6b,
  [0x059a] = 0x23b6c, [0x059b] = 0x23d0e, [0x059c] = 0x23d43, [0x059e] = 0x23d0f,
  [0x059f] = 0x23d44, [0x05a0] = 0x23d45, [0x05a1] = 0x23d46, [0x05a2] = 0x23d47,
  [0x05a3] = 0x23d48, [0x05a4] = 0x23d49, [0x05a5] = 0x23d10, [0x05a6] = 0x23d4a,
  [0x05a7] = 0x23d4b, [0x05a8] = 0x23d4c, [0x05a9] = 0x23d4d, [0x05aa] = 0x3cdf,
  [0x05ab] = 0x23d4e, [0x05ac] = 0x23d4f, [0x05ad] = 0x23d50, [0x05ae] = 0x23d51,
  [0x05af] = 0x23d52, [0x05b0] = 0x23d53, [0x05b1] = 0x23d54, [0x05b2] = 0x23d55,
  [0x05b3] = 0x23d56, [0x05b4] = 0x23d57, [0x05b5] = 0x23d58, [0x05b6] = 0x23d59,
  [0x05b7] = 0x23d5a, [0x05b8] = 0x23d1d, [0x05b9] = 0x23d5b, [0x05ba] = 0x23d5c,
  [0x05bb] = 0x23d5d, [0x05bc] = 0x2f906, [0x05be] = 0x241fe, [0x05bf] = 0x241ff,
  [0x05c0] = 0x24200, [0x05c1] = 0x24201, [0x05c2] = 0x24202, [0x05c3] = 0x24203,
  [0x05c4] = 0x24204, [0x05c5] = 0x24205, [0x05c6] = 0x24206, [0x05c7] = 0x24207,
  [0x05c8] = 0x24218, [0x05c9] = 0x24208, [0x05ca] = 0x24209, [0x05cb] = 0x241f3,
  [0x05cc] = 0x2420a, [0x05cd] = 0x2420b, [0x05ce] = 0x2421b, [0x05cf] = 0x2420c,
  [0x05d0] = 0x2420d, [0x05d1] = 0x2420e, [0x05d2] = 0x2420f, [0x05d3] = 0x24210,
  [0x05d4] = 0x2450f, [0x05d5] = 0x24561, [0x05d6] = 0x24579, [0x05d7] = 0x247c1,
  [0x05d8] = 0x247ac, [0x05d9] = 0x247ad, [0x05da] = 0x24943, [0x05db] = 0x24944,
  [0x05dc] = 0x24945, [0x05dd] = 0x24946, [0x05de] = 0x24947, [0x05df] = 0x24948,
  [0x05e0] = 0x24949, [0x05e1] = 0x24964, [0x05e2] = 0x2494a, [0x05e3] = 0x2494b,
  [0x05e4] = 0x2494c, [0x05e5] = 0x2494d, [0x05e6] = 0x3ec7, [0x05e7] = 0x2f92d,
  [0x05e8] = 0x24b39, [0x05e9] = 0x24c57, [0x05ea] = 0x24c58, [0x05eb] = 0x24c59,
  [0x05ec] = 0x24c5f, [0x05ed] = 0x24c5a, [0x05ee] = 0x24d6a, [0x05ef] = 0x24f2b,
  [0x05f0] = 0x24f5d, [0x05f1] = 0x24f5e, [0x05f2] = 0x24f5f, [0x05f4] = 0x24f60,
  [0x05f5] = 0x24f61, [0x05f6] = 0x25058, [0x05f7] = 0x25059, [0x05f8] = 0x2514a,
  [0x05f9] = 0x2514b, [0x05fa] = 0x2514c, [0x05fb] = 0x2514d, [0x05fc] = 0x2514e,
  [0x05fd] = 0x25457, [0x05fe] = 0x253b3, [0x05ff] = 0x253b4, [0x0600] = 0x2546e,
  [0x0601] = 0x2545b, [0x0602] = 0x2545c, [0x0604] = 0x2545d, [0x0605] = 0x2545e,
  [0x0607] = 0x2546c, [0x0608] = 0x2563d, [0x0609] = 0x25654, [0x060a] = 0x2563e,
  [0x060b] = 0x25655, [0x060c] = 0x25656, [0x060d] = 0x2563f, [0x060e] = 0x25657,
  [0x060f] = 0x25658, [0x0610] = 0x25659, [0x0611] = 0x2565a, [0x0612] = 0x2565b,
  [0x0613] = 0x2565c, [0x0614] = 0x25788, [0x0615] = 0x25789, [0x0617] = 0x25958,
  [0x0618] = 0x25959, [0x0619] = 0x2595a, [0x061b] = 0x25a67, [0x061c] = 0x25a68,
  [0x061d] = 0x25a69, [0x061e] = 0x25a6a, [0x0620] = 0x25a6b, [0x0621] = 0x25a6c,
  [0x0622] = 0x25afe, [0x0623] = 0x25aff, [0x0624] = 0x25b00, [0x0625] = 0x25b01,
  [0x0626] = 0x25b02, [0x0627] = 0x25b03, [0x0628] = 0x25b04, [0x0629] = 0x25b05,
  [0x062a] = 0x25e39, [0x062b] = 0x25e3a, [0x062c] = 0x25e3b, [0x062d] = 0x25e3c,
  [0x0630] = 0x25fb2, [0x0631] = 0x25fb3, [0x0632] = 0x25fb4, [0x0633] = 0x25fb5,
  [0x0634] = 0x25fb6, [0x0635] = 0x25fb7, [0x0636] = 0x25fb8, [0x0637] = 0x7f3c,
  [0x0638] = 0x262a2, [0x0639] = 0x26361, [0x063a] = 0x26362, [0x063b] = 0x26414,
  [0x063d] = 0x264eb, [0x063e] = 0x26542, [0x063f] = 0x26548, [0x0641] = 0x26543,
  [0x0642] = 0x2654e, [0x0643] = 0x26544, [0x0644] = 0x26545, [0x0645] = 0x266b4,
  [0x0646] = 0x266b5, [0x064a] = 0x266b6, [0x064c] = 0x268e1, [0x064d] = 0x26905,
  [0x064f] = 0x8273, [0x0650] = 0x26b9a, [0x0652] = 0x26b9b, [0x0653] = 0x26b9c,
  [0x0654] = 0x26b9d, [0x0655] = 0x26b9e, [0x0656] = 0x26b9f, [0x0657] = 0x2f99a,
  [0x0658] = 0x26ba0, [0x0659] = 0x26ba1, [0x065a] = 0x26ba2, [0x065b] = 0x26bb5,
  [0x065c] = 0x26ba3, [0x065d] = 0x26ba4, [0x065f] = 0x26ba5, [0x0660] = 0x26ba6,
  [0x0661] = 0x26ba7, [0x0662] = 0x26ba8, [0x0663] = 0x26baf, [0x0664] = 0x26ba9,
  [0x0665] = 0x26baa, [0x0666] = 0x26bab, [0x0667] = 0x26bac, [0x0668] = 0x26bad,
  [0x0669] = 0x26bb3, [0x066b] = 0x271a0, [0x066c] = 0x27256, [0x066d] = 0x27634,
  [0x066e] = 0x4627, [0x066f] = 0x2763c, [0x0670] = 0x2780c, [0x0672] = 0x27c3e,
  [0x0673] = 0x27ca3, [0x0674] = 0x27d30, [0x0675] = 0x27d31, [0x0676] = 0x27e97,
  [0x0677] = 0x27e98, [0x0678] = 0x27e99, [0x0679] = 0x27e9a, [0x067a] = 0x27e9b,
  [0x067b] = 0x282b2, [0x067c] = 0x282ab, [0x067d] = 0x282ac, [0x067e] = 0x286fc,
  [0x067f] = 0x286e8, [0x0680] = 0x286e9, [0x0681] = 0x286ea, [0x0682] = 0x287f1,
  [0x0683] = 0x2893f, [0x0684] = 0x28cce, [0x0685] = 0x28e6d, [0x0686] = 0x28e6e,
  [0x0687] = 0x28e6f, [0x0688] = 0x28e70, [0x0689] = 0x28e71, [0x068a] = 0x28e72,
  [0x068b] = 0x28e73, [0x068c] = 0x96be, [0x068d] = 0x29078, [0x068e] = 0x2983a,
  [0x068f] = 0x20167, [0x0690] = 0x66fa, [0x0691] = 0x20164, [0x0692] = 0x20165,
  [0x0693] = 0x20307, [0x0694] = 0x202c4, [0x0695] = 0x202c5, [0x0696] = 0x202c6,
  [0x0697] = 0x202c7, [0x0698] = 0x202c8, [0x0699] = 0x202c9, [0x069a] = 0x202e5,
  [0x069b] = 0x202ca, [0x069c] = 0x202cb, [0x069e] = 0x202cc, [0x069f] = 0x202cd,
  [0x06a0] = 0x202ce, [0x06a1] = 0x202cf, [0x06a2] = 0x202d0, [0x06a3] = 0x20497,
  [0x06a4] = 0x2049a, [0x06a5] = 0x2049b, [0x06a6] = 0x205ea, [0x06a7] = 0x205ec,
  [0x06a8] = 0x205ed, [0x06a9] = 0x205f7, [0x06aa] = 0x205ee, [0x06ab] = 0x205ef,
  [0x06ac] = 0x2065b, [0x06ad] = 0x2065c, [0x06ae] = 0x2065d, [0x06af] = 0x2076d,
  [0x06b0] = 0x2076e, [0x06b2] = 0x2076f, [0x06b3] = 0x20881, [0x06b4] = 0x20882,
  [0x06b5] = 0x2094f, [0x06b6] = 0x209ad, [0x06b7] = 0x2870a, [0x06b8] = 0x20a7e,
  [0x06b9] = 0x20b69, [0x06ba] = 0x20d54, [0x06bb] = 0x20d55, [0x06bc] = 0x20d56,
  [0x06bd] = 0x20d57, [0x06be] = 0x35ab, [0x06bf] = 0x20d58, [0x06c0] = 0x20d59,
  [0x06c1] = 0x20d5a, [0x06c2] = 0x20d5b, [0x06c3] = 0x20d5c, [0x06c4] = 0x20d5d,
  [0x06c7] = 0x20d5e, [0x06c8] = 0x20d5f, [0x06c9] = 0x2133b, [0x06ca] = 0x21336,
  [0x06cb] = 0x2133c, [0x06cc] = 0x2133d, [0x06cd] = 0x2133e, [0x06ce] = 0x2133f,
  [0x06cf] = 0x21340, [0x06d0] = 0x21341, [0x06d1] = 0x21342, [0x06d2] = 0x364b,
  [0x06d3] = 0x21343, [0x06d4] = 0x21344, [0x06d5] = 0x21345, [0x06d6] = 0x21337,
  [0x06d9] = 0x21346, [0x06da] = 0x21347, [0x06db] = 0x21348, [0x06dc] = 0x21349,
  [0x06dd] = 0x2152d, [0x06de] = 0x215a5, [0x06df] = 0x2163d, [0x06e0] = 0x2163f,
  [0x06e1] = 0x2196e, [0x06e2] = 0x217e7, [0x06e3] = 0x21775, [0x06e4] = 0x21776,
  [0x06e5] = 0x21777, [0x06e6] = 0x21778, [0x06e7] = 0x21779, [0x06e8] = 0x2177a,
  [0x06e9] = 0x2177b, [0x06ea] = 0x2177c, [0x06eb] = 0x2178e, [0x06ec] = 0x2177d,
  [0x06ed] = 0x2177e, [0x06ee] = 0x5a72, [0x06ef] = 0x2177f, [0x06f0] = 0x21780,
  [0x06f1] = 0x21781, [0x06f2] = 0x21782, [0x06f3] = 0x21783, [0x06f4] = 0x21784,
  [0x06f5] = 0x21785, [0x06f6] = 0x21786, [0x06f7] = 0x21787, [0x06f8] = 0x21788,
  [0x06f9] = 0x21789, [0x06fa] = 0x2178a, [0x06fb] = 0x2178b, [0x06fc] = 0x2178c,
  [0x06fd] = 0x2196f, [0x06fe] = 0x21970, [0x06ff] = 0x21971, [0x0700] = 0x21972,
  [0x0701] = 0x21a2a, [0x0702] = 0x21a2b, [0x0703] = 0x21a2c, [0x0704] = 0x21a2d,
  [0x0705] = 0x21a2e, [0x0706] = 0x21a2f, [0x0707] = 0x21a30, [0x0708] = 0x21a31,
  [0x0709] = 0x21b83, [0x070a] = 0x21c84, [0x070b] = 0x21c85, [0x070c] = 0x21c93,
  [0x070e] = 0x21c86, [0x070f] = 0x21c87, [0x0710] = 0x21c88, [0x0711] = 0x21c89,
  [0x0712] = 0x21c8a, [0x0713] = 0x21c8b, [0x0714] = 0x21e36, [0x0715] = 0x21e38,
  [0x0716] = 0x21e39, [0x0717] = 0x21e3a, [0x0718] = 0x37e2, [0x0719] = 0x21e3b,
  [0x071a] = 0x21e3c, [0x071b] = 0x21e3d, [0x071c] = 0x21e3e, [0x071d] = 0x21e3f,
  [0x071f] = 0x21e40, [0x0720] = 0x21e41, [0x0721] = 0x21e42, [0x0722] = 0x220de,
  [0x0723] = 0x220df, [0x0724] = 0x220e0, [0x0725] = 0x220e1, [0x0726] = 0x2224a,
  [0x0727] = 0x2224b, [0x0728] = 0x2224c, [0x0729] = 0x2224d, [0x072a] = 0x2224e,
  [0x072b] = 0x2224f, [0x072c] = 0x22250, [0x072d] = 0x22327, [0x072e] = 0x223f9,
  [0x072f] = 0x223fa, [0x0730] = 0x223fb, [0x0731] = 0x22498, [0x0732] = 0x22499,
  [0x0733] = 0x2249a, [0x0735] = 0x2251e, [0x0736] = 0x2269e, [0x0737] = 0x226f7,
  [0x0738] = 0x2269f, [0x0739] = 0x226a0, [0x073a] = 0x226f0, [0x073b] = 0x226f8,
  [0x073c] = 0x226f9, [0x073d] = 0x226a1, [0x073e] = 0x226a2, [0x073f] = 0x226a3,
  [0x0740] = 0x226a4, [0x0741] = 0x226a5, [0x0742] = 0x226fa, [0x0743] = 0x226fb,
  [0x0744] = 0x226fc, [0x0745] = 0x226fd, [0x0746] = 0x226fe, [0x0747] = 0x2271f,
  [0x0748] = 0x226ff, [0x0749] = 0x22700, [0x074a] = 0x2278c, [0x074b] = 0x22701,
  [0x074c] = 0x229c2, [0x074d] = 0x229c3, [0x074f] = 0x229d3, [0x0750] = 0x22a4e,
  [0x0751] = 0x22baf, [0x0752] = 0x22bb0, [0x0753] = 0x22bb1, [0x0754] = 0x22bb2,
  [0x0755] = 0x22bb3, [0x0756] = 0x22bc6, [0x0757] = 0x22bb4, [0x0759] = 0x22bb5,
  [0x075a] = 0x22bb6, [0x075b] = 0x22bb7, [0x075c] = 0x22bb8, [0x075d] = 0x22bb9,
  [0x075e] = 0x22bba, [0x075f] = 0x22bbb, [0x0760] = 0x22bbc, [0x0761] = 0x22bbd,
  [0x0762] = 0x22bbe, [0x0763] = 0x22bbf, [0x0764] = 0x22bc1, [0x0765] = 0x22bc0,
  [0x0766] = 0x22bc2, [0x0767] = 0x22bc3, [0x0768] = 0x22f51, [0x0769] = 0x22f52,
  [0x076a] = 0x22f53, [0x076b] = 0x22f54, [0x076c] = 0x22f55, [0x076d] = 0x2305a,
  [0x076e] = 0x23050, [0x076f] = 0x23051, [0x0770] = 0x23078, [0x0771] = 0x230f8,
  [0x0772] = 0x230f9, [0x0773] = 0x231c6, [0x0774] = 0x231cb, [0x0775] = 0x231cc,
  [0x0776] = 0x231cd, [0x0777] = 0x231ce, [0x0778] = 0x231cf, [0x0779] = 0x231d0,
  [0x077a] = 0x231d1, [0x077b] = 0x231d2, [0x077c] = 0x231d3, [0x077d] = 0x231d4,
  [0x077e] = 0x231d5, [0x0780] = 0x23337, [0x0781] = 0x23338, [0x0782] = 0x23373,
  [0x0783] = 0x23490, [0x0784] = 0x23491, [0x0785] = 0x234bf, [0x0786] = 0x23492,
  [0x0787] = 0x23493, [0x0788] = 0x23494, [0x0789] = 0x23495, [0x078a] = 0x23496,
  [0x078b] = 0x23497, [0x078c] = 0x23498, [0x078d] = 0x23499, [0x078e] = 0x68bd,
  [0x078f] = 0x2349a, [0x0790] = 0x2349b, [0x0791] = 0x2349c, [0x0792] = 0x2349d,
  [0x0793] = 0x2349e, [0x0794] = 0x2349f, [0x0795] = 0x234a0, [0x0796] = 0x234a1,
  [0x0797] = 0x234a2, [0x0798] = 0x234a3, [0x0799] = 0x234a4, [0x079a] = 0x234a5,
  [0x079c] = 0x234a6, [0x079d] = 0x234a7, [0x079e] = 0x238bf, [0x079f] = 0x238c0,
  [0x07a0] = 0x238c1, [0x07a1] = 0x238c2, [0x07a2] = 0x238c3, [0x07a3] = 0x238c4,
  [0x07a4] = 0x238c5, [0x07a5] = 0x238c6, [0x07a6] = 0x23975, [0x07a7] = 0x23a43,
  [0x07a8] = 0x23a12, [0x07a9] = 0x23a9f, [0x07aa] = 0x23a91, [0x07ab] = 0x2f8f6,
  [0x07ac] = 0x23a92, [0x07ad] = 0x23a93, [0x07ae] = 0x23a94, [0x07af] = 0x23b7e,
  [0x07b0] = 0x23b7f, [0x07b1] = 0x23b80, [0x07b2] = 0x23b81, [0x07b3] = 0x23b82,
  [0x07b4] = 0x23c4c, [0x07b5] = 0x23c63, [0x07b6] = 0x23dc0, [0x07b7] = 0x23dc1,
  [0x07b8] = 0x23dc2, [0x07b9] = 0x2423a, [0x07ba] = 0x23dc3, [0x07bb] = 0x23dc4,
  [0x07bc] = 0x23dc5, [0x07bd] = 0x23dc6, [0x07be] = 0x6e15, [0x07bf] = 0x23dc7,
  [0x07c1] = 0x23dc8, [0x07c2] = 0x23d73, [0x07c3] = 0x23dc9, [0x07c4] = 0x23dca,
  [0x07c5] = 0x23dcb, [0x07c6] = 0x23dcc, [0x07c7] = 0x23dcd, [0x07c8] = 0x3d0e,
  [0x07cb] = 0x23dce, [0x07cc] = 0x23dcf, [0x07cd] = 0x23dd0, [0x07ce] = 0x23dd1,
  [0x07cf] = 0x23dd2, [0x07d0] = 0x23dd3, [0x07d1] = 0x23dd4, [0x07d2] = 0x23dd5,
  [0x07d3] = 0x2423b, [0x07d4] = 0x2423c, [0x07d5] = 0x2423d, [0x07d6] = 0x2423e,
  [0x07d8] = 0x2423f, [0x07d9] = 0x24240, [0x07da] = 0x24241, [0x07db] = 0x24242,
  [0x07dc] = 0x24243, [0x07dd] = 0x3dc1, [0x07de] = 0x24244, [0x07df] = 0x24245,
  [0x07e0] = 0x24259, [0x07e1] = 0x24246, [0x07e2] = 0x24247, [0x07e3] = 0x24248,
  [0x07e5] = 0x24249, [0x07e6] = 0x24513, [0x07e7] = 0x24582, [0x07e8] = 0x24583,
  [0x07e9] = 0x24666, [0x07ea] = 0x24667, [0x07eb] = 0x24965, [0x07ec] = 0x7413,
  [0x07ed] = 0x24966, [0x07ee] = 0x3ed9, [0x07ef] = 0x24975, [0x07f0] = 0x24974,
  [0x07f1] = 0x24967, [0x07f2] = 0x2496d, [0x07f3] = 0x24976, [0x07f4] = 0x24968,
  [0x07f5] = 0x24969, [0x07f6] = 0x2496a, [0x07f7] = 0x2496b, [0x07f8] = 0x2494f,
  [0x07f9] = 0x2496c, [0x07fa] = 0x24af8, [0x07fb] = 0x24b45, [0x07fc] = 0x74f8,
  [0x07fd] = 0x24b46, [0x07fe] = 0x24be4, [0x07ff] = 0x24c85, [0x0800] = 0x24c6f,
  [0x0801] = 0x24c70, [0x0802] = 0x24d82, [0x0803] = 0x24d83, [0x0804] = 0x24d84,
  [0x0805] = 0x24d8b, [0x0806] = 0x24f2d, [0x0807] = 0x24f68, [0x0808] = 0x24f69,
  [0x0809] = 0x24f77, [0x080a] = 0x24f64, [0x080b] = 0x24f6a, [0x080d] = 0x24ff2,
  [0x080e] = 0x24fe2, [0x080f] = 0x25063, [0x0810] = 0x25064, [0x0811] = 0x25065,
  [0x0812] = 0x25066, [0x0813] = 0x7b3d, [0x0814] = 0x25067, [0x0815] = 0x25068,
  [0x0816] = 0x76d8, [0x0818] = 0x25172, [0x0819] = 0x2517d, [0x081a] = 0x25369,
  [0x081b] = 0x25484, [0x081c] = 0x25486, [0x081d] = 0x25487, [0x081e] = 0x25488,
  [0x081f] = 0x2549e, [0x0821] = 0x25489, [0x0822] = 0x2548a, [0x0823] = 0x25496,
  [0x0824] = 0x2548b, [0x0825] = 0x2548c, [0x0826] = 0x2548d, [0x0827] = 0x25679,
  [0x0828] = 0x25683, [0x0829] = 0x2567a, [0x082a] = 0x2565d, [0x082b] = 0x2567b,
  [0x082d] = 0x2579f, [0x082e] = 0x4148, [0x082f] = 0x257a0, [0x0830] = 0x257a1,
  [0x0831] = 0x257a2, [0x0832] = 0x79fc, [0x0833] = 0x2596a, [0x0834] = 0x2596b,
  [0x0835] = 0x2596c, [0x0836] = 0x2596d, [0x0837] = 0x2596e, [0x0838] = 0x25a78,
  [0x083a] = 0x25a79, [0x083b] = 0x25a7c, [0x083c] = 0x25b21, [0x083d] = 0x25b22,
  [0x083e] = 0x25b23, [0x083f] = 0x25b24, [0x0840] = 0x7b39, [0x0841] = 0x25b25,
  [0x0842] = 0x25b26, [0x0843] = 0x25e51, [0x0844] = 0x25e52, [0x0845] = 0x25e56,
  [0x0847] = 0x7d4b, [0x0848] = 0x25fe1, [0x084b] = 0x25fd6, [0x084c] = 0x25fd7,
  [0x084d] = 0x25fd8, [0x084e] = 0x25fd9, [0x084f] = 0x262b5, [0x0850] = 0x2636c,
  [0x0851] = 0x2636f, [0x0852] = 0x26370, [0x0853] = 0x264bc, [0x0854] = 0x264bd,
  [0x0855] = 0x264ed, [0x0856] = 0x2655b, [0x0857] = 0x2655c, [0x0858] = 0x2655d,
  [0x0859] = 0x2655e, [0x085a] = 0x26686, [0x085b] = 0x26687, [0x085c] = 0x266ed,
  [0x085e] = 0x266ee, [0x0860] = 0x2690e, [0x0861] = 0x2690f, [0x0863] = 0x26a2d,
  [0x0864] = 0x26a2a, [0x0865] = 0x26a2e, [0x0866] = 0x26acc, [0x0867] = 0x26bf9,
  [0x0868] = 0x26bfa, [0x0869] = 0x26bfb, [0x086a] = 0x26bfc, [0x086b] = 0x83b9,
  [0x086c] = 0x26bfd, [0x086d] = 0x26c1b, [0x086e] = 0x26bfe, [0x086f] = 0x26bff,
  [0x0870] = 0x26c00, [0x0871] = 0x26c01, [0x0872] = 0x26c02, [0x0873] = 0x26c03,
  [0x0874] = 0x26c21, [0x0875] = 0x26c04, [0x0876] = 0x26c05, [0x0877] = 0x26c06,
  [0x0878] = 0x26c07, [0x0879] = 0x26c08, [0x087a] = 0x26c09, [0x087b] = 0x26c0a,
  [0x087c] = 0x26c0b, [0x087d] = 0x26c0c, [0x087e] = 0x26c0d, [0x087f] = 0x26c0e,
  [0x0880] = 0x26c0f, [0x0881] = 0x26c10, [0x0882] = 0x26c11, [0x0883] = 0x26c12,
  [0x0884] = 0x26c13, [0x0885] = 0x26c14, [0x0886] = 0x26c15, [0x0887] = 0x2727b,
  [0x0888] = 0x2727c, [0x0889] = 0x2727d, [0x088a] = 0x2727e, [0x088b] = 0x86cf,
  [0x088c] = 0x2727f, [0x088d] = 0x27280, [0x088e] = 0x27266, [0x088f] = 0x27281,
  [0x0890] = 0x27282, [0x0891] = 0x27283, [0x0892] = 0x275b2, [0x0894] = 0x2767f,
  [0x0895] = 0x27672, [0x0896] = 0x2765a, [0x0897] = 0x277eb, [0x0898] = 0x27815,
  [0x089a] = 0x27816, [0x089b] = 0x2f9ca, [0x089c] = 0x27817, [0x089e] = 0x278c5,
  [0x089f] = 0x278c6, [0x08a0] = 0x278c7, [0x08a1] = 0x27988, [0x08a2] = 0x27989,
  [0x08a4] = 0x2798a, [0x08a5] = 0x2798b, [0x08a6] = 0x2798c, [0x08a7] = 0x2798d,
  [0x08a8] = 0x27bdf, [0x08aa] = 0x27d3b, [0x08ab] = 0x27d3c, [0x08ac] = 0x27d3d,
  [0x08ad] = 0x27d3e, [0x08ae] = 0x27d3f, [0x08af] = 0x27d40, [0x08b0] = 0x27d41,
  [0x08b1] = 0x27d42, [0x08b2] = 0x27e5c, [0x08b3] = 0x27eaf, [0x08b4] = 0x27ea9,
  [0x08b5] = 0x27eaa, [0x08b6] = 0x27eab, [0x08b7] = 0x27eac, [0x08b8] = 0x27eb5,
  [0x08b9] = 0x27ead, [0x08ba] = 0x27eae, [0x08bb] = 0x27fe2, [0x08bc] = 0x28224,
  [0x08bd] = 0x8eae, [0x08be] = 0x282c4, [0x08c0] = 0x284ea, [0x08c1] = 0x284d3,
  [0x08c2] = 0x284d4, [0x08c3] = 0x284d5, [0x08c4] = 0x28686, [0x08c6] = 0x28687,
  [0x08c7] = 0x286fd, [0x08c8] = 0x28805, [0x08c9] = 0x28949, [0x08ca] = 0x28c5f,
  [0x08cb] = 0x28cd3, [0x08cc] = 0x28cd5, [0x08cd] = 0x28e8a, [0x08ce] = 0x28e8b,
  [0x08cf] = 0x28e8c, [0x08d0] = 0x28e8d, [0x08d1] = 0x28e8e, [0x08d2] = 0x28e8f,
  [0x08d3] = 0x28e90, [0x08d4] = 0x28e91, [0x08d5] = 0x28e92, [0x08d6] = 0x28e93,
  [0x08d7] = 0x2907d, [0x08d8] = 0x2907e, [0x08d9] = 0x96eb, [0x08dc] = 0x291ec,
  [0x08dd] = 0x2925b, [0x08de] = 0x29c87, [0x08df] = 0x29c86, [0x08e1] = 0x20046,
  [0x08e2] = 0x20047, [0x08e3] = 0x20102, [0x08e4] = 0x2016d, [0x08e5] = 0x2016e,
  [0x08e6] = 0x2016f, [0x08e7] = 0x2030d, [0x08e8] = 0x2030e, [0x08e9] = 0x2030f,
  [0x08ea] = 0x20310, [0x08eb] = 0x20311, [0x08ec] = 0x20312, [0x08ed] = 0x20313,
  [0x08ee] = 0x20314, [0x08ef] = 0x2049f, [0x08f0] = 0x204a0, [0x08f1] = 0x204a1,
  [0x08f2] = 0x204a3, [0x08f3] = 0x20521, [0x08f4] = 0x20522, [0x08f5] = 0x2337c,
  [0x08f6] = 0x20595, [0x08f7] = 0x20596, [0x08f8] = 0x20661, [0x08f9] = 0x20667,
  [0x08fa] = 0x20791, [0x08fb] = 0x20792, [0x08fc] = 0x20793, [0x08fd] = 0x20794,
  [0x08fe] = 0x2088a, [0x08ff] = 0x2088b, [0x0900] = 0x208f7, [0x0901] = 0x20924,
  [0x0902] = 0x209b0, [0x0903] = 0x209b1, [0x0904] = 0x20a8e, [0x0905] = 0x20b04,
  [0x0906] = 0x20b6a, [0x0907] = 0x20b6b, [0x0908] = 0x20b6c, [0x0909] = 0x20b6d,
  [0x090a] = 0x20de8, [0x090b] = 0x20e0c, [0x090c] = 0x20de9, [0x090d] = 0x20dea,
  [0x090e] = 0x20e0d, [0x0910] = 0x20deb, [0x0911] = 0x20dec, [0x0912] = 0x20ded,
  [0x0913] = 0x20dee, [0x0914] = 0x20def, [0x0915] = 0x20df0, [0x0916] = 0x20df1,
  [0x0917] = 0x20df2, [0x0918] = 0x20e16, [0x0919] = 0x20df3, [0x091a] = 0x20e04,
  [0x091b] = 0x20e1d, [0x091c] = 0x20df4, [0x091d] = 0x20df5, [0x091e] = 0x20df6,
  [0x091f] = 0x20df7, [0x0920] = 0x55b0, [0x0921] = 0x21202, [0x0922] = 0x21203,
  [0x0923] = 0x21204, [0x0924] = 0x2139a, [0x0925] = 0x2137c, [0x0926] = 0x213c5,
  [0x0927] = 0x2137d, [0x0928] = 0x2137e, [0x0929] = 0x2137f, [0x092a] = 0x21380,
  [0x092b] = 0x21381, [0x092c] = 0x5840, [0x092d] = 0x5842, [0x092e] = 0x21382,
  [0x092f] = 0x21383, [0x0930] = 0x21384, [0x0931] = 0x21385, [0x0932] = 0x21386,
  [0x0933] = 0x21387, [0x0934] = 0x215ae, [0x0935] = 0x2164d, [0x0936] = 0x217b0,
  [0x0937] = 0x217dc, [0x0938] = 0x217b6, [0x0939] = 0x217b7, [0x093a] = 0x217b8,
  [0x093b] = 0x217b9, [0x093c] = 0x217ba, [0x093d] = 0x217bb, [0x093e] = 0x217db,
  [0x093f] = 0x217bc, [0x0940] = 0x217bd, [0x0941] = 0x217be, [0x0943] = 0x217bf,
  [0x0944] = 0x217c0, [0x0945] = 0x217c1, [0x0946] = 0x217c2, [0x0947] = 0x217c3,
  [0x0948] = 0x217ce, [0x0949] = 0x3701, [0x094a] = 0x217d9, [0x094b] = 0x217c4,
  [0x094c] = 0x217c5, [0x094d] = 0x217da, [0x094e] = 0x217c6, [0x094f] = 0x217c7,
  [0x0950] = 0x217c8, [0x0951] = 0x217c9, [0x0952] = 0x217ca, [0x0953] = 0x2197e,
  [0x0954] = 0x2197b, [0x0955] = 0x21a4e, [0x0956] = 0x21a4f, [0x0957] = 0x21a50,
  [0x0958] = 0x21a51, [0x0959] = 0x21a52, [0x095a] = 0x21a53, [0x095b] = 0x21a54,
  [0x095c] = 0x21a55, [0x095d] = 0x21b8f, [0x095e] = 0x21bfc, [0x095f] = 0x21ca6,
  [0x0960] = 0x21ca7, [0x0961] = 0x21ca8, [0x0962] = 0x21d23, [0x0963] = 0x21e8b,
  [0x0965] = 0x21e8c, [0x0966] = 0x21e8d, [0x0967] = 0x21e8e, [0x0968] = 0x21e8f,
  [0x0969] = 0x21ea4, [0x096a] = 0x20599, [0x096b] = 0x22024, [0x096c] = 0x22046,
  [0x096d] = 0x22047, [0x096e] = 0x22048, [0x096f] = 0x220fd, [0x0970] = 0x220fe,
  [0x0972] = 0x2226e, [0x0973] = 0x2227c, [0x0974] = 0x2226f, [0x0975] = 0x22270,
  [0x0976] = 0x22271, [0x0977] = 0x22272, [0x0978] = 0x22329, [0x0979] = 0x2232a,
  [0x097a] = 0x2235f, [0x097b] = 0x224a2, [0x097c] = 0x22530, [0x097d] = 0x2252b,
  [0x097e] = 0x22761, [0x097f] = 0x22702, [0x0980] = 0x22703, [0x0981] = 0x22704,
  [0x0982] = 0x22705, [0x0983] = 0x22706, [0x0984] = 0x22707, [0x0986] = 0x22708,
  [0x0987] = 0x22709, [0x0988] = 0x22762, [0x0989] = 0x2270a, [0x098a] = 0x2270b,
  [0x098b] = 0x2270c, [0x098c] = 0x2270d, [0x098d] = 0x2270e, [0x098e] = 0x22763,
  [0x098f] = 0x22727, [0x0990] = 0x2270f, [0x0992] = 0x2271b, [0x0993] = 0x22764,
  [0x0994] = 0x22710, [0x0995] = 0x22711, [0x0996] = 0x22781, [0x0997] = 0x22712,
  [0x0998] = 0x22765, [0x0999] = 0x22713, [0x099a] = 0x229ce, [0x099b] = 0x22c29,
  [0x099c] = 0x22bc4, [0x099d] = 0x22c2a, [0x099f] = 0x22c2b, [0x09a0] = 0x22c2c,
  [0x09a1] = 0x22c2d, [0x09a2] = 0x22c2e, [0x09a3] = 0x22c2f, [0x09a4] = 0x22c30,
  [0x09a5] = 0x22c31, [0x09a6] = 0x22c32, [0x09a7] = 0x22c33, [0x09a8] = 0x22c34,
  [0x09aa] = 0x22c35, [0x09ab] = 0x22c36, [0x09ac] = 0x22be3, [0x09ad] = 0x22c37,
  [0x09ae] = 0x22c38, [0x09af] = 0x22c39, [0x09b0] = 0x22c3a, [0x09b1] = 0x22c3b,
  [0x09b2] = 0x22c4d, [0x09b3] = 0x22c3c, [0x09b4] = 0x22c3d, [0x09b5] = 0x22c3e,
  [0x09b7] = 0x22ed0, [0x09b8] = 0x22ed1, [0x09b9] = 0x23055, [0x09ba] = 0x23102,
  [0x09bb] = 0x23104, [0x09bc] = 0x231fc, [0x09bd] = 0x23207, [0x09be] = 0x231fd,
  [0x09bf] = 0x231fe, [0x09c1] = 0x231ff, [0x09c2] = 0x23200, [0x09c3] = 0x23208,
  [0x09c4] = 0x23201, [0x09c5] = 0x23202, [0x09c6] = 0x23235, [0x09c7] = 0x23203,
  [0x09c8] = 0x23204, [0x09ca] = 0x2333b, [0x09cb] = 0x23389, [0x09cc] = 0x23381,
  [0x09cd] = 0x23382, [0x09ce] = 0x26733, [0x09cf] = 0x234eb, [0x09d0] = 0x234ec,
  [0x09d2] = 0x234ed, [0x09d3] = 0x234ee, [0x09d4] = 0x234ef, [0x09d6] = 0x234f0,
  [0x09d7] = 0x234f1, [0x09d8] = 0x234f2, [0x09d9] = 0x692b, [0x09da] = 0x234f3,
  [0x09db] = 0x6916, [0x09dc] = 0x234f4, [0x09dd] = 0x234f5, [0x09de] = 0x2004b,
  [0x09df] = 0x234f6, [0x09e0] = 0x234f7, [0x09e1] = 0x234f8, [0x09e2] = 0x234f9,
  [0x09e3] = 0x234fa, [0x09e4] = 0x234fb, [0x09e5] = 0x234fc, [0x09e6] = 0x234fd,
  [0x09e7] = 0x234fe, [0x09e8] = 0x234ff, [0x09e9] = 0x23500, [0x09ea] = 0x691b,
  [0x09eb] = 0x23501, [0x09ec] = 0x23502, [0x09ed] = 0x23503, [0x09ee] = 0x23504,
  [0x09ef] = 0x23505, [0x09f0] = 0x23506, [0x09f1] = 0x23507, [0x09f2] = 0x6927,
  [0x09f3] = 0x23508, [0x09f4] = 0x23509, [0x09f5] = 0x2350a, [0x09f6] = 0x2350b,
  [0x09f7] = 0x2350c, [0x09f8] = 0x2350d, [0x09f9] = 0x238d7, [0x09fa] = 0x238d8,
  [0x09fb] = 0x23a29, [0x09fc] = 0x23a9c, [0x09fd] = 0x23af8, [0x09fe] = 0x23b98,
  [0x09ff] = 0x23ba4, [0x0a00] = 0x23b99, [0x0a01] = 0x23b9a, [0x0a02] = 0x23b9b,
  [0x0a03] = 0x23b97, [0x0a04] = 0x23b9c, [0x0a05] = 0x6bf5, [0x0a06] = 0x23b9d,
  [0x0a08] = 0x23b9e, [0x0a09] = 0x23b9f, [0x0a0a] = 0x23c65, [0x0a0b] = 0x23dd6,
  [0x0a0c] = 0x3d20, [0x0a0d] = 0x23e32, [0x0a0e] = 0x23e33, [0x0a0f] = 0x23e34,
  [0x0a10] = 0x6e82, [0x0a11] = 0x23e35, [0x0a12] = 0x23e36, [0x0a13] = 0x23e37,
  [0x0a14] = 0x23dd7, [0x0a16] = 0x23dd8, [0x0a17] = 0x23e38, [0x0a18] = 0x23e39,
  [0x0a19] = 0x23e3a, [0x0a1a] = 0x23e3b, [0x0a1b] = 0x23e3c, [0x0a1c] = 0x23e3d,
  [0x0a1d] = 0x23e3e, [0x0a1e] = 0x23e3f, [0x0a1f] = 0x23e40, [0x0a20] = 0x23dd9,
  [0x0a21] = 0x23e41, [0x0a22] = 0x23e42, [0x0a23] = 0x3d21, [0x0a24] = 0x23e43,
  [0x0a25] = 0x23e44, [0x0a26] = 0x6e7a, [0x0a27] = 0x2427c, [0x0a28] = 0x2427d,
  [0x0a29] = 0x2427e, [0x0a2a] = 0x2427f, [0x0a2b] = 0x24280, [0x0a2c] = 0x24281,
  [0x0a2d] = 0x24282, [0x0a2f] = 0x24283, [0x0a30] = 0x24284, [0x0a31] = 0x24285,
  [0x0a32] = 0x24286, [0x0a33] = 0x242a5, [0x0a34] = 0x24287, [0x0a35] = 0x24288,
  [0x0a36] = 0x24289, [0x0a37] = 0x2428a, [0x0a38] = 0x2428b, [0x0a39] = 0x2428c,
  [0x0a3a] = 0x2428d, [0x0a3b] = 0x2428e, [0x0a3e] = 0x2428f, [0x0a40] = 0x24290,
  [0x0a41] = 0x24291, [0x0a42] = 0x24292, [0x0a43] = 0x24293, [0x0a44] = 0x24294,
  [0x0a45] = 0x24295, [0x0a47] = 0x7129, [0x0a48] = 0x24296, [0x0a49] = 0x24556,
  [0x0a4a] = 0x24557, [0x0a4b] = 0x245e1, [0x0a4c] = 0x2460e, [0x0a4d] = 0x2460f,
  [0x0a4e] = 0x2467a, [0x0a4f] = 0x2467b, [0x0a50] = 0x2468b, [0x0a51] = 0x247f2,
  [0x0a52] = 0x247f3, [0x0a53] = 0x24993, [0x0a54] = 0x24994, [0x0a55] = 0x24995,
  [0x0a56] = 0x24996, [0x0a57] = 0x24997, [0x0a58] = 0x249ad, [0x0a59] = 0x24998,
  [0x0a5a] = 0x249ab, [0x0a5b] = 0x2497b, [0x0a5c] = 0x24999, [0x0a5d] = 0x2499a,
  [0x0a5e] = 0x2499b, [0x0a5f] = 0x24985, [0x0a60] = 0x2499c, [0x0a61] = 0x2499d,
  [0x0a62] = 0x3eda, [0x0a64] = 0x249ac, [0x0a65] = 0x24979, [0x0a66] = 0x2499e,
  [0x0a67] = 0x249a4, [0x0a68] = 0x24afd, [0x0a69] = 0x24beb, [0x0a6a] = 0x24bea,
  [0x0a6b] = 0x24c8f, [0x0a6c] = 0x24c90, [0x0a6d] = 0x24c86, [0x0a6e] = 0x24c87,
  [0x0a6f] = 0x24c88, [0x0a70] = 0x24dac, [0x0a71] = 0x24dad, [0x0a72] = 0x24fec,
  [0x0a73] = 0x25073, [0x0a74] = 0x25074, [0x0a76] = 0x251aa, [0x0a77] = 0x251ab,
  [0x0a78] = 0x251ac, [0x0a79] = 0x253d1, [0x0a7a] = 0x253d7, [0x0a7b] = 0x254b9,
  [0x0a7c] = 0x254ba, [0x0a7d] = 0x254bb, [0x0a7e] = 0x254bc, [0x0a7f] = 0x2568e,
  [0x0a80] = 0x2568f, [0x0a81] = 0x25690, [0x0a82] = 0x2567c, [0x0a83] = 0x25691,
  [0x0a84] = 0x25692, [0x0a85] = 0x25693, [0x0a86] = 0x25694, [0x0a87] = 0x2f954,
  [0x0a88] = 0x2569a, [0x0a89] = 0x25695, [0x0a8a] = 0x25696, [0x0a8b] = 0x25697,
  [0x0a8c] = 0x257bd, [0x0a8d] = 0x257c7, [0x0a8e] = 0x257be, [0x0a90] = 0x257bf,
  [0x0a92] = 0x25993, [0x0a93] = 0x21a5e, [0x0a94] = 0x25a86, [0x0a95] = 0x25a87,
  [0x0a96] = 0x25a88, [0x0a97] = 0x25a89, [0x0a98] = 0x25b41, [0x0a99] = 0x25b42,
  [0x0a9a] = 0x25b43, [0x0a9b] = 0x25b44, [0x0a9c] = 0x25b45, [0x0a9d] = 0x25b46,
  [0x0a9e] = 0x25b4a, [0x0a9f] = 0x25b47, [0x0aa0] = 0x25b48, [0x0aa1] = 0x25b49,
  [0x0aa2] = 0x25e6b, [0x0aa3] = 0x25e66, [0x0aa5] = 0x25e67, [0x0aa6] = 0x7cab,
  [0x0aa7] = 0x25e68, [0x0aa8] = 0x7cac, [0x0aaa] = 0x25ffd, [0x0aab] = 0x25ffe,
  [0x0aac] = 0x25fff, [0x0aad] = 0x26000, [0x0aae] = 0x26001, [0x0aaf] = 0x26002,
  [0x0ab0] = 0x26003, [0x0ab1] = 0x26004, [0x0ab2] = 0x26005, [0x0ab3] = 0x26006,
  [0x0ab4] = 0x26233, [0x0ab5] = 0x262c2, [0x0ab6] = 0x262d0, [0x0ab7] = 0x2f974,
  [0x0ab8] = 0x26395, [0x0ab9] = 0x26396, [0x0aba] = 0x26435, [0x0abb] = 0x26436,
  [0x0abc] = 0x26567, [0x0abd] = 0x26568, [0x0abe] = 0x26569, [0x0abf] = 0x2656a,
  [0x0ac0] = 0x26572, [0x0ac1] = 0x2656b, [0x0ac2] = 0x2656c, [0x0ac3] = 0x2656d,
  [0x0ac4] = 0x2656e, [0x0ac5] = 0x2656f, [0x0ac6] = 0x26722, [0x0ac9] = 0x26723,
  [0x0aca] = 0x26741, [0x0acb] = 0x26724, [0x0acc] = 0x26725, [0x0acd] = 0x26726,
  [0x0ace] = 0x26727, [0x0acf] = 0x26728, [0x0ad0] = 0x26729, [0x0ad1] = 0x266b7,
  [0x0ad5] = 0x269da, [0x0ad6] = 0x26acf, [0x0ad7] = 0x26ada, [0x0ad8] = 0x26c89,
  [0x0ad9] = 0x26cb7, [0x0ada] = 0x26c8a, [0x0adb] = 0x26c8b, [0x0adc] = 0x26d5e,
  [0x0add] = 0x26c8c, [0x0adf] = 0x26c8d, [0x0ae0] = 0x26c8e, [0x0ae1] = 0x26c8f,
  [0x0ae2] = 0x26c90, [0x0ae3] = 0x26c91, [0x0ae4] = 0x26c92, [0x0ae5] = 0x26c93,
  [0x0ae6] = 0x26c94, [0x0ae7] = 0x26cb6, [0x0ae8] = 0x26c95, [0x0ae9] = 0x26c96,
  [0x0aea] = 0x26c97, [0x0aeb] = 0x83f7, [0x0aed] = 0x26c98, [0x0aee] = 0x26c99,
  [0x0aef] = 0x26c9a, [0x0af0] = 0x44ea, [0x0af2] = 0x26c9b, [0x0af3] = 0x26cbd,
  [0x0af4] = 0x26c9c, [0x0af5] = 0x26c9d, [0x0af6] = 0x26c9e, [0x0af7] = 0x26c9f,
  [0x0af8] = 0x26cc0, [0x0af9] = 0x26ca0, [0x0afa] = 0x26ca1, [0x0afb] = 0x26ca2,
  [0x0afc] = 0x26ca3, [0x0afd] = 0x26ca4, [0x0afe] = 0x26ca5, [0x0aff] = 0x26ca6,
  [0x0b00] = 0x26ca7, [0x0b01] = 0x26ca8, [0x0b02] = 0x26ca9, [0x0b03] = 0x26caa,
  [0x0b05] = 0x271b4, [0x0b06] = 0x271b5, [0x0b07] = 0x271b6, [0x0b08] = 0x272b9,
  [0x0b09] = 0x272ba, [0x0b0a] = 0x272bb, [0x0b0b] = 0x27291, [0x0b0c] = 0x272bc,
  [0x0b0d] = 0x275ee, [0x0b0e] = 0x275ef, [0x0b0f] = 0x27696, [0x0b10] = 0x27697,
  [0x0b11] = 0x27673, [0x0b12] = 0x27698, [0x0b13] = 0x27699, [0x0b14] = 0x277ed,
  [0x0b15] = 0x277ee, [0x0b16] = 0x278de, [0x0b17] = 0x279b0, [0x0b18] = 0x279b1,
  [0x0b19] = 0x279b2, [0x0b1a] = 0x279b3, [0x0b1b] = 0x279b4, [0x0b1c] = 0x279b5,
  [0x0b1d] = 0x279b6, [0x0b1e] = 0x2f9ce, [0x0b1f] = 0x279b7, [0x0b20] = 0x27be3,
  [0x0b21] = 0x27cb9, [0x0b22] = 0x27d55, [0x0b23] = 0x27d7e, [0x0b24] = 0x27d56,
  [0x0b25] = 0x27d57, [0x0b26] = 0x27d58, [0x0b27] = 0x27d59, [0x0b28] = 0x27d5a,
  [0x0b2a] = 0x27ec4, [0x0b2b] = 0x27ec5, [0x0b2c] = 0x27ec6, [0x0b2d] = 0x27ec7,
  [0x0b2e] = 0x27ec8, [0x0b2f] = 0x27ec9, [0x0b30] = 0x27eee, [0x0b31] = 0x27ecf,
  [0x0b32] = 0x27eca, [0x0b33] = 0x27ecb, [0x0b34] = 0x27ecc, [0x0b35] = 0x28009,
  [0x0b36] = 0x2f9da, [0x0b37] = 0x28003, [0x0b38] = 0x2822f, [0x0b39] = 0x28230,
  [0x0b3a] = 0x28231, [0x0b3b] = 0x28232, [0x0b3c] = 0x2850e, [0x0b3d] = 0x2850f,
  [0x0b3e] = 0x28510, [0x0b3f] = 0x28511, [0x0b40] = 0x28512, [0x0b42] = 0x28513,
  [0x0b43] = 0x28514, [0x0b44] = 0x28515, [0x0b45] = 0x284ff, [0x0b46] = 0x2871f,
  [0x0b47] = 0x28720, [0x0b48] = 0x28721, [0x0b49] = 0x2872d, [0x0b4b] = 0x2881b,
  [0x0b4c] = 0x2881c, [0x0b4d] = 0x2881d, [0x0b4e] = 0x2881e, [0x0b4f] = 0x28953,
  [0x0b50] = 0x28954, [0x0b51] = 0x28955, [0x0b52] = 0x28956, [0x0b53] = 0x28964,
  [0x0b54] = 0x28959, [0x0b55] = 0x28957, [0x0b57] = 0x28958, [0x0b58] = 0x28c6b,
  [0x0b59] = 0x4989, [0x0b5b] = 0x28ce7, [0x0b5c] = 0x28ce8, [0x0b5d] = 0x28ce9,
  [0x0b5e] = 0x28cea, [0x0b5f] = 0x28ceb, [0x0b60] = 0x28cec, [0x0b61] = 0x9596,
  [0x0b62] = 0x28ced, [0x0b63] = 0x28eb1, [0x0b64] = 0x28eb4, [0x0b66] = 0x2908a,
  [0x0b67] = 0x2908d, [0x0b68] = 0x2908c, [0x0b69] = 0x291f2, [0x0b6a] = 0x291f3,
  [0x0b6c] = 0x22108, [0x0b6d] = 0x29417, [0x0b6e] = 0x4ab2, [0x0b6f] = 0x2959f,
  [0x0b70] = 0x296a0, [0x0b72] = 0x2986d, [0x0b73] = 0x2307d, [0x0b74] = 0x29f90,
  [0x0b75] = 0x2a5c4, [0x0b76] = 0x2004c, [0x0b77] = 0x20174, [0x0b78] = 0x20175,
  [0x0b79] = 0x20176, [0x0b7a] = 0x20349, [0x0b7b] = 0x2034a, [0x0b7c] = 0x2034b,
  [0x0b7d] = 0x2034c, [0x0b7e] = 0x2034d, [0x0b7f] = 0x2034e, [0x0b80] = 0x2034f,
  [0x0b81] = 0x20350, [0x0b82] = 0x20351, [0x0b83] = 0x20352, [0x0b84] = 0x20353,
  [0x0b85] = 0x20354, [0x0b86] = 0x20355, [0x0b87] = 0x20356, [0x0b88] = 0x204a4,
  [0x0b89] = 0x204a9, [0x0b8a] = 0x2052a, [0x0b8b] = 0x2052b, [0x0b8c] = 0x207b3,
  [0x0b8d] = 0x207c1, [0x0b8e] = 0x20898, [0x0b8f] = 0x208fb, [0x0b90] = 0x208fc,
  [0x0b91] = 0x209bc, [0x0b92] = 0x20a96, [0x0b93] = 0x20b0a, [0x0b95] = 0x20ea2,
  [0x0b97] = 0x20e81, [0x0b98] = 0x20e82, [0x0b99] = 0x20e83, [0x0b9a] = 0x20e84,
  [0x0b9b] = 0x20e85, [0x0b9c] = 0x20e86, [0x0b9d] = 0x20e87, [0x0b9e] = 0x20e88,
  [0x0b9f] = 0x20e89, [0x0ba0] = 0x20e8a, [0x0ba1] = 0x20e8b, [0x0ba2] = 0x20e8c,
  [0x0ba3] = 0x55f1, [0x0ba4] = 0x20e8d, [0x0ba5] = 0x20ed1, [0x0ba6] = 0x21214,
  [0x0ba7] = 0x213c9, [0x0ba8] = 0x213ca, [0x0ba9] = 0x213cb, [0x0baa] = 0x29afd,
  [0x0bab] = 0x213cc, [0x0bac] = 0x213cd, [0x0bad] = 0x213ce, [0x0bae] = 0x213cf,
  [0x0baf] = 0x213d0, [0x0bb0] = 0x213d1, [0x0bb1] = 0x213d2, [0x0bb2] = 0x213d3,
  [0x0bb3] = 0x213d4, [0x0bb4] = 0x213d5, [0x0bb5] = 0x21571, [0x0bb6] = 0x2004d,
  [0x0bb7] = 0x21662, [0x0bb8] = 0x2180e, [0x0bb9] = 0x217fe, [0x0bba] = 0x217ff,
  [0x0bbb] = 0x21800, [0x0bbd] = 0x21801, [0x0bbe] = 0x21802, [0x0bbf] = 0x21803,
  [0x0bc0] = 0x21815, [0x0bc1] = 0x21804, [0x0bc2] = 0x21805, [0x0bc4] = 0x21987,
  [0x0bc5] = 0x21989, [0x0bc6] = 0x21a6e, [0x0bc7] = 0x21a6f, [0x0bc8] = 0x21a70,
  [0x0bc9] = 0x21a71, [0x0bca] = 0x2f86e, [0x0bcb] = 0x21a72, [0x0bcc] = 0x21a73,
  [0x0bcd] = 0x21a74, [0x0bce] = 0x21b9d, [0x0bcf] = 0x21cb8, [0x0bd0] = 0x21cb9,
  [0x0bd1] = 0x21cba, [0x0bd2] = 0x21ec2, [0x0bd3] = 0x21ec3, [0x0bd4] = 0x21ed5,
  [0x0bd5] = 0x21ec4, [0x0bd6] = 0x21ec0, [0x0bd7] = 0x21ec5, [0x0bd8] = 0x21ec6,
  [0x0bd9] = 0x2228a, [0x0bda] = 0x2228b, [0x0bdb] = 0x2228c, [0x0bdc] = 0x22364,
  [0x0bdd] = 0x25b8c, [0x0bde] = 0x5f41, [0x0bdf] = 0x2240d, [0x0be0] = 0x224a8,
  [0x0be1] = 0x2253b, [0x0be2] = 0x2253c, [0x0be3] = 0x2253d, [0x0be4] = 0x2253e,
  [0x0be5] = 0x2253f, [0x0be6] = 0x22540, [0x0be7] = 0x22766, [0x0be8] = 0x227cb,
  [0x0be9] = 0x227b7, [0x0bea] = 0x227b8, [0x0beb] = 0x227b9, [0x0bec] = 0x227ba,
  [0x0bed] = 0x22767, [0x0bee] = 0x227bb, [0x0bef] = 0x22768, [0x0bf0] = 0x22769,
  [0x0bf1] = 0x2276a, [0x0bf2] = 0x227bc, [0x0bf4] = 0x229db, [0x0bf5] = 0x22c81,
  [0x0bf6] = 0x22c82, [0x0bf7] = 0x22c83, [0x0bf8] = 0x22c84, [0x0bf9] = 0x22c85,
  [0x0bfa] = 0x22c86, [0x0bfb] = 0x22c87, [0x0bfc] = 0x22c88, [0x0bfd] = 0x22c89,
  [0x0bfe] = 0x2f8bf, [0x0bff] = 0x22c8a, [0x0c00] = 0x22c8b, [0x0c02] = 0x22c8c,
  [0x0c03] = 0x22c8d, [0x0c04] = 0x22c8e, [0x0c05] = 0x22c8f, [0x0c06] = 0x22c90,
  [0x0c07] = 0x22ca1, [0x0c08] = 0x22c91, [0x0c09] = 0x22ed6, [0x0c0a] = 0x22f9f,
  [0x0c0b] = 0x23058, [0x0c0c] = 0x23085, [0x0c0d] = 0x2310b, [0x0c0e] = 0x2310c,
  [0x0c0f] = 0x23237, [0x0c10] = 0x23238, [0x0c11] = 0x2f8ce, [0x0c12] = 0x23239,
  [0x0c13] = 0x2323a, [0x0c14] = 0x2323b, [0x0c15] = 0x2323c, [0x0c16] = 0x2323d,
  [0x0c17] = 0x2323e, [0x0c18] = 0x23240, [0x0c19] = 0x23241, [0x0c1a] = 0x23242,
  [0x0c1c] = 0x23243, [0x0c1d] = 0x2338a, [0x0c1e] = 0x2338b, [0x0c1f] = 0x23590,
  [0x0c20] = 0x23559, [0x0c21] = 0x2355b, [0x0c22] = 0x2355c, [0x0c23] = 0x2355d,
  [0x0c24] = 0x2355e, [0x0c25] = 0x698a, [0x0c26] = 0x2355f, [0x0c27] = 0x23560,
  [0x0c28] = 0x23561, [0x0c29] = 0x23562, [0x0c2a] = 0x23563, [0x0c2b] = 0x23564,
  [0x0c2c] = 0x23593, [0x0c2d] = 0x23565, [0x0c2e] = 0x23566, [0x0c2f] = 0x23567,
  [0x0c30] = 0x23568, [0x0c31] = 0x23569, [0x0c32] = 0x2356a, [0x0c33] = 0x2356b,
  [0x0c34] = 0x23595, [0x0c35] = 0x2356c, [0x0c36] = 0x2356d, [0x0c37] = 0x2356e,
  [0x0c38] = 0x2356f, [0x0c39] = 0x698c, [0x0c3a] = 0x23570, [0x0c3b] = 0x23571,
  [0x0c3c] = 0x23572, [0x0c3d] = 0x23573, [0x0c3e] = 0x23574, [0x0c3f] = 0x23575,
  [0x0c40] = 0x23576, [0x0c41] = 0x23577, [0x0c42] = 0x23578, [0x0c43] = 0x23579,
  [0x0c44] = 0x6980, [0x0c45] = 0x2357a, [0x0c46] = 0x697f, [0x0c47] = 0x2357b,
  [0x0c48] = 0x235a9, [0x0c49] = 0x2357c, [0x0c4a] = 0x238f1, [0x0c4b] = 0x238eb,
  [0x0c4c] = 0x2398a, [0x0c4d] = 0x23a3d, [0x0c4e] = 0x23aa6, [0x0c4f] = 0x23aa7,
  [0x0c50] = 0x23bba, [0x0c52] = 0x23bbb, [0x0c53] = 0x23bbc, [0x0c54] = 0x23bbd,
  [0x0c55] = 0x23e91, [0x0c56] = 0x23e92, [0x0c57] = 0x23e93, [0x0c58] = 0x23e94,
  [0x0c59] = 0x23e95, [0x0c5a] = 0x23e96, [0x0c5b] = 0x23e97, [0x0c5c] = 0x23dda,
  [0x0c5d] = 0x23e98, [0x0c5e] = 0x23e99, [0x0c5f] = 0x23e9a, [0x0c62] = 0x23e9b,
  [0x0c63] = 0x23e9c, [0x0c64] = 0x23e9d, [0x0c65] = 0x23e9e, [0x0c66] = 0x23e9f,
  [0x0c67] = 0x23ea0, [0x0c68] = 0x23ebf, [0x0c69] = 0x23ea1, [0x0c6a] = 0x23ea2,
  [0x0c6b] = 0x23ea3, [0x0c6c] = 0x23ea4, [0x0c6d] = 0x23ea5, [0x0c6e] = 0x23ea6,
  [0x0c6f] = 0x23ea7, [0x0c70] = 0x23e5b, [0x0c71] = 0x2f90d, [0x0c72] = 0x23ea8,
  [0x0c73] = 0x242cb, [0x0c75] = 0x3dda, [0x0c76] = 0x242cc, [0x0c77] = 0x2434f,
  [0x0c78] = 0x242cd, [0x0c79] = 0x242ce, [0x0c7a] = 0x242cf, [0x0c7b] = 0x2434e,
  [0x0c7c] = 0x242d0, [0x0c7d] = 0x242d1, [0x0c7e] = 0x242d2, [0x0c80] = 0x242d3,
  [0x0c81] = 0x242d4, [0x0c82] = 0x242d5, [0x0c83] = 0x242d6, [0x0c84] = 0x242d7,
  [0x0c85] = 0x242d8, [0x0c86] = 0x242d9, [0x0c87] = 0x242fa, [0x0c88] = 0x242da,
  [0x0c89] = 0x242db, [0x0c8a] = 0x2f91c, [0x0c8b] = 0x242dc, [0x0c8e] = 0x242dd,
  [0x0c8f] = 0x242de, [0x0c90] = 0x242df, [0x0c91] = 0x242e0, [0x0c92] = 0x242e1,
  [0x0c93] = 0x242e2, [0x0c94] = 0x242e3, [0x0c95] = 0x242e4, [0x0c96] = 0x242e5,
  [0x0c97] = 0x3ddb, [0x0c98] = 0x242e6, [0x0c99] = 0x242e7, [0x0c9a] = 0x2458e,
  [0x0c9b] = 0x24698, [0x0c9c] = 0x24699, [0x0c9e] = 0x2480a, [0x0ca0] = 0x24826,
  [0x0ca1] = 0x24827, [0x0ca2] = 0x249bd, [0x0ca3] = 0x249df, [0x0ca4] = 0x3ee2,
  [0x0ca5] = 0x249be, [0x0ca6] = 0x249de, [0x0ca7] = 0x249bf, [0x0ca8] = 0x249c0,
  [0x0ca9] = 0x249c1, [0x0caa] = 0x249c2, [0x0cab] = 0x2499f, [0x0cac] = 0x249c3,
  [0x0cad] = 0x249c4, [0x0cae] = 0x249c5, [0x0caf] = 0x249c6, [0x0cb0] = 0x249c7,
  [0x0cb1] = 0x3ee3, [0x0cb2] = 0x249c8, [0x0cb3] = 0x249c9, [0x0cb4] = 0x249e3,
  [0x0cb5] = 0x249ca, [0x0cb6] = 0x249cb, [0x0cb7] = 0x249cc, [0x0cb8] = 0x249cd,
  [0x0cb9] = 0x249ce, [0x0cba] = 0x249cf, [0x0cbb] = 0x249d0, [0x0cbc] = 0x249d1,
  [0x0cbd] = 0x249d2, [0x0cbe] = 0x249d3, [0x0cbf] = 0x2f92f, [0x0cc0] = 0x249d4,
  [0x0cc1] = 0x24b04, [0x0cc2] = 0x24b65, [0x0cc3] = 0x24bee, [0x0cc4] = 0x24ca0,
  [0x0cc5] = 0x24ca1, [0x0cc6] = 0x24ddd, [0x0cc7] = 0x24dde, [0x0cc8] = 0x24f83,
  [0x0cc9] = 0x24ffa, [0x0cca] = 0x2507f, [0x0ccb] = 0x25080, [0x0ccc] = 0x251e8,
  [0x0ccd] = 0x251e9, [0x0cce] = 0x251ea, [0x0ccf] = 0x235a6, [0x0cd0] = 0x253e1,
  [0x0cd2] = 0x254db, [0x0cd3] = 0x254dc, [0x0cd4] = 0x254dd, [0x0cd5] = 0x254de,
  [0x0cd6] = 0x254df, [0x0cd7] = 0x254e0, [0x0cd8] = 0x254e1, [0x0cd9] = 0x254e2,
  [0x0cda] = 0x254e3, [0x0cdb] = 0x254e4, [0x0cdc] = 0x254e5, [0x0cdd] = 0x254e6,
  [0x0cde] = 0x789c, [0x0cdf] = 0x254e7, [0x0ce0] = 0x256b0, [0x0ce1] = 0x25698,
  [0x0ce2] = 0x256b1, [0x0ce3] = 0x25699, [0x0ce4] = 0x256b2, [0x0ce5] = 0x256b3,
  [0x0ce6] = 0x256b4, [0x0ce7] = 0x256b5, [0x0ce8] = 0x256b6, [0x0ce9] = 0x256b7,
  [0x0cea] = 0x256b8, [0x0ceb] = 0x256b9, [0x0cec] = 0x257e2, [0x0ced] = 0x257e3,
  [0x0cee] = 0x257e4, [0x0cef] = 0x257e5, [0x0cf0] = 0x257e6, [0x0cf1] = 0x257e7,
  [0x0cf2] = 0x257e8, [0x0cf3] = 0x257e9, [0x0cf4] = 0x259ad, [0x0cf5] = 0x259cf,
  [0x0cf6] = 0x259ae, [0x0cf7] = 0x259af, [0x0cf8] = 0x259b0, [0x0cf9] = 0x259b1,
  [0x0cfa] = 0x25a92, [0x0cfb] = 0x25a93, [0x0cfc] = 0x25a94, [0x0cfd] = 0x25b79,
  [0x0cfe] = 0x7b7b, [0x0cff] = 0x25b7a, [0x0d00] = 0x25b7b, [0x0d01] = 0x25b7c,
  [0x0d02] = 0x25b7d, [0x0d03] = 0x25b7e, [0x0d04] = 0x25b7f, [0x0d05] = 0x25b80,
  [0x0d06] = 0x25b81, [0x0d07] = 0x25b82, [0x0d08] = 0x25b83, [0x0d09] = 0x25b84,
  [0x0d0a] = 0x25e97, [0x0d0b] = 0x25e84, [0x0d0c] = 0x25e85, [0x0d0d] = 0x25e89,
  [0x0d0e] = 0x25e86, [0x0d0f] = 0x25e87, [0x0d10] = 0x2602c, [0x0d11] = 0x2602d,
  [0x0d12] = 0x2602e, [0x0d13] = 0x2602f, [0x0d14] = 0x26030, [0x0d15] = 0x26031,
  [0x0d16] = 0x26032, [0x0d17] = 0x26033, [0x0d18] = 0x26034, [0x0d19] = 0x26035,
  [0x0d1a] = 0x26036, [0x0d1b] = 0x26037, [0x0d1c] = 0x262d2, [0x0d1d] = 0x26444,
  [0x0d1e] = 0x26500, [0x0d1f] = 0x26583, [0x0d20] = 0x26584, [0x0d21] = 0x2f986,
  [0x0d22] = 0x2677a, [0x0d23] = 0x2677b, [0x0d24] = 0x2677c, [0x0d25] = 0x2677d,
  [0x0d26] = 0x2677e, [0x0d28] = 0x2677f, [0x0d29] = 0x26787, [0x0d2a] = 0x26780,
  [0x0d2b] = 0x26a43, [0x0d2c] = 0x26d34, [0x0d2d] = 0x4749, [0x0d2e] = 0x26d35,
  [0x0d2f] = 0x26d36, [0x0d30] = 0x26d37, [0x0d31] = 0x26d38, [0x0d32] = 0x26d39,
  [0x0d33] = 0x26d3a, [0x0d34] = 0x26d3b, [0x0d35] = 0x26d3c, [0x0d36] = 0x26d3d,
  [0x0d37] = 0x26d3e, [0x0d38] = 0x26d3f, [0x0d39] = 0x26d40, [0x0d3a] = 0x26d41,
  [0x0d3b] = 0x26d42, [0x0d3c] = 0x26d43, [0x0d3d] = 0x26d44, [0x0d3e] = 0x26d45,
  [0x0d3f] = 0x26d46, [0x0d40] = 0x26d47, [0x0d41] = 0x26d48, [0x0d42] = 0x26d49,
  [0x0d43] = 0x26d4a, [0x0d45] = 0x26d4b, [0x0d46] = 0x26d4c, [0x0d47] = 0x26d4d,
  [0x0d48] = 0x26d4e, [0x0d49] = 0x272e7, [0x0d4a] = 0x272e8, [0x0d4b] = 0x272e9,
  [0x0d4c] = 0x272ea, [0x0d4d] = 0x272eb, [0x0d4e] = 0x272ec, [0x0d4f] = 0x272ed,
  [0x0d50] = 0x272ee, [0x0d51] = 0x275f7, [0x0d52] = 0x2769a, [0x0d53] = 0x276cd,
  [0x0d54] = 0x2f9c8, [0x0d55] = 0x276a3, [0x0d56] = 0x276ce, [0x0d57] = 0x27831,
  [0x0d58] = 0x27832, [0x0d59] = 0x27833, [0x0d5a] = 0x279e4, [0x0d5c] = 0x279e5,
  [0x0d5d] = 0x279e6, [0x0d5e] = 0x27cc8, [0x0d5f] = 0x27d74, [0x0d60] = 0x27d75,
  [0x0d61] = 0x27d76, [0x0d62] = 0x27d77, [0x0d63] = 0x27d78, [0x0d64] = 0x27d79,
  [0x0d65] = 0x27d7a, [0x0d66] = 0x27e61, [0x0d67] = 0x27edf, [0x0d68] = 0x27ee0,
  [0x0d69] = 0x27ee1, [0x0d6a] = 0x27ee2, [0x0d6b] = 0x27ee3, [0x0d6c] = 0x27ee4,
  [0x0d6d] = 0x27ee5, [0x0d6e] = 0x27ee6, [0x0d6f] = 0x47f3, [0x0d70] = 0x28031,
  [0x0d71] = 0x2823d, [0x0d72] = 0x2823e, [0x0d73] = 0x2823f, [0x0d74] = 0x282f4,
  [0x0d75] = 0x282f5, [0x0d76] = 0x282f6, [0x0d78] = 0x28417, [0x0d79] = 0x28541,
  [0x0d7a] = 0x28542, [0x0d7b] = 0x28543, [0x0d7c] = 0x28544, [0x0d7d] = 0x28545,
  [0x0d7e] = 0x28546, [0x0d7f] = 0x28547, [0x0d80] = 0x28548, [0x0d81] = 0x28549,
  [0x0d82] = 0x2854a, [0x0d83] = 0x2854b, [0x0d84] = 0x90d2, [0x0d85] = 0x286ba,
  [0x0d86] = 0x28744, [0x0d87] = 0x2892a, [0x0d88] = 0x2896d, [0x0d89] = 0x9274,
  [0x0d8a] = 0x2f9e7, [0x0d8b] = 0x2897e, [0x0d8c] = 0x2896e, [0x0d8d] = 0x2896f,
  [0x0d8e] = 0x28970, [0x0d8f] = 0x28971, [0x0d90] = 0x28976, [0x0d91] = 0x28c79,
  [0x0d92] = 0x95a0, [0x0d94] = 0x28d04, [0x0d95] = 0x28cf8, [0x0d96] = 0x28d05,
  [0x0d97] = 0x28eda, [0x0d98] = 0x28edb, [0x0d99] = 0x28edc, [0x0d9a] = 0x28edd,
  [0x0d9b] = 0x28ede, [0x0d9c] = 0x28edf, [0x0d9d] = 0x28ee0, [0x0d9e] = 0x28ee1,
  [0x0d9f] = 0x290c5, [0x0da0] = 0x2909a, [0x0da1] = 0x2909b, [0x0da2] = 0x2909c,
  [0x0da3] = 0x291d8, [0x0da4] = 0x291d9, [0x0da5] = 0x291da, [0x0da6] = 0x2926e,
  [0x0da7] = 0x4a57, [0x0da8] = 0x2946b, [0x0da9] = 0x27d83, [0x0dab] = 0x296d6,
  [0x0dac] = 0x296bd, [0x0dae] = 0x2322e, [0x0daf] = 0x29afc, [0x0db0] = 0x29d53,
  [0x0db1] = 0x29db7, [0x0db2] = 0x2a28c, [0x0db3] = 0x2a38e, [0x0db4] = 0x2a2ff,
  [0x0db5] = 0x20076, [0x0db6] = 0x20106, [0x0db7] = 0x2017e, [0x0db8] = 0x20384,
  [0x0db9] = 0x20385, [0x0dba] = 0x20386, [0x0dbb] = 0x20387, [0x0dbc] = 0x20388,
  [0x0dbd] = 0x203a7, [0x0dbe] = 0x20389, [0x0dbf] = 0x2038a, [0x0dc0] = 0x2038b,
  [0x0dc1] = 0x2038c, [0x0dc2] = 0x2038d, [0x0dc3] = 0x2038e, [0x0dc4] = 0x2038f,
  [0x0dc5] = 0x203a3, [0x0dc6] = 0x20390, [0x0dc7] = 0x20391, [0x0dc8] = 0x20392,
  [0x0dc9] = 0x20393, [0x0dca] = 0x203ac, [0x0dcb] = 0x20394, [0x0dcc] = 0x20395,
  [0x0dcd] = 0x20396, [0x0dce] = 0x204aa, [0x0dcf] = 0x204ab, [0x0dd0] = 0x2059e,
  [0x0dd1] = 0x51a9, [0x0dd2] = 0x20608, [0x0dd3] = 0x20609, [0x0dd4] = 0x20668,
  [0x0dd5] = 0x20669, [0x0dd6] = 0x2066a, [0x0dd7] = 0x2069a, [0x0dd8] = 0x207d9,
  [0x0dd9] = 0x207da, [0x0dda] = 0x207db, [0x0ddb] = 0x208a8, [0x0ddc] = 0x20901,
  [0x0ddd] = 0x20928, [0x0dde] = 0x20977, [0x0ddf] = 0x209bd, [0x0de0] = 0x209be,
  [0x0de1] = 0x20f60, [0x0de2] = 0x20aa0, [0x0de3] = 0x20aa2, [0x0de4] = 0x20b0f,
  [0x0de5] = 0x20b85, [0x0de6] = 0x20efe, [0x0de7] = 0x20eff, [0x0de8] = 0x20f00,
  [0x0de9] = 0x20f01, [0x0dea] = 0x20f16, [0x0deb] = 0x20f02, [0x0dec] = 0x20f03,
  [0x0ded] = 0x20f04, [0x0dee] = 0x20f05, [0x0def] = 0x20f06, [0x0df0] = 0x20f07,
  [0x0df1] = 0x20f08, [0x0df2] = 0x20f22, [0x0df3] = 0x20f09, [0x0df4] = 0x20f0a,
  [0x0df5] = 0x20f0b, [0x0df6] = 0x20f0c, [0x0df7] = 0x20f14, [0x0df8] = 0x20f29,
  [0x0df9] = 0x20e6e, [0x0dfa] = 0x20f0d, [0x0dfb] = 0x20f0e, [0x0dfc] = 0x20f0f,
  [0x0dfd] = 0x20f10, [0x0dfe] = 0x20f11, [0x0dff] = 0x20f12, [0x0e00] = 0x20f13,
  [0x0e01] = 0x2121b, [0x0e02] = 0x2121c, [0x0e03] = 0x2121d, [0x0e04] = 0x21413,
  [0x0e05] = 0x213ff, [0x0e06] = 0x21400, [0x0e07] = 0x21401, [0x0e08] = 0x21402,
  [0x0e09] = 0x21403, [0x0e0a] = 0x21404, [0x0e0b] = 0x2140a, [0x0e0c] = 0x21405,
  [0x0e0d] = 0x21406, [0x0e0e] = 0x21407, [0x0e0f] = 0x21408, [0x0e10] = 0x2153a,
  [0x0e11] = 0x21573, [0x0e12] = 0x2166f, [0x0e13] = 0x21670, [0x0e14] = 0x21671,
  [0x0e15] = 0x20052, [0x0e17] = 0x21a9a, [0x0e18] = 0x21860, [0x0e19] = 0x2f869,
  [0x0e1a] = 0x2182b, [0x0e1b] = 0x21824, [0x0e1c] = 0x2182c, [0x0e1d] = 0x2182d,
  [0x0e1e] = 0x2182e, [0x0e1f] = 0x2182f, [0x0e20] = 0x21830, [0x0e21] = 0x21831,
  [0x0e22] = 0x21832, [0x0e23] = 0x21833, [0x0e24] = 0x21834, [0x0e25] = 0x21835,
  [0x0e26] = 0x21845, [0x0e27] = 0x21836, [0x0e28] = 0x21992, [0x0e29] = 0x21993,
  [0x0e2a] = 0x21a88, [0x0e2b] = 0x21a89, [0x0e2c] = 0x21a8a, [0x0e2d] = 0x21a8b,
  [0x0e2e] = 0x21b3f, [0x0e30] = 0x21cca, [0x0e31] = 0x21ef6, [0x0e32] = 0x21ef7,
  [0x0e33] = 0x21ef8, [0x0e34] = 0x21ef9, [0x0e35] = 0x2212b, [0x0e37] = 0x221a4,
  [0x0e38] = 0x221ce, [0x0e3a] = 0x2229e, [0x0e3b] = 0x2229f, [0x0e3c] = 0x222a0,
  [0x0e3d] = 0x222a1, [0x0e3e] = 0x222a2, [0x0e3f] = 0x222a3, [0x0e40] = 0x222a4,
  [0x0e41] = 0x222a5, [0x0e42] = 0x222a6, [0x0e43] = 0x22418, [0x0e44] = 0x22560,
  [0x0e45] = 0x22561, [0x0e46] = 0x22562, [0x0e47] = 0x22563, [0x0e48] = 0x22564,
  [0x0e49] = 0x227bd, [0x0e4a] = 0x227be, [0x0e4b] = 0x2280b, [0x0e4c] = 0x227bf,
  [0x0e4d] = 0x227c0, [0x0e4f] = 0x2280c, [0x0e50] = 0x2278d, [0x0e51] = 0x2280d,
  [0x0e52] = 0x2280e, [0x0e53] = 0x2285e, [0x0e54] = 0x2280f, [0x0e55] = 0x22810,
  [0x0e56] = 0x227c1, [0x0e57] = 0x22811, [0x0e59] = 0x227c2, [0x0e5a] = 0x229ea,
  [0x0e5b] = 0x229eb, [0x0e5c] = 0x22ce3, [0x0e5d] = 0x22ce4, [0x0e5e] = 0x22ce5,
  [0x0e5f] = 0x22ce6, [0x0e60] = 0x22ce7, [0x0e61] = 0x22c92, [0x0e62] = 0x22d4a,
  [0x0e64] = 0x22ce8, [0x0e65] = 0x22ce9, [0x0e66] = 0x22cea, [0x0e67] = 0x22d01,
  [0x0e68] = 0x22ceb, [0x0e69] = 0x22cec, [0x0e6a] = 0x22cef, [0x0e6b] = 0x22ced,
  [0x0e6c] = 0x22cee, [0x0e6d] = 0x22cf0, [0x0e6e] = 0x22cf1, [0x0e6f] = 0x22cf2,
  [0x0e70] = 0x22cf3, [0x0e71] = 0x22cf4, [0x0e72] = 0x22cf5, [0x0e73] = 0x22cf6,
  [0x0e74] = 0x22cf7, [0x0e75] = 0x22cf8, [0x0e76] = 0x22cf9, [0x0e77] = 0x22cfe,
  [0x0e78] = 0x22cfa, [0x0e79] = 0x22cfb, [0x0e7a] = 0x22cfc, [0x0e7b] = 0x23263,
  [0x0e7c] = 0x23264, [0x0e7d] = 0x23265, [0x0e7e] = 0x23266, [0x0e7f] = 0x23267,
  [0x0e80] = 0x23268, [0x0e81] = 0x2360b, [0x0e82] = 0x2360d, [0x0e83] = 0x3bbc,
  [0x0e84] = 0x2360e, [0x0e85] = 0x2360f, [0x0e86] = 0x3ba4, [0x0e87] = 0x23614,
  [0x0e88] = 0x235d4, [0x0e89] = 0x235d5, [0x0e8a] = 0x235d6, [0x0e8b] = 0x235d7,
  [0x0e8c] = 0x235d8, [0x0e8d] = 0x235d9, [0x0e8e] = 0x235da, [0x0e8f] = 0x235db,
  [0x0e90] = 0x235dc, [0x0e91] = 0x235dd, [0x0e92] = 0x235de, [0x0e93] = 0x235df,
  [0x0e94] = 0x235e0, [0x0e95] = 0x235e1, [0x0e96] = 0x235e2, [0x0e97] = 0x235e3,
  [0x0e98] = 0x235e4, [0x0e99] = 0x235e5, [0x0e9a] = 0x235e6, [0x0e9b] = 0x235e7,
  [0x0e9c] = 0x235e8, [0x0e9d] = 0x235e9, [0x0e9e] = 0x235ea, [0x0e9f] = 0x235eb,
  [0x0ea0] = 0x235ec, [0x0ea1] = 0x236ec, [0x0ea2] = 0x235ed, [0x0ea3] = 0x235ee,
  [0x0ea4] = 0x235ef, [0x0ea5] = 0x23600, [0x0ea6] = 0x235f0, [0x0ea7] = 0x23904,
  [0x0ea9] = 0x23905, [0x0eaa] = 0x2f8f3, [0x0eab] = 0x23ab5, [0x0eac] = 0x2f8f9,
  [0x0ead] = 0x23bd6, [0x0eae] = 0x23bd7, [0x0eaf] = 0x23bd8, [0x0eb0] = 0x23bd9,
  [0x0eb1] = 0x23ea9, [0x0eb2] = 0x23f00, [0x0eb3] = 0x23f01, [0x0eb4] = 0x23f02,
  [0x0eb5] = 0x23f03, [0x0eb6] = 0x23f04, [0x0eb7] = 0x23f05, [0x0eb8] = 0x23f06,
  [0x0eb9] = 0x23f3f, [0x0eba] = 0x23f07, [0x0ebb] = 0x23f08, [0x0ebc] = 0x23f09,
  [0x0ebd] = 0x23f0a, [0x0ebe] = 0x23eb6, [0x0ebf] = 0x23f0b, [0x0ec0] = 0x23f0c,
  [0x0ec3] = 0x23f0d, [0x0ec5] = 0x23f0e, [0x0ec6] = 0x23f0f, [0x0ec7] = 0x23f10,
  [0x0ec8] = 0x23f11, [0x0ec9] = 0x23f12, [0x0eca] = 0x23f35, [0x0ecb] = 0x23f13,
  [0x0ecc] = 0x23f14, [0x0ecd] = 0x23eaa, [0x0ecf] = 0x23fc1, [0x0ed0] = 0x23f15,
  [0x0ed1] = 0x23f16, [0x0ed2] = 0x23f17, [0x0ed3] = 0x2431e, [0x0ed4] = 0x2431f,
  [0x0ed5] = 0x24320, [0x0ed6] = 0x24321, [0x0ed8] = 0x24322, [0x0ed9] = 0x24323,
  [0x0eda] = 0x7195, [0x0edb] = 0x24324, [0x0edc] = 0x24325, [0x0edd] = 0x24326,
  [0x0ede] = 0x24327, [0x0edf] = 0x24328, [0x0ee0] = 0x24329, [0x0ee1] = 0x2432a,
  [0x0ee2] = 0x2432b, [0x0ee3] = 0x24307, [0x0ee4] = 0x2432c, [0x0ee5] = 0x2432d,
  [0x0ee6] = 0x2432e, [0x0ee7] = 0x3de8, [0x0ee9] = 0x2432f, [0x0eea] = 0x24330,
  [0x0eeb] = 0x24348, [0x0eec] = 0x24331, [0x0eed] = 0x24332, [0x0eee] = 0x7198,
  [0x0eef] = 0x24333, [0x0ef1] = 0x24334, [0x0ef2] = 0x24335, [0x0ef3] = 0x24336,
  [0x0ef4] = 0x24559, [0x0ef5] = 0x24591, [0x0ef6] = 0x245e7, [0x0ef7] = 0x246b8,
  [0x0ef8] = 0x246b9, [0x0ef9] = 0x246ba, [0x0efa] = 0x246bb, [0x0efb] = 0x246c8,
  [0x0efc] = 0x2484d, [0x0efd] = 0x24889, [0x0efe] = 0x24850, [0x0eff] = 0x24828,
  [0x0f00] = 0x249fb, [0x0f01] = 0x249fc, [0x0f02] = 0x249fd, [0x0f03] = 0x249fe,
  [0x0f04] = 0x249ff, [0x0f05] = 0x7478, [0x0f06] = 0x24a00, [0x0f07] = 0x24a01,
  [0x0f08] = 0x24a02, [0x0f09] = 0x249e4, [0x0f0a] = 0x249e5, [0x0f0b] = 0x24a03,
  [0x0f0c] = 0x24a04, [0x0f0d] = 0x24a05, [0x0f0e] = 0x24a06, [0x0f0f] = 0x24a07,
  [0x0f10] = 0x24a13, [0x0f11] = 0x24a08, [0x0f12] = 0x24a09, [0x0f13] = 0x249e6,
  [0x0f14] = 0x24a0a, [0x0f15] = 0x24a0b, [0x0f16] = 0x24a0c, [0x0f17] = 0x24b0b,
  [0x0f18] = 0x24bf4, [0x0f19] = 0x24c0a, [0x0f1a] = 0x24cb1, [0x0f1b] = 0x24e1d,
  [0x0f1c] = 0x2f93b, [0x0f1d] = 0x24f8b, [0x0f1e] = 0x24f8c, [0x0f1f] = 0x24f8f,
  [0x0f20] = 0x24f8d, [0x0f21] = 0x24f8e, [0x0f22] = 0x25006, [0x0f23] = 0x25086,
  [0x0f24] = 0x25222, [0x0f25] = 0x25223, [0x0f26] = 0x25224, [0x0f27] = 0x25225,
  [0x0f28] = 0x25226, [0x0f29] = 0x25227, [0x0f2a] = 0x25228, [0x0f2b] = 0x253f1,
  [0x0f2c] = 0x78b9, [0x0f2d] = 0x25523, [0x0f2e] = 0x25512, [0x0f30] = 0x25513,
  [0x0f31] = 0x25514, [0x0f32] = 0x25515, [0x0f33] = 0x25516, [0x0f34] = 0x25517,
  [0x0f35] = 0x25518, [0x0f36] = 0x25519, [0x0f37] = 0x2554b, [0x0f38] = 0x2551a,
  [0x0f39] = 0x2551b, [0x0f3a] = 0x2551c, [0x0f3c] = 0x2551d, [0x0f3d] = 0x2551e,
  [0x0f3e] = 0x2551f, [0x0f3f] = 0x25520, [0x0f40] = 0x256ca, [0x0f41] = 0x222ac,
  [0x0f42] = 0x256cb, [0x0f43] = 0x256cc, [0x0f44] = 0x256cd, [0x0f45] = 0x256ce,
  [0x0f47] = 0x256cf, [0x0f48] = 0x25810, [0x0f49] = 0x25811, [0x0f4a] = 0x25812,
  [0x0f4b] = 0x25813, [0x0f4c] = 0x25814, [0x0f4d] = 0x2580f, [0x0f4e] = 0x25815,
  [0x0f4f] = 0x25816, [0x0f50] = 0x25817, [0x0f51] = 0x7a33, [0x0f52] = 0x25818,
  [0x0f53] = 0x259c6, [0x0f54] = 0x25a9f, [0x0f55] = 0x25aa0, [0x0f56] = 0x25aa4,
  [0x0f57] = 0x25aa1, [0x0f58] = 0x25bb5, [0x0f59] = 0x25bb6, [0x0f5a] = 0x25bb7,
  [0x0f5b] = 0x25bb8, [0x0f5c] = 0x25bb9, [0x0f5d] = 0x25bba, [0x0f5e] = 0x25bd0,
  [0x0f5f] = 0x25bc6, [0x0f60] = 0x25bbb, [0x0f61] = 0x25bbc, [0x0f62] = 0x25bbd,
  [0x0f63] = 0x25bbe, [0x0f64] = 0x25bbf, [0x0f65] = 0x25bc0, [0x0f66] = 0x25bc1,
  [0x0f67] = 0x25bc2, [0x0f68] = 0x2f960, [0x0f69] = 0x25ea7, [0x0f6a] = 0x25ea8,
  [0x0f6b] = 0x25ea9, [0x0f6c] = 0x25eaa, [0x0f6d] = 0x25eab, [0x0f6e] = 0x25eac,
  [0x0f6f] = 0x25ebc, [0x0f70] = 0x25ead, [0x0f71] = 0x25eae, [0x0f72] = 0x7cc0,
  [0x0f73] = 0x25eaf, [0x0f74] = 0x25eb0, [0x0f75] = 0x25eb1, [0x0f76] = 0x7cc1,
  [0x0f77] = 0x25eb4, [0x0f78] = 0x25eb2, [0x0f79] = 0x26066, [0x0f7a] = 0x26067,
  [0x0f7b] = 0x26068, [0x0f7c] = 0x26069, [0x0f7d] = 0x2606a, [0x0f7e] = 0x2606b,
  [0x0f7f] = 0x2606c, [0x0f80] = 0x2606d, [0x0f81] = 0x2606e, [0x0f82] = 0x2606f,
  [0x0f84] = 0x2623e, [0x0f85] = 0x2623f, [0x0f86] = 0x26240, [0x0f87] = 0x262e8,
  [0x0f88] = 0x26452, [0x0f89] = 0x26453, [0x0f8a] = 0x26454, [0x0f8b] = 0x26455,
  [0x0f8c] = 0x26591, [0x0f8d] = 0x26592, [0x0f8e] = 0x26593, [0x0f8f] = 0x26594,
  [0x0f90] = 0x26595, [0x0f91] = 0x26596, [0x0f92] = 0x26597, [0x0f93] = 0x26598,
  [0x0f95] = 0x26599, [0x0f96] = 0x2659a, [0x0f97] = 0x2659b, [0x0f98] = 0x267b7,
  [0x0f99] = 0x267b8, [0x0f9a] = 0x267b9, [0x0f9c] = 0x267ba, [0x0f9d] = 0x267bb,
  [0x0f9e] = 0x267bd, [0x0fa0] = 0x267bc, [0x0fa1] = 0x2f988, [0x0fa2] = 0x2f987,
  [0x0fa3] = 0x268eb, [0x0fa4] = 0x268ec, [0x0fa5] = 0x26916, [0x0fa6] = 0x26944,
  [0x0fa7] = 0x269e3, [0x0fa8] = 0x26a58, [0x0fa9] = 0x26a53, [0x0faa] = 0x26a54,
  [0x0fab] = 0x26ae0, [0x0fac] = 0x26db1, [0x0fad] = 0x26db2, [0x0fae] = 0x26db3,
  [0x0faf] = 0x26dea, [0x0fb1] = 0x26db4, [0x0fb2] = 0x26df0, [0x0fb3] = 0x26db5,
  [0x0fb4] = 0x26db6, [0x0fb5] = 0x26db7, [0x0fb6] = 0x26db8, [0x0fb7] = 0x26db9,
  [0x0fb8] = 0x26dba, [0x0fb9] = 0x26dbb, [0x0fba] = 0x26dbc, [0x0fbb] = 0x26dbd,
  [0x0fbc] = 0x26dbe, [0x0fbd] = 0x26dbf, [0x0fbe] = 0x26dc0, [0x0fbf] = 0x26dc1,
  [0x0fc0] = 0x26dc2, [0x0fc1] = 0x26dc3, [0x0fc3] = 0x26dc4, [0x0fc4] = 0x26dc5,
  [0x0fc5] = 0x26dc6, [0x0fc6] = 0x26dfc, [0x0fc7] = 0x26dc7, [0x0fc8] = 0x26dc8,
  [0x0fc9] = 0x26dc9, [0x0fca] = 0x26dca, [0x0fcb] = 0x26dcb, [0x0fcc] = 0x26dcc,
  [0x0fcd] = 0x26e84, [0x0fce] = 0x26dcd, [0x0fcf] = 0x26dce, [0x0fd0] = 0x26dcf,
  [0x0fd1] = 0x26dd0, [0x0fd2] = 0x26dd1, [0x0fd3] = 0x26dd2, [0x0fd4] = 0x271ca,
  [0x0fd5] = 0x271cb, [0x0fd6] = 0x27336, [0x0fd7] = 0x27337, [0x0fd8] = 0x2739f,
  [0x0fd9] = 0x27338, [0x0fda] = 0x27339, [0x0fdb] = 0x2733a, [0x0fdc] = 0x8744,
  [0x0fdd] = 0x276be, [0x0fde] = 0x276f0, [0x0fdf] = 0x276bf, [0x0fe0] = 0x276c0,
  [0x0fe1] = 0x277f7, [0x0fe2] = 0x277f8, [0x0fe3] = 0x277fa, [0x0fe4] = 0x27843,
  [0x0fe5] = 0x27844, [0x0fe6] = 0x27a0f, [0x0fe7] = 0x27a10, [0x0fe8] = 0x27a11,
  [0x0fe9] = 0x27a12, [0x0fea] = 0x8aad, [0x0feb] = 0x27a13, [0x0fec] = 0x27a14,
  [0x0fed] = 0x27a15, [0x0fee] = 0x27a16, [0x0ff1] = 0x27cd3, [0x0ff2] = 0x27cd7,
  [0x0ff4] = 0x27db0, [0x0ff5] = 0x27d98, [0x0ff6] = 0x27d90, [0x0ff7] = 0x27d91,
  [0x0ff8] = 0x27d92, [0x0ff9] = 0x27d97, [0x0ffa] = 0x27d93, [0x0ffb] = 0x27e64,
  [0x0ffc] = 0x27f01, [0x0ffd] = 0x27f02, [0x0ffe] = 0x27f03, [0x0fff] = 0x27f04,
  [0x1000] = 0x27f05, [0x1001] = 0x27f06, [0x1002] = 0x27f07, [0x1003] = 0x2805a,
  [0x1004] = 0x2805b, [0x1005] = 0x2824e, [0x1006] = 0x2824f, [0x1007] = 0x28250,
  [0x1008] = 0x28319, [0x1009] = 0x2831a, [0x100a] = 0x2831b, [0x100c] = 0x2831c,
  [0x100e] = 0x28419, [0x100f] = 0x2841a, [0x1010] = 0x4886, [0x1011] = 0x2898a,
  [0x1012] = 0x2856e, [0x1013] = 0x2856f, [0x1014] = 0x9064, [0x1015] = 0x28570,
  [0x1016] = 0x28571, [0x1017] = 0x28572, [0x1018] = 0x28573, [0x1019] = 0x28574,
  [0x101a] = 0x28575, [0x101b] = 0x28576, [0x101c] = 0x2883a, [0x101d] = 0x2883b,
  [0x101e] = 0x2883c, [0x101f] = 0x2885a, [0x1020] = 0x2883d, [0x1021] = 0x28842,
  [0x1022] = 0x2898b, [0x1023] = 0x289a0, [0x1024] = 0x2898c, [0x1025] = 0x2898d,
  [0x1026] = 0x2898e, [0x1027] = 0x289ab, [0x1028] = 0x2898f, [0x1029] = 0x9277,
  [0x102a] = 0x28990, [0x102b] = 0x28991, [0x102c] = 0x28992, [0x102d] = 0x289aa,
  [0x102e] = 0x92af, [0x102f] = 0x28993, [0x1030] = 0x28994, [0x1031] = 0x2899e,
  [0x1032] = 0x289a8, [0x1035] = 0x28f00, [0x1036] = 0x28f01, [0x1037] = 0x28f02,
  [0x1038] = 0x28f03, [0x1039] = 0x28f04, [0x103a] = 0x28f05, [0x103b] = 0x28fc4,
  [0x103c] = 0x28fc5, [0x103e] = 0x290ae, [0x103f] = 0x290b2, [0x1042] = 0x291f7,
  [0x1043] = 0x29213, [0x1044] = 0x2939e, [0x1045] = 0x29485, [0x1046] = 0x25380,
  [0x1047] = 0x294c8, [0x1048] = 0x29841, [0x1049] = 0x29881, [0x104a] = 0x29882,
  [0x104b] = 0x29883, [0x104c] = 0x29884, [0x104d] = 0x29ac4, [0x104e] = 0x29ac5,
  [0x104f] = 0x29ac6, [0x1050] = 0x29c95, [0x1051] = 0x29c96, [0x1052] = 0x29c97,
  [0x1053] = 0x29d58, [0x1054] = 0x29d59, [0x1056] = 0x29fa9, [0x1057] = 0x29faa,
  [0x1058] = 0x29fab, [0x1059] = 0x2a28e, [0x105a] = 0x2a303, [0x105b] = 0x2a3b3,
  [0x105c] = 0x20b82, [0x105d] = 0x20053, [0x105e] = 0x20183, [0x105f] = 0x20184,
  [0x1060] = 0x20185, [0x1061] = 0x203cc, [0x1062] = 0x203cd, [0x1063] = 0x203ce,
  [0x1064] = 0x203cf, [0x1065] = 0x203d0, [0x1067] = 0x203d1, [0x1068] = 0x203d2,
  [0x1069] = 0x203da, [0x106a] = 0x204b3, [0x106b] = 0x204b4, [0x106d] = 0x20613,
  [0x106e] = 0x2060e, [0x106f] = 0x2066b, [0x1070] = 0x20903, [0x1071] = 0x20961,
  [0x1072] = 0x209c1, [0x1073] = 0x209c2, [0x1074] = 0x20ab5, [0x1075] = 0x20140,
  [0x1076] = 0x20f98, [0x1077] = 0x20f99, [0x1078] = 0x20f9a, [0x1079] = 0x20f9b,
  [0x107a] = 0x20f9c, [0x107b] = 0x20f9d, [0x107c] = 0x20f9e, [0x107d] = 0x20f9f,
  [0x107e] = 0x20fee, [0x107f] = 0x20fa0, [0x1080] = 0x20fa1, [0x1081] = 0x20f86,
  [0x1082] = 0x21225, [0x1083] = 0x21434, [0x1084] = 0x21435, [0x1085] = 0x21451,
  [0x1086] = 0x2142d, [0x1087] = 0x24362, [0x1088] = 0x21454, [0x1089] = 0x21452,
  [0x108a] = 0x366d, [0x108b] = 0x21436, [0x108c] = 0x21437, [0x108d] = 0x21438,
  [0x108e] = 0x21439, [0x108f] = 0x2143a, [0x1090] = 0x2143b, [0x1091] = 0x21468,
  [0x1092] = 0x2143c, [0x1093] = 0x366e, [0x1094] = 0x21577, [0x1095] = 0x23296,
  [0x1096] = 0x2167e, [0x1097] = 0x2167f, [0x1098] = 0x21869, [0x1099] = 0x2186a,
  [0x109a] = 0x2186b, [0x109b] = 0x2186c, [0x109c] = 0x2186d, [0x109d] = 0x2186e,
  [0x109e] = 0x2186f, [0x109f] = 0x2185d, [0x10a0] = 0x21870, [0x10a1] = 0x21885,
  [0x10a2] = 0x21871, [0x10a3] = 0x21872, [0x10a4] = 0x21873, [0x10a5] = 0x21874,
  [0x10a6] = 0x21875, [0x10a7] = 0x21876, [0x10a8] = 0x21884, [0x10a9] = 0x21877,
  [0x10aa] = 0x21883, [0x10ab] = 0x2199a, [0x10ac] = 0x21aa7, [0x10ad] = 0x21aa8,
  [0x10ae] = 0x21aa9, [0x10af] = 0x21aaa, [0x10b0] = 0x21aab, [0x10b1] = 0x21aac,
  [0x10b2] = 0x21ab7, [0x10b3] = 0x21aad, [0x10b4] = 0x21aae, [0x10b6] = 0x21b45,
  [0x10b7] = 0x21cd3, [0x10b8] = 0x21cd4, [0x10b9] = 0x21cd5, [0x10ba] = 0x21cd6,
  [0x10bb] = 0x21cd7, [0x10bc] = 0x21f26, [0x10bd] = 0x21f27, [0x10be] = 0x21f28,
  [0x10c0] = 0x21f29, [0x10c1] = 0x21f2a, [0x10c2] = 0x22005, [0x10c3] = 0x5e64,
  [0x10c4] = 0x221d0, [0x10c5] = 0x222b7, [0x10c7] = 0x222b8, [0x10c8] = 0x222b9,
  [0x10c9] = 0x22422, [0x10ca] = 0x22423, [0x10cb] = 0x22573, [0x10cc] = 0x22574,
  [0x10cd] = 0x2288f, [0x10ce] = 0x22812, [0x10cf] = 0x22813, [0x10d0] = 0x22814,
  [0x10d1] = 0x22815, [0x10d2] = 0x22816, [0x10d3] = 0x22817, [0x10d4] = 0x22818,
  [0x10d5] = 0x22861, [0x10d6] = 0x22866, [0x10d7] = 0x22867, [0x10d8] = 0x22819,
  [0x10d9] = 0x22868, [0x10da] = 0x2281a, [0x10db] = 0x22869, [0x10dc] = 0x22a04,
  [0x10dd] = 0x22a05, [0x10df] = 0x22a06, [0x10e0] = 0x229fd, [0x10e1] = 0x22d50,
  [0x10e2] = 0x22d51, [0x10e3] = 0x22d52, [0x10e4] = 0x22d53, [0x10e5] = 0x22d54,
  [0x10e6] = 0x22d55, [0x10e7] = 0x22d63, [0x10e8] = 0x2f8c4, [0x10e9] = 0x22d56,
  [0x10ea] = 0x22d57, [0x10eb] = 0x22d58, [0x10ec] = 0x22d59, [0x10ed] = 0x22d5a,
  [0x10ee] = 0x22d5b, [0x10f0] = 0x22d5c, [0x10f1] = 0x22d5d, [0x10f2] = 0x22d5e,
  [0x10f3] = 0x22d5f, [0x10f4] = 0x22fd7, [0x10f6] = 0x23060, [0x10f7] = 0x23061,
  [0x10f8] = 0x23117, [0x10f9] = 0x23139, [0x10fa] = 0x2328f, [0x10fb] = 0x232c8,
  [0x10fc] = 0x23290, [0x10fd] = 0x23291, [0x10fe] = 0x23398, [0x10ff] = 0x23399,
  [0x1100] = 0x23644, [0x1101] = 0x23645, [0x1102] = 0x23646, [0x1103] = 0x23659,
  [0x1104] = 0x236ee, [0x1105] = 0x6a2b, [0x1106] = 0x23647, [0x1107] = 0x23648,
  [0x1109] = 0x23649, [0x110a] = 0x2364a, [0x110b] = 0x2364b, [0x110c] = 0x2364c,
  [0x110d] = 0x2364d, [0x110e] = 0x2364e, [0x110f] = 0x2364f, [0x1110] = 0x23650,
  [0x1111] = 0x23651, [0x1112] = 0x23652, [0x1113] = 0x23653, [0x1114] = 0x23654,
  [0x1115] = 0x23655, [0x1116] = 0x23656, [0x1117] = 0x23915, [0x1118] = 0x2399a,
  [0x1119] = 0x23a58, [0x111a] = 0x23afc, [0x111b] = 0x23afd, [0x111c] = 0x23f86,
  [0x111d] = 0x23f87, [0x111e] = 0x23f88, [0x111f] = 0x23f89, [0x1120] = 0x23f8a,
  [0x1121] = 0x23f8b, [0x1122] = 0x23f8c, [0x1124] = 0x2401f, [0x1125] = 0x23f8d,
  [0x1126] = 0x23f8e, [0x1127] = 0x23f8f, [0x1128] = 0x23f90, [0x1129] = 0x23f91,
  [0x112a] = 0x23f92, [0x112b] = 0x23f93, [0x112c] = 0x2f911, [0x112d] = 0x23f94,
  [0x112e] = 0x23f95, [0x112f] = 0x23eca, [0x1130] = 0x23f18, [0x1132] = 0x23f96,
  [0x1133] = 0x23f97, [0x1134] = 0x23f98, [0x1135] = 0x23f99, [0x1136] = 0x23f9a,
  [0x1137] = 0x23f9b, [0x1138] = 0x23f9c, [0x1139] = 0x23f9d, [0x113a] = 0x23f9e,
  [0x113b] = 0x23f9f, [0x113c] = 0x6f46, [0x113e] = 0x23fa0, [0x113f] = 0x23fa1,
  [0x1140] = 0x6f9a, [0x1141] = 0x23fb7, [0x1143] = 0x24396, [0x1144] = 0x24368,
  [0x1145] = 0x24369, [0x1146] = 0x2438c, [0x1147] = 0x2436a, [0x1148] = 0x2436b,
  [0x1149] = 0x2436c, [0x114a] = 0x2436d, [0x114b] = 0x2436e, [0x114c] = 0x2436f,
  [0x114d] = 0x2437f, [0x114e] = 0x24384, [0x114f] = 0x24370, [0x1150] = 0x24371,
  [0x1151] = 0x24385, [0x1152] = 0x24372, [0x1153] = 0x24373, [0x1154] = 0x24374,
  [0x1155] = 0x24375, [0x1156] = 0x24376, [0x1158] = 0x24377, [0x1159] = 0x24378,
  [0x115a] = 0x24379, [0x115b] = 0x2437a, [0x115c] = 0x2437b, [0x115d] = 0x2437c,
  [0x115e] = 0x2f91e, [0x115f] = 0x24594, [0x1160] = 0x246d5, [0x1161] = 0x24829,
  [0x1162] = 0x24a2b, [0x1163] = 0x24a2c, [0x1164] = 0x24a2d, [0x1165] = 0x24a2e,
  [0x1166] = 0x24a2f, [0x1167] = 0x24a30, [0x1168] = 0x24a3e, [0x1169] = 0x24a31,
  [0x116a] = 0x24a32, [0x116b] = 0x24a33, [0x116c] = 0x24a34, [0x116d] = 0x24a35,
  [0x116e] = 0x24b11, [0x116f] = 0x3f53, [0x1170] = 0x24cc0, [0x1171] = 0x24e42,
  [0x1172] = 0x24e43, [0x1173] = 0x24e44, [0x1174] = 0x24e45, [0x1175] = 0x24e46,
  [0x1176] = 0x2508c, [0x1177] = 0x2508d, [0x1178] = 0x2508e, [0x1179] = 0x2508f,
  [0x117a] = 0x25090, [0x117b] = 0x25091, [0x117c] = 0x25263, [0x117d] = 0x25264,
  [0x117e] = 0x25265, [0x1180] = 0x2f94a, [0x1182] = 0x253f6, [0x1183] = 0x25536,
  [0x1184] = 0x25537, [0x1185] = 0x2552c, [0x1186] = 0x25538, [0x1187] = 0x25539,
  [0x1189] = 0x2553a, [0x118a] = 0x256e3, [0x118b] = 0x256e2, [0x118c] = 0x256f5,
  [0x118d] = 0x25847, [0x118f] = 0x25848, [0x1190] = 0x25849, [0x1191] = 0x2584a,
  [0x1192] = 0x2584b, [0x1193] = 0x2584c, [0x1194] = 0x25aa8, [0x1195] = 0x25aa9,
  [0x1196] = 0x25aaa, [0x1198] = 0x25c0b, [0x1199] = 0x25c06, [0x119a] = 0x25c73,
  [0x119b] = 0x25c0c, [0x119c] = 0x25c0d, [0x119d] = 0x25c0e, [0x119e] = 0x25c0f,
  [0x119f] = 0x25c10, [0x11a1] = 0x25c11, [0x11a2] = 0x2f962, [0x11a3] = 0x25c12,
  [0x11a4] = 0x25c14, [0x11a5] = 0x25c15, [0x11a6] = 0x25c16, [0x11a8] = 0x25c13,
  [0x11a9] = 0x25c17, [0x11aa] = 0x25ed9, [0x11ab] = 0x25eda, [0x11ac] = 0x25edb,
  [0x11ae] = 0x25edc, [0x11af] = 0x260a6, [0x11b0] = 0x260a7, [0x11b1] = 0x260a8,
  [0x11b2] = 0x260a9, [0x11b3] = 0x260aa, [0x11b4] = 0x260ab, [0x11b6] = 0x2f96d,
  [0x11b7] = 0x260ac, [0x11b8] = 0x260ad, [0x11b9] = 0x260ae, [0x11bc] = 0x260af,
  [0x11bd] = 0x260b0, [0x11be] = 0x260b1, [0x11bf] = 0x26244, [0x11c0] = 0x26245,
  [0x11c1] = 0x262f7, [0x11c2] = 0x262f8, [0x11c3] = 0x2646b, [0x11c4] = 0x2646c,
  [0x11c5] = 0x26461, [0x11c6] = 0x26512, [0x11c7] = 0x265af, [0x11c8] = 0x265b0,
  [0x11c9] = 0x265b1, [0x11ca] = 0x265b2, [0x11cb] = 0x265b3, [0x11cc] = 0x265b4,
  [0x11cd] = 0x265b5, [0x11ce] = 0x265b6, [0x11cf] = 0x26622, [0x11d0] = 0x267ea,
  [0x11d1] = 0x267eb, [0x11d2] = 0x267ec, [0x11d3] = 0x267ed, [0x11d4] = 0x443d,
  [0x11d5] = 0x268ed, [0x11d6] = 0x2691b, [0x11d7] = 0x26947, [0x11d8] = 0x269e8,
  [0x11d9] = 0x26ad0, [0x11da] = 0x26ae7, [0x11db] = 0x26ae8, [0x11dc] = 0x26ae9,
  [0x11dd] = 0x26e4e, [0x11de] = 0x26e4f, [0x11df] = 0x26e50, [0x11e0] = 0x26e72,
  [0x11e1] = 0x26e51, [0x11e2] = 0x26e52, [0x11e3] = 0x2f9a7, [0x11e4] = 0x4526,
  [0x11e5] = 0x26e53, [0x11e6] = 0x26e54, [0x11e7] = 0x26e55, [0x11e8] = 0x26e56,
  [0x11e9] = 0x26e57, [0x11ea] = 0x26e77, [0x11eb] = 0x26e58, [0x11ec] = 0x26e59,
  [0x11ed] = 0x26e5a, [0x11ee] = 0x26e5b, [0x11ef] = 0x26e5c, [0x11f0] = 0x26e6e,
  [0x11f1] = 0x26e5d, [0x11f2] = 0x4527, [0x11f4] = 0x26e5e, [0x11f5] = 0x4528,
  [0x11f6] = 0x26e5f, [0x11f8] = 0x271d7, [0x11f9] = 0x27389, [0x11fa] = 0x2738a,
  [0x11fc] = 0x2f9be, [0x11fd] = 0x2738b, [0x11fe] = 0x2738c, [0x11ff] = 0x2738d,
  [0x1200] = 0x2738e, [0x1201] = 0x2f9bc, [0x1202] = 0x2738f, [0x1204] = 0x2739a,
  [0x1205] = 0x27390, [0x1206] = 0x27391, [0x1207] = 0x27392, [0x1208] = 0x27393,
  [0x1209] = 0x27712, [0x120a] = 0x276f1, [0x120c] = 0x277fe, [0x120d] = 0x277fb,
  [0x120e] = 0x27857, [0x120f] = 0x27a49, [0x1210] = 0x27a4a, [0x1211] = 0x27a4b,
  [0x1212] = 0x27a4c, [0x1214] = 0x27a4d, [0x1215] = 0x27a59, [0x1216] = 0x27a4e,
  [0x1217] = 0x27a4f, [0x1218] = 0x27bf4, [0x1219] = 0x27c65, [0x121a] = 0x27da6,
  [0x121b] = 0x27da7, [0x121c] = 0x27da8, [0x121d] = 0x27f22, [0x121e] = 0x27f23,
  [0x121f] = 0x2808b, [0x1220] = 0x2808c, [0x1221] = 0x2808d, [0x1222] = 0x2808e,
  [0x1223] = 0x28258, [0x1225] = 0x28259, [0x1226] = 0x2833b, [0x1227] = 0x2833c,
  [0x1228] = 0x2833d, [0x1229] = 0x2833e, [0x122a] = 0x28448, [0x122b] = 0x28594,
  [0x122c] = 0x28595, [0x122d] = 0x28596, [0x122e] = 0x28597, [0x122f] = 0x28598,
  [0x1230] = 0x2877d, [0x1231] = 0x286fe, [0x1232] = 0x2877e, [0x1233] = 0x2877f,
  [0x1235] = 0x2885b, [0x1236] = 0x2885c, [0x1237] = 0x289e4, [0x1238] = 0x289c1,
  [0x1239] = 0x289fd, [0x123a] = 0x289c2, [0x123b] = 0x289d6, [0x123c] = 0x289c3,
  [0x123d] = 0x289c4, [0x123f] = 0x289dc, [0x1240] = 0x289c5, [0x1241] = 0x289c6,
  [0x1242] = 0x289c7, [0x1243] = 0x289e1, [0x1244] = 0x289c8, [0x1245] = 0x289c9,
  [0x1246] = 0x289ca, [0x1247] = 0x289cb, [0x1248] = 0x92f2, [0x1249] = 0x289cc,
  [0x124a] = 0x289cd, [0x124b] = 0x28d35, [0x124c] = 0x28d36, [0x124d] = 0x28d47,
  [0x124e] = 0x28d37, [0x124f] = 0x28f1b, [0x1250] = 0x28f1c, [0x1251] = 0x28f1d,
  [0x1252] = 0x28f1e, [0x1253] = 0x28f1f, [0x1254] = 0x28f20, [0x1255] = 0x28f7b,
  [0x1256] = 0x28f7a, [0x1257] = 0x28ffe, [0x1258] = 0x290c6, [0x1259] = 0x290c7,
  [0x125a] = 0x290c8, [0x125b] = 0x290c9, [0x125c] = 0x290ca, [0x125e] = 0x294a0,
  [0x125f] = 0x294a1, [0x1260] = 0x295cc, [0x1261] = 0x295cd, [0x1262] = 0x296f1,
  [0x1263] = 0x296d7, [0x1264] = 0x29842, [0x1265] = 0x29846, [0x1266] = 0x298a3,
  [0x1267] = 0x298a5, [0x1268] = 0x298a6, [0x1269] = 0x29acb, [0x126a] = 0x29acc,
  [0x126b] = 0x29b26, [0x126c] = 0x29b27, [0x126d] = 0x29b28, [0x126e] = 0x29b29,
  [0x126f] = 0x29cae, [0x1270] = 0x29caf, [0x1271] = 0x29cb0, [0x1272] = 0x29d73,
  [0x1274] = 0x29d7c, [0x1275] = 0x29d74, [0x1277] = 0x29f80, [0x1278] = 0x29d75,
  [0x1279] = 0x9b79, [0x127b] = 0x29fcf, [0x127d] = 0x2a38f, [0x127e] = 0x20670,
  [0x127f] = 0x20085, [0x1280] = 0x20397, [0x1281] = 0x203f6, [0x1282] = 0x203f7,
  [0x1283] = 0x203f8, [0x1284] = 0x203fa, [0x1285] = 0x203fb, [0x1286] = 0x204ba,
  [0x1287] = 0x204bb, [0x1288] = 0x204bc, [0x1289] = 0x204bd, [0x128a] = 0x20534,
  [0x128b] = 0x20535, [0x128c] = 0x205a2, [0x128d] = 0x20614, [0x128e] = 0x20611,
  [0x128f] = 0x2066e, [0x1290] = 0x2066f, [0x1291] = 0x20808, [0x1292] = 0x208b8,
  [0x1293] = 0x20a09, [0x1294] = 0x20abd, [0x1295] = 0x20abe, [0x1296] = 0x20b88,
  [0x1297] = 0x20ffe, [0x1298] = 0x567a, [0x1299] = 0x20fff, [0x129a] = 0x21000,
  [0x129b] = 0x21001, [0x129c] = 0x21002, [0x129d] = 0x21003, [0x129e] = 0x21004,
  [0x129f] = 0x21005, [0x12a0] = 0x21006, [0x12a2] = 0x21007, [0x12a3] = 0x21008,
  [0x12a4] = 0x21009, [0x12a5] = 0x2100a, [0x12a6] = 0x2100b, [0x12a7] = 0x2100c,
  [0x12a8] = 0x2100d, [0x12a9] = 0x2100e, [0x12aa] = 0x2100f, [0x12ae] = 0x2122c,
  [0x12af] = 0x2146b, [0x12b1] = 0x2146c, [0x12b2] = 0x21487, [0x12b3] = 0x2146d,
  [0x12b4] = 0x21578, [0x12b5] = 0x2168b, [0x12b6] = 0x218a4, [0x12b7] = 0x218a5,
  [0x12b8] = 0x218a6, [0x12b9] = 0x218a7, [0x12ba] = 0x218be, [0x12bb] = 0x218a8,
  [0x12bc] = 0x218a9, [0x12bd] = 0x218aa, [0x12be] = 0x2188d, [0x12bf] = 0x218ab,
  [0x12c0] = 0x218ac, [0x12c1] = 0x218bf, [0x12c3] = 0x372c, [0x12c4] = 0x218ad,
  [0x12c5] = 0x218ae, [0x12c6] = 0x219a4, [0x12c7] = 0x219a5, [0x12c8] = 0x21ac7,
  [0x12c9] = 0x21ac8, [0x12ca] = 0x21ac9, [0x12cb] = 0x27dbc, [0x12cc] = 0x21aca,
  [0x12cd] = 0x21acb, [0x12ce] = 0x21acc, [0x12cf] = 0x21acd, [0x12d0] = 0x21ace,
  [0x12d1] = 0x21c19, [0x12d2] = 0x21ce6, [0x12d3] = 0x21f4c, [0x12d4] = 0x21f4d,
  [0x12d5] = 0x21f4e, [0x12d6] = 0x22006, [0x12d8] = 0x222cd, [0x12d9] = 0x222ce,
  [0x12da] = 0x222cf, [0x12db] = 0x222d0, [0x12dc] = 0x5f5c, [0x12dd] = 0x22581,
  [0x12de] = 0x22582, [0x12df] = 0x20533, [0x12e0] = 0x2286a, [0x12e1] = 0x2286b,
  [0x12e2] = 0x228b2, [0x12e3] = 0x2286c, [0x12e4] = 0x2286d, [0x12e5] = 0x228cc,
  [0x12e6] = 0x228ae, [0x12e7] = 0x228b3, [0x12e8] = 0x22891, [0x12ea] = 0x2286e,
  [0x12eb] = 0x228b4, [0x12ec] = 0x2286f, [0x12ed] = 0x22dc9, [0x12ee] = 0x22da7,
  [0x12ef] = 0x22da8, [0x12f0] = 0x22da9, [0x12f1] = 0x22daa, [0x12f2] = 0x22dab,
  [0x12f3] = 0x22dac, [0x12f4] = 0x22dad, [0x12f5] = 0x22dae, [0x12f6] = 0x22daf,
  [0x12f7] = 0x22db0, [0x12f8] = 0x22d60, [0x12f9] = 0x22db1, [0x12fa] = 0x22ee1,
  [0x12fb] = 0x22fe3, [0x12fc] = 0x22ff6, [0x12fd] = 0x65d9, [0x12fe] = 0x232af,
  [0x12ff] = 0x232b0, [0x1300] = 0x232b1, [0x1301] = 0x232ba, [0x1302] = 0x232b2,
  [0x1303] = 0x232b3, [0x1304] = 0x2334c, [0x1306] = 0x233a0, [0x1307] = 0x233a1,
  [0x1308] = 0x236b0, [0x1309] = 0x236b1, [0x130a] = 0x236b2, [0x130b] = 0x236b3,
  [0x130c] = 0x236b4, [0x130d] = 0x2f8eb, [0x130e] = 0x236b5, [0x130f] = 0x236b6,
  [0x1310] = 0x236b7, [0x1311] = 0x236b8, [0x1312] = 0x236b9, [0x1313] = 0x236ba,
  [0x1314] = 0x6a72, [0x1315] = 0x236bb, [0x1316] = 0x236bc, [0x1317] = 0x236bd,
  [0x1318] = 0x236be, [0x1319] = 0x236bf, [0x131a] = 0x236df, [0x131b] = 0x236c0,
  [0x131c] = 0x236c1, [0x131d] = 0x236c2, [0x131e] = 0x236c3, [0x131f] = 0x236c4,
  [0x1320] = 0x6a78, [0x1322] = 0x236c5, [0x1323] = 0x236c6, [0x1324] = 0x236c7,
  [0x1325] = 0x236c8, [0x1326] = 0x236c9, [0x1327] = 0x236ca, [0x1328] = 0x2373c,
  [0x1329] = 0x236cb, [0x132a] = 0x236cc, [0x132b] = 0x236cd, [0x132c] = 0x236ce,
  [0x132d] = 0x236cf, [0x132e] = 0x236d0, [0x132f] = 0x236d1, [0x1330] = 0x6b5a,
  [0x1331] = 0x2391d, [0x1332] = 0x2391e, [0x1333] = 0x23a64, [0x1334] = 0x23a65,
  [0x1335] = 0x23ac5, [0x1336] = 0x23ac6, [0x1337] = 0x23ac7, [0x1338] = 0x23aff,
  [0x1339] = 0x23bff, [0x133a] = 0x23c00, [0x133b] = 0x23c01, [0x133c] = 0x23c02,
  [0x133d] = 0x23ff5, [0x133e] = 0x2405c, [0x133f] = 0x23ff6, [0x1340] = 0x23ff7,
  [0x1341] = 0x23fb0, [0x1342] = 0x23ff8, [0x1343] = 0x23ff9, [0x1344] = 0x2405d,
  [0x1345] = 0x24050, [0x1346] = 0x23ffa, [0x1347] = 0x23ffb, [0x1348] = 0x23fa2,
  [0x1349] = 0x23ffc, [0x134a] = 0x23ffd, [0x134b] = 0x23ffe, [0x134c] = 0x23fff,
  [0x134d] = 0x2f913, [0x134e] = 0x24000, [0x134f] = 0x2f91f, [0x1350] = 0x243c1,
  [0x1351] = 0x243c2, [0x1352] = 0x243c3, [0x1353] = 0x243c4, [0x1354] = 0x243c5,
  [0x1355] = 0x243c6, [0x1356] = 0x243c7, [0x1357] = 0x243c8, [0x1358] = 0x243c9,
  [0x1359] = 0x243ca, [0x135a] = 0x243cb, [0x135b] = 0x290ef, [0x135c] = 0x243d7,
  [0x135d] = 0x243cc, [0x135e] = 0x243cd, [0x135f] = 0x243ce, [0x1360] = 0x24422,
  [0x1361] = 0x243cf, [0x1362] = 0x243d0, [0x1364] = 0x243d1, [0x1365] = 0x243d2,
  [0x1366] = 0x243e9, [0x1367] = 0x243d3, [0x1368] = 0x243d4, [0x1369] = 0x243d5,
  [0x136a] = 0x24538, [0x136b] = 0x2103b, [0x136c] = 0x2487d, [0x136d] = 0x2056f,
  [0x136e] = 0x248a1, [0x136f] = 0x24a53, [0x1370] = 0x24a54, [0x1371] = 0x3efb,
  [0x1372] = 0x24a55, [0x1373] = 0x24a3c, [0x1374] = 0x24a56, [0x1376] = 0x24a66,
  [0x1377] = 0x3efc, [0x1378] = 0x24a57, [0x1379] = 0x24a58, [0x137a] = 0x24a59,
  [0x137b] = 0x24a5a, [0x137c] = 0x24a5b, [0x137d] = 0x24a67, [0x137e] = 0x24a5c,
  [0x1381] = 0x24a6a, [0x1382] = 0x24a5d, [0x1383] = 0x24a3d, [0x1384] = 0x24a5e,
  [0x1385] = 0x24a65, [0x1386] = 0x24ce1, [0x1387] = 0x24ccd, [0x1388] = 0x2a011,
  [0x1389] = 0x24e6f, [0x138a] = 0x24e70, [0x138b] = 0x2501a, [0x138c] = 0x2509a,
  [0x138d] = 0x2509b, [0x138e] = 0x2509c, [0x138f] = 0x2509e, [0x1390] = 0x2509f,
  [0x1391] = 0x2f93f, [0x1392] = 0x2528e, [0x1394] = 0x253fd, [0x1396] = 0x2555e,
  [0x1397] = 0x2555f, [0x1398] = 0x25560, [0x1399] = 0x25565, [0x139a] = 0x25567,
  [0x139b] = 0x25568, [0x139c] = 0x25569, [0x139e] = 0x2556a, [0x139f] = 0x2558f,
  [0x13a0] = 0x2556b, [0x13a1] = 0x2556c, [0x13a2] = 0x2556d, [0x13a3] = 0x256f7,
  [0x13a4] = 0x256f8, [0x13a5] = 0x256f9, [0x13a6] = 0x256e4, [0x13a7] = 0x256fa,
  [0x13a8] = 0x258b1, [0x13a9] = 0x25875, [0x13aa] = 0x25876, [0x13ab] = 0x259ff,
  [0x13ac] = 0x25a00, [0x13ad] = 0x25a01, [0x13ae] = 0x25a02, [0x13af] = 0x25a03,
  [0x13b0] = 0x25aae, [0x13b1] = 0x25aad, [0x13b2] = 0x25c51, [0x13b3] = 0x25c52,
  [0x13b5] = 0x25c53, [0x13b6] = 0x25c54, [0x13b7] = 0x25c55, [0x13b8] = 0x25c56,
  [0x13b9] = 0x25c57, [0x13ba] = 0x25c58, [0x13bb] = 0x25c59, [0x13bc] = 0x25c5a,
  [0x13bd] = 0x25c5b, [0x13be] = 0x25c5c, [0x13bf] = 0x25c5d, [0x13c0] = 0x25c5e,
  [0x13c1] = 0x25c5f, [0x13c2] = 0x25c60, [0x13c3] = 0x25c61, [0x13c4] = 0x25c62,
  [0x13c5] = 0x25c63, [0x13c6] = 0x25c64, [0x13c7] = 0x25c4c, [0x13c8] = 0x25c66,
  [0x13c9] = 0x25ef1, [0x13ca] = 0x25ef2, [0x13cb] = 0x260f2, [0x13cc] = 0x260f3,
  [0x13cd] = 0x260f4, [0x13ce] = 0x260f5, [0x13cf] = 0x260f6, [0x13d0] = 0x260f7,
  [0x13d1] = 0x260f8, [0x13d2] = 0x260f9, [0x13d3] = 0x260fa, [0x13d4] = 0x2624c,
  [0x13d5] = 0x2624b, [0x13d6] = 0x2631b, [0x13d7] = 0x263ce, [0x13d8] = 0x263b3,
  [0x13d9] = 0x2647f, [0x13da] = 0x2f979, [0x13db] = 0x2f97c, [0x13dc] = 0x2651a,
  [0x13dd] = 0x265c4, [0x13de] = 0x265c5, [0x13df] = 0x265c6, [0x13e0] = 0x265c7,
  [0x13e1] = 0x265c8, [0x13e2] = 0x265c9, [0x13e4] = 0x26819, [0x13e5] = 0x2681c,
  [0x13e6] = 0x2681a, [0x13e7] = 0x2694c, [0x13e8] = 0x2698e, [0x13e9] = 0x26a7e,
  [0x13ea] = 0x26eeb, [0x13eb] = 0x26eec, [0x13ec] = 0x26eed, [0x13ed] = 0x26eee,
  [0x13ee] = 0x26eef, [0x13ef] = 0x26ef0, [0x13f0] = 0x26ef1, [0x13f1] = 0x26ed8,
  [0x13f2] = 0x26ed9, [0x13f3] = 0x26eda, [0x13f4] = 0x26edb, [0x13f5] = 0x26ef2,
  [0x13f6] = 0x26ef3, [0x13f7] = 0x26ef4, [0x13f8] = 0x26f07, [0x13fa] = 0x2f9ac,
  [0x13fc] = 0x26ef5, [0x13fd] = 0x26f15, [0x13ff] = 0x26ef6, [0x1400] = 0x26ef7,
  [0x1401] = 0x26ef8, [0x1402] = 0x26ef9, [0x1403] = 0x26efa, [0x1404] = 0x26efb,
  [0x1405] = 0x26efc, [0x1406] = 0x26efd, [0x1407] = 0x26efe, [0x1408] = 0x26fa7,
  [0x1409] = 0x26eff, [0x140a] = 0x26f00, [0x140b] = 0x26f01, [0x140c] = 0x26f02,
  [0x140d] = 0x26f03, [0x140e] = 0x26f04, [0x140f] = 0x26f05, [0x1410] = 0x271e9,
  [0x1411] = 0x271ea, [0x1412] = 0x271eb, [0x1413] = 0x273de, [0x1414] = 0x273df,
  [0x1416] = 0x273e0, [0x1417] = 0x273e1, [0x1418] = 0x273e2, [0x1419] = 0x273e3,
  [0x141a] = 0x2773e, [0x141b] = 0x2773f, [0x141c] = 0x27740, [0x141d] = 0x27741,
  [0x141e] = 0x27871, [0x141f] = 0x27872, [0x1420] = 0x27873, [0x1421] = 0x27a88,
  [0x1422] = 0x27a89, [0x1423] = 0x27a8a, [0x1424] = 0x27a8b, [0x1425] = 0x27a8c,
  [0x1426] = 0x27a8d, [0x1427] = 0x27a8e, [0x1428] = 0x27a8f, [0x1429] = 0x27a90,
  [0x142a] = 0x27a91, [0x142b] = 0x27a92, [0x142d] = 0x27cf1, [0x142e] = 0x27dbe,
  [0x142f] = 0x27dbf, [0x1430] = 0x27dc0, [0x1431] = 0x27dc1, [0x1432] = 0x27dc2,
  [0x1433] = 0x27dc3, [0x1434] = 0x27e6e, [0x1435] = 0x27f37, [0x1436] = 0x27f57,
  [0x1437] = 0x27f38, [0x1438] = 0x27f39, [0x1439] = 0x27f3a, [0x143a] = 0x27f3b,
  [0x143b] = 0x27f3c, [0x143c] = 0x27f3d, [0x143d] = 0x27f3e, [0x143e] = 0x27f27,
  [0x143f] = 0x27f3f, [0x1442] = 0x280c0, [0x1443] = 0x280c1, [0x1444] = 0x8ebe,
  [0x1445] = 0x28267, [0x1447] = 0x28268, [0x1448] = 0x28358, [0x1449] = 0x28365,
  [0x144a] = 0x28359, [0x144b] = 0x2835a, [0x144c] = 0x28424, [0x144d] = 0x285cd,
  [0x144e] = 0x285ce, [0x144f] = 0x285cf, [0x1450] = 0x285d2, [0x1451] = 0x2879f,
  [0x1452] = 0x28722, [0x1453] = 0x2887d, [0x1454] = 0x2887e, [0x1455] = 0x289ff,
  [0x1456] = 0x28a25, [0x1457] = 0x28a00, [0x1458] = 0x28a01, [0x1459] = 0x933b,
  [0x145a] = 0x28a02, [0x145b] = 0x28a03, [0x145c] = 0x28a04, [0x145d] = 0xfa28,
  [0x145e] = 0x28a05, [0x145f] = 0x28a06, [0x1460] = 0x28a07, [0x1461] = 0x28a16,
  [0x1462] = 0x28a08, [0x1463] = 0x28a17, [0x1464] = 0x28a09, [0x1465] = 0x28a0a,
  [0x1466] = 0x9340, [0x1467] = 0x28a0b, [0x1468] = 0x28a0c, [0x1469] = 0x28a29,
  [0x146a] = 0x933a, [0x146b] = 0x28a0d, [0x146d] = 0x28a0e, [0x146e] = 0x28a10,
  [0x146f] = 0x28a11, [0x1470] = 0x28a1d, [0x1471] = 0x28a12, [0x1472] = 0x28a13,
  [0x1474] = 0x28a14, [0x1475] = 0x28a15, [0x1476] = 0x28c8d, [0x1477] = 0x28d54,
  [0x1478] = 0x28d55, [0x1479] = 0x28d56, [0x147a] = 0x28f33, [0x147b] = 0x28f34,
  [0x147c] = 0x28f35, [0x147d] = 0x28f36, [0x147e] = 0x28ffa, [0x147f] = 0x290e3,
  [0x1480] = 0x290e6, [0x1481] = 0x290ed, [0x1482] = 0x290e7, [0x1483] = 0x290e8,
  [0x1484] = 0x290e9, [0x1485] = 0x290ea, [0x1486] = 0x290eb, [0x1487] = 0x29227,
  [0x1488] = 0x29220, [0x1489] = 0x2f9f8, [0x148a] = 0x2942a, [0x148b] = 0x294c9,
  [0x148d] = 0x294ca, [0x148e] = 0x294cb, [0x148f] = 0x294d9, [0x1490] = 0x294cc,
  [0x1491] = 0x294cd, [0x1492] = 0x294ce, [0x1494] = 0x295e1, [0x1495] = 0x2971b,
  [0x1496] = 0x2971c, [0x1497] = 0x2971d, [0x1498] = 0x2971e, [0x1499] = 0x29847,
  [0x149a] = 0x298d1, [0x149b] = 0x298c9, [0x149c] = 0x29ad0, [0x149d] = 0x29ad1,
  [0x149e] = 0x29b3f, [0x149f] = 0x29b40, [0x14a0] = 0x29b41, [0x14a1] = 0x2fa09,
  [0x14a2] = 0x29b42, [0x14a3] = 0x29cc4, [0x14a4] = 0x29d9b, [0x14a5] = 0x29d92,
  [0x14a6] = 0x29d93, [0x14a7] = 0x9b96, [0x14a8] = 0x29d94, [0x14a9] = 0x29d95,
  [0x14ac] = 0x29ff3, [0x14ad] = 0x29ff4, [0x14ae] = 0x29ff5, [0x14af] = 0x29ff6,
  [0x14b0] = 0x2a260, [0x14b1] = 0x2a29e, [0x14b2] = 0x2a2a0, [0x14b3] = 0x2a2a1,
  [0x14b4] = 0x2a31c, [0x14b5] = 0x2a39f, [0x14b6] = 0x20056, [0x14b7] = 0x20412,
  [0x14b8] = 0x2041f, [0x14b9] = 0x20415, [0x14ba] = 0x20416, [0x14bb] = 0x20417,
  [0x14bc] = 0x2f80c, [0x14bd] = 0x20418, [0x14be] = 0x204c2, [0x14bf] = 0x204c3,
  [0x14c0] = 0x204c4, [0x14c1] = 0x205a1, [0x14c2] = 0x2061b, [0x14c3] = 0x2061c,
  [0x14c4] = 0x20671, [0x14c5] = 0x20813, [0x14c6] = 0x20b17, [0x14c7] = 0x20b8f,
  [0x14c8] = 0x2105f, [0x14c9] = 0x21060, [0x14ca] = 0x210ac, [0x14cd] = 0x21061,
  [0x14ce] = 0x21062, [0x14cf] = 0x21063, [0x14d0] = 0x21064, [0x14d1] = 0x21065,
  [0x14d3] = 0x21499, [0x14d4] = 0x2149a, [0x14d5] = 0x2149b, [0x14d6] = 0x2149c,
  [0x14d7] = 0x2149d, [0x14d8] = 0x2149e, [0x14d9] = 0x2149f, [0x14da] = 0x214a0,
  [0x14db] = 0x218da, [0x14dc] = 0x218db, [0x14dd] = 0x218dc, [0x14de] = 0x218dd,
  [0x14df] = 0x218de, [0x14e0] = 0x218df, [0x14e1] = 0x21ad5, [0x14e2] = 0x21ad6,
  [0x14e3] = 0x21ad7, [0x14e4] = 0x21ad8, [0x14e5] = 0x21ae0, [0x14e6] = 0x21bb4,
  [0x14e8] = 0x21f6e, [0x14e9] = 0x21f6f, [0x14ea] = 0x21f70, [0x14eb] = 0x22163,
  [0x14ed] = 0x222d5, [0x14ee] = 0x222d6, [0x14ef] = 0x222d7, [0x14f0] = 0x222d8,
  [0x14f1] = 0x22430, [0x14f2] = 0x228b5, [0x14f3] = 0x228b6, [0x14f4] = 0x228c1,
  [0x14f5] = 0x228f7, [0x14f6] = 0x39a0, [0x14f7] = 0x228cb, [0x14f8] = 0x228e8,
  [0x14fa] = 0x228e9, [0x14fb] = 0x228b7, [0x14fc] = 0x22a15, [0x14fd] = 0x22de6,
  [0x14fe] = 0x22de7, [0x14ff] = 0x22de8, [0x1500] = 0x22de9, [0x1501] = 0x22dea,
  [0x1502] = 0x22deb, [0x1503] = 0x2f8ca, [0x1504] = 0x2300b, [0x1505] = 0x2300c,
  [0x1506] = 0x23064, [0x1507] = 0x232d4, [0x1508] = 0x232d5, [0x1509] = 0x232d6,
  [0x150a] = 0x232d8, [0x150b] = 0x233a5, [0x150c] = 0x2005b, [0x150d] = 0x2370f,
  [0x150e] = 0x23710, [0x1510] = 0x23711, [0x1511] = 0x23712, [0x1512] = 0x23713,
  [0x1513] = 0x23714, [0x1514] = 0x23715, [0x1515] = 0x23716, [0x1516] = 0x23717,
  [0x1517] = 0x23718, [0x1518] = 0x23719, [0x1519] = 0x2371a, [0x151a] = 0x2372f,
  [0x151b] = 0x2371b, [0x151c] = 0x2371c, [0x151d] = 0x2373f, [0x151e] = 0x2371d,
  [0x151f] = 0x2371e, [0x1520] = 0x2372c, [0x1521] = 0x2371f, [0x1522] = 0x23924,
  [0x1523] = 0x239a4, [0x1524] = 0x239a5, [0x1525] = 0x26251, [0x1526] = 0x23afe,
  [0x1527] = 0x23c11, [0x1528] = 0x23c12, [0x1529] = 0x23c13, [0x152a] = 0x23c54,
  [0x152b] = 0x2405e, [0x152c] = 0x24021, [0x152d] = 0x2403f, [0x152e] = 0x24040,
  [0x152f] = 0x24041, [0x1530] = 0x24042, [0x1531] = 0x24043, [0x1532] = 0x24044,
  [0x1533] = 0x24020, [0x1534] = 0x24045, [0x1536] = 0x24046, [0x1537] = 0x240b0,
  [0x1538] = 0x24047, [0x1539] = 0x24057, [0x153a] = 0x24001, [0x153b] = 0x240b1,
  [0x153c] = 0x2405a, [0x153d] = 0x24048, [0x153e] = 0x24404, [0x153f] = 0x24405,
  [0x1540] = 0x71f5, [0x1541] = 0x2444f, [0x1542] = 0x24450, [0x1543] = 0x24406,
  [0x1544] = 0x24407, [0x1545] = 0x24408, [0x1546] = 0x24409, [0x1547] = 0x2440a,
  [0x1548] = 0x2440b, [0x1549] = 0x2440c, [0x154a] = 0x2440d, [0x154b] = 0x2440e,
  [0x154c] = 0x2440f, [0x154d] = 0x24410, [0x154e] = 0x24411, [0x154f] = 0x24533,
  [0x1550] = 0x24534, [0x1551] = 0x24535, [0x1552] = 0x245fc, [0x1553] = 0x245fd,
  [0x1554] = 0x248b0, [0x1555] = 0x248b1, [0x1556] = 0x248b2, [0x1557] = 0x248b3,
  [0x1558] = 0x3f01, [0x1559] = 0x24a7b, [0x155a] = 0x24a7c, [0x155b] = 0x24a7d,
  [0x155c] = 0x24a7e, [0x155d] = 0x24a7f, [0x155e] = 0x24a80, [0x155f] = 0x24a81,
  [0x1560] = 0x24a82, [0x1561] = 0x24bf9, [0x1562] = 0x24cda, [0x1563] = 0x24ea1,
  [0x1566] = 0x250ad, [0x1567] = 0x250ae, [0x1569] = 0x252c8, [0x156a] = 0x252c9,
  [0x156b] = 0x252ca, [0x156c] = 0x25582, [0x156d] = 0x25583, [0x156e] = 0x25584,
  [0x156f] = 0x25585, [0x1570] = 0x25586, [0x1571] = 0x25587, [0x1572] = 0x25588,
  [0x1573] = 0x25589, [0x1574] = 0x2558a, [0x1576] = 0x25707, [0x1577] = 0x25708,
  [0x1578] = 0x25701, [0x1579] = 0x25709, [0x157a] = 0x25702, [0x157b] = 0x2570a,
  [0x157c] = 0x2570b, [0x157d] = 0x2570c, [0x157e] = 0x258a7, [0x157f] = 0x25898,
  [0x1580] = 0x25899, [0x1581] = 0x2589a, [0x1582] = 0x2589b, [0x1583] = 0x2589c,
  [0x1584] = 0x7a50, [0x1585] = 0x2589d, [0x1586] = 0x25a1a, [0x1587] = 0x25ab2,
  [0x1588] = 0x25cb0, [0x1589] = 0x25cb1, [0x158a] = 0x7c14, [0x158b] = 0x25cb2,
  [0x158c] = 0x25cb3, [0x158d] = 0x25cb4, [0x158e] = 0x25cb5, [0x158f] = 0x25cb6,
  [0x1590] = 0x25cc0, [0x1591] = 0x25cb7, [0x1592] = 0x25cb8, [0x1593] = 0x25cb9,
  [0x1594] = 0x25cba, [0x1595] = 0x25d1d, [0x1596] = 0x25f0c, [0x1597] = 0x25f0d,
  [0x1598] = 0x25f0e, [0x1599] = 0x25f35, [0x159a] = 0x25f0f, [0x159b] = 0x25f10,
  [0x159c] = 0x25f11, [0x159d] = 0x25f12, [0x159e] = 0x26122, [0x159f] = 0x26123,
  [0x15a0] = 0x26124, [0x15a2] = 0x26125, [0x15a3] = 0x26126, [0x15a4] = 0x26127,
  [0x15a5] = 0x26128, [0x15a6] = 0x26129, [0x15a7] = 0x2612a, [0x15a8] = 0x2612b,
  [0x15a9] = 0x2612c, [0x15aa] = 0x2612d, [0x15ab] = 0x2612e, [0x15ac] = 0x26252,
  [0x15ad] = 0x26253, [0x15ae] = 0x26319, [0x15af] = 0x263bf, [0x15b0] = 0x2648b,
  [0x15b1] = 0x2648c, [0x15b2] = 0x26523, [0x15b3] = 0x26524, [0x15b4] = 0x265f1,
  [0x15b5] = 0x265d8, [0x15b6] = 0x2f97f, [0x15b7] = 0x2684e, [0x15b8] = 0x2684f,
  [0x15b9] = 0x26855, [0x15ba] = 0x26850, [0x15bb] = 0x26851, [0x15bc] = 0x26852,
  [0x15bd] = 0x26853, [0x15be] = 0x26802, [0x15bf] = 0x26854, [0x15c0] = 0x268f1,
  [0x15c1] = 0x26924, [0x15c2] = 0x26ad1, [0x15c3] = 0x26f7c, [0x15c4] = 0x26f7d,
  [0x15c5] = 0x26f7e, [0x15c6] = 0x26f7f, [0x15c7] = 0x26f80, [0x15c8] = 0x26f9f,
  [0x15c9] = 0x26f81, [0x15ca] = 0x26f82, [0x15cb] = 0x26f83, [0x15cc] = 0x26f84,
  [0x15cd] = 0x26f85, [0x15ce] = 0x26f86, [0x15cf] = 0x26f87, [0x15d0] = 0x26f88,
  [0x15d1] = 0x26f9b, [0x15d2] = 0x26f89, [0x15d3] = 0x26f8a, [0x15d4] = 0x26f8b,
  [0x15d5] = 0x26f8c, [0x15d7] = 0x26fa1, [0x15d8] = 0x26f8d, [0x15d9] = 0x26f8e,
  [0x15da] = 0x271f3, [0x15db] = 0x271f4, [0x15dc] = 0x271f5, [0x15dd] = 0x27424,
  [0x15de] = 0x27425, [0x15df] = 0x27426, [0x15e0] = 0x27427, [0x15e1] = 0x27428,
  [0x15e2] = 0x27429, [0x15e3] = 0x2742a, [0x15e5] = 0x2742b, [0x15e6] = 0x275cd,
  [0x15e7] = 0x27601, [0x15e8] = 0x2258d, [0x15e9] = 0x27770, [0x15ea] = 0x27880,
  [0x15eb] = 0x27881, [0x15ec] = 0x27ac0, [0x15ed] = 0x27ac1, [0x15ee] = 0x27ac2,
  [0x15ef] = 0x27ac3, [0x15f0] = 0x27ac4, [0x15f1] = 0x27ac5, [0x15f2] = 0x27ac6,
  [0x15f3] = 0x27ac7, [0x15f4] = 0x27ac8, [0x15f5] = 0x27cfe, [0x15f7] = 0x27dd9,
  [0x15f8] = 0x27ddc, [0x15f9] = 0x27dda, [0x15fa] = 0x27e75, [0x15fb] = 0x27f58,
  [0x15fc] = 0x27f59, [0x15fd] = 0x27f5a, [0x15fe] = 0x27f5b, [0x15ff] = 0x27f5c,
  [0x1600] = 0x280ec, [0x1601] = 0x481a, [0x1602] = 0x280f3, [0x1603] = 0x28272,
  [0x1604] = 0x28273, [0x1605] = 0x28382, [0x1606] = 0x28383, [0x1607] = 0x2842a,
  [0x1608] = 0x2842b, [0x1609] = 0x2842c, [0x160a] = 0x2842d, [0x160b] = 0x285f6,
  [0x160c] = 0x28600, [0x160d] = 0x285f7, [0x160e] = 0x285f8, [0x160f] = 0x285f9,
  [0x1611] = 0x28894, [0x1612] = 0x28895, [0x1613] = 0x28896, [0x1614] = 0x2891b,
  [0x1615] = 0x28a86, [0x1616] = 0x28a4c, [0x1617] = 0x28a4d, [0x1618] = 0x28a83,
  [0x161a] = 0x28a82, [0x161b] = 0x28a4e, [0x161c] = 0x28a4f, [0x161d] = 0x28a50,
  [0x161e] = 0x28a51, [0x161f] = 0x28a81, [0x1620] = 0x9387, [0x1621] = 0x28a52,
  [0x1622] = 0x28a53, [0x1623] = 0x28a54, [0x1624] = 0x28a55, [0x1626] = 0x28a56,
  [0x1627] = 0x28a57, [0x1628] = 0x9385, [0x1629] = 0x28a58, [0x162a] = 0x28a5b,
  [0x162b] = 0x28a5c, [0x162c] = 0x28a5d, [0x162d] = 0x28a5e, [0x162e] = 0x28a5f,
  [0x162f] = 0x28a60, [0x1630] = 0x28a61, [0x1631] = 0x28a7c, [0x1632] = 0x28a62,
  [0x1633] = 0x493c, [0x1635] = 0x28a63, [0x1636] = 0x28a64, [0x1637] = 0x28d73,
  [0x1638] = 0x28f45, [0x1639] = 0x28f46, [0x163a] = 0x28f47, [0x163b] = 0x28f4c,
  [0x163c] = 0x29012, [0x163e] = 0x29013, [0x163f] = 0x29139, [0x1640] = 0x29103,
  [0x1641] = 0x2910d, [0x1642] = 0x29230, [0x1643] = 0x292d5, [0x1644] = 0x293c0,
  [0x1645] = 0x294f2, [0x1646] = 0x294f3, [0x1647] = 0x294f4, [0x1648] = 0x294f5,
  [0x1649] = 0x294f6, [0x164a] = 0x294f7, [0x164b] = 0x295fa, [0x164c] = 0x295fb,
  [0x164d] = 0x29745, [0x164e] = 0x29746, [0x164f] = 0x29849, [0x1650] = 0x298e7,
  [0x1651] = 0x298e8, [0x1652] = 0x29ad8, [0x1653] = 0x29ad9, [0x1654] = 0x29b55,
  [0x1655] = 0x29b56, [0x1656] = 0x29b57, [0x1657] = 0x29cda, [0x1658] = 0x29cdb,
  [0x1659] = 0x29cdc, [0x165a] = 0x29db8, [0x165b] = 0x29db9, [0x165c] = 0x29dba,
  [0x165d] = 0x9bb1, [0x165e] = 0x29dbb, [0x165f] = 0x29f89, [0x1660] = 0x29dbc,
  [0x1661] = 0x2a030, [0x1662] = 0x2a031, [0x1663] = 0x2a032, [0x1664] = 0x2a033,
  [0x1666] = 0x2a034, [0x1667] = 0x2a035, [0x1668] = 0x2a036, [0x1669] = 0x9d47,
  [0x166a] = 0x2a037, [0x166b] = 0x2a2b2, [0x166c] = 0x2a390, [0x166d] = 0x2a326,
  [0x166e] = 0x2a3a2, [0x166f] = 0x2a438, [0x1671] = 0x2042e, [0x1673] = 0x2042f,
  [0x1674] = 0x20430, [0x1675] = 0x2042a, [0x1676] = 0x2042b, [0x1677] = 0x204c9,
  [0x1678] = 0x20537, [0x1679] = 0x20620, [0x167a] = 0x20672, [0x167b] = 0x2081f,
  [0x167c] = 0x20965, [0x167d] = 0x209ca, [0x167e] = 0x20ac0, [0x167f] = 0x20ac1,
  [0x1680] = 0x20ac2, [0x1681] = 0x210b7, [0x1682] = 0x210b8, [0x1683] = 0x210b9,
  [0x1684] = 0x210ba, [0x1685] = 0x210bb, [0x1688] = 0x210bc, [0x1689] = 0x210bd,
  [0x168a] = 0x210be, [0x168b] = 0x210f2, [0x168c] = 0x214b5, [0x168d] = 0x214b7,
  [0x168e] = 0x214b8, [0x168f] = 0x210df, [0x1690] = 0x214b9, [0x1691] = 0x214d5,
  [0x1692] = 0x214ba, [0x1693] = 0x214bb, [0x1694] = 0x2154a, [0x1695] = 0x218f2,
  [0x1696] = 0x218f3, [0x1697] = 0x218f4, [0x1698] = 0x218fa, [0x1699] = 0x218f5,
  [0x169a] = 0x21902, [0x169b] = 0x21ae2, [0x169c] = 0x21ae3, [0x169d] = 0x21f8e,
  [0x169e] = 0x21f8b, [0x169f] = 0x222e4, [0x16a0] = 0x222eb, [0x16a1] = 0x222e5,
  [0x16a2] = 0x22435, [0x16a3] = 0x22593, [0x16a4] = 0x228ea, [0x16a5] = 0x228eb,
  [0x16a7] = 0x228ec, [0x16a8] = 0x228ed, [0x16a9] = 0x22917, [0x16aa] = 0x228ee,
  [0x16ab] = 0x228ef, [0x16ac] = 0x228f0, [0x16ad] = 0x22e0e, [0x16ae] = 0x22e0f,
  [0x16af] = 0x22e10, [0x16b0] = 0x22e19, [0x16b1] = 0x22e18, [0x16b2] = 0x22e11,
  [0x16b3] = 0x22e12, [0x16b4] = 0x22e13, [0x16b5] = 0x2f8c7, [0x16b6] = 0x22e14,
  [0x16b7] = 0x22e15, [0x16b8] = 0x3a6b, [0x16b9] = 0x22e16, [0x16ba] = 0x23065,
  [0x16bb] = 0x232ea, [0x16bc] = 0x232eb, [0x16bd] = 0x232ec, [0x16be] = 0x232ed,
  [0x16c0] = 0x232ee, [0x16c1] = 0x233ac, [0x16c2] = 0x237c2, [0x16c3] = 0x23769,
  [0x16c4] = 0x23781, [0x16c5] = 0x2376a, [0x16c6] = 0x2376b, [0x16c7] = 0x2376c,
  [0x16c8] = 0x2376d, [0x16c9] = 0x2376e, [0x16ca] = 0x2376f, [0x16cb] = 0x23770,
  [0x16cc] = 0x3bf4, [0x16cd] = 0x23771, [0x16ce] = 0x23772, [0x16cf] = 0x23773,
  [0x16d0] = 0x2378c, [0x16d1] = 0x23774, [0x16d2] = 0x23775, [0x16d3] = 0x23776,
  [0x16d4] = 0x23c24, [0x16d5] = 0x23c2a, [0x16d6] = 0x2408e, [0x16d7] = 0x240af,
  [0x16d8] = 0x2408f, [0x16d9] = 0x24090, [0x16da] = 0x24091, [0x16dc] = 0x24092,
  [0x16dd] = 0x24093, [0x16de] = 0x24094, [0x16df] = 0x24095, [0x16e0] = 0x24096,
  [0x16e1] = 0x24097, [0x16e2] = 0x24098, [0x16e3] = 0x24099, [0x16e4] = 0x2409a,
  [0x16e5] = 0x2409b, [0x16e6] = 0x2409c, [0x16e7] = 0x24439, [0x16e8] = 0x2443b,
  [0x16e9] = 0x2443a, [0x16ea] = 0x2443c, [0x16eb] = 0x2443d, [0x16ec] = 0x2443e,
  [0x16ed] = 0x2443f, [0x16ee] = 0x24470, [0x16ef] = 0x24440, [0x16f0] = 0x24441,
  [0x16f1] = 0x248c0, [0x16f2] = 0x248c1, [0x16f3] = 0x24aa4, [0x16f4] = 0x24a97,
  [0x16f5] = 0x24aa5, [0x16f6] = 0x24a98, [0x16f7] = 0x24a99, [0x16f8] = 0x24a9a,
  [0x16fa] = 0x24a9b, [0x16fb] = 0x24aa6, [0x16fc] = 0x24a9c, [0x16fd] = 0x24a9d,
  [0x16fe] = 0x24a9e, [0x16ff] = 0x24a9f, [0x1700] = 0x28ace, [0x1701] = 0x24ac8,
  [0x1702] = 0x24aa0, [0x1703] = 0x24b19, [0x1704] = 0x24c0d, [0x1705] = 0x250bf,
  [0x1706] = 0x252e9, [0x1707] = 0x252ea, [0x1708] = 0x255a2, [0x1709] = 0x255a3,
  [0x170a] = 0x255a4, [0x170b] = 0x255a5, [0x170c] = 0x255a6, [0x170d] = 0x25715,
  [0x170e] = 0x25717, [0x170f] = 0x258bf, [0x1710] = 0x258c0, [0x1711] = 0x258c1,
  [0x1712] = 0x258c2, [0x1713] = 0x258c3, [0x1714] = 0x258c4, [0x1715] = 0x258c5,
  [0x1716] = 0x2190d, [0x1717] = 0x25a23, [0x1718] = 0x25abe, [0x1719] = 0x25d03,
  [0x171a] = 0x25d04, [0x171b] = 0x25d05, [0x171c] = 0x25d06, [0x171d] = 0x25d07,
  [0x171e] = 0x25d08, [0x171f] = 0x25d09, [0x1720] = 0x25d0a, [0x1721] = 0x25d0b,
  [0x1722] = 0x25d0c, [0x1723] = 0x25d20, [0x1724] = 0x25d0d, [0x1725] = 0x25d0e,
  [0x1726] = 0x25d0f, [0x1728] = 0x25f26, [0x1729] = 0x25f27, [0x172a] = 0x25f28,
  [0x172b] = 0x2615d, [0x172c] = 0x2615e, [0x172d] = 0x2615f, [0x172e] = 0x26160,
  [0x172f] = 0x26161, [0x1730] = 0x26162, [0x1731] = 0x26163, [0x1732] = 0x26164,
  [0x1733] = 0x26165, [0x1734] = 0x26257, [0x1735] = 0x263cf, [0x1736] = 0x263d0,
  [0x1737] = 0x26499, [0x1738] = 0x2649a, [0x1739] = 0x265ea, [0x173a] = 0x265eb,
  [0x173b] = 0x265ec, [0x173c] = 0x2687e, [0x173d] = 0x2687f, [0x173e] = 0x26880,
  [0x1741] = 0x26881, [0x1742] = 0x26882, [0x1743] = 0x268f3, [0x1744] = 0x26925,
  [0x1745] = 0x2699f, [0x1746] = 0x269a0, [0x1747] = 0x269f3, [0x1748] = 0x26a00,
  [0x1749] = 0x26feb, [0x174a] = 0x26fec, [0x174b] = 0x26fed, [0x174c] = 0x26fee,
  [0x174d] = 0x26fef, [0x174f] = 0x455c, [0x1750] = 0x26ff0, [0x1751] = 0x26ff1,
  [0x1752] = 0x26ff2, [0x1753] = 0x26ff3, [0x1754] = 0x26ff4, [0x1755] = 0x26ff5,
  [0x1756] = 0x26ff9, [0x1757] = 0x26ffa, [0x1758] = 0x26ffb, [0x1759] = 0x26ffc,
  [0x175a] = 0x26ffd, [0x175b] = 0x26ffe, [0x175c] = 0x26fff, [0x175d] = 0x27000,
  [0x175e] = 0x2f9ae, [0x175f] = 0x27001, [0x1760] = 0x27002, [0x1761] = 0x27203,
  [0x1762] = 0x27486, [0x1763] = 0x27479, [0x1765] = 0x2747a, [0x1766] = 0x2747b,
  [0x1767] = 0x2747c, [0x1768] = 0x27484, [0x1769] = 0x2747d, [0x176a] = 0x2747e,
  [0x176b] = 0x2747f, [0x176c] = 0x27786, [0x176d] = 0x27787, [0x176e] = 0x27800,
  [0x176f] = 0x27801, [0x1770] = 0x27888, [0x1771] = 0x27889, [0x1772] = 0x2788a,
  [0x1773] = 0x27af5, [0x1775] = 0x27b25, [0x1776] = 0x27af6, [0x1777] = 0x27deb,
  [0x1778] = 0x27dec, [0x1779] = 0x27ded, [0x177a] = 0x27dee, [0x177b] = 0x4787,
  [0x177c] = 0x27def, [0x177d] = 0x27df0, [0x177e] = 0x27f6c, [0x177f] = 0x28121,
  [0x1780] = 0x28122, [0x1781] = 0x28123, [0x1782] = 0x28124, [0x1783] = 0x28125,
  [0x1784] = 0x28126, [0x1785] = 0x2827b, [0x1786] = 0x2827c, [0x1787] = 0x28391,
  [0x1788] = 0x28392, [0x1789] = 0x28393, [0x178a] = 0x2860e, [0x178b] = 0x2860f,
  [0x178c] = 0x28610, [0x178d] = 0x288b3, [0x178e] = 0x28936, [0x178f] = 0x28ac6,
  [0x1790] = 0x28aa1, [0x1791] = 0x28aa2, [0x1792] = 0x93b9, [0x1793] = 0x28aa3,
  [0x1794] = 0x28acc, [0x1795] = 0x28aa4, [0x1796] = 0x28aa5, [0x1797] = 0x28aa6,
  [0x1798] = 0x28acb, [0x1799] = 0x28aa7, [0x179b] = 0x28aa8, [0x179c] = 0x28aa9,
  [0x179d] = 0x28aaa, [0x179f] = 0x28aab, [0x17a0] = 0x28aac, [0x17a1] = 0x28aad,
  [0x17a2] = 0x93bf, [0x17a3] = 0x28aae, [0x17a4] = 0x28aaf, [0x17a5] = 0x28ab0,
  [0x17a6] = 0x28d8c, [0x17a7] = 0x28d8d, [0x17a8] = 0x28d8e, [0x17a9] = 0x2f9f1,
  [0x17aa] = 0x28d92, [0x17ab] = 0x28d8f, [0x17ac] = 0x28f52, [0x17ad] = 0x28f53,
  [0x17ae] = 0x28f54, [0x17af] = 0x28f55, [0x17b0] = 0x29129, [0x17b2] = 0x292f3,
  [0x17b3] = 0x292f4, [0x17b4] = 0x2950d, [0x17b5] = 0x2950e, [0x17b6] = 0x2950f,
  [0x17b7] = 0x29510, [0x17b8] = 0x29614, [0x17b9] = 0x29771, [0x17ba] = 0x29772,
  [0x17bb] = 0x2990f, [0x17bc] = 0x29904, [0x17bd] = 0x29906, [0x17bf] = 0x29907,
  [0x17c0] = 0x29adc, [0x17c1] = 0x29add, [0x17c2] = 0x29b6d, [0x17c3] = 0x29b79,
  [0x17c4] = 0x29b6e, [0x17c6] = 0x29cec, [0x17c7] = 0x29ddd, [0x17c8] = 0x29dde,
  [0x17c9] = 0x29ddf, [0x17ca] = 0x29de0, [0x17cb] = 0x29de1, [0x17cc] = 0x29de2,
  [0x17cd] = 0x29de3, [0x17ce] = 0x29de4, [0x17cf] = 0x29e3a, [0x17d0] = 0x29de5,
  [0x17d1] = 0x9bcf, [0x17d2] = 0x29de6, [0x17d3] = 0x2a06b, [0x17d4] = 0x2a06c,
  [0x17d5] = 0x2444e, [0x17d6] = 0x9d64, [0x17d7] = 0x2a2bd, [0x17d8] = 0x2a2be,
  [0x17d9] = 0x2a2bf, [0x17da] = 0x2a334, [0x17db] = 0x9ebf, [0x17dc] = 0x2a512,
  [0x17dd] = 0x20440, [0x17de] = 0x20441, [0x17df] = 0x21102, [0x17e0] = 0x21103,
  [0x17e1] = 0x21104, [0x17e2] = 0x21105, [0x17e3] = 0x21106, [0x17e4] = 0x214d7,
  [0x17e5] = 0x214cc, [0x17e6] = 0x21698, [0x17e7] = 0x21906, [0x17e8] = 0x21907,
  [0x17e9] = 0x219af, [0x17ea] = 0x219b0, [0x17eb] = 0x21aea, [0x17ec] = 0x21af2,
  [0x17ed] = 0x21fa2, [0x17ee] = 0x21fa3, [0x17ef] = 0x21fa4, [0x17f0] = 0x22176,
  [0x17f1] = 0x222f5, [0x17f2] = 0x222f6, [0x17f3] = 0x222f7, [0x17f5] = 0x2293b,
  [0x17f6] = 0x22925, [0x17f7] = 0x2293a, [0x17f8] = 0x2293c, [0x17f9] = 0x2293d,
  [0x17fa] = 0x2293e, [0x17fb] = 0x22e37, [0x17fc] = 0x22e38, [0x17fd] = 0x22e39,
  [0x17fe] = 0x22e3a, [0x17ff] = 0x22e3b, [0x1800] = 0x22e3c, [0x1801] = 0x22e3d,
  [0x1802] = 0x22e3e, [0x1803] = 0x22e3f, [0x1804] = 0x22e4b, [0x1806] = 0x22e40,
  [0x1807] = 0x23022, [0x1808] = 0x23067, [0x1809] = 0x230d3, [0x180a] = 0x232fa,
  [0x180c] = 0x237bc, [0x180f] = 0x237a5, [0x1810] = 0x3c07, [0x1811] = 0x237b6,
  [0x1813] = 0x237a6, [0x1814] = 0x237bb, [0x1815] = 0x237a7, [0x1816] = 0x237a8,
  [0x1817] = 0x237a9, [0x1818] = 0x237aa, [0x1819] = 0x237ab, [0x181a] = 0x237ac,
  [0x181b] = 0x237bf, [0x181c] = 0x237ad, [0x181d] = 0x237ae, [0x181e] = 0x237af,
  [0x181f] = 0x2392e, [0x1820] = 0x23b00, [0x1821] = 0x2409d, [0x1822] = 0x2409e,
  [0x1823] = 0x240cb, [0x1824] = 0x240cc, [0x1825] = 0x240cd, [0x1826] = 0x240ce,
  [0x1827] = 0x240cf, [0x1829] = 0x2409f, [0x182a] = 0x240d0, [0x182b] = 0x240d1,
  [0x182c] = 0x240d2, [0x182d] = 0x240d3, [0x182e] = 0x240d4, [0x1831] = 0x2445d,
  [0x1832] = 0x2445e, [0x1833] = 0x2445f, [0x1834] = 0x24460, [0x1835] = 0x3f05,
  [0x1836] = 0x24aba, [0x1837] = 0x24abc, [0x1838] = 0x3f06, [0x1839] = 0x24ab5,
  [0x183a] = 0x24abd, [0x183b] = 0x24ab6, [0x183c] = 0x24ab7, [0x183d] = 0x24ab8,
  [0x183e] = 0x24aa8, [0x183f] = 0x2f932, [0x1840] = 0x24bfe, [0x1841] = 0x24bfc,
  [0x1842] = 0x24ee1, [0x1843] = 0x250c4, [0x1844] = 0x250c5, [0x1845] = 0x25314,
  [0x1846] = 0x255ae, [0x1847] = 0x255af, [0x1848] = 0x255b0, [0x1849] = 0x255b1,
  [0x184a] = 0x255b9, [0x184b] = 0x258e7, [0x184c] = 0x25d46, [0x184e] = 0x25d47,
  [0x184f] = 0x25d48, [0x1850] = 0x25d49, [0x1851] = 0x25d1e, [0x1852] = 0x25d1f,
  [0x1853] = 0x25d4a, [0x1854] = 0x25d4b, [0x1855] = 0x25d4c, [0x1856] = 0x25d4d,
  [0x1857] = 0x25d4e, [0x1859] = 0x25f46, [0x185a] = 0x25f4b, [0x185b] = 0x26190,
  [0x185c] = 0x26191, [0x185e] = 0x26192, [0x185f] = 0x26193, [0x1860] = 0x26194,
  [0x1861] = 0x2625b, [0x1862] = 0x265f2, [0x1863] = 0x265f8, [0x1864] = 0x265f9,
  [0x1865] = 0x26894, [0x1866] = 0x268a4, [0x1867] = 0x26897, [0x1868] = 0x26898,
  [0x1869] = 0x26929, [0x186a] = 0x269ae, [0x186b] = 0x26aaf, [0x186c] = 0x27059,
  [0x186d] = 0x2705a, [0x186f] = 0x2705b, [0x1870] = 0x2705c, [0x1871] = 0x2705d,
  [0x1872] = 0x2705e, [0x1873] = 0x2705f, [0x1874] = 0x27060, [0x1875] = 0x27061,
  [0x1876] = 0x26faa, [0x1877] = 0x27062, [0x1878] = 0x27063, [0x1879] = 0x27064,
  [0x187a] = 0x27065, [0x187b] = 0x27066, [0x187c] = 0x27067, [0x187d] = 0x27068,
  [0x187e] = 0x27069, [0x187f] = 0x2706a, [0x1880] = 0x2706b, [0x1881] = 0x2706c,
  [0x1882] = 0x2706d, [0x1883] = 0x2706e, [0x1884] = 0x2706f, [0x1886] = 0x274b2,
  [0x1887] = 0x274b3, [0x1888] = 0x274b4, [0x1889] = 0x45f9, [0x188a] = 0x274b5,
  [0x188b] = 0x274b6, [0x188c] = 0x274b7, [0x188d] = 0x274b8, [0x188e] = 0x274b9,
  [0x188f] = 0x274ba, [0x1890] = 0x274bb, [0x1891] = 0x27788, [0x1892] = 0x89b8,
  [0x1893] = 0x27b19, [0x1894] = 0x27b1a, [0x1895] = 0x27b1b, [0x1896] = 0x27c16,
  [0x1897] = 0x27e0c, [0x1898] = 0x27e0d, [0x1899] = 0x27e0e, [0x189a] = 0x27e0f,
  [0x189c] = 0x27f83, [0x189d] = 0x27f84, [0x189e] = 0x27f85, [0x189f] = 0x28283,
  [0x18a0] = 0x28284, [0x18a1] = 0x283b0, [0x18a3] = 0x28626, [0x18a4] = 0x28627,
  [0x18a5] = 0x28636, [0x18a6] = 0x28628, [0x18a7] = 0x28629, [0x18aa] = 0x28ae7,
  [0x18ab] = 0x28ae8, [0x18ac] = 0x28ae9, [0x18ad] = 0x4953, [0x18ae] = 0x28aea,
  [0x18af] = 0x28aeb, [0x18b1] = 0x28aec, [0x18b2] = 0x28aed, [0x18b3] = 0x28aee,
  [0x18b4] = 0x28aef, [0x18b5] = 0x28af0, [0x18b6] = 0x28af1, [0x18b7] = 0x28b0c,
  [0x18b8] = 0x28af2, [0x18b9] = 0x28af3, [0x18ba] = 0x28af4, [0x18bb] = 0x28af5,
  [0x18bc] = 0x28af6, [0x18bd] = 0x28af7, [0x18be] = 0x28af8, [0x18bf] = 0x28da3,
  [0x18c0] = 0x28da4, [0x18c1] = 0x28da5, [0x18c2] = 0x28da6, [0x18c3] = 0x28da7,
  [0x18c5] = 0x28f5c, [0x18c7] = 0x2914b, [0x18c8] = 0x291e1, [0x18c9] = 0x29315,
  [0x18cb] = 0x29316, [0x18cc] = 0x29317, [0x18cd] = 0x2952c, [0x18ce] = 0x2952d,
  [0x18cf] = 0x29622, [0x18d0] = 0x29623, [0x18d1] = 0x29796, [0x18d2] = 0x29797,
  [0x18d4] = 0x2992e, [0x18d5] = 0x29930, [0x18d6] = 0x29931, [0x18d8] = 0x29adf,
  [0x18d9] = 0x29b8b, [0x18da] = 0x29b8c, [0x18dc] = 0x29b8d, [0x18dd] = 0x29cfc,
  [0x18de] = 0x29cfd, [0x18df] = 0x29cfe, [0x18e0] = 0x29cff, [0x18e1] = 0x29e17,
  [0x18e2] = 0x29e2d, [0x18e3] = 0x29e18, [0x18e4] = 0x9bf3, [0x18e5] = 0x29e19,
  [0x18e7] = 0x29e12, [0x18e8] = 0x29e1a, [0x18e9] = 0x29e1b, [0x18ea] = 0x29e1c,
  [0x18eb] = 0x4c69, [0x18ed] = 0x2a097, [0x18ee] = 0x4ce5, [0x18ef] = 0x2a098,
  [0x18f0] = 0x2a099, [0x18f2] = 0x2a3c4, [0x18f3] = 0x2044f, [0x18f4] = 0x204cd,
  [0x18f5] = 0x20ac7, [0x18f6] = 0x2111c, [0x18f7] = 0x2111d, [0x18f8] = 0x2111e,
  [0x18f9] = 0x214e7, [0x18fa] = 0x214e9, [0x18fb] = 0x214ea, [0x18fc] = 0x214eb,
  [0x18fd] = 0x21915, [0x18fe] = 0x21cf1, [0x18ff] = 0x21cf2, [0x1900] = 0x21fb6,
  [0x1901] = 0x225a0, [0x1902] = 0x2294a, [0x1903] = 0x22a20, [0x1904] = 0x22e60,
  [0x1905] = 0x22e61, [0x1906] = 0x22e41, [0x1907] = 0x22e62, [0x1908] = 0x23303,
  [0x1909] = 0x23304, [0x190a] = 0x237d9, [0x190b] = 0x237da, [0x190c] = 0x237db,
  [0x190d] = 0x237dc, [0x190e] = 0x237dd, [0x1910] = 0x237de, [0x1911] = 0x237df,
  [0x1912] = 0x237e0, [0x1913] = 0x237e1, [0x1914] = 0x237e2, [0x1915] = 0x3c12,
  [0x1916] = 0x237e3, [0x1918] = 0x23c31, [0x1919] = 0x24105, [0x191a] = 0x24106,
  [0x191b] = 0x24107, [0x191c] = 0x24472, [0x191d] = 0x2448a, [0x191e] = 0x2448b,
  [0x191f] = 0x24488, [0x1921] = 0x2448c, [0x1922] = 0x2448d, [0x1923] = 0x2448e,
  [0x1924] = 0x2448f, [0x1925] = 0x24490, [0x1926] = 0x24542, [0x1927] = 0x24543,
  [0x1929] = 0x24ac1, [0x192a] = 0x24ac2, [0x192b] = 0x24ac3, [0x192c] = 0x24ac4,
  [0x192d] = 0x250cb, [0x192e] = 0x25322, [0x1931] = 0x255c3, [0x1932] = 0x255c4,
  [0x1933] = 0x255c5, [0x1934] = 0x255c6, [0x1935] = 0x2571e, [0x1936] = 0x25725,
  [0x1937] = 0x258f3, [0x1938] = 0x258f4, [0x193a] = 0x258f5, [0x193b] = 0x25a38,
  [0x193c] = 0x25a48, [0x193d] = 0x25aca, [0x193e] = 0x7c4f, [0x193f] = 0x25d7a,
  [0x1940] = 0x25d7b, [0x1941] = 0x25f55, [0x1942] = 0x25f56, [0x1943] = 0x261b1,
  [0x1944] = 0x261b2, [0x1945] = 0x261b3, [0x1947] = 0x261b4, [0x1948] = 0x261b5,
  [0x1949] = 0x264a4, [0x194a] = 0x26600, [0x194b] = 0x26601, [0x194c] = 0x26602,
  [0x194d] = 0x268ac, [0x194f] = 0x269b3, [0x1950] = 0x270b6, [0x1951] = 0x270b7,
  [0x1952] = 0x270b8, [0x1953] = 0x270b9, [0x1954] = 0x270ba, [0x1955] = 0x270bb,
  [0x1956] = 0x270bc, [0x1957] = 0x270bd, [0x1958] = 0x270be, [0x1959] = 0x270bf,
  [0x195a] = 0x270c0, [0x195b] = 0x270c1, [0x195c] = 0x270c2, [0x195d] = 0x270c3,
  [0x195e] = 0x270c4, [0x195f] = 0x270c5, [0x1960] = 0x270c6, [0x1961] = 0x270c7,
  [0x1962] = 0x274e3, [0x1963] = 0x274e4, [0x1964] = 0x274e5, [0x1965] = 0x274e6,
  [0x1966] = 0x274e7, [0x1968] = 0x277b5, [0x1969] = 0x277a1, [0x196a] = 0x27895,
  [0x196b] = 0x27b3b, [0x196c] = 0x27b3c, [0x196d] = 0x27b3d, [0x196e] = 0x27b3e,
  [0x196f] = 0x27c14, [0x1970] = 0x27e20, [0x1971] = 0x27e21, [0x1972] = 0x27e22,
  [0x1973] = 0x27e23, [0x1974] = 0x27e24, [0x1975] = 0x27f96, [0x1976] = 0x27f97,
  [0x1977] = 0x27f98, [0x1978] = 0x28194, [0x1979] = 0x28195, [0x197a] = 0x28196,
  [0x197b] = 0x28197, [0x197c] = 0x28198, [0x197d] = 0x2828f, [0x197e] = 0x28290,
  [0x197f] = 0x283bc, [0x1980] = 0x2843b, [0x1981] = 0x2863d, [0x1983] = 0x288df,
  [0x1984] = 0x2891e, [0x1985] = 0x28b23, [0x1986] = 0x28b24, [0x1987] = 0x28b49,
  [0x1988] = 0x28b25, [0x1989] = 0x28b26, [0x198a] = 0x28b27, [0x198b] = 0x28b28,
  [0x198c] = 0x9425, [0x198d] = 0x28b29, [0x198e] = 0x28b2a, [0x198f] = 0x28b2e,
  [0x1990] = 0x28b2f, [0x1991] = 0x28b4e, [0x1992] = 0x28b30, [0x1993] = 0x2f9eb,
  [0x1994] = 0x28b31, [0x1995] = 0x28b32, [0x1996] = 0x28b33, [0x1997] = 0x28b34,
  [0x1998] = 0x28b35, [0x1999] = 0x28b36, [0x199b] = 0x28b37, [0x199c] = 0x28b50,
  [0x199d] = 0x28b38, [0x199f] = 0x28b39, [0x19a0] = 0x28b3a, [0x19a1] = 0x28b3b,
  [0x19a2] = 0x28b3c, [0x19a3] = 0x28b3d, [0x19a4] = 0x28b3e, [0x19a5] = 0x28db8,
  [0x19a6] = 0x28db9, [0x19a7] = 0x28f62, [0x19a8] = 0x28f63, [0x19a9] = 0x29041,
  [0x19aa] = 0x2916d, [0x19ac] = 0x29352, [0x19ad] = 0x29337, [0x19ae] = 0x293db,
  [0x19af] = 0x2953f, [0x19b0] = 0x29634, [0x19b1] = 0x2985b, [0x19b2] = 0x29946,
  [0x19b3] = 0x2995a, [0x19b4] = 0x2995b, [0x19b5] = 0x29ae1, [0x19b6] = 0x29ae2,
  [0x19b7] = 0x29ae3, [0x19b8] = 0x29ae4, [0x19b9] = 0x29ba7, [0x19ba] = 0x29ba8,
  [0x19bb] = 0x29ba9, [0x19bc] = 0x29d0f, [0x19bd] = 0x29d2b, [0x19be] = 0x29e56,
  [0x19bf] = 0x29e57, [0x19c0] = 0x29e58, [0x19c1] = 0x29e5e, [0x19c2] = 0x29e59,
  [0x19c3] = 0x29e5a, [0x19c4] = 0x29e5b, [0x19c5] = 0x29e5c, [0x19c7] = 0x29e68,
  [0x19c9] = 0x2a12c, [0x19ca] = 0x2a0e2, [0x19cb] = 0x2a0f3, [0x19cc] = 0x2a2d2,
  [0x19cd] = 0x2a356, [0x19ce] = 0x2a3ce, [0x19cf] = 0x2a461, [0x19d0] = 0x2a69b,
  [0x19d1] = 0x2a6b4, [0x19d2] = 0x2045c, [0x19d3] = 0x2f810, [0x19d4] = 0x204d2,
  [0x19d5] = 0x20571, [0x19d6] = 0x20673, [0x19d7] = 0x2082c, [0x19d8] = 0x20b94,
  [0x19da] = 0x2113f, [0x19db] = 0x21140, [0x19dc] = 0x214f9, [0x19dd] = 0x214fa,
  [0x19de] = 0x214fd, [0x19df] = 0x214fb, [0x19e0] = 0x2191c, [0x19e1] = 0x2191d,
  [0x19e2] = 0x21aff, [0x19e3] = 0x21b00, [0x19e4] = 0x21cf7, [0x19e5] = 0x24cfe,
  [0x19e7] = 0x224b6, [0x19e8] = 0x22954, [0x19e9] = 0x22e77, [0x19ea] = 0x22e79,
  [0x19eb] = 0x22e7a, [0x19ec] = 0x22e7b, [0x19ed] = 0x23069, [0x19ee] = 0x23802,
  [0x19ef] = 0x23803, [0x19f0] = 0x23804, [0x19f1] = 0x23805, [0x19f2] = 0x23806,
  [0x19f3] = 0x23807, [0x19f4] = 0x23808, [0x19f5] = 0x23809, [0x19f6] = 0x2380a,
  [0x19f7] = 0x2380b, [0x19f8] = 0x23835, [0x19fa] = 0x23c39, [0x19fb] = 0x2411e,
  [0x19fc] = 0x2411f, [0x19fd] = 0x24120, [0x19fe] = 0x24121, [0x1a00] = 0x24122,
  [0x1a01] = 0x24123, [0x1a02] = 0x24124, [0x1a03] = 0x2412c, [0x1a04] = 0x244b3,
  [0x1a05] = 0x244b5, [0x1a06] = 0x244b6, [0x1a07] = 0x244b7, [0x1a08] = 0x244b8,
  [0x1a09] = 0x24ad0, [0x1a0a] = 0x24cfa, [0x1a0b] = 0x24f07, [0x1a0c] = 0x24fbd,
  [0x1a0d] = 0x250d7, [0x1a0e] = 0x250d2, [0x1a0f] = 0x255db, [0x1a10] = 0x255d6,
  [0x1a11] = 0x2045d, [0x1a12] = 0x25a43, [0x1a13] = 0x25acf, [0x1a14] = 0x25da3,
  [0x1a15] = 0x25da4, [0x1a16] = 0x25da5, [0x1a17] = 0x25da6, [0x1a18] = 0x25da7,
  [0x1a19] = 0x25da8, [0x1a1a] = 0x25da9, [0x1a1b] = 0x25f5d, [0x1a1c] = 0x25f5e,
  [0x1a1d] = 0x261c9, [0x1a1e] = 0x261ca, [0x1a1f] = 0x261cb, [0x1a20] = 0x261cc,
  [0x1a21] = 0x26606, [0x1a22] = 0x26607, [0x1a24] = 0x268c5, [0x1a25] = 0x268f5,
  [0x1a26] = 0x269b4, [0x1a27] = 0x269b8, [0x1a28] = 0x269b5, [0x1a29] = 0x270fd,
  [0x1a2a] = 0x4577, [0x1a2b] = 0x270fe, [0x1a2c] = 0x270ff, [0x1a2d] = 0x27100,
  [0x1a2e] = 0x27101, [0x1a2f] = 0x27102, [0x1a30] = 0x27103, [0x1a31] = 0x27109,
  [0x1a32] = 0x27104, [0x1a33] = 0x27105, [0x1a34] = 0x27106, [0x1a35] = 0x27107,
  [0x1a36] = 0x27511, [0x1a37] = 0x27512, [0x1a38] = 0x27513, [0x1a39] = 0x27514,
  [0x1a3a] = 0x27515, [0x1a3b] = 0x8977, [0x1a3c] = 0x27898, [0x1a3d] = 0x27899,
  [0x1a3e] = 0x27e3c, [0x1a3f] = 0x27e2d, [0x1a40] = 0x27e2e, [0x1a42] = 0x281bb,
  [0x1a43] = 0x281bc, [0x1a44] = 0x4878, [0x1a45] = 0x2843d, [0x1a46] = 0x28647,
  [0x1a47] = 0x28648, [0x1a48] = 0x28649, [0x1a49] = 0x2864a, [0x1a4a] = 0x287e1,
  [0x1a4b] = 0x287e7, [0x1a4c] = 0x288eb, [0x1a4d] = 0x28920, [0x1a4e] = 0x28938,
  [0x1a4f] = 0x28b62, [0x1a50] = 0x28b67, [0x1a51] = 0x28b68, [0x1a52] = 0x28b69,
  [0x1a53] = 0x28b6a, [0x1a54] = 0x28b6b, [0x1a55] = 0x28b6d, [0x1a56] = 0x28b6e,
  [0x1a57] = 0x28b6f, [0x1a58] = 0x28b70, [0x1a59] = 0x28b71, [0x1a5a] = 0x28b72,
  [0x1a5b] = 0x28b8f, [0x1a5c] = 0x28b73, [0x1a5d] = 0x28b74, [0x1a5e] = 0x28b75,
  [0x1a5f] = 0x28b76, [0x1a60] = 0x28b77, [0x1a61] = 0x28dc9, [0x1a62] = 0x28dca,
  [0x1a63] = 0x95e6, [0x1a64] = 0x28dcb, [0x1a65] = 0x28dcc, [0x1a66] = 0x28dcd,
  [0x1a67] = 0x29051, [0x1a6a] = 0x29245, [0x1a6b] = 0x29559, [0x1a6c] = 0x2955a,
  [0x1a6d] = 0x2955b, [0x1a6e] = 0x29640, [0x1a6f] = 0x297cf, [0x1a70] = 0x2985e,
  [0x1a71] = 0x2997b, [0x1a72] = 0x29a93, [0x1a73] = 0x29bbc, [0x1a74] = 0x29d16,
  [0x1a75] = 0x29d30, [0x1a77] = 0x29e8d, [0x1a78] = 0x29e8e, [0x1a79] = 0x9c2f,
  [0x1a7a] = 0x29e8f, [0x1a7b] = 0x29e90, [0x1a7c] = 0x29e91, [0x1a7d] = 0x29e92,
  [0x1a7e] = 0x29e93, [0x1a7f] = 0x29e94, [0x1a80] = 0x29e95, [0x1a81] = 0x29e96,
  [0x1a82] = 0x2a123, [0x1a83] = 0x2a114, [0x1a84] = 0x2a2db, [0x1a85] = 0x2a40a,
  [0x1a86] = 0x2a5ce, [0x1a87] = 0x20466, [0x1a88] = 0x20468, [0x1a89] = 0x20467,
  [0x1a8a] = 0x204d3, [0x1a8b] = 0x21163, [0x1a8c] = 0x21504, [0x1a8d] = 0x2169e,
  [0x1a8e] = 0x21926, [0x1a8f] = 0x2192a, [0x1a90] = 0x2296c, [0x1a91] = 0x22964,
  [0x1a92] = 0x22e8f, [0x1a93] = 0x22e90, [0x1a94] = 0x23034, [0x1a95] = 0x2330e,
  [0x1a96] = 0x23825, [0x1a97] = 0x23826, [0x1a98] = 0x23827, [0x1a9a] = 0x23828,
  [0x1a9b] = 0x6b0c, [0x1a9d] = 0x23829, [0x1a9e] = 0x2382a, [0x1a9f] = 0x2382b,
  [0x1aa0] = 0x2382c, [0x1aa1] = 0x23c3e, [0x1aa2] = 0x24142, [0x1aa3] = 0x24143,
  [0x1aa4] = 0x24133, [0x1aa5] = 0x244c1, [0x1aa6] = 0x244c2, [0x1aa7] = 0x244c3,
  [0x1aa8] = 0x24ad6, [0x1aa9] = 0x24ad7, [0x1aaa] = 0x255e1, [0x1aab] = 0x255e2,
  [0x1aac] = 0x2572b, [0x1aad] = 0x25ad1, [0x1aae] = 0x25dcc, [0x1aaf] = 0x25dcd,
  [0x1ab0] = 0x25dce, [0x1ab1] = 0x25dcf, [0x1ab2] = 0x261de, [0x1ab3] = 0x261df,
  [0x1ab4] = 0x261e0, [0x1ab5] = 0x261e1, [0x1ab6] = 0x268f6, [0x1ab7] = 0x2712c,
  [0x1ab9] = 0x27154, [0x1aba] = 0x2712d, [0x1abb] = 0x2712e, [0x1abc] = 0x27541,
  [0x1abd] = 0x27542, [0x1abe] = 0x277c7, [0x1abf] = 0x278a0, [0x1ac0] = 0x278a1,
  [0x1ac1] = 0x27b70, [0x1ac2] = 0x27b71, [0x1ac3] = 0x27b72, [0x1ac5] = 0x27e34,
  [0x1ac6] = 0x27e35, [0x1ac7] = 0x27e36, [0x1ac8] = 0x27fa6, [0x1ac9] = 0x281cc,
  [0x1acb] = 0x28299, [0x1acc] = 0x288f4, [0x1acd] = 0x28b9b, [0x1ace] = 0x28b9e,
  [0x1acf] = 0x28b9f, [0x1ad0] = 0x28ba0, [0x1ad1] = 0x28ba1, [0x1ad2] = 0x28ba2,
  [0x1ad3] = 0x28ba3, [0x1ad4] = 0x28ba4, [0x1ad5] = 0x28ba5, [0x1ad6] = 0x28ba6,
  [0x1ad7] = 0x28ba7, [0x1ad8] = 0x28ba8, [0x1ad9] = 0x28ddb, [0x1ada] = 0x28ddc,
  [0x1adb] = 0x28ddd, [0x1adc] = 0x28dd7, [0x1add] = 0x28dde, [0x1ade] = 0x28ddf,
  [0x1adf] = 0x29059, [0x1ae0] = 0x29446, [0x1ae1] = 0x29565, [0x1ae2] = 0x29649,
  [0x1ae4] = 0x297dd, [0x1ae5] = 0x29995, [0x1ae6] = 0x29996, [0x1ae7] = 0x29997,
  [0x1ae8] = 0x29bd4, [0x1ae9] = 0x29bd6, [0x1aea] = 0x29d26, [0x1aeb] = 0x9c47,
  [0x1aec] = 0x4c88, [0x1aed] = 0x29ec8, [0x1aee] = 0x29ec5, [0x1aef] = 0x29ec6,
  [0x1af0] = 0x29ec7, [0x1af1] = 0x2a151, [0x1af2] = 0x2a152, [0x1af3] = 0x2a153,
  [0x1af4] = 0x2a154, [0x1af5] = 0x2a155, [0x1af6] = 0x2a2de, [0x1af8] = 0x2046d,
  [0x1af9] = 0x204d5, [0x1afa] = 0x21172, [0x1afb] = 0x221ae, [0x1afc] = 0x2230f,
  [0x1afd] = 0x2296b, [0x1afe] = 0x2296d, [0x1aff] = 0x2296e, [0x1b00] = 0x22978,
  [0x1b01] = 0x22e98, [0x1b02] = 0x22e99, [0x1b03] = 0x22e9a, [0x1b04] = 0x23844,
  [0x1b05] = 0x2385c, [0x1b06] = 0x23845, [0x1b07] = 0x23b01, [0x1b08] = 0x24158,
  [0x1b09] = 0x24159, [0x1b0a] = 0x2415a, [0x1b0c] = 0x244d9, [0x1b0d] = 0x244d6,
  [0x1b0e] = 0x24ada, [0x1b0f] = 0x24adf, [0x1b10] = 0x24adb, [0x1b11] = 0x24d03,
  [0x1b12] = 0x255ef, [0x1b13] = 0x7936, [0x1b14] = 0x25733, [0x1b15] = 0x25ad2,
  [0x1b16] = 0x25de1, [0x1b17] = 0x25f6